/*
 * Program to read/write raw diskette images directly to a floppy disk.
 *
 * ?COPY.TXT 1991-2005 Dave Dunfield
 *  -- see COPY.TXT --
 *
 * Permission granted to use for personal (non-commercial) use only.
 *
 * Compile command: cc xdisk -fop
 */
#include <stdio.h>
#include <file.h>
#define	Debug(a)	//printf a;
#define	Debug1(a)	//printf a;

#define	MAXSECTOR	18	/* Maximum # sectors on a diskette */

/* Disk track buffers */
unsigned char buffer[MAXSECTOR*1024], buffer1[MAXSECTOR*1024];

/* General variables */
unsigned
	V,
	critical_error[2];/* Saved critical error handler vector */
char
	drive_type;		/* Storage for original diskette setup */

#define	O_READ	0x01
#define	O_WRITE	0x02
#define	O_VERIF	0x04
#define	O_DOUB	0x08
#define	O_FMT	0x10
#define	O_LOW	0x20

/* Copy procedure global variables */
unsigned char
	*Ptr,
	Opt = O_DOUB,
	Tmin, Tmax,
	drive = -1,		/* Active copy drive */
	tracks,			/* Number of tracks on drive */
	sectors,		/* Number of sectors on each track */
	*DO,			/* Current operation */
	DS;				/* Error status of operation */

static char *drive_type_table[] = {
	"None",
	"5.25\" low density (360K)",
	"5.25\" high density (1.2M)",
	"3.5\" low density (720K)",
	"3.5\" high density (1.44M)" };

#include "R:\Help.h"

/*
 * Read a master file from a diskette
 */
read_disk(char *file)
{
	unsigned track, size;
	HANDLE fp;

	if(!(fp = open(file, F_WRITE))) {
		printf("Unable to open image file '%s' for WRITE\n", file);
		return -1; }

	/* Set up drive */
	if(DS = initialize_drive())
		goto disk_error;

	size = (unsigned)sectors * 512;
	for(track = Tmin; track < Tmax; ++track) {
		printf("\b\b%-2u", track);
		if(read_track(buffer, track, 0))
			goto disk_error;
		if((Opt & O_DOUB) && read_track(buffer+size, track, 1))
			goto disk_error;

		write(buffer, size, fp);
		if(Opt & O_DOUB)
			write(buffer+size, size, fp); }

	close(fp);
	return 0;

disk_error:
	printf("Disk error: Operation=%s, status=%04x\n", DO, DS);
	delay(1000);
	if(fp)
		close(fp);
	return -2;
}

/*
 * Write the diskette from a master file
 */
write_disk(char *file)
{
	unsigned i, track, size, retry;
	unsigned char *ptr, sbuf[MAXSECTOR*4];
	HANDLE fp;

	/* Open the master file */
	if(!(fp = open(file, F_READ))) {
		printf("Cannot open image file '%s' for READ\n", file);
		return -2; }

	/* Set up drive */
	retry = 3;
	if(DS = initialize_drive())
		goto disk_error1;

	size = (unsigned)sectors * 512;
	for(track = Tmin; track < Tmax; ++track) {
		printf("\b\b%-2u", track);
		/* Read in buffer for transfer */
		if(read(buffer, size, fp) != size) {
			printf("Error reading image file.\n");
			goto exit; }
		if((Opt & O_DOUB) && (read(buffer+size, size, fp) != size)) {
			printf("Error reading image file.\n");
			goto exit; }
retry_disk:
		/* Format the disk if selected */
		if(Opt & O_FMT) {
			ptr = sbuf;
			for(i=1; i <= sectors; ++i) {
				*ptr++ = track;
				*ptr++ = 0;
				*ptr++ = i;
				*ptr++ = 2; }
			if(format_track(sbuf, track, 0))
				goto disk_error;
			if(Opt & O_DOUB) {
				ptr = sbuf + 1;
				for(i=1; i <= sectors; ++i) {
					*ptr = 1;
					ptr += 4; }
				if(format_track(sbuf, track, 1))
					goto disk_error; } }

		/* Write the data on the diskette */
		if(write_track(buffer, track, 0))
			goto disk_error;
		if((Opt & O_DOUB) && write_track(buffer+size, track, 1))
			goto disk_error;

		/* Verify the data on the diskette */
		if(!(Opt & O_VERIF)) {
			if(read_track(buffer1, track, 0))
				goto disk_error;
			if(read_track(buffer1+size, track, 1))
				goto disk_error;

			if(compare(buffer, buffer1, (Opt & O_DOUB) ? size*2 : size)) {
				DO = "VERIFY";
				goto disk_error; } } }

	close(fp);
	return 0;

disk_error:
	if(--retry) {
		initialize_drive();
		goto retry_disk; }
disk_error1:
	printf("Disk error: Operation=%s, status=%04x\n", DO, DS);
	delay(1000);
exit:
	if(fp)
		close(fp);
	return -1;
}

/*
 * High speed compare of two block of memory
 */
compare(block1, block2, size)
asm {
		PUSH	DS				; Save DATA segment
		POP		ES				; Set EXTRA segment
		MOV		SI,8[BP]		; Get first string
		MOV		DI,6[BP]		; Get second string
		MOV		CX,4[BP]		; Get count
		XOR		AX,AX			; Assume success
	REPE CMPSB					; Do the compare
		JZ		cok				; Its ok
		DEC		AX				; Set -1 (fail)
cok:
}

/*
 * Identify the type of diskette to be used
 */
initialize_drive()
{
	DO = "INITIALIZE";

	asm {
; Reset the drive
		MOV		DL,DGRP:_drive	; Get drive id
		XOR		AH,AH			; Function 0 - reset disk
		INT		13h				; Try it out
; Select the drive type
		MOV		CL,DGRP:_sectors; Get sectors
		MOV		CH,DGRP:_tracks	; Get tracks
		DEC		CH				; Adjust
		MOV		DL,DGRP:_drive	; Get drive number
		MOV		AH,18h			; Set media type for format
		INT		13h				; Call BIOS
		MOV		AL,AH			; Get result
		JC		error			; Error has occured
; Copy over parameter table
		MOV		SI,DI			; Si = source offset
		PUSH	DS				; Save data seg
		PUSH	ES				; Save extra
		XOR		BX,BX			; Get zero
		MOV		DS,BX			; ES = 0
		MOV		BX,0078h		; Point to table
		MOV		DI,[BX]			; Get dest offset
		INC		BX				; Advance
		INC		BX				; One word
		MOV		ES,[BX]			; Get dest segment
		POP		DS				; Get source segment
		MOV		CX,11			; size of table
	rep	MOVSB					; Perform the move
		POP		DS				; Restore data seg
		XOR		AX,AX			; Zero result
error:
	}
}

/*
 * Format a track on the floppy diskette
 */
format_track(buffer, track, head)
{
	DO="FORMAT";

	asm {
		MOV		DI,5			; Retry count
fagain:	PUSH	DS				; Save data seg
		POP		ES				; Set extra seg
		MOV		BX,8[BP]		; Get buffer address
		MOV		CH,6[BP]		; Get track
		MOV		DH,4[BP]		; Get head
		MOV		DL,DGRP:_drive	; Get drive
		MOV		AL,DGRP:_sectors; Get # sectors
		MOV		AH,05h			; Format track command
		INT		13h				; Call bios
		JNC		fok				; It worked
		MOV		DGRP:_DS,AH		; Save command status
		XOR		AH,AH			; Reset command
		INT		13h				; Ask BIOS
		DEC		DI				; Reduce count
		JNZ		fagain			; Do it again sam
		MOV		AX,-1			; Indicate failure
		JMP	SHORT fexit			; And leave
fok:	XOR		AX,AX			; Indicate success
fexit:
	}
}

/*
 * Write a track on the floppy diskette
 */
write_track(buffer, track, head)
{
	DO="WRITE";

	asm {
		MOV		DI,5			; Retry count
wagain:	PUSH	DS				; Save data seg
		POP		ES				; Set extra seg
		MOV		BX,8[BP]		; Get buffer address
		MOV		CH,6[BP]		; Get track
		MOV		DH,4[BP]		; Get head
		MOV		DL,DGRP:_drive	; Get drive
		MOV		AL,DGRP:_sectors; Get # sectors
		MOV		AH,03h			; WRITE command
		MOV		CL,1			; Start with sector 0
		INT		13h				; Call bios
		JNC		wok				; It worked
		MOV		DGRP:_DS,AH		; Save command status
		XOR		AH,AH			; Reset command
		INT		13h				; Ask BIOS
		DEC		DI				; Reduce count
		JNZ		wagain			; Do it again sam
		MOV		AX,-1			; Indicate failure
		JMP	SHORT wexit			; And leave
wok:	XOR		AX,AX			; Indicate success
wexit:
	}
}

/*
 * Read a track from the floppy diskette
 */
read_track(buffer, track, head)
{
	DO="READ";

	asm {
		MOV		DI,5			; Retry count
ragain:	PUSH	DS				; Save data seg
		POP		ES				; Set extra seg
		MOV		BX,8[BP]		; Get buffer address
		MOV		CH,6[BP]		; Get track
		MOV		DH,4[BP]		; Get head
		MOV		DL,DGRP:_drive	; Get drive
		MOV		AL,DGRP:_sectors; Get # sectors
		MOV		AH,02h			; READ command
		MOV		CL,1			; Start with sector 0
		INT		13h				; Call bios
		JNC		rok				; It worked
		MOV		DGRP:_DS,AH		; Record the status
		XOR		AH,AH			; Reset command
		INT		13h				; Ask BIOS
		DEC		DI				; Reduce count
		JNZ		ragain			; Do it again sam
		MOV		AX,-1			; Indicate failure
		JMP	SHORT rexit			; And leave
rok:	XOR		AX,AX			; Indicate success
rexit:
	}
}

unsigned Value(void)
{
	unsigned c;
	unsigned char *p;
	V = 0;
	p = Ptr;
Debug1(("[%s{", Ptr))
	while((c = *Ptr - '0') < 10) {
Debug1(("%c", *Ptr))
		V = (V * 10) + c;
		++Ptr; }
Debug1(("}%s]\n", Ptr))
	if(Ptr == p)
		return 255;
	return *Ptr;
}

main(int argc, char *argv[])
{
	int i;
	char *filename, c;

	filename =  i = 0;
	while(++i < argc) {
		if(*(Ptr = argv[i]) == '-') {
			++Ptr;
/*ChtTxt R:\Help.H

Use:	XDISK <-R or -W> <drive> <image file> [start,end] [options]

opts:	-R		Read disk image into file
		-W		Write disk image from file
		-L		Low density (360 in 1.2 or 720 in 1.44)
		-S		Single sided (default is 2 sided)
		-F		Format diskette while writing
		-V		inhibit read after write Verify

?COPY.TXT 1991-2022 Dave Dunfield.
 -- see COPY.TXT -- - Free for personal use.
*/
o1:			switch(c = toupper(*Ptr++)) {
			default	:
				Debug1(("Bad -parameter"))
				goto he;
			case 'R':	c = O_READ;		goto o2;
			case 'W':	c = O_WRITE;	goto o2;
			case 'S':	c = O_DOUB;		goto o2;
			case 'L':	c = O_LOW;		goto o2;
			case 'F':	c = O_FMT;		goto o2;
			case 'V':	c =  O_VERIF;
o2:				Opt ^= c; }
			if(*Ptr) goto o1;
			continue; }
		if(!strcmp(Ptr+1, ":")) switch(c = toupper(*Ptr)) {
			default	:	goto he1;
			case 'A':
			case 'B':
				Debug1(("Drive:%c\n", c))
				drive = c - 'A';
				continue; }
		if(!filename) {
			filename = Ptr;
			continue; }
		if(Tmin || Tmax) {
			Debug1(("Already set range"))
			goto he; }
		c = Value();
		Tmin = V;
		switch(c) {
		default	:
			Debug1(("Bad,"))
			goto he;
		case 0	:	continue;
		case ',':	++Ptr; }
		if(Value()) {
			Debug1(("BadEOL"))
			goto he; }
		Tmax = V; }
	Debug1(("Tmin=%u Tmax=%u\n", Tmin, Tmax))
	if(drive > 1) {
he1:	Ptr = "Must specify drive A: or B:";
he0:	fputs(Ptr, stdout);
		putc('\n', stdout);
he:		Ptr = Help;
		while(c = *Ptr++) {
			if(c & 0x80) {
				while(c-- & 0x7F)
					putc(' ', stdout);
				continue; }
			putc(c, stdout); }
		return; }
	if(!filename) {
		Ptr = "Must specify a filename.";
		goto he0; }
	switch(Opt & (O_READ|O_WRITE)) {
	default	:
		Ptr = "Must specify ONE of -R or -W";
		goto he0;
	case O_READ:
		if(Opt & O_FMT) {
			Ptr = "-F is valid only with -W";
			goto he0; }
		if(Opt & O_VERIF) {
			Ptr = "-V is valid only with -W";
			goto he0; }
	case O_WRITE:	; }
		
	/* Read original floppy drive types from CMOS */
	asm {
		MOV		AL,10h		; Floppy drive types
		CLI					; Inhibit interrupts
		OUT		70h,AL		; Write type
		IN		AL,71h		; Read data
		STI					; Re-allow interrupts
		MOV		DGRP:_drive_type,AL; Set old drive types
	}
	drive_type = drive ? drive_type & 0x0F : drive_type >> 4;
	printf("Drive %c: %s\n", drive+'A', (drive_type > 4) ? "Unknown" : drive_type_table[drive_type]);
	switch(drive_type) {
		case 0x01 :			/* 5.25 360k */
			if(!(Opt & O_LOW)) {
				sectors = 9;
				tracks = 40;
				break; }
	nolow:	Ptr = "-L cannot be used with this drive.";
			goto he0;
		case 0x02 :			/* 5.25 1.2m */
			if(Opt & O_LOW)
				sectors = 9, tracks = 40;
			else
				sectors = 15, tracks = 80;
			break;
		case 0x03 :			/* 720 */
			if(Opt & O_LOW) goto nolow;
			sectors = 9;
			tracks = 80;
			break;
		case 0x04 :			/* 1.44 */
			if(Opt & O_LOW)
				sectors = 9, tracks = 80;
			else
				sectors = 18, tracks = 80;
			break;
		default:
			Ptr = "This drive type cannot be used.";
			goto he0; }

	if((Tmax > tracks) || !Tmax)
		Tmax = tracks;
	printf("Image=%uk (%u tracks, %u sectors, %s, %s density)\n",
		(Tmax-Tmin)*sectors/((Opt & O_DOUB) ? 1 : 2),
		Tmax-Tmin, sectors, (Opt & O_DOUB) ? "2 sides" : "1 side",
			(Opt & O_LOW) ? "Low" : "High");

/* Install our own critical error handler */
	asm {
		MOV		AX,3524h	; Get critical error handler
		INT		21h			; Call DOS
		MOV		DGRP:_critical_error,BX; Save offset
		MOV		DGRP:_critical_error+2,ES; Save segment
		MOV		DX,OFFSET chand; Get handler offset
		MOV		AX,2524h	; Set critical error handler
		INT		21h			; Call DOS
	}

	if(Opt & O_WRITE) {
		if(Opt & O_FMT)
			printf("Formatting and ");
		printf("Writing track: --");
		write_disk(filename); }
	else {
		printf("Reading track: --");
		read_disk(filename); }

	asm {
		PUSH	DS		; Save our segment
		MOV		DX,DGRP:_critical_error; Get offset
		MOV		DS,DGRP:_critical_error+2; Get segment
		MOV		AX,2524h; Set critical error handler
		INT		21h		; Call DOS
		POP		DS		; Restore DS
	}
}

/*
 * Critical error handler
 */
asm {
chand:	MOV		AL,3		; Return "FAIL" status
		IRET				; Go home
}
