/*
 * I/R code file editor.
 *
 * ?COPY.TXT 1994-2005 Dave Dunfield
 *  -- see COPY.TXT --.
 */
#include <stdio.h>			/* General I/O definitions */
#include <window.h>			/* Window  I/O definitions */
#include <comm.h>			/* Serial  I/O definitions */

#define	MAX_CODES	100		/* Maximum number of codes */

unsigned char IR_codes[MAX_CODES][116];
unsigned IR_top = 0, comport = 1;

int jmp_buf[3];				/* Longjump return buffer */

/*
 * Get BIOS tick counter
 */
unsigned get_ticks(void) asm
{
	MOV	AX,40h
	MOV	ES,AX
	MOV	BX,6Ch
	MOV	AX,ES:[BX]
}

/*
 * Get input character in timeout
 */
int Cgett(unsigned ticks)
{
	int c;
	unsigned f;
	f = get_ticks();
	do {
		if((c = Ctestc()) != -1)
			return c; }
	while((get_ticks() - f) <= ticks);
	return -1;
}

/*
 * Send a command to the controller
 */
void send_command(int c)
{
	while(Ctestc() != -1);
	Cputc(c);
	if(Cgett(18) != c)
		longjmp(jmp_buf, "Command did not echo!");
}

/*
 * Record an I/R code
 */
int record(unsigned char *ptr)
{
	int c, d;
again:
	wprintf("\nPress REMOTE key...");
	send_command('L');
	if(Cgett(500) != '*') {
		wprintf("Error! - retry");
		goto again; }
	wprintf("Count=%d ", *ptr++ = c = d = Cgett(18));
	while(c--)
		*ptr++ = Cgett(18);
	wprintf("K=%d", *ptr = Cgett(18));
	if(Cgett(18) != '.')
		longjmp(jmp_buf, "Error! no '.'!");
	return d;
}

/*
 * Transmit an I/R code
 */
void transmit(unsigned char *ptr)
{
	int c;
	/* Send command to remote */
	if(c = *ptr++) {
		send_command('S');
		Cputc(c);
		while(c--)
			Cputc(*ptr++);
		Cputc(*ptr);
		if(Cgett(18) != '.')
			longjmp(jmp_buf, "Error! no '.'!"); }
}

/*
 * Main I/R editor program
 */
main(int argc, char *argv[])
{
	int i, j, k;
	unsigned char *ptr;
	FILE *fp;

	fputs("\nI/R Code Editor, Version 1.0\n\n?COPY.TXT 1994-2005 Dave Dunfield\n -- see COPY.TXT --.\n", stderr);

	if(argc < 2)
		abort("\nUse: iredit <filename> [comport]\n");

	/* If file exists, load it in */
	if(fp = fopen(argv[1], "rvb")) {
		while((i = getc(fp)) >= 0) {
			ptr = IR_codes[IR_top++];
			*ptr++ = i++;
			while(i--) {
				if((j = getc(fp)) < 0)
					abort("Incorrect file format!");
				*ptr++ = j; } }
		fclose(fp); }

	if(argc > 2)
		comport = atoi(argv[2]);

	/* Open COMM port */
	if(Copen(comport, _9600, PAR_NO|DATA_8|STOP_1, OUTPUT_2))
		abort("Cannot open comm port");
	Cflags |= TRANSPARENT;

	/* Open a full-screen window */
	wopen(0, 0, 80, 24, WSAVE|WCOPEN|WBOX1|NORMAL);
	k = 0;

	if(ptr = setjmp(jmp_buf)) {
		wclwin();
		wgotoxy(1, 10);
		wprintf("I/R command protocol error: %s\n", ptr);
		wprintf("\n\nPress a key...");
		wgetc();
		wclwin(); }

	/* Display the current IR code information */
display:
	if(k < 0)
		k = 0;
	wgotoxy(4, 1);
	ptr = IR_codes[k];
	wprintf("Code %-2u of %-2u: Length=%-3u Gap=%-3u\n", k, IR_top, j = *ptr, ptr[j+1]);
	for(i=0; i < j; ++i) {
		if(!(i & 0x0F)) {
			wprintf("\n    %04x:", i);
			wcleol(); }
		wprintf(" %02x", ptr[i+1]); }
	wcleow();

	/* Prompt for user command & execute */
	wgotoxy(1, 20);
	wprintf("Commands:  D)elete I)nsert N)ew P)lay R)ecord\n ");
	wprintf("Up=previous Down=next Home=first End=last ESC=exit ?");
	switch(toupper(wgetc())) {
		case _KUA :		/* Up: previous code */
			if(k)
				--k;
			break;
		case _KDA :		/* Down: next code */
			if((k+1) < IR_top)
				++k;
			break;
		case _KHO :		/* Home: first code */
			k = 0;
			break;
		case _KEN :		/* End: last code */
			k = IR_top - 1;
			break;
		case 'D' :		/* Delete this code */
			if(k >= IR_top)
				break;
			for(i=k; i < IR_top; ++i)
				memcpy(IR_codes[i], IR_codes[i+1], sizeof(IR_codes[0]));
			if(k >= --IR_top)
				k = IR_top - 1;
			break;
		case 'I' :		/* Insert a code */
			if(IR_top >= MAX_CODES)
				break;
			for(i=IR_top++; i > k; --i)
				memcpy(IR_codes[i], IR_codes[i-1], sizeof(IR_codes[0]));
			memset(ptr, 0, sizeof(IR_codes[0]));
			break;
		case 'P' :		/* Playback a code */
			transmit(ptr);
			break;
		case 'N' :		/* New code (at end) */
			k = IR_top;
		case 'R' :		/* Record over code */
			wclwin();
			record(ptr = IR_codes[k]);
			if((k >= IR_top) && *ptr)
				++IR_top;
			wclwin();
			break;
		case 0x1B :		/* ESCAPE: exit program */
			goto quit; }
	goto display;

	/* Close resources, write file & exit */
quit:
	wclwin();
	wgotoxy(15, 10);
	wprintf("R)esume  S)ave-file  Q)uit/lose-changes");
	switch(toupper(wgetc())) {
		default:
			goto quit;
		case 'R' :
			goto display;
		case 'S' :
			wclose();
			if(fp = fopen(argv[1], "wvb")) {
				for(i=0; i < IR_top; ++i) {
					ptr = IR_codes[i];
					putc(j = *ptr++, fp);
					++j;
					while(j--)
						putc(*ptr++, fp); }
				fclose(fp); }
			break;
		case 'Q' :
			wclose(); }
	Cclose();
}
