/*
 * AVR LOADer
 *
 * ?COPY.TXT 1998-2005 Dave Dunfield
 *  -- see COPY.TXT --.
 *
 * Compile with DDS Micro-C/PC: cc avrload -fop [TARGET=n]
 */
#include <stdio.h>

/* Compile modes (target interfaces) [use TARGET=n] */
#define	_DIRECT_	0		/* Direct (DDS) LPT connection */
#define	_KANDA_		1		/* Kanda basic dongal */
#define	_KDONG1_	2		/* Kanda enhanced dongal */
#define	_MP3_		3		/* Rick's MP3 box */

#if TARGET == _DIRECT_
#define Target Direct LPT
/* Sig    Pin   Dir   Addr.Bit  S   LPT Name
 * -----------------------------------------
 * !MOSI =  1 : OUT : LPT+2.0 : - : -STROBE
 * !MISO = 11 : IN  : LPT+1.7 : - : BUSY
 * SCK   = 16 : OUT : LPT+2.2 : + : -INIT
 * !RESET= 17 : OUT : LPT+2.3 : - : -SLCTIN
 * !LED  = 14 : OUT : LPT+2.1 : - : -AutoFD
 *								   xxxxRCLD */
#define	ASSERT_RESET	0x0A	/* 00001011 Reset=0, Clk=0, Led=0, Data=0 */
#define	RAISE_CLOCK		0x0F	/* 00001111 Reset=0, Clk=1, Led=0, Data=0 */
#define	RELEASE_RESET	0x03	/* 00000011 Reset=1, Clk=0, Led=0, Data=0 */
#define	FINISH_RESET	0x0C	/* 00001100 Reset=0, Clk=1, Led=1, Data=1 */
#define	FINISH_RUN		0x04	/* 00000100 Reset=1, Clk=1, Led=1, Data=1 */

#elif TARGET == _KANDA_
#define Target Kanda Dongal
/* Sig    Pin   Dir   Addr.Bit  S   LPT Name
 * -----------------------------------------
 * MOSI  =  5 : OUT : LPT+0.5 : + : Data 3
 * MISO  = 10 : IN  : LPT+1.6 : + : SLCT
 * SCK   =  4 : OUT : LPT+0.4 : + : Data 2
 * !RESET=  7 : OUT : LPT+0.7 : + : Data 5
 * !LED  =  6 : OUT : LPT+0.6 : + : Data 4
 * !PRG  =  2 : OUT : LPT+0.2 : + : Data 0 \_ Hold low to enable dongle
 * !MON  =  3 : OUT : LPT+0.3 : + : Data 1 /
 *                                 RLDCMPxx */
#define	ASSERT_RESET	0x00	/* 00000000 Rst=0, Led=0, Data=0, Clk=0, M/P=0 */
#define	RAISE_CLOCK		0x10	/* 00010000 Rst=0, Led=0, Data=0, Clk=1, M/P=0 */
#define	RELEASE_RESET	0x80	/* 10000000 Rst=1, Led=0, Data=0, Clk=0, M/P=0 */ 
#define	FINISH_RESET	0x70	/* 01110000 Rst=0, Led=1, Data=1, Clk=1, M/P=0 */
#define	FINISH_RUN		0xFF	/* 11111111 Rst=1, Led=1, Data=0, Clk=0, M/P=1 */

#elif TARGET == _KDONG1_
#define Target Kanda Smart Dongal
/* Sig    Pin   Dir   Addr.Bit  S   LPT Name
 * -----------------------------------------
 * !RESET=  1 : OUT : LPT+2.0 : - : -STROBE
 * !MISO = 11 : IN  : LPT+1.7 : - : BUSY
 * MOSI  = 16 : OUT : LPT+2.2 : + : -INIT
 * !SCK  = 14 : OUT : LPT+2.1 : - : -AutoFD
 *								   xxxxxDCR */
#define	ASSERT_RESET	0x03	/* 00000011 Reset=0, Clk=0, Data=0 */
#define	RAISE_CLOCK		0x01	/* 00000001 Reset=0, Clk=1, Data=0 */
#define	RELEASE_RESET	0x02	/* 00000010 Reset=1, Clk=0, Data=0 */
#define	FINISH_RESET	0x05	/* 00000101 Reset=0, Clk=1, Data=1 */
#define	FINISH_RUN		0x04	/* 00000100 Reset=1, Clk=1, Data=1 */

#elif TARGET == _MP3_
#define Target MP3 Box
/* Sig    Pin   Dir   Addr.Bit  S   LPT Name
 * -----------------------------------------
 * MOSI  = 16 : OUT : LPT+2.2 : + : -INIT
 * MISO  =  9 : IN  : LPT+0.7 : + : D7
 * SCK1  = 14 : OUT : LPT+2.1 : - : -AutoFD
 * SCK2  = 17 : OUT : LPT+2.3 : - : -SLCTIN
 * !RESET=  1 : OUT : LPT+2.0 : - : -STROBE
 *                                   00I0CDCR */
unsigned char ASSERT_RESET= 0x23; /* 00100011 Reset=0, Clk=0, Data=0 */
#define		  Assert_reset  0x29  /* Alternate Clk=0 */
#define		  RAISE_CLOCK	0x21  /* 00100001 Reset=0, Clk=1, Data=0 */
unsigned char RELEASE_RESET=0x22; /* 00100010 Reset=1, Clk=0, Data=0 */
#define		  Release_reset 0x28  /* Alternate Clk=0 */
#define 	  FINISH_RESET 	0x05  /* 00000101 Reset=0, Clk=1, Data=1 */
#define		  FINISH_RUN 	0x04  /* 00000100 Reset=1, Clk=1, Data=1 */

#else
	#error TARGET is not defined correctly.
#endif

#message Compiling for Target

#define	NOSIZE			0xFF55	/* Indicates no size set */
#define	CAL_FACTOR		90		/* Delay loop calibration factor */

char help_g[] = { "\n\
AVR Loader v1.1 - " #
	#if TARGET == _DIRECT_
via PC LPT port
	#elif TARGET == _KANDA_
via Kanda ISP Dongal
	#elif TARGET == _KDONG1_
for Kanda Smart Dongal
	#elif TARGET == _MP3_
for MP3 player
	#endif
".\n\n\
Use: avrload <command [filename]>... [options]\n\n\
?COPY.TXT 1998-2005 Dave dunfield\n\
 -- see COPY.TXT --.\n\n\
http://www.dunfield.com\n" };

char help_o[] = { "Options:\n\n\
  -B		save Binary files\n\
  -I		save Intel format files\n\
  -M		save Motorola format files (default)\n\
  -P		forced Paged load mode (override device)\n\
  -Q		Quiet: no informational messages\n" #
#if TARGET == _MP3_
  -X		load alternate processor\n
#endif
"\
  B=address	set Base address (default from first record)\n\
  D=value	set Default 'fill' value (FF)\n\
  E=size	set size of Eeprom memory (override device)\n\
  F=size	set size of Flash memory  (override device)\n\
  P=1-3/addr	set Parallel port number/address\n\
  R=length	set Record length for intel/motorola output files\n\
  U=maxFF	set maximum Unprogrammed values before file split\n\
  ~P=value	override Programming delay (microseconds)\n\
  ~R=value	override Reset delay (milliseconds)\n\
  ~S=value	override SPI clock frequency (delay loop count: " };
char help_o1[] = { ")\n\n\
Values can be entered as: decimal, $hex, @octal or %binary\n\
Options have effect for all commands entered at one time.\n" };

#if TARGET == _DIRECT_
char help_w[] = { "Wiring:\n\n\
 LPT-Pin#   Target connection\n\
    1		MOSI	(SPI data  from PC to Target)\n\
   11		MISO	(SPI data  from Target to PC)\n\
   14		!LED	(Driven low when programming)\n\
   16		SCK	(SPI clock from PC to target)\n\
   17		!RESET 	(Driven low to reset target)\n\
 18-25		Ground\n\n\
To protect the target from being overdriven by the PC (especially systems\n\
operating at < 5V),  The MOSI and SCK signals should be connected through\n\
two simple diode/pull-up circuits:\n\
				+--/\\/\\/\\--- +5v (Target)\n\
			   *	|    1K\n\
     MOSI/SCK from PC -----|<---+-----------  MOSI/SCK to Target\n\n\
* Any low drop, high-speed diode should work.\n\n\
AVRLOAD sets MOSI and SCK high when it exit's, which gives the added\n\
benefit of not driving these signals when this circuit is used.\n" };

#elif TARGET == _KANDA_
char help_w[] = { "Wiring:\n\n\
This version of AVRLOAD works with the ISP Dongle from Kanda Systems.\n\n\
Ready made cables are available from Kanda, and included with their workshop\n\
boards. The ISP Dongle mates with this ISP connector on the target system:\n\n\
	+---------------+	1 = MOSI	7=SCK\n\
	| 2  4  6  8  10|	2 = VCC		9=MISO\n\
	| 1  3  5  7  9 |	3 = !Led	4,6,8,10 = Gound\n\
	+-----[   ]-----+	5 = !Reset\n\n\
For more information, contact:\n\n\
Kanda Systems Limited           Tel: 44-01970621030 Fax: 44-01970621040\n\
Unit 17 Glanyrafon Enterprise Park Aberystwyth Ceredigion SY23 3JQ U.K.\n\
email: kevin@kanda-systems.com  http://www.kanda-systems.com\n" };
/*
 LPT-Pin#   Target connection (Kanda ISP Dongle)\n\
   2-5		!ENABLE	(Driven LOW to enable Dongle)\n\
    6		SCK	(SPI clock from PC to target)\n\
    7		MOSI	(SPI data  from PC to Target)\n\
    8		!LED	(Driven low when programming)\n\
    9		!RESET 	(Driven low to reset target)\n\
   10		MISO	(SPI data  from Target to PC)\n\
 18-25		Ground\n\n\
*/

#elif TARGET == _KDONG1_
char help_w[] = { "Wiring:\n\n\
This version of AVRLOAD loads the firmware into the ISP Smart Dongle from\n\
Kanda Systems. It DOES NOT perform ISP programming via the dongle.\n\n\
There is no wiring, simply plug the dongle into your PC printer port.\n\n\
For more information, contact:\n\n\
Kanda Systems Limited           Tel: 44-01970621030 Fax: 44-01970621040\n\
Unit 17 Glanyrafon Enterprise Park Aberystwyth Ceredigion SY23 3JQ U.K.\n\
email: kevin@kanda-systems.com  http://www.kanda-systems.com\n" };

#elif TARGET == _MP3_
char help_w[] = { "NO HELP\n" };
#endif

char help_c[] = { "\n\
Multiple commands allowed. eg: AVRLOAD ERA FP myfile.hex FV RUN\n\
- Erase the device\n\
- Program flash with MYFILE.HEX (verifies each byte as programmed)\n\
- Verify complete image after programming (not really necessary)\n\
- Release reset and allow target to run\n" };

char *cmd_tab[] = {
	"ERASE\t\t\tErase device (FLASH & EEPROM)",
	"FREAD <file>\t\tRead FLASH  into file",
	"EREAD <file>\t\tRead EEPROM into file",
	"FPROGRAM <file>\tProgram FLASH  from file",
	"EPROGRAM <file>\tProgram EEPROM from file",
	"FVERIFY\t\tVerify buffer with FLASH  (Blank check if empty)",
	"EVERIFY\t\tVerify buffer with EEPROM (Blank check if empty)",
	"LOAD <file>\t\tLoad file into memory buffer (for Verify)",
	"LOCK <value>\t\tProgram lock bits (0-3)",
	"LOW\t\t\tSelect LOW 64k when programming Mega (default)",
	"HIGH\t\t\tSelect HIGH 64k when programming Mega",
	"RUN\t\t\tRelease RESET on exit" };

// Command encoding
#define	ERASE		1		// Erase device
#define	FREAD		2		// Read flash
#define	EREAD		3		// Read EEPROM
#define	FPROGRAM	4		// Program flash
#define	EPROGRAM	5		// Program EEPROM
#define	FVERIFY		6		// Verify flash
#define	EVERIFY		7		// Verify EEPROM
#define	LOAD		8		// Load file
#define	LOCK		9		// Set LOCK bits
#define	LOW			10		// Select LOW 64k for page mode
#define	HIGH		11		// Select HIGH 64k for page mode
#define	RUN			12		// Execute on exit

unsigned
	base = -1,				// Base device address offset
	image,					// Memory image segment
	cptr,					// Command index pointer
	hmask,					// High address mask
	fsize = NOSIZE,			// Flash size
	esize = NOSIZE,			// EEPROM size
	maxff = 5,				// Maximum FF's (unprogrammed) before break
	reclen = 34,			// ASCII/HEX record length
	cport,					// IO port address
	xspidel,				// SPI delay loop counter
	xpgmdel,				// Programming delay (microseconds)
	xresdel=50;				// Reset delay (55ms ticks)
unsigned char
	*ptr,					// General purpose pointer
	*string,				// String/Parsing pointer
	buffer[256],			// General purpose temp/buffer
	exit_condition = FINISH_RESET,	// Run state at program exit
	outfmt,					// Output file format (bin/mot/int)
	verbose = -1,			// Verbose message flag
	defval = 0xFF,			// Default fill value
	Npage = -1,				// NOT paged mode flag
	*cmds[20];				// Command buffer pointers

/*
 * Formatted print of an informational message
 */
register message(args)
	unsigned args;
{
	char buffer[81];
	unsigned l;

	l = _format_(nargs() * 2 + &args, buffer);
	if(verbose) {
		fputs("AVRLOAD: ", stdout);
		fputs(buffer, stdout);
		putc('\n', stdout); }
	return l;
}

/*
 * Formatted print of error message & exit
 */
register error(args)
	unsigned args;
{
	char buffer[81];
	unsigned l;

	l = _format_(nargs() * 2 + &args, buffer);
	if(cport)
		write_pp(FINISH_RESET);
	fputs("AVRLOAD: ", stderr);
	fputs(buffer, stderr);
	putc('\n', stderr);
	exit(-1);
}

/*
 * Calibrate the pulse delay loop
 * This is a cheesy way to insure that the SPI pulses will be
 * long enough. Not terribly accurate, but good enough.
 */
cal_delay() asm
{
		MOV		AX,40h			; BIOS segment
		MOV		ES,AX			; Address it
		XOR		CX,CX			; Clear counter
		XOR		DX,DX			; ""
		MOV		AX,ES:[006Ch]	; Read clock tick
cal1:	ADD		CX,1			; Increment value
		ADC		DX,0			; Carry into high
		JC		cal3			; Timeout
		MOV		BX,ES:[006Ch]	; Get value
		CMP		AX,BX			; Match?
		JZ		cal1			; It matches
		XOR		CX,CX			; Zero low
		XOR		AX,AX			; Zero high
cal2:	ADD		CL,1			; Advance CX
		ADC		AX,0			; Carry into DX
		JC		cal3			; Report error
		CALL	goret			; Break pipeline
		CALL	goret			; Break pipeline
		CMP		BX,ES:[006Ch]	; Does it match?
		JZ		cal2			; Wait for it
		JMP SHORT cal4			; And exit
cal3:	MOV		AX,-1			; Report high
cal4:
} asm { /* Delay loop stub */
xdelay:	MOV		BX,DGRP:_xspidel; Get X delay value
ydelay:	AND		BX,BX			; Delay = 0
		JZ		goret			; Yes, skip
ydel1:	CALL	goret			; Stall pipeline
		CALL	goret			; Stall pipeline
		DEC		BX				; Reduce count
		JNZ		ydel1			; Wait till it expires
goret:	RET
}

/*
 * Write a value to the parallel port outputs
 */
write_pp(value) asm
{
		MOV		AL,4[BP]			; Get value to write
		MOV		DX,DGRP:_cport		; Get I/O port
		OUT		DX,AL				; Write it
}

/*
 * Write a byte to the SPI port
 */
write_spi(c) asm
{
		MOV		CX,8				; Shift 8 bits
		MOV		DX,DGRP:_cport		; Get I/O port
		MOV		AH,4[BP]			; Get data to send
#if TARGET == _DIRECT_		/* Write via DIRECT LPT connection: xxxxRCxD */
wrspi1:	MOV		AL,00001011b		; Reset=0, clk=0, Data=0, Led=0
		ROL		AH,1				; Get bit
		SBB		AL,0				; If bit high, data=0
		OUT		DX,AL				; Write to port
		CALL	xdelay				; Wait a bit
		OR		AL,00000100b		; clk=1
		OUT		DX,AL				; Write it
		CALL	xdelay				; Wait a bit
		LOOP	wrspi1				; Write it all
		AND		AL,11111011b		; clk=0
		OUT		DX,AL				; Write it
#elif TARGET == _KANDA_		/* Write via KANDA Dongle: RLDCMPxx */
wrspi1:	MOV		AL,00000000b		; Rst=0, Led=0, Data=0, Clk=0
		ROL		AH,1				; Get bit
		JNC		wrspi2				; Data=0
		OR		AL,00100000b		; Set data high
wrspi2:	OUT		DX,AL				; Write to port
		CALL	xdelay				; Wait a bit
		OR		AL,00010000b		; clk=1
		OUT		DX,AL				; Write it
		CALL	xdelay				; Wait a bit
		LOOP	wrspi1				; Write it all
		AND		AL,11101111b		; clk=0
		OUT		DX,AL				; Write it
#elif TARGET == _KDONG1_		/* Write via DIRECT LPT connection: xxxxxDCR */
wrspi1:	MOV		AL,00000011b		; Reset=0, clk=0, Data=0
		ROL		AH,1				; Get bit
		JNC		wrspi2				; Data=0
		OR		AL,00000100b		; Set data high
wrspi2:	OUT		DX,AL				; Write to port
		CALL	xdelay				; Wait a bit
		AND		AL,11111101b		; clk=1
		OUT		DX,AL				; Write it
		CALL	xdelay				; Wait a bit
		LOOP	wrspi1				; Write it all
		OR		AL,00000010b		; Clk=0
		OUT		DX,AL				; Write it
#elif TARGET == _MP3_			/* Write via mp3 board xxxxCDCR */
wrspi1:	MOV		AL,DGRP:_ASSERT_RESET ; Reset=0, clk=0, data=0
		ROL		AH,1				; Get bit
		JNC		wrspi2				; Data=0, ok
		OR		AL,00000100b		; Set data high
wrspi2:	OUT		DX,AL				; Write to port
		CALL	xdelay				; Wait a bit
		XOR		AL,DGRP:_RELEASE_RESET; Raise clock
		OUT		DX,AL				; Write it
		CALL	xdelay				; Wait a bit
		LOOP	wrspi1				; Do all bits
		XOR		AL,DGRP:_RELEASE_RESET; Drop clock
		OUT		DX,AL				; Write it
#endif
}

/*
 * Read a byte from the SPI port
 */
read_spi() asm
{
		MOV		CX,8				; Shift 8 bits
		MOV		DX,DGRP:_cport		; Get I/O port
#if TARGET == _DIRECT_		/* Read via direct LPT connection: xxxxRCxD */
rdspi1:	MOV		AL,00001101b		; Reset=0, Clk=1, Data=0
		OUT		DX,AL				; Write to port
		CALL	xdelay				; Wait a bit
; Contrary to Atmels docs, it seems you shift in data on rising edge
		DEC		DX					; Backup to input
		IN		AL,DX				; Read I/O bit
		ROL		AX,1				; Shift into received data
		INC		DX					; Back to output
		MOV		AL,00001011b		; Reset=0, clk=0, Data=0, Led=0
		OUT		DX,AL				; Write to port
; This is the way Atmel claims that it works, however if you look at it
; on a scope, the data invalidates IMMEDIATELY after the clock drops!
;		DEC		DX					; Backup to input
;		IN		AL,DX				; Read I/O bit
;		INC		DX					; Restore
;		ROL		AX,1				; Get data
		CALL	xdelay				; Wait a bit
		LOOP	rdspi1				; Read it all
		MOV		AL,AH				; Get in low
		XOR		AH,AH				; Zero high
		NOT		AL					; Invert
#elif TARGET == _KANDA_		/* Read via KANDA basic Dongle: RLDCMPxx */
rdspi1:	MOV		AL,00010000b		; Rst=0, Led=0, DATA=0, Clk=1
		OUT		DX,AL				; Write to port
		CALL	xdelay				; Wait a bit
		INC		DX					; Advance to input
		IN		AL,DX				; Read I/O bit
		ROL		AL,1				; Get into bit 7
		ROL		AX,1				; Shift into received data
		DEC		DX					; Back to output
		MOV		AL,00000000b		; Reset=0, clk=0, Data=0
		OUT		DX,AL				; Write to port
		CALL	xdelay				; Wait a bit
		LOOP	rdspi1				; Read it all
		MOV		AL,AH				; Get in low
		XOR		AH,AH				; Zero high
#elif TARGET == _KDONG1_		/* Read via KANDA new dongal: xxxxxDCR */
rdspi1:	MOV		AL,00000001b		; Reset=0, Clk=1, Data=0
		OUT		DX,AL				; Write to port
		CALL	xdelay				; Wait a bit
		DEC		DX					; Backup to input
		IN		AL,DX				; Read I/O bit
		ROL		AX,1				; Shift into received data
		INC		DX					; Back to output
		MOV		AL,00000011b		; Reset=0, clk=0, Data=0
		OUT		DX,AL				; Write to port
		CALL	xdelay				; Wait a bit
		LOOP	rdspi1				; Read it all
		MOV		AL,AH				; Get in low
		XOR		AH,AH				; Zero high
		NOT		AL					; Invert
#elif TARGET == _MP3_			/* Read via MP3 interface: xxxxCDCR */
rdspi1:	MOV		AL,00100001b		; Set clock high
		OUT		DX,AL				; Write it
		CALL	xdelay				; Wait a bit
		DEC		DX					; Back to...
		DEC		DX					; Input field
		IN		AL,DX				; Read data
		ROL		AX,1				; Shift into result
		INC		DX					; Back to...
		INC		DX					; output latch
		MOV		AL,DGRP:_ASSERT_RESET; Set clock low
		OUT		DX,AL				; Write it
		CALL	xdelay				; Wait a bit
		LOOP	rdspi1				; Do all bits
		MOV		AL,AH				; Get result
		XOR		AH,AH				; Zero high
#endif
}

/* Wait for a delay in microseconds */
m_delay(micros) asm
{
		MOV		DX,4[BP]			; Get delay (microseconds)
		XOR		CX,CX				; Zero high
		MOV		AH,86h				; BIOS delay function
		INT		15h					; Call BIOS
}

/*
 * Reset target device & enable in-circuit programming
 */
enable_programming()
{
	unsigned a, b, i;

	/* Reset the target CPU */
	write_pp(ASSERT_RESET);			/* Reset low */
	delay(xresdel);
	write_pp(RELEASE_RESET);		/* Reset high */
	delay(xresdel*2);
	write_pp(ASSERT_RESET);			/* Set low again */

	/* Issue reset command */
	delay(xresdel);
	write_spi(0xAC);
	write_spi(0x53);
	a = read_spi();
	b = read_spi();

	/* Identify device */
	for(i=0; i < 3; ++i) {
		write_spi(0x30);
		write_spi(0x00);
		write_spi(i);
		buffer[i] = read_spi(); }
	message("Device id code: %02x %02x %02x", buffer[0], buffer[1], buffer[2]);
	switch(buffer[0]) {
		case 0x1E :	message("Manufacturer is Atmel"); }
	switch((buffer[1] << 8) | buffer[2]) {
		case 0x9001: a=1024; b=64;				ptr="90S1200";	goto shosiz;
		case 0x9101: a=2048; b=128;				ptr="90S2313";	goto shosiz;
		case 0x9201: a=4096; b=256;				ptr="90S4414";	goto shosiz;
		case 0x9203: a=4096; b=256;				ptr="90S4433";	goto shosiz;
		case 0x9301: a=8192; b=512;				ptr="90S8515";	goto shosiz;
		case 0x9303: a=8192; b=512;				ptr="90S8535";	goto shosiz;
		case 0x0601: a=-1;	 b=2048; Npage=0;	ptr="Mega106";	goto shosiz;
		case 0x0101: a=-1;	 b=4096; Npage=0;	ptr="Mega103";
	shosiz: message("Device: %s, Fsize=%u Esize=%u", ptr, a, b);
		if(fsize == NOSIZE) fsize = a;
		if(esize == NOSIZE) esize = b;
		return; }

	verbose = -1;
	message("Device type is not knowm!");
	if((fsize == NOSIZE) || (esize == NOSIZE))
		error("Set fsize= and esize= manually!");

/*	i = 9;
	do {
		delay(50);
		write_spi(0xAC);
		write_spi(0x53);
		a = read_spi();
		b = read_spi();
		if(a == 0x53)
			return;
		message("Re-syncing clock...");
		write_pp(RAISE_CLOCK);
		write_pp(ASSERT_RESET); }
	while(--i);
	error("Init failed."); */
}


/*
 * Read file into memory image
 */
read_file(filename, msize)
	char *filename;
	unsigned msize;
{
	unsigned a, b, chksum, loadptr, lcount, count, maxsize;
	FILE *fp;

	if(!(fp = fopen(filename,"r")))
		error("Unable to read: %s", filename);

	fill_buffer();
	lcount = maxsize = 0;
	for(;;) {
		if(!fgets(ptr = buffer, 80, fp)) {
			fprintf(stderr,"AVRLOAD: No end of file record.\n");
			goto quit; }
		again: switch(*ptr++) {
			case 'S' :	/* Motorola HEX format */
				if(!outfmt) outfmt = 2;
				if(*ptr == '9') goto quit;
				if(*ptr++ != '1') continue;
				lcount += count = (chksum = get_byte()) - 3;
				chksum += a = get_byte();
				chksum += b = get_byte();
				loadptr = (a << 8) + b;
				if(base == -1) base = loadptr;
				if(loadptr < base)
					error("First record is not beginning, Use B=");
				while(count--) {
					chksum += a = get_byte();
					poke(image, loadptr++, a); }
				if((255 & ~chksum) != get_byte())
					goto quitbad;
				break;
			case ':' :		/* Intel HEX format */
				if(!outfmt) outfmt = 3;
				if(!(count = get_byte())) goto quit;
				lcount += count;
				chksum = (a = get_byte()) + count;
				chksum += b = get_byte();
				loadptr = (a << 8) + b;
				if(base == -1) base = loadptr;
				if(loadptr < base)
					error("First record is not beginning, Use B=");
				chksum += get_byte();
				while(count--) {
					chksum += a = get_byte();
					poke(image, loadptr++, a); }
				if((255 & -chksum) != get_byte())
					goto quitbad;
				break;
			case ' ' :	/* Space */
			case '\t' :	/* Tab */
				goto again;
			case 0 :		/* Null line */
				continue;
			default:
				error("Invalid record format."); }
		if(loadptr > maxsize)
			maxsize = loadptr; }

quitbad:
	error("Bad record checksum.");
quit:
	fclose(fp);

	message("Base address is %04x", base);
	message("%u bytes loaded, Image size is %u", lcount, maxsize);

	if(maxsize > msize)
		error("Image size (%u) exceeds device size (%u)", maxsize, msize);
}

/* Load a byte from the hex file */
get_byte()
{
	return (get_hex(*ptr++) << 4) + get_hex(*ptr++);
}

/* Test for HEX digit & get value*/
get_hex(c)
	int c;
{
	if((c >= '0') && (c <= '9'))
		return c-'0';
	if((c >= 'A') && (c <= 'F'))
		return c-'A'+10;
	if((c >= 'a') && (c <= 'f'))
		return c-'a'+10;
	error("Bad hex digit");
}

/*
 * Write file from memory image
 */
write_file(name, size)
	char *name;
	unsigned size;
{
	unsigned cnt, locn, adr1;
	char fflag;
	FILE *fp;

	cnt = locn = fflag = 0;

	if(!outfmt) outfmt = 2;

	if(!(fp = fopen(name, (outfmt > 1) ? "w" : "wb")))
		error("Unable to write: '%s'", name);

	if(outfmt > 1) {
		while(locn < size) {
			if(!cnt) {
				adr1 = base + locn;
				buffer[cnt++] = adr1 >> 8;
				buffer[cnt++] = adr1 & 255; }
			if((buffer[cnt++] = peek(image, locn++)) == 255) {
				if(!fflag) {
					for(adr1 = locn; (peek(image,adr1) == 255) && (adr1 < size); ++adr1);
					if((adr1 - locn) >= maxff) {	/* within range of MAXFF */
						write_record(fp, cnt-1);
						locn = adr1;
						cnt = 0; }
					else
						fflag = -1; } }
			else
				fflag = 0;
			if(cnt >= reclen) {
				write_record(fp, cnt);
				cnt = 0; } }
		write_record(fp, cnt);
		switch(outfmt) {
			case 2 :
				fprintf(fp,"S9030000FC\n");
				break;
			case 3 :
				fprintf(fp,":00000001FF\n"); } }

	else			/* pure binary output */
		while(locn < size)
			putc(peek(image, locn++), fp);

	fclose(fp);
}

/* write mhx record to a file */
write_record(fp, count)
	FILE *fp;
	unsigned count;
{
	unsigned chk, i, chr;

	if(count > 2) {
		switch(outfmt) {
			case 2 :		/* Motorola */
				fprintf(fp,"S1%02x", chk = count + 1);
				for(i=0; i < count; ++i) {
					fprintf(fp,"%02x", chr = buffer[i]);
					chk += chr; }
				fprintf(fp,"%02x\n",255&~chk);
				break;
			case 3 :		/* Intel */
				chk = buffer[0] + buffer[1] + count - 2;
				fprintf(fp,":%02x%02x%02x00",count-2, buffer[0], buffer[1]);
				for(i=2; i < count; ++i) {
					fprintf(fp,"%02x",chr = buffer[i]);
					chk += chr; }
				fprintf(fp,"%02x\n",255 & (0-chk)); } }
}

/*
 * get a decimal/hex/octal/binary number from passed string
 */
get_num(term)
	char term;
{
	register unsigned value;
	register char chr;

	value = 0;

	if(isdigit(*string))				/* decimal number */
		while(isdigit(chr=*string++))
			value = (value * 10) + chr - '0';
	else if('$'==(chr=*string++))		/* hexidecimal number */
		while((isdigit(chr=toupper(*string++))) || ((chr >= 'A') && (chr <= 'F')))
			value = (value << 4) + ((chr < 'A') ? (chr - '0') : (chr - '7'));
	else if(chr=='@')					/* octal number */
		while(('0'<=(chr=*string++)) && (chr < '8'))
			value = (value << 3) + (chr - '0');
	else if(chr=='%')					/* binary number */
		while(('0'==(chr=*string++)) || (chr=='1'))
			value = (value << 1) + (chr - '0') ;
	else chr=-1;
	if(chr != term)
		error("Invalid operand\n");
	return(value);
}

/*
 * Test for beginning of command string
 */
match(char *p)
{
	char *p1;
	p1 = string;
	while(*p1)
		if(toupper(*p1++) != *p++)
			return 0;
	return 1;
}

/*
 * High-speed fill of device buffer with default value
 */
fill_buffer() asm
{
		MOV		ES,DGRP:_image		; Get segment
		MOV		AL,DGRP:_defval		; Default value
		MOV		AH,AL				; Make it a word
		XOR		DI,DI				; Zero offset
		MOV		CX,32768			; Get count
		REP STOSW					; Write it
}

/*
 * High speed compare of buffer with internal buffer
 */
test_buffer() asm
{
		MOV		ES,DGRP:_image		; Get segment
		MOV		DI,4[BP]			; Get offset into segment
		MOV		SI,OFFSET DGRP:_buffer ;Point to our buffer
		MOV		CX,128				; Compare 128 words
		XOR		AX,AX				; Assume success
		REP CMPSW					; Do the compare
		JZ		testb1				; Success
		DEC		AX					; Indicate failure
testb1:
}

/* AVR ID bits:
AT90S1200	$1E	- Indicates manufactured by Atmel
F=1k		$90	- Indicates 1K flash memory
E=64		$01	- Indicates 90S1200 when 1=$90

AT90S2313	$1E	- Indicates Atmel
F=2K		$91	- 2K flash
E=128		$01	- Indicates 90S2313 when 1=$91

AT90S4414	$1E	- Indicates Atmel
F=4K		$92	- Indicates 4K flash
E=256		$01	- Indicates 90S4414 when 1=$92

AT90S8515	$1E	- Indicates Atmel
F=8K		$93	- Indicates 8K flash
E=512		$01	- Indicates 90S8515 when 1=$93
*/

main(int argc, char *argv[])
{
	unsigned cmd, i, j, k, wa;
	unsigned char d, e;

	cport = peekw(0x40, 0x08);		/* Get LPT1 address */
	xspidel = cal_delay() / CAL_FACTOR;

	for(i=1; i < argc; ++i) {
		if(*(string = argv[i]) == '-') {		/* Enable switch */
			while(*++string) switch(toupper(*string)) {
				case 'B' : outfmt = 1;		break;		/* Binary out */
				case 'M' : outfmt = 2;		break;		/* Motorola out */
				case 'I' : outfmt = 3;		break;		/* Intel out */
				case 'Q' : verbose= 0;		break;		/* Quiet mode */
				case 'P' : Npage = 0;		break;		/* Force paged mode */
#if TARGET == _MP3_
				case 'X' :								/* Alternate box */
					ASSERT_RESET = Assert_reset;
					RELEASE_RESET = Release_reset;
					break;
#endif
				default: goto badopt; }
			continue; }
		if((*string == '~') && (string[2] == '=')) {
			string += 3;
			switch(toupper(*(string-2))) {
				case 'S' : xspidel = get_num(0); continue;
				case 'P' : xpgmdel = get_num(0); continue;
				case 'R' : xresdel = get_num(0); continue; }
			goto badopt; }
		if(*++string == '=') switch(toupper(*(string++ - 1))) {
			case 'B' : base = get_num(0);	continue;	/* Specify base */
			case 'D' : defval = get_num(0);	continue;	/* Default value */
			case 'R' : reclen = get_num(0)+2;continue;	/* Record length */
			case 'F' : fsize = get_num(0);	continue;	/* Flash size */
			case 'E' : esize = get_num(0);	continue;	/* Eprom size */
			case 'U' : maxff = get_num(0);	continue;	/* Maximum FF's */
			case 'P' :
				if(!(cport = get_num(0)))
					error("LPT must be 1-3 or address");
				if(cport < 4)
					cport = peekw(0x40, (cport*2)+6);
				continue;
			default:
			badopt:
				error("Invalid option: %s", argv[i]); }
		cmds[cptr++] = argv[i]; }

	if(!cptr) {
		cmd = 'G';
		for(;;) {
			switch(toupper(cmd)) {
				case 'G' :	fputs(help_g, stdout);	break;
				case 'O' :
					fputs(help_o, stdout);
					printf("%u", xspidel);
					fputs(help_o1, stdout);
					break;
				case 'W' :	fputs(help_w, stdout);	break;
				case 'C' :	fputs("Commands:\n\n", stdout);
					for(i=0; i < (sizeof(cmd_tab)/sizeof(cmd_tab[0])); ++i)
						printf("  %s\n", cmd_tab[i]);
					fputs(help_c, stdout);
					break;
				case 'E' :
				case '\r':
				case ' ' :
				case 0x03:
				case 0x1B:
					return; }
			printf("\nG)eneral C)ommands O)ptions W)iring E)xit ?");
			cmd = kbget(); } }

	if(!cport)
		error("LPT port not found!");

	switch(cport) {
		default: message("Using non-standard LPT address $%04x", cport);
		case 0x3BC :
		case 0x378 :
		case 0x278 : }

#if TARGET == _DIRECT_
	cport += 2;			/* Offset to output latch */
#elif TARGET == _KANDA_
	out(cport+2, 0x0C);	/* Insure it is an output */
#elif TARGET == _KDONG1_
	out(cport, 0xFF);	/* Power dongal */
	cport += 2;			/* Offset to output latch */
#elif TARGET == _MP3_
	out(cport += 2, 0xF0);
#endif

	if(!(image = alloc_seg(4096)))
		error("Cannot allocate memory");

	fill_buffer();

	enable_programming();

	for(cmd = 0; cmd < cptr; ++cmd) {
		string = cmds[cmd];
		i = j = 0;
		while(i < (sizeof(cmd_tab)/sizeof(cmd_tab[0]))) {
			if(match(cmd_tab[i++])) {
				if(j)
					error("Ambiguous command: '%s'", string);
				j = i; } }
		switch(j) {
		case ERASE :
			write_spi(0xAC);
			write_spi(0x80);
			read_spi();
			read_spi();
			i = verbose;
			verbose = 0;
			enable_programming();
			verbose = i;
			message("Device erased!");
			continue;
		case LOAD :
			if(++cmd >= cptr)
				error("Filename required");
			read_file(cmds[cmd], -1);
			continue;
		case FPROGRAM :
			if(++cmd >= cptr)
				error("Filename required");
			read_file(cmds[cmd], fsize);
			if(Npage) {
				for(i=0; i < fsize; ++i) {
					if((d = peek(image, i)) == 0xFF)
						continue;
					wa = (i >> 1) | hmask;
					write_spi((i & 1) ? 0x48 : 0x40);
					write_spi(wa >> 8);
					write_spi(wa);
					write_spi(d);
					m_delay(xpgmdel || 5000);
					write_spi((i & 1) ? 0x28 : 0x20);
					write_spi(wa >> 8);
					write_spi(wa);
					e = read_spi();
					if(d != e)
						error("Verify error at: %04x, Expected: %02x, Found: %02x", i, d, e); } }
			else {
				memset(buffer, -1, sizeof(buffer));
				i = 0;
				while(i < fsize) {
					if(!test_buffer(k = i)) {
						if(i += 256)
							continue;
						break; }
					for(j=0; j < 256; ++j) {
						d = peek(image, i);
						wa = (i >> 1) | hmask;
						write_spi((i & 1) ? 0x48 : 0x40);
						write_spi(wa >> 8);
						write_spi(wa & 0x7F);
						write_spi(d);
						++i; }
					wa = (k >> 1) | hmask;
					write_spi(0x4C);
					write_spi(wa >> 8);
					write_spi(wa);
					write_spi(0);
					m_delay(xpgmdel || 50000);
					for(j=0; j < 128; ++j) {
						d = peek(image, k);
						wa = (k >> 1) | hmask;
						write_spi((k & 1) ? 0x28 : 0x20);
						write_spi(wa >> 8);
						write_spi(wa);
						e = read_spi();
						if(d != e)
							error("Verify error at: %04x, Expected: %02x, Found: %02x", k, d, e);
						++k; }
					if(!i) break; } }

			message("Programmed and verified.");
			continue;
		case FREAD :
			if(++cmd >= cptr)
				error("Filename required.\n");
			if(base == -1)
				base = 0;
			for(i=0; i < fsize; ++i) {
				wa = (i >> 1) | hmask;
				write_spi((i & 1) ? 0x28 : 0x20);
				write_spi(wa >> 8);
				write_spi(wa);
				poke(image, i, read_spi()); }
			write_file(cmds[cmd], fsize);
			message("Read complete.");
			continue;
		case EPROGRAM :
			if(++cmd >= cptr)
				error("Filename required.\n");
			read_file(cmds[cmd], esize);
			for(i=0; i < esize; ++i) {
				d = peek(image, i);
				write_spi(0xC0);
				write_spi(i >> 8);
				write_spi(i);
				write_spi(d);
				if(Npage) {		/* Normal, 4ms delay */
					m_delay(xpgmdel || 5000);
					write_spi(0xA0);
					write_spi(i >> 8);
					write_spi(i);
					e = read_spi(); }
				else {			/* EE verification cycle */
					if((d == 0x7F) || (d == 0x80) || xpgmdel) {
						m_delay(xpgmdel || 10000);
						write_spi(0xA0);
						write_spi(i >> 8);
						write_spi(i);
						e = read_spi(); }
					else {
						for(k=0; k < 10; ++k) {
							m_delay(1000);
							write_spi(0xA0);
							write_spi(i >> 8);
							write_spi(i);
							e = read_spi();
							if(e == d)
								break; } } }
				if(d != e)
					error("Verify error at: %04x, Expected: %02x, Found: %02x", i, d, e); }
			message("Programmed and verified.");
			continue;
		case EREAD :
			if(++cmd >= cptr)
				error("Filename required.\n");
			if(base == -1)
				base = 0;
			for(i=0; i < esize; ++i) {
				write_spi(0xA0);
				write_spi(i >> 8);
				write_spi(i);
				poke(image, i, read_spi()); }
			write_file(cmds[cmd], esize);
			message("Read complete.");
			continue;
		case FVERIFY :
			for(i=0; i < fsize; ++i) {
				d = peek(image, i);
				wa = (i >> 1) | hmask;
				write_spi((i & 1) ? 0x28 : 0x20);
				write_spi(wa >> 8);
				write_spi(wa);
				e = read_spi();
				if(d != e)
					error("Verify error at: %04x, Expected: %02x, Found: %02x", i, d, e); }
			message("Verify complete.");
			continue;
		case EVERIFY :
			for(i=0; i < esize; ++i) {
				d = peek(image, i);
				write_spi(0xA0);
				write_spi(i >> 8);
				write_spi(i);
				e = read_spi();
				if(d != e)
					error("Verify error at: %04x, Expected: %02x, Found: %02x", i, d, e); }
			message("Verify complete.");
			continue;
		case LOCK :
			if(++cmd >= cptr)
				error("Value required.\n");
			string = cmds[cmd];
			if((j = get_num(0)) > 3)
				error("Lock bits must be 0-3");
			write_spi(0xAC);
			write_spi((j << 1) | 0xE0);
			write_spi(0);
			write_spi(0);
			message("Lock bits written.");
			continue;
		case LOW :
			hmask = 0;
			message("Programming LOW 64k");
			continue;
		case HIGH :
			hmask = 0x8000;
			message("Programming HIGH 64k");
			continue;
		case RUN :
			exit_condition = FINISH_RUN;
			continue; }
		error("Unknown command: '%s'", string); }

	write_pp(exit_condition);
}
