/*
 * Main module for the PC / RS-232 Data Line Monitor.
 *
 * ?COPY.TXT 1989-2005 Dave Dunfield
 *  -- see COPY.TXT --.
 */
#include <stdio.h>
#include <file.h>
#include <video.h>
#include <window.h>

#define MSGROW	20
#define COMROW	3
#define	PTUP	30
#define	PTDN	31
#define	CUR		'>'
#define	VBAR	179
#define	HBAR	'-'
#define	SPC		' '
#define	LON		0x02
#define	LOFF	0x01

/* Command menu tables */
char *main_menu[] = {
	"Begin data capture",
	"Review captured data",
	"Clear capture buffer",
	"RS-232 Signal monitor",
	"Serial port parameters",
	"Disk transfer functions",
	"Exit to DOS",
	0 }
char *setup_menu[] = {
	"Configure DTE and DCE",
	"Configure DTE settings",
	"Configure DCE settings",
	"Adjust baud rate table",
	0 }
char *setup_menu1[] = {
	"Baudrate",
	"Data bits",
	"Parity",
	"Stop bits",
	0 };

char *disk_menu[] = {
	"Load Data Image",
	"Load Configuration",
	"Save Data Image",
	"Save Configuration",
	0 };

/* Static Uart configuration static data tables */
char *databits[] = { "Five", "Six", "Seven", "Eight", 0 };
char *parity[] = { "Odd", "Even", "Mark", "Space", "None", 0 };
char *stopbits[] = { "One", "Two", 0 };

/* Uart dynamic data tables (Beginning of config memory) */
int uart_config[2][5] = {
	0x3f8, 6, 3, 4, 0,		/* Uart1: 9600,8,None,1 */
	0x2f8, 6, 3, 4, 0 }		/* Uart2: 9600,8,None,1 */
unsigned dte_attr = 0x0F00;
unsigned char cap_mask = 0xFF;
char view_mode = 0;
char image_file[51] = "IMAGE.DLI";
char config_file[51] = "CONFIG.DLC";
//int baud[] = { 110, 150, 300, 1200, 2400, 4800, 9600, 19200, 38400, 57600, 0 }
unsigned baud[] = { 1047, 768, 384, 96, 48, 24, 12, 6, 3, 2, 1 };

#define CONFIG_SIZE (\
	sizeof(uart_config)\
	+sizeof(dte_attr)\
	+sizeof(cap_mask)\
	+sizeof(view_mode)\
	+sizeof(image_file)\
	+sizeof(config_file)\
	+sizeof(baud) )

char	/* Font special characters (must be in order) */
	Spc = SPC,
	Hbar = HBAR,
	Vbar = VBAR,
	Ptup = PTUP,
	Ptdn = PTDN,
	Cur = CUR;

unsigned RDPTR = 0, WRPTR = 0, ODTE = 0, ODCE = 0;

unsigned char font_flag = -1, font[4097];

#define	POINTER_SIZE	8		/* Size of data pointers (in bytes) */

extern int PSP;

char help[] = { "\n\
Use:	DLM [opts]\n\n\
opts:	DTE=1-4		Set DTE port as COM1-COM4\n\
	DTE=5+		Set DTE port to address (hex)\n\
	DTE!		Disable DTE port\n\
	DCE=1-4		Set DCE port as COM1-COM4\n\
	DCE=5+		Set DCE port to address (hex)\n\
	DCE!		Disable DCE port\n\
	/R		Review mode (no COM ports)\n\
	A=xx		Set video attribute for DTE hex display\n\
	C=cfgfile[.DLC]	Load DLM configuration file\n\
	F=fntfile[.DLF]	Load VGA font for data display\n\
	M=xx		Set mask for displayed characters\n" };

/*
 * Open a file for read or write
 */
FILE *openf(fname, rw)
	char *fname, rw;
{
	char c, omsg[80], *ptr;
	FILE *fp;

	ptr = "read";
	fp = open(fname, F_READ);	/* First try and read the file */
	if(rw) {					/* If writing the file */
		ptr = "write";
		if(fp) {
			close(fp);
			sprintf(omsg, "Overwrite existing %s (Y/N) ?", fname);
			message(omsg);
			do {
				c = toupper(vgetc());
				if((c == 0x1B) || (c == 'N'))
					return 0; }
			while(c != 'Y'); }
		fp = open(fname, F_WRITE); }
	if(!fp) {
		sprintf(omsg,"Cannot %s %s (Press a key)", ptr, fname);
		message(omsg);
		vgetc(); }
	return fp;
}

/*
 * Get a COM port designation
 */
unsigned get_com(char *ptr)
{
	unsigned i;
	static unsigned addresses[] = {
		0x3F8, 0x2F8, 0x3E8, 0x2E8 };
	if(toupper(*ptr) == 'E')
		++ptr;
	switch(*ptr++) {
		case '!' : return 0;
		case '=' : }
	if((i = atox(ptr)) < 5) {
		if(!i)
			abort("Invalid COM port specification\n");
		return addresses[i-1]; }
	return i;
}

/*
 * Open a file for the startup argument parsing
 */
unsigned open_startup(char *ptr, char *dest, char *ext)
{
	unsigned fh;
	char flag, *ptr1;

	flag = -1;
	ptr1 = dest;
	while(*ptr1 = toupper(*ptr++)) {
		if(*ptr1++ == '.')
			flag = 0; }
	if(flag)
		strcpy(ptr1, ext);
	if(!(fh = open(dest, F_READ))) {
		printf("Cannot open: %s\n", dest);
		exit(-1); }
	return fh;
}

/*
 *---------------------------
 * Main program & subroutines
 *---------------------------
 */
main(int argc, char *argv[])
{
	int i;
	char *ptr, *ptr1;

	for(i=1; i < argc; ++i) {
		ptr = argv[i];
		switch((toupper(*ptr++) << 8) | toupper(*ptr++)) {
			case 'DT' : uart_config[0][0] = get_com(ptr);	continue;
			case 'DC' : uart_config[1][0] = get_com(ptr);	continue;
			case '?'<<8:
			case '/?' :
			case '-?' :
			case '/H' :
			case '-H' :
				abort(help);
			case '/R' :
			case '-R' : uart_config[0][0] = uart_config[1][0] = 0; continue;
			case 'A=' : dte_attr = atox(ptr) << 8;			continue;
			case 'M=' : cap_mask = atox(ptr);				continue;
			case 'F=' :		/* Load font file */
				ptr1 = open_startup(ptr, font, ".DLF");
				read(font, 3, ptr1);
				if((font[0] != 'D') || (font[1] != 'L') || (font[2] != 'M'))
					goto badfont;
				read(&Spc, 6, ptr1);	/* Read the special characters */
				if(read(font, sizeof(font), ptr1) != 4096) {
				badfont:
					abort("Invalid font file!"); }
				close(ptr1);
				font_flag = 0;
				continue;
			case 'C=' :		/* Load config file */
				ptr1 = open_startup(ptr, config_file, ".DLC");
				read(uart_config, CONFIG_SIZE, ptr1);
				close(ptr1);
				continue;
			default:
				printf("Unknown option: %s\n", argv[i]);
				abort(help); } }

	if(uart_config[0][0] == uart_config[1][0]) {
		if(uart_config[0][0] || uart_config[1][0]) {
			printf("DTE and DCE ports must be different!\n");
			abort(help); } }

	if(test_uart(uart_config[0][0]) | test_uart(i = uart_config[1][0]))
		abort(help);

	if(resize_seg(PSP, 4096*3))
		abort("Not enough memory!!!\n");

	i = 0;
	update_uart(0);
	update_uart(1);
	vopen();
	vcursor_off();
redraw:
	vclscr();
	V_ATTR = REVERSE;
	vdraw_box(0, 0, 79, 2);
	vgotoxy(1, 1);
	vputf("", 21);
	vputf("RS-232 Data Line Monitor Version 1.9", 57);
	V_ATTR = NORMAL;
	vdraw_box(0, COMROW, 79, 2);
	show_settings();

	for(;;) {
		message("Select function and press ENTER");
		if(vmenu(0, 6, main_menu, 0, &i))
			continue;
		switch(i) {
			case 0 :		/* begin data capture */
				if(uart_config[0][0] || uart_config[1][0]) {
					capture();
					goto redraw; }
			review_only:
				message("No COM ports are enabled");
				vgetc();
				continue;
			case 1 :		/* review captured data */
				review();
				goto redraw;
			case 2 :		/* Clear capture buffer */
				clear_buffer();
				continue;
			case 3 :		/* Light BOX */
				if(!(uart_config[0][0]|uart_config[1][0]))
					goto review_only;
				light_box();
				continue;
			case 4 :		/* Settings and Options */
				if(!(uart_config[0][0]|uart_config[1][0]))
					goto review_only;
				setup();
				continue;
			case 5 :		/* disk transfer functions */
				disk_transfer();
				continue;
			case 6 :		/* exit to dos */
				vclscr();
				vdraw_box(0,0,79,5);
				vgotoxy(2, 1);
				vputs("The PC / RS-232 Data Line Monitor");
				vgotoxy(2,3);
				vputs("?COPY.TXT 1989-2005 Dave Dunfield");
				vgotoxy(2,4);
				vputs(" -- see COPY.TXT --.");
				vgotoxy(0,6);
				vcursor_line();
				fputs(help, stdout);
				exit(0); } }
}

/*
 * Review the captured data
 */
review()
{
	int page, cursor, entry, i;
	unsigned size, x, y, mark, t, tt;
	char chr, *ptr;

	if(RDPTR == WRPTR) {
		message("Capture buffer is empty (Press a key)");
		vgetc();
		return; }

	size = (WRPTR-RDPTR)/2;
	page = cursor = 0;

	mark = peekw(PSP+8192, RDPTR);

redraw:
	draw_screen();
	vgotoxy(0,0);
	V_ATTR = REVERSE;
	vputs("REVIEW (ESC = exit, F1 = mode)");
	V_ATTR = NORMAL;
	font_on();

refresh:
	review_page((page*2) + RDPTR);

	for(;;) {
		entry = peekw(PSP+4096, t = ((page+cursor)*2) + RDPTR);
		t = peekw(PSP+8192, t) - mark;
		vgotoxy(x = cursor % 80, y = ((cursor / 80) * 3) + 3);
		V_ATTR = REVERSE;
		x = (y*160) + (x*2);
		if(entry & 0x0800) {			/* DCE data */
			pokew(V_BASE, x, 0x7000|'C');
			ptr = "FPOD XSC"; }
		else {							/* DTE data */
			pokew(V_BASE, x, 0x7000|'T');
			ptr = "FPOD XTR";
			entry |= 0x0800; }
		vgotoxy(35, 0);
		vprintf("%5u.%02u", tt = t / 18, ((t%18)*3640)/655);
		vgotoxy(45,0);
		vprintf((entry & 0x1000) ? "%02x " : "-- ", entry & 255);
		vputc(' ');
		for(i=0; i < 8; ++i) {
			vputc((entry & 0x8000) ? ptr[i] : '-');
			vputc(' ');
			entry <<= 1; }
		vprintf(" %5u %5u", page+cursor+1, size);
		V_ATTR = NORMAL;
		chr = vgetc();
		pokew(V_BASE, x, 0x0700|'-');
		x = cursor; y = page;
		switch(chr) {
			case _KLA:			/* left arrow */
				if(--cursor >= 0)
					break;
				cursor = 0;
				if(page)
					--page;
				goto refresh;
			case _KUA:			/* up arrow */
				if((cursor -= 80) >= 0)
					break;
				cursor += 80;
				if((page -= 80) < 0)
					page = cursor = 0;
				goto refresh;
			case _KRA:			/* Right arrow */
				if(++x >= 640) {
					x = 639;
					++y; }
tsthigh:		if((x+y) >= size)
					goto doend;
				cursor = x;
				if(y != page) {
					page = y;
					goto refresh; }
				break;
			case _KDA:			/* Down arrow */
				if((x += 80) >= 640) {
					x -= 80;
					y += 80; }
				goto tsthigh;
			case _KPD:			/* Page down */
				y += 640;
				goto tsthigh;
			case _KPU:			/* page up */
				if((page -= 640) >= 0)
					goto refresh;
			case _KHO:			/* Home key */
				page = cursor = 0;
				goto refresh;
			case _KEN:			/* End key */
			doend:
				page = (size > 640) ? size - 640 : 0;
				cursor = size - page - 1;
				goto refresh;
			case _K1 :			/* Toggle view mode */
				if(++view_mode > 1)
					view_mode = 0;
				goto redraw;
			case 0x1B:			/* Escape */
				font_off();
				return;
			default:
				vputc(7); } }
}

/*
 * Clear the data transfer buffer
 */
clear_buffer()
{
	char c;

	vdraw_box(25,7,32,4);
	vgotoxy(27, 9);
	vprintf("Buffer contains %5u entries", (WRPTR-RDPTR)/2);
	message("Press ENTER to clear (ESC = cancel)");
	do {
		if((c = vgetc()) == '\n') {
			RDPTR = WRPTR = ODTE = ODCE = 0;
			break; } }
	while(c != '\x1b');
	vclear_box(25, 7, 32, 4);
}

/*
 * RS-232 Light BOX
 */
light_box()
{
	int stat1, stat2, new1, new2, old1, old2;

	stat1 = uart_config[0][0];		/* DTE status register */
	if(stat1) stat1 += 6;
	stat2 = uart_config[1][0];		/* DCE status register */
	if(stat2) stat2 += 6;

	old1 = old2 = -1;

	vdraw_box(25, 7, 27, 10);
	vgotoxy(30, 9); vputs("DTR:        DSR:");
	vgotoxy(30,11); vputs("RTS:        CTS:");
	vgotoxy(30,13); vputs("DCD:         RI:");
	vgotoxy(30,15); vputs("AS1:        AS2:");
	message("LIGHT BOX (ESC = exit)");

	do {
		if(stat1) {
			if((new1 = in(stat1) & 0xf0) != old1) {
				old1 = new1;
				vgotoxy(35, 9); vputc((new1 & 0x20) ? LON : LOFF);
				vgotoxy(35,11); vputc((new1 & 0x10) ? LON : LOFF);
				vgotoxy(35,13); vputc((new1 & 0x40) ? LON : LOFF);
				vgotoxy(35,15); vputc((new1 & 0x80) ? LON : LOFF); } }
		if(stat2) {
			if((new2 = in(stat2) & 0xf0) != old2) {
				old2 = new2;
				vgotoxy(47, 9); vputc((new2 & 0x20) ? LON : LOFF);
				vgotoxy(47,11); vputc((new2 & 0x10) ? LON : LOFF);
				vgotoxy(47,13); vputc((new2 & 0x40) ? LON : LOFF);
				vgotoxy(47,15); vputc((new2 & 0x80) ? LON : LOFF); } } }
	while(vtstc() != 0x1b);
	vclear_box(25, 7, 27, 10);
}

char *getbaud(unsigned b)
{
	static char btext[7];
	switch(b = baud[b]) {
	case 0 : return "??????";
	case 1 : return "115200"; }
	sprintf(btext, "%u", scale(57600, 2, b));
	return btext;
}

/*
 * Setup mode
 */
setup()
{
	unsigned port, i, j, l, m1, m2, bp;
	char *bptr[20], btext[11][7];

	port = m1 = m2 = 0;
refresh:
	for(bp=0; bp < (sizeof(baud)/2); ++bp)
		strcpy(bptr[bp] = btext[bp], getbaud(bp));
	bptr[bp] = 0;
	message("Select SETUP function (ESC = cancel)");
	if(vmenu(25,7,setup_menu,0, &port) & 0xf000) {
		update_uart(0);
		update_uart(1);
		return; }
	if(port == 3) {
		if(vmenu(49, 8, bptr, bptr[bp] = 0, &m2))
			goto refresh;
		message("Up/Down= \xF11, PgUp/PgDn = \xF110 Home/End = \xF1100");
		vdraw_box(57, 9, 20, 3);
		vgotoxy(65, 10); vputs("Div.   Speed");
		j = l = baud[m2];
		for(;;) {
			vgotoxy(58, 11);
			baud[m2] = j;
			vprintf("Speed: %-5u %6s", j, getbaud(m2));
			switch(vgetc()) {
				case _KDA : i = 1;
			testd:	j = ((j+i) < j) ? 65535 : (j+i);
					continue;
				case _KUA : i = 1;
			testu:	j = (j > i) ? (j-i) : 0;
					continue;
				case _KPD : i = 10;		goto testd;
				case _KPU : i = 10;		goto testu;
				case _KEN : i = 100;	goto testd;
				case _KHO : i = 100;	goto testu;
				case 0x1B : baud[m2] = l;
				case '\n' :
					vclear_box(49, 8, 28, 15);
					show_settings();
					goto refresh; } } }
redo:
	if(i=port) {
		if(!uart_config[--i][0]) {
			message("This port is disabled (Press a key)");
			vgetc();
			goto refresh; } }
	else
		for(i=4; i; --i)
			uart_config[1][i] = uart_config[0][i];
	show_settings();
	if(vmenu(49, 8, setup_menu1, 0, &m1))
		goto refresh;
	switch(m1) {
		case 0 :	/* baudrate */
			vmenu(60,9,bptr,-1,&uart_config[i][1]);
			break;
		case 1 :	/* Data bits */
			vmenu(60,9,databits,-1,&uart_config[i][2]);
			break;
		case 2 :	/* Parity */
			vmenu(60,9,parity,-1,&uart_config[i][3]);
			break;
		case 3 :	/* Stop bits */
			vmenu(60,9,stopbits,-1,&uart_config[i][4]); }
	goto redo;
}

/*
 * Disk transfer functions
 */
disk_transfer()
{
	int func;
	char *fptr;
	FILE *fp;

	func = 0;
refresh:
	message("Select DISK TRANSFER function (ESC = cancel)");
	if(vmenu(25, 7, disk_menu, 0, &func))
		return;
	if(vgets(5, MSGROW, (func & 0x01) ? "CONFIG filename? " : "IMAGE filename? ",
		fptr = (func & 0x01) ? config_file : image_file, 50))
		goto refresh;
	if(!(fp = openf(fptr, func >= 2)))
		goto refresh;
	switch(func) {
		case 0 :		/* Read image */
			read(&RDPTR, POINTER_SIZE, fp);
			xread(PSP+4096, 0, 32768, fp);
			xread(PSP+4096, 32768, 32768, fp);
			xread(PSP+8192, 0, 32768, fp);
			xread(PSP+8192, 32768, 32768, fp);
			break;
		case 1 :		/* Read config */
			read(uart_config, CONFIG_SIZE, fp);
			show_settings();
			update_uart(0);
			update_uart(1);
			break;
		case 2 :		/* Save image */
			write(&RDPTR, POINTER_SIZE, fp);
			xwrite(PSP+4096, 0, 32768, fp);
			xwrite(PSP+4096, 32768, 32768, fp);
			xwrite(PSP+8192, 0, 32768, fp);
			xwrite(PSP+8192, 32768, 32768, fp);
			break;
		case 3 :		/* Write config */
			write(uart_config, CONFIG_SIZE, fp); }
	close(fp);
	goto refresh;
}

/*
 * Show the current COM port settings
 */
show_settings()
{
	vgotoxy(13, COMROW+1);
	vprintf("DTE: ");
	if(uart_config[0][0]) {
		vprintf("%6s,%2d,%5s,%2d", getbaud(uart_config[0][1]),
			uart_config[0][2]+5, parity[uart_config[0][3]], uart_config[0][4]+1); }
	else
		vprintf("Disabled");
	vgotoxy(45, COMROW+1);
	vprintf("DCE: ");
	if(uart_config[1][0]) {
		vprintf("%6s,%2d,%5s,%2d", getbaud(uart_config[1][1]),
			uart_config[1][2]+5, parity[uart_config[1][3]], uart_config[1][4]+1); }
	else
		vprintf("Disabled");
}

/*
 * Update the uart with its com port settings
 */
update_uart(port)
	int port;
{
	unsigned base, i, b;

	if(base = uart_config[port][0]) {
//		if(b = uart_config[port][1])
//			b = scale(57600, 2, b);
//		else
//			b = 1;
		b = baud[uart_config[port][1]];
		i = ((uart_config[port][3] << 4) & 0x30) |	/* parity type */
			(uart_config[port][2] & 0x03) |			/* # data bits */
			((uart_config[port][4] << 2) & 0x04) |	/* # stop bits */
			((uart_config[port][3] < 4) << 3);		/* parity enable */

		out(base+3, i | 0x80);
		out(base, b);
		out(base+1, b >> 8);
		out(base+3, i);
		out(base+1, 0);		/* disable all interrupts */
		out(base+4, 3); }	/* Enable DTR and RTS */
}

/*
 * Test for the existance of a serial port
 */
test_uart(base)
	int base;
{
	char *ptr;
	if(base) {
		if( (in(base+1) | in(base+4)) & 0x80) {
			ptr = "";
			switch(base) {
			case 0x3F8 : ptr = " (COM1)"; break;
			case 0x2F8 : ptr = " (COM2)"; break;
			case 0x3E8 : ptr = " (COM3)"; break;
			case 0x2E8 : ptr = " (COM4)"; }
			printf("No COM port found at %04x%s\n", base, ptr);
			return -1; } }
	return 0;
}

/*
 * Display a message
 */
message(ptr)
	char *ptr;
{
	vgotoxy(0, MSGROW);
	vcleos();
	vmessage(38 - strlen(ptr)/2, MSGROW, ptr);
}

/*
 *-------------------------------------------
 * 8086 Assembly language routines to perform
 * high speed data capture & display.
 *-------------------------------------------
 */
extern unsigned _V_BASE;
// asm "	EXTRN	_V_BASE:word";

/*
 * Draw the data capture screen
 */
draw_screen() asm
{
		MOV		AX,DGRP:_V_BASE; Get video address
		MOV		ES,AX		; Set up extra
		MOV		DI,0		; Begin at zero offset
		MOV		AH,07h		; Normal video
		MOV		AL,DGRP:_Spc; Get space char
		MOV		CX,80*25	; Size of screen
	REP	STOSW				; Clear screen
		MOV		DI,0		; Reset to start
		MOV		AX,7020h	; Reverse video spaces
		MOV		CX,80		; Write 80 words
	REP	STOSW				; Clear line
		MOV		AH,07h		; Normal video space
		MOV		AL,DGRP:_Hbar; Get horizontal bar
		ADD		DI,160*2	; Skip adjust lines
ds1:	MOV		CX,80		; Write 80 words
	REP	STOSW				; Clear line
		ADD		DI,320		; Skip two lines
		CMP		DI,160*25	; Are we over?
		JB		ds1			; Do them all
}

/*
 * Main data CAPTURE routine
 */
capture()
{
/*		-30,	-28,  	-26,	-24 */
	int hbr,	cur,	vbr,	spc;
/*		-22,	-21	*/
	char ptup, ptdn;
/*		-20		-18			-16			-14 */		
	int cmask, bios_seg, data_seg, time_seg;
/*		-12		-10		-8		-6		-4		-2		stack offsets */
	int dte_d,	dce_d,	dte_o,	dce_o,	dte_s,	dce_s;

	hbr = 0x0700 + (unsigned)Hbar ;
	cur = 0x7000 + (unsigned)Cur  ;
	vbr = 0x0700 + (unsigned)Vbar ;
	spc = 0x0700 + (unsigned)Spc  ;

	ptup = Ptup;
	ptdn = Ptdn;
	cmask = cap_mask;
	bios_seg = 0x40;
	time_seg = (data_seg = PSP + 4096) + 4096;

	draw_screen(160*2);
	dte_s = (dte_d = uart_config[0][0]) ? dte_d + 5 : 0;
	dce_s = (dce_d = uart_config[1][0]) ? dce_d + 5 : 0;
	vgotoxy(0,0);
	V_ATTR = REVERSE;
	vputs("DATA CAPTURE (ESC = exit)");
	V_ATTR = NORMAL;
	dte_o = ODTE;
	dce_o = ODCE;
	font_on();
	asm {
		MOV		SI,DGRP:_WRPTR; Get write pointer
		MOV		CX,DGRP:_RDPTR; Get read pointer
		MOV		DS,-16[BP]	; Get data segment
		MOV		DI,320		; Start of new line
		MOV		BX,0		; Position zero in line
;
cap1:	MOV		AX,-28[BP]							; Get cursor
		MOV		WORD PTR ES:160[BX+DI],AX			; Write cursor
		MOV		AX,-24[BP]							; Get space
		MOV		WORD PTR ES:-160[BX+DI],AX			; Write space
		MOV		WORD PTR ES:0[BX+DI],AX				; Write space
; Poll DTE and record any changes
cap2:	MOV		DX,-4[BP]	; Get UART1 address
		AND		DX,DX		; Enabled?
		JZ		cap10a		; No, skip it
		IN		AL,DX		; Read Line Status Register
		MOV		AH,AL		; Save upper
		INC		DX			; Point to next
		IN		AL,DX		; Read Modem Status Register
; We have DTE event to record
		SHL		AX,1		; Shift	
		SHL		AX,1		; Status bits
		SHL		AX,1		; Into
		SHL		AX,1		; High word
		AND		AH,0f7h		; Mask unwanted bits
		CMP		AH,-8[BP]	; Compare against old
		JZ		cap10		; No change
		TEST	AH,10h		; Data received?
		JNZ		cap3		; Yes, we have it
; Record a signal transition
		MOV		AL,-22[BP]	; Upper character
		PUSH	AX			; Save AX
		MOV		AX,-26[BP]	; Get vertical indicator
		MOV		WORD PTR ES:0[BX+DI],AX	; Indicate transition
		POP		AX			; Restore AX
		JMP		SHORT cap4	; And proceed
cap10a:	JMP		SHORT cap10
; Record a received character
cap3:	MOV		DX,-12[BP]	; Get data address
		IN		AL,DX		; Read character
cap4:	MOV		[SI],AX		; Save data
		MOV		DS,-18[BP]	; Get BIOS segment
		MOV		DX,DS:[006Ch]	; Get data
		MOV		DS,-14[BP]	; Get time segment
		MOV		[SI],DX		; Save data
		MOV		DS,-16[BP]	; Get data segment back
		ADD		SI,2		; Advance to next
		CMP		SI,CX		; Buffer full?
		JNZ		cap4a		; No, its ok
		ADD		CX,2		; Drop oldest character
cap4a:	AND		AH,0EFh		; Remove data ready bit
		MOV		-8[BP],AH	; Save old mask
		MOV		AH,07h		; Set video attribute
		AND		AL,-20[BP]	; Apply display mask
		MOV		ES:-160[BX+DI],AX; Write char
; Advance cursor to next char
		MOV		AX,-30[BP]	; Get horizontal bar
		MOV		WORD PTR ES:160[BX+DI],AX
		ADD		BL,2		; Advance to next char
		CMP		BL,160		; Are we over?
		JB		cap5		; No, its ok
; Advance to next line
		ADD		DI,160*3	; Offset to next line
		MOV		BL,0		; Reset to start
		CMP		DI,25*160	; Past end of screen?
		JB		cap5		; No, its ok
; Wrap to top of screen
		MOV		DI,320		; Top of screen
; Write next cursor
cap5:	MOV		AX,-28[BP]							; Get cursor
		MOV		WORD PTR ES:160[BX+DI],AX			; Write cursor
		MOV		AX,-24[BP]							; Get space
		MOV		WORD PTR ES:-160[BX+DI],AX			; Write space
		MOV		WORD PTR ES:0[BX+DI],AX				; Write space
; Poll DCE and record any changes
cap10:	MOV		DX,-2[BP]	; Get UART2 address
		AND		DX,DX		; Enabled?
		JZ		cap20		; No, skip it
		IN		AL,DX		; Read Line Status Register
		MOV		AH,AL		; Save upper
		INC		DX			; Point to next
		IN		AL,DX		; Read Modem Status Register
; We have DTE event to record
		SHL		AX,1		; Shift	
		SHL		AX,1		; Status bits
		SHL		AX,1		; Into
		SHL		AX,1		; High word
		AND		AH,0f7h		; Mask unwanted bits
		OR		AH,08h		; Indicate DCE data
		CMP		AH,-6[BP]	; Any signals?
		JZ		cap20		; No change
		TEST	AH,10h		; Date received?
		JNZ		cap11		; Yes, we have it
; Record a signal transition
		PUSH	AX			; Save AX
		MOV		AX,-26[BP]	; Get vertical indicator
		MOV		WORD PTR ES:-160[BX+DI],AX; Indicate transition
		POP		AX			; Restore AX
		MOV		AL,-21[BP]	; Lower character
		JMP		SHORT cap12	; And proceed
; Record a received character
cap11:	MOV		DX,-10[BP]	; Get data address
		IN		AL,DX		; Read character
cap12:	MOV		[SI],AX		; Save data
		MOV		DS,-18[BP]	; Get BIOS segment
		MOV		DX,DS:[006Ch]	; Get data
		MOV		DS,-14[BP]	; Get time segment
		MOV		[SI],DX		; Save data
		MOV		DS,-16[BP]	; Get data segment back
		ADD		SI,2		; Advance to next
		CMP		SI,CX		; Buffer full?
		JNZ		cap12a		; No, its ok
		ADD		CX,2		; Drop oldest character
cap12a:	AND		AH,0EFh		; Remove data ready bit
		MOV		-6[BP],AH	; Save old mask
		MOV		AH,07h		; Video attribute
		AND		AL,-20[BP]	; Apply display mask
		MOV		ES:0[BX+DI],AX; Display character
; Advance cursor to next char
		MOV		AX,-30[BP]	; Get horizontal bar
		MOV		WORD PTR ES:160[BX+DI],AX	; Remove old cursor
		ADD		BL,2		; Advance to next char
		CMP		BL,160		; Are we over?
		JB		cap13		; No, its ok
; Advance to next line
		ADD		DI,160*3	; Offset to next line
		MOV		BL,0		; Reset to start
		CMP		DI,25*160	; Past end of screen?
		JB		cap13		; No, its ok
; Wrap to top of screen
		MOV		DI,320		; Top of screen
cap13:	JMP		cap1		; Next cursor
; No data received, check for keypress
cap20:	MOV		AH,01h		; Check for key
		INT		16h			; Ask bios
		JZ		cap21		; No key waiting
		XOR		AH,AH		; Function 0 - get key
		INT		16h			; Ask bios
		CMP		AL,1Bh		; Escape?
		JZ		cap22		; Yes, exit
cap21:	JMP		cap1		; Play it again sam
; Restore DATA SEGMENT before exiting
cap22:	MOV		AX,SS		; Get stack segment
		MOV		DS,AX		; Restore data seg
		MOV		DGRP:_RDPTR,CX; Resave read pointer
		MOV		DGRP:_WRPTR,SI; Resave write pointer
	}

	font_off();

	ODTE = dte_o;
	ODCE = dce_o;
}

/*
 * Draw a page of characters from the capture bufffer.
 */
review_page(unsigned start)
{
	unsigned entry, offset, x, d1, d2, a;
	int o1, o2;
	unsigned char c;

	offset = 320;
	x = 0;
	do {
		if(start != WRPTR) {			/* Entry available in buffer */
			c = (entry = peekw(PSP+4096, start)) & cap_mask;
			if(entry & 0x0800)			/* DCE entry */
				o1 = 0, o2 = -160, a = 0x0700;
			else						/* DTE entry */
				o1 = -160, o2 = 0, a = view_mode ? dte_attr : 0x0700;
			if(entry & 0x1000) {		/* Data value */
				if(view_mode) {
					o1 = -160; o2 = 0;
					if((d1 = (c >> 4) & 15) > 9)
						d1 += 7;
					d1 += '0';
					if((d2 = c & 15) > 9)
						d2 += 7;
					d2 += '0'; }
				else {
					d1 = c;
					d2 = Spc; } }
			else {
				d1 = (entry & 0x0800) ? Ptdn : Ptup;
				d2 = Vbar; }
			pokew(V_BASE, offset+o1, d1|a);
			pokew(V_BASE, offset+o2, d2|a);
			start += 2; }
		else {
			pokew(V_BASE, offset, (0x0700 | (unsigned)Spc));
			pokew(V_BASE, offset-160, (0x0700 | (unsigned)Spc)); }
		offset += 2;
		if(++x >= 80) {
			x = 0;
			offset += (160*2); } }
	while(offset < (25*160));
}

/*
 * Read from EXTENDED segment (DS+64k)
 */
xread(segment, addr, size, fp) asm
{
		MOV		DX,8[BP]	; Get buffer address
		MOV		CX,6[BP]	; Get number of bytes
		MOV		BX,4[BP]	; Get file handle
		PUSH	DS			; Save MICRO-C data segment
		MOV		DS,10[BP]	; Get segment
		MOV		AH,3Fh		; DOS READ function
		INT		21h			; Read file
		JNC		read1		; A-OK, return
		XOR		AX,AX		; Zero indicates error
read1:	POP		DS			; Restore MICRO-C data segment
}

/*
 * Write to the EXTENDED segment (DS+64k)
 */
xwrite(seg, addr, size, fp) asm
{
		MOV		DX,8[BP]	; Get pointer to block
		MOV		CX,6[BP]	; Get number of bytes
		MOV		BX,4[BP]	; Get file handle
		PUSH	DS			; Save MICRO-C data segment
		MOV		DS,10[BP]	; Get segment
		MOV		AH,40h		; DOS WRITE Function
		INT		21h			; Write to file
		JC		putc2		; Failure, preserve retcode
		XOR		AX,AX		; Zero indicates success
putc2:	POP		DS			; Restore MICRO-C data segment
}

/*
 * Scale to a fractional value
 */
scale(value, multiply, divide) asm
{
		MOV		AX,8[BP]		; Get value
		MUL		WORD PTR 6[BP]	; Multiply to 32 bit product
		MOV		BX,4[BP]		; Get divisor
		DIV		BX				; Divide back to 16 bit result
		SHR		BX,1			; /2 for test
		INC		DX				; .5 rounds up
		SUB		BX,DX			; Set 'C' if remainder > half
		ADC		AX,0			; Increment result to scale
}

/*
 * Enable the custom font
 */
font_on() asm
{
		MOV		AL,DGRP:_font_flag ; Get font enabled flag
		AND		AL,AL		; Is it enabled?
		JNZ		fo4			; No, exit
		PUSH	ES			; Save ES
		MOV		AX,1130h	; Get font information
		MOV		BH,0		; Current font
		INT		10h			; Ask BIOS
		PUSH	DS			; Get DS
		POP		ES			; ES = DS
		XOR		DX,DX		; Start with character 0
		CMP		CX,16		; Is current font > 16 points?
		JA		fo1			; Yes, special case
; Font is 16 points or less, just use one we have
		MOV		AX,1100h	; Set font
		MOV		BX,1000h	; 16 point font
		MOV		CX,0100h	; 256 charcters
		MOV		BP,OFFSET DGRP:_font ; Use font table "as is"
		INT		10h			; Ask BIOS
		JMP SHORT fo3		; And exit
; Font is greater than 16 points
; We must translate our 16 point font to match (zero pad)
fo1:	MOV		BH,CL		; BH = points
		XOR		BL,BL		; * 256
		SUB		SP,BX		; Allocate space for new font
		MOV		DI,SP		; Dest is new buffer
		MOV		SI,OFFSET DGRP:_font ; Source is our font
; Copy in our font data
fo2:	MOV		CX,16		; Copy 16 bytes
	REP	MOVSB				; Copy the character
		MOV		CL,BH		; Get point size
		SUB		CL,16		; Calculate # bytes remaining
		XOR		AX,AX		; Get zero
	REP STOSB				; Clear the end
		CMP		SI,OFFSET DGRP:_font+4096
		JB		fo2			; Do all data
		MOV		AX,1100h	; Set font
		MOV		CX,0100h	; 256 characters
		MOV		BP,SP		; Use new font table
		INT		10h			; Ask BIOS
		MOV		SP,DI		; Fix SP
fo3:	POP		ES			; Restore ES
fo4:
}

/*
 * Disable the custom font
 */
font_off()
{
	if(!font_flag) asm {
		MOV		AX,1104h
		MOV		BL,00h
		INT		10h
	}
}
