#include <stdio.h>
#include <window.h>

#define	LINES	24
#define	MSGS	1024
#define	MSIZE	64

#define	WSTD	0x17
#define	WPRI	0x67
#define	WERR	0x47
#define	WCMD	0x57

#define	STD		1
#define	PRI		2
#define	ERR		3
#define	CMD		5

struct WINDOW
	*swin,
	*mwin;

unsigned
	Maxerr = 10;	// Maximum error messages to "stick"

unsigned
	Seg,			// Segment for text
	Read,			// Read pointer
	Write,			// Write pointer
	Base,			// Base pointer
	Mcount,			// Count of messages
	Ecount;			// Count of error messages

unsigned char
	Hour[MSGS],
	Minute[MSGS],
	Second[MSGS],
	Attr[MSGS];		// Attributes

/*
 * -------------------- PACKET DRIVER INTERFACE --------------------
 */

/* PKT_INT must be in DECIMAL, as it is used by C and inline assembly */
#define	PKT_INT		96				/* Packet driver interrupt (0x60) */
#define	PKT_DATA	1000			/* Size of data area in packet */
#define	PKT_SIZE	PKT_DATA+14		/* Size of packet with overhead */
#define	PKT_EASIZE	6				/* Size of ethernet address */
/* PKT_WINDOW must be a power of 2, so we can mask with PKT_WINDOW-1 */
#define	PKT_WINDOW	32				/* Size of packet receive queue */

/*
 * Structure of information returned by pkt_driver_info()
 */
struct PD_driver_info {
	unsigned version;				/* Driver version */
	unsigned char class;			/* Interface class */
	unsigned type;					/* Interface type */
	unsigned char number;			/* Interface number */
	unsigned char functionality;	/* Driver functionality */
	unsigned char name[9]; };		/* Driver name (?) */

/*
 * Structure of packet as sent/received by driver
 */
struct PD_packet {
	unsigned char to_addr[PKT_EASIZE];
	unsigned char from_addr[PKT_EASIZE];
	unsigned ptype;
	unsigned char data[PKT_DATA]; };

/*
 * Structure of a packet receive queue entry
 */
struct PD_queue {
	unsigned handle;
	unsigned length;
	struct PD_packet packet; };

/*
 * Packet driver interface global variables
 */
struct PD_queue
	pkt_queue[PKT_WINDOW];			/* Queue for received packets */

unsigned
	pkt_head = 0,					/* Head of RX packet queue */
	pkt_tail = 0;					/* Tail of RX packet queue */

/*
 * Obtain packet driver information
 *
 * This function *MUST* be called before pkt_access_type() is called, in
 * order to insure that the SAVEDS location is initialized with the Micro-C
 * data segment. Another reason is that you need to obtain the interface
 * class/type/number so that you can pass them to pkt_access_type();
 *
 * Arguments:
 *	info_struct	= "PD_driver_info" structure to fill in.
 *
 * Returns:
 *	0			= Success
 *	!0			= Packet driver error code
 */
int pkt_driver_info(info_struct) asm
{
		MOV		CS:SAVEDS,DS		; Save out data seg
		MOV		AX,01ffh			; Request info
pi1:	INT		PKT_INT				; Ask packet driver
		JC		di2					; Error occured
		PUSH	DS					; Save driver DS
		MOV		DS,CS:SAVEDS		; Restore DATA segment
		POP		ES					; Point to driver data
		MOV		DI,4[BP]			; Get struct
		MOV		0[DI],BX			; Save version
		MOV		2[DI],CH			; Save Class
		MOV		3[DI],DX			; Save type
		MOV		5[DI],CL			; Save number
		MOV		6[DI],AL			; Save functionality
		LEA		BX,7[DI]			; Point to name
di1:	MOV		AL,ES:[SI]			; Get data
		MOV		[BX],AL				; Copy it
		INC		SI					; Advance source
		INC		DI					; Advance dest
		AND		AL,AL				; End of string?
		JNZ		di1					; Copy it all
		XOR		AX,AX				; Return 0
		JMP		di3
; Report driver error
di2:	MOV		AL,DH				; Get error code
		MOV		AH,0FFh				; Insure non-zero
di3:
}

/*
 * Obtain access to a packet type
 *
 * Arguments:
 *	if_class	= Interface class  \
 *	if_type		= Interface type    > - From pkt_driver_info()
 *	if_number	= Interface number /
 *	type		= Pointer to packet type to access
 *	typelen		= Length of packet type (0=Access ALL packets)
 *	handle		= Pointer to 'int' variable to receive handle
 *
 * Returns:
 *	0			= Success
 *	!0			= Packet driver error code
 */
int pkt_access_type(if_class, if_type, if_number, type, typelen, handle) asm
{
		MOV		AL,14[BP]			; Get if_class
		MOV		BX,12[BP]			; Get if_type
		MOV		DL,10[BP]			; Get if_number
		MOV		SI,8[BP]			; Get type pointer
		MOV		CX,6[BP]			; Get typelen
		MOV		DI,OFFSET pktrdy	; Point to handler
		PUSH	CS					; Save code segment
		POP		ES					; ES = handler segment
		MOV		AH,02h				; Indicate "access type" request
pi2:	INT		PKT_INT				; Call packet driver
		JC		di2					; Report error
		MOV		BX,4[BP]			; Get handle
		MOV		[BX],AX				; Save handle
		XOR		AX,AX				; Indicate success
}

/*
 * Release access to a packet type
 *
 * Arguments:
 *	handle		= Handle returned by pkt_access_type()
 *
 * Returns:
 *	0			= Success
 *	!0			= Packet driver error code
 */
int pkt_release_type(handle) asm
{
		MOV		BX,4[BP]			; Get handle
		MOV		AH,03h				; Indicate "release type" request
pi3:	INT		PKT_INT				; Ask for it
		JC		di2					; Indicate failed
		XOR		AX,AX				; Indicate success
}

/*
 * Send a packet
 *
 * Arguments:
 *	buffer		= Pointer to buffer to send
 *	length		= Length of buffer to send
 *
 * Returns:
 *	0			= Success
 *	!0			= Packet driver error code
 */
int pkt_send(buffer, length) asm
{
		MOV		SI,6[BP]			; Get buffer address
		MOV		CX,4[BP]			; Get length
		MOV		AH,04h				; Indicate "send packet" request
pi4:	INT		PKT_INT				; Ask for it
		JC		di2					; Report error
		XOR		AX,AX				; Indicate success
}

/*
 * Get our ethernet address
 *
 * Arguments:
 *	handle		= Handle returned by pkt_access_type()
 *	buffer		= Pointer to buffer to receive address
 *	length		= Size of buffer (maximum)
 *	alen		= Pointer to 'int' variable to receive size of address
 *
 * Returns:
 *	0			= Success
 *	!0			= Packet driver error code
 */
int pkt_get_address(handle, buffer, length, alen) asm
{
		MOV		BX,10[BP]			; Get handle
		MOV		DI,8[BP]			; Get buffer
		MOV		CX,6[BP]			; Get length
		PUSH	DS					; Save out segment
		POP		ES					; Set ES
		MOV		AH,06h				; Indicate "get address" request
pi5:	INT		PKT_INT				; Ask driver
		JC		di2					; Report error
		MOV		BX,4[BP]			; Get received length
		MOV		[BX],CX				; Save length
		XOR		AX,AX				; Indicate success
}

/*
 * Set the receive mode for a packet
 *
 * Arguments:
 *	handle		= Handle returned by pkt_access_type()
 *	mode		=	1: Turn off receiver
 *					2: Receive only packets sent to our address
 *					3: mode 2 + broadcast packets
 *					4: mode 3 + limited multicast packets
 *					5: mode 3 + all multicast packets
 *					6: all packets
 *
 * Returns:
 *	0			= Success
 *	!0			= Packet driver error code
 */
int pkt_set_receive_mode(handle, mode) asm
{
		MOV		BX,6[BP]			; Get handle
		MOV		CX,4[BP]			; Get mode
		MOV		AH,20				; Set function
pi6:	INT		PKT_INT				; Ask driver
		JC		di2					; Report error
		XOR		AX,AX				; Report success
}

/*
 * Receive packet handler. Called by packet driver when a packet
 * has been received for us.
 *
 * Check to see if we have enough space, and buffer the packet in our
 * receive packet queue, notify the mainline when received.
 */
asm {
; Entry:
;  AX = flag: 0=Allocate buffer, 1=Transfer complete
;  BX = handle
;  CX = Length
; Exit:
;  ES:DI = Buffer to receive packet (0:0=Not available)
pktrdy:	PUSH	DS					; Save our data segment
		MOV		DS,CS:SAVEDS		; Restore Micro-C DS
		AND		AX,AX				; Is this allocate?
		JNZ		pktr2				; No, try next
; Allocate space for a packet
		CMP		CX,PKT_SIZE			; Is packet too large?
		JA		pktr1				; Yes, Fail reception
		MOV		AX,DGRP:_pkt_head	; Get head of queue
		INC		AX					; Advance to next
		AND		AX,PKT_WINDOW-1		; Mask for wrap
		CMP		AX,DGRP:_pkt_tail	; Is queue full?
		JZ		pktr1				; Yes, Fail reception
		MOV		AX,DGRP:_pkt_head	; Get head back
		MOV		DI,PKT_SIZE+4		; Get size
		MUL		DI					; Compute offset
		ADD		AX,OFFSET DGRP:_pkt_queue ; Offset to location
		MOV		DI,AX				; Point to location
		MOV		[DI],BX				; Save handle
		MOV		2[DI],CX			; Save length
		ADD		DI,4				; Skip to data
		PUSH	DS					; Get DS
		POP		ES					; Copy to ES
		JMP SHORT pktrx				; And exit
; No space for buffer, or packet too large - reject packet
pktr1:	MOV		DI,0				; Indicate no buffer
		MOV		ES,DI				; ""
		JMP SHORT pktrx				; And exit
; Packet has been copied into our buffer, notify application
pktr2:	MOV		AX,DGRP:_pkt_head ; Get packet queue head
		INC		AX					; Skip to next
		AND		AX,PKT_WINDOW-1		; Wrap circular buffer
		MOV		DGRP:_pkt_head,AX ; Save new queue head
pktrx:	POP		DS					; Restore data seg
		DB		0CBh				; Far return
SAVEDS	DW		?
}

/*
 * Set the packet interrupt by patching the code
 */
set_interrupt(inum) asm
{
		MOV		AL,4[BP]			; Get the interrupt number
		MOV		byte ptr pi1+1,AL	; Patch
		MOV		byte ptr pi2+1,AL	; Patch
		MOV		byte ptr pi3+1,AL	; Patch
		MOV		byte ptr pi4+1,AL	; Patch
		MOV		byte ptr pi5+1,AL	; Patch
		MOV		byte ptr pi6+1,AL	; Patch
}

/*
 * -------------- Debug console Application ---------------
 */

/*
 * Update status line
 */
register status(unsigned args)
{
	unsigned char buffer[80];
	_format_(nargs() * 2 + &args, buffer);
	w_clwin(swin);
	w_puts(buffer, swin);
}

/*
 * Draw a line at a position on the screen
 */
void draw_line(unsigned x, unsigned m)
{
	unsigned t, i;
	unsigned char a;
	int c;
	static unsigned Attrs[] = {
		WSTD, WSTD, WPRI, WERR, WERR, WCMD };
	wgotoxy(0, x);
	*W_OPEN = Attrs[a = Attr[m]];
	if(a) {
		t = ((Base - m)-1) & (MSGS-1);
		wprintf("%-4u %02u:%02u:%02u  ", t+1, Hour[m], Minute[m], Second[m]);
		t = (m * MSIZE);
		for(i=0; i < 64; ++i) {
			if(!(c = peek(Seg, t++)))
				break;
			wputc(c); } }
	wcleol();
}

/*
 * Update the onscreen display
 */
void update(void)
{
	unsigned i, r, e, f;
	unsigned char a;
	r = Base = Write;
	i = LINES;
	e = Ecount;
	f = MSGS;
	do {
		a = Attr[r = (r-1) & (MSGS-1)];
		if(a == ERR) {
			--e;
			draw_line(--i, r);
			continue; }
		if((i > Maxerr) || (i > e))
			draw_line(--i, r); }
	while(i && --f);
}

/*
 * Record an entry in the log
 */
void log(unsigned char a, unsigned char *text)
{
	unsigned i, p, h, m, s;
	int c;

	if(Attr[Write] == ERR)
		--Ecount;
	get_time(&h, &m, &s);
	p = Write * 64;
	for(i=0; i < 64; ++i) {
		poke(Seg, p++, c = *text++);
		if(!c)
			break; }
	Hour[Write] = h;
	Minute[Write] = m;
	Second[Write] = s;
	if((Attr[Write++] = a) == ERR)
		++Ecount;
	if(Mcount < MSGS)
		++Mcount;
	Write &= (MSGS-1);
}

/*
 * Review captured log
 */
void review(void)
{
	unsigned i, r, p, p1;
	unsigned char f;
	p = Write;
	for(;;) {
		p &= (MSGS-1);
		p1 = ((Base = Write) - p) & (MSGS-1);
		f = 255;
		r = p;
		i = LINES;
		f = 255;
		do {
			--i;
			if(f) {
				draw_line(i, r = (r - 1) & (MSGS-1));
				if(r == Base) {
					*W_OPEN = WSTD;
					f = 0; }
				continue; }
			wgotoxy(0, i);
			wcleol(); }
		while(i);
		status("Review: %u of %u : Up/Dn/PgUp/PgDn/Home/End=Position ESC=exit",
			p1+1, Mcount);
		switch(winput()) {
		case -1: continue;
		case _KDA : if(p1) ++p;				continue;
		case _KUA : if(p1 < Mcount) --p;	continue;
		case _KPD :
			if(p1 > LINES) {
				p += (LINES-1);
				continue; }
		case _KEN : p = Write;				continue;
		case _KPU :
			if((p1 + LINES) < Mcount) {
				p -= (LINES-1);
				continue; }
		case _KHO : p = (Write-Mcount)+1;		continue;
		case 0x1B:
			return; } }
}

unsigned
	_mode = 3,
	_type = 0xEDEB,
	Cpos;

unsigned char
	Pkt_int = PKT_INT,
	Cbuffer[14+65];

main(int argc, char *argv[])
{
	unsigned ps, po;
	unsigned h1;
	unsigned char my_address[PKT_EASIZE];
	unsigned alen;
	struct PD_driver_info info;
	static char pkt_driver_id[] = { "PKT DRVR" };
	struct PD_packet *p;
	unsigned i, t;

	memset(Cbuffer.to_addr, 0xFF, 6);
	for(i=1; i < argc; ++i) {
		p = argv[i];
		switch((toupper(*p++) << 8) | toupper(*p++)) {
		case 'I=' : set_interrupt(Pkt_int = atox(p));		continue;
		case 'M=' : _mode = atoi(p);						continue;
		case 'T=' : _type = atox(p);						continue;
		case 'W=' : Maxerr = atoi(p);						continue;
		case '?'<<8:
		case '-?' :
		case '/?' :
			abort("No help yet"); }
		printf("Bad option: '%s' - use ? for help\n", p-2);
		exit(-1); }
	
	/* Check that a packet driver is present */
	po = peekw(0, Pkt_int*4) + 3;	/* Offset to packet ID text */
	ps = peekw(0, (Pkt_int*4)+2);	/* Segment of packet driver */
	t = 0;
	do {
		if(pkt_driver_id[t] != peek(ps, po+t)) {
			printf("No packet driver on INT %02x [%04x:%04x]\n", Pkt_int, ps, po-3);
			exit(-1); } }
	while(pkt_driver_id[++t]);

	/* Obtain information from driver and display */
	if(t = pkt_driver_info(info)) {
		printf("Unable to obtain packet driver information, RC=%04x\n", t);
		exit(-1); }
	printf("Pkt driver: version=%u ", info.version);
	printf("Class/Type/Num=%02x/%04x/%02x ", info.class, info.type, info.number);
	printf("Func=%02x ", info.functionality);
	printf("Name='%s'\n", info.name);

	/* Obtain access to our packet type */
	t = _type ? sizeof(_type) : 0;
	if(t = pkt_access_type(info.class, info.type, 0, &_type, t, &h1)) {
		printf("Unable to obtain packet access, RC=%04x\n", t);
		exit(-1); }

	/* Get our ethernet address */
	if(t=pkt_get_address(h1, my_address, sizeof(my_address), &alen)) {
		printf("Unable to read ethernet address!, RC=%04x\n", t);
		goto release_driver; }

	if(t=pkt_set_receive_mode(h1, _mode)) {
		printf("Unable to set access mode!, RC=%04x\n", t);
		goto release_driver; }

	printf("Monitoring: ");
	switch(_mode) {
	case 6 : printf("All packets"); goto dotype;
	case 5 : printf("All"); goto domulti;
	case 4 : printf("Limited");	
	domulti: printf(" multicast+");
	case 3 : printf("Broadcast+");
	case 2 :
		for(i=0; i < alen; ++i) {
			if(i) putc(':', stdout);
			printf("%02x", my_address[i]); }
	dotype: 
		printf(_type ? " of type:%04x\n" : " of all types\n", _type);
		break;
	case 1 : printf("No packets!\n"); }

	Seg = alloc_seg(4096);
	swin = wopen(0, LINES, 80, 1, WSAVE|WCOPEN|REVERSE);
	mwin = wopen(0, 0, 80, LINES, WSAVE|WCOPEN|WSCROLL|WSTD);
	wcursor_off();

	memcpy(Cbuffer.from_addr, my_address, alen);
	Cbuffer.ptype = _type;

restat:
	status("Logging");
redisp:
	update();
	for(;;) {
		switch(i = winput()) {
		case -1: goto redisp;
		case _K10: if(Mcount) review(); goto restat;
		case '\b' :
		case _KLA :
		case _KBS :
			if(Cpos) {
				Cbuffer.data[--Cpos] = 0;
				if(!Cpos) goto restat; }
recmd:		status("Cmd: %s_", Cbuffer.data);
			continue;
		case '\n' :
			if(Cpos) {
				pkt_send(Cbuffer, sizeof(Cbuffer));
				Cpos = 0;
				log(CMD, Cbuffer.data);
				goto restat; }
			continue;
		case 0x1B:
			wclose();
			wclose();
		release_driver:
			if(t = pkt_release_type(h1))
				printf("Failed to release packet driver, RC=%04x\n", t);
			return; }
		if((Cpos < (sizeof(Cbuffer)-1)) && (i >= ' ') && (i <= '~')) {
			Cbuffer.data[Cpos++] = i;
			Cbuffer.data[Cpos] = 0;
			goto recmd; } }
}

winput()
{
	unsigned a, l;
	struct PD_packet *p;
	for(;;) {
		if(pkt_head != pkt_tail) {
			do {
				p = pkt_queue[pkt_tail].packet;
				if(p->ptype == _type) {
					l = pkt_queue[pkt_tail].length;
					a = p->data[0];
					if(a && (a < 4))
						log(p->data[0], (p->data)+1);
					else
						log(PRI, "**Bad Debug Packet**"); }
				pkt_tail = (pkt_tail+1) & (PKT_WINDOW-1); }
			while(pkt_head != pkt_tail);
pret:		return -1; }
		if(l = wtstc()) { switch(l) {
			case _K1 : log(STD, "**Standard Packet**");	goto pret;
			case _K2 : log(PRI, "**Priority Packet**");	goto pret;
			case _K3 : log(ERR, "**Error Packet**");	goto pret;
			} return l; } }
}
