#include <stdio.h>
#include <window.h>

/* Parallel port I/O lines */
#define	DATA		0x04	// Data line
#define	CLOCK		0x08	// Clock line
#define	LPT_flip	0x0B	// Lines to flip

#define	MX			25		// X position to start display
#define	MY			3		// Y position to start display
#define	RECLEN		34		// Record length for write file

unsigned
	Port = 0x278,			// Parallel port
	Device = -1,			// Device selection index
	Edit,					// Edit address
	Slow,					// Slowdown factor
	Base,					// Base load address
	Size,					// Maximum size of device
	Seg;					// Segment for RAM buffer

unsigned char
	*ptr,					// General parsing pointer
	P = DATA|CLOCK,			// LPT port output state
	Fill = 0xFF,			// Fill value
	Outfmt,					// Output format
	Devtype,				// Device type
	Page,					// Page write mask
	Bflag,					// Base set flag
	Emode,					// Edit mode (HEX/ASCII)
	Run = -1,				// Run flag
	Address = 0xA0,			// I2C address
	filename[65],			// File to read/write
	buffer[256];


#define	CPROMPT		0		// Color index for prompt bar
#define	CMAIN		1		// Color index for main window
#define	CMESSAGE	2		// Color index for message window
#define	CEDIT		3		// Color index for edit
#define	CMENU		4		// Color index for pop-up menus

unsigned char C_color[] = {
	0x70,					/* Prompt bar at bottom */
	0x17,					/* Main view screen */
	0x67,					/* Message window */
	0x47,					/* Preview window */
	0x70 };					/* Pop-up menues */
unsigned char M_color[] = {
	0x70,					/* Prompt bar at bottom */
	0x07,					/* Main view screen */
	0x07,					/* Message window */
	0x70,					/* Preview window */
	0x70 };					/* Pop-up menus */
unsigned char *color = &C_color;

struct WINDOW
	*swin,					/* Status window */
	*pwin,					/* Prompt window */
	*mwin;					/* Main window */

char mprompt[] = {
"  A)ddr  C)lear  D)evice  E)dit  L)oad  R)ead  S)ave  V)erify  W)rite  F10:Exit" };
char eprompt[] = {
	" F1:Hex/ASCII  F2:Goto  F3:Fill  \x1B\x18\x19\x1A/Home/End/Pu/Pd/^Pu/^Pd:Move  ESC:Exit " };
char *file_form[] = {
	76 << 8 | 3,
	"\x00\x00\x41Filename?",
	0 };

char *device_list[] = {
	"24C01",
	"24C02",
	"24C04",
	"24C08",
	"24C16",
	"24C32",
	"24C64",
	"24C128",
	"24C256",
	"24C512",
	0 };

struct  {
	unsigned devSize;
	unsigned char devPage;
	unsigned char devType; } device_data[] = {

	int 128, char 0x07, 0x00,		// 24C01
	int 256, char 0x07, 0x00,		// 24C02
	int 512, char 0x0F, 0x00,		// 24C04
	int 1024, char 0x0F, 0x00,		// 24C08
	int 2048, char 0x0F, 0x00,		// 24C16
	int 4096, char 0x1F, 0xFF,		// 24C32
	int 8192, char 0x1F, 0xFF,		// 24C64
	int 16384,char 0x3F, 0xFF,		// 24C128
	int 32768,char 0x3F, 0xFF,		// 24C256
	int 0000, char 0x7F, 0xFF		// 24C512
	};

/*
 * Command line help text
 */
char help[] = { "\n\
Use: 	I2CE [options]\n\n\
opts:	/C	- Force no Color (Monochrome) video\n\
	/B	- write files in Binary format\n\
	/I	- write files in Intel format\n\
	/M	- write files in Motorola format (default)\n\
	/W	- display Wiring information\n\
	/X	- iint lpt & eXit (turn off device)\n\
	A=0-7	- select i2c device Address\n\
	B=xxxx	- specify load/save Base address\n\
	D=name	- select Device (otherwise prompt)\n\
	F=xx	- set Fill (clear) value\n\
	P=n/adr	- select lpt Port (1-3) or address\n\
	S=n	- Slow down by 'n' IN cycles\n\n\
?COPY.TXT 2000-2005 Dave Dunfield.\n -- see COPY.TXT --.\n" };

char wire[] = { "\n\
I2CE adapter connections for standard DB-25 PC parallel port\n\n\
8-pin DIP	LPT Port\n\
------------------------\n\
1 A0		18-25 Ground\n\
2 A1		18-25 Ground\n\
3 A2		18-25 Ground\n\
4 Gnd		18-25 Ground\n\
5 DATA		16 -INIT\n\
6 CLOCK		17 -SLCTIN\n\
7 WP		18-25 Ground\n\
8 VCC		2-9 Data *\n\n\
* = NOTE: I2CE always drives the data pins ALL HIGH  (to power the device),\n\
    or ALL LOW (to unpower the device).  If any other data value is written\n\
    to the parallel port while the I2CE adapter is installed, damage to the\n\
    parallel port may result.  For this reason, you should install the I2CE\n\
    adapter **AFTER** you have run I2CE at least once, and remove it before\n\
    you run anything else which may write to the printer port.\n"};

/*
 * Formatted print to message window
 *
 * Input: Format string, Variable argument list
 */
static void register message(unsigned args)
{
	unsigned char buffer[81];

	_format_(nargs() * 2 + &args, buffer);
	w_puts(buffer, mwin);
}

/*
 * Issue a prompt
 */
satic void prompt(char *s)
{
	w_clwin(pwin);
	w_puts(s, pwin);
}

/*
 * Edit global memory 
 */
static void edit()
{
	unsigned dt, dc, i, j, k;

	wopen(3, 3, 73, 18, WSAVE|WCOPEN|WBOX1|color[CEDIT]);
	title("MEMORY EDIT");
	wcursor_line();

	prompt(eprompt);

redraw:
	dt = Edit & 0xFF00;
	for(i=0; i < 16; ++i) {
		wgotoxy(0, i);
		draw_line(dt+(i*16)); }
prompt:
	if(Edit > (Size-1))
		Edit &= (Size-1);
	if((Edit & 0xFF00) != dt)
		goto redraw;
	dc = Edit & 0xFF;
	wgotoxy(Emode ? (dc%16)+55 : ((dc%16)*3)+6, dc/16);
	switch(toupper(i = wgetc())) {
		case 0x1B :
			wclose();
			return;
		case _K1 :		/* Toggle mode */
			Emode = !Emode;
			break;
		case _K2 :		/* Go to address */
			get_number("Edit address? ", 16, &Edit);
			break;
		case _K3 :		/* Fill memory */
		regetstart:
			if(get_number("Starting address? ", 16, &dc))
				break;
			if(dc > (Size-1)) {
				wputc(7);
				goto regetstart; }
		regetend:
			if(get_number("Ending address? ", 16, &dt))
				break;
			if((dt > (Size-1)) || (dt < dc)) {
				wputc(7);
				goto regetend; }
			if(get_number("Fill with? ", 16, &i))
				break;
			do
				poke(Seg, dc, i);
			while(dc++ < dt);
			break;
		case _KLA :
			--Edit;
			break;
		case _KPU :
			Edit -= 256;
			break;
		default:
			if(Emode) {
				if(i >= 128)
					break;
				poke(Seg, Edit, i); }
			else {
				if((j = gethex(i)) >= 16)
					break;
				wputc(i);
				if((k = gethex(wgetc())) < 16)
					poke(Seg, Edit, (j<<4)+k); }
			wgotoxy(0, i = dc/16);
			draw_line(dt+(i*16));
		case _KRA :
			++Edit;
			break;
		case _KPD :
			Edit += 256;
			break;
		case _KHO :
			Edit = (Edit-1) & 0xFFF0;
			break;
		case _KUA :
			Edit -= 16;
			break;
		case _KEN :
			Edit = (Edit+1) | 0x000F;
			break;
		case _KDA :
			Edit += 16;
			break;
		case _CPU :
			Edit = 0;
			break;
		case _CPD :
			Edit = Size-1; }
	goto prompt;
}

/*
 * Display a line for the editor
 */
draw_line(addr)
	int addr;
{
	unsigned int i, c, ptr;
	unsigned char buffer[16];

	ptr = addr;
	wprintf("%04x  ", addr);
	for(i=0; i < 16; ++i)
		wprintf("%02x ", buffer[i] = peek(Seg, ptr++));
	wputc(' ');
	for(i=0; i < 16; ++i) {
		c = buffer[i];
		wputc(((c >= ' ') && (c <= 0x7E)) ? c : '.'); }
}

/* Test for HEX digit & get value*/
gethex(c)
	int c;
{
	if((c >= '0') && (c <= '9'))
		return c-'0';
	if((c >= 'A') && (c <= 'F'))
		return c-'A'+10;
	if((c >= 'a') && (c <= 'f'))
		return c-'a'+10;
	return -1;
}

/*
 * Display a title on a window
 */
title(ptr)
	char *ptr;
{
	char buffer[8];
	memcpy(buffer, W_OPEN, sizeof(buffer));
	*W_OPEN = REVERSE;
	W_OPEN[1] &= (~WBOX3 >> 8);
	--W_OPEN[3];
	wgotoxy(0,0);
	wprintf(" %s ", ptr);
	memcpy(W_OPEN, buffer, sizeof(buffer));
}

/*
 * Get an numeric value at a prompt
 */
get_number(prompt, base, var)
	char *prompt;
	int base, *var;
{
	unsigned int i, j, x, value, len;

	i = (len = (base == 10) ? 5 : 4) + strlen(prompt) + 3;
	wopen((80-i)/2, 10, i, 3, WSAVE|WBOX3|WCOPEN|REVERSE);
	wputs(prompt);
	wcursor_line();
	value = x = 0;
	for(;;) {
		switch(i = wgetc()) {
			case '\n' :
				if(x) {
					*var = value;
					wclose();
					return 0; }
			case 0x1B :
				wclose();
				return i;
			case _KBS :
			case _KLA :
				if(x) {
					--x;
					wputs("\b \b");
					value /= base; }
				break;
			default:
				if(isdigit(j = toupper(i)))		/* convert numeric digits */
					j -= '0';
				else if(j >= 'A')				/* convert alphabetics */
					j -= ('A' - 10);
				else							/* not a valid "digit" */
					break;
				if((x < len) && (j < base)) {
					wputc(i);
					++x;
					value = (value * base) + j; } } }
}

/*
 *------------------------
 * I2C interface functions
 *------------------------
 */

/*
 * Write a byte to the I2C control port
 */
ICout(unsigned char v)
{
	unsigned i;

	out(Port+2, v ^ LPT_flip);
	if(Slow) {
		i = Slow;
		do
			in(Port+1);
		while(--i); }
}

/*
 * Generate a start condition on the I2C interface
 */
ICstart()
{
	ICout(P |= CLOCK);		// Raise clock
	ICout(P &= ~DATA);		// Drop data = Start
	ICout(P &= ~CLOCK);		// Drop clock
}

/*
 *  Check for an ACK from the I2C interface
 */
ICack(char *reason)
{
	unsigned char a;
	ICout(P |= DATA);		// Raise data for input
	ICout(P |= CLOCK);
	a = in(Port+2) & DATA;
	ICout(P &= ~CLOCK);
	if(a) {
		if(reason)
			message("\n%s ACK not reported!", reason); }
	return a;
}

/*
 * Send an ack condition to the I2C interface
 */
ICsack()
{
	ICout(P &= ~DATA);
	ICout(P |= CLOCK);
	ICout(P &= ~CLOCK);
}

/*
 *  Send a data byte to the I2C interface
 */
ICsend(unsigned char v)
{
	unsigned i;

	for(i=0; i < 8; ++i) {
		P &= ~DATA;
		if(v & 0x80)
			P |= DATA;
		ICout(P);
		ICout(P |= CLOCK);
		v <<= 1;
		ICout(P &= ~CLOCK); }
}

/*
 * Generate a STOP condition on the I2C interface
 */
ICstop()
{
	ICout(P &= ~DATA);
	ICout(P |= CLOCK);
	ICout(P |= DATA);
}

/*
 * Send an address to the I2C interface
 */
ICaddr(unsigned address)
{
	unsigned t, l, c;
	unsigned char a;

	t = l = peek(0x40, 0x6C);
	c = 2;
	a = Address;
	if(!Devtype)
		a |= (address >> 7) & 0x0E;

	for(;;) {
		t = peek(0x40, 0x6C);
		if(t != l) {
			l = t;
			if(!--c) {
				message("\nDevice not responding!");
				return -1; } }
		ICstart();
		ICsend(a);			// Write with address
		if(!ICack(0))
			break;
		ICstop(); }

	if(Devtype) {
		ICsend(address >> 8);
		if(ICack("ADDRH"))
			return -1; }
	ICsend(address);
	if(ICack("ADDRL"))
		return -1;

	return 0;
}

/*
 * Start a "current" read through the I2C interface
 */
ICcurr()
{
	unsigned c;
	c = 10;
	do {
		if(!--c) {
			message("\nDevice not responding (2)");
			return -1; }
		ICstart();
		ICsend(Address | 1); }
	while(ICack("CURR"));
	return 0;
}

/*
 * Read a data byte through the I2C interface
 */
ICread()
{
	unsigned i;
	unsigned char v;

	ICout(P |= DATA);

	for(i=v=0; i < 8; ++i) {
		ICout(P |= CLOCK);
		v <<= 1;
		if(in(Port+2) & DATA)
			v |= 1;
		ICout(P &= ~CLOCK); }

	return v;
}

/*
 * Reset the I2C device
 */
ICreset()
{
	unsigned i;

	ICout(P |= DATA);
	for(i=0; i < 9; ++i) {
		ICout(P |= CLOCK);
		if(in(Port+2) & DATA)
			return;
		ICout(P &= !CLOCK); }
	message("Reset at end!");
}

/*
 * Read the I2C memory device
 */
read_device(unsigned address, unsigned length)
{
	message("\nReading ....");
	if(ICaddr(address))
		return -1;
	ICstop();
	if(ICcurr())
		return -1;
	for(;;) {
		if(!(address & 0x0F)) {
			message("\b\b\b\b%04x", address);
			if(wtstc() == 0x1B) {
				ICstop();
				message(" Canceled!");
				return -1; } }
		poke(Seg, address++, ICread());
		if(!--length) {
			ICstop();
			break; }
		ICsack(); }

	message("\b\b\b\b%04x Done!", address-1);
	return 0;
}

/*
 * Verify the I2C memory device
 */
verify_device(unsigned address, unsigned length)
{
	unsigned e, a;
	unsigned char c1, c2, e1, e2;
	e = 0;
	message("\nVerifying ....");
	if(ICaddr(address))
		return -1;
	ICstop();
	if(ICcurr())
		return -1;
	for(;;) {
		if(!(address & 0x0F)) {
			message("\b\b\b\b%04x", address);
			if(wtstc() == 0x1B) {
				ICstop();
				message(" Canceled!");
				return -1; } }
		if((c1=ICread()) != (c2=peek(Seg, address++))) {
			if(!e) {
				a = address-1;
				e1 = c1;
				e2 = c2; }
			++e; }
		if(!--length) {
			ICstop();
			break; }
		ICsack(); }

	message("\b\b\b\b%04x Done! - ", address-1);
	if(e)
		message("%u errors [%04x: %02x-%02x]", e, a, e1, e2);
	else
		message("No errors");
	return 0;
}

/*
 * Write the I2C memory device
 */
write_device(unsigned address, unsigned length)
{
	message("\nWriting ....");
readdr:
	if(ICaddr(address))
		return -1;
	for(;;) {
		if(!(address & 0x0F)) {
			message("\b\b\b\b%04x", address);
			if(wtstc() == 0x1B) {
				ICstop();
				message(" Canceled!");
				return -1; } }
		ICsend(peek(Seg, address++));
		if(ICack("DWRITE"))
			return -1;
		if(!--length)
			break;
		if(!(address & Page)) {
			ICstop();
			goto readdr; } }
	ICstop();
	delay(100);
	ICaddr(0);
	ICstop();

	message("\b\b\b\b%04x Done!", address-1);
	return 0;
}

/*
 *------------------------
 * File I/O functions
 *------------------------
 */

/*
 * Read file into memory image
 */
int read_file(filename)
	char *filename;
{
	unsigned a, b, chksum, loadptr, count, lcount, maxsize, base;
	char xflag, bflag;
	FILE *fp;

	base = Base;
	bflag = Bflag;

	if(!(fp = fopen(filename,"r"))) {
		message("\nUnable to read file: %s", filename);
		return 0; }

	xflag = lcount = maxsize = 0;
	for(;;) {
		if(!fgets(ptr = buffer, 80, fp)) {
			message("\nNo end of file record!");
			goto quit; }
		again: switch(*ptr++) {
			case 'S' :	/* Motorola HEX format */
				if(!Outfmt) Outfmt = 2;
				if(*ptr == '9') goto quit;
				if(*ptr++ != '1') continue;
				lcount += count = (chksum = get_byte()) - 3;
				chksum += a = get_byte();
				chksum += b = get_byte();
				loadptr = (a << 8) + b;
				if(!bflag) {
					base = loadptr;
					bflag = -1; }
				if(loadptr < base)
					goto quitbase;
				loadptr -= base;
				while(count--) {
					chksum += a = get_byte();
					poke(Seg, loadptr++, a); }
				if((255 & ~chksum) != get_byte())
					goto quitbad;
				break;
			case ':' :		/* Intel HEX format */
				if(!Outfmt) Outfmt = 3;
				if(!(count = get_byte())) goto quit;
				lcount += count;
				chksum = (a = get_byte()) + count;
				chksum += b = get_byte();
				loadptr = (a << 8) + b;
				if(!bflag) {
					base = loadptr;
					bflag = -1; }
				if(loadptr < base)
					goto quitbase;
				loadptr -= base;
				chksum += get_byte();
				while(count--) {
					chksum += a = get_byte();
					poke(Seg, loadptr++, a); }
				if((255 & -chksum) != get_byte())
					goto quitbad;
				break;
			case ' ' :	/* Space */
			case '\t' :	/* Tab */
				goto again;
			case 0 :		/* Null line */
				continue;
			default:
				message("\nInvalid record format!");
				goto quit; }
		if(loadptr > maxsize)
			maxsize = loadptr;
		if(loadptr > Size) {
			if(!xflag)
				message("\nDevice size exceeded!");
			xflag = -1; } }
quitbase:
	message("\nRecord < base");
	goto quit;
quitbad:
	message("\nBad record checksum!");
quit:
	fclose(fp);
	message(" (%u bytes, image=%u, base=%04x)", lcount, maxsize, base);
	return -1;
}

/* Load a byte from the hex file */
get_byte()
{
	return (get_hex(*ptr++) << 4) + get_hex(*ptr++);
}

/* Test for HEX digit & get value*/
get_hex(c)
	int c;
{
	if((c >= '0') && (c <= '9'))
		return c-'0';
	if((c >= 'A') && (c <= 'F'))
		return c-'A'+10;
	if((c >= 'a') && (c <= 'f'))
		return c-'a'+10;
	abort("HEXFMT: Bad hex digit.");
}

/*
 * Write file from memory image
 */
int write_file(name, size)
	char *name;
	unsigned size;
{
	unsigned cnt, locn, adr1;
	FILE *fp;

	cnt = locn = 0;

	if(!Outfmt)
		Outfmt = 2;

	if(!(fp = fopen(name, (Outfmt > 1) ? "w" : "wb"))) {
		message("\nUnable to write file: %s", name);
		return 0; }

	if(Outfmt > 1) {
		while(locn < size) {
			if(!cnt) {
				adr1 = locn + Base;
				buffer[cnt++] = adr1 >> 8;
				buffer[cnt++] = adr1 & 255; }
			buffer[cnt++] = peek(Seg, locn++);
			if(cnt >= RECLEN) {
				write_record(fp, cnt);
				cnt = 0; } }
		write_record(fp, cnt);
		switch(Outfmt) {
			case 2 :
				fprintf(fp,"S9030000FC\n");
				break;
			case 3 :
				fprintf(fp,":00000001FF\n"); } }

	else			/* pure binary output */
		while(locn < size)
			putc(peek(Seg, locn++), fp);

	fclose(fp);
	return -1;
}

/* write mhx record to a file */
write_record(fp, count)
	FILE *fp;
	unsigned count;
{
	unsigned chk, i, chr;

	if(count > 2) {
		switch(Outfmt) {
			case 2 :		/* Motorola */
				fprintf(fp,"S1%02x", chk = count + 1);
				for(i=0; i < count; ++i) {
					fprintf(fp,"%02x", chr = buffer[i]);
					chk += chr; }
				fprintf(fp,"%02x\n",255&~chk);
				break;
			case 3 :		/* Intel */
				chk = buffer[0] + buffer[1] + count - 2;
				fprintf(fp,":%02x%02x%02x00",count-2, buffer[0], buffer[1]);
				for(i=2; i < count; ++i) {
					fprintf(fp,"%02x",chr = buffer[i]);
					chk += chr; }
				fprintf(fp,"%02x\n",255 & (0-chk)); } }
}

/*
 * Main program
 */
main(int argc, char *argv[])
{
	unsigned i;

	/* Parse command line options */
	Port = peekw(0x40, 0x08);		/* Get LPT1 address */
	for(i=1; i < argc; ++i) {
		ptr = argv[i];
		switch((toupper(*ptr++) << 8) | toupper(*ptr++)) {
#ifndef DEMO
		case ('-'<<8)|'B' :		/* Save in binary */
		case ('/'<<8)|'B' :
			Outfmt = 1;
			continue;
		case ('-'<<8)|'M' :		/* Save in Motorola */
		case ('/'<<8)|'M' :
			Outfmt = 2;
			continue;
		case ('-'<<8)|'I' :		/* Save in Intel */
		case ('/'<<8)|'I' :
			Outfmt = 3;
			continue;
#else
		case ('-'<<8)|'B' :
		case ('/'<<8)|'B' :
		case ('-'<<8)|'M' :
		case ('/'<<8)|'M' :
		case ('-'<<8)|'I' :
		case ('/'<<8)|'I' :
			abort("/B /I /M options not available in demo.");
#endif
		case ('-'<<8)|'W' :		/* Wiring information */
		case ('/'<<8)|'W' :
			fputs(wire, stdout);
			return;
		case ('-'<<8)|'C' :		/* Disable color screen */
		case ('/'<<8)|'C' :
			color = M_color;
			continue;
		case ('-'<<8)|'X' :
		case ('/'<<8)|'X' :
			Run = 0;
			continue;
		case ('A'<<8)|'=' :
			if((Address = atox(ptr)) > 7) {
				fputs("Address must be 0-7\n", stderr);
				goto gohelp; }
			Address = (Address << 1) | 0xA0;
			continue;
		case ('P'<<8)|'=' :
			if(!(Port = atox(ptr))) {
				fputs("LPT must be 1-3 or address\n", stderr);
				goto gohelp; }
			if(Port < 4)
				Port = peekw(0x40, (Port*2)+6);
			continue;
		case ('B'<<8)|'=' :		/* Base address */
			Base = atox(ptr);
			Bflag = -1;
			continue;
		case ('D'<<8)|'=' :		/* Device type */
			strupr(ptr);
			for(Device=0; i = device_list[Device]; ++Device) {
				if(!strcmp(i, ptr))
					goto found_dev; }
			fprintf(stderr,"Unknown device: %s\n", ptr);
			goto gohelp;
		case ('F'<<8)|'=' :		/* fill value */
			Fill = atox(ptr);
		found_dev:
			continue;
		case ('S'<<8)|'=' :		/* Slowdown factor */
			Slow = atoi(ptr);
			continue;
		default:
			fprintf(stderr,"Invalid option: %s\n", argv[i]);
		case ('?'<<8):
		case ('-'<<8)|'?':
		case ('/'<<8)|'?':
		gohelp:
			abort(help); } }

	if(!Port) {
		fputs("LPT port not found!\n", stderr);
		goto gohelp; }

	ICout(0);
	out(Port, 0);

	if(!Run)
		return;

	if(!(Seg = alloc_seg(4096)))
		abort("Not enough memory.");
	i = 0;
	do
		poke(Seg, i, Fill);
	while(++i);

	/* Open windows */
	swin = wopen(0, 0, 80, 1, WSAVE|WCOPEN|REVERSE);
	wputs("I2C EEPROM pgmr 1.1 - ?COPY.TXT 2000-2005 Dave Dunfield -  -- see COPY.TXT --.");
	if(W_BASE == 0xB000)
		color = M_color;
	pwin = wopen(0,24, 80, 1, WSAVE|WCOPEN|color[CPROMPT]);
	mwin = wopen(0, 17, 80, 23-16, WSAVE|WCOPEN|WSCROLL|color[CMESSAGE]);
	wopen(0, 1, 80,16, WSAVE|WCOPEN|color[CMAIN]);
	wcursor_off();
#ifndef DEMO
	message("Ready");
#else
	message("Demo version");
#endif

	if(Device > 15) {
		Device = 0;
		if(wmenu(10, 3, WSAVE|WCOPEN|WBOX1|color[CMENU], device_list, &Device))
			goto doexit; }

reselect:
	Size = device_data[Device].devSize;
	Page = device_data[Device].devPage;
	Devtype = device_data[Device].devType;
	Edit = 0;

	wclwin();
	wgotoxy(MX-2, MY-1); wputs("*** I2C Device Information ***");
	wgotoxy(MX, MY+2); wprintf("I2C Address: %u", (Address >> 1) & 7);
	wgotoxy(MX, MY+4); wprintf("Device: %s", device_list[Device]);
	wgotoxy(MX, MY+6); wputs("Size: ");
	if(Size)
		wprintf("%-5u (%04x)", Size, Size);
	else
		wputs("65536 (10000)");
	wgotoxy(MX, MY+8); wprintf("Address type: %s byte", Devtype ? "2" : "1");
	wgotoxy(MX, MY+10);wprintf("Write page size: %u", Page+1);
	if((Device >= 2) && (Device <= 4)) {
		wgotoxy(MX+22, MY+8);
		wputs("+I2C Address"); }

reprompt:
	prompt(mprompt);
command:
	switch(toupper(wgetc())) {
		case 'A' :
//			Address = ((Address+2) & 0x0E) | 0xA0;
			select_address();
			goto reselect;
		case 'R' :
			ICout(P = DATA|CLOCK);
			out(Port, -1);
			delay(500);
			read_device(0, Size);
		off:
			ICout(0);
			out(Port, 0);
			goto command;
		case 'V' :
			ICout(P = DATA|CLOCK);
			out(Port, -1);
			delay(500);
			verify_device(0, Size);
			goto off;
		case 'W' :
			ICout(P=DATA|CLOCK);
			out(Port, -1);
			delay(500);
			write_device(0, Size);
			goto off;
		case 'D' :
			wmenu(10, 3, WSAVE|WCOPEN|WBOX1|color[CMENU], device_list, &Device);
			goto reselect;
		case 'E' :
			edit();
			goto reprompt;
		case 'C' :
			i=0;
			do
				poke(Seg, i, Fill);
			while(++i);
			message("\nBuffer cleared.");
			goto command;
		case 'L' :
			message("\nLoad file ");
			if(wform(2, 8, WSAVE|WCOPEN|WBOX1|color[CMENU], file_form, filename)) {
				message(" Canceled!");
				goto command; }
			message(filename);
			if(read_file(filename))
				message(" Done.");
			goto command;
		case 'S' :
			message("\nSave file ");
#ifndef DEMO
			if(wform(2, 8, WSAVE|WCOPEN|WBOX1|color[CMENU], file_form, filename)) {
				message(" Canceled!");
				goto command; }
			message(filename);
			if(write_file(filename, Size))
				message(" Done!");
#else
			message(" not available in demo.");
#endif
			goto command;
		case _K10 :
		doexit:
			wclose();
			wclose();
			wclose();
			wclose();
			free_seg(Seg);
			return; }
	message("\nUnknown command");
	goto command;
}
select_address()
{
	unsigned k;
	unsigned char c, a;
	prompt("  Select value (0-7 or Arrow) press ENTER to save, ESC to cancel");
	c = *W_OPEN;
	*W_OPEN = (c >> 4)|(c << 4);
	a = (Address >> 1) & 7;
	for(;;) {
		wgotoxy(MX+12, MY+2);
		wprintf(" %u ", a &= 7);
		switch(k = wgetc()) {
		case _KRA :
		case _KUA :	++a; continue;
		case _KLA :
		case _KDA : --a; continue;
		case '\n' :
			Address = (a << 1) | 0xA0;
		case 0x1B :
			*W_OPEN = c;
			return;
		case '0' :
		case '1' :
		case '2' :
		case '3' :
		case '4' :
		case '5' :
		case '6' :
		case '7' :
			a = k-'0'; } }
}
