#include <stdio.h>
#include <lrg.h>
#include <keys.h>
#include <comm.h>

// Mouse buttons
#define	MOUSE_LEFT		0x01	/* Select button */
#define	MOUSE_RIGHT		0x02	/* Cancel button */
#define	MOUSE_CENTER	0x04	/* Center button */
#define	MOUSE_KEY		0x08	/* Key was entered */

// Screen colors
#define	ON			0
#define	GREY		24
#define	OFF			25
#define	MON			20
#define	MGREY		25
#define	MOFF		29
#define	CURSOR		0x0E
#define	TEXT		6
#define	BTEXT		2
#define	ETEXT		4
#define	MTEXT		2
#define	MARK		(OFF<<8)+1

#define	XSIZE		320
#define	YSIZE		102
#define	MAX_SEG		1024
#define	CLIPBD		16384

char *help[] = {
	//---------------------------------------
	"1:Base 2:Bddr 3:Saddr 4:BaddrAdj 5:Map",
	"6:Bmap 7:Cgen 8:Audit 9:Emulate 10:File",
	"Copy Fill Line Map Paste Redraw Snap",
	"Ins=Full Del=Zoom \x1B\x18\x19\x1A=Move View",
	"[^]Pu/Pd=Seg [^]Ho/En=Map ^\x1B=Con ^\x1A=Bri",
	0 };

char
	snap,
	*emsg,
	inc = -1,
	incstat,
	marky[10],
	lcdname[25],
	cgname[25],
	temp[30],
	cursor_flag;
unsigned
	key,
	bright = 4,
	grey = GREY-2,		// Contrast level
	xseg,				// Segment for LCD storage
	xbase,
	ybase,
	xbs,
	xbe,
	ybs,
	ybe,
	select,				// Selected color
	map,				// Map value
	offset,				// Address offset
	magnify,			// Magnification
	magnify1,			// Updated magnification
	mxbase,				// Mouse X base
	mybase,				// Mouse Y base
	mxmax = XSIZE-1,	// Mouse X max
	mymax = YSIZE-1,	// Mouse Y max
	mousex,				// Mouse X position
	mousey,				// Mouse Y position
	clipx,				// Clipboard X dimension
	clipy,				// Clipboard Y dimension
	markx[10],			// Marker X positions
	smap[MAX_SEG+1],	// Segment map
	clipboard[CLIPBD];	// Clipboard
FILE
	*fp;

/*
 * Initializes the mouse driver, returns with -1 if successful.
 */
int init_mouse(void) asm
{
; Initialize & test for mouse
		XOR		AX,AX			; Init functions.
		INT		33h				; Call mouse driver
		AND		AX,AX			; Mouse present
		JZ		initm1			; No, skip it
; Set mouse limits (some drivers do not do it properly on reset)
		XOR		CX,CX			; Lower limit is zero
		MOV		DX,638			; Upper horizontal limit
		MOV		AX,7			; Set horizontal limit
		INT		33h				; Call mouse driver
		MOV		DX,199			; Upper vertical limit
		MOV		AX,8			; Set vertical limit
		INT		33h				; Call mouse driver
		MOV		AX,-1			; Indicate mouse ok
initm1:
}

openf(char *f, char *a, char *e)
{
	char *p, d;
	p = temp;
	d = -1;
	while(*p = *f++) {
		if(*p++ == '.')
			d = 0; }
	if(d)
		strcpy(p, e);

	if(fp = fopen(temp, a))
		return 0;
	while(*p)
		++p;
	sprintf(p, " -Can't %s", (*a == 'W') ? "write" : "read");
	emsg = temp;
	return -1;
}

update_bounds()
{
	magnify1 = magnify+1;
	mxmax = (mxbase + (320 / magnify1)) - 1;
	mymax = (mybase + (102 / magnify1)) - 1;
	while(mxmax >= XSIZE) {
		--mxbase;
		--mxmax; }
	while(mymax >= YSIZE) {
		--mybase;
		--mymax; }
/*	if(mxmax >= XSIZE) {
		mxbase -= (mxmax - (XSIZE-1));
		mxmax  -= (mxmax - (XSIZE-1)); }
	if(mymax >= YSIZE) {
		mybase -= (mymax - (YSIZE-1));
		mymax  -= (mymax - (YSIZE-1)); */
	lrg_printf(0, 113, TEXT, "x%-2u [%4u,%-4u-%4u,%-4u]", magnify1,
		mxbase,mybase,mxmax,mymax);
	lrg_fbox(0, 92, 310, 10, BLACK);
	lrg_fbox(310, 0, 10, 102, BLACK);
}

plot(unsigned x, unsigned y, unsigned char c)
{
	if((x < mxbase) || (x > mxmax)) return;
	if((y < mybase) || (y > mymax)) return;
	if(!magnify) {
		lrg_plot(x, y, c);
		return; }
	x = (x - mxbase) * magnify1;
	y = (y - mybase) * magnify1;
	lrg_fbox(x, y, magnify1, magnify1, c);
}

refresh()
{
	unsigned x, y, z, c;
	if(magnify) {
		for(y=mybase; y <= mymax; ++y)
			for(x = mxbase; x <= mxmax; ++x)
				update(x, y); }
	else {
		for(y=z=0; y < YSIZE; ++y) {
			for(x=0; x < XSIZE; ++x) {
				if(c = peekw(xseg, z))
					lrg_plot(x, y, (c == select) ? ON : grey);
				else
					lrg_plot(x, y, OFF);
				z += 2; } } }

	for(z=0; z < 10; ++z) {
		if(y = marky[z]) {
			--y;
			x = markx[z];
			if((x < mxbase) || (x > mxmax)) continue;
			if((y < mybase) || (y > mymax)) continue;
			x = (x - mxbase) * magnify1;
			y = (y - mybase) * magnify1;
			lrg_putc(x, y, MARK, z+'0'); } }
}

plot_select(unsigned x, unsigned y)
{
	plot(x, y, MON);
}
set_select(unsigned x, unsigned y)
{
	set_pixel(x, y, select);
}

update(unsigned x, unsigned y)
{
	unsigned c;
	c = (y * XSIZE) + x;
	if(c = peekw(xseg, c+c)) {
		plot(x, y, (c == select) ? ON : grey);
		return; }
	plot(x, y, OFF);
}

update_cursor()
{
	
	if(peek(0x40, 0x6C) & 0x04) {
		plot(mousex, mousey, CURSOR);
		cursor_flag = -1;
		return; }
	if(cursor_flag) {
		update(mousex, mousey);
		cursor_flag = 0; }
}
set_pixel(unsigned x, unsigned y, unsigned s)
{
	unsigned c;
	c = (y * XSIZE) + x;
	pokew(xseg, c+c, s);
}
get_pixel(unsigned x, unsigned y)
{
	unsigned c;
	c = (y * XSIZE) + x;
	return peekw(xseg, c+c);
}

/*
 * Update mouse position and on-screen cursor. If any button is
 * activated, remove cursor, wait for button to be released, and
 * report it.
 */
int mouse_status(void)
{
	unsigned x, y, z;

	/* If no cursor on screen, draw one */
newcursor:
	update_cursor();
	/* Get mouse position and button status */
	asm {
		MOV		AX,0003h	; Mouse status function
		INT		33h			; Call mouse driver
		MOV		-2[BP],BX	; Save buttons
		MOV		-4[BP],DX	; Save Y position
		MOV		-6[BP],CX	; Save X position
		}
	// Compute screen position from mouse driver output
	y = (y * (mymax-mybase) / 199) + mybase;
	x = lrg_scale(x, mxmax-mxbase, 638) + mxbase;

	/* If snap enabled, force cursor alignment */
	if(snap) {
		x = (x / snap) * snap;
		y = (y / snap) * snap; }

	/* If cursor position changed, update cursor and display*/
	if((x != mousex) || (y != mousey)) {
		update(mousex, mousey);
		cursor_flag = 0;
		lrg_printf(249, 113, TEXT, "%4d,%-4d", (mousex = x)-xbase, (mousey = y)-ybase);
		goto newcursor; }

	/* If any buttons are activated, wait for release, remove cursor */
	if(z & (MOUSE_LEFT|MOUSE_RIGHT|MOUSE_CENTER)) {
		asm {
			mloop1:	MOV		AX,0003h		; Mouse status function
					INT		33h				; Call mouse driver
					AND		BL,07h			; Any buttons down?
					JNZ		mloop1			; Wait till clear
			} }

	switch(key = kbtst()) {
	case '\r' :
		return MOUSE_LEFT;
	case 0x1B :
		return MOUSE_RIGHT;
	case ' ' :
		if(incstat & 0x01) {	// Enable updates
			incstat |= 0x80;
			inc = inc ? 0 : -1; }
	/* default: z |= MOUSE_KEY;
	case 0: */ }

	if(incstat & 0x80) {
		lrg_puts(296, 103, MTEXT, inc ? "Inc" : "   ");
		incstat &= 0x7F; }

	/* Pass back button status to caller */
	return z;
}

plot_square(unsigned x, unsigned y, unsigned xx, unsigned yy, unsigned c)
{
	unsigned i;
	if(x > xx) {
		i = x;
		x = xx;
		xx = i; }
	if(y > yy) {
		i = y;
		y = yy;
		yy = i; }

	if(c) {
		for(i=x; i < xx; ++i) {
			plot(i, y, c);
			plot(i, yy, c); }
		for(i=y; i < yy; ++i) {
			plot(x, i, c);
			plot(xx, i, c); }
		return; }
	for(i=x; i < xx; ++i) {
		update(i, y);
		update(i, yy); }
	for(i=y; i < yy; ++i) {
		update(x, i);
		update(xx, i); }
}

do_zoom()
{
	static unsigned zm = 3;
	unsigned i, m, mx, my, x, y, lx, ly, lw, lh;
	char d;
	m = magnify;
	mx = mxbase;
	my = mybase;
	magnify = mxbase = mybase = 0;
	update_bounds();
	refresh();
	mouse_status();
	lx = ly = 0;
	lw = lh = d = 1;
	lrg_puts(0, 103, BTEXT, "Select ZOOM - use +/- to change size");
	for(;;) {
		if(d) {
			if((x = (mousex + (320/zm))-1) >= XSIZE)
				x = XSIZE-1;
			if((y = (mousey + (102/zm))-1) >= YSIZE)
				y = YSIZE-1;
			plot_square(lx = mousex, ly = mousey, lw = x, lh = y, TEXT);
			d = 0; }
		i = mouse_status();
		if(i & MOUSE_RIGHT) {
			magnify = m;
			mxbase = mx;
			mybase = my;
			return; }
		if(i & MOUSE_LEFT) {
			magnify = zm-1;
			mxbase = mousex;
			mybase = mousey;
			return; }
		switch(key) {
		case '=' :
		case '+' : if(zm > 1) { --zm;	d = -1;	} break;
		case '_' :
		case '-' : if(zm < 10) { ++zm; d = -1; } }

		if((mousex != lx) ||  (mousey != ly) || d) {
			plot_square(lx, ly, lw, lh, 0);
			d = -1; } }
	refresh();
}
	
select_point(char *p)
{
	unsigned i;
	lrg_printf(0, 103, BTEXT, "%s - select position", p);
	do {
		i = mouse_status();
		if(i & MOUSE_RIGHT)
			return 0; }
	while(!(i & MOUSE_LEFT));
	return -1;
}

select_block(char *p)
{
	unsigned i, bx, by, lx, ly;

	lrg_printf(0, 103, BTEXT, "%s - starting point", p);
	do {
		i = mouse_status();
		if(i & MOUSE_RIGHT)
			return 0; }
	while(!(i & MOUSE_LEFT));

	bx = lx = mousex;
	by = ly = mousey;
	lrg_printf(0, 103, BTEXT, "%s - ending point  ", p);
	for(;;) {
		if((mousex != lx) || (mousey != ly)) {
			plot_square(bx, by, lx, ly, 0);
			plot_square(bx, by, lx = mousex, ly = mousey, TEXT); }
		i = mouse_status();
		if(i & MOUSE_LEFT) {
			plot_square(bx, by, lx, ly, 0);
			if(bx <= lx)
				{ xbs = bx; xbe = lx; }
			else
				{ xbs = lx; xbe = bx; }
			if(by <= ly)
				{ ybs = by; ybe = ly; }
			else
				{ ybs = ly; ybe = by; }
			return -1; }
		if(i & MOUSE_RIGHT) {
			plot_square(bx, by, lx, ly, 0);
			return 0; } }
}

draw_clip(unsigned x, unsigned y)
{
	unsigned i, j, z, c;
	for(j=z=0; j < clipy; ++j) {
		for(i=0; i < clipx; ++i) {
			if(c = clipboard[z++])
				plot(x+i, j+y, (c == select) ? MON : MGREY);
			else
				plot(x+i, j+y, MOFF); } }
}
erase_clip(unsigned x, unsigned y)
{
	unsigned i, j;
	for(j=0; j < clipy; ++j) {
		for(i=0; i < clipx; ++i) {
			update(x+i, y+j); } }
}

paste()
{
	unsigned i, x, y, lx, ly;
	char d;
	if(!(clipx && clipy)) {
		emsg = "Clipboard empty";
		return 0; }
	lrg_puts(0, 103, BTEXT, "PASTE - Select position");
	lx = ly = 0;
	d = -1;
	for(;;) {
		if(d) {
			draw_clip(lx = mousex, ly = mousey);
			d = 0; }
		i = mouse_status();
		if(i & MOUSE_RIGHT) {
			erase_clip(lx, ly);
			return 0; }
		if(i & MOUSE_LEFT) {
			for(i=ly = 0; ly < clipy; ++ly) {
				for(lx = 0; lx < clipx; ++lx) {
					if(((x = mousex + lx) < XSIZE) && ((y = mousey + ly) < YSIZE))
						set_pixel(x, y, clipboard[i]);
					++i; } }
			return -1; }
		if((mousex != lx) || (mousey != ly) || d) {
			erase_clip(lx, ly);
			d = -1; } }
}

/*
 * Draw a line from point (x1, y1) to (x2, y2)
 */
draw_line(int x1, int y1, int x2, int y2, int *func)
{
	int i, w, h;
	/* If 'X' is greater, increment through 'X' coordinate */
	if((w = abs(x1 - x2)) >= (h = abs(y1 - y2))) {
		if(x1 > x2) {
			i = x1;
			x1 = x2;
			x2 = i;
			i = y1;
			y1 = y2;
			y2 = i; }
		if(y1 < y2) {
			for(i=0; i < w; ++i)
				(*func)(x1+i, y1+lrg_scale(i, h, w)); }
		else {
			for(i=0; i < w; ++i)
				(*func)(x1+i, y1-lrg_scale(i, h, w)); } }
	/* If 'Y' is greater, increment through 'Y' coordinate */
	else {
		if(y1 > y2) {
			i = x1;
			x1 = x2;
			x2 = i;
			i = y1;
			y1 = y2;
			y2 = i; }
		if(x1 < x2) {
			for(i=0; i < h; ++i)
				(*func)(x1+lrg_scale(i, w, h), y1+i); }
		else {
			for(i=0; i < h; ++i)
				(*func)(x1-lrg_scale(i, w, h), y1+i); } }
	(*func)(x2, y2);
}

line()
{
	unsigned i, bx, by, lx, ly;
	char d;
	if(!select_point("LINE START"))
		return;
	bx = mousex;
	by = mousey;
	d = -1;
	lrg_puts(0, 103, BTEXT, "Select LINE ending position");
	for(;;) {
		if(d) {
			draw_line(bx, by, lx = mousex, ly = mousey, &plot_select);
			d = 0; }
		i = mouse_status();
		if(i & MOUSE_RIGHT)
			return;
		if(i & MOUSE_LEFT) {
			draw_line(bx, by, lx, ly, &set_select);
			return; }
		if((mousex != lx) || (mousey != ly) || d) {
			draw_line(bx, by, lx, ly, &update);
			d = -1; } }
}

dohelp(char *x[], unsigned offset, unsigned color)
{
	unsigned i, p;
	for(i = 0; p = x[i]; ++i)
		lrg_puts(0, (i*10)+offset, color, p);
}

input(char *p, char *d, unsigned len)
{
	unsigned i, l, x;
	unsigned c, cf;
reset:
	lrg_fbox(0, 103, 320, 10, BLACK);
	lrg_puts(0, 103, BTEXT, p);
	i = strlen(p) * 8;
	l = strlen(d);
	lrg_puts(i, 103, TEXT, d);
	cf = 0;
	for(;;) {
		if(peek(0x40, 0x6c) & 0x04) {
			x = (l * 8) + i;
			if(!cf) {
				lrg_fbox(x, 103, 10, 8, CURSOR);
				cf = -1; } }
		else {
			if(cf) {
				lrg_fbox(x, 103, 10, 8, BLACK);
				cf = 0; } }
		if(c = kbtst()) {
			if(cf) {
				lrg_fbox(x, 103, 10, 8, BLACK);
				cf = 0; }
			switch(c) {
			case 0 : continue;
			case 0x1B :
				return 0;
			case '\b' :
				if(l) {
					--l;
					lrg_putc((l*8)+i, 103, TEXT, ' '); }
				continue;
			case '\n' :
			case '\r' :
				d[l] = 0;
				return *d;
			case _HO :
			case _EN :
			case _INS :
			case _DEL :
				d[l=0] = 0;
				goto reset; }
			if((c & 0xFF00) || (l >= len)) {
				beep(1000, 100);
				continue; }
			lrg_putc((l*8)+i, 103, TEXT, c);
			d[l++] = c; } } 
}
print_segment()
{
	if(select) {
		lrg_printf(0, 123, TEXT, "Seg: %-5u [%04x %04x.%u]", select,
			((select+offset)-1), (select/8)+offset, select & 7); }
	else
		lrg_fbox(0, 123, 320, 10, BLACK);
	if(map)
		lrg_printf(0, 133, TEXT, "Map: %-5u %04x", map, map);
	else
		lrg_fbox(0, 133, 320, 10, BLACK);
}
clear_emsg()
{
	if(emsg)
		lrg_fbox(emsg = 0, 103, 320, 10, BLACK);
}
write_emsg()
{
	lrg_fbox(incstat = 0, 103, 320, 10, BLACK);
	if(emsg) {
		if(*emsg == '!')
			lrg_puts(0, 103, MTEXT, emsg+1);
		else {
			lrg_puts(0, 103, ETEXT, emsg);
			beep(1000, 100); } }
}

loadfile()
{
	unsigned i, j;
	if(openf(lcdname, "rb", ".LCD"))
		return;
	fget(smap+2, sizeof(smap)-2, fp);
	for(i=0; i < (XSIZE*YSIZE); ++i) {
		if(fget(&j, sizeof(j), fp) != sizeof(j)) {
			emsg = "Bad file format";
			fclose(fp);
			return; }
		pokew(xseg, i+i, j); }
	fclose(fp);
}

main(int argc, char *argv[])
{
	unsigned i, k, j, x, y;
	unsigned char *p;

	if(!(xseg = alloc_seg(4096)))
		abort("Not enough memory");
	i = 0; do {
		pokew(xseg, i, 0); }
	while(i += 2);
	emsg = "!LCD 1.0 - (C) 2002-2005 Dave Dunfield";

	for(i=1; i < argc; ++i) {
		p = argv[i];
		switch((toupper(*p++) << 8) | toupper(*p++)) {
		case 'C=' : grey = (GREY+3)-atoi(p);			continue;
		case 'B=' : bright = (j=atoi(p)*2) ?j-4 : 99;	continue;
		case 'O=' :	offset = atox(p); 					continue;
		case 'S=' : snap = atoi(p);						continue; }
		if(*lcdname)
			abort("Too many arguments");
		strcpy(lcdname, argv[i]);
		loadfile(); }

	if(!init_mouse())
		abort("Mouse required");
	if(lrg_open())
		abort("VGA required");

	adjust_brightness();
dorefresh:
	update_bounds();
dorefresh1:
	refresh();
	dohelp(help, 150, TEXT);
doclear:
	write_emsg();
	while(kbtst());
	for(;;) {
		i = mouse_status();
		if(i & MOUSE_RIGHT) {
			clear_emsg();
			map = smap[select = get_pixel(mousex, mousey)];
			goto reselect1; }
		if(i & MOUSE_LEFT) {
			clear_emsg();
			set_pixel(mousex, mousey, select); }
		if(key) {
			clear_emsg();
			switch(toupper(key)) {
			case '+' :
			case '=' :
				if(magnify < 9) {
					++magnify;
					goto dorefresh; }
				continue;
			case '-' :
			case '_' :
				if(magnify) {
					--magnify;
					goto dorefresh; }
				continue;
			case _LA :
				if(mxbase) {
					if((mxbase -= 10 - magnify) & 0x8000)
						mxbase = 0;
					goto dorefresh; }
				continue;
			case _RA :
				if(mxmax < (XSIZE-1)) {
					mxbase += 10 - magnify;
					goto dorefresh; }
				continue;
			case _UA :
				if(mybase) {
					if((mybase -= 10 - magnify) & 0x8000)
						mybase = 0;
					goto dorefresh; }
				continue;
			case _DA :
				if(mymax < (YSIZE-1)) {
					mybase += 10 - magnify;
					goto dorefresh; }
				continue;
			case _CPU :
				select += 9;
			case _PU :
				++select;
			reselect:
				if(select > 65530)
					select = MAX_SEG;
				if(select > MAX_SEG)
					select = 0;
			reselect1:
				map = smap[select];
			reselect2:
				print_segment();
				goto dorefresh1;
			case _CPD :
				select -= 9;
			case _PD :
				--select;
				goto reselect;
			case _CHO :
				map += 9;
			case _HO :
				++map;
			remap:
				if(map > 65530)
					map = MAX_SEG;
				if(map > MAX_SEG)
					map = 0;
				goto reselect2;
			case _CEN :
				map -= 9;
			case _EN :
				--map;
				goto remap;
			case 'F' :
				if(select_block("Fill")) {
					for(x=xbs; x <= xbe; ++x) {
						for(y=ybs; y <= ybe; ++y) {
							i = (y * XSIZE) + x;
							pokew(xseg, i+i, select); } } }
				goto dorefresh;
			case 'C' :
				if(select_block("Copy")) {
					x = (xbe - xbs)+1;
					y = (ybe - ybs)+1;
					if((x * y) > CLIPBD) {
						emsg = "Too large for clipboard";
						goto dorefresh; }
					i = 0;
					clipx = x;
					clipy = y;
					for(y=ybs; y <= ybe; ++y) {
						for(x = xbs; x <= xbe; ++x) {
							j = (y * XSIZE) + x;
							clipboard[i++] = peekw(xseg, j+j); } } }
				goto dorefresh1;
			case 'M' :
				lrg_puts(0, 103, BTEXT, "Map: C)lear F)ind H)ardware");
				for(;;) switch(kbget()) {
				case 'c' :
				case 'C' :
					memset(smap, 0, sizeof(smap));
					goto dorefresh;
				case 'f' :
				case 'F' :
					for(i=1; i < MAX_SEG; ++i) {
						if(smap[i] == map) {
							select = i;
							print_segment();
							goto dorefresh1; } }
					emsg = "Not found";
					goto doclear;
				case 'h' : 
				case 'H' :
					memset(smap, 0, sizeof(smap));
					for(x=0; x < (XSIZE*YSIZE); ++x) {
						if(i = peekw(xseg, x+x))
							smap[i] = i; }
					goto dorefresh;
				case 0x1B :
					goto doclear; }
			case 'P' :
				paste();
			case 'R' :	// Redraw
				goto dorefresh1;
			case 'S' :	// Snap
				sprintf(temp, "%u", snap);
				if(input("Snap?", temp, 2))
					snap = atoi(temp);
				goto doclear;
			case 'L' :
				line();
				goto dorefresh;
			case _F1 :	// Set base point
				x = xbase;
				y = ybase;
				xbase = ybase = 0;
				if(select_point("Base Reference")) {
					xbase = mousex;
					ybase = mousey;
					goto doclear; }
				xbase = x;
				ybase = y;
				goto doclear;
			case _F2 :	// Block-Address
				incstat = 0x81;
				while(select_block("Block Address")) {
					j = 0;
					for(y=ybs; y <= ybe; ++y) {
						for(x=xbs; x <= xbe; ++x) {
							if(get_pixel(x, y)) {
								set_pixel(x, y, select);
								j = -1; } } }
					if(!j) {
						emsg = "No segments found";
						goto doclear; }
					refresh();
					if(inc)
						++select;
					print_segment(); }
				goto dorefresh1;
			case _F3 :	// Shape-Address
				incstat = 0x81;
				while(select_point("Shape Address")) {
					if(block_select(clipboard))
						continue;
					clipx = clipy = 0;
					refresh();
					if(inc)
						++select;
					print_segment(); }
				goto dorefresh1;
			case _F4 :	// Block-Adjust
				incstat = 0x81;
				if(select_block("Adjust Block")) {
					j = -1; k=0;
					for(y=ybs; y <= ybe; ++y) {
						for(x=xbs; x <= xbe; ++x) {
							if(i = get_pixel(x, y)) {
								if(i < j)
									j=i;
								if(i > k)
									k = i; } } }
					if(j == -1) {
						emsg = "No segments found";
						goto doclear; }
					for(y=ybs; y <= ybe; ++y) {
						for(x=xbs; x <= xbe; ++x) {
							if(i = get_pixel(x, y))
								set_pixel(x, y, (i-j) + select); } }
					if(inc)
						select += ((k-j) + 1);
					print_segment(); }
				goto dorefresh1;
			case _F5 :	// Map
				incstat = 0x81;
				while(select_point("Map segment")) {
					if(!(select = get_pixel(mousex, mousey))) {
						beep(1000, 100);
						continue; }
					smap[select] = map;
					refresh();
					if(inc)
						++map;
					print_segment(); }
				goto dorefresh;
			case _F6 :	// Block-Map
				incstat = 0x81;
				k = map;
				while(select_block("Map Block")) {
					memset(clipboard, clipx = clipy = 0, sizeof(clipboard));
					for(y=ybs; y <= ybe; ++y) {
						for(x=xbs; x <= xbe; ++x) {
							if(i = get_pixel(x, y)) {
								if(!clipboard[select = i]) // Not mapped
									smap[i] = clipboard[i] = map++; } } }
					refresh();
					print_segment(); }
				if(!inc)
					map = k;
				print_segment();
				goto dorefresh1;
			case _CRA :	// Brightness
				adjust_brightness();
				continue;
			case _CLA :	// Contrast
				adjust_contrast();
				goto dorefresh1;
			case _F7 :	// Character Generation
				cgen(clipboard);
				print_segment();
				goto dorefresh;
			case _F8 :	// LCD Audit
				audit();
				print_segment();
				goto dorefresh;
			case _F9 :	// Emulate
				emulate();
				print_segment();
				goto dorefresh;
			case _DEL :
				do_zoom();
				goto dorefresh;
			case _INS :
				mxbase = mybase = 0;
				magnify = 0;
				goto dorefresh;
			case _F10 :
				lrg_puts(0, 103, BTEXT, "File:  L)load  S)ave  Q)uit");
				for(;;) switch(kbget()) {
				case 'l' :
				case 'L' :
					if(!input("Load file?", lcdname, sizeof(lcdname)-1))
						goto doclear;
					loadfile();
					goto dorefresh;
				case 's' :
				case 'S' :
#ifdef DEMO
					emsg = "Cannot S)ave in DEMO";
#else
					if(!input("Save file?", lcdname, sizeof(lcdname)-1))
						goto doclear;
					if(openf(lcdname, "wb", ".LCD"))
						goto doclear;
					fput(smap+2, sizeof(smap)-2, fp);
					for(i=0; i < (XSIZE*YSIZE); ++i) {
						j = peekw(xseg, i+i);
						fput(&j, sizeof(j), fp); }
					fclose(fp);
#endif
				case 0x1B: 
					goto doclear;
				case 'q' :
				case 'Q' :
					goto endit;
				default:
					beep(1000, 100); }
			default:
				if(isdigit(key)) {
					if(marky[key -= '0'])
						marky[key] = 0;
					else {
						markx[key] = mousex;
						marky[key] = mousey + 1; }
					goto dorefresh1; } } } }
endit:
	lrg_close();
}

block_select(unsigned char map[XSIZE][YSIZE])
{
	unsigned x, y, x1, y1, i, fc;
	unsigned char flag;
	static int offset_x[] = { 0, -1, 1, 0 };
	static int offset_y[] = { -1, 0, 0, 1 };

	if(!(fc = get_pixel(mousex, mousey))) {
		beep(1000, 100);
		return -1; }
	memset(map, 0, sizeof(map));
	map[mousex][mousey] = 1;
	do {
		flag = 0;
		for(y=0; y < YSIZE; ++y) {
			for(x=0; x < XSIZE; ++x) {
				if(map[x][y] == 1) {
					map[x][y] = 2;
					set_pixel(x, y, select);
					flag = -1;
					for(i=0; i < 4; ++i) {
						x1 = x + offset_x[i];
						y1 = y + offset_y[i];
						if((get_pixel(x1, y1) == fc) && !map[x1][y1])
							map[x1][y1] = 1;  } } } } }
	while(flag);
	return 0;
}
unsigned lx, ly, eco;
nl()
{
	lx = 0;
	if((ly += 10) >= 190)
		ly = 103;
	lrg_fbox(0, ly, 320, 10, BLACK);
}
pc(unsigned c)
{
	if((c == '\n') || (c == '\r')) {
		lx = 321;
		return; }
	if(c >= ' ') {
		if(lx > 310)
			nl();
		lrg_putc(lx, ly, eco, c);
		lx += 10; }
}
ps(char *s)
{
	while(*s)
		pc(*s++);
}

/*
 * Emulate LCD over serial port
 * Emulation commands:
 * 0xxxxxxx = Display ASCII character
 * 100xxxxx = Display Event indicator
 * 101hhhhh = Set HIGH bits of segment
 * 110lllll = Turn segment ON
 * 111lllll = Turn segment OFF
 */
emulate()
{
	unsigned i, x, y, z, c, h;
	char cc, buffer[100], state[MAX_SEG];
	static unsigned com=1, speed=_38400;

	for(;;) {
		lrg_printf(0, 103, MTEXT, "Emulate: COM%-5u Baud=%-5u", com, lrg_scale(57600, 2, speed));
		switch(kbget()) {
		case _DA : com = (com > 1) ? com-1 : 4;		continue;
		case _UA : com = (com < 4) ? com+1 : 1;		continue;
		case _LA : ++speed;							continue;
		case _RA : if(speed > 2) --speed;			continue;
		case 0x1B:
			return;
		case '\n' :
		case '\r' :
			goto emulate1; } }

emulate1:
	if(Copen(com, speed, PAR_NO|DATA_8|STOP_1, SET_RTS|SET_DTR|OUTPUT_2)) {
		emsg = "Cannot open COM port";
		return; }

	// Insure screen is 1:1
	magnify = mxbase = mybase = 0;
	update_bounds();
	refresh();
	mouse_status();
	update(mousex, mousey);

	// Create inverse map using Clipboard
	clipx = clipy = h = 0;
	memset(state, 0, sizeof(state));
	for(i=1; i < (MAX_SEG+1); ++i) {
		if(z = smap[i])
			clipboard[z-1] = i; }

	// Zero the text output
	lrg_fbox(0, 103, 320, 97, BLACK);
	lx = 0;
	ly = 103;
	x = 0; y = 0; z=0;
	eco = MTEXT;
	ps("Emulating LCD - ESC to exit\n");
resc:
	eco = TEXT;
	for(;;) {
		if((c=Ctestc()) != -1) {	// Character received
			switch(c & 0xE0) {
			case 0x80 :	// Signal event
				if(c == 0x80) {		// Clear Screen
					memset(state, 0x80, sizeof(state));
					continue; }
				sprintf(buffer, "[%u]", c & 0x1F);
				eco = ETEXT;
				ps(buffer);
				goto resc;
			case 0xA0 :	// Set High
				h = (c & 0x1F) << 5;
				continue;
			case 0xC0 :	// Set OFF
				if(i = clipboard[(c & 0x1F) | h])
					state[i-1] = 0x80;
				continue;
			case 0xE0 :	// Set ON
				if(i = clipboard[(c & 0x1F) |h])
					state[i-1] = 0x81;
				continue; }
			pc(c);
			continue; }
		for(i=0; i < 10; ++i) {
			if(c = peekw(xseg, z)) {
				cc = state[c-1];
				lrg_plot(x, y, (cc & 0x01) ? ON : grey); }
			z += 2;
			if(++x >= XSIZE) {
				x = 0;
				if(++y >= YSIZE)
					y = z = 0; } }
		switch(kbtst()) {
		case _CRA :
			adjust_brightness();
			continue;
		case _CLA :
			adjust_contrast();
			continue;
		case _INS :
		case _DEL :
			memset(state, 0, sizeof(state));
			continue;
		case _F1 :
			lrg_fbox(lx = 0, ly = 103, 320, 97, BLACK);
			continue;
		case 0x1B:	// Exit
			Cclose();
			lrg_fbox(0, 103, 320, 97, BLACK);
			print_segment();
			return; } }
}

cgen(char *temp)
{
	unsigned base, end, size, i, x, y, m, o, omx, omy, fs;
	unsigned char b, *cp;
	static unsigned char c;


	memset(clipboard, clipx = clipy = fs = 0, sizeof(clipboard));

	if(input("Load file?", cgname, sizeof(cgname)-1)) {
		if(openf(cgname, "rb", ".LCG"))
			return;
		fget(&fs, sizeof(size), fp);
		i = ((fs+7) / 8) * 256;
		if((fget(temp, i, fp) != i) || (getc(fp) != -1)) {
			fclose(fp);
			emsg = "Bad file format";
			return; }
		fclose(fp); }

	if(!select_block("CGEN"))
		return;

	omx = mousex; omy = mousey;

	base = -1;
	end = 0;
	for(y=ybs; y <= ybe; ++y) {
		for(x=xbs; x <= xbe; ++x) {
			if(i = get_pixel(x, y)) {
				i = smap[i];
				temp[10000+i] = -1;
				if(i < base)
					base=i;
				if(i > end)
					end = i; } } }
	if(!end) {
		emsg = "No segments found";
		return; }
	for(i=base; i <= end; ++i) {
		if(!temp[10000+i]) {
			emsg = "Segments not linear";
			return; } }

	size = (end - base) + 1;

	if(fs) {
		if(size != fs) {
			emsg = "File/Block size mismatch";
			return; } }

	if((!base) || !end) {
		emsg = "Segment not mapped";
		return; }

	if(size > 128) {
		emsg = "Block too large (max 128)";
		return; }

	lrg_puts(0, 103, MTEXT, "Character Generator");
	lrg_fbox(0, 123, 320, 77, BLACK);
	lrg_printf(0, 123, TEXT, "Base=%u End=%u Size=%u",
		base, end, size);

dorefresh:
	update_bounds();
dorefresh1:
	select = 0;
	refresh();

select_char:
	lrg_printf(0, 133, MTEXT, "Code %02x: %c", c, ((c < ' ') || (c > 0x7E)) ? ' ' : c);
	cp = (((size+7)/8) * c) + temp;
refresh:
	for(y=ybs; y <= ybe; ++y) {
		for(x=xbs; x <= xbe; ++x) {
			if(m = get_pixel(x, y)) {		// Segment
				if(m = smap[m]) {			// Mapped
					m -= base;
					o = m >> 3;
					b = 1 << (m & 7);
					plot(x, y, (cp[o] & b) ? ON : grey); } } } }
doclear:
	write_emsg();
	for(;;) {
		select = 0;
		if(x = get_pixel(mousex, mousey)) {
			if(m = smap[x]) {
				if((m >= base) && (m <= end)) {
					m -= base;
					o = m >> 3;
					b = 1 << (m & 7);
					if(cp[o] & b)
						select = x; } } }
		i = mouse_status();
		if((mousex != omx) || (mousey != omy)) {
			if(m = get_pixel(omx, omy)) {
				if(m = smap[m]) {
					if((m >= base) && (m <= end)) {
						m -= base;
						o = m >> 3;
						b = 1 << (m & 7);
						plot(omx, omy, (cp[o] & b) ? ON : grey); } } }
			omx = mousex; omy = mousey; }

		if(i & MOUSE_LEFT) {
			clear_emsg();
			if(!(m = get_pixel(mousex, mousey))) {
				emsg = "No segment";
				goto doclear; }
			if(!(m = smap[m])) {
				emsg = "Segment not mapped";
				goto doclear; }
			if((m < base) || (m > end)) {
				emsg = "Out of block";
				goto doclear; }
			m -= base;
			o = m >> 3;
			b = 1 << (m & 7);
			cp[o] ^= b;
			goto refresh; }
		if(i & MOUSE_RIGHT) {
			clear_emsg();
			lrg_puts(0, 103, BTEXT, "CGEN: S)ave Q)uit");
			for(;;) {
				if(mouse_status() & (MOUSE_LEFT|MOUSE_RIGHT))
					goto refresh;
				switch(key) {
				case 'q' :
				case 'Q' :
					return;
				case 's' :
				case 'S' :
#ifdef DEMO
					emsg = "Cannot S)ave in DEMO";
#else
					if(input("Save file?", cgname, sizeof(cgname)-1)) {
						if(openf(cgname, "wb", ".LCG"))
							goto doclear;
						fput(&size, sizeof(size), fp);
						i = ((size + 7) / 8) * 256;
						fput(temp, i, fp);
						fclose(fp); }
#endif
					goto refresh; } } }
		if(key) {
			clear_emsg();
			switch(key) {
			case _RA : ++c;	goto select_char;
			case _LA : --c; goto select_char;
			case _UA : c += 16;	goto select_char;
			case _DA : c -= 16; goto select_char;
			case _CRA :
				adjust_brightness();
				continue;
			case _CLA :	// Contrast
				adjust_contrast();
				goto dorefresh1;
			case _DEL :	// Zoom
				do_zoom();
				goto dorefresh;
			case _INS :
				mxbase = mybase = 0;
				magnify = 0;
				goto dorefresh; }
			if(key < 0x80) {
				c = key;
				goto select_char; } } }
}

adjust_brightness()
{
	unsigned i;
	unsigned char a[3];
	if((bright+=2) > 12)
		bright = 0;
	show_display();
	for(i=20; i < 30; ++i) {
		a[2] = a[1] = a[0] = i+bright;
		lrg_setpal(i, a); }
}
adjust_contrast()
{
	if((--grey < (GREY-4)))
		grey = GREY+2;
	show_display();
}
show_display()
{
	lrg_printf(249, 113, TEXT, " C:%u B:%u", 7-(grey-(GREY-4)), (bright/2)+1);
}

unsigned audit_max;
audit_error(unsigned y, unsigned *px, unsigned v)
{
	unsigned x;
	if((x=*px) < audit_max) {
		lrg_printf(x % 320, ((x / 320)*10)+y, ETEXT, "%u", v+1);
		*px += (8*5); }
}
#define	A_MD	10
#define	A_MS	50
#define	A_HS	90
#define	A_ME	4
#define	A_HE	11
audit()
{
	unsigned i, m, sc, hs, hm, hx, dx, sx, x, y, z;

	lrg_fbox(0, 0, 320, 200, BLACK);
	memset(clipboard, clipx = clipy = hs = hm = sc = 0, sizeof(clipboard));
	// Scan for segments unused
	for(y=z=0; y < YSIZE; ++y) {
		for(x=0; x < XSIZE; ++x) {
			if(m = peekw(xseg, z)) {
				if(m > hs)
					hs = m;
				if(!clipboard[1999+m])
					++sc;
				clipboard[1999+m] = -1; }
			z += 2; } }
	// Scan for maps
	for(i=1; i < MAX_SEG; ++i) {
		if(m = smap[i]) {
			if(m > hm)
				hm = m;
			++clipboard[m-1]; } }

	lrg_printf(0, 0, MTEXT, "High map:%u  High seg:%u  [%u used]", hm, hs, sc);
	lrg_puts(0, A_MD, TEXT, "Dup  map:");
	lrg_puts(0, A_MS, TEXT, "Skip map:");
	lrg_puts(0, A_HS, TEXT, "Skip seg:");
	hx = dx = sx = 80;
	audit_max = 320*A_ME;
	for(i=0; i < hm; ++i) {
		m = clipboard[i];
		if(clipboard[i] > 1)
			audit_error(A_MD, &dx, i);
		if(!clipboard[i])
			audit_error(A_MS, &sx, i); }
	if(dx == 80)
		lrg_puts(dx, A_MD, TEXT, "None");
	if(sx == 80)
		lrg_puts(sx, A_MS, TEXT, "None");
	if(dx >= audit_max)
		lrg_putc(310, A_MD+((A_ME-1)*10), ETEXT, '+');
	if(sx >= audit_max)
		lrg_putc(310, A_MS+((A_ME-1)*10), ETEXT, '+');
	audit_max = 320*A_HE;
	for(i=0; i < hs; ++i) {
		if(!clipboard[2000+i])
			audit_error(A_HS, &hx, i); }
	if(hx == 80)
		lrg_puts(hx, A_HS, TEXT, "None");
	if(hx >= audit_max)
		lrg_putc(310, A_HS+((A_HE-1)*10), ETEXT, '+');
	kbget();
	lrg_fbox(0, 0, 320, 200, BLACK);
}
