/*
 * Main module for the PC / RS-232 Line Error Simulator
 *
 * ?COPY.TXT 1990-2005 Dave Dunfield
 *  -- see COPY.TXT --.
 */
#include c:\mc\stdio.h
#include c:\mc\file.h
#include c:\mc\video.h

#define	MENROW	11
#define	MSGROW	20
#define	NUMERR	5
#define	NUMOUT	2166

char *main_menu[] = {
	"Begin Error Simulation",
	"Configure error types",
	"Configure serial ports",
	"Disk transfer functions",
	"Exit to DOS",
	0 };

char *serial_menu[] = {
	"Baudrate",
	"Data bits",
	"Parity",
	"Stop bits",
	0 };

char *error_menu[] = {
	"Error 1",
	"Error 2",
	"Error 3",
	"Error 4",
	"Error 5",
	0 };

char *disk_menu[] = {
	"Load configuration",
	"Save configuration",
	0 };

unsigned baudvalue[] = { 1040, 384, 96, 48, 24, 12, 6 };
char *btext[] =
	{ "110", "300", "1200", "2400", "4800", "9600", "19200", 0 };
char *dtext[] = { "Five", "Six", "Seven", "Eight", 0 };
char *ptext[] = { "Odd", "Even", "Mark", "Space", "None", 0 };
char *stext[] = { "One", "Two", 0 };
char *etext[] = { "Disabled", "Drop char", "Replace char", "Flip bits", 0};
char *atext[] = { "BOTH", "DTE", "DCE", 0 };

/* COMM port parameters */
int baud = 5, dbits = 3, parity = 4, sbits = 0;
unsigned com1 = 0x3F8, com2 = 0x2F8;

/* Induced error tables */
/* type, apply, freq, data, count */
int error_table[NUMERR][5] = 0 ;

char filename[51] = "CONFIG.LES";

char help[] = { "\n\
Use:	LES [opts]\n\n\
opts:	DTE=1-4		Set DTE port as COM1-COM4\n\
	DTE=5+		Set DTE port to address (hex)\n\
	DCE=1-4		Set DCE port as COM1-COM4\n\
	DCE=5+		Set DCE port to address (hex)\n" };

#define	CONFIGSIZ	109

/*
 * Get a COM port designation
 */
unsigned get_com(char *ptr)
{
	unsigned i;
	static unsigned addresses[] = {
		0x3F8, 0x2F8, 0x3E8, 0x2E8 };
	if(toupper(*ptr) == 'E')
		++ptr;
	switch(*ptr++) {
		case '!' : return 0;
		case '=' : }
	if((i = atox(ptr)) < 5) {
		if(!i)
			abort("Invalid COM port specification\n");
		return addresses[i-1]; }
	return i;
}

/*
 * Main program
 */
main(int argc, char *argv[])
{
	int i;
	char *ptr;

	for(i=1; i < argc; ++i) {
		ptr = argv[i];
		switch((toupper(*ptr++) << 8) | toupper(*ptr++)) {
			case 'DT' : com1 = get_com(ptr); break;
			case 'DC' : com2 = get_com(ptr); break;
			case '?'<<8:
			case '/?' :
			case '-?' :
			case '/H' :
			case '-H' :
				abort(help);
			default:
				printf("Unknown option: %s\n", argv[i]);
				abort(help); } }

	if(com1 == com2) {
		printf("DTE and DCE ports must be different!\n");
		abort(help); }

	if(test_uart(com1) || test_uart(com2))
		abort(help);

	vopen();
	vcursor_off();
	i = 0;

	vclscr();
	V_ATTR = REVERSE;
	vdraw_box(0, 0, 79, 2);
	vgotoxy(1, 1);
	vputf("", 21);
	vputf("RS-232 Line Error Simulator - Version 1.1", 57);
	V_ATTR = NORMAL;
	vdraw_box(0, 3, 79, 7);
	show_settings();

	for(;;) {
		message("Select function and press ENTER");
		if(vmenu(0, MENROW, main_menu, 0, &i))
			continue;
		switch(i) {
			case 0 :		/* Begin simulation */
				simulate();
				break;
			case 1 :		/* Setup error types */
				setup_error();
				break;
			case 2 :		/* Setup serial ports */
				setup_serial();
				break;
			case 3 :		/* Disk transfer functions */
				disk_transfer();
				break;
			case 4 :		/* Exit to DOS */
				vclscr();
				vdraw_box(0,0,79,5);
				vgotoxy(2, 1); vputs("The PC / RS-232 Line Error SImulator");
				vgotoxy(2, 3); vputs("?COPY.TXT 1990-2005 Dave Dunfield");
				vgotoxy(2, 4); vputs(" -- see COPY.TXT --.");
				vgotoxy(0,6);
				vcursor_line();
				fputs(help, stdout);
				exit(0); } }
}

/*
 * Set up the serial port settings
 */
setup_serial()
{
	int i;

	message("Select setup function (ESC to cancel)");
	i = 0;
	for(;;) {
		if(vmenu(25, MENROW, serial_menu, 0, &i))
			return;
		switch(i) {
			case 0 :	/* Baudrate */
				vmenu(36, MENROW, btext, -1, &baud);
				break;
			case 1 :	/* Data bits */
				vmenu(36, MENROW, dtext, -1, &dbits);
				break;
			case 2 :	/* Parity */
				vmenu(36, MENROW, ptext, -1, &parity);
				break;
			case 3 :	/* Stop Bits */
				vmenu(36, MENROW, stext, -1, &sbits); }
		show_settings(); }
}

/*
 * Setup error types
 */
setup_error()
{
	int i;

	message("Select setup function (ESC to cancel)");
	i = 0;
	for(;;) {
		if(vmenu(25, MENROW, error_menu, 0, &i))
			return;
		if(vmenu(34, MENROW, atext, -1, &error_table[i][1]))
			continue;
		if(vmenu(34, MENROW, etext, -1, &error_table[i][0]))
			continue;
		switch(error_table[i][0]) {
			case 0 :
				error_table[i][2] = 0;
				break;
			case 2 :		/* Replace char */
				get_num(34, MENROW, "Replace with (hex)? ", 16, 255, &error_table[i][4]);
				goto getfreq;
			case 3 :
				get_num(34, MENROW, "Bit toggle mask (hex)? ", 16, 255, &error_table[i][4]);
			case 1 :
			getfreq:
				get_num(34, MENROW, "Frequency (dec)? ", 10, -1, &error_table[i][2]); }
		show_settings(); }
}

/*
 * Get a number for the screen
 */
get_num(x, y, prompt, base, limit, var)
	unsigned x, y, base, limit, *var;
	char *prompt;
{
	unsigned h, i, p, value, c, c1;

	h = strlen(prompt)+7;
	vdraw_box(x, y, h, 2);
	vgotoxy(x+1, y+1);
	vputs(prompt);
	value = p = 0;
	vcursor_line();
	for(;;) {
		if(isdigit(c1 = c = vgetc()))
			c -= '0';
		else if(c >= 'a')
			c -= 'a' - 10;
		else if(c >= 'A')
			c -= 'A' - 10;
		else
			c = -1;
		if(c < base) {
			i = (value * base) + c;
			if((i <= limit) && (p < 5)) {
				++p;
				value = i;
				vputc(c1);
				continue; } }
		switch(c1) {
			case '\n' :
				*var = value;
			case 0x1B :
				vclear_box(x, y, h, 2);
				vcursor_off();
				return c1 == 0x1B;
			case _KBS :
				if(p) {
					--p;
					value /= base;
					vputs("\x08 \x08");
					break; }
			default:
				vputc(7); } }
}

/*
 * Show current settings
 */
show_settings()
{
	int i;
	char buffer[25];

	vgotoxy(2, 4);
	vprintf("Serial: %5s, %u, %5s, %u", btext[baud], dbits+5, ptext[parity], sbits+1);
	for(i=0; i < NUMERR; ++i) {
		vgotoxy(2, i+5);
		strcpy(buffer, etext[*error_table[i]]);
		switch(*error_table[i]) {
			case 2 :
				sprintf(buffer,"Replace char with %02x", (unsigned)error_table[i][4]);
				break;
			case 3 :
				sprintf(buffer,"Flip bits, Mask=%02x", (unsigned)error_table[i][4]); }
		vprintf("Error %u: %-4s %-20s, Frequency=%-5u", i+1, atext[error_table[i][1]],
			buffer, error_table[i][2]); }
}

/*
 * Test for the existance of a serial port
 */
test_uart(base)
	int base;
{
	char *ptr;
	if( (in(base+1) | in(base+4)) & 0x80) {
		ptr = "";
		switch(base) {
		case 0x3F8 : ptr = " (COM1)"; break;
		case 0x2F8 : ptr = " (COM2)"; break;
		case 0x3E8 : ptr = " (COM3)"; break;
		case 0x2E8 : ptr = " (COM4)"; }
		printf("No COM port found at %04x%s\n", base, ptr);
		return -1; }
	return 0;
}

/*
 * Update the uart with its com port settings
 */
update_uart(base)
	int base;
{
	int i;

	i = ((parity << 4) & 0x30) |	/* parity type */
		(dbits & 0x03) |			/* # data bits */
		((sbits << 2) & 0x04) |		/* # stop bits */
		((parity < 4) << 3);		/* parity enable */

	out(base+3, i | 0x80);
	out(base, baudvalue[baud]);
	out(base+1, baudvalue[baud] >> 8);
	out(base+3, i);
	out(base+1, 0);		/* disable all interrupts */
	out(base+4, 3);		/* Enable DTR and RTS */
}

/*
 * Display a message
 */
message(ptr)
	char *ptr;
{
	vgotoxy(0, MSGROW);
	vcleos();
	vmessage(38-strlen(ptr)/2, 20, ptr);
}

/*
 * Disk transfer
 */
disk_transfer()
{
	int i;
	FILE *fp;

	message("Select disk transfer function (ESC to cancel)");
	i = 0;
	for(;;) {
		if(vmenu(25, MENROW, disk_menu, 0, &i))
			return;
		if(vgets(5, MSGROW, "Filename?", filename, 50))
			continue;
		if(!(fp = open(filename, i ? F_WRITE : F_READ))) {
			message("Cannot open file");
			continue; }
		i ? write(&baud, CONFIGSIZ, fp) : read(&baud, CONFIGSIZ, fp);
		close(fp);
		show_settings(); }
}

/*
 * Perform error simulation
 */
simulate()
{
	int a, b;

	a = b = 0;
	update_uart(com1);
	update_uart(com2);

	vdraw_box(30, MENROW+1, 19, 4);
	vgotoxy(32, MENROW+2); vputs("DTE Count:");
	vgotoxy(32, MENROW+4); vputs("DCE Count:");
	message("Simulation in progress.. <ESCAPE> to exit");
	do {
		if(dosim(com1, com2, 1))
			outnum(++a, NUMOUT);
		if(dosim(com2, com1, 2))
			outnum(++b, NUMOUT+320); }
	while(vtstc() != 0x1B);
	vclear_box(30, MENROW+1, 19, 4);
}

/*
 * Main data-transfer routine
 */
dosim(local, dest, type) asm
{
		MOV		DX,8[BP]		; Get COM1 address
		ADD		DX,5			; Offset to status
		IN		AL,DX			; Read data
		TEST	AL,01h			; Character received?
		JZ		dosim8			; No, its ok
		SUB		DX,5			; Backup to data port
		IN		AL,DX			; Read data
		LEA		SI,_error_table	; Point to error table
		MOV		DI,5			; Count
; Scan table looking for errors
dosim1:	MOV		BX,[SI]			; Get "type"
		AND		BX,BX			; Enabled?
		JZ		dosim7			; No, keep going
		MOV		CX,2[SI]		; Get "apply"
		AND		CX,CX			; BOTH?
		JZ		dosim2			; Yes, its ok
		CMP		CX,4[BP]		; Is it us?
		JNZ		dosim7			; No, skip it
; Test frequency
dosim2:	MOV		CX,6[SI]		; Get value
		INC		CX				; Advance
		CMP		CX,4[SI]		; Are we there?
		JB		dosim6			; no, its ok
; Type 1 - drop character
		DEC		BX				; Type 1?
		JZ		dosim5			; Yes, ignore char
; Type 2 - replace character
		DEC		BX				; Type 2
		JNZ		dosim3			; No, try next
		MOV		AL,8[SI]		; Get replacement
		JMP	SHORT dosim4		; and proceed
; Type 3 - Flip bits
dosim3:	DEC		BX				; Type 3?
		JNZ		dosim7			; No, try next
		XOR		AL,8[SI]		; Toggle the bits
; Write the modified character
dosim4:	MOV		DX,6[BP]		; Get dest uart
		OUT		DX,AL			; Write the character
; Zero count
dosim5:	XOR		CX,CX			; Reset to zero
		MOV		6[SI],CX		; Reset count
		POP		BP				; Restore caller
		MOV		AX,-1			; Indicate activity
		RET
dosim6:	MOV		6[SI],CX		; Resave count
; Advance to next test and try it
dosim7:	ADD		SI,10			; Skip to next
		DEC		DI				; Reduce count
		JNZ		dosim1			; keep looking
		MOV		DX,6[BP]		; Get output port
		OUT		DX,AL			; Write the character
		POP		BP				; Restore caller
		MOV		AX,-1			; Set flag
		RET
dosim8:	XOR		AX,AX			; Indicate no activity
}

outnum(value, addr) asm
{
		EXTRN	_V_BASE:WORD
		MOV		AX,_V_BASE		; Get screen address
		MOV		ES,AX			; Set up ES
		MOV		AX,6[BP]		; Get value
		MOV		DI,4[BP]		; Get offset
		MOV		BL,5			; 5 digits
		MOV		CX,10			; Divide by 10
outn1:	XOR		DX,DX			; Zero high
		DIV		CX				; Perform divide
		PUSH	DX				; Save remainder
		DEC		BL				; Reduce count
		JNZ		outn1			; Keep going
		MOV		BL,5			; 5 digits
outn2:	POP		AX				; Get last digit
		ADD		AL,'0'			; Convert to ASCII
		MOV		ES:[DI],AL		; Write to display
		INC		DI				; Advance (attribute)
		INC		DI				; Advance (character)
		DEC		BL				; Reduce count
		JNZ		outn2			; And proceed
}
