#include <stdio.h>
#include <lrg.h>
#include <keys.h>
#include "tssound.c"

#define	BLACK	0
#define	BLUE	1
#define	GREEN	2
#define	CYAN	3
#define	RED		4
#define	VIOLET	5
#define	BROWN	6
#define	WHITE	7
#define	BGREEN	10
#define	YELLOW	14

// Parallel I/O bits in LPT port
#define	LPT_HOOK	0x0F	// Set for off-hook

// Macros to control LPT bits
#define	lpt_set(a)	out(lpt, (lpm |= (a)))
#define	lpt_clr(a)	out(lpt, (lpm &= ~(a)))

unsigned
	lpt,				// LPT port offset
	foff,				// File offset indicator
	vseg,				// Active voice segment
	voc_seg,			// Base voc segment for driver
	vsize,				// Size of voice block
	vsample,			// Voc sample rate
	offset,				// Display offset into voice block
	hscale,				// Horizontal display scale
	lbound = 0,			// Left bound value
	rbound = 0,			// right bound value
	sample_rate=40000,	// Record sample rate
	block_size=1000,	// Input block size
	cadon = 1000,		// Cadence ON time
	cadoff = 1000,		// Cadence OFF time
	trig_level=20,		// Trigger level
	sfreq = 1000,		// Sine freqency
	ssize = 10000,		// Sine sample size
	nvalue = 100;		// Normalization value
int
	vadjust = 128,		// Vertical adjustment
	vscale = 50;		// Vertical display scale
unsigned char
	header[20],			// static File header
	head1[4],			// Dynamic file header
	afile[40],			// Audio File name buffer
	cfile[40],			// Config file name buffer
	active_buffer,		// Currently active buffer
	cflag,				// Changed flag
	bflag,				// Bounder flag
	lpm,				// LPT port mirror
	hook,				// Hookswitch state
	smode = 1,			// Scale mode
	senable = -1,		// Sound card enable
	overlay,			// Overlay mode
	trig_sense = 0x77,	// Trigger sense(pos/neg)
	scaden,				// Sine cadence
	soverlay,			// Sine overlay mode
	arun,				// Auto-run mode
	temp,				// Temp flag
	vtype,				// Voc block type
	vformat;			// Voc format

unsigned
	Pdata[10],			// Play data
	Rdata[10];			// Record data

struct Input {
	char type;				// Type of data (0=Fill, 1=Num, 2=N/Y, 3=Char)
	char *prompt;			// Prompt to display
	unsigned *address;		// Address to store
	unsigned min;			// Minimum value
	unsigned max;			// Maximum value
	} input_table[] = {
	char 1, int "Trig level =%5u", &trig_level, 0, 127,					// 0
	char 2, int "Trig edge  =%5s", &trig_sense, "-", "+",				// 1
	char 1, int "#samples   =%5u", &block_size, MIN_VOC_BLOCK, 65000,	// 2
	char 1, int "Max value  =%5u", &nvalue, 1, 127,						// 3
	char 1, int "Frequency  =%5u", &sfreq, 50, 22000,					// 4
	char 1, int "#samples   =%5u", &ssize, 10, 65000,					// 5
	char 2, int "Cadence    =%5s", &scaden, "No", "Yes",				// 6
	char 1, int "On  Samples=%5u", &cadon, 1, 65000,					// 7
	char 1, int "Off samples=%5u", &cadoff, 1, 65000,					// 8
	char 1, int "Sample rate=%5u", &sample_rate, 1000, 44000,			// 9
	char 2, int "Overlay    =%5s", &soverlay, "No", "Yes",				// 10
	char 2, int "Exit LSCOPE=%5s", &temp, "No", "Yes",					// 11
	char 2, int "Auto run   =%5s", &arun, "No", "Yes",					// 12
	char 3, int "?%38s", &afile, 0x100, 38,								// 13
	char 3, int "?%38s", &cfile, 0x100, 38,								// 14
	char 0, int "Load audio file:", 0, 0, 0,							// 15
	char 0, int "Save audio file:", 0, 0, 0,							// 16
	char 0, int "Load config file:", 0, 0, 0,							// 17
	char 0, int "Save config file:", 0, 0, 0,							// 18
	char 0, int "Cadence:", 0, 0, 0,									// 19
	char 0, int "", 0, 0, 0,											// 20
	char 0, int "Auto-Run >.1 sec", 0, 0, 0,							// 21
	char 0, int "=poor KB response", 0, 0, 0,							// 22
	char 2, int "Proceed? %3s", &temp, "No",  "Yes",					// 23
	0 };
#define	IWIDTH			17
#define	Itrig_level		0
#define	Itrig_sense		1
#define	Iblock_size		2
#define	Invalue			3
#define	Isfreq			4
#define	Issize			5
#define	Iscaden			6
#define	Icadon			7
#define	Icadoff			8
#define	Isample_rate	9
#define Isoverlay		10
#define	Iexit			11
#define	Iarun			12
#define	Iafile			13
#define	Icfile			14
#define	IPload_audio	15
#define	IPsave_audio	16
#define	IPload_config	17
#define	IPsave_config	18
#define	IPcadence		19
#define	IPnull			20
#define	IPwarn1			21
#define	IPwarn2			22
#define	IPproceed		23

char *gkeys[] = {
//....-....1....-....2....-....3....-...
" ------- L I N E S C O P E  1.0 -------",
"",
"Left/Right   = Move view left/right",
"PgUp/PgDn    = Page view left/right",
"Home/end     = Move to beginning/end",
"Up/Down      = Set vertical gain",
"^left/^right = Set horizontal scale",
"^Home        = Set hscale=1, vscale=50",
"Tab          = Align to hscale",
"0-9,*,#,A-D  = Dial DTMF digit",
"+            = Offset +",
"-            = Offset -",
"Space        = Toggle function keys",
"",
//....-....1....-....2....-....3....-...
"E = Entire/Bounded M = Mixer Control",
"P = Play audio     O = Overlay",
"R = Record audio   S = Scale mode",
"H = Hookswitch     W = DTMF Waveform",
"Q = Quit",
"Press SPACE to proceed...",
0 };

/*
 * -- Sound Card Mixer interfacing functions --
 */

#define	MIX_MASTER		0			/* Master volume */
#define	MIX_VOICE		1			/* Voice level */
#define	MIX_MIDI		2			/* MIDI level */
#define	MIX_CD			3			/* CD level */
#define	MIX_LINE		4			/* Line level */
#define	MIX_MIKE		5			/* Mike volume */
#define	MIX_INPUT		6			/* Input select */

char *mixnames[] = {
	"MASTER",
	"VOICE",
	"MIDI",
	"CDAUDIO",
	"LINE",
	"MIKE",
	"INPUT",
	0 };

char *inpnames[] = {
	"LINE",
	"MIKE",
	"CD",
	"MIDI",
	0 };

unsigned mlevels[7][2] = {
	31, 31,				// Master left/right
	0, 20,				// Voice left/right
	0, 0,				// Midi left/right
	0, 0,				// CD	left/right
	31, 0,				// line left/right
	0,	0,				// Microphone
	0, 0 };				// Input select

/*
 * Set the (L,R) output levels for a channel
 * Level goes 0-31 (5 bits)
 */
int mix_level(unsigned channel, unsigned l, unsigned r)
{
	unsigned a, v1, v2;
//	if((l|r) > 31)
//		return 13;
	switch(sct) {
	case 2 :		// SB-PRO
	case 4 :		// SP-PRO2
		v1 = ((l << 3) & 0xE0) | ((r >> 1) & 0x0E);
		switch(channel) {
		default: return 13;
		case MIX_MASTER : a = 0x22;	break;
		case MIX_VOICE	: a = 0x04;	break;
		case MIX_MIDI	: a = 0x26;	break;
		case MIX_CD		: a = 0x28; break;
		case MIX_LINE	: a = 0x2E; break;
		case MIX_MIKE	: a = 0x0A; v1 >>= 1; }
		mix_write(a, v1);
		return 0;
	case 3 :		// SB2
		v1 = (l >> 1) & 0x0E;
		switch(channel) {
		default: return 13;
		case MIX_MASTER : a = 0x00;	break;
		case MIX_MIDI	: a = 0x05;	break;
		case MIX_CD		: a = 0x08; break;
		case MIX_VOICE	:
			v1 = (l >> 2) & 0x06;
			a = 0x0A; }
		mix_write(a, v1);
		return 0;
	case 6 :		// SB-16
		v1 = l << 3;
		v2 = r << 3;
		switch(channel) {
		default: return 13;
		case MIX_MIKE	:
			mix_write(0x3A, v1);
			return 0;
		case MIX_MASTER	: a = 0x30;	break;
		case MIX_VOICE	: a = 0x32;	break;
		case MIX_MIDI	: a = 0x34;	break;
		case MIX_CD		: a = 0x36;	break;
		case MIX_LINE	: a = 0x38; }
		mix_write(a,	v1);	// Set left channel
		mix_write(a+1,	v2);	// Set right channel
		return 0; }
	return 14;
}

update_mix()
{
	static unsigned char mx4in[] = { 0x06, 0x00, 0x02, 0x04 };
	static unsigned char mx6inl[] = { 0x10, 0x01, 0x04, 0x40 };
	static unsigned char mx6inr[] = { 0x08, 0x01, 0x02, 0x20 };
//	mix_level(MIX_LINE, 0, 0);
	mix_level(MIX_MASTER, mlevels[MIX_MASTER][0], mlevels[MIX_MASTER][1]);
	mix_level(MIX_VOICE, mlevels[MIX_VOICE][0], mlevels[MIX_VOICE][1]);
	mix_level(MIX_MIDI, mlevels[MIX_MIDI][0], mlevels[MIX_MIDI][1]);
	mix_level(MIX_CD, mlevels[MIX_CD][0], mlevels[MIX_CD][1]);
	mix_level(MIX_LINE, mlevels[MIX_LINE][0], mlevels[MIX_LINE][1]);
	mix_level(MIX_MIKE, mlevels[MIX_MIKE][0], mlevels[MIX_MIKE][1]);
//	mix_write(0x0C, 0x03);
	mix_write(0x0C, mx4in[mlevels[MIX_INPUT][0]]);
	switch(sct) {
//	case 4 :
//		break;
	case 6 :
		mix_write(0x3D, mx6inl[mlevels[MIX_INPUT][0]]);
		mix_write(0x3E, mx6inr[mlevels[MIX_INPUT][1]]);
//		mix_write(0x3D, 0x1F);
//		mix_write(0x3E, 0x00);
		mix_write(0x3F, 0xE0); }
}

unsigned sample()
{
#ifdef DEMO
	static int x = 0x80;
	int a, b;
	unsigned r;
	r = rand();
	b = peekw(0x40, 0x6C);
	a = r > 30000;
	b = (b & 7) + 1;
	b -= (r%b);
	if(x >= 0x80) {
		if(a)
			x -= b;
		else
			x += b; }
	else {
		if(a)
			x += b;
		else
			x -= b; }
	if(x < 10)
		x = 10;
	if(x > 245)
		x = 245;
	return x;
#else
	while(in(scb+0x0C) & 0x80);		// Wait till ready
	out(scb+0x0C, 0x20);
	while(!(in(scb+0x0E) & 0x80));	// Wait for data
	return in(scb+0x0A);
#endif
}

edit_mix()
{
	unsigned i, y;
	int l, r;
	unsigned char *p;
	static unsigned select;
	static unsigned char *prompts[] = {
		"Up/Down    = Select Channel",
		"Left/right = Adj. both channels",
		"Home/End   = Adj. left  channel only",
		"PgUp/PgDn  = Adj. right channel only",
		"Ins/Del    = Max/Min",
		"Enter      = Exit",
		0 };

	lrg_fbox(0, 0, 320, 200, BLACK);
	lrg_fbox(55,  8, 200, 16, WHITE);
	lrg_puts(60, 12, (WHITE<<8)|RED, "Sound Card Mixer Control");
	for(i=0; p = prompts[i]; ++i)
		lrg_puts(10, (i*12) + 125, WHITE, p);

redraw:
	i = 0;
	for(;;) {
		lrg_printf(20, y = (i*12)+40, (i == select) ? WHITE : GREEN,
			"%-10s", mixnames[i]);
		if(i >= 6) {
			lrg_printf(85, y, BROWN, "%-10s", p = inpnames[mlevels[MIX_INPUT][0]]);
			break; }
		lrg_fbox( 85, y, 31*3, 8, BROWN);
		lrg_fbox( 85, y, mlevels[i][0] * 3, 8, WHITE);
		if(i < 5) {
			lrg_fbox(185, y, 31*3, 8, BROWN);
			lrg_fbox(185, y, mlevels[i][1] * 3, 8, WHITE); }
		++i; }
	l = mlevels[select][0];
	r = mlevels[select][1];
	switch(select) {
	case 5 :
		lrg_printf(195, (5*12)+41, BLUE, "   V:%2u   ", l);
		break;
	case 6 :
		lrg_printf(195, (5*12)+41, BLUE, "%-11s", p);
		break;
	default:
		lrg_printf(195, (5*12)+41, BLUE, "L:%2u  R:%2u", l, r); }

	while(!(i = kbtst())) {
		if((y = sample()) < 0x80)
			y = 0x80-y;
		else
			y -= 128;
		y *= 2;
		lrg_retrace();
		lrg_fbox(20, 30, y, 3, WHITE);
		lrg_fbox(y+20, 30, 256-y, 3, BROWN); }
	switch(i) {
		case '\r':
		case 0x1B:	update_mix();					return;
		case _UA :	select = select ? select-1 : 6;	goto redraw;
		case _DA :	if(++select > 6) select=0;		goto redraw;
		case _RA :	r = ++l;						goto setlev;
		case _LA :	r = --l;						goto setlev;
		case _DEL:	r = l = 0;						goto setlev;
		case _INS: 	r = l = 31;						goto setlev;
		case _HO :	++l;							goto setlev;
		case _EN :	--l;							goto setlev;
		case _PU :	++r;							goto setlev;
		case _PD :	--r;
		setlev:
			if(l < 0) l = 0;
			if(r < 0) r = 0;
			if(select == 6) {
				if(l > 3)
					l = 3;
				r = l; }
			mlevels[select][0] = (l > 31) ? 31 : l;
			mlevels[select][1] = (r > 31) ? 31 : r;
			update_mix();
			goto redraw; }
	goto redraw;
}

/*
 * -- SineWave generation and tone/waveform editing functions --
 */

#define	LRGW	250		// Width of graphic display
#define	LRGH	128		// Height of graphic display
#define	LRGX	1		// X offset for graphic display
#define	LRGY	13		// Y offset for graphic display
#define	GAP		20		// Minumum quadrant size

unsigned
	q1 = 250,		// Quadrant 1 end, Quadrant 2 start
	q2 = 500,		// Quadrant 2 end, Quadrant 3 start
	q3 = 750,		// Quadrant 3 end, Quadrant 4 start
	select;			// Current selected edit quadrant
unsigned char
	amplitude[2] = { 50, 50 },			// Q1/2, Q3/4 amplitude
	slope[4] = { 100, 100, 100, 100 };		// Quadrant slope

/*
 * Quadrant sine table - this is 1/4 of a complete sine wave
 */
char qst[] = {
    0,    0,    1,    2,    3,    3,    4,    5,
    6,    7,    7,    8,    9,   10,   11,   11,
   12,   13,   14,   15,   15,   16,   17,   18,
   19,   19,   20,   21,   22,   23,   23,   24,
   25,   26,   26,   27,   28,   29,   30,   30,
   31,   32,   33,   33,   34,   35,   36,   36,
   37,   38,   39,   40,   40,   41,   42,   43,
   43,   44,   45,   46,   46,   47,   48,   48,
   49,   50,   51,   51,   52,   53,   54,   54,
   55,   56,   56,   57,   58,   59,   59,   60,
   61,   61,   62,   63,   63,   64,   65,   66,
   66,   67,   68,   68,   69,   70,   70,   71,
   72,   72,   73,   74,   74,   75,   75,   76,
   77,   77,   78,   79,   79,   80,   80,   81,
   82,   82,   83,   83,   84,   85,   85,   86,
   86,   87,   88,   88,   89,   89,   90,   90,
   91,   92,   92,   93,   93,   94,   94,   95,
   95,   96,   96,   97,   97,   98,   98,   99,
   99,  100,  100,  101,  101,  102,  102,  103,
  103,  104,  104,  105,  105,  105,  106,  106,
  107,  107,  108,  108,  108,  109,  109,  110,
  110,  110,  111,  111,  112,  112,  112,  113,
  113,  113,  114,  114,  114,  115,  115,  115,
  116,  116,  116,  117,  117,  117,  118,  118,
  118,  118,  119,  119,  119,  120,  120,  120,
  120,  121,  121,  121,  121,  121,  122,  122,
  122,  122,  123,  123,  123,  123,  123,  123,
  124,  124,  124,  124,  124,  124,  125,  125,
  125,  125,  125,  125,  125,  125,  125,  126,
  126,  126,  126,  126,  126,  126,  126,  126,
  126,  126,  126,  126,  126,  126,  126,  126,
  126,  126};
char
	st[sizeof(qst)*4],	// Active waveform
	dst[sizeof(qst)*4];	// DTMF waveform

/*
 * DTMF tone pairs
 */
unsigned dtmf_tones[16][2] = {
	697, 1209,		// 1
	697, 1336,		// 2
	697, 1477,		// 3
	770, 1209,		// 4
	770, 1336,		// 5
	770, 1477,		// 6
	852, 1209,		// 7
	852, 1336,		// 8
	852, 1477,		// 9
	941, 1336,		// 0
	941, 1209,		// *
	941, 1477,		// #
	697, 1633,		// A
	770, 1633,		// B
	852, 1633,		// C
	941, 1633 };	// D
// unsigned char dtmf_digits[] = { "1234567890*#ABCD" };


/*
 * Build a quadrant consisting of 1/4 sine wave
 *	x1,x2	= starting,ending x position
 *	a		= amplitude (0-100)
 *	s		= slope (0-100)
 *	d		= Direction (0=Up, 1=Down)
 */
void build_quad(unsigned x1, unsigned x2, unsigned a, unsigned s, char d)
{
	unsigned i, dx;
	char x, y, xst[sizeof(qst)];

	dx = x2 - x1;

	for(i=0; i < sizeof(qst); ++i) {
		x = qst[i];						// Size component
		x = (x * s) / 100;
		y = (i * 128) / sizeof(qst);	// Flat component
		y = ((100-s) * y) / 100;
		xst[i] = ((x+y) * a) / 100; }

	switch(d & 3) {
	case 0 :
		for(i=0; i < dx; ++i)
			st[x1+i] = xst[lrg_scale(i, 250, dx)];
		return;
	case 1 :
		for(i=0; i < dx; ++i)
			st[x1+i] = xst[(sizeof(xst)-1) - lrg_scale(i, 250, dx)];
		return;
	case 2 :
		for(i=0; i < dx; ++i)
			st[x1+i] = -xst[lrg_scale(i, 250, dx)];
		return;
	case 3 :
		for(i=0; i < dx; ++i)
			st[x1+i] = -xst[(sizeof(xst)-1) - lrg_scale(i, 250, dx)]; }
}

/*
 * Build the entire sine wave by generating each of the 4 quadrants
 */
void build_sine()
{
	build_quad(0, q1, amplitude[0], slope[0], 0);
	build_quad(q1, q2, amplitude[0], slope[1], 1);
	build_quad(q2, q3, amplitude[1], slope[2], 2);
	build_quad(q3, 1000, amplitude[1], slope[3], 3);
}

/*
 * Display graphical representation of generated sine waveform
 */
void display_sine()
{
	unsigned i;
	int j, l;
	unsigned char c;

	c = (select == 0) ? 4 : 6;
	for(i=0; i < LRGW; ++i) {
		j = st[l = i*4] / 2;
		if(l > q1) {
			c = (select == 1) ? 4 : 6;
			if(l > q2) {
				c = (select == 2) ? 4 : 6;
				if(l > q3)
					c = (select == 3) ? 4 : 6; } }
		lrg_vline(i+LRGX, LRGY, LRGH-1, c);
		lrg_plot(i+LRGX, (64 - j) + (LRGY-1), 7); }
}

/*
 * Edit the sine waveform
 */
int edit_sine()
{
	cflag = 0;

	lrg_fbox(0, 0, 320, 200, BLACK);

	lrg_printf(20, 0, 7, "*** SINE WAVEFORM EDITOR ***");

	// Draw box
	lrg_box(LRGX-1, LRGY-1, LRGW+1, LRGH, 1);

	// Draw permanent prompts
//	lrg_printf(270, 10+LRGY, 7, w ? "High" : "Low");
	lrg_printf(270, 20+LRGY, 7, "Tone");
	lrg_printf(0, 154+LRGY, 7, "amp-U/D Sine/Flat Left/Right Normal");
	lrg_printf(0, 167+LRGY, 7, "SPACE=quadrant ENTER=exit/save ESC=quit");

rebuild:
	// Build the sinewave and display the quadrant markers
	build_sine();
	lrg_fbox(0, 129+LRGY, 330, 16, 0);	// Clear old
	lrg_printf(0, 130+LRGY, 7, "0");
	lrg_printf((q1-10)>>2, 138+LRGY, 7, ".%u", q1);
	lrg_printf((q2-10)>>2, 130+LRGY, 7, ".%u", q2);
	lrg_printf((q3-10)>>2, 138+LRGY, 7, ".%u", q3);
	lrg_printf(245, 130+LRGY, 7, "1");

redraw:
	// Update the quadrant dynamic displays and prompt for cmmand
	lrg_printf(270, 50+LRGY, 7, "A:%3u", amplitude[select/2]);
	lrg_printf(270, 70+LRGY, 7, "S:%3u", slope[select]);
	for(;;) {
		display_sine(st);
		switch(toupper(kbget())) {
		case ' ' :			// Select quadrant
			select = (select + 1) & 3;
			goto redraw;
		case _UA :			// Increase amplitude
			if(amplitude[select/2] < 100)
				++amplitude[select/2];
		chg:cflag = -1;
			goto rebuild;
		case _DA :			// Decrease amplitude
			if(amplitude[select/2])
				--amplitude[select/2];
			goto chg;
		case 'S' :			// Move toward "sine" waveform
			if(slope[select] < 100)
				++slope[select];
			goto chg;
		case 'F' :			// Move toward "flat" waveform
			if(slope[select])
				--slope[select];
			goto chg;
		case _RA :			// Adjust quadrant end to the right
			switch(select) {
			case 0 :
				if(q1 < (q2-GAP))
					++q1;
				goto chg;
			case 1 :
				if(q2 < (q3 - GAP))
					++q2;
				goto chg;
			case 2 :
				if(q3 < (1000-GAP))
					++q3;
				goto chg; }
			continue;
		case _LA :			// Adjust quadrant end to the left
			switch(select) {
			case 0 :
				if(q1 > GAP)
					--q1;
				goto chg;
			case 1 :
				if(q2 > (q1+GAP))
					--q2;
				goto chg;
			case 2 :
				if(q3 > (q2+GAP))
					--q3;
				goto chg; }
			continue;
		case 'N' :			// Return to "normal" waveform
			amplitude[0] = amplitude[1] = 50;
			slope[0] = slope[1] = slope[2] = slope[3] = 100;
			q1 = 250; q2 =  500; q3 = 750;
			goto chg;
		case '\r' :
			return -1;
		case 0x1B:
			return 0; } }
}

/*
 * Build a DTMF tone-pair
 */
void build_dtmf(unsigned f, unsigned duration)
{
#ifdef DEMO
	beep((dtmf_tones[f][0]+dtmf_tones[f][1])/2, 200);
#else
	int z1, z2;
	unsigned i, o1, o2, f1, f2, s, vo;
	f1 = dtmf_tones[f][0];
	f2 = dtmf_tones[f][1];

	poke(voc_seg, 0, 1);	// Voice data record
	pokew(voc_seg, 1, duration+2);
	poke(voc_seg, 3, 0);	// Zero high byte of length
	poke(voc_seg, 4, 256 - lrg_scale(1000, 1000, sample_rate));
	poke(voc_seg, 5, 0);	// 8-bit PCM
	vo = 6;
	s = sample_rate / 1000;
	for(i=o1=o2=0; i < duration; ++i) {
		z1 = dst[o1/s] / 2;
		z2 = dst[o2/s] / 2;
		poke(voc_seg, vo++, (z1 + z2) ^ 0x80);
		if((o1 += f1) >= sample_rate)
			o1 -= sample_rate;
		if((o2 += f2) >= sample_rate)
			o2 -= sample_rate; }
	while(vo)
		poke(voc_seg, vo++, 0);
#endif
}

/*
 * Build a single tone sample
 */
void build_tone(unsigned f, unsigned duration, char overlay, unsigned on, unsigned off)
{
	int z, z1;
	unsigned i, o, s, v, t;

	v = t = z = 0;
	s = (vsample = sample_rate) / 1000;
	for(i=o=0; i < duration; ++i) {
		z = st[o/s];
		if(t >= on) {
			if((z & 0x80) ^ (z1 & 0x80)) {
				z=0;
				goto skip; } }
		z1 = z;
		if((o += f) >= sample_rate)
			o -= sample_rate;
	skip:
		if(t++ >= off)
			t = o = 0;
		if(overlay && (v < vsize)) {
			z += (peek(vseg, v) - 0x80);
			if(z < -128)
				z = -128;
			else if(z > 127)
				z = 127; }
		poke(vseg, v++, z ^ 0x80); }
	if(overlay) {
		if(v < vsize)
			return; }
	poke(vseg, vsize = v, 0);
}


register message(unsigned args)
{
	char buffer[51];
	unsigned l, x;

	x = 160 - (l = _format_(nargs() * 2 + &args, buffer) * 8) / 2;
	lrg_fbox(x-5, 84, l+7, 20, WHITE);
	lrg_puts(x, 90, (WHITE<<8)|RED, buffer);
	lrg_puts(70, 200-10, (WHITE<<8)|RED, " Press ENTER to proceed ");
	while(kbget() != '\r');
}

test_sound()
{
	if(!senable) {
		message("Sound Card Disabled");
		return -1; }
#ifdef DEMO
	lrg_puts(0, 0, (WHITE<<8)|RED, "**SOUND FUNCTION SIMULATED FOR DEMO**");
#else
	while(voc_status);
#endif
	return 0;
}

#ifndef DEMO
FILE *fileopen(char *n, char *e, char *a)
{
	char *p, f;
	f = 0;
	p = n;
	for(;;) {
		switch(*p++) {
		case 0 :
			if(!f)
				strcpy(p-1, e);
			return fopen(n, a);
		case '.' :	f = -1;	break;
		case ':' :
		case '\\':	f = 0; } }
}
write_config(char *f, char *e)
{
	FILE *fp;
	if(!(fp = fileopen(f, e, "wb"))) {
		message("Unable to WRITE: %s", f);
		return; }
	fput(&sample_rate, sizeof(sample_rate), fp);
	fput(&block_size, sizeof(block_size), fp);
	fput(&trig_level, sizeof(trig_level), fp);
	fput(&trig_sense, sizeof(trig_sense), fp);
	fput(&scaden, sizeof(scaden), fp);
	fput(&cadon, sizeof(cadon), fp);
	fput(&cadoff, sizeof(cadoff), fp);
	fput(&hscale, sizeof(hscale), fp);
	fput(&vscale, sizeof(vscale), fp);
	fput(&sfreq, sizeof(sfreq), fp);
	fput(&ssize, sizeof(ssize), fp);
	fput(&nvalue, sizeof(nvalue), fp);
	fput(&q1, sizeof(q1), fp);
	fput(&q2, sizeof(q2), fp);
	fput(&q3, sizeof(q3), fp);
	fput(amplitude, sizeof(amplitude), fp);
	fput(slope, sizeof(slope), fp);
	fclose(fp);
}

load_config(char *f, char *e)
{
	FILE *fp;
	if(!(fp = fileopen(f, e, "rb"))) {
		message("Unable to READ: %s", f);
		return; }
	fget(&sample_rate, sizeof(sample_rate), fp);
	fget(&block_size, sizeof(block_size), fp);
	fget(&trig_level, sizeof(trig_level), fp);
	fget(&trig_sense, sizeof(trig_sense), fp);
	fget(&scaden, sizeof(scaden), fp);
	fget(&cadon, sizeof(cadon), fp);
	fget(&cadoff, sizeof(cadoff), fp);
	fget(&hscale, sizeof(hscale), fp);
	fget(&vscale, sizeof(vscale), fp);
	fget(&sfreq, sizeof(sfreq), fp);
	fget(&ssize, sizeof(ssize), fp);
	fget(&nvalue, sizeof(nvalue), fp);
	fget(&q1, sizeof(q1), fp);
	fget(&q2, sizeof(q2), fp);
	fget(&q3, sizeof(q3), fp);
	fget(amplitude, sizeof(amplitude), fp);
	fget(slope, sizeof(slope), fp);
	fclose(fp);
}

/*
 * Load file into memory
 */
load_file(char *f, char *e)
{
	int c;
	unsigned s;
	FILE *fp;

	if(!(fp = fileopen(f, e, "rb"))) {
		message("Unable to READ: %s", f);
		return; }
	fget(header, sizeof(header), fp);
	fget(&foff, sizeof(foff), fp);
	if(foff != 0x1A) {
		fclose(fp);
		message("VOC offset %04x wrong", foff);
		return; }
	fget(head1, sizeof(head1), fp);
	vtype = getc(fp);
	fget(&vsize, sizeof(vsize), fp);
	if(getc(fp) || (vsize > 65529)) {
		fclose(fp);
		message("VOC record too big!");
		return; }
	vsample = lrg_scale(1000, 1000, 256-getc(fp));
	vformat = getc(fp);
	vsize -= 2;
	for(s = 0; s < vsize; ++s) {
		if((c = getc(fp)) & 0xFF00) {
			fclose(fp);
			message("Premature end of file!");
			return; }
		poke(vseg, s, c); }
	c = getc(fp);
	if(!(c & 0xFF00)) {
		fclose(fp);
		message("Extra data: %02x", c);
		return 4; }
	fclose(fp);
	return 0;
}

write_file(char *f, char *e)
{
	unsigned i;
	FILE *fp;
	static char vh1[] = { 'C','r','e','a','t','i','v','e',' ',
		'V','o','i','c','e',' ','F','i','l','e', 0x1A, 0x1A, 0x00,
		0x0A, 0x01, 0x29, 0x11 };

	if(!(fp = fileopen(f, e, "wvqb"))) {
		message("Unable to WRITE: %s", f);
		return; }

	fput(vh1, sizeof(vh1), fp);
	putc(1, fp);
	i = vsize + 2;
	fput(&i, sizeof(i), fp);
	putc(0, fp);
	putc(256 - lrg_scale(1000, 1000, vsample), fp);
	putc(0, fp);	// 8-bit PCM
	for(i=0; i < vsize; ++i)
		putc(peek(vseg, i), fp);
	fclose(fp);
}
#endif
/*
wait_voc()
{
	while(voc_status) {
		if(debug_flag)
			if(kbtst() == 0x1B)
				break; }
} */

/*
 * Play audio sample
 */
void play_sample(unsigned l, unsigned h)
{
	unsigned o;

#ifdef DEMO
	o = 0;
	while((h-l) > o) {
		sound((sample()*11) + 500);
		delay(55);
		o += sample_rate/55; }
	sound(0);
#else
/*	asm {
		MOV		BX,4		; Set speaker
		MOV		AX,1		; ON
		CALL	callsb		; Call driver
	} */
	if(debug_flag)
		lrg_printf(1, 0, WHITE, "%u %u", h-l, vsample);
	if((h-l) >= MIN_VOC_BLOCK) {
		poke(voc_seg, 0, 1);
		pokew(voc_seg, 1, (h-l)+2);
		poke(voc_seg, 3, 0);
		poke(voc_seg, 4, 256 - lrg_scale(1000, 1000, vsample));
		poke(voc_seg, o=5, 0);	// 8-bit PCM
		while(l < h)
			poke(voc_seg, ++o, peek(vseg, l++));
		while(++o)
			poke(voc_seg, o, 0);
		play_voc(voc_seg); }
#endif
}

trigger()
{
	unsigned c, t;
	unsigned char x;

	sample();
	lrg_plot(0, 0, YELLOW);
	if(trig_sense) {		// Positive edge trigger
		t = trig_level + 128;
		while(sample() >= t) {		/* Wait for negative */
			if(!++x) {
				if(c = kbtst())
					return c; } }
		while(sample() < t) {		/* Wait for positive */
			if(!++x) {
				if(c = kbtst())
					return c; } }
		lrg_plot(0, 0, GREEN);
		return 0; }

	// Negative edge trigger
	t = 128 - trig_level;
	while(sample() <= t) {			/* Wait for positive */
		if(!++x) {
			if(c = kbtst())
				return c; } }
	while(sample() > t) {			/* Wait for negative */
		if(!++x) {
			if(c = kbtst())
				return c; } }
	lrg_plot(0, 0, GREEN);
	return 0;
}
	
record_sample()
{
#ifdef DEMO
	for(vsize=0; vsize < block_size; ++vsize)
		poke(vseg, vsize, sample());
	delay(block_size/sample_rate * 1000);
#else
	unsigned i, o;

	asm {
		MOV		BX,4		; Set speaker
		MOV		AX,0		; OFF
		CALL	callsb		; Call driver
	}
	asm {
		MOV		BX,7			; Start voice input
		MOV		AX,DGRP:_sample_rate; Sample rate
		MOV		CX,DGRP:_block_size; Maximum length
		ADD		CX,100
		MOV		ES,DGRP:_voc_seg; Get voice segment
		MOV		DI,0			; Zero offset
		MOV		DX,0			; Zero high length
		CALL	callsb			; Inform card
	}
	while(voc_status);
//	asm {
//		MOV		BX,8	; Stop voice function
//		CALL	callsb	; Call the driver
//	}
	vtype = peek(voc_seg, 0);
	vsize = peekw(voc_seg, 1);
	if(vtype == 9) {
		vsize -= 12;
		vsample = peekw(voc_seg, 4);
		vformat = peek(voc_seg, 10);
		o = 15;	 }
	else {
		vsize -= 2;
		vsample = lrg_scale(1000, 1000, 256-peek(voc_seg, 4));
		vformat = peek(voc_seg, 5);
		o = 5; }
	if(vsize > block_size)
		vsize = block_size;
	for(i=0; i < vsize; ++i)
		poke(vseg, i, peek(voc_seg, ++o));
#endif
}

/*
 * Show the audio waveform
 */
void show_wave()
{
	unsigned i, s, v, ly, v1;
	int c;
	unsigned char f;

	v = offset;

	f = 0;
	for(i=0; i < 320; ++i) {
		c = peek(vseg, v1 = v++);
		for(s=0; s < hscale; ++s)
			c = (c + peek(vseg, v++)) >> 1;
		if(v < vsize) {
			c = ((c-vadjust) * vscale) / 100;
			lrg_vline(i, 10, 128, (((v-1) >= lbound) && (v1 <= rbound)) ? BROWN : BLUE);
			lrg_plot(i, 10+63, BLACK);
			if(c < -64) f=2, c = -64;
			if(c > 63) f=2, c = 63;
			c = (137-64) - c;
			if(f) {
				if(i)
					lrg_line(i-1, ly, i, c, RED);
				else
					lrg_plot(i, c, RED);
				--f;
				ly = c;
				continue; }
			if(i)
				lrg_line(i-1, ly, i, c, WHITE);
			else
				lrg_plot(i, c, WHITE);
			ly = c;
			continue; }
		lrg_vline(i, 10, 128, CYAN); }
}

/*
 * Show the alternate (overlay) waveform
 */
void show_overlay()
{
	unsigned i, s, v, ly, v1, *p;
	unsigned vseg, vsize /*, hscale */;
	int vscale, vadjust;
	int c;
	unsigned char f;

	if(active_buffer) {
		vseg = voc_seg + 4096;
		p = Pdata; }
	else {
		vseg = voc_seg + 8192;
		p = Rdata; }
	vsize = p[0];
	v = p[5];
	if(overlay == 2)
		v += offset;
//	hscale = p[2];
	vscale = p[3];
	vadjust = p[4];

	f = 0;
	for(i=0; i < 320; ++i) {
		c = peek(vseg, v1 = v++);
		for(s=0; s < hscale; ++s)
			c = (c + peek(vseg, v++)) >> 1;
		if(v < vsize) {
			c = ((c-vadjust) * vscale) / 100;
			if(c < -64) f=2, c = -64;
			if(c > 63) f=2, c = 63;
			c = (137-64) - c;
			if(f) {
				if(i)
					lrg_line(i-1, ly, i, c, VIOLET);
				else
					lrg_plot(i, c, VIOLET);
				--f;
				ly = c;
				continue; }
			if(i)
				lrg_line(i-1, ly, i, c, GREEN);
			else
				lrg_plot(i, c, GREEN);
			ly = c;
			continue; }
		break; }
}

char *fk1[] = {	// Boundary movement prompts
	"Left  Bound <-",
	"Left  Bound ->",
	"Right Bound <-",
	"Right Bound ->",
	"Set Left+Right",
	"Goto Left Bound",
	"Goto Right Bound",
	"Normalize",
	"Crop",
	"Cut" };
char *fk2[] = {	// File access prompts
	"",
	"Load Audio",
	"Save Audio",
	"Load Config",
	"Save Config",
	"Build Sinewave",
	"Play/Record",
	"Statistics",
	"",
	"" };

char **prompts[] = { &fk1, &fk2, 0 };

select_buffer(char b)
{
	unsigned *old, *new;
	old = active_buffer ? Rdata : Pdata;
	if(active_buffer = b) {		// Recording data
		new = Rdata;
		vseg = voc_seg + 8192;
		fk2[0] = "Output Buffer"; }
	else {
		new = Pdata;
		vseg = voc_seg + 4096;
		fk2[0] = "Input Buffer"; };
	// Save old values
	old[0] = vsize;
	old[1] = vsample;
	old[2] = hscale;
	old[3] = vscale;
	old[4] = vadjust;
	old[5] = offset;
	old[6] = lbound;
	old[7] = rbound;

	// Read stored array values
	vsize  = new[0];
	vsample= new[1];
	hscale = new[2];
	vscale = new[3];
	vadjust= new[4];
	offset = new[5];
	lbound = new[6];
	rbound = new[7];
}

/*
 * Display key prompts
 */
void mprompt(unsigned char *p[])
{
	unsigned i, x;
	lrg_fbox(0, 150, 320, 50, BLACK);
	for(i=0; i < 10; ++i) {
		x = (i/5)*160;
		if(i==9) x -= 8;
		lrg_printf(x, ((i%5)*10)+150, WHITE, "F%u:%s", i+1, p[i]); }
	lrg_puts(273, 200-10, WHITE, "?=keys");
}

/*
 * Normalize the sample to a preset maximum value
 */
void normalize(unsigned l, unsigned h, unsigned mv)
{
	unsigned i, c, max;
	max = 0;

	// Determine widest range of values
	for(i=l; i < h; ++i) {
		if((c = peek(vseg, i)) & 0x80)		// Positive value
			c -= 0x80;
		else								// Negative value
			c = 0x80 - c;
		if(c > max)
			max = c; }

	// Adjust values to match
	for(i=l; i < h; ++i) {
		if((c = peek(vseg, i)) & 0x80) {	// Positive
			c = ((c - 0x80) * mv) / max;
			poke(vseg, i, c + 0x80); }
		else {
			c = ((0x80 - c) * mv) / max;
			poke(vseg, i, 0x80 - c); } }
}

/*
 * Handle an input form
 */
register input(unsigned args)
{
	unsigned na, *ap, w, h, x, y, i, c, *ip, s;
	struct Input *p;
	unsigned char *cp, f;

	ap = (na = nargs()) * 2 + &args;	// Pointer to args
	w = ((*--ap) * 8) + 1;
	h = (--na * 9);
	lrg_fbox(x = 160-(w/2), y = 100-(h/2), w, h, WHITE);
	--ap; ++x; ++y;
	s = 0;
	while(!input_table[ap[-s]].type)
		++s;
	f = -1;
top:
	for(i=0; i < na; ++i) {
		h = (i*9)+y;
		ip = cp = (p = input_table[ap[-i]])->address;
		c = (i==s) ? (WHITE<<8)|RED : (WHITE<<8)|BLUE;
		switch(p->type) {
case 0 : lrg_printf((p->min)+x, h, (WHITE<<8)|BLACK, p->prompt);	break;
case 1 : lrg_printf(x, h, c, p->prompt, *ip);						break;
case 2 : lrg_printf(x, h, c, p->prompt, *cp ? p->max : p->min);		break;
case 3 : lrg_printf(x, h, c, p->prompt, ip); } }

	ip = cp = (p = input_table[ap[-s]])->address;

	switch(c = kbget()) {		// General movement & exits
	case _DA :
	case _UA :
	case 0x1B:
	case '\r' :
		f = -1;
		for(i=0; i < na; ++i) {
			ip = cp = (p = input_table[ap[-i]])->address;
			if((p->type == 1) && (*ip < p->min)) {
				s = i;
				*ip = p->min;
				goto top; }
			if((p->type == 3) && (strlen(cp) < (p->min&0xFF))) {
				s = i;
				goto top; } }
		do switch(c) {
			case 0x1B:	return 0;
			case '\r':	return -1;
			case _DA:	if(++s >= na) s = 0;	break;
			case _UA:	s = s ? s-1 : na-1; }
		while !input_table[ap[-s]].type;
		goto top; }

	if(p->type == 1) switch(c) {	// Numeric input
		case _HO : *ip += 90;
		case _PU : *ip += 9;
		case _RA : ++*ip;
		tstset:	f = -1;
				if(*ip < p->min) *ip = p->min;
		tstmax:	if(*ip > p->max) *ip = p->max;
			goto top;
		case _EN : if(*ip > 90) *ip -= 90;
		case _PD : if(*ip > 9) *ip -=9;
		case _LA : if(*ip) --*ip;	goto tstset;
		case _DEL: *ip = p->min;	goto tstset;
		case _INS: *ip = p->max;	goto tstset;
		case '\b': *ip /= 10;		goto top;
		default: if(isdigit(c)) {
			if(f)
				*ip = f = 0;
			*ip = (*ip * 10) + (c - '0');
			goto tstmax; } }

	if(p->type == 2) switch(c) {	// Yes/No input
		case _INS:	*cp = 0x77;	goto top;
		case _DEL:	*cp = 0;	goto top;
		case _RA :
		case _PU :
		case _HO :
		case _LA :
		case _PD :
		case _EN : *cp = *cp ? 0 : 0x77; goto top; }

	if(p->type == 3) switch(i = strlen(cp), c) {
		case '\b' :
			if(i)
				cp[i-1] = 0;
			goto top;
		case ' ' :
			if(p->min & 0x100)
				goto top;
		default: if((c >= ' ') && (c <= 0x7F) && (i < p->max)) {
			cp[i++] = (p->min & 0x200) ? toupper(c) : c;
			cp[i] = 0; } }

	goto top;
}

statistics(unsigned start, unsigned end)
{
	unsigned z, d, c;
	unsigned char h, l, s, p, df;
	char la[4], le[4], ln[4];

	h = l = 0x80;
	longset(le, z = d = c = 0);
	longset(la, 0x80);

	lrg_fbox(0, 0, 320, 200, BLACK);
	lrg_fbox(50,  8, 200, 16, WHITE);
	lrg_printf(60, 12, (WHITE<<8)|RED, "Statistics: %5u-%-5u", start, end);

//	lrg_printf(10, 10, WHITE, "Start: %-5u End: %u", start, end);

	p = peek(vseg, start);
	df = peek(vseg, start) > peek(vseg, start+1);
	while(start < end) {
		s = peek(vseg, start++);
		if(s > h)
			h = s;
		if(s < l)
			l = s;
		if(df) {
			if(s > p) {
				df = 0;
				++d; } }
		else {
			if(s < p) {
				df = -1;
				++d; } }
		longset(ln, s);
		longadd(la, ln);
		longset(ln, (s & 0x80) ? s-0x80 : 0x80 - s);
		longadd(le, ln);
		++c;
		if((s ^ p) & 0x80)
			++z;
		p = s; }
	longset(ln, c+1);
	longdiv(la, ln);
	if(c) {
		longset(ln, c);
		longdiv(le, ln); }
	lrg_printf(60, 40, WHITE, "Samples processed: %u", c);
	lrg_printf(60, 50, WHITE, "Highest value    : %d", (char)h-0x80);
	lrg_printf(60, 60, WHITE, "Lowest  value    : %d", (char)l-0x80);
	lrg_printf(60, 70, WHITE, "Average value    : %d", *la-0x80);
	lrg_printf(60, 80, WHITE, "Average energy   : %u", *(unsigned*)le);
	lrg_printf(60, 90, WHITE, "Zero crossings   : %u", z);
	lrg_printf(60,100, WHITE, "Direction changes: %u", d);
	lrg_puts(60, 200-10, (WHITE<<8)|RED, " Press ENTER to proceed ");
	while(kbget() != '\r');
}

xxget()
{
	unsigned c, t;
	if(!arun)
		return kbget();
	t = trig_level + 128;
	for(;;) {
		while(voc_status) {
			if(c = kbtst())
				return c;  }
		if(arun == 0x99) {
			show_wave();
			arun = 0x77;
			if(c = kbtst())
				return c; }
		if(c = trigger()) {
			lrg_plot(0, 0, BLACK);
			return c; }
		record_sample();
		lrg_plot(0, 0, BLACK);
		arun = 0x99; }
}

/*
 * Edit a sample wave in memory
 */
edit_wave()
{
	int c;
	unsigned i, j, k, s1, s2, l1, o;
	static unsigned char mode = 1;

rescreen:
	lrg_fbox(0, 0, 320, 200, BLACK);
newprompt:
	mprompt(prompts[mode]);
redraw:
	k = smode ? GREEN : RED;
	lrg_hline(0, 9, 320, BLACK);
	lrg_fbox(0, 131+7, 320, 10, BLACK);
	s1 = hscale + 1;
	o = offset/s1;
	if(j = smode) {
		if(j = o % 10)
			j = 10 - j; }
	while(j < 320) {
		lrg_plot(j, 9, BLUE);
		lrg_plot(j,129+9, BLUE);
		j += 10; }
	if(j = smode) {
		if(j = o % 50)
			j = 50 - j; }
	while(j < 320) {
		lrg_plot(j, 9, GREEN);
		lrg_plot(j,129+9, GREEN);
		j += 50; }
	if(i = s2 = smode) {
		if(i = o % 100)
			i = 100 - i;
		s2 = offset + (i*s1); }
	while(i < 320) {
		l1 = 16;
		if(s2 >= 100) {
			l1 += 8;
			if(s2 >= 1000) {
				l1 += 8;
				if(s2 >= 10000)
					l1 += 8; } }
		j = l1 / 2;
		if(i < j)
			j = i;
		if((i+j) > 319) {
			i = 320 - l1;
			j = 0; }
		lrg_printf(i-j, 131+9, k, "%u", s2);
		s2 += (s1*100);
		i += 100; }

	show_wave();
	if(overlay)
		show_overlay();
	l1 = (vsize > 319) ? vsize-319 : 0;
retitle:
	lrg_printf(1, 0, 7, "%5u-%-5u/%-5u %3u %-3u %5u-%-5u%c%c%c",
		offset+1, (s2 = s1*320)+offset, vsize, s1, vscale, lbound+1, rbound+1,
		hook ? 'H' : ' ',
		overlay ? (overlay==1 ? 'A' : 'R') : ' ',
		bflag ? 'B' : 'E');
	for(;;) {
		switch(c = toupper(xxget())) {
		case '?' :
			lrg_fbox(0, 0, 320, 200, BLACK);
			for(i=0; j = gkeys[i]; ++i)
				lrg_puts(0, i*10, WHITE, j);
			while(kbget() != ' ');
			goto rescreen;
		case _RA :	if((offset+s1) <= l1) offset+=s1;	goto redraw;
		case _LA :  if(offset > s1) { offset -= s1;		goto redraw; }
		case _HO :	offset = 0;							goto redraw;
		case _EN :	offset = l1;						goto redraw;
		case _PU :	offset = (offset > s2) ? offset-s2 : 0;	goto redraw;
		case _PD :	if((offset+s2) <= l1) offset+= s2;	goto redraw;
		case _CRA:	if((vsize / s1) >= 320) ++hscale;	goto redraw;
		case _CLA:	if(hscale) --hscale;				goto redraw;
		case _UA:	if(vscale < 250) ++vscale;			goto redraw;
		case _DA:	if(vscale) --vscale;				goto redraw;
		case _CHO:	hscale=0; vscale=50; vadjust=128;	goto redraw;
		case ' ':	if(!prompts[++mode]) mode = 0;		goto newprompt;
		case 'H':
			if(hook = !hook)
				lpt_set(LPT_HOOK);
			else
				lpt_clr(LPT_HOOK);
			goto retitle;
		case 'P':
				if(test_sound()) goto rescreen;
				if(bflag) {
	 				if((rbound - lbound) > 5)
						play_sample(lbound, rbound+1);
					continue; }
				play_sample(0, vsize);
				continue;
		case 'R' :
			if(test_sound()) goto rescreen;
		again:
			if(input(IWIDTH, Itrig_level, Itrig_sense, Iblock_size, Isample_rate, Iarun)) {
				if(arun && ((sample_rate / block_size) < 10)) {
					temp = 0;
					if(!(input(IWIDTH, IPnull, IPwarn1, IPwarn2, IPnull, IPproceed) && temp))
						goto again; }
				for(;;) switch(trigger()) {
					case 0 : record_sample();
					case 0x1B: lrg_plot(0, 0, BLACK);
						goto redraw; } }
			goto redraw;
		case  'S' :
			smode = smode ? 0 : -1;
			goto redraw;
		case 'O' :
			if(++overlay > 2) overlay=0;
			goto redraw;
		case 'Q' :
			temp = -1;
			if(input(IWIDTH, Iexit) && temp)
				return;
			goto rescreen;
		case 'M' :
			if(!test_sound())
				edit_mix();
			goto rescreen;
		case 'W' :
			if(edit_sine())
				memcpy(dst, st, sizeof(dst));
			goto rescreen;
		case 'E' : bflag = bflag ? 0 : -1;				goto retitle;
		case '=' :
		case '+' :	if(vadjust > 0) --vadjust;			goto redraw;
		case '_' :
		case '-' :	if(vadjust < 255) ++vadjust;		goto redraw;
		case '\t':	offset = (offset / s1) * s1;		goto redraw;
		case '0' :	c = 9;	goto dodtmf;
		case '*' :	c =10;	goto dodtmf;
		case '#' :	c =11;	goto dodtmf;
		case 'A' :	c =12;	goto dodtmf;
		case 'B' :	c =13;	goto dodtmf;
		case 'C' :	c =14;	goto dodtmf;
		case 'D' :	c =15;	goto dodtmf;
		case '1' :
		case '2' :
		case '3' :
		case '4' :
		case '5' :
		case '6' :
		case '7' :
		case '8' :
		case '9' : c -= '1';
		dodtmf:
			if(test_sound()) goto rescreen;
			build_dtmf(c, sample_rate/6);
			play_voc(voc_seg);
			continue; }

		if(!mode) switch(c) {
			case _F1 :
				if(lbound >= s1)
					lbound -= s1;
				else if(lbound)
					--lbound;
				goto redraw;
			case _F2 :
				if(lbound < (vsize-s1))
					lbound += s1;
				else if(lbound < vsize)
					++lbound;
				if(rbound < lbound)
					rbound = lbound;
				goto redraw;
			case _F3 :
				if(rbound >= s1)
					rbound -= s1;
				else if(rbound)
					--rbound;
				if(lbound > rbound)
					lbound = rbound;
				goto redraw;
			case _F4 :
				if(rbound < (vsize-s1))
					rbound += s1;
				else if(rbound < vsize)
					++rbound;
				goto redraw;
			case _F5 :	lbound = rbound = offset;				goto redraw;
			case _F6 :	offset = lbound;						goto redraw;
			case _F7 :	offset = rbound;						goto redraw;
			case _F8 :	// Normalize
				if(input(IWIDTH, Invalue)) {
					if(bflag)
						normalize(lbound, rbound, nvalue);
					else
						normalize(0, vsize, nvalue); }
				goto redraw;
			case _F9 :	// Crop
				if(rbound != lbound) {
					vsize = 0;
					while(lbound <= rbound)
						poke(vseg, vsize++, peek(vseg, lbound++));
					lbound = 0;
					rbound = vsize-1; }
				goto redraw;
			case _F10 :	// Cut
				i = (vsize - rbound)+1;
				j = lbound;
				vsize -= (rbound - lbound);
				while(i) {
					poke(vseg, j++, peek(vseg, ++rbound));
					--i; }
				rbound = lbound;			
				goto redraw; }

	if(mode == 1) switch(c) {
		case _F1 :	// Toggle active buffer
			select_buffer(!active_buffer);
			goto newprompt;
#ifdef DEMO
		case _F2 :
		case _F3 :
		case _F4 :
		case _F5 :
			message("File Functions disabled in DEMO");
			goto rescreen;
#else
		case _F2 :		// Load audio
			if(input(40, IPload_audio, Iafile))
				load_file(afile, ".VOC");
			goto rescreen;
		case _F3 :		// Save audio
			if(input(40, IPsave_audio, Iafile))
				write_file(afile, ".VOC");
			goto rescreen;
		case _F4 :		// Load config
			if(input(40, IPload_config, Icfile))
				load_config(cfile, ".CFG");
			goto rescreen;
		case _F5 :		// Save config
			if(input(40, IPsave_config, Icfile))
				write_config(cfile, ".CFG");
			goto rescreen;
#endif
		case _F6 :		// Sine editor
			if(!vsize)
				soverlay = 0;
			if(!input(IWIDTH, Isfreq, Isample_rate, Issize, Iscaden, Isoverlay))
				goto rescreen;
			if(!vsize)
				soverlay = 0;
			j = k = -1;
			if(scaden) {
				if(!input(IWIDTH, IPnull, IPcadence, IPnull, Icadon, Icadoff))
					goto rescreen;
				k = (j = cadon) + cadoff; }
			if(edit_sine())
				build_tone(sfreq, ssize, soverlay, j, k);
			goto rescreen;
		case _F7 :		// Play/Record
			if(test_sound()) goto rescreen;
			if(input(IWIDTH, Iblock_size, Isample_rate)) {
				select_buffer(1);
				play_sample(0, vsize);
				select_buffer(0);
				while(voc_status);
				record_sample(); }
			goto newprompt;
		case _F8 :
			if(bflag)
				statistics(lbound, rbound);
			else
				statistics(0, vsize);
			goto rescreen;
	} }
}

char help[] = { "\n\
Use:	LSCOPE	[options]\n\n\
opts:	-D		- enable Debug messages\n\
	-S		- disable Soundcard interface\n\
	P=1-3/address	- specify Parallel port\n\
\n?COPY.TXT 2002-2005 Dave Dunfield\n -- see COPY.TXT --.\n" };

main(int argc, char *argv[])
{
	unsigned i;
	unsigned char *p;

	lpt = peekw(0x40, 0x08);
	for(i=1; i < argc; ++i) {
		p = argv[i];
		switch((toupper(*p++) << 8) | toupper(*p++)) {
		case '-D' :
		case '/D' : debug_flag = -1;		continue;
		case '-S' :
		case '/S' :	senable = 0;			continue;
		case 'P=' :		// Specify LPT port
			if(!(lpt = atox(p)))
				abort("LPT must be 1-3 or address\n");
			if(lpt < 4)
				lpt = peekw(0x40, (lpt*2)+6);
			continue;
		default:
			printf("Unknown option: %s\n", argv[i]);
		case '?' << 8:
		case '-?':
		case '/?': }
			abort(help); }
	if(!lpt)
		abort("LPT port not found.");
	debug("LPT=%04x\n", lpt);
	lpt_clr(0xFF);

	if(!(voc_seg = alloc_seg(4096*3)))
		abort("Not enough memory");
	vsample = sample_rate;
	select_buffer(active_buffer = 0);
	select_buffer(active_buffer = 1);
	build_sine();
	memcpy(dst, st, sizeof(dst));
	if(senable)
		load_ct_voice();
	if(debug_flag)
		kbget();
	lrg_open();
	if(senable)
		update_mix();
	edit_wave();
	lrg_close();
	if(senable)
		unload_ct_voice();
}
