/*
 * Multi-Channel Data Line Monitor
 *
 * ?COPY.TXT 2001-2005 Dave Dunfield
 *  -- see COPY.TXT --.
 *
 * Compile with DDS Micro-C/PC: cc MDLM -fop
 */
#include <stdio.h>			// Standard I/O definitions
#include <window.h>			// Window library definitions
#include <file.h>			// File system definitions

#ifdef DEMO
	#define	MAX_PORTS	2	// Maximum # ports supported
	#define	DEMOTXT		" (Demo limit)"
#else
	#define	MAX_PORTS	24	// Maximum # ports supported
	#define	DEMOTXT		""
#endif
#define	MAX_PROCESS	10		// Maximum # of post-processors
#define	DATA_HOLD	10		// Data hold timer in signal-monitor

/*
 * Extended memory structure
 *
 * 64k = UartStatus/ModemStatus
 * 64k = Channel/DataChar
 * 64K = timestamp
 */
unsigned
	stat_seg,				// Status segment
	data_seg,				// Data segment
	time_seg;				// Time segment

/*
 * Channel control structure
 */
struct dlm_channel {
	unsigned com_port;					// Uart status register address
	unsigned last_status;				// Last status read from uart
	unsigned video_offset;				// Offset to 1st screen line
	unsigned char bit_sigs[10];			// Bit identifiers
	unsigned speed;						// Uart baudrate divisor
	unsigned char data, parity, stop;	// Data format
	unsigned char channel_name[9];		// Channel/port name
	unsigned char ri_name[9];			// Ring-indicator name
	unsigned char cd_name[9];			// Carrier detect name
	unsigned char dsr_name[9];			// Data-set-ready name
	unsigned char cts_name[9];			// Clear-to-send name
	} dlm_channels[MAX_PORTS];
#define	STRUCT_SIZE	66					// Cannot use sizeof() in asm{}

/*
 * Post processor control structure
 */
struct processor_struct {
	char process_name[51];				// Menu display for post processor
	char process_file[65];				// File to execute
	char process_tail[81];				// Command tail
	} process_list[MAX_PROCESS];

/*
 * Video attribute indexes (for colors[])
 */
#define	CO_TITLE	0		// Title screen
#define	CO_MAIN		1		// Main screen
#define	CO_MENU		2		// Pop-up menus
#define	CO_DATA		3		// Data capture
#define	CO_FDATA	4		// First data capture
#define	CO_SIGNAL	5		// Signal transition
#define	CO_FSIGNAL	6		// First signal transition
#define	CO_CURSOR	7		// Data capture cursor
#define	CO_FCURSOR	8		// First data capture cursor
#define	CO_DREVIEW	9		// Review cursor - data
#define	CO_SREVIEW	10		// Review cursor - status

/*
 * Video attribute storage
 */
unsigned char default_colors[] = {
	0x70,				// Title bar
	0x17,				// Main screen
	0x70,				// Pop-up windows
	0x1B,				// Channel data
	0x6B,				// Last channel data
	0x10,				// Channel background
	0x60,				// Last channel background
	0x12,				// Channel cursor
	0x62,				// Last channel cursor
	0x4B,				// Review cursor - data
	0x40 },				// Review cursor - status
	colors[sizeof(default_colors)];

/*
 * Misc global variables
 */
unsigned
	year,				// Year of last capture
	month,				// Month of last capture
	day,				// Day of last capture
	hour,				// Hour of last capture
	minute,				// Minute of last capture
	second,				// Second of last capture
	pm_select,			// Process menu selector
	buf_wptr,			// Buffer write pointer
	buf_rptr,			// Buffer read pointer
	video_wcol,			// Video write pointer
	video_wline,		// Video line
	video_spacing,		// Video spacing of channels
	video_end,			// End of video screen (wrap point)
	video_size,			// Size of video screen
	review_scr,			// Review screen position
	review_cur;			// Review cursor position
unsigned char
	channel,			// Current channel being scanned
	nchannel,			// Number of channels defined
	processors,			// Number of processors defined
	vcursor,			// Video cursor
	vsignal,			// Indicates signal change
	enable_capture,		// Enable capture
	delete_temp,		// Delete temp file
	comment[81],		// Descriptor
	*error,				// Error indicator
	tempfile[65],		// Temporary filename
	*pptr,				// General global parsing pointer
	font,				// Currently selected font
	font_max,			// Number of fonts loaded
	font_page,			// VGA page holding selected font
	font_c2[4],			// Dual (C2) mode select switches
	font_name[4][9],	// Names of loaded fonts
	font_data[4][4097];	// Storage for loaded fonts

/*
 * Preset filename storage
 */
static char
	cfgfile[65] = { "MDLM.CFG" },		// Configuration filename
	datafile[65] = { "MDLM.DAT" };		// Data-Image filename

/*
 * Translate table for parity text display
 */
static char *parity[] = {
	"ODD", "EVEN", "MARK", "SPACE", "NONE", 0 };

/*
 * Menu & form storage
 */
static char *main_menu[] = {
	"Begin data capture",
	"Review captured data",
	"Clear capture buffer",
	"Signal monitor",
	"Configure I/O ports",
	"Disk transfer functions",
	"Post-Processor menu",
	"Edit comment",
	"Exit MDLM",
	0 };
static char *file_menu[] = {
	"Load configuration",
	"Load data image",
	"Save data image",
	0 };
static char *filename_form[] = {
	72<<8|3,
	"\x00\x00\x40File:",
	0 };
static char edit_prompt[] = { "\n\
  \x1B      = Move left.\n\
  \x1A      = Move right.\n\
  DEL    = Delete character under cursor\n\
  BKSP   = Backup cursor and delete\n\
  INSERT = Toggle insert mode\n\
  HOME   = Position to start of text\n\
  END    = Position to end of text\n\
  PgUp   = Clear entire field\n\
  PgDn   = Clear to end of field\n\
  ENTER  = Accept data\n\
  ESC    = Cancel" };
static char help[] = { "\n\
Use:	MDLM [options]\n\n\
Opts:	/C		- Begin immediate capture\n\
	C=file		- Specify configuration file\n\
	D=file		- Load data file at startup\n\
\n?COPY.TXT 2001-2005 Dave Dunfield -  -- see COPY.TXT --.\n" };
static char welcome[] = { 
"MDLM version 1.0 - ?COPY.TXT 2001-2005 Dave Dunfield -  -- see COPY.TXT --." };

/* --- Misc. support functions --- */

/*
 * Scale to a fractional value using 32 bit intermediate
 */
unsigned scale(value, multiply, divide) asm
{
		MOV		AX,8[BP]		; Get value
		MUL		WORD PTR 6[BP]	; Multiply to 32 bit product
		MOV		BX,4[BP]		; Get divisor
		DIV		BX				; Divide back to 16 bit result
		SHR		BX,1			; /2 for test
		INC		DX				; .5 rounds up
		SUB		BX,DX			; Set 'C' if remainder > half
		ADC		AX,0			; Increment result to scale
}

/*
 * Test for VGA adapter
 */
int test_vga(void) asm
{
		MOV		AX,1A00h		; Get display code
		INT		10h				; Call BIOS
		CMP		AL,1Ah			; VGA supported?
		XOR		AX,AX			; 0=Yes
		JZ		initv1			; It does exist
		DEC		AX				; -1 = Not available
initv1:
}

/*
 * Enable a prevusly loaded font
 */
void font_on(void) asm
{
		PUSH	ES				; Save ES
		MOV		AX,1130h		; Get font information
		MOV		BH,0			; Current font
		INT		10h				; Ask BIOS
		PUSH	DS				; Get DS
		POP		ES				; ES = DS
		CMP		CX,16			; Is current font > 16 points?
		JA		fo1				; Yes, special case
; Font is 16 points or less, just use one we have
		MOV		AX,1100h		; Set font
		MOV		BH,10h			; 16 point font
		MOV		BL,DGRP:_font_page	; Select this page
		MOV		CX,0100h		; 256 charcters
		XOR		DX,DX			; Start with character 0
		MOV		BP,DGRP:_pptr	; Use font table "as is"
		INT		10h				; Ask BIOS
		JMP SHORT fo3			; And exit
; Font is greater than 16 points
; We must translate our 16 point font to match (zero pad)
fo1:	MOV		BH,CL			; BH = points
		XOR		BL,BL			; * 256
		SUB		SP,BX			; Allocate space for new font
		MOV		DI,SP			; Dest is new buffer
		MOV		SI,DGRP:_pptr	; Source is our font
		MOV		DX,SI			; Copy
		ADD		DX,4096			; Adjust to buffer size
; Copy in our font data
fo2:	MOV		CX,16			; Copy 16 bytes
	REP	MOVSB					; Copy the character
		MOV		CL,BH			; Get point size
		SUB		CL,16			; Calculate # bytes remaining
		XOR		AX,AX			; Get zero
	REP STOSB					; Clear the end
		CMP		SI,DX			; Are we at end?
		JB		fo2				; Do all data
		MOV		AX,1100h		; Set font
		MOV		BL,DGRP:_font_page	; Get page to load
		MOV		CX,0100h		; 256 characters
		XOR		DX,DX			; Character 0
		MOV		BP,SP			; Use new font table
		INT		10h				; Ask BIOS
		MOV		SP,DI			; Fix SP
fo3:	POP		ES				; Restore ES
}

/*
 * Disable loaded font / enable standard VGA font
 */
void font_off(void) asm
{
		MOV		AX,1104h		; Load ROM 8x16 font
		MOV		BL,00h			; Font bank 0
		INT		10h				; Ask BIOS
}

/*
 * Enable dual character set mode
 */
void enable_c2(void) asm
{
		MOV		AX,1103h		; Set block specifier
		MOV		BL,04h			; !att3=Block0 att3=Block1
		INT		10h				; Ask BIOS
}

/*
 * Disable dual character set mode
 */
void disable_c2(void) asm
{
		MOV		AX,1103h		; Set block specifier
		MOV		BL,00h			; Always block0
		INT		10h				; Ask BIOS
}

/*
 * Select a font by index (0=PC/Standard, 1-n=loaded)
 */
void select_font(unsigned f)
{
	if(font_max) {				// Fonts installed?
		if(!f) {				// 0=Disable
			disable_c2();
			font_off();
			return; }
		pptr = font_data[f-1];
		if(font_c2[f-1]) {		// Dual character set
			font_off();
			font_page = 1;
			font_on();
			enable_c2();
			return; }
		font_page = 0;			// Single character set
		disable_c2();
		font_on(); }
}

/*
 * Display message & prompt for key input
 */
register message(unsigned args)
{
	char buffer[81], *p, *p1;
	unsigned l, y, ll;

	// Perform "printf()" formatting of message
	l = _format_(nargs() * 2 + &args, buffer);

	// Detect prompt from leading '|'
	p = "Press ENTER to continue.";
	ll = 25;
	if(*(p1 = buffer) == '|') {
		p = "Press ENTER to continue, ESC to cancel.";
		++p1;
		--l;
		ll = 40; }

	// Calculate window position/size, open and display message
	y = 4;
	if(l < ll) l = ll;
	if(l > 70) {
		++y;
		l = 70; }
	wopen(38-(l/2), 10, l+2, y, ((unsigned)colors[CO_MENU])|(WSAVE|WCOPEN|WWRAP|WBOX2));
	wputs(p1);
	wgotoxy(0, y-3);
	wputs(p);

	// Wait for ENTER or ESC to terminate message box
	for(;;) switch(wgetc()) {
		case '\n' :		// Enter
			wclose();
			return 0;
		case 0x1B :		// ESC
			wclose();
			return -1; }
}

/*
 * Display the title at the top of the screen
 */
register title(unsigned args)
{
	char buffer[81];

	_format_(nargs() * 2 + &args, buffer);
	wgotoxy(0, 0);
	*W_OPEN = colors[CO_TITLE];
	wputs(buffer);
	wcleol();
}

/*
 * Draw the data capture/review screen
 */
void draw_screen(void)
{
	unsigned i, i1;
	wgotoxy(0, 0); *W_OPEN = colors[CO_TITLE]; wcleol();
	for(i=i1=0; i < video_size; ++i) {
		if(i1 > video_end)
			break;
		wgotoxy(0, i+1);
		*W_OPEN = colors[(i % nchannel) ? CO_SIGNAL : CO_FSIGNAL];
		wcleol();
		i1 += 160; }

	// Reset positions after inital draw
	video_wline = video_wcol = 0;
}

/*
 * Install default configuration
 */
void install_default(void)
{
	*comment = nchannel = processors = 0;
	memcpy(colors, default_colors, sizeof(colors));
	vcursor = 0xB1;
	vsignal = 0xFE;
	video_wline = video_wcol = review_scr = review_cur =
	buf_rptr = buf_wptr = video_size = pm_select =
	year = month = day = hour = minute = second = 0;
	delete_temp = -1;
	if(!enable_capture)
		enable_capture = -1;
	strcpy(tempfile, "$MDLM$.TMP");
	strcpy(comment, welcome);
	font = font_max = 0;
}

/*
 * Setup the uart with its com port settings
 */
int setup_uart(unsigned c)
{
	unsigned i, a, j;
	unsigned b, speed, p, d, s;

	b = dlm_channels[c].com_port - 5;
	speed = dlm_channels[c].speed;
	d = dlm_channels[c].data - 5;
	p = dlm_channels[c].parity;
	s = dlm_channels[c].stop - 1;

	i = ((p << 4) & 0x30) |		/* parity type */
		(d & 0x03) |			/* # data bits */
		((s << 2) & 0x04) |		/* # stop bits */
		((p < 4) << 3);			/* parity enable */

	out(a = b+3, j = i | 0x80);
	if(in(a) != j) goto fail;
	out(a = b, j = speed & 255);
	if(in(b) != j) goto fail;
	out(a = b+1, j = speed >> 8);
	if(in(a) != j) goto fail;
	out(a = b+3, i);
	if(in(a) != i) goto fail;
	out(a = b+1, 0);		/* disable all interrupts */
	if(in(a) != 0) goto fail;
	out(a = b+4, 3);		/* Enable DTR and RTS */
	if(in(a) != 3) goto fail;
	return 0;
fail:
	return -1;
}

/*
 * Skip blanks in the input line
 */
int skip_blanks(void)
{
	while(isspace(*pptr))
		++pptr;
	return *pptr;
}

/*
 * Parse an item from the input line
 */
int parse(char *dest)
{
	int x;
	x  = *pptr;
	while((*pptr) && !isspace(*pptr))
		*dest++ =  toupper(*pptr++);
	*dest = 0;
	skip_blanks();
	return x;
}

/*
 * Get a hex value from the input pointer
 */
unsigned get_hex(unsigned l, unsigned h)
{
	unsigned x;
	char item[50];
	parse(item);
	if(isxdigit(*item)) {
		x = atox(item);
		if((x >= l) && (x <= h))
			return x; }
	error = "Bad HEX value";
	return l;
}

/*
 * Get a decimal value from the input pointer
 */
unsigned get_dec(unsigned l, unsigned h)
{
	unsigned x;
	char item[50];
	parse(item);
	if(isxdigit(*item)) {
		x = atoi(item);
		if((x >= l) && (x <= h))
			return x; }
	error = "Bad decimal value";
	return l;
}

/*
 * Copy a filename and provide a default extension
 */
void copy_file(char *dest, char *src, char *ext)
{
	char flag, *ptr1;

	flag = -1;
	ptr1 = dest;
	while(*ptr1 = toupper(*src++)) {
		if(*ptr1++ == '.')
			flag = 0; }
	if(flag)
		strcpy(ptr1, ext);
}

/*
 * Read and configure a signal identifier with optional name
 */
void read_signal(unsigned n)
{
	unsigned i;
	char *p;
	struct dlm_channel *c;

	if(*pptr) {		// Identifier is present
		c = dlm_channels[nchannel];
		p = 0;
		switch(n) {
		case 8 : i=0; break;
		case 7 : i=1; break;
		case 6 : i=2; break;
		case 5 : i=3; break;
		case 4 : i=4; break;
		case 3 : i=5; p = c->cd_name; break;
		case 2 : i=6; p = c->ri_name; break;
		case 1 : i=7; p = c->dsr_name; break;
		case 0 : i=8; p = c->cts_name; }
		c->bit_sigs[i] = *pptr++;
		switch(*pptr) {
		case ':' :	// Name is supplied
			if(!p) {
				error = "Condition flag cannot be named!";
				return; }
			while(*++pptr && !isspace(*pptr))
				*p++ = *pptr;
			*p = 0;
			skip_blanks();
			return;
		case 0 :
		case ' ' :	// No name is supplied
			if(p)
				*p=0;
			skip_blanks();
			return; }
		// Format is not recognized
		error = "Bad signal name format"; }
}

/*
 * Read the configuration file and process it
 */
int read_config(void)
{
	unsigned t, t1, t2, t3, i, sp, db, pa, sb, line;
	char buffer[200], item[65], file[65], *ptr, dflag, eflag, vflag;
	FILE *fp;
	static char *keywords[] = {
		"PORT", "TITLE", "DATA", "BACKGROUND", "ENDATA", "CURSOR", "LINES",
		"MAIN", "MENU", "FONT", "COMMENT", "PROCESS", "TEMPFILE",
		0 };

	// Setup filename and open it
	copy_file(file, cfgfile, ".CFG");
	dflag = eflag = line = 0;
	if(!(fp = fopen(file, "r"))) {
		wprintf("Unable to open: %s\n", file);
		wputs("Press a key...");
		wgetc();
		return -1; }

	W_OPEN->WINattr |= WSCROLL;

	// Install default, then read and apply configuration statements
	install_default();
	vflag = test_vga();
	while(fgets(pptr = buffer, sizeof(buffer)-1, fp)) {
		++line;
		switch(skip_blanks()) {
		case ';' :					// Comment
		case 0 : continue; }		// or Blank line - ignore
		// Obtain & lookup keyword then execute handler
		parse(item); 
		for(t=0; ptr = keywords[t]; ++t) {
			if(strbeg(ptr, item))
				break; }
		error = 0;
		switch(t) {
		case 0 :		// PORT
			if(nchannel >= MAX_PORTS) {
				error = "Maximum of "#MAX_PORTS" channels exceeded."#DEMOTXT;
				break; }
			// Read port configuration
			t = get_hex(1, 65535);
			sp = scale(57600, 2, t2 = get_dec(1, 57600));
			db = get_dec(5, 8);
			parse(item);
			for(pa=0; ptr = parity[pa]; ++pa) {
				if(strbeg(ptr, item))
					goto pfound; }
			error = "Bad parity specifier";
			break;
		pfound:
			sb = get_dec(1, 2);

			// Input the channel name & signal identifier bits
			parse(dlm_channels[t1 = nchannel].channel_name);
			strcpy(dlm_channels[nchannel].bit_sigs, "BFPODCRST");
			for(i=0; i < 9; ++i)
				read_signal(i);

			// Store misc. channel parameters
			dlm_channels[nchannel].com_port = t + 5;
			dlm_channels[nchannel].last_status = -1;
			dlm_channels[nchannel].video_offset = (160*nchannel) + 160;
			dlm_channels[nchannel].speed = sp;
			dlm_channels[nchannel].data = db;
			dlm_channels[nchannel].parity = pa;
			dlm_channels[nchannel].stop = sb;

			// Display channel info & match speed
			wprintf("Channel %u : %04x %u", ++nchannel, t, t3 = scale(57600, 2, sp));
			wprintf(" %u/%s/%u", db, parity[pa], sb);
			if(t2 != t3) {
				wprintf(" !!!SPEED ADJUSTED!!! %u->%u", t2, t3);
				dflag = -1; }

			// Initialize the channel uart
			if(setup_uart(t1)) {
				wprintf(" !!!FAILED!!!");
				enable_capture = 0;
				error = "Serial port failed init"; }
			wputc('\n');
			break;
		case 1 :		// TITLE
			colors[CO_TITLE] = get_hex(1, 255);
			break;
		case 2 :		// DATA
			colors[CO_FDATA] = get_hex(1, 255);
			colors[CO_DATA] = get_hex(1, 255);
			break;
		case 3 :		// SIGNAL
			colors[CO_FSIGNAL] = get_hex(1, 255);
			colors[CO_SIGNAL] = get_hex(1, 255);
			vsignal = get_hex(1, 255);
			break;
		case 4 :		// CURSOR
			colors[CO_FCURSOR] = get_hex(1, 255);
			colors[CO_CURSOR] = get_hex(1, 255);
			vcursor = get_hex(1, 255);
			break;
		case 5 :		// Review cursor
			colors[CO_DREVIEW] = get_hex(1, 255);
			colors[CO_SREVIEW] = get_hex(1, 255);
			break;
		case 6 :		// Video size
			video_size = get_dec(10, 100) - 1;
			break;
		case 7 :		// Main window attribute
			colors[CO_MAIN] = get_hex(1, 255);
			break;
		case 8 :		// Menu window attributes
			colors[CO_MENU] = get_hex(1, 255);
			break;
		case 9 :		// Font selection
			if(vflag) {
				vflag = 0x7F;
				break; }
			if(font_max >= 4)
				abort("Too many font's");
			parse(item);
			copy_file(file, item, ".DLF");
			if(!(t = open(file, F_READ))) {
				error = "Cannot open file";
				break; }
			read(item, 3, t);
			if((item[0] != 'D') || (item[1] != 'L') || (item[2] != 'M'))
				goto badfont;
			read(item, 6, t);	// Read & discard special characters
			if(read(font_data[(unsigned)font_max], sizeof(font_data[0]), t) != 4096) {
			badfont: error = "Invalid font file";
				close(t);
				break; }
			parse(item);
			if(toupper(*item) == 'Y')	// Enable dual ?
				font_c2[(unsigned)font_max] = -1;
			parse(font_name[(unsigned)font_max]);
			++font_max;
			close(t);
			break;
		case 10 :		// Comment
			strcpy(comment, pptr);
			break;
		case 11 :		// Processor
			if(processors >= MAX_PROCESS) {
				error = "To many processors defined"#DEMOTXT;
				break; }
			ptr = process_list[processors].process_name;
			while(*pptr != '|') {
				if(!*pptr) {
					error = "Process must have |filename";
					goto test_error; }
				*ptr++ = *pptr++; }
			++pptr;
			skip_blanks();
			*ptr = 0;
			parse(process_list[processors].process_file);
			strcpy(process_list[processors++].process_tail, pptr);
			break;
		case 12 :			// TEMPFILE
			parse(item);
			copy_file(tempfile, item, ".DAT");
			break;
		default:
			error = "Unknown keyword"; }
	test_error:
		if(error) {
			wprintf("Line %u: %s\n>>", line, error);
			wputs(buffer);
			wputc('\n');
			error = 0;
			eflag = -1; }
			continue;  }

	fclose(fp);

	// Check that at least one channel is defined
	if(!nchannel) {
		eflag = -1;
		wputs("No capture channels defined!\n"); }

	// Check for FONT / non-vga conflict
	if(vflag == 0x7F) {
		wputs("Use of FONT requires a VGA display - fonts disabled.\n");
		dflag = -1; }

	// Check bit3 for dual-font configurations
	for(i=t1=0; i < font_max; ++i)
		t1 |= font_c2[i];
	if(t1) {
		t2 = colors[CO_FDATA]&colors[CO_DATA]&colors[CO_DREVIEW];
		if(!(t2 & 0x08)) {
			eflag = -1;
			wputs("Bit3 must be SET in data attributes when using dual-fonts!\n"); }
		t2	= colors[CO_FSIGNAL]|colors[CO_SIGNAL]|colors[CO_TITLE]
			| colors[CO_FCURSOR]|colors[CO_CURSOR]|colors[CO_SREVIEW];
		if(t2 & 0x08) {
			eflag = -1;
			wputs("Bit3 must be CLEAR in title/signal/cursor attributes when using dual-fonts!\n"); } }

	// If an error occured, reset to default & disable capture
	if(eflag) {
		install_default();
		enable_capture = 0; }

	// Compute the video size if not predefined
	if(!video_size) {
		video_size = peek(0x40, 0x84);
		if((video_size < 24) && (video_size > 49))
			video_size = 24; }

	// Compute screen geometry
	video_spacing = ((unsigned)nchannel) * 160;
	if(eflag) {
		video_end = 160*12;
		wputs("Press a key...");
		wgetc();
		W_OPEN->WINattr &= ~WSCROLL;
		return -1; }
	for(t=t1=0; t < video_size; ++t) {
		if((t % nchannel) == (nchannel-1)) {
			video_end = t1; }
		t1 += 160; }

	// If warnings, inject a delay
	if(dflag) {
		for(i=10; i; --i) {
			wprintf("\rProceeding in %u seconds... Press key to bypass. ", i);
			for(t=0; t < 10; ++t) {
				if(wtstc()) goto doexit;
				delay(100); } }
	doexit: wputc('\n'); }

	W_OPEN->WINattr &= ~WSCROLL;
	return 0;
}

/*
 * Poll uarts and capture data / update screen in real time
 */
void poll_capture(void) asm
{
		MOV		BX, OFFSET DGRP:_dlm_channels	; Point to data structure
		MOV		DI,DGRP:_buf_wptr		; Get write pointer
		XOR		SI,SI					; Indicate no write data
; Read this channel
cap1:	MOV		DX,[BX]					; Read I/O port
		IN		AL,DX					; Read line status register
		MOV		AH,AL					; Save upper
		INC		DX						; Point to next
		IN		AL,DX					; Read modem status register
		AND		AX,1FF0h				; Save only bits we care about
		TEST	AH,01h					; Any data received
		JZ		cap2					; No, try next
; Data has been received
		MOV		CX,AX					; Save status
		CALL	updspc					; Update video
		MOV		AX,word ptr DGRP:_colors+CO_DATA	; Get colors
		MOV		DX,4[BX]				; Get video offset
		ADD		SI,DX					; Adjust
		CMP		DX,160					; first line
		JZ		cap1a					; No, try next
		MOV		AH,AL					; Save data
cap1a:	MOV		DX,[BX]					; Get I/O port address
		SUB		DX,5					; Backup to data port
		IN		AL,DX					; Read data byte
		MOV		ES:[SI],AX				; Save data
		JMP	SHORT cap3					; Do next
; Test for status change from last time
cap2:	CMP		AL,2[BX]				; Have any signals changed
; If you want reports of the BFPO signals going OFF, then
; remove the above CMP, and uncomment the three lines below
;		MOV		CX,2[BX]				; Get old status
;		AND		CH,0FEh					; Remove data indicator
;		CMP		AX,CX					; Different from last status?
		JZ		cap6					; No, no change
		MOV		CX,AX					; CX = Status
		CALL	updspc					; Update video
		ADD		SI,4[BX]				; Add video offset
		MOV		AL,DGRP:_vsignal		; Get signal indicator
		MOV		ES:[SI],AL				; Write it
		XOR		AL,AL					; AL = data (none)
; Save event for later display
cap3:	MOV		2[BX],CX				; Save last status
		MOV		ES,DGRP:_stat_seg		; Get status segment
		MOV		ES:[DI],CX				; Save status
		MOV		AH,DGRP:_channel		; Get current channel
		MOV		ES,DGRP:_data_seg		; Get data segment
		MOV		ES:[DI],AX				; Save data
		MOV		AX,40h					; Bios segment
		MOV		ES,AX					; Setup segment
		MOV		CX,WORD PTR ES:[06Ch]	; Read data
		MOV		ES,DGRP:_time_seg		; Point to time segment
		MOV		ES:[DI],CX				; Save data
		INC		DI						; Skip to next
		INC		DI						; Position in buffer
		MOV		DGRP:_buf_wptr,DI		; Resave DI
		MOV		AX,DGRP:_buf_rptr		; Get read pointer
		CMP		AX,DI					; Wrapped buffer?
		JNZ		cap3a					; No problem
		INC		AX						; Skip ahead
		INC		AX						; Skip ahead
		MOV		DGRP:_buf_rptr,AX		; Resave
; Update video position
cap3a:	MOV		AX,DGRP:_video_wcol		; Get column
		INC		AX						; Advance
		INC		AX						; Advance
		CMP		AX,160					; Are we over?
		JB		cap5					; No, its OK
		MOV		AX,DGRP:_video_wline	; Get current line
		ADD		AX,DGRP:_video_spacing	; Skip to next
		CMP		AX,DGRP:_video_end		; Are we over line
		JB		cap4					; No, its OK
		XOR		AX,AX					; Reset line
cap4:	MOV		DGRP:_video_wline,AX	; Resave line
		XOR		AX,AX					; Reset col
cap5:	MOV		DGRP:_video_wcol,AX		; Resave col
		MOV		AL,DGRP:_vcursor		; Cursor display
		MOV		DX,word ptr DGRP:_colors+CO_CURSOR ; Get colors
		CALL	updvid					; Update video
; Advance to the next channel
cap6:	ADD		BX,STRUCT_SIZE			; Skip to next
		MOV		AL,DGRP:_channel		; Point to channel
		INC		AL						; Skip to next
		MOV		DGRP:_channel,AL		; Resave
		CMP		AL,DGRP:_nchannel		; Are we over?
		JAE		cap7					; Yes, exit
		JMP		cap1					; Do them all
cap7:
}

/*
 * Update current video position with attribute and data
 */
asm {
updspc:	MOV		DX,word ptr DGRP:_colors+CO_SIGNAL ; Get colors
		MOV		AL,' '					; Get space
updvid:	MOV		ES,DGRP:_W_BASE			; Offset
		MOV		SI,DGRP:_video_wline	; Offset to current line
		ADD		SI,DGRP:_video_wcol		; Offset to current column
		MOV		AH,DGRP:_nchannel		; Get # channels
updv1:	ADD		SI,160					; Next line
		MOV		ES:[SI],AL				; Write character
		MOV		ES:1[SI],DH				; Save data attribute
		MOV		DH,DL
		DEC		AH						; Reduce count
		JNZ		updv1					; Do them all
		SUB		SI,DGRP:_video_spacing	; Get data back
		RET
}

/*
 * Display a single-screen of review data
 */
void review_screen(unsigned p)
{
	unsigned c, s, d, t, o, o1;
	char flag;

	flag = video_wline = video_wcol = 0;

	do {
		s = peekw(stat_seg, p);
		c = (d = peekw(data_seg, p)) >> 8;
		t = peekw(time_seg, p);
		p += 2;
		o = (o1 = dlm_channels[c].video_offset) + video_wline + video_wcol;

		// Update display with data/signal
		if(s & 0x100) {		// Data available
			asm {
				CALL	updspc
			}
			poke(W_BASE, o, d);
			poke(W_BASE, o+1, colors[(o1 == 160) ? CO_FDATA : CO_DATA]); }
		else {				// Signal change
			asm {
				call	updspc
			}
			poke(W_BASE, o, vsignal); }

		// Advance to next column
dovideo:
		if((video_wcol += 2) >= 160) {
			video_wcol = 0;
			if((video_wline += video_spacing) >= video_end)
				return; } }
	while(p != buf_wptr);
	if(flag) asm {
		CALL	updspc
	} else asm {
		MOV		DX,word ptr DGRP:_colors+CO_CURSOR
		MOV		AL,DGRP:_vcursor
		CALL	updvid
	}
	flag = -1;
	goto dovideo;
}

/*
 * Backup the review cursor by 'x' spaces
 */
void backup_cursor(unsigned x)
{
	while(x--) {
		if(review_cur == buf_rptr)
			return;
		review_cur -= 2; }
}

/*
 * Advance the review cursor by 'x' spaces
 */
void advance_cursor(unsigned x)
{
	unsigned t;
	t = buf_wptr - 2;
	while(x--) {
		if(review_cur == t)
			return;
		review_cur += 2; }
}

/*
 * Backup the review screen base position by 'x' spaces
 */
void backup_scr(unsigned x)
{
	while(x--) {
		if(review_scr == buf_rptr)
			return;
		review_scr -= 2; }
}

/*
 * Advance the review screen base position by 'x' spaces
 */
void advance_scr(unsigned x)
{
	unsigned t;
	t = buf_wptr - 2;
	while(x--) {
		if(review_scr == t)
			return;
		review_scr += 2; }
}

/*
 * Review the captured data
 */
void review(void)
{
	unsigned s, d, t, tb, i, c, ch, l, vs;
	unsigned char sc[MAX_PORTS], adj_flag;

	tb = peekw(time_seg, buf_rptr);
	vs = video_size/nchannel * 80;
	// Redraw the entire screen
redraw:
	adj_flag = 0;
	title("REVIEW: ESC=Main-menu F1=Font %-8s %5u:%-5u",
		font ? font_name[font-1] : "",
		(buf_wptr - buf_rptr)/2, (review_scr - buf_rptr)/2 +1);
	review_screen(review_scr);
	// Reposition the review cursor
reposition:
	// Validate the current cursor position to insure it is
	// located within the displayed screen
	i = nchannel * 160;
	c = review_cur - review_scr;		// Cursor position
	l = 160;
	while(c >= 160) {
		c -= 160;
		l += i; }
	l = (l + c) + 1;
	// If cursor is out of bounds, adjust position & redraw
	if((l < 160) || (l > video_end)) {
		switch(adj_flag) {
		case 1 :	// Advancing by 1
			advance_scr(1);
			goto redraw;
		case 2 :	// Backup by 80
			backup_scr(80);
			goto redraw;
		case 3 :	// Advancing by 80
			advance_scr(80);
			goto redraw; }
		review_scr = review_cur;
		goto redraw; }
	adj_flag = 0;

	// Display information in title bar
	c = l;
	s = peekw(stat_seg, review_cur);
	ch = (d = peekw(data_seg, review_cur)) >> 8;
	t = peekw(time_seg, review_cur) - tb;
	for(i=0; i < nchannel; ++i) {
		sc[i] = peek(W_BASE, l);
		poke(W_BASE, l, colors[((ch == i) && (s & 0x100)) ? CO_DREVIEW : CO_SREVIEW]);
		l += 160; }
	wgotoxy(50, 0); wprintf("%5u:%-2u ",(review_cur - buf_rptr)/2 + 1, ch+1);
	if(s & 0x100)
		wprintf("%02x ", d & 255);
	else
		wputs("-- ");
	pptr = dlm_channels[d>>8].bit_sigs;
	while(*pptr) {
		wputc((s & 0x1000) ? *pptr : '-');
		s <<= 1;
		++pptr; }
	wprintf(" %5u.%02u", t / 18, ((t%18)*3640)/655);

	// Get command, then remove the current review cursor
	l = wgetc();
	for(i=0; i < nchannel; ++i) {
		poke(W_BASE, c, sc[i]);
		c += 160; }

	// Execute command key
	switch(l) {
	case _KHO :		// Back to beginning
		review_cur = review_scr = buf_rptr;
		goto redraw;
	case _KEN :		// Advance to end
		review_cur = review_scr = buf_wptr - 2;
		backup_scr(vs - 2);
		goto redraw;
	case _KLA :		// Backone char
		backup_cursor(1);
		goto reposition;
	case _KRA :		// Advance one char
		advance_cursor(1);
		adj_flag = 1;
		goto reposition;
	case _KUA :		// Back one line
		backup_cursor(80);
		adj_flag = 2;
		goto reposition;
	case _KDA :		// Advance one line
		advance_cursor(80);
		adj_flag = 3;
		goto reposition;
	case _KPU :		// Back one page
		backup_cursor(vs);
		goto reposition;
	case _KPD :		// Advance one page
		advance_cursor(vs);
		goto reposition;
	case _K1 :		// Toggle font selection
		if(++font > font_max)
			font = 0;
		select_font(font);
		goto redraw;
	case 0x1B :		// Exit
		return; }
	goto reposition;
}

/*
 * Edit the Channel/Signal names
 */
void edit_names(unsigned c)
{
	unsigned select;
	char *menu[6], buffer[5][25], temp[10], *p;
	struct dlm_channel *cp;

	cp = dlm_channels[c];
	select = 0;

retitle:	// Display title bar
	wcursor_off();
	title("CHANNEL NAME EDITOR: \x18\x19=Select ENTER=Edit ESC=Config-menu");
	wgotoxy(0, 1);
	*W_OPEN = colors[CO_MAIN];
	wcleow();

	// Build menu
	sprintf(menu[0] = buffer[0], "Channel name: %s", cp->channel_name);
	sprintf(menu[1] = buffer[1], "RI      name: %s", cp->ri_name);
	sprintf(menu[2] = buffer[2], "CD      name: %s", cp->cd_name);
	sprintf(menu[3] = buffer[3], "DSR     name: %s", cp->dsr_name);
	sprintf(menu[4] = buffer[4], "CTS     name: %s", cp->cts_name);
	menu[5] = 0;

	// Display menu & select name to edit
	if(wmenu(25, 7, ((unsigned)colors[CO_MENU])|(WSAVE|WCOPEN|WBOX1),
		menu, &select))
		return;
	switch(select) {
	case 1 : p = cp->ri_name;	break;
	case 2 : p = cp->cd_name;	break;
	case 3 : p = cp->dsr_name;	break;
	case 4 : p = cp->cts_name;	break;
	default: p = cp->channel_name; }

	strcpy(temp, p);

	title("CHANNEL NAME EDITOR");
	*W_OPEN = colors[CO_MAIN];
	wgotoxy(2, 3);
	wputs(menu[select]);
	wgotoxy(0, 10);
	wputs(edit_prompt);

	// Edit name, wait for ENTER to save or ESC to cancel
	*W_OPEN = colors[CO_MENU];
	for(;;) {
		switch(wgets(14+2, 3, temp, 8)) {
		case '\n' :
			strcpy(p, temp);
			goto retitle;
		case 0x1B :
			goto retitle; } }
}

/*
 * Display and configure the current port settings
 */
void configure_ports(void)
{
	unsigned i, y, s;
	unsigned char n, r, changed_flag[MAX_PORTS];
	static unsigned char p=0;

	n = colors[CO_MAIN];
	r = (n >> 4) | (n << 4);
	y = 12 - (nchannel/2);

	memset(changed_flag, 0, sizeof(changed_flag));

retitle:
	title("\x18/\x19=Select  \x1B/\x1A=speed  PgUp/Dn=speed+  D)ata  P)arity  S)top  N)ames  ESC=Menu");
	wgotoxy(0, 1);
	*W_OPEN = colors[CO_MAIN];
	wcleow();

redraw:
	for(i=0; i < nchannel; ++i) {
		wgotoxy(4, y+i);
		if(p == i)
			*W_OPEN = r;
		wprintf("%2u-%-8s [%04x] :  ", i+1, dlm_channels[i].channel_name, dlm_channels[i].com_port-5);
		wprintf("Speed=%-5u-%-5u  Dbits=%u  Parity=%-5s  Sbits=%u",
			scale(57600, 2, dlm_channels[i].speed),
			dlm_channels[i].speed,
			dlm_channels[i].data,
			parity[dlm_channels[i].parity],
			dlm_channels[i].stop);
		*W_OPEN = n; }

	// Handle input keystrokes to perform editing functions
	for(;;) switch(wgetc()) {
		case _KUA :		// Previous channel
			p = p ? p-1 : nchannel-1;
			goto redraw;
		case _KDA :		// Next channel
			if(++p >= nchannel)
				p = 0;
			goto redraw;
		case _KPU :		// Fast reduce speed
			s = (dlm_channels[p].speed/10) +1;
			dlm_channels[p].speed += (s-1);
		case _KLA :		// Slow reduce speed
			++dlm_channels[p].speed;
			goto check_range;
		case _KPD :		// Fast increase speed
			s = (dlm_channels[p].speed/10) +1;
			if(dlm_channels[p].speed > s)
				dlm_channels[p].speed -= (s-1);
			else
				dlm_channels[p].speed = 3;
		case _KRA :		// Slow increase speed
			--dlm_channels[p].speed;
		check_range:
			if(dlm_channels[p].speed < 2)
				dlm_channels[p].speed = 2;
			else if(dlm_channels[p].speed > 57600)
				dlm_channels[p].speed = 57600;
		changed:
			changed_flag[p] = -1;
			goto redraw;
		case 'd' :
		case 'D' :		// Data bits
			if(++dlm_channels[p].data > 8)
				dlm_channels[p].data = 5;
			goto changed;
		case 'p' :
		case 'P' :		// Parity
			if(++dlm_channels[p].parity > 4)
				dlm_channels[p].parity = 0;
			goto changed;
		case 's' :
		case 'S' :		// Stop bits
			if(++dlm_channels[p].stop > 2)
				dlm_channels[p].stop = 1;
			goto changed;
		case 'n' :
		case 'N' :		// Name editor
			edit_names(p);
			goto retitle;
		case '\n' :
		case 0x1B :		// Exit
			enable_capture = -1;
			for(i=0; i < nchannel; ++i) {
				if(changed_flag[i])
					if(setup_uart(i))
						enable_capture = 0; }
			return; }
}

/*
 * RS-232 signal monitor
 */
void signal_monitor(void)
{
	unsigned i, a, s, y, t, dt[MAX_PORTS];
	unsigned char *p;

	title("SIGNAL MONITOR: ESC=Main-menu");
	wgotoxy(0, 1);
	*W_OPEN = colors[CO_MAIN];
	wcleow();

	y = 12 - (nchannel/2);
	memset(dt, 0, sizeof(dt));

	// Display the channel names/addresses
	for(i=0; i < nchannel; ++i) {
		wgotoxy(5, y+i);
		wprintf("%2u-%-8s [%04x] :", i+1, dlm_channels[i].channel_name,
			dlm_channels[i].com_port - 5); }

	// Loop and update display whenever change detected
	do {
		t = peekw(0x40, 0x6C);
		for(i=0; i < nchannel; ++i) {
			a = dlm_channels[i].com_port;
			s = in(a) << 8;
			s = (s | in(a+1)) & 0x1FF0;
			if(s & 0x100) {			// Data present
				if(!dt[i])
					dt[i] = t;
				else {				// No data - timeout indicator
					if((t - dt[i]) > DATA_HOLD) {
						in(a-5);
						dt[i] = 0; } } }
			p = dlm_channels[i].bit_sigs;
			wgotoxy(25, y+i);
			while(*p) {				// Display all signal indicators
				wputc(' ');
				wputc(' ');
				wputc(*p++);
				wputc(':');
				wputc((s & 0x1000) ? 'Y' : 'n');
				s <<= 1; } } }
	while(wtstc() != 0x1B);
}

/*
 * Load data and embedded configuration from a file
 */
int load_data(char *filename)
{
	unsigned i, s, n;
	FILE *fp;

	// Open file - report failure
	if(!(fp = fopen(filename, "rb")))
		return -1;

	// Read comment
	pptr = comment;
	while(i = getc(fp)) {
		if(i == -1) {
			fclose(fp);
			return -2; }
		*pptr++ = i; }
	*pptr = 0;

	// Read # configured channels
	n = getc(fp);
	if((n < 1) || (n > MAX_PORTS)) {
	bad_file:
		fclose(fp);
		return -2; }

	// Read channel control structure entries
	nchannel = n;
	for(i=0; i < n; ++i) {
		if(fget(dlm_channels[i], STRUCT_SIZE, fp) != STRUCT_SIZE)
			goto bad_file; }

	// Read capture timestamp
	year = getc(fp);
	year |= (getc(fp) << 8);
	month = getc(fp);
	day = getc(fp);
	hour = getc(fp);
	minute = getc(fp);
	second = getc(fp);

	// Read # data capture entries
	s = getc(fp);
	s |= getc(fp) << 8;
	if(s & 0x8000)
		goto bad_file;

	// Read individual data capture entries
	buf_rptr = buf_wptr = 0;
	while(s) {
		if((i = getc(fp)) & 0xFF00) goto bad_file;
		poke(data_seg, buf_wptr+1, i);
		if((i = getc(fp)) & 0xFF00) goto bad_file;
		poke(stat_seg, buf_wptr+1, i);
		if((i = getc(fp)) & 0xFF00) goto bad_file;
		poke(stat_seg, buf_wptr, i);
		if((i = getc(fp)) & 0xFF00) goto bad_file;
		poke(data_seg, buf_wptr, i);
		if((i = getc(fp)) & 0xFF00) goto bad_file;
		poke(time_seg, buf_wptr, i);
		if((i = getc(fp)) & 0xFF00) goto bad_file;
		poke(time_seg, buf_wptr+1, i);
		buf_wptr += 2;
		--s; }

	// Test for End-Of-File - if not, bad format
	if(getc(fp) != -1)
		goto bad_file;

	fclose(fp);

	// Compute screen geometry
	video_spacing = ((unsigned)nchannel) * 160;
	for(i=s=0; i < video_size; ++i) {
		if((i % nchannel) == (nchannel-1)) {
			video_end = s; }
		s += 160; }

	return 0;
}

#ifndef DEMO
/*
 * Save data to file with embedded configuration information
 */
int save_data(char *filename)
{
	unsigned i, s;
	FILE *fp;

	// Open file - report failure
	if(!(fp = fopen(filename, "wb")))
		return -1;

	// Write configuration information to the file
	fputs(comment, fp);					// Comment
	putc(0, fp);						// Terminate comment
	putc(nchannel, fp);					// # of configured channels
	for(i=0; i < nchannel; ++i)			// Channel control structures
		fput(dlm_channels[i], STRUCT_SIZE, fp);

	// Write capture timestamp
	putc(year, fp);
	putc(year >> 8, fp);
	putc(month, fp);
	putc(day, fp);
	putc(hour, fp);
	putc(minute, fp);
	putc(second, fp);

	// Write number of data capture entries
	s = (buf_wptr - buf_rptr)/2;
	putc(s, fp);
	putc(s >> 8, fp);

	// Write individual data capture entries
	i = buf_rptr;
	while(s) {
		putc(peek(data_seg, i+1), fp);
		putc(peek(stat_seg, i+1), fp);
		putc(peek(stat_seg, i), fp);
		putc(peek(data_seg, i), fp);
		putc(peek(time_seg, i), fp);
		putc(peek(time_seg, i+1), fp);
		i += 2;
		--s; }

	fclose(fp);
	return 0;
}

/*
 * Perform disk transfer functions
 */
void disk_transfer(void)
{
	unsigned i;
	static unsigned select = 0;

again:
	title("DISK TRANSFER FUNCTIONS");
	wgotoxy(0, 1);
	*W_OPEN = colors[CO_MAIN];
	wcleow();
	if(wmenu(25, 7, ((unsigned)colors[CO_MENU])|(WSAVE|WCOPEN|WBOX1),
	file_menu, &select))
		return;
	switch(select) {
	case 0 :	// Load configuration
		if(buf_wptr - buf_rptr) {
			if(message("|Capture buffer will be cleared."))
				goto again; }
		wgotoxy(0, 8);
		wputs(edit_prompt);
		if(wform(4, 3, ((unsigned)colors[CO_MENU])|(WSAVE|WCOPEN|WBOX2),
		filename_form, cfgfile))
			goto again;
		wclwin();
		if(read_config())
			goto again;
		return;
	case 1 :	// Load Data
		wgotoxy(0, 8);
		wputs(edit_prompt);
		if(wform(4, 3, ((unsigned)colors[CO_MENU])|(WSAVE|WCOPEN|WBOX2),
		filename_form, datafile))
			goto again;
		switch(load_data(datafile)) {
		case -1 : message("Unable to open: %s", datafile); goto again;
		default : message("Bad file format!"); goto again;
		case 0 : }
		enable_capture = -1;
		for(i=0; i < nchannel; ++i) {
			if(setup_uart(i))
				enable_capture = 0; }
		return;
	case 2 :	// Save Data
		if(buf_wptr == buf_rptr) {
			message("There are no entries in the capture buffer.");
			goto again; }
		wgotoxy(0, 8);
		wputs(edit_prompt);
		if(wform(4, 3, ((unsigned)colors[CO_MENU])|(WSAVE|WCOPEN|WBOX2),
		filename_form, datafile))
			goto again;
		switch(save_data(datafile)) {
		case 0 : return;
		default: message("Unable to open: %s", datafile); goto again; } }
}

/*
 * Edit the comment field
 */
void edit_comment(void)
{
	char temp[sizeof(comment)];

	strcpy(temp, comment);
	wgotoxy(0, 5);
	wputs(edit_prompt);

	*W_OPEN = colors[CO_MENU];
	for(;;) {
		switch(wgets(0, 2, temp, 80)) {
		case '\n' :		// ENTER - accept
			strcpy(comment, temp);
		case 0x1B :		// ESC - cancel
			wcursor_off();
			return; } }
}

/*
 * Create the temp filename with the necessary conversions
 */
void process_tempfile(char *d)
{
	unsigned v;
	char *p, c;
	static unsigned x;

	p = tempfile;
	while(c = *p++) {
		if(c == '%') {
			switch(c = *p++) {
			case 'K' : delete_temp = 0;	continue;	// Keep temps
			case 'C' : v = ++x % 10000;	goto num4;	// 4 digit count
			case 'Y' : v = year;					// 4 digit year
			num4:sprintf(d, "%04u", v);	goto skip;
			case 'E' : v = year % 100;	goto num2;	// 2 digit year
			case 'M' : v = month;		goto num2;	// 2 digit month
			case 'D' : v = day;			goto num2;	// 2 digit day
			case 'H' : v = hour;		goto num2;	// 2 digit hour
			case 'I' : v = minute;		goto num2;	// 2 digit minute
			case 'S' : v = second;		goto num2;	// 2 digit second
			case 'O' : v = (++x) % 100;				// 2 digit count
			num2: sprintf(d, "%02u", v);
			skip: while(*d) ++d;		continue; } }
		*d++ = c; }
	*d = 0;
}

/*
 * Handle the post-processor menu
 */
void process_menu(void)
{
	unsigned i;
	char *plist[MAX_PROCESS+1], tail[128], temp[10], tfile[65];
	char *p, *p1, *p2, c, ff, mf;

	if(!processors) {
		message("No post-processors are defined");
		return; }

	// Build menu
	for(i=0; i < processors; ++i)
		plist[i] = process_list[i].process_name;
	plist[i] = 0;

	// Present menu
	if(wmenu(20, 4, ((unsigned)colors[CO_MENU])|(WSAVE|WCOPEN|WBOX1), plist, &pm_select))
		return;

	// Process the command tail and perform '%' substitutions
	p = process_list[pm_select].process_tail;
	p1 = tail;
	ff = mf = 0;
	while(c = *p++) {
		if(c == '%') switch(*p) {
			case 'S' :		// capture Segment address
				sprintf(p2 = temp, "%u", stat_seg); goto copy;
			case 'R' :		// buffer Read pointer
				sprintf(p2 = temp, "%u", buf_rptr); goto copy;
			case 'W' :		// buffer Write pointer
				sprintf(p2 = temp, "%u", buf_wptr); goto copy;
			case 'D' :		// Data segment address
				sprintf(p2 = temp, "%u", get_ds()); goto copy;
			case '#' :		// # of configured channels
				sprintf(p2 = temp, "%u", nchannel); goto copy;
			case 'C' :		// offset to Channel control structure
				sprintf(p2 = temp, "%u", dlm_channels); goto copy;
			case 'T' :		// offset to capture Timestamp
				sprintf(p2 = temp, "%u", &year); goto copy;
			case 'N' :		// offset to comment/Name
				sprintf(p2 = temp, "%u", comment); goto copy;
			case 'M' :		// tempfile / save Memory
				mf = -1;
			case 'F' :		// tempFile
				if(!ff)
					process_tempfile(tfile);
				ff = -1;
				p2 = tfile;
			copy:
				while(*p2)
				*p1++ = *p2++;
				++p;
				continue;
			case '%' :		// %% (insert 1 '%')
				++p; }
		*p1++ = c; }
	*p1 = 0;

	if(ff)		// File active - save to tempfile
		save_data(tfile);
	wclose();	// Close window before running external
	if(mf)		// MemorySave active - release capture memory
		free_seg(stat_seg);
	// Execute post-processor and get DOS exit status
	i = exec(p = process_list[pm_select].process_file, tail);
	if(mf) {	// MemorySave active - reallocate capture memory & reload
		if(!(stat_seg = alloc_seg(4096*3)))
			abort("Not enough memory");
		data_seg = stat_seg + 4096;
		time_seg = data_seg + 4096;
		load_data(tfile); }
	if(ff && delete_temp)	// File active & delete enabled - delete
		delete(tfile);
	// Re-open window for MDLM screens
	wopen(0, 0, 80, video_size + 1, WSAVE|WCOPEN|NORMAL);
	wcursor_off();
	if(i)					// Report non-zero exit status
		message("%s exit status=%u", p, i);
}
#endif

/*
 * Main program
 */
main(int argc, char *argv[])
{
	unsigned i;
	static unsigned select = 0;
	static char lflag = 0;

	// Allocate capture memory
	if(!(stat_seg = alloc_seg(4096*3)))
		abort("Not enough memory");
	data_seg = stat_seg + 4096;
	time_seg = data_seg + 4096;

	// Process command line arguments
	for(i=1; i < argc; ++i) {
		pptr = argv[i];
		switch((toupper(*pptr++) << 8) | toupper(*pptr++)) {
		case '-C' :
		case '/C' :		// immediate Capture
			enable_capture = -2;
			continue;
		case 'C=' :		// specify Config file
			copy_file(cfgfile, pptr, ".CFG");
			continue;
		case 'D=' :		// load Data file
			copy_file(datafile, pptr, ".DAT");
			lflag = -1;
			continue;
		default:		// Unknown option
			printf("Bad option: %s\n", argv[i]);
		case '?'<<8 :
		case '-?' :		// Help request
		case '/?' : }
		abort(help); }

	// Open main window
	wopen(0, 0, 80, 25, WSAVE|WCOPEN|NORMAL);
	wputs(welcome);
	wputs("\n\n");

	// Read initial configuration
	if(read_config()) {
		wclose();
		exit(-1); }

	// If not 24 lines, close & reopen to resize
	if(video_size != 24) {
		wclose();
		wopen(0, 0, 80, video_size + 1, WSAVE|WCOPEN|NORMAL); }

	wcursor_off();		// turn off cursor
	select_font(0);		// Setup initial font

	// Load data file if specified
	if(lflag) {
		switch(load_data(datafile)) {
		case -1 : message("Unable to open: %s", datafile); break;
		default : message("Bad file format!");
		case 0 : } }

	// Main loop - redraw title and offer main menu
	for(;;) {
#ifdef DEMO
		title("MDLM 1.0 -DEMO- %u entries in buffer.", (buf_wptr - buf_rptr)/2);
#else
		title("MDLM 1.0  ---  %u entries in buffer.", (buf_wptr - buf_rptr)/2);
#endif
		if(!enable_capture) {	// If capture is inhibited - display indicator
			wgotoxy(60, 0);
			wputs("[CAPTURE INHIBITED]"); }
		// Clear remainder of main screen to selected color
		wgotoxy(0, 1);
		*W_OPEN = colors[CO_MAIN];
		wcleow();
		if(*comment) {			// If comment exists - display
			wgotoxy((80-strlen(comment))/2, 2);
			wputs(comment); }
		if(hour) {				// If capture timestamp exists - display
			wgotoxy(25, 20);
			wprintf("Last capture: %2u/%02u/%04u %2u:%02u:%02u", day, month, year, hour, minute, second); }
		if(enable_capture != -2) {	// Unless immediate capture - issue menu
			if(wmenu(28, 6, ((unsigned)colors[CO_MENU])|(WSAVE|WCOPEN|WBOX1), main_menu, &select))
				continue; }
		// If no channels configured - allow only certain functions
		if(!nchannel) switch(select) {
			default: message("There are no I/O ports configured");
				continue;
			case 5 :		// Disk transfer menu
			case 7 :		// Edit comment
			case 8 : }		// Exit
		// Process selected function
		switch(select) {
		case 0 :		// Begin data capture
			if(!enable_capture) {
				message("Serial ports failed init. - Capture inhibited");
				continue; }
			enable_capture = -1;	// Reset immediate capture option
			draw_screen();
			get_date(&day, &month, &year);		// Set capture timestamp
			get_time(&hour, &minute, &second);	// ""
		retitle:
			select_font(font);
			title("CAPTURE: ESC=Main-menu F1=font    %s", font ? font_name[font-1] : "");
		recapture:
			// Loop on poll_capture() until a key is pressed
			do {
				channel = 0;
				poll_capture(); }
			while(!(i = wtstc()));
			switch(i) {		// Process keypress
			case 0x1B :		// ESC - exit to main menu
				review_scr = review_cur = buf_rptr;
				select_font(0);
				continue;
			case _K1 :		// Font selection
				if(++font > font_max)
					font = 0;
				goto retitle; }
			goto recapture;
		case 1 :		// Review captured data
			if(buf_wptr == buf_rptr) {
				message("There are no entries in the capture buffer.");
				continue; }
			select_font(font);
			draw_screen();
			review();
			select_font(0);
			continue;
		case 2 :		// Clear capture buffer
			if(message("|Buffer contains %u entries.", (buf_wptr - buf_rptr)/2))
				continue;
			review_scr = review_cur = buf_rptr = buf_wptr = 0;
			for(i=0; i < nchannel; ++i)
				dlm_channels[i].last_status = -1;
			continue;
		case 3 :			// Signal monitor
			signal_monitor();
			continue;
		case 4 :			// Configure I/O ports
			if(buf_wptr - buf_rptr) {
				if(message("|Capture buffer will be cleared."))
					continue; }
			buf_wptr = buf_rptr = 0;
			configure_ports();
			continue;
		case 5 :			// Disk transfer functions
#ifdef DEMO
			message("Disk transfer menu is not available in DEMO");
#else
			disk_transfer();
#endif
			continue;
		case 6 :			// Post-Processor menu
#ifdef DEMO
			message("Post-processor menu is not available in DEMO");
#else
			if(buf_wptr == buf_rptr) {
				if(message("|There are no entries in the capture buffer."))
					continue; }
			process_menu();
#endif
			continue;
		case 7 :			// Edit comment
#ifdef DEMO
			message("Edit comment function is not available in DEMO");
#else
			edit_comment();
#endif
			continue;
		case 8 :			// Exit to DOS
			wclose();
			return; }
		beep(1000, 250); }
}
