/*
 * -------------------- PACKET DRIVER INTERFACE --------------------
 * cc packets -pofm PKT_INT=96
 * rename packets.obj packet1.obj
 * cc packets -pofm PKT_INT=97
 * rename packets.obj packet2.obj
 */

/* PKT_INT must be in DECIMAL, as it is used by C and inline assembly */
#ifndef PKT_INT
	#define	PKT_INT		96				// Packet driver interrupt (0x60)
	#define	PACKET		pkt				// default access name
#else
	#if PKT_INT == 96
		#define	PACKET	pkt1			// 0x60
	#elif PKT_INT == 97
		#define	PACKET	pkt2			// 0x61
	#elif PKT_INT == 98
		#define	PACKET	pkt3			// 0x62
	#elif PKT_INT == 99
		#define	PACKET	pkt4			// 0x63
	#else
		#error "PKT_INT" must be 96-99
	#endif
#endif

#define	PKT_DATA	242				/* Size of data area in packet */
#define	PKT_SIZE	PKT_DATA+14		/* Size of packet with overhead */
#define	PKT_EASIZE	6				/* Size of ethernet address */
/* PKT_WINDOW must be a power of 2, so we can mask with PKT_WINDOW-1 */
#define	PKT_WINDOW	256				/* Size of packet receive queue */

/*
 * Structure of information returned by pkt_driver_info()
 */
struct PD_driver_info {
	unsigned version;				/* Driver version */
	unsigned char class;			/* Interface class */
	unsigned type;					/* Interface type */
	unsigned char number;			/* Interface number */
	unsigned char functionality;	/* Driver functionality */
	unsigned char name[9]; };		/* Driver name (?) */

/*
 * Structure of packet as sent/received by driver
 */
struct PD_packet {
	unsigned char to_addr[PKT_EASIZE];
	unsigned char from_addr[PKT_EASIZE];
	unsigned ptype;
	unsigned char data[PKT_DATA]; };

/*
 * Structure of a packet receive queue entry
 */
struct PD_queue {
	unsigned handle;
	unsigned length; };

/*
 * Packet driver interface global variables
 */
struct PD_queue
	PACKET/**/_queue[PKT_WINDOW];			/* Queue for received packets */

unsigned
	PACKET/**/_int = PKT_INT,				// Packet interrupt
	PACKET/**/_seg;							// Packet segment
unsigned char
	PACKET/**/_head = 0,					/* Head of RX packet queue */
	PACKET/**/_tail = 0;					/* Tail of RX packet queue */

/*
 * Get our ethernet address
 *
 * Arguments:
 *	handle		= Handle returned by pkt_access_type()
 *	buffer		= Pointer to buffer to receive address
 *	length		= Size of buffer (maximum)
 *	alen		= Pointer to 'int' variable to receive size of address
 *
 * Returns:
 *	0			= Success
 *	!0			= Packet driver error code
 */
int PACKET/**/_get_address(handle, buffer, length, alen) asm
{
		MOV		BX,10[BP]			; Get handle
		MOV		DI,8[BP]			; Get buffer
		MOV		CX,6[BP]			; Get length
		PUSH	DS					; Save our segment
		POP		ES					; Set ES
		MOV		AH,06h				; Indicate "get address" request
pi5:	INT		PKT_INT				; Ask driver
		JC		di2					; Report error
		MOV		BX,4[BP]			; Get received length
		MOV		[BX],CX				; Save length
		XOR		AX,AX				; Indicate success
}

/*
 * Obtain packet driver information
 *
 * This function *MUST* be called before pkt_access_type() is called, in
 * order to insure that the SAVEDS location is initialized with the Micro-C
 * data segment. Another reason is that you need to obtain the interface
 * class/type/number so that you can pass them to pkt_access_type();
 *
 * Arguments:
 *	info_struct	= "PD_driver_info" structure to fill in.
 *
 * Returns:
 *	0			= Success
 *	!0			= Packet driver error code
 */
int PACKET/**/_driver_info(info_struct) asm
{
		MOV		CS:SAVEDS,DS		; Save out data seg
		MOV		AX,01ffh			; Request info
pi1:	INT		PKT_INT				; Ask packet driver
		JC		di2					; Error occured
		PUSH	DS					; Save driver DS
		MOV		DS,CS:SAVEDS		; Restore DATA segment
		POP		ES					; Point to driver data
		MOV		DI,4[BP]			; Get struct
		MOV		0[DI],BX			; Save version
		MOV		2[DI],CH			; Save Class
		MOV		3[DI],DX			; Save type
		MOV		5[DI],CL			; Save number
		MOV		6[DI],AL			; Save functionality
		LEA		BX,7[DI]			; Point to name
di1:	MOV		AL,ES:[SI]			; Get data
		MOV		[BX],AL				; Copy it
		INC		SI					; Advance source
		INC		DI					; Advance dest
		AND		AL,AL				; End of string?
		JNZ		di1					; Copy it all
		XOR		AX,AX				; Return 0
		JMP		di3
; Report driver error
di2:	MOV		AL,DH				; Get error code
		MOV		AH,0FFh				; Insure non-zero
di3:
}

/*
 * Obtain access to a packet type
 *
 * Arguments:
 *	if_class	= Interface class  \
 *	if_type		= Interface type    > - From pkt_driver_info()
 *	if_number	= Interface number /
 *	type		= Pointer to packet type to access
 *	typelen		= Length of packet type (0=Access ALL packets)
 *	handle		= Pointer to 'int' variable to receive handle
 *
 * Returns:
 *	0			= Success
 *	!0			= Packet driver error code
 */
int PACKET/**/_access_type(if_class, if_type, if_number, type, typelen, handle) asm
{
		MOV		AL,14[BP]			; Get if_class
		MOV		BX,12[BP]			; Get if_type
		MOV		DL,10[BP]			; Get if_number
		MOV		SI,8[BP]			; Get type pointer
		MOV		CX,6[BP]			; Get typelen
		MOV		DI,OFFSET pktrdy	; Point to handler
		PUSH	CS					; Save code segment
		POP		ES					; ES = handler segment
		MOV		AH,02h				; Indicate "access type" request
pi2:	INT		PKT_INT				; Call packet driver
		JC		di2					; Report error
		MOV		BX,4[BP]			; Get handle
		MOV		[BX],AX				; Save handle
		XOR		AX,AX				; Indicate success
}

/*
 * Release access to a packet type
 *
 * Arguments:
 *	handle		= Handle returned by pkt_access_type()
 *
 * Returns:
 *	0			= Success
 *	!0			= Packet driver error code
 */
int PACKET/**/_release_type(handle) asm
{
		MOV		BX,4[BP]			; Get handle
		MOV		AH,03h				; Indicate "release type" request
pi3:	INT		PKT_INT				; Ask for it
		JC		di2					; Indicate failed
		XOR		AX,AX				; Indicate success
}

/*
 * Send a packet
 *
 * Arguments:
 *	buffer		= Pointer to buffer to send
 *	length		= Length of buffer to send
 *
 * Returns:
 *	0			= Success
 *	!0			= Packet driver error code
 */
int PACKET/**/_send(buffer, length) asm
{
		MOV		SI,6[BP]			; Get buffer address
		MOV		CX,4[BP]			; Get length
		MOV		AH,04h				; Indicate "send packet" request
		MOV		DS,8[BP]			; Get packet segment
pi4:	INT		PKT_INT				; Ask for it
		MOV		DS,CS:SAVEDS		; restore data segment
		JC		di2					; Report error
		XOR		AX,AX				; Indicate success
}

/*
 * Set the receive mode for a packet
 *
 * Arguments:
 *	handle		= Handle returned by pkt_access_type()
 *	mode		=	1: Turn off receiver
 *					2: Receive only packets sent to our address
 *					3: mode 2 + broadcast packets
 *					4: mode 3 + limited multicast packets
 *					5: mode 3 + all multicast packets
 *					6: all packets
 *
 * Returns:
 *	0			= Success
 *	!0			= Packet driver error code
 */
int PACKET/**/_set_receive_mode(handle, mode) asm
{
		MOV		BX,6[BP]			; Get handle
		MOV		CX,4[BP]			; Get mode
		MOV		AH,20				; Set function
pi6:	INT		PKT_INT				; Ask driver
		JC		di2					; Report error
		XOR		AX,AX				; Report success
}

/*
 * Receive packet handler. Called by packet driver when a packet
 * has been received for us.
 *
 * Check to see if we have enough space, and buffer the packet in our
 * receive packet queue, notify the mainline when received.
 */
asm {
; Entry:
;  AX = flag: 0=Allocate buffer, 1=Transfer complete
;  BX = handle
;  CX = Length
; Exit:
;  ES:DI = Buffer to receive packet (0:0=Not available)
pktrdy:	PUSH	DS					; Save our data segment
		MOV		DS,CS:SAVEDS		; Restore Micro-C DS
		AND		AX,AX				; Is this allocate?
		JNZ		pktr2				; No, try next
; Allocate space for a packet
		CMP		CX,PKT_SIZE			; Is packet too large?
		JA		pktr1				; Yes, Fail reception
		MOV		AL,DGRP:_/**/PACKET/**/_head	; Get head of queue
		INC		AL					; Advance to next
		CMP		AL,DGRP:_/**/PACKET/**/_tail	; Is queue full?
		JZ		pktr1				; Yes, Fail reception
		DEC		AL					; Get head back
		XOR		AH,AH				; Zero high
		MOV		DI,4				; Get size
		MUL		DI					; Compute offset
		ADD		AX,OFFSET DGRP:_/**/PACKET/**/_queue ; Offset to location
		MOV		DI,AX				; Point to location
		MOV		[DI],BX				; Save handle
		MOV		2[DI],CX			; Save length
		MOV		AH,DGRP:_/**/PACKET/**/_head	; Get position
		XOR		AL,AL				; Zero low
		MOV		DI,AX				; Calculate offset
		MOV		ES,DGRP:_/**/PACKET/**/_seg		; Get segment
		POP		DS					; Restore DS
		DB		0CBh				; Far return
; No space for buffer, or packet too large - reject packet
pktr1:	MOV		DI,0				; Indicate no buffer
		MOV		ES,DI				; ""
		POP		DS					; Restore DS
		DB		0CBh				; Far return
; Packet has been copied into our buffer, notify application
pktr2:	INC		DGRP:_/**/PACKET/**/_head	; Advance head pointer
		POP		DS					; Restore data seg
		DB		0CBh				; Far return
SAVEDS	DW		?
}

/*
 * Set the packet interrupt by patching the code
 */
PACKET/**/_set_interrupt(inum) asm
{
		MOV		AL,4[BP]			; Get the interrupt number
		MOV		byte ptr pi1+1,AL	; Patch
		MOV		byte ptr pi2+1,AL	; Patch
		MOV		byte ptr pi3+1,AL	; Patch
		MOV		byte ptr pi4+1,AL	; Patch
		MOV		byte ptr pi5+1,AL	; Patch
		MOV		byte ptr pi6+1,AL	; Patch
}
