/*
 * PCLA - PC based Logic Analyser
 *
 * ?COPY.TXT 2000-2005 Dave Dunfield
 *  -- see COPY.TXT --.
 *
 * cc pcla -fmop
 * masm/ml pclacap;
 * lc pcla pclacap
 */
#include <stdio.h>
#include <window.h>
#include "c:\project\demos\demo.h"

/* General system parameters */
#define LPT_flip	0x80	// Inverted bit mask in LPT+1 input
#define	LPT_test	0x07	// Test bits for preview data
#define	TRIG_BITS	5		// Number of trigger bits
#define	VIEW_WIDTH	75		// Width of viewing area
#define	VIEW_START	1		// Starting row for viewers

/* Offsets for CMOS time data */
#define	SECOND		0
#define	MINUTE		2
#define	HOUR		4
#define	DAY			7
#define	MONTH		8
#define	YEAR		9

/* Meanings for Exitmode */
#define	EXIT_ROUT	0x55	/* Restore to output on exit */
#define	EXIT_RNOW	0xAA	/* Exit and restore to output now */

/* Special characters to output */
#define	WALL		'|'		/* Wall at either end */
#define	BIT_LOW		'_'		/* Bit is LOW */
#define	BIT_HIGH	'-'		/* Bit is high */
#define	BIT_NA		0x01FA	/* Bit is not available */
#define	POS_1		0x01F9	/* Position is /1 only */
#define	POS_5		0x0107	/* Position is /5 */
#define	POS_10		0x01FE	/* Position is /10 */

/* Size of data to save in file operations */
#define	WSIZE		4		// Number of bytes from WORD parms to save
#define	BSIZE		5		// Number of bytes from BYTE parms to save

DEMODEF(xck1)

unsigned
	Seg1,					// Segment for data recording
	Seg2,					// Segment for trigger recording
	Port,					// LPT port
	Delcal,					// Delay loop calibration
	Interval = 1000,		// F: Interval (ms)
	Samples = 1000,			// F: Number of samples to take
	Mdelay = 50,			// Manual mode delay
	Start = -1,				// Starting position in buffer
	End = 0,				// Ending position in buffer
	Rpos = -1;				// Viewing position in review buffer
unsigned char
	*pptr,					// General parsing pointer
	Preview = 0,			// F: Position in preview buffer
	CTmask = 0,				// F: Trigger mask for capture
	CTmatch = 0,			// F: Trigger match for capture
	CDmask = 0,				// F: Data mask for capture
	CDmatch = 0,			// F: Data match for capture
	STmask = 0,				// Trigger mask for search
	STmatch = 0,			// Trigger match for search
	SDmask = 0,				// Data mask for search
	SDmatch = 0,			// Data match for search
	PTmask,					// Trigger mask for pulse counting
	PTmatch,				// Trigger match for pulse counting
	PDmask,					// Data mask for pulse counting
	PDmatch,				// Data match for pulse counting
	Intmode,				// Interrupt stop mode
	Exitmode,				// Exit mode (0=Restore output)
	Debug,					// Debug messages
	Pre1[256],				// Storage for pre-trigger first
	Pre2[256],				// Storage for pre-trigger second
	Ppos,					// Viewing position in preview buffer
	Pdetect,				// Mask to detect preview
	long1[4],				// Long math working register
	long2[4];				// ""

/* Conversion table of bit index to bit mask values */
static unsigned char imask[] =
	{ 0x80, 0x40, 0x20, 0x10, 0x08, 0x04, 0x02, 0x01 };

#define	CPROMPT		0		// Color index for prompt bar
#define	CMAIN		1		// Color index for main window
#define	CMESSAGE	2		// Color index for message window
#define	CPREVIEW	3		// Color index for preview
#define	CREVIEW		4		// Color index for review
#define	CMENU		5		// Color index for pop-up menus

unsigned char C_color[] = {
	0x70,					/* Prompt bar at bottom */
	0x17,					/* Main view screen */
	0x67,					/* Message window */
	0x47,					/* Preview window */
	0x07,					/* Review window */
	0x70 };					/* Pop-up menues */
unsigned char M_color[] = {
	0x70,					/* Prompt bar at bottom */
	0x07,					/* Main view screen */
	0x07,					/* Message window */
	0x70,					/* Preview window */
	0x07,					/* Review window */
	0x70 };					/* Pop-up menus */
unsigned char *color = &C_color;

DEMODEF(xck2)

struct WINDOW
	*swin,					/* Status window */
	*pwin,					/* Prompt window */
	*mwin;					/* Main window */

char *setup_form[] = {
	32 << 8 | 4,
	"\x01\x00\x85Number of samples    :",
	"\x01\x01\x85Sampling interval(us):",
	0 };
char *monitor_form[] = {
	31 << 8 | 3,
	"\x01\x00\x85Monitor interval(ms):",
	0 };
#ifndef DEMO
char *file_form[] = {
	76 << 8 | 3,
	"\x00\x00\x41Filename?",
	0 };
char *file_menu[] = {
	"Load PCLA image",
	"Save PCLA image",
	0 };
#endif

char Done[] = { "Done!" };
char Canceled[] = { "Canceled!" };

/*
 * Prompts for the prompt bar
 */
char main_prompt[] =
	{ " C)apture  F)ile  M)onitor  P)review  R)eview  S)etup  T)rigger   F10:Exit" };
char review_prompt[] =
/*
 xx:+/-1 yy:+/-10 Pu/Pd:+/-74 Home/End F1:Search F2:Again F3:Count ESC/F10:Exit
*/
	{ " \x1B\x1A:+/-1 \x18\x19:+/-10 Pu/Pd:+/-74 Home/End F1:Search F2:Again F3:Count ESC/F10:Exit" };
char preview_prompt[] =
	{ " \x1B\x1A:+/-1  \x18\x19:+/-10  Home/End  F1:Search  F2:Again  ESC/F10:Exit" };
char bit_prompt[] =
	{" Press '0-7'/'A-E' to toggle X/0/1, ESC to cancel, ENTER/F10 to save." };

static char *bit_select[] = {
	"X - Don't Care",
	"0 - Zero",
	"1 - One" };
static char *bit_change[] = {
	"X - Don't Care",
	"* - Do care",
	"* - Do care" };

/*
 * Command line help text
 */
char help[] = { "\n\
Use: 	PCLA [options]\n\n\
opts:	/D	- Disable unused entry detection in preview\n\
	/I	- Disable interrupts during capture\n\
	/I*	- \"\" but do not reset clock\n\
	/M	- Force monochrome screens\n\
	/O	- Revert LPT DATA to outputs on exit\n\
	/O*	- \"\" but exit immediately (no interactive)\n\
	/W	- Display wiring information\n\
	I=n	- Set default capture interval (us)\n\
	M=n	- Set default monitor update interval(ms)\n\
	P=n/adr	- Select LPT port (1-3) or address\n\
	S=n	- Set default capture size\n\n\
?COPY.TXT 2000-2005 Dave Dunfield.\n -- see COPY.TXT --.\n" };

DEMODEF(xck3)

char wire[] = { "\n\
PCLA connections for standard DB-25 PC parallel port\n\n\
Pin Signal	Pin Signal\n\
-----------------------------\n\
 2 -Data 0	15 -Data A\n\
 3 -Data 1	13 -Data B\n\
 4 -Data 2	12 -Data C\n\
 5 -Data 3	10 -Data D\n\
 6 -Data 4	11 -Data E\n\
 7 -Data 5\n\
 8 -Data 6\n\
 9 -Data 7   18-25 -Ground\n" };

/*
 * Display a number as 1-65535 (instead of 0-65535)
 */
char *xnum(unsigned n)
{
	static char buffer[6];
	if(n)
		sprintf(buffer, "%u", n);
	else
		strcpy(buffer, "65536");
	return buffer;
}

/*
 * Formatted print to status window
 *
 * Input: Format string, Variable argument list
 */
static void register status(unsigned args)
{
	unsigned i;
	unsigned char d, buffer[81];

	_format_(nargs() * 2 + &args, buffer);
	w_putc('\n', swin);
	w_puts(buffer, swin);
	w_cleol(swin);

	w_gotoxy(48, 0, swin);
	w_puts("T:", swin);
	for(i=0; i < TRIG_BITS; ++i) {
		if(imask[i] & CTmask)
			d = (imask[i] & CTmatch) ? '1' : '0';
		else
			d = (imask[i] & CTmatch) ? '?' : 'X';
		w_putc(d, swin); }
	w_putc('-', swin);
	for(i=0; i < 8; ++i) {
		if(imask[i] & CDmask)
			d = (imask[i] & CDmatch) ? '1' : '0';
		else
			d = (imask[i] & CDmatch) ? '?' : 'X';
		w_putc(d, swin); }
	w_printf(swin, " S:%-5s P:%04x", xnum(End-Start), Port);
}

/*
 * Formatted print to message window
 *
 * Input: Format string, Variable argument list
 */
static void register message(unsigned args)
{
	unsigned char buffer[81];

	_format_(nargs() * 2 + &args, buffer);
	w_puts(buffer, mwin);
}

/*
 * Issue a prompt
 */
satic void prompt(char *s)
{
	w_clwin(pwin);
	w_puts(s, pwin);
}

/*
 * Prompt for bit match pattern for searches
 */
static int bit_pattern(unsigned char *op, unsigned char Maxval, char *bit_desc[])
{
	unsigned i;
	unsigned char d, Tp[TRIG_BITS], Dp[8];
	unsigned char Tmask, Tmatch, Dmask, Dmatch;

	Tmask = *op;
	Tmatch = *++op;
	Dmask = *++op;
	Dmatch = *++op;

	wopen(28, 4, 21, TRIG_BITS+11, WSAVE|WCOPEN|WBOX1|color[CMENU]);
	prompt(bit_prompt);

	for(i=0; i < TRIG_BITS; ++i) {	/* Decode trigger bits */
		d = 0;
		if(imask[i] & Tmask) {
			d = 1;
			if(imask[i] & Tmatch)
				d = 2; }
		Tp[i] = d; }
	for(i=0; i < 8; ++i) {			/* Decode data bits */
		d = 0;
		if(imask[i] & Dmask) {
			d = 1;
			if(imask[i] & Dmatch)
				d = 2; }
		Dp[i] = d; }

	/* Mask Match
	 *  1     1   = 1
     *  1     0   = 0
     *  0     0   = X */
	for(;;) {
		for(i=0; i < TRIG_BITS; ++i) {
			wgotoxy(1, i);
			wprintf("%c: %s", 'E'-i, bit_desc[Tp[i]]);
			wcleol(); }
		for(i=0; i < 8; ++i) {
			wgotoxy(1, (TRIG_BITS+1)+i);
			wprintf("%c: %s", '7'-i, bit_desc[Dp[i]]);
			wcleol(); }
		switch(d = toupper(wgetc())) {
			case '\n' :
			case _K10 :
				Tmask = Tmatch = Dmask = Dmatch = 0;
				for(i=0; i < TRIG_BITS; ++i) {
					switch(Tp[i]) {
						case 2 :
							Tmatch |= imask[i];
						case 1 :
							Tmask  |= imask[i]; } }
				for(i=0; i < 8; ++i) {
					switch(Dp[i]) {
						case 2 :
							Dmatch |= imask[i];
						case 1 :
							Dmask  |= imask[i]; } }
				wclose();
				*op = Dmatch;
				*--op = Dmask;
				*--op = Tmatch;
				*--op = Tmask;
				return -1;
			case 0x1B :
				message(Canceled);
				wclose();
				return 0;
			default:
				continue;
			case '7' :
			case '6' :
			case '5' :
			case '4' :
			case '3' :
			case '2' :
			case '1' :
			case '0' :
				i = 7 - (d - '0');
				if(++Dp[i] > Maxval)
					Dp[i] = 0;
				continue;
			case 'E' :
			case 'D' :
			case 'C' :
			case 'B' :
			case 'A' :
				i = TRIG_BITS - (d - ('A'-1));
				if(++Tp[i] > Maxval)
					Tp[i] = 0; } }
}

/*
 * View data in the review buffers
 */
static void review()
{
	unsigned i, j, k;
	unsigned char d, t, buffer[10];

	wopen(0, VIEW_START, VIEW_WIDTH+5, 16, WSAVE|WCOPEN|WBOX1|color[CREVIEW]);
	status("Review");

reprompt:
	prompt(review_prompt);
	for(;;) {
		for(i=0; i < TRIG_BITS; ++i) {
			wgotoxy(0, i);
			wputc('E'-i);
			wputc(WALL);
			k = Rpos;
			for(j=0; j < VIEW_WIDTH; ++j) {
				if((k == End) && j) {
					wputc(BIT_NA);
					continue; }
				d = peek(Seg2, k++) ^ LPT_flip;
				wputc((imask[i] & d) ? BIT_HIGH : BIT_LOW); }
//			wgotoxy(VIEW_WIDTH+2, i);
			wputc(WALL); }

		for(i=0; i < 8; ++i) {
			wgotoxy(0, i+(TRIG_BITS+1));
			wputc('7'-i);
			wputc(WALL);
			k = Rpos;
			for(j=0; j < VIEW_WIDTH; ++j) {
				if((k == End) && j) {
					wputc(BIT_NA);
					continue; }
				d = peek(Seg1, k++);
				wputc((imask[i] & d) ? BIT_HIGH : BIT_LOW); }
//			wgotoxy(VIEW_WIDTH+2, i+(TRIG_BITS+1));
			wputc(WALL); }

		wgotoxy(2, TRIG_BITS);
		k = (Rpos - Start)+1;
		for(i=0; i < VIEW_WIDTH; ++i) {
			j = k + i;
			if(!(j % 5))
				wputc((j % 10) ? POS_5 : POS_10);
			else
				wputc(POS_1); }
		wgotoxy(1, TRIG_BITS);
		wputs(xnum(k));
		strcpy(buffer, xnum(k + (VIEW_WIDTH-1)));
		wgotoxy((VIEW_WIDTH+3)-strlen(buffer), TRIG_BITS);
		wputs(buffer);

		switch(d = wgetc()) {
			case _KRA :
				if(Rpos < (End-(VIEW_WIDTH-1)))
					++Rpos;
				continue;
			case _KLA :
				if(Rpos > Start)
					--Rpos;
				continue;
			case _KUA :
				for(i=0; i < 10; ++i)
					if(Rpos < (End-(VIEW_WIDTH-1)))
						++Rpos;
				continue;
			case _KDA :
				for(i=0; i < 10; ++i)
					if(Rpos > Start)
						--Rpos;
				continue;
			case _KPU :
				for(i=0; i < (VIEW_WIDTH-1); ++i)
					if(Rpos < (End-(VIEW_WIDTH-1)))
						++Rpos;
				continue;
			case _KEN :
				Rpos = End-1;
			case _KPD :
				for(i=0; i < (VIEW_WIDTH-1); ++i)
					if(Rpos > Start)
						--Rpos;
				continue;
			case _KHO :
				Rpos = Start;
				continue;
			case _K2 :
				message("\nSearch again ");
				i = Rpos + 1;
				if(i == End)
					continue;
				goto dosearch;
			case _K1 :
				message("\nSearch ");
				if(!bit_pattern(&STmask, 2, bit_select))
					goto reprompt;
				i = Rpos;
			dosearch:
				message("from %s...", xnum((i-Start)+1));
				do {
					t = (peek(Seg2, i) ^ LPT_flip) & STmask;
					d = peek(Seg1, i) & SDmask;
					if((t == STmatch) && (d == SDmatch))
						goto found; }
				while(++i != End);
				message(" Not found!");
				goto reprompt;
			found:
				Rpos = i;
				message(" Found at %s", xnum((i-Start)+1));
				goto reprompt;
			case _K3 :
				message("\nState changes ");
				if(!bit_pattern(&PTmask, 1, bit_change))
					goto reprompt;
				i = Rpos;
				PTmatch = peek(Seg2, i) & PTmask;
				PDmatch = peek(Seg1, i) & PDmask;
				j = 0;
				while(++i != End) {
					t = peek(Seg2, i) & PTmask;
					d = peek(Seg1, i) & PDmask;
					if((t != PTmatch) || (d != PDmatch)) {
						PTmatch = t;
						PDmatch = d;
						++j; } }
				message("from %s to end: %u", xnum((Rpos-Start)+1), j);
				goto reprompt;
			case 0x1B :
			case _K10 :
				wclose();
				return; } }
}

/*
 * View data in the preview buffers
 */
static void preview()
{
	unsigned i, j;
	unsigned char d, t, k, buffer[10];

	wopen(0, VIEW_START, VIEW_WIDTH+5, 16, WSAVE|WCOPEN|WBOX1|color[CPREVIEW]);
	status("Preview");

reprompt:
	prompt(preview_prompt);
	for(;;) {
		for(i=0; i < TRIG_BITS; ++i) {
			wgotoxy(0, i);
			wputc('E'-i);
			wputc(WALL);
			k = (Preview - Ppos)-VIEW_WIDTH;
			for(j=0; j < VIEW_WIDTH; ++j) {
				d = Pre2[k++] ^ LPT_flip;
				if((d & LPT_test) == Pdetect) {
					wputc(BIT_NA);
					continue; }
				wputc((imask[i] & d) ? BIT_HIGH : BIT_LOW); }
//			wgotoxy(VIEW_WIDTH+2, i);
			wputc(WALL); }

		for(i=0; i < 8; ++i) {
			wgotoxy(0, i+(TRIG_BITS+1));
			wputc('7'-i);
			wputc(WALL);
			k = (Preview - Ppos)-VIEW_WIDTH;
			for(j=0; j < VIEW_WIDTH; ++j) {
				d = Pre1[k];
				if((Pre2[k++] & LPT_test) == Pdetect) {
					wputc(BIT_NA);
					continue; }
				wputc((imask[i] & d) ? BIT_HIGH : BIT_LOW); }
//			wgotoxy(VIEW_WIDTH+2, i+(TRIG_BITS+1));
			wputc(WALL); }

		wgotoxy(2, TRIG_BITS);
		k = (Ppos + VIEW_WIDTH)-1;
		j = k+1;
		for(i=0; i < VIEW_WIDTH; ++i) {
			if(!(--j % 5))
				wputc((j % 10) ? POS_5 : POS_10);
			else
				wputc(POS_1); }
		wgotoxy(1, TRIG_BITS);
		wprintf("%d", k);
		sprintf(buffer, "%d", Ppos);
		wgotoxy((VIEW_WIDTH+3)-strlen(buffer), TRIG_BITS);
		wputs(buffer);

		switch(d = wgetc()) {
			case _KUA :
				if(Ppos >= 10) {
					Ppos -= 10;
					continue; }
			case _KHO :
				Ppos = 0;
				continue;
			case _KRA :
				if(Ppos)
					--Ppos;
				continue;
			case _KDA :
				if(Ppos < ((256-10)-VIEW_WIDTH)) {
					Ppos += 10;
					continue; }
			case _KEN :
				Ppos = 256-VIEW_WIDTH;
				continue;
			case _KLA :
				if(Ppos != (256-VIEW_WIDTH))
					++Ppos;
				continue;
			case _K2 :
				message("\nSearch again ");
				i =  Ppos + 1;
				goto dosearch;
			case _K1 :
				message("\nSearch ");
				if(!bit_pattern(&STmask, 2, bit_select))
					goto reprompt;
				i = Ppos;
			dosearch:
				message("from %u...", i);
				do {
					k = (Preview - i)-1;
					t = (Pre2[k] ^ LPT_flip) & STmask;
					d = Pre1[k] & SDmask;
					if((t == STmatch) && (d == SDmatch))
						goto found;
					++i; }
				while(i < 256);
				message(" Not found!");
				goto reprompt;
			found:
				Ppos = i;
				message(" Found at %u", i);
				goto reprompt;
			case 0x1B :
			case _K10 :
				wclose();
				return; } }
}

/*
 * Read the CMOS time/date  into a buffer
 */
static void read_cmos(buffer) asm
{
rdcm1:	CLI
		MOV		AL,10			; Read Register A
		OUT		70h,AL			; Write it
		NOP
		IN		AL,71h			; Read the data
		AND		AL,10000000b	; Ready?
		JZ		rdcm2			; Yes, handle it
		STI						; Re-allow interrupts
		JMP		SHORT rdcm1		; And continue
rdcm2:	XOR		AH,AH			; Start with zero
		MOV		BX,4[BP]		; Point to buffer
rdcm3:	MOV		AL,AH			; Get address
		OUT		70h,AL			; Write to device
		INC		AH				; Advance to next
		IN		AL,71h			; Read the data
		MOV		[BX],AL			; Write it
		INC		BX				; And advance
		CMP		AH,10			; Are we at end?
		JB		rdcm3			; Do them all
		STI						; Re-enable interrupts
; Convert read values from BCD to binary
		MOV		BX,4[BP]		; Point to buffer again
		MOV		CL,10			; Convert 4 values
rdcm4:	MOV		AL,[BX]			; Get value
		MOV		AH,AL			; Save for later
		AND		AH,0Fh			; Save only low
		SHR		AL,1			; Shift over
		SHR		AL,1			; ""
		SHR		AL,1			; ""
		SHR		AL,1			; ""
		MOV		CH,AL			; Copy
		ADD		AL,AL			; *2
		ADD		AL,AL			; *4
		ADD		AL,CH			; *5
		ADD		AL,AL			; *10
		ADD		AL,AH			; Include low digit
		MOV		[BX],AL			; Rewrite it
		INC		BX				; Advance to next
		DEC		CL				; Reduce count
		JNZ		rdcm4			; Do them all
}

/*
 * Show the current CMOS and PC time with a heading
 */
static void show_time(char *p)
{
	unsigned char buffer[10];
	unsigned y, m, d, h, mi, s;

	read_cmos(buffer);
	get_time(&h, &mi, &s);
	get_date(&d, &m, &y);
	y %= 100;
	message("\n%s time = CMOS: %u:%02u:%02u %u/%02u/%02u  PC: %u:%02u:%02u %u/%02u/%02u",
		p,
		buffer[HOUR], buffer[MINUTE], buffer[SECOND],
		buffer[DAY], buffer[MONTH], buffer[YEAR],
		h, mi, s, d, m, y % 100);
}

/*
 * Update system time from CMOS clock time
 */
static void update_time()
{
	unsigned char buffer[10];
	unsigned year, month, day, hour, minute, second;

	if(Debug)
		show_time("Before");
	for(;;) {
		read_cmos(buffer);
		year = buffer[YEAR];
		month = buffer[MONTH];
		day = buffer[DAY];
		hour = buffer[HOUR];
		minute = buffer[MINUTE];
		second = buffer[SECOND];
		if((hour < 23) || (minute < 59) || (second < 58)) {
			set_date(day, month, year+2000);
			set_time(hour, minute, second);
			if(Debug)
				show_time("After ");
			return; }
		delay(2500); }
}

/*
 * Local display (Monitor mode), wait for key
 */
static int local_display()
{
	unsigned i, j, l, t;
	unsigned char d, p, k, p1[256], p2[256];

	DEMOCHECK(xck3)
	l = VIEW_WIDTH;
	memset(p1, 0, sizeof(p1));
	memset(p2, 0, sizeof(p2));
	p = 0;
	t = peek(0x40, 0x6C);
	for(;;) {
		p1[p] = in(Port);
		p2[p++] = in(Port+1) ^ LPT_flip;

		for(i=0; i < TRIG_BITS; ++i) {
			wgotoxy(1, i+1);
			wputc('E'-i);
			wputc(WALL);
			k = p-VIEW_WIDTH;
			for(j=0; j < VIEW_WIDTH; ++j) {
				d = p2[k++];
				if(j < l) {
					wputc(BIT_NA);
					continue; }
				wputc((imask[i] & d) ? BIT_HIGH : BIT_LOW); }
			wgotoxy(VIEW_WIDTH+3, i+1);
			wputc(WALL); }

		for(i=0; i < 8; ++i) {
			wgotoxy(1, i+(TRIG_BITS+2));
			wputc('7'-i);
			wputc(WALL);
			k = p-VIEW_WIDTH;
			for(j=0; j < VIEW_WIDTH; ++j) {
				d = p1[k++];
				if(j < l) {
					wputc(BIT_NA);
					continue; }
				wputc((imask[i] & d) ? BIT_HIGH : BIT_LOW); }
			wgotoxy(VIEW_WIDTH+3, i+(TRIG_BITS+2));
			wputc(WALL); }
		if(l)
			--l;

		if(!(i = (Mdelay+27) / 55))
			i = 1;
		do {
			do {
				j = peek(0x40, 0x6C); }
			while(j == t);
			t = j;
			if(j = wtstc())
				return j; }
		while(--i); }
}

/*
 * Calibrate the delay loop
 */
static unsigned calibrate() asm
{
		MOV		AX,40h			; Point to BIOS segment
		MOV		ES,AX			; Set segment
		MOV		SI,6Ch			; Set offset
		MOV		AL,ES:[SI]		; Get initial value
		XOR		CX,CX			; Zero counter
		MOV		DX,DGRP:_Port	; Get port name
cal1:	MOV		BL,ES:[SI]		; Get next value
		CMP		AL,BL			; Has it changed?
		JZ		cal1			; Wait till it changes
cal2:	IN		AX,DX			; Read I/O port
		INC		CX				; Advance counter
cal3:	MOV		AL,ES:[SI]		; Get count
		CMP		AL,BL			; Does it match?
		JZ		cal2			; Wait for change
		MOV		AX,CX			; Get count value
}

/*
 * Perform file operations
 */
static void file()
{
#ifndef DEMO
	unsigned i, t, d;
	FILE *fp;
	static char filename[66] = { "PCLA.DAT" };

	status("File");
	i = 0;
	message("\nFile ");
	if(wmenu(30, 8, WSAVE|WCOPEN|WBOX1|color[CMENU], file_menu, &i)) {
		message(Canceled);
		return; }

	message(i ? "Save " : "Load ");

	if(wform(2, 8, WSAVE|WCOPEN|WBOX1|color[CMENU], file_form, filename)) {
		message(Canceled);
		return; }

	message("%s... ", filename);

	if(!i) {
		if(!(fp = fopen(filename, "rb"))) {
			message("Unable to open!");
			return; }
		if(fget(&Interval, WSIZE, fp) != WSIZE)
			goto read_error;
		if(fget(&Preview, BSIZE, fp) != BSIZE)
			goto read_error;
		Ppos = 0;
		if(fget(Pre2, sizeof(Pre2), fp) != sizeof(Pre2))
			goto read_error;
		if(fget(Pre1, sizeof(Pre1), fp) != sizeof(Pre1))
			goto read_error;
		if(fget(&i, sizeof(i), fp) != 2)
			goto read_error;
		Start = Rpos = i = -i;
		End = 0;
		do {
			if((t = getc(fp)) & 0xFF00)
				goto read_error;
			if((d = getc(fp)) & 0xFF00)
				goto read_error;
			poke(Seg2, i, t);
			poke(Seg1, i, d); }
		while(++i);
		if(getc(fp) != -1)
			message("Extra data in file - ignored ");
		fclose(fp);
		message(Done);
		return;
	read_error:
		message("Invalid file format!");
		fclose(fp);
		return; }

	/* Write data to file */
	if(!(fp = fopen(filename, "wb"))) {
		message("Unable to open!");
		return; }
	fput(&Interval, WSIZE, fp);		/* Write WORD parameter data */
	fput(&Preview, BSIZE, fp);		/* Write BYTE parameter data */
	fput(Pre2, sizeof(Pre2), fp);	/* Write trigger preview */
	fput(Pre1, sizeof(Pre1), fp);	/* Write data preview */
	i = End - Start;
	fput(&i, sizeof(i), fp);		/* Write size of buffer */
	i = Start;
	do {
		putc(peek(Seg2, i), fp);	/* Write trigger information */
		putc(peek(Seg1, i), fp); }	/* Write data informatino */
	while(++i);
	fclose(fp);
	message(Done);
#else
	DEMOCHECK(xck2);
	message("\nFILE functions are not provided in the DEMO version");
#endif
}

/*
 * Main program
 */
int main(int argc, char *argv[])
{
	unsigned i, j;
	unsigned char t, d;

	/* Parse command line options */
	Port = peekw(0x40, 0x08);		/* Get LPT1 address */
	for(i=1; i < argc; ++i) {
		pptr = argv[i];
		switch((toupper(*pptr++) << 8) | toupper(*pptr++)) {
		case ('-'<<8)|'D' :		/* Disable bad entry detection */
		case ('/'<<8)|'D' :
			Pdetect = LPT_flip;
			continue;
		case ('-'<<8)|'I' :		/* Interrupt mode */
		case ('/'<<8)|'I' :
			Intmode = (*pptr == '*') ? 1 : 2;
			continue;
		case ('-'<<8)|'M' :		/* Monochrome mode */
		case ('/'<<8)|'M' :
			color = M_color;
			continue;
		case ('-'<<8)|'O' :		/* Revert to output */
		case ('/'<<8)|'O' :
			Exitmode = (*pptr == '*') ? EXIT_RNOW : EXIT_ROUT;
			continue;
		case ('-'<<8)|'W' :		/* Wiring information */
		case ('/'<<8)|'W' :
			fputs(wire, stdout);
			return;
		case ('$'<<8)|'D' :		/* Enable Debug messages */
			Debug = -1;
			continue;
		case ('P'<<8)|'=' :
			if(!(Port = atox(pptr))) {
				fputs("LPT must be 1-3 or address\n", stderr);
				goto gohelp; }
			if(Port < 4)
				Port = peekw(0x40, (Port*2)+6);
			continue;
		case ('M'<<8)|'=' :		/* Set monitor delay */
			Mdelay = atoi(pptr);
			continue;
		case ('S'<<8)|'=' :		/* Set number of samples */
			Samples = atoi(pptr);
			continue;
		case ('I'<<8)|'=' :		/* Set capture interval */
			Interval = atoi(pptr);
			continue;
		default:
			fprintf(stderr,"Invalid option: %s\n", argv[i]);
		case ('?'<<8):
		case ('-'<<8)|'?':
		case ('/'<<8)|'?':
		gohelp:
			abort(help); } }

	if(!Port) {
		fputs("LPT port not found!\n", stderr);
		goto gohelp; }

	if(Exitmode == EXIT_RNOW) {
		out(Port+2, 0x0C);
		return; }

	/* Configure port for input */
	out(Port+2, 0x2C);		/* Set as input port */
	out(Port, -1);			/* Drive all lines high */

	/* Allocate data storage buffers (2 * 64 = 128k) */
	if(!(Seg1 = alloc_seg(8192)))
		abort("Not enough memory");
	Seg2 = Seg1 + 4096;

	/* Open windows */
	DEMOCHECK(xck1)
	swin = wopen(0, 0, 80, 1, WSAVE|WCOPEN|REVERSE);
	if(W_BASE == 0xB000)
		color = M_color;
	pwin = wopen(0,24, 80, 1, WSAVE|WCOPEN|color[CPROMPT]);
	mwin = wopen(0, 17, 80, 23-16, WSAVE|WCOPEN|WSCROLL|color[CMESSAGE]);
	wopen(0, 1, 80,16, WSAVE|WCOPEN|color[CMAIN]);
	wcursor_off();
#ifdef DEMO
	message("PCLA v1.0 (Demo) - ?COPY.TXT 2000-2005 Dave Dunfield -  -- see COPY.TXT --.");
#else
	message("PCLA version 1.0 - ?COPY.TXT 2000-2005 Dave Dunfield -  -- see COPY.TXT --.");
#endif

	switch(Port) {
		default: message("\nUsing non-standard LPT address $%04x", Port);
		case 0x3BC :
		case 0x378 :
		case 0x278 : }

	Delcal = calibrate();
	Pdetect |= (~in(Port+1)) & LPT_test;
	if(Debug)
		message("\nSeg1=%u Seg2=%u Delcal=%u Pdetect=%04x", Seg1, Seg2,
			Delcal, Pdetect);

	longset(long1, 55555);
	longset(long2, 100);
	longmul(long1, long2);
	longset(long2, Delcal);
	longdiv(long1, long2);
	i = *(unsigned*)long1;
	longset(long1, 10000);
	longset(long2, 10);
	longmul(long1, long2);
	longset(long2, i);
	longdiv(long1, long2);
	message("\nHighest sample rate is %u.%02uus (%ukhz)",
		i/100, i%100, *(unsigned*)long1);


	memset(Pre2, Pdetect | LPT_flip, sizeof(Pre2));
	i=0;
	do {
		poke(Seg1, i, 0);
		poke(Seg2, i, LPT_flip);  }
	while(++i);

restat:
	status("Monitoring");
reprompt:
	prompt(main_prompt);
	for(;;) {
		switch(toupper(local_display())) {
		case 'R' :			/* Review data */
			review();
			goto restat;
		case 'P' :			/* Preview data */
			preview();
			goto restat;
		case 'T' :			/* Set trigger */
			message("\nSet Trigger... ");
			if(bit_pattern(&CTmask, 2, bit_select))
				message(Done);
			goto restat;
		case 'S' :			/* Setup capture */
			message("\nCapture Setup... ");
			i = Samples;
			j = Interval;
			if(wform(25, 7, WSAVE|WCOPEN|WBOX1|color[CMENU], setup_form, &Samples, &Interval)) {
				Samples = i;
				Interval = j;
				message(Canceled);
				goto reprompt; }
#ifdef DEMO
			if(Samples > 1000) {
				message("\nDEMO limits samples to <= 1000 ");
				Samples = 1000; }
			if(Interval < 100) {
				message("\nDEMO limits interval to >= 100us ");
				Interval = 100; }
#endif
			message(Done);
			goto reprompt;
		case 'M' :			/* Monitor interval */
			i = Mdelay;
			message("\nMonitor Interval... ");
			if(wform(25, 7, WSAVE|WCOPEN|WBOX1|color[CMENU], monitor_form, &Mdelay)) {
				Mdelay = i;
				message(Canceled);
				goto reprompt; }
			message(Done);
			goto reprompt;
		case 'C':			/* Begin capture */
			message("\nCapture... ");
			Start = -Samples;
			longset(long1, Interval);			// Desire 100 us
			longset(long2, Delcal);				// Loops for 50ms
			longmul(long1, long2);				// User time * 50ms
			longset(long2, 55555);				// 50ms
			longdiv(long1, long2);				// User time in ms
			i = *(unsigned*)long1;
			if(Debug)
				message("Delay loop count: %u ", i);
			status("[T] Trigger/Acquire %s samples, at %uus", xnum(Samples), Interval);
			prompt(" CAPTURE MODE: Press ESC to abort...");
			memset(Pre1, 0, sizeof(Pre1));
			memset(Pre2, Pdetect | LPT_flip, sizeof(Pre2));
			t = (CTmatch ^ LPT_flip) & CTmask;
			d = (CDmatch ^ LPT_flip) & CDmask;
			capture((t << 8) | d, i);
			Rpos = Start;
			Ppos = 0;
			message(Done);
			if(Intmode == 2)
				update_time();
			goto restat;
		case 'F' :
			file();
			goto restat;
		case _K10 :
			goto endprog; } }

endprog:
	wclose();
	wclose();
	wclose();
	wclose();
	free_seg(Seg1);
	if(Exitmode == EXIT_ROUT)
		out(Port+2, 0x0C);
}
