/*
 * Packet load generator
 *
 * ?COPY.TXT 2003-2005 Dave Dunfield -  -- see COPY.TXT --.
 *
 * Compile command: CC pktlg -pof
 */

#include <stdio.h>
#include <window.h>

/*
 * -------------------- PACKET DRIVER INTERFACE --------------------
 */

/* PKT_INT must be in DECIMAL, as it is used by C and inline assembly */
#define	PKT_INT		96				/* Packet driver interrupt (0x60) */
#define	PKT_DATA	500				/* Size of data area in packet */
#define	PKT_SIZE	PKT_DATA+14		/* Size of packet with overhead */
#define	PKT_EASIZE	6				/* Size of ethernet address */
/* PKT_WINDOW must be a power of 2, so we can mask with PKT_WINDOW-1 */
#define	PKT_WINDOW	64				/* Size of packet receive queue */
#define	MAX_NODES	20				// Number of nodes supported

/*
 * Structure of information returned by pkt_driver_info()
 */
struct PD_driver_info {
	unsigned version;				/* Driver version */
	unsigned char class;			/* Interface class */
	unsigned type;					/* Interface type */
	unsigned char number;			/* Interface number */
	unsigned char functionality;	/* Driver functionality */
	unsigned char name[9]; };		/* Driver name (?) */

/*
 * Structure of packet as sent/received by driver
 */
struct PD_packet {
	unsigned char to_addr[PKT_EASIZE];
	unsigned char from_addr[PKT_EASIZE];
	unsigned ptype;
	unsigned char data[PKT_DATA]; };

/*
 * Structure of a packet receive queue entry
 */
struct PD_queue {
	unsigned handle;
	unsigned length;
	struct PD_packet packet; };

/*
 * Packet driver interface global variables
 */
struct PD_queue
	pkt_queue[PKT_WINDOW];			/* Queue for received packets */

unsigned
	Pkt_int = PKT_INT,				// Packet interrupt
	pkt_head = 0,					/* Head of RX packet queue */
	pkt_tail = 0;					/* Tail of RX packet queue */

/*
 * Obtain packet driver information
 *
 * This function *MUST* be called before pkt_access_type() is called, in
 * order to insure that the SAVEDS location is initialized with the Micro-C
 * data segment. Another reason is that you need to obtain the interface
 * class/type/number so that you can pass them to pkt_access_type();
 *
 * Arguments:
 *	info_struct	= "PD_driver_info" structure to fill in.
 *
 * Returns:
 *	0			= Success
 *	!0			= Packet driver error code
 */
int pkt_driver_info(info_struct) asm
{
		MOV		CS:SAVEDS,DS		; Save out data seg
		MOV		AX,01ffh			; Request info
pi1:	INT		PKT_INT				; Ask packet driver
		JC		di2					; Error occured
		PUSH	DS					; Save driver DS
		MOV		DS,CS:SAVEDS		; Restore DATA segment
		POP		ES					; Point to driver data
		MOV		DI,4[BP]			; Get struct
		MOV		0[DI],BX			; Save version
		MOV		2[DI],CH			; Save Class
		MOV		3[DI],DX			; Save type
		MOV		5[DI],CL			; Save number
		MOV		6[DI],AL			; Save functionality
		LEA		BX,7[DI]			; Point to name
di1:	MOV		AL,ES:[SI]			; Get data
		MOV		[BX],AL				; Copy it
		INC		SI					; Advance source
		INC		DI					; Advance dest
		AND		AL,AL				; End of string?
		JNZ		di1					; Copy it all
		XOR		AX,AX				; Return 0
		JMP		di3
; Report driver error
di2:	MOV		AL,DH				; Get error code
		MOV		AH,0FFh				; Insure non-zero
di3:
}

/*
 * Obtain access to a packet type
 *
 * Arguments:
 *	if_class	= Interface class  \
 *	if_type		= Interface type    > - From pkt_driver_info()
 *	if_number	= Interface number /
 *	type		= Pointer to packet type to access
 *	typelen		= Length of packet type (0=Access ALL packets)
 *	handle		= Pointer to 'int' variable to receive handle
 *
 * Returns:
 *	0			= Success
 *	!0			= Packet driver error code
 */
int pkt_access_type(if_class, if_type, if_number, type, typelen, handle) asm
{
		MOV		AL,14[BP]			; Get if_class
		MOV		BX,12[BP]			; Get if_type
		MOV		DL,10[BP]			; Get if_number
		MOV		SI,8[BP]			; Get type pointer
		MOV		CX,6[BP]			; Get typelen
		MOV		DI,OFFSET pktrdy	; Point to handler
		PUSH	CS					; Save code segment
		POP		ES					; ES = handler segment
		MOV		AH,02h				; Indicate "access type" request
pi2:	INT		PKT_INT				; Call packet driver
		JC		di2					; Report error
		MOV		BX,4[BP]			; Get handle
		MOV		[BX],AX				; Save handle
		XOR		AX,AX				; Indicate success
}

/*
 * Release access to a packet type
 *
 * Arguments:
 *	handle		= Handle returned by pkt_access_type()
 *
 * Returns:
 *	0			= Success
 *	!0			= Packet driver error code
 */
int pkt_release_type(handle) asm
{
		MOV		BX,4[BP]			; Get handle
		MOV		AH,03h				; Indicate "release type" request
pi3:	INT		PKT_INT				; Ask for it
		JC		di2					; Indicate failed
		XOR		AX,AX				; Indicate success
}

/*
 * Send a packet
 *
 * Arguments:
 *	buffer		= Pointer to buffer to send
 *	length		= Length of buffer to send
 *
 * Returns:
 *	0			= Success
 *	!0			= Packet driver error code
 */
int pkt_send(buffer, length) asm
{
		MOV		SI,6[BP]			; Get buffer address
		MOV		CX,4[BP]			; Get length
		MOV		AH,04h				; Indicate "send packet" request
pi4:	INT		PKT_INT				; Ask for it
		JC		di2					; Report error
		XOR		AX,AX				; Indicate success
}

/*
 * Get our ethernet address
 *
 * Arguments:
 *	handle		= Handle returned by pkt_access_type()
 *	buffer		= Pointer to buffer to receive address
 *	length		= Size of buffer (maximum)
 *	alen		= Pointer to 'int' variable to receive size of address
 *
 * Returns:
 *	0			= Success
 *	!0			= Packet driver error code
 */
int pkt_get_address(handle, buffer, length, alen) asm
{
		MOV		BX,10[BP]			; Get handle
		MOV		DI,8[BP]			; Get buffer
		MOV		CX,6[BP]			; Get length
		PUSH	DS					; Save out segment
		POP		ES					; Set ES
		MOV		AH,06h				; Indicate "get address" request
pi5:	INT		PKT_INT				; Ask driver
		JC		di2					; Report error
		MOV		BX,4[BP]			; Get received length
		MOV		[BX],CX				; Save length
		XOR		AX,AX				; Indicate success
}

/*
 * Set the receive mode for a packet
 *
 * Arguments:
 *	handle		= Handle returned by pkt_access_type()
 *	mode		=	1: Turn off receiver
 *					2: Receive only packets sent to our address
 *					3: mode 2 + broadcast packets
 *					4: mode 3 + limited multicast packets
 *					5: mode 3 + all multicast packets
 *					6: all packets
 *
 * Returns:
 *	0			= Success
 *	!0			= Packet driver error code
 */
int pkt_set_receive_mode(handle, mode) asm
{
		MOV		BX,6[BP]			; Get handle
		MOV		CX,4[BP]			; Get mode
		MOV		AH,20				; Set function
pi6:	INT		PKT_INT				; Ask driver
		JC		di2					; Report error
		XOR		AX,AX				; Report success
}

/*
 * Receive packet handler. Called by packet driver when a packet
 * has been received for us.
 *
 * Check to see if we have enough space, and buffer the packet in our
 * receive packet queue, notify the mainline when received.
 */
asm {
; Entry:
;  AX = flag: 0=Allocate buffer, 1=Transfer complete
;  BX = handle
;  CX = Length
; Exit:
;  ES:DI = Buffer to receive packet (0:0=Not available)
pktrdy:	PUSH	DS					; Save our data segment
		MOV		DS,CS:SAVEDS		; Restore Micro-C DS
		AND		AX,AX				; Is this allocate?
		JNZ		pktr2				; No, try next
; Allocate space for a packet
		CMP		CX,PKT_SIZE			; Is packet too large?
		JA		pktr1				; Yes, Fail reception
		MOV		AX,DGRP:_pkt_head	; Get head of queue
		INC		AX					; Advance to next
		AND		AX,PKT_WINDOW-1		; Mask for wrap
		CMP		AX,DGRP:_pkt_tail	; Is queue full?
		JZ		pktr1				; Yes, Fail reception
		MOV		AX,DGRP:_pkt_head	; Get head back
		MOV		DI,PKT_SIZE+4		; Get size
		MUL		DI					; Compute offset
		ADD		AX,OFFSET DGRP:_pkt_queue ; Offset to location
		MOV		DI,AX				; Point to location
		MOV		[DI],BX				; Save handle
		MOV		2[DI],CX			; Save length
		ADD		DI,4				; Skip to data
		PUSH	DS					; Get DS
		POP		ES					; Copy to ES
		JMP SHORT pktrx				; And exit
; No space for buffer, or packet too large - reject packet
pktr1:	MOV		DI,0				; Indicate no buffer
		MOV		ES,DI				; ""
		JMP SHORT pktrx				; And exit
; Packet has been copied into our buffer, notify application
pktr2:	MOV		AX,DGRP:_pkt_head ; Get packet queue head
		INC		AX					; Skip to next
		AND		AX,PKT_WINDOW-1		; Wrap circular buffer
		MOV		DGRP:_pkt_head,AX ; Save new queue head
pktrx:	POP		DS					; Restore data seg
		DB		0CBh				; Far return
SAVEDS	DW		?
}

/*
 * Set the packet interrupt by patching the code
 */
set_interrupt(inum) asm
{
		MOV		AL,4[BP]			; Get the interrupt number
		MOV		byte ptr pi1+1,AL	; Patch
		MOV		byte ptr pi2+1,AL	; Patch
		MOV		byte ptr pi3+1,AL	; Patch
		MOV		byte ptr pi4+1,AL	; Patch
		MOV		byte ptr pi5+1,AL	; Patch
		MOV		byte ptr pi6+1,AL	; Patch
}


unsigned
	Dtime = 4,				// Display update interval
	Pkt_type = 0x5555,		// Packet type
	Btime = (18*10),		// Broadcast interval
	Blength = 50,			// Broadcast packet length
	Ttime = 1,				// Transmit interval
	Tcount = 28,			// Transmit packet count (# per interval)
	Tlength = 75;			// Transmit length

struct Setup {
	unsigned *Var;			// Pointer to variable
	unsigned Type;			// Type of value
	unsigned Low, High;		// Low/High values
	unsigned char *Text;
	} Setup_items[] = { int
	&Ttime,		2,	1,	9999,		"TX Pkt/Sec",
	&Tlength,	0,	1,	PKT_DATA,	"TX    size",
	&Btime,		1,	1,	9999,		"Bcast rate",
	&Blength,	0,	1,	PKT_DATA,  	"Bcast size" };
#define	SETUP_ITEMS	4

struct node {
	unsigned char Address[PKT_EASIZE];
	unsigned rx[2];
	unsigned rb[2];
	unsigned tx[2]; } node_list[MAX_NODES];

unsigned
	h1,								// Packet handle
	rx[2],							// Global RX count
	rb[2],							// Global BRX count
	tx[2],							// Global TX count
	tb[2],							// Global BTX count
	time,							// Current time
	dtick,							// Display interval tick
	btick,							// Broadcast interval tick
	ttick,							// Transmit interval tick
	dnode,							// Current display node
	tcount,							// Transmit count
	setup,							// Setup selection
	node_top;						// Top of node list
unsigned char
	Xfill,							// Fill type
	Xdata;							// Fill data;

static char help[] = { "\n\
Use:	PKTLG	[options]\n\n\
opts:	B=n	Broadcast packet length	( 1-"#PKT_DATA")		[50]\n\
	D=n	Data packet length	( 1-"#PKT_DATA")		[75]\n\
	F=n	Fill data (00-FF R=Random S0-65535=RndSeed)	[00]\n\
	I=n	broadcast Interval	( 1-3600 seconds)	[10]\n\
	P=xx	Packet interrupt	(00-FF)			[60]\n\
	R=n	data packet Rate	( 1-5000 pkts/second)	[500]\n\
	S=n	Screen update rate	(55-1000 ms)		[250]\n\
	T=xxxx	specify packet Type				[5555]\n\
\n?COPY.TXT 2003-2005 Dave Dunfield\n -- see COPY.TXT --.\n\
" };

static char *prompt[] = {
	"\x18/\x19     = Select",
	"\x1B/\x1A     = Adjust by \xF11",
	"Pup/Pdn = Adjust by \xF110",
	"ESC     = Exit PKTLG",
	0 };

extern unsigned RAND_SEED;

void show_address(unsigned char *a)
{
	wprintf("%02x:%02x:%02x:%02x:%02x:%02x", a[0], a[1], a[2], a[3], a[4], a[5]);
}

void increment(unsigned ct[2])
{
	if(++*ct >= 10000) {
		++ct[1];
		*ct = 0; }
}

void show_count(unsigned ct[2])
{
	if(ct[1]) {
		wprintf("%6u%04u", ct[1], *ct);
		return; }
	wprintf("%10u", *ct);
}

void in_packet(struct PD_packet *p)
{
	unsigned i;
	struct node *n;

	for(i=0; i < node_top; ++i) {
		if(!memcmp((n=node_list[i])->Address, p->from_addr, PKT_EASIZE))
			goto found; }

	if(node_top < MAX_NODES)
		memcpy((n=node_list[node_top++])->Address, p->from_addr, PKT_EASIZE);
found:
	if(p->to_addr[0] == 0xFF) {
		increment(rb);
		increment(n->rb);
		return; }
	increment(rx);
	increment(n->rx);
}

static unsigned char banner[] = {
"PKTLG v1.0 - ?COPY.TXT 2003-2005 Dave Dunfield -  -- see COPY.TXT --.\n" };

/*
 * Main demonstration program
 */
main(int argc, char *argv[])
{
	unsigned ps, po;							/* Driver segment,offset */
	unsigned char my_address[PKT_EASIZE];		/* Our ethernet address */
	unsigned alen;								/* Length of address */
	struct PD_driver_info info;					/* Info from driver */
	struct PD_packet send_buffer;				/* Packet to send */
	struct node *n;
	static char pkt_driver_id[] = {"PKT DRVR"};	/* Driver ID string */
	unsigned i, t;								/* Misc variables */

	RAND_SEED = peekw(0x40, 0x6C);
	for(i=1; i < argc; ++i) {
		n = argv[i];
		switch((toupper(*n++)<<8)|toupper(*n++)) {
		case 'B=' : Blength = atoi(n);	continue;
		case 'D=' : Tlength = atoi(n);	continue;
		case 'I=' : Btime = atoi(n)*18;	continue;
		case 'T=' : Pkt_type = atox(n);	continue;
		case 'S=' : Dtime = atoi(n)/55;	continue;
		case 'P=' : set_interrupt(Pkt_int = atox(n));	continue;
		case 'F=' :
			if(isxdigit(*n)) {
				Xdata = atox(n);
				continue; }
			switch(*n) {
				case 's' :
				case 'S' : RAND_SEED = atoi(n+1);
				case 'r' :
				case 'R' : Xfill = 0x55; continue; }
			abort("Fill must be 00-FF, R or S1-65535\n");
		case 'R=' :	if(!(t = atoi(n))) abort("Data rate must be 1-10000\n");
			if(t <= 18) {
				Tcount = 1;
				Ttime = 18 / t; }
			else {
				Ttime = 1;
				Tcount = t / 18; }
			continue;
		default: printf("\nUnknown option: %s\n", n-2);
		case '?'<<8:
		case '/?' :
		case '-?' :
			abort(help); } }

	fputs(banner, stdout);

	if((Blength > PKT_DATA) || !Blength)
		abort("Broadcast length must be 1-"#PKT_DATA"\n");
	if((Tlength > PKT_DATA) || !Tlength)
		abort("Data length must be 1-"#PKT_DATA"\n");
	if((Btime > 64800) || !Btime)
		abort("Broadcast interval must be 1-3600\n");
	if(Tcount > (10000/18))
		abort("Data rate must be 1-10000\n");
	if((Dtime > 18) || !Dtime)
		abort("Screen refresh must be 55-1000\n");

	/* Check that a packet driver is present */
	po = peekw(0, Pkt_int*4) + 3;	/* Offset to packet ID text */
	ps = peekw(0, (Pkt_int*4)+2);	/* Segment of packet driver */
	t = 0;
	do {
		if(pkt_driver_id[t] != peek(ps, po+t)) {
			printf("No packet driver on INT %02x [%04x:%04x]\n", Pkt_int, ps, po-3);
			exit(-1); } }
	while(pkt_driver_id[++t]);

	/* Obtain information from driver and display */
	if(t = pkt_driver_info(info)) {
		printf("Unable to obtain packet driver information, RC=%04x\n", t);
		exit(-1); }
	printf("Pkt driver: version=%u ", info.version);
	printf("Class/Type/Num=%02x/%04x/%02x ", info.class, info.type, info.number);
	printf("Func=%02x ", info.functionality);
	printf("Name='%s'\n", info.name);

	/* Obtain access to our packet type */
	if(t = pkt_access_type(info.class, info.type, 0, &Pkt_type, sizeof(Pkt_type), &h1)) {
		printf("Unable to obtain packet access, RC=%04x\n", t);
		exit(-1); }

	/* Get our ethernet address */
	if(t=pkt_get_address(h1, my_address, sizeof(my_address), &alen)) {
		printf("Unable to read ethernet address!, RC=%04x\n", t);
	do_exit:
		pkt_release_type(h1);
		return; }

	if(t=pkt_set_receive_mode(h1, 6)) {
		printf("Unable to set access mode!, RC=%04x\n", t);
		goto do_exit; }

/* Obtain information from driver and display
	printf("driver_info=%u ", pkt_driver_info(info));
	printf("ver=%u ", info.version);
	printf("Class/Type/Num=%02x/%04x/%02x ", info.class, info.type, info.number);
	printf("Func=%02x ", info.functionality);
	printf("Name='%s'\n", info.name);

	/* Obtain access to our packet type */
	t = Pkt_type;
	printf("access_type=%u ", pkt_access_type(info.class, info.type, 0, &t, sizeof(t), &h1));
	printf("Handle=%04x\n", h1);

	/* Get our ethernet address & display */
	printf("get_address=%u ", pkt_get_address(h1, my_address, sizeof(my_address), &alen));
	printf("alen=%u Address=%02x:%02x:%02x:%02x:%02x:%02x\n", alen,
		my_address[0], my_address[1], my_address[2], my_address[3], my_address[4], my_address[5]);
	pkt_set_receive_mode(h1, 6); */

	wopen(0, 0, 80, 25, WSAVE|WCOPEN|0x17);
	wcursor_off(); wputs(banner);
	show_address(my_address);
	wgotoxy(1, 2);
	//     xx:xx:xx:xx:xx:xx  --------- --------- --------- ---------
	wputs( "--Node MAC Id--   Bcast-RX    Pkt-RX    Pkt-TX  Bcast-TX");

	wgotoxy(56, 4); wprintf("Pkt type  : %04x", Pkt_type);
	wgotoxy(56, 5); wputs("Fill      : ");
	if(Xfill == 0x55) wprintf("Seed(%u)", RAND_SEED);
	else wprintf("Data(%02x)", Xdata);

	/*
	 * Enter into a loop, monitoring the receive packet queue.
	 * If a packet is received, display it and remove from the queue.
	 * Also monitor keyboard, and exit if the ESC key is pressed.
	 * '0' causes transmission of a multicast packet.
	 * '1-9' transmits to collected addresses.
	 */
	memset(send_buffer, 0, sizeof(send_buffer));
	memcpy(send_buffer.from_addr, my_address, PKT_EASIZE);
	send_buffer.ptype = Pkt_type;
	for(i=0; i < PKT_DATA; ++i) {
		if(Xfill == 0x55)
			Xdata = rand();
		send_buffer.data[i] = Xdata; }

	for(i=0; i < SETUP_ITEMS; ++i)
		show_item(i);
	i += 8;
	for(t=0; n = prompt[t]; ++t) {
		wgotoxy(56, t+i);
		wputs(n); }

	btick = ((dtick = peekw(0x40, 0x6C))-Btime) + 10;
	for(;;) {
		while(pkt_head != pkt_tail) {
			in_packet(pkt_queue[pkt_tail].packet);
			pkt_tail = (pkt_tail+1) & (PKT_WINDOW-1); }
		time = peekw(0x40, 0x6C);

		if((time - dtick) >= Dtime) {	// Display update pending
			dtick = time;
			if(dnode < node_top) {
				wgotoxy(0, dnode+3);
				show_address((n = node_list[dnode])->Address);
				show_count(n->rb);
				show_count(n->rx);
				show_count(n->tx);
				++dnode; }
			else {
				dnode = 0;
				wgotoxy(17, 1);
				show_count(rb);
				show_count(rx);
				show_count(tx);
				show_count(tb); }
			if(i = wtstc())
				process_key(i); }

		if((time - btick) >= Btime) {	// Broadcast pending
			memset(send_buffer.to_addr, 0xFF, PKT_EASIZE);
			pkt_send(send_buffer, Blength);
			increment(tb);
			btick = time; }

		if((time - ttick) >= Ttime) {	// Transmit tick pending
			wgotoxy(75, 1);
			wputs(tcount ? "OVRN" : "    ");
			tcount = Tcount;
			ttick = time; }

		if(tcount) {
			for(i=0; i < node_top; ++i) {
				memcpy(send_buffer.to_addr, (n = node_list[i])->Address, PKT_EASIZE);
				pkt_send(send_buffer, Tlength);
				increment(tx);
				increment(n->tx); }
			--tcount; }
		}
}

process_key(int c)
{
	unsigned t;
	switch(c) {
	case _KDA :
		t = setup;
		if(++setup >= SETUP_ITEMS)
			setup = 0;
	new_setup:
		wgotoxy(55, t+7); wputc(' ');
		show_item(setup);
		return;
	case _KRA :
		increment_item(setup);
		show_item(setup);
		return;
	case _KPU :
		for(t=0; t < 10; ++t)
			increment_item(setup);
		show_item(setup);
		return;
	case _KLA :
		decrement_item(setup);
		show_item(setup);
		return;
	case _KPD :
		for(t=0; t < 10; ++t)
			decrement_item(setup);
		show_item(setup);
		return;
	case _KUA :
		t = setup;
		setup = (setup ? setup : SETUP_ITEMS) - 1;
		goto new_setup;
	case _K1 : node_top = MAX_NODES; return;
	case 0x1B :
		wclose();
		pkt_release_type(h1);
		exit(0); }
}

show_item(unsigned n)
{
	struct Setup *p;
	unsigned *v, t;

	t = *(v = (p = Setup_items[n])->Var);
	wgotoxy(55, n+7);
	wputc(n == setup ? 0x10 : ' ');
	wprintf("%-10s: ", p->Text);
	switch(p->Type) {
	case 0 : wprintf("%5u", t); break;
	case 1 : wprintf("%5u.%02u", t / 18, ((t % 18) * 5)+5); break;
	case 2 : wprintf("%5u", (*(v+2) * 18) / t);
	}
}

increment_item(unsigned n)
{
	struct Setup *p;
	unsigned *v, *v1, t, t1;
	t = *(v = (p = Setup_items[n])->Var);
	switch(p->Type) {
	case 0 :
	case 1 :	if(t < p->High) ++*v; return;
	case 2 :
		v1 = v+2;
		t1 = (*v1 * 18) / t;
		if(t1 < p->High) {
			if(t > 1) {
				--*v;
				*v1 = 1;
				return; }
			++*v1; } }
}

decrement_item(unsigned n)
{
	struct Setup *p;
	unsigned *v, *v1, t, t1;
	t = *(v = (p = Setup_items[n])->Var);
	switch(p->Type) {
	case 0 :
	case 1 :	if(t > p->Low) --*v; return;
	case 2 :
		v1 = v+2;
		t1 = (*v1 * 18) / t;
		if(t1 > p->Low) {
			if(*v1 > 1) {
				--*v1;
				*v = 1;
				return; }
			++*v; } }
}
