/*
 * Serial Debug Terminal
 *
 * SDT is a terminal (TTY) program with features to assist in the
 * debugging of RS-232 serial connections.
 * - DTR and RTS signals can be controlled from the keyboard
 * - RI, CD, DSR and CTS states are displayed on the status line
 * - Input/Output can be independantly switched to HEX or ASCII
 * - All baud rates/data formats available on the PC are supported
 * - Pop-up HEX/ASCII code chart and serial port pinout screens
 *
 * Compile with DDS Micro-C/PC: cc SDT -fop
 *
 * Copyright 2000-2005 Dave Dunfield
 * All rights reserved.
 */
#include <stdio.h>		// Standard I/O definitions
#include <window.h>		// Text windowing definitions
#include <comm.h>		// Communications definitions

unsigned
	Seg,			// Capture segment
	Spos,			// Segment position
	Hmode,
	Uaddr,
	X, MaxX, MaxY,
	baud=_9600;		// COM speed divisor
struct WINDOW
	*mwin;
FILE
	*fpi,
	*fpo;
unsigned char
	*Ptr,
	port=1,			// COM port
	dtr=0,			// DTR asserted
	rts=0,			// RTS asserted
	Brk=0,			// BREAD control
	parity=0,		// Parity
	dbits=3,		// Data bits
	sbits=0,		// Stop bits
	hex = 4,		// HEX/ASCII mode (auto-select)
	btext[8],		// Baudrate text
	temp[25],		// Temp. buffer
	Fname[65],
	Dmask[8192];	// Data used mask

// Parity -> text table
char *ptext[] = { "None", "Even", "Odd", "Mark", "Space" };

// Hex mode -> text table
char *htext[] = { "Asc/Asc", "Asc/Hex", "Hex/Asc", "Hex/Hex" };

// Help menu
char *hmenu[] = {
	"General help",
	"Command line",
	"Function keys",
	"HEX/ASCII chart",
	"RS-232 port pinouts",
	0 };

//ChtTxt R:\Help.h
#include "R:\\Help.h"

/*ChtTxt GenHELP
SDT is a terminal (TTY) program with features to assist in the
debugging of RS-232 serial connections.

	DTR and RTS signals can be controlled from the keyboard
	RI, CD, DSR and CTS states are displayed on the status line
	Input/Output can be independantly switched to HEX or ASCII
	All baud rates/data formats available on the PC are supported
	Pop-up HEX/ASCII code chart and serial port pinout screens
*/
/*ChtTxt CmdHELP
Use:	SDT [file] [options]

opts:	+/-D				set startup DTR state
		+/-R					""		RTS state
		B=baud[ParDataStop]		""		Baudrate/Parity
		M=AA|AH|HA|HH			""		HEX/ASCII Mode
		P=1-4				set COM Port to use

	if [file], preset PgUp/PgDn filename

		eg:  SDT upload.hex +DR B=1200E72 M=AH P=2

Dave Dunfield	-	https://dunfield.themindfactory.com
*/

/*ChtTxt FncHELP
F1		Toggle DTR state
F2		Toggle RTS state
F3		Toggle sendBREAK (TD low)
F4		Toggle ASCII-HEX mode (Tx/Rx)
F5		Setup baudrate/parity/data/stop
F6		Clear the screen
F7		Review received data
F8		Display help text
PgUp	Upload ASCII
PgDn	Download ASCII
F10		Exit
*/

/*ChtTxt PinHELP
Signal name				   Origin	DB-25	DB-9
-------------------------------------------------
Protective Ground	(GND)	---		  1		  -
Transmit Data		(TXD)	DTE		  2		  3
Receive Data		(RXD)	DCE		  3		  2
Request To Send		(RTS)	DTE		  4		  7
Clear To Send		(CTS)	DCE		  5		  8
Data Set Ready		(DSR)	DCE		  6		  6
Signal Ground		(GND)	---		  7		  5
Data Carrier Detect	(DCD)	DCE		  8		  1
Data Terminal Ready	(DTR)	DTE		 20		  4
Ring Indicator		(RI )	DCE		 22		  9
*/

void Ps(unsigned char *p)	{ fputs(p, stdout); }

/*
 * Scale to a fractional value with 32 bit intermediate
 */
unsigned scale(value, multiply, divide) asm
{
		MOV		AX,8[BP]		; Get value
		MUL		WORD PTR 6[BP]	; Multiply to 32 bit product
		MOV		BX,4[BP]		; Get divisor
		DIV		BX				; Divide back to 16 bit result
		SHR		BX,1			; /2 for test
		INC		DX				; .5 rounds up
		SUB		BX,DX			; Set 'C' if remainder > half
		ADC		AX,0			; Increment result to scale
}

/*
 * Set baudrate text based on baud divisor
 */
void set_btext(void)
{
	// Special case for 115200 since we can't sprintf > 16 bits
	if(baud == 1)
		strcpy(btext, "115200");
	else
		sprintf(btext, "%u", scale(57600, 2, baud));
}

/*
 * Set the RS-232 comm parameters
 */
void set_comm(void)
{
	int c, count;
	char obtext[10];

	wopen(24, 6, 31, 11, WSAVE|WCOPEN|WBOX1|REVERSE);
	wcursor_off();
	wputs("\x18/\x19 = Adjust baud by 1 step\n");
	wputs("\x1B/\x1A = Adjust baud by 10 steps\n");
	wputs(" P  = Adjust Parity\n");
	wputs(" D  = Adjust Data bits\n");
	wputs(" S  = Adjust Stop bits\n");
	wputs("\n\n\nPress ENTER/F10/ESC to Exit");
	for(;;) {
		wgotoxy(6, 6);
		wprintf("%6s %5s-%u-%u", btext, ptext[parity], dbits+5, sbits+1);
		c = wgetc();
		count = 0;
	renew:
		switch(c) {
		case 'p' :				// Adjust parity
		case 'P' : if(++parity > 4) parity = 0;		continue;
		case 'd' :				// Adjust data bits
		case 'D' : if(++dbits > 3) dbits = 0;		continue;
		case 's' :				// adjust stop bits
		case 'S' : sbits = !sbits;					continue;
		case _KRA : count = 9; c = _KUA;
		case _KUA :				// Adjust speed (-)
			if(baud <= 1)
				continue;
			--baud;
			goto recalc;
		case _KLA : count = 9; c = _KDA;
		case _KDA :				// Adjust speed (+)
			if(baud == -1)
				continue;
			++baud;
		recalc:
			set_btext();
			if(!strcmp(btext, obtext))
				goto renew;
			strcpy(obtext, btext);
			if(count) {
				--count;
				goto renew; }
			continue;
		case '\n' :
		case _K10 :
		case 0x1B :				// Exit this function
				if(dbits < 2)	// < 7 bit cannot display ASCII
					hex = 3;
				wclose();
				return; } }
}

void Phc(unsigned char c)
{
	switch(Hmode) {
	case 0x55:	wputc(c);			return;
	case 0xAA:
		if(c == '\n') {
			++MaxY;
			if(X > MaxX)
				MaxX = X;
			X = 0;
			return; }
		++X;
		return; }
	putc(c, stdout);
}

void PutHelp(unsigned char *p)
{
	unsigned char c;
	while(c = *p++) {
		if(c & 0x80) {
			while(c-- & 0x7F)
				Phc(' ');
			continue; }
		Phc(c); }
}

/*
 * Perform the help menu
 */
void help(void)
{
	unsigned i; //x, mx, my;
	unsigned char *ptr;
	static unsigned select =  0;

	// Issue help menu
	if(wmenu(30, 8, WSAVE|WCOPEN|WBOX1|REVERSE, hmenu, &select))
		return;
	// Perform selected help action
	switch(select) {
		case 0 : ptr = GenHELP; break;	// General help
		case 1 : ptr = CmdHELP; break;	// Command line help
		case 2 : ptr = FncHELP; break;	// Function key help
		case 3 :						// HEX/ASCII chart
			wopen(0, 0, 80, 24, WSAVE|WCOPEN|NORMAL);
			wcursor_off();
			for(i=0; i < 256; ++i) {
				wgotoxy((i / 16) * 5, (i & 15)+4);
				wprintf("%02x=", i);
				wputc(i | 0x100); }
			*W_OPEN = REVERSE;
			wgotoxy(28,  2);
			wputs(" HEX/ASCII Code Chart ");
			wgotoxy(27, 21);
			wputs(" ENTER/F10/ESC to exit ");
			for(;;) switch(wgetc()) {
				case '\n' :
				case _K10 :
				case 0x1B :
					wclose();
					return; }
		case 4 : ptr = PinHELP; }		// RS-232 pinouts

	// Scan text - compute required window size
	MaxX = MaxY = X = 0;
	Hmode = 0xAA;
	PutHelp(ptr);
	Hmode = 0x55;
	// Open window, display text, wait for key, close & exit
	wopen(39-(MaxX/2), 10-(MaxY/2), MaxX+2, MaxY+2, WSAVE|WCOPEN|WBOX1|REVERSE);
	wcursor_off();
	PutHelp(ptr);

	for(;;) switch(wgetc()) {
		case '\n' :
		case _K10 :
		case 0x1B :
			wclose();
			return; }
}

int bitset(index) asm
{
		MOV		BX,4[BP]		// Get index
		MOV		CL,BL			// Save bit number
		AND		CL,7			// Keep bit number
		MOV		AX,1			// Get mask
		SHL		AL,CL			// Into position
		MOV		CL,3			// Shift bits
		SHR		BX,CL			// Get address
		AND		AL,DGRP:_Dmask[BX]	// Set bit
}

/*
 * Review buffered data
 */
void review()
{
	int c;
	unsigned i, j, p, t;
	w_clwin(mwin);
	w_puts("REVIEW: \x1B\x18\x19\x1A:Move  PgUp/PgDn:Page  Home:Reset  ESC:Exit", mwin);
	wopen(0, 0, 80, 24, WSAVE|WCOPEN|NORMAL);
	wcursor_off();
home:
	p = Spos;
redraw:
	t = p - (16*24);
	for(i=0; i < 24; ++i) {
		wgotoxy(1, i);
		if(j = Spos - t)
			wprintf("-%-5u", Spos - t);
		else
			wputs("-65536");
		for(j=0; j < 16; ++j) {
			if(!(j & 3))
				wputc(' ');
			if(bitset(t+j))
				wprintf(" %02x", peek(Seg, t+j));
			else
				wputs(" \xFA\xFA"); }
		wputs("  ");
		for(j=0; j < 16; ++j) {
			c = peek(Seg, t++);
			wputc( ((c < ' ') || (c > 0x7E)) ? 0xFA : c); } }
	for(;;) switch(wgetc()) {
		case 0x1B:
		case _K10:
			wclose();
			return;
		case _KHO: 					goto home;
		case _KLA: --p;				goto redraw;
		case _KRA: ++p;				goto redraw;
		case _KUA: p -= 16;			goto redraw;
		case _KDA: p += 16;			goto redraw;
		case _KPU: p -= (16*14);	goto redraw;
		case _KPD: p += (16*24);	goto redraw; }
}

void Close(void)
{
	if(fpi) fclose(fpi);
	if(fpo) fclose(fpo);
	fpi = fpo = 0;
}

/*
 * Serial Debug Terminal
 */
main(int argc, char *argv[])
{
	int c;
	unsigned t1, t2, oldsig;
	unsigned char uflag, xval, *ptr;

	// Parse arguments
	for(t1 = 1; t1 < argc; ++t1) {
		if(*(ptr = argv[t1]) == '+') {		// '+' options
			while(*++ptr) switch(toupper(*ptr)) {
			case 'D' : dtr = 1; continue;	// Assert DTR
			case 'R' : rts = 1; continue;	// Assert RTS
			default: goto badopt; }
			continue; }
		if(*ptr == '-') {					// '-' options
			while(*++ptr) switch(toupper(*ptr)) {
			case 'D' : dtr = 0; continue;	// Deassert DTR
			case 'R' : rts = 0; continue;	// Deassert RTS
			default: goto badopt; }
			continue; }
		if(*(ptr+1) == '=') {				// '=' options
			c = toupper(*ptr);
			ptr += 2;
			switch(c) {
			case 'P' :						// Set Parity
				port = atoi(ptr);
				if((port < 1) || (port > 4))
					abort("port must be 1-4");
				continue;
			case 'B' :						// Set Baudrate
				t2 = 0;
				while(isdigit(*ptr))
					btext[t2++] = *ptr++;
				btext[t2] = 0;
				if(!strcmp(btext, "115200"))
					baud = 1;
				else if((t2 = atoi(btext)) >= 2)
					baud = scale(57600, 2, t2);
				else
					abort("Bad baud rate");
				if(c = toupper(*ptr++)) {
					parity = 0;
					while(*ptext[parity] != c) {
						if(++parity > 4)
							abort("Bad parity"); }
					dbits = *ptr++ - '5';
					if(dbits > 3)
						abort("Bad data bits");
					sbits = *ptr++ - '1';
					if(sbits > 1)
						abort("Bad stop bits"); }
				continue;
			case 'M' :						// Set hex/ascii mode
				for(t2=hex=0; t2 < 2; ++t2) {
					c = toupper(*ptr++);
					if(c == 'H')
						hex |= (2 >> t2);
					else if(c != 'A')
						goto badmode; }
				if(*ptr) {
				badmode:
					abort("Bad mode"); }
				continue; } }
		if(*ptr == '?') {					// Command line help
			Ps("Serial Debug Terminak\n\n");
he:			PutHelp(CmdHELP);
			PutHelp(GenHELP);
			exit(0); }
		if(*Fname) {
badopt:		printf("Bad option: %s\n\n", argv[t1]);
			goto he; }
		strcpy(Fname, ptr); }

	set_btext();		// Initialize baudrate text
	if(hex == 4)		// Set initial hex/ascii mode by databits
		hex = (dbits < 2) ? 3 : 0;

	Seg = alloc_seg(4096);
	do {
		pokew(Seg, Spos, 0); }
	while(Spos -= 2);

	// Open windows
	mwin = wopen(xval = 0, 24, 80, 1, WSAVE|WCOPEN|REVERSE);
	wopen(0, 0, 80, 24, WSAVE|WCOPEN|WSCROLL|WWRAP|NORMAL);

	// (re)open the COM port
	// Build Copen parameters from current selections
open_com:
	t1 = dbits | (sbits ? STOP_2 : STOP_1);
	t2 = OUTPUT_2;
	switch(parity) {
		case 1 : t1 |= PAR_EVEN;	break;
		case 2 : t1 |= PAR_ODD;		break;
		case 3 : t1 |= PAR_MARK;	break;
		case 4 : t1 |= PAR_SPACE; }
	if(rts) t2 |= SET_RTS;
	if(dtr) t2 |= SET_DTR;

	if(Copen(port, baud, t1, t2)) {
		wclose();
		wclose();
		printf("Cannot open COM%u:\n", port);
		goto he; }
	asm {
		ADD	DX,3
		MOV	DGRP:_Uaddr,DX
		IN	AL,DX
		MOV	DGRP:_Brk,AL
	}
	Cflags |= TRANSPARENT;	// Raw binary mode

	// (re)display status line (message window)
restat:
	sprintf(temp,"%s-%s-%u-%u", btext, ptext[parity], dbits+5, sbits+1);
	w_gotoxy(0, 0, mwin);
	w_printf(mwin, "1:%s 2:%s 3:%s 4:%s 5:%-16s 6:Cls 7:Rev 9:Help",
		dtr ? "DTR":"dtr", rts ? "RTS":"rts", (Brk & 0x40) ? "BRK":"brk",
		htext[hex], temp);
	if(hex & 2)		// Currently active xmit hex value
		w_printf(mwin, "%02x", xval);
	else
		w_puts("  ", mwin);
	uflag = oldsig = -1;

	// Main terminal loop
	for(;;) {
		if(uflag) {						// Cursor update pending
			wupdatexy();
			uflag = 0; }
		if((c = Ctestc()) != -1) {		// COMM Data received
			asm {
				MOV		BX,DGRP:_Spos	// Get value
				MOV		CL,BL			// Save bit number
				AND		CL,7			// Keep bit number
				MOV		AL,1			// Get mask
				SHL		AL,CL			// Into position
				MOV		CL,3			// Shift bits
				SHR		BX,CL			// Get address
				OR		DGRP:_Dmask[BX],AL	// Set bit
			}
			poke(Seg, Spos++, c);		// Save for capture
			if(hex & 1)					// Display in hex
				wprintf(".%02x", c);
			else {						// Display in ASCII
				switch(c) {
				case 7 : c |= 0x100; }	// disable audible BELL
				wputc(c); }
			if(fpo)
				putc(c, fpo);
			uflag = -1;
			continue; }
		if(fpi) switch(c = getc(fpi)) {
			case'\n':	c = '\r';
			default	:	goto a1;
			case EOF:	Close(); }
		if(c = wtstc()) {				// Process keyboard entries
			if(c & 0xFF00) switch(c) {		// Function key
				case _K1 : dtr = !dtr;				goto reopen; // toggle DTR
				case _K2 : rts = !rts;				goto reopen; // toggle RTS
				case _K3 : out(Uaddr, Brk ^= 0x40);	goto restat; // Toggle BREAK
				case _K4 : if(++hex > 3) hex=0;		goto restat; // Hex/ASCII
				case _K5 : set_comm();		// COMM setup menu
				reopen:
					Cclose();
					goto open_com;
				case _K6 : wclwin();				continue;	// Clear screen
				case _K7 : review();				goto restat;// Review data
				case _K10:					// Exit
					Cclose();
					wclose();
					wclose();
					Close();
					return;
				case _KBS:
					if(!(hex & 2))
						Cputc('\b');
					continue;
				case _KPU:
				case _KPD:
					ptr = W_OPEN;
					W_OPEN = mwin;
					wclwin();
					wprintf("%sload File? ", (c == _KPU) ? "Up" : "Down");
					if(wgets(W_OPEN->WINcurx, 0, Fname, 64) == '\n') {
						if(*Fname) {
							wclwin();
							if(c == _KPU)
								fpi = t1 = fopen(Fname, "r");
							else
								fpo = t1 = fopen(Fname, "w");
							if(!t1) {
								wclwin();
								wputs("?open? ");
								wputs(Fname);
								wgetc(); } } }
					W_OPEN = ptr;
					goto restat;
				case _KEN:
					Close();
					continue;
				case _K9 :									// Help
					help(); }
			else {							// Data to send
				if(hex & 2) {					// Hex entry mode
					if(isdigit(c))
						c -= '0';
					else if((c >= 'a') && (c <= 'f'))
						c -= ('a'-10);
					else if((c >= 'A') && (c <= 'F'))
						c -= ('A'-10);
					else if(c == '\n') {
						Cputc(xval);
						continue; }
					else
						continue;
					xval = (xval << 4) + c;
					goto restat; }
				else {							// ASCII entry mode
a1:					Cputc(c == '\n' ? '\r' : c); } }
			continue; }
		// Monitor COMM signal changes & update displays
		if((t1 = Csignals() & (CD|RI|DSR|CTS)) != oldsig) {
			w_gotoxy(67, 0, mwin);
			w_printf(mwin, "%s.%s.%s.%s",
				(t1 & CD) ? "CD" : "cd",
				(t1 & RI) ? "RI" : "ri",
				(t1 & DSR) ? "DSR" : "dsr",
				(t1 & CTS) ? "CTS" : "cts");
			uflag = -1;
			oldsig = t1; } }
}
