#include <stdio.h>
#include <window.h>
#include <lrg.h>
#include <keys.h>

// Mixer sources
#define	MASTER	0
#define	VOICE	1
#define	MIDI	2
#define	CD		3
#define	LINE	4

unsigned char
	cflag,		// Change flag
	*emsg = "?COPY.TXT 2002-2005 Dave Dunfield  -   -- see COPY.TXT --.";

// Function prototypes for access by all modules
void draw_main_screen(void);
void draw_fkeys(void);
void draw_times(void);
void draw_fb(void);
void message(unsigned char *m);

// Subsystem modules
#include "tssound.c"	// Sound card interface functions
// #include "tsdtmf.c"		// DTMF generation and editing functions

// Parallel I/O bits in LPT port
#define	LPT_HOOK	0x0F	// Set for off-hook
#define	LPT_DTMF	0xF0	// Set for DTMF output

// Macros to control LPT bits
#define	lpt_set(a)	out(lpt, (lpm |= (a)))
#define	lpt_clr(a)	out(lpt, (lpm &= ~(a)))

struct WINDOW
	*mwin;		// Main window

unsigned
	tduration = 1000,	// Tone duration
	sduration = 1000,	// Silence duration
	fduration = 2400,	// Flash duration
	pduration = 2000,	// Pause duration
	DTMFlevel = 20,		// DTMF output level
	DTMFmain = 31,		// Main DTMF output level
	SPKlevel = 31,		// Speaker output level
	SPKmain = 31,		// Main SPK output level
	lpt,				// Lpt port address
	fbp;				// Function buffer pointer
unsigned char
	ebuffer[79],		// Error message buffer
	lpm,				// LPT mirror
	hook,				// Current hook state
	*pp,				// Parse pointer
	fw,					// Buffer wrapped flag
	fb[72],				// Function buffer
	fkeys[10][50];		// Function key handlers

unsigned char
	filename[50] = { "TESTSET" };

static char help[] = { "\n\
Use:	TESTSET [options]\n\n\
opts:	-D		= enable Debug messages		[Off]\n\
	C=file[.INI]	= load Configuration file	[None]\n\
	M=dtmf,spk	= set Mixer output levels	[31]\n\
	P=1-3/address	= specify Parallel port		[Lpt1]\n\
\n?COPY.TXT 2002-2005 Dave Dunfield\n -- see COPY.TXT --.\n" };

/*
 * -- DTMF generation and tone/waveform editing functions --
 */

#define	LRGW	250		// Width of graphic display
#define	LRGH	128		// Height of graphic display
#define	LRGX	1		// X offset for graphic display
#define	LRGY	13		// Y offset for graphic display
#define	GAP		20		// Minumum quadrant size

unsigned
	lwave[9] = 		// Low wave parameters
		{ 250, 500, 750, 75, 75, 100, 100, 100, 100 },
	hwave[9] =		// High wave parameters
		{ 250, 500, 750, 75, 75, 100, 100, 100, 100 },
	q1 = 250,		// Quadrant 1 end, Quadrant 2 start
	q2 = 500,		// Quadrant 2 end, Quadrant 3 start
	q3 = 750,		// Quadrant 3 end, Quadrant 4 start
	select,			// Current selected edit quadrant
	voc_seg,		// Voice file storage segment
	voc_off;		// Voice file storage offset
unsigned char
	amplitude[2] = { 75, 75 },			// Q1/2, Q3/4 amplitude
	slope[4] = { 100, 100, 100, 100 };		// Quadrant slope

/*
 * Quadrant sine table - this is 1/4 of a complete sine wave
 */
char qst[] = {
    0,    0,    1,    2,    3,    3,    4,    5,
    6,    7,    7,    8,    9,   10,   11,   11,
   12,   13,   14,   15,   15,   16,   17,   18,
   19,   19,   20,   21,   22,   23,   23,   24,
   25,   26,   26,   27,   28,   29,   30,   30,
   31,   32,   33,   33,   34,   35,   36,   36,
   37,   38,   39,   40,   40,   41,   42,   43,
   43,   44,   45,   46,   46,   47,   48,   48,
   49,   50,   51,   51,   52,   53,   54,   54,
   55,   56,   56,   57,   58,   59,   59,   60,
   61,   61,   62,   63,   63,   64,   65,   66,
   66,   67,   68,   68,   69,   70,   70,   71,
   72,   72,   73,   74,   74,   75,   75,   76,
   77,   77,   78,   79,   79,   80,   80,   81,
   82,   82,   83,   83,   84,   85,   85,   86,
   86,   87,   88,   88,   89,   89,   90,   90,
   91,   92,   92,   93,   93,   94,   94,   95,
   95,   96,   96,   97,   97,   98,   98,   99,
   99,  100,  100,  101,  101,  102,  102,  103,
  103,  104,  104,  105,  105,  105,  106,  106,
  107,  107,  108,  108,  108,  109,  109,  110,
  110,  110,  111,  111,  112,  112,  112,  113,
  113,  113,  114,  114,  114,  115,  115,  115,
  116,  116,  116,  117,  117,  117,  118,  118,
  118,  118,  119,  119,  119,  120,  120,  120,
  120,  121,  121,  121,  121,  121,  122,  122,
  122,  122,  123,  123,  123,  123,  123,  123,
  124,  124,  124,  124,  124,  124,  125,  125,
  125,  125,  125,  125,  125,  125,  125,  126,
  126,  126,  126,  126,  126,  126,  126,  126,
  126,  126,  126,  126,  126,  126,  126,  126,
  126,  126};
char
	*st,					// Active waveform
	lst[sizeof(qst)*4],		// Generated low sine wave
	hst[sizeof(qst)*4];		// Generated high sine wave

/*
 * DTMF tone pairs
 */
unsigned dtmf_otones[16][2] = {
	697, 1209,		// 1
	697, 1336,		// 2
	697, 1477,		// 3
	770, 1209,		// 4
	770, 1336,		// 5
	770, 1477,		// 6
	852, 1209,		// 7
	852, 1336,		// 8
	852, 1477,		// 9
	941, 1336,		// 0
	941, 1209,		// *
	941, 1477,		// #
	697, 1633,		// A
	770, 1633,		// B
	852, 1633,		// C
	941, 1633 };	// D
unsigned char dtmf_digits[] = { "1234567890*#ABCD" };
unsigned dtmf_tones[16][2];

/*
 * Select low/high waveform into working set
 */
void select_wave(char w)
{
	unsigned *p;
	if(w) {
		st = hst;
		p = hwave; }
	else {
		st = lst;
		p = lwave; }
	q1 = p[0];
	q2 = p[1];
	q3 = p[2];
	amplitude[0] = p[3];
	amplitude[1] = p[4];
	slope[0] = p[5];
	slope[1] = p[6];
	slope[2] = p[7];
	slope[3] = p[8];
}

/*
 * Update waveform settings from working set
 */
void update_wave(char w)
{
	unsigned *p;
	p = w ? hwave : lwave;
	p[0] = q1;
	p[1] = q2;
	p[2] = q3;
	p[3] = amplitude[0];
	p[4] = amplitude[1];
	p[5] = slope[0];
	p[6] = slope[1];
	p[7] = slope[2];
	p[8] = slope[3];
}

/*
 * Build a quadrant consisting of 1/4 sine wave
 *	x1,x2	= starting,ending x position
 *	a		= amplitude (0-100)
 *	s		= slope (0-100)
 *	d		= Direction (0=Up, 1=Down)
 */
void build_quad(unsigned x1, unsigned x2, unsigned a, unsigned s, char d)
{
	unsigned i, dx;
	char x, y, xst[sizeof(qst)];

	dx = x2 - x1;

	for(i=0; i < sizeof(qst); ++i) {
		x = qst[i];						// Size component
		x = (x * s) / 100;
		y = (i * 128) / sizeof(qst);	// Flat component
		y = ((100-s) * y) / 100;
		xst[i] = ((x+y) * a) / 100; }

	switch(d & 3) {
	case 0 :
		for(i=0; i < dx; ++i)
			st[x1+i] = xst[lrg_scale(i, 250, dx)];
		return;
	case 1 :
		for(i=0; i < dx; ++i)
			st[x1+i] = xst[(sizeof(xst)-1) - lrg_scale(i, 250, dx)];
		return;
	case 2 :
		for(i=0; i < dx; ++i)
			st[x1+i] = -xst[lrg_scale(i, 250, dx)];
		return;
	case 3 :
		for(i=0; i < dx; ++i)
			st[x1+i] = -xst[(sizeof(xst)-1) - lrg_scale(i, 250, dx)]; }
}

/*
 * Build the entire sine wave by generating each of the 4 quadrants
 */
void build_sine()
{
	build_quad(0, q1, amplitude[0], slope[0], 0);
	build_quad(q1, q2, amplitude[0], slope[1], 1);
	build_quad(q2, q3, amplitude[1], slope[2], 2);
	build_quad(q3, 1000, amplitude[1], slope[3], 3);
}

/*
 * Display graphical representation of generated sine waveform
 */
void display_sine()
{
	unsigned i;
	int j, l;
	unsigned char c;

	c = (select == 0) ? 4 : 6;
	for(i=0; i < LRGW; ++i) {
		j = st[l = i*4] / 2;
		if(l > q1) {
			c = (select == 1) ? 4 : 6;
			if(l > q2) {
				c = (select == 2) ? 4 : 6;
				if(l > q3)
					c = (select == 3) ? 4 : 6; } }
		lrg_vline(i+LRGX, LRGY, LRGH-1, c);
		lrg_plot(i+LRGX, (64 - j) + (LRGY-1), 7); }
}

/*
 * Edit the sine waveform
 */
int edit_sine(char w)
{
	cflag = 0;
	select_wave(w);

relrg:
	if(lrg_open())
		return -1;

	lrg_printf(20, 0, 7, "*** SINE WAVEFORM EDITOR ***");

	// Draw box
	lrg_box(LRGX-1, LRGY-1, LRGW+1, LRGH, 1);

	// Draw permanent prompts
	lrg_printf(270, 10+LRGY, 7, w ? "High" : "Low");
	lrg_printf(270, 20+LRGY, 7, "Tone");
	lrg_printf(0, 154+LRGY, 7, "amp-U/D Sine/Flat Left/Right Normal");
	lrg_printf(0, 167+LRGY, 7, "SPACE=quadrant ENTER=exit/save ESC=quit");

rebuild:
	// Build the sinewave and display the quadrant markers
	build_sine(st);
	lrg_fbox(0, 129+LRGY, 330, 16, 0);	// Clear old
	lrg_printf(0, 130+LRGY, 7, "0");
	lrg_printf((q1-10)>>2, 138+LRGY, 7, ".%u", q1);
	lrg_printf((q2-10)>>2, 130+LRGY, 7, ".%u", q2);
	lrg_printf((q3-10)>>2, 138+LRGY, 7, ".%u", q3);
	lrg_printf(245, 130+LRGY, 7, "1");

redraw:
	// Update the quadrant dynamic displays and prompt for cmmand
	lrg_printf(270, 50+LRGY, 7, "A:%3u", amplitude[select/2]);
	lrg_printf(270, 70+LRGY, 7, "S:%3u", slope[select]);
	for(;;) {
		display_sine(st);
		switch(toupper(kbget())) {
		case ' ' :			// Select quadrant
			select = (select + 1) & 3;
			goto redraw;
		case _UA :			// Increase amplitude
			if(amplitude[select/2] < 100)
				++amplitude[select/2];
		chg:cflag = -1;
			goto rebuild;
		case _DA :			// Decrease amplitude
			if(amplitude[select/2])
				--amplitude[select/2];
			goto chg;
		case 'S' :			// Move toward "sine" waveform
			if(slope[select] < 100)
				++slope[select];
			goto chg;
		case 'F' :			// Move toward "flat" waveform
			if(slope[select])
				--slope[select];
			goto chg;
		case _RA :			// Adjust quadrant end to the right
			switch(select) {
			case 0 :
				if(q1 < (q2-GAP))
					++q1;
				goto chg;
			case 1 :
				if(q2 < (q3 - GAP))
					++q2;
				goto chg;
			case 2 :
				if(q3 < (1000-GAP))
					++q3;
				goto chg; }
			continue;
		case _LA :			// Adjust quadrant end to the left
			switch(select) {
			case 0 :
				if(q1 > GAP)
					--q1;
				goto chg;
			case 1 :
				if(q2 > (q1+GAP))
					--q2;
				goto chg;
			case 2 :
				if(q3 > (q2+GAP))
					--q3;
				goto chg; }
			continue;
		case 'N' :			// Return to "normal" waveform
			amplitude[0] = amplitude[1] = 75;
			slope[0] = slope[1] = slope[2] = slope[3] = 100;
			q1 = 250; q2 =  500; q3 = 750;
			goto chg;
		case '\r' :
			update_wave(w);
			lrg_close();
			wclwin();
			draw_main_screen();
			draw_fkeys();
			draw_times();
			draw_fb();
			return;
		case 0x1B:
			lrg_close();
			wclwin();
			draw_main_screen();
			draw_fkeys();
			draw_times();
			draw_fb();
			if(!confirm("DISCARD CHANGES"))
				goto relrg;
			return; } }
}

/*
 * Build a DTMF tone-pair
 */
void build_dtmf(unsigned f, unsigned duration)
{
#ifdef DEMO
	message("SOUND CARD DISABLED FOR DEMO");
	beep((dtmf_tones[f][0]+dtmf_tones[f][1])/2, duration/8);
#else
	int z1, z2;
	unsigned i, o1, o2, f1, f2;
	f1 = dtmf_tones[f][0];
	f2 = dtmf_tones[f][1];

	poke(voc_seg, 0, 1);	// Voice data record
	pokew(voc_seg, 1, duration+2);
	poke(voc_seg, 3, 0);	// Zero high byte of length
	poke(voc_seg, 4, 0x83);	// 8000khz
	poke(voc_seg, 5, 0);	// 8-bit PCM
	voc_off = 6;
	for(i=o1=o2=0; i < duration; ++i) {
		z1 = lst[o1>>3] / 2;
		z2 = hst[o2>>3] / 2;
		poke(voc_seg, voc_off++, (z1 + z2) ^ 0x80);
		if((o1 += f1) >= 8000)
			o1 -= 8000;
		if((o2 += f2) >= 8000)
			o2 -= 8000; }
	poke(voc_seg, voc_off, 0);
#endif
}

/*
 * Build a block of silence
 */
void build_silence(unsigned duration)
{
#ifdef DEMO
	delay(duration/8);
#else
	unsigned i;

	poke(voc_seg, 0, 1);	// Voice data record
	pokew(voc_seg, 1, duration+2);
	poke(voc_seg, 3, 0);	// Zero high byte of length
	poke(voc_seg, 4, 0x83);	// 8000khz
	poke(voc_seg, 5, 0);	// 8-bit PCM
	voc_off = 6;
	for(i=0; i < duration; ++i)
		poke(voc_seg, voc_off++, 0x7F);
	poke(voc_seg, voc_off, 0);
#endif
}

/*
 * DTMF tone pair editor
 */
void edit_tones(void)
{
	unsigned i, c, tone, dt[16][2];
	char f;
	static unsigned char p, t;

	wopen(23, 3, 32, 17, WSAVE|WCOPEN|WBOX1|0x17);
	memcpy(dt, dtmf_tones, sizeof(dt));
	wcursor_off();
	wgotoxy(4, 0);
	wputs("*** DTMF TONE EDITOR ***");
	wgotoxy(0, 11);
	wputs("\x1B\x18\x19\x1A=select      PgUp/PgDn=+/-\n");
	wputs("0-9=entry        BS=correction\n");
	wputs("   --undo--        --normal--\n");
	wputs("F1=dig F2=all    F3=dig F4=all");
	cflag = f = 0;
	for(;;) {
		tone = dt[p &= 15][t &= 1];
		for(i=0; i < 16; ++i) {
			wgotoxy((i < 8) ? 1 : 18, (i & 7)+2);
			wprintf("%c:", dtmf_digits[i]);
			if((i == p) && !t)
				*W_OPEN = 0x71;
			wprintf("%4u", dt[i][0]);
			*W_OPEN = 0x17;
			wputc('/');
			if((i == p) && t)
				*W_OPEN = 0x71;
			wprintf("%-4u", dt[i][1]);
			*W_OPEN = 0x17; }
		if(isdigit(c=wgetc())) {
			if(!f) {
				tone = 0;
				f = -1; }
			if((tone = (tone * 10) + (c - '0')) > 8000)
				tone = 8000;
			dt[p][t] = tone;
		chg:cflag = -1;
			continue; }
		if(f) {
			if(c == _KBS) {
				dt[p][t] = tone = tone / 10;
				goto chg; }
			if(tone < 100)
				dt[p][t] = tone = 100;
			f = 0; }
		switch(c) {
		case '\n' :
			memcpy(dtmf_tones, dt, sizeof(dtmf_tones));
			wclose();
			return;
		case 0x1B :
			if(!confirm("DISCARD CHANGES"))
				continue;
			wclose();
			return;
		case _KDA :	++p;	continue;
		case _KUA :	--p;	continue;
		case _KRA :
			if(++t > 1)
				p = (p + 8) & 15;
			continue;
		case _KLA :
			if(!(t--))
				p = (p + 8) & 15;
			continue;
		case _KPU : if(tone < 8000) ++tone;		goto settone;
		case _KPD : if(tone > 1)	--tone;
		settone: dt[p][t] = tone;				goto chg;
		case _K1 :	tone = dtmf_tones[p][t];	goto settone;
		case _K2 :	memcpy(dt, dtmf_tones, sizeof(dt));	goto chg;
		case _K3 :	tone = dtmf_otones[p][t];	goto settone;
		case _K4 :	memcpy(dt, dtmf_otones, sizeof(dt)); goto chg;
	} }
}

/*
 * -- Sound-Blaster MIXER control functions --
 */

// Set the (L,R) output levels for a channel
// Level goes 0-31 (5 bits)
void set_level(unsigned channel, unsigned l, unsigned r)
{
	unsigned a, v1, v2;
	if(sct == 6) {	// SB16 mixer
		v1 = l << 3;
		v2 = r << 3;
		switch(channel) {
		case MASTER	: a = 0x30;	goto w16;
		case VOICE	: a = 0x32;	goto w16;
		case MIDI	: a = 0x34;	goto w16;
		case CD		: a = 0x36;	goto w16;
		case LINE	: a = 0x38;
		w16:
			mix_write(a,	v1);	// Set left channel
			mix_write(a+1,	v2); }	// Set right channel
		return; }
	// SPpro mixer
	v1 = ((l << 3) & 0xE0) | ((r >> 1) & 0x0E);
	switch(channel) {
	case MASTER : a = 0x22;	goto wpro;
	case VOICE	: a = 0x04;	goto wpro;
	case MIDI	: a = 0x26;	goto wpro;
	case CD		: a = 0x28; goto wpro;
	case LINE	: a = 0x2E;
	wpro:
		mix_write(a, v1); }
}

// Initialize the sound card mixer
void update_mix(void)
{
	static char reset = 0;
	if(!reset) {
		mix_write(0x00, 0);
		reset = -1; }
	set_level(MASTER, SPKmain, DTMFmain);
	set_level(VOICE, 0, DTMFlevel);
	set_level(MIDI, 0, 0);
	set_level(CD, 0, 0);
	set_level(LINE, SPKlevel, 0);
}

/*
 * Display a bar graph of mixer setting
 */
void bar_graph(unsigned bar)
{
	unsigned s, x;
	wputc(' ');
	x = W_OPEN->WINcurx;
	s = bar;
	while(s--)
		wputc(0xFE);
	wcleol();
	W_OPEN->WINcurx = x + 33;
	wprintf("[%2u]", bar);
}

/*
 * Edit mixer levels
 */
void edit_mix(void)
{
	unsigned step, *p, DTMF, SPK;
	static unsigned select;

	DTMF = DTMFlevel;
	SPK = SPKlevel;

	step = (sct == 6) ? 1 : 4;
	wopen(15, 9, 52, 5, WSAVE|WCOPEN|WBOX1|0x17);
	wcursor_off();
	wgotoxy(13, cflag = 0); wputs("*** Mixer controls ***");
redraw:
	wgotoxy(1, 1);
	if(select == 0) {
		*W_OPEN = 0x71;
		p = &DTMF; }
	wputs("DTMF level:");
	*W_OPEN = 0x17;
	bar_graph(DTMF);
	wgotoxy(1, 2);
	if(select == 1) {
		*W_OPEN = 0x71;
		p = &SPK; }
	wputs("SPK  level:");
	*W_OPEN = 0x17;
	bar_graph(SPK);
	for(;;) switch(wgetc()) {
	case _KUA :
	case _KDA :
		select = !select;
		goto redraw;
	case _KRA :
		if((*p + step) <= 31)
			*p += step;
		else
			*p = 31;
	chg:cflag = -1;
		goto redraw;
	case _KLA :
		if(*p > step)
			*p -= step;
		else
			*p = 0;
		goto chg;
	case 0x1B :
		if(!confirm("DISCARD CHANGES"))
			continue;
		wclose();
		return;
	case '\n' :
		DTMFlevel = DTMF;
		SPKlevel = SPK;
		update_mix();
		wclose();
		return; }
}

/*
 * Issue confirmation prompt & want for Y/N response
 */
int confirm(char *p)
{
	unsigned l;
	if(cflag) {
		l = strlen(p) + 17;
		wopen(38-(l/2), 10, l, 3, WSAVE|WCOPEN|WBOX1|0x4E);
		wprintf("Confirm %s (Y/N)?", p);
		for(;;) switch(wgetc()) {
		case '\n' :
		case 'y' :
		case 'Y' :
			wclose();
			return -1;
		case 0x1B :
		case 'n' :
		case 'N' :
			wclose();
			return 0; } }
	return -1;
}

/*
 * Copy a filename & append extension
 */
void copy_file(char *d, char *p)
{
	unsigned char c, dot;
	dot = -1;
	while(*d = toupper(c = *p++)) {
		if(c == '.')
			dot = 0;
		++d; }
	*d = 0;
	if(dot)
		strcpy(d, ".INI");
}

/*
 * Log an entry to the function buffer
 */
void log_fb(unsigned char e)
{
	fb[fbp] = e;
	fbp = (fbp + 1) % sizeof(fb);
}

/*
 * Display the function buffer
 */
void draw_fb(void)
{
	unsigned i, p;

	wgotoxy(3, 3);
	*W_OPEN = 0x70;
	i = p = 0;
	if(fw) {
		p = fbp;
		wputc(fb[p++]);
		++i; }
	do {
		p = p % sizeof(fb);
		if(p == fbp)
			wputc(' ');
		else
			wputc(fb[p++]); }
	while(++i < sizeof(fb));
	*W_OPEN = 0x67;
}

/*
 * Draw a key on the terminal window in specified color
 */
void draw_t_key(unsigned k, unsigned char attr)
{
	unsigned x, y;
	unsigned char a;
	static unsigned tx[] = {
		0, 1, 2,  4, 5, 6,  8, 9, 10,  13, 12, 14,  3, 7, 11, 15 };

	static char *keys1[] = {
	"qz.", "abc", "def", "   ",
	"ghi", "jkl", "mno", "   ",
	"prs", "tuv", "wxy", "   ",
	"   ", "   ", "   ", "   " };

	static char *keys2[] = {
	" 1 ", " 2 ", " 3 ", " A ",
	" 4 ", " 5 ", " 6 ", " B ",
	" 7 ", " 8 ", " 9 ", " C ",
	" * ", " 0 ", " # ", " D " };

	k = tx[k];
	x = ((k % 4) * 7) + 5;
	y = ((k / 4) * 2) + 5;

	a = *mwin;
	if(attr)
		*mwin = (a >> 4) | (a << 4);
	w_gotoxy(x, y, mwin); w_printf(mwin, "\xDA%s\xBF", keys1[k]);
	w_gotoxy(x,++y,mwin); w_printf(mwin, "\xC0%s\xD9", keys2[k]);
	*mwin = a;
}

/*
 * Build a DTMF/silence digit and play it
 */
void dtmf(unsigned d)
{
	lpt_set(LPT_DTMF);
	draw_t_key(d, -1);
	log_fb(dtmf_digits[d]);
	while(voc_status);
	build_dtmf(d, tduration);
	play_voc(voc_seg);
	while(voc_status);
	draw_t_key(d, 0);
	build_silence(sduration);
	play_voc(voc_seg);
	lpt_clr(LPT_DTMF);
}

/*
 * Flash the switchhook
 */
void flash()
{
	if(hook) {
		wopen(14, 14, 10, 3, WSAVE|WCOPEN|WBOX1|REVERSE);
		wputs(" FLASH  ");
		log_fb('F');
		lpt_clr(LPT_HOOK);
		while(voc_status);
		build_silence(fduration);
		play_voc(voc_seg);
		while(voc_status);
		lpt_set(LPT_HOOK);
		wclose(); }
}

/*
 * Pause for an interval
 */
void pause()
{
	wopen(14, 14, 10, 3, WSAVE|WCOPEN|WBOX1|REVERSE);
	wputs(" PAUSE  ");
	log_fb('P');
	while(voc_status);
	build_silence(pduration);
	play_voc(voc_seg);
	while(voc_status);
	wclose();
}

/*
 * Set on/off hook status
 */
void hookswitch(unsigned char s)
{
	wgotoxy(13, 14);
	if(hook != s) {
		log_fb('H');
		if(hook = s)
			lpt_set(LPT_HOOK);
		else
			lpt_clr(LPT_HOOK); }
	if(hook) {
		*W_OPEN = 0x74;
		wputs(" OFF HOOK ");
		*W_OPEN = 0x67; }
	else
		wputs(" ON  HOOK ");
}

/*
 * Display a duration in samples & milliseconds
 */
show_duration(unsigned d)
{
	wprintf("%5u (%4u.%u ms)", d, d / 8, ((d%8)*9)/7);
}

/*
 * Edit the time table
 */
edit_time(unsigned *t, char *p)
{
	unsigned c, v;
	unsigned char f;

	v = *t;
	cflag = f = 0;
	wopen(20, 8, 40, 8, WSAVE|WCOPEN|WBOX1|0x17);
	wgotoxy(0, 2);
	wputs(" \x1B\x1A  = Adjust by 1\n");
	wputs(" \x18\x19  = Adjust by 10\n");
	wputs(" PuPd= Adjust by 100\n");
	wputs(" 0-9 = direct entry  (BS=correct)");
	for(;;) {
		wgotoxy(0, 0);
		wprintf("%s: ", p);
		show_duration(v);
		if(isdigit(c = wgetc())) {
			if(!f) {
				v = 0;
				f =  -1; }
			if((v=(v * 10) + (c - '0')) > 65500)
				v = 65500;
		chg:cflag = -1;
			continue; }
		if(c == _KBS) {
			f = -1;
			v /= 10;
			goto chg; }
		f = 0;
		switch(c) {
		case _KLA: if(v) --v;						goto chg;
		case _KRA: if(v < 65500) ++v;				goto chg;
		case _KDA: v = (v < 10) ? 0 : v-10;			goto chg;
		case _KUA: v = (v < 65490) ? v+10 : 65500;	goto chg;
		case _KPD: v = (v < 100) ? 0 : v-100;		goto chg;
		case _KPU: v = (v < 65400) ? v+100 : 65500;	goto chg;
		case '\n' :
			*t = v;
			wclose();
			return;
		case 0x1B :
			if(confirm("DISCARD CHANGES")) {
				wclose();
				return; } } }
}

/*
 * Edit and perform a function key
 */
void edit_fkey(unsigned k)
{
	unsigned c, i;
	unsigned char *p;
	wopen(10, 10, 60, 3, WSAVE|WCOPEN|WBOX1|0x17);
	p = fkeys[k];
	for(;;) {
		wclwin();
		wprintf("F%-2u: ", k+1);
		i = 0;
		while(p[i])
			wputc(p[i++]);
		switch(c = toupper(wgetc())) {
		case '1' :
		case '2' :
		case '3' :
		case '4' :
		case '5' :
		case '6' :
		case '7' :
		case '8' :
		case '9' :
		case '0' :
		case '*' :
		case '#' :
		case 'A' :
		case 'B' :
		case 'C' :
		case 'D' :
		case 'F' :
		case 'H' :
		case 'P' :
		case '-' :
		case ' ' :
			if(i < (sizeof(fkeys[0])-1)) {
				p[i++] = c;
				p[i] = 0; }
			continue;
		case _KHO :
			i = 1;
		case _KBS :
			if(i)
				p[--i] = 0;
			continue;
		case 0x1B :
			wclose();
			return;
		case '\n' :
			wclose();
			while(c = *p++) {
				draw_fb();
				switch(c) {
				case 'F' :	flash();						continue;
				case 'P' :	pause();						continue;
				case 'H' :	hookswitch(hook ? 0 : -1);		continue; }
				for(i=0; i < 16; ++i) {
					if(dtmf_digits[i] == c) {
						dtmf(i);
						break; } } }
			return; } }
}

/*
 * Build the main screen layout
 */
void draw_main_screen(void)
{
	unsigned i;
	wgotoxy(24, 0); wputs("*** DTMF/Flash Test Set ***");

	wgotoxy(40, 17); wputs("0-9, * , #, A-D = DTMF dialing");
	wgotoxy(40, 18); wputs("H)ookswitch   F)lash     P)ause");
	wgotoxy(40, 19); wputs("E)rase-log    M)ixer     S)torage");
	wgotoxy(40, 20); wputs("Q)uit");
	wgotoxy(40, 21); wputs("edit: T)ones  W)aveform  I)ntervals");
	wcursor_off();
	for(i=0; i < 16; ++i)
		draw_t_key(i, 0);
	hookswitch(0);
}

/*
 * Display the function key settings
 */
void draw_fkeys(void)
{
	unsigned i;
	for(i=0; i < 10; ++i) {
		wgotoxy(40, 5+i);
		wprintf("F%-2u: %s",i+1, fkeys[i]); }
}

/*
 * Display the timing intervals
 */
void draw_times(void)
{
	wgotoxy(4, 17); wputs("Tone   :"); show_duration(tduration);
	wgotoxy(4, 18); wputs("Silence:"); show_duration(sduration);
	wgotoxy(4, 19); wputs("Flash  :"); show_duration(fduration);
	wgotoxy(4, 20); wputs("Pause  :"); show_duration(pduration);
}

/*
 * Display a message in the main window
 */
void message(unsigned char *m)
{
	w_gotoxy(0, 1, mwin);
	w_cleol(mwin);

	if(m) {
		if(*m == '!') {
			*mwin = 0x4E;
			beep(500, 300);
			++m; }
		else
			*mwin = 0x17;
		w_gotoxy(36 - (strlen(m)/2), 1, mwin);
		w_putc(' ', mwin);
		w_puts(m, mwin);
		w_putc(' ', mwin);
		*mwin = 0x67; }
}

/*
 * Edit a string
 */
int edit_string(unsigned char *prompt, unsigned char name[], unsigned length)
{
	unsigned i, j;
	i = (j = strlen(prompt)) + length + 4;
	wopen(40-(i/2), 8, i, 3, WSAVE|WCOPEN|WBOX1|0x17);
	wputs(prompt);
	wputs(": ");
	for(;;) {
		switch(wgets(j+2, 0, name, length)) {
		case 0x1B :
			wclose();
			return 0;
		case '\n' :
			wclose();
			return -1; }
		beep(500, 300); }
}

/*
 * Skip blanks in the input stream
 */
int skip_blanks(void)
{
	while(isspace(*pp))
		++pp;
	return *pp;
}

/*
 * Parse a delimited element from the input stream
 *
 * Returns: 0=Found terminator, 1=Hit-end, 2=Length exceeded 3=NoData
 */
int parse(char *d, char t, unsigned l)
{
	unsigned char c, r;
	if(!skip_blanks())
		return 3;
	r = 1;
	while(c = toupper(*pp++)) {
		if(isspace(c))	// Ignore embedded spaces
			continue;
		if(c == t) {	// Hit terminator, end
			r = 0;
			break; }
		if(!l--) {		// Too long
			r = 2;
			break; }
		*d++ = c; }
	*d = 0;
	return r;
}

/*
 * Get a decimal number from the input stream
 */
unsigned get_number()
{
	unsigned char word[50];
	parse(word, ',', sizeof(word)-1);
	return atoi(word);
}

#ifndef DEMO
/*
 * Read a configuration file and establish tester settings
 */
int read_config(void)
{
	unsigned i;
	FILE *fp;
	unsigned char buffer[100], word[10], *p;
	static unsigned char *inames[] = {
		"F1", "F2", "F3", "F4", "F5", "F6", "F7", "F8", "F9", "F10",	// 0-9
		"T1", "T2", "T3", "T4", "T5", "T6", "T7", "T8", "T9", "T0",
		"T*", "T#", "TA", "TB", "TC", "TD",			// 10-25
		"TIME",		// 26
		"LWAVE",	// 27
		"HWAVE",	// 28
		"MIX",		// 29
	0 };
	copy_file(buffer, filename);
	if(!(fp = fopen(buffer, "r"))) {
		sprintf(emsg = ebuffer, "!Cannot READ: %s", buffer);
		return -1; }
	while(fgets(pp = buffer, sizeof(buffer)-1, fp)) {
		switch(parse(word, '=', sizeof(word)-1)) {
		default:
			sprintf(emsg = ebuffer, "!Bad input: %s", buffer);
			fclose(fp);
			return -1;
		case 3 :	continue;
		case 0 : }

		for(i=0; p = inames[i]; ++i) {
			if(!strcmp(word, p))
				goto found; }
		sprintf(emsg = ebuffer, "!Unknown item: %s", word);
		fclose(fp);
		return -1;

	found:
		skip_blanks();
		if(i < 10) {
			strcpy(fkeys[i], pp);
			continue; }
		if(i < 26) {
			dtmf_tones[i-=10][0] = get_number();
			dtmf_tones[i][1] = get_number();
			continue; }
		switch(i) {
		case 26 :		// TIME
			tduration = get_number();
			sduration = get_number();
			fduration = get_number();
			pduration = get_number();
			continue;
		case 27 :		// Lwave
			for(i=0; i < 9; ++i)
				lwave[i] = get_number();
			continue;
		case 28 :		// Hwave
			for(i=0; i < 9; ++i)
				hwave[i] = get_number();
			continue;
		case 29 :		// Mix
			DTMFlevel = get_number();
			SPKlevel = get_number();
			DTMFmain = get_number();
			SPKmain = get_number();
			continue; }
	}

	fclose(fp);
	return 0;
}

/*
 * Write current tester settings to a config file
 */
void write_config(void)
{
	unsigned i;
	char buffer[60];
	FILE *fp;

	copy_file(buffer, filename);
	if(!(fp = fopen(buffer, "w"))) {
		sprintf(emsg = ebuffer, "!Cannot WRITE: %s", buffer);
		return; }
	fprintf(fp, "TIME=%u,%u,%u,%u\n", tduration, sduration, fduration, pduration);
	fprintf(fp, "LWAVE=");
	for(i=0; i < 9; ++i) {
		if(i) putc(',', fp);
		fprintf(fp, "%u", lwave[i]); }
	fprintf(fp, "\nHWAVE=");
	for(i=0; i < 9; ++i) {
		if(i) putc(',', fp);
		fprintf(fp, "%u", hwave[i]); }
	fprintf(fp, "\nMIX=%u,%u,%u,%u\n", DTMFlevel, SPKlevel, DTMFmain, SPKmain);

	for(i=0; i < 16; ++i)
		fprintf(fp, "T%c=%u,%u\n", dtmf_digits[i], dtmf_tones[i][0], dtmf_tones[i][1]);

	for(i=0; i < 10; ++i) {
		if(*fkeys[i])
			fprintf(fp, "F%u=%s\n", i+1, fkeys[i]); }
	fclose(fp);
}
#endif
	
main(int argc, char *argv[])
{
	unsigned i, c;
	static unsigned char *imenu[] = {
		"DTMF on-time",
		"DTMF spacing",
		"FLASH duration",
		"PAUSE duration",
		0 };
	static unsigned char *cmenu[] = {
		"Read config",
		"Write config",
		0 };
	static unsigned char *smenu[] = {
		"Low tone waveform",
		"High tone waveform",
		"Copy low to high",
		"Copy high to low",
		0 };
	static unsigned is, cs, ws;

	lpt = peekw(0x40, 0x08);

	for(i=1; i < argc; ++i) {
		pp = argv[i];
		switch((toupper(*pp++) << 8) | toupper(*pp++)) {
		case '/D' :		// Enable debug messages
		case '-D' : debug_flag = -1;			continue;
		case 'C=' :		// Load config file
#ifdef DEMO
			abort("FILE FUNCTIONS DISABLED FOR DEMO");
#else
			copy_file(filename, pp);
			if(read_config())
				abort(emsg+1);
			continue;
#endif
		case 'M=' :		// Override mixer main levels
			DTMFmain = get_number();
			SPKmain = get_number();
			if((DTMFmain | SPKmain) > 31)
				abort("Mixer values must be 0-31\n");
			continue;
		case 'P=' :		// Specify LPT port
			if(!(lpt = get_number()))
				abort("LPT must be 1-3 or address\n");
			if(lpt < 4)
				lpt = peekw(0x40, (lpt*2)+6);
			continue;
		default:
			printf("Unknown option: %s\n", argv[i]);
		case '?'<<8:
		case '/?' :
		case '-?' :
			fputs(help, stdout);
		return; } }

	if(!lpt)
		abort("LPT port not found!\n");
	debug("LPT=%04x\n", lpt);
	lpt_clr(0xFF);

	// Set default sinewave and dtmf tones
	memcpy(dtmf_tones, dtmf_otones, sizeof(dtmf_tones));
	select_wave(0);
	build_sine();
	select_wave(1);
	build_sine();

	load_ct_voice();
	if(!(voc_seg = alloc_seg(4096)))
		abort("Cannot allocate memory");

	update_mix();

	// Open main terminal window and draw initial keypad
	mwin = wopen(0, 0, 80, 25, WSAVE|WBOX1|WCOPEN|0x67);

	draw_main_screen();

redraw15:
	draw_fkeys();

redraw2:
	draw_times();

redraw3:
	draw_fb();
	for(;;) {
		if(emsg)
			message(emsg);
		c = toupper(wgetc());
		if(emsg)
			message(emsg = 0);
		cflag = 0;
		switch(c) {
		case '1' : i = 0;	goto dotone;
		case '2' : i = 1;	goto dotone;
		case '3' : i = 2;	goto dotone;
		case '4' : i = 3;	goto dotone;
		case '5' : i = 4;	goto dotone;
		case '6' : i = 5;	goto dotone;
		case '7' : i = 6;	goto dotone;
		case '8' : i = 7;	goto dotone;
		case '9' : i = 8;	goto dotone;
		case '0' : i = 9;	goto dotone;
		case '*' : i = 10;	goto dotone;
		case '#' : i = 11;	goto dotone;
		case 'A' : i = 12;	goto dotone;
		case 'B' : i = 13;	goto dotone;
		case 'C' : i = 14;	goto dotone;
		case 'D' : i = 15;
		dotone: dtmf(i);	goto redraw3;
		case 'F' :			// Flash
			flash();
			goto redraw3;
		case 'P' :			// Pause
			pause();
			goto redraw3;
		case 'W' :			// Waveform editor
			for(;;) {
				if(wmenu(28, 8, WSAVE|WCOPEN|WBOX1|0x17, smenu, &ws)) {
					message(0);
					goto redraw15; }
				switch(ws) {
				case 0 :
				case 1 : edit_sine(ws);	continue;
				case 2 :
					memcpy(hwave, lwave, sizeof(hwave));
					select_wave(1);
					build_sine();
					message("Low->High copied");
					continue;
				case 3 :
					memcpy(lwave, hwave, sizeof(lwave));
					select_wave(0);
					build_sine();
					message("High->Low copied");
					continue; } }
		case 'T' :			// Frequency editor
			edit_tones();
			continue;
		case 'M' :			// Mixer settings
			edit_mix();
			continue;
		case 'I' :			// Interval editor
			for(;;) {
				if(wmenu(28, 8, WSAVE|WCOPEN|WBOX1|0x17, imenu, &is))
					goto redraw2;
				switch(is) {
				case 0 : edit_time(&tduration, imenu[0]); continue;
				case 1 : edit_time(&sduration, imenu[1]); continue;
				case 2 : edit_time(&fduration, imenu[2]); continue; }
				edit_time(&pduration, imenu[3]); }
		case 'S' :			// Storage
#ifdef DEMO
			message("FILE FUNCTIONS DISABLED FOR DEMO");
			continue;
#else
			if(wmenu(28, 8, WSAVE|WCOPEN|WBOX1|0x17, cmenu, &cs))
				continue;
			if(!edit_string("File", filename, sizeof(filename)-1))
				continue;
			switch(cs) {
			case 0 :
				if(!read_config()) {
					select_wave(0);
					build_sine();
					select_wave(1);
					build_sine();
					update_mix(); }
				goto redraw15;
			case 1 : write_config();	goto redraw15; }
#endif
		case 'H' : hookswitch(hook ? 0 : -1);	goto redraw2;
		case _K1 : i=0;	goto fkey;
		case _K2 : i=1;	goto fkey;
		case _K3 : i=2;	goto fkey;
		case _K4 : i=3;	goto fkey;
		case _K5 : i=4;	goto fkey;
		case _K6 : i=5;	goto fkey;
		case _K7 : i=6;	goto fkey;
		case _K8 : i=7;	goto fkey;
		case _K9 : i=8;	goto fkey;
		case _K10: i=9;
		fkey: edit_fkey(i);	goto redraw15;
		case 'E' :
			fbp = fw = 0;
			goto redraw3;
		case 'Q' :
			wclose();
			unload_ct_voice();
			return; }
	strcpy(emsg = ebuffer, "!Bad command key"); }

}
