/*
 * Sound Card voice interfacing functions
 */
#ifndef _MICROC_
	#define	STANDALONE
	#include <stdio.h>
	#define	DURATION	2000
	#define	FREQUENCY	20
	unsigned
		voc_seg,
		voc_off;
#endif

/* Sound BLASTER definitions */
#define	MIN_VOC_BLOCK	10			/* Minimum .VOC block size */

#define	VDATA			0x01		/* Voice DATA block */
#define	VCONT			0x02		/* Voice CONTINUATION block */
#define	VSILENCE		0x03		/* Voice SILENCE block */
#define	VMARKER			0x04		/* Voice MARKER block */
#define	VASCII			0x05		/* Voice ASCII block */
#define	VEXTEND			0x08		/* Extended block */
#define	VEDATA			0x09		/* Extended voice data */

unsigned
	sct,				/* Sound card type */
	scb,				/* Sound card base address */
	ct_seg,				/* CT-VOICE load segment */
	voc_status;			/* Status word from VOC buffer */
char
	debug_flag = 0;		/* Debugging enabled flag */

extern char *ARGV[];

#ifdef DEMO
void debug() { }
void load_ct_voice() { sct=6; scb=0x220; }
void unload_ct_voice() { }
void play_voc() { voc_status = 0; }
void mix_write() { voc_status = 0; }
asm {
callsb:	XOR	AX,AX
	MOV	DGRP:_voc_status,AX
	RET
}
#else
/*
 * Debugging output
 */
register debug(unsigned args)
{
	unsigned *ptr;
	char buffer[81];

	ptr = nargs() * 2 + &args;
	if(debug_flag) {
		_format_(ptr, buffer);
		fputs(buffer, stdout); }
}

/*
 * Unload CT-VOICE driver
 */
void unload_ct_voice(void)
{
	debug("Voice driver unloaded.\n");
	asm {
		MOV		BX,9			; Terminate driver
		CALL	callsb			; Call driver
	}
	free_seg(ct_seg);
}

/*
 * Catastrophic sound-blaster failure
 */
register sb_fail(unsigned args)
{
	char buffer[81];

	_format_(nargs() * 2 + &args, buffer);
	fputs(buffer, stdout);
	unload_ct_voice();
	exit(-1);
}

/*
 * Load a far memory buffer from an open file
 */
load_far(unsigned seg, unsigned offset, unsigned size, FILE *fp)
{
	unsigned char buffer[1024];
	unsigned s;

	while(size) {
		if(s = fget(buffer, (size > sizeof(buffer)) ? sizeof(buffer) : size, fp)) {
			copy_seg(seg, offset, get_ds(), buffer, s);
			offset += s;
			size -= s;
			continue; }
		break; }

	return offset;
}

/*
 * Load CT-VOICE driver
 */
void load_ct_voice(void)
{
	unsigned sh, sl, j;
	char buffer[100], name[13], *ptr;
	FILE *fp;

	/* Read SOUND variable & build path to driver */
	if(getenv("SOUND", buffer))
		strcat(buffer, "\\DRV\\");
	else {
		strcpy(buffer, ARGV[0]);
		sh = sl = 0;
		while(j = buffer[sh++]) {
			if(j == '\\')
				sl = sh; }
		buffer[sl] = 0; }
	strcat(buffer, "CT-VOICE.DRV");

	/* Determine size of driver file */
	if(find_first(buffer, 0, name, &sh, &sl, &j, &j, &j)) {
		printf("Cannot locate: %s\n", buffer);
		exit(-1); }

	/* Allocate external (far) buffers */
	if(	!(ct_seg = alloc_seg((sl+15)/16)) )
		abort("Cannot allocate memory");

	/* Load CT-VOICE driver into memory*/
	fp = fopen(buffer, "rvbq");
	debug("Voice driver: seg=%04x size=%u", ct_seg,
		load_far(ct_seg, 0, sl, fp));
	fclose(fp);

	asm {	/* Patch in far calling vector */
		MOV		AX,DGRP:_ct_seg			; Get segment
		MOV		word ptr CS:ctseg,AX	; Set segment
		XOR		BX,BX					; Get zero
		CALL	callsb					; Get version
	}
	j = nargs();
	debug(" %u.%u", j>>8, j & 0xFF);

	/* Assume settings from BLASTER environment variable if it exists */
	if(getenv("BLASTER", ptr = buffer)) {
		while(*ptr) switch(toupper(*ptr++)) {
			case 'A' :		/* Address */
				debug(" A%x", scb = atox(ptr));
				scb;
				asm {
					MOV		BX,1
					CALL	callsb
					}
				break;
			case 'I' :		/* Interrupt */
				debug(" I%u", j=atoi(ptr));
				j;
				asm {
					MOV		BX,2
					CALL	callsb
				}
				break;
			case 'D' :		/* DMA channel */
				debug(" D%u", j=atoi(ptr));
				j;
				asm {
					MOV		BX,19
					CALL	callsb
				}
				break;
			case 'T' :		/* Card type */
				debug(" T%u", sct=atoi(ptr));
				sct;
				asm {
					MOV		BX,20
					CALL	callsb;
				} } }
	debug("\n");

	asm {	/* Initialize the card */
		MOV		BX,3			; Init card
		XOR		AX,AX			; Function 0
		CALL	callsb			; Call driver
	}

	/* Non-zero (accumulator content) means initialziation failed */
	if(nargs())
		sb_fail("Sound card failed init.\n");

	asm {	/* Establish pointer to status word */
		MOV		BX,5			; Set status word
		PUSH	DS				; Get data seg
		POP		ES				; Into extra
		MOV		DI,OFFSET DGRP:_voc_status; Point to status word
		CALL	callsb			; Call the blaster
	}
}


/*
 * Play a VOC file
 */
void play_voc(segment) asm
{
;		XOR		AX,AX
		MOV		BX,6			; Start voice function
		MOV		ES,4[BP]		; Get voice segment
		MOV		DI,0			; Offset zero
		CALL	callsb			; Call the driver
}

// Write a byte to the sound card mixer
mix_write(unsigned char a, unsigned char d)
{
	out(scb+4, a);
	out(scb+5, d);
}
/* Read a byte from the sound card mixer
unsigned mix_read(unsigned char a)
{
	out(scb+4, a);
	return in(scb+5);
} */

/*
 * Assembly stub to perform inter-segment call to CT-VOICE driver
 */
asm {
callsb:	DB		9Ah				; Call instruction
		DW		0h				; Offset 0
ctseg:	DW		0h				; Segment
		RET
}
#endif

#ifdef STANDALONE
main()
{
	unsigned i;
	char c, d;

	debug_flag = -1;		// Enable debug messages

	load_ct_voice();
	voc_seg = alloc_seg(4096);

	// Build a triangle wave for output
	poke(voc_seg, 0, 1);	// Voice data record
	pokew(voc_seg, 1, DURATION+2);
	poke(voc_seg, 3, 0);	// Zero high byte of length
	poke(voc_seg, 4, 0x83);	// 8000khz
	poke(voc_seg, 5, 0);	// 8-bit PCM
	voc_off = 6;
	for(i=0; i < DURATION; ++i) {
		poke(voc_seg, voc_off++, c);
		if(d) {		// Negative waveform
			if((c -= FREQUENCY) <= -100)
				d = 0; }
		else {		// Positive waveform
			if((c += FREQUENCY) >= 100)
				d = -1; } }
	poke(voc_seg, voc_off, 0);

	// Play the wave
	play_voc(voc_seg);

	// Wait for it to complete
	while(voc_status);

	// Release the driver
	unload_ct_voice();
}
#endif
