/*
 * MacMon
 *
 * compile with: cc MM -pof M=S
 */
#include <stdio.h>
#include <window.h>
#include <file.h>

/*
 * -------------------- PACKET DRIVER INTERFACE --------------------
 */

/* PKT_INT must be in DECIMAL, as it is used by C and inline assembly */
#define	PKT_INT		96				/* Packet driver interrupt (0x60) */
#define	PKT_DATA	1500			/* Size of data area in packet */
#define	PKT_SIZE	PKT_DATA+14		/* Size of packet with overhead */
#define	PKT_EASIZE	6				/* Size of ethernet address */
/* PKT_WINDOW must be a power of 2, so we can mask with PKT_WINDOW-1 */
#define	PKT_WINDOW	32				/* Size of packet receive queue */

/*
 * Structure of information returned by pkt_driver_info()
 */
struct PD_driver_info {
	unsigned version;				/* Driver version */
	unsigned char class;			/* Interface class */
	unsigned type;					/* Interface type */
	unsigned char number;			/* Interface number */
	unsigned char functionality;	/* Driver functionality */
	unsigned char name[9]; };		/* Driver name (?) */

/*
 * Structure of packet as sent/received by driver
 */
struct PD_packet {
	unsigned char to_addr[PKT_EASIZE];
	unsigned char from_addr[PKT_EASIZE];
	unsigned ptype;
	unsigned char data[PKT_DATA]; };

/*
 * Structure of a packet receive queue entry
 */
struct PD_queue {
	unsigned handle;
	unsigned length;
	struct PD_packet packet; };

/*
 * Packet driver interface global variables
 */
struct PD_queue
	pkt_queue[PKT_WINDOW];			/* Queue for received packets */

unsigned
	pkt_head = 0,					/* Head of RX packet queue */
	pkt_tail = 0;					/* Tail of RX packet queue */

unsigned reverse(value) asm
{
	MOV		AH,4[BP]
	MOV		AL,5[BP]
}

/*
 * Obtain packet driver information
 *
 * This function *MUST* be called before pkt_access_type() is called, in
 * order to insure that the SAVEDS location is initialized with the Micro-C
 * data segment. Another reason is that you need to obtain the interface
 * class/type/number so that you can pass them to pkt_access_type();
 *
 * Arguments:
 *	info_struct	= "PD_driver_info" structure to fill in.
 *
 * Returns:
 *	0			= Success
 *	!0			= Packet driver error code
 */
int pkt_driver_info(info_struct) asm
{
		MOV		CS:SAVEDS,DS		; Save our data seg
		MOV		AX,01ffh			; Request info
pi1:	INT		PKT_INT				; Ask packet driver
		JC		di2					; Error occured
		PUSH	DS					; Save driver DS
		MOV		DS,CS:SAVEDS		; Restore DATA segment
		POP		ES					; Point to driver data
		MOV		DI,4[BP]			; Get struct
		MOV		0[DI],BX			; Save version
		MOV		2[DI],CH			; Save Class
		MOV		3[DI],DX			; Save type
		MOV		5[DI],CL			; Save number
		MOV		6[DI],AL			; Save functionality
		LEA		BX,7[DI]			; Point to name
di1:	MOV		AL,ES:[SI]			; Get data
		MOV		[BX],AL				; Copy it
		INC		SI					; Advance source
		INC		DI					; Advance dest
		AND		AL,AL				; End of string?
		JNZ		di1					; Copy it all
		XOR		AX,AX				; Return 0
		JMP		di3
; Report driver error
di2:	MOV		AL,DH				; Get error code
		MOV		AH,0FFh				; Insure non-zero
di3:
}

/*
 * Obtain access to a packet type
 *
 * Arguments:
 *	if_class	= Interface class  \
 *	if_type		= Interface type    > - From pkt_driver_info()
 *	if_number	= Interface number /
 *	type		= Pointer to packet type to access
 *	typelen		= Length of packet type (0=Access ALL packets)
 *	handle		= Pointer to 'int' variable to receive handle
 *
 * Returns:
 *	0			= Success
 *	!0			= Packet driver error code
 */
int pkt_access_type(if_class, if_type, if_number, type, typelen, handle) asm
{
		MOV		AL,14[BP]			; Get if_class
		MOV		BX,12[BP]			; Get if_type
		MOV		DL,10[BP]			; Get if_number
		MOV		SI,8[BP]			; Get type pointer
		MOV		CX,6[BP]			; Get typelen
		MOV		DI,OFFSET pktrdy	; Point to handler
		PUSH	CS					; Save code segment
		POP		ES					; ES = handler segment
		MOV		AH,02h				; Indicate "access type" request
pi2:	INT		PKT_INT				; Call packet driver
		JC		di2					; Report error
		MOV		BX,4[BP]			; Get handle
		MOV		[BX],AX				; Save handle
		XOR		AX,AX				; Indicate success
}

/*
 * Release access to a packet type
 *
 * Arguments:
 *	handle		= Handle returned by pkt_access_type()
 *
 * Returns:
 *	0			= Success
 *	!0			= Packet driver error code
 */
int pkt_release_type(handle) asm
{
		MOV		BX,4[BP]			; Get handle
		MOV		AH,03h				; Indicate "release type" request
pi3:	INT		PKT_INT				; Ask for it
		JC		di2					; Indicate failed
		XOR		AX,AX				; Indicate success
}

/*
 * Send a packet
 *
 * Arguments:
 *	buffer		= Pointer to buffer to send
 *	length		= Length of buffer to send
 *
 * Returns:
 *	0			= Success
 *	!0			= Packet driver error code
 */
int pkt_send(buffer, length) asm
{
		MOV		SI,6[BP]			; Get buffer address
		MOV		CX,4[BP]			; Get length
		MOV		AH,04h				; Indicate "send packet" request
pi4:	INT		PKT_INT				; Ask for it
		JC		di2					; Report error
		XOR		AX,AX				; Indicate success
}

/*
 * Get our ethernet address
 *
 * Arguments:
 *	handle		= Handle returned by pkt_access_type()
 *	buffer		= Pointer to buffer to receive address
 *	length		= Size of buffer (maximum)
 *	alen		= Pointer to 'int' variable to receive size of address
 *
 * Returns:
 *	0			= Success
 *	!0			= Packet driver error code
 */
int pkt_get_address(handle, buffer, length, alen) asm
{
		MOV		BX,10[BP]			; Get handle
		MOV		DI,8[BP]			; Get buffer
		MOV		CX,6[BP]			; Get length
		PUSH	DS					; Save out segment
		POP		ES					; Set ES
		MOV		AH,06h				; Indicate "get address" request
pi5:	INT		PKT_INT				; Ask driver
		JC		di2					; Report error
		MOV		BX,4[BP]			; Get received length
		MOV		[BX],CX				; Save length
		XOR		AX,AX				; Indicate success
}

/*
 * Set the receive mode for a packet
 *
 * Arguments:
 *	handle		= Handle returned by pkt_access_type()
 *	mode		=	1: Turn off receiver
 *					2: Receive only packets sent to our address
 *					3: mode 2 + broadcast packets
 *					4: mode 3 + limited multicast packets
 *					5: mode 3 + all multicast packets
 *					6: all packets
 *
 * Returns:
 *	0			= Success
 *	!0			= Packet driver error code
 */
int pkt_set_receive_mode(handle, mode) asm
{
		MOV		BX,6[BP]			; Get handle
		MOV		CX,4[BP]			; Get mode
		MOV		AH,20				; Set function
pi6:	INT		PKT_INT				; Ask driver
		JC		di2					; Report error
		XOR		AX,AX				; Report success
}

/*
 * Receive packet handler. Called by packet driver when a packet
 * has been received for us.
 *
 * Check to see if we have enough space, and buffer the packet in our
 * receive packet queue, notify the mainline when received.
 */
asm {
; Entry:
;  AX = flag: 0=Allocate buffer, 1=Transfer complete
;  BX = handle
;  CX = Length
; Exit:
;  ES:DI = Buffer to receive packet (0:0=Not available)
pktrdy:	PUSH	DS					; Save our data segment
		MOV		DS,CS:SAVEDS		; Restore Micro-C DS
		AND		AX,AX				; Is this allocate?
		JNZ		pktr2				; No, try next
; Allocate space for a packet
		CMP		CX,PKT_SIZE			; Is packet too large?
		JA		pktr1				; Yes, Fail reception
		MOV		AX,DGRP:_pkt_head	; Get head of queue
		INC		AX					; Advance to next
		AND		AX,PKT_WINDOW-1		; Mask for wrap
		CMP		AX,DGRP:_pkt_tail	; Is queue full?
		JZ		pktr1				; Yes, Fail reception
		MOV		AX,DGRP:_pkt_head	; Get head back
		MOV		DI,PKT_SIZE+4		; Get size
		MUL		DI					; Compute offset
		ADD		AX,OFFSET DGRP:_pkt_queue ; Offset to location
		MOV		DI,AX				; Point to location
		MOV		[DI],BX				; Save handle
		MOV		2[DI],CX			; Save length
		ADD		DI,4				; Skip to data
		PUSH	DS					; Get DS
		POP		ES					; Copy to ES
		JMP SHORT pktrx				; And exit
; No space for buffer, or packet too large - reject packet
pktr1:	MOV		DI,0				; Indicate no buffer
		MOV		ES,DI				; ""
		JMP SHORT pktrx				; And exit
; Packet has been copied into our buffer, notify application
pktr2:	MOV		AX,DGRP:_pkt_head ; Get packet queue head
		INC		AX					; Skip to next
		AND		AX,PKT_WINDOW-1		; Wrap circular buffer
		MOV		DGRP:_pkt_head,AX ; Save new queue head
pktrx:	POP		DS					; Restore data seg
		DB		0CBh				; Far return
SAVEDS	DW		?
}

/*
 * Set the packet interrupt by patching the code
 */
set_interrupt(inum) asm
{
		MOV		AL,4[BP]			; Get the interrupt number
		MOV		byte ptr pi1+1,AL	; Patch
		MOV		byte ptr pi2+1,AL	; Patch
		MOV		byte ptr pi3+1,AL	; Patch
		MOV		byte ptr pi4+1,AL	; Patch
		MOV		byte ptr pi5+1,AL	; Patch
		MOV		byte ptr pi6+1,AL	; Patch
}

/*
 * -------------------- SNIFFER APPLICATION --------------------
 */
#define	MAXMAC	100
#define	MAXNAME	15

#define	FLAG_LOG	0x80		// Log Active/Inactive
#define	FLAG_IP		0x40		// Log IP DNS lookups
#define	FLAG_ALARM	0x03		// Alarms 1-3

#define	TICK	peekw(0x40,0x6C)

unsigned
	H1,		// pkt handle
	Pkt_int = PKT_INT,
	Pkt_type = 0,
	Pkt_mode = 6,
	Day,
	Month,
	Year,
	LastDay,
	Hour,
	Minite,
	Second,
	Csecond,
	LastTick,
	LastSecond,
	LastMinite,
	LastHour,
	HM,
	Mactop,
	Beep = 700,
	ActiveTime = 10,
	Alarm,
	Alarms[3][2];
FILE
	*fp;

struct WINDOW
	*swin,
	*mwin;

unsigned char
	*ptr,
	*Prefix = "",
	*Macfile = "MM.MAC",
	Aactive = 0x0E,
	Sound,
	Mac[MAXMAC][6],
	Flags[MAXMAC],
	Active[MAXMAC],
	Name[MAXMAC][MAXNAME+1],
	Temp[128],
	Temp1[256];

void quit(void)
{
	if(swin) {
		wclose();
		wclose(); }
	if(H1)
		pkt_release_type(H1);
	exit(-1);
}

/*
 * Report an error and exit
 */
register error(unsigned args)
{
	unsigned char *p, buf[80];
	_format_(nargs() * 2 + &args, buf);
	if(fp) {
		fprintf(stderr, "%u: %s\n", Day, buf);
		fputs(p=Temp, stderr);
		putc('\n', stderr);
		while(p < ptr) {
			putc(' ', stderr);
			++p; }
		fputs("^\n", stderr); }
	else {
		fputs(buf, stdout);
		putc('\n', stdout); }
	quit();
}

/*
 * Skip ahead in text file
 */
int skip()
{
	while(isspace(*ptr))
		++ptr;
	return *ptr;
}

/*
 * Expect a specific character in the file
 */
void expect(unsigned char c)
{
	if(skip() != c)
		error("Expected '%c'", c);
	++ptr;
}

/*
 * Expect the end of a token
 */
void end(void)
{
	if(!*ptr)
		return;
	if(!isspace(*ptr))
		error("Syntax error");
}

/*
 * Parse a word from the input line
 */
void parse(unsigned char *d)
{
	while(!isspace(*ptr))
		*d++ = toupper(*ptr++);
	*d = 0;
}

/*
 * Get hex value from command line
 */
unsigned value(unsigned b, unsigned l, unsigned h)
{
	unsigned v, c;
	unsigned char f;
	v = f = 0;
	skip();
	for(;;) {
		c = *ptr;
		if((c >= '0') && (c <= '9'))
			c -= '0';
		else if((c >= 'A') && (c <= 'F'))
			c -= ('A'-10);
		else if((c >= 'a') && (c <= 'f'))
			c -= ('a'-10);
		else {
			if(!f)
				error("Number required");
			if((v < l) || (v > h))
				error("Out of range");
			return v; }
		v = (v * b) + c;
		++ptr;
		f = 255; }
}

/*
 * Read MACADDR.TXT file into memory array
 */
void readmacfile()
{
	unsigned i, j, k;
	unsigned char *p, f;
	static unsigned char *keywords[] = {
		"CURFEW", "ACTIVE", "BEEP",
		0 };

	fp = fopen(Macfile, "rvq");
	while(fgets(ptr = Temp, sizeof(Temp)-1, fp)) {
		++Day;
		switch(skip()) {
		case '#' :
			++ptr;
			parse(Temp1);
			for(i=0; p = keywords[i]; ++i) {
				if(!strcmp(p, Temp1))
					goto found; }
			error("Bad directive: #%s", Temp1);
found:		switch(i) {
			case 0 :		// Alarm
				i = value(10, 1, 3);
				end();
				j = value(10, 0, 23);
				expect(':');
				k = value(10, 0, 59);
				end();
				Alarms[i][0] = (j*60)+k;
				j = value(10, 0, 23);
				expect(':');
				k = value(10, 0, 59);
				end();
				Alarms[i][1] = (j*60)+k;
				continue;
			case 1 :		// Active timer
				ActiveTime = value(10, 1, -1);
				end();
				continue;
			case 2 :		// Beep frequency
				Beep = value(10, 1, 10000);
				end();
				continue; }
		case ';' :
		case 0 : continue; }
		if(Mactop >= MAXMAC)
			error("Too many MACs");
		p = Mac[Mactop];
		for(i=0; i < 6; ++i) {
			if(i)
				expect(':');
			*p++ = value(16, 0, 255); }
		end();

		skip();
		f = FLAG_LOG;
again:	switch(*ptr) {
		case '#' :
			++ptr;
			i = value(10, 0, 3);
			f |= i;
			goto again;
		case '!' : f &= ~FLAG_LOG;	goto next;
		case '+' : f |= FLAG_IP;
next:		++ptr;
			goto again; }
		Flags[Mactop] = f;
		p = Name[Mactop];
		while(*ptr && !isspace(*ptr))
			*p++ = *ptr++;
		*p = 0;
		if(strlen(Name[Mactop++]) > MAXNAME)
			error("Name too long");
		switch(skip()) {
		default: error("Unexpected data");
		case ';' :
		case 0 : ; }
	}
	fclose(fp);
	fp = 0;
	return;
}

/*
 * Locate MAC in database
 */
unsigned findmac(mac) asm
{
		MOV		SI,4[BP]				// Get source mac
		MOV		DI,OFFSET DGRP:_Mac		// Get mac address list
		XOR		CX,CX					// Zero count
; Search for MAC address
TST1:	MOV		AX,[SI]					// Get first
		CMP		AX,[DI]					// Match?
		JNZ		TST2					// No
		MOV		AX,2[SI]				// Second
		CMP		AX,2[DI]				// Match?
		JNZ		TST2					// No
		MOV		AX,4[SI]				// Get third
		CMP		AX,4[DI]				// Match?
		JZ		TST3					// Yes
; Mac not found - advance to next
TST2:	ADD		DI,6					// Skip to next
		INC		CX						// Next
		CMP		CX,DGRP:_Mactop			// In range?
		JB		TST1					// Continue
		MOV		CX,0FFFFh				// Not found (-1)
TST3:	MOV		AX,CX					// Get value
}

unsigned char *mac(unsigned char *m)
{
//	static unsigned char v, xm[13][2];
//	unsigned char *p;
//	p = xm[++v & 1];
	sprintf(Temp, "%02x%02x%02x%02x%02x%02x", m[0], m[1], m[2], m[3], m[4], m[5]);
	return Temp;
}

/*
 * Write record to log
 */
void log(unsigned i, unsigned char ftype, unsigned msg)
{
	sprintf(Temp1, "\n%u:%02u:%02u %s %s",
		Hour, Minite, Csecond, Name[i], msg);
	wputs(Temp1);
	if(Flags[i] & ftype) {
		fputs(Temp1+1, fp);
		putc('\n', fp); }
}

register status(unsigned args)
{
	unsigned char buf[81];
	_format_(nargs() * 2 + &args, buf);
	W_OPEN = swin;
	wgotoxy(0, 0);
	*swin = (Aactive & 2) ? 0x74 : 0x72;
	wputc('1');
	*swin = (Aactive & 4) ? 0x74 : 0x72;
	wputc('2');
	*swin = (Aactive & 8) ? 0x74 : 0x72;
	wputc('3');
	*swin = REVERSE;
	wputs(buf);
	wcleol();
	W_OPEN = mwin;
}

unsigned char Stext[] = { " 4:Quiet 5:Test  MACMON by Dave Dunfield - "#__DATE__"" };

unsigned char Help[] = { "\n\
Use: MM [log_prefix] [options]\n\n\
opts:	D=xx		- packet Driver interrupt	[60]\n\
	M=filename	- Mac definition file		[MM.MAC]\n\
\nDave Dunfield - "#__DATE__"\n" };

//unsigned z;
/*
 * Main demonstration program
 */
main(int argc, char *argv[])
{
	unsigned ps, po;							/* Driver segment,offset */
	unsigned char my_address[PKT_EASIZE];		/* Our ethernet address */
	unsigned alen;								/* Length of address */
	struct PD_driver_info info;					/* Info from driver */
	static char pkt_driver_id[] = {"PKT DRVR"};	/* Driver ID string */
	struct PD_packet *p;						/* Pointer to current packet */
	unsigned l, i;								/* Length of current packet */
	unsigned char *d;

	for(i=1; i < argc; ++i) {
		ptr = argv[i];
		switch((toupper(*ptr++)<<8) | toupper(*ptr++)) {
		case '?'<<8:
		case '-?' :
		case '/?' : abort(Help);
		case 'D=' : Pkt_int = value(16, 0, 255);	continue;
		case 'M=' : Macfile = ptr;					continue;
		} if(*Prefix)
			abort(Help);
		Prefix = ptr-2; }

	readmacfile();
	get_time(&Hour, &Minite, &Second);
	get_date(&Day, &Month, &Year);
	HM = (Hour * 60) + Minite;
	LastTick = TICK;
	Csecond = Second;

	/* Check that a packet driver is present */
	po = peekw(0, Pkt_int*4) + 3;	/* Offset to packet ID text */
	ps = peekw(0, (Pkt_int*4)+2);	/* Segment of packet driver */
	i = l = 0;
	do {
		if(pkt_driver_id[i] != peek(ps, po+i)) {
			printf("No packet driver on INT %02x [%04x:%04x]\n", Pkt_int, ps, po-3);
			quit(); } }
	while(pkt_driver_id[++i]);
	if(Pkt_int != PKT_INT)
		set_interrupt(Pkt_int);

	/* Obtain information from driver and display */
	if(i = pkt_driver_info(info)) {
		printf("Unable to obtain packet driver information, RC=%04x\n", i);
		quit(); }
	printf("Pkt driver: version=%u ", info.version);
	printf("Class/Type/Num=%02x/%04x/%02x ", info.class, info.type, info.number);
	printf("Func=%02x ", info.functionality);
	printf("Name='%s' ", info.name);

	/* Obtain access to our packet type */
	i = Pkt_type ? sizeof(Pkt_type) : 0;
	if(i = pkt_access_type(info.class, info.type, 0, &Pkt_type, i, &H1)) {
		printf("Unable to obtain packet access, RC=%04x\n", i);
		quit(); }

	/* Get our ethernet address */
	if(i=pkt_get_address(H1, my_address, sizeof(my_address), &alen)) {
		printf("Unable to read ethernet address!, RC=%04x\n", i);
		goto do_exit; }

	if(i=pkt_set_receive_mode(H1, Pkt_mode)) {
		printf("Unable to set access mode!, RC=%04x\n", i);
		goto do_exit; }

	swin = wopen(0, 24, 80, 1, WCOPEN|WSAVE|REVERSE);
	mwin = wopen(0, 0, 80, 24, WCOPEN|WSAVE|WWRAP|WSCROLL|0x17);
	wcursor_off();
	status(Stext);
	IOB_size = 2048;
	ptr = "wva";
newfile:
	sprintf(Temp, "%s%02u%02u%02u.LOG", Prefix, i=Year%100, Month, Day);
	if(!(fp = fopen(Temp, ptr)))
		quit();
	fprintf(fp, "%u/%02u/%02u %u:%02u:%02u Begin logging\n",
		LastDay=Day, Month, i, LastHour=Hour, Minite, Csecond);

	for(;;) {
		/* Check for received packets, and display */
		if(pkt_head != pkt_tail) {		/* Packet has been received */
			p = pkt_queue[pkt_tail].packet;
			l = pkt_queue[pkt_tail].length;
			if((i = findmac(p->from_addr)) == -1) {
				if(Mactop >= MAXMAC) {
					wprintf("\nCannot add: %s", mac(p->from_addr));
					goto drop_pkt2; }
				memcpy(Mac[Mactop], d = p->from_addr, 6);
				sprintf(Name[Mactop], "%02x%02x%02x%02x%02x%02x",
					d[0], d[1], d[2], d[3], d[4], d[5]);
				Flags[Mactop] = FLAG_LOG|FLAG_IP;
				Active[i=Mactop] = ActiveTime;
				log(Mactop++, FLAG_LOG, "Discovered");
				goto drop_pkt1; }
			if(!Active[i])
				log(i, FLAG_LOG, "Active");
			Active[i] = ActiveTime;
drop_pkt1:	if(p->ptype == 0x0008)
				process_ip(i, p->data);
			// Process pending alarms
			if(l = Flags[i] & FLAG_ALARM) {
				if((HM < Alarms[l][0]) || (HM > Alarms[l][1])) {
					if((1<<l) & Aactive)
						Alarm = 60; } }
drop_pkt2:	pkt_tail = (pkt_tail+1) & (PKT_WINDOW-1);
			continue; }
		// 18.2*60 = 1092 ticks/minite
		l = (i = TICK) - LastTick;
		Csecond = (l*10)/182 + Second;
		if(Csecond >= 60) {
//wputc('.');
			get_time(&Hour, &Minite, &Second);
			HM = (Hour * 60) + Minite;
			Csecond = Second;
			LastTick = i; }
// if(Csecond != z) { z = Csecond; wprintf("%u:%02u:%02u\n", Hour%100, Minite, Csecond); } */

		if(Csecond == LastSecond)
			continue;
		LastSecond = Csecond;
		switch(wtstc()) {
		case _K1 : i = 2;	goto setalm;
		case _K2 : i = 4;	goto setalm;
		case _K3 : i = 8;	setalm:
			Aactive ^= i;
			status(Stext);
			Alarm = 0;
			break;
		case _K4:
			Alarm = 1;
			break;
		case _K5:
			Alarm = 60;
			break;
		case 0x1B:
			i = malloc(1);
			wclose();
			wclose();
			sound_off();
			printf("Top: %04x\n", i);
			goto do_exit; }

		if(Alarm) {
			--Alarm;
			if(Sound) {
				sound_off();
				Sound = 0; }
			else {
				sound(Beep);
				Sound = 255; } }
		else if(Sound) {
			sound_off();
			Sound = 0; }

		if(Minite == LastMinite)
			continue;
		LastMinite = Minite;
		if(Hour != LastHour) {
			LastHour = Hour;
			get_date(&Day, &Month, &Year);
			if(LastDay != Day) {
				fprintf(fp, "%u/%02u/%02u New day\n", LastDay = Day, Month, Year%100);
				fclose(fp);
				ptr = "wv";	// Don't append on new day
				goto newfile; } }
		for(i=0; i < Mactop; ++i) {
			if(Active[i])
				if(!--Active[i])
					log(i, FLAG_LOG, "Inactive"); }
		}

do_exit:
	if(fp)
		fclose(fp);
	if(i = pkt_release_type(H1))
		printf("Failed to release packet driver, RC=%04x\n", i);
}

process_ip(unsigned index, unsigned char *data)
{
	unsigned i;
	unsigned char *p, *p1;
	static unsigned Lindex = -1;
	static char Ldata[128];

	if(data[9] == 0x11) switch(*(unsigned*)(data+22)) {
	case 0x3500:		// DNS
		if((data[28+2] & 0xF8) == 0) {	// Query
			p = data+(28+12);
			p1 = Temp;
			while(i = *p++) {
				do {
					*p1++ = *p++; }
				while(--i);
				*p1++ = '.'; }
			*--p1 = 0;
			if((index == Lindex) && !strcmp(Temp, Ldata)) {
//				log(index, FLAG_IP, "Dup");
				return; }
			strcpy(Ldata, Temp);
			log(Lindex = index, FLAG_IP, Temp); } }
}

