/*
 * MAE: Micro Ascii Encoder
 * Base on LZSS Adaptive Huffman Coding
 */
#include <stdio.h>
#include <dir.h>

int cn = 0, c1 = 0, c2 = 0, c3 = 0, width = 70;

/* General parameters */
#define PRINT_STEP	1024	/* Increments to update count */
#define FILE_NAME	65		/* Size of filename */
#define NUM_CMDS	20		/* Maximum number of input variables */

/* LZSS Parameters */
#define SBSIZE		4096	/* Size of string buffer */
#define LASIZE		60		/* Size of look-ahead buffer */
#define THRESHOLD	2
#define NIL			SBSIZE	/* End of tree's node */

/* Huffman coding parameters */
#define N_CHAR  	(256-THRESHOLD+LASIZE)	/* char code (= 0..N_CHAR-1) */
#define TSIZE 		(N_CHAR * 2 - 1)		/* Size of table */
#define ROOT		(TSIZE - 1)				/* root position */
#define MAX_FREQ	0x8000					/* update when cumulative */
										/* frequency reaches this value */
/*
 * Tables for encoding/decoding upper 6 bits of
 * sliding dictionary pointer
 */

/* encoder tables */
unsigned char p_len[64] = {
	0x03, 0x04, 0x04, 0x04, 0x05, 0x05, 0x05, 0x05,
	0x05, 0x05, 0x05, 0x05, 0x06, 0x06, 0x06, 0x06,
	0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06,
	0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07,
	0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07,
	0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07,
	0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08,
	0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08 };

unsigned char p_code[64] = {
	0x00, 0x20, 0x30, 0x40, 0x50, 0x58, 0x60, 0x68,
	0x70, 0x78, 0x80, 0x88, 0x90, 0x94, 0x98, 0x9C,
	0xA0, 0xA4, 0xA8, 0xAC, 0xB0, 0xB4, 0xB8, 0xBC,
	0xC0, 0xC2, 0xC4, 0xC6, 0xC8, 0xCA, 0xCC, 0xCE,
	0xD0, 0xD2, 0xD4, 0xD6, 0xD8, 0xDA, 0xDC, 0xDE,
	0xE0, 0xE2, 0xE4, 0xE6, 0xE8, 0xEA, 0xEC, 0xEE,
	0xF0, 0xF1, 0xF2, 0xF3, 0xF4, 0xF5, 0xF6, 0xF7,
	0xF8, 0xF9, 0xFA, 0xFB, 0xFC, 0xFD, 0xFE, 0xFF };

/* decoder tables */
unsigned char d_code[256] = {
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
	0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
	0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02,
	0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02,
	0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03,
	0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03,
	0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04,
	0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05,
	0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06,
	0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07,
	0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08,
	0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09,
	0x0A, 0x0A, 0x0A, 0x0A, 0x0A, 0x0A, 0x0A, 0x0A,
	0x0B, 0x0B, 0x0B, 0x0B, 0x0B, 0x0B, 0x0B, 0x0B,
	0x0C, 0x0C, 0x0C, 0x0C, 0x0D, 0x0D, 0x0D, 0x0D,
	0x0E, 0x0E, 0x0E, 0x0E, 0x0F, 0x0F, 0x0F, 0x0F,
	0x10, 0x10, 0x10, 0x10, 0x11, 0x11, 0x11, 0x11,
	0x12, 0x12, 0x12, 0x12, 0x13, 0x13, 0x13, 0x13,
	0x14, 0x14, 0x14, 0x14, 0x15, 0x15, 0x15, 0x15,
	0x16, 0x16, 0x16, 0x16, 0x17, 0x17, 0x17, 0x17,
	0x18, 0x18, 0x19, 0x19, 0x1A, 0x1A, 0x1B, 0x1B,
	0x1C, 0x1C, 0x1D, 0x1D, 0x1E, 0x1E, 0x1F, 0x1F,
	0x20, 0x20, 0x21, 0x21, 0x22, 0x22, 0x23, 0x23,
	0x24, 0x24, 0x25, 0x25, 0x26, 0x26, 0x27, 0x27,
	0x28, 0x28, 0x29, 0x29, 0x2A, 0x2A, 0x2B, 0x2B,
	0x2C, 0x2C, 0x2D, 0x2D, 0x2E, 0x2E, 0x2F, 0x2F,
	0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
	0x38, 0x39, 0x3A, 0x3B, 0x3C, 0x3D, 0x3E, 0x3F };

unsigned char d_len[256] = {
	0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03,
	0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03,
	0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03,
	0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03,
	0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04,
	0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04,
	0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04,
	0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04,
	0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04,
	0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04,
	0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05,
	0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05,
	0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05,
	0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05,
	0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05,
	0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05,
	0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05,
	0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05,
	0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06,
	0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06,
	0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06,
	0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06,
	0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06,
	0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06,
	0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07,
	0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07,
	0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07,
	0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07,
	0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07,
	0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07,
	0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08,
	0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08 };

FILE *infile, *outfile;

unsigned textsize, codesize, printcount;

unsigned char text_buf[SBSIZE + LASIZE - 1], getlen, putlen,
	filename[FILE_NAME];

int	match_position, match_length, lson[SBSIZE + 1], rson[SBSIZE + 257],
	dad[SBSIZE + 1], prnt[TSIZE + N_CHAR], son[TSIZE];

unsigned freq[TSIZE + 1], getbuf, putbuf;

char verbose = -1, ecmd = 0;
char *cmds[NUM_CMDS];
int ccount = 0;

char pcount = 0, print_chars[] = { '-', '\\', '|', '/' };

unsigned char outtab[64] =
{ "abcdefghijklmnopqrstuvwxyz0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ.," };
unsigned char intab[256], start = '{', end = '}';

InitTree()  /* Initialize tree */
{
	register int i;

	for(i = SBSIZE + 1; i <= SBSIZE + 256; ++i)
		rson[i] = NIL;			/* root */
	for(i = 0; i < SBSIZE; ++i)
		dad[i] = NIL;			/* node */
	for(i=0; i < (SBSIZE + LASIZE - 1); ++i)
		text_buf[i] = 0;
}

InsertNode(r)  /* Insert node to the tree */
	int r;
{
	register int i, p;
	int cmp;
	unsigned char *key;
	unsigned c;

	cmp = 1;
	key = &text_buf[r];
	p = SBSIZE + 1 + key[0];
	rson[r] = lson[r] = NIL;
	match_length = 0;
	for(;;) {
		if(cmp >= 0) {
			if(rson[p] != NIL)
				p = rson[p];
			else {
				rson[p] = r;
				dad[r] = p;
				return; } }
		else {
			if(lson[p] != NIL)
				p = lson[p];
			else {
				lson[p] = r;
				dad[r] = p;
				return; } }
		for(i = 1; i < LASIZE; ++i)
			if(cmp = key[i] - text_buf[p + i])
				break;
		if(i > THRESHOLD) {
			if(i > match_length) {
				match_position = ((r - p) & (SBSIZE - 1)) - 1;
				if((match_length = i) >= LASIZE)
					break; }
			if(i == match_length) {
				if ((c = ((r - p) & (SBSIZE - 1)) - 1) < match_position)
					match_position = c; } } }
	dad[r] = dad[p];
	lson[r] = lson[p];
	rson[r] = rson[p];
	dad[lson[p]] = r;
	dad[rson[p]] = r;
	if(rson[dad[p]] == p)
		rson[dad[p]] = r;
	else
		lson[dad[p]] = r;
	dad[p] = NIL;  /* remove p */
}

DeleteNode(p)  /* Delete node from tree */
	int p;
{
	register int q;

	if(dad[p] == NIL)
		return;			/* unregistered */
	if(rson[p] == NIL)
		q = lson[p];
	else if (lson[p] == NIL)
		q = rson[p];
	else {
		q = lson[p];
		if(rson[q] != NIL) {
			do
				q = rson[q];
			while(rson[q] != NIL);
			rson[dad[q]] = lson[q];
			dad[lson[q]] = dad[q];
			lson[q] = lson[p];
			dad[lson[p]] = q; }
		rson[q] = rson[p];
		dad[rson[p]] = q; }
	dad[q] = dad[p];
	if(rson[dad[p]] == p)
		rson[dad[p]] = q;
	else
		lson[dad[p]] = q;
	dad[p] = NIL;
}

int GetBit()	/* get one bit */
{
	register int i;

	if(!getlen)
		getbuf = xgetc() << (getlen = 8);
	i = getbuf;
	getbuf <<= 1;
	--getlen;
	return i < 0;
}

int GetByte()	/* get a byte */
{
	register unsigned i;

	if(getlen < 8) {
		getbuf |= xgetc() << (8 - getlen);
		getlen += 8; }
	i = getbuf;
	getbuf <<= 8;
	getlen -= 8;
	return i >> 8;
}

Putcode(l, c)		/* output c bits */
	int l;
	unsigned c;
{
	putbuf |= c >> putlen;
	if((putlen += l) >= 8) {
		rwrite(putbuf >> 8);
		if((putlen -= 8) >= 8) {
			rwrite(putbuf);
			codesize += 2;
			putlen -= 8;
			putbuf = c << (l - putlen); }
		else {
			putbuf <<= 8;
			++codesize; } }
}

/*
 * Initialize Huffman frequency tree
 */
StartHuff()
{
	register int i, j;

	getlen = putlen = getbuf = putbuf = 0;

	for(i = 0; i < N_CHAR; ++i) {
		freq[i] = 1;
		son[i] = i + TSIZE;
		prnt[i + TSIZE] = i; }
	i = 0; j = N_CHAR;
	while(j <= ROOT) {
		freq[j] = freq[i] + freq[i + 1];
		son[j] = i;
		prnt[i] = prnt[i + 1] = j;
		i += 2; ++j; }
	freq[TSIZE] = 0xffff;
	prnt[ROOT] = 0;
}

/*
 * Reconstruct freq tree
 */
reconst()
{
	register int i, j;
	int k;
	unsigned f, l;

	/* halve cumulative freq for leaf nodes */
	j = 0;
	for(i = 0; i < TSIZE; ++i) {
		if(son[i] >= TSIZE) {
			freq[j] = (freq[i] + 1) / 2;
			son[j] = son[i];
			++j; } }
	/* make a tree : first, connect children nodes */
	for(i = 0, j = N_CHAR; j < TSIZE; i += 2, ++j) {
		k = i + 1;
		f = freq[j] = freq[i] + freq[k];
		for(k = j - 1; f < freq[k]; --k);
		++k;
		l = (j - k) * 2;
		memmove(&freq[k + 1], &freq[k], l);
		freq[k] = f;
		memmove(&son[k + 1], &son[k], l);
		son[k] = i; }

	/* connect parent nodes */
	for(i = 0; i < TSIZE; ++i) {
		if((k = son[i]) >= TSIZE)
			prnt[k] = i;
		else
			prnt[k] = prnt[k + 1] = i; }
}

/* update freq tree */
update(c)
	int c;
{
	register int i, k;
	int j, l;

	if(freq[ROOT] == MAX_FREQ)
		reconst();
	c = prnt[c + TSIZE];
	do {
		k = ++freq[c];
		/* swap nodes to keep the tree freq-ordered */
		if(k > freq[l = c + 1]) {
			while (k > freq[++l]);
			--l;
			freq[c] = freq[l];
			freq[l] = k;
			i = son[c];
			prnt[i] = l;
			if(i < TSIZE)
				prnt[i + 1] = l;
			j = son[l];
			son[l] = i;
			prnt[j] = c;
			if(j < TSIZE)
				prnt[j + 1] = c;
			son[c] = j;
			c = l; } }
	while(c = prnt[c]);		/* do until we reach the root */
}

EncodeChar(c)
	unsigned c;
{
	register unsigned i;
	register int j;
	int k;

	k = prnt[c + TSIZE];

	/* search connections from leaf node to the root */
	i = j = 0;
	do {
		i >>= 1;
		if(k & 1) 			/* 0= Even address, 1 = Odd address */
			i |= 0x8000;
		++j; }
	while((k = prnt[k]) != ROOT);
	Putcode(j, i);
/*	code = i;
	len = j; */
	update(c);
}

EncodePosition(c)
	unsigned c;
{
	register unsigned i;

	/* output upper 6 bits with encoding */
	i = c >> 6;
	Putcode(p_len[i], (unsigned)p_code[i] << 8);

	/* output lower 6 bits directly */
	Putcode(6, (c & 0x3f) << 10);
}

EncodeEnd()
{
	if(putlen) {
		rwrite(putbuf >> 8);
		++codesize; }
}

int DecodeChar()
{
	register unsigned c;

	/*
	 * search the tree from the root to leaves.
	 * choose node #(son[]) if input bit == 0
	 * else choose #(son[]+1) (input bit == 1)
	 */
	c = son[ROOT];
	while(c < TSIZE) {
		c += GetBit();
		c = son[c]; }
	c -= TSIZE;
	update(c);
	return c;
}

int DecodePosition()
{
	register unsigned i, j;
	unsigned c;

	/* decode upper 6 bits from given table */
	i = GetByte();
	c = (unsigned)d_code[i] << 6;
	j = d_len[i] - 2;

	/* input lower 6 bits directly */
	while(j--)
		i = (i << 1) + GetBit();
	return c | (i & 0x3f);
}

/*
 * Main compression routine
 */
Encode()
{
	register int i, len;
	int c, r, s, last_match_length;

	textsize = codesize = printcount = 0;		/* reset size to zero */
	StartHuff();
	InitTree();
	s = 0;
	r = SBSIZE - LASIZE;
	for(i = s; i < (SBSIZE-LASIZE); ++i)
		text_buf[i] = 0;	/* was ' ' */
	for(len = 0; len < LASIZE && (c = getc(infile)) != EOF; ++len)
		text_buf[r + len] = c;
	textsize = len;
	for(i = 1; i <= LASIZE; ++i)
		InsertNode(r - i);
	InsertNode(r);
	do {
		if(match_length > len)
			match_length = len;
		if(match_length <= THRESHOLD) {
			match_length = 1;
			EncodeChar(text_buf[r]); }
		else {
			EncodeChar(255 - THRESHOLD + match_length);
			EncodePosition(match_position); }
		last_match_length = match_length;
		for(i = 0; i < last_match_length && (c = getc(infile)) != EOF; ++i) {
			DeleteNode(s);
			text_buf[s] = c;
			if(s < LASIZE - 1)
				text_buf[s + SBSIZE] = c;
			s = (s + 1) & (SBSIZE - 1);
			r = (r + 1) & (SBSIZE - 1);
			InsertNode(r); }
		if((textsize += i) > printcount) {
			if(verbose) {
				putc(8, stderr);
				putc(print_chars[++pcount & 0x03], stderr); }
			printcount += PRINT_STEP; }
		while(i++ < last_match_length) {
			DeleteNode(s);
			s = (s + 1) & (SBSIZE - 1);
			r = (r + 1) & (SBSIZE - 1);
			if(--len)
				InsertNode(r); } }
	while(len > 0);
	EncodeEnd();
}

/*
 * Main decompression routine
 */
Decode()
{
	register int i, k;
	int j, r, c;
	unsigned count;

	printcount = count = 0;
	StartHuff();
	for(i = 0; i < (SBSIZE-LASIZE); ++i)
		text_buf[i] = 0;	/* was ' ' */
	r = SBSIZE - LASIZE;
	while(count < textsize) {
		c = DecodeChar();
		if(c < 256) {
			putc(c, outfile);
			text_buf[r++] = c;
			r &= (SBSIZE - 1);
			++count; }
		else {
			i = (r - DecodePosition() - 1) & (SBSIZE - 1);
			j = c - 255 + THRESHOLD;
			for (k = 0; k < j; ++k) {
				c = text_buf[(i + k) & (SBSIZE - 1)];
				putc(c, outfile);
				text_buf[r++] = c;
				r &= (SBSIZE - 1);
				++count; } }
		if(count > printcount) {
			if(verbose) {
				putc(8, stderr);
				putc(print_chars[--pcount & 0x03], stdout); }
			printcount += PRINT_STEP; } }
}

/*
 * Match a filename against a pattern using DOS rules.
 */
fmatch(filename, pattern)
	char *filename, *pattern;
{
	char f, p;

	do {
		if(f = *filename)
			++filename;
		if((p = toupper(*pattern)) != f) {
			if(p == '*') {				/* Multi wildcard */
				if((f == '.') || !f) {	/* DOT or END resets */
					while(*pattern && (*pattern++ != '.'));
					p = *pattern; }
				continue; }
			else if(p != '?')			/* Single wildcard */
				return 0; }
		++pattern; }
	while(f || p);

	return -1;
}

/*
 * Open a file & display message/exit if error
 */
FILE *openf(filename, mode, suffix)
	char *filename, *mode, *suffix;
{
	char buffer[FILE_NAME], *ptr, flag;
	FILE *fp;

	/* Copy over filename */
	ptr = buffer;
	flag = -1;
	while(*ptr = *filename++) switch(*ptr++) {
		case '.' :	flag = 0;	break;
		case '\\':
		case '/' :	flag = -1; }
	if(flag) 
		while(*ptr++ = *suffix++);

	if(!(fp = fopen(buffer, mode))) {
		fprintf(stderr,"\n\007MAE: Unable to access '%s'\n", buffer);
		exit(-1); }
	return fp;
}

/*
 * Get next file from the input stream (archive)
 */
get_next()
{
	int c;
	char *ptr;

	ptr = filename;
	do
		*ptr++ = c = xgetc();
	while(c > 0);
	if(!c) {
		if((!xread(&textsize, sizeof(textsize))) ||
		   (!xread(&codesize, sizeof(codesize)))) {
				fprintf(stderr,"\n\007MAE: Corrupt archive file\n");
				exit(-1); }
		return -1; }

	return 0;
}

/*
 * Draw a line of dashes
 */
draw_line(num)
	int num;
{
	while(num--)
		putc('-', stdout);
	putc('\n', stdout);
}

/*
 * Display the file information
 */
show_file(name)
	char *name;
{
	fprintf(stdout,"%-12s %8u %8u  %3u%%\n", name, textsize,
		codesize, (textsize - codesize) /
		((textsize < 100) ? 1 : textsize / 100));
}

/*
 * Main program, parse options & perform archive functions
 */
main(argc, argv)
	int argc;
	char *argv[];
{
	int i, j, k;
	unsigned long total1, total2;
	struct ffblk ffblk;
	char *ptr, *ptr1, *ptr2;

	rinit();		/* Set up for RAM file */

	for(i=1; i < argc; ++i) {
		ptr = argv[i];
		switch((toupper(*ptr++) << 8) | toupper(*ptr++)) {
			case 'V-' :		/* turn off verbose output */
				verbose = 0;
				break;
			case 'A+' :		/* Archive files */
				ecmd = 1;
				break;
			case 'R+' :		/* Retreive files */
				ecmd = 2;
				break;
			case '=W' :		/* Specify width */
				width = atoi(ptr);
				break;
			case '=T' :		/* Specify encoder table */
				infile = openf(ptr, "r", ".MAT");
				start = getc(infile);
				end = getc(infile);
				for(k=0; k < sizeof(outtab); ++k) {
					if(((j = getc(infile)) <= ' ') || (j > 0x7E)) {
						fprintf(stderr,"\n\007MAE: Non-printable character in table (%02x)\n", j);
						exit(-1); }
					outtab[k] = j; }
				if(getc(infile) != '\n') {
					fprintf(stderr,"\n\007MAE: Too many/few character in table\n");
					exit(-1); }
				fclose(infile);
				for(k=1; k < sizeof(outtab); ++k)
					for(j = 0; j < k; ++j)
						if(outtab[k] == outtab[j]) {
							fprintf(stderr,"\n\007MAE: Duplicate characters in table\n");
							exit(-1); }
				break;
			default:
				cmds[ccount++] = argv[i]; } }

	if(!ccount) {
		fputs("Micro Ascii Encoder v1.2\n\nUse: mae [+a +r t=table -v w=width] <archive> [files...]\n\n?COPY.TXT 1989-2005 Dave Dunfield\nFreely Distributable.\n", stderr);
		exit(-1); }

	/* Initialize the input translate table */
	for(i=0; i < sizeof(intab); ++i)
		intab[i] = -1;

	for(i=0; i < sizeof(outtab); ++i)
		intab[outtab[i]] = i;

	if(ccount < 2)		/* If no parameters, default to all files */
		cmds[ccount++] = "*.*";

	switch(ecmd) {
		case 0 :			/* list files in archive */
			infile = openf(cmds[0], "r", ".MAE");
			get_start();
			total1 = total2 = j = 0;
			while(get_next()) {
				for(i = 1; i < ccount; ++i) {
					if(fmatch(filename, cmds[i])) {
						total1 += textsize;
						total2 += codesize;
						if(!j++) {
							fputs("Filename       Before    After Ratio\n", stdout);
							draw_line(36); }
						show_file(filename);
						break; } }
					xseek(codesize); }
			if(j) {
				draw_line(36);
				fprintf(stdout,"Total: %-5u %8lu %8lu  %3u%%\n", j,
					total1, total2, (total1 - total2) /
					((total1 < 100) ? 1 : total1 / 100)); }
			else
				fprintf(stderr,"\n\007MAE: File not found in archive.\n");
			fclose(infile);
			break;
		case 1 :			/* Write files to archive */
			outfile = openf(cmds[0], "w", ".MAE");
			putc(start, outfile);
			for(i = 1; i < ccount; ++i) {
				if(findfirst(ptr = cmds[i], &ffblk, 0)) {
					fprintf(stderr,"\n\007MAE: No files found\n");
					exit(-1); }
				ptr2 = ptr1 = filename;
				while(*ptr) {
					switch(*ptr1++ = *ptr++) {
						case ':' :
						case '\\' :
							ptr2 = ptr1; } }
				do {
					strcpy(ptr2, ffblk.ff_name);
					infile = openf(filename, "rb", "");
					rreset();
					if(verbose) {
						pcount = 0;
						fprintf(stdout, "Archiving %s -", ffblk.ff_name); }
					ptr = ffblk.ff_name;
					do
						xputc(*ptr);
					while(*ptr++);
					Encode();

					if(codesize >= textsize) {	/* no compression */
						rewind(infile);
						codesize = textsize; }

					xwrite(&textsize, sizeof(textsize));
					xwrite(&codesize, sizeof(codesize));

					if(codesize != textsize) {
						while((j = rread()) != -1)
							xputc(j); }
					else {
						while(j = fread(text_buf, 1, SBSIZE, infile))
							xwrite(text_buf, j); }
					fclose(infile);

					if(verbose) {
						putc('\r', stdout);
						show_file(ffblk.ff_name); }
					fclose(infile); }
				while(!findnext(&ffblk)); }
			xflush();
			putc(end, outfile);
			putc('\n', outfile);
			fclose(outfile);
			break;
		case 2 :			/* Retrieve files from archive */
			infile = openf(cmds[0], "r", ".MAE");
			get_start();
			while(get_next()) {
				for(i = 1; i < ccount; ++i) {
					if(fmatch(filename, cmds[i])) {
						outfile = openf(filename,"wb", "");
						if(verbose) {
							pcount = 0;
							fprintf(stdout, "Retrieving %s -", filename); }
						if(textsize) {
							if(textsize == codesize) {		/* stored file, retreive it */
								for(total1 = 0; total1 < textsize; total1 += total2) {
									if((total2 = (textsize - total1)) > SBSIZE)
										total2 = SBSIZE;
									xread(text_buf, (unsigned int)total2);
									fwrite(text_buf, 1, (unsigned int)total2, outfile); } }
							else			/* Compressed file */
								Decode(); }
						fclose(outfile);
						if(verbose) {
							putc('\r', stdout);
							show_file(filename); }
						break; } }
				if(i >= ccount)		/* Didn't restore, skip */
					xseek(codesize); }
			fclose(infile); }
}

get_start()
{
	int i;
	while((i = getc(infile)) != start)
		if(i & 0xFF00) {
			fputs("\n\007MAE: Missing START marker\n", stderr);
			exit(-1); }
}

xwrite(ptr, size)
	char *ptr;
	unsigned size;
{
	while(size--)
		xputc(*ptr++);
}
xputc(c)
	char c;
{
	switch(cn++) {
		case 0 :	c1 = c;		break;
		case 1 :	c2 = c;		break;
		case 2 :	c3 = c;		xflush(); }
}
xflush()
{
	if(cn) {
		putc(outtab[(c1&0xc0)>>2 | ((c2&0xC0)>>4) | ((c3&0xC0)>>6)], outfile);
		xlimit();
		putc(outtab[c1 & 0x3f], outfile);
		xlimit(); }
	if(cn > 1) {
		putc(outtab[c2 & 0x3F], outfile);
		xlimit(); }
	if(cn > 2) {
		putc(outtab[c3 & 0x3F], outfile);
		xlimit(); }

	cn = c1 = c2 = c3 = 0;
}
xlimit()
{
	static int outcount = 1;

	if(++outcount >= width) {
		putc('\n', outfile);
		outcount = 0; }
}
xread(ptr, size)
	char *ptr;
	int size;
{
	int i, c;
	i = 0;
	while(i < size) {
		if((c = xgetc()) & 0xFF00)
			break;
		*ptr++ = c;
		++i; }
	return i;
}
xgetc()
{
	int c;
	static int d;

	switch(cn++) {
		case 0 :
			if((d = xgetc1(0)) & 0xFF00)
				return -1;
			if((c = xgetc1((d << 2) & 0xC0)) & 0xFF00)
				return -1;
			break;
		case 1 :
			if((c = xgetc1((d << 4) & 0xC0)) & 0xFF00)
				return -1;
			break;
		case 2 :
			cn = 0;
			if((c = xgetc1((d << 6) & 0xC0)) & 0xFF00)
				return -1; }
	return c;
}
xgetc1(high)
	int high;
{
	unsigned c, d;

	for(;;) {
		if((c = getc(infile)) <= ' ')
			continue;
		if(c == end)
			return -1;
	if(c & 0xFF00) {
		fputs("\n\007MAE: Premature EOF in archive\n", stderr);
		exit(-1); }
	if((d = intab[c]) > 64) {
		fprintf(stderr, "\n\007MAE: Invalid character %04X in archive\n", c);
		exit(-1); }
	return d | high; }
}
xseek(size)
	unsigned size;
{
	while(size--)
		xgetc();
}
