/*
 * C source reporter
 *
 * Provides basic statistics about a 'C' program, including:
 * - Number of char/lines in file/comments/whitespace
 * - Number of significant chars/lines
 * - Occurances of common C constructions ({, }, ; and comments)
 *
 * ?COPY.TXT 2000-2003 Dave Dunfield
 *  -- see COPY.TXT --.
 *
 * Permission granted for personal (non-commercial) use only.
 *
 * Compile command: cc cstats -fop
 */
#include <stdio.h>

unsigned char
	lines[4],		/* Total number of lines */
	chars[4],		/* Total number of chars */
	white[4],		/* Total number of whitespace chars */
	ccount[4],		/* Total number of comments */
	cchars[4],		/* Total number of comment chars */
	ulines[4],		/* Total number of useful lines */
	obrace[4],		/* Total number of opening braces */
	cbrace[4],		/* Total number of closing braces */
	semis[4],		/* Total number of semicolons */
	temp1[4];		/* Temporary long register */

unsigned char		/* Fixed long numbers */
	one[4] = { 1, 0, 0, 0 },
	two[4] = { 2, 0, 0, 0 };

FILE *fp;			// General file pointer

/*
 * Collect statistics about a specific file
 */
void do_stat(char *filename)
{
	int c;
	char incomment, inquote, uflag, ctype;

	if(!(fp = fopen(filename, "rv")))
		return;

	incomment = inquote = uflag = ctype = 0;

	while((c = getc(fp)) != EOF) {
	again:
		longadd(chars, one);
		if(incomment)
			longadd(cchars, one);
		switch(c) {
		case '\n' :
			longadd(lines, one);
			if(uflag) {
				longadd(ulines, one);
				uflag = 0; }
			if(ctype)
				incomment = ctype = 0;
			continue;
		case ' ' :
		case '\t' :
			if(!inquote)
				longadd(white, one);
			continue;
		case '/' :
			if(inquote) break;
			c = getc(fp);
			longadd(chars, one);
			switch(c) {
			case '*' :
				++incomment;
				longadd(ccount, one);
				longadd(cchars, two);
				continue;
			case '/' :
				if(incomment) continue;
				++incomment;
				longadd(ccount, one);
				longadd(cchars, two);
				ctype = -1;
				continue; }
			goto again;
		case '*' :
			if(!incomment) break;
			if(ctype) break;
			if((c = getc(fp)) != '/')
				goto again;
			--incomment;
			longadd(chars, one);
			longadd(cchars, one);
			continue;
		case '\'' :
		case '"' :
			if(incomment) continue;
			if(inquote == c) {
				inquote = 0;
				continue; }
			if(inquote)
				continue;
			inquote = c;
			continue;
		case '\\' :
			c = getc(fp);
			longadd(chars, one);
			continue;
		case '{' :
			if(incomment || inquote) break;
			longadd(obrace, one);
			break;
		case '}' :
			if(incomment || inquote) break;
			longadd(cbrace, one);
			break;
		case ';' :
			if(incomment || inquote) break;
			longadd(semis, one); }
		if(!incomment)
			uflag = -1; }
	fclose(fp);
}

/*
 * Convert long number to static printable string for use in printf()
 * 4 strings are cycled, allowing up to 4 numbers to be output in a
 * single printf() statement.
 */
char *ln(unsigned char *n)
{
	static char buffer[4][25];
	static unsigned p;
	char *ptr;

	ptr = buffer[p = (p + 1) & 3];

	ltoa(n, ptr, 10);
	return ptr;
}

main(int argc, char *argv[])
{
	int i;

	if(argc < 2)
		abort("\nUse: CSTATS <filename> ...\n");

	for(i=1; i < argc; ++i)
		do_stat(argv[i]);

	longcpy(temp1, chars);
	longsub(temp1, cchars);
	longsub(temp1, white);
	printf("Characters:\n");
	printf("  in file(s)   : %s\n", ln(chars));
	printf("  in comments  : %s\n", ln(cchars));
	printf("  whitespace   : %s\n", ln(white));
	printf("  significant  : %s\n", ln(temp1));
	longcpy(temp1, lines);
	longsub(temp1, ulines);
	printf("Lines:\n");
	printf("  in file(s)   : %s\n", ln(lines));
	printf("  blank/comment: %s\n", ln(temp1));
	printf("  significant  : %s\n", ln(ulines));
	printf("Cism's:\n");
	printf("  '{'s         : %s\n", ln(obrace));
	printf("  '}'s         : %s\n", ln(cbrace));
	printf("  ';'s         : %s\n", ln(semis));
	printf("  comments     : %s\n", ln(ccount));
}
