/*
 * This utility finds matching {} brackets in a C program
 *
 * The program listing is output with the '{' and '}' replaced with
 * '{n' and 'n}' where 'n' is the nesting depth of the brackets.
 *
 * If no line number is specified, the entire file is output. If
 * a line number is specified, only as much of the file is output
 * as is covered by the first {} block found starting from that line.
 *
 * ?COPY.TXT 2000-2003 Dave Dunfield
 *  -- see COPY.TXT --.
 *
 * Permission granted for personal (non-commercial) use only.
 *
 * Compile command: cc fb -fop
 */

#include <stdio.h>
#include <file.h>

#define	BUFF_SIZE	16384	// Disk IO buffer size
#define	LINE_SIZE	2000	// Maximum size of line

// Exemption flags for skipping sections of the file
#define	EXEMPT_SQ	1		// Single-quote
#define	EXEMPT_DQ	2		// Double-quote
#define	EXEMPT_SC	3		// Standard C comment
#define	EXEMPT_EC	4		// end-of-line comment
// Exemption flag->Character transaction
#define	EXEMPT_STR	" '\"*/"

unsigned
	line;					// Input line count
int
	count;					// Count of brackets
char
	exempt,					// Scanning exempt states (See EXEMPT_* above)
	buffer[LINE_SIZE],		// Input buffer
	*ptr,					// Input pointer
// Command line options
	dline = -1,				// Display line numbers
	cppc = -1,				// Allow C++ style comments
	trunc,					// Truncate output at last bracket
	edebug,					// Exemption debugging
	re_sq,					// Reset single-quote exempt at end of line
	re_dq;					// Reset double-quote exempt at end of line

FILE
	*fp;					// Input file pointer

char help[] = { "\n\
Use:	FP filename [options]\n\n\
opts:	/C	= Do not scan for // comments\n\
	/D	= Debug: output scanning state at start of line\n\
	/L	= Do not output line numbers\n\
	/'	= Reset single-quote at start of line\n\
	/\"	= Reset double-quote at start of line\n\
	L=n	= Begin scanning from this line\n\n\
?COPY.TXT 2001-2003 Dave Dunfield\n -- see COPY.TXT --.\n" };

main(int argc, char *argv[])
{
	unsigned i, l;
	char quit, c;

	IOB_size = BUFF_SIZE;
	l = line = 1;
	for(i=1; i < argc; ++i) {
		ptr = argv[i];
		switch((toupper(*ptr++) << 8) | toupper(*ptr++)) {
		case ('/'<<8)|'C' :
		case ('-'<<8)|'C' :	cppc = 0;		continue;	// Disallow // comments
		case ('/'<<8)|'\'':
		case ('-'<<8)|'\'': re_sq= 0;		continue;	// Reset ' exemption
		case ('/'<<8)|'"' :
		case ('-'<<8)|'"' : re_dq= 0;		continue;	// Reset " exemption
		case ('/'<<8)|'L' :
		case ('-'<<8)|'L' : dline= 0;		continue;	// No line #'s
		case ('/'<<8)|'D' :
		case ('-'<<8)|'D' : edebug=-1;		continue;	// Exemption debug
		case ('L'<<8)|'=' :
			if(!(l = line = atoi(ptr)))
				abort("Bad L= value");
			trunc = -1;
			continue; }
		if(fp)
			abort("Illegal option or multiple filenames");
		fp = fopen(argv[i], "rvq"); }

	if(!fp)
		abort(help);

	stdout = setbuf(stdout, BUFF_SIZE);

	// Search for beginning line number
	while(--l) {
		if(!fgets(buffer, sizeof(buffer)-1, fp)) {
			printf("End of file before starting line (%u - %u)\n", line, line-l);
			fclose(stdout);
			fclose(fp);
			exit(-1); } }

	quit = 0;
	while(fgets(ptr = buffer, sizeof(buffer)-1, fp)) {
		if(edebug)
			putc(EXEMPT_STR[exempt], stdout);
		if(dline)
			printf("%5u: ", line);
		++line;
		while(c = *ptr++) {
			switch(c) {
			case '\'' :	/* Single quote */
				if(exempt) {
					if(exempt == EXEMPT_SQ)
						exempt = 0;
					break; }
				exempt = EXEMPT_SQ;
				break;
			case '"' :	/* Double quote */
				if(exempt) {
					if(exempt == EXEMPT_DQ)
						exempt = 0;
					break; }
				exempt = EXEMPT_DQ;
				break;
			case '/' :	/* Possible comment */
				switch(*ptr) {
				case '*' :	/* normal comment */
					if(!exempt) {
						exempt = EXEMPT_SC;
						putc(c, stdout);
						c = '*';
						++ptr; }
					break;
				case '/' :	/* end-of-line comment */
					if(cppc && !exempt)
						exempt = EXEMPT_EC; }
				break;
			case '*' :	/* Possible closing comment */
				if((exempt == EXEMPT_SC) && (*ptr == '/')) {
					exempt = 0;
					putc(c, stdout);
					c = '/';
					++ptr; }
				break;
			case '\\' :	/* Protected character */
				if(*ptr) {
					putc(c, stdout);
					c = *ptr++; }
				break;
			case '{' :	/* Opening brace */
				if(!exempt) {
					putc('{', stdout);
					++count;
					printf("%d", count);
					continue; }
				break;
			case '}' :	/* Closing brace */
				if(!exempt) {
					printf("%d", count);
					putc('}', stdout);
					switch(--count) {
					case 0 :
					case -1 :
						if(trunc) {
							count = 0;
							quit = -1; } }
					continue; } }
			putc(c, stdout); }
		putc('\n', stdout);
		if(quit)
			break;
		// Handle exemption resets
		switch(exempt) {
		case EXEMPT_SQ :	if(!re_sq)	break;
		case EXEMPT_DQ :	if(!re_dq)	break;
		case EXEMPT_EC :	exempt = 0; } }
	fclose(stdout);
	fclose(fp);
}
