/*
 * Program to "fix" JWASM by modifying Micro-C generated source code
 * to be compatible (see Help text).
 *
 * Compile command: CC JWAFIX -fop
 */
#include <stdio.h>
#include <file.h>

unsigned
	Lines,				// Lines processed
	Count;				// Index operations fixup count
unsigned char
	*Ptr,				// General pointer
	Cmt,				// Enable comment generation
	Quiet,				// Enable quiet mode
	Buffer[512];		// General buffer
FILE
	*fpi,				// Input file
	*fpo;				// Output file

unsigned char Hello[] = { "JWAFIX by Dave Dunfield - "#__DATE__"\n" };

unsigned char Help[] = { "\n\
Use:	JWAFIX infile[.ASM] outfile[.ASM] [options]\n\n\
opts:	/C	= include original source as Comments\n\
	/Q	= Quiet; suppress informational output\n\
\nJWASM does not support the standard INTEL/MASM syntax of: offset[index]\n\
which I (and my compiler) use extensively.  This program will process a\n\
.ASM source file and convert such constructs to: [index+/-offset].\n\n\
eg:	MOV AX,5[BX]	->	MOV AX,[BX+5]\n\
	LEA SI,-10[BP]	->	LEA SI,[BP-10]\n" };

/*
 * Test for valid characters in an offset field, these include:
 *	Numbers		: 0-9
 *	Symbols		: a-z,A-Z,_ (+numbers)
 *	Arithmetic	: +,-
 */
int isoffset(int c)
{
	if((c >= 'a') && (c <= 'z'))	// Lower case alpha
		return 255;
	if((c >= 'A') && (c <= 'Z'))	// Upper case alpha
		return 255;
	if((c >= '0') && (c <= '9'))	// Numeric
		return 255;
	switch(c) {						// Special characters
	case '_' :
	case '+' :
	case '-' :
		return 255; }
	return 0;
}

/*
 * Process filename into buffer & add extension if not supplied
 */
void filename(unsigned char *d)
{
	unsigned char f;
	f = 0;
	for(;;) switch(*d++ = *Ptr++) {
		case 0 :
			if(!f)
				strcpy(d-1, ".ASM");
			return;
		case '.' : f = 255;	continue;
		case ':' :
		case '\\': f = 0; }
}

/*
 * Main program
 */
main(int argc, char *argv[])
{
	unsigned i;
	unsigned char f, q, c, *p, *p1, *p2, *p3;

	// Process command line arguments
	for(i=1; i < argc; ++i) {
		Ptr = argv[i];
		switch((toupper(*Ptr++)<<8)|toupper(*Ptr++)) {
		case '-C' :
		case '/C' : Cmt = 255;		continue;
		case '-Q' :
		case '/Q' : Quiet = 255;	continue;
		} Ptr -= 2;
		if(!*(Buffer+256)) {		// Input filename
			filename(Buffer+256);
			continue; }
		if(*Buffer) {				// Output filename
help:		fputs(Hello, stderr);
			abort(Help); }
		filename(Buffer); }

	if(!*Buffer)		// Need two filenames
		goto help;
	if(!Quiet)			// Output welcome message if enabled
		fputs(Hello, stdout);
	IOB_size = 8192;

	fpi = fopen(Buffer+256, "rvq");
	fpo = fopen(Buffer, "wvq");
	while(fgets(Ptr = p = Buffer, sizeof(Buffer)-1, fpi)) {
		++Lines;
		f = Cmt;
		q = 0;
		for(;;) switch(c = *Ptr++) {
			case ';' :				// Comment - stop processing
				if(q)
					continue;
				while(*Ptr++);
			case 0 : goto done;		// End of line
			case '\'':
			case '"' :
				if(!q) {
					q = c;
					continue; }
				if(c == q)
					q = 0;
				continue;
			case '[' :				// Indexing operation
				if(q)
					continue;
				p1 = p2 = (p3 = Ptr)-1;
				while(isspace(*p3))	// Skip to reg
					++p3;
				switch((toupper(*p3++)<<8)|toupper(*p3++)) {
				default: continue;
				case 'BX' :			// Confirm valid register
				case 'SI' :
				case 'DI' :
				case 'BP' : }
				while(isspace(*p3))	// Skip to ']'
					++p3;
				if(*p3 != ']')		// Not valid index
					continue;
				while((p1 > Buffer) && isoffset(*(p1-1)))
					--p1;
				if(p1 == p2)		// No offset
					continue;
				++Count;
				if(f) {				// Output comment if enabled
					putc(';', fpo);
					fputs(Buffer, fpo);
					putc('\n', fpo);
					f = 0; }
				fput(p, p1-p, fpo);	// Output up to offset
				putc('[', fpo);		// Open index
				while(Ptr < p3)		// Output register
					putc(*Ptr++, fpo);
				if(*p1 != '-')		// Not subtract, add '+'
					putc('+', fpo);
				while(p1 < p2)		// Output offset
					putc(*p1++, fpo);
				p = Ptr;			// Set new pending position
		}
done:	if(--Ptr - p)
			fput(p, Ptr-p, fpo);
		putc('\n', fpo); }
	fclose(fpo);
	fclose(fpi);
	if(!Quiet)
		printf("%u input lines, %u index operations converted.\n", Lines, Count);
}
