/*
 * This is a DOS command line program to record and play .VOC sound files
 * using the CT-VOICE driver included with Sound Blaster cards. CT-VOICE.DRV
 * must be located in the \DRV subdirectory of the sound-blaster directory
 * identified by the SOUND environment variable. See your Sound Blaster
 * installation docs for more information. This was written as part of my
 * CAM project for handicapped children, and has the following features not
 * found in most similar utilities:
 * 
 * 	- Insures .VOC blocks will be less than 64K.
 * 	- Allows .VOC blocks to be split at defined points.
 * 	- Allows markers to be inserted into the file interactively.
 * 
 * command line: SOUND [/R /Bsize /D /P /Q /Srate /Ttime] filename ...
 * /R		- RECORD to files, if not given: PLAY files
 * /Bsize	- Sets block size (record only - default 65500)
 * /D		- Enable debugging messages
 * /P		- Playback/Recording PAUSE mode at startup
 * /Q		- QUIET! - inhibit normal operating messages
 * /Srate	- Set sampling rate (record only - default 8000)
 * /Ttime	- Set time limit (time = 1/18th second intervals)
 * 
 * Keyboard commands during playback:
 * ESC		- Terminate playback and exit
 * SPACE	- Pause output (any other key resumes)
 * 
 * Keyboard commands during recording:
 * ESC		- Stop recording and exit
 * SPACE	- Pause recording (any other key resumes)
 * ENTER	- Begin a new block
 * 1-9		- Insert marker (1-9) and begin a new block
 *
 * ?COPY.TXT 1995-2003 Dave Dunfield
 *  -- see COPY.TXT --.
 *
 * Compile command: cc sound -fop
 */
#include <stdio.h>

/* Sound BLASTER definitions */
#define	VOC_SEG_SIZE	4096		/* 64K for VOC buffer */
#define	MIN_VOC_BLOCK	10			/* Minimum .VOC block size */

#define	VDATA			0x01		/* Voice DATA block */
#define	VCONT			0x02		/* Voice CONTINUATION block */
#define	VSILENCE		0x03		/* Voice SILENCE block */
#define	VMARKER			0x04		/* Voice MARKER block */
#define	VASCII			0x05		/* Voice ASCII block */
#define	VEDATA			0x09		/* Extended voice data */

unsigned
	ct_seg,				/* CT-VOICE load segment */
	voc_seg,			/* Active VOC buffer segment */
	voc_base_seg,		/* Base VOC segment address */
	voc_last,			/* Last VOC segment issued */
	voc_status,			/* Status word from VOC buffer */
	voc_marker,			/* Last marker encountered */
	start_time,			/* Timestamp of start */
	time_limit = 0,		/* Time limit of recording */
	sample_rate = 8000,	/* Command line sample rate */
	block_size = 65500;	/* Minimum buffers size */
char
	debug_flag = 0,		/* Debugging enabled flag */
	playback = -1,		/* Playback function enabled (0=Record) */
	verbose = -1,		/* Verbose mode enabled */
	pause = 0,			/* Playback/Recording pause mode is active */
	voc_text[80];		/* Text read from VOC ASCII data */

FILE voc_fp;

/*
 * Debugging output
 */
register debug(unsigned args)
{
	unsigned *ptr;
	char buffer[81];

	ptr = nargs() * 2 + &args;
	if(debug_flag) {
		_format_(ptr, buffer);
		fputs(buffer, stdout); }
}

/*
 * Catastrophic sound-blaster failure
 */
register sb_fail(unsigned args)
{
	char buffer[81];

	_format_(nargs() * 2 + &args, buffer);
	fputs(buffer, stdout);
	fclose(voc_fp);
	unload_ct_voice();
	exit(-1);
}
	
/*
 * Load CT-VOICE driver
 */
load_ct_voice()
{
	unsigned sh, sl, j;
	char sbpath[65], buffer[100], *ptr;
	FILE *fp;

	/* Read SOUND variable & build path to driver */
	if(!getenv("SOUND", sbpath))
		abort("SOUND environment variable not set!");
	concat(buffer, sbpath, "\\DRV\\CT-VOICE.DRV");

	/* Determine size of driver file */
	if(find_first(buffer, 0, sbpath, &sh, &sl, &j, &j, &j))
		abort("Cannot locate CT-VOICE driver");

	/* Allocate external (far) buffers */
	if(	!(	(ct_seg = alloc_seg((sl+15)/16))
		&&	(voc_base_seg = alloc_seg(VOC_SEG_SIZE*2)) ) )
		abort("Cannot allocate memory");

	/* Load CT-VOICE driver into memory*/
	fp = fopen(buffer, "rvbq");
	debug("Voice driver: seg=%04x size=%u vseg=%04x", ct_seg,
		load_far(ct_seg, 0, sl, fp), voc_base_seg);
	fclose(fp);

	asm {	/* Patch in far calling vector */
		MOV		AX,DGRP:_ct_seg			; Get segment
		MOV		word ptr CS:ctseg,AX	; Set segment
		XOR		BX,BX					; Get zero
		CALL	callsb					; Get version
	}
	j = nargs();
	debug(" %u.%u", j>>8, j & 0xFF);

	/* Assume settings from BLASTER environment variable if it exists */
	if(getenv("BLASTER", ptr = buffer)) {
		while(*ptr) switch(toupper(*ptr++)) {
			case 'A' :		/* Address */
				debug(" A%x", j = atox(ptr));
				j;
				asm {
					MOV		BX,1
					CALL	callsb
					}
				break;
			case 'I' :		/* Interrupt */
				debug(" I%u", j=atoi(ptr));
				j;
				asm {
					MOV		BX,2
					CALL	callsb
				}
				break;
			case 'D' :		/* DMA channel */
				debug(" D%u", j=atoi(ptr));
				j;
				asm {
					MOV		BX,19
					CALL	callsb
				}
				break;
			case 'T' :		/* Card type */
				debug(" T%u", j=atoi(ptr));
				j;
				asm {
					MOV		BX,20
					CALL	callsb;
				} } }
	debug("\n");

	asm {	/* Initialize the card */
		MOV		BX,3			; Init card
		XOR		AX,AX			; Function 0
		CALL	callsb			; Call driver
	}

	/* Non-zero (accumulator content) means initialziation failed */
	if(nargs())
		sb_fail("Sound card failed init.\n");

	asm {	/* Establish pointer to status word */
		MOV		BX,5			; Set status word
		PUSH	DS				; Get data seg
		POP		ES				; Into extra
		MOV		DI,OFFSET DGRP:_voc_status; Point to status word
		CALL	callsb			; Call the blaster
	}
}

/*
 * Unload CT-VOICE driver
 */
unload_ct_voice()
{
	debug("Voice driver unloaded.\n");
	asm {
		MOV		BX,9			; Terminate driver
		CALL	callsb			; Call driver
	}
	free_seg(voc_base_seg);
	free_seg(ct_seg);
}

/*
 * Load a far memory buffer from an open file
 */
load_far(unsigned seg, unsigned offset, unsigned size, FILE *fp)
{
	unsigned char buffer[1024];
	unsigned s;

	while(size) {
		if(s = fget(buffer, (size > sizeof(buffer)) ? sizeof(buffer) : size, fp)) {
			copy_seg(seg, offset, get_ds(), buffer, s);
			offset += s;
			size -= s;
			continue; }
		break; }

	return offset;
}

/*
 * Assembly stub to perform inter-segment call to CT-VOICE driver
 */
asm {
callsb:	DB		9Ah				; Call instruction
		DW		0h				; Offset 0
ctseg:	DW		0h				; Segment
		RET
}

/*
 * Load a voice block into extended memory
 */
load_voice_block()
{
	int type;
	unsigned size;
	switch(type = getc(voc_fp)) {
		case -1:
		case 0 :
			debug("End of VOC file!\n");
			return 0;
		default:
			sb_fail("Unknown VOC block type %02x\n", type);
		case VDATA		:	/* Voice data */
		case VCONT		:	/* Voice continuation */
		case VSILENCE	:	/* Silence */
		case VMARKER	:	/* Marker */
		case VASCII		:	/* ASCII data */
		case VEDATA		:	/* Extended voice data */ }

	fget(&size, sizeof(size), voc_fp);
	if((size > 65500) || getc(voc_fp))
		sb_fail("VOC block length exceeds 65500\n");

	debug("VOC BLOCK: Type=%u Size=%u", type, size);
	poke(voc_seg, 0, type);
	pokew(voc_seg, 1, size);
	poke(voc_seg, 3, 0);
	load_far(voc_seg, 4, size, voc_fp);
	poke(voc_seg, size + 4, 0);

	switch(type) {
		case VMARKER :
			debug(" Marker=%u", voc_marker = peekw(voc_seg, 4));
			break;
		case VASCII :
			copy_seg(get_ds(), voc_text, voc_seg, 4, sizeof(voc_text));
			debug(" Text=%s", voc_text); }

	debug("\n");
	return type;
}

/*
 * Save a pending block
 */
save_voice_block(char *prompt)
{
	unsigned size, s, o;
	unsigned char buffer[1024];

	if(voc_last) {
		if(*prompt)
			asm {
				MOV		BX,8	; Stop voice function
				CALL	callsb	; Call the driver
			}
		if((size = peekw(voc_last, 1) + 4) >= MIN_VOC_BLOCK) {
			printf("Block size %u %s\n", size, prompt);
			o = 0;
			do {
				copy_seg(get_ds(), buffer, voc_last, o, s = (size > sizeof(buffer)) ?
					sizeof(buffer) : size);
				fput(buffer, s, voc_fp);
				o += s; }
			while(size -= s);
			voc_last = 0;
			return; }
		voc_last = 0; }

	if(*prompt)
		printf("%s\n", prompt);
}

/*
 * Play a voice file:
 */
play_file(char *filename)
{
	unsigned i;
	char buffer[50];

	concat(buffer, filename, ".VOC");
	if(!(voc_fp = fopen(buffer, "rvb")))
		return;

	if(verbose)
		printf("Playing %s:\nEsc=Stop  Space=Pause\n", buffer);

	fget(buffer, 0x14, voc_fp);
	fget(&i, sizeof(i), voc_fp);
	if(i != 0x1A)
		sb_fail("VOC file offset %04x is wrong\n", i);
	fget(buffer, 4, voc_fp);

	if(pause) {
		printf("(paused)\n");
		kbget();
		printf("Resuming\n"); }

	for(;;) {
		voc_seg = (voc_seg == voc_base_seg) ? voc_base_seg+VOC_SEG_SIZE : voc_base_seg;
	nextblock:
		i = load_voice_block();
		while(voc_status) switch(kbtest()) {
			case 0x1B:		/* Exit */
			pexit:
				asm {
					MOV		BX,8	; Stop voice function
					CALL	callsb	; Call the driver
				}
				fclose(voc_fp);
				return;
			case ' ' :		/* Pause output */
				asm {
					MOV		BX,10	; Pause output
					CALL	callsb	; Call the driver
				}
				printf("(Paused)\n");
				if(kbget() == 0x1B)
					goto pexit;
				printf("Resuming\n");
				asm {
					MOV		BX,11	; Resume output
					CALL	callsb	; Call the driver
				} }
		switch(i) {
			case 0 :
				fclose(voc_fp);
				return;
			case VMARKER :
				goto nextblock; }

		asm {
			MOV		BX,6			; Start voice function
			MOV		ES,DGRP:_voc_seg; Get voice segment
			MOV		DI,0			; Offset zero
			CALL	callsb			; Call the driver
		} }
}

/*
 * Record a file
 */
record_file(char *filename)
{
	unsigned i;
	char buffer[50];
	static char mstring[] = { "Marker 0" };
	static char vh1[] = { 'C','r','e','a','t','i','v','e',' ',
		'V','o','i','c','e',' ','F','i','l','e', 0x1A, 0x1A, 0x00,
		0x0A, 0x01, 0x29, 0x11 };

	concat(buffer, filename, ".VOC");
	if(!(voc_fp = fopen(buffer, "wvb")))
		return;

	fput(vh1, sizeof(vh1), voc_fp);

	asm {
		MOV		BX,4		; Set speaker
		MOV		AX,0		; OFF
		CALL	callsb		; Call driver
;		MOV		BX,16
;		MOV		AX,0
;		CALL	callsb
;		MOV		BX,17
;		MOV		AX,0
;		CALL	callsb
	}

	voc_seg = voc_last = voc_status = 0;

	if(verbose) {
		printf("Recording '%s', SampleRate=%u BlockSize=%u\n", buffer, sample_rate, block_size);
		printf("Esc=Stop  Space=Pause  Enter=NewBlock  1-9=Marker\n"); }

	if(pause) {
		voc_seg = voc_base_seg;
		goto rpause; }

	for(;;) {
		voc_last = voc_seg;
		voc_seg = (voc_seg == voc_base_seg) ? voc_base_seg+VOC_SEG_SIZE : voc_base_seg;
		while(voc_status) if(i = kbtest()) {
newkey:	switch(i) {
			case 0x1B :		/* Stop recording */
				save_voice_block("(Ending)");
				putc(0, voc_fp);
				fclose(voc_fp);
				return;
			case '1' :		/* Insert marker */
			case '2' :
			case '3' :
			case '4' :
			case '5' :
			case '6' :
			case '7' :
			case '8' :
			case '9' :
				mstring[7] = i;
				save_voice_block(mstring);
				putc(4, voc_fp);
				putc(2, voc_fp);
				putc(0, voc_fp);
				putc(0, voc_fp);
				putc(i-'0', voc_fp);
				putc(0, voc_fp);
				break;
			case ' ' :		/* Pause recording */
			rpause:
				save_voice_block("(Paused)");
				if((i = kbget()) == ' ')
					i = 0;
				printf("Resuming\n");
				goto newkey;
			case '\r' :		/* Start new block */
				asm {
					MOV		BX,8	; Stop voice function
					CALL	callsb	; Call the driver
				} } }

		asm {
			MOV		BX,7			; Start voice input
			MOV		AX,DGRP:_sample_rate; Sample rate
			MOV		CX,DGRP:_block_size; Maximum length
			MOV		ES,DGRP:_voc_seg; Get voice segment
			MOV		DI,0			; Zero offset
			MOV		DX,0			; Zero high length
			CALL	callsb			; Inform card
		}

		if(i = nargs()) {
			printf("Record failed: %04x\n", i);
			fclose(voc_fp);
			unload_ct_voice();
			exit(-1); }

		save_voice_block(""); }
}

/* ------------- */

/*
 * Get a parameter and check it against limits
 */
get_parm(char *ptr, unsigned low, unsigned high)
{
	unsigned v;

	v = atoi(ptr);
	if((v < low) || (v > high)) {
		printf("Invalid option value: %s\n", ptr - 2);
		printf("Value must range between %u and %u\n", low, high);
		exit(-1); }
	return v;
}

/*
 * Command line recorder
 */
main(int argc, char *argv[])
{
	int i, file;
	char *ptr, playback, *files[20];
	static char help[] = { "Use: SOUND [options] file...\n\n\
Opts:	/B<size>	- Set block size (record only)\n\
	/D		- Enable debug output\n\
	/P		- Pause before playing/recording\n\
	/Q		- Quiet! (inhibit output)\n\
	/R		- Record to file(s) (otherwise play)\n\
	/S<rate>	- Set sample rate (record only)\n\
	/T<time>	- Set time limit (1/18 second intervals)\n" };

	file = 0;
	playback = -1;

	for(i=1; i < argc; ++i) {
		files[file] = ptr = argv[i];
		switch((toupper(*ptr++) << 8) | toupper(*ptr++)) {
			case '-B' :		/* Set block size */
			case '/B' :
				block_size = get_parm(ptr, 4000, 65500);
				continue;
			case '-D' :		/* Enable debug manual */
			case '/D' :
				debug_flag = -1;
				continue;
			case '-P' :		/* Pause mode */
			case '/P' :
				pause = -1;
				continue;
			case '-Q' :		/* Be quiet */
			case '/Q' :
				verbose = 0;
				continue;
			case '-R' :		/* Record to file */
			case '/R' :
				playback = 0;
				continue;
			case '-S' :		/* Set sample rate */
			case '/S' :
				sample_rate = get_parm(ptr, 4000, 44000);
				continue;
			case '/T' :		/* Set time limit */
			case '-T' :
				time_limit = get_parm(ptr, 1, 65000);
				continue; }
		++file; }

	if(verbose)
		printf("SOUND - VOC recorder/player - ?COPY.TXT 1995-2003 Dave Dunfield.\n\n");

	if(!file)
		abort(help);

	load_ct_voice();

	start_time = peekw(0x40, 0x6C);

	if(playback) {
		for(i=0; i < file; ++i)
			play_file(files[i]); }
	else {
		for(i=0; i < file; ++i)
			record_file(files[i]); }

	unload_ct_voice();
}

/*
 * Test keyboard, inject ESC if time limit expires
 */
kbtest()
{
	if(time_limit) {
		if((peekw(0x40, 0x6C) - start_time) > time_limit)
			return 0x1B; }
	return kbtst();
}
