#define L_PRINT					/* include printing services */

/* LNUM -- Large Number Package for Micro-C/PC

This file is intended to be #included in another source file.
To compile as a demo, use: cc LNUM -pof DEMO=

Function list
--------------
void l_copy(word *a, word *b)						[ Copy the bignum from a to b ]
void l_clear(word *a)								[ Set a to zero ]
void l_set(word *a, word n)							[ Set a to the digit n ]

int l_iszero(word *a)								[ is a == 0 ]
int l_cmp(word *a, word *b)							[ compare a and b ]
int l_cmp_d(word *a, word b)						[ compare a and 'b' ]

int l_shr(word *a, word *b)							[ b = a/2  (return carry) ]
int l_shr_s(word *a)								[ a = a/2  (return carry) ]
int l_shl(word *a, word *b)							[ b = 2a (return carry) ]
int l_shl_s(word *a)								[ a = 2a (return carry) ]

void l_add(word *a, word *b, word *c)				[ c = a + b ]
void l_add_s(word *a, word *b)						[ a += b ]
void l_add_d(word *a, word b)						[ a += 'b' ]
void l_sub(word *a, word *b, word *c)				[ c = a - b ]
void l_sub_s(word *a, word *b)						[ a -= b ]
void l_sub_d(word *a, word b)						[ a -= 'b']
void l_mul(word *a, word *b, word *c)				[ c = ab ]
void l_mul_s(word *a, word *b)						[ a *= b ]
void l_mul_d(word *a, word b)						[ a *= 'b']
void l_div(word *a, word *b, word *q, word *r)		[ q = a/b, r = a%b ]
void l_div_s(word *a, word *b)						[ a /= b ]
void l_div_d(word *a, word b)						[ a /= 'b']
void l_mod(word *a, word *b, word *c)				[ c = a%b ]
void l_mod_s(word *a, word *b)						[ a %= b ]
word l_mod_d(word *a, word b)						[ returns a mod b ]

void l_addmod(word *a, word *b, word *m, word *c)	[ c = (a+b) mod m ]
void l_submod(word *a, word *b, word *m, word *c)	[ c = (a-b) mod m ]
void l_mulmod(word *a, word *b, word *m, word *c)	[ c = (ab) mod m ]

void l_sqr(word *a)									[ c = a^2 ]
void l_sqrmod(word *a, word *m, word *c)			[ c = a^2 mod m ]
void l_expt(word *a, word *b, word *c)				[ c = a^b ]
void l_exptmod(word *a, word *b, word *m, word *c)  [ c = a^b mod m ]

void l_gcd(word *a, word *b, word *c)				[ c = gcd(a, b) ]
void l_lcm(word *a, word *b, word *c)				[ c = lcm(a, b) ]
void l_invmod(word *a, word *n, word *b)			[ b = a^-1 mod n ]
void l_sqrt(word *n, word *N)						[ N = n^1/2 ]

void l_print(word *a, word radix, FILE *out)		[ Output ]
void l_println(word *a, word radix, FILE *out)		[ Output with new line ]

[ radixes from 2 to 64 ]
void l_toradix(word *a, word radix, char *out)		[ put in buffer ]
void l_fromradix(char *a, word radix, word *b)		[ read string ]

void l_readraw(word *a, unsigned char *in, int len) [ read binary ]
int l_toraw(word *a, unsigned char *out)			[ write binary ]

Notes:

	You have to configure this package before you use it.  All you have to
choose is how big you want your large numbers to be.  Which is essentially just

		LSIZE = (bits_total / (BITS-1)) + rem?1:0

	Where BITS is the number of bits per 'word', and 'rem' is set true if
'bits_total' (the total number of bits you want) is not a multiple of (BITS-1).

	For example to get [at least] 128 bit words from a 16-bit target you need

		LSIZE	= 128 / 15 + rem?1:0
				= 8 + 1
				= 9

	Note that 9 words is actually 9 * 15 = 135 bits in this case.  Which is
large enough for the 128 bit words.

------------------------------------------------------------------------------
	Tom St Denis (tom@dasoft.org), Jan 2000.  Distribute like mad!!!
*/
#include <limits.h>

static unsigned char radix_data[]
	= "0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz{}";

/* You have to set these up */
#ifndef LSIZE
	#define LSIZE		9			/* How many words per large num? */
#endif

#ifndef word
	#define word		unsigned	/* should be BITS bits in size */
#endif

#define BITS		(CHAR_BIT*sizeof(word))
#define BUFSIZE		128			/* if you use l_toradix set this up!!! */
#define MASK		(((word)1 << (BITS-1)) - (word)1)

#define L_LT		-1			/* Less than */
#define L_EQ		0			/* Equal to */
#define L_GT		1			/* Greater then */

/* Errors */
#define L_OK		0			/* All ok */
#define L_DIVZERO	1			/* Division by zero */
#define L_INVNUM	2			/* Invalid number */

word l_error=L_OK;
char *l_error_str[] = { "All ok", "Division by zero", "Invalid parameter" };

#ifdef L_PRINT
	#include <stdio.h>
#endif

/* b = a */
void l_copy(word *a, word *b)
{
	memcpy(b, a, sizeof(word) * LSIZE);
}

/* a = 0 */
void l_clear(word *a)
{
	memset(a, 0, sizeof(word) * LSIZE);
}

/* a = 'n' */
void l_set(word *a, word n)
{
	l_clear(a);
	a[0] = n & MASK;
}

/* is zero? */
int l_iszero(word *a)
{
	int i;

	for (i = 0; i < LSIZE; i++)
		if (a[i])
			return 0;

	return 1;
}

int l_cmp(word *a, word *b)
{
	word i;

	for (i = LSIZE-1; i != (word)-1; i--) {
		if (a[i] < b[i])
			return L_LT;
		else if (a[i] > b[i])
			return L_GT;
	}

	return L_EQ;
}

int l_cmp_d(word *a, word b)
{
	word temp[LSIZE], temp2;

	l_set(temp, b);
	temp2 = l_cmp(a, temp);

	return temp2;
}

/* b = a/2 (return carry) */
int l_shr(word *a, word *b)
{
	int carry, i;

	carry = a[0] & 1;
	for (i = 0; i < (LSIZE-1); i++)
		b[i] = ((a[i] >> 1) | (a[i+1]<<(BITS-2))) & MASK;
	b[LSIZE-1] = a[LSIZE-1]>>1;

	return carry;
}

/* a = a/2 return carry */
int l_shr_s(word *a)
{
	word temp[LSIZE], carry;

	carry = l_shr(a, temp);
	l_copy(temp, a);

	return carry;
}

/* b = 2a (return carry) */
int l_shl(word *a, word *b)
{
	int carry, i;

	carry = a[LSIZE-1] >> (BITS-2);
	for (i = LSIZE-1; i > 0; i--)
		b[i] = ((a[i] << 1) | (a[i - 1] >> (BITS-2))) & MASK;
	b[0] = (a[0] << 1) & MASK;

	return carry;
}

/* a = 2a return carry */
int l_shl_s(word *a)
{
	word temp[LSIZE], carry;

	carry = l_shl(a, temp);
	l_copy(temp, a);

	return carry;
}

/* c = a + b */
void l_add(word *a, word *b, word *c)
{
	word carry, i;

	for (i = carry = 0; i < LSIZE; i++) {
		carry = (c[i] = (a[i] + b[i] + carry)) >> (BITS-1);
		c[i] &= MASK;
	}
}

/* a = a+b */
void l_add_s(word *a, word *b)
{
	word temp[LSIZE];

	l_add(a, b, temp);
	l_copy(temp, a);

}

/* a = a + 'b' */
void l_add_d(word *a, word b)
{
	word temp[LSIZE];

	l_set(temp, b);
	l_add_s(a, temp);
}

/* c = a - b */
void l_sub(word *a, word *b, word *c)
{
	word carry, i;

	for (i = carry = 0; i < LSIZE; i++) {
		carry = (c[i] = (a[i] - (b[i] + carry))) >> (BITS-1);
		c[i] &= MASK;
	}
}

/* a = a-b */
void l_sub_s(word *a, word *b)
{
	word temp[LSIZE];

	l_sub(a, b, temp);
	l_copy(temp, a);
}

/* a = a - 'b' */
void l_sub_d(word *a, word b)
{
	word temp[LSIZE];

	l_set(temp, b);
	l_sub_s(a, temp);
}

void l_mul_big_shl(word *a, word cnt)
{
	word i, j, tmp[LSIZE];

	l_clear(tmp);
	for (i = cnt, j = 0; i < LSIZE; )
		tmp[i++] = a[j++];

	l_copy(tmp, a);
}

void l_mul_shl(word *a, word cnt)
{
	word b[LSIZE], i;

	for (i = LSIZE-1; i > 0; i--)
		b[i] = ((a[i] << cnt) | (a[i - 1] >> (BITS-1-cnt))) & MASK;
	b[0] = (a[0] << cnt) & MASK;
	l_copy(b, a);
}

/* c = ab, Using the add/double method */
void l_mul(word *a, word *b, word *c)
{
	word mult[LSIZE], mult2[LSIZE], i, j, k, l;

	/* copy into multiplier buffer */
	l_clear(c);

	/* make the shift register the smaller of the two */
	if (l_cmp(a, b) == L_LT) {
		l_copy(b, mult);
		l_copy(a, mult2);
	} else {
		l_copy(a, mult);
		l_copy(b, mult2);
	}

	/* loop */
	for (i = 1, k = j = 0; j < LSIZE; ) {
		/* if coefficient, add to result */
		if (mult2[j] & i) {
			if ((l = k / (BITS-1)))
				l_mul_big_shl(mult, l);
			if ((k %= (BITS-1)))
				l_mul_shl(mult, k);
			k = 0;
			l_add_s(c, mult);
		}

		/* move bitmask if required */
		i += i; ++k;
		if (!(i & MASK)) { i = 1; ++j; }
	}
}

/* a = ab */
void l_mul_s(word *a, word *b)
{
	word temp[LSIZE];

	l_mul(a, b, temp);
	l_copy(temp, a);
}

/* square a number */
void l_sqr(word *a)
{
	l_mul_s(a, a);
}

/* a = a'b' */
void l_mul_d(word *a, word b)
{
	word temp[LSIZE];

	l_set(temp, b);
	l_mul_s(a, temp);
}

/* find such that b = aq + r */
void l_div(word *a, word *b, word *q, word *r)
{
	word cy, cy2, t;

	/* division by zero? */
	if (l_iszero(b)) {
		l_error = L_DIVZERO;
		return;
	}

	l_copy(a, q);
	l_clear(r);
	t = ((BITS - 1) * LSIZE) + 1;

carry_zero:
	cy = 0;
	for (;;) {
		cy2 = cy;
		cy = l_shl_s(q);
		l_add_d(q, cy2);

		if (!--t)
			return;

		l_shl_s(r);
		l_add_d(r, cy);

		if (l_cmp(r, b) == L_LT)
			goto carry_zero;

		l_sub_s(r, b);
		cy = 1;
	}
}

/* c = a mod b */
void l_mod(word *a, word *b, word *c)
{
	word temp[LSIZE];

	l_div(a, b, temp, c);
}

/* a = a mod b */
void l_mod_s(word *a, word *b)
{
	word temp[LSIZE];

	l_mod(a, b, temp);
	l_copy(temp, a);
}

/* take the remainder and return it */
word l_mod_d(word *a, word b)
{
	word temp[LSIZE], temp2[LSIZE], r;

	l_set(temp, b);
	l_copy(a, temp2);
	l_mod_s(temp2, temp);
	r = temp2[0];

	return r;
}

/* a = a/b */
void l_div_s(word *a, word *b)
{
	word temp[2][LSIZE];

	l_div(a, b, temp[0], temp[1]);
	l_copy(temp[0], a);
}

/* a = a/'b' */
void l_div_d(word *a, word b)
{
	word temp[LSIZE];

	l_set(temp, b);
	l_div_s(a, temp);
}

/* c = a + b (mod m) */
void l_addmod(word *a, word *b, word *m, word *c)
{
	word temp[LSIZE];

	l_add(a, b, temp);
	l_mod(temp, m, c);
}

/* c = a - b (mod m) */
void l_submod(word *a, word *b, word *m, word *c)
{
	word temp[LSIZE];

	l_sub(a, b, temp);
	l_mod(temp, m, c);
}

/* c = ab (mod m) */
void l_mulmod(word *a, word *b, word *m, word *c)
{
	word temp[LSIZE];

	l_mul(a, b, temp);
	l_mod(temp, m, c);
}

/* modular square */
void l_sqrmod(word *a, word *m, word *c)
{
	l_mulmod(a, a, m, c);
}

/* Compute a power using the multiply and square method */
void l_expt(word *a, word *b, word *c)
{
	word temp[LSIZE], j;
	unsigned char bits[(((BITS-1)*LSIZE)/8)+1];

	/* zero? */
	l_set(c, 1);

	if (l_iszero(b))
		return;

	/* grab bits from exponent */
	j = 0;
	l_copy(b, temp);
	memset(bits, 0, sizeof(bits));
	while (!l_iszero(temp)) {
		bits[j >> 3] |= l_shr_s(temp) << (j & 7);
		++j;
	}

	/* calc power */
	while (--j != (word)-1) {
		l_sqr(c);
		if (bits[j >> 3] & (1 << (j & 7)))
			l_mul_s(c, a);
	}
}

/* modular power function, using multiply-and-square */
void l_exptmod(word *a, word *b, word *m, word *c)
{
	word temp[LSIZE], j;
	unsigned char bits[(((BITS-1)*LSIZE)/8)+1];

	/* zero? */
	l_set(c, 1);

	if (l_iszero(b))
		return;

	/* grab bits from exponent */
	j = 0;
	l_copy(b, temp);
	memset(bits, 0, sizeof(bits));
	while (!l_iszero(temp)) {
		bits[j >> 3] |= l_shr_s(temp) << (j & 7);
		++j;
	}

	/* calc power */
	while (--j != (word)-1) {
		l_sqrmod(c, m, temp);
		if (bits[j >> 3] & (1 << (j & 7)))
			l_mulmod(temp, a, m, c);
		else
			l_copy(temp, c);
	}
}

/* Greatest Common Divisor, using the Generalized Euclidean Algorithm */
void l_gcd(word *a, word *b, word *c)
{
	word r[LSIZE], A[LSIZE], B[LSIZE];

	if (l_cmp(a, b) == L_GT) {
		l_copy(a, A);
		l_copy(b, B);
	} else {
		l_copy(a, B);
		l_copy(b, A);
	}

	do {
		l_mod(A, B, r);				/* r = a mod b */
		l_copy(B, A);				/* A = B */
		l_copy(r, B);				/* B = r */
	} while (!l_iszero(r));

	l_copy(A, c);					/* c = GCD */
}

/* Lowest common multiple */
void l_lcm(word *a, word *b, word *c)
{
	word t[3][LSIZE];

	/* valid parm? */
	if (l_iszero(a) || l_iszero(b)) {
		l_error = L_INVNUM;
		return;
	}

	l_mul(a, b, t[0]);
	l_gcd(a, b, t[1]);
	l_div(t[0], t[1], c, t[2]);
}

/* multiplicative inverse */
void l_invmod(word *a, word *n, word *b)
{
	word u[3][LSIZE], v[3][LSIZE], g[3][LSIZE], y[LSIZE], temp[LSIZE];

	/* valid parms? */
	l_gcd(a, n, b);
	if (l_cmp_d(b, 1) != L_EQ) {
		l_error = L_INVNUM;
		return ;
	}

	memset(u, 0, sizeof(u));
	memset(v, 0, sizeof(v));
	l_set(u[0], 1);
	l_set(v[1], 1);
	l_copy(n, g[0]);
	l_copy(a, g[1]);
	l_clear(g[2]);
	l_clear(y);

	while( !l_iszero(g[1]) )
	{  /* gi = ui*n + vi*a */
		l_div(g[0], g[1], y, temp);
		l_mul(y, g[1], temp);
		l_sub(g[0], temp, g[2]);
		l_mul(y, u[1], temp);
		l_sub(u[0], temp, u[2]);
		l_mul(y, v[1], temp);
		l_sub(v[0], temp, v[2]);
		l_copy(g[1], g[0]);
		l_copy(g[2], g[1]);
		l_copy(u[1], u[0]);
		l_copy(u[2], u[1]);
		l_copy(v[1], v[0]);
		l_copy(v[2], v[1]);
	}
	l_copy(v[0], b);
}

/* Square root using a binary search method.  Takes roughly log2(n) steps
to find the integer square root */
void l_sqrt(word *n, word *N)
{
	word a[LSIZE], b[LSIZE], c[LSIZE], d[LSIZE], t[LSIZE], t2[LSIZE];

	/* check parm */
	if (l_iszero(n)) {
		l_error = L_INVNUM;
		return ;
	}

	l_copy(n, a);
	l_set(b, 1);
	l_shr(a, c);
	l_mul(c, c, d);

	for (;;) {
		l_sub(n, d, t);
		l_add(c, c, t2);
		l_sub_d(t2, 1);
		if ( ! ((l_cmp(d, n) == L_GT) || (l_cmp(t, t2) == L_GT)))
			break;

		if (l_cmp(d, n) == L_GT)
			l_copy(c, a);
		else
			l_copy(c, b);

		l_sub(a, b, c);
		l_add_d(c, 1);
		l_shr_s(c);
		l_add_s(c, b);
		l_mul(c, c, d);
	}
	l_copy(c, N);
}

/* read a string */
void l_fromradix(char *a, word radix, word *b)
{
	int i;

	l_clear(b);
	while (*a) {
		for (i = 0; i < 64; i++)
			if (radix_data[i] == *a)
				break;
		l_add_d(b, i);
		if (*++a)
			l_mul_d(b, radix);
	}
}

void l_toradix(word *a, word radix, char *out)
{
	char buf[BUFSIZE];
	word T[3][LSIZE], D[LSIZE], j;

	j = 0;
	l_set(D, radix);
	l_copy(a, T[0]);
	do  {
		l_div(T[j&1], D, T[~j & 1], T[2]);
		buf[j++] = radix_data[T[2][0]];
	} while (!l_iszero(T[j&1]));

	while (j--)
		*out++ = buf[j];
	*out = 0;
}

#ifdef L_PRINT

void l_print(word *a, word radix, FILE *out)
{
	char buf[BUFSIZE+1];

	l_toradix(a, radix, buf);
	fprintf(out, "%s", buf);
}

void l_println(word *a, word radix, FILE *out)
{
	l_print(a, radix, out);
	fprintf(out, "\n");
}

/* grab a number */
void l_readln(word *a, word radix, FILE *in)
{
	char buf[BUFSIZE+1];

	fgets(buf, sizeof(buf), in);

	/* trim new line (if present) */
	if (buf[strlen(buf)-1] == '\n')
		buf[strlen(buf)-1] = 0;

	l_fromradix(buf, radix, a);
}

#endif	/* L_PRINT */

int l_toraw(word *a, unsigned char *out)
{
	word temp[2][LSIZE], r[LSIZE], b[LSIZE], j, j2;
	unsigned char t[BUFSIZE];

	memset(t, 0, sizeof(t));
	l_copy(a, temp[0]);
	l_set(b, 256);
	j = 0;
	while (!l_iszero(temp[j&1])) {
		l_div(temp[j&1], b, temp[~j & 1], r);
		t[j++] = r[0] & 255;
	}

	j2 = j;
	while (--j != (word)-1)
		*out++ = t[j];

	return j2;
}

void l_readraw(word *a, unsigned char *in, int len)
{
	l_clear(a);
	while (len--) {
		l_mul_d(a, 256);
		l_add_d(a, *in++);
	}
}

#ifdef DEMO
/* Large Number Demo using LNUM
 *
 * Tom St Denis
 */

main(int argc,char **argv)
{
	word a[LSIZE], b[LSIZE], c[LSIZE], temp[LSIZE];

	if (argc == 1)
		abort("use: LNUM num op num");

	/* get the two numbers */
	l_fromradix(argv[1], 10, a);
	l_fromradix(argv[3], 10, b);

	switch (argv[2][0]) {
		case '+':	l_add(a, b, c); break;
		case '-':	l_sub(a, b, c); break;
		case '*':	l_mul(a, b, c); break;
		case '/':	l_div(a, b, c, temp); break;
		case '%':	l_mod(a, b, c); break;
		case '^':	l_expt(a, b, c); break;
	}
	if (l_error != L_OK)
		printf("Error: %s\n", l_error_str[l_error]);
	else {
		printf("Result: ");
		l_println(c, 10, stdout);
	}
}
#endif
