/* Simple Path Finder

	This program demonstrates a simple dynamic pathfinding algorithm.  It
avoids walls and tries not to walk on the 'grass'(green spots).  It basically
searches from the START marker to find the END marker then it backtracts.
While it is searching it displays the searched areas which should help explain
how the algorithm works. - Tom St Denis, July 1999
*/

/*
	The algorithm works by starting from the START marker then it tries to
find adjacent NORMAL or SLOW cells.  It then adds one or two respectively
based on the adjacent cell, to the adjacent cell.  Once it finds the END cell
it can start following the path.  The path is basically a set of numbers
from 1(start) to n(end).  It just finds adjacent lower numbers to find
it's way home.  This works because the numbers increase as you trace a
path AWAY from the START.

	In reality pre-computation would be used to speed things along.  A
line of sight type algorithm could be used, calling these routines only when
it is stuck.  Basically you don't want to call this version too often(it can
be customized however).  You could build maps for every 4th cell then tracing
a path to any point would be very easy(table lookups).  This requires

	m =  w(x**4)
		---------		(w = word size in bytes, n = divisor)
			n

	m bytes of ram.  For example with a square map of 60x60 with 16-bit path
data and a divisor of 4, this requires 6.17 MB of ram.  Obviously this
optimization is only practical where large amounts of memory are directly
accessible.  It's possible to build these maps and copy them to EMS for
usage.  This type of application should cache maps in local NEAR memory
for speed.  Each map in this case would require(xxw)/n or 1800 bytes of
ram and you could fit 9 maps per EMS page.

	A simple variation on the algorithm is to modify create_path() to search
more irregularly.  This has the effect of tracing faster but not as evenly.
On average this variation is faster and requires less ram.  It is suggested
that in real applications this(or any other) variant be used while being
optimized in assembly.  Here are the steps:

	1) Remove 'path2[][]' and performs stores to 'path[][]'(in do/while loop)
	2) Replace 'else if' with 'if' in do/while loop

	That's it.

	This demo loops indefinetately.  You can hit a key at any time to stop it
though.

	Colors:

		Brite White						-> Wall
		White							-> Normal Passable
		Bright Blue						-> Searched Normal Passable
		Magenta							-> Runner
		Green							-> Grass(slow ground to avoid)

		Multi(Cycling Colors)			-> Start and End(makes it more visable)

	Compile with: cc pathfind -fop m=s
*/

#include <stdio.h>
#include <lrg.h>

/* screen size */
#define	SW		100				/* Tiles Wide */
#define SH		100				/* Tiles High */
#define	BW		(320/SW)		/* Block width */
#define BH		(200/SH)		/* block height */
#define	HO		((320-SW*BW)/2)	/* Horisontal offset to center */
#define ITEMS	((SW*SH)/4)		/* Walls and grass items to place */

/* types of objects in GRID */
#define START	0				/* Starting point */
#define END		1				/* Ending point */
#define WALL	2				/* Impassable */
#define NORMAL	3				/* Normal(passable) */
#define WALKED	4				/* Previously walked on */
#define WALKEDN 5				/* Previously walked on grass */
#define WALK	6				/* Current possition of runner */
#define SLOW	7				/* Slower material */

/* grid is SWxSH and so is the path data */
unsigned grid[SW][SH], path[SW][SH];


/* prototypes required */
void draw(void);

/*
	The RNG used is called a 'Lagged Fibonacci Generator' which are adaptions
of 'Linear Feedback Generators'(LFRSs).  This type of additive generator is
fast in software.  This particular generator(denoted as F(7, 1, -)) has a
period of 127 * 32768 or 2^21.  The choice of generator and polynomial was
of speed, simplicity and cause.  The normal linear congruetial generator does
not make even outputs(in make()) and a LFSR is too slow.  Longer gererators
would be F(55, 24, -) and F(52, 19, -) which are ideally small and have much
longer periods.  These longer generators also tend to have somewhat better
'randomness', however the F(7, 1, -) generator will work fine here.  Also note
that subtraction(instead of addition which is the normal function) was used
so that the carries will move downwards.  This is because in make() only the
first ~6 bits are used of the rng output.  If addition was used this generator
would have a period of 127*50 or 6350.
*/
unsigned seed[7], sx, sy;	/* rng data */
unsigned rng(void)
{
	unsigned temp;

	/* update */
	temp =(seed[sx] -= seed[sy]);

	/* clock */
	sx =(sx + 1) % 7;
	sy =(sy + 1) % 7;

/*	can code the above as(avoid the modulus operation)
	sx =(sx < 6) ? ++sx : 0;
	sy =(sy < 6) ? ++sy : 0; */

	return temp;
}

/* truncated results(similar to random(n)) */
unsigned rnga(unsigned n)
{
	return rng() % n;
}

/* create path(grid[][] must already be constructed) */
void create_path()
{
	int sx, sy, ex, ey, x, y, done, i;
	int path2[SW][SH];

	/* clear path */
	memset(path, 0, sizeof(path));
	memset(path2, 0, sizeof(path2));

	/* find start and end */
	for(x = 0; x < SW; ++x)
		for(y = 0; y < SH; ++y)
			if(grid[x][y] == START) {
				sx = x;
				sy = y; }
			else if(grid[x][y] == END) {
				ex = x;
				ey = y; }

	/* mark starting place */
	i = done = 0;
	path2[sx][sy] = path[sx][sy] = 1;

	do {
		if(++i == 10000)					/* Time out */
			return;

		draw();

		/* look for already marked cells and update */
		for(x = 0; x < SW; ++x)
			for(y = 0; y < SH; ++y)
				if(path[x][y] != 0) {
					/* try to add down cell */
					if((y < SH-1) && (grid[x][y+1] != WALL) && !path[x][y+1])
						path2[x][y+1] = path[x][y] + (grid[x][y+1] == SLOW ? 2 : 1);

					/* try to add up cell */
					else if((y > 0) && (grid[x][y-1] != WALL) && !path[x][y-1])
						path2[x][y-1] = path[x][y] + (grid[x][y-1] == SLOW ? 2 : 1);

					/* try to add left cell */
					else if((x > 0) && (grid[x-1][y] != WALL) && !path[x-1][y])
						path2[x-1][y] = path[x][y] + (grid[x-1][y] == SLOW ? 2 : 1);

					/* try to add right cell */
					else if((x < SW-1) && (grid[x+1][y] != WALL) && !path[x+1][y])
						path2[x+1][y] = path[x][y] + (grid[x+1][y] == SLOW ? 2 : 1);

					if((ex == x) && (ey == y))
						done = 1; }

		memcpy(path, path2, sizeof(path)); }	/* Update path data */
	while(!done && !kbhit());
}

/* draw the grid on screen */
void draw(void)
{
	int x, y;
	static int c;

	for(x = 0; x < SW; ++x)
		for(y = 0; y < SH; ++y)
			switch(grid[x][y]) {
			case START	: lrg_fbox(x * BW + HO, y * BH, BW, BH, ++c); break;
			case END	: lrg_fbox(x * BW + HO, y * BH, BW, BH, ++c); break;
			case NORMAL	: lrg_fbox(x * BW + HO, y * BH, BW, BH, path[x][y] ? 9 : 7); break;
			case SLOW	: lrg_fbox(x * BW + HO, y * BH, BW, BH, path[x][y] ? 10 : 2); break;
			case WALKED	: lrg_fbox(x * BW + HO, y * BH, BW, BH, 14); break;
			case WALKEDN: lrg_fbox(x * BW + HO, y * BH, BW, BH, 4); break;
			case WALK	: lrg_fbox(x * BW + HO, y * BH, BW, BH, 5); break;
			case WALL	: lrg_fbox(x * BW + HO, y * BH, BW, BH, 15); break;
			default		: lrg_fbox(x * BW + HO, y * BH, BW, BH, 0); }
}

/* make a map(demo only) */
void make(void)
{
	int x, y, i;

	for(x = 0; x < SW; ++x)
		for(y = 0; y < SH; ++y)
			grid[x][y] = NORMAL;

	/* set start and end points */
	grid[rnga(SW)][rnga(SH)] = END;
	grid[rnga(SW)][rnga(SH)] = START;

	for(i = 0; i < ITEMS; ++i) {
		x = rnga(SW);
		y = rnga(SH);
		if(grid[x][y] == NORMAL)
			grid[x][y] = WALL;

		x = rnga(SW);
		y = rnga(SH);
		if(grid[x][y] == NORMAL)
			grid[x][y] = SLOW; }
}

/* Pick a new(x, y) based on adjacent cells */
void pick(int *x, int *y)
{
	unsigned u, v;

	u = *x; v = *y;

	/* check above */
	if((v > 0) &&(path[u][v] > path[u][v - 1]) && path[u][v - 1]) {
		*y = v - 1;
		return; }

	/* check below */
	if((v < SH-1) && (path[u][v] > path[u][v + 1]) && path[u][v + 1]) {
		*y = v + 1;
		return; }

	/* check right */
	if((u < SW-1) && (path[u][v] > path[u + 1][v]) && path[u + 1][v]) {
		*x = u + 1;
		return; }

	/* check left */
	if((u > 0) && (path[u][v] > path[u - 1][v]) && path[u - 1][v]) {
		*x = u - 1;
		return; }

	/* force move by making current cell unattractive and recursion */
	if(path[u][v] != 1) {
		path[u][v] += 10;
		pick(x, y);
		path[u][v] += 10; }
}

/* follow a map, this will pick new values for x and y based on the
path data.  In reality this would be replaced with a vector translator
(to make smaller way-points). */
void follow()
{
	int ex, ey, x, y, i, temp;

	i = 0;
	/* find end */
	for(x = 0; x < SW; ++x)
		for(y = 0; y < SH; ++y)
			if(grid[x][y] == END) {
				ex = x;
				ey = y;
				break; }

	x = ex;
	y = ey;
	while(!kbhit()) {
		if(++i == 10000)			/* time out */
			return;

		pick(&x, &y);

		if(grid[x][y] == START)
			return;

		temp = grid[x][y];
		grid[x][y] = WALK;
		draw();
		grid[x][y] =(temp == NORMAL) ? WALKED : WALKEDN; }
}

/* simple demo */
void main()
{
	int a;

	/* seed RNG and run for a bit */
	get_time(&seed[0], &seed[1], &seed[2]);
	get_date(&seed[3], &seed[4], &seed[5]);
	seed[6] = 1;
	sx = 0;
	sy = 6;

	for(a = 0; a < 10000; ++a)
		rng();

	lrg_open();

	do {
		make();
		create_path();
		follow();
		delay(2000);
	} while(!kbtst());

	lrg_close();
}
