/*
 * Example program to show how to re-size the DOS memory block to make
 * the program smaller. On startup, MICRO-C/PC allocates an entire 64K
 * block for data. This allows the maximum possible stack and heap growth.
 * This is fine for general programs, but "shell" programs and other
 * specialized applications may wish to reduce the amount of memory used
 * while they are running. This program is an example of how you can re-
 * size the memory allocation. NOTE: If you are writing a TSR, the tsr()
 * function supplied with MICRO-C/PC already re-sizes the memory.
 *
 * In TINY model, there is only one memory block, which contains the
 * program PSP, CODE and DATA. In SMALL model, there are three blocks,
 * one for the PSP, one for the CODE, and one for the DATA. In both
 * cases, the size of the PSP and CODE is fixed, so the only thing you
 * can really re-size is the data block.
 *
 * When resizing the data block, you must:
 *	1) move the SP so that it remains in the allocated part of the block
 *	2) Insure that you leave enough memory for expected stack and heap
 *	   use. In TINY model, you have to include the size of the PSP (256),
 *	   plus code and static data in this calculation. In SMALL model,
 *	   you need only consider static data, since the PSP+CODE are in
 *	   separate physical segments.
 *
 * Example:	Program has 5K of code and 3K of static data. We desire 2K
 *			of memory for stack and heap growth:
 *
 *	  In TINY model: PSP+CODE+SDATA+STACK= 256+5120+3072+2048= 10496
 *				Therefore: MOV BX,656  and  MOV SP,10496
 *
 *	  In SMALL model: SDATA+STACK= 3072+2048= 5120
 *				Therefore: ADD BX,320  and  MOV SP,5120
 *
 * NOTE: The values given in BX are for 16 byte pages, therefore, divide
 *       the value loaded into SP by 16 to get this number.
 *
 * ?COPY.TXT 1996-2003 Dave Dunfield
 *  -- see COPY.TXT --.
 *
 * Permission granted for personal (non-commercial) use only.
 *
 * To compile this program in TINY  model: CC resize -fop TINY=
 * To compile this program in SMALL model: CC resize -fop m=s SMALL=
 */
#include <stdio.h>

/*
 * Main program, resize block, the shell to DOS to display memory use
 */
main()
{
#ifdef	TINY
	#define	MODEL_OK
	// TINY model: Resize the common PSP/CODE/DATA block
	asm {
		extrn	_PSP:word	; Declare PSP
		MOV		BX,256		; 256 pages (4096 bytes) for program
		MOV		SP,4096		; Drop SP (Keep in allocated memory)
		MOV		ES,DGRP:_PSP; Resizing PSP segment
		MOV		AH,4Ah		; DOS resize memory block function
		INT 	21h			; Call DOS
	}
#endif

#ifdef	SMALL
	#define	MODEL_OK
	// SMALL model: Calculate PSP+CODE size, then resize for data
	asm {
		extrn	_PSP:word	; Declare PSP
		MOV		BX,DGRP		; Get data group segment address
		SUB		BX,DGRP:_PSP; Subtract PSP segment (Get PSP+CODE size)
		ADD		BX,256		; Add 256 pages (4096 bytes) for data
		MOV		SP,4096		; Drop SP (Keep in allocated memory)
		MOV		ES,DGRP:_PSP; Resizing PSP segment
		MOV		AH,4Ah		; DOS resize memory block function
		INT 	21h			; Call DOS
	}
#endif

#ifndef	MODEL_OK
	#error Memory model not defined for re-size
#endif

	system("mem /c");		// Call DOS program to display memory use

	// Since we moved SP, the return address is not longer on our stack
	// Therefore we *HAVE* to call exit. If main() were to return, the
	// program would crash!
	exit(0);
}
