/*
 * The MS-DOS exit code set by a terminating program can be read only ONCE
 * via DOS INT 21, Function 4D ... This function also clears the exit code,
 * so that it cannot be retrieved again!   This can cause a problem if you
 * need to know the exit status of a program when you will not directly
 * receive control after it terminates (example: Calling a program through
 * another program, such as when using COMMAND.COM to parse command line
 * arguments, COMMAND.COM retrieves the exit status (destroying it in the
 * process), and does not pass it back to you... making it lost forever).
 *
 * This program demonstrates saving the MS-DOS exit status by trapping
 * INT-21/4D, and storing the exit codes as they are retrieved.
 *
 * This program also calls itself "recursively"... When passed a command
 * line argument, it simply exits with the specified value as the exit
 * code ...  When executed without arguments, the program will perform the
 * demonstration, running itself (with arguments) as an example executable
 * program where we can know what the exit code is supposed to be.
 *
 * Compile with DDS Micro-C/PC (http://www.dunfield.com): cc SAVEXIT -fop
 */
#include <stdio.h>

#define	RC_QUEUE	8			// Must be power of 2

unsigned
	rc[RC_QUEUE],				// Circular return code queue
	rcp,						// Return code queue pointer
	rcc;						// Return code count

/*
 * Interrupt 21 handler
 *
 * Pass all but function 4D directly to original DOS vector.
 * For Func 4D, call DOS, then store retrieved exit code in our table.
 */
asm {
int21:	CMP		AH,4Dh			; "Get return code"?
		JNZ		int21a			; No, proceed directly to DOS
; Function 4D - Get exit code - Call DOS first
		PUSHF					; Save flags (for DOS IRET)
		PUSH	CS				; Save segment (for DOS IRET)
		CALL	int21a			; Call it
; Then save the returned exit code
		PUSH	DS				; Save DS
		PUSH	CS				; Get CS
		POP		DS				; DS = CS
		PUSH	BX				; Save BX
		MOV		BX,DGRP:_rcp	; Get return code pointer
		SHL		BX,1			; x2 for word entries
		MOV		DGRP:_rc[BX],AX	; Save 
		SHR		BX,1			; /2 back to byte
		INC		BX				; Skip to next
		AND		BX,RC_QUEUE-1	; Mask to wrap queue
		MOV		DGRP:_rcp,BX	; Resave new queue pointer
		INC		DGRP:_rcc		; Advance the return code count
		POP		BX				; Restore BX
		POP		DS				; Restore DS
		IRET					; Back to original caller
; Long jump to saved DOS vector
int21a:	DB		0EAh			; Intersegment JMP
int21o	DW		0				; Offset
int21s	DW		0				; Segment
}

/*
 * Retrieve the DOS error level
 */
unsigned get_errorlevel(void) asm
{
		MOV		AH,4Dh			; Ask for errorlevel
		INT		21h				; Ask DOS (return in AX)
}

int main(int argc, char *argv[])
{
	unsigned r;

	/* Called with an argument, just exit with specified code */
	if(argc > 1) {
		printf("SAVEXIT: Terminating with exit code %u\n", r = atoi(argv[1]));
		exit(r); }

	/* Hook INT21 vector so we can "look in" on DOS calls */
	printf("Hooking int21\n");
	asm {
		MOV		AX,3521h		; Get interrupt 21
		INT		21h				; Call DOS
		MOV		CS:int21o,BX	; Save offset
		MOV		CS:int21s,ES	; Save segment
		MOV		AX,2521h		; Set interrupt 21
		MOV		DX,OFFSET int21	; Get offset
		INT		21h				; Set the segment
	}

	rcp = rcc = 0;			// Clear our table pointers

	/*
	 * This example executes a program directly, when it exits,
     * The "exec()" function retrieves the exit code via INT 21/4D
	 * and passes it back to us.
	 */
	printf("\nExecuting program directly\n");
	printf("Return from exec()=%u\n", exec("SAVEXIT.COM", "5"));

	/*
	 * This demonstrates that INT 21/4D cannot be called more than
	 * once... The first call by exec() above has cleared the exit
	 * code and this will only print a zero (exit code is lost).
	 */
	printf("Errorlevel=%u\n", get_errorlevel());

	/*
	 * This example executes the program by using system(), which
	 * loads COMMAND.COM, allowing it to parse the command line.
	 * Unfortunately, COMMAND.COM will call INT 21/4D, thereby
	 * clearing the exit code, which it DOES NOT pass back to it's
	 * caller... COMMAND.COM always returns 0.
	 *
	 * Although RC is going to terminate with an exit code of 6, the
	 * calling program has ABSOLUTELY NO WAY to retrieve this exit
	 * code (except for the fact that we have hooked INT 21/4D).
	 */
	printf("\nExecuting program via COMMAND.COM\n");
	printf("Return from system()=%u\n", system("SAVEXIT 6"));

	/*
	 * Display the INT 21/4D calls which have occurred
	 * This will tell us that the return codes were (newest to oldest):
	 *  0 - Returned by COMMAND.COM and retrieved by system();
	 *	6 - Returned by 2nd run of RC, retrieved by COMMAND.COM and lost!
	 *  0 - Usless/Cleared value returned by get_errorlevel()
	 *	5 - Returned by 1st RC, retrieved by exec()
	 */
	printf("\nReturn code count=%u\n", rcc);
	if(rcc > RC_QUEUE)			// Truncate to queue size of higher
		rcc  = RC_QUEUE;
	printf("Last %u return codes (newest to oldest): ", rcc);
	for(r=0; r < rcc; ++r)
		printf(" %u", rc[rcp = (rcp - 1) & (RC_QUEUE-1)]);
	putc('\n', stdout);

	/* Release the INT21 vector */
	printf("\nUnhooking int21\n");
	asm	{
		PUSH	DS				; Save DS
		MOV		AX,2521h		; Set interrupt 10
		MOV		DX,CS:int21o	; Get offset
		MOV		DS,CS:int21s	; Get segment
		INT		21h				; Call DOS
		POP		DS				; Restore DS
	}
}
