/*
 * Signed long math functions for Micro-C
 *
 * The LONGMATH library included with Micro-C operates on unsigned long
 * numbers, however many of the functions will operate equally well on
 * signed long numbers. In fact, only the longdiv(), longcmp(), atol()
 * and ltoa() functions are problematic with signed numbers.
 *
 * This program demonstrates replacement slongdiv(), slongcmp(), satol(),
 * and sltoa() functions which work correctly with signed long numbers.
 * This is accomplished by manipulating the sign bits and negating the
 * values where necessary to make them suitable for the standard unsigned
 * versions of these functions.
 *
 * Like any two's-compliment number, the sign bit is stored in the highest
 * bit of the value. In the LONGMATH implementation, this represented in
 * bit7 of the highest byte of the character array holding the long value.
 *
 * This program includes a very simple long number "calculator", which
 * operates with signed long numbers.
 *
 * The ltoa() and atol() functions are included in the Micro-C/PC library.
 * For processors not having these functions in their library, you can
 * extract them from the LONGCALC.C demo program.
 *
 * Compile command: cc signlong -fop
 * Dave Dunfield - Oct 2000
 */
#include <stdio.h>

#define	LSIZE	4		/* 4 byte (32 bit) longs */

char l1[LSIZE], l2[LSIZE];

/*
 * Signed long division function
 *
 * Divide absolute value of numbers, then negate if only one
 * of the input quantities was negative.
 */
void slongdiv(char *num1, char *num2)
{
	char flag, lt[LSIZE];
	flag = 0;
	if(num1[LSIZE-1] & 0x80) {		/* Numerator is negative */
		longcpy(lt, num1);
		longset(num1, 0);
		longsub(num1, lt);
		flag = 1; }
	if(num2[LSIZE-1] & 0x80) {		/* Denominator is negative */
		longset(lt, 0);
		longsub(lt, num2);
		longdiv(num1, lt);
		flag ^= 1; }
	else
		longdiv(num1, num2);
	if(flag) {						/* Result should be negative */
		longcpy(lt, num1);
		longset(num1, 0);
		longsub(num1, lt); }
}

/*
 * Signed long comparison function
 *
 * If signs are different, return -1/1 based on which one is negative.
 */
int slongcmp(char *num1, char *num2)
{
	if((num1[LSIZE-1] ^ num2[LSIZE-1]) & 0x80)	/* Signs differ */
		return (num1[LSIZE-1] & 0x80) ? -1 : 1;
	return longcmp(num1, num2);
}

/*
 * Signed ASCII to LONG function
 *
 * If leading '-' : negate number after conversion
 */
int satol(char *string, char *num, int radix)
{
	int r;
	char lt[LSIZE];

	if(*string == '-') {			/* Input is negative */
		r = atol(string+1, lt, radix);
		longset(num, 0);
		longsub(num, lt);
		return r; }
	return atol(string, num, radix);
}

/*
 * Signed LONG to ASCII function
 *
 * If number is negative, output '-' and negate before conversion
 */
char *sltoa(char *num, char *string, int radix)
{
	char lt[LSIZE];
	if(num[LSIZE-1] & 0x80) {		/* Number is negative */
		*string++ = '-';
		longset(lt, 0);
		longsub(lt, num);
		return ltoa(lt, string, radix); }
	return ltoa(num, string, radix);
}


/*
 * Demo "calculator"
 */
main(int argc, char *argv[])
{
	char outbuf[25];

	if(argc < 3)
		abort("Use: sldemo <num1> +-*/? <num2>\n");

	satol(argv[1], l1, 10);
	satol(argv[3], l2, 10);
	switch(*argv[2]) {
		case '+' :				/* Addition */
			longadd(l1, l2);
			break;
		case '-' :				/* Subtraction */
			longsub(l1, l2);
			break;
		case '*' :				/* Multiplication */
			longmul(l1, l2);
			break;
		case '/' :				/* Division */
			slongdiv(l1, l2);
			break;
		case '?' :				/* Comparison */
			printf("Compare result: %d\n", slongcmp(l1, l2));
			return;
		default:
			abort("Bad operator"); }

	sltoa(l1, outbuf, 10);
	printf("Arithmetic result: %s\n", outbuf);
}
