// LZSS Huffman compression
#define	Int	unsigned

// LZSS Parameters
#define SBSIZE		4096	// Size of string buffer
#define LASIZE		60		// Size of look-ahead buffer
#define THRESHOLD	2
#define NIL			SBSIZE	// End of tree's node

// Huffman coding parameters
#define N_CHAR  	(256-THRESHOLD+LASIZE)	// char code (= 0..N_CHAR-1)
#define TSIZE 		(N_CHAR * 2 - 1)		// Size of table
#define ROOT		(TSIZE - 1)				// root position
#define MAX_FREQ	0x8000					// update when cumulative

unsigned
	Ltmp[2],			// Long temp
	Usize[2],			// UncompressedSize
	Csize[2],			// CompressedSize
	PrintCount[2];		// Next -/-\ update
unsigned
	MatchPosition,
	MatchLength,
	Lson[SBSIZE + 1],
	Rson[SBSIZE + 257],
	Dad[SBSIZE + 1],
	Prnt[TSIZE + N_CHAR],
	Son[TSIZE],
	Freq[TSIZE + 1],
	PutBuf;
unsigned char
	verbose = 7,
	Pcount,				// -/-\ position
	TextBuf[SBSIZE + LASIZE - 1],
	PutLen;

unsigned char
	PrintChars[] = { '-', '\\', '|', '/' };

// encoder tables
unsigned char p_len[64] = {
	0x03, 0x04, 0x04, 0x04, 0x05, 0x05, 0x05, 0x05,
	0x05, 0x05, 0x05, 0x05, 0x06, 0x06, 0x06, 0x06,
	0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06,
	0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07,
	0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07,
	0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07,
	0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08,
	0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08 };
unsigned char p_code[64] = {
	0x00, 0x20, 0x30, 0x40, 0x50, 0x58, 0x60, 0x68,
	0x70, 0x78, 0x80, 0x88, 0x90, 0x94, 0x98, 0x9C,
	0xA0, 0xA4, 0xA8, 0xAC, 0xB0, 0xB4, 0xB8, 0xBC,
	0xC0, 0xC2, 0xC4, 0xC6, 0xC8, 0xCA, 0xCC, 0xCE,
	0xD0, 0xD2, 0xD4, 0xD6, 0xD8, 0xDA, 0xDC, 0xDE,
	0xE0, 0xE2, 0xE4, 0xE6, 0xE8, 0xEA, 0xEC, 0xEE,
	0xF0, 0xF1, 0xF2, 0xF3, 0xF4, 0xF5, 0xF6, 0xF7,
	0xF8, 0xF9, 0xFA, 0xFB, 0xFC, 0xFD, 0xFE, 0xFF };

unsigned L1[] = {	1,	0	};	// Long 1
unsigned L2[] = {	2,	0	};	// Long 2
unsigned Lps[]= { 1024, 0	};	// Long printstep

extern Rinit(), Rreset(), Rread(), Rwrite();	// RamFile functions

void InitTree()  // Initialize tree
{
	Int i;

	for(i = SBSIZE + 1; i <= (SBSIZE + 256); ++i)
		Rson[i] = NIL;			// root
	for(i = 0; i < SBSIZE; ++i)
		Dad[i] = NIL;			// node
	for(i=0; i < (SBSIZE + LASIZE - 1); ++i)
		TextBuf[i] = 0;
}

void InsertNode(Int r)  // Insert node to the tree
{
	Int i, p;
	int cmp;
	unsigned char *key;
	unsigned c;

	cmp = 1;
	key = &TextBuf[r];
	p = ((unsigned)SBSIZE + 1) + key[0];
	Rson[r] = Lson[r] = NIL;
	MatchLength = 0;
	for(;;) {
		if(cmp >= 0) {
			if(Rson[p] != NIL)
				p = Rson[p];
			else {
				Rson[p] = r;
				Dad[r] = p;
				return; } }
		else {
			if(Lson[p] != NIL)
				p = Lson[p];
			else {
				Lson[p] = r;
				Dad[r] = p;
				return; } }
		for(i = 1; i < LASIZE; ++i)
			if(cmp = key[i] - TextBuf[p + i])
				break;
		if(i > THRESHOLD) {
			if(i > MatchLength) {
				MatchPosition = ((r - p) & (SBSIZE - 1)) - 1;
				if((MatchLength = i) >= LASIZE)
					break; }
			if(i == MatchLength) {
				if ((c = ((r - p) & (SBSIZE - 1)) - 1) < MatchPosition)
					MatchPosition = c; } } }
	Dad[r] = Dad[p];
	Lson[r] = Lson[p];
	Rson[r] = Rson[p];
	Dad[Lson[p]] = r;
	Dad[Rson[p]] = r;
	if(Rson[Dad[p]] == p)
		Rson[Dad[p]] = r;
	else
		Lson[Dad[p]] = r;
	Dad[p] = NIL;  // remove p
}

void DeleteNode(Int p)  // Delete node from tree
{
	Int q;

	if(Dad[p] == NIL)
		return;			// unregistered
	if(Rson[p] == NIL)
		q = Lson[p];
	else if (Lson[p] == NIL)
		q = Rson[p];
	else {
		q = Lson[p];
		if(Rson[q] != NIL) {
			do
				q = Rson[q];
			while(Rson[q] != NIL);
			Rson[Dad[q]] = Lson[q];
			Dad[Lson[q]] = Dad[q];
			Lson[q] = Lson[p];
			Dad[Lson[p]] = q; }
		Rson[q] = Rson[p];
		Dad[Rson[p]] = q; }
	Dad[q] = Dad[p];
	if(Rson[Dad[p]] == p)
		Rson[Dad[p]] = q;
	else
		Lson[Dad[p]] = q;
	Dad[p] = NIL;
}

void Putcode(Int l, unsigned c)		// output c bits
{
	PutBuf |= c >> PutLen;
	if((PutLen += l) >= 8) {
		Rwrite(PutBuf >> 8);
		if((PutLen -= 8) >= 8) {
			Rwrite(PutBuf);
			longadd(Csize, L2);
			PutLen -= 8;
			PutBuf = c << (l - PutLen); }
		else {
			PutBuf <<= 8;
			longadd(Csize, L1); } }
}

void reconst()		// Reconstruct freq tree
{
	Int i, j;
	Int k;
	unsigned f, l;

	// halve cumulative freq for leaf nodes
	j = 0;
	for(i = 0; i < TSIZE; ++i) {
		if(Son[i] >= TSIZE) {
			Freq[j] = (Freq[i] + 1) / 2;
			Son[j] = Son[i];
			++j; } }
	// make a tree : first, connect children nodes
	for(i = 0, j = N_CHAR; j < TSIZE; i += 2, ++j) {
		k = i + 1;
		f = Freq[j] = Freq[i] + Freq[k];
		for(k = j - 1; f < Freq[k]; --k);
		++k;
		l = (j - k) * 2;
		memmove(&Freq[k + 1], &Freq[k], l);
		Freq[k] = f;
		memmove(&Son[k + 1], &Son[k], l);
		Son[k] = i; }

	// connect parent nodes
	for(i = 0; i < TSIZE; ++i) {
		if((k = Son[i]) >= TSIZE)
			Prnt[k] = i;
		else
			Prnt[k] = Prnt[k + 1] = i; }
}

// update freq tree
void update(Int c)
{
	Int i, k;
	Int j, l;

	if(Freq[ROOT] == MAX_FREQ)
		reconst();
	c = Prnt[c + TSIZE];
	do {
		k = ++Freq[c];
		if(k > Freq[l = c + 1]) {	// swap nodes to keep tree freq-ordered
			while (k > Freq[++l]);
			--l;
			Freq[c] = Freq[l];
			Freq[l] = k;
			i = Son[c];
			Prnt[i] = l;
			if(i < TSIZE)
				Prnt[i + 1] = l;
			j = Son[l];
			Son[l] = i;
			Prnt[j] = c;
			if(j < TSIZE)
				Prnt[j + 1] = c;
			Son[c] = j;
			c = l; } }
	while(c = Prnt[c]);		// do until we reach root
}

void EncodeChar(unsigned c)
{
	unsigned i;
	Int j;
	Int k;

	k = Prnt[c + TSIZE];

	// search connections from leaf node to root
	i = j = 0;
	do {
		i >>= 1;
		if(k & 1) 			// Odd address?
			i |= 0x8000;
		++j; }
	while((k = Prnt[k]) != ROOT);
	Putcode(j, i);
	update(c);
}

void EncodePosition(unsigned c)
{
	unsigned i;

	// output upper 6 bits with encoding
	i = c >> 6;
	Putcode(p_len[i], (unsigned)p_code[i] << 8);
	// output lower 6 bits directly
	Putcode(6, (c & 0x3f) << 10);
}

void EncodeEnd()
{
	if(PutLen) {
		Rwrite(PutBuf >> 8);
		longadd(Csize, L1); }
}

// Initialize Huffman frequency tree
void StartHuff()
{
	Int i, j;

	PutLen = PutBuf = 0;

	for(i = 0; i < N_CHAR; ++i) {
		Freq[i] = 1;
		Son[i] = i + TSIZE;
		Prnt[i + TSIZE] = i; }
	i = 0; j = N_CHAR;
	while(j <= ROOT) {
		Freq[j] = Freq[i] + Freq[i + 1];
		Son[j] = i;
		Prnt[i] = Prnt[i + 1] = j;
		i += 2; ++j; }
	Freq[TSIZE] = 0xffff;
	Prnt[ROOT] = 0;
}

// Main compression routine
#define	Clr(x)	memset(x,sizeof(x),0)
void Encode()
{
	Int i, len;
	Int c, r, s, last_match_length;

#if 0
Clr(Ltmp);
Clr(Usize);
Clr(Csize);
Clr(PrintCount);
Clr(MatchPosition);
Clr(MatchLength);
Clr(Lson);
Clr(Rson);
Clr(Dad);
Clr(Prnt);
Clr(Son);
Clr(Freq);
Clr(PutBuf);
Clr(Pcount);
Clr(TextBuf);
Clr(PutLen);
#endif
	Rreset();
	longset(Usize, 0);
	longset(Csize, 0);
	longset(PrintCount, 0);

	StartHuff();
	InitTree();
	s = 0;
	r = SBSIZE - LASIZE;
	for(i = s; i < (SBSIZE-LASIZE); ++i)
		TextBuf[i] = 0;	//DD was ' '
	for(len = 0; len < LASIZE && (c = getc(fpi)) != EOF; ++len)
		TextBuf[r + len] = c;
	longset(Ltmp, len);
	longadd(Usize, Ltmp);
	for(i = 1; i <= LASIZE; ++i)
		InsertNode(r - i);
	InsertNode(r);
	do {
		if(MatchLength > len)
			MatchLength = len;
		if(MatchLength <= THRESHOLD) {
			MatchLength = 1;
			EncodeChar(TextBuf[r]); }
		else {
			EncodeChar(255 - THRESHOLD + MatchLength);
			EncodePosition(MatchPosition); }
		last_match_length = MatchLength;
		for(i = 0; i < last_match_length && (c = getc(fpi)) != EOF; ++i) {
			DeleteNode(s);
			TextBuf[s] = c;
			if(s < LASIZE - 1)
				TextBuf[s + SBSIZE] = c;
			s = (s + 1) & (SBSIZE - 1);
			r = (r + 1) & (SBSIZE - 1);
			InsertNode(r); }
		longset(Ltmp, i);
		longadd(Usize, Ltmp);
		if(longcmp(Usize, PrintCount) == 1) {
			if(verbose) {
				putc(PrintChars[++Pcount & 0x03], stderr);
				putc(8, stderr); }
			longadd(PrintCount, Lps); }
		while(i++ < last_match_length) {
			DeleteNode(s);
			s = (s + 1) & (SBSIZE - 1);
			r = (r + 1) & (SBSIZE - 1);
			if(--len)
				InsertNode(r); } }
	while(len > 0);
	EncodeEnd();
}
