;************************************************************
;*      Software debug monitor for the 68HC908GP32          *
;************************************************************
;*    This monitor is setup for use on a 68HC908GP32,       *
;*  which is installed in the BiPOM  MINI-MAX/908-C board   *
;************************************************************
;* Based on Dunfield 68HC12 monitor (MON12)                 *
;* Copyright 2001-2005 BiPOM Electronics                    *
;* Portions ?COPY.TXT 1996-2005 Dave Dunfield               *
;*  -- see COPY.TXT --.                                     *
;* Date: 10/12/2001                                         *
;* Last revised: Feb 5, 2002 - Dave Dunfield                *
;* This monitor is NOT supported by DDS - Contact BiPom.    *
;************************************************************
R_VEC		EQU	$FFFE	;Reset Vector
BREAK_VEC	EQU	$FFFC	;BRK Module Vector
START_MONITOR	EQU	$E600	;MON08 Start Address
RAM_MONITOR	EQU	$00A0	;MON08 RAM Data Memory(System Area)
RAM_DATA	EQU	$0100	;MON08 RAM Data Memory(Data Area)
STACK		EQU	$0240	;MON08 Stack
;
; ADDRESSING MODES
;
INH		EQU	$00
IMM		EQU	$10
DIR		EQU	$20
EXT		EQU	$30
DD		EQU	$40
IX_PLUS_D	EQU	$50
REL		EQU	$60
IX		EQU	$70
IX1		EQU	$80
IX2		EQU	$90
IMD		EQU	$A0
DIX_PLUS	EQU	$B0
SP1		EQU	$C0
SP2		EQU	$D0
IX_PLUS		EQU	$E0
IX1_PLUS	EQU	$F0
;
; DEFAULTS REGISTERS OF USER APPLICATION
;
DEFAULT_A	EQU	0	;A
DEFAULT_H	EQU	0	;H
DEFAULT_X	EQU	0	;X
DEFAULT_SP	EQU	$00FF	;SP
DEFAULT_PC	EQU	$8000	;PC
DEFAULT_CC	EQU	$69	;CC
;
; CPU REGISTERS
;
PTA		EQU	0	;Port A Data Register
PTB		EQU	1	;Port B Data Register
PTC		EQU	2	;Port C Data Register
PTD		EQU	3	;Port D Data Register
DDRA		EQU	4	;Data Direction Register A
DDRB		EQU	5	;Data Direction Register B
DDRC		EQU	6	;Data Direction Register C
DDRD		EQU	7	;Data Direction Register D
PTE		EQU	8	;Port E Data Register
DDRE		EQU	$000C	;Data Direction Register E
PTAPUE		EQU	$000D	;Port A Input PullUp
PTCPUE		EQU	$0000E	;Port C Input PullUp
;
SCC1		EQU	$0013	;SCI Control Register 1
SCC2		EQU	$0014	;SCI Control Register 2
SCC3		EQU	$0015	;SCI Control Register 3
SCS1		EQU	$0016	;SCI Status Register 1
SCS2		EQU	$0017	;SCI Status Register 2
SCDR		EQU	$0018	;SCI Data Register 1
SCBR		EQU	$0019	;SCI Baud Rate Register 1
;
FLCR		EQU	$FE08	;Flash Control Register
FLBPR		EQU	$FF7E	;Flash Block Protect Register
;
SBSR		EQU	$FE00	;SIM Break Status Register
SBFCR		EQU	$FE03	;SIM Break Flag Control Register
BRKH		EQU	$FE09	;Break Address Register High
BRKL		EQU	$FE0A	;Break Address Register Low
BRKSCR		EQU	$FE0B	;Break Status and Control Register

		ORG	RAM_MONITOR
;
; REGISTERS OF USER APPLICATION
;
REG_A		DS	1	;A
REG_H		DS	1	;H
REG_X		DS	1	;X
REG_SP		DS	2	;SP
REG_PC		DS	2	;PC
REG_CC		DS	1	;CC
;
; Monitor variables
;
REG_TEMP	DS	2		;Temporary register
START_SOURCE	EQU	*		;Doubles with START_ADDR
START_ADDR	DS	2		;?
START_DIST	EQU	*		;Doubles with HIGH/LOW-BYTE
FINISH_ADDR	EQU	*		;Doubles with HIGH/LOW_BYTE
HIGH_BYTE	DS	1		;High byte of word entry
LOW_BYTE	DS	1		;Low byte of work entry
COUNTER		DS	1		;Counter
FINISH_SOURCE	EQU	*		;Doubles with TEMP_ADDR
TEMP_ADDR	DS	2		;Temporary address
FLASH_BYTE	EQU	*		;Doubles with ACCUMULATOR
OPCODE_LENGTH	EQU	*		;Doubles with ACCUMULATOR
ACCUMULATOR	DS	1		;Temporary A
FLASH_ADDR	DS	2
MNEMONIC_ADDR	EQU	*		;Doubles with COMPARE_ADDR
COMPARE_ADDR	DS	2
;
FLASH_COUNTER	EQU	*		;Doubles with TEMP_COUNTER
TEMP_COUNTER	DS	1

		ORG	RAM_DATA
;
BUFFER		DS	32
START_R		DS	1		;RAM resident code starts here
;***********************************************************************
	ORG	START_MONITOR	;Position to monitor code
;
;BEGIN	BSET	0,$1F		;Disable COP module
BEGIN	equ	*		;SIM08 doesn't allow COP access
	LDHX	#STACK		;Set initial stack
	TXS			;Set it up
	BSET	1,PTCPUE	;Disable PIC communications
; Enable the serial communications
	BSET	6,SCC1		;Enable the SCI
	BSET	3,SCC2		;Enable the transmitter
	BSET	2,SCC2		;Enable the receiver
; Configure Baud Rate Generator
	LDA	#$21		;PD = 4; BD = 2
	STA	SCBR		;Baud Rate = 19200
; Set defaults for registers of user application
	LDHX	#REG_A		;Set pointer to first user register
;	CLR	,X		;Zero user A
;	CLR	1,X		;Zero user H
;	CLR	2,X		;Zero user X
	LDA	#=DEFAULT_SP	;Get default of high byte SP
	STA	3,X		;Set user SP (high)
	LDA	#DEFAULT_SP	;Get default of low byte SP
	STA	4,X		;Set user SP (low)
;	LDA	#=DEFAULT_PC	;Get default of high byte PC
;	STA	5,X		;Set user PC (high)
;	LDA	#DEFAULT_PC	;Get default of low byte PC
;	STA	6,X		;Set user PC (low)
	LDA	#DEFAULT_CC	;Get default of CC
	STA	7,X		;Set user CC
; Display welcome message
	JSR	DISP_MSG	;Output message
	FCB	$0A,$0D
	FCC	'MON08 version 1.02'
	FCB	$0A,$0D
	FCC	'Copyright 2001-2005 BiPOM Electronics'
	FCB	$0A,$0D
	FCCZ	'All rights reserved.'
; Issue command prompt
PROMPT	BSR	DISP_MSG	;Output message
	FCB	$0A,$0D		;Newline
	FCCZ	'>'		;Prompt
	CLR	REG_TEMP	;Zero second char
	JSR	GET_CHR		;Get a command symbol
	JSR	PUT_CHR		;Send a command symbol to console
	LDHX	#COMMAND_TABLE	;Point to command table
; Test for command match
cmd1	CMP	,X		;Does first char match?
	BNE	cmd5		;No, try next
	PSHA			;Save first
	LDA	REG_TEMP	;Get second symbol
cmd2	CMP	1,X		;Does second symbol match?
	BNE	cmd3		;No, try next
	PULA			;Clean stack
; Command was found - execute from table entry
exec	JSR	PUT_SPACE	;Echo a space
	LDA	2,X		;Get high
	LDX	3,X		;Get low
	PSHA			;Copy A
	PULH			;to H
	JSR	,X		;Execute command
	BRA	PROMPT		;Reprompt
; See if possible second character required
cmd3	TSTA			;Second char required?
	BNE	cmd4		;No, it's OK
	BSR	GET_CHR		;Get character
	BSR	PUT_CHR		;Echo it
	STA	REG_TEMP	;Set new second symbol
	BRA	cmd2		;And test again
cmd4	PULA			;Restore first
; Skip to next command entry
cmd5	AIX	#4		;Skip to next
	TST	,X		;End of table
	BNE	cmd1		;Keep looking
; Command was not found in the table
UNK_CMD	BSR	DISP_MSG	;Output message
	FCCZ	' UNKNOWN COMMAND - USE "?" FOR HELP'
	BRA	PROMPT
;
; Utility ROUTINES
;
; Send string(PC) to console
DISP_MSG
	STHX	REG_TEMP	;Store H:X to REG_TEMP
	PULH			;Get high byte PCH
	PULX			;Get low byte PCL
disp1	LDA	,X		;Get char from string
	AIX	#1		;Advance to next
	BEQ	disp2		;End-exit
	BSR	PUT_CHR		;Write character
	BRA	disp1		;Do entire string
disp2	PSHX			;Set new PCL
	PSHH			;Set new PCH
	LDHX	REG_TEMP	;Restore H:X from REG_TEMP
	RTS
; Send new line command to console
NEW_LINE
	BSR	DISP_MSG	;Output message
	FCB	$0A,$0D,0	;LF+CR
	RTS
; Print word *H:X on console
PRINT_WORD_HEX
	LDA	#'$'
	BSR	PUT_CHR
PRINT_WORD
	LDA	,X
	BSR	PRINT_BYTE
	AIX	#1		;Increment pointer
	LDA	,X
	BRA	PRINT_BYTE
; Print byte (A) on console
PRINT_BYTE_HEX
	PSHA
	LDA	#'$'
	BSR	PUT_CHR
	PULA
PRINT_BYTE
	PSHA			;Store low nibble
	NSA			;Move high into low
	BSR	NIBBLE		;Output high nibble
	PULA			;Get low nibble
; Print nibble (A) on console
NIBBLE	AND	#%00001111	;Drop high
	ADD	#'0'		;Convert to ASCII
	CMP	#'9'		;Numeric?
	BLS	PUT_CHR		;Yes, its OK
	ADD	#7		;Convert
; Put character (A) to serial port
PUT_CHR	BRCLR	7,SCS1,*	;Transmitting in progress?
	STA	SCDR		;Send a char to console
	RTS
; Get character (A) from serial port
GET_CHR	BRCLR	5,SCS1,*	;Receiving in progress?
	LDA	SCDR		;Load a char to (A)
	CMP	#'a'		;lower-case alpha?
	BLO	GET_CHR_0	;No, its not
	CMP	#'z'		;lower-case alpha?
	BHI	GET_CHR_0	;No, its not
	AND	#%11011111	;Convert to upper case
GET_CHR_0
	RTS
; Send a space symbol to console
PUT_SPACE
	PSHA			;Save A register
	LDA	#' '		;Load space symbol
	BSR	PUT_CHR		;Send a symbol
	PULA			;Restore A register
	RTS
; Read a NIBBLE into A
R_NIBBLE
	BSR	GET_CHR		;Get character
	CBEQA	#' ',FAILURE	;Space symbol is break symbol
	BSR	PUT_CHR		;Put char to console
	SUB	#'0'		;In range?
	CMP	#10		;Test for OK
	BLO	EXIT_R_NIBBLE	;Yes, its OK
	SUB	#7		;Convert HEX's
	CMP	#$0A		;In range?
	BLO	ERROR_R_NIBBLE	;No, its invalid
	CMP	#$0F		;In range?
	BHI	ERROR_R_NIBBLE	;Its invalid
EXIT_R_NIBBLE
	CLC			;No carry, its OK
	RTS
ERROR_R_NIBBLE
	BSR	DISP_MSG	;Output message
	FCB	$0A,$0D
	FCCZ	'ERROR SYMBOL'
FAILURE	SEC			;Indicate failure
	RTS
; Read a BYTE into A
READ_BYTE
	BSR	R_NIBBLE	;Get HIGH nibble
	BCS	EXIT_READ_BYTE	;Error
	NSA			;Move into high nibble
	STA	REG_TEMP	;Save for later
	BSR	R_NIBBLE	;Get HIGH nibble
	BCS	EXIT_READ_BYTE	;Error
	ORA	REG_TEMP	;Combine
;	CLC			;Indicate success (already clear)
EXIT_READ_BYTE
	RTS
; Read Start and Finish Address
READ_START_FINISH
	BSR	READ_WORD	;Get word from console
	BCS	EXIT_READ_START_FINISH
	LDHX	FINISH_ADDR	;Get the entered word
	STHX	START_ADDR	;Store Start address
	LDA	#','		;
	BSR	PUT_CHR		;Put "," to console
; Read a WORD into HIGH_ADDR:LOW_ADDR
READ_WORD
	BSR	READ_BYTE	;Get high byte
	BCS	EXIT_READ_WORD	;Error
	STA	HIGH_BYTE	;Store high byte
	BSR	READ_BYTE	;Get high byte
	BCS	EXIT_READ_WORD	;Error
	STA	LOW_BYTE	;Store low byte
;	CLC			;Indicate success (already clear)
EXIT_READ_WORD
EXIT_READ_START_FINISH
EXIT_MOVE_MEMORY
	RTS
; Move memory
; Move from START_ADDR-TEMP_ADDR to FINISH_ADDR
MOVE_MEMORY_ROUTINE
	LDHX	START_ADDR	;Get current location for reading
NEXT_MOVE_MEMORY
	LDA	,X		;Get data byte from current location
	LDHX	FINISH_ADDR	;Get current location for writing
	STA	,X		;Write data byte to memory
	AIX	#1		;Increment pointer
	STHX	FINISH_ADDR	;Store current location for writing
	LDHX	START_ADDR	;Get current location for reading
	CPHX	TEMP_ADDR	;Finish ?
	BEQ	EXIT_MOVE_MEMORY ;Yes, go to exit
	AIX	#1		;Increment pointer
	STHX	START_ADDR	;Store current location for reading
	BRA	NEXT_MOVE_MEMORY ;Go to next location

WR_FBUFFER_ROUTINE
	LDA	BUFFER
	STA	FLASH_BYTE
	BRA	WR_FBYTE_0
WR_FBYTE_ROUTINE
	LDA	FLASH_BYTE
	STA	BUFFER
	LDA	#1
	STA	FLASH_COUNTER
WR_FBYTE_0
; Write data byte to Flash location
; FLASH_ADDR - Flash location address
; FLASH_BYTE - Data byte
	LDHX	#START_R	;Start address of RAM resident program
	STHX	START_DIST	;Load the RAM start address
	LDHX	#WR_FBYTE_START	;Start address of Flash code
	STHX	START_SOURCE	;Load the Flash start address
	LDHX	#WR_FBYTE_END	;Finish address of Flash code
	STHX	FINISH_SOURCE	:Load the Finish address
	BSR	MOVE_MEMORY_ROUTINE
	JMP	START_R
WR_FBYTE_START
	LDA	#1		; Set the PGM bit
	STA	FLCR
	LDA	FLBPR		; Read Flash Block Protect register
	LDHX	FLASH_ADDR	; Load H:X with the current address
	LDA	FLASH_BYTE	; Load byte
	STA	,X		; Write a data to Flash Address
	LDA	#4
	BSR	PageDelay	; Delay 10us
	LDA	#9		; Set PGM and HVEN bits
	STA	FLCR
	LDA	#2
	BSR	PageDelay	; Delay 5us
;---------------------
	PSHH
	PSHX
;---
	LDHX	#BUFFER
NEXT_FLASH
	LDA	,X		; Load byte
	AIX	#1
	PSHH
	PSHX
	LDHX	FLASH_ADDR
	CPHX	#(BEGIN-1)
	BHI	NEXT_FL_BODY
	BRA	NEXT_FL_RV
NEXT_FL_BODY
	CPHX	#END_OF_MONITOR_BODY
	BHI	NEXT_FL_RV
	BRA	NEXT_FLASH_INCR
;-- CHECK RESET VECTOR --
NEXT_FL_RV
	CPHX	#$FFFE
	BNE	NEXT_FL_0
	STA	REG_PC
	BRA	NEXT_FLASH_INCR
NEXT_FL_0
	CPHX	#$FFFF
	BNE	NEXT_FL_1
	STA	REG_PC+1
	BRA	NEXT_FLASH_INCR
;-- CHECK BRK MODULE VECTOR
NEXT_FL_1
	CPHX	#$FFFC
	BEQ	NEXT_FLASH_INCR
	CPHX	#$FFFD
	BEQ	NEXT_FLASH_INCR
;-----------------------------
	STA	,X		; Write a data to Flash memory
NEXT_FLASH_INCR
	AIX	#1
	STHX	FLASH_ADDR

	PULX
	PULH
	DBNZ	FLASH_COUNTER,NEXT_FLASH
;---
	PULX
	PULH
	STHX	FLASH_ADDR
;-----------------------
	LDA	#12
	BSR	PageDelay	; Delay 30us
	LDA	#8		; Clear the PGM bit
	STA	FLCR
	LDA	#2
	BSR	PageDelay	; Delay 5us
	CLRA
	STA	FLCR		; Clear the HVEN bit
	JMP	WR_FBYTE_END
; Delay 100 us
PageDly_100us
; Oguz: Everywhere else, you call PageDelay with a comment
; indicating that each count = 2.5us, yet in this case, it
; would appear that each count = 1.7 us - What gives?
	LDA	#59		; 2 Cycles
; Software timing delay loop -  Obviously depending on clock frequency
; Total time = 2(LDA#)+4(BSR)+4(RTS)+(A*4) cycles per call
; Add 1 cycle if called by JSR extended
PageDelay
	DECA			; 1 Cycle
	BNE PageDelay		; 3 cycles
	RTS			; 4 cycles
WR_FBYTE_END
	LDHX	FLASH_ADDR	; H:X = current address
	LDA	FLASH_BYTE	; A = byte
	CBEQ	X+,WR_GOOD_STATUS ; if (A != M)
	SEC			;Indicate failure
	RTS
WR_GOOD_STATUS
	CLC			;Indicate success
	RTS
; Erase Page of Flash Memory
; FLASH_ADDR - Flash location address
ERASE_PAGE_ROUTINE
	LDHX	#START_R	;Start address of RAM resident program
	STHX	START_DIST	;Load the RAM start address
	LDHX	#ER_PAGE_START	;Start address of Flash code
	STHX	START_SOURCE	;Load the Flash start address
	LDHX	#ER_PAGE_END	;Finish address of Flash code
	STHX	FINISH_SOURCE	;Load the Finish address
	JSR	MOVE_MEMORY_ROUTINE
	JMP	START_R
ER_PAGE_START
	LDA	#%00000010
	STA	FLCR		;Set ERASE bit
	LDA	FLBPR		;Read from Block Protect Register
	LDHX	FLASH_ADDR	;Get the page address
	STA	,X		;Write any data
	LDHX	#FLCR		;Set HX = FLCR
	LDA	#4
	BSR	PageDelay	;Delay 10us
	LDA	#%00001010
	STA	,X		;Set HVEN bit
	LDA	#40
	BSR	PageDly_N100us	;Delay 4ms
	LDA	#%00001000
	STA	,X		;Clr ERASE bit
	BSR	PageDly_100us	;Delay 100us
	CLRA
	STA	,X		;Clr HVEN bit
	RTS
; Delay A * 100us
PageDly_N100us
	PSHA
	BSR PageDly_100us
	PULA
	DECA
	BNE PageDly_N100us
	RTS
ER_PAGE_END
;
; COMMANDS OF MON08
;
SET_BREAK_POINT
;Set break point
	JSR	READ_WORD
	BCS	EXIT_SBP
; -- ENABLE BREAK POINT --
	LDHX	FINISH_ADDR
	PSHH
	PULA
	STA	BRKH
	STX	BRKL
	CPHX	#0			; Clearing breakpoint?
	BEQ	CLR_BRK			; Yes, clear it
	LDA	#$80
	STA	SBFCR			;Enable the clearable status bits
	STA	BRKSCR			;Enable 16-bit address match
	RTS
CLR_BRK	LDA	#$80
	STA	SBFCR		;Enable the clearable status bits
	CLRA
	STA	BRKSCR		;Disable 16-bit address match
	STA	SBFCR		;Disable the clearable status bits
EXIT_SBP
EXIT_CLEAR_FLASH
	RTS
;
CLEAR_FLASH
;Edit Flash Memory
	JSR	READ_START_FINISH ;Get dump address
	BCS	EXIT_CLEAR_FLASH ;Error
	LDHX	START_ADDR	;
	STHX	FLASH_ADDR	;FLASH_ADDR = START_ADDR
	LDA	#$80
	AND	FLASH_ADDR+1
	STA	FLASH_ADDR+1	;FLASH_ADDR &=0xFF80

	LDA	#$80
	AND	FINISH_ADDR+1
	STA	FINISH_ADDR+1	;FINISH_ADDR &=0xFF80
	LDHX	FINISH_ADDR
	STHX	COMPARE_ADDR	;COMPARE_ADDR = FINISH_ADDR
NEW_CLEAR_PAGE
	JSR	NEW_LINE	;Make a new line on console
	LDA	FLASH_ADDR	;Get high byte of address
	JSR	PRINT_BYTE	;Print high byte of address
	LDA	FLASH_ADDR+1	;Get low byte of address
	JSR	PRINT_BYTE	;Print low byte of address
;-- DISABLE A ERASING MONITOR BODY --
	LDHX	FLASH_ADDR
	CPHX	#(BEGIN-1)	;Begin of monitor body
	BHI	NEXT_CF_BODY	;Is this location a monitor body?
	BRA	NEXT_CF_RV	;Maybe it is monitor body, check next
NEXT_CF_BODY
	CPHX	#END_OF_MONITOR_BODY ; End of monitor body
	BHI	NEXT_CF_RV	;Is this location a monitor body?
	BRA	CF_BAD_LOCATION	;This location from monitor body
;-- CHECK RESET VECTOR --
NEXT_CF_RV
	CPHX	#$FF80
	BEQ	CF_BAD_LOCATION
;-- ERASE THE PAGE --
	JSR	ERASE_PAGE_ROUTINE
;-- CHECK THE RESULT
	LDHX	FLASH_ADDR
	LDA	#128
	STA	COUNTER
NEXT_CF_LOCATION
	LDA	,X
	CMP	#$FF
	BNE	CF_BAD_LOCATION
	AIX	#1
	DBNZ	COUNTER,NEXT_CF_LOCATION
;
	JSR	DISP_MSG	;Output message
	FCCZ	' - OK '
	BRA	INCR_CF
CF_BAD_LOCATION
	JSR	DISP_MSG	;Output message
	FCCZ	' - BAD'
	BRA	INCR_CF
INCR_CF
	LDHX	FLASH_ADDR	;Get current address of page
	CPHX	COMPARE_ADDR	;Finish?
	BEQ	EXIT_CLEAR_FLASH ;Yes, go to exit
	AIX	#127		;Increment location memory
	AIX	#1
	STHX	FLASH_ADDR	;Store current location
	BRA	NEW_CLEAR_PAGE	;Print new line
;
CHANGE_REGISTER
;Change CPU register
	JSR	GET_CHR		;Get a symbol form console
	JSR	PUT_CHR		;Put a symbol to console
	LDHX	#REG_A		;Point at register-A
	CBEQA	#'A',GET_BYTE_VALUE
	AIX	#1		;Advance to H
	CBEQA	#'H',GET_BYTE_VALUE
	AIX	#1		;Advance to X
	CBEQA	#'X',GET_BYTE_VALUE
	AIX	#1		;Advance to SP
	CBEQA	#'S',GET_SP
	AIX	#2		;Advance to PC
	CBEQA	#'P',GET_PC
	AIX	#2		;Advance to CC
	CBEQA	#'C',GET_CC
	JSR	DISP_MSG	;Output message
	FCB	$0A,$0D
	FCCZ	'UNKNOWN REGISTER'
	RTS
GET_SP	LDA	#'P'
	BRA	get_wv1
GET_PC	LDA	#'C'
GET_WV1	JSR	PUT_CHR		;Echo second character of register
GET_WORD_VALUE
	JSR	PUT_SPACE	;Put a space symbol
	JSR	READ_WORD	;Get word from console
	BCS	EXIT_CHANGE_REGISTER
	LDA	HIGH_BYTE	;Get high byte of word
	STA	,X		;Save high byte of CPU register
	LDA	LOW_BYTE	;Get high byte of word
	STA	1,X		;Save low byte of CPU register
	BRA	EXIT_CHANGE_REGISTER
GET_CC	LDA	#'C'
	JSR	PUT_CHR
GET_BYTE_VALUE
	JSR	PUT_SPACE	;Put a space symbol
	JSR	READ_BYTE	;Get byte from console
	BCS	EXIT_CHANGE_REGISTER
	STA	,X		;Change CPU register to new value
EXIT_CHANGE_REGISTER
	JMP	DISPLAY_REG	;Display all CPU registers
;
DISPLAY_BREAKPOINTS
;Display breakpoints
	JSR	DISP_MSG	;Output message
	FCCZ	'BP='
	LDHX	#BRKH		;Point to breakpoint register
	LDA	,X		;Get high
	ORA	1,X		;Test with low
	BEQ	nobp		;Not set
	JMP	PRINT_WORD	;Display value
nobp	JSR	DISP_MSG	;Output message
	FCCZ	'XXXX'
EXIT_DISPLAY_ASM
EXIT_DISPLAY_HEX
EXIT_EDIT
	RTS
;
DISPLAY_ASM
;Display memory in asm format
	JSR	READ_START_FINISH	;Get dump address
	BCS	EXIT_DISPLAY_ASM	;Error
NEXT_DISPLAY_ASM
	LDHX	START_ADDR
	CPHX	FINISH_ADDR		; Finish?
	BHI	EXIT_DISPLAY_ASM	;Yes, go to exit
	JSR	NEW_LINE
	JSR	DISPLAY_MNEMONIC_LINE
	BRA	NEXT_DISPLAY_ASM
;
DISPLAY_HEX
;Display memory in hex format
	JSR	READ_START_FINISH ;Get dump address
	BCS	EXIT_DISPLAY_HEX  ;Error
NEW_LINE_DISPLAY_HEX
	JSR	NEW_LINE	;Make a new line on console
	LDA	START_ADDR	;Get high byte of address
	PSHA			;Store to stack
	JSR	PRINT_BYTE	;Print high byte of address
	LDA	START_ADDR+1	;Get low byte of address
	PSHA			;Store to stack
	JSR	PRINT_BYTE	;Print low byte of address
	JSR	PUT_SPACE	;Print space
	PULX			;Get low byte of address
	PULH			;Get high byte of address
	LDA	#16		;
	STA	COUNTER		;Load counter
NEXT_DISPLAY_HEX
	LDA	,X		;Get data byte from memory
	JSR	PRINT_BYTE	;Print data byte
	JSR	PUT_SPACE	;Print space
	CPHX	FINISH_ADDR	;Finish?
	BEQ	EXIT_DISPLAY_HEX ;Yes, go to exit
	AIX	#1		;Increment location memory
	DBNZ	COUNTER,NEXT_DISPLAY_HEX ; New line? No, go to next byte
	STHX	START_ADDR	;Store current location
	BRA	NEW_LINE_DISPLAY_HEX ;Print new line
;
EDIT_MEMORY
	JSR	READ_WORD	;Get memory address
	BCS	EXIT_EDIT	;Error, go to exit
NEW_EDIT_LINE
	JSR	NEW_LINE
	LDA	FINISH_ADDR	;Get high byte of address
	PSHA			;Store to stack
	JSR	PRINT_BYTE	;Print high byte of address
	LDA	FINISH_ADDR+1	;Get low byte of address
	PSHA			;Store to stack
	JSR	PRINT_BYTE	;Print low byte of address
	JSR	PUT_SPACE	;Print space
	PULX			;Get low byte of address
	PULH			;Get high byte of address
	LDA	,X		;Get data byte from memory
	JSR	PRINT_BYTE	;Print data byte
	JSR	PUT_SPACE	;Print space
	JSR	READ_BYTE	;Get new value
	BCS	EXIT_EDIT	;Error, go to exit
	STA	,X		;Write new value to current location
	CBEQ	X+,EDIT_GOOD_STATUS ; if (A = M) Good
	JSR	DISP_MSG	;Output message
	FCB	$0A,$0D
	FCCZ	'BAD RAM LOCATION'
EDIT_GOOD_STATUS
	STHX	FINISH_ADDR	;Store current location
	BRA	NEW_EDIT_LINE	;Print new line
;
DISPLAY_REG
;Display CPU registers
	LDHX	#REG_A		;Pointer to user registers
	JSR	DISP_MSG	;Output message
	FCCZ	' A='
	BSR	NEXT_REGISTER	;Print A register
	JSR	DISP_MSG	;Output message
	FCCZ	' H:X='
	BSR	NEXT_REGISTER	;Print H register
	BSR	NEXT_REGISTER	;Print X register
	JSR	DISP_MSG	;Output message
	FCCZ	' SP='
	BSR	NEXT_REGISTER	;Print SPH
	BSR	NEXT_REGISTER	;Print SPL
	JSR	DISP_MSG	;Output message
	FCCZ	' PC='
	BSR	NEXT_REGISTER	;Print PCH
	BSR	NEXT_REGISTER	;Print PCL
	JSR	DISP_MSG	;Output message
	FCCZ	' CC='
NEXT_REGISTER
	LDA	,X		;Get user register
	AIX	#1		;Pointer to next register
	JMP	PRINT_BYTE	;Print CC register
;
FILL_MEMORY
;Fill memory
	JSR	READ_START_FINISH ;Get dump address
	BCS	EXIT_FILL_MEMORY
	JSR	PUT_SPACE	;Put space to console
	JSR	READ_BYTE	;Get data byte
	BCS	EXIT_FILL_MEMORY
	LDHX	START_ADDR	;Get Start address
NEXT_FILL_MEMORY
	STA	,X		;Write data byte to memory
	PSHX			;Store X into stack
	PSHH			;Store H into stack
	CBEQ	X+,FILL_GOOD	; if (A = M) Good
	JSR	DISP_MSG	;Output message
	FCB	$0A,$0D
	FCCZ	'FIRST BAD LOCATION IS '
	PULA			;Get high byte
	JSR	PRINT_BYTE	;Print high byte of bad location
	PULA			;Get low byte
	JSR	PRINT_BYTE	;Print low byte of bad location
EXIT_EDIT_FLASH
EXIT_FILL_MEMORY
	RTS
FILL_GOOD
	PULH			;Restore H
	PULX			;Restore X
	CPHX	FINISH_ADDR	;Finish ?
	BEQ	EXIT_FILL_MEMORY ;Yes, go to exit
	AIX	#1		;Increment pointer
	BRA	NEXT_FILL_MEMORY ;Go to next location
;
EDIT_FLASH
;Edit Flash Memory
	JSR	READ_WORD	;Get memory address
	BCS	EXIT_EDIT_FLASH	;Error, go to exit
	LDHX	FINISH_ADDR
	STHX	FLASH_ADDR
NEW_EDIT_FLASH_LINE
	JSR	NEW_LINE
	LDA	FLASH_ADDR	;Get high byte of address
	PSHA			;Store to stack
	JSR	PRINT_BYTE	;Print high byte of address
	LDA	FLASH_ADDR+1	;Get low byte of address
	PSHA			;Store to stack
	JSR	PRINT_BYTE	;Print low byte of address
	JSR	PUT_SPACE	;Print space
	PULX			;Get low byte of address
	PULH			;Get high byte of address
	LDA	,X		;Get data byte from memory
	JSR	PRINT_BYTE	;Print data byte
	JSR	PUT_SPACE	;Print space
	JSR	READ_BYTE	;Get new value
	BCS	EXIT_EDIT_FLASH	;Error, go to exit
	STA	FLASH_BYTE	;Write new value to current location
	JSR	WR_FBYTE_ROUTINE ;Write data byte to Flash memory
	BCC	GOOD_F_WRITING	;Data byte is written
	JSR	DISP_MSG	;Output message
	FCB	$0A,$0D
	FCCZ	'BAD FLASH LOCATION'
GOOD_F_WRITING
	LDHX	FLASH_ADDR	;Restore pointer
	AIX	#1		;Increment location memory
	STHX	FLASH_ADDR	;Store current location
	BRA	NEW_EDIT_FLASH_LINE ;Print new line
;
FILL_FLASH
;Fill memory
	JSR	READ_START_FINISH ;Get dump address
	BCS	EXIT_FILL_FLASH
	JSR	PUT_SPACE	;Put space to console
	JSR	READ_BYTE	;Get data byte
	BCS	EXIT_FILL_FLASH
	STA	FLASH_BYTE	;This byte will be written
	LDHX	START_ADDR	;
	STHX	FLASH_ADDR	;FLASH_ADDR = START_ADDR
	LDHX	FINISH_ADDR
	STHX	COMPARE_ADDR	;COMPARE_ADDR = FINISH_ADDR
NEXT_FILL_FLASH
	JSR	WR_FBYTE_ROUTINE ;Write data byte to Flash memory
	BCC	GOOD_FILL_FLASH
	JSR	DISP_MSG	;Output message
	FCB	$0A,$0D
	FCCZ	'FIRST BAD LOCATION IS '
	LDA	FLASH_ADDR	;Print address of bad location
	JSR	PRINT_BYTE
	LDA	FLASH_ADDR+1
	JSR	PRINT_BYTE
	BRA	EXIT_FILL_FLASH	;Error, goto exit
GOOD_FILL_FLASH
	LDHX	FLASH_ADDR	;Restore current location
	CPHX	COMPARE_ADDR	;Finish ?
	BEQ	EXIT_FILL_FLASH	;Yes, go to exit
	AIX	#1		;Increment pointer
	STHX	FLASH_ADDR	;Store new current location
	BRA	NEXT_FILL_FLASH	;Go to next location
;
EXECUTE_PROGRAM
;Execute user application
	JSR	READ_WORD	;Get user address
	BCS	EXIT_EXECUTE_PROGRAM
	LDHX	FINISH_ADDR
	STHX	REG_PC
EXECUTE_PROGRAM_0
	LDHX	REG_SP		;Get user stack
	TXS			;Set user stack
	LDHX	REG_PC		;Get start address
	PSHX			;PCL to stack
	PSHH			;PCH to stack
	LDHX	REG_H		;Set user H:X
	LDA	REG_CC		;Set user CC
	TPA
	LDA	REG_A		;Set user A
EXIT_FILL_FLASH
EXIT_EXECUTE_PROGRAM
EXIT_HELP
	RTS			;Start user application
;
MOVE_MEMORY
;Move memory
	JSR	READ_START_FINISH ;Get dump address
	BCS	EXIT_MOVE_MEM	;Error, go to exit
	LDHX	FINISH_ADDR	;Store finish address
	STHX	TEMP_ADDR	;to TEMP_ADDR
	JSR	PUT_SPACE	;Put space to console
	JSR	READ_WORD	;Get third address
	BCS	EXIT_MOVE_MEM	;Error, go to exit
	JMP	MOVE_MEMORY_ROUTINE ;Move memory
;
; Display help text
;
HELP	LDHX	#HELP_TEXT-1	;Point to help text
help1	JSR	NEW_LINE	;New output line
	CLR	COUNTER		;Zero counter
help2	AIX	#1		;Skip to next character
	LDA	,X		;Get char from string
	BEQ	help1		;New line
	BMI	EXIT_HELP	;End of help
	CBEQA	#'~',help3	;Special separator
	JSR	PUT_CHR		;Output
	INC	COUNTER		;Advance the counter
	BRA	help2		;No, keep going
help3	JSR	PUT_SPACE	;Space over
	INC	COUNTER		;Advance counter
	LDA	COUNTER		;Get counter
	CMP	#25		;Over limit?
	BLO	help3		;Continue
	BRA	help2		;And continue
;
GET_BYTE_PLUS_CHECK_SUM
	JSR	READ_BYTE
	BCS	GBPCS_EXIT
	PSHA
	ADD	ACCUMULATOR
	STA	ACCUMULATOR
	PULA
	CLC			;No carry, its OK
GBPCS_EXIT
EXIT_MOVE_MEM
	RTS
LOAD_MEMORY
;Load hex file into memory
	CLR	ACCUMULATOR
	LDHX	#BUFFER
	BSR	GET_BYTE_PLUS_CHECK_SUM
	BCS	LOAD_MEMORY_BAD_CS
	STA	COUNTER			; numBytes
	STA	FLASH_COUNTER
	BSR	GET_BYTE_PLUS_CHECK_SUM
	BCS	LOAD_MEMORY_BAD_CS
	STA	FLASH_ADDR		; High Address
	BSR	GET_BYTE_PLUS_CHECK_SUM
	BCS	LOAD_MEMORY_BAD_CS
	STA	FLASH_ADDR+1		; Low Address
LOAD_MEMORY_NEXT
	BSR	GET_BYTE_PLUS_CHECK_SUM	; Data Byte
	BCS	LOAD_MEMORY_BAD_CS
	STA	,X
	AIX	#1
	DBNZ	COUNTER,LOAD_MEMORY_NEXT
	LDA	FLASH_COUNTER
	STA	COUNTER
	BSR	GET_BYTE_PLUS_CHECK_SUM
	BCS	LOAD_MEMORY_BAD_CS
	TST	ACCUMULATOR
	BNE	LOAD_MEMORY_BAD_CS
	JSR	WR_FBUFFER_ROUTINE
;Check a result of writing
	LDHX	#BUFFER		; H:X -> BUFFER
LOAD_MEMORY_0
	LDA	,X		; Get the next byte from BUFFER
	PSHH			; Store HighPointer to BUFFER
	PSHX			; Store LowPointer  to BUFFER

	LDHX	FLASH_ADDR	; H:X = current address
	CPHX	#$FFFE
	BEQ	LOAD_MEMORY_VECTOR
	CPHX	#$FFFF
	BEQ	LOAD_MEMORY_VECTOR
	CPHX	#$FFFC
	BEQ	LOAD_MEMORY_VECTOR
	CPHX	#$FFFD
	BEQ	LOAD_MEMORY_VECTOR
	CBEQ	X+,LOAD_MEMORY_1
	PULX
	PULH
	BRA	LOAD_MEMORY_BAD_WR
LOAD_MEMORY_1
	STHX	FLASH_ADDR	; CurrentAddress++
	PULX			; Restore LowPointer  to DataBuffer
	PULH			; Restore HighPointer to Databuffer
	AIX	#1		; Incrememt Pointer
	DBNZ	COUNTER,LOAD_MEMORY_0 ; Is it finish ? No, go to next byte
	LDA	#'.'
LM_PUT_CHR
	JMP	PUT_CHR
LOAD_MEMORY_BAD_CS
	LDA	#'X'
	BRA	LM_PUT_CHR
LOAD_MEMORY_BAD_WR
	LDA	#'W'
	BRA	LM_PUT_CHR
LOAD_MEMORY_VECTOR
	AIX	#1
	BRA	LOAD_MEMORY_1
;
SINGLE_STEP
;Single step execution
	JSR	NEW_LINE
	LDHX	REG_PC
	STHX	START_ADDR
	JSR	DISPLAY_MNEMONIC_LINE	;Display Assembler mnemonic
	LDHX	REG_PC
;Oguz: Shouldn't this advance by the size of the instruction?
	AIX	#1			;Increase to prevent a stop
	PSHH				;Get H
	PULA				;A=H
	STA	BRKH			;Write high
	STX	BRKL			;Write low
;
	LDA	#$80
	STA	SBFCR			;Enable the clearable status bits
	STA	BRKSCR			;Enable 16-bit address match
	JMP	EXECUTE_PROGRAM_0
;
;	TABLE OF COMMANDS
;
COMMAND_TABLE
	STRZ	'L'
	FDB	LOAD_MEMORY		;Load hex file into memory
	STRZ	'S'
	FDB	SINGLE_STEP		;Single step execution
	STRZ	'G'
	FDB	EXECUTE_PROGRAM		;Execute user application
	STRZ	'?'
	FDB	HELP			;Help
	STR	'BS'
	FDB	SET_BREAK_POINT		;Set break point
	STR	'CF'
	FDB	CLEAR_FLASH		;Clear Flash Memory
	STR	'CR'
	FDB	CHANGE_REGISTER		;Change CPU register
	STR	'DB'
	FDB	DISPLAY_BREAKPOINTS	;Display breakpoints
	STR	'DI'
	FDB	DISPLAY_ASM		;Display memory in asm format
	STR	'DM'
	FDB	DISPLAY_HEX		;Display memory in hex format
	STR	'DR'
	FDB	DISPLAY_REG		;Display CPU registers
	STR	'EM'
	FDB	EDIT_MEMORY		;Edit RAM
	STR	'EF'
	FDB	EDIT_FLASH		;Edit Flash Memory
	STR	'FM'
	FDB	FILL_MEMORY		;Fill memory
	STR	'FF'
	FDB	FILL_FLASH		;Fill Flash memory
	STR	'MM'
	FDB	MOVE_MEMORY		;Move memory
	FDB	$0			;End of COMMAND TABLE


DISPLAY_MNEMONIC_LINE
	LDHX	#START_ADDR		;Get address
	JSR	PRINT_WORD		;Print address
	JSR	PUT_SPACE
	JSR	GET_OPCODE_INFORMATION	;Get information from table
	AND	#$0F
	STA	COUNTER			;Store number of size
	LDA	#16
	STA	TEMP_COUNTER
	LDHX	START_ADDR
	STHX	TEMP_ADDR		;Store address for next using
NEXT_OPCODE_DML
	LDA	,X
	JSR	PRINT_BYTE		;Print opcodes
	DEC	TEMP_COUNTER
	DEC	TEMP_COUNTER
	JSR	PUT_SPACE
	DEC	TEMP_COUNTER
	AIX	#1
	STHX	START_ADDR
	DBNZ	COUNTER,NEXT_OPCODE_DML
SPACE_DML
	JSR	PUT_SPACE
	DBNZ	TEMP_COUNTER,SPACE_DML
;
	LDHX	TEMP_ADDR		;Get current opcode
	LDA	,X
; BRSET X,BRCLR X (X =0..7) ?
	CMP	#$0F
	BHI	NO_BRANCH_BIT_DML
	ASRA
	PSHA
	BCS	BRCLR_DETECTED_DML
	JSR	DISP_MSG
	FCCZ	'BRSET '
	BRA	BR_BIT_DML
BRCLR_DETECTED_DML
	JSR	DISP_MSG
	FCCZ	'BRCLR '
BR_BIT_DML
	PULA
	AND	#$07
	ADD	#'0'
	JSR	PUT_CHR
	LDA	#','
	JSR	PUT_CHR
	BRA	OPERANDS_DML
NO_BRANCH_BIT_DML
; BSET X,BCLR X (X =0..7) ?
	CMP	#$1F
	BHI	NO_BIT_DML
	ASRA
	PSHA
	BCS	BCLR_DETECTED_DML
	JSR	DISP_MSG
	FCCZ	'BSET '
	BRA	BIT_DML
BCLR_DETECTED_DML
	JSR	DISP_MSG
	FCCZ	'BCLR '
BIT_DML
	BRA	BR_BIT_DML
NO_BIT_DML
;
; OTHER OPCODES
;
OTHER

	LDA	#5
	STA	COUNTER
	LDHX	MNEMONIC_ADDR

PRINT_MNEMONIC_DML
	LDA	,X
	JSR	PUT_CHR
	AIX	#1
	DBNZ	COUNTER,PRINT_MNEMONIC_DML
	JSR	PUT_SPACE
;
; OPERANDS
;
OPERANDS_DML
	LDHX	TEMP_ADDR
	AIX	#1
	LDA	ACCUMULATOR
	PSHA
	AND	#$0F
	DECA
	STA	COUNTER		;Number of opcode bytes
	PULA
	AND	#$F0
	CBEQA	#DIR,DIR_ADDRESSING
	CBEQA	#REL,L_REL_ADDRESSING
	CBEQA	#INH,L_INH_ADDRESSING
	CBEQA	#IMM,L_IMM_ADDRESSING
	CBEQA	#EXT,L_EXT_ADDRESSING
	CBEQA	#IX,L_IX_ADDRESSING
	CBEQA	#IX1,L_IX1_ADDRESSING
	CBEQA	#IX2,L_IX2_ADDRESSING
	CBEQA	#SP1,L_SP1_ADDRESSING
	CBEQA	#SP2,L_SP2_ADDRESSING
	CBEQA	#IMD,L_IMD_ADDRESSING
	CBEQA	#DD,L_DD_ADDRESSING
	CBEQA	#IX_PLUS_D,L_IX_PLUS_D_ADDRESSING
	CBEQA	#DIX_PLUS,L_DIX_PLUS_ADDRESSING
	CBEQA	#IX_PLUS,L_IX_PLUS_ADDRESSING
	CBEQA	#IX1_PLUS,L_IX1_PLUS_ADDRESSING
	RTS
L_REL_ADDRESSING	JMP	REL_ADDRESSING
L_INH_ADDRESSING	JMP	INH_ADDRESSING
L_IMM_ADDRESSING	JMP	IMM_ADDRESSING
;L_EXT_ADDRESSING	JMP	EXT_ADDRESSING
L_EXT_ADDRESSING	JMP	PRINT_WORD_HEX
L_IX_ADDRESSING		JMP	IX_ADDRESSING
L_IX1_ADDRESSING	JMP	IX1_ADDRESSING
L_IX2_ADDRESSING	JMP	IX2_ADDRESSING
L_SP1_ADDRESSING	JMP	SP1_ADDRESSING
L_SP2_ADDRESSING	JMP	SP2_ADDRESSING
L_IMD_ADDRESSING	JMP	IMD_ADDRESSING
L_DD_ADDRESSING		JMP	DD_ADDRESSING
L_IX_PLUS_D_ADDRESSING	JMP	IX_PLUS_D_ADDRESSING
L_DIX_PLUS_ADDRESSING	JMP	DIX_PLUS_ADDRESSING
L_IX_PLUS_ADDRESSING	JMP	IX_PLUS_ADDRESSING
L_IX1_PLUS_ADDRESSING	JMP	IX1_PLUS_ADDRESSING
SP1_ADDRESSING
	AIX	#1
	DEC	COUNTER
	LDA	,X
	JSR	PRINT_BYTE_HEX
SP1_ADDR_0
	JSR	DISP_MSG
	FCCZ	',SP'
	AIX	#1
	DBNZ	COUNTER,DIR_REL
	RTS
SP2_ADDRESSING
	AIX	#1
	JSR	PRINT_WORD_HEX
	DEC	COUNTER
	DEC	COUNTER
	BRA	SP1_ADDR_0
DIR_ADDRESSING
	LDA	,X
	AIX	#1
	JSR	PRINT_BYTE_HEX
	DBNZ	COUNTER,DIR_REL
	RTS
DIR_REL
	LDA	#','
	JSR	PUT_CHR
REL_ADDRESSING
	CLRA
	TST	,X
	BPL	REL_NEGATIVE
	COMA
REL_NEGATIVE
	PSHA
	LDA	,X	;GET REL
	LDHX	#TEMP_ADDR+1
	ADD	,X
	STA	,X
	AIX	#-1
	PULA
	ADC	,X
	STA	,X
;
	AIX	#1
	LDA	ACCUMULATOR
	AND	#$0F
	ADD	,X
	STA	,X
	AIX	#-1
	CLRA
	ADC	,X
	STA	,X
	LDHX	#TEMP_ADDR
	JMP	PRINT_WORD_HEX
INH_ADDRESSING
	DBNZ	COUNTER,INH_ADDR_0
	BRA	REL_ADDRESSING
INH_ADDR_0
	AIX	#-1
	LDA	,X
	LDHX	MNEMONIC_ADDR
	CPHX	#A_UNKNOWN
	BNE	L_EXIT_DML
	JMP	PRINT_BYTE_HEX
IMM_ADDRESSING
	JSR	DISP_MSG
	FCCZ	'#$'
	AIX	#-1
	LDA	,X
	AIX	#1
	CBEQA	#$41,IMM_ADDR_1	; CBEQA opcode?
	CBEQA	#$51,IMM_ADDR_1	; CBEQX opcode?
IMM_ADDR_0
	LDA	,X
	AIX	#1
	JSR	PRINT_BYTE
	DBNZ	COUNTER,IMM_ADDR_0
L_EXIT_DML
	RTS
IMM_ADDR_1
	LDA	,X
	AIX	#1
	JSR	PRINT_BYTE
L2_DIR_REL
	JMP	DIR_REL
;EXT_ADDRESSING
;	JMP	PRINT_WORD_HEX
IX_ADDRESSING
	DBNZ	COUNTER,IX_ADDR
	LDA	#'X'
	JSR	PUT_CHR
	BRA	DIR_REL
IX_ADDR
	JSR	DISP_MSG
	FCCZ	',X'
	RTS
IX1_ADDRESSING
	LDA	,X
	AIX	#1
	JSR	PRINT_BYTE_HEX
IX1_ADDR_0
	JSR	DISP_MSG
	FCCZ	',X'
	DBNZ	COUNTER,L2_DIR_REL
	RTS
IX2_ADDRESSING
	JSR	PRINT_WORD_HEX
	DEC	COUNTER
	BRA	IX1_ADDR_0
IMD_ADDRESSING
	LDA	#'#'
	JSR	PUT_CHR
	LDA	,X
	AIX	#1
	JSR	PRINT_BYTE_HEX
IMD_ADDR_0
	LDA	#','
	JSR	PUT_CHR
	LDA	,X
	JMP	PRINT_BYTE_HEX
DD_ADDRESSING
	LDA	,X
	AIX	#1
	JSR	PRINT_BYTE_HEX
	LDA	#','
	JSR	PUT_CHR
	LDA	,X
	JMP	PRINT_BYTE_HEX
IX_PLUS_D_ADDRESSING
	JSR	DISP_MSG
	FCCZ	'X+'
	BRA	IMD_ADDR_0
DIX_PLUS_ADDRESSING
	LDA	,X
	JSR	PRINT_BYTE_HEX
	JSR	DISP_MSG
	FCCZ	',X+'
	RTS
IX_PLUS_ADDRESSING
	JSR	DISP_MSG
	FCCZ	'X+'
	JMP	DIR_REL
IX1_PLUS_ADDRESSING
	LDA	,X
	AIX	#1
	JSR	PRINT_BYTE_HEX
	LDA	#','
	JSR	PUT_CHR
	BRA	IX_PLUS_ADDRESSING
;
GET_OFFSET
	TAX	; X = A
	CLRA	; A = 0
	LSLX	; CARRY = X.7, X.0 =0; X<<=1
	ROLA	; A<<=1, A.0 = CARRY
	LSLX	; CARRY = X.7, X.0 =0; X<<=1
	ROLA	; A<<=1, A.0 = CARRY
	PSHA	; H =
	PULH	; A
	RTS
;
GET_OPCODE_INFORMATION
	LDHX	START_ADDR
	LDA	,X			;Get current opcode
GENERATE_FCB
	CMP	#$0F			;Is it BRSET,BRCLR ?
	BHI	NO_BRANCH_BIT_GOI
	ASRA
	LDHX	#A_BRCLR
	BCS	BRCLR_DETECTED_GOI
	LDHX	#A_BRSET
BRCLR_DETECTED_GOI
	STHX	MNEMONIC_ADDR
	LDA	#(3+DIR)
	BRA	GET_OL_0
NO_BRANCH_BIT_GOI
	CMP	#$1F			;Is it BSET,BCLR?
	BHI	NO_BIT_GOI
	ASRA
	LDHX	#A_BCLR
	BCS	BCLR_DETECTED_GOI
	LDHX	#A_BSET
BCLR_DETECTED_GOI
	STHX	MNEMONIC_ADDR
	LDA	#(2+DIR)
	BRA	GET_OL_0
NO_BIT_GOI
	CBEQA	#$9E,PREFIX_GOI	;Is it 9e prefix?
	BSR	GET_OFFSET
	LDA	OPCODE_MAP-$80,X
	STA	MNEMONIC_ADDR
	AIX	#1
	LDA	OPCODE_MAP-$80,X
	STA	MNEMONIC_ADDR+1
	AIX	#1
	LDA	OPCODE_MAP-$80,X
	BRA	GET_OL_0
PREFIX_GOI
	AIX	#1
	LDA	,X			;Get opcode
	PSHA
	AND	#$0F
	BSR	GET_OFFSET
	PULA
	AND	#$F0
	CBEQA	#$60,GET_OL_60
	CBEQA	#$D0,GET_OL_D0
	CBEQA	#$E0,GET_OL_E0
	LDA	#$82			;ERROR OPCODE
	BRA	GENERATE_FCB
GET_OL_60
	LDA	PREFIX_9E6,X
	STA	MNEMONIC_ADDR
	AIX	#1
	LDA	PREFIX_9E6,X
	STA	MNEMONIC_ADDR+1
	AIX	#1
	LDA	PREFIX_9E6,X
	BRA	GET_OL_0
GET_OL_D0
	LDA	PREFIX_9ED,X
	STA	MNEMONIC_ADDR
	AIX	#1
	LDA	PREFIX_9ED,X
	STA	MNEMONIC_ADDR+1
	AIX	#1
	LDA	PREFIX_9ED,X
	BRA	GET_OL_0
GET_OL_E0
	LDA	PREFIX_9EE,X
	STA	MNEMONIC_ADDR
	AIX	#1
	LDA	PREFIX_9EE,X
	STA	MNEMONIC_ADDR+1
	AIX	#1
	LDA	PREFIX_9EE,X
GET_OL_0
	STA	ACCUMULATOR
	RTS
;
;
;	OPCODE MAP
;
OPCODE_MAP
;
;20
	FDB	A_BRA
	FCB	2+REL,3
;21
	FDB	A_BRN
	FCB	2+REL,3
;22
	FDB	A_BHI
	FCB	2+REL,3
;23
	FDB	A_BLS
	FCB	2+REL,3
;24
	FDB	A_BCC
	FCB	2+REL,3
;25
	FDB	A_BCS
	FCB	2+REL,3
;26
	FDB	A_BNE
	FCB	2+REL,3
;27
	FDB	A_BEQ
	FCB	2+REL,3
;28
	FDB	A_BHCC
	FCB	2+REL,3
;29
	FDB	A_BHCS
	FCB	2+REL,3
;2A
	FDB	A_BPL
	FCB	2+REL,3
;2B
	FDB	A_BMI
	FCB	2+REL,3
;2C
	FDB	A_BMC
	FCB	2+REL,3
;2D
	FDB	A_BMS
	FCB	2+REL,3
;2E
	FDB	A_BIL
	FCB	2+REL,3
;2F
	FDB	A_BIH
	FCB	2+REL,3
;30
	FDB	A_NEG
	FCB	2+DIR,4
;31
	FDB	A_CBEQ
	FCB	3+DIR,5
;32
	FDB	A_UNKNOWN
	FCB	1,0
;33
	FDB	A_COM
	FCB	2+DIR,4
;34
	FDB	A_LSR
	FCB	2+DIR,4
;35
	FDB	A_STHX
	FCB	2+DIR,4
;36
	FDB	A_ROR
	FCB	2+DIR,4
;37
	FDB	A_ASR
	FCB	2+DIR,4
;38
	FDB	A_LSL
	FCB	2+DIR,4
;39
	FDB	A_ROL
	FCB	2+DIR,4
;3A
	FDB	A_DEC
	FCB	2+DIR,4
;3B
	FDB	A_DBNZ
	FCB	3+DIR,5
;3C
	FDB	A_INC
	FCB	2+DIR,4
;3D
	FDB	A_TST
	FCB	2+DIR,3
;3E
	FDB	A_UNKNOWN
	FCB	1,0
;3F
	FDB	A_CLR
	FCB	2+DIR,3
;40
	FDB	A_NEGA
	FCB	1+INH,1
;41
	FDB	A_CBEQA
	FCB	3+IMM,4
;42
	FDB	A_MUL
	FCB	1+INH,5
;43
	FDB	A_COMA
	FCB	1+INH,1
;44
	FDB	A_LSRA
	FCB	1+INH,1
;45
	FDB	A_LDHX
	FCB	3+IMM,3
;46
	FDB	A_RORA
	FCB	1+INH,1
;47
	FDB	A_ASRA
	FCB	1+INH,1
;48
	FDB	A_LSLA
	FCB	1+INH,1
;49
	FDB	A_ROLA
	FCB	1+INH,1
;4A
	FDB	A_DECA
	FCB	1+INH,1
;4B
	FDB	A_DBNZA
	FCB	2+INH,3
;4C
	FDB	A_INCA
	FCB	1+INH,1
;4D
	FDB	A_TSTA
	FCB	1+INH,1
;4E
	FDB	A_MOV
	FCB	3+DD,5
;4F
	FDB	A_CLRA
	FCB	1+INH,1
;50
	FDB	A_NEGX
	FCB	1+INH,1
;51
	FDB	A_CBEQX
	FCB	3+IMM,4
;52
	FDB	A_DIV
	FCB	1+INH,7
;53
	FDB	A_COMX
	FCB	1+INH,1
;54
	FDB	A_LSRX
	FCB	1+INH,1
;55
	FDB	A_LDHX
	FCB	2+DIR,4
;56
	FDB	A_RORX
	FCB	1+INH,1
;57
	FDB	A_ASRX
	FCB	1+INH,1
;58
	FDB	A_LSLX
	FCB	1+INH,1
;59
	FDB	A_ROLX
	FCB	1+INH,1
;5A
	FDB	A_DECX
	FCB	1+INH,1
;5B
	FDB	A_DBNZX
	FCB	2+INH,3
;5C
	FDB	A_INCX
	FCB	1+INH,1
;5D
	FDB	A_TSTX
	FCB	1+INH,1
;5E
	FDB	A_MOV
	FCB	2+DIX_PLUS,4
;5F
	FDB	A_CLRX
	FCB	1+INH,1
;60
	FDB	A_NEG
	FCB	2+IX1,4
;61
	FDB	A_CBEQ
	FCB	3+IX1_PLUS,5
;62
	FDB	A_NSA
	FCB	1+INH,3
;63
	FDB	A_COM
	FCB	2+IX1,4
;64
	FDB	A_LSR
	FCB	2+IX1,4
;65
	FDB	A_CPHX
	FCB	3+IMM,3
;66
	FDB	A_ROR
	FCB	2+IX1,4
;67
	FDB	A_ASR
	FCB	2+IX1,4
;68
	FDB	A_LSL
	FCB	2+IX1,4
;69
	FDB	A_ROL
	FCB	2+IX1,4
;6A
	FDB	A_DEC
	FCB	2+IX1,4
;6B
	FDB	A_DBNZ
	FCB	3+IX1,5
;6C
	FDB	A_INC
	FCB	2+IX1,4
;6D
	FDB	A_TST
	FCB	2+IX1,3
;6E
	FDB	A_MOV
	FCB	3+IMD,4
;6F
	FDB	A_CLR
	FCB	2+IX1,3
;70
	FDB	A_NEG
	FCB	1+IX,3
;71
	FDB	A_CBEQ
	FCB	2+IX_PLUS,4
;72
	FDB	A_DAA
	FCB	1+INH,2
;73
	FDB	A_COM
	FCB	1+IX,3
;74
	FDB	A_LSR
	FCB	1+IX,3
;75
	FDB	A_CPHX
	FCB	2+DIR,4
;76
	FDB	A_ROR
	FCB	1+IX,3
;77
	FDB	A_ASR
	FCB	1+IX,3
;78
	FDB	A_LSL
	FCB	1+IX,3
;79
	FDB	A_ROL
	FCB	1+IX,3
;7A
	FDB	A_DEC
	FCB	1+IX,3
;7B
	FDB	A_DBNZ
	FCB	2+IX,4
;7C
	FDB	A_INC
	FCB	1+IX,3
;7D
	FDB	A_TST
	FCB	1+IX,2
;7E
	FDB	A_MOV
	FCB	2+IX_PLUS_D,4
;7F
	FDB	A_CLR
	FCB	1+IX,2
;80
	FDB	A_RTI
	FCB	1+INH,7
;81
	FDB	A_RTS
	FCB	1+INH,4
;82
	FDB	A_UNKNOWN
	FCB	1,0
;83
	FDB	A_SWI
	FCB	1+INH,9
;84
	FDB	A_TAP
	FCB	1+INH,2
;85
	FDB	A_TPA
	FCB	1+INH,1
;86
	FDB	A_PULA
	FCB	1+INH,2
;87
	FDB	A_PSHA
	FCB	1+INH,2
;88
	FDB	A_PULX
	FCB	1+INH,2
;89
	FDB	A_PSHX
	FCB	1+INH,2
;8A
	FDB	A_PULH
	FCB	1+INH,2
;8B
	FDB	A_PSHH
	FCB	1+INH,2
;8C
	FDB	A_CLRH
	FCB	1+INH,1
;8D
	FDB	A_UNKNOWN
	FCB	1,0
;8E
	FDB	A_STOP
	FCB	1+INH,1
;8F
	FDB	A_WAIT
	FCB	1+INH,1
;90
	FDB	A_BGE
	FCB	2+REL,3
;91
	FDB	A_BLT
	FCB	2+REL,3
;92
	FDB	A_BGT
	FCB	2+REL,3
;93
	FDB	A_BLE
	FCB	2+REL,3
;94
	FDB	A_TXS
	FCB	1+INH,2
;95
	FDB	A_TSX
	FCB	1+INH,2
;96
	FDB	A_UNKNOWN
	FCB	1,0
;97
	FDB	A_TAX
	FCB	1+INH,1
;98
	FDB	A_CLC
	FCB	1+INH,1
;99
	FDB	A_SEC
	FCB	1+INH,1
;9A
	FDB	A_CLI
	FCB	1+INH,2
;9B
	FDB	A_SEI
	FCB	1+INH,2
;9C
	FDB	A_RSP
	FCB	1+INH,1
;9D
	FDB	A_NOP
	FCB	1+INH,1
;9E
	FDB	0
	FCB	0,0
;9F
	FDB	A_TXA
	FCB	1+INH,1
;A0
	FDB	A_SUB
	FCB	2+IMM,2
;A1
	FDB	A_CMP
	FCB	2+IMM,2
;A2
	FDB	A_SBC
	FCB	2+IMM,2
;A3
	FDB	A_CPX
	FCB	2+IMM,2
;A4
	FDB	A_AND
	FCB	2+IMM,2
;A5
	FDB	A_BIT
	FCB	2+IMM,2
;A6
	FDB	A_LDA
	FCB	2+IMM,2
;A7
	FDB	A_AIS
	FCB	2+IMM,2
;A8
	FDB	A_EOR
	FCB	2+IMM,2
;A9
	FDB	A_ADC
	FCB	2+IMM,2
;AA
	FDB	A_ORA
	FCB	2+IMM,2
;AB
	FDB	A_ADD
	FCB	2+IMM,2
;AC
	FDB	A_UNKNOWN
	FCB	1,0
;AD
	FDB	A_BSR
	FCB	2+REL,4
;AE
	FDB	A_LDX
	FCB	2+IMM,2
;AF
	FDB	A_AIX
	FCB	2+IMM,2
;B0
	FDB	A_SUB
	FCB	2+DIR,3
;B1
	FDB	A_CMP
	FCB	2+DIR,3
;B2
	FDB	A_SBC
	FCB	2+DIR,3
;B3
	FDB	A_CPX
	FCB	2+DIR,3
;B4
	FDB	A_AND
	FCB	2+DIR,3
;B5
	FDB	A_BIT
	FCB	2+DIR,3
;B6
	FDB	A_LDA
	FCB	2+DIR,3
;B7
	FDB	A_STA
	FCB	2+DIR,3
;B8
	FDB	A_EOR
	FCB	2+DIR,3
;B9
	FDB	A_ADC
	FCB	2+DIR,3
;BA
	FDB	A_ORA
	FCB	2+DIR,3
;BB
	FDB	A_ADD
	FCB	2+DIR,3
;BC
	FDB	A_JMP
	FCB	2+DIR,2
;BD
	FDB	A_JSR
	FCB	2+DIR,4
;BE
	FDB	A_LDX
	FCB	2+DIR,3
;BF
	FDB	A_STX
	FCB	2+DIR,3
;C0
	FDB	A_SUB
	FCB	3+EXT,4
;C1
	FDB	A_CMP
	FCB	3+EXT,4
;C2
	FDB	A_SBC
	FCB	3+EXT,4
;C3
	FDB	A_CPX
	FCB	3+EXT,4
;C4
	FDB	A_AND
	FCB	3+EXT,4
;C5
	FDB	A_BIT
	FCB	3+EXT,4
;C6
	FDB	A_LDA
	FCB	3+EXT,4
;C7
	FDB	A_STA
	FCB	3+EXT,4
;C8
	FDB	A_EOR
	FCB	3+EXT,4
;C9
	FDB	A_ADC
	FCB	3+EXT,4
;CA
	FDB	A_ORA
	FCB	3+EXT,4
;CB
	FDB	A_ADD
	FCB	3+EXT,4
;CC
	FDB	A_JMP
	FCB	3+EXT,3
;CD
	FDB	A_JSR
	FCB	3+EXT,5
;CE
	FDB	A_LDX
	FCB	3+EXT,4
;CF
	FDB	A_STX
	FCB	3+EXT,4
;D0
	FDB	A_SUB
	FCB	3+IX2,4
;D1
	FDB	A_CMP
	FCB	3+IX2,4
;D2
	FDB	A_SBC
	FCB	3+IX2,4
;D3
	FDB	A_CPX
	FCB	3+IX2,4
;D4
	FDB	A_AND
	FCB	3+IX2,4
;D5
	FDB	A_BIT
	FCB	3+IX2,4
;D6
	FDB	A_LDA
	FCB	3+IX2,4
;D7
	FDB	A_STA
	FCB	3+IX2,4
;D8
	FDB	A_EOR
	FCB	3+IX2,4
;D9
	FDB	A_ADC
	FCB	3+IX2,4
;DA
	FDB	A_ORA
	FCB	3+IX2,4
;DB
	FDB	A_ADD
	FCB	3+IX2,4
;DC
	FDB	A_JMP
	FCB	3+IX2,4
;DD
	FDB	A_JSR
	FCB	3+IX2,6
;DE
	FDB	A_LDX
	FCB	3+IX2,4
;DF
	FDB	A_STX
	FCB	3+IX2,4
;E0
	FDB	A_SUB
	FCB	2+IX1,3
;E1
	FDB	A_CMP
	FCB	2+IX1,3
;E2
	FDB	A_SBC
	FCB	2+IX1,3
;E3
	FDB	A_CPX
	FCB	2+IX1,3
;E4
	FDB	A_AND
	FCB	2+IX1,3
;E5
	FDB	A_BIT
	FCB	2+IX1,3
;E6
	FDB	A_LDA
	FCB	2+IX1,3
;E7
	FDB	A_STA
	FCB	2+IX1,3
;E8
	FDB	A_EOR
	FCB	2+IX1,3
;E9
	FDB	A_ADC
	FCB	2+IX1,3
;EA
	FDB	A_ORA
	FCB	2+IX1,3
;EB
	FDB	A_ADD
	FCB	2+IX1,3
;EC
	FDB	A_JMP
	FCB	2+IX1,3
;ED
	FDB	A_JSR
	FCB	2+IX1,5
;EE
	FDB	A_LDX
	FCB	2+IX1,3
;EF
	FDB	A_STX
	FCB	2+IX1,4
;F0
	FDB	A_SUB
	FCB	1+IX,2
;F1
	FDB	A_CMP
	FCB	1+IX,2
;F2
	FDB	A_SBC
	FCB	1+IX,2
;F3
	FDB	A_CPX
	FCB	1+IX,2
;F4
	FDB	A_AND
	FCB	1+IX,2
;F5
	FDB	A_BIT
	FCB	1+IX,2
;F6
	FDB	A_LDA
	FCB	1+IX,2
;F7
	FDB	A_STA
	FCB	1+IX,2
;F8
	FDB	A_EOR
	FCB	1+IX,2
;F9
	FDB	A_ADC
	FCB	1+IX,2
;FA
	FDB	A_ORA
	FCB	1+IX,2
;FB
	FDB	A_ADD
	FCB	1+IX,2
;FC
	FDB	A_JMP
	FCB	1+IX,2
;FD
	FDB	A_JSR
	FCB	1+IX,4
;FE
	FDB	A_LDX
	FCB	1+IX,2
;FF
	FDB	A_STX
	FCB	1+IX,2
PREFIX_9E6
;60
	FDB	A_NEG
	FCB	3+SP1,5
;61
	FDB	A_CBEQ
	FCB	4+SP1,6
;62
	FDB	A_UNKNOWN
	FCB	1,0
;63
	FDB	A_COM
	FCB	3+SP1,5
;64
	FDB	A_LSR
	FCB	3+SP1,5
;65
	FDB	A_UNKNOWN
	FCB	1,0
;66
	FDB	A_ROR
	FCB	3+SP1,5
;67
	FDB	A_ASR
	FCB	3+SP1,5
;68
	FDB	A_LSL
	FCB	3+SP1,5
;69
	FDB	A_ROL
	FCB	3+SP1,5
;6A
	FDB	A_DEC
	FCB	3+SP1,5
;6B
	FDB	A_DBNZ
	FCB	4+SP1,6
;6C
	FDB	A_INC
	FCB	3+SP1,5
;6D
	FDB	A_TST
	FCB	3+SP1,4
;6E
	FDB	A_UNKNOWN
	FCB	1,0
;6F
	FDB	A_CLR
	FCB	3+SP1,4
PREFIX_9ED
;D0
	FDB	A_SUB
	FCB	4+SP2,5
;D1
	FDB	A_CMP
	FCB	4+SP2,5
;D2
	FDB	A_SBC
	FCB	4+SP2,5
;D3
	FDB	A_CPX
	FCB	4+SP2,5
;D4
	FDB	A_AND
	FCB	4+SP2,5
;D5
	FDB	A_BIT
	FCB	4+SP2,5
;D6
	FDB	A_LDA
	FCB	4+SP2,5
;D7
	FDB	A_STA
	FCB	4+SP2,5
;D8
	FDB	A_EOR
	FCB	4+SP2,5
;D9
	FDB	A_ADC
	FCB	4+SP2,5
;DA
	FDB	A_ORA
	FCB	4+SP2,5
;DB
	FDB	A_ADD
	FCB	4+SP2,5
;DC
	FDB	A_UNKNOWN
	FCB	1,0
;DD
	FDB	A_UNKNOWN
	FCB	1,0
;DE
	FDB	A_LDX
	FCB	4+SP2,5
;DF
	FDB	A_STX
	FCB	4+SP2,5
PREFIX_9EE
;D0
	FDB	A_SUB
	FCB	3+SP1,4
;D1
	FDB	A_CMP
	FCB	3+SP1,4
;D2
	FDB	A_SBC
	FCB	3+SP1,4
;D3
	FDB	A_CPX
	FCB	3+SP1,4
;D4
	FDB	A_AND
	FCB	3+SP1,4
;D5
	FDB	A_BIT
	FCB	3+SP1,4
;D6
	FDB	A_LDA
	FCB	3+SP1,4
;D7
	FDB	A_STA
	FCB	3+SP1,4
;D8
	FDB	A_EOR
	FCB	3+SP1,4
;D9
	FDB	A_ADC
	FCB	3+SP1,4
;DA
	FDB	A_ORA
	FCB	3+SP1,4
;DB
	FDB	A_ADD
	FCB	3+SP1,4
;DC
	FDB	A_UNKNOWN
	FCB	1,0
;DD
	FDB	A_UNKNOWN
	FCB	1,0
;DE
	FDB	A_LDX
	FCB	3+SP1,4
;DF
	FDB	A_STX
	FCB	3+SP1,4
A_BRA	FCC	'BRA  '
A_BRN	FCC	'BRN  '
A_BHI	FCC	'BHI  '
A_BLS	FCC	'BLS  '
A_BCC	FCC	'BCC  '
A_BCS	FCC	'BCS  '
A_BNE	FCC	'BNE  '
A_BEQ	FCC	'BEQ  '
A_BHCC	FCC	'BHCC '
A_BHCS	FCC	'BHCS '
A_BPL	FCC	'BPL  '
A_BMI	FCC	'BMI  '
A_BMC	FCC	'BMC  '
A_BMS	FCC	'BMS  '
A_BIL	FCC	'BIL  '
A_BIH	FCC	'BIH  '
A_NEG	FCC	'NEG  '
A_CBEQ	FCC	'CBEQ '
A_COM	FCC	'COM  '
A_LSR	FCC	'LSR  '
A_STHX	FCC	'STHX '
A_ROR	FCC	'ROR  '
A_ASR	FCC	'ASR  '
A_LSL	FCC	'LSL  '
A_ROL	FCC	'ROL  '
A_DEC	FCC	'DEC  '
A_DBNZ	FCC	'DBNZ '
A_INC	FCC	'INC  '
A_TST	FCC	'TST  '
A_CLR	FCC	'CLR  '
A_NEGA	FCC	'NEGA '
A_CBEQA	FCC	'CBEQA'
A_MUL	FCC	'MUL  '
A_COMA	FCC	'COMA '
A_LSRA	FCC	'LSRA '
A_RORA	FCC	'RORA '
A_ASRA	FCC	'ASRA '
A_LSLA	FCC	'LSLA '
A_ROLA	FCC	'ROLA '
A_DECA	FCC	'DECA '
A_DBNZA	FCC	'DBNZA'
A_INCA	FCC	'INCA '
A_TSTA	FCC	'TSTA '
A_MOV	FCC	'MOV  '
A_CLRA	FCC	'CLRA '
A_NEGX	FCC	'NEGX '
A_CBEQX	FCC	'CBEQX'
A_DIV	FCC	'DIV  '
A_COMX	FCC	'COMX '
A_LSRX	FCC	'LSRX '
A_LDHX	FCC	'LDHX '
A_RORX	FCC	'RORX '
A_ASRX	FCC	'ASRX '
A_LSLX	FCC	'LSLX '
A_ROLX	FCC	'ROLX '
A_DECX	FCC	'DECX '
A_DBNZX	FCC	'DBNZX'
A_INCX	FCC	'INCX '
A_TSTX	FCC	'TSTX '
A_CLRX	FCC	'CLRX '
A_NSA	FCC	'NSA  '
A_CPHX	FCC	'CPHX '
A_DAA	FCC	'DAA  '
A_RTI	FCC	'RTI  '
A_RTS	FCC	'RTS  '
A_SWI	FCC	'SWI  '
A_TAP	FCC	'TAP  '
A_TPA	FCC	'TPA  '
A_PULA	FCC	'PULA '
A_PSHA	FCC	'PSHA '
A_PULX	FCC	'PULX '
A_PSHX	FCC	'PSHX '
A_PULH	FCC	'PULH '
A_PSHH	FCC	'PSHH '
A_CLRH	FCC	'CLRH '
A_STOP	FCC	'STOP '
A_WAIT	FCC	'WAIT '
A_BGE	FCC	'BGE  '
A_BLT	FCC	'BLT  '
A_BGT	FCC	'BGT  '
A_BLE	FCC	'BLE  '
A_TXS	FCC	'TXS  '
A_TSX	FCC	'TSX  '
A_TAX	FCC	'TAX  '
A_CLC	FCC	'CLC  '
A_SEC	FCC	'SEC  '
A_CLI	FCC	'CLI  '
A_SEI	FCC	'SEI  '
A_RSP	FCC	'RSP  '
A_NOP	FCC	'NOP  '
A_TXA	FCC	'TXA  '
A_SUB	FCC	'SUB  '
A_CMP	FCC	'CMP  '
A_SBC	FCC	'SBC  '
A_CPX	FCC	'CPX  '
A_AND	FCC	'AND  '
A_BIT	FCC	'BIT  '
A_LDA	FCC	'LDA  '
A_AIS	FCC	'AIS  '
A_EOR	FCC	'EOR  '
A_ADC	FCC	'ADC  '
A_ORA	FCC	'ORA  '
A_ADD	FCC	'ADD  '
A_BSR	FCC	'BSR  '
A_LDX	FCC	'LDX  '
A_AIX	FCC	'AIX  '
A_JMP	FCC	'JMP  '
A_JSR	FCC	'JSR  '
A_STX	FCC	'STX  '
A_STA	FCC	'STA  '
A_BRSET	FCC	'BRSET'
A_BRCLR	FCC	'BRCLR'
A_BSET	FCC	'BSET '
A_BCLR	FCC	'BCLR '
A_UNKNOWN FCC	'FCB '
;
;	HELP INFORMATION
;
HELP_TEXT
	FCCZ	'BS <addr>~Set breakpoint (0000=remove)'
	FCCZ	'CF <addr>,<addr>~Clear Flash Memory'
	FCCZ	'CR <reg> <data>~Change register'
	FCCZ	'DB~Display breakpoint'
	FCCZ	'DI <addr>,<addr>~Display memory in assembly format'
	FCCZ	'DM <addr>,<addr>~Display memory in hex dump format'
	FCCZ	'DR~Display processor registers'
	FCCZ	'EM <addr>~Edit memory(RAM),  "SPACE" exit'
	FCCZ	'EF <addr>~Edit memory(FLASH),"SPACE" exit'
	FCCZ	'FM <addr>,<addr> <data>~Fill memory(RAM)'
	FCCZ	'FF <addr>,<addr> <data>~Fill memory(FLASH)'
	FCCZ	'G  <addr>~Go (execute program)'
	FCCZ	'L~Load an image into memory'
	FCCZ	'MM <addr>,<addr> <addr>~Move memory (to RAM)'
	FCCZ	'S~Single step execution'
	FCB	-1		;End of list
;
;
;   BREAK MODULE ROUTINE
;
START_BRK
	PSHH			;User H reg to Stack
	PULA
	STA	REG_H		;Store user H to REG_H
	PULA
	ORA	#$08		;Prevent interrupts
	STA	REG_CC		;Store user CCR to REG_CC
	PULA
	STA	REG_A		;Store user A to REG_A
	PULA
	STA	REG_X		;Store user X to REG_X
	PULA
	STA	REG_PC		;Store user PCH to REG_PCH
	PULA
	STA	REG_PC+1	;Store user PCL to REG_PCL
	TSX
	STHX	REG_SP		;Store user SP to REG_SP
;
	LDA	#$80
	STA	SBFCR
	CLRA
	STA	BRKSCR		;Disable 16-bit address match
	STA	SBFCR		;Disable the clearable status bits
;
	LDHX	#EXIT_START_BRK
	PSHX	;PCL
	PSHH	;PCH
	PSHA
	PSHA
	PSHA
	RTI
EXIT_START_BRK
	SEI			;Disable other interrupts
	LDHX	#STACK		;Get initial stack value
	TXS			;Set up stack pointer
	JSR	DISPLAY_REG	;Display the registers
	JMP	PROMPT		;And prompt for command
END_OF_MONITOR_BODY
	NOP
;***********************************************************************
;* 68HC08 machine vectors
;***********************************************************************
	ORG	BREAK_VEC	;Position to BRK module routine
	FDB	START_BRK	;Set break module routine address
	ORG	R_VEC		;Position to reset vector
	FDB	START_MONITOR	;Set start address
	END
