*
* MON11: A software debug monitor for the 68HC11
*
* This monitor is setup for use on a 68HC11D3, which is a low cost, version
* of the 68HC11 that is available in a 40 pin DIP package, making it an ideal
* device for the home designer. You may have to modify the memory addresses
* and initialization code to use this on other versions of the 68HC11.
*
* The '11D3' has its internal registers at address ZERO, and comes up with
* the COP enabled!!!. Internal ROM (when in external mode) is enabled at
* address $7000-$7FFF. My development system uses simple "A15" decoding,
* which supports a 32K RAM chip ($0000-$7FFF), and 32K of ROM ($8000-$FFFF).
*
* ?COPY.TXT 1991-2007 Dave Dunfield
*  -- see COPY.TXT --.
*
* Memory addresses
INTREGS	EQU	$0000		Internal registers
ROM	EQU	$E000		Monitor code goes here
RAM	EQU	$0080		Monitor data goes here
USRSP	EQU	$7FFF		Initial user SP
USRPC	EQU	$0100		Initial user PC
* Internal registers
SCIBAUD	EQU	INTREGS+$2B	SCI Baudrate
SCICCR1	EQU	INTREGS+$2C	SCI Control reg 1
SCICCR2	EQU	INTREGS+$2D	SCI Control reg 2
SCISTAT	EQU	INTREGS+$2E	SCI Status Port
SCIDATA	EQU	INTREGS+$2F	SCI Data Port
CONFIG	EQU	INTREGS+$3F	CONFIG register
*
* Internal monitor RAM based variables
*
	ORG	RAM		RAM location
INTVEC	RMB	18*2		Interrupt vectors
IOCTL	RMB	1		I/O control byte
SAVCC	RMB	1		Saved Condition Codes
SAVA	RMB	1		Saved A register
SAVB	RMB	1		Saved B register
SAVX	RMB	2		Saved X register
SAVY	RMB	2		Saved Y register
SAVPC	RMB	2		Saved Program Counter
SAVSP	RMB	2		Saved Stack Pointer
BRKPTS	RMB	8*3		Breakpoint table
TEMP	RMB	5		Temporary locations
BUFFER	EQU	*		Misc. buffer
STACK	EQU	RAM+$7F		Stack goes here
*
	ORG	ROM		Code goes here
*
* Beginning of monitor code
*
BEGIN	LDS	#STACK		Initial stack
* CONFIG of 68HC711D3 must be written at startup
	LDAA	#%00000100	Disable COP and internal ROM
	STAA	CONFIG		Write config register
* Initialize the SCI device
	LDAA	#%00110000	SCP=3, SCR=0 : 9600 @ 8Mhz
	STAA	SCIBAUD		Set up baud rate
	CLR	SCICCR1		8 bit data format
	LDAA	#%00001100	No ints, enable TX & RX
	STAA	SCICCR2		Write to SCI
* Zero internal memory
	LDX	#RAM		Point to RAM
INIT1	CLR	,X		Zero one byte
	INX			Advance
	CPX	#STACK		At end?
	BLO	INIT1		No, keep going
* Establish default values
	LDD	#USRSP		Point to user stack
	STD	SAVSP		Set user PS
	LDD	#USRPC		Point to user program counter
	STD	SAVPC		Set user PC
	LDAA	#$D0		Default condition codes
	STAA	SAVCC		Set user CC
* Issue welcome message
	JSR	WRMSG		Output signon message
	FCC	'MON11 version 1.0'
	FCB	$0A,$0D,$0A
	FCC	'?COPY.TXT 1991-2007 Dave Dunfield'
	FCB	$0A,$0D
	FCC	' -- see COPY.TXT --.'
	FCB	$0A,0
*
* Newline and then execute command
*
NXTLFCR	JSR	WRLFCR		New line
*
* Prompt for and execute next command
*
NXTCMD	LDS	#STACK		Reset stack incase error
	JSR	WRMSG		Output prompt
	FCCZ	'* '		Prompt text
	JSR	RDCHRE		Read a character
	CLRB			Zero Second char
NXTCMD1	LDX	#CTABLE		Point to command table
	CLR	TEMP		Reset found flag
NXTCMD2	CPD	,X		Does it match?
	BEQ	NXTCMD4		We have it
	CMPA	,X		First char match?
	BNE	NXTCMD3		No, don't record
	STAA	TEMP		Set flag
NXTCMD3	INX			Advance pointer
	INX			to next
	INX			command
	INX			table entry
	TST	,X		End of table?
	BNE	NXTCMD2		No, keep looking
	TSTB			Already have two chars?
	BNE	ERROR		Invalid command
	LDAB	TEMP		Get flag
	BEQ	ERROR		No partial match
	PSHA			Save old char
	JSR	RDCHRE		New char
	TAB			Second char
	PULA			Restore old char
	BRA	NXTCMD1		Try again
* Command was found - execute it
NXTCMD4	JSR	WRSPACE		Separator
	LDX	2,X		Get address
	JMP	,X		And proceed
* Error has occured, report it
ERROR	JSR	WRMSG		Output message
	FCC	' ?'		Message
	FCB	$FF		Newline terminator
	BRA	NXTCMD		Get next command
*
* Write string(PC) to console
*
WRMSG	PSHX			Save 'X'
	PSHY			Save 'Y'
	TSY			Address stack
	LDX	4,Y		Get message
	BSR	WRSTR		Output string
	STX	4,Y		Resave return address
	PULY			Restore 'Y'
	PULX			Restore 'X'
	RTS
*
* Write out string(x) to console
*
WRSTR	PSHA			Save ACC
WRSTR1	LDAA	,X		Get char from X
	INX			Advance
	TSTA			End of string?
	BEQ	WRSPAC1		End, EXIT
	CMPA	#$FF		CRLF exit?
	BEQ	WRLFCR1		Newline & exit
	BSR	WRCHR		Output char
	BRA	WRSTR1		And proceed
*
* Write out a space
*
WRSPACE	PSHA			Save ACC
	LDAA	#' '		Get space
	BSR	WRCHR		Output character
WRSPAC1	PULA			Restore A
	RTS
*
* Write out line-feed and carriage-return
*
WRLFCR	PSHA			Save ACC
WRLFCR1	LDAA	#$0A		Get LF
	BSR	WRCHR		Output
	LDAA	#$0D		Get CR
	BSR	WRCHR		Output
	PULA			Restore A
	RTS
*
* Check for ESCAPE and exit to main program if received.
* Wait for SPACE if output flow control is in effect.
*
CHKESC	LDAA	IOCTL		Get I/O control byte
	BITA	#%01000000	Test PAUSE bit
	BNE	CHKE3		Pause is in effect
* Test for ESCAPE key - abort
	BSR	RDCHR		Any character?
	CMPA	#$1B		Escape?
	BNE	CHKE2		No, its OK
CHKE1	JMP	NXTCMD		Abort
* Test for SPACE - pause output
CHKE2	CMPA	#' '		Space?
	BNE	CHKE5		No, its OK
* Wait for output control key before exiting
CHKE3	LDAA	IOCTL		Get I/O control byte
	ANDA	#%10111111	Clear I/O flag
	STAA	IOCTL		Resave
CHKE4	BSR	RDCHR		Get key
	CMPA	#$1B		Escape?
	BEQ	CHKE1		Yes, ABORT
	CMPA	#$0D		Free-run?
	BEQ	CHKE5		Yes, allow it
	CMPA	#' '		Allow one line?
	BNE	CHKE4		No, try again
	LDAA	IOCTL		Get I/O control
	ORAA	#%01000000	Set PAUSE bit
	STAA	IOCTL		Resave
CHKE5	RTS
*
* Write value of D as hex word
*
WRWORD	BSR	WRBYTE		Output first byte
	TBA			Get low word
*
* Write value in A as hex byte
*
WRBYTE	PSHA			Save for later
	LSRA			Shift
	LSRA			high
	LSRA			nibble into
	LSRA			low nibble
	BSR	WRNIB		Output
	PULA			Restore LOW bute
*
* Write value in A as a hex number
*
WRNIB	ANDA	#%00001111	Drop high
	ADDA	#'0'		Convert to ASCII
	CMPA	#'9'		In range?
	BLS	WRCHR		Yes, its OK
	ADDA	#7		Convert
*
* Write character in 'A' to console
*
WRCHR	TST	IOCTL		Check output
	BMI	WRCHR2		Disabled!
	PSHA			Save char
WRCHR1	LDAA	SCISTAT		Get SCI status
	BITA	#%10000000	Ok to transmit?
	BEQ	WRCHR1		No, keep trying
	PULA			Restore char
	STAA	SCIDATA		Write SCI data port
WRCHR2	RTS
*
* Read a character - 0 = no char received
*
RDCHR	LDAA	SCISTAT		Read SCI status
	ANDA	#%00100000	RX ready?
	BEQ	RDCHR1		No, its not
	LDAA	SCIDATA		Read char from SCI
RDCHR1	RTS
*
* Wait for a character, echo on console & convert to upper case
*
RDCHRE	BSR	RDCHR		Read character
	BEQ	RDCHRE		None ready, try again
	CMPA	#' '		Control code?
	BLS	RDCHR1		Don't echo
	BSR	WRCHR		Output the character
	CMPA	#'a'		In range?
	BLO	RDCHR1		No, its not
	CMPA	#'z'		In range?
	BHI	RDCHR1		No, its not
	ANDA	#%11011111	Convert to upper case
	RTS
*
* Read a NIBBLE into A
*
RDNIB	BSR	RDCHRE		Get character & echo
	SUBA	#'0'		In range?
	CMPA	#10		Test for OK
	BLO	RDNIB1		Yes, its OK
	SUBA	#7		Convert HEX's
	CMPA	#$0A		In range?
	BLO	RDNIB2		No, its invalid
	CMPA	#$0F		In range?
	BHI	RDNIB2		Its invalid
RDNIB1	CLC			No carry, its OK
	RTS
* Test for single quote - ASCII input
RDNIB2	ADDA	#$37		Normalize character
RDNIB3	SEC			Indicate failure
	RTS
*
* Read a BYTE into A
*
RDBYTE	BSR	RDNIB		Get HIGH nibble
	BCS	RDBYTE2		Error
	LSLA			Move
	LSLA			into
	LSLA			high
	LSLA			nibble
	STAA	TEMP		Save for later
	BSR	RDNIB		Get LOW nibble
	BCS	RDBYTE1		Error
	ORAA	TEMP		Combine
RDBYTE1	RTS
RDBYTE2	CMPA	#$27		Single quote?
	BNE	RDNIB3		Report error
RDBYTE3	BSR	RDCHR		Get character
	BEQ	RDBYTE3		Wait for it
	BSR	WRCHR		Echo it
	CLC			Indicate success
	RTS
*
* Read a word into D
*
RDWORD	BSR	RDBYTE		Read HIGH byte
	BCS	RDWORD1		Check special cases
	TAB			B = HIGH
	BSR	RDBYTE		Get LOW byte
	BCS	RDBYTE1		Error
	PSHA			Save char
	TBA			A = HIGH
	PULB			B = LOW
	RTS
* Test for special case register names
RDWORD1	PSHX			Save X
	LDX	SAVX		Assume X
	CMPA	#'X'		Is it X?
	BEQ	RDWORD2		Yes, use it
	LDX	SAVY		Assume Y
	CMPA	#'Y'		Is it Y?
	BEQ	RDWORD2		Yes, use it
	LDX	SAVSP		Assume SP
	CMPA	#'S'		Is it SP?
	BEQ	RDWORD2		Yes, use it
	LDX	SAVPC		Assume PC
	CMPA	#'P'		Is it PC?
	BEQ	RDWORD2		Yes, use it
	PULX			Restore X
	SEC			Indicate error
	RTS
RDWORD2	LDAA	#'='		Separator
	JSR	WRCHR		Display it
	XGDX			D = value
	PULX			Restore X
	PSHA			Save ACCA
	JSR	WRWORD		Output the value
	PULA			Restore ACC
	CLC			Indicate success
	RTS
*
* Read a byte into A with error checking
*
RDBYTEX	BSR	RDBYTE		Get value
	BCS	ERROR1		Report error
RDBX1	RTS
*
* Read a word into A with error checking
*
RDWORDX	BSR	RDWORD		Get value
	BCC	RDBX1		Its OK
ERROR1	JMP	ERROR		Report error
*
* Read an address range into X, (temp+1) = end
*
RDADDR	BSR	RDWORDX		Get starting value
RDADDR1	XGDX			X = Beginning
	LDAA	#','		Seperator
	JSR	WRCHR		Output
	BSR	RDWORD		Get ending address
	BCC	RDADDR2		Its OK
	CMPA	#' '		Space?
	BNE	ERROR1		No, report error
	JSR	WRMSG		Output
	FCCZ	'FFFF'		Indicate the value
	LDD	#$FFFF		Default to top
RDADDR2	STD	TEMP+1		Save for later
	RTS
*
* Display user registers
*
DISREG	LDX	#REGMSG		Point to message
	LDY	#SAVCC		Point to saved CC
	LDAB	,Y		Get CC value
	BSR	DISBYTE		Display a 'CC='
	JSR	WRSTR		Next portion
	PSHX			Save X
	LDX	#CCBITS		Point to bit table
DISR1	LDAA	,X		Get flag character
	INX			Advance
	ROLB			Test bit
	BCS	DISR2		OK to display
	LDAA	#'-'		Indicate not set
DISR2	JSR	WRCHR		Output it
	CPX	#CCBITS+8	Done all bits?
	BLO	DISR1		No, keep going
	PULX			Restore messages
	BSR	DISBYTE		Display 'A='
	BSR	DISBYTE		Display 'B='
	BSR	DISWORD		Display 'X='
	BSR	DISWORD		Display 'Y='
	BSR	DISWORD		Display 'PC='
	BSR	DISWORD		Display 'SP='
	JMP	WRLFCR		New line & exit
* Display byte register
DISBYTE	JSR	WRSTR		Output string
	LDAA	,Y		Get value
	INY			Advance
	JMP	WRBYTE		Output in hex
DISWORD	JSR	WRSTR		Output string
	LDD	,Y		Get value
	INY			Advance
	INY			Advance
	JMP	WRWORD		Display
* Text tables for register display
REGMSG	FCCZ	' CC='
	FCCZ	' ['
	FCCZ	'] A='
	FCCZ	' B='
	FCCZ	' X='
	FCCZ	' Y='
	FCCZ	' PC='
	FCCZ	' SP='
CCBITS	FCC	'SXHINZVC'
*
* Software Interrupt Handler, save registers & enter monitor
*
SWIHND	TSX			Address stack
* Search breakpoint table to see if this is breakpoint
	LDD	7,X		Get Program Counter
	SUBD	#1		Backup to opcode
	LDX	#BRKPTS		Point to breakpoint table
SWI1	CPD	,X		Does it match?
	BEQ	SWI2		We found it
	INX			Skip HIGH address
	INX			Skip LOW address
	INX			Skip saved opcode
	CPX	#BRKPTS+24	Over limit?
	BNE	SWI1		Keep looking
* SWI was not result of a breakpoint, pass it on
	TSX			Address stack
	LDAA	2,X		Get PC back
	LDAB	#16		SWI interrupt
	JMP	REVECT		Revector interrupt
* Breakpoint is real, save registers & say hello
SWI2	XGDX			D = breakpoint table
	SUBD	#BRKPTS		Convert to offset
	LDX	#3		/3 for offset
	IDIV			X=D/3
	XGDX			B = breakpoint #
	STAB	TEMP		Save for later
	PULA			Get CCR
	STAA	SAVCC		Set user CC
	PULB			Get saved B
	PULA			Get saved A
	STD	SAVA		Set user D
	PULX			Get X
	STX	SAVX		Set user X
	PULX			Get Y
	STX	SAVY		Set user Y
	PULX			Get PC
	DEX			Backup to real address
	STX	SAVPC		Set user PC
	STS	SAVSP		Set user SP
* Announce breakpoint, and display user registers
	LDS	#STACK		Switch to monitor stack
	JSR	WRMSG		Output message
	FCCZ	'**Breakpoint '
	LDAA	TEMP		Get number back
	ADDA	#'0'		Convert to ASCII
	JSR	WRCHR		Output
SWI3	JSR	WRLFCR		New line
	JSR	DISREG		Show user registers
* Remove breakpoints from user code & enter monitor
	LDX	#BRKPTS		Point to saved opcodes
RMBRK1	LDY	,X		Get address
	BEQ	RMBRK2		No breakpoint
	LDAA	2,X		Get saved opcode
	STAA	,Y		Write it back
RMBRK2	INX			Skip HIGH address
	INX			Skip LOW address
	INX			Skip saved opcode
	CPX	#BRKPTS+24	Are we over?
	BLO	RMBRK1		No, keep going
	JMP	NXTCMD		New line & enter monitor
*
*************************
* Command handler table *
*************************
*
CTABLE	FCB	'D','R'		Display Registers
	FDB	DREGIST
	FCB	'D','M'		Display Memory
	FDB	DMEMORY
	FCB	'D','I'		DIsassemble memory
	FDB	DISMEM
	FCB	'D','B'		Display Breakpoints
	FDB	DBREAKP
	FCB	'D','V'		Display Vectors
	FDB	DINTVEC
	FCB	'C','R'		Change Registers
	FDB	CREGIST
	FCB	'C','V'		Change Vector
	FDB	CINTVEC
	FCB	'B',0		set Breakpoint
	FDB	SBREAKP
	FCB	'E',0		Edit Memory
	FDB	MEDIT
	FCB	'F','M'		Fill memory
	FDB	MFILL
	FCB	'L',0		Load file
	FDB	DLOAD
	FCB	'M','M'		Move Memory
	FDB	MMOVE
	FCB	'G',0		Go (begin execution)
	FDB	GO
	FCB	'S',0		Single step
	FDB	STEPC
	FCB	'W',0		Write value
	FDB	WRITE
	FCB	'R','R'		Repeating Read
	FDB	RREAD
	FCB	'R','W'		Repeating write
	FDB	RWRITE
	FCB	'M','T'		Memory Test
	FDB	RAMTEST
	FCB	'+',0		Hex Addition
	FDB	HEXADD
	FCB	'-',0		Hex Subtraction
	FDB	HEXSUB
	FCB	'?',0		Help
	FDB	HELP
	FCB	0		End of table
*
* Display registers
*
DREGIST	JSR	DISREG		Display registers
	JMP	NXTCMD		Get next command
*
* Display memory in "dump" format
*
DMEMORY	JSR	RDADDR		Get address
	JSR	WRLFCR		New line
DMEM1	PSHX			Save Address
	XGDX			Get for print
	PULX			Restore address
	JSR	WRWORD		Write address
	JSR	WRSPACE		Output space
* Display HEX fields
	PSHX			Save address
	LDAB	#16		16 bytes/line
DMEM2	JSR	WRSPACE		Output
	LDAA	,X		Get char
	JSR	WRBYTE		Display it
	INX			Advance X
	DECB			Reduce count
	BNE	DMEM2		And proceed
	JSR	WRMSG		Output seperator
	FCCZ	'    '
* Display the character data
	PULX			Restore address
	LDAB	#16		16 bytes/line
DMEM3	LDAA	,X		Get value
	BSR	WRPRINT		Output as printable
	INX			Advance
	DECB			Reduce count
	BNE	DMEM3		And proceed
	JSR	WRLFCR		New line
	JSR	CHKESC		Test for EXIT
	CPX	TEMP+1		Are we over?
	BLS	DMEM1		No, keep going
	JMP	NXTCMD		New command
* Display character only if it is printable
WRPRINT	CMPA	#' '		Lowest printable
	BLO	WRPRIN1		Yes, invaliud
	CMPA	#$7E		Highest printable
	BLS	WRPRIN2		OK to output
WRPRIN1	LDAA	#'.'		Convert to DOT
WRPRIN2	JMP	WRCHR		Output char
*
* Disassemble memory command
*
DISMEM	JSR	RDADDR		Get address
	JSR	WRLFCR		New line
DISM1	JSR	DODISS		Do one line
	JSR	CHKESC		Test for ESCAPE exit
	CPX	TEMP+1		Are we over?
	BLS	DISM1		No, keep going
	JMP	NXTCMD		Get next command
*
* Display breakpoints
*
DBREAKP	LDX	#BRKPTS		Point to breakpoint table
	LDAB	#'0'		Initial brkpt number
DBREAK1	JSR	WRSPACE		Seperator
	LDAA	#'B'		Header
	JSR	WRCHR		Output
	TBA			Get number
	JSR	WRCHR		Output
	LDAA	#'='		Trailer
	JSR	WRCHR		Output
	PSHB			Save number
	LDD	,X		Get value
	INX			Skip HIGH address
	INX			Skip LOW address
	INX			Skip saved opcode
	JSR	WRWORD		Display it
	PULB			Restore number
	INCB			Advance
	CMPB	#'8'		In range?
	BLO	DBREAK1		No, keep going
	JMP	NXTLFCR		New line & exit
*
* Display interrupt vectors
*
DINTVEC	JSR	WRLFCR		New line
	LDY	#INTVEC		Point to vectors
	CLRB			Zero ID
DINTV1	PSHB			Save name
	TBA			Get address
	JSR	WRBYTE		Output
	JSR	WRSPACE		Seperator
	LDD	,Y		Get value
	JSR	WRWORD		Output value
	JSR	WRSPACE
	PULB			Restore number
	JSR	DINTNAM		Display name
	JSR	WRLFCR		New line
	JSR	CHKESC		Test for ESCAPE exit
	INY			Advance
	INY			Advance
	INCB			Advance
	CMPB	#18		All done?
	BLO	DINTV1		No, keep going
	JMP	NXTCMD		Next command
*
* Change registers
*
CREGIST	JSR	RDCHRE		Get register ID
	LDX	#RTABLE		Point to table
	LDY	#SAVCC		Point to saved CC
CREGI1	CMPA	,X		Is this it?
	BEQ	CREGI2		No, its not
	INX			Advance
	LDAB	,X		Get length
	INX			Advance
	ABY			Next register
	TST	,X		End of table?
	BNE	CREGI1		No, keep looking
ERROR2	JMP	ERROR		Report error
CREGI2	JSR	WRSPACE		Separator
	LDAB	1,X		Get register length
	DECB			Test for length=1
	BEQ	CREGI3		No, use 8 bit
	JSR	RDWORDX		Get 16 bit value
	STD	,Y		Write to register
	JMP	NXTLFCR		New line & exit
CREGI3	JSR	RDBYTEX		Get 8 bit value
	STAA	,Y		Write to register
	JMP	NXTLFCR		New line & exit
* Register display table
RTABLE	FCB	'C',1		Condition code register
	FCB	'D',0		A+B
	FCB	'A',1		A accumulator
	FCB	'B',1		B accumulator
	FCB	'X',2		X index register
	FCB	'Y',2		Y index register
	FCB	'P',2		Program counter
	FCB	'S',2		Stack Pointer
	FCB	0
*
* Change interrupt vector
*
CINTVEC	JSR	RDBYTEX		Get vector number
	CMPA	#18		In range?
	BHS	ERROR2		Report error
	LDX	#INTVEC		Point to vector
	TAB			B = value
	ABX			Offset
	ABX			16 bit entries
	JSR	WRSPACE		Separator
	JSR	RDWORDX		Get new address
	STD	,X		Set new vector
	JMP	NXTLFCR		New line & next command
*
* Set breakpoint
*
SBREAKP	JSR	RDCHRE		Get char
	SUBA	#'0'		Convert to binary
	CMPA	#8		In range?
	BHS	ERROR2		Report failure
	TAB			B = brkpt #
	LDX	#BRKPTS		Point to breakpoint table
	ABX			Offset for HIGH address
	ABX			Offset for LOW address
	ABX			Offset for saved opcode
	JSR	WRSPACE		Seperator
	JSR	RDWORDX		Get value
	STD	,X		Write value
	JMP	NXTLFCR		New line & next command
*
* Write a value to memory
*
WRITE	JSR	RDWORDX		Get address
	XGDX			Swap to X
	JSR	WRSPACE		Separator
	JSR	RDBYTEX		Get value
	STAA	,X		Write it
	JMP	NXTLFCR		New line & next command
*
* Repeating Read
*
RREAD	JSR	RDWORDX		Get address
	XGDX			Swap to X
	JSR	WRLFCR		New line
RREAD1	LDAA	,X		Read byte
	JSR	RDCHR		Test for jey
	CMPA	#$1B		Escape?
	BNE	RREAD1		Keep going
	JMP	NXTCMD		New command
*
* Repeating Write
*
RWRITE	JSR	RDWORDX		Get address
	XGDX			Swap to X
	JSR	WRSPACE		Separator
	JSR	RDBYTEX		Get value
	TAB			B = value
	JSR	WRLFCR		New line
RWRITE1	STAB	,X		Write the value
	JSR	RDCHR		Test for key
	CMPA	#$1B		Escape?
	BNE	RWRITE1		No, keep going
	JMP	NXTCMD		New command
*
* Download to memory
*
DLOAD	JSR	WRLFCR		New line
	LDY	#0		Begin with zero
DLOAD1	JSR	DLREC		Get a record
	BCS	DLOAD1		More.. get them
	XGDY			D = # bytes
	JSR	WRWORD		Output value
	JSR	WRMSG		Output message
	FCCZ	' Bytes'
	JMP	NXTLFCR		New line & next command
*
* 'MT' - MEMORY TEST
*
RAMTEST	JSR	RDADDR		Get address range
	STX	TEMP+3		Save start address
	LDY	#-1		Always loop once
	JSR	WRLFCR		New line
RAM0	CLR	,X		Zap one byte
	INX			Advance
	CPX	TEMP+1		Are we over?
	BLS	RAM0		No, clear em all
RAM1	LDX	TEMP+3		Reset starting address
	LDAA	#$0D		Get CR
	JSR	WRCHR		Back to start of line
	JSR	WRMSG		Output messahe
	FCCZ	'Pass: '
	INY			Advance count
	STY	BUFFER		Save for later
	LDD	BUFFER		Get value
	JSR	WRWORD		Output
	JSR	WRSPACE		Space over
RAM2	JSR	RDCHR		Test for characer
	CMPA	#$1B		Escape?
	BEQ	RAM5		Quit & restart monitor
	LDAB	BUFFER+1	Get extected value
	CMPB	,X		Did it keep its value
	BNE	RAM7		No, error
	LDAA	#%00000001	First data value
RAM3	STAA	,X		Write to memory
	CMPA	,X		Same ?
	BNE	RAM6		Failed
	LSLA			Shift to next bit
	BNE	RAM3		Do all bits
RAM4	INCB			Advance to next value
	STAB	,X		Save in location
	INX			Advance to next
	CPX	TEMP+1		More to do?
	BLS	RAM2		Yes, its OK
	BRA	RAM1		And go again
RAM5	JMP	NXTLFCR		RE-enter monitor
* Verify of location failed
RAM6	PSHA			Save value written
	JSR	WRMSG		OUTPUT MESSAGE
	FCCZ	'Addr: '
	PSHX			Copy 'X'
	PULA			Restore to D (HIGH)
	PULB			Restore to D (LOW
	JSR	WRWORD		Output address
	JSR	WRMSG		Output message
	FCCZ	', Wrote: '
	PULA			Restore value written
	JSR	WRBYTE		Output value
	JSR	WRMSG		Output message
	FCCZ	', Read: '
	LDAA	,X		Get value read
	BRA	RAM8		Continue
* DATA WAS CORRUPTED BY OTHER WRITES
RAM7	JSR	WRMSG		Output message
	FCCZ	'Addr: '
	PSHX			Copy address
	PULA			Restore to D (HIGH)
	PULB			Restore to D (LOW)
	JSR	WRWORD		Output address
	JSR	WRMSG		Output message
	FCCZ	', Expected: '
	LDAA	BUFFER+1	Get expected value
	JSR	WRBYTE		Output value
	JSR	WRMSG		Output message
	FCCZ	', Read: '
	LDAA	,X		Get value read
RAM8	JSR	WRBYTE		Output value
	JSR	WRLFCR		New line
	LDAB	BUFFER+1	Get correct next value
	JMP	RAM4		And proceed
*
* Edit memory command
*
MEDIT	JSR	RDWORDX		Get address
	XGDX			X = address
* Prompt with address
MEDIT1	JSR	WRLFCR		New line
	PSHX			Save X
	XGDX			D = X
	PULX			Restore X
	JSR	WRWORD		Output address
	LDAB	#8		Max count
* Prompt with location contents
MEDIT2	JSR	WRSPACE		Output space
	LDAA	,X		Get value
	JSR	WRBYTE		Output
	LDAA	#'-'		Get seperator
	JSR	WRCHR		Output
	JSR	RDBYTE		Read a byte
	BCS	MEDIT5		Special command
MEDIT3	STAA	,X		Write to address
MEDIT4	INX	X		Advance
	DECB			Through range?
	BNE	MEDIT2		No, keep going
	BRA	MEDIT1		And proceed
* Space: Next address
MEDIT5	CMPA	#' '		Next location?
	BNE	MEDIT6		No, try next
	JSR	WRSPACE		Extra space
	BRA	MEDIT4		And proceed
* Backspace: previous address
MEDIT6	CMPA	#$08		Backspace?
	BNE	MEDIT7		No, try next
	DEX			Previous address
	BRA	MEDIT1		And proceed
* Return: exit
MEDIT7	CMPA	#$0D		Return?
	BNE	ERROR3		No, errot
	JMP	NXTLFCR		Proceed
*
* Fill memory with a value
*
MFILL	JSR	RDADDR		Read in an address
	JSR	WRSPACE		Seperator
	JSR	RDBYTEX		Get value
MFILL1	STAA	,X		Write to location
	INX			Advance X
	CPX	TEMP+1		Are we over?
	BLS	MFILL1		No, do them all
	JMP	NXTLFCR		Proceed
*
* Move a block of memory
*
MMOVE	JSR	RDADDR		Read in an address range
	JSR	WRSPACE		Output space
	JSR	RDWORDX		Get word of memory
	XGDY			Y = destination
MMOVE1	LDAA	,X		Read from source
	STAA	,Y		Write to dest
	INX			Advance source
	INY			Advance dest
	CPX	TEMP+1		Are we over?
	BLS	MMOVE1		No, do them all
	JMP	NXTLFCR		And proceed
* Report an error
ERROR3	JMP	ERROR		Report error
*
* Hex addition
*
HEXADD	JSR	RDWORDX		Read in a word
	STD	TEMP+1		Save for later
	LDAA	#'+'		Seperator
	JSR	WRCHR		Output
	JSR	RDWORDX		Read in second
	ADDD	TEMP+1		Perform addition
	BRA	HEXOUT		Output answer
*
* Hex Subtraction
*
HEXSUB	JSR	RDWORDX		Read in a word
	STD	TEMP+1		Save for later
	XGDX			X = value
	LDAA	#'-'		Seperator
	JSR	WRCHR		Output
	JSR	RDWORDX		Read in second
	STD	TEMP+1		Save second
	XGDX			D = value
	SUBD	TEMP+1		Calculate result
HEXOUT	JSR	WRMSG		Output message
	FCCZ	'='		PRoceed
	JSR	WRWORD		Output result
	JMP	NXTLFCR		New line & exit
*
* Get PC address - Default to SAVPC
*
GETPC	JSR	RDWORD		Get value
	BCC	GETPC1		Its OK
	CMPA	#' '		Space = default?
	BNE	ERROR3		No, report error
	JSR	WRMSG		Output message
	FCCZ	'->'
	LDD	SAVPC		Get user PC
	PSHA			Save high
	JSR	WRWORD		Output
	PULA			Restore high
GETPC1	JMP	WRLFCR		New line
*
* Dispatch the user program
*
GO	BSR	GETPC		Get PC address
	STD	SAVPC		Set user PC
	XGDY			Y = address
	JSR	DISASS		Get length
	XGDY			D = out address
	SUBD	SAVPC		Calculate length
	LDY	SAVPC		Restore address
	JSR	STEP		Step over first brkpt
* Insert breakpoints into user program
	LDX	#BRKPTS		Point to breakpoint table
	LDAB	#$3F		Get SWI (breakpoint)
GO1	LDY	,X		Get address
	BEQ	GO2		Not a breakpoint
	LDAA	,Y		Get user opcode
	STAA	2,X		Save in table
	STAB	,Y		Insert breakpoint
GO2	INX			Skip HIGH address
	INX			Skip LOW address
	INX			Skip saved opcode
	CPX	#BRKPTS+24	End of table?
	BLO	GO1		No, keep going
* Restore user registers and return to him
	LDS	SAVSP		Get user stack
	LDX	SAVPC		Get user PC
	PSHX			Stack it
	LDX	SAVY		Get user Y
	PSHX			Stack it
	LDX	SAVX		Get user X
	PSHX			Stack it
	LDD	SAVA		Get user A,B
	PSHA			Save user A
	PSHB			Save user B
	LDAA	SAVCC		Get user CC
	PSHA			Save it
	RTI			Execute user program
*
* STEP command
*
STEPC	BSR	GETPC		Get PC value
STEPC2	XGDX			X = address
	PSHX			Save address
	BSR	DODISS		Perform disassembly
	PULY			Restore Y
	LDAB	TEMP+3		Get length
	JSR	STEP		Perform stemp
	LDAA	IOCTL		Get I/O control
	BITA	#%00100000	Register display disabled?
	BNE	STEPC3		Yes, don't output
STEPC2A	JSR	DISREG		Display registers
STEPC3	JSR	RDCHR		Read character
	CMPA	#' '		Continue?
	BNE	STEPC4		No, try next
	LDD	SAVPC		Get address
	BRA	STEPC2		And proceed
STEPC4	CMPA	#'?'		Display registers?
	BEQ	STEPC2A		Yes, do it
	CMPA	#$0D		Toggle reg display?
	BNE	STEPC5		No, try next
	LDAA	IOCTL		Get contol byte
	EORA	#%00100000	Toggle flag
	STAA	IOCTL		Resave
	BRA	STEPC3		And get another key
STEPC5	CMPA	#$1B		ESCAPE?
	BNE	STEPC3		No, wait for a key
	JMP	NXTCMD
*
* Display dissassembler output (X=address)
*
DODISS	PSHX			Save output poiner
* Get the disassembled instruction
	PULY			Y = address
	PSHX			Resave
	BSR	DISASS		Perform disassembly
	TSX			Address stack
	XGDY			D = Ending address
	SUBD	,X		Calculate length
	STAB	TEMP+3		Save length
* Display address at beginning of line
	LDD	,X		D = instruction address
	PULX			X = instruction address
	JSR	WRWORD		Output value
	JSR	WRSPACE		Separator
* Display HEX bytes
	CLRB			Zero count
	PSHX			Save address
DODISS1	LDAA	,X		Get value from opcide
	INX			Advance
	JSR	WRBYTE		Output
	JSR	WRSPACE		New value
	INCB			Advance count
	CMPB	TEMP+3		Are we over?
	BLO	DODISS1		No, its OK
* Fill for even boundaries
DODISS2	CMPB	#5		Done am all?
	BHS	DODISS3		Yes, its OK
	INCB			Advance position
	JSR	WRMSG		Output string
	FCCZ	'   '		Separator
	BRA	DODISS2		Do them all
* Display ASCII equivalent
DODISS3	PULX			Restore address
	CLRB			Zero count
DODISS4	LDAA	,X		Get byte
	INX			Advance
	JSR	WRPRINT		Output if printable
	INCB			Advance count
	CMPB	TEMP+3		All done?
	BLO	DODISS4		No, keep going
* Fill for even boundary
DODISS5	CMPB	#6		Done am all?
	BHS	DODISS6		Yes, we have
	INCB			Advance count
	JSR	WRSPACE		Output space
	BRA	DODISS5		And go again
* Display the instruction
DODISS6	PSHX			Save address
	LDX	#BUFFER		Point at buffer
	JSR	WRSTR		Output it
	PULX			Restore X
	JMP	WRLFCR		New line
*
* Search shift table for opcode
*
DISHIFT	LDAB	,Y		Get next opcode bute
	INY			Advance
DISH1	LDAA	,X		Get table opcode
	BEQ	DISA0		End of table...
	INX			Skip opcode
	CBA			Match our opcode?
	BEQ	DISA0		Yes, process
	INX			Skip text id
	INX			Skip operand type
	BRA	DISH1		Try again
*
* Main disassembler, fill BUFFER with disassembly of (Y)
*
DISASS	LDD	#BUFFER		Point to output buffer
	STD	TEMP+3		Save output pointer
	LDAB	,Y		Get opcode
	INY			Advance
	LDX	#OPTAB1		Assume shift table 1
	CMPB	#$18		Shift-1 opcode?
	BEQ	DISHIFT		Yes, handle it
	LDX	#OPTAB2		Assume shift table 2
	CMPB	#$1A		Shift-2 opcode?
	BEQ	DISHIFT		Yes, handle it
	LDX	#OPTAB3		Assume shift table 3
	CMPB	#$CD		Shift-3 opcode?
	BEQ	DISHIFT		YEs, handle it
* Lookup opcode in main table
	CLRA			Zero high
	LSLD			*2
	ADDD	#OPTAB0		Offset to table
	XGDX			X = table address
DISA0	LDD	,X		Get info
	STAB	TEMP		Save opcode data
* Copy instruction opcode to output buffer
	TAB			B = textid
	CLRA			Zero high
	LSLD			*2
	LSLD			*4
	ADDD	#INSTXT		Offset to table
	XGDX			X = table address
	LDAB	#4		Copy 4 chars
DISA1	LDAA	,X		Get char
	INX			Advance
	JSR	PUTCHR		Write to output buffer
	DECB			Reduce count
	BNE	DISA1		Do them all
* Append a space
	LDAA	#' '		Get space
	JSR	PUTCHR		Write it
* Process operands
	LDAB	TEMP		Get type back
	ANDB	#%00000111	Save only type
	BEQ	DEXIT		No argument
* Type 1 - Immediate 8 bit value
	DECB			Type 1?
	BNE	DTYPE2		No, try next
	LDAA	#'#'		Immediate indication
	JSR	PUTCHR		Write it
	BRA	DWR8B		Output
* Type 2 - Immediate 16 bit value
DTYPE2	DECB			Type2?
	BNE	DTYPE3		No, try next
	LDAA	#'#'		Immediate indication
	JSR	PUTCHR		Write it
	BRA	DWR16B		Output 16 bit value
* Type3 - 8 bit direct address
DTYPE3	DECB			Type3?
	BNE	DTYPE4		No, try next
DWR8B	LDAA	,Y		Get value
	INY			Advance
	BSR	PUTHEX		Write it
	JMP	DEXIT		Finish up
* Type4 - 16 bit extended address
DTYPE4	DECB			Type 4?
	BNE	DTYPE5		No, try next
DWR16B	LDD	,Y		Get value
	INY			Advance
	INY			Adbance
	BRA	DWR16C		Write both bytes
* Type5 - Indexed from X
DTYPE5	DECB			Type5?
	BNE	DTYPE6		No, try next
	LDAB	#'X'		Offset from X
	BRA	DWRIDX		And proceed
* Type6 - Indexed from Y
DTYPE6	DECB			Type6?
	BNE	DTYPE7		No, try next
	LDAB	#'Y'		Offset from Y
DWRIDX	LDAA	,Y		Get offset value
	INY			Advance
	BSR	PUTHEX		Write it
	LDAA	#','		Get seperator
	JSR	PUTCHR		Write it out
	TBA			A = index reg
	BSR	PUTCHR		Write it
	BRA	DEXIT		Finish up
* Type7 - PC relative address
DTYPE7	CLRA			Assume positive
	LDAB	,Y		Get offset
	BPL	DTYPE7A		Assumption correct
	COMA			Convert to negative
DTYPE7A	INY			Advance Y
	PSHY			Write out Y
	TSX			Addres stack
	ADDD	,X		Calculate address
	PULY			Restore y
DWR16C	BSR	PUTHEX		Out first byte
	TBA			A = low value
	BSR	PUTHEX		Out second byte
*
* Handle any post processing
*
DEXIT	LDAB	TEMP		Get type byte
* If bit mask follows, append-it
	BITB	#%00001000	MASK follows?
	BEQ	DEXIT1		No, its OK
	LDAA	#';'		Separator
	BSR	PUTCHR		Output
	LDAA	,Y		Get value
	INY			Advance
	BSR	PUTHEX		And proceed
* Append PCR address if needed
DEXIT1	BITB	#%00010000	PCR address follows?
	BEQ	DEXIT2		No, its OK
	BITB	#%00001111	Any other operands?
	BEQ	DEXIT1A		No, its OK
	LDAA	#' '		Separator
	JSR	PUTCHR		Output
DEXIT1A	CLRA			Assume positice
	LDAB	,Y		Get offset
	BPL	DEXIT1B		Assumption correct
	DECA			Convert to negative
DEXIT1B	INY			Advance Y
	PSHY			Write out Y
	TSX			Addres stack
	ADDD	,X		Calculate address
	PULY			Restore y
	BSR	PUTHEX		Out first byte
	TBA			A = low value
	BSR	PUTHEX		Out second byte
DEXIT2	LDX	TEMP+3		Get output pointer
	CLR	,X		Zero end of display
	RTS
* Write byte(A) to output buffer
PUTHEX	PSHA			Save value
	LSRA			Shift
	LSRA			high nibble
	LSRA			into
	LSRA			low nibble
	BSR	PUTNIB		Write it
	PULA			Restore value
* Write nibble(A) to output buffer
PUTNIB	ANDA	#%00001111	Save only lower nibble
	ADDA	#'0'		Offset
	CMPA	#'9'		In range?
	BLS	PUTCHR		Yes, write
	ADDA	#7		Convert alphas
* Write character(A) to output buffer
PUTCHR	PSHX			Save X
	LDX	TEMP+3		Get output pointer
	STAA	,X		Write char
	INX			Advance position
	STX	TEMP+3		Resave output pointer
	PULX			Restore X
	RTS
*
* Disassembler main opcode table, first byte is text number, second byte:
* Bits 0-2:
*	000 = No operand
*	001 = Immediate 8 bit value
*	010 = Immediate 16 bit value
*	011 = Direct (8 bit) address
*	100 = Extended (16 bit address)
*	101 = Indexed from X
*	110 = Indexed from Y
* Bit 3: Instruction is followed by bit MASK
* Bit 4: Instruction is followed by PC relative address
*
OPTAB0	FCB	125,%00000000	00 TEST
	FCB	086,%00000000	01 NOP
	FCB	059,%00000000	02 IDIV
	FCB	058,%00000000	03 FDIV
	FCB	081,%00000000	04 LSRD
	FCB	077,%00000000	05 LSLD
	FCB	123,%00000000	06 TAP
	FCB	126,%00000000	07 TPA
	FCB	064,%00000000	08 INX
	FCB	054,%00000000	09 DEX
	FCB	040,%00000000	0A CLV
	FCB	110,%00000000	0B SEV
	FCB	037,%00000000	0C CLC
	FCB	108,%00000000	0D SEC
	FCB	038,%00000000	0E CLI
	FCB	109,%00000000	0F SEI
	FCB	105,%00000000	10 SBA
	FCB	036,%00000000	11 CBA
	FCB	031,%00011011	12 BRSET dir
	FCB	029,%00011011	13 BRCLR dir
	FCB	032,%00001011	14 BSET dir
	FCB	015,%00001011	15 BCLR dir
	FCB	122,%00000000	16 TAB
	FCB	124,%00000000	17 TBA
	FCB	000,%00000000	18 Shift-1
	FCB	049,%00000000	19 DAA
	FCB	000,%00000000	1A Shift-2
	FCB	001,%00000000	1B ABA
	FCB	032,%00001101	1C BSET ,x
	FCB	015,%00001101	1D BSET ,x
	FCB	031,%00011101	1E BRSET ,x
	FCB	029,%00011101	1F BRCLR ,x
	FCB	028,%00010000	20 BRA
	FCB	030,%00010000	21 BRN
	FCB	020,%00010000	22 BHI
	FCB	023,%00010000	23 BLS
	FCB	014,%00010000	24 BCC
	FCB	016,%00010000	25 BCS
	FCB	026,%00010000	26 BNE
	FCB	017,%00010000	27 BEQ
	FCB	034,%00010000	28 BVC
	FCB	035,%00010000	29 BVS
	FCB	027,%00010000	2A BPL
	FCB	025,%00010000	2B BMI
	FCB	018,%00010000	2C BGE
	FCB	024,%00010000	2D BLT
	FCB	019,%00010000	2E BGT
	FCB	022,%00010000	2F BLE
	FCB	130,%00000000	30 TSX
	FCB	063,%00000000	31 INS
	FCB	093,%00000000	32 PULA
	FCB	094,%00000000	33 PULB
	FCB	053,%00000000	34 DES
	FCB	132,%00000000	35 TXS
	FCB	089,%00000000	36 PSHA
	FCB	090,%00000000	37 PSHB
	FCB	095,%00000000	38 PULX
	FCB	104,%00000000	39 RTS
	FCB	002,%00000000	3A ABX
	FCB	103,%00000000	3B RTI
	FCB	091,%00000000	3C PSHX
	FCB	082,%00000000	3D MUL
	FCB	134,%00000000	3E WAI
	FCB	121,%00000000	3F SWI
	FCB	084,%00000000	40 NEGA
	FCB	000,%00000000	41 ???
	FCB	000,%00000000	42 ???
	FCB	044,%00000000	43 COMA
	FCB	079,%00000000	44 LSRA
	FCB	000,%00000000	45 ???
	FCB	101,%00000000	46 RORA
	FCB	012,%00000000	47 ASRA
	FCB	075,%00000000	48 LSLA
	FCB	098,%00000000	49 ROLA
	FCB	051,%00000000	4A DECA
	FCB	000,%00000000	4B ???
	FCB	061,%00000000	4C INCA
	FCB	128,%00000000	4D TSTA
	FCB	000,%00000000	4E ???
	FCB	137,%00000000	4F CLRA
	FCB	085,%00000000	50 NEGB
	FCB	000,%00000000	51 ???
	FCB	000,%00000000	52 ???
	FCB	045,%00000000	53 COMB
	FCB	080,%00000000	54 LSRB
	FCB	000,%00000000	55 ???
	FCB	102,%00000000	56 RORB
	FCB	013,%00000000	57 ASRB
	FCB	076,%00000000	58 LSLB
	FCB	099,%00000000	59 ROLB
	FCB	052,%00000000	5A DECB
	FCB	000,%00000000	5B ???
	FCB	062,%00000000	5C INCB
	FCB	129,%00000000	5D TSTB
	FCB	000,%00000000	5E ???
	FCB	138,%00000000	5F CLRB
	FCB	083,%00000101	60 NEG ,x
	FCB	000,%00000000	61 ???
	FCB	000,%00000000	62 ???
	FCB	043,%00000101	63 COM ,x
	FCB	078,%00000101	64 LSR ,x
	FCB	000,%00000000	65 ???
	FCB	100,%00000101	66 ROR ,x
	FCB	011,%00000101	67 ASR ,x
	FCB	074,%00000101	68 LSL ,x
	FCB	097,%00000101	69 ROL ,x
	FCB	050,%00000101	6A DEC ,x
	FCB	000,%00000000	6B ???
	FCB	060,%00000101	6C INC ,x
	FCB	127,%00000101	6D TST ,x
	FCB	066,%00000101	6E JMP ,x
	FCB	039,%00000101	6F CLR ,x
	FCB	083,%00000100	70 NEG ext
	FCB	000,%00000000	71 ???
	FCB	000,%00000000	72 ???
	FCB	043,%00000100	73 COM ext
	FCB	078,%00000100	74 LSR ext
	FCB	000,%00000000	75 ???
	FCB	100,%00000100	76 ROR ext
	FCB	011,%00000100	77 ASR ext
	FCB	074,%00000100	78 LSL ext
	FCB	097,%00000100	79 ROL ext
	FCB	050,%00000100	7A DEC ext
	FCB	000,%00000000	7B ???
	FCB	060,%00000100	7C INC ext
	FCB	127,%00000100	7D TST ext
	FCB	066,%00000100	7E JMP ext
	FCB	039,%00000100	7F CLR ext
	FCB	118,%00000001	80 SUBA #
	FCB	041,%00000001	81 CMPA #
	FCB	106,%00000001	82 SBCA #
	FCB	120,%00000010	83 SUBD #
	FCB	009,%00000001	84 ANDA #
	FCB	139,%00000001	85 BITA #
	FCB	068,%00000001	86 LDAA #
	FCB	000,%00000000	87 ???
	FCB	056,%00000001	88 EORA #
	FCB	004,%00000001	89 ADCA #
	FCB	087,%00000001	8A ORAA #
	FCB	006,%00000001	8B ADDA #
	FCB	047,%00000010	8C CPX #
	FCB	033,%00010000	8D BSR
	FCB	071,%00000010	8E LDS #
	FCB	135,%00000000	8F XGDX
	FCB	118,%00000011	90 SUBA dir
	FCB	041,%00000011	91 CMPA dir
	FCB	106,%00000011	92 SBCA dir
	FCB	120,%00000011	93 SUBD dir
	FCB	009,%00000011	94 ANDA dir
	FCB	139,%00000011	95 BITA dir
	FCB	068,%00000011	96 LDAA dir
	FCB	111,%00000011	97 STAA dir
	FCB	056,%00000011	98 EORA dir
	FCB	004,%00000011	99 ADCA dir
	FCB	087,%00000011	9A ORAA dir
	FCB	006,%00000011	9B ADDA dir
	FCB	047,%00000011	9C CPX dir
	FCB	067,%00000011	9D JSR dir
	FCB	071,%00000011	9E LDS dir
	FCB	115,%00000011	9F STS dir
	FCB	118,%00000101	A0 SUBA ,x
	FCB	041,%00000101	A1 CMPA ,x
	FCB	106,%00000101	A2 SBCA ,x
	FCB	120,%00000101	A3 SUBD ,x
	FCB	009,%00000101	A4 ANDA ,x
	FCB	139,%00000101	A5 BITA ,x
	FCB	068,%00000101	A6 LDAA ,x
	FCB	111,%00000101	A7 STAA ,x
	FCB	056,%00000101	A8 EORA ,x
	FCB	004,%00000101	A9 ADCA ,x
	FCB	087,%00000101	AA ORAA ,x
	FCB	006,%00000101	AB ADDA ,x
	FCB	047,%00000101	AC CPX ,x
	FCB	067,%00000101	AD JSR ,x
	FCB	071,%00000101	AE LDS ,x
	FCB	115,%00000101	AF STS ,x
	FCB	118,%00000100	B0 SUBA ext
	FCB	041,%00000100	B1 CMPA ext
	FCB	106,%00000100	B2 SBCA ext
	FCB	120,%00000100	B3 SUBD ext
	FCB	009,%00000100	B4 ANDA ext
	FCB	139,%00000100	B5 BITA ext
	FCB	068,%00000100	B6 LDAA ext
	FCB	111,%00000100	B7 STAA ext
	FCB	056,%00000100	B8 EORA ext
	FCB	004,%00000100	B9 ADCA ext
	FCB	087,%00000100	BA ORAA ext
	FCB	006,%00000100	BB ADDA ext
	FCB	047,%00000100	BC CPX ext
	FCB	067,%00000100	BD JSR ext
	FCB	071,%00000100	BE LDS ext
	FCB	115,%00000100	BF STS ext
	FCB	119,%00000001	C0 SUBB #
	FCB	042,%00000001	C1 CMPB #
	FCB	107,%00000001	C2 SBCB #
	FCB	008,%00000010	C3 ADDD #
	FCB	010,%00000001	C4 ANDB #
	FCB	140,%00000001	C5 BITB #
	FCB	069,%00000001	C6 LDAB #
	FCB	000,%00000000	C7 ???
	FCB	057,%00000001	C8 EORB #
	FCB	005,%00000001	C9 ADCB #
	FCB	088,%00000001	CA ORAB #
	FCB	007,%00000001	CB ADDB #
	FCB	070,%00000010	CC LDD #
	FCB	000,%00000000	CD Shift-3
	FCB	072,%00000010	CE LDX #
	FCB	114,%00000000	CF STOP
	FCB	119,%00000011	D0 SUBB dir
	FCB	042,%00000011	D1 CMPB dir
	FCB	107,%00000011	D2 SBCB dir
	FCB	008,%00000011	D3 SUBD dir
	FCB	010,%00000011	D4 ANDB dir
	FCB	140,%00000011	D5 BITB dir
	FCB	069,%00000011	D6 LDAB dir
	FCB	112,%00000011	D7 STAB dir
	FCB	057,%00000011	D8 EORB dir
	FCB	005,%00000011	D9 ADCB dir
	FCB	088,%00000011	DA ORAB dir
	FCB	007,%00000011	DB ADDB dir
	FCB	070,%00000011	DC LDD dir
	FCB	113,%00000011	DD STD dir
	FCB	072,%00000011	DE LDX dir
	FCB	116,%00000011	DF STX dir
	FCB	119,%00000101	E0 SUBB ,x
	FCB	042,%00000101	E1 CMPB ,x
	FCB	107,%00000101	E2 SBCB ,x
	FCB	008,%00000101	E3 SUBD ,x
	FCB	010,%00000101	E4 ANDB ,x
	FCB	140,%00000101	E5 BITB ,x
	FCB	069,%00000101	E6 LDAB ,x
	FCB	112,%00000101	E7 STAB ,x
	FCB	057,%00000101	E8 EORB ,x
	FCB	005,%00000101	E9 ADCB ,x
	FCB	088,%00000101	EA ORAB ,x
	FCB	007,%00000101	EB ADDB ,x
	FCB	070,%00000101	EC LDD ,x
	FCB	113,%00000101	ED STD ,x
	FCB	072,%00000101	EE LDX ,x
	FCB	116,%00000101	EF STX ,x
	FCB	119,%00000100	F0 SUBB ext
	FCB	042,%00000100	F1 CMPB ext
	FCB	107,%00000100	F2 SBCB ext
	FCB	008,%00000100	F3 SUBD ext
	FCB	010,%00000100	F4 ANDB ext
	FCB	140,%00000100	F5 BITB ext
	FCB	069,%00000100	F6 LDAB ext
	FCB	112,%00000100	F7 STAB ext
	FCB	057,%00000100	F8 EORB ext
	FCB	005,%00000100	F9 ADCB ext
	FCB	088,%00000100	FA ORAB ext
	FCB	007,%00000100	FB ADDB ext
	FCB	070,%00000100	FC LDD ext
	FCB	113,%00000100	FD STD ext
	FCB	072,%00000100	FE LDX ext
	FCB	116,%00000100	FF STX ext
*
* Shifted Opcode tables:
* Format as above, except opcode prefix's
*
* Shifted opcode table 1 ($18)
OPTAB1	FCB	$08,065,%00000000	INY
	FCB	$09,055,%00000000	DEY
	FCB	$1C,032,%00001110	BSET ,y
	FCB	$1D,015,%00001110	BCLR ,y
	FCB	$1E,031,%00011110	BRSET ,y
	FCB	$1F,029,%00011110	BRCLR ,y
	FCB	$30,131,%00000000	TSY
	FCB	$35,133,%00000000	TYS
	FCB	$38,096,%00000000	PULY
	FCB	$3A,003,%00000000	ABY
	FCB	$3C,092,%00000000	PSHY
	FCB	$60,083,%00000110	NEG ,y
	FCB	$63,043,%00000110	COM ,y
	FCB	$64,078,%00000110	LSR ,y
	FCB	$66,100,%00000110	ROR ,y
	FCB	$67,011,%00000110	ASR ,y
	FCB	$68,074,%00000110	LSL ,y
	FCB	$69,097,%00000110	ROL ,y
	FCB	$6A,050,%00000110	DEC ,y
	FCB	$6C,060,%00000110	INC ,y
	FCB	$6D,127,%00000110	TST ,y
	FCB	$6E,066,%00000110	JMP ,y
	FCB	$6F,039,%00000110	CLR ,y
	FCB	$8C,048,%00000010	CPY #
	FCB	$8F,136,%00000000	XGDY
	FCB	$9C,048,%00000011	CPY dir
	FCB	$A0,118,%00000110	SUBA ,y
	FCB	$A1,041,%00000110	CMPA ,y
	FCB	$A2,106,%00000110	SPCA ,y
	FCB	$A3,120,%00000110	SUBD ,y
	FCB	$A4,009,%00000110	ANDA ,y
	FCB	$A5,139,%00000110	BITA ,y
	FCB	$A6,068,%00000110	LDAA ,y
	FCB	$A7,111,%00000110	STAA ,y
	FCB	$A8,056,%00000110	EORA ,y
	FCB	$A9,004,%00000110	ADCA ,y
	FCB	$AA,087,%00000110	ORAA ,y
	FCB	$AB,006,%00000110	ADDA ,y
	FCB	$AC,048,%00000110	CPY ,y
	FCB	$AD,067,%00000110	JSR ,y
	FCB	$AE,071,%00000110	LDS ,y
	FCB	$AF,115,%00000110	STS ,y
	FCB	$BC,048,%00000100	CPY ext
	FCB	$CE,073,%00000010	LDY #
	FCB	$DE,073,%00000011	LDY dir
	FCB	$DF,117,%00000011	STY dir
	FCB	$E0,119,%00000110	SUBB ,y
	FCB	$E1,042,%00000110	CMPB ,y
	FCB	$E2,107,%00000110	SPCB ,y
	FCB	$E3,008,%00000110	ADDD ,y
	FCB	$E4,010,%00000110	ANDB ,y
	FCB	$E5,140,%00000110	BITB ,y
	FCB	$E6,069,%00000110	LDAB ,y
	FCB	$E7,112,%00000110	STAB ,y
	FCB	$E8,057,%00000110	EORB ,y
	FCB	$E9,005,%00000110	ADCB ,y
	FCB	$EA,088,%00000110	ORAB ,y
	FCB	$EB,007,%00000110	ADDB ,y
	FCB	$EC,070,%00000110	LDD ,y
	FCB	$ED,113,%00000110	STD ,y
	FCB	$EE,073,%00000110	LDY ,y
	FCB	$EF,117,%00000110	STY ,y
	FCB	$FE,073,%00000100	LDY ext
	FCB	$FF,117,%00000100	STY ext
	FCB	0,%00000000		Invalid
* Shifted opcode table 2 ($1A)
OPTAB2	FCB	$83,046,%00000010	CPD #
	FCB	$93,046,%00000011	CPD dir
	FCB	$A3,046,%00000101	CPD ,x
	FCB	$AC,048,%00000101	CPY ,x
	FCB	$B3,046,%00000100	CPD ext
	FCB	$EE,073,%00000101	LDY ,x
	FCB	$EF,117,%00000101	STY ,x
	FCB	0,%00000000		Invalid
* Shifted opcode table 3 ($CD)
OPTAB3	FCB	$A3,046,%00000110	CPD ,y
	FCB	$AC,047,%00000110	CPX ,y
	FCB	$EE,072,%00000110	LDX ,y
	FCB	$EF,116,%00000110	STX ,y
	FCB	0,%00000000		Invalid
*
* Disassembler instruction text table
*
INSTXT	FCC	'????'		0
	FCC	'ABA '		1
	FCC	'ABX '		2
	FCC	'ABY '		3
	FCC	'ADCA'		4
	FCC	'ADCB'		5
	FCC	'ADDA'		6
	FCC	'ADDB'		7
	FCC	'ADDD'		8
	FCC	'ANDA'		9
	FCC	'ANDB'		10
	FCC	'ASR '		11
	FCC	'ASRA'		12
	FCC	'ASRB'		13
	FCC	'BCC '		14
	FCC	'BCLR'		15
	FCC	'BCS '		16
	FCC	'BEQ '		17
	FCC	'BGE '		18
	FCC	'BGT '		19
	FCC	'BHI '		20
	FCC	'BIT '		21
	FCC	'BLE '		22
	FCC	'BLS '		23
	FCC	'BLT '		24
	FCC	'BMI '		25
	FCC	'BNE '		26
	FCC	'BPL '		27
	FCC	'BRA '		28
	FCC	'BRCL'		29
	FCC	'BRN '		30
	FCC	'BRSE'		31
	FCC	'BSET'		32
	FCC	'BSR '		33
	FCC	'BVC '		34
	FCC	'BVS '		35
	FCC	'CBA '		36
	FCC	'CLC '		37
	FCC	'CLI '		38
	FCC	'CLR '		39
	FCC	'CLV '		40
	FCC	'CMPA'		41
	FCC	'CMPB'		42
	FCC	'COM '		43
	FCC	'COMA'		44
	FCC	'COMB'		45
	FCC	'CPD '		46
	FCC	'CPX '		47
	FCC	'CPY '		48
	FCC	'DAA '		49
	FCC	'DEC '		50
	FCC	'DECA'		51
	FCC	'DECB'		52
	FCC	'DES '		53
	FCC	'DEX '		54
	FCC	'DEY '		55
	FCC	'EORA'		56
	FCC	'EORB'		57
	FCC	'FDIV'		58
	FCC	'IDIV'		59
	FCC	'INC '		60
	FCC	'INCA'		61
	FCC	'INCB'		62
	FCC	'INS '		63
	FCC	'INX '		64
	FCC	'INY '		65
	FCC	'JMP '		66
	FCC	'JSR '		67
	FCC	'LDAA'		68
	FCC	'LDAB'		69
	FCC	'LDD '		70
	FCC	'LDS '		71
	FCC	'LDX '		72
	FCC	'LDY '		73
	FCC	'LSL '		74
	FCC	'LSLA'		75
	FCC	'LSLB'		76
	FCC	'LSLD'		77
	FCC	'LSR '		78
	FCC	'LSRA'		79
	FCC	'LSRB'		80
	FCC	'LSRD'		81
	FCC	'MUL '		82
	FCC	'NEG '		83
	FCC	'NEGA'		84
	FCC	'NEGB'		85
	FCC	'NOP '		86
	FCC	'ORAA'		87
	FCC	'ORAB'		88
	FCC	'PSHA'		89
	FCC	'PSHB'		90
	FCC	'PSHX'		91
	FCC	'PSHY'		92
	FCC	'PULA'		93
	FCC	'PULB'		94
	FCC	'PULX'		95
	FCC	'PULY'		96
	FCC	'ROL '		97
	FCC	'ROLA'		98
	FCC	'ROLB'		99
	FCC	'ROR '		100
	FCC	'RORA'		101
	FCC	'RORB'		102
	FCC	'RTI '		103
	FCC	'RTS '		104
	FCC	'SBA '		105
	FCC	'SBCA'		106
	FCC	'SBCB'		107
	FCC	'SEC '		108
	FCC	'SEI '		109
	FCC	'SEV '		110
	FCC	'STAA'		111
	FCC	'STAB'		112
	FCC	'STD '		113
	FCC	'STOP'		114
	FCC	'STS '		115
	FCC	'STX '		116
	FCC	'STY '		117
	FCC	'SUBA'		118
	FCC	'SUBB'		119
	FCC	'SUBD'		120
	FCC	'SWI '		121
	FCC	'TAB '		122
	FCC	'TAP '		123
	FCC	'TBA '		124
	FCC	'TEST'		125
	FCC	'TPA '		126
	FCC	'TST '		127
	FCC	'TSTA'		128
	FCC	'TSTB'		129
	FCC	'TSX '		130
	FCC	'TSY '		131
	FCC	'TXS '		132
	FCC	'TYS '		133
	FCC	'WAI '		134
	FCC	'XGDX'		135
	FCC	'XGDY'		136
	FCC	'CLRA'		137
	FCC	'CLRB'		138
	FCC	'BITA'		139
	FCC	'BITB'		140
*
* Debug routine
*
DEBUG	PSHB			Save B
	PSHA			Save A
	PSHX			Save X
	PSHY			Save Y
	TSY			Address stack
	LDX	6,Y		Get RETURN address
	JSR	WRSTR		Output string
	STX	6,Y		Write
	LDX	#DEBMSG		Point to strings
	JSR	WRSTR		Introduce D
	LDD	4,Y		Get D
	JSR	WRWORD		Display
	JSR	WRSTR		Introduce X
	LDD	2,Y		Get X
	JSR	WRWORD		Display
	JSR	WRSTR		Introduce Y
	LDD	,Y		Get Y
	JSR	WRWORD		Display
	JSR	WRSTR		Closing arguments
	PULY			Restore Y
	PULX			Restore X
	PULA			Restore A
	PULB			Restore B
	RTS
DEBMSG	FCCZ	': D='
	FCCZ	' X='
	FCCZ	' Y-'
	FCB	$0A,$0D,0
*
* Interrupt handlers
*
FFD6	CLRB			SCI
	BRA	REVECT
FFD8	LDAB	#1		SPI
	BRA	REVECT
FFDA	LDAB	#2		Pulse ACC Input
	BRA	REVECT
FFDC	LDAB	#3		Pulse ACC overflow
	BRA	REVECT
FFDE	LDAB	#4		Timer overflow
	BRA	REVECT
FFE0	LDAB	#5		Timer IC4/OC5
	BRA	REVECT
FFE2	LDAB	#6		Timer out comp 4
	BRA	REVECT
FFE4	LDAB	#7		Timer out comp 3
	BRA	REVECT
FFE6	LDAB	#8		Timer out comp 2
	BRA	REVECT
FFE8	LDAB	#9		Timer out comp 1
	BRA	REVECT
FFEA	LDAB	#10		Timer inp comp 3
	BRA	REVECT
FFEC	LDAB	#11		Timer inp comp 2
	BRA	REVECT
FFEE	LDAB	#12		Timer inp comp 1
	BRA	REVECT
FFF0	LDAB	#13		Real-Time
	BRA	REVECT
FFF2	LDAB	#14		IRQ
	BRA	REVECT
FFF4	LDAB	#15		XIRQ
	BRA	REVECT
FFF6	LDAB	#16		SWI
	BRA	REVECT
FFF8	LDAB	#17		Illegal opcode
*
* ReVector interrupt
*
REVECT	LDX	#INTVEC		Point to vectors
	ABX			Offset to vector
	ABX			Offset to vector
	LDX	,X		Get address
	BEQ	UNEXINT		Unexpected interrupt
	PSHX			Stack for return
	TSX			Point to stack
	LDAB	3,X		Restore B
	LDX	5,X		Restore X
	RTS			Execute handler
* Report an unexpected interrupt
UNEXINT	STAB	TEMP		Save interrupt number
	PULA			Get CCR
	STAA	SAVCC		Set user CC
	PULB			Get user B
	PULA			Get user A
	STD	SAVA		Set user D
	PULX			Get X
	STX	SAVX		Set user X
	PULX			Get Y
	STX	SAVY		Set user Y
	PULX			Get PC
	STX	SAVPC		Set user PC
	STS	SAVSP		Set user SP
	LDS	#STACK		Switch to internal stack
	JSR	WRMSG		Output message
	FCCZ	'**Unexpected interrupt: '
	LDAA	TEMP		Get interrupt number
	TAB			Save for later
	JSR	WRBYTE		Display number
	BSR	DINTNAM		Display name
	JMP	SWI3		Re-enter monitor
* Display interrupt name
DINTNAM	JSR	WRSPACE		Output space
	PSHB			Save number
	LDX	#INTNTAB	Point to table
	TSTB			We have it?
	BEQ	DINTN2		Yes, this is it
DINTN1	LDAA	,X		Get char
	INX			Advance
	TSTA			At end?
	BNE	DINTN1		No, keep looking
	DECB			Reduce count
	BNE	DINTN1		Keep going
DINTN2	PULB			Restore number
	JMP	WRSTR		Output interrupt name
* Table of interrupt names
INTNTAB	FCCZ	'SCI'
	FCCZ	'SPI'
	FCCZ	'PULSE ACC INP'
	FCCZ	'PULSE ACC OVFL'
	FCCZ	'TIMER OVFL'
	FCCZ	'TIMER IC4/OC5'
	FCCZ	'TIMER OCMP 4'
	FCCZ	'TIMER OCMP 3'
	FCCZ	'TIMER OCMP 2'
	FCCZ	'TIMER OCMP 1'
	FCCZ	'TIMER ICAP 3'
	FCCZ	'TIMER ICAP 2'
	FCCZ	'TIMER ICAP 1'
	FCCZ	'REAL TIME'
	FCCZ	'IRQ'
	FCCZ	'XIRQ'
	FCCZ	'SWI'
	FCCZ	'BAD OPCODE'
*
* Load a record in HEX format
*
DLREC	LDAA	IOCTL		Get I/O control byte
	PSHA			Save for later
	ORAA	#%10000000	Inhibit output
	STAA	IOCTL		Set value
DLREC1	JSR	RDCHR		Read character
	CMPA	#':'		Intel format?
	BEQ	DLINT		Yes, download
	CMPA	#'S'		Motorola format?
	BNE	DLREC1		No, keep looking
* Download record in MOTOROLA format
DLMOT	JSR	RDCHR		Read type byte
	BEQ	DLMOT		Wait for char
	CMPA	#'0'		Header record?
	BEQ	DLREC1		Skip it
	CMPA	#'9'		End of file?
	BEQ	DLEOF		Yes, handle it
	CMPA	#'1'		Data record?
	BNE	DLERR		No, report error
* Get record count
	JSR	RDBYTE		Get count
	BCS	DLERR		Report error
	STAA	TEMP+1		Start checksum
	SUBA	#3		Get data length
	STAA	TEMP+2		Save for later
	TAB			B = value
	ABY			Adjust total
* Get record address
	JSR	RDWORD		Read the address
	BCS	DLERR		Report error
	PSHB			Save LOW
	PSHA			Save HIGH
	PULX			Copy to A
	ADDA	TEMP+1		Include in checksum
	ABA			Include second byte
	STAA	TEMP+1		Resave checksum
* Get data bytes
DLMOT1	JSR	RDBYTE		Get data byte
	BCS	DLERR		Report error
	STAA	,X		Write to memory
	INX			Advance to next
	ADDA	TEMP+1		Include in checksum
	STAA	TEMP+1		Resave checksum
	DEC	TEMP+2		Reduce count
	BNE	DLMOT1		Read them all
* Get checksum
	JSR	RDBYTE		Get data byte
	BCS	DLERR		Report error
	ADDA	TEMP+1		Test with checksum
	INCA			Is it OK?
	BNE	DLERR		Report error
* Download was OK
DLOK	SEC			Indicate more data
DLEOF	PULA			Restore IOCTL
	STAA	IOCTL		Resave
	RTS
* Download failed
DLERR	PULA			Restore IOCTL
	STAA	IOCTL		Resave
	JMP	ERROR		Report error
* Download record in INTEL format
DLINT	JSR	RDBYTE		Get count
	BCS	DLERR		Report error
	STAA	TEMP+1		Start checksum
	STAA	TEMP+2		Record length
	BEQ	DLEOF		End of file
	TAB			B = length
	ABY			Adjust total
* Get address
	JSR	RDWORD		Get data address
	BCS	DLERR		Report error
	PSHB			Save LOW
	PSHA			Save HIGH
	PULX			X = address
	ADDA	TEMP+1		Include checksum
	ABA			Include second
	STAA	TEMP+1		Resave checksum
* Get record type
	JSR	RDBYTE		Get type
	BCS	DLERR		Report error
	ADDA	TEMP+1		Include checksum
	STAA	TEMP+1		Resave checksum
* Get data bytes
DLINT1	JSR	RDBYTE		Get data byte
	BCS	DLERR		Report error
	STAA	,X		Write to memory
	INX			Advance
	ADDA	TEMP+1		Include checksum
	STAA	TEMP+1		Resave checksum
	DEC	TEMP+2		Reduce length
	BNE	DLINT1		Do them all
* Get checksum
	JSR	RDBYTE		Read a byte
	BCS	DLERR		Report error
	ADDA	TEMP+1		Include checksum
	BEQ	DLOK		Report success
	BRA	DLERR		Report failure
*
* Step one instruction
*
* Y=address, B=length
STEP	PSHY			Save address
* Copy instruction into buffer area
	LDX	#BUFFER		Point to buffer
	LDAA	,X		Get disassembled instruction
	CMPA	#'?'		Unknown?
	BNE	STEP1		No, Its OK
	JSR	WRMSG		Output message
	FCCZ	'Bad Opcode'	Message
	JMP	ERROR		Report error
STEP1	LDAA	,Y		Read byte from opcode
	STAA	,X		Write to buffer
	INY			Advance source
	INX			Advance dest
	DECB			Reduce count
	BNE	STEP1		Copy them all
	STY	SAVPC		Set new address
* Append a 'JMP CSTEP'
	LDAA	#$7E		'JMP' opcode
	STAA	,X		Write it
	LDD	#CSTEP		Get address
	STD	1,X		Write it
* Read opcode and process
	PULY			Restore address
	LDD	,Y		Get opcode
	CMPA	#$18		Prefix?
	BNE	SBRA		No, its not
	INY			Skip prefix
* Test for simple relative branches
SBRA	CMPA	#$12		BRSET dir?
	BEQ	SBRA1		Do it
	CMPA	#$13		BRCLR dir?
	BEQ	SBRA1		Do it
	CMPA	#$1E		BRSET ,x?
	BEQ	SBRA1		Do it
	CMPA	#$1F		BRSET ,x?
	BEQ	SBRA1		Do it
	CPD	#$181E		BRSET ,y
	BEQ	SBRA1		Do it
	CPD	#$181F		BRCLR ,y
	BEQ	SBRA1		Do it
	CMPA	#$20		< BRA?
	BLO	SJMP		Not branch
	CMPA	#$2F		> BLE?
	BHI	SJMP		Not branch
SBRA1	DEX			Backup to address
	LDAA	,X		Get old offset
	PSHA			Save for later
	LDD	#$015C		1 and 'INCB'
	STD	,X		Write to buffer
	LDD	#$3900		'RTS' and 0
	STAA	2,X		Write RTS
	LDX	SAVX		Incase BRCLR/BRSET
	LDY	SAVY		"" ""
	LDAA	SAVCC		Get Condition code
	ORAA	#%11010000	Mask S, X and I
	TAP			Set CCR
	JSR	BUFFER		Perform test
	TSTB			Did it happen?
	PULB			Get offset back
	BNE	SNOJMP		No jump occured
* Jump with 8 bit PC relative offset
SBRA2	CLRA			Zero
	TSTB			Test sign
	BPL	SBRA3		Positive
	COMA			Negative
* Jump with 16 bit PC relative offset
SBRA3	ADDD	SAVPC		Calculate offset
* Save new program counter
SNEWPC	STD	SAVPC		Resave
SNOJMP	RTS			All done
* JMP & JSR ,x
SJMP	CMPA	#$6E		JMP ,X?
	BEQ	SJMP1		Yes, process
	CMPA	#$AD		JSR ,X?
	BNE	SJMP2		No, try next
	BSR	PSHPC		Stack PC
SJMP1	LDAB	1,Y		Get offset
	CLRA			Zero high
	ADDD	SAVX		Calculate address
	BRA	SNEWPC		Set new PC
* JMP and JSR ext
SJMP2	CMPA	#$7E		JMP ext?
	BEQ	SJMP3		Yes, handle it
	CMPA	#$BD		JSR ext?
	BNE	SJMP4		No, try next
	BSR	PSHPC		Stack PC
SJMP3	LDD	1,Y		Get new address
	BRA	SNEWPC		Set new PC
* JMP and JSR ,y
SJMP4	CPD	#$186E		JMP ,y?
	BEQ	SJMP5		Yes, handle it
	CPD	#$18AE		JSR ,y?
	BNE	SJMP6		No, try next
	BSR	PSHPC		Stack PC
SJMP5	LDAB	1,Y		Get offset
	CLRA			Zero high
	ADDD	SAVY		Calculate address
	BRA	SNEWPC		Set new PC
* JSR dir
SJMP6	CMPA	#$9D		JSR dir?
	BNE	SBSR		No, try next
	BSR	PSHPC		Stack current PC
	LDAB	1,Y		Get address
	CLRA			Zero high
	JMP	SNEWPC		Set new PC
* BSR
SBSR	CMPA	#$8D		BSR?
	BNE	SRTS		No, try next
	BSR	PSHPC		Stack PC
	LDAB	1,Y		Get relative offset
	BRA	SBRA2		Perform relative branch
* RTS
SRTS	CMPA	#$39		RTS?
	BNE	SNORMAL		No, try next
	LDX	SAVSP		Get user SP
	INX			Advance
	LDD	,X		Get HIGH
	INX			Advance
	STX	SAVSP		Resave user SP
	BRA	SNEWPC		Set new PC
* Normal instruction, set up registers & execute
SNORMAL	STS	TEMP+1		Save monitor stack
	LDS	SAVSP		Get user stack
	LDX	#BUFFER		Addres to execute
	PSHX			Save it
	LDX	SAVY		Get user Y
	PSHX			Save it
	LDX	SAVX		Get user X
	PSHX			Save it
	LDD	SAVA		Get user A,B
	PSHA			Save user B
	PSHB			Save user A
	LDAA	SAVCC		Get user CC
	PSHA			Save it
	RTI			Execute user program
*
* Stack progam counter
*
PSHPC	LDX	SAVSP		Get user stack
	LDD	SAVPC		Get
	DEX			Release space
	STD	,X		Save data
	DEX			Release space
	STX	SAVSP		Resave user stack
	RTS
*
* Continue from a single stepped instruction
*
CSTEP	PSHA			Save A
	TPA			Get CC
	STAA	SAVCC		Save CC
	PULA			Restore A
	STD	SAVA		Save A+B
	STX	SAVX		Save user X
	STY	SAVY		Save user Y
	STS	SAVSP		Save user SP
	LDS	TEMP+1		Restore monitor stack
	RTS			Execute buffered instruction
*
* Help request
*
HELP	LDX	#HLPTXT		Point to help text
HELP1	LDAB	#25		Column counter
HELP2	LDAA	,X		Get character
	INX			Advance
	CMPA	#'|'		Separator?
	BEQ	HELP3		Yes, process
	TSTA			End of line?
	BEQ	HELP4		Exit this line
	JSR	WRCHR		Output character
	DECB			Reduce count
	BRA	HELP2		And continue
HELP3	JSR	WRSPACE		Space over
	DECB			Reduce count
	BNE	HELP3		Continue
	JSR	WRMSG		Output message
	FCCZ	'- '		Separator
	BRA	HELP2		Keep displaying
HELP4	JSR	WRLFCR		New line
	JSR	CHKESC		Test for ESCAPE exit
	LDAA	,X		More data?
	BPL	HELP1		Do it all
	JMP	NXTCMD		Get next command
*
* Help information
*
HLPTXT	FCB	0		New line to start
	FCCZ	'B 0-7 <addr>|Set breakpoint (0000=remove)'
	FCCZ	'CR <reg> <data>|Change register'
	FCCZ	'CV <vec> <addr>|Change interrupt vector'
	FCCZ	'DB|Display breakpoints'
	FCCZ	'DI <addr>,<addr>|Display memory in assembly format'
	FCCZ	'DM <addr>,<addr>|Display memory in hex dump format'
	FCCZ	'DR|Display processor registers'
	FCCZ	'DV|Display interrupt vectors'
	FCCZ	'E <addr>|Edit memory'
	FCCZ	'FM <addr>,<addr> <data>|Fill memory'
	FCCZ	'G [<addr>]|Go (execute program)'
	FCCZ	'L|Load an image into memory'
	FCCZ	'MM <addr>,<addr> <addr>|Move memory'
	FCCZ	'MT <addr>,<addr>|Memory test'
	FCCZ	'RR <addr>|Repeating READ access'
	FCCZ	'RW <addr> <data>|Repeating WRITE access'
	FCCZ	'S [<addr>]|Single step execution'
	FCCZ	'W <addr> <data>|Write to memory'
	FCCZ	'+ <value>+<value>|Hexidecimal addition'
	FCCZ	'- <value>-<value>|Hexidecimal subtraction'
	FCB	-1		End of list
*
* Interrupt vectors
*
	ORG	ROM+$1FD6
	FDB	FFD6		SCI
	FDB	FFD8		SPI
	FDB	FFDA		Pulse ACC Input
	FDB	FFDC		Pulse ACC overflow
	FDB	FFDE		Timer overflow
	FDB	FFE0		Timer IC4/OC5
	FDB	FFE2		Timer out comp 4
	FDB	FFE4		Timer out comp 3
	FDB	FFE6		Timer out comp 2
	FDB	FFE8		Timer out comp 1
	FDB	FFEA		Timer inp comp 3
	FDB	FFEC		Timer inp comp 2
	FDB	FFEE		Timer inp comp 1
	FDB	FFF0		Real-Time
	FDB	FFF2		IRQ
	FDB	FFF4		XIRQ
	FDB	SWIHND		SWI (Breakpoint)
	FDB	FFF8		Illegal opcode
	FDB	BEGIN		FFFA COP failure
	FDB	BEGIN		FFFC CLOCK monitor
	FDB	BEGIN		FFFE Reset
