*
* MON12: A software debug monitor for the 68HC12
*
* This monitor is setup for use on a 68HC812A4, which is installed in the
* Motorola 68HC12 EVB, and is configured to run out of RAM under the D-Bug12
* monitor supplied on the EVB.
*
* Search for the string '***' to find comments which highlite areas which
* may require changes when porting to another device or memory configuration.
*
* ?COPY.TXT 1996-2007 Dave Dunfield
*  -- see COPY.TXT --.
*
*** Memory/Addressing setup
REGS	EQU	0		68HC12 internal registers
MONCODE	EQU	$6800		MON12 Program code
MONDATA	EQU	$0960		MON12 data area (160 bytes)
USERPC	EQU	$4000		Default user PC
USERSP	EQU	MONDATA		Default user SP (right below mon12 data)
*** Monitor features setup
NBRK	EQU	8		Number of breakpoints supported
NUMINT	EQU	31		Number of interrupt vectors handled
* Monitor internal RAM memory area (160 bytes required)
	ORG	MONDATA
INTVEC	DS	NUMINT*2	Interrupt vector table
* Saved registers from applicaton
SAVD	DS	2		Saved A:B
SAVX	DS	2		Saved X
SAVY	DS	2		Saved Y
SAVSP	DS	2		Saved SP
SAVPC	DS	2		Saved PC
SAVCC	DS	1		Saved CC
* Disassembler control variables
ADDR	DS	2		Address of disassembled instruction
INSTPTR	DS	2		Point to display text
VTABLE	DS	8		Table of extracted operand values
VINDEX	DS	1		Index into table
* Misc variables
* IOCTL bits: 7=InhibitOutput, 6=Pauseing, 5=FlushOnError
IOCTL	DS	1		I/O Control byte
BRKPTS	DS	NBRK*3		Breakpoint addresses
TEMP	DS	3		Temporary storage
STBUFF	DS	8		Single-step copy buffer
STACK	EQU	MONDATA+$A0	Set stack here
*
	ORG	MONCODE	
* Monitor code starts here:
BEGIN	LDS	#STACK		Set up stack
*
*** Patch D-Bug12 vector table to gain control of interrupts
*
* This is required only when testing MON12 under D-Bug12, the monitor
* provided by Motorola with the 68HC12 EVB. If not using D-Bug12, replace
* this block with code to disable the COP, initialize the chip selects,
* serial port and other on-chip peripherals in a manner appropriate to
* your target system hardware configuration.
*
	LDD	#28		TRAP, D-Bug12's highest interrupt
	LDX	#FFF8		Start with our TRAP handler
ptdbug	PSHD			Save vector number
	PSHX			Pass address on stack
	JSR	[$FE1A-*-4,PC]	Execute via D-Bug12 function vector
	PULX			Restore X
	PULD			Restore D
	LEAX	4,X		Handlers are 4 bytes in size
	DECB			Reduce interrupt number
	CMPB	#7		Lowest documented interrupt in D-Bug12
	BHS	ptdbug		Do them all (End of D-Bug12 setup block)
* Zero monitor data zrea
	LDX	#MONDATA	Point to data area
beg1	CLR	1,X+		Zero one byte
	CPX	#TEMP		At end?
	BLO	beg1		Do them all
* Set defaults for user registers
	LDD	#USERSP		Get user SP
	STD	SAVSP		Set initial SP
	LDD	#USERPC		Get user Code
	STD	SAVPC		Set default PC
	TPA			Get reasonable CC
	STAA	SAVCC		Set default CC
* Patch our SWI vector to perform breakpoint processing
	LDD	#BREAKPT	Point to breakpoint handler
	STD	INTVEC+6	Install it
* Issue welcome message
	JSR	WRMSG		Output message
	FCC	'MON12 version 1.0'
	FCB	$0A,$0D,$0A
	FCC	'?COPY.TXT 1996-2007 Dave Dunfield'
	FCB	$0A,$0D
	FCC	' -- see COPY.TXT --.'
	FCB	$0A,0
* Newline and prompt for command
NEXTCMD	CLR	IOCTL		Reset in case error
	JSR	WRLFCR		New line
NXTCMD1	LDS	#STACK		Reset stack
	JSR	WRMSG		Output message
	FCCZ	'* '		Prompt
	CLRB			Zero second char
	STAB	TEMP		Zero partial match flag
nxtcmd2	JSR	RDCHR		Get character
	LDX	#CMDTAB		Point to command table
nxtcmd3	LDY	2,X+		Get address
	BEQ	nxtcmd5		Not found
	CPD	1,X+		Does it match?
	BEQ	nxtcmd4		Yes, we found it
	CMPA	1,X+		Partial match?
	BNE	nxtcmd3		No, don't set
	DEC	TEMP		Indicate partial match
	BRA	nxtcmd3		And proceed
nxtcmd4	JSR	WRSPACE		Issue a space
	JSR	,Y		Execute command
	BRA	NEXTCMD
nxtcmd5	TSTB			Have we already done two?
	BNE	ERROR		Yes, it's an error
	LDAB	TEMP		Partial match?
	TFR	A,B		Set low digit
	BNE	nxtcmd2		Get next char
* Issue error message & abort
ERROR	BCLR	IOCTL;%11011111	Clear all but FLUSH
	JSR	WRMSG		Output message
	FCCZ	' ?'		Message to output
	BRCLR	IOCTL;%00100000;NEXTCMD No flush
FLUSH	LDX	#0		Loop 1000 times
error2	JSR	CHKCHR		Check for char
	BNE	FLUSH		Reset if found
	DEX			Reduce count
	BNE	error2		And continue
	BRA	NEXTCMD		And re-prompt
*
* Edit memory
*
EDIT	JSR	REWORD		Get memory address
	TFR	D,Y		Y = address
edit1	JSR	WRLFCR		New line
	TFR	Y,D		Get address
	JSR	WRWORD		Output
	CLRB			Zero count
edit2	JSR	WRSPACE		Space over
	LDAA	,Y		Get data
	JSR	WRBYTE		Output
	LDAA	#'-'		Prompt
	JSR	WRCHR		Output
	JSR	RDBYTE		Get data
	BCS	edit4		Bad, stop
	STAA	,Y		Save data
edit3	INY			Advance Y
	INCB			Advance B
	BITB	#%00000111	Are we over 8
	BNE	edit2		No, its OK
	BRA	edit1		Start new line
edit4	CMPA	#$1B		End of edit?
	BEQ	gorts1		Yes, quit
	CMPA	#$0D		End of edit?
	BEQ	gorts1		Yes, quit
	CMPA	#$08		Backup
	BNE	edit5		No, it's not
	DEY			Backup
	BRA	edit1		And display new line
edit5	CMPA	#' '		Space?
	BNE	ERROR		No, report error
	JSR	WRSPACE		Space over
	JSR	WRSPACE		Space over
	BRA	edit3		And continue
*
* Dump memory
*
DUMP	JSR	RDADDR		Get address
dump1	JSR	WRLFCR		New line
	TFR	X,D		Get address
	JSR	WRWORD		Output
	LDAB	#16		16 bytes/line
dump2	BITB	#%00000011	At space point?
	BNE	dump3		No, don't space
	JSR	WRSPACE		Space over
dump3	JSR	WRSPACE		Output space
	LDAA	1,X+		Get char
	JSR	WRBYTE		Output
	DECB			Reduce count
	BNE	dump2		And proceed
	JSR	WRSPACE		Space over
	JSR	WRSPACE		Space over
	LEAX	-16,X		Backup
	LDAB	#16		16 bytes/line
dump4	LDAA	1,X+		Get char
	BSR	WRPRINT		Display as printable
	DECB			Reduce count
	BNE	dump4		Do them all
	CPX	TEMP+1		Are we at end?
	BLS	dump1		Output another line
gorts1	RTS
* Display A as printable character
WRPRINT	CMPA	#' '		Printable?
	BLO	wrpr1		No
	CMPA	#$7F		Printable
	BLS	wrpr2		Yes, its OK
wrpr1	LDAA	#'.'		Convert
wrpr2	JMP	WRCHR		Output
*
* Disassemble memory
*
DISMEM	JSR	RDADDR		Get an address range
dism1	JSR	WRLFCR		New line
	JSR	DISASM		Disassemble opcode
	PSHX			Save ending address
	JSR	DISPASM		Display
	PULX			Restore
	CPX	TEMP+1		Are we at end?
	BLS	dism1		Proceed
	RTS
*
* Change register
*
CHGREG	JSR	RDCHR		Get character
	LDY	#SAVD		Point to registers
	LDX	#CHRTAB		Point to table
chreg1	CMPA	1,X+		Is this the one?
	BEQ	chreg2		Yes, it is
	LDAB	1,X+		Get length
	BMI	GOERR1		End of list
	ABY			Advance pointer
	BRA	chreg1		And proceed
chreg2	JSR	WRSPACE		Space over
	BRCLR	,X;1;chreg3	Change WORD
	JSR	REBYTE		Get value
	STAA	,Y		Write it
	RTS
chreg3	JSR	REWORD		Get value
	STD	,Y		Write it
chreg4	RTS
*
* Display CPU registers
*
DISREG	LDX	#REGTXT		Point to register text
	LDY	#SAVD		Point to first register
* Output 16 bit registers
disr1	JSR	WRSTR		Output string
	LDD	2,Y+		Get data
	JSR	WRWORD		Output
	JSR	WRSPACE		Space to next
	CPY	#SAVCC		Are we over?
	BLO	disr1		No, do them all
* Output CC register
	JSR	WRSTR		Output string
	LDAA	,Y		Get CC
	JSR	WRBYTE		Output
	JSR	WRSTR		Output prefix
	LDAB	1,Y+		Get CC value
	LDAA	1,X+		Get bit value
disr2	LSLB			Shift into carry
	BCS	disr3		Its OK
	LDAA	#'-'		Indicate OFF
disr3	JSR	WRCHR		Output
	LDAA	1,X+		Get next
	BNE	disr2		And proceed
	LDAA	#']'		Closing brace
	JMP	WRCHR
GOERR1	JMP	ERROR
*
* Set a breakpoint
*
SETBRK	JSR	RDNIB		Get breakpoint number
	CMPA	#$08		In range?
	BHS	GOERR1		No, report error
	LDX	#BRKPTS		Point to breakpoint table
	LEAX	A,X		x1
	LSLA			x2
	LEAX	A,X		x3
	JSR	WRSPACE		Space over
	JSR	REWORD		Read a word
	STD	,X		Write it
GORTS2	RTS
*
* Change an interrupt vector
*
CHGINT	JSR	REBYTE		Get vector number
	CMPA	#NUMINT		Are we in range?
	BHS	GOERR1		No, report error
	LDX	#INTVEC		Point to vector table
	LSLA			x2 for word entries
	LEAX	A,X		Offset to table
	JSR	WRSPACE		Output
	JSR	REWORD		Read IRQ vector
	STD	,X		Write to table
	RTS
*
* Display interrupt vectors
*
DISINT	CLRA			Begin with IRQ 0
	LDX	#INTVEC		Point to vector table
disi1	JSR	WRLFCR		New line
disi2	PSHA			Save number
	JSR	WRBYTE		Output number
	LDAA	#'-'		Separator
	JSR	WRCHR		Output
	LDD	2,X+		Get vector contents
	JSR	WRWORD		Output
	PULA			Restore A
	INCA			Advance to next
	CMPA	#NUMINT		Are we over?
	BHS	GORTS2		Quit
	BITA	#%00000011	Are we on even boundary?
	BEQ	disi1		New line
	JSR	WRSPACE		Space over
	JSR	WRSPACE		Space over
	BRA	disi2		And go again
*
* Display breakpoints
*
DISBRK	LDY	#BRKPTS		Point to breakpoints
	LDAA	#'0'		Start count
disbr1	PSHA			Save for later
	JSR	WRMSG		Output message
	FCCZ	' B'		Lead in
	LDAA	,SP		Get bp number
	JSR	WRCHR		Output
	LDAA	#'='		Indicator
	JSR	WRCHR		Output
	LDD	3,Y+		Get break value
	JSR	WRWORD		Output
	PULA			Get break indicator
	INCA			Advance
	CMPA	#'0'+NBRK	Are we past end?
	BLO	disbr1		No, keep displayng
	RTS
*
* Download to memory
*
LOAD	JSR	WRLFCR		New line
	LDY	#0		Begin with zero
LOAD1	JSR	DLREC		Get a record
	BCS	LOAD1		More.. get them
	TFR	Y,D		D = # bytes
	JSR	WRWORD		Output value
	JSR	WRMSG		Output message
	FCCZ	' Bytes'
	JMP	FLUSH		Next command
*
* Single write to memory
*
WRIMEM	JSR	REWORD		Get address
	TFR	D,X		X = address
	JSR	WRSPACE		Space over
	JSR	REBYTE		Get data to read
	STAA	,X		Write it
	RTS
*
* Repeating write to memory
*
RWRITE	JSR	REWORD		Get address
	TFR	D,X		X = address
	JSR	WRSPACE		Space over
	JSR	REBYTE		Get data to write
	TFR	A,B		Save for later
	JSR	WRLFCR		New line
rwrite1	STAB	,X		Write to memory
	JSR	CHKCHR		Check for abort
	CMPA	#$1B		Escape?
	BNE	rwrite1		No, keep going
	JMP	NXTCMD1		No new line
*
* Repeading read of meory
*
RREAD	JSR	REWORD		Get address
	TFR	D,X		X = address
	JSR	WRLFCR		New line
rread1	LDAA	,X		Read memory
	JSR	CHKCHR		Test for end
	CMPA	#$1B		Keyboard abort?
	BNE	rread1		Keep going
	JMP	NXTCMD1		No new line
*
* Fill memory
*
FILLMEM	JSR	RDADDR		Get address range
	JSR	WRSPACE		Space over
	JSR	REBYTE		Get data to write
fill1	STAA	1,X+		Write to memory
	CPX	TEMP+1		Are we finished?
	BLS	fill1		No, do them all
	RTS
*
* Move memory
*
MOVEMEM	JSR	RDADDR		Get address range
	JSR	WRSPACE		Space over
	JSR	REWORD		Get destination
	TFR	D,Y		Y = Destination
move1	LDAA	1,X+		Get byte from source
	STAA	1,Y+		Write to dest
	CPX	TEMP+1		Are we finished?
	BLS	move1		No, do them all
	RTS
*
* HEX Addition and subtraction
*
HEXMATH	PSHA			Save operation
	JSR	REWORD		Get first value
	TFR	D,Y		Save for later
	LDAA	,SP		Get operation
	JSR	WRCHR		Output
	JSR	REWORD		Get second
	STD	TEMP		Save
	LDAA	#'='		Get option
	JSR	WRCHR		Output
	PULA			Get operation back
	CMPA	#'-'		Subtract?
	TFR	Y,D		Back to acc
	BEQ	hexm1		Perform subtraction
	ADDD	TEMP		Perform addition
	JMP	WRWORD		Output
hexm1	SUBD	TEMP		Perform subtract
	JMP	WRWORD		Output
*
* Help request
*
HELP	LDX	#HLPTXT		Point to help text
HELP1	LDAB	#25		Column counter
HELP2	LDAA	1,X+		Get character
	BEQ	HELP4		End of line
	CMPA	#'|'		Separator?
	BEQ	HELP3		Yes, process
	JSR	WRCHR		Output character
	DECB			Reduce count
	BRA	HELP2		And continue
HELP3	JSR	WRSPACE		Space over
	DECB			Reduce count
	BNE	HELP3		Continue
	JSR	WRMSG		Output message
	FCCZ	'- '		Separator
	BRA	HELP2		Keep displaying
HELP4	JSR	WRLFCR		New line
	LDAA	,X		More data?
	BPL	HELP1		Do it all
	JMP	NXTCMD1		No new line
*
* 'G'o command, execute program
*
GO	JSR	REWORD		Get word
	STD	SAVPC		Set new go location
	JSR	WRLFCR		New line
	LDX	SAVPC		Get PC address
	JSR	DISASM		Compute instruction length
	STX	TEMP+1		Save ending address
	JSR	SSTEP1		Step 1 instruction
* Insert SWI instructoions at breakpoints
	LDY	#BRKPTS		Point to breakpoint table
insb1	LDX	3,Y+		Get breakpoint address
	BEQ	insb2		None set here
	LDAA	,X		Get opcode
	STAA	-1,Y		Save in table
	MOVB	#$3F;0,X	Write SWI to code
insb2	CPY	#BRKPTS+(NBRK*3) At end?
	BLO	insb1		Do them all
* Restore user registers & execute code
	LDX	SAVX		Get X
	LDY	SAVY		Get Y
	LDAA	SAVCC		Get Saved CC
	LDS	SAVSP		Get saved SP
	PSHA			Stash CC
	LDD	SAVD		Restore A:B
	PULC			Setup CC
	JMP	[SAVPC-*-4,PC]	Execute user code
*
* Write string(PC) to console
*
WRMSG	PSHX			Save X
	LDX	2,SP		Get PC
	BSR	WRSTR		Output string
	STX	2,SP		Reset PC
	PULX			Restore X
	RTS
*
* Write a LINE-FEED and CARRIAGE-RETURN to the console
*
WRLFCR	PSHA			Save ACC
wrlf1	BSR	CHKCHR		Check for character
	BNE	wrlf3		Handle it
	BRSET	IOCTL;%01000000;wrlf1	Halted, wait for it
wrlf2	LDAA	#$0A		Get LF
	BSR	WRCHR		Output
	LDAA	#$0D		Get CR
	BSR	WRCHR		Output
	PULA			Restore ACC
	RTS
wrlf3	CMPA	#' '		Space - allow 1 line
	BNE	wrlf4		No, try next
	BRSET	IOCTL;%01000000;wrlf2 Already set, allow 1
	BSET	IOCTL;%01000000	Inhibit output
	BRA	wrlf1		And wait
wrlf4	CMPA	#$0D		Return - Allow full
	BNE	wrlf5		No, try next
	BCLR	IOCTL;%01000000	Allow output
	BRA	wrlf2		And proceed
wrlf5	CMPA	#$1B		Escape - terminate
	BNE	wrlf1		No, wait
	JMP	NEXTCMD		New line
*
* Write a SPACE to the console
*
WRSPACE	PSHA			Save ACC
	LDAA	#' '		Get space
	BSR	WRCHR		Output
	PULA			Restore ACC
wrspc1	RTS
*
* Write string (X) to console
*
WRSTR	LDAA	1,X+		Get char from string
	BEQ	wrspc1		End of string
	BSR	WRCHR		Output
	BRA	wrstr		And proceed
*
* Write word (D) to serial port
*
WRWORD	BSR	WRBYTE		Output high
	TBA			Get low
*
* Write byte (A) to serial port
*
WRBYTE	PSHA			Save LOW
	LSRA			Shift
	LSRA			High
	LSRA			Into
	LSRA			Low
	BSR	WRNIB		Output high nibble
	PULA			Get low nibble
* Write nibble (A) to serial port
WRNIB	ANDA	#%00001111	Drop high
	ADDA	#'0'		Convert to ASCII
	CMPA	#'9'		Digit?
	BLS	WRCHR		Yes, its OK
	ADDA	#7		Convert
*
* Write character (A) to serial port
*** Change this function to write to a different serial port device.
*
WRCHR	TST	IOCTL		Check I/O control
	BMI	chkc1		No output
	BRCLR	REGS+$C4;%10000000;*
	STAA	REGS+$C7	Write character
	RTS
*
* Check for character from serial port (A=chr if found, else 0)
*** Change this function to read from a different serial port device.
*
CHKCHR	CLRA			Assume none
	BRCLR	REGS+$C4;%00100000;chkc1
	LDAA	REGS+$C7
chkc1	RTS
*
* Get character, echo & convert to UCASE
*
RDCHR	BSR	CHKCHR		Check for char
	BEQ	RDCHR		None found
	CMPA	#' '		Control code?
	BLS	chkc1		Don't echo
	BSR	WRCHR		Output
	CMPA	#'a'		Lower case?
	BLO	chkc1		No, its OK
	CMPA	#'z'		Lower case?
	BHI	chkc1		No, its OK
	ANDA	#%11011111	Convert to UCASE
	RTS
*
* Read a NIBBLE into A
*
RDNIB	BSR	RDCHR		Get character & echo
	SUBA	#'0'		In range?
	CMPA	#10		Test for OK
	BLO	rdnib1		Yes, its OK
	SUBA	#7		Convert HEX's
	CMPA	#$0A		In range?
	BLO	rdnib2		No, its invalid
	CMPA	#$0F		In range?
	BHI	rdnib2		Its invalid
RDNIB1	CLC			No carry, its OK
	RTS
* Invalid hex char - return to normal
rdnib2	ADDA	#'0'+7		Normalize character
rdnib3	SEC			Indicate failure
	RTS
*
* Read a BYTE into A
*
RDBYTE	BSR	RDNIB		Get HIGH nibble
	BCS	rdbyte2		Error
	LSLA			Move
	LSLA			into
	LSLA			high
	LSLA			nibble
	STAA	TEMP		Save for later
	BSR	RDNIB		Get LOW nibble
	BCS	rdbyte1		Error
	ORAA	TEMP		Combine
rdbyte1	RTS
rdbyte2	CMPA	#$27		Single quote?
	BNE	rdnib3		Report error
rdbyte3	BSR	CHKCHR		Get character
	BEQ	rdbyte3		Wait for it
	PSHA			Save char
	JSR	WRPRINT		Echo it
	PULA			Restore char
	CLC			Indicate success
	RTS
*
* Read a word into D
*
RDWORD	BSR	RDBYTE		Read HIGH byte
	BCS	rdword1		Check special cases
	TAB			B = HIGH
	BSR	RDBYTE		Get LOW byte
	BCS	refail		Report error
	EXG	A,B		Swap
	RTS
* Test for special case register names
rdword1	PSHX			Save X
	LDX	SAVX		Assume X
	CMPA	#'X'		Is it X?
	BEQ	rdword2		Yes, use it
	LDX	SAVY		Assume Y
	CMPA	#'Y'		Is it Y?
	BEQ	rdword2		Yes, use it
	LDX	SAVSP		Assume SP
	CMPA	#'S'		Is it SP?
	BEQ	rdword2		Yes, use it
	LDX	SAVPC		Assume PC
	CMPA	#'P'		Is it PC?
	BEQ	rdword2		Yes, use it
	PULX			Restore X
	SEC			Indicate error
	RTS
rdword2	LDAA	#'='		Separator
	JSR	WRCHR		Display it
	TFR	X,D		D = value
	PULX			Restore X
	PSHA			Save ACCA
	JSR	WRWORD		Output the value
	PULA			Restore ACC
	CLC			Indicate success
	RTS
* Version of the 'RD' functions which exit on error
REBYTE	BSR	RDBYTE		Get byte
	BCC	reok		Its OK
refail	JMP	ERROR		Report error
REWORD	BSR	RDWORD		Get word
	BCS	refail		Report error
reok	RTS
*
* Read an address range into X:(temp+1)
*
RDADDR	BSR	REWORD		Get word in D
	TFR	D,X		X = start address
	LDAA	#','		Get separator
	JSR	WRCHR		Output
	BSR	RDWORD		Get word in D
	BCC	rdaddr1		It's OK
	CMPA	#' '		Space?
	BNE	refail		Report error
	JSR	WRMSG		Output message
	FCCZ	'FFFF'
	LDD	#$FFFF		Set to FFFF
RDADDR1	STD	TEMP+1		Set ending address
	RTS
*
* Disassemble code (X), set length & values
*
DISASM	STX	ADDR		Set instruction address
	CLR	VINDEX		Set starting index
	LDAA	#1		Assume initial length
	LDY	#ITABLE1	Point to instruction table
	LDAA	1,X+		Get opcode
	CMPA	#$18		Shift opcode?
	BNE	disa1		No, its OK
	LDAA	1,X+		Get next opcode
	LDY	#ITABLE2	Use shift table
* Search instruction table for opcode
disa1	LDAB	1,Y+		Get opcode from table
disa2	CBA			Does it match?
	BEQ	disa4		We found it
disa3	LDAB	1,Y+		Get next byte
	BNE	disa3		Loop till end
	LDAB	1,Y+		Get next opcode
	BNE	disa2		Not end of table
disa4	STY	INSTPTR		Save instruction pointer
* We have opcode, process definitions
disa5	LDAB	1,Y+		Get next entry from table
	CMPB	#$09		Special entry?
	BHS	disa5		No, keep going
	LSLB			x2 for two byte entries
	CLRA			Zero high
	JMP	[D,PC]		Execute handler
	FDB	DEXIT		00 = End of string
	FDB	DSWAP		01 = Swap values & end
	FDB	DBYTE		02 = Byte value
	FDB	DWORD		03 = Word value
	FDB	DINDEX		04 = Indexed mode
	FDB	DREL8		05 = Relative 8
	FDB	DREL9		06 = Relative 9
	FDB	DREL16		07 = Relative 16
	FDB	DEXTRA		08 = Extra byte for register
* Absorb an extra byte from the instruction stream
DEXTRA	INX			Advance
	BRA	disa5		And continue
* Handle byte data
DBYTE	LDAB	1,X+		Get data byte
	CLRA			Zero high
* Store D in VTABLE[VINDEX++]
SET16	BSR	SETVAL		Save D in value table
	BRA	disa5		And continue
* Handle word data
DWORD	LDD	2,X+		Get data word
	BRA	SET16		And save it
* Handle relative 8 bit data
DREL8	LDAB	1,X+		Get offset
	SEX	B,D		Extend sign
* Calculate relative address from (D,X)
SETREL	PSHX			Address value
	ADDD	2,SP+		Add to value
	BRA	SET16		And save it
* Handle relative 9
DREL9	LDD	2,X+		Get code & offset
	ANDA	#%00010000	Is it negative?
	BEQ	SETREL		No, zero high
	LDAA	#$FF		extend
	BRA	SETREL		And save it
* Handle relative 16
DREL16	LDD	2,X+		Get offset
	BRA	SETREL		And save it
* Handle indexed addressing
DINDEX	LDAB	1,X+		Get postbyte
	CLRA			Zero high
	BSR	SETVAL		Save in value table
* Handle 5 bit indexing
	BITB	#%00100000	Test for 5 bit
	BNE	dind1		Not 5 bit
	ANDB	#%00011111	Convert to 5 bit value
	BITB	#%00010000	Is it negative?
	BEQ	SET16		No, save positive value
	COMA			Negate high value
	ORAB	#%11110000	Complete low negative bits
	BRA	SET16
* Handle other index values
dind1	ANDB	#%11100100	Mask off register & option bits
	CMPB	#%11100000	Does this mode have an offset value
	BNE	disa5		No, no further action necessary
	LDAB	1,X+		Get 1st value byte
	LDAA	-2,X		Get postbyte back
	BITA	#%00000010	16 bit indicated?
	BNE	dind2		Yes, handle it
	ANDA	#%00000001	Negative?
	BEQ	SET16		No, save positive value
	LDAA	#$FF		Set negative
	BRA	SET16		And save it
dind2	TFR	B,A		Set high byte
	LDAB	1,X+		Get low byte
	BRA	SET16		And save it
* Save 16 bit value (D) on the VTABLE stack
SETVAL	PSHX			Save pointer
	PSHY			Save Y
	TFR	D,Y		Y = value
	LDX	#VTABLE		X = table
	LDAB	VINDEX		Get index
	LSLB			x2
	STY	B,X		Save it
	INC	VINDEX		Advance
	TFR	Y,D		Restore value
	PULY			Restore Y
	PULX			Restore X
	RTS
* Swap last argument in VTABLE to first, and stop processing
DSWAP	STY	INSTPTR		Set new instruction pointer
	PSHX			Save offset
	LDX	#VTABLE		Point to VTABLE
	LDAB	VINDEX		Get table index
	LSLB			x2
	LEAX	B,X		Offset to end of table
	LDY	2,-X		Get last value
dswap1	LDD	2,-X		Get preceeding value
	STD	2,X		Set new value
	CPX	#VTABLE		Are we at beginning
	BHI	dswap1		Keep going
	STY	,X		Set new 0 value
	PULX			Restore memory address
* Terminate processing
DEXIT	RTS
*
* Display disassembled instructions (X at end of INST)
*
DISPASM	LDD	ADDR		Get address
	JSR	WRWORD		Output address
	TFR	X,D		Get address
	SUBD	ADDR		Get length
	PSHB			Save length
	LDY	ADDR		Get address
* Display the address bytes
dispa1	JSR	WRSPACE		Extra space
	LDAA	1,Y+		Get byte
	JSR	WRBYTE		Output byte
	DECB			Reduce count
	BNE	dispa1		Do them all
* Fill with spaces
	PULB			Restore length
dispa2	JSR	WRSPACE		Output space
	JSR	WRSPACE		Output space
	JSR	WRSPACE		Output space
	INCB			Increase
	CMPB	#7		At end?
	BLO	dispa2		Do them all
* Display instruction
	LDY	INSTPTR		Y = instruction
	CLR	VINDEX		Zero table index
* Display next value from instruction table
dispa3	LDAA	1,Y+		Get char from table
	BPL	dispa5		No special
* Display special "compressed" text
	ANDA	#%01111111	Mask flag bit
	LDX	#INAMES		Point to inst names
	LEAX	A,X		x1
	LEAX	A,X		x3
	LEAX	A,X		x2
	LDAB	#3		Output 3 chars
dispa4	LDAA	1,X+		Get char
	JSR	WRCHR		Output
	DECB			Reduce count
	BNE	dispa4		Do them all
	BRA	dispa3		And proceed
* Display characters from instruction string
dispa5	CMPA	#$10		Special mode?
	BLO	dispa6		Yes, handle it
	JSR	WRCHR		Output
	BRA	dispa3		And proceed
* Display a byte value
PBYTE	BSR	GETVAL		Get value
	TFR	B,A		Get low value
pbyte1	JSR	WRBYTE		Output
	BRA	dispa3		And continue
* Display a WORD value
PWORD	BSR	GETVAL		Get value
	JSR	WRWORD		Output
	BRA	dispa3		And continue
* Display a registre from lower nibble of byte1
PREGL	LDAA	1,X		Get byte1 value
preg	LDX	#TFRREG		Point to register table
preg1	BSR	SHOWREG		Display the register
	BRA	dispa3
* Display EXG/SEX/TFR instruction
PEST	LDAA	1,X		Get byte1
	LDX	#TEST		Assume EXG
	TSTA			Test for EXG
	BMI	pest2		Yes, output
	LEAX	4,X		Advance to next (SEX)
	TAB			Save a copy
	ANDA	#%00001110	Mask lower register
	ANDB	#%01110000	Mask high register
	CMPA	#$04		In range?
	BLO	pest1		No, go to next
	CMPB	#3		In range?
	BLO	pest2		Yes, use this one
pest1	LEAX	4,X		Advance to next (TFR)
pest2	JSR	WRSTR		Output string
	BRA	dispa3
* Display opcode value
POPCD1	INX			Skip to byte1
POPCODE	LDAA	,X		Get opcode value
	BRA	pbyte1		And output
* Display register from upper nibble of byte1
PREGH	LDAA	1,X		Get byte1 value
	LSRA			Shift
	LSRA			High
	LSRA			Into
	LSRA			Low
	BRA	preg		output register
* Display DBEQ family opcode from byte1
PDBEQ	LDAA	1,X		Get byte1 value
	LSRA			Shift
	LSRA			Over
	LSRA			Leaving
	LSRA			Only high
	LSRA			3 bits (instruction)
	LDX	#DBEQT		Point to table
	BRA	preg1		Output table value
* Get 16 bit value (D) on the VTABLE stack
GETVAL	LDX	#VTABLE		X = table
	LDAB	VINDEX		Get index
	LSLB			x2
	LDD	B,X		Get value
	INC	VINDEX		Advance
PEXIT	RTS
* Show register text(X) from index in A
SHOWREG	ANDA	#%00000111	Select 0-7
	BEQ	showr2		Yes, proceed
showr1	TST	1,X+		Try next char
	BNE	showr1		Go to end
	DECA			Reduce count
	BNE	showr1		Go to end
showr2	JMP	WRSTR		Output string
* Execute special handler for inputs
dispa6	LDX	ADDR		Point to instruction
	TFR	A,B		Set low value
	LSLB			x2 for two byte entries
	CLRA			Zero high
	JMP	[D,PC]		Execute handler
	FDB	PEXIT		00 = Exit
	FDB	PEXIT		01 = Should never occur
	FDB	PBYTE		02 = Byte value
	FDB	PWORD		03 = Word value
	FDB	PINDEX		04 = Index value
	FDB	PWORD		05 = 8 bit relative
	FDB	PWORD		06 = 9 bit relative
	FDB	PWORD		07 = 16 bit relative
	FDB	PREGH		08 = Register from HIGH nibble
	FDB	PREGL		09 = Register from LOW nibble
	FDB	PDBEQ		0A = DBEQ family instruction
	FDB	PEST		0B = EXG/SEX/TFR instruction
	FDB	POPCODE		0C = Opcode value
	FDB	POPCD1		0D = Byte1 value
* Display an index value
* Locals:	0,SP = postbyte
*		1,SP = register name
*		2,SP = ACC names + indirect flag(7)
*		3,SP = +/- flag
PINDEX	LEAS	-4,SP		Reserve 2 local variables
	BSR	GETVAL		Get postbyte value
	STAB	,SP		Save postbyte
	CLR	2,SP		Zero ACC indicator
	CLR	3,SP		Zero +/- indicator
* Handle 5 bit addressing
	BITB	#%00100000	5 bit indexed?
	BNE	pind1		No, try next
	LSLD			Get register
	LSLD			Into A
	STAA	1,SP		Save register
	BRA	pind2		And proceed
pind1	LSRB			Shift
	LSRB			register
	LSRB			over
	STAB	1,SP		Save
	LDAA	,SP		Get postbyte
	ANDA	#%11100111	Drop registers
* Handle numeric offset types
	CMPA	#%11100000	n,R (8)+
	BEQ	pind2		Handle it
	CMPA	#%11100001	n,R (8)-
	BEQ	pind2		Handle it
	CMPA	#%11100010	n,R (16)
	BEQ	pind2		Handle it
	CMPA	#%11100011	[n,r] (16)
	BNE	pind3		No, try next
	LDAA	#'['		Opening brace
	JSR	WRCHR		Output
	MOVB	#$80;2,SP	Indicate indirect
pind2	BSR	GETVAL		Get value
	JSR	WRWORD		Output
	BRA	pind7		Finish
* Handle accumulator offset modes
pind3	LDAB	#'A'		Assume A offset
	CMPA	#%11100100	A,r?
	BEQ	pind4		Yes, we have it
	LDAB	#'B'		Assume B offset
	CMPA	#%11100101	B,r?
	BEQ	pind4		Yes, we have it
	LDAB	#'D'		Assume D offset
	CMPA	#%11100110	D,r?
	BEQ	pind4		Yes, we have it
	CMPA	#%11100111	[D,r]?
	BNE	pind5		No, try next
	LDAA	#'['		Opening brace
	JSR	WRCHR		Output
	LDAB	#'D'+$80	Assume D indirect
pind4	STAB	2,SP		Set acc mode
	BRA	pind7		Finish
* Handle auto-inc/dec modes
pind5	LDAB	,SP		Get postbute
	LDAA	,SP		Save a copy
	ANDA	#%00000111	Save amount
	INCA			Range 1-8
	MOVB	#'+';3,SP	Assume increment
	BITB	#%00001000	Decrement?
	BEQ	pind6		No, it's increment
	MOVB	#'-';3,SP	Set decrement indicator
	PSHA			Save value
	LDAA	#9		Get adjustment
	SUBA	1,SP+		Convert
pind6	ADDA	#'0'		Convert to ASCII
	JSR	WRCHR		Output
	LSLD			Shift over
	LSLD			Shift over
	STAA	1,SP		Save register
* Clean up and output pending data
pind7	LDAA	2,SP		Restore
	ANDA	#%01111111	ACC to output?
	BEQ	pind8		No, keep going
	JSR	WRCHR		Output
pind8	LDAA	#','		Get comma
	JSR	WRCHR		Output
	LDAA	3,SP		Get +/- indicator
	BEQ	pind9		None
	LDAB	,SP		Get postbyte back
	BITB	#%00010000	Pre/Post
	BNE	pind9		Not now
	JSR	WRCHR		Output
	CLR	3,SP		Don't do it later
pind9	LDAA	1,SP		Get register names
	ANDA	#%00000011	Mask unused values
	LDX	#IDXREG		Point to table
	JSR	SHOWREG		Output
	LDAA	3,SP		Get +/- indicator
	BEQ	pind10		None
	JSR	WRCHR		Output
pind10	LDAA	2,SP		Get IND indicator
	BPL	pind11		None
	LDAA	#']'		Get closing
	JSR	WRCHR		Output
pind11	LEAS	4,SP		Clean stack
	JMP	dispa3		Continue
*
* Single-step one instruction
*
SSTEP	LDX	SAVPC		Get PC address
	JSR	DISASM		Compute instruction length
	STX	TEMP+1		Save ending address
	JSR	DISPASM		Display instruction
* Copy instruction into temp buffer so we can abuse it
sstep1	LDD	TEMP+1		Get ending address
	SUBD	SAVPC		Compute length
	LDY	#STBUFF		Point to temp buffer
	LDX	SAVPC		Get address of instruction
sstep2	LDAA	1,X+		Get byte from inst
	STAA	1,Y+		Save in buffer
	DECB			Reduce length count
	BNE	sstep2		Copy entire instruction
* Examine opcode to determine instruction affecting program transfer
* These require emulation, as execution would give up control.
	LDAA	STBUFF		Get opcode
* Handle 'BRA' type instructions
	CMPA	#$20		Is it a BRANCH?
	BLO	stjmp1		No, try next
	CMPA	#$2F		Second test for branch?
	BHI	stjmp1		No, try next
* Test branch by calling subroutine
	LDAB	#$01		Branch offset = 1
	STAB	-1,Y		Adjust branch
STBRA	LDD	#$513D		'COMB' and 'RTS'
	STD	,Y		Set subsequent
	LDAA	SAVCC		Get condition code
	CLRB			Zero B
	TAP			Set CC
	JSR	STBUFF		Call test function
	TSTB			Was branch taken?
	BEQ	SETV0		Yes, set to address
SETNXT	LDD	TEMP+1		Get address of next instruction
set0	STD	SAVPC		Set new PC
	RTS			Return to caller
* Push following address onto stack
PUSHPC	LDX	SAVSP		Get saved SP
	LDD	TEMP+1		Get following address
	STD	2,-X		Write t stack
	STX	SAVSP		Resave SP
	RTS
* Handle JMP and JSR instructions
stjmp1	CMPA	#$06		'JMP ext'?
	BEQ	SETV0		Go to handler
	CMPA	#$05		'JSR idx'?
	BNE	stjsr1		Handle it
	BRA	stjmp4		And perform it
stjsr1	CMPA	#$17		'JSR dir'
	BEQ	stjsr2		Yes, handle it
	CMPA	#$16		'JSR dir'?
	BNE	stjsr3		No, try next
stjsr2	BSR	PUSHPC		Save next PC on stack
	BRA	SETV0		And execute jump
stjsr3	CMPA	#$15		'JSR idx'
	BNE	strts1		No, try next
	BSR	PUSHPC		Save next PC on stack
stjmp4	JSR	STINDEX		Compute index value
	BRA	set0		And proceed
* Handle 'RTS', 'RTC' and 'RTI' instructions
strts1	CMPA	#$3D		'RTS'?
	BNE	stdbeq		No, try next
	LDX	SAVSP		Get saved SP
	LDD	2,X+		Get return value
	STX	SAVSP		Resave
SETV0	LDD	VTABLE		Get VTABLE0 address
	BRA	set0		And proceed
* Handle 'DBEQ' type instructions
stdbeq	CMPA	#$04		'DBEQ' family opcode
	BNE	stcbad		No, try next
	LDAA	STBUFF+1	Get operand
	ANDA	#%11101111	Insure offset is positive
	LDAB	#2		Offset 2 to skip CLR
	STD	STBUFF+1	Resave
	LDD	#$69C3		'CLR 3,PC'
	STD	STBUFF+3	Set it
	LDD	#$06FF		Get JMP and $FF
	LDX	#stdbret	Return address
	STAA	STBUFF+5	Save JMP
	STX	STBUFF+6	Save return address
	STAB	STBUFF+8	Save value
	LDD	SAVD		Get saved D
	LDX	SAVX		Get saved X
	LDY	SAVY		Get saved Y
	STS	ADDR		Save out stack
	LDS	SAVSP		Get saved SP
	JMP	STBUFF		Execute instruction
stdbret	STS	SAVSP		Save SP
	LDS	ADDR		Retore our stack
	STY	SAVY		Resave Y
	STX	SAVX		Resave X
	STD	SAVD		Resave D
	LDAA	STBUFF+8	Get flag byte
	BNE	SETV0		Indicate taken
	JMP	SETNXT		Indicate NOT taken
* Check for "bad" opcode that cannot be stepped
STCBAD	TSTA			'BGND'
	BEQ	stbad		Don't allow
	CMPA	#$4A		'CALL'
	BEQ	stbad		Don't allow
	CMPA	#$4B		'CALL'
	BEQ	stbad		Don't allow
	CMPA	#$0A		'RTC'
	BEQ	stbad		Don't allow
	CMPA	#$0B		'RTI'
	BEQ	stbad		Don't allow
	CMPA	#$3F		'SWI'
	BEQ	stbad		Don't allow
	CMPA	#$3E		'WAI'
	BNE	STLBRA		It's OK
STBAD	JSR	WRMSG		Output message
	FCCZ	" **Can't step!"
	JMP	NEXTCMD		And proceed
* Handle long branch instructions
STLBRA	CMPA	#$18		Shift opcode?
	BNE	stmem		No, this is safe to handle
	LDAA	STBUFF+1	Get 2nd opcode
	CMPA	#$3E		'STOP'
	BEQ	STBAD		Don't allow
	CMPA	#$20		LBRA set?
	BLO	stmem		No, this is safe to handle
	CMPA	#$2F		LBRA set?
	BHI	stmem		No, this is safe to handle
	LDD	#$0001		Relative offset = 1
	STD	-2,Y		Save it
	JMP	STBRA		And perform test
* This instruction is save to execute in our local buffer...
STMEM	LDAA	#$06		'JMP ext' instruction
	STAA	,Y		Write to buffer
	LDD	#STPRET		Point to STEP return
	STD	1,Y		Write to buffer
	LDD	TEMP+1		Get ending address
	STD	SAVPC		Set new ending address
* Load CPU register and execute instruction
	LDX	SAVX		Get saved X
	LDY	SAVY		Get Saved Y
	LDAA	SAVCC		Get saved CC
	STS	ADDR		Save our stack
	LDS	SAVSP		Reset user stack
	PSHA			Save it
	LDD	SAVD		Get saved D
	PULC			Restore CC
	JMP	STBUFF		Execute memory buffer
* Return from step instructon, re-save registers
STPRET	PSHC			Save CC
	STD	SAVD		Save D
	PULA			Get A
	STS	SAVSP		Save user SP
	LDS	ADDR		Restore our stack
	STAA	SAVCC		Save CC
	STX	SAVX		Save X
	STY	SAVY		Save Y
	RTS			Return to caller
*
* Compute destination from indexed addressing mode
*
STINDEX	LDAA	VTABLE+1	Get postbyte
* Handle 5 bit offsets
	BITA	#%00100000	5 bit offset?
	BNE	stidx1		No, try next
	BSR	STIREG6		Get register
	ADDD	VTABLE+2	Adjust offset
	RTS
* Handle 9 and 16 bit offsets
stidx1	TFR	A,B		Save postbyte
	ANDA	#%11100000	Test for non-auto inc
	CMPA	#%11100000	Is it true?
	BNE	stauto		No, it's an auto inc
	ANDB	#%00000111	Keep only opertion bits
	CLRA			Zero high
	LSLB			x2
	JMP	[D,PC]		Execute handler
	FDB	stidxv		000: Nine bit (+)
	FDB	stidxv		001: Nine bit (-)
	FDB	stidxv		010: 16 bit
	FDB	stind16		011: 16 bit double indirect
	FDB	stacca		100: A accumulator offset
	FDB	staccb		101: B accumulator offset
	FDB	staccd		110: D accumulator offset
	FDB	stindd		111: D double indirect
* Perform auto-increment/decrement
stauto	LDAB	VTABLE+1	Get postbyte
	ORAB	#%11110000	Assume negative
	BITB	#%00001000	Is it negative?
	BNE	stauto1		Yes, it is
	ANDB	#%00001111	Convert back to positive
	INCB			Adjust to 1-8 value.
stauto1	SEX	B,D		Get full adjust value
	STD	INSTPTR		Save for later
	BSR	STIREG6		Get register
	BRSET	VTABLE+1;%00010000;stauto2 Post adjust
	ADDD	INSTPTR		Adjust value
	STD	,X		Resave
	RTS
stauto2	PSHD			Save value
	ADDD	INSTPTR		Adjust value
	STD	,X		Resave
	PULD			Restore value
	RTS
* Get register value into D from bits 6:7 (pointer in X)
STIREG6	LDAA	VTABLE+1	Get postbyte
	LSRA
	LSRA
	LSRA
	BRA	stir3a		And proceed
* Get register value into D from bits 3:4 (pointer in X)
STIREG3	LDAA	VTABLE+1	Get postbute
stir3a	LSRA
	LSRA
	ANDA	#%00000110	Save only register bits
	LDX	#SAVX		Point to register
	LEAX	A,X		Offset to correct one
	LDD	,X		Get value
	RTS
* 8 and 16 bit offsets
stidxv	BSR	STIREG3		Get register
	ADDD	VTABLE+2	Adjust offset
	RTS
* 16 bit double indirect
stind16	BSR	STIREG3		Get register
	ADDD	VTABLE+2	Add offset
stgoind	TFR	D,X		Address
	LDD	,X		Get address
	RTS
* A accumulator offset
stacca	BSR	STIREG3		Get register
	ADDB	SAVD		Add saved A
	ADCA	#0		Adjust
	RTS
* B accumulator offset
staccb	BSR	STIREG3		Get register
	ADDB	SAVD+1		Add saved B
	ADCA	#0		Adjust
	RTS
* D accumulator offset
staccd	BSR	STIREG3		Get register
	ADDD	SAVD		Add saved D
	RTS
* D double indirect
stindd	BSR	STIREG3		Get register
	ADDD	SAVD		Add saved D
	BRA	stgoind		Fetch address
*
* Load a record in HEX format
*
DLREC	BSET	IOCTL;%10100000	Inhibit output & enable flush
DLREC1	JSR	RDCHR		Read character
	CMPA	#':'		Intel format?
	BEQ	DLINT		Yes, download
	CMPA	#'S'		Motorola format?
	BNE	DLREC1		No, keep looking
* Download record in MOTOROLA format
DLMOT	JSR	RDCHR		Read type byte
	BEQ	DLMOT		Wait for char
	CMPA	#'0'		Header record?
	BEQ	DLREC1		Skip it
	CMPA	#'9'		End of file?
	BEQ	DLEOF		Yes, handle it
	CMPA	#'1'		Data record?
	BNE	DLERR		No, report error
* Get record count
	JSR	RDBYTE		Get count
	BCS	DLERR		Report error
	STAA	TEMP+1		Start checksum
	SUBA	#3		Get data length
	STAA	TEMP+2		Save for later
	LEAY	A,Y		Adjust total
* Get record address
	JSR	RDWORD		Read the address
	BCS	DLERR		Report error
	TFR	D,X		X = record address
	ADDA	TEMP+1		Include in checksum
	ABA			Include second byte
	STAA	TEMP+1		Resave checksum
* Get data bytes
DLMOT1	JSR	RDBYTE		Get data byte
	BCS	DLERR		Report error
	STAA	1,X+		Write to memory
	ADDA	TEMP+1		Include in checksum
	STAA	TEMP+1		Resave checksum
	DEC	TEMP+2		Reduce count
	BNE	DLMOT1		Read them all
* Get checksum
	JSR	RDBYTE		Get data byte
	BCS	DLERR		Report error
	ADDA	TEMP+1		Test with checksum
	INCA			Is it OK?
	BNE	DLERR		Report error
* Download was OK
DLOK	SEC			Indicate more data
DLEOF	MOVB	#0;IOCTL	Reset without affecting CY
	RTS
DLERR	JMP	ERROR		Indicate error
* Download record in INTEL format
DLINT	JSR	RDBYTE		Get count
	BCS	DLERR		Report error
	STAA	TEMP+1		Start checksum
	STAA	TEMP+2		Record length
	BEQ	DLEOF		End of file
	LEAY	A,Y		Adjust total length
* Get address
	JSR	RDWORD		Get data address
	BCS	DLERR		Report error
	TFR	D,X		X = Record address
	ADDA	TEMP+1		Include checksum
	ABA			Include second
	STAA	TEMP+1		Resave checksum
* Get record type
	JSR	RDBYTE		Get type
	BCS	DLERR		Report error
	ADDA	TEMP+1		Include checksum
	STAA	TEMP+1		Resave checksum
* Get data bytes
DLINT1	JSR	RDBYTE		Get data byte
	BCS	DLERR		Report error
	STAA	1,X+		Write to memory
	ADDA	TEMP+1		Include checksum
	STAA	TEMP+1		Resave checksum
	DEC	TEMP+2		Reduce length
	BNE	DLINT1		Do them all
* Get checksum
	JSR	RDBYTE		Read a byte
	BCS	DLERR		Report error
	ADDA	TEMP+1		Include checksum
	BEQ	DLOK		Report success
	BRA	DLERR		Indicate error
* Vector handlers
FFFC	LDAB	#0		Clock monitor failire
	BRA	revect
FFFA	LDAB	#1		COP timeout
	BRA	revect
FFF8	LDAB	#2		Unimplemented opcode
	BRA	revect
FFF6	LDAB	#3		SWI
	BRA	revect
FFF4	LDAB	#4		XIRQ
	BRA	revect
FFF2	LDAB	#5		IRQ
	BRA	revect
FFF0	LDAB	#6
	BRA	revect
FFEE	LDAB	#7
	BRA	revect
FFEC	LDAB	#8
	BRA	revect
FFEA	LDAB	#9
	BRA	revect
FFE8	LDAB	#10
	BRA	revect
FFE6	LDAB	#11
	BRA	revect
FFE4	LDAB	#12
	BRA	revect
FFE2	LDAB	#13
	BRA	revect
FFE0	LDAB	#14
	BRA	revect
FFDE	LDAB	#15
	BRA	revect
FFDC	LDAB	#16
	BRA	revect
FFDA	LDAB	#17
	BRA	revect
FFD8	LDAB	#18
	BRA	revect
FFD6	LDAB	#19
	BRA	revect
FFD4	LDAB	#20
	BRA	revect
FFD2	LDAB	#21
	BRA	revect
FFD0	LDAB	#22
	BRA	revect
FFCE	LDAB	#23
	BRA	revect
FFCC	LDAB	#24
	BRA	revect
FFCA	LDAB	#25
	BRA	revect
FFC8	LDAB	#26
	BRA	revect
FFC6	LDAB	#27
	BRA	revect
FFC4	LDAB	#28
	BRA	revect
FFC2	LDAB	#29
	BRA	revect
FFC0	LDAB	#30
* Handle interrupt
REVECT	LDX	#INTVEC		Point to interrupt vector table
	ABX			x1
	LDX	B,X		Point to vector
	BEQ	UNEXINT		Unexpected
	JMP	,X		Execute user handler
* This interrupt not recognized... Save registers & transfer
UNEXINT	STAB	TEMP		Save IRQ number
	PULA			Get CC
	STAA	SAVCC		Save CC
	PULD			Get D
	EXG	A,B		Swap because stacked out of order
	STD	SAVD		Save D
	PULX			Get X
	STX	SAVX		Save X
	PULY			Get Y
	STY	SAVY		Save Y
	PULX			Get PC
	STS	SAVSP		Save SP
	LDS	#STACK		Switch to our stack
	STX	SAVPC		Save it
	JSR	WRMSG		Output message
	FCCZ	'**Interrupt '	Leading message
	LDAA	TEMP		Get ID back
	JSR	WRBYTE		Output interrupt ID
	JSR	WRLFCR		New line
* Remove breakpoints
RMBRK	LDY	#BRKPTS		Point to breakpoint
	CLRB			Zero bp number
rmbr1	LDX	3,Y+		Get address
	BEQ	rmbr2		None set here
	LDAA	-1,Y		Get opcode
	STAA	,X		Save it
	CPX	SAVPC		Is this a break?
	BNE	rmbr2		No, it's not
	JSR	WRMSG		Output message
	FCCZ	'**Breakpoint '	Leading message
	TFR	B,A		Get breakpoint
	JSR	WRNIB		Output
	JSR	WRLFCR		New line
rmbr2	INCB			Advance break number
	CPY	#BRKPTS+(NBRK*3) Are we at end?
	BLO	rmbr1		Do them all
	JSR	DISREG		Display registers
	JMP	NEXTCMD		And hit next command
*
* Breakpoint is encountered
*
BREAKPT	PULA			Get CC
	STAA	SAVCC		Save CC
	PULD			Get D
	EXG	A,B		Swap because stacked out of order
	STD	SAVD		Save D
	PULX			Get X
	STX	SAVX		Save X
	PULY			Get Y
	STY	SAVY		Save Y
	PULX			Get PC
	STS	SAVSP		Save SP
	LDS	#STACK		Switch to our stack
	DEX			Backup to inst
	STX	SAVPC		Save it
	BRA	RMBRK		Remove brkpts & proceed
*
*------------------------------------------------
*
* Command handler table
*
CMDTAB	FDB	DUMP		Dump memory
	RDB	'DM'
	FDB	DISMEM		Disassembly memory
	RDB	'DI'
	FDB	DISREG		Display registers
	RDB	'DR'
	FDB	DISBRK		Display breakpoints
	RDB	'DB'
	FDB	DISINT		Display vectors
	RDB	'DV'
	FDB	CHGREG		Change register
	RDB	'CR'
	FDB	CHGINT		Change vector
	RDB	'CV'
	FDB	FILLMEM		Fill memory
	RDB	'FM'
	FDB	MOVEMEM		Move memory
	RDB	'MM'
	FDB	RREAD		Repeating read
	RDB	'RR'
	FDB	RWRITE		Repeating write
	RDB	'RW'
	FDB	EDIT		Edit memory
	RDB	'E'
	FDB	SETBRK		Set/Display breakpoint
	RDB	'B'
	FDB	WRIMEM		Single write to memory
	RDB	'W'
	FDB	SSTEP		Single-step
	RDB	'S'
	FDB	GO		Go (execute)
	RDB	'G'
	FDB	LOAD		Load records
	RDB	'L'
	FDB	HEXMATH		Hex addition
	RDB	'+'
	FDB	HEXMATH		Hex subtraction
	RDB	'-'
	FDB	HELP		Help request
	RDB	'?'
	FDB	0		End of table
* Register display text for 'Display Register' command
REGTXT	FCCZ	'D(A:B)='
	FCCZ	'X='
	FCCZ	'Y='
	FCCZ	'SP='
	FCCZ	'PC='
	FCCZ	'CC='
	FCCZ	' ['
	FCCZ	'SXHINZVC'
* Register parse text for 'Change Register' command
CHRTAB	FCB	'D',0		Special case
	FCB	'A',1
	FCB	'B',1
	FCB	'X',2
	FCB	'Y',2
	FCB	'S',2
	FCB	'P',2
	FCB	'C',-1
* Table of registers for TFR/EXG etc. disassembly.
TFRREG	FCCZ	'A'
	FCCZ	'B'
	FCCZ	'CCR'
	FCCZ	'TMP'
	FCCZ	'D'
* Table of registers for index mode disassembly.
IDXREG	FCCZ	'X'
	FCCZ	'Y'
	FCCZ	'SP'
	FCCZ	'PC'
* Table of DBEQ family instruction names
DBEQT	FCCZ	'DBEQ'
	FCCZ	'DBNE'
	FCCZ	'TBEQ'
	FCCZ	'TBNE'
	FCCZ	'IBEQ'
	FCCZ	'IBNE'
	FCCZ	'?'
	FCCZ	'?'
* Names of the EXG/SEX/TFR instruction names
TEST	FCCZ	'EXG'
	FCCZ	'SEX'
	FCCZ	'TFR'
*
* Disassembler "compressed name" table
*
inames	FCB	'A','D','C'
	FCB	'A','D','D'
	FCB	'A','N','D'
	FCB	'A','S','L'
	FCB	'A','S','R'
	FCB	'B','I','T'
	FCB	'C','L','R'
	FCB	'C','M','P'
	FCB	'C','O','M'
	FCB	'C','P','X'
	FCB	'D','E','C'
	FCB	'E','O','R'
	FCB	'I','N','C'
	FCB	'J','M','P'
	FCB	'J','S','R'
	FCB	'L','D','A'
	FCB	'L','D','S'
	FCB	'L','S','R'
	FCB	'N','E','G'
	FCB	'O','R','A'
	FCB	'P','S','H'
	FCB	'P','U','L'
	FCB	'R','O','R'
	FCB	'R','O','L'
	FCB	'S','B','C'
	FCB	'S','T','A'
	FCB	'S','T','S'
	FCB	'S','U','B'
	FCB	'T','S','T'
	FCB	'L','D','X'
	FCB	'S','T','X'
	FCB	'L','D','D'
	FCB	'S','T','D'
	FCB	'S','E','T'
	FCB	'C','P','Y'
	FCB	'L','D','Y'
	FCB	'S','T','Y'
	FCB	'C','P','D'
	FCB	'F','C','B'
	FCB	'B','E','Q'
	FCB	'B','N','E'
	FCB	'L','E','A'
	FCB	'C','P','S'
	FCB	'M','O','V'
	FCB	'M','A','X'
	FCB	'M','I','N'
	FCB	'M','U','L'
	FCB	'D','I','V'
*
* Disassembler opcode -> text translation table for basic instruction map
*
itable1	FCB	$00,'B','G','N','D',0
	FCB	$01,'M','E','M',0
	FCB	$02,'I','N','Y',0
	FCB	$03,'D','E','Y',0
	FCB	$04,$0A,' ',$09,';',$06,0
	FCB	$05,$8D,' ',$04,0
	FCB	$06,$8D,' ',$03,0
	FCB	$07,'B','S','R',' ',$05,0
	FCB	$08,'I','N','X',0
	FCB	$09,'D','E','X',0
	FCB	$0A,'R','T','C',0
	FCB	$0B,'R','T','I',0
	FCB	$0C,'B',$A1,' ',$04,';',$02,0
	FCB	$0D,'B',$86,' ',$04,';',$02,0
	FCB	$0E,'B','R',$A1,' ',$04,';',$02,';',$05,0
	FCB	$0F,'B','R',$86,' ',$04,';',$02,';',$05,0
	FCB	$10,$82,'C','C',' ','#',$02,0
	FCB	$11,'E',$AF,0
	FCB	$12,$AE,0
	FCB	$13,'E',$AE,0
	FCB	$14,'O','R','C','C',' ','#',$02,0
	FCB	$15,$8E,' ',$04,0
	FCB	$16,$8E,' ',$03,0
	FCB	$17,$8E,' ',$02,0
	FCB	$19,$A9,'Y',' ',$04,0
	FCB	$1A,$A9,'X',' ',$04,0
	FCB	$1B,$A9,'S',' ',$04,0
	FCB	$1C,'B',$A1,' ',$03,';',$02,0
	FCB	$1D,'B',$86,' ',$03,';',$02,0
	FCB	$1E,'B','R',$A1,' ',$03,';',$02,';',$05,0
	FCB	$1F,'B','R',$86,' ',$03,';',$02,';',$05,0
	FCB	$20,'B','R','A',' ',$05,0
	FCB	$21,'B','R','N',' ',$05,0
	FCB	$22,'B','H','I',' ',$05,0
	FCB	$23,'B','L','S',' ',$05,0
	FCB	$24,'B','C','C',' ',$05,0
	FCB	$25,'B','C','S',' ',$05,0
	FCB	$26,$A8,' ',$05,0
	FCB	$27,$A7,' ',$05,0
	FCB	$28,'B','V','C',' ',$05,0
	FCB	$29,'B','V','S',' ',$05,0
	FCB	$2A,'B','P','L',' ',$05,0
	FCB	$2B,'B','M','I',' ',$05,0
	FCB	$2C,'B','G','E',' ',$05,0
	FCB	$2D,'B','L','T',' ',$05,0
	FCB	$2E,'B','G','T',' ',$05,0
	FCB	$2F,'B','L','E',' ',$05,0
	FCB	$30,$95,'X',0
	FCB	$31,$95,'Y',0
	FCB	$32,$95,'A',0
	FCB	$33,$95,'B',0
	FCB	$34,$94,'X',0
	FCB	$35,$94,'Y',0
	FCB	$36,$94,'A',0
	FCB	$37,$94,'B',0
	FCB	$38,$95,'C',0
	FCB	$39,$94,'C',0
	FCB	$3A,$95,'D',0
	FCB	$3B,$94,'D',0
	FCB	$3D,'R','T','S',0
	FCB	$3E,'W','A','I',0
	FCB	$3F,'S','W','I',0
	FCB	$40,$92,'A',0
	FCB	$41,$88,'A',0
	FCB	$42,$8C,'A',0
	FCB	$43,$8A,'A',0
	FCB	$44,$91,'A',0
	FCB	$45,$97,'A',0
	FCB	$46,$96,'A',0
	FCB	$47,$84,'A',0
	FCB	$48,$83,'A',0
	FCB	$49,$91,'D',0
	FCB	$4A,'C','A','L','L',' ',$03,';',$02,0
	FCB	$4B,'C','A','L','L',' ',$04,';',$02,0
	FCB	$4C,'B',$A1,' ',$02,';',$02,0
	FCB	$4D,'B',$86,' ',$02,';',$02,0
	FCB	$4E,'B','R',$A1,' ',$02,';',$02,';',$05,0
	FCB	$4F,'B','R',$86,' ',$02,';',$02,';',$05,0
	FCB	$50,$92,'B',0
	FCB	$51,$88,'B',0
	FCB	$52,$8C,'B',0
	FCB	$53,$8A,'B',0
	FCB	$54,$91,'B',0
	FCB	$55,$97,'B',0
	FCB	$56,$96,'B',0
	FCB	$57,$84,'B',0
	FCB	$58,$83,'B',0
	FCB	$59,$83,'D',0
	FCB	$5A,$99,'A',' ',$02,0
	FCB	$5B,$99,'B',' ',$02,0
	FCB	$5C,$A0,' ',$02,0
	FCB	$5D,$A4,' ',$02,0
	FCB	$5E,$9E,' ',$02,0
	FCB	$5F,$9A,' ',$02,0
	FCB	$60,$92,' ',$04,0
	FCB	$61,$88,' ',$04,0
	FCB	$62,$8C,' ',$04,0
	FCB	$63,$8A,' ',$04,0
	FCB	$64,$91,' ',$04,0
	FCB	$65,$97,' ',$04,0
	FCB	$66,$96,' ',$04,0
	FCB	$67,$84,' ',$04,0
	FCB	$68,$83,' ',$04,0
	FCB	$69,$86,' ',$04,0
	FCB	$6A,$99,'A',' ',$04,0
	FCB	$6B,$99,'B',' ',$04,0
	FCB	$6C,$A0,' ',$04,0
	FCB	$6D,$A4,' ',$04,0
	FCB	$6E,$9E,' ',$04,0
	FCB	$6F,$9A,' ',$04,0
	FCB	$70,$92,' ',$03,0
	FCB	$71,$88,' ',$03,0
	FCB	$72,$8C,' ',$03,0
	FCB	$73,$8A,' ',$03,0
	FCB	$74,$91,' ',$03,0
	FCB	$75,$97,' ',$03,0
	FCB	$76,$96,' ',$03,0
	FCB	$77,$84,' ',$03,0
	FCB	$78,$83,' ',$03,0
	FCB	$79,$86,' ',$03,0
	FCB	$7A,$99,'A',' ',$03,0
	FCB	$7B,$99,'B',' ',$03,0
	FCB	$7C,$A0,' ',$03,0
	FCB	$7D,$A4,' ',$03,0
	FCB	$7E,$9E,' ',$03,0
	FCB	$7F,$9A,' ',$03,0
	FCB	$80,$9B,'A',' ','#',$02,0
	FCB	$81,$87,'A',' ','#',$02,0
	FCB	$82,$98,'A',' ','#',$02,0
	FCB	$83,$9B,'D',' ','#',$03,0
	FCB	$84,$82,'A',' ','#',$02,0
	FCB	$85,$85,'A',' ','#',$02,0
	FCB	$86,$8F,'A',' ','#',$02,0
	FCB	$87,$86,'A',0
	FCB	$88,$8B,'A',' ','#',$02,0
	FCB	$89,$80,'A',' ','#',$02,0
	FCB	$8A,$93,'A',' ','#',$02,0
	FCB	$8B,$81,'A',' ','#',$02,0
	FCB	$8C,$A5,' ','#',$03,0
	FCB	$8D,$A2,' ','#',$03,0
	FCB	$8E,$89,' ','#',$03,0
	FCB	$8F,$AA,' ','#',$03,0
	FCB	$90,$9B,'A',' ',$02,0
	FCB	$91,$87,'A',' ',$02,0
	FCB	$92,$98,'A',' ',$02,0
	FCB	$93,$9B,'D',' ',$02,0
	FCB	$94,$82,'A',' ',$02,0
	FCB	$95,$85,'A',' ',$02,0
	FCB	$96,$8F,'A',' ',$02,0
	FCB	$97,$9C,'A',0
	FCB	$98,$8B,'A',' ',$02,0
	FCB	$99,$80,'A',' ',$02,0
	FCB	$9A,$93,'A',' ',$02,0
	FCB	$9B,$81,'A',' ',$02,0
	FCB	$9C,$A5,' ',$02,0
	FCB	$9D,$A2,' ',$02,0
	FCB	$9E,$89,' ',$02,0
	FCB	$9F,$AA,' ',$02,0
	FCB	$A0,$9B,'A',' ',$04,0
	FCB	$A1,$87,'A',' ',$04,0
	FCB	$A2,$98,'A',' ',$04,0
	FCB	$A3,$9B,'D',' ',$04,0
	FCB	$A4,$82,'A',' ',$04,0
	FCB	$A5,$85,'A',' ',$04,0
	FCB	$A6,$8F,'A',' ',$04,0
	FCB	$A7,'N','O','P',0
	FCB	$A8,$8B,'A',' ',$04,0
	FCB	$A9,$80,'A',' ',$04,0
	FCB	$AA,$93,'A',' ',$04,0
	FCB	$AB,$81,'A',' ',$04,0
	FCB	$AC,$A5,' ',$04,0
	FCB	$AD,$A2,' ',$04,0
	FCB	$AE,$89,' ',$04,0
	FCB	$AF,$AA,' ',$04,0
	FCB	$B0,$9B,'A',' ',$03,0
	FCB	$B1,$87,'A',' ',$03,0
	FCB	$B2,$98,'A',' ',$03,0
	FCB	$B3,$9B,'D',' ',$03,0
	FCB	$B4,$82,'A',' ',$03,0
	FCB	$B5,$85,'A',' ',$03,0
	FCB	$B6,$8F,'A',' ',$03,0
	FCB	$B7,$0B,' ',$08,',',$09,0
	FCB	$B8,$8B,'A',' ',$03,0
	FCB	$B9,$80,'A',' ',$03,0
	FCB	$BA,$93,'A',' ',$03,0
	FCB	$BB,$81,'A',' ',$03,0
	FCB	$BC,$A5,' ',$03,0
	FCB	$BD,$A2,' ',$03,0
	FCB	$BE,$89,' ',$03,0
	FCB	$BF,$AA,' ',$03,0
	FCB	$C0,$9B,'B',' ','#',$02,0
	FCB	$C1,$87,'B',' ','#',$02,0
	FCB	$C2,$98,'B',' ','#',$02,0
	FCB	$c3,$81,'D',' ','#',$03,0
	FCB	$C4,$82,'B',' ','#',$02,0
	FCB	$C5,$85,'B',' ','#',$02,0
	FCB	$C6,$8F,'B',' ','#',$02,0
	FCB	$C7,$86,'B',0
	FCB	$C8,$8B,'B',' ','#',$02,0
	FCB	$C9,$80,'B',' ','#',$02,0
	FCB	$CA,$93,'B',' ','#',$02,0
	FCB	$CB,$81,'B',' ','#',$02,0
	FCB	$CC,$9F,' ','#',$03,0
	FCB	$CD,$A3,' ','#',$03,0
	FCB	$CE,$9D,' ','#',$03,0
	FCB	$CF,$90,' ','#',$03,0
	FCB	$D0,$9B,'B',' ',$02,0
	FCB	$D1,$87,'B',' ',$02,0
	FCB	$D2,$98,'B',' ',$02,0
	FCB	$D3,$81,'D',' ',$02,0
	FCB	$D4,$82,'B',' ',$02,0
	FCB	$D5,$85,'B',' ',$02,0
	FCB	$D6,$8F,'B',' ',$02,0
	FCB	$D7,$9C,'B',0
	FCB	$D8,$8B,'B',' ',$02,0
	FCB	$D9,$80,'B',' ',$02,0
	FCB	$DA,$93,'B',' ',$02,0
	FCB	$DB,$81,'B',' ',$02,0
	FCB	$DC,$9F,' ',$02,0
	FCB	$DD,$A3,' ',$02,0
	FCB	$DE,$9D,' ',$02,0
	FCB	$DF,$90,' ',$02,0
	FCB	$E0,$9B,'B',' ',$04,0
	FCB	$E1,$87,'B',' ',$04,0
	FCB	$E2,$98,'B',' ',$04,0
	FCB	$E3,$81,'D',' ',$04,0
	FCB	$E4,$82,'B',' ',$04,0
	FCB	$E5,$85,'B',' ',$04,0
	FCB	$E6,$8F,'B',' ',$04,0
	FCB	$E7,$9C,' ',$04,0
	FCB	$E8,$8B,'B',' ',$04,0
	FCB	$E9,$80,'B',' ',$04,0
	FCB	$EA,$93,'B',' ',$04,0
	FCB	$EB,$81,'B',' ',$04,0
	FCB	$EC,$9F,' ',$04,0
	FCB	$ED,$A3,' ',$04,0
	FCB	$EE,$9D,' ',$04,0
	FCB	$EF,$90,' ',$04,0
	FCB	$F0,$9B,'B',' ',$03,0
	FCB	$F1,$87,'B',' ',$03,0
	FCB	$F2,$98,'B',' ',$03,0
	FCB	$F3,$81,'D',' ',$03,0
	FCB	$F4,$82,'B',' ',$03,0
	FCB	$F5,$85,'B',' ',$03,0
	FCB	$F6,$8F,'B',' ',$03,0
	FCB	$F7,$9C,' ',$03,0
	FCB	$F8,$8B,'B',' ',$03,0
	FCB	$F9,$80,'B',' ',$03,0
	FCB	$FA,$93,'B',' ',$03,0
	FCB	$FB,$81,'B',' ',$03,0
	FCB	$FC,$9F,' ',$03,0
	FCB	$FD,$A3,' ',$03,0
	FCB	$FE,$9D,' ',$03,0
	FCB	$FF,$90,' ',$03,0
	FCB	$00,'F','C','B',' ',$0C,0
* Disassembler opcode -> text translation table for shifted instruction map
itable2	FCB	$00,$04,$03,$01,$AB,'W',' ','#',$03,';',$04,0
	FCB	$01,$04,$03,$01,$AB,'W',' ',$03,';',$04,0
	FCB	$02,$AB,'W',' ',$04,';',$04,0
	FCB	$03,$AB,'W',' ','#',$03,';',$03,0
	FCB	$04,$AB,'W',' ',$03,';',$03,0
	FCB	$05,$AB,'W',' ',$04,';',$03,0
	FCB	$06,'A','B','A',0
	FCB	$07,'D','A','A',0
	FCB	$08,$04,$02,$01,$AB,'B',' ','#',$02,';',$04,0
	FCB	$09,$04,$03,$01,$AB,'B',' ',$03,';',$04,0
	FCB	$0A,$AB,'B',' ',$04,';',$04,0
	FCB	$0B,$AB,'B',' ','#',$02,';',$03,0
	FCB	$0C,$AB,'B',' ',$03,';',$03,0
	FCB	$0D,$AB,'B',' ',$04,';',$03,0
	FCB	$0E,'T','A','B',0
	FCB	$0F,'T','B','A',0
	FCB	$10,'I',$AF,0
	FCB	$11,'F',$AF,0
	FCB	$12,'E','M','A','C','S',' ',$03,0
	FCB	$13,'E',$AE,'S',0
	FCB	$14,'E',$AF,'S',0
	FCB	$15,'I',$AF,'S',0
	FCB	$16,'S','B','A',0
	FCB	$17,'C','B','A',0
	FCB	$18,$AC,'A',' ',$04,0
	FCB	$19,$AD,'A',' ',$04,0
	FCB	$1A,'E',$AC,'D',' ',$04,0
	FCB	$1B,'E',$AD,'D',' ',$04,0
	FCB	$1C,$AC,'M',' ',$04,0
	FCB	$1D,$AD,'M',' ',$04,0
	FCB	$1E,'E',$AC,'M',' ',$04,0
	FCB	$1F,'E',$AD,'M',' ',$04,0
	FCB	$20,'L','B','R','A',' ',$07,0
	FCB	$21,'L','B','R','N',' ',$07,0
	FCB	$22,'L','B','H','I',' ',$07,0
	FCB	$23,'L','B','L','S',' ',$07,0
	FCB	$24,'L','B','C','C',' ',$07,0
	FCB	$25,'L','B','C','S',' ',$07,0
	FCB	$26,'L',$A8,' ',$07,0
	FCB	$27,'L',$A7,' ',$07,0
	FCB	$28,'L','B','V','C',' ',$07,0
	FCB	$29,'L','B','V','S',' ',$07,0
	FCB	$2A,'L','B','P','L',' ',$07,0
	FCB	$2B,'L','B','M','I',' ',$07,0
	FCB	$2C,'L','B','G','E',' ',$07,0
	FCB	$2D,'L','B','L','T',' ',$07,0
	FCB	$2E,'L','B','G','T',' ',$07,0
	FCB	$2F,'L','B','L','E',' ',$07,0
	FCB	$3A,'R','E','V',0
	FCB	$3B,'R','E','V','W',0
	FCB	$3C,'W','A','V',0
	FCB	$3D,'T','B','L',' ',$04,0
	FCB	$3E,'S','T','O','P',0
	FCB	$3F,'E','T','B','L',' ',$04,0
	FCB	$00,'T','R','A','P',' ',$0C,0
*
* Help information
*
HLPTXT	FCB	0		New line to start
	FCCZ	'B 0-7 <addr>|Set breakpoint (0000=remove)'
	FCCZ	'CR <reg> <data>|Change register'
	FCCZ	'CV <vec> <addr>|Change interrupt vector'
	FCCZ	'DB|Display breakpoints'
	FCCZ	'DI <addr>,<addr>|Display memory in assembly format'
	FCCZ	'DM <addr>,<addr>|Display memory in hex dump format'
	FCCZ	'DR|Display processor registers'
	FCCZ	'DV|Display interrupt vectors'
	FCCZ	'E <addr>|Edit memory'
	FCCZ	'FM <addr>,<addr> <data>|Fill memory'
	FCCZ	'G <addr>|Go (execute program)'
	FCCZ	'L|Load an image into memory'
	FCCZ	'MM <addr>,<addr> <addr>|Move memory'
	FCCZ	'RR <addr>|Repeating READ access'
	FCCZ	'RW <addr> <data>|Repeating WRITE access'
	FCCZ	'S|Single step execution'
	FCCZ	'W <addr> <data>|Write to memory'
	FCCZ	'+ <value>+<value>|Hexidecimal addition'
	FCCZ	'- <value>-<value>|Hexidecimal subtraction'
	FCB	-1		End of list
*
* Interrupt vectors.
*
*** To add additional vectors, you must:
*
* -Change ORG below to new starting address & add new FDB statements
* -Add new handlers around the REVECT routine
* -Increase the equate NUMINT to the new number of vectors
* -Modify Monitor DATA area/STACK location to accomodate larger INTVEC
*
* NOTE when loading under D-Bug12, these vectors will cause errors with
* failed writes to memory (the vectors are in EPROM on the EVB). This is
* not a problem, because the startup code calls D-Bug12 to acquire the
* vectors, and these (physical) vectors are not used in this configuration.
* The vectors are *REQUIRED* when running MON12 on the "bare metal".
*
	ORG	$FFC0		Allow room for 31 vectors + reset

	FDB	FFC0
	FDB	FFC2
	FDB	FFC4
	FDB	FFC6
	FDB	FFC8
	FDB	FFCA
	FDB	FFCC
	FDB	FFCE
	FDB	FFD0
	FDB	FFD2
	FDB	FFD4
	FDB	FFD6
	FDB	FFD8
	FDB	FFDA
	FDB	FFDC
	FDB	FFDE
	FDB	FFE0
	FDB	FFE2
	FDB	FFE4
	FDB	FFE6
	FDB	FFE8
	FDB	FFEA
	FDB	FFEC
	FDB	FFEE
	FDB	FFF0
	FDB	FFF2		IRQ
	FDB	FFF4		XIRQ
	FDB	FFF6		SWI
	FDB	FFF8		Unimplemented Opcode
	FDB	FFFA		COP timeout
	FDB	FFFC		Clock monitor
	FDB	BEGIN		Hard reset
