;
; Quick and simple AVR monitor in less than 1K
; ?COPY.TXT 1998-2007 Dave Dunfield -  -- see COPY.TXT --.
;
; Resources for a debugger are very limited on the AVR. This monitor has
; been set up to work on the 90S8515, which has the greatest amount of
; program and data memory to work with (at the time of this writing). Even
; if you are designing for a smaller AVR device, you may want to use the
; 8515 for development, as it will accomodate the additional code and
; data requirements of the monitor.
;
; MONAVR commands:
;
; Daaaa		- Dump memory at address 'aaaa' (SPACE for next line)
; Eaaaa		- Edit memory at address 'aaaa', SPACE=Skip to next,
;		  BACKSPACE=Previous location, 'vv'= Enter new value
; Raaaa		- Read and display single byte at 'aaaa'
; Waaaa vv	- Write value 'vv' to address 'aaaa'
; V		- View processor registers
; Crr cc vv	- Change register 'rr', current value 'cc' is displayed,
;		  then enter new value 'vv'.
; X		- Execute user program.
; G		- Go at address in PC (continue from breakpoint)
; S		- Single-step one instruction
;	NOTE: You can abort any command at a prompt by entering ESC.
;
; The AVR has no capability to write to code memory, making normal break-
; points impossible. MONAVR allows you to insert breakpoints into your
; code as calls to the "MONAVR" entry point in this monitor. Once entered,
; You may view and modify memory and registers, then enter 'G' to continue
; at the location immediately after the call to MONAVR.
;
; You may also enter 'S' to single-step one instruction, this is done by
; programming Timer0 to cause an interrupt, 1 cycle into the execution
; of the user program. See the notes at the label "STEP".
;
CODE	EQU	$0000		; Code goes here
RAM	EQU	$0060		; Data memory goes here
STACK	EQU	$015F		; AVR stack initialized to here
;BAUD	EQU	47		; Baudrate divisor for 9600 @ 7.37Mhz
BAUD	EQU	25		; Baudrate divisor for 9600 @ 4.00Mhz
; Special function register definitions for: AVR 90S8515
ACSR	EQU	$08		; Analog Comparator Control and Status Register
UBRR	EQU	$09		; UART Baud Rate Register
UCR	EQU	$0A		; UART Control Register
USR	EQU	$0B		; UART Status Register
UDR	EQU	$0C		; UART I/O Data Register
SPCR	EQU	$0D		; SPI Control Register
SPSR	EQU	$0E		; SPI Status Register
SPDR	EQU	$0F		; SPI I/O Data Register
PIND	EQU	$10		; Input Pins, Port D
DDRD	EQU	$11		; Data Direction Register, Port D
PORTD	EQU	$12		; Data Register, Port D
PINC	EQU	$13		; Input Pins, Port C
DDRC	EQU	$14		; Data Direction Register, Port C
PORTC	EQU	$15		; Data Register, Port C
PINB	EQU	$16		; Input Pins, Port B
DDRB	EQU	$17		; Data Direction Register, Port B
PORTB	EQU	$18		; Data Register, Port B
PINA	EQU	$19		; Input Pins, Port A
DDRA	EQU	$1A		; Data Direction Register, Port A
PORTA	EQU	$1B		; Data Register, Port A
EECR	EQU	$1C		; EEPROM Control Register
EEDR	EQU	$1D		; EEPROM Data Register
EEAR	EQU	$1E		; EEPROM low Address Register
WDTCR	EQU	$21		; Watchdog Timer Control Register
ICR1	EQU	$24		; T/C 1 Input Capture Register
OCR1B	EQU	$28		; Timer/Counter1 Output Compare Register B
OCR1A	EQU	$2A		; Timer/Counter1 Output Compare Register A
TCNT1	EQU	$2C		; Timer/Counter 1
TCCR1B	EQU	$2E		; Timer/Counter 1 Control and Status Register
TCCR1A	EQU	$2F		; Timer/Counter 1 Control Register
OCR0	EQU	$31		; Output Compare Register 0
TCNT0	EQU	$32		; Timer/Counter 0
TCCR0	EQU	$33		; Timer/Counter 0 Control Register
MCUCR	EQU	$35		; MCU general Control Register
TIFR	EQU	$38		; Timer/Counter Interrupt FLAG register
TIMSK	EQU	$39		; Timer/Counter Interrupt MASK register
GIMSK	EQU	$3B		; General Interrupt MASK register
SP	EQU	$3D		; Stack Pointer
SREG	EQU	$3F		; Status REGister
;
; Reserved memory areas
;
	ORG	$0060		; SRAM starts here
; Save areas used by MONAVR
;
; Normally, you would place this at the top of the memory available in
; your particular AVR device, or at some other address where it will not
; conflict with the memory requirements of the user application.
SAVR0	RMB	1		; Saved R0
SAVR28	RMB	1		; Saved R28
SAVR29	RMB	1		; Saved R29
SAVR30	RMB	1		; Saved R30
SAVR31	RMB	1		; Saved R31
SAVSREG	RMB	1		; Saved SREG
SAVPC	RMB	2		; Current PC position
SAVSP	RMB	2		; Stack pointer on entry
;
	ORG	CODE
	RJMP	BEGIN		; Execute start code
; Interrupt vectors (if you need them) go here!
; Unused vectors should point to the 'MONAVR' entry point
	RJMP	MONAVR		;1 External interrupt 0
	RJMP	MONAVR		;2 External interrupt 1
	RJMP	MONAVR		;3 Timer1 input capture
	RJMP	MONAVR		;4 Timer1 output compareA
	RJMP	MONAVR		;5 Timer1 output compareB
	RJMP	MONAVR		;6 Timer1 overflow
	RJMP	MONAVR		;7 Timer0 overflow
	RJMP	MONAVR		;8 SPI transfer
	RJMP	MONAVR		;9 UART RX
	RJMP	MONAVR		;A UART Data reg empty
	RJMP	MONAVR		;B UART TX complete
	RJMP	MONAVR		;C Analog comparator
;
; Text block - Test addressed with only R30 (via PUTTXT)
;
WELCOME	STRZ	"MonAVR"
RTEXT	DB	$0A,$0D
	STRZ	"R00-15="
RTEXT1	DB	$0A,$0D
	STRZ	"R16-31="
RTEXT2	DB	$0A,$0D
	STRZ	"SREG(32)-PCL(33)-PCH(34)="
;
; Console command processor handler table
;
CMDTAB	DB	'D'		; Dump memory
	DW	DUMP/2
	DB	'E'		; Edit memory
	DW	EDIT/2
	DB	'R'		; Read memory
	DW	READ/2
	DB	'W'		; Write memory
	DW	WRITE/2
	DB	'V'		; View registers
	DW	RSHOW/2
	DB	'C'		; Change register
	DW	RSET/2
	DB	'G'		; Go/Continue
	DW	GO/2
	DB	'S'		; Step
	DW	STEP/2
	DB	'X'		; Execute user program
	DW	USER/2
	DB	0
;
	ORG	(*+1)&$FFFE	; Word align
;
; Monitor dynamic entry point.
; Save context of user program, and enter the interactive monitor.
;
MONAVR	STS	SAVR30,R30	; Save R30
	STS	SAVR31,R31	; Save R31
	LDI	R30,SAVR0	; Point to registers
	LDI	R31,=SAVR0	; ""
	STD	Z+0,R0		; Save R0
	STD	Z+1,R28		; Save R28
	STD	Z+2,R29		; Save R29
	IN	R28,SREG	; Get SREG
	STD	Z+5,R28		; Save SREG
	POP	R29		; Get PC high
	POP	R28		; Get PC low
	LSL	R28		; x2 for byte address
	ROL	R29		; ""
	STD	Z+6,R28		; Save PCL
	STD	Z+7,R29		; Save PCH
	IN	R28,SP		; Get SPL
	IN	R29,SP+1	; Get SPH
	STD	Z+8,R28		; Save SPL
	STD	Z+0,R29		; Save SPH
	RCALL	RSHOW		; Display the registers
	RJMP	CMD		; Enter monitor	
;
; Main monitor startup code
;
BEGIN	LDI	R30,STACK	; Get low hardware stack
	LDI	R31,=STACK	; Get high hardware stack
	OUT	SP,R30		; Set low hardware stack
	OUT	SP+1,R31	; Set high hardware stack
	STS	SAVSP,R30	; Save for later
	STS	SAVSP+1,R31	; ""
; Place additional initialization here
	CLR	R28		; Get a zero
	OUT	MCUCR,R28	; Disable external SRAM
; Setup serial port
	LDI	R28,BAUD	; Baud rate for 9600 & 4.0 Mhz
	OUT	UBRR,R28	; Write it
	LDI	R28,%00011000	; No interrupts, enable RX+TX, 8bit
	OUT	UCR,R28		; Write it
; Issue welcome message
	LDI	R30,WELCOME	; Point to welcome message
	RCALL	PUTTXT		; Output it
; Reset stack pointer incase we are aborting
RESET	LDS	R30,SAVSP	; Reset LOW stack
	LDS	R31,SAVSP+1	; Reset HIGH stack
	OUT	SP,R30		; Set low hardware stack
	OUT	SP+1,R31	; Set high hardware stack
; Prompt for, and execute commands
cmd	RCALL	NEWLINE		; New line
	LDI	R28,'>'		; Prompt
	RCALL	PUTCH		; Write it
	RCALL	SPACE		; Space over
	RCALL	GETCHE		; Get command character
	LDI	R30,CMDTAB	; Point to command table (low)
	LDI	R31,=CMDTAB	; Point to command table (high)
cmd1	LPM			; Get byte from command table
	TST	R0		; End of table?
	BREQ	ERROR		; Report error
	ADIW	Z,1		; Skip to next
	CP	R28,R0		; Is this is?
	BREQ	cmd2		; Yes, execute command
	ADIW	Z,2		; Skip to next
	RJMP	cmd1		; And proceed
cmd2	LPM			; Get LOW address
	ADIW	Z,1		; Skip to next
	MOV	R29,R0		; Save LOW address
	LPM			; Get high address
	MOV	R31,R0		; Set high
	MOV	R30,R29		; Restore LOW
	ICALL			; Execute handler
	RJMP	cmd		; And go again
; Set a register
RSET	RCALL	GETDEC		; Get register number
	CPI	R28,35		; Valid register
	BRSH	ERROR		; Out of range
	MOV	R29,R28		; Save it
	RCALL	SPACE		; Space over
	PUSH	R29		; Save index
	RCALL	GETREG		; Get register value
	POP	R29		; Restore index
	RCALL	PUTHEX		; Display value
	RCALL	SPACE		; Space over
	RCALL	GETBYTE		; Get value
	MOV	R0,R28		; Save it
	RJMP	SETREG		; Set it
;
; Report an error & re-enter command processor
;
ERROR	RCALL	SPACE		; SPACE over
	LDI	R28,'?'		; Get indicator
	RCALL	PUTCH		; output
	RJMP	RESET		; And proceed
;----------------------
; Monitor I/O functions
;----------------------
;
; Read a character from the serial port
;
GETCH	SBIS	USR,7		; Receive ready?
	RJMP	GETCH		; No, wait
	IN	R28,UDR		; Read data
	CPI	R28,$1B		; Escape?
	BREQ	RESET		; Exit
	CPI	R28,'a'		; Upper case?
	BRLO	getch1		; Yes, it's OK
	CPI	R28,'z'+1	; Upper case?
	BRSH	getch1		; Don't convert
	ANDI	R28,$DF		; Convert to upper
getch1	RET
;
; Get character with echo
;
GETCHE	RCALL	GETCH		; Get character
	RJMP	PUTCH		; Echo it
;
; Get a decimal number (00-99) into R28
;
GETDEC	CLR	R0		; No carry in
	RCALL	getde1		; Get digit1
	MOV	R0,R28		; Make copy
	ADD	R0,R0		; x2
	ADD	R0,R0		; x4
	ADD	R0,R28		; x5
	ADD	R0,R0		; x10
getde1	RCALL	GETCHE		; Get first number
	SUBI	R28,'0'		; Convert
	CPI	R28,9+1		; In range
	BRSH	ERROR		; No, error
	ADD	R28,R0		; 
	RET
;
; Get a hex number (00-FF) into R28
;
GETBYTE	RCALL	GETCHE		; Get char
getbyt1	CLR	R0		; No carry in
	RCALL	getbyt2		; Process first char
	SWAP	R28		; Get in high
	MOV	R0,R28		; Save for later
	RCALL	GETCHE		; Get char
getbyt2	SUBI	R28,'0'		; Decimal
	BRLO	ERROR		; No, error
	CPI	R28,$A		; In range?
	BRLO	getbyt3		; It's OK
	SUBI	R28,7		; Convert
	BRLO	ERROR		; Error
	CPI	R28,$10		; In range
	BRSH	ERROR		; Error
getbyt3	ADD	R28,R0		; Include old
	RET
;
; Get word into R30:31
;
GETWORD	RCALL	GETBYTE		; Get high
	MOV	R31,R28		; Copy
	RCALL	GETBYTE		; Get low
	MOV	R30,R28		; Copy
	RET
;
; Write word(R31:R30) value in HEX
;
PUTWORD	PUSH	R28		; Save LOW
	MOV	R28,R31		; Get high
	RCALL	PUTHEX		; Write it
	MOV	R28,R30		; Get low
	RCALL	PUTHEX		; Write it
	POP	R28		; Restore LOW
	RET
;
; Write byte(R28) in hex
;
PUTHEX	PUSH	R28		; Save R28
	SWAP	R28		; Get high nibble
	RCALL	puth1		; Output high
	POP	R28		; Restore R28
	PUSH	R28		; Resave
	RCALL	puth1		; Output low
	POP	R28		; Restore
	RET
;
; Write nibble(R28) to serial
;
puth1	ANDI	R28,%00001111	; Mask off low
	CPI	R28,$0A		; In range?
	BRLO	puth2		; Its OK
	ADIW	R28,7		; Adjust
puth2	ADIW	R28,'0'		; Convert to ASCII
	RCALL	PUTCH		; Write char
puth3	RET
; Display value & space over
rshowx	RCALL	PUTHEX		; Display it
;
; Write SPACE
;
SPACE	LDI	R28,' '		; Get SPACE
	RJMP	PUTCH		; And write it
;
; Write NEWLINE
;
NEWLINE	LDI	R28,$0A		; Get NEWLINE
;
; Write character(R28) to serial port
;
PUTCH	SBIS	USR,5		; Transmit empty?
	RJMP	PUTCH		; No, wait
	OUT	UDR,R28		; Write to uart
	CPI	R28,$0A		; Newline?
	BRNE	puth3		; No, exit
	LDI	R28,$0D		; Return
	RJMP	PUTCH		; And go again
;
; Write string(R30) from the text block to the serial port
;
PUTTXT	LDI	R31,=CODE	; Setup text pointer
;
; Write string(Z) to the serial port
;
PUTSTR	LPM			; Get byte from memory
	ADIW	Z,1		; Advance to next
	TST	R0		; End of string?
	BREQ	puth3		; Yes, exit
puts1	SBIS	USR,5		; Transmit empty?
	RJMP	puts1		; No, wait
	OUT	UDR,R0		; Write data
	RJMP	PUTSTR		; And proceed
;-------------------------
; Console command handlers
;-------------------------
;
; Dump working memory
;
DUMP	RCALL	GETWORD		; Get address
dump1	RCALL	NEWLINE		; New line
	RCALL	PUTWORD		; Write address
	LDI	R29,16		; Copy 16 bytes
dump2	RCALL	SPACE		; Space over
	LD	R28,Z+		; Get data
	RCALL	PUTHEX		; Output
	DEC	R29		; Reduce count
	BRNE	dump2		; Do them all
	RCALL	SPACE		; Space over
	SBIW	Z,16		; Backup
	LDI	R29,16		; Get count
dump3	LD	R28,Z+		; Get char
	CPI	R28,' '		; Control?
	BRLO	dump4		; Yes, control code
	CPI	R28,$7F		; In range?
	BRLO	dump5		; It's OK
dump4	LDI	R28,'.'		; Convert to .
dump5	RCALL	PUTCH		; Output
	DEC	R29		; Reduce count
	BRNE	dump3		; Do them all
dump6	RCALL	GETCH		; Get character
	CPI	R28,' '		; Continue?
	BREQ	dump1		; Next line
	RJMP	dump6		; And proceed
;
; Edit memory
;
EDIT	RCALL	GETWORD		; Get address
edit1	RCALL	NEWLINE		; New line
	RCALL	PUTWORD		; Write address
	LDI	R29,8		; Reset count
edit2	RCALL	SPACE		; Space over
	LD	R28,Z		; Get value
	RCALL	PUTHEX		; Write it
	LDI	R28,'-'		; Prompt
	RCALL	PUTCH		; Output
	RCALL	GETCHE		; Get character
	CPI	R28,$08		; BackSPACE?
	BREQ	edit4		; Handle it
	CPI	R28,' '		; Next
	BREQ	edit5		; Handle it
	RCALL	GETBYT1		; Get a byte
	ST	Z+,R28		; Write it
edit3	DEC	R29		; At end of line?
	BRNE	edit2		; Keep going
	RJMP	edit1		; And proceed
edit4	SBIW	Z,1		; Backup
	RJMP	edit1		; And proceed
edit5	RCALL	SPACE		; Advance
	ADIW	Z,1		; And proceed
	RJMP	edit3		; End
;
; Read a memory location
;
READ	RCALL	GETWORD		; Get address
	RCALL	SPACE		; Space over
	LD	R28,Z		; Read the data
	RJMP	PUTHEX		; Display
;
; Write a memory location
;
WRITE	RCALL	GETWORD		; Get address
	RCALL	SPACE		; Space over
	RCALL	GETBYTE		; Read data
	ST	Z,R28		; Write data
rshow3	RET
;
; Show current registers
;
RSHOW	LDI	R30,RTEXT	; Point to show
	CLR	R29		; Begin with R0
rshow1	RCALL	PUTTXT		; Display first block
rshow2	RCALL	SPACE		; Display space
	PUSH	R29		; Save R29
	RCALL	GETREG		; Get register
	POP	R29		; Restore it
	RCALL	PUTHEX		; Display it
	INC	R29		; Advance to next
	LDI	R30,RTEXT1	; Next text block
	CPI	R29,16		; Next block?
	BREQ	rshow1		; New prompt
	LDI	R30,RTEXT2	; Next text block
	CPI	R29,32		; Special reg's
	BREQ	rshow1		; New prompt
	CPI	R29,35		; At end?
	BREQ	rshow3		; Yes, exit
	MOV	R28,R29		; Get value
	ANDI	R28,%00000011	; Extra space?
	BRNE	rshow2		; No handle it
	RCALL	SPACE		; Extra space
	RJMP	rshow2		; And continue
;
; Begin execution
;
GO	RCALL	NEWLINE		; New line
go1	LDI	R30,SAVR0	;1 Point to memory
	LDI	R31,=SAVR0	;1 Point to memory
	LDD	R28,Z+8		;2 Get SPL
	LDD	R29,Z+9		;2 Get SPH
	OUT	SP,R28		;1 Write SP high
	OUT	SP+1,R29	;1 Write SP low
	LDD	R0,Z+0		;2 Restore R0
	LDD	R28,Z+6		;2 Get PC low
	LDD	R29,Z+7		;2 Get PC high
	LSR	R29		;1 /2 for word address
	ROR	R28		;1 ""
	PUSH	R28		;2 Stack low
	PUSH	R29		;2 Stack high
	LDD	R28,Z+5		;2 Get SREG
	OUT	SREG,R28	;1 Restore it
	LDD	R28,Z+1		;2 Restore R28
	LDD	R29,Z+2		;2 Restore R29
	LDD	R30,Z+3		;2 Restore R30
	LDS	R31,SAVR31	;3 Restore R31
	RET			;4 Launch program
;
; Single-step one instruction by programming Timer0 to cause an interrupt 1
; cycle into the user program. Please note the following:
; - The Timer0 interrupt vector **MUST** point to the MONAVR entry point.
; - This will corrupt any programming of Timer0 done by the user program.
; - Global interrupts are forced enabled before stepping.
; - When the interrupt occurs, the AVR automatically clears the 'I' bit.
;   If you wish global interrupts to remain enabled after stepping, you must
;   manually set bit 7 of SREG register (via C32) command before 'G'.
;
STEP	RCALL	NEWLINE		; New line
	LDS	R28,SAVSREG	; Get saved SREF
	ORI	R28,%10000000	; Enable interrupts
	STS	SAVSREG,R28	; Resave it
	CLR	R28		; Get zero
	OUT	TCCR0,R28	; Stop timer
; This count must wait until all of the instructions up to the
; preceeding 'RET' at the end of 'go1' are executed.
	LDI	R28,-39		; Get count value
	OUT	TCNT0,R28	; Set initial count value
	LDI	R28,%00000010	; Timer0 overflow
	OUT	TIFR,R28	; Clear any pending interrupt
	IN	R28,TIMSK	; Get Timer control
	ORI	R28,%00000010	; Enable timer0 overflow
	OUT	TIMSK,R28	; Write it
	LDI	R28,%00000001	; Prescale=0
	OUT	TCCR0,R28	; Start timer
	SEI			;1 Enable global interrupts
	RJMP	go1		;2 and continue
;
; Get register value (R29=reg)
;
GETREG	LDI	R30,RGTAB/2	; Point to handler table
	LDI	R31,=(RGTAB/2)	; ""
getr0	CLR	R28		; Get zero
	ADD	R29,R29		; x2 instructions
	ADD	R30,R29		; Offsete
	ADC	R31,R28		; Carry over
	LDI	R28,SAVR0	; Point to Save table
	LDI	R29,=SAVR0	; ""
	IJMP			; Call handler
;
; Set register value (R29=Reg, R0=Value)
;
SETREG	LDI	R30,RSTAB/2	; Point to handler table
	LDI	R31,=(RSTAB/2)	; ""
	RJMP	getr0		; Process it
; Table of register read handlers
RGTAB	LDD	R28,Y+0		; Get SAVR0
	RET
	MOV	R28,R1		; Get register
	RET
	MOV	R28,R2		; Get register
	RET
	MOV	R28,R3		; Get register
	RET
	MOV	R28,R4		; Get register
	RET
	MOV	R28,R5		; Get register
	RET
	MOV	R28,R6		; Get register
	RET
	MOV	R28,R7		; Get register
	RET
	MOV	R28,R8		; Get register
	RET
	MOV	R28,R9		; Get register
	RET
	MOV	R28,R10		; Get register
	RET
	MOV	R28,R11		; Get register
	RET
	MOV	R28,R12		; Get register
	RET
	MOV	R28,R13		; Get register
	RET
	MOV	R28,R14		; Get register
	RET
	MOV	R28,R15		; Get register
	RET
	MOV	R28,R16		; Get register
	RET
	MOV	R28,R17		; Get register
	RET
	MOV	R28,R18		; Get register
	RET
	MOV	R28,R19		; Get register
	RET
	MOV	R28,R20		; Get register
	RET
	MOV	R28,R21		; Get register
	RET
	MOV	R28,R22		; Get register
	RET
	MOV	R28,R23		; Get register
	RET
	MOV	R28,R24		; Get register
	RET
	MOV	R28,R25		; Get register
	RET
	MOV	R28,R26		; Get register
	RET
	MOV	R28,R27		; Get register
	RET
	LDD	R28,Y+1		; Get R28
	RET
	LDD	R28,Y+2		; Get R29
	RET
	LDD	R28,Y+3		; Get R30
	RET
	LDD	R28,Y+4		; Get R31
	RET
	LDD	R28,Y+5		; Get SREG
	RET
	LDD	R28,Y+6		; Get PC (low)
	RET
	LDD	R28,Y+7		; Get PC (high)
	RET
; Table of register write handlers
RSTAB	STD	Y+0,R0		; Save R0
	RET
	MOV	R1,R0		; Set register
	RET
	MOV	R2,R0		; Set register
	RET
	MOV	R3,R0		; Set register
	RET
	MOV	R4,R0		; Set register
	RET
	MOV	R5,R0		; Set register
	RET
	MOV	R6,R0		; Set register
	RET
	MOV	R7,R0		; Set register
	RET
	MOV	R8,R0		; Set register
	RET
	MOV	R9,R0		; Set register
	RET
	MOV	R10,R0		; Set register
	RET
	MOV	R11,R0		; Set register
	RET
	MOV	R12,R0		; Set register
	RET
	MOV	R13,R0		; Set register
	RET
	MOV	R14,R0		; Set register
	RET
	MOV	R15,R0		; Set register
	RET
	MOV	R16,R0		; Set register
	RET
	MOV	R17,R0		; Set register
	RET
	MOV	R18,R0		; Set register
	RET
	MOV	R19,R0		; Set register
	RET
	MOV	R20,R0		; Set register
	RET
	MOV	R21,R0		; Set register
	RET
	MOV	R22,R0		; Set register
	RET
	MOV	R23,R0		; Set register
	RET
	MOV	R24,R0		; Set register
	RET
	MOV	R25,R0		; Set register
	RET
	MOV	R26,R0		; Set register
	RET
	MOV	R27,R0		; Set register
	RET
	STD	Y+1,R0		; Set R28
	RET
	STD	Y+2,R0		; Set R29
	RET
	STD	Y+3,R0		; Set R30
	RET
	STD	Y+4,R0		; Set R31
	RET
	STD	Y+5,R0		; Set SREG
	RET
	STD	Y+6,R0		; Set PC low
	RET
	STD	Y+7,R0		; Set PC high
	RET
;
; Sample user code application - PLACE YOUR CODE TO DEBUG HERE
;
USER	CLR	R30		; Start with zero
	CLR	R31		; ""
loop	RCALL	PUTWORD		; Display number
	ADIW	Z,1		; Increment
	RCALL	MONAVR		; Enter monitor to debug
	RJMP	loop		; And proceed
