/*
 * VT-100 compatible terminal program with many features
 * (See PC100.TXT for detailed information on this program)
 *
 * ?COPY.TXT 1991-2014 Dave Dunfield
 *  -- see COPY.TXT --.
 *
 * Permission granted for personal (non-commercial) use only.
 *
 * Compile with DDS MICRO-C compiler in 8086 TINY model.
 *
 * Compile command: cc pc100 -fop
 */
#include <stdio.h>
#include <window.h>
#include <comm.h>
#include <file.h>
#define	NAME	PC100
#define	VERSION	2.3
//#define	MEMMON

/* General definitions */
#define	FSIZE			32		// Max. size of a file name
#define	NFILES			8		// Number of file entries
#define	FWIDTH			65		// Width of file entries
#define	KWIDTH			65		// Width of key entries
#define	CFGVERSION		101		// Configuration version

#define	HEIGHT			24
#define	WIDTH			80

/* Script processor definitions */
#define	NUM_VARS		26		// Number of variables allowed
#define	CAPTURE_SIZE	256		// Sizeof TX capture buffer

#define	TICK	peekw(0x40,0x6C)

/*
 * Configuration parameters
 * These MUST all be initialized in order to have default values,
 * and to insure that they are of the same storage class.
 */
char homedir[FSIZE+1] = 0;			// Default home directory
char answerback[FSIZE+1] = 0;		// Answerback message
char filename[FSIZE+1] = { "*.*" };	// Last filename accessed
char scriptfile[FSIZE+1]= { "*" };	// Script file
unsigned
	com_port = 0,					// Com port (1)
	baud = 14,						// Baud rate (9600)
	parity = 4,						// Parity (None)
	dbits = 3,						// Data bits (8)
	sbits = 0,						// Stop bits (1)
	Flow = 0,						// Flow control (disabled)
	Margin = 999,					// Margin Bell position
	Mbell = 1000,					// Margin Bell frequency
	Bell = 1000,					// Bell frequency
	Bduration = 5,					// Bell duration
	Breaklen = 500,					// Break length
	hangup_delay = 3000,			// Delay to hangup modem
	key_string_delay = 0,			// Delay between chars in TX'd strings
	upl_char_delay = 0,				// Delay between characters on upload
	upl_line_delay = 0,				// Delay between lines on upload
	upl_sync_char = 0,				// Upload synchronize character
	DAvalue = 2,					// DA response value
	M132 = 0;						// 132 column mode
char alarm = 0,						// Sound alarm on errors
	echotty = 0,					// Echo data sent in tty
	echosnd = 0,					// Echo data sent in scripts
	sendlf = 0,						// Line-feed in ASCII upload
	sendnull = 0,					// Send space on null lines
	hangmdm = 0,					// Hangup modem on exit
	vt52 = 255,						// VT52 compatibility
	keypad = 0,						// Keypad application mode
	NewlineI = 0,					// Newline mode (input
	NewlineO = 0,					// Newline mode (output)
	Wrap = 255,						// Wrap at end of line
	Krepeat = 255,					// Keyboard repeat
	Uk = 0,							// UK character set
	Cfont = 255,					// Load custom font
	Cattr = 255,					// VGA attributes
	C132 = 0,						// 132 column
	Sscroll = 0,					// Slow scroll
	Cblock = 0;						// Cursor block
unsigned char attrs[] = {			// Window attributes
	0x07,							// 0: Main TTY screen
	0x70,							// 1: REVERSE
	0x01,							// 2: UNDERLINE
	0x10,							// 3: UNDERLINE+REVERSE
	0x0F,							// 4: BOLD
	0x74,							// 5: BOLD+REVERSE
	0x09,							// 6: BOLD+UNDERLINE
	0x19,							// 7: BOLD+UNDERLINE+REVERSE
	0x87,							// 8: BLINK
	0xF0,							// 9: BLINK+REVERSE
	0x81,							// 10: BLINK+UNDERLINE
	0xF1,							// 11: BLINK+UNDERLINE+REVERSE
	0x8F,							// 12: BLINK+BOLD
	0xF4,							// 13: BLINK+BOLD+REVERSE
	0x89,							// 14: BLINK+BOLD+UNDERLINE
	0xF9,							// 15: BLINK+BOLD+UNDERLINE+REVERSE
	0x70,							// 16: Status line
	0x70,							// 17: Error Messages
	0x70,							// 18: String prompts
	0x70,							// 19: Setup screen 1
	0x70,							// 20: Setup screen 2
#define	ATTR_EMULATE				   21
	0x07, 0x70, 0x0F, 0x78, 0x05, 0x50, 0x0D, 0x58,		// Emulated
	0x87, 0xF0, 0x8F, 0xF8, 0x85, 0xD0, 0x8D, 0xD8 };	// Attributes
char tabs[132] = { 0 };
#define	CONFIG_SIZE		223
char protocols[NFILES][FWIDTH] = { "ASCII \\F\\A" };
char scripts[NFILES][FWIDTH] = { 0 };
struct {
	char	Name[8];			// Function key name
	char	String[KWIDTH+1];	// Function key string
}	Fkeys[30] = { char
"{Esc}Oq",
"^[Oq",0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
"{Esc}Or",
"^[Or",0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
"{Esc}Os",
"^[Os",0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
"{Esc}Ot",
"^[Ot",0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
"{Esc}Ou",
"^[Ou",0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
"{Esc}Ov",
"^[Ov",0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
"{Esc}Ow",
"^[Ow",0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
"{Esc}Ox",
"^[Ox",0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
"{Esc}Oy",
"^[Oy",0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
"{Esc}Op",
"^[Op",0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
};
#include "keyboard.h"

/* Window attribute identifiers */
#define	TTY		0
#define	STATUS	16
#define	ERROR	17
#define	STRING	18
#define	SETUP1	19
#define	SETUP2	20
#define	_STATUS_	WSAVE|WCOPEN|attrs[STATUS]
#define	_ERROR_		WSAVE|WCOPEN|WBOX2|attrs[ERROR]
#define	_STRING_	WSAVE|WCOPEN|WBOX3|attrs[STRING]
#define	_SETUP1_	WSAVE|WCOPEN|WBOX1|attrs[SETUP1]
#define	_SETUP2_	(unsigned)WCOPEN|WBOX2|attrs[SETUP2]

/*
 * ANSI (VT100) Function key translation table
 */
unsigned char *ansi_keys[] = {
	"\x1B[A", "\x1B[B", "\x1B[D", "\x1B[C",		// Arrow keys
	"\x1BOP", "\x1BOQ", "\x1BOR", "\x1BOS" };	// PgUp, Pgdn, Home, End
unsigned char *vt52_keys[] = {
	"\x1BA", "\x1BB", "\x1BD", "\x1BC",			// Arrow keys
	"\x1BP", "\x1BQ", "\x1BR", "\x1BS" };		// PgUp, Pgdn, Home, End

unsigned char *ansi_keypad[] = {
	"\x1BOM", "\x1BOn",										// ENT ,
	"\x1BOq", "\x1BOr", "\x1BOs", "\x1BOt", "\x1BOu",		// 1-5
	"\x1BOv", "\x1BOw", "\x1BOx", "\x1BOy", "\x1BOp",		// 6-9 0
	"\x1BOm", "\x1BOl" };									// - ,
unsigned char *vt52_keypad[] = {
	"\x1B?M", "\x1B?n",										// ENT ,
	"\x1B?q", "\x1B?r", "\x1B?s", "\x1B?t", "\x1B?u",		// 1-5
	"\x1B?v", "\x1B?w", "\x1B?x", "\x1B?y", "\x1B?p",		// 6-9 0
	"\x1B?m", "\x1B?l" };									// - ,
unsigned char std_keypad[] = {
	0x0D, '.',
	'1', '2', '3', '4', '5', '6', '7', '8', '9', '0',
	'-', ',' };

char *main_menu[] = {		/* Main function menu */
	"Function menu",
	"Download file",
	"Upload file",
	"Kill capture",
	"Perform script",
	"Hangup modem",
	"Reset terminal",
	"Clear screen",
	"Configuration",
	"Shell to DOS",
	"Exit to DOS",
	0 };

char *config_menu[] = {		/* System configuration menu */
	"General switches",
	"General parameters",
	"Serial port settings",
	"Transfer protocols",
	"Tab stops",
	"Function keys",
	"Function menu scripts",
	"Video attributes",
	"Keyboard mapping",
	"Load configuration",
	"Save configuration",
	0 }

char *switches[] = {		/* General switch names */
	"Sound ALARM in messages",			// 0
	"Echo data sent in tty",			// 1
	"Echo data sent in scripts",		// 2
	"Send LF on ASCII uploads",			// 3
	"Send space in NULL lines",			// 4
	"Drop DTR+RTS on EXIT",				// 5
	"VT100/ANSI mode (N=VT52)",			// 6
	"Keypad application mode",			// 7
	"Newline mode (input)",				// 8
	"Newline mode (output)",			// 9
	"Wrap at end of line",				// 10
	"Keyboard auto-repeat",				// 11
	"UK charset (Shift-3 = '\x9C')",	// 12
	"VT-100 character font (VGA)",		// 13
	" + Emulate attributes (VGA)",		// 14
	"132 column mode",					// 15
	"Slow scroll",						// 16
	"Block cursor (otherwise line)" };	// 17
#define	SWITCH_VGA	13
#define	SWITCH_ATTR	14
#define	SWITCH_C132	15

char *serial_menu[] = {		/* Serial configuration menu */
	"Comm port",
	"Baudrate",
	"Parity",
	"Data bits",
	"Stop bits",
	"Flow control",
	0 }

char *setup_form[] = {		/* To set general parameters */
	52<<8|16,
	"\x01\x00\x20Home directory:",
	"\x01\x01\x20Answerback MSG:",
	"\x01\x02\x84Bell duration  (ms)    :",
	"\x01\x03\x84Bell frequency (hz)    :",
	"\x01\x04\x84Margin bell frequency  :",
	"\x01\x05\x83Margin bell position   :",
	"\x01\x06\x84Break length (ms)      :",
	"\x01\x07\x85Modem hangup delay (ms):",
	"\x01\x08\x85String TX char delay   :",
	"\x01\x09\x85ASCII Upload char delay:",
	"\x01\x0A\x85ASCII Upload line delay:",
	"\x01\x0B\x83ASCII Upload sync. char:",
	"\x01\x0C\x81VT100 DA response value:",
	"\x01\x0D\x83132 column BIOS mode   :",
	0 };

char *string_form[] = {		/* External protocols & scripts */
	68<<8|10,
	"\x00\x00\x401:",
	"\x00\x01\x402:",
	"\x00\x02\x403:",
	"\x00\x03\x404:",
	"\x00\x04\x405:",
	"\x00\x05\x406:",
	"\x00\x06\x407:",
	"\x00\x07\x408:",
	0 };

char *menus[] = {
	"Normal", "", "", "", "", "", "", "",
	"", "", "", "", "", "", "", "",
	"Status line",
	"Error message box's",
	"String input fields",
	"Main menu's",
	"Sub menu's" };

char *commands[] = {		/* Script commands */
	"print", "echo", "assign", "equate", "input", "read", "menu", "if#",
	"ifeq",	"ifne", "ifnot", "if", "goto", "skip", "stop", "call", "chain",
	"send", "scan", "flush", "wait", "upload", "download", "close", "dos",
	"visible", "invisible", "abort", "error", "message", "hangup", "terminal",
	"clear", "config", "exittodos", "log", "rem", 0 };

char *operators[] = {		/* Operators in script expression parser */
	"+", "-", "*", "/", "%", "&", "|", "^", "<<", ">>",
	"==", "!=", ">=", "<=", ">", "<",
	0 };

/* Common strings */
char cnfprompt[] = { "Configuration filename" };
char cnfext[] = { ".CFG" };

/* Uart configuration static data tables */
unsigned baudvalue[] = {
	2304,	1536,	1047,	857,	768,
	576,	384,	192,	96, 	64,
	58,		48,		32,		24,		12,
	6,		3,		2,		1 };
char *baudtext[] = {
	"50", "75", "110", "134.5",	"150",
	"200", "300", "600", "1200", "1800",
	"2000", "2400", "3600", "4800", "9600",
	"19200", "38400", "57600", "115200", 0 };
char *paritytext[] = { "Odd", "Even", "Mark", "Space", "None", 0 };
char *onetwo[] = { "One", "Two", 0 };
char *onefour[] = { "One", "Two", "Three", "Four", 0 };
char *fiveight[] = { "Five", "Six", "Seven", "Eight", 0 };
char *flowtext[] = { "Disabled", "Hardware", "Xon/Xoff", "Both", 0 };

/* Open window pointers */
struct WINDOW *tty, *status;

/* Global window pointer (to retain postion) */
unsigned int
	main_p,
	config_p,
	serial_p,
	proto_p,
	script_p;

/* Misc global variables */
unsigned
	Width = 80,				// Screen width
	Widthm1 = 79,			// Screen width-1
	Widthx24 = 80*24,		// Screen width*24
	key_delay,
	status_name_ptr,
	signals = SET_DTR|SET_RTS|OUTPUT_2;
unsigned char
	Leds,
	*Attrs,
	*key_ptr,
	*status_names[10],
	sent_cr,
	last_rx,
	active_keys,
	CapLK,				// Caps Lock enabled
	xy_flag,			// Cursor position update required
	cnffile[FSIZE+1] = { ""#NAME"" };
FILE
	*fp,
	*log_fp,		// Capture file pointer
	*upl_fp; 		// Upload/Download file pointers

/* Globals used by the script interpreter */
unsigned
	line,
	cptr,
	savenv[3];
unsigned char
	*optr,
	invisible,
	abort_flag,
	interactive = 255,
	SaveCfg[CONFIG_SIZE],			// Saved configuration
	capture[CAPTURE_SIZE],
	buffer[120];

extern register message_box(), write_status();
extern unsigned Cseg;

unsigned char font[] =
#include "vgafont.h"

unsigned char tty_translate[] = {
0x00,0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08,0x09,0x0A,0x0B,0x0C,0x0D,0x0E,0x0F,
0x10,0x11,0x12,0x13,0x14,0x15,0x16,0x17,0x18,0x19,0x1A,0x1B,0x1C,0x1D,0x1E,0x1F,
0x20,0x21,0x22,0x23,0x24,0x25,0x26,0x27,0x28,0x29,0x2A,0x2B,0x2C,0x2D,0x2E,0x2F,
0x30,0x31,0x32,0x33,0x34,0x35,0x36,0x37,0x38,0x39,0x3A,0x3B,0x3C,0x3D,0x3E,0x3F,
0x40,0x41,0x42,0x43,0x44,0x45,0x46,0x47,0x48,0x49,0x4A,0x4B,0x4C,0x4D,0x4E,0x4F,
0x50,0x51,0x52,0x53,0x54,0x55,0x56,0x57,0x58,0x59,0x5A,0x5B,0x5C,0x5D,0x5E,0x5F,
0x60,0x61,0x62,0x63,0x64,0x65,0x66,0x67,0x68,0x69,0x6A,0x6B,0x6C,0x6D,0x6E,0x6F,
0x70,0x71,0x72,0x73,0x74,0x75,0x76,0x77,0x78,0x79,0x7A,0x7B,0x7C,0x7D,0x7E,0x7F,
0xB1 };
unsigned char tty_text[] = { 0x5F,
0x60,0x61,0x62,0x63,0x64,0x65,0x66,0x67,0x68,0x69,0x6A,0x6B,0x6C,0x6D,0x6E,0x6F,
0x70,0x71,0x72,0x73,0x74,0x75,0x76,0x77,0x78,0x79,0x7A,0x7B,0x7C,0x7D,0x7E };
unsigned char tty_graph[] = { 0x20,
0x04,0xB1,0x80,0x81,0x82,0x83,0xF8,0xF1,0x84,0x85,0xD9,0xBF,0xDA,0xC0,0xC5,0x86,
0x87,0xC4,0x88,0x89,0xC3,0xB4,0xC1,0xC2,0xB3,0xF3,0xF2,0xE3,0xD8,0x9C,0xF9 };

unsigned
	kb_save[2];				// Saved interrupt handler

unsigned char
	KBscan,					// Last scancode received
	KBpre,					// Last keyboard prefix
	KBflags,				// Shift flags
	KBwp,					// Keyboard write pointer
	KBrp,					// Keyboard read pointer
	KBuffer[16];			// Type-ahead buffer

unsigned char KBpt[] = {
	0x48, 0x50, 0x4B, 0x4D,				// Up, Down, Left, Right
	0x52, 0x53, 0x47, 0x4F, 0x49, 0x51,	// Ins, Del, Home, End, PgUp, PgDn
	0x35, 0x1C, 0x1D, 0x38,				// KP/, KPent, Rctrl, Ralt
	0x37, 0x45, 0x5B, 0x5C,				// SYSrq, PAUSE, Lwin, Rwin
	0 };

kb_on() asm
{
		MOV		DX,OFFSET kbint		// Point at interrupt
		MOV		AX,2509h			// Set interrupt 9
		INT		21h					// Ask DOS
}

kb_off() asm
{
		PUSH	DS					// Save DS
		MOV		DX,DGRP:_kb_save+2	// Get offset
		MOV		DS,DGRP:_kb_save	// Get segment
		MOV		AX,2509h			// Set interrupt
		INT		21h					// Ask DOS
		POP		DS					// Restore DS
}

asm {
kbint:	PUSH	DS					// Save DS
		PUSH	AX					// Save AX
		PUSH	BX					// Save BX
		MOV		AX,CS				// Get our segment
		MOV		DS,AX				// Address
		IN		AL,60h				// Read data
		MOV		AH,AL				// Save for later
		IN		AL,61h				// Read control
		OR		AL,80h				// Set reset bit
		OUT		61h,AL				// Write it
		AND		AL,7Fh				// Clear reset bit
		OUT		61h,AL				// Write it
		MOV		AL,20h				// Reset PIC
		OUT		20h,AL				// Write it
// Check for prefix code on incoming data
		CMP		AH,0E0h				// Prefix code?
		JB		nopre				// Not a prefix
		MOV		DGRP:_KBpre,AH		// Save for later
		JMP 	kbexit				// And exit
// Check for last prefix = 0E1 - pause special prefix
nopre:	CMP BYTE PTR DGRP:_KBpre,0E1h 	// Special prefix?
		JNZ		nopre1				// No, try next
		MOV		DGRP:_KBpre,0E0h 	// Reset to E0
		JMP		kbexit				// And stop
// Check for last prefix = 0E0 - normal prefix
nopre1:	CMP BYTE PTR DGRP:_KBpre,0E0h 	// Normal prefix?
		JNZ		norkey				// No - standard keycode
		MOV		DGRP:_KBpre,0		// Reset to none
		MOV		AL,AH				// Copy key
		AND		AL,07Fh				// Remove clear bit
		XOR		BX,BX				// Zero counter
nopre2:	CMP		AL,_KBpt[BX]		// Does this one match?
		JZ		nopre3				// Yes, we have it
		INC		BX					// Advance
		TEST	BYTE PTR _KBpt[BX],0FFh	// At end?
		JNZ		nopre2				// Keep looking
		JMP SHORT kbexit			// Not found
doflag:	TEST	AH,80h				// Release?
		JNZ		doflg1				// Handle it
		OR		DGRP:_KBflags,BL	// Set the flag
		JMP	SHORT kbexit			// All done
doflg1:	NOT		BL					// Compliment
		AND		DGRP:_KBflags,BL	// Clear the flag
		JMP SHORT kbexit			// All done
nopre3:	AND		AH,80h				// Keep only break
		ADD		AH,BL				// Offset to key
		ADD		AH,06Eh				// Offset to available codes
// We have a keypress (AH) - Check for Shift or CTRL
norkey:	CMP		AH,DGRP:_KBscan		// Same as last?
		JNZ		nork1				// Allow it
		TEST	BYTE PTR DGRP:_Krepeat,0FFh	// Keyboard repeat set?
		JZ		kbexit				// Ignore this one
nork1:	MOV		DGRP:_KBscan,AH		// Save key
		MOV		AL,DGRP:_KBflags 	// Get shift flags
		AND		AL,3				// Only SHIFT & CTRL
		MOV		BX,OFFSET DGRP:_KB_ctrl // Assume CTRL
		CMP		AL,1				// Is CTRL key pressed?
		JZ		dokey				// Handle it
		MOV		BX,OFFSET DGRP:_KB_shift // Assume SHIFT
		CMP		AL,2				// Is SHIFT key pressed?
		JZ		dokey				// Handle it
		MOV		BX,OFFSET DGRP:_KB_std 	// Must be standard
dokey:	MOV		AL,AH				// Get char
		AND		AL,7Fh				// Mask
		ADD		BL,AL				// Index into key array
		ADC		BH,0				// Carry forward
		MOV		AL,[BX]				// Get function
		AND		AL,AL				// Dead key?
		JZ		kbexit				// Ignore
		MOV		BX,1				// CTRL
		CMP		AL,0B6h				// Control?
		JZ		doflag				// Handle it
		MOV		BL,2				// SHIFT
		CMP		AL,0B5h				// Shift?
		JZ		doflag				// Handle it
		TEST	AH,80h				// Key-up?
		JNZ		kbexit				// Dont process
		MOV		BL,_KBwp			// Get write pointer
		MOV		DGRP:_KBuffer[BX],AL // Save code
		INC		BL					// Next
		AND		BL,15				// Mask
		MOV		DGRP:_KBwp,BL		// Save it
kbexit:	POP		BX					// Restore BX
		POP		AX					// Restore AX
		POP		DS					// Restore DS
		IRET
}

wrkbd(c) asm
{
x1:		IN		AL,64h				// Read status
		AND		AL,02h				// Test ready
		JNZ		x1					// Wait for it
		MOV		AL,4[BP]			// Get code
		OUT		60h,AL				// Write it
}


/*
 * Display a message in a box
 */
register message_box(args)
	unsigned args;
{
	char buf[80];
	unsigned l;

	l = _format_(nargs() * 2 + &args, buf) + 16;

	wopen((Width-l)/2, 10, l+2, 3, _ERROR_);
	wcursor_off();
	wprintf(" %s (Press ENTER) ", buf);
	if(alarm) wputc(7);
	while(wgetc() != '\n');
	wclose();
}

/*
 * Write a message into the status line
 */
register write_status(args)
	unsigned args;
{
	char buf[80];

	_format_(nargs() * 2 + &args, buf);
	w_printf(status,"\r%s", buf);
	w_cleow(status);
	xy_flag = 255;
}

/*
 * Update status line
 */
update_status()
{
	int i, j;
	unsigned char a, a1;

	xy_flag = 255;
	switch(active_keys & 3) {
	case 0 :
	default:
		w_printf(status,"\rCOM%u:%6s-%u%c%u%c\xB3%3s\xB3",
			com_port+1, baudtext[baud], dbits+5, *paritytext[parity], sbits+1,
			*flowtext[Flow], log_fp ? "Cap" : "");
		for(i=0; i < status_name_ptr; ++i)
			w_printf(status," %s", status_names[i]);
		w_cleow(status);
		w_gotoxy(Width-5, 0, status);
		if(!vt52)
			w_puts("VT52", status);
		else if(a=Leds) {
			for(i=0; i < 4; ++i) {
				w_putc((a & 1) ? '*' : '-', status);
				a >>= 1; } }
		if(keypad) {
			w_gotoxy(Widthm1, 0, status);
			w_putc('K', status); }
		return;
	case 0x01: j = 20; goto dokey;
	case 0x02: j = 10; goto dokey;
	case 0x03: j = 0;  dokey:
		a = *status;
		a1 = (a << 4) | (a >> 4);
		for(i=0; i < 10; ++i) {
			w_gotoxy(i*8, 0, status);
			*status = a1;
			w_printf(status, "%u", (i+1)%10);
			*status = a;
			w_printf(status, "%-7s", Fkeys[i+j].Name); } }
}

//unsigned Z1, Z2, Z3, Z4, Z6//
asm {	// Support functions
//		EXTRN	_W_BASE:WORD
		EXTRN	_W_COLS:byte
// Setup pointers to window and video buffer
XSETUP:	POP		AX					// Get return
		MOV		SI,DGRP:_tty		// Get tty window
		PUSH	ES					// Save ES
		MOV		ES,DGRP:_W_BASE		// Point to video base
		MOV		DX,6[SI]			// Get 'X' and 'Y' position
		JMP		AX					// Return to caller
// Calculate screen address in BX (DH=Y DL=X)
XYADDR:	PUSH	AX					// Save AX
		XOR		AH,AH				// Zero high
		MOV		AL,DH				// Get 'Y'
		ADD		AL,3[SI]			// Offset from start of screen
		MUL		DGRP:_W_COLS		// Calculate 'Y' offset
		MOV		BL,DL				// Get 'X' position
		ADD		BL,2[SI]			// Offset from start of screen
		XOR		BH,BH				// Zero high
		ADD		BX,AX				// BX = character position
		SHL		BX,1				// Adjust for words
		POP		AX
		RET
}

// Clear start of window
wclsow() asm
{
		CALL	XSETUP				// Address screen
		CALL	XYADDR				// BX = video address
		MOV		DI,BX				// DI = out address
		XOR		DX,DX				// X=Y=0
		JMP		short xclr			// And clear
}

wclsol() asm
{
		CALL	XSETUP				// Address screen
		CALL	XYADDR				// BX=Our address
		MOV		DI,BX				// DI=Our address
		XOR		DL,DL				// Zero X
xclr:	CALL	XYADDR				// BX=Start of screen
		MOV		AH,[SI]				// Get attribute
		MOV		AL,' '				// Space
		SUB		BX,2				// Pre-adjust
clsol1:	ADD		BX,2				// Advance
		MOV		ES:[BX],AX			// Write to screen
		CMP		BX,DI				// Are we there?
		JB		clsol1				// Do all
		POP		ES					// Restore ES
}

rscroll() asm
{
		CALL	XSETUP				// Address screen
		XOR		DX,DX				// Get base address
		CALL	XYADDR				// BX=Base address
		MOV		CX,BX				// CX=Base address
		MOV		DX,4[SI]			// Get window size
		DEC		DH
		CALL	XYADDR				// BX=window size
		MOV		DI,BX				// Get end address
		XOR		DH,DH				// Zero high
		SUB		BX,DX				// Adjust to previous line
		SUB		BX,DX				// x2 for word entries
// CX=Base address DI=End address BX=END-1line
rsc1:	CMP		BX,CX				// Are we at start
		JBE		rsc2				// All done
		SUB		BX,2				// Backup
		SUB		DI,2				// Backup
		MOV		AX,ES:[BX]			// Get from source
		MOV		ES:[DI],AX			// Write data
		JMP		SHORT rsc1			// Do it all
rsc2:	MOV		AH,[SI]				// Get attribute
		MOV		AL,' '				// Get space
rsc3:	MOV		ES:[BX],AX			// Clear one
		ADD		BX,2				// Advance to next
		DEC		DL					// Reduce count
		JNZ		rsc3				// Do it all
		POP		ES					// Restore
}

unsigned
	Tseg,				// Text segment
	Aseg,				// Attribute segment
	Cpos,				// Current position
	Blines,				// Buffered lines
	Mlines = 819,		// Maximum lines
	SaveBuf[24][132];	// Screen save buffer

// Save screen to SaveBuf
void save() asm
{
		MOV		DX,DS						// Save DS
		MOV		DI,offset DGRP:_SaveBuf		// Point to save buffer
		XOR		SI,SI						// Save from 00
		MOV		CX,DGRP:_Widthx24			// Get width * 24
		MOV		ES,DX						// Set ES
		MOV		DS,DGRP:_W_BASE				// Video base segment
	rep	MOVSW								// Copy it
		MOV		DS,DX						// Restore DS
}
// Restore screen from SaveBuf
void restore() asm
{
		MOV		SI,offset DGRP:_SaveBuf		// Point to save buffer
		XOR		DI,DI						// Restore to 00
		MOV		ES,DGRP:_W_BASE				// Get video segment
		MOV		CX,DGRP:_Widthx24			// Width*24
	rep	MOVSW								// Copy it
}

// Save top line of screen to external segments
void saveline(void) asm
{
		PUSH	SI
		PUSH	DX
		PUSH	ES
		MOV		CX,DGRP:_Width				// Do 80 bytes
		MOV		DI,DGRP:_Cpos				// Get current position
		SUB		DI,CX						// Backup
		MOV		DGRP:_Cpos,DI				// Resave
		XOR		SI,SI						// From sero
		MOV		BX,DGRP:_Tseg				// Text segment
		MOV		DX,DGRP:_Aseg				// Attribute segment
		MOV		DS,DGRP:_W_BASE				// Video segment
sl1:	MOV		AX,[SI]						// Get from DS (video)
		MOV		ES,BX						// Text
		MOV		ES:[DI],AL					// Save text
		MOV		ES,DX						// Attrib
		MOV		ES:[DI],AH					// Save Attrib
		ADD		SI,2						// Next in source
		INC		DI							// Next in dest
		LOOP	sl1							// Save all
		MOV		AX,SS						// Get our DS
		MOV		DS,AX						// Restore DS
		POP		ES							// Restore ES
		POP		DX							// Restore DX
		POP		SI							// Restore SI
		MOV		AX,DGRP:_Blines				// Get line counter
		CMP		AX,DGRP:_Mlines				// At max?
		JNC		sl2							// Yes, no advance
		INC		AX							// Increment
		MOV		DGRP:_Blines,AX				// And resave
sl2:
}

// Get line from review-log to screen
void getline(to, from) asm
{
		MOV		AX,23					// Max Y
		SUB		AX,6[BP]				// Convert 'to' to reverse offset
		MOV		CX,DGRP:_Width			// Size of screen
		SHL		CX,1					// *2 for word entries
		MUL		CX						// Adjust
		MOV		DI,AX					// Destination address
		MOV		AX,4[BP]				// Get 'from'
		SUB		AX,24					// Local?
		JNC		gl1						// No - get from segment
// Lines 0-23 from SaveBuf
		MOV		AX,23					// Max Y
		SUB		AX,4[BP]				// Convert 'from' to reverse index
		MUL		CX						// Adjust
		ADD		AX,offset DGRP:_SaveBuf	// Point to buffer
		MOV		SI,AX					// Set source
		MOV		ES,DGRP:_W_BASE			// Point to screen
		SHR		CX,1					// Back to width
	rep	MOVSW							// Move them
		POP		BP
		RET
// Lines 24+ from external segment
gl1:	SHR		CX,1					// Back to width
		MUL		CX						// Adjust
		ADD		AX,DGRP:_Cpos			// And get current position
		MOV		SI,AX					// Set source
		MOV		BX,DGRP:_Tseg			// Text segment
		MOV		DX,DGRP:_Aseg			// Attribute segment
		MOV		DS,DGRP:_W_BASE			// Video segment
gl2:	MOV		ES,BX					// Text segment
		MOV		AL,ES:[SI]				// Get text
		MOV		ES,DX					// Attribute segment
		MOV		AH,ES:[SI]				// Get attribute
		MOV		[DI],AX					// Write to video
		INC		SI						// Next sourcer
		ADD		DI,2					// Next video
		LOOP	gl2						// Do them all
		MOV		AX,SS					// Get DS
		MOV		DS,AX					// Restore DS
}

// Get line from review-log to screen
void gettext(dest, from) asm
{
		MOV		DI,6[BP]				// Get destination
		MOV		AX,4[BP]				// Get 'from'
		MOV		CX,DGRP:_Width			// Size of line
		SUB		AX,24					// Local?
		JNC		gt2						// No - get from segment
// Lines 0-23 from SaveBuf
		MOV		AX,23					// Max Y
		SUB		AX,4[BP]				// Convert 'from' to reverse index
		SHL		CX,1					// *2 for word entries
		MUL		CX						// Adjust
		ADD		AX,offset DGRP:_SaveBuf	// Point to buffer
		MOV		SI,AX					// Set source
		SHR		CX,1					// Back to width
gt1:	MOV		AX,[SI]					// Get from source
		MOV		[DI],AL					// Save
		ADD		SI,2					// Next in source
		INC		DI						// Next in dest
		LOOP	gt1						// Do them all
		JMP short gt4					// Clean up
// Lines 24+ from external segment
gt2:	MUL		CX						// Adjust
		ADD		AX,DGRP:_Cpos			// And get current position
		MOV		SI,AX					// Set source
		MOV		ES,DGRP:_Tseg			// Text segment
gt3:	MOV		AL,ES:[SI]				// Get text
		MOV		[DI],AL					// Write to video
		INC		SI						// Next in source
		INC		DI						// Next dest
		LOOP	gt3						// Do them all
		MOV		AX,SS					// Get DS
		MOV		DS,AX					// Restore DS
gt4:	MOV		SI,6[BP]				// Get pointer
gt5:	MOV		AL,-1[DI]				// Get previous
		CMP		AL,' '					// Space?
		JNZ		gt6						// No - stop here
		DEC		DI						// Backup
		CMP		DI,SI					// At begining?
		JA		gt5						// Keep looking
gt6:	MOV	byte ptr [DI],0				// Zero terminate
}

// Review terminal capture
void review()
{
	unsigned i, p;

	save();
	wcursor_off();
	p = 0;
	write_status("%8sof %u  -  REVIEW  -  Esc=EXIT  F1=SAVE  F2=CLEAR", "", Blines+24);
redraw:
	if(p > Blines)
		p = Blines;
	w_printf(status, "\r%3u-%-3u", p+1, p+24);
	for(i=0; i < 24; ++i)
		getline(i, p+i);
	for(;;) switch(toupper(wgetc())) {
	case _KUA: ++p;			goto redraw;
	case _KDA: if(p) --p;	goto redraw;
	case _KPU: p += 24;		goto redraw;
	case _KHO: p = 65535;	goto redraw;
	case _KPD:
		if((p -= 24) < Blines)
			goto redraw;
	case _KEN: p = 0;		goto redraw;
	case 0x1B:
		restore();
		return;
	case _K1:			// Save
		concat(buffer, "File to ", "SAVE");
		if(!get_string(buffer, filename, FSIZE)) {
			restore();
			i = open_file(filename, "w");
			save();
			if(i) {
				i = Blines + 24;
x1:				gettext(buffer, --i);
				fputs(buffer, fp);
				if(i) {
					putc('\n', fp);
					goto x1; }
				*(unsigned*)buffer = 'N';
				if(yesno("Continue capture", 'N')) {
					if(log_fp)
						fclose(log_fp);
					log_fp = fp;
					continue; }
				fclose(fp); } }
			goto redraw;
	case _K2:
		if(yesno("Clear review buffer", 'Y')) {
			Cpos = Blines = 0;
			if(yesno("Clear screen", 'Y')) {
				restore();
				wclwin();
				save(); } }
		goto redraw; }
}

unsigned char
	VGA,					// 0==VGA available
	VGA_font,				// VGA font is active
	VGA_attr,				// VGA special attributes are active
	Font_page,				// Current font page
	Font_save[4];			// Save area

/*
 * Initialize VGA display & Record FONT information
 */
void init_vga() asm
{
		MOV		AX,1A00h			// Get display code
		INT		10h					// Call BIOS
		SUB		AL,1Ah				// Return 0 if VGA
		MOV		_VGA,AL				// Save for later
		JNZ		ivga1				// No more action
// Read existing color level
		MOV		AX,1007h			// Read individual register
		MOV		BL,12h				// Register
		INT		10h					// Ask BIOS
		MOV		DGRP:_Font_save,BH	// Save value
// Read Color-5 palette entry
		MOV		AX,1015h			// Get color
		MOV		BX,5				// Get color
		INT		10h					// Read it
		MOV		BYTE PTR DGRP:_Font_save+1,DH
		MOV		WORD PTR DGRP:_Font_save+2,CX
ivga1:
}

/*
 * Enable a prevusly loaded font
 */
void font_on(void) asm
{
		MOV		DGRP:_VGA_font,255
		PUSH	ES					// Save ES
		MOV		AX,1130h			// Get font information
		MOV		BH,0				// Current font
		INT		10h					// Ask BIOS
		PUSH	DS					// Get DS
		POP		ES					// ES = DS
		CMP		CX,16				// Is current font > 16 points?
		JA		fo1					// Yes, special case
// Font is 16 points or less, just use one we have
		MOV		AX,1100h			// Set font
		MOV		BH,10h				// 16 point font
		MOV		BL,DGRP:_Font_page	// Select this page
		MOV		CX,0100h			// 256 charcters
		XOR		DX,DX				// Start with character 0
		MOV		BP,OFFSET DGRP:_font
		INT		10h					// Ask BIOS
		JMP SHORT fo3				// And exit
// Font is greater than 16 points
// We must translate our 16 point font to match (zero pad)
fo1:	MOV		BH,CL				// BH = points
		XOR		BL,BL				// * 256
		SUB		SP,BX				// Allocate space for new font
		MOV		DI,SP				// Dest is new buffer
		MOV		SI,OFFSET DGRP:_font
		MOV		DX,SI				// Copy
		ADD		DX,4096				// Adjust to buffer size
// Copy in our font data
fo2:	MOV		CX,16				// Copy 16 bytes
	REP	MOVSB						// Copy the character
		MOV		CL,BH				// Get point size
		SUB		CL,16				// Calculate # bytes remaining
		XOR		AX,AX				// Get zero
	REP STOSB						// Clear the end
		CMP		SI,DX				// Are we at end?
		JB		fo2					// Do all data
		MOV		AX,1100h			// Set font
		MOV		BL,DGRP:_Font_page	// Get page to load
		MOV		CX,0100h			// 256 characters
		XOR		DX,DX				// Character 0
		MOV		BP,SP				// Use new font table
		INT		10h					// Ask BIOS
		MOV		SP,DI				// Fix SP
fo3:	POP		ES					// Restore ES
}

/*
 * Disable loaded font / enable standard VGA font
 */
void font_off(void) asm
{
		MOV		DGRP:_VGA_font,0
		MOV		AX,1104h			// Load ROM 8x16 font
		MOV		BL,00h				// Font bank 0
		INT		10h					// Ask BIOS
}

/*
 * Generate and enable the underline font
 */
void enable_c2(void)
{
	unsigned i, j;
	unsigned char buf[sizeof(font)/16];

	VGA_attr = 255;
	j = 0;
	for(i=14; i < sizeof(font); i += 16) {
		buf[j++] = font[i];
		font[i] = 0xFF; }
	Font_page = 1;
	font_on();
	asm {
// Direct BIOS to allow two character fonts
		MOV		AX,1103h			// Set block specifier
		MOV		BL,04h				// !att3=Block0 att3=Block1
		INT		10h					// Ask BIOS
// Set new color level
		MOV		AX,1000h			// Set color level
		MOV		BX,0712h			// Set it
		INT		10h					// As BIOS
// Set new Color-5 palette entry
		MOV		AX,1010h			// Set color
		MOV		BX,5				// Color-5
		MOV		DH,3Fh				// Get color
		MOV		CH,DH				// Set
		MOV		CL,DH				// Set
		INT		10h					// Ask BIOS
	}
	Font_page = j = 0;
	for(i=14; i < sizeof(font); i += 16)
		font[i] = buf[j++];
}

/*
 * Disable dual character set mode
 */
void disable_c2(void) asm
{
		MOV		DGRP:_VGA_attr,0
// Disallow two character sets
		MOV		AX,1103h			// Set block specifier
		MOV		BL,00h				// Always block0
		INT		10h					// Ask BIOS
// Set new color level
		MOV		AX,1000h			// Set color level
		MOV		BL,12h				// Set it
		MOV		BH,DGRP:_Font_save
		INT		10h					// As BIOS
// Set new Color-5 palette entry
		MOV		AX,1010h			// Set color
		MOV		BX,5				// Color-5
		MOV		DH,BYTE PTR DGRP:_Font_save+1
		MOV		CX,WORD PTR DGRP:_Font_save+2
		INT		10h					// Ask BIOS
}

unsigned char help[] = { "\n\
Use:	"#NAME" [options] [script [arguments]]\n\n\
Opts:	-I		- inhibit Interactive session (script only)\n\
	C=file[.CFG]	- specify Configuration file\n\
	F=file[.VGA]	- specify Font file\n\
	P=1-4		- specify com Port\n\
\n?COPY.TXT 1991-2014 Dave Dunfield ("#__DATE__")\n -- see COPY.TXT --.\n" };

unsigned SaveX, SaveY, ScrollT, ScrollB = HEIGHT-1;
unsigned char Attr, Cset, SaveA, SaveC, Cursor, Origin, G0set, G1set, Pnl;

int load_config(void)
{
	unsigned v;
	fget(&v, sizeof(v), fp);
	if(v == CFGVERSION) {
		fget(homedir, ansi_keys-homedir, fp);
		fclose(fp);
		open_comm(); }
	else {
		fclose(fp);
		message_box("Config file is incorrect version"); }
}

setattr()
{
	Attrs = attrs;
	if(Cattr)
		Attrs += ATTR_EMULATE;
}

open_windows()
{
	if((VGA_font = Cfont) && !VGA) {
		font_on();
		if(VGA_attr = Cattr)
			enable_c2(); }
	else
		VGA_font = VGA_attr = Cfont = Cattr = 0;
	setattr();
	status = wopen(0, 24, Width, 1, _STATUS_);
	tty = wopen(0, 0, Width, 24, WSAVE|WLF|WSCROLL|*Attrs);
}

void setmode() asm
{
// If no 132 column mode - do nothing
		MOV		AX,DGRP:_M132		// Get 132 column mode
		AND		AX,AX				// Does it exist?
		JNZ		setm1				// Ok
		MOV		DGRP:_C132,0		// Turn feature OFF
		POP		BP
		RET
// Mode is a VESA mode
vesa:	MOV		BX,AX				// BX = mode
		MOV		AX,DS				// Get segment
		MOV		ES,AX				// Set in ES
		LEA		DI,256[BP]			// Free memory
		MOV		AX,4F02h			// VESA setmode
		INT		10h					// Ask BIOS
		CMP		AX,004Fh			// Did it work?
		JZ		setm2				// Yes, success
		JMP		SHORT sfail			// No
setw:	MOV		DGRP:_Mlines,CX		// Save max lines
		MOV		DGRP:_Widthm1,AX	// Set width-1
		INC		AX					// Adjust to width
		MOV		DGRP:_Width,AX		// Set width
		MOV		CX,24				// *24
		MUL		CX					// Calculate screen size
		MOV		DGRP:_Widthx24,AX	// Save size
		XOR		AX,AX				// Get zero
		MOV		_Cpos,AX			// Clear position
		MOV		_Blines,AX			// Clear buffered lines
		RET
// Set 132 or 80 column mode
setm1:	CALL	_wclose				// Close tty
		CALL	_wclose				// Close status
		TEST BYTE PTR DGRP:_C132,0FFh	// 132 column mode?
		JZ		s80					// No, 80 cols
		MOV		AX,131				// Width-1
		MOV		CX,496				// Maximum lines
		CALL	setw				// Set width
		MOV		AX,DGRP:_M132		// 132 column mode
		AND		AH,AH				// Vesa mode?
		JNZ		vesa				// Do vesa mode
		INT		10h					// Ask BIOS
		MOV		AH,0Fh				// Get mode
		INT		10h					// Ask BIOS
		CMP		AL,BYTE PTR DGRP:_M132 // Does it match?
		JZ		setm2				// Yes, its OK
sfail:	MOV		DGRP:_C132,0		// Reset mode
s80:	MOV		AX,79				// Width-1
		MOV		CX,819
		CALL	setw				// Set width
		MOV		AX,0003h			// Mode-3
		INT		10h					// Ask BIOS
		MOV		AH,0Fh				// Get mode
		INT		10h					// Ask BIOS
		CMP		AL,3				// Did it take?
		JZ		setm2				// Yes, mode is OK
		MOV		AX,0007h			// Try monochrome mode
		INT		10h					// Ask BIOS
setm2:	CALL	_open_windows		// Open the windows
}

getupper() asm
{
		MOV		BX,DGRP:_optr		// Get pointer
		MOV		AL,[BX]				// Get value
		INC		BX					// Next
		MOV		DGRP:_optr,BX		// Resave
		CMP		AL,'a'				// Lower?
		JB		tok					// No
		CMP		AL,'z'				// Lower?
		JA		tok					// No
		AND		AL,0DFh				// Convert to upper
tok:	XOR		AH,AH				// Zero high
}

/*
 * Main program
 */
main(argc, argv)
	int argc;
	char *argv[];
{
#ifdef MEMMON
	asm {
		MOV		SI,offset DGRP:?heap + 16
		MOV		DI,SP
		SUB		DI,256
fml:	MOV		byte ptr [SI],0A5h
		INC		SI
		CMP		SI,DI
		JB		fml
	}
#endif
	if(!(Cseg = alloc_seg(4096*3)))
		abort("No memory");
	Aseg = (Tseg = Cseg + 4096) + 4096;
	line = 0; do {
		pokew(Cseg, line, 0);
		pokew(Aseg, line, 0x0C0C);
		pokew(Tseg, line, 0xA8A8); }
	while(line -= 2);

	for(line=1; line < argc; ++line) {
		optr = argv[line];
		switch((getupper() << 8) | getupper()) {
			case 'C=' :		/* Specify CONFIG file */
				SaveX = optr;
				continue;
			case 'F=' :		// Font file
				parse_filename(buffer, ".VGA");
				fp = fopen(buffer, "rvqb");
				if(	(getc(fp) != 'V') || (getc(fp) != 'G') || (getc(fp) != 'A')) {
		badfnt:		abort("Corrupt font file\n"); }
				if(fget(font, sizeof(font)+1, fp) != sizeof(font))
					goto badfnt;
				fclose(fp);
				continue;
			case 'P=' : SaveY = *optr-'1';	continue;
			case '/I' :		/* Disable interactive session */
			case '-I' :
				interactive = 0;
				continue;
			case '/?' :
			case '-?' :
			case '?' << 8:
				abort(help);
			default:
				while(argc < 20)
					argv[argc++] = 0;
				argc = 0; } }
	set_tabs(8);

	optr = SaveX || cnffile;
	parse_filename(buffer, cnfext);
	if(fp = fopen(buffer, "rb"))
		goto lc;
	optr = argv[0];
	while(*optr) ++optr;
	strcpy(optr-4, cnfext);
	if(fp = fopen(argv[0], "rb")) {
lc:		load_config(); }
	if(SaveY)
		com_port = SaveY;

	memcpy(SaveCfg, &com_port, sizeof(SaveCfg));
	init_vga();
	open_windows();

	if(SaveX && !fp)
		message_box("Unable to open CONFIG file: %s", buffer);

	IOB_size = 512;
	asm {
		MOV		AX,3509h			// Get interrupt 9
		INT		21h					// Ask DOS
		MOV		DGRP:_kb_save,ES	// Save segment
		MOV		DGRP:_kb_save+2,BX	// Save offset
	}

	tty_reset();
	w_gotoxy(22, 0, status);
	w_puts(""#NAME" "#VERSION" by Dave Dunfield ", status);

	Findkey("  %s=MENU", 0xB4);
	Findkey("  %s=REVIEW", 0xBA);

	if(!argc)
		script(&argv[line-1]);

	while(interactive) {
		if(ansi_term() == -2) {
			review();
			update_status();
			continue; }
		if(*(unsigned*)buffer == 0xAABB) {		// Pending script
			*(unsigned*)buffer = 0;
			script(buffer+2);
			continue; }
		if(!wmenu(5, 3, _SETUP1_, main_menu, &main_p)) switch(main_p) {
			case 0 :	/* Fast access script */
				if(optr = file_menu(scripts, &script_p, 0)) {
					parse_args(buffer);
					script(buffer); }
				break;
			case 1 :	/* Download */
				if(optr = file_menu(protocols, &proto_p, 0))
					file_transfer(0, 0);
				break;
			case 2 :	/* Upload */
				if(optr = file_menu(protocols, &proto_p, 0))
					file_transfer(-1, 0);
				break;
			case 3 :	/* Close capture */
				if(log_fp)
					fclose(log_fp);
				log_fp = 0;
				update_status();
				break;
			case 4 :	/* Script by name */
				if(!get_string("Script", optr = scriptfile, FSIZE)) {
					parse_args(buffer);
					script(buffer); }
				break;
			case 5 :	/* Hangup */
				hangup();
				break;
			case 6 :	// Reset
				tty_reset();
				break;
			case 7 :	/* Clear screen */
				wclwin();
				break;
			case 8 :	/* Configure */
				configure();
				memcpy(SaveCfg, &com_port, sizeof(SaveCfg));
				break;
			case 9 :	/* Shell to DOS */
				doshell(0);
				break;
			case 10 :	/* Exit */
				interactive = 0; } }

	if(log_fp)
		fclose(log_fp);
	if(hangmdm) {
		signals &= ~(SET_DTR|SET_RTS);
		open_comm(); }
	Cclose();
	if(C132) {
		C132 = 0;
		setmode(); }
	wclose();
	wclose();
	if(VGA_font) {
		font_off();
		if(VGA_attr)
			disable_c2(); }
#ifdef MEMMON
	asm {
		EXTRN	?heap:near
		MOV		SI,offset DGRP:?heap + 16
		MOV		DI,0FF80h
		XOR		CX,CX
al1:	XOR		BX,BX
al2:	CMP		SI,DI
		JA		al4
		MOV		AL,[SI]
		INC		SI
		CMP		AL,0A5H
		JNZ		al3
		INC		BX
		JMP short al2
al3:	AND		BX,BX
		JZ		al1
		CMP		BX,CX
		JBE		al1
		MOV		CX,BX
		JMP		al1
al4:	MOV		DGRP:_Cseg,CX
	} printf("\n%u %u=%u %u\n", Cseg, protocols-&com_port, sizeof(SaveCfg), sizeof(font));
#endif

}

/*
 * Open comm port with correct settings
 */
open_comm()
{
	int mode;

	/* Calculate the communications parameter value */
	mode =	((parity << 4) & 0x30) |	/* parity type */
			(dbits & 0x03) |			/* # data bits */
			((sbits << 2) & 0x04) |		/* # stop bits */
			((parity < 4) << 3);		/* parity enable */

	/* Open the communications port */
	if(Copen(com_port+1, baudvalue[baud], mode, signals)) {
		message_box("Cannot open COM port %u!", com_port+1);
		return 0; }

	/* Remove transparency if XON/XOFF flow control */
	disable();
	Cflags &= ~(TRANSPARENT|HFLOW);
	if(Flow & 1) Cflags |= HFLOW;
	if(!(Flow & 2)) Cflags |= TRANSPARENT;
	enable();

	return -1;
}

/*
 * Hangup the modem
 */
hangup()
{
	write_status("Hanging up the modem...");
	signals &= ~(SET_DTR|SET_RTS);
	open_comm();
	wait_clear(hangup_delay);
	signals |= (SET_DTR|SET_RTS);
	open_comm();
	update_status();
}

/*
 * Write a string to the comm port
 */
Cputs(ptr)
	char *ptr;
{
	int c;
	unsigned timeout;

	while(*ptr) {
		timeout = key_string_delay;
		do {
			if((c = Ctestc()) != -1)
				receive_data(c);
			if((c = Ctestc()) != -1)
				receive_data(c); }
		while(timeout--);
		Cwrite(*ptr++); }
	testabort();
}

/*
 * Write char to comm port with echo if enabled
 */
Cwrite(c)
	int c;
{
	Cputc(c);
	if(echosnd)
		wputc(c);
}


/*
 * Get a string from the console
 */
get_string(prompt, name, length)
	char *prompt, *name;
	int length;
{
	int i, j;

	i = (j = strlen(prompt))+length+7;
	wopen((Width-i)/2, 10, i, 3, _STRING_);
	for(;;) {
		wgotoxy(1, 0);
		wprintf("%s ? ", prompt);
		switch(wgets(j+4, 0, name, length)) {
			case 0x1B :		/* Exit */
				wclose();
				return -1;
			case '\n' :		/* Select */
				wclose();
				return 0; } }
}

int yesno(unsigned char *prompt, int c)
{
	*(unsigned*)buffer = c;
	for(;;) {
		if(!get_string(prompt, buffer, 1)) switch(*buffer) {
			case 'y' :
			case 'Y' :
				return 255;
			case 'n' :
			case 'N' :
			case 0x1B:
				return 0; } }
}

/*
 * Open a file with options
 */
open_file(filename, options)
	char *filename, *options;
{
	int n, p, c, i;
	char **names, *nptr, *ptr, *ptr1;
	names = SaveBuf;

	for(ptr = filename; i = *ptr; ++ptr)
		if((i == '*') || (i == '?'))
			break;
	if(i) {
		if(find_first(filename, n = 0, nptr = SaveBuf+200, &i, &i, &i, &i, &i)) {
			message_box("No files matching: '%s'", filename);
			return 0; }
		wopen(13, 5, 53, 12, _STRING_);
		wcursor_off();
		do {
			for(i=0; i < n; ++i)
				if(strcmp(names[i], nptr) == 1)
					break;
			for(c = ++n; c > i; --c)
				names[c] = names[c-1];
			names[i] = nptr;
			while(*nptr++); }
		while((n < 100) && !find_next(nptr, &i, &i, &i, &i, &i));
		for(i=n; i < 100; ++i)
			names[i] = "";

		c = p = 0;
		for(;;) {
			for(i=0; i < 40; ++i) {
				wgotoxy((i%4)*13, i/4);
				*W_OPEN = (i == c) ? ((*W_OPEN >> 4)|(*W_OPEN << 4)) : attrs[STRING];
				wputf(names[i+p], 12); }
			switch(wgetc()) {
				case 0x1B :
					wclose();
					return 0;
				case '\n' :
					ptr = ptr1 = filename;
					while(i = *ptr++) {
						if((i == '\\') || (i == ':'))
							ptr1 = ptr; }
					strcpy(ptr1, names[p+c]);
					wclose();
					goto open_file;
				case _KRA : i = c + 1; goto position;
				case _KLA : i = c - 1; goto position;
				case _KUA : i = c - 4; goto position;
				case _KDA : i = c + 4;
				position:
					if(i < 0) {
						c = 0;
						if((p - 4) >= 0) {
							c = i + 4;
							p -= 4; }
						break; }
					if(i >= 40) {
						if((p + 40) < n) {
							i -= 4;
							p += 4; } }
					c = ((p + i) < n) ? i : (n - p) - 1; } } }

open_file:
	if(!options)		/* Solve wildcards only ... don't open */
		return -1;
	if(!(fp = fopen(filename, options)))
		message_box("Unable to access: '%s'", filename);
	return fp;
}

/*
 * Function key setup
 */
setup_fkey()
{
	unsigned i, j, s;
	unsigned char a, a1, *ptr;
	static unsigned p, sp;
	static char type[] = { ' ', 0x18, '^' };
	wopen(3, 1, 73, 22, _SETUP1_);
	a = attrs[SETUP1];
	a1 = (a >> 4) | (a << 4);
	for(;;) {
		i = 0;
		if(sp > 20) sp = 20;
		do {
			j = (i/2) + sp;
			wgotoxy(0, i);
			wprintf("%cF%-2u: ", type[j / 10], (j % 10)+1);
			if(i == p) {
				s = 7;
				ptr = Fkeys[j].Name; }
			wputs(Fkeys[j].Name); wcleol();
			wgotoxy(6, ++i);
			if(i == p) {
				s = KWIDTH;
				ptr = Fkeys[j].String; }
			wcleol(); wputs(Fkeys[j].String); }
		while(++i < 20);
again:	*W_OPEN = a1;
		i = wgets(6, p, ptr, s);
		*W_OPEN = a;
		switch(i) {
		case 0x1B:
		case _KEN:
			wclose();
			return;
		case _KUA:
			if(p) { --p; continue; }
			if(sp) { --sp; continue; }
			goto again;
		case _KDA:
			if(p < 19) { ++p; continue; }
			if(sp < 30) { ++sp; continue; }
			goto again;
		case _CPU:	sp = (sp < 10) ? 0 : sp-10;			continue;
		case _CPD:	sp += 10;							continue;
		case _CHO:	sp = p = 0;							continue;
		case _CEN:	sp = 20; p = 19;					continue;
		} beep(1000, 100); goto again; }
}

/*
 * Set tab stops at predefined intervals
 */
set_tabs(unsigned interval)
{
	unsigned i;
	for(i=0; i < sizeof(tabs); ++i)
		tabs[i] = (i%interval) ? 0 : 255;
//	*tabs = 0;
}

update_switch(unsigned i)
{
	switch(i) {
	case SWITCH_ATTR :
		if(!Cfont) {
			Cattr = 0;
			return; }
	case SWITCH_VGA :
		if(VGA) {
			Cfont = Cattr = 0;
			return; }
		if(VGA_font) {			// Font already active
			if(!Cfont) {			// Disable font
				font_off();
				disable_c2();
				Cattr = 0;
				return; }
			if(Cattr)
				enable_c2();
			else
				disable_c2();
			return; }
		if(Cfont) {		// Turn on VGA
			font_on();
			return; }
		if(VGA_attr)
			disable_c2();
		return;
	case SWITCH_C132 :
		if(C132 && !M132)
			C132 = 0;
	}
}

/*
 * Configuration menu
 */
configure()
{
	int c, i, f, b;
	unsigned o;
	char a, a1, *ptr, X132;
	static unsigned t;

top:
	wopen(5, 3, 30, (sizeof(config_menu)/2)+1, WSAVE);
	wcursor_off();
	while(!wmenu(5, 3, _SETUP2_, config_menu, &config_p)) switch(config_p) {
		case 0 :	/* General switches */
			X132 = C132;
			ptr = &alarm;
			wopen(28, 3, 40, (sizeof(switches)/2)+2, _SETUP1_);
			for(;;) {
				for(i=0; i < (sizeof(switches)/2); ++i) {
					wgotoxy(0, i);
					wprintf("%c: %-31s= %c", i+'A', switches[i], ptr[i] ? 'Y' : 'N'); }
				if((i = wgetc()) == 0x1B) {
					tty_cset(Uk ? 1 : 0);
					wclose();
					break; }
				if(i >= 'a') i -= ('a'-'A');
				if((i -= 'A') < (sizeof(switches)/2)) {
					ptr[i] = ptr[i] ? 0 : -1;
					update_switch(i);
					continue; } }
				if(X132 != C132) {	// Mode has changed
					wclose();
					setmode();
					wclwin();
					update_status();
					goto top; }
			break;
		case 1 : 	/* General parameters */
			o = Bduration * 55;
			wform(28, 3, _SETUP1_, setup_form, homedir, answerback,
				&o, &Bell, &Mbell, &Margin, &Breaklen,
				&hangup_delay, &key_string_delay, &upl_char_delay,
				&upl_line_delay, &upl_sync_char, &DAvalue, &M132);
			Bduration = (o+10) / 55;
			break;
		case 2 :	/* Serial port setup */
			wopen(28, 3, 30, (sizeof(serial_menu)/2)+1, WSAVE);
			while(!wmenu(28, 3, _SETUP2_, serial_menu, &serial_p)) {
				switch(serial_p) {
					case 0 : wmenu(42, 3, _SETUP1_, onefour, &com_port);	break;
					case 1 : wmenu(42, 3, _SETUP1_, baudtext, &baud);		break;
					case 2 : wmenu(42, 3, _SETUP1_, paritytext, &parity);	break;
					case 3 : wmenu(42, 3, _SETUP1_, fiveight, &dbits);		break;
					case 4 : wmenu(42, 3, _SETUP1_, onetwo, &sbits);		break;
					case 5 : wmenu(42, 3, _SETUP1_, flowtext, &Flow);		}
				update_status(); }
			open_comm();
			wclose();
			break;
		case 3 :	/* Protocol setup */
			wform(5, 13, _SETUP1_, string_form, protocols[0], protocols[1],
				protocols[2], protocols[3], protocols[4], protocols[5],
				protocols[6], protocols[7]);
			break;
		case 4 :	/* Tab stops */
			wopen(0, 20, Width, 2, a = WSAVE|WCOPEN|attrs[SETUP1]);
			a1 = (a >> 4) | (a << 4);
		rd:	if(t > Widthm1) t = Widthm1;
			for(i=0; i < Width; ++i) {
				*W_OPEN = ((i/10) & 1) ? a1 : a;
				wgotoxy(i, 0); wputc(((i+1)%10)+'0');
				*W_OPEN = (i == t) ? a1 : a;
				wgotoxy(i, 1); wputc(tabs[i] ? 'T' : ' '); }
			rk:	switch(c=wgetc()) {
				default: goto rk;
				case _KLA: if(t) --t;		goto rd;
				case _KRA: ++t;				goto rd;
				case _KHO: t = 0;			goto rd;
				case _KEN: t = Width;		goto rd;
				default:
					if(isdigit(c)) {
						if(!(c -= '0'))
							c = 10;
						set_tabs(c); }
					goto rd;
				case '\t':
					do
						++t;
					while((t < Width) && !tabs[t]);
					goto rd;
				case _KBS:
					while(t) {
						--t;
						if(tabs[t])
							break; }
					goto rd;
				case ' ' :
				case '\n':
				case 't' :
				case 'T' : tabs[t] = tabs[t] ? 0 : 255;		goto rd;
				case 'c' :
				case 'C' : memset(tabs, 0, sizeof(tabs));	goto rd;
				case 0x1B: }
			wclose();
			break;
		case 5 :	/* Function keys */
			setup_fkey();
			break;
		case 6 :	/* Script setup */
			wform(5, 13, _SETUP1_, string_form, scripts[0], scripts[1],
				scripts[2], scripts[3], scripts[4], scripts[5], scripts[6],
				scripts[7]);
			break;
		case 7 :	/* Video attributes */
			i = 0;
			do {
				if(((o=i) < 16) && Cattr)
					o += ATTR_EMULATE;
				f = attrs[o] & 0x0F;
				b = (attrs[o] >> 4);
				wopen(28, 3, 40, 9, WSAVE|WCOPEN|WBOX1|(b<<4)|f);
				*W_OPEN = 0x70;
				if(i < 16) {
					wputs("TTY: ");
					if(i & 8) wputs("BLINK ");
					if(i & 4) wputs("BOLD ");
					if(i & 2) wputs("UNDERLINE ");
					if(i & 1) wputs("REVERSE"); }
				wputs(menus[i]); wcleol();
				*W_OPEN = (b << 4) + f;
				wputs("\n\nLeft/Right = Foreground\nUp/Down    = Background\n");
				wputs("PgUp/PgDn  = Select window\n\nPress ESCAPE to exit.");
				c = wgetc();
				wclose();
				switch(c) {
					case _KRA : f = (f+1) & 0x0F; goto newattr;
					case _KLA : f = (f-1) & 0x0F; goto newattr;
					case _KUA : b = (b+1) & 0x0F; goto newattr;
					case _KDA : b = (b-1) & 0x0F;
					newattr:
						attrs[o] = (b << 4) | f;
						break;
					case _KPU : if(++i >= (sizeof(menus)/2)) i = 0;	break;
					case _KPD : if(--i < 0) i = (sizeof(menus)/2)-1; } }
			while(c != 0x1B);
			break;
		case 8 : map_keys();		break;
		case 9 :	/* Load configuration */
			if(get_string(cnfprompt, optr = cnffile, FSIZE))
				break;
			parse_filename(buffer, cnfext);
			if(open_file(buffer, "rb"))
				load_config();
			break;
		case 10 :	/* Save configuration */
			if(get_string(cnfprompt, optr = cnffile, FSIZE))
				break;
			parse_filename(buffer, cnfext);
			if(open_file(buffer, "wb")) {
				i = CFGVERSION;
				fput(&i, sizeof(i), fp);
				fput(homedir, ansi_keys-homedir, fp);
				fclose(fp); }
		}
	wclose();
	update_config();
}

update_config()
{
	setattr();
	if(*tty != *Attrs) {
		*tty = *Attrs;
		w_clwin(tty); }
	*status = attrs[STATUS];
	update_status();
}

tty_parm(unsigned n)
{
	static unsigned char ptable[] = { 4, 5, 6, 7, 1 };
	static unsigned char dtable[] = { 4, 3, 2, 1 };
	static unsigned char btable[] = {
		0, 8, 16, 24, 32, 40, 48, 56, 64, 72, 80, 88, 96, 104, 112, 120,
		128, 136, 144 };
	sprintf(key_ptr = buffer, "\x1B[%u;%u;%u;%u;%u;1;0x",
		n, ptable[parity], dtable[dbits], btable[baud], btable[baud]);
}

tty_cset(unsigned c)
{
	tty_translate[0x23] = '#';
	switch(Cset = c) {
	case 1 : tty_translate[0x23] = 0x9C;				// UK
	case 0 :											// US
		memcpy(tty_translate+0x5F, tty_text, sizeof(tty_text));
		return;
	case 2 :											// Graphics
		memcpy(tty_translate+0x5F, tty_graph, sizeof(tty_graph)); }
}

tty_reset()
{
	unsigned char X132;
	X132 = C132;
	memcpy(&com_port, SaveCfg, sizeof(SaveCfg));
	CapLK = Attr = Leds = Origin = Cursor = SaveA = SaveC = G0set = G1set = Pnl = ScrollT = SaveX = SaveY = 0;
	tty_cset(Uk ? 1 : 0);
	ScrollB = HEIGHT-1;
	setattr();
	*tty = *Attrs;
	if(X132 != C132)
		setmode();
	wclwin();
	open_comm();
	update_status();
}

tty_mode(unsigned mode, int value)
{
	switch(mode) {
	case 20 : NewlineI = NewlineO = value;	return;
	case 1001: Cursor = value;				return;
	case 1002: vt52 = value;				return;
	case 1003:
		if(C132 != value) {
			C132 = value;
			setmode();
			wclwin();
			update_status(); }
		return;
	case 1004: Sscroll = value;				return;
	case 1006: Origin = value;				return;
	case 1007: Wrap = value;				return;
	case 1008: Krepeat = value;	}
}

keypad_on()
{
	keypad = 255;
	update_status();
}

/*
 * Terminal mode using ANSI (VT100) emulation
 */
int ansi_term()
{
	unsigned x, y, c, state, value, value1, parm, parms[8], Sound;
	unsigned char Scrollf;
	static unsigned char *cursor_keys[] = {
		"\x1B[A", "\x1B[B", "\x1B[D", "\x1B[C",		// ANSI
		"\x1BOA", "\x1BOB", "\x1BOD", "\x1BOC" };	// APP

	xy_flag = 255;		/* Force initial cursor update */
	if(Cblock) wcursor_block(); else wcursor_line();
	setattr();
	KBflags = KBrp = KBwp = Scrollf = 0;
	kb_on();
reset1:
	wrkbd(0xED);
	wrkbd(CapLK & 7);

reset:
	y = Cursor ? 4 : 0;
	for(x=0; x < 4; ++x)
		ansi_keys[x] = cursor_keys[x+y];
	Sound = state = 0;		/* Not receiving a control sequence */
	for(;;) {
		if(Sound) {
			if((TICK - Sound) >= Bduration) {
				Sound = 0;
				sound_off(); } }
		if(Scrollf) {
			if((TICK - value) < 3)
				goto skiprx;
			Scrollf = 0; }
		/* Process any input from the comm port */
		if((c = Ctestc()) != -1) {
			last_rx = c;
ansi_out:
			if(log_fp && ((c >= ' ') || (c == '\t') || (c == '\n')))
				putc(c, log_fp);
			xy_flag = 255;
			if((c &= 0x7F) < ' ') {
				state = 0;
				switch(c) {
				case 0x1B :			// ESC lead-in
					state = vt52 ? 1 : 100;
					memset(parms, value = value1 = parm = 0, sizeof(parms));
				default:
					continue;
				case '\t' :			// TAB
					x = tty[6]; y = tty[7];
					while(x < Widthm1) {
						if(tabs[++x])
							break; }
					goto xy;
				case '\r' : wputc('\r'); nopnl:	Pnl = 0;	continue;
				case 0x07 :
					sound(Bell);
					if(!(Sound = TICK))
						Sound = 1;
					continue;
				case 0x08 : wputc(8);			goto nopnl;	// Backspace
				case '\n' :	// Line-Feed
				case 0x0B :	// VT
				case 0x0C : // FF
					if(NewlineI) {
			nl1: 		tty[6] = 0; }
			nl2:	Pnl = 0;
					if(tty[7] == ScrollB) {
						tty[3] = ScrollT;
						tty[5] = (ScrollB-ScrollT)+1;
						wputc('\n');
						tty[3] = 0;
						tty[5] = HEIGHT;
						if(Scrollf = Sscroll)
							value = TICK;
						continue; }
					if(tty[7] < (HEIGHT-1)) {
						++tty[7];
						xy_flag = 255; }
					continue;
				case 0x0F : tty_cset(G0set);	continue;	// SI
				case 0x0E : tty_cset(G1set);	continue;	// SO
				case 0x05 : key_ptr = answerback; continue;
				case 0x18 :		// CAN
				case 0x1A :		// SUB
					c = 0x80; } }
			switch(state) {
			case 0 :				/* No special processing */
		outc:	if(!(c & 0xE0))
					continue;
				if(tty[6] >= Widthm1) {
					if(Wrap) {
						if(Pnl) {
							tty[3] = ScrollT;
							tty[5] = (ScrollB-ScrollT)+1;
							wputc('\n');
							tty[3] = tty[6] = 0;
							tty[5] = HEIGHT;
							xy_flag = 255;
							goto outc1; }
						Pnl = 255; goto outc2; } }
		outc1:	Pnl = 0;
		outc2:	wputc(tty_translate[c]);
				if(tty[6] == Margin) {
					sound(Mbell);
					if(!(Sound = TICK))
						Sound = 1; }
				continue;
			case 1 :				/* Escape received (ANSI) */
				state = 0;
				switch(c) {
				case '[' : state = 2;					continue;
				case '7' :
					SaveX=tty[6];
					SaveY=tty[7];
					SaveC = Cset;
					SaveA = Attr;
					continue;
				case '8' :
					tty_cset(SaveC);
					wgotoxy(SaveX, SaveY);
					*tty = Attrs[SaveA];
					Pnl = 0;
					continue;
				case 'H' : tabs[tty[6]] = 255;			continue;
				case '=' : keypad_on();					continue;
				case '>' : keypad=0; update_status();	continue;
				case 'M' :
					if(tty[7] == ScrollT) {
						tty[3] = ScrollT;
						tty[5] = (ScrollB-ScrollT)+1;
						rscroll();
						tty[3] = 0;
						tty[5] = HEIGHT;
						if(Scrollf = Sscroll)
							value = TICK;
						continue; }
					if(tty[7]) {
						--tty[7];
						Pnl = 0;
						xy_flag = 255; }
					continue;
				case 'c' :
					tty_reset();
					goto reset1;
				case 'E' : goto nl1;
				case 'D' : goto nl2;
				case 'Z' : goto dar;
				case '(' : value = 255;
				case ')' : state = 3; continue;
				case '#' : state = 4; }
				continue;
			case 2 :				/* Waiting for '#' parms (ANSI) */
				if(c == '?') {
					value1 = 1000;
					continue; }
				if(isdigit(c)) {
					value = (value * 10) + (c - '0');
					continue; }
				parms[parm++] = (value+=value1);	// Save & advance
				if(c == ';') {						// More coming
					value = 0;
					continue; }
				state = 0;
				x = tty[6]; y = tty[7];
				if(!(value1 = value))
					value1 = 1;
				switch(c) {
				case 'H' :		// Cursor position (1)
				case 'f' :		// Cursor position (2)
					if(y = parms[0])
						--y;
					if(x = parms[1])
						--x;
					if(Origin) {
						if((y += ScrollT) > ScrollB)
							y = ScrollB; }
			xy:		wgotoxy((x > Widthm1) ? Widthm1 : x, (y > 23) ? 23 : y);
					Pnl = 0;
					continue;
				case 'J' :		// Erase in display
					c = *tty; *tty = *Attrs;
					switch(value) {
					case 0 : wcleow();	break;
					case 1 : wclsow();	break;
					case 2 : wclwin(); }
					*tty = c;
					goto xy;
				case 'K' :		// Erase in line
					c = *tty; *tty = *Attrs;
					switch(value) {
					case 2 : tty[6] = 0;
					case 0 : wcleol();	break;
					case 1 : wclsol(); }
					*tty = c;
					goto xy;
				case 'A' :		// Cursor UP
					y = (value1 > y) ? 0 : y-value1;
					if(Origin && (y < ScrollT))
						y = ScrollT;
					goto xy;
				case 'B' :		// Cursor down
					y += value1;
					if(Origin && (y > ScrollB))
						y = ScrollB;
					goto xy;
				case 'C' :		// Cursor forward
					x += value1;
					goto xy;
				case 'D' :		// Cursor backward
					x = (value1 > x) ? 0 : x-value1;
					goto xy;
				case 'g' :		// Tab clear
					switch(value) {
					case 0 : tabs[x] = 0;	continue;
					case 3 : memset(tabs, 0, sizeof(tabs)); }
					continue;
				case 'q' :		// Load Leds
					for(x=0; x < parm; ++x) switch(parms[x]) {
						case 0 : Leds = 0;	break;
						case 1 : Leds |= 1;	break;
						case 2 : Leds |= 2;	break;
						case 3 : Leds |= 4;	break;
						case 4 : Leds |= 8;	}
					update_status();
					continue;
				case 'm' :		// Select attributes
					for(x=0; x < parm; ++x) switch(parms[x]) {
						case 0 : Attr=0;	continue;
						case 1 : Attr |= 4;	continue;	// Bold
						case 4 : Attr |= 2;	continue;	// Underline
						case 5 : Attr |= 8;	continue;	// Blink
						case 7 : Attr |= 1;	}			// Reverse
					*tty = Attrs[Attr];
					continue;
				case 'r' :		// Set margins
					x = parms[0]; y = parms[1];
					if(x-- && y--) {
						if((x < (HEIGHT-1)) && (y <= (HEIGHT-1)) && (x < y)) {
							ScrollT = x;
							ScrollB = y; }
						continue; }
					ScrollT = 0;
					ScrollB = HEIGHT-1;
					continue;
				case 'h' :		// Set mode
					for(x=0; x < parm; ++x)
						tty_mode(parms[x], 255);
					goto reset;
				case 'l' :		// Reset mode
					for(x=0; x < parm; ++x)
						tty_mode(parms[x], 0);
					goto reset;
				case 'n' :		//  Device Status Report
					switch(value) {
					case 5 : key_ptr = "\x1B[0n";	continue;
					case 6 : sprintf(key_ptr=buffer, "\x1B[%u;%uR", y+1, x+1); }
					continue;
				case 'x' :		// Tarminal Parameters
					if(value < 2)
						tty_parm(2);
					continue;
				case 'c' :		// ID request
					if(!value) {
						dar: sprintf(key_ptr=buffer, "\x1B[?1;%uc", DAvalue); }
					continue;
				case 'y' :
					if((parms[0] == 2) && !parms[1]) {
						tty_reset();
						goto reset; }
					continue;
				} goto outc;
			case 3 :		// Escape-( received
				state = 0;
				switch(c) {
				case 'A' : c = 1;	goto setc;
				case 'B' : c = 0;	goto setc;
				case '0' : c = 2;	setc:
					if(value) G0set = c; else G1set = c; }
				continue;
			case 4 :		// Escape-# received
				state = Pnl = 0;
				switch(c) {
				case '8' :
					wgotoxy(0, 0);
					for(value = 0; value < 3840; value += 2)
						poke(W_BASE, value, 'E');
				} continue;
			case 100 :		// Escape received (VT52)
				x = tty[6]; y = tty[7];
				state = 0;
				switch(c) {
				case 'A' : if(y) --y;					goto xy;
				case 'B' : if(++y > 23) y = 23;			goto xy;
				case 'C' : if(++x > Widthm1) x = Widthm1; goto xy;
				case 'D' : if(x) --x;					goto xy;
				case 'F' : tty_cset(2);					continue;
				case 'G' : tty_cset(0);					continue;
				case 'H' : x=y=0;						goto xy;
				case 'I' :
					if(tty[7])
						--tty[7];
					else
						rscroll();
					continue;
				case 'J' : wcleow();					continue;
				case 'K' : wcleol();					continue;
				case 'Y' : state = 101;					continue;
				case 'Z' : key_ptr = "\x1B/Z";			continue;
				case '=' : keypad_on();					continue;
				case '>' : keypad = 0;					continue;
				case '<' : vt52 = 255; }
				continue;
			case 101 : state = 102; value1 = c - ' ';	continue;
			case 102 : state = 0; y = value1; x = c - ' ';
					if((y < 24) && (x < Width)) goto xy; } }
	else if(xy_flag) {				/* Cursor has moved */
		wupdatexy();
		xy_flag = 0; }

skiprx:
		// Update status line if SHIFT/ALT/CTRL keys change
		if(KBflags != active_keys) {
			active_keys = KBflags;
			update_status(); }

		/* Process any input from the keyboard */
		if(c = get_key()) {
			if(c & 0x8000) {
				wrkbd(0xED);
				wrkbd((peek(0, 0x417) >> 4) & 7);
				kb_off();
				active_keys = 0;
				update_status();
				return c; }
			Cputc(c);
			if((c == '\r') && NewlineO)
				key_ptr = "\n";
			if(echotty)
				goto ansi_out; } }

}

/*
 * Display key assignment
 */
showkey(unsigned char c)
{
	static unsigned char *fs[] = {
		"", "\x18", "^" };
	static unsigned char *vt100[] = {
		"Up", "Down", "Left", "Right",
		"PF1", "PF2", "PF3", "PF4" };
	static unsigned char *keypad[] = {
		"ENTER", ".", "1", "2", "3", "4", "5", "6", "7", "8", "9", "0",
		"-", "," };
	static unsigned Skeys[] = {
		"Menu", "Shift", "Ctrl", "CapsLK", "NoScroll", "Break", "Review" };

	if(!(c & 0x80)) {			// ASCII
		if(!c) {
			wputs("None");
			return; }
		wputs("ascii ");
		if(c < ' ')
			wprintf("^%c [%02x]", c+'@', c);
		else
			wprintf("%c  [%02x]", c, c);
		return; }
	if(c < 0x9E) {				// Function
		c -= 0x80;
		wputs(fs[c/10]);
		c %= 10;
		wprintf("F%u", c+1);
		return; }
	if(c < 0xA6) {				// VT-100 keys
		wputs(vt100[c-0x9E]);
		return; }
	if(c < 0xB4) {				// Keypad
		wprintf("kp %s", keypad[c-0xA6]);
		return; }
	if(c <= 0xBA) {
		wputs(Skeys[c-0xB4]);
		return; }
	wputs("?unknown");
}

unsigned char KB_text[] = {
	0x01,'E','S','C',0x82,
	0x3B,'F','1',0x81,
	0x3C,'F','2',0x81,
	0x3D,'F','3',0x81,
	0x3E,'F','4',0x82,
	0x3F,'F','5',0x81,
	0x40,'F','6',0x81,
	0x41,'F','7',0x81,
	0x42,'F','8',0x82,
	0x43,'F','9',0x81,
	0x44,'F','1','0',0x81,
	0x57,'F','1','1',0x81,
	0x58,'F','1','2',0x82,
	0x7C,'S','Y',0x81,
	0x46,'S','L',0x81,
	0x7D,'P','A',0x80,
	0x29,'`',0x83,
	0x02,'1',0x82,
	0x03,'2',0x82,
	0x04,'3',0x82,
	0x05,'4',0x82,
	0x06,'5',0x82,
	0x07,'6',0x82,
	0x08,'7',0x82,
	0x09,'8',0x82,
	0x0A,'9',0x82,
	0x0B,'0',0x82,
	0x0C,'-',0x82,
	0x0D,'=',0x82,
	0x2B,'\\',0x82,
	0x0E,'B','S',0x82,
	0x72,'I','N',0x81,
	0x74,'H','O',0x81,
	0x76,'P','U',0x82,
	0x45,'N','L',0x81,
	0x78,'K','/',0x81,
	0x37,'K','*',0x81,
	0x4A,'K','-',0x80,
	0x0F,'T','a','b',0x83,
	0x10,'Q',0x82,
	0x11,'W',0x82,
	0x12,'E',0x82,
	0x13,'R',0x82,
	0x14,'T',0x82,
	0x15,'Y',0x82,
	0x16,'U',0x82,
	0x17,'I',0x82,
	0x18,'O',0x82,
	0x19,'P',0x82,
	0x1A,'[',0x82,
	0x1B,']',0x87,
	0x73,'D','L',0x81,
	0x75,'E','N',0x81,
	0x77,'P','D',0x82,
	0x47,'K','7',0x81,
	0x48,'K','8',0x81,
	0x49,'K','9',0x80,
	0x3A,'C','a','p','s',0x83,
	0x1E,'A',0x82,
	0x1F,'S',0x82,
	0x20,'D',0x82,
	0x21,'F',0x82,
	0x22,'G',0x82,
	0x23,'H',0x82,
	0x24,'J',0x82,
	0x25,'K',0x82,
	0x26,'L',0x82,
	0x27,';',0x82,
	0x28,'\'',0x82,
	0x1C,'E','n','t','e','r',0x8C,
	0x4B,'K','4',0x81,
	0x4C,'K','5',0x81,
	0x4D,'K','6',0x81,
	0x4E,'K','+',0x80,
	0x2A,'L','s','h','i','f','t',0x82,
	0x2C,'Z',0x82,
	0x2D,'X',0x82,
	0x2E,'C',0x82,
	0x2F,'V',0x82,
	0x30,'B',0x82,
	0x31,'N',0x82,
	0x32,'M',0x82,
	0x33,',',0x82,
	0x34,'.',0x82,
	0x35,'/',0x83,
	0x36,'R','s','h','i','f','t',0x85,
	0x6E,'U','P',0x85,
	0x4F,'K','1',0x81,
	0x50,'K','2',0x81,
	0x51,'K','3',0x80,
	0x1D,'L','c','t','l',0x81,
	0x7E,'L','w','i','n',0x81,
	0x38,'L','a','l','t',0x81,
	0x39,' ',' ',' ',' ',' ','S','P','A','C','E',' ',' ',' ',' ',0x81,
	0x7B,'R','a','l','t',0x82,
	0x7F,'R','w','i','n',0x81,
	0x7A,'R','c','t','l',0x82,
	0x70,'L','E',0x81,
	0x6F,'D','N',0x81,
	0x71,'R','I',0x82,
	0x52,'-','K','0','-',0x82,
	0x53,'K','.',0x81,
	0x79,'K','E',0x80,
	0 };

/*
 * Draw keyboard and hilite code
 */
drawkey(unsigned code)
{
	unsigned char a, a1, c, s, *p;

	a = *W_OPEN;
	a1 = (a << 4) | (a >> 4);
	wgotoxy(0, 0);
	p = KB_text;
	while(c = *p++) {
		*W_OPEN = (c == code) ? a1 : a;
		while(!(*p & 0x80))
			wputc(*p++);
		*W_OPEN = a;
		if(!(s = *p++ & 0x7F)) {
			wputc('\n');
			continue; }
		while(s--)
			wputc(' '); }
}

// Find and display key by code
Findkey(c) asm
{
		MOV		DI,offset DGRP:_buffer	// Output buffer
		MOV		DX,DI
		MOV		AL,4[BP]				// Get char
		MOV		SI,offset DGRP:_KB_text	// Translate table
		XOR		BX,BX					// Offset zero
		MOV		AH,018h					// Shift indicator
fk1:	CMP		AL,DGRP:_KB_std[BX]		// Standard char?
		JZ		fk4						// Yes
		CMP		AL,DGRP:_KB_shift[BX]	// Shift character?
		JZ		fk3						// Yes
		CMP		AL,DGRP:_KB_ctrl[BX]	// Control character?
		JZ		fk2						// Yes
		INC		BL						// Next
		CMP		BL,128					// At end?
		JB		fk1						// Keep going
// Not found - return 0
fk1a:	XOR		AX,AX					// Not found
		JMP short fke					// exit
// Skip to next lookup table entry
fk1b:	MOV		AL,[SI]					// Get entry
		INC		SI						// Advance
		AND		AL,AL					// Test high bit
		JNS		fk1b					// Go tillset
		JMP short  fk4					// and continue
// We have key - lookup text to display
fk2:	MOV		AH,'^'					// Ctrl indicator
fk3:	MOV		[DI],AH					// Save prefix
		INC		DI						// Next
fk4:	MOV		AL,[SI]					// Get from table
		AND		AL,AL					// EOT?
		JZ		fk1a					// Exit
		INC		SI						// Next
		CMP		AL,BL					// Match keycode?
		JNZ		fk1b					// No - skip this one
		MOV		AL,[SI]					// Get code
fk5:	INC		SI						// Next in source
		CMP		AL,' '					// Space?
		JZ		fk6						// Don't display
		MOV		[DI],AL					// Save
		INC		DI						// Next in dest
fk6:	MOV		AL,[SI]					// Get code
		AND		AL,AL					// End?
		JNS		fk5						// Yes, exit
		MOV byte ptr [DI],0				// Terminate
// Call w_print to display
		PUSH	DGRP:_status			// pass Status window
		MOV		AX,6[BP]				// Get format string
		PUSH	AX						// pass format
		PUSH	DX						// pass display text
		MOV		AX,3					// 3 arguments
		CALL	_w_printf
		ADD		SP,6					// clean stack
fke:
}

#define	SIZ	10
map_keys()
{
	int d;
	unsigned i, k, lk;
	unsigned char c, *p, a, a1;
	static unsigned char *kt[] = { "NORM", "SHIFT", "CTRL" };

	wopen(5, 3, 70, 13, WSAVE|WBOX1|_SETUP1_);
	a = *W_OPEN;
	a1 = (a >> 4) | (a << 4);
newkey:
	wclwin();
	drawkey(lk=0);
	wputs("\nPress key to display - twice to select");
nextkey:
	kb_on();
	do {
		KBscan = 0;
		while(!(k = KBscan)); }
	while(k & 0x80);
	kb_off();
shokey:
	drawkey(k);
shocmd:
	wgotoxy(0, 7); wcleow();
	wprintf("    Keycode: %02x", KBscan);
	wputs("\n    Normal : "); showkey(KB_std[KBscan]);
	wputs("\n    Shift  : "); showkey(KB_shift[KBscan]);
	wputs("\n    Ctrl   : "); showkey(KB_ctrl[KBscan]);
	if(k != lk) {
		lk = k;
		wgotoxy(40, 8); wputs("Press hilighted key to edit");
		wgotoxy(43, 9); wputs("or select another key");
		goto nextkey; }
	for(i=0; i < 4; ++i) {
		wgotoxy(0,i+7);
		wprintf("F%c:", i+'1'); }
	wgotoxy(40, 8); wprintf("F5: Clear"); wcleol();
	wgotoxy(39, 9); wprintf("ESC: Exit"); wcleol();
	update_status();
getcmd: switch(wgetc()) {
	default: goto getcmd;
	case _K1 : goto newkey;
	case 0x1B: wclose();	return;
	case _K5 : KB_std[k] = KB_shift[k] = KB_ctrl[k] = 0; goto shocmd;
	case _K2 : c=0;	p = KB_std;		break;
	case _K3 : c=1;	p = KB_shift;	break;
	case _K4 : c=2;	p = KB_ctrl;	}

	write_status(" F1=AssignASCII  F2=AssignALL  ENTER=Assign%s  ESC=Cancel", kt[c]);
	wclwin();
	if((d = p[k]) & 0x80) {
		d -= 0x80;
		goto redraw; }
	d = 0;
redraw:	if(d < 0) d = (0xBA-0x80);
	if(d > (0xBA-0x80)) d = 0;
	for(i=0; i <= (0xBA-0x80); ++i) {
		wgotoxy((i/SIZ) * 10, i%SIZ);
		*W_OPEN = (i == d) ? a1 : a;
		showkey(i|0x80); }
	*W_OPEN = a;
	for(;;) switch(wgetc()) {
		case _K1:			// Assign ASCII
			wclwin();
			wputs("Press ASCII code key:");
			switch(i = wgetc()) {
			case _KBS: i = 0x08; break;
			case _KDL: i = 0x7F; }
			if(!(i & 0xFF80)) {
				p[k] = i;
				wclwin();
				goto shokey; }
			wclwin();
			goto redraw;
		case _K2:			// Assign ALL
			KB_std[k] = KB_shift[k] = KB_ctrl[k] = d|0x80;
		case '\n' :			// Assign ONE
			p[k] = d|0x80;
		case 0x1B :			// Cancel
			wclwin();
			goto shokey;
		case _KRA : d+=SIZ;	goto redraw;
		case _KLA : d-=SIZ;	goto redraw;
		case _KUA : --d;	goto redraw;
		case _KDA : ++d;	goto redraw; }
}

/*
 * Read a key from the keyboard
 */
get_key()
{
	int c;
	unsigned i, j;
	static unsigned lt;
	static unsigned char Etick;

	if(key_delay) {		// Delay pending
		i = TICK;
		switch(Etick) {
		case 3 :
			asm {
				MOV		AH,86h		// Delay function
				MOV		DX,1000		// millisconds
				XOR		CX,CX		// Zero high
				INT		15h			// Ask BIOS
			} --key_delay;
			return 0;		// loop delay
		case 2 : lt = i; --Etick;			// Set first tick
		case 1 : if(i == lt) return 0;		// Wait for tick
			lt = i;
			--Etick; }
		if(i != lt) {						// Expire delay
			lt = i;
			--key_delay; }
		return 0; }

	if(upl_fp) {
		if(c = wtstc()) switch(c) {
			case 0x1B :
				abort_flag = -1;
				goto abort;
			case ' ' :
				last_rx = upl_sync_char; }
		key_delay = upl_char_delay;
		Etick = 3;
		switch(sent_cr) {
			case ' ' :			/* Expanded NULL line */
				goto sendcr;
			case '\r' :			/* Just sent return */
				if(sendlf)
					return sent_cr = 0x0A;
			case 0x0A :			/* Just sent line-feed */
				if(upl_sync_char) {
					if(last_rx != upl_sync_char)
						return key_delay = 0; }
				sent_cr = -1; }
		if((c = getc(upl_fp)) != EOF) {
			if(c == '\n') {
				if(sendnull && sent_cr)
					return sent_cr = ' ';
sendcr:			key_delay = (upl_line_delay+27)/55;
				Etick = 2;
				return sent_cr = '\r'; }
			sent_cr = 0;
			return c; }
abort:	fclose(upl_fp);
		upl_fp = sent_cr = key_delay = 0;
		update_status(); }

	if(key_ptr) {
		if(c = *key_ptr++) {
			if(c == '^') switch(c=*key_ptr++) {
				case 0: --key_ptr; break;
				case '$':
					i = 22;
					j = 1;
					for(;;) {
					lp1: switch(*key_ptr) {
						case ' ' :
						case '\t': ++key_ptr; goto lp1;
						case ';' : ++key_ptr;
						case 0:
							if(i > 22) {
								while(j < 11)
									((unsigned*)buffer)[j++] = 0;
								*(unsigned*)buffer = 0xAABB;
								return -1; }
							return 0; }
						((unsigned*)buffer)[j++] = buffer+i;
						while((c = *key_ptr) && (*key_ptr++ != ';') && !isspace(c))
							buffer[i++] = c;
						buffer[i++] = 0;  }
				default:
					if(isdigit(c)) {
						if(!(c -= '0')) c = 10;
						key_delay = c * 9;
						Etick = 2;
						return 0; }
					c &= 0x1F;
				case '^': ; }
			key_delay = key_string_delay;
			Etick = 3;
			return c; }
		key_ptr = 0; }

	if(KBrp != KBwp) {			// Key available
		c = KBuffer[KBrp];
		KBrp = (KBrp+1) & 15;
		if(c & 0x80) {			// Special key
			if(c < 0x9E) {			// Function keys
				key_ptr = Fkeys[c-0x80].String;
				return 0; }
			if(c < 0xA6) {			// VT-100/52 standard keys
				key_ptr = vt52 ? ansi_keys[c-0x9E] : vt52_keys[c-0x9E];
				return 0; }
			if(c < 0xB4) {			// VT-100/52 keypad
				c -= 0xA6;
				if(keypad) {
					key_ptr = vt52 ? ansi_keypad[c] : vt52_keypad[c];
					return 0 ; }
				return std_keypad[c]; }
			switch(c) {
			case 0xB4 :	return -1;
			case 0xBA : return -2;
			case 0xB7 :
				wrkbd(0xED);
				wrkbd((CapLK ^= 0x04) & 7);
				break;
			case 0xB8:
				wrkbd(0xED);
				wrkbd((CapLK ^= 0x01) & 7);
				Cputc((CapLK & 0x01) ? 0x13 : 0x11);
				break;
			case 0xB9:
				Copen(com_port+1, _9600, SEND_BREAK, signals);
				delay(Breaklen);
				open_comm();
			} return 0; }
		if(CapLK & 0x04) {
			if((c >= 'a') && (c <= 'z'))
				c -= 0x20;
			else if((c >= 'A') && (c <= 'Z'))
				c += 0x20; }
		return c; }
	return 0;
}

void xsave()
{
	save();
	wopen(0, 0, Width, 25, WCOPEN|0x07);
}
void xrestore()
{
	wclose();
	restore();
	update_status();
}

/*
 * Execute a sub-shell
 */
doshell(command)
	char *command;
{
	char comspec[65], tail[80];

	if(!getenv("COMSPEC", comspec)) {
		message_box("Cannot locate COMSPEC");
		return; }
	*tail = 0;
	if(command)
		concat(tail, "/C ", command);
	xsave();
	wupdatexy();
	exec(comspec, tail);
	xrestore();
}

/*
 * Perform a file transfer
 */
file_transfer(upload, name)
	char upload, *name;
{
	int i, c;
	char buf[6], cmd[100], zap, wait, ascii;
	char *ptr1, *ptr2, *ptr3;

	wait = zap = ascii = i = 0;
	ptr1 = parse_filename(cmd, ".COM");

	ptr2 = ++ptr1;
	while(c = *optr++) {
		if(c == '\\') switch(c = *optr++) {
			case 'Z' : zap = -1;	continue;				/* Zap screen */
			case 'W' : wait = -1;	continue;				/* Wait after run */
			case 'A' : ascii = -1;	continue;				/* ASCII transfer */
			case 'P' : ptr3 =paritytext[parity];goto copy;	/* Parity */
			case 'B' : ptr3 = baudtext[baud];	goto copy;	/* Baudrate */
			case 'H' : ptr3 = homedir;			goto copy;	/* Home dir */
			case 'N' : c = dbits + 5;			goto wrnum;	/* Data bits */
			case 'S' : c = sbits + 1;			goto wrnum;	/* Stop bits */
			case 'C' : c = com_port + 1;					/* Comm port */
			wrnum:
				sprintf(ptr3 = buf,"%u", c);
			copy:
				while(c = *ptr3++)
					*ptr1++ = c;
				continue;
			case 'G' :		/* Upload filename */
			case 'Y' :		/* Upload with wildcards */
				if(!upload)
					continue;
			case 'F' :		/* Filename */
			case 'X' :		/* Allow wildcards */
				if(!(ptr3 = name)) {
					concat(buffer, "File to ", upload ? "UPLOAD" : "DOWNLOAD");
					if(get_string(buffer, ptr3 = name = filename, FSIZE))
						return; }
				if(((c == 'F') || (c == 'G')) && !open_file(name, 0))
					return;
				goto copy;
			case 'U' :		/* Upload string */
				while((c = *optr++) && (c != '\\'))
					if(upload)
						*ptr1++ = c;
				continue;
			case 'D' :		/* Download string */
				while((c = *optr++) && (c != '\\'))
					if(!upload)
						*ptr1++ = c;
				continue; }
		*ptr1++ = c; }
	*ptr1 = 0;

	/* Process ASCII transfers */
	if(ascii) {
		while(isspace(*ptr2)) ++ptr2;
		if(upload) {
			if(upl_fp = open_file(ptr2, "r"))
				write_status("ASCII upload of '%s'...", ptr2); }
		else {
			if(log_fp)
				fclose(log_fp);
				log_fp = open_file(ptr2, "w");
				update_status(); }
		return; }

	/* Execute the external protocol */
	write_status("%s of '%s'...", upload ? "UPLOAD" : "DOWNLOAD", name);
	Cclose();
	if(zap)
		xsave();
	wupdatexy();
	i = exec(cmd, ptr2);
	if(wait) {
		write_status("Press enter to continue...");
		while(wtstc() != '\n'); }
	if(zap)
		xrestore();
	open_comm();
	if(i)
		message_box("Error %u returned from '%s'!", i, cmd);
}

/*
 * Build an entry from a file list
 */
file_menu(files, pointer, name)
	char files[NFILES][FWIDTH], *name;
	int *pointer;
{
	int i, j, c;
	char *names[NFILES+1], *pptr[NFILES], buf[100], *ptr, *ptr1, *ptr2;

	/* Build menu from filenames */
	ptr1 = buf;
	for(i=j=0; i < NFILES; ++i) {
		if(*(ptr = files[i])) {
			pptr[j] = ptr;
			names[j++] = ptr2 = ptr1;
			while(c = *ptr++) {
				if(isspace(c) || (c == ';'))
					break;
				else if((c == '\\') || (c == ':'))
					ptr1 = ptr2;
				else
					*ptr1++ = c; }
			*ptr1++ = 0; }
			if(name && strbeg(ptr2, name)) {
				ptr = ptr1 = pptr[j-1];
				goto strip_name; } }
	names[j] = 0;

	if(!j) {
		message_box("No selections configured!");
		return 0; }

	if(wmenu(5, 3, _SETUP1_, names, pointer))
		return 0;

	/* Strip off any leading name */
	ptr = ptr1 = pptr[*pointer];
strip_name:
	while((c = *ptr++) && !isspace(c)) {
		if(c == ';')
			return ptr; }
	return ptr1;
}

/*
 * Execute a script command file
 */
script(argv)
	char *argv[];
{
	unsigned c, i;
	char buf[80], buf1[80], *ptr, *ptr1;
	char variables[NUM_VARS][50];
	FILE *sfp;

chain_script:
	abort_flag = invisible = 0;
	status_names[status_name_ptr++] = optr = argv[0];
	parse_filename(buf, ".SCR");
	if(!(sfp = open_file(buf, "r")))
		goto exit1;
	update_status();

	/* zero out the variables */
	memset(variables, line=0, sizeof(variables));
	flush_capture(0);

	while(fgets(optr = buf, sizeof(buf), sfp)) {
		++line;
		/* Skip any label */
		if(skip_blanks() == ':')
			while(*optr && !isspace(*optr))
				++optr;

next_cmd_string:
		wupdatexy();
		/* Substitute any variables etc. */
		ptr = buf1;
		while((c = *optr++) && (c != ';')) {
			if(c == '\\') switch(c = *optr++) {
				case 'b' :	c = 0x08;	break;
				case 'd' :	c = 0x7F;	break;
				case 'e' :	c = 0x1B;	break;
				case 'r' :	c = 0x0D;	break;
				case '#' :
					c = 0;
					while(isdigit(*optr))
						c = (c*10) + (*optr++ - '0');
					break;
				case 's' :
					sprintf(ptr, "%u", Csignals());
					while(*++ptr);
					continue;
				default:
					if((i = c - '0') < 10)
						ptr1 = argv[i];
					else if((i = c - 'A') < NUM_VARS)
						ptr1 = variables[i];
					else if(c == 'n')
						ptr1 = "\r\n";
					else if(c == 'h')
						ptr1 = homedir;
					else
						break;
					if(ptr1) while(*ptr1)
						*ptr++ = *ptr1++;
					continue; }
			else if(c == '^')
				c = *optr++ & 0x1f;
			*ptr++ = c; }
		while(isspace(*--ptr) && (ptr >= buf1));
		*++ptr = 0;
		optr = buf1;

continue_command:
		testabort();
		if(abort_flag)
			goto exit;
		if(*optr) switch(i = lookup(commands)) {
			case 0 :	/* Print */
				wprintf("%s\n\r", optr);
				break;
			case 1 :	/* Echo */
				wputs(optr);
				break;
			case 2 :	/* Assign */
			case 3 :	/* Equate */
			case 4 :	/* Input */
			case 5 :	/* Read */
			case 6 :	/* Menu */
				if((c = *optr++ - 'A') >= NUM_VARS) {
					script_error("Bad variable name");
					goto exit; }
				skip_blanks();
				ptr = variables[c];
				switch(i) {
					case 2 : strcpy(ptr, optr); 			break;
					case 3 : sprintf(ptr, "%d", eval());	break;
					case 4 :
						if(get_string(optr, ptr, sizeof(variables[0])-1))
							goto exit;
						break;
					case 5 :
						Cputs(optr);
						i = 0;
						do {
							do
								if(testabort()) goto exit;
							while((c = Ctestc()) == -1);
							if((c == '\b') || (c == 0x7f)) {
								if(i) {
									--i;
									Cputs("\b \b"); } }
							else if(c >= ' ')
								Cwrite(ptr[i++] = c); }
						while(c != '\r');
						ptr[i] = 0;
						Cputs("\n\r");
						break;
					case 6 :
						parse_args(buf);
						c = 0;
						sprintf(ptr,"%d", wmenu(5, 3, _SETUP1_, buf, &c) ?
							-1 : c); }
				break;
			case 7 :	/* If# */
				if(eval()) {
					expect(':');
					goto continue_command; }
				break;
			case 8 :	/* Ifeq */
			case 9 :	/* ifne */
				parse_string(buf);
				parse_string(buf+40);
				expect(':');
				if(i == 8) {
					if(!strcmp(buf, buf+40))
						goto continue_command; }
				else {
					if(strcmp(buf, buf+40))
						goto continue_command; }
				break;
			case 10 :	/* Ifnot */
				parse_string(buf);
				expect(':');
				if(!search_buffer(buf))
					goto continue_command;
				break;
			case 11 :	/* If */
				parse_string(buf);
				expect(':');
				if(search_buffer(buf))
					goto continue_command;
				break;
			case 12 :	/* Goto */
				if(*optr == '+')
					++optr;
				else {
					rewind(sfp);
					line = 0; }
				ptr = optr;
			search_next:
				while(fgets(optr = buf, sizeof(buf), sfp)) {
					++line;
					if(skip_blanks() == ':') {
						++optr;
						ptr1 = ptr;
						while(*optr && !isspace(*optr)) {
							if(*optr++ != *ptr1++)
								goto search_next; }
						if(isspace(*ptr1) || !*ptr1)
							goto next_cmd_string; } }
				script_error("Bad label");
			case 13 :	/* Skip */
				for(i = eval(); i; --i)
					if(!fgets(buf, sizeof(buf), sfp))
						goto exit;
			case 14 :	/* Stop */
				goto exit;
			case 15 :	/* Call */
				if(status_name_ptr > 4) {
					script_error("Calls to deep");
					goto exit; }
				i = line;
				c = invisible;
				parse_args(buf);
				script(buf);
				invisible = c;
				line = i;
				break;
			case 16 :	/* Chain */
				fclose(sfp);
				parse_args(argv);
				--status_name_ptr;
				goto chain_script;
			case 17 :	/* Send */
				Cputs(optr);
				break;
			case 18 :	/* Scan */
				flush_capture(0);
				if(parse_string(buf))
					if(!wait_string(buf, eval()))
						if(*optr) {
							expect(':');
							goto continue_command; }
				break;
			case 19 :	/* Flush */
				while((c = Ctestc()) != -1)
					receive_data(c);
				flush_capture(*buf ? buf : 0);
				break;
			case 20 :	/* Wait */
				wait_clear(eval());
				break;
			case 21 :	/* Upload */
			case 22 :	/* Download */
				ptr = optr;
				while((c = *optr) && !isspace(c)) ++optr;
				if(c)
					*optr++ = 0;
				ptr1 = skip_blanks() ? optr : 0;
				if(optr = file_menu(protocols, &proto_p, ptr)) {
					if(i == 21) {
						file_transfer(-1, ptr1);
						while(upl_fp) {
							while((c = Ctestc()) != -1)
								receive_data(c);
							if(c = get_key())
								Cwrite(c); } }
					else
						file_transfer(0, ptr1); }
				break;
			case 23 :	/* Close */
				if(log_fp) {
					fclose(log_fp);
					log_fp = 0; }
				update_status();
				break;
			case 24 :	/* Dos */
				doshell(optr);
				break;
			case 25 :	invisible = 0;	break;	/* Visible */
			case 26 :	invisible = -1;	break;	/* Invisible */
			case 27 :	/* Abort */
				script_error(optr);
				break;
			case 28 :	/* Error */
				abort_flag = -1;
			case 29 :	/* Message */
				message_box(optr);
				break;
			case 30 :	/* Hangup */
				hangup();
				break;
			case 31 :	/* Terminal */
				ansi_term();
				break;
			case 32 :	/* Clear */
				wclwin();
				break;
			case 33 :	/* Config */
				parse_filename(buf, cnfext);
				if(open_file(buf, "rb")) {
					fget(homedir, ansi_keys-homedir, fp);
					fclose(fp);
					open_comm();
					update_config(); }
				break;
			case 34 :	/* Exittodos */
				interactive = 0;
				abort_flag = -1;
				goto exit;
			default:
				script_error("Bad command");
				goto exit;
			case 35 :	/* Log */
				if(log_fp) {
					fputs(optr, log_fp);
					putc('\n', log_fp); }
			case 36 : } } /* Remark */
exit:
	fclose(sfp);
exit1:
	if(upl_fp) {
		fclose(upl_fp);
		upl_fp = 0; }
	--status_name_ptr;
	update_status();
}

/*
 * Evaluate an expression and return its numeric value
 */
eval()
{
	int o, v, v1;
	char vflag, mflag;

	v1 = o = 0;
	do {
		if(!*optr) {
			script_error("Bad expression");
			return 0; }
		mflag = vflag = v = 0;
		if(*optr == '-') {
			++optr;
			mflag = -1; }
		if(*optr == '(') {
			++optr;
			v = eval();
			if(expect(')'))
				return; }
		else {
			while(isdigit(*optr)) {
				v = (v * 10) + (*optr++ - '0');
				vflag = -1; }
			if(!vflag) {
				script_error("Bad number");
				return 0; } }
		if(mflag) v = -v;
		switch(o) {
			case 0 : v1 += v;		break;
			case 1 : v1 -= v;		break;
			case 2 : v1 *= v;		break;
			case 3 : v1 /= v;		break;
			case 4 : v1 %= v;		break;
			case 5 : v1 &= v;		break;
			case 6 : v1 |= v;		break;
			case 7 : v1 ^= v;		break;
			case 8 : v1 <<= v;		break;
			case 9 : v1 >>= v;		break;
			case 10: v1 = v1 == v;	break;
			case 11: v1 = v1 != v;	break;
			case 12: v1 = v1 >= v;	break;
			case 13: v1 = v1 <= v;	break;
			case 14: v1 = v1 > v;	break;
			case 15: v1 = v1 < v;	}
		o = lookup(operators); }
	while(operators[o]);
	if((o = *optr) && (o != ':') && (o != ')')) {
		script_error("Bad operation");
		return 0; }
	return v1;
}

/*
 * Determine if a character is a space
 */
isspace(c)
	int c;
{
	return (c == ' ') || (c == '\t');
}

/*
 * Lookup in table entry in command strinf
 */
lookup(table)
	char **table;
{
	int i;
	char *ptr;

	skip_blanks();
	for(i=0; ptr = table[i]; ++i)
		if(strbeg(optr, ptr)) {
			optr += strlen(ptr);
			break; }
	skip_blanks();
	return i;
}

/*
 * Skip ahead past leading blanks
 */
skip_blanks()
{
	while(isspace(*optr))
		++optr;
	return *optr;
}

/*
 * Expect a character in the input buffer
 */
expect(c)
	int c;
{
	if(skip_blanks() == c) {
		++optr;
		return 0; }
	script_error("Syntax error");
	return -1;
}

/*
 * Test for the abort key and indicate if pressed
 */
testabort()
{
	if(wtstc() == 0x1B) {
		script_error("User abort");
		return -1; }
	return 0;
}

/*
 * Indicate an error occuring in a script file
 */
script_error(string)
	char *string;
{
	wprintf("** line %u: %s\n\r", line, string);
	abort_flag = -1;
}

/*
 * Parse a delimited string from (optr)
 */
parse_string(output)
	char *output;
{
	int d, c;
	if(d = skip_blanks()) {
		while(c = *++optr) {
			if(c == d) {
				++optr;
				*output = 0;
				skip_blanks();
				return -1; }
			*output++ = c; } }
	script_error("Bad string");
	return 0;
}

/*
 * Parse string(OPTR) into 10 ARGC, ARGV parameters
 */
parse_args(outbuf)
	char *outbuf[];
{
	int i;
	char *ptr;

	i = 0;
	ptr = &outbuf[10];
	while(skip_blanks()) {
		outbuf[i++] = ptr;
		while(*optr && !isspace(*optr))
			*ptr++ = *optr++;
		*ptr++ = 0; }

	while(i < 10)
		outbuf[i++] = 0;
}

/*
 * Parse a filename from (optr)
 */
parse_filename(dest, suffix)
	char *dest, *suffix;
{
	int c, d;
	char *ptr;

	/* Test for direcrtory supplied */
	d = '\\';
	ptr = optr;
	while((c = *ptr++) && !isspace(c)) {
		if((c == '\\') || (c == ':'))
			d = 0; }
	/* If no directory, prepend home directory */
	if(d) {
		ptr = homedir;
		while(c = *ptr++)
			*dest++ = d = c;
		if(d != '\\')
			*dest++ = '\\'; }
	/* Copy in filename */
	d = -1;
	while((c = *optr) && !isspace(c)) {
		++optr;
		if((*dest++ = c) == '.')
			d = 0; }
	/* Append suffix if requested */
	if(d && suffix) {
		while(c = *suffix++)
			*dest++ = c; }
	*dest = 0;
	return dest;
}

/*
 * Flush the capture buffer and optionally insert a string
 */
flush_capture(ptr)
	char *ptr;
{
//	register unsigned i;
//	for(cptr = i = 0; i < CAPTURE_SIZE; ++i)
//		capture[i] = 0;
	memset(capture, cptr = 0, CAPTURE_SIZE);
	if(ptr) {
		while(*ptr)
			capture[cptr++] = *ptr++; }
}

/*
 * Handle data received from the remote
 */
receive_data(c)
	char c;
{
	capture[cptr] = last_rx = c;
	if(++cptr >= CAPTURE_SIZE)
		cptr = 0;

/* echo char if selected */
	if(!invisible)
		wputc(c);

/* Write download file if selected */
	if(log_fp && ((c >= ' ') || (c == '\t') || (c == '\n')))
		putc(c, log_fp);
}

/*
 * Wait for string from remote
 */
wait_string(char *string, unsigned timeout)
{
	int c;
	unsigned ts;
	char *ptr, *ptr1, *ptr2, buf[50], flag;

	timeout = (timeout+27) / 55;
	ts = TICK;
	flag = 0;
	ptr = buf;

	do {
		if((c = Ctestc()) >= 0) {
			receive_data(*ptr++ = c);
			*ptr = 0; flag = -1; }
		else if(testabort())
			return 0;
		if(flag) {
			ptr1 = string;
			ptr2 = buf;
			while(*ptr2 && (*ptr2 == *ptr1)) {
				++ptr1; ++ptr2; }
			if(*ptr2) {				/* strings do not match at all */
				for(ptr1 = buf; ptr1 < ptr; ++ptr1)
					*ptr1 = *(ptr1+1);
				--ptr; }
			else if(*ptr1)			/* Partial match, wait for more chars */
				flag = 0;
			else					/* Full match - we found it */
				return -1 ; } }
	while((TICK - ts) <= timeout);
	return 0;
}

/*
 * Search the capture buffer for a string
 */
search_buffer(string)
	char *string;
{
	char *ptr;
	unsigned sptr, rptr;

	sptr = cptr;

	do {
		rptr = sptr;
		ptr = string;
		while(*ptr && (*ptr == capture[rptr])) {
			++ptr;
			if(++rptr >= CAPTURE_SIZE)
				rptr = 0; }

		if(!*ptr)					/* we found it */
			return 11;
		if(++sptr >= CAPTURE_SIZE)	/* advance scan pointer */
			sptr = 0; }
	while(sptr != cptr);
	return 0;
}

/*
 * Wait until no data is received
 */
wait_clear(unsigned timeout)
{
	int c;
	unsigned ts;

	timeout = (timeout+27)/55;
	ts = TICK;
	do {
		while((c = Ctestc()) != -1) {
			receive_data(c);
			ts = TICK; }
		if(testabort())
			return; }
	while((TICK - ts) <= timeout);
}
