/*
 * Virtual ALTAIR
 *
 * ?COPY.TXT 2003-2024 Dave Dunfield
 *  -- see COPY.TXT --
 */
#include <stdio.h>
#include <window.h>
#include <keys.h>
#include <file.h>
#include <comm.h>
#define	MemTst(a)
#define	TICK	peekw(0x40,0x6C)

#define	PANEL	0x1700		// Panel background
#define	DWIN	0x30		// Debug window
#define	RUN		0x17		// While running
#define	DLOFF	1004h		// Disk LED - OFF
#define	DLON	0x1C04		// Disk LED - ON
#define	HTEXT	0x70		// Help text
#define HLINK	0x71		// Help link
#define	HLINKH	0x17		// Hilited link
#define	SWON	0x1300		// Attribute for ON switch
#define	SWHIL	0x1A00		// Attribute for HILITE switch
#define	LED0	0x10DB		// Attribute for OFF LED
#define LED1	0x14B0		// Attribute for DIM LED
#define	LED2	0x14B1		// Attribute for MEDIUM LED
#define	LED3	0x14B2		// Attribute for BRIGHT LED
#define	LED4	0x14DB		// Attribute for ON LED

// Help indexes
#define	HCMD	1
#define	HSTOP	2
#define	HRUN	3
#define	HEDITM	4
#define	HEDITR	5
#define	HTRACE	6
#define	HFILE	7
#define	HENTER	8

// Disk system parameters
#define	DTICK	50			// Disk TICK service interval
#define	DMOTOR	255			// Disk mtor on timeout
#define	DLEDS	3988		// Offset to disk LEDs
#define	SFILES	110			// Number of files to allow select

#define	O_BIO	0x01		//Ignore I/O
#define	O_GO	0x02		//Go
#define	O_NEP	0x04		//NoExitPrompt
#define	O_HEX	0x08		//HexPanel

unsigned char
	*Dip,			// Active directory pointer
	Directory[70];	// Directory name buffer

struct WINDOW
	*awin,			// Assembly window
	*rwin,			// Register window
	*mwin,			// Memory window
	*pwin,			// Panel window
	*dwin;			// Debug window
unsigned char
	Copt = O_BIO,
	eflag = 255,	// Error message flag
	Tflag,			// TTY delay
	Cmode=1,		// Console mode flag
	Cport,			// Console I/O port redirection
	Cport1,			// Physical port
	Cfmt = PAR_NO|DATA_8|STOP_1,		// Comm format
	Csig = SET_DTR|SET_RTS|OUTPUT_2,	// Comm signals
	Cdata,			// Console data
	Cdata1,			// Console RX data flag
	Udata,			// Uart RX data
	Udata1,			// Uart RX data flag
	Uctrl,			// Uart control register
	Umode,			// Uart mode register
	Umode1,			// PC uart control
	*ptr,			// General pointer
	cx,				// Console X position
	cy,				// Console Y position
	cdata[24][80];	// Console data buffer
unsigned
	Step = 1024,	// Steps/update
	Speed = 0,		// Simulation rate
	Switch = 0xE900, // Switch settings
	Cbaud = _9600,	// COMM baudrate
	SWhil,			// Switch hilight
	LPC,			// Load PC
	Lsize,			// Load size
	LEDS,			// Led port
	Address,		// Panel address
	Daddress,		// Last disassembly address
	Maddress,		// Last memory address
	daddress[64];	// Disassembly address

extern unsigned char
	A,
	B,
	C,
	D,
	E,
	H,
	L,
	PSW,
	EI,
	MEMR,
	IN,
	M1,
	OUT,
	STACK,
	BRKSK,
	W_PAGE;
extern unsigned
	SP,
	PC,
	SEG,
	STEP,
	BREAK,
	TRCPTR,
	SLOWCT,
	INSTCT,
	TRCBUF[];

extern int multi_step(void);
extern void setslow(unsigned count);
extern unsigned char *ARGV[];
extern unsigned Longreg[];

FILE
	fp;			// General file pointer

#include "panel.h"

unsigned char *Ptext[] = {
	"A0",	"A1",	"A2",	"A3",	"A4",	"A5",	"A6",	"A7",
	"A8",	"A9",	"A10",	"A11",	"A12",	"A13",	"A14",	"A15",
	"D0",	"D1",	"D2",	"D3",	"D4",	"D5",	"D6",	"D7",
	"INTE",	"PROT",	"MEMR",	"INP",	"M1",	"OUT",	"HLTA",	"STACK",
	"W0",	"INT",	"WAIT",	"HLDA",
	"Stop",	"sTep",	"Examine",	"Deposit",	"Reset", "Protect",	"aux",	"aux",
	"Go",	"Next",	"neXt",	"Clr",	"Unprot"/*,	"Y", "Z"*/ };

static unsigned char BROM[] = {		// Disk BOOT ROM
0x31,0x14,0x21,0x06,0x0A,0xC5,0x3E,0x59,0x32,0x00,0x20,0x32,0x03,0x20,0x01,
0x01,0x00,0x79,0x16,0x04,0x59,0x21,0x00,0x20,0xCD,0x1E,0xE9,0xC3,0x98,0xE9,
0xF5,0xE5,0xD5,0xC5,0x06,0xEB,0xCD,0xE0,0xE9,0x21,0xFF,0x34,0x09,0x7E,0xEE,
0x59,0xE5,0xCC,0x64,0xE9,0xE1,0xF1,0xCD,0x64,0xE9,0xC1,0xCD,0xCE,0xE9,0x3A,
0x30,0xEB,0xE6,0x0F,0xB8,0xC2,0x38,0xE9,0xE1,0x0D,0xFA,0x0A,0x20,0xC2,0x07,
0x20,0x06,0x8C,0x11,0x50,0xEB,0x0E,0x00,0x3A,0x10,0xEB,0xE6,0x04,0xC2,0xAE,
0xE9,0x05,0xC2,0x53,0xE9,0x3E,0x01,0xC3,0xAB,0xE9,0x57,0x96,0x72,0xC8,0x21,
0x1D,0xEB,0x4F,0xF2,0x7B,0xE9,0x2F,0x3C,0x4F,0x3A,0x10,0xEB,0xE6,0x01,0xC0,
0x21,0x1C,0xEB,0x7E,0x3A,0x09,0xEB,0xE3,0xE3,0x3A,0x08,0xEB,0x16,0x02,0xCD,
0xD0,0xE9,0x3A,0x10,0xEB,0xE6,0x01,0xCA,0x93,0xE9,0x0E,0x01,0x0D,0xC2,0x7C,
0xE9,0xC9,0xC1,0xCA,0x04,0x20,0x05,0xC2,0x05,0xE9,0xC3,0xA0,0xE9,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0xC1,0xB7,0xC9,0x41,0x1A,0x77,0xA8,0x07,0x47,
0x23,0x0D,0xC2,0xAF,0xE9,0x1A,0xA8,0xCA,0xC4,0xE9,0x78,0x3E,0x02,0xC3,0xAB,
0xE9,0xF1,0x3D,0xC8,0xF5,0xCD,0xCE,0xE9,0xC3,0x4C,0xE9,0x16,0x01,0x3A,0x14,
0xEB,0x3A,0x90,0xEB,0xE6,0x80,0xCA,0xD3,0xE9,0x15,0xC8,0xC3,0xD0,0xE9,0x3A,
0x90,0xEB,0xE6,0x10,0xC2,0xF0,0xE9,0x16,0x32,0xCD,0xD0,0xE9,0xC3,0xF5,0xE9,
0x3A,0x03,0x20,0xB9,0xC8,0x0A,0x79,0x32,0x03,0x20,0x16,0x0D,0xC3,0xD0,0xE9,
0x00 };

static unsigned L10[2] = { 10, 0 };			// Fixed long 10
static unsigned L60[2] = { 60, 0 };			// Fixed long 60
static unsigned L256[2] = { 256, 0 };		// Fixed long 256
static unsigned L1000[2] = { 1000, 0 };		// Fixed long 1000
static unsigned L10000[2] = { 10000, 0 };	// Fixed long 10000
static unsigned L115200[2] = { 0xC200, 1 };	// Fixed long 115200

unsigned char  *Dmenu[] = {
	"Drive 1 - RO",
	"Drive 1 - R/W",
	"Drive 2 - RO",
	"Drive 2 - R/W",
	"Drive 3 - RO",
	"Drive 3 - R/W",
	0 };

// Disk system globals
unsigned char
	Dstate,			// Disk system state
	Dcheck,			// Disk check character
	Dtrack[4],		// Current track
	Dsector,		// Current sector
	Dmotor,			// Motors on
	Dselect,		// Disk select
	Dsflag,			// Disk sector flag
	Dstep,			// Step direction
	Dfreeze,		// Freeze during read/write
	Dbuffer[256],	// Sector buffer
	Dname[3][9],	// Disk names
	Dwrite[4];		// Write enable flags
unsigned
	Dp;				// Disk buffer pointer
HANDLE
	Dhandle[4];	// Virtual Disk handles

// Clock subsystem globals
unsigned char
	Cset,			// Clock set mode
	Csetx;			// Slow setting pacer
unsigned
	Clast,			// Last TICK value
	Cdate = 372,	// Date (Day and Month)
	Ctime[2];		// Current time (in ms)

/*
 * Perform a single "tick" setclock function
 */
void setclock(void)
{
	unsigned char c, f;
	unsigned lt[2];

	switch((c = Cset) & 0xF0) {
	default:	return;
	case 0x30 :
		if(c == 0x37) {
			Ctime[0] = Ctime[1] = 0;
			return; }
		Cdate = 0;
	default:
		return;
	case 0x20 :		// Fast set
		f = 1;		// Assmue slow
		switch(c & 0x0F) {
		case 2 :	// Minute 10's
		case 3 :	// Minute units
		case 0xB:	// Day units
			goto goset; }
		if(c & 8) {	// Date set
			if(Cdate < 368)
				f = 3;
			goto goset; }
		// Time
		longcpy(lt, Ctime);		// Get time
		longdiv(lt, L10000);	// Get units ms
		if(*Longreg < 7940)		// Cannot overflow
			f = 3;
		break;
	case 0x10 :		// Slow set
		if(++Csetx < 3)
			return;
		Csetx = 0;
		f = 1; }

goset:	do {
		if(c & 8) {		// Setting day
			if(++Cdate >= 372)
				Cdate = 0; }
		else {			// Setting minutes
			asm {
				MOV		AX,30000	; Advance 30 seconds
				CALL	BMPsec		; Bump time
				MOV		AX,30000	; Advance 30 seconds
				CALL	BMPsec		; Bump time
			} } }
	while(--f);
}

/*
 * Clock tick handler
 */
asm
{
upclock: XOR	AX,AX			; Get zero
		MOV		ES,AX			; Address 0
		MOV		AX,ES:[046Ch]	; Read tick
clktick:MOV		BX,AX			; Save copy
		SUB		AX,DGRP:_Clast	; Same as last?
		JZ		upclk4			; Has not ticked
		MOV		CL,DGRP:_Cset	; Setting clock?
		AND		CL,30h			; Setting function?
		JZ		upclk1			; Do normal
; Setting, call setclock for each tick
upclk0:	PUSH	BX				; Save tick value
		CALL	_setclock		; Set the clock
		POP		BX				; Restore value
		INC		WORD PTR DGRP:_Clast; Advance last
		MOV		AX,DGRP:_Clast	; Get count
		CMP		AX,BX			; Are they same?
		JNZ		upclk0			; Do them all
		RET
; Update real time
upclk1:	MOV		DGRP:_Clast,BX	; Set new value
		MOV		BX,55			; 55ms tick
		IMUL	BX				; AX = ms count
BMPsec:	MOV		BX,offset DGRP:_Ctime
		ADD		AX,[BX]			; Add to low
		MOV		[BX],AX			; Resave
		JNC		upclk2			; No carry
		INC		WORD PTR 2[BX]	; Advance high
upclk2: MOV		AX,2[BX]		; Get high
		CMP		AX,526h			; End of day?
		JC		upclk4			; Not yet
		MOV		AX,[BX]			; Get low
		SUB		AX,5C00h		; End of day?
		JC		upclk4			; Not yet
; Rollover - reduce day counter by 24 hours
		MOV		[BX],AX			; Resave
		XOR		AX,AX			; Zero high
		MOV		2[BX],AX		; Resave
; Advance month
		MOV		AX,DGRP:_Cdate	; Get date
		INC		AX				; New data
		CMP		AX,372			; Past 12 months?
		JC		upclk3			; No, its OK
		XOR		AX,AX			; Reset date
upclk3:	MOV	DGRP:_Cdate,AX	; Set data
upclk4:	RET
}

/*
 * Clear the disk activity LEDs
 */
void clrdiskled(void) asm
{
	MOV		ES,DGRP:_W_BASE
	MOV		BX,DLEDS
	MOV		AX,DLOFF
	MOV		ES:[BX],AX
	MOV		ES:4[BX],AX
	MOV		ES:8[BX],AX
}

/*
 * Disk timer tick - service background functions
 */
void disktick(void)
{
	if(Dfreeze) {
		if(--Dfreeze)
			return; }
	Dsflag = 0x80;
	if(!Dstate)
		Dstate = 1;
	if(++Dsector > 9)
		Dsector = 0;
	Dcheck = Dp = 0;
	if(Dmotor) {
		if(!--Dmotor) {
			clrdiskled();
			Dselect = Dstate = 0; } }
}

/*
 * Read a sector from the disk controller hardware
 */
unsigned diskread(unsigned a)
{
	unsigned char c, r;
	unsigned s[2], t[2];
	static char rs;		// Last status sector

	r = 0;				// Default return of zero
	if((a & 0xFF00) == 0xEA00) {
		if(Dstate < 100)
			return 0;
		if(Dstate < 116) {
			++Dstate;
			Dp = Dcheck = 0;
			return 0; }
		if(Dstate == 116)
			Dfreeze = 10;
		Dbuffer[Dp++] = a;
		if(Dp == 256) {
			if(Dwrite[Dselect]) {		// Not write protected
				longset(s, Dtrack[Dselect]);
				longmul(s, L10);
				longset(t, rs /* Dsector */);
				longadd(s, t);
				longmul(s, L256);
				if(*t = Dhandle[Dselect]) {
					lseek(*t, s[1], s[0], 0);
					write(Dbuffer, 256, *t); } }
			Dstate = Dp = 0; }
		return; }

	if(a & 0x40) {	// Read data
		if(Dstate == 1) {		// Reading
			if(Dp < 256) {
				if(!Dp) {	// Read sector from file
					Dfreeze = 10;
					longset(s, Dtrack[Dselect]);
					longmul(s, L10);
					longset(t, rs /* Dsector */);
					longadd(s, t);
					longmul(s, L256);
					if(*t = Dhandle[Dselect]) {
						lseek(*t, s[1], s[0], 0);
						read(Dbuffer, 256, *t); }
					Dcheck = 0; }
				r = c = Dbuffer[Dp++];
				c ^= Dcheck;
				Dcheck = (c << 1) | (c >> 7); }
			else if(Dp == 256) {
				STEP = 1;
				r = Dcheck;
				Dp = 500; } } // }
		if(a & 0x80)
			Dmotor = DMOTOR;
		return r; }

	rs = Dsector;
	Dfreeze = 0;

	if(a & 0x20) {	// Read B status
		r = Dsflag | Dsector;
		if(Dmotor)
			r |= 0x10; }
	else {						// Read A status
		r = Dsflag;
		if(Dmotor)				// Motors ON
			r |= 0x10;
		if(Dstate >= 100)		// Write mode
			r |= 0x08;
		else if(Dstate) {		// Read Body
			if(Dhandle[Dselect])
				r |= 0x04; }
		if(!Dwrite[Dselect])	// Write protect
			r |= 0x02;
		if(!Dtrack[Dselect])	// Track 0
			r |= 0x01; }
	switch(a & 0x1C) {		// Command
	case 0x00 :				// 000 - Select
		clrdiskled();
		if(Dselect = a & 3)
			pokew(W_BASE, ((unsigned)Dselect*4)+(DLEDS-4), DLON);
		break;
	case 0x04 :				// 001 - Write
		Dstate = 100;
		break;
	case 0x08 :				// 010 - Step
		if(a & 1) {
			if(Dstep) {
				if(Dtrack[Dselect] < 34)
					++Dtrack[Dselect]; }
			else {
				if(Dtrack[Dselect])
					--Dtrack[Dselect]; } }
		break;
	case 0x0C :				// 011 - Interrupt armed
	case 0x10 :				// 100 - No operation
		break;
	case 0x14 :				// 101 - Reset sector flag
		Dsflag = 0;
		break;
	case 0x18:				// 110 - Reset controller
		Dselect = Dstate = Dsflag = 0;
		Dmotor = 1;
		break;
	case 0x1C:				// Load step direction
		Dstep = a & 1; }

	if(a & 0x80)
		Dmotor = DMOTOR;
	return r;
}

/*
 * Reset virtual CPU
 */
void reset(void)
{
	A = B = C = D = H = L = PSW = SP = EI = MEMR = IN = M1 = OUT = STACK = 0;
	PC = LPC;
	copy_seg(SEG, 0xE900, get_ds(), BROM, 256);
	Uctrl = 0x40;
	Umode = 0x37;
	LEDS = 0;
}

/*
 * Draw text on virtual front panel
 */
void draw_text(unsigned x, unsigned attr)
{
	unsigned char *p;
	p = Ptext[x];
	x = Panel[x];
	while(*p) {
		pokew(W_BASE, x, (unsigned)*p++ | attr);
		x += 2; }
}

/*
 * Draw front panel address lights
 */
void draw_switch(void)
{
	unsigned i;
	unsigned b;
	b = 1;
	for(i=0; i < 16; ++i) {
		draw_text(i, (Switch & b) ? SWON : PANEL);
		b <<= 1; }
}
void draw_sws(void)
{
	unsigned i, b;
	b = SWhil;
	for(i=0; i < 16; ++i) {
		poke(W_BASE, Panel[i]+((160*2)+2), (b & 1) ? '^' : ' ');
		b >>= 1; }
}

/*
 * Draw the virtual front panel
 */
void draw_panel(void)
{
	unsigned i;
	w_clwin(pwin);
	draw_switch();
	for(i=16; i < (sizeof(Panel)/2); ++i)
		draw_text(i, PANEL);
	draw_sws();
	for(i=0; i < (sizeof(Leds)/2); ++i)
		pokew(W_BASE, Leds[i], LED0);
}

/*
 * Update the data LED values
 */
void write_data(c) asm
{
		MOV		AX,4[BP]
		MOV		DI,OFFSET DGRP:_Leds
		MOV		CX,8
wrd1:	MOV		BX,[DI]
		INC		DI
		INC		DI
		RCR		AX,1
		JC		wrd2
		MOV		DX,10DBh
		JMP		SHORT wrd3
wrd2:	MOV		DX,14DBh
wrd3:	MOV		ES:[BX],DX
		LOOP	wrd1
}

/*
 * Updte the address LED values
 */
void write_address(c) asm
{
		MOV		AX,4[BP]
		MOV		DI,(OFFSET DGRP:_Leds)+16
		MOV		CX,16
wra1:	MOV		BX,[DI]
		INC		DI
		INC		DI
		RCR		AX,1
		JC		wra2
		MOV		DX,10DBh
		JMP		SHORT wra3
wra2:	MOV		DX,14DBh
wra3:	MOV		ES:[BX],DX
		LOOP	wrd1
}

/*
 * Update front panel LED's
 */
void upleds() asm
{
		MOV		ES,DGRP:_W_BASE			; Get base video
		MOV		DI,(OFFSET DGRP:_Leds)+48 ; Get INTE
		MOV		AL,DGRP:_EI				; Get interrupt enabled
		CALL	Lonoff					; On/OFF led
		INC		DI						; Skip
		INC		DI						; PROT
		MOV		AL,DGRP:_MEMR			; Get MEMR
		CALL	Lvar					; Set LED
		MOV		AL,DGRP:_IN				; Get IN
		CALL	Lvar					; Set LED
		MOV		AL,DGRP:_M1				; Get M1
		CALL	Lvar					; Set LED
		MOV		AL,DGRP:_OUT			; Get OUT
		CALL	Lvar					; Set LED
		INC		DI						; Skip
		INC		DI						; HLTA
		MOV		AL,DGRP:_STACK			; Get STACK
		CALL	Lvar					; Set LED
} asm {
Lonoff:	MOV		BX,[DI]					; Get LED address
		INC		DI						; Next
		INC		DI						; Next]
		AND		AL,AL					; Enabled?
		JNZ		Led4					; Fill brightness
Led0:	MOV		AX,10DBh				; LED = Off
		MOV		ES:[BX],AX
		RET
Led1:	MOV		AX,14B0h
		MOV		ES:[BX],AX
		RET
Led2:	MOV		AX,14B1h
		MOV		ES:[BX],AX
		RET
Led3:	MOV		AX,14B2h
		MOV		ES:[BX],AX
		RET
Lvar:	MOV		BX,[DI]
		INC		DI
		INC		DI
		AND		AL,AL
		JZ		Led0
		CMP		AL,16
		JC		Led1
		CMP		AL,64
		JC		Led2
		CMP		AL,128
		JC		Led3
Led4:	MOV		AX,14DBh				; LED = Full
		MOV		ES:[BX],AX
		RET
}

// Disassembler globals
/*
 * Table of instruction opcodes: MASK, COMPARE, TYPE/LENGTH, TEXT
 */
unsigned char itable[] = {
	0xFF, 0xFE, 0x02, 'C', 'P', 'I', ' ', 0,
	0xFF, 0x3A, 0x03, 'L', 'D', 'A', ' ', 0,
	0xFF, 0x32, 0x03, 'S', 'T', 'A', ' ', 0,
	0xFF, 0x2A, 0x03, 'L', 'H', 'L', 'D', ' ', 0,
	0xFF, 0x22, 0x03, 'S', 'H', 'L', 'D', ' ', 0,
	0xFF, 0xF5, 0x01, 'P', 'U', 'S', 'H', ' ', 'P', 'S', 'W', 0,
	0xFF, 0xF1, 0x01, 'P', 'O', 'P', ' ', 'P', 'S', 'W', 0,
	0xFF, 0x27, 0x01, 'D', 'A', 'A', 0,
	0xFF, 0x76, 0x01, 'H', 'L', 'T', 0,
	0xFF, 0xFB, 0x01, 'E', 'I', 0,
	0xFF, 0xF3, 0x01, 'D', 'I', 0,
	0xFF, 0x37, 0x01, 'S', 'T', 'C', 0,
	0xFF, 0x3F, 0x01, 'C', 'M', 'C', 0,
	0xFF, 0x2F, 0x01, 'C', 'M', 'A', 0,
	0xFF, 0xEB, 0x01, 'X', 'C', 'H', 'G', 0,
	0xFF, 0xE3, 0x01, 'X', 'T', 'H', 'L', 0,
	0xFF, 0xF9, 0x01, 'S', 'P', 'H', 'L', 0,
	0xFF, 0xE9, 0x01, 'P', 'C', 'H', 'L', 0,
	0xFF, 0xDB, 0x02, 'I', 'N', ' ', 0,
	0xFF, 0xD3, 0x02, 'O', 'U', 'T', ' ', 0,
	0xFF, 0x07, 0x01, 'R', 'L', 'C', 0,
	0xFF, 0x0F, 0x01, 'R', 'R', 'C', 0,
	0xFF, 0x17, 0x01, 'R', 'A', 'L', 0,
	0xFF, 0x1F, 0x01, 'R', 'A', 'R', 0,
	0xFF, 0xC6, 0x02, 'A', 'D', 'I', ' ', 0,
	0xFF, 0xCE, 0x02, 'A', 'C', 'I', ' ', 0,
	0xFF, 0xD6, 0x02, 'S', 'U', 'I', ' ', 0,
	0xFF, 0xDE, 0x02, 'S', 'B', 'I', ' ', 0,
	0xFF, 0xE6, 0x02, 'A', 'N', 'I', ' ', 0,
	0xFF, 0xF6, 0x02, 'O', 'R', 'I', ' ', 0,
	0xFF, 0xEE, 0x02, 'X', 'R', 'I', ' ', 0,
	0xFF, 0x00, 0x01, 'N', 'O', 'P', 0,
	/*  8085 specific instructions */
//	0xFF, 0x20, 0x01, 'R', 'I', 'M', 0,
//	0xFF, 0x30, 0x01, 'S', 'I', 'M', 0,
	/*  Jumps, Calls & Returns */
	0xFF, 0xC3, 0x0B, 'J', 'M', 'P', ' ', 0,
	0xFF, 0xCA, 0x43, 'J', 'Z', ' ', 0,
	0xFF, 0xC2, 0x4B, 'J', 'N', 'Z', ' ', 0,
	0xFF, 0xDA, 0x13, 'J', 'C', ' ', 0,
	0xFF, 0xD2, 0x1B, 'J', 'N', 'C', ' ', 0,
	0xFF, 0xEA, 0x23, 'J', 'P', 'E', ' ', 0,
	0xFF, 0xE2, 0x2B, 'J', 'P', 'O', ' ', 0,
	0xFF, 0xFA, 0x83, 'J', 'M', ' ', 0,
	0xFF, 0xF2, 0x8B, 'J', 'P', ' ', 0,
	0xFF, 0xCD, 0x0B, 'C', 'A', 'L', 'L', ' ', 0,
	0xFF, 0xCC, 0x43, 'C', 'Z', ' ', 0,
	0xFF, 0xC4, 0x4B, 'C', 'N', 'Z', ' ', 0,
	0xFF, 0xDC, 0x13, 'C', 'C', ' ', 0,
	0xFF, 0xD4, 0x1B, 'C', 'N', 'C', ' ', 0,
	0xFF, 0xEC, 0x23, 'C', 'P', 'E', ' ', 0,
	0xFF, 0xE4, 0x2B, 'C', 'P', 'O', ' ', 0,
	0xFF, 0xFC, 0x83, 'C', 'M', ' ', 0,
	0xFF, 0xF4, 0x8B, 'C', 'P', ' ', 0,
	0xFF, 0xC9, 0x05, 'R', 'E', 'T', 0,
	0xFF, 0xC8, 0x45, 'R', 'Z', 0,
	0xFF, 0xC0, 0x4D, 'R', 'N', 'Z', 0,
	0xFF, 0xD8, 0x15, 'R', 'C', 0,
	0xFF, 0xD0, 0x1D, 'R', 'N', 'C', 0,
	0xFF, 0xE8, 0x25, 'R', 'P', 'E', 0,
	0xFF, 0xE0, 0x2D, 'R', 'P', 'O', 0,
	0xFF, 0xF8, 0x85, 'R', 'M', 0,
	0xFF, 0xF0, 0x8D, 'R', 'P', 0,
	/*  Register based instructions */
	0xC0, 0x40, 0x01, 'M', 'O', 'V', ' ', 'd', ',', 's', 0,
	0xC7, 0x06, 0x02, 'M', 'V', 'I', ' ', 'd', ',', 0,
	0xF8, 0x90, 0x01, 'S', 'U', 'B', ' ', 's', 0,
	0xF8, 0x98, 0x01, 'S', 'B', 'B', ' ', 's', 0,
	0xF8, 0x80, 0x01, 'A', 'D', 'D', ' ', 's', 0,
	0xF8, 0x88, 0x01, 'A', 'D', 'C', ' ', 's', 0,
	0xF8, 0xA0, 0x01, 'A', 'N', 'A', ' ', 's', 0,
	0xF8, 0xB0, 0x01, 'O', 'R', 'A', ' ', 's', 0,
	0xF8, 0xA8, 0x01, 'X', 'R', 'A', ' ', 's', 0,
	0xF8, 0xB8, 0x01, 'C', 'M', 'P', ' ', 's', 0,
	0xC7, 0x04, 0x01, 'I', 'N', 'R', ' ', 'd', 0,
	0xC7, 0x05, 0x01, 'D', 'C', 'R', ' ', 'd', 0,
	/*  Register pair instructions */
	0xCF, 0x01, 0x03, 'L', 'X', 'I', ' ', 'p', ',', 0,
	0xEF, 0x0A, 0x01, 'L', 'D', 'A', 'X', ' ', 'p', 0,
	0xEF, 0x02, 0x01, 'S', 'T', 'A', 'X', ' ', 'p', 0,
	0xCF, 0x03, 0x01, 'I', 'N', 'X', ' ', 'p', 0,
	0xCF, 0x0B, 0x01, 'D', 'C', 'X', ' ', 'p', 0,
	0xCF, 0x09, 0x01, 'D', 'A', 'D', ' ', 'p', 0,
	0xCF, 0xC5, 0x01, 'P', 'U', 'S', 'H', ' ', 'p', 0,
	0xCF, 0xC1, 0x01, 'P', 'O', 'P', ' ', 'p', 0,
	/*  Restart instruction */
	0xC7, 0xC7, 0x01, 'R', 'S', 'T', ' ', 'v', 0,
	/*  This entry always matches invalid opcodes */
	0x00, 0x00, 0x01, 'D', 'B', ' ', 'o', 0 };

/* Tables to convert register index into actual names */
unsigned char regtab[]	= { 'B','C','D','E','H','L','M','A' };
unsigned char rptab[]	= { 'B','D','H','S' };

/*
 * Disassemble an 8080 instruction
 */
void disassemble(unsigned length, unsigned char display)
{
	unsigned a, i, l, ll, t;
	unsigned char c, o, *p;

	for(i=0; i < length; ++i) {
		if(daddress[i] == Daddress) {
			a = daddress[0];
			goto dodis; } }
	a = Daddress;
dodis:
	for(ll=0; ll < length; ++ll) {
		*awin = (a == Daddress) ? 0x76 : 0x67;
		o = Mread(daddress[ll] = a);
		p = itable;
again:	if((o & *p++) != *p++) {
			while(*p++);
				goto again; }
		l = *p++ & 3;
		if(display && !Cmode) {
			w_gotoxy(0, ll, awin);
			w_printf(awin, "%04x", a);
			for(i=0; i < 3; ++i) {
				if(i < l)
					w_printf(awin, " %02x", Mread(a+i));
				else
					w_puts("   ", awin); }
			i = 0;
			w_puts("  ", awin);
			while(c = *p++) switch(c) {
			case 'v' :		/* Interrupt vector */
				w_putc(((o >> 3) & 7) + '0', awin);
				goto iplus;
			case 'p' :		/* Register PAIR */
				w_putc(c = rptab[(o >> 4) & 3], awin);
				if(c == 'S') {
					w_putc('P', awin);
					++i; }
				goto iplus;
			case 'd' :		/* Destination register */
				w_putc(regtab[(o >> 3) & 7], awin);
				goto iplus;
			case 's' :		/* Source register */
				w_putc(regtab[o & 7], awin);
				goto iplus;
			case 'o' :		/* Output opcode byte */
				i += w_printf(awin,"$%02x", o);
				break;
			case ' ' :		/* Separator */
				do
					w_putc(' ', awin);
				while(++i < 8);
				break;
			default:
				w_putc(c, awin);
			iplus:
				++i; }
			w_cleol(awin);
			switch(l) {
			case 2 :		/* Single byte operand */
				w_printf(awin, "%02x", Mread(a+1));
				break;
			case 3 :		/* Double byte operand */
				t = Mread(a+1);
				w_printf(awin, "%04x", (Mread(a+2) << 8)|t); } }
		a += l; }
}

/*
 * Dump a block of memory
 */
void memory(void)
{
	unsigned i, j, a;
	unsigned char c;
	if(!Cmode) {
		a = Maddress;
		for(j=0; j < 16; ++j) {
			w_gotoxy(0, j, mwin);
			w_printf(mwin, "%04x ", a);
			for(i=0; i < 8; ++i)
				w_printf(mwin, " %02x", Mread(a+i));
			w_puts("  ", mwin);
			for(i=0; i < 8; ++i) {
				c = Mread(a++);
				w_putc(((c>=' ') && (c < 0x7F)) ? c : '.', mwin); } } }
}

/*
 * Update 8080 register display
 */
void dumpreg(void)
{
	unsigned i;
	unsigned char c;
	static char Fs[] = { 'S', 'Z', '?', 'A', '?', 'P', '?', 'C' };
	static char Fc[] = { 's', 'z', '-', 'a', '-', 'p', '-', 'c' };
	if(!Cmode) {
		w_gotoxy(0, 0, rwin); w_cleol(rwin);
		w_printf(rwin, "\n A :%02x", A);			w_cleol(rwin);
		w_printf(rwin, "\n BC:%02x %02x", B, C);	w_cleol(rwin);
		w_printf(rwin, "\n DE:%02x %02x", D, E);	w_cleol(rwin);
		w_printf(rwin, "\n HL:%02x %02x", H, L);	w_cleol(rwin);
		w_printf(rwin, "\n SP:%04x", SP);			w_cleol(rwin);
		w_printf(rwin, "\n PC:%04x", PC);			w_cleol(rwin);
		w_printf(rwin, "\nPSW:%02x", c = PSW);		w_cleol(rwin);
		w_puts("\n ", rwin);
		for(i=0; i < 8; ++i) {
			w_putc( ((c & 0x80) ? Fs : Fc)[i], rwin);
			c <<= 1; }
		w_cleow(rwin); }
}

/*
 * Update console X/Y position
 */
void cupdatexy() asm
{
		MOV		DX,WORD PTR DGRP:_cx ; Get X/Y
		MOV		BH,DGRP:_W_PAGE	; Get display page
		MOV		AH,02h			; Set cursor function
		INT		10h				; Call DOS
}

/*
 * Open the console
 */
void open_console()
{
	unsigned x, y, i;
	y = i = 0;
	if(Cmode == 1) {
		draw_panel();
		y = 8;
		i = 1280; }
	while(y < 24) {
		for(x=0; x < 80; ++x) {
			pokew(W_BASE, i, (unsigned)cdata[y][x] | 0x0700);
			i += 2; }
		++y; }
	wcursor_line();
	cupdatexy();
}

/*
 * Close the console
 */
void close_console(void)
{
	if(Cmode) {
		Cmode = 0;
		draw_panel();
		disassemble(16, 255);
		memory();
		dumpreg();
		wcursor_off(); }
}

/*
 * Perform newline delay if enabled
 */
void newline(void)
{
	if(Tflag)
		delay(110);
}

/*
 * Scroll console
 */
void cscroll(void)
{
	unsigned i;

	memcpy(cdata, cdata+80, sizeof(cdata)-80);
	memset(&cdata[22][80], ' ', 80);

	switch(Cmode) {
	default: return;
	case 1 : copy_seg(W_BASE, 1280, W_BASE, 1440, 15*160);
		break;
	case 2 :
		copy_seg(W_BASE, 0, W_BASE, 160, 23*160); }

	for(i=3680; i < 3840; i+=2)
		poke(W_BASE, i, ' ');
}

/*
 * Write a character to the console
 */
void cputc(unsigned char c)
{
	unsigned i;
	static unsigned char cstate = 255;

	// Handle leading cursor position
	switch(cstate) {
	case 0 : break;
	case 1 :
		if(c != '=') {
			cstate = 0;
			break; }
		cstate = 2;
		return;
	case 2:
		if((c < ' ') || (c > (' '+79))) {
			cstate = 0;
			break; }
		cstate = c - (' ' - 3);
		return;
	default:
		if((c < ' ') || (c > (' '+23))) {
			cstate = 0;
			break; }
		cx = cstate - 3;
		cy = c - ' ';
		cstate = 0;
		goto upxy;
	case 255 :
		if(Cmode == 1) {
			cy = 8;
			switch(c) {
			case 'K'-0x40 :
			case 'Z'-0x40 :
			case '^' - 0x40:
			case 8 :
				return; } }
		cstate = 0; }

	switch(c) {
	case '\r' :				// Carriage return
		cx = 0;
		goto upxy;
	case '\n' :				// Line-feed
		newline();
		if(cy < 23) {
			++cy;
			goto upxy; }
		cscroll();
		goto upxy;
	case 'K'-0x40:			// Up line
		if(cy)
			--cy;
		goto upxy;
	case 'G'-0x40:			// Beep
		beep(1000, 100);
		return;
	case 'L'-0x40:			// Forward space
		goto advance;
	case 'Z'-0x40:			// Clear screen
		memset(cdata, ' ', sizeof(cdata));
		i = 0;
		if(Cmode) {
			i = (Cmode == 1) ? 1280 : 0;
			do {
				pokew(W_BASE, i, 0x0720); }
			while((i += 2) < 3840); }
	case '^'-0x40:			// Home Cursor
		cx = cy = 0;
		if(Cmode == 1) cy = 8;
		goto upxy;
	case 0x1B :				// Cursor position
		cstate = 1;
		return;
	case 8 :				// Backspace
		if(cx)
			--cx;
		goto upxy; }

	if((c < ' ') || (c >= 0x7F))
		return;

	cdata[cy][cx] = c;
	if(Cmode) {
		if((Cmode == 2) || (cy > 7))
			poke(W_BASE, (cy*160)+(cx*2), c); }

advance:
	if(++cx > 79) {
		cx = 0;
		newline();
		if(cy >= 23)
			cscroll();
		else
			++cy; }

upxy:	if(Cmode) cupdatexy();
}

/*
 * Cancel stepping
 */
register error(unsigned args)
{
	unsigned char emsg[128];
	_format_(nargs() * 2 + &args, emsg);
	if(!pwin) {
		fputs(emsg, stdout);
		exit(0); }
	*dwin = DWIN;
	wclwin();
	wputs(emsg);
	eflag = 255;
}

/*
 * Display status line
 */
register status(unsigned args)
{
	unsigned char buffer[128];
	_format_(nargs() * 2 + &args, buffer);
	wclwin();
	wputs(buffer);
}

/*
 * "wgetc" function with background clock update
 */
int wgetct() asm
{
		EXTRN	_wtstc:NEAR
wgetc1:	CALL	_wtstc			; Any characters?
		AND		AX,AX			; Data?
		JNZ		wgetc2			; Yes, exit
		CALL	upclock			; Update clock
		JMP SHORT wgetc1		; Do them all
wgetc2:
}

/*
 * Hexidecimal input function
 */
int gethex(unsigned char in, unsigned *dest, unsigned length)
{
	unsigned c, v, l;
	unsigned char i, d;
	v = l = 0;
	i = in;
	for(;;) {
		if(!(c = i)) {
			wupdatexy();
			c = wgetct(); }
		d = c;
		i = 0;
		if((c >= '0') && (c <= '9'))
			c -= '0';
		else if((c >= 'A') && (c <= 'F'))
			c -= ('A' - 10);
		else if((c >= 'a') && (c <= 'f'))
			c -= ('a' - 10);
		else switch(c) {
			case '?' :
			case _K1 : help(0); continue;
			case 0x1B:
			case _K10: return 0;
			case _KBS:
				if(l) {
					--l;
					v >>= 4;
					wputc('\b');
					continue; }
			default:	
				beep(1000, 250);
				if(in && !l)
					return 0;
				continue; }
		wputc(d);
		v = (v << 4) | c;
		if(++l >= length) {
			*dest = v;
			return 255; } }
}

/*
 * Interactive memory editor
 */
void edit_memory(void)
{
	unsigned xc, xh, y, c, s, v;
	static int offset;
	static char mode;

	close_console();
	status("Memory Editor: F1=help  F2=Mode  F3=Address  F10=Exit");
	W_OPEN = mwin;
	wcursor_line();
redraw:
	memory();
	for(;;) {
		if(offset < 0) {
			do {
				offset += s;
				Maddress -= s; }
			while(offset < 0);
			goto redraw; }
		if(offset > 127) {
			do {
				offset -= s;
				Maddress += s; }
			while(offset > 127);
			goto redraw; }
		y = offset >> 3;
		xc = offset & 7;
		xh = (xc * 3) + 6;
		xc += 31;
		wgotoxy(mode ? xc : xh, y);
		s = 1;
		switch(c = wgetct()) {
		case _KRA: ++offset;	s=1;	continue;
		case _KLA: --offset;	s=1;	continue;
		case _KUA: offset -= (s=8); 	continue;
		case _KDA: offset += (s=8);		continue;
		case _KPU: Maddress -= 128;	goto redraw;
		case _KPD: Maddress += 128;	goto redraw;
		case '?' : if(mode) break;
		case _K1 : help(HEDITM);	continue;
		case _K2:	mode = mode ? 0 : 255; continue;
		case _K3:
			wgotoxy(0, 0);
			if(gethex(0, &Maddress, 4))
				offset = 0;
			goto redraw;
		case 0x1B : if(mode) break;
		case _K10:	W_OPEN = dwin; wcursor_off();	return; }
		if(c & 0xFF00) continue;
		if(!mode) {
			if(!gethex(c, &v, 2))
				goto redraw;
			c = v; }
		poke(SEG, offset++ + Maddress, c);
		wgotoxy(xh, y);
		wprintf("%02x", c);
		wgotoxy(xc, y);
		wputc( ((c < ' ') || (c > 0x7E)) ? '.' : c); }
}

/*
 * Interactive register editor
 */
void edit_register(void)
{
	unsigned x, y, v;
	unsigned char *bp;
	unsigned *wp;

	close_console();
redraw:
	W_OPEN = dwin;
	dumpreg();
	for(;;) {
		status("Select Register: A B C D E H L Sp Pc psW  ESC=exit? ");
		wcursor_line(); wupdatexy();
		switch(toupper(wgetct())) {
		case 'A' : y = 1; x = 4; bp = &A; goto in1;
		case 'B' : y = 2; x = 4; bp = &B; goto in1;
		case 'C' : y = 2; x = 7; bp = &C; goto in1;
		case 'D' : y = 3; x = 4; bp = &D; goto in1;
		case 'E' : y = 3; x = 7; bp = &E; goto in1;
		case 'H' : y = 4; x = 4; bp = &H; goto in1;
		case 'L' : y = 4; x = 7; bp = &L; goto in1;
		case 'W' : y = 7; x = 4; bp = &PSW;
		in1: status("Enter value: ESC=cancel");
			W_OPEN = rwin;
			wgotoxy(x, y);
			if(gethex(0, &v, 2))
				*bp = v;
			goto redraw;
		case 'S' : y = 5; x = 4; wp = &SP; goto in2;
		case 'P' : y = 6; x = 4; wp = &PC;
		in2: status("Enter value: ESC=cancel");
			W_OPEN = rwin;
			wgotoxy(x, y);
			if(gethex(0, &v, 4))
				*wp = v;
			goto redraw;
		case '?' :
		case _K1 : help(HEDITR); continue;
		case 0x1B:
		case _K10: return; }
		beep(1000, 250); }
}

/*
 * Step a block of 8080 instructions
 */
unsigned XT1, XT2, XT3, XT4;
unsigned char *CMnames[3] = { "Panel", "Tty  ", "Debug" };
unsigned char DBtick, XT5;
int dostep(unsigned s)
{
	unsigned i, t;

	M1 = 255;
	if(!(t = s)) {
		*dwin = RUN;
		M1 = 25;
		//      0....*....1....*....2....*....3....*....4....*....5....*....6....*....
		status("Run: F1=Help  F2=%s  F3=Stop  F4=Reset  F5=Clear  F6=%s",
			CMnames[Cmode],
			Tflag ? "Fast" : "Slow" );
		clrdiskled();
		if(Cmode)
			cupdatexy(); }
	INSTCT = XT3 = XT4 = XT5 = 0;
	BRKSK = 255;	// Skip first breakpoint
top:
	if(i = kbtst()) switch(i) {
		case _F1 : help(HRUN); break;
		case _F2 :					// Toggle console mode
			wgotoxy(17, 0);
			switch(++Cmode) {
			case 1 :	// 1/2 terminal
			case 2 :	// Full terminal
				wputs(CMnames[Cmode]);
				open_console();
				break;
			default:	// Control
				wputs(CMnames[0]);
				close_console(); }
			break;
		case _F3 :	error("Stop");				return 1;
		case _F4 :	reset(); error("Reset");	return 1;
		case _F5 : cputc(0x1A);					break;
		case _F6 :					// Toggle TTY speed
			wgotoxy(56, 0);
			if(Tflag) {
				wputs("Slow");
				Tflag = 0; }
			else {
				wputs("Fast");
				Tflag = 255; }
			if(Cmode) cupdatexy();
			break;
		case _DEL : i = 0x7F;
		default:
			if(i < 256) {
				Cdata = i;
				Cdata1 = 255; } }

	if(s) {
		STEP = (t < Step) ? t : Step;
		t -= STEP; }
	else {
		t = 255;
		STEP = Step; }

	if(Cmode != 2) {
		asm " MOV ES,DGRP:_W_BASE";
		upleds();
		write_address(PC);
		write_data(LEDS);
		if(DBtick) {
			DBtick = 0;
			if(!Cmode) {
				Daddress = PC;
				disassemble(16, 255);
				dumpreg();
				memory(); } } }

	asm {
; Wait for clock tick for housekeeping functions
		XOR		AX,AX			; Seg 0
		MOV		ES,AX			; Set ES
		MOV		AX,WORD PTR ES:[046Ch] ; Read tick
		CMP		AX,DGRP:_XT1	; Changed?
		JZ		diskt1			; No
		MOV		DGRP:_XT1,AX	; New value
; Tick occured - first update clock
		AND		AL,0FEh			; Mask 1s
		MOV		DGRP:_DBtick,AL	; Set tick
		CALL	clktick			; Update clock
; Update instruction rate
		MOV		AL,DGRP:_XT5	; Get update flag
		AND		AL,AL			; Test
		JZ		newsp			; First tick
		MOV		AX,DGRP:_INSTCT	; Get current speed
		XOR		BX,BX			; Get zero
		MOV		DGRP:_INSTCT,BX	; Reset counter
		MOV		BX,DGRP:_SLOWCT	; Get slow count
		CMP		AX,DGRP:_Speed	; Compare to desired
		JC		spdup			; Must go faster
		INC		BX				; Slow down a little
		JMP		SHORT savspd	; And continue
newsp:	MOV		AX,DGRP:_Speed	; Get speed setting
		OR		AL,AH			; Test for zero
		MOV		DGRP:_XT5,AL	; Save
		JMP		SHORT nospd		; And proceed
spdup:	CMP		BX,2			; Can we go faster?
		JC		nospd			; No - stay here
		DEC		BX				; Speed up a little
savspd:	MOV		DGRP:_SLOWCT,BX	; Resave it
; Update disk controller value
nospd:	MOV		AX,DGRP:_XT2	; Get count
		XOR		DX,DX			; Get zero
		MOV		BX,3			; / 3
		IDIV	BX				; AX = AX / 3
		MOV		DGRP:_XT3,AX	; Save 1st third
		ADD		AX,AX			; Calculate 2nd third
		MOV		DGRP:_XT4,AX	; Save step
		CALL	_disktick		; Perform disk tick
		XOR		AX,AX			; Get zero
		MOV		DGRP:_XT2,AX	; Reset count
		JMP	SHORT diskt3		; And proceed
diskt1:	INC		WORD PTR DGRP:_XT2 ; Advance count
		MOV		AX,DGRP:_XT2	; Get step
		CMP		AX,DGRP:_XT3	; Time to step?
		JZ		diskt2			; Yes
		CMP		AX,DGRP:_XT4	; Time to step?
		JNZ		diskt3			; No
diskt2:	CALL	_disktick		; Perform tick
diskt3:
	}

	switch(i = multi_step()) {
	case 0 : if(t) goto top; *dwin = DWIN;					return 0;
	case 1 : error("HLT instruction");						break;
	case 2 : error("Breakpoint!");							break;
	case -1 : error("Illegal instruction encountered.");	break;
	case -2 : error("Unknown I/O port");					break;
	default: error("STOP reason: %d", i); }
	return -1;
}

/*
 * Load a byte from the hex file
 */
unsigned load_byte(void)
{
	unsigned char c, v, ct;
	ct = 2;
	do {
		c = *ptr++;
		if((c >= '0') && (c <= '9'))
			c -= '0';
		else if((c >= 'A') && (c <= 'F'))
			c -= ('A'-10);
		else if((c >= 'a') && (c <= 'f'))
			c -= ('a'-10);
		else {
			printf("%u: Bad hex digit '%c'[%02x]\n", PC, c, c);
			exit(-1); }
		v = (v << 4) | c; }
	while(--ct);
	return v;
}

/*
 * Copy filename to Dbuffer & append extension
 */
void filename(unsigned char *file, unsigned char *extension, unsigned char *n)
{
	unsigned char *d, *p, *e;
	d = Dbuffer;
	e = 0;
	p = file;
	while(*d = *file++) {
		switch(*d++) {
		case '.' : e = file-1; continue;
		case '\\':
		case ':' : p = file; e = 0; } }
	if(e)
		*e = 0;
	else
		strcpy(d, extension);
	if(n)
		strcpy(n, p);
}

/*
 * Read file into memory image
 */
unsigned load_file(void)
{
	unsigned a, b, chksum, addr, length, count;
	unsigned char buffer[80];
	FILE *fp;

	filename(LPC, ".HEX", 0);
	if(!(fp = fopen(Dbuffer, "r"))) {
		error("can't open: %s", Dbuffer);
		goto r1; }

	for(Lsize = PC = 0; fgets(ptr = buffer, 80, fp);) {
		++PC;
		again: switch(*ptr++) {
			case 'S' :	/* Motorola HEX format */
				if(*ptr == '9') goto quitload;
				if(*ptr++ != '1') continue;
				length = count = (chksum = load_byte()) - 3;
				chksum += a = load_byte();
				chksum += b = load_byte();
				addr = (a << 8) + b;
				if(!Lsize)
					LPC = addr;
				while(count--) {
					chksum += a = load_byte();
					poke(SEG, addr++, a); }
				if((255 & ~chksum) != load_byte())
					goto quitbad;
				break;
			case ':' :		/* Intel HEX format */
				if(!(length = count = load_byte())) goto quitload;
				chksum = (a = load_byte()) + length;
				chksum += b = load_byte();
				addr = (a << 8) + b;
				chksum += load_byte();
				if(!Lsize)
					LPC = addr;
				while(count--) {
					chksum += a = load_byte();
					poke(SEG, addr++, a); }
				if((255 & -chksum) != load_byte())
					goto quitbad;
				break;
			case ' ' :		/* Space */
			case '\t' :		/* Tab */
				goto again;
			case 0 :		/* Null line */
				continue;
			default:
				error("%u: Invalid record format", PC);
				goto quit; }
		Lsize += length; }

quitbad:
	error("%u: Bad record checksum", PC);
quit:
	fclose(fp);
r1:	return 255;
quitload:
	return 0;
}

unsigned save_file(unsigned mf, unsigned mt, unsigned char ty)
{
	unsigned i;
	unsigned char h, l, c, ck;
	FILE *fp;
	filename(Directory, ".HEX", 0);
	if(mf > mt) {
		i = mf;
		mf = mt;
		mt = i; }
	if(!(fp = fopen(Dbuffer, "w"))) {
		error("can't open: %s", Dbuffer);
		return 255; }
	++mt;
	while(mf < mt) {
		if((i = mt - mf) > 32) i = 32;
		h = mf >> 8;
		l = mf & 255;
		if(ty) {	//Intel
			ck = i + h + l;
			fprintf(fp, ":%02x%02x%02x00", i, h, l);
			while(i--) {
				fprintf(fp, "%02x", c = Mread(mf++));
				ck += c; }
			fprintf(fp, "%02x\n", 255 & (0-ck)); }
		else {	// Motorola
			ck = (i + h + l)+3;
			fprintf(fp, "S1%02x%02x%02x", i+3, h, l);
			while(i--) {
				fprintf(fp, "%02x", c = Mread(mf++));
				ck += c; }
			fprintf(fp, "%02x\n", 255 & ~ck); }
	}
	if(ty)	fprintf(fp,":00000001FF\n");
	else	fprintf(fp,"S9030000FC\n");
	fclose(fp);
	return 0;
}

#if 0
		if(intel) {					/* intel hex format */
			chk = outrec[0] + outrec[1] + ocnt - 2;
			fprintf(hex_fp,":%02x%02x%02x00", ocnt-2, outrec[0], outrec[1]);
			for(i=2; i<ocnt; ++i) {
				fprintf(hex_fp,"%02x", c = outrec[i]);
				chk += c; }
			fprintf(hex_fp,"%02x\n", 255 & (0-chk)); }
		else {						/* motorola hex format */
			chk = ocnt + 1;
			fprintf(hex_fp,"S1%02x", ocnt + 1);
			for(i=0; i<ocnt; ++i) {
				fprintf(hex_fp,"%02x", c = outrec[i]);
				chk += c; }
			fprintf(hex_fp,"%02x\n", 255 & ~chk); }
#endif
				
/*
 * Parse a command line parameter
 */
void cmdparm(unsigned char *p)
{
	unsigned char c, w;
	unsigned l1[2], l2[2];
	static char Dindex;

	while(isspace(*p))
		++p;

	w = 0;
	switch((toupper(*p++) << 8) | toupper(*p++)) {
	case 'C3' : ++w;
	case 'C2' : ++w;
	case 'C1' : if(*p != '=') goto badopr;
		if(Cport) {
			printf("Cannot redirect more than one device");
			break; }
		Cport = w | 0xF0;
		Cport1 = atoi(p+1);
		if((Cport1 < 1) || (Cport1 > 4)) {
			printf("Com port must be 1-4");
			break; }
		while(*p++ != ':') {
			if(!*p)
				return; }
		switch(*p++) {
		default:
		badfmt: printf("Bad format."); goto quit;
		case '8' : Cfmt = DATA_8; break;
		case '7' : Cfmt = DATA_7; break;
		case '6' : Cfmt = DATA_6; break;
		case '5' : Cfmt = DATA_5; }
		switch(*p++) {
		default: goto badfmt;
		case '1' : w = STOP_1;	break;
		case '2' : w = STOP_2; }
		Cfmt |= w;
		switch(toupper(*p++)) {
		default: goto badfmt;
		case 'N' : w = PAR_NO;		break;
		case 'E' : w = PAR_EVEN;	break;
		case 'O' : w = PAR_ODD;		break;
		case 'M' : w = PAR_MARK;	break;
		case 'S' : w = PAR_SPACE; }
		Cfmt += w;
		Cport1 |= 0xF0;
		longset(l1, atoi(p));
		if(!longtst(l1)) goto badfmt;
		longcpy(l2, L115200);
		longdiv(l2, l1);
		Cbaud = *l2;
		return;
	case 'L=' : LPC = p;			return;		// Load file
	case 'S=' : Switch	= atox(p);	return;		// Initial Switches
	case 'M=' : Maddress= atox(p);	return;		// Initial Memory view
	case 'F=' :									// Fill memory value
		PC = atox(p) & 255;
		PC |= PC << 8;
		return;
	case 'W=' : w = 255;						// Writeable disk
	case 'R=' :									// Read only disk
		if(++Dindex > 3) {
			printf("Too many R=/W= drives (Max 3)");
			break; }
		filename(p, ".NSI", Dname[Dindex-1]);
		if(Dhandle[Dindex] = open(Dbuffer, w ? (F_READ|F_WRITE) : F_READ)) {
			Dwrite[Dindex] = w;
			return; }
		printf("Unable to access: %s", Dbuffer);
		break;
	case '/I' :	c = O_BIO;
a1:		Copt ^= c;
		return;
	case '/G':	c = O_GO;	goto a1;
	case '/E':	c = O_NEP;	goto a1;
	case '/H':	c = O_HEX;	goto a1;
	case '?'<<8 :
	case '-?' :
	case '/?' : help(HCMD); exit(0);
	case '/P' : switch(toupper(*p++)) {
		case 'T' :	SP = 0x1234;					return;
		case ':' :	Speed = atoi(p);				return;
		case 0	:	Speed = 350;					return;
		case 'D' :	w = 255;
		case 'I' :	Speed = 0;
			if(*p == ':') {
				if(w) setslow(atoi(p+1)); else Step = atoi(p+1);
				return; } }
		--p;
	badopr:
	default: printf("Unknown option: '%s'", p-2); }
quit:
	putc('\n', stdout);
	exit(-1);
}

/*
 * Execute a sub-shell
 */
doshell()
{
	unsigned char comspec[65], tail[80];

	if(!getenv("COMSPEC", comspec)) {
		error("Cannot locate COMSPEC");
		return; }
	*tail = 0;
	wopen(0, 0, 80, 25, WSAVE|WCOPEN|NORMAL);
	wcursor_line();
	wupdatexy();
	exec(comspec, tail);
	wclose();
}

/*
 * Calibrate CPU performance
 */
void calibrate(void)
{
	unsigned t, t1, i, aj, SPC;

	if(!(Speed = (Speed + 9) / 18))
		return;

	SPC = PC;
	aj = 1000;

	wopen(15, 8, 50, 7, WSAVE|WBOX1|WCOPEN|REVERSE|WSCROLL);
	wgotoxy(9, 0); wputs(" >>> Calibrating CPU speed <<<");
	wgotoxy(10, 4); wputs(" --- Press ESC to bypass ---");

	PC = 0xE9A0;	// Jump to self
	setslow(1);
	t1 = TICK;
	while((t = TICK) == t1);	// Wait for clock to tick
	while(kbtst() != 0x1B) {
		t1 = t;
		INSTCT = 0;
		do {
			STEP = Step;
			multi_step(); }
		while((t = TICK) == t1);
		wgotoxy(15, 2);
		wprintf("%-5u  %-5u  %-5u", INSTCT, SLOWCT, aj);
		if(INSTCT > Speed) {
			SLOWCT += aj;
			continue; }
		i = aj;
		if(aj /= 10) {
			SLOWCT = (SLOWCT > i) ? (SLOWCT - i) : 1;
			continue; }
		break; }
	wclose();
	PC = SPC;
}

trace()
{
	unsigned t, i, j, s, st;
	close_console();
home:
	st = j = TRCPTR / 2;
reshow:
	wcursor_off();
	j &= 4095;
	if(i = (st-j) & 4095)
		Daddress = s = TRCBUF[t = j];
	else {
		t = st;
		Daddress = s = PC; }
	status("TRACE %-4u: F1=Help  F2=SearchF  F3=SearchB  F4=NextF  F5=NextB  F10=Exit", i);
	disassemble(16, 255);
	for(;;) switch(wgetct()) {
	case '?' :
	case _K1 : help(HTRACE);			continue;
	case _K2 :
		status("Search Forward?"); i = 1;
		goto dosearch;
	case _K3 :
		status("Search Backward?"); i = -1;
	dosearch:
		wcursor_line();
		if(gethex(0, &s, 4)) {
dosr:			j = t;
			do {
				j = (j + i) & 4095;
				if(TRCBUF[j] == s)
					goto reshow; }
			while(j != st);
			beep(1000, 100); }
		j = t;
		goto reshow;
	case _K4 : i = 1; goto dosr;
	case _K5 : i = -1; goto dosr;
	case _KHO : j = st+1;				goto reshow;
	case _KEN : goto home;
	case _KUA : j = t - 1;				goto reshow;
	case _KDA : j = t + 1;				goto reshow;
	case _KPU : j = t - 100;			goto reshow;
	case _KPD :	j = t + 100;			goto reshow;
	case _K10 :
	case 0x1B : return;
	default: if(eflag) { eflag = 0; goto reshow; } }
}

#ifndef MemTst
	void MemTst(unsigned char x)
	{
		unsigned i, j;
		unsigned char *p, *p1;
		static unsigned char *mp;

		if(x) {
			p1 = &i - 16;
			free(mp = p = malloc(8));
			while(p < p1)
				*p++ = 0xA5;
			return; }

		i = 0;
		p1 = &x;
		printf("Mem: %04x-", mp);
	a1:	j = 0;
		while(mp < p1) {
			if(*mp++ != 0xA5) {
				if(j > i) {
					i = j;
					p = mp; }
				goto a1; }
			++j; }
		printf("%04x %u\n", p, i);
	}
#endif


main(int argc, char *argv[])
{
	unsigned i, j, k, t;

	MemTst(7);
	Directory[0] = get_drive() + 'A';
	Directory[1] = ':';
	Directory[2] = '\\';
	getdir(Directory+3);
	for(i=1; i < argc; ++i)
		cmdparm(argv[i]);

	if(Copt & O_HEX) {
		memcpy(Panel, HPanel, sizeof(HPanel));
		memcpy(Leds, HLeds, sizeof(HLeds)); }

	SEG = alloc_seg(4096);
	i = 0;
	do {
		pokew(SEG, i, PC); }
	while(i += 2);

	if(SP == 0x1234) {		// Performance analysis
		printf("Blocks of %u insts/sec., press ESC to exit\n", Step);
		reset(); PC=0xE9A0;
		while(kbtst() != 0x1B) {
			i = TICK;
			while((t = TICK) == i);
			i = 0;
			ptr = "%u\n";
			do {
				STEP = Step; 
				multi_step();
				if(!++i)
					ptr = ">65535\n"; }
			while((TICK - t) < 18);
			printf(ptr, i); }
		return; }

	if(LPC)
		load_file();
#if 0
strcpy(Directory, "B");
save_file(0x3000, (0x3000+Lsize)-1, 7);
return;
#endif
	if(Cport) {
		if(Copen(Cport1 & 15, Cbaud, Cfmt, Csig))
			error("Cannot open COM port");
		Cflags |= TRANSPARENT; }

	pwin = wopen(0, 0, 80, 8, WSAVE|WCOPEN|0x17);
	awin = wopen(0, 8, 31, 16, WSAVE|WCOPEN|0x67);
	rwin = wopen(31,8, 10, 16, WSAVE|WCOPEN|0x57);
	mwin = wopen(41, 8, 39, 16, WSAVE|WCOPEN|0x67);
	dwin = wopen(0, 24, 80, 1, WSAVE|WCOPEN|DWIN);
	draw_panel();
	open_console();
	memset(cdata, ' ', sizeof(cdata));
	reset();
	W_OPEN = dwin;
	wcursor_off();
	if(Lsize)
		status("%s - %u bytes at %04x", Dbuffer, Lsize, LPC);
	else
		status(" Virtual Altair - Dave Dunfield - "#__DATE__"");
	calibrate();
	wgotoxy(71, 0); wputs("F1=Help");
	Clast = TICK;
	if(Copt & O_GO)
		goto go;
stop:
	t = TRCPTR/2;
//	j = Mread(Address = Daddress = PC);
	Address = Daddress = PC;
	if(MEMR) MEMR = 255;
	if(IN)	IN = 255;
	if(M1)	M1 = 255;
	if(OUT)	OUT = 255;
	if(STACK) STACK=255;
	if(Cmode != 2) {
		upleds();
		write_address(Address);
		write_data(LEDS); }
	disassemble(16, 255);
	dumpreg();
	memory();
cmsg:
	if(!eflag)
		status("F1=Help F2=%s F3=EdMem F4=EdReg F5=BkPt F6=Dis F7=Mount F8=Trc F9=Dos F10=Off",
			CMnames[Cmode]);
	eflag = 0;
	wcursor_off();
cmd:
	switch(k = toupper(wgetc())) {
	case _K10 :
		if(Copt & O_NEP)
			goto ex;
		status("Exit ALTAIR simulator (Y/N)?");
		for(;;) switch(wgetct()) {
		case 'n' :
		case 'N' :
		case 0x1B:
			goto cmsg;
		case 'y' :
		case 'Y' :
		case '\n' :
ex:			if(Cport)
				Cclose();
			wclose();
			wclose();
			wclose();
			wclose();
			wclose();
			for(i=0; i < 4; ++i) {
				if(Dhandle[i])
					close(Dhandle[i]); }
			MemTst(0);
			return; }
	case 'G' :
go:		clrdiskled();
		if(Cmode)
			wcursor_line();
		LEDS = 255;
		dostep(0);
		goto stop;
	case 'T' :
		dostep(1);
		LEDS = A;
		goto stop;
	case 'R' :
		reset();
		error("Reset");
		goto stop;
	case '0' :
	case '1' :
	case '2' :
	case '3' :
	case '4' :
	case '5' :
		if(SWhil & 0xFF00)	k += 10;
	case '6' :
	case '7' :
	case '8' :
	case '9' :
		k -= '0';
		Switch ^= (1 << k);
		if(SWhil == 0xFC00) {
			SWhil = 0;
			draw_sws(); }
		draw_switch();
		if(Cmode == 2) close_console();
		goto cmsg;
	case '=':	SWhil = 0xFC00;	goto a1;
	case '[':	SWhil = 0xFFC0;	goto a1;
	case ']':	SWhil = 0x3FF;
a1:		draw_sws();
		goto cmd;
	case '-':
		SWhil = 0;
		goto a1;
	case 'N' :		// Examine next
		++Address;
		goto readm1;
	case 'E' :		// Examine
		PC = Address = Switch;
		dumpreg();
	readm1:
		M1 = 0;
		j = Mread(Address);
		if(Cmode != 2)
			upleds();
		write_address(Address);
		write_data(j);
		goto cmd;
	case 'X' :		// Deposit next
		++Address;
	case 'D' :		// Deposit
		poke(SEG, Address, Switch);
		goto readm1;
	case _K2 :	// TTY mode
		if(++Cmode > 2) {
			close_console();
			goto cmsg; }
		open_console();
		wcursor_off();
		goto cmsg;
	case _KDA:
		j = daddress[1];
		goto dodis;
	case _KPD:
		j = daddress[15];
	dodis:
		Daddress = daddress[0] = j;
	dodis1:
		close_console();
		disassemble(16, 255);
		goto cmsg;
	case _KUA: i = 1; goto dotb;
	case _KPU:
		i = 15;
	dotb:
		close_console();
		k = Daddress;
		Daddress = daddress[0] = (Daddress - 58);
		disassemble(64, 0);
		for(j=15; j < 64; ++j) {
			if(daddress[j] >= k)
				break; }
		j = daddress[j-i];
//		if(j > Daddress)
//			j = 0;
		goto dodis;
	case _K3:	edit_memory();		goto cmsg;
	case _K4:	edit_register();	goto cmsg;
	case _K5:
		status("Break=%04x?", BREAK);
		wcursor_line();
		if(gethex(0, &j, 4))
			BREAK = j;
		goto cmsg;
	case _K6:
		close_console();
		W_OPEN = awin;
		wgotoxy(0, 0);
		wcursor_line();
		if(gethex(0, &Daddress, 4))
			daddress[0] = Daddress;
		W_OPEN = dwin;
		wcursor_off();
		goto dodis1;
	case _K7:
		wclwin();
		for(j=1; j < 4; ++j)
			wprintf(" D%u-%c: %-12s", j, Dwrite[j] ? 'W' : 'R', Dname[j-1]);
		/* *Temp = */ j = 0;
		if(!wmenu(30, 9, WSAVE|WCOPEN|WBOX1|0x70, Dmenu, &j)) {
			k = (j >> 1) + 1;	// Drive number
			if(Dhandle[k]) {
				close(Dhandle[k]);
				*Dname[k-1] = Dhandle[k] = Dwrite[k] = 0; }
			if(selectfile(".NSI")) {
				filename(Directory, ".NSI", Dname[k-1]);
				if(!(Dhandle[k] = open(Dbuffer, (j&1) ? (F_READ|F_WRITE) : F_READ))) {
					*Dname[k-1] = 0;
					status("Cannot access: %s", Dbuffer);
					goto cmd; }
				if(j & 1) Dwrite[k] = 255; } }
		wcursor_off();
		goto cmsg;
	case _K8 : trace();	goto cmsg;
	case _K9 : doshell(); goto cmsg;
	case 'M':
		wclwin();
		wputs("memory Load/saveIntel/saveMotorola?");
		wcursor_line();
c1:		switch(k = toupper(wgetc())) {
		default	:	goto c1;
		case 'L':
			if(selectfile(".HEX")) {
				LPC = Directory;
				if(!load_file())
					error("%s - %u bytes at %04x", Dbuffer, Lsize, LPC); }
		case 0x1B:
		case _K10:
			goto cmsg;
		case 'M':	k = 0;
		case 'I':	wclwin(); }
		wputc(k || 'M');
		wputs("save ");
		if(!gethex(0, &i, 4)) goto cmsg;
		wputc('-');
		if(!gethex(0, &j, 4)) goto cmsg;
		if(selectfile(".HEX")) {
			if(!save_file(i, j, k))
				error("%csave %04x-%04x to %s", k || 'M', i, j, Dbuffer); }
		goto cmsg;
	case _K1 :
	case '?' : help(HSTOP); goto cmd;
	case 'W' :	k = 4;			goto b1;
	case 'B' :	k = 2;
b1:		status((k & 4) ? "Word?" : "Byte?");
		wcursor_line();
		if(gethex(0, &j, k)) {
			Switch &= (k & 2) ? 0xFF00 : 0;
			Switch |= j;
			draw_switch(); }
		goto cmsg;
	default: goto cmsg; }
}
#include "altair.cc1"
