#include <stdio.h>
#include <keys.h>
#include <window.h>
#include <file.h>
#include <comm.h>
#define	MemTst(a)
#define	TICK	peekw(0x40, 0x6C)

#define	TTY_LINES		24
#define	DEBUG_LINES		10
#define	SFILES			110
#define	TAB_SIZE		4
#define	LINKS			25
#define	HTEXT			0x70
#define	HLINK			0x71
#define	HLINKH			0x17
#define	LINK_S			('N'-0x40)
#define	LINK_E			('O'-0x40)

#define	HCMD	1
#define	HSTOP	2
#define	HRUN	3
#define	HEDITM	4
#define	HEDITR	5
#define	HTRACE	6
#define	HFILE	7
#define	HENTER	8

extern unsigned
	X,						// 6809 X register
	Y,						// 6809 Y register
	U,						// 6809 U register
	S,						// 6809 Stack pointer
	P,						// 6809 Program counter
	SEG,					// 6809 memory segment
	STEP,					// Virtual CPU step count
	BREAK,					// Virtual CPU breapoint
	TRCPTR,					// Trace pointer
	SPSTOP,					// Stack bound flag
	TRCBUF[4096];			// Trace buffer
extern unsigned char
	A,						// 6809 A register
	B,						// 6809 B register
	CC,						// 6809 Condition Codes
	DP,						// 680 Direct Page register
	*ARGV[];				// Program argument vector

unsigned
	Kbc,					// Console Fake character
	Memaddr,				// Current memory display address
	Rtick,					// Runtime tick
	Disaddr,				// Current disassembler address
	Dislist[128];			// List of disassembler lines

unsigned char
	*ptr,					// General pointer
	Cx,						// console X
	Cy,						// console Y
	Cmode = 255,			// Console mode
	Ssr,					// SSR report mode
	Ceffect,				// Console effect mode
	Cdelay,					// Console delay flag
	Cstate,					// Console TTY state
	Cdata[TTY_LINES][80],	// Console data buffer
	Uport,					// COM port
	Uctrl1,					// Uart control word1
	Uctrl2,					// Uart control word2
	Ubaud = 14,				// Uart baudrate
	Upending,				// Uart data pending
	Udata,					// Data byte
	Eflag,					// Error message flag
	Swapio,					// Swap I/O flag
	Directory[65],			// Input directory
	*Romfile = "D6809";		// Rom load file

struct WINDOW
	*dwin,					// Disassembly window
	*rwin,					// Register window
	*mwin,					// Memory window
	*swin,					// Status window
	*xwin;					// Main window

FILE
	*INfp,
	*OUTfp,
	*fp;

// Disk controller variables
unsigned
	Ddrive,					// Active drive
	Dactive,				// Activity count
	Dcylinder,				// Selected cylinder
	Dhead,					// Selected head
	Dsector,				// Selected sector
	Dnhead,					// # of heads
	Dnsector,				// # of sectors
	Dtick,					// Disk tick count
	Dptr,					// Data pointer
	Dop;					// Active operation/status
unsigned char
	Ddelay = 255,			// Simulate Disk Delay
	Dname[4][9],			// Disk names
	Dwrite[4],				// Disk write flags
	Dbuffer[512];			// Disk data buffer
HANDLE
	Dhandle[4];				// Open drive handles

unsigned char *Dmenu[] = {
	"A: RO",
	"A: R/W",
	"B: RO",
	"B: R/W",
	"C: RO",
	"C: R/W",
	"D: RO",
	"D: R/W",
	0 };

/*
 * Disassembler
 */
#define	Xputc(c)	wputc(c)
#define	Xprintf		wprintf
#define	Xputs(s)	wputs(s)

/* Table of instruction names and substrings */
unsigned char inames[] = {
	'N','E','G',	/* 80 */
	'C','O','M',	/* 81 */
	'L','S','R',	/* 82 */
	'R','O','R',	/* 83 */
	'A','S','R',	/* 84 */
	'L','S','L',	/* 85 */
	'R','O','L',	/* 86 */
	'D','E','C',	/* 87 */
	'I','N','C',	/* 88 */
	'T','S','T',	/* 89 */
	'J','M','P',	/* 8A */
	'C','L','R',	/* 8B */
	'L','B','R',	/* 8C */
	'A','N','D',	/* 8D */
	'B','R','A',	/* 8E */
	'B','R','N',	/* 8F */
	'B','H','I',	/* 90 */
	'B','L','S',	/* 91 */
	'B','C','C',	/* 92 */
	'B','C','S',	/* 93 */
	'B','N','E',	/* 94 */
	'B','E','Q',	/* 95 */
	'B','V','C',	/* 96 */
	'B','V','S',	/* 97 */
	'B','P','L',	/* 98 */
	'B','M','I',	/* 99 */
	'B','G','E',	/* 9A */
	'B','L','T',	/* 9B */
	'B','G','T',	/* 9C */
	'B','L','E',	/* 9D */
	'L','E','A',	/* 9E */
	'P','S','H',	/* 9F */
	'P','U','L',	/* A0 */
	'S','W','I',	/* A1 */
	'S','U','B',	/* A2 */
	'C','M','P',	/* A3 */
	'S','B','C',	/* A4 */
	'B','I','T',	/* A5 */
	'L','D','A',	/* A6 */
	'E','O','R',	/* A7 */
	'A','D','C',	/* A8 */
	'O','R','A',	/* A9 */
	'A','D','D',	/* AA */
	'B','S','R',	/* AB */
	'L','D','X',	/* AC */
	'S','T','A',	/* AD */
	'J','S','R',	/* AE */
	'S','T','X',	/* AF */
	'L','D','B',	/* B0 */
	'O','R','B',	/* B1 */
	'L','D','D',	/* B2 */
	'L','D','U',	/* B3 */
	'S','T','B',	/* B4 */
	'S','T','D',	/* B5 */
	'S','T','U',	/* B6 */
	'F','C','B',	/* B7 */
	'L','B','L',	/* B8 */
	'L','B','C',	/* B9 */
	'L','B','V',	/* BA */
	'L','B','G',	/* BB */
	'L','D','Y',	/* BC */
	'S','T','Y',	/* BD */
	'L','D','S',	/* BE */
	'S','T','S'		/* BF */
};

/* Table of 6809 opcodes: OPCODE, SYMBOL-FLAGS|LENGTH, TEXT */
/* SYMBOL-FLAGS: 00=None 10=8bitData 20=8bitAddress 30=16Bitdata/address */
/* 40=8bitRel 50=16BitRel 60=Indexed */
unsigned char itable[] = {
	0x3F, 0x02, "SSR b",
	0x00, 0x22, "\x80 <b",		/* NEG */
	0x03, 0x22, "\x81 <b",		/* COM */
	0x04, 0x22, "\x82 <b",		/* LSR */
	0x06, 0x22, "\x83 <b",		/* ROR */
	0x07, 0x22, "\x84 <b",		/* ASR */
	0x08, 0x22, "\x85 <b",		/* LSL */
	0x09, 0x22, "\x86 <b",		/* ROL */
	0x0A, 0x22, "\x87 <b",		/* DEC */
	0x0C, 0x22, "\x88 <b",		/* INC */
	0x0D, 0x22, "\x89 <b",		/* TST */
	0x0E, 0x22, "\x8A <b",		/* JMP */
	0x0F, 0x22, "\x8B <b",		/* CLR */

	0x12, 0x01, "NOP",
	0x13, 0x01, "SYNC",
	0x16, 0x53,	"\x8CA l",		/* LBRA */
	0x17, 0x53, "L\xAB l",		/* LBSR */
	0x19, 0x01, "DAA",
	0x1A, 0x02, "ORCC #b",
	0x1C, 0x02, "\x8DCC #b",	/* ANDCC */
	0x1D, 0x01, "SEX",
	0x1E, 0x02, "EXG t",
	0x1F, 0x02, "TFR t",

	0x20, 0x42,	"\x8E r",		/* BRA */
	0x21, 0x42, "\x8F r",		/* BRN */
	0x22, 0x42,	"\x90 r",		/* BHI */
	0x23, 0x42,	"\x91 r",		/* BLS */
	0x24, 0x42,	"\x92 r",		/* BCC */
	0x25, 0x42,	"\x93 r",		/* BCS */
	0x26, 0x42,	"\x94 r",		/* BNE */
	0x27, 0x42,	"\x95 r",		/* BEQ */
	0x28, 0x42,	"\x96 r",		/* BVC */
	0x29, 0x42,	"\x97 r",		/* BVS */
	0x2A, 0x42,	"\x98 r",		/* BPL */
	0x2B, 0x42,	"\x99 r",		/* BMI */
	0x2C, 0x42,	"\x9A r",		/* BGE */
	0x2D, 0x42,	"\x9B r",		/* BLT */
	0x2E, 0x42,	"\x9C r",		/* BGT */
	0x2F, 0x42,	"\x9D r",		/* BLE */

	0x30, 0x62, "\x9EX x",		/* LEAX */
	0x31, 0x62, "\x9EY x",		/* LEAY */
	0x32, 0x62, "\x9ES x",		/* LEAS */
	0x33, 0x62, "\x9EU x",		/* LEAU */
	0x34, 0x02, "\x9FS p",		/* PSHS */
	0x35, 0x02, "\xA0S p",		/* PULS */
	0x36, 0x02, "\x9FU q",		/* PSHU */
	0x37, 0x02, "\xA0U q",		/* PULU */
	0x39, 0x01, "RTS",
	0x3A, 0x01, "ABX",
	0x3B, 0x01, "RTI",
	0x3C, 0x02, "CWAI #b",
	0x3D, 0x01, "MUL",

	0x3F, 0x01, "\xA1",			/* SWI */

	0x40, 0x01, "\x80A",		/* NEGA */
	0x43, 0x01, "\x81A",		/* COMA */
	0x44, 0x01, "\x82A",		/* LSRA */
	0x46, 0x01, "\x83A",		/* RORA */
	0x47, 0x01, "\x84A",		/* ASRA */
	0x48, 0x01, "\x85A",		/* LSLA */
	0x49, 0x01, "\x86A",		/* ROLA */
	0x4A, 0x01, "\x87A",		/* DECA */
	0x4C, 0x01, "\x88A",		/* INCA */
	0x4D, 0x01, "\x89A",		/* TSTA */
	0x4F, 0x01, "\x8BA",		/* CLRA */

	0x50, 0x01, "\x80B",		/* NEGB */
	0x53, 0x01, "\x81B",		/* COMB */
	0x54, 0x01, "\x82B",		/* LSRB */
	0x56, 0x01, "\x83B",		/* RORB */
	0x57, 0x01, "\x84B",		/* ASRB */
	0x58, 0x01, "\x85B",		/* LSLB */
	0x59, 0x01, "\x86B",		/* ROLB */
	0x5A, 0x01, "\x87B",		/* DECB */
	0x5C, 0x01, "\x88B",		/* INCB */
	0x5D, 0x01, "\x89B",		/* TSTB */
	0x5F, 0x01, "\x8BB",		/* CLRB */

	0x60, 0x62, "\x80 x",		/* NEG */
	0x63, 0x62, "\x81 x",		/* COM */
	0x64, 0x62, "\x82 x",		/* LSR */
	0x66, 0x62, "\x83 x",		/* ROR */
	0x67, 0x62, "\x84 x",		/* ASR */
	0x68, 0x62, "\x85 x",		/* LSL */
	0x69, 0x62, "\x86 x",		/* ROL */
	0x6A, 0x62, "\x87 x",		/* DEC */
	0x6C, 0x62, "\x88 x",		/* INC */
	0x6D, 0x62, "\x89 x",		/* TST */
	0x6E, 0x62, "\x8A x",		/* JMP */
	0x6F, 0x62, "\x8B x",		/* CLR */

	0x70, 0x33, "\x80 w",		/* NEG */
	0x73, 0x33, "\x81 w",		/* COM */
	0x74, 0x33, "\x82 w",		/* LSR */
	0x76, 0x33, "\x83 w",		/* ROR */
	0x77, 0x33, "\x84 w",		/* ASR */
	0x78, 0x33, "\x85 w",		/* LSL */
	0x79, 0x33, "\x86 w",		/* ROL */
	0x7A, 0x33, "\x87 w",		/* DEC */
	0x7C, 0x33, "\x88 w",		/* INC */
	0x7D, 0x33, "\x89 w",		/* TST */
	0x7E, 0x33, "\x8A w",		/* JMP */
	0x7F, 0x33, "\x8B w",		/* CLR */

	0x80, 0x12, "\xA2A #b",		/* SUBA */
	0x81, 0x12, "\xA3A #b",		/* CMPA */
	0x82, 0x12, "\xA4A #b",		/* SBCA */
	0x83, 0x33, "\xA2D #w",		/* SUBD */
	0x84, 0x12, "\x8DA #b",		/* ANDA */
	0x85, 0x12, "\xA5A #b",		/* BITA */
	0x86, 0x12, "\xA6 #b",		/* LDA */
	0x88, 0x12, "\xA7A #b",		/* EORA */
	0x89, 0x12, "\xA8A #b",		/* ADCA */
	0x8A, 0x12, "\xA9 #b",		/* ORA */
	0x8B, 0x12, "\xAAA #b",		/* ADDA */
	0x8C, 0x33, "\xA3X #w",		/* CMPX */
	0x8D, 0x42, "\xAB r",		/* BSR */
	0x8E, 0x33, "\xAC #w",		/* LDX */

	0x90, 0x22, "\xA2A <b",		/* SUBA */
	0x91, 0x22, "\xA3A <b",		/* CMPA */
	0x92, 0x22, "\xA4A <b",		/* SBCA */
	0x93, 0x22, "\xA2D <b",		/* SUBD */
	0x94, 0x22, "\x8DA <b",		/* ANDA */
	0x95, 0x22, "\xA5A <b",		/* BITA */
	0x96, 0x22, "\xA6 <b",		/* LDA */
	0x97, 0x22, "\xAD <b",		/* STA */
	0x98, 0x22, "\xA7A <b",		/* EORA */
	0x99, 0x22, "\xA8A <b",		/* ADCA */
	0x9A, 0x22, "\xA9 <b",		/* ORA */
	0x9B, 0x22, "\xAAA <b",		/* ADDA */
	0x9C, 0x22, "\xA3X <b",		/* CMPX */
	0x9D, 0x22, "\xAE <b",		/* JSR */
	0x9E, 0x22, "\xAC <b",		/* LDX */
	0x9F, 0x22, "\xAF <b",		/* STX */

	0xA0, 0x62, "\xA2A x",		/* SUBA */
	0xA1, 0x62, "\xA3A x",		/* CMPA */
	0xA2, 0x62, "\xA4A x",		/* SBCA */
	0xA3, 0x62, "\xA2D x",		/* SUBD */
	0xA4, 0x62, "\x8DA x",		/* ANDA */
	0xA5, 0x62, "\xA5A x",		/* BITA */
	0xA6, 0x62, "\xA6 x",		/* LDA */
	0xA7, 0x62, "\xAD x",		/* STA */
	0xA8, 0x62, "\xA7A x",		/* EORA */
	0xA9, 0x62, "\xA8A x",		/* ADCA */
	0xAA, 0x62, "\xA9 x",		/* ORA */
	0xAB, 0x62, "\xAAA x",		/* ADDA */
	0xAC, 0x62, "\xA3X x",		/* CMPX */
	0xAD, 0x62, "\xAE x",		/* JSR */
	0xAE, 0x62, "\xAC x",		/* LDX */
	0xAF, 0x62, "\xAF x",		/* STX */

	0xB0, 0x33, "\xA2A w",		/* SUBA */
	0xB1, 0x33, "\xA3A w",		/* CMPA */
	0xB2, 0x33, "\xA4A w",		/* SBCA */
	0xB3, 0x33, "\xA2D w",		/* SUBD */
	0xB4, 0x33, "\x8DA w",		/* ANDA */
	0xB5, 0x33, "\xA5A w",		/* BITA */
	0xB6, 0x33, "\xA6 w",		/* LDA */
	0xB7, 0x33, "\xAD w",		/* STA */
	0xB8, 0x33, "\xA7A w",		/* EORA */
	0xB9, 0x33, "\xA8A w",		/* ADCA */
	0xBA, 0x33, "\xA9 w",		/* ORA */
	0xBB, 0x33, "\xAAA w",		/* ADDA */
	0xBC, 0x33, "\xA3X w",		/* CMPX */
	0xBD, 0x33, "\xAE w",		/* JSR */
	0xBE, 0x33, "\xAC w",		/* LDX */
	0xBF, 0x33, "\xAF w",		/* STX */

	0xC0, 0x12, "\xA2B #b",		/* SUBB */
	0xC1, 0x12, "\xA3B #b",		/* CMPB */
	0xC2, 0x12, "\xA4B #b",		/* SBCB */
	0xC3, 0x33, "\xAAD #w",		/* ADDD */
	0xC4, 0x12, "\x8DB #b",		/* ANDB */
	0xC5, 0x12, "\xA5B #b",		/* BITB */
	0xC6, 0x12, "\xB0 #b",		/* LDB */
	0xC8, 0x12, "\xA7B #b",		/* EORB */
	0xC9, 0x12, "\xA8B #b",		/* ADCB */
	0xCA, 0x12, "\xB1 #b",		/* ORB */
	0xCB, 0x12, "\xAAB #b",		/* ADDB */
	0xCC, 0x33, "\xB2 #w",		/* LDD */
	0xCE, 0x33, "\xB3 #w",		/* LDU */

	0xD0, 0x22, "\xA2B <b",		/* SUBB */
	0xD1, 0x22, "\xA3B <b",		/* CMPB */
	0xD2, 0x22, "\xA4B <b",		/* SBCB */
	0xD3, 0x22, "\xAAD <b",		/* ADDD */
	0xD4, 0x22, "\x8DB <b",		/* ANDB */
	0xD5, 0x22, "\xA5B <b",		/* BITB */
	0xD6, 0x22, "\xB0 <b",		/* LDB */
	0xD7, 0x22, "\xB4 <b",		/* STB */
	0xD8, 0x22, "\xA7B <b",		/* EORB */
	0xD9, 0x22, "\xA8B <b",		/* ADCB */
	0xDA, 0x22, "\xB1 <b",		/* ORB */
	0xDB, 0x22, "\xAAB <b",		/* ADDB */
	0xDC, 0x22, "\xB2 <b",		/* LDD */
	0xDD, 0x22, "\xB5 <b",		/* STD */
	0xDE, 0x22, "\xB3 <b",		/* LDU */
	0xDF, 0x22, "\xB6 <b",		/* STU */

	0xE0, 0x62, "\xA2B x",		/* SUBB */
	0xE1, 0x62, "\xA3B x",		/* CMPB */
	0xE2, 0x62, "\xA4B x",		/* SBCB */
	0xE3, 0x62, "\xAAD x",		/* ADDD */
	0xE4, 0x62, "\x8DB x",		/* ANDB */
	0xE5, 0x62, "\xA5B x",		/* BITB */
	0xE6, 0x62, "\xB0 x",		/* LDB */
	0xE7, 0x62, "\xB4 x",		/* STB */
	0xE8, 0x62, "\xA7B x",		/* EORB */
	0xE9, 0x62, "\xA8B x",		/* ADCB */
	0xEA, 0x62, "\xB1 x",		/* ORB */
	0xEB, 0x62, "\xAAB x",		/* ADDB */
	0xEC, 0x62, "\xB2 x",		/* LDD */
	0xED, 0x62, "\xB5 x",		/* STD */
	0xEE, 0x62, "\xB3 x",		/* LDU */
	0xEF, 0x62, "\xB6 x",		/* STU */

	0xF0, 0x33, "\xA2B w",		/* SUBB */
	0xF1, 0x33, "\xA3B w",		/* CMPB */
	0xF2, 0x33, "\xA4B w",		/* SBCB */
	0xF3, 0x33, "\xAAD w",		/* ADDD */
	0xF4, 0x33, "\x8DB w",		/* ANDB */
	0xF5, 0x33, "\xA5B w",		/* BITB */
	0xF6, 0x33, "\xB0 w",		/* LDB */
	0xF7, 0x33, "\xB4 w",		/* STB */
	0xF8, 0x33, "\xA7B w",		/* EORB */
	0xF9, 0x33, "\xA8B w",		/* ADCB */
	0xFA, 0x33, "\xB1 w",		/* ORB */
	0xFB, 0x33, "\xAAB w",		/* ADDB */
	0xFC, 0x33, "\xB2 w",		/* LDD */
	0xFD, 0x33, "\xB5 w",		/* STD */
	0xFE, 0x33, "\xB3 w",		/* LDU */
	0xFF, 0x33, "\xB6 w",		/* STU */
	0xFF, 0, "\xB7 o" };		/* FCB */

unsigned char itable10[] = {
	0x21, 0x53, "\x8CN l",		/* LBRN */
	0x22, 0x53, "L\x90 l",		/* LBHI */
	0x23, 0x53, "\xB8S l",		/* LBLS */
	0x24, 0x53, "\xB9C l",		/* LBCC */
	0x25, 0x53, "\xB9S l",		/* LBCS */
	0x26, 0x53, "L\x94 l",		/* LBNE */
	0x27, 0x53, "L\x95 l",		/* LBEQ */
	0x28, 0x53, "\xBAC l",		/* LBVC */
	0x29, 0x53, "\xBAS l",		/* LBVS */
	0x2A, 0x53, "L\x98 l",		/* LBPL */
	0x2B, 0x53, "L\x99 l",		/* LBMI */
	0x2C, 0x53, "\xBBE l",		/* LBGE */
	0x2D, 0x53, "\xB8T l",		/* LBLT */
	0x2E, 0x53, "\xBBT l",		/* LBGT */
	0x2F, 0x53, "\xB8E l",		/* LBLE */

	0x3F, 0x01, "\xA12",		/* SWI2 */

	0x83, 0x33, "\xA3D #w",		/* CMPD */
	0x8C, 0x33, "\xA3Y #w",		/* CMPY */
	0x8E, 0x33, "\xBC #w",		/* LDY */

	0x93, 0x22, "\xA3D <b",		/* CMPD */
	0x9C, 0x22, "\xA3Y <b",		/* CMPY */
	0x9E, 0x22, "\xBC <b",		/* LDY */
	0x9F, 0x22, "\xBD <b",		/* STY */

	0xA3, 0x62, "\xA3D x",		/* CMPD */
	0xAC, 0x62, "\xA3Y x",		/* CMPY */
	0xAE, 0x62, "\xBC x",		/* LDY */
	0xAF, 0x62, "\xBD x",		/* STY */

	0xB3, 0x33, "\xA3D w",		/* CMPD */
	0xBC, 0x33, "\xA3Y w",		/* CMPY */
	0xBE, 0x33, "\xBC w",		/* LDY */
	0xBF, 0x33, "\xBD w",		/* STY */

	0xCE, 0x33, "\xBE #w",		/* LDS */
	0xDE, 0x22, "\xBE <b",		/* LDS */
	0xDF, 0x22, "\xBF <b",		/* STS */
	0xEE, 0x62, "\xBE x",		/* LDS */
	0xEF, 0x62, "\xBF x",		/* STS */
	0xFE, 0x33, "\xBE w",		/* LDS */
	0xFF, 0x33, "\xBF w",		/* STS */

	0xFF, 0, "\xB7 $10,o" };	/* FCB */

unsigned char itable11[] = {
	0x3F, 0x01, "\xA13",		/* SWI3 */
	0x83, 0x33, "\xA3U #w",		/* CMPU */
	0x8C, 0x33, "\xA3S #w",		/* CMPS */
	0x93, 0x22, "\xA3U <b",		/* CMPU */
	0x9C, 0x22, "\xA3S <b",		/* CMPS */
	0xA3, 0x62, "\xA3U x",		/* CMPU */
	0xAC, 0x62, "\xA3S x",		/* CMPS */
	0xB3, 0x33, "\xA3U w",		/* CMPU */
	0xBC, 0x33, "\xA3U w",		/* CMPU */
	0xFF, 0, "\xB7 $11,o" };	/* FCB */

unsigned char *treg[] = {
/*	 0   1   2   3   4   5    6   7   8   9   A    B    C   D   E   F */
	"D","X","Y","U","S","PC","?","?","A","B","CC","DP","?","?","?","?"
	};
unsigned char *preg[] = { "CC", "A", "B", "DP", "X", "Y", "U", "PC" };
unsigned char *qreg[] = { "CC", "A", "B", "DP", "X", "Y", "S", "PC" };
	
unsigned char
	memory[6];
unsigned
	Length;

#ifndef MemTst
	void MemTst(unsigned char x)
	{
		unsigned i, j;
		unsigned char *p, *p1;
		static unsigned char *mp;

		if(x) {
			p1 = &i - 16;
			free(mp = p = malloc(8));
			while(p < p1)
				*p++ = 0xA5;
			return; }

		i = 0;
		p1 = &x;
		printf("Mem: %04x-", mp);
	a1:	j = 0;
		while(mp < p1) {
			if(*mp++ != 0xA5) {
				if(j > i) {
					i = j;
					p = mp; }
				goto a1; }
			++j; }
		printf("%04x %u\n", p, i);
	}
#endif

/*
 * Get a word from memory in the proper "endian"
 * 6809 uses "big endian", 1st byte is HIGH, 2nd byte is LOW
 */
static unsigned get_word(index)
	unsigned index;
{
	return (memory[index] << 8) | memory[index+1];
}

/*
 * Disassemble one instruction
 */
unsigned disassemble(unsigned Address, char Show)
{
	unsigned i, type, shift_offset;
	unsigned char c, d, opcode, *ptr, **ptr1, *inst_ptr;
	int c1;

	shift_offset = i = 1;
	inst_ptr = Ssr ? (itable+8) : itable;
	switch(opcode = *memory = peek(SEG, Address)) {		/* Record opcode for later */
		case 0x10 : inst_ptr = itable10; goto doshift;
		case 0x11 : inst_ptr = itable11;
		doshift:
			opcode = memory[1] = peek(SEG, Address+1);
			shift_offset = i = 2; }

	/* Search instruction table for this opcode */
	while(*inst_ptr++ != opcode) {
		if(!*inst_ptr++) {		/* 0 length = FCB */
			Length = shift_offset;
			goto doinst; }
		while(*inst_ptr++); }
	Length = (((type = *inst_ptr++) & 0x0F) + shift_offset) - 1;

	/* Load instruction into memory */
	while(i < Length) {
		memory[i] = peek(SEG, Address+i);
		++i; }

	if((type & 0xF0) == 0x60) {
		switch(memory[shift_offset] & 0x9F) {
			case 0x89 :	// 16 bit offset
			case 0x99 :
			case 0x8D :	// 16 bit PCR
			case 0x9D :
			case 0x9F :	// extended indirect
				++Length;
				memory[i]  = peek(SEG, Address+i);
				++i;
			case 0x88 :	// 8 bit offset
			case 0x98 :
			case 0x8C :	// 8 bit PCR
			case 0x9C :
				++Length;
				memory[i]  = peek(SEG, Address+i);
				++i; } }

doinst:
	if(Show) {
		Xprintf("%04x", Address);
		for(i=0; i < 5; ++i) {
			if(i < Length)
				Xprintf(" %02x", memory[i]);
			else
				Xputs("   "); }

		i = 0;
		while(*inst_ptr) switch(c = *inst_ptr++) {
			case ' ' :		/* Separator */
				do
					Xputc(' ');
				while(++i < 8);
				break;
			case 'o' :		/* Output opcode byte */
				Xprintf("$%02x", opcode);
				break;
			case 'b' :		/* Byte operand */
				Xprintf("%02x", memory[shift_offset]);
				break;
			case 'w' :		/* Word operand */
				Xprintf("%04x", get_word(shift_offset));
				break;
			case 'r' :		/* 8 bit Relative address */
				c = memory[shift_offset];
				Xprintf("%04x", (Address+shift_offset+1)+(int)c);
				break;
			case 'l' :		/* 16 bit relative address */
				c1 = get_word(shift_offset);
				Xprintf("%04x", (Address+shift_offset+2)+c1);
				break;
			case 't' :		/* tfr/exg register set */
				c =  memory[shift_offset];
				Xprintf("%s,%s", treg[(unsigned)c >> 4], treg[c & 0x0F]);
				break;
			case 'p' :		/* PSHS/PULS */
				ptr1 = preg;
			dostack:
				c = memory[shift_offset];
				while(c) {
					if(c & 0x01) {
						Xputs(*ptr1);
						if(c & 0xFE)
							Xputc(','); }
					(unsigned char)c >>= 1;
					++ptr1; }
				break;
			case 'q' :		/* PSHU/PULU */
				ptr1 = qreg;
				goto dostack;
			case 'x' :		/* Indexed */
				d = "XYUS"[((c = memory[shift_offset]) >> 5) & 0x03];
				if(!(c & 0x80)) {	/* 5 bit offset */
					if((c &= 0x1F) & 0x10)
						c |= 0xE0;
					Xprintf("%d,%c", c, d);
					break; }
				if(c & 0x10)
					Xputc('[');
				switch(c & 0x0F) {
					case 0x04 :	// No offset
						Xprintf(",%c", d);	break;
					case 0x08 :	// 8 bit offset
						Xprintf("%d,%c", (char)memory[shift_offset+1], d); break;
					case 0x09 :	// 16 bit offset
						Xprintf("%d,%c", get_word(shift_offset+1), d); break;
					case 0x06 :	// Accumulator A offset
						Xprintf("A,%c", d);	break;
					case 0x05 :	// Accumulator B offset
						Xprintf("B,%c", d);	break;
					case 0x0B :	// Accumulator D offset
						Xprintf("D,%c", d);	break;
					case 0x00 :	// +
						Xprintf(",%c+", d);	break;
					case 0x01 :	// ++
						Xprintf(",%c++", d);	break;
					case 0x02 : // -
						Xprintf(",-%c", d);	break;
					case 0x03 :	// --
						Xprintf(",--%c", d);	break;
					case 0x0C :	// 8 bit offset from PC
						Xprintf("%d,PC", (char)memory[shift_offset+1]); break;
					case 0x0D :	// 16 bit offset from PC
						Xprintf("%d,PC", get_word(shift_offset+1) + Address + Length); break;
					case 0x0F :	// 16 bit indirect
						Xprintf("%04x", get_word(shift_offset+1)); }
				if(c & 0x10)
					Xputc(']');
				break;
			default:
				if(c & 0x80) {
					ptr = inames + (c & 0x7F)*3;
					Xputc(*ptr++);
					Xputc(*ptr++);
					i += 2;
					c = *ptr; }
				++i;
				Xputc(c); } }
	return Length;
}

/*
 * Display simulator status line
 */
register status(unsigned args)
{
	char buffer[81];
	_format_(nargs()*2 + &args, buffer);
	w_clwin(swin);
	w_puts(buffer, swin);
}

/*
 * Display error message
 */
register error(unsigned args)
{
	char buffer[81];
	_format_(nargs()*2 + &args, buffer);
	*swin = 0x4E;
	w_clwin(swin);
	w_puts(buffer, swin);
	Eflag = 255;
}

/*
 * Update console X/Y position
 */
void Cupdatexy() asm
{
		MOV		DX,WORD PTR DGRP:_Cx	; Get X&Y
		MOV		BH,DGRP:_W_PAGE			; Get display page
		MOV		AH,02h					; Set cursor function
		INT		10h						; Ask BIOS
}

/*
 * Close transient file
 */
int closefp(FILE *fp)
{
	if(*fp) {
		fclose(*fp);
		error("File closed");
		Cupdatexy();
		return *fp=0; }
	return 255;
}

/*
 * Display a title on a window
 */
void wintitle(char *ptr)
{
	char buffer[6];
	memcpy(buffer, W_OPEN, 6);
	W_OPEN[1] &= (~WBOX3 >> 8);
	--W_OPEN[3];
	wgotoxy(0,0);
	wprintf(" %s ", ptr);
	memcpy(W_OPEN, buffer, 6);
}

/*
 * Update disassembly window
 */
void updis(unsigned H)
{
	unsigned i, a;
	W_OPEN = dwin;
	a = Disaddr;
	for(i=0; i < DEBUG_LINES; ++i) {
		wgotoxy(0, i);
		*W_OPEN = (a == H) ? 0x71 : 0x17;
		wcleol();
		a += disassemble(Dislist[i] = a, 255); }
	*W_OPEN = 0x17;
	W_OPEN = xwin;
}

/*
 * Scroll disassembly window back
 */
void backdis(unsigned n)
{
	unsigned i, a;
	a = Disaddr - (DEBUG_LINES*5);
	for(i=0; i < (sizeof(Dislist)/2); ++i) {
		a += disassemble(Dislist[i] = a, 0);
		if(a >= Disaddr)
			break; }
	Disaddr = Dislist[i-n];
}

/*
 * Insure that an address will appear in the disassembly window
 */
void showdis(unsigned addr)
{
	unsigned i;
	for(i=0; i < DEBUG_LINES; ++i) {
		if(Dislist[i] == addr)
			return; }
	Disaddr = addr;
}

/*
 * Update register display window
 */
void upreg(void)
{
	unsigned i;
	W_OPEN = rwin;
	wgotoxy(0, 0);
	wprintf("D:%02x%02x\n", A, B);
	wprintf("X:%04x\nY:%04x\nU:%04x\nS:%04x\nP:%04x", X, Y, U, S, P);
	wprintf("\nDP:%02x  \nCC:%02x  \n ",DP, CC);
	for(i=0; i < 8; ++i) {
		if(i == 4)
			wputs(" \n ");
		wputc( (((0x80 >> i)&CC) ? "EFHINZVC": "efhinzvc")[i]); }
	wputc(' ');
	W_OPEN = xwin;
}

/*
 * Update memory display window
 */
void upmem()
{
	unsigned a, i, j;
	unsigned char c;
	W_OPEN = mwin;
	a = Memaddr;
	for(i=0; i < DEBUG_LINES; ++i) {
		wgotoxy(0, i);
		wprintf("%04x",  a);
		for(j=0; j < 8; ++j)
			wprintf(" %02x", peek(SEG, a++));
		wputc(' ');
		a -= 8;
		for(j=0; j < 8; ++j) {
			c = peek(SEG, a++);
			wputc( ((c >= ' ') && (c < 0x7F)) ? c : 0xF9); } }
	W_OPEN = xwin;
}

/*
 * Help commnd - Implements "hypertext" screens
 */
void help(unsigned section)
{
	int c, lx, ly, ox, oy;
	unsigned i, j, size, ncount;
	unsigned char buffer[65], *ptr;
	unsigned char xs[LINKS], ys[LINKS], names[LINKS][25], link[LINKS];
	FILE *fp;

	ptr = ARGV[i=j=0];
	while(*ptr) {
		if((buffer[i++] = *ptr++) == '.')
			j = i; }
	strcpy(buffer+j, "HLP");
	if(!(fp = fopen(buffer, "rb"))) {
		beep(1000, 250);
		return; }

	/* Locate the section in question */
lookup:
	size = getc(fp);
	size |= getc(fp) << 8;
	if(section--) {
		fseek(fp, 0, size, 1);
		goto lookup; }

	/* Draw the screen */
	wopen(5, 4, 70, 17, WSAVE|WCOPEN|WBOX1|HTEXT);
	wcursor_off();
	i = ncount = ptr = 0;
	while(buffer[i++] = getc(fp));
	wintitle(buffer);
	wgotoxy(0, 0);
	while(i++ < size) switch(c = getc(fp)) {
		case LINK_S :	/* Start link */
			ptr = names[ncount];
			xs[ncount] = W_OPEN->WINcurx;
			ys[ncount] = W_OPEN->WINcury;
			*W_OPEN = HLINK;
			break;
		case LINK_E :	/* End link */
			link[ncount++] = getc(fp);
			*ptr = ptr = 0;
			*W_OPEN = HTEXT;
#ifndef USER_CODE
			++i;
			break;
#else
		extra:
			++i;
			break;
		case USER_CODE:	/* User control code */
			user_help(getc(fp));
			goto extra;
#endif
		case '\t' :		/* Horizontal TAB */
			do
				wputc(' ');
			while(W_OPEN->WINcurx % TAB_SIZE);
			break;
		case 0 :
			c = '\n';
		default:
			wputc(c);
			if(ptr)
				*ptr++ = c; }

	/* Allow user to select field & link to new screen */
	i = section = 0;
	for(;;) {
		wgotoxy(xs[section], ys[section]);
		*W_OPEN = HLINK;
		wputs(names[section]);
		wgotoxy(xs[i], ys[i]);
		*W_OPEN = HLINKH;
		wputs(names[section = i]);
		*W_OPEN = HTEXT;
		switch(c = wgetc()) {		/* User keystroke */
			case _KLA :				/* Left arrow - previous field */
				i = (i ? i : ncount) - 1;
				break;
			case _KRA :				/* Right arrow - next field */
				i = (i + 1) % ncount;
				break;
			case _KUA : ox = oy = -1000; goto dofind;
			case _KDA : ox = oy = 1000;
			dofind:
				size = i;
				for(j = 0; j < ncount; ++j) {
					lx = (int)xs[j] - (int)xs[i];
					ly = (int)ys[j] - (int)ys[i];
					if(c == _KUA) {
						if((ly >= 0) || (ly < oy)) continue; }
					else {
						if((ly <= 0) || (ly > oy))	continue; }
					if(abs(lx) > abs(ox)) continue;
					size = j;
					ox = lx;
					oy = ly; }
				i = size;
				break;
			case '\n' :				/* Enter - chain to menu */
				rewind(fp);
				section = link[i];
				wclose();
				goto lookup;
			case 0x1B:				/* Escape exit */
				wclose();
				fclose(fp);
				return; } }
}

/*
 * Hexidecimal input function
 */
int gethex(unsigned char in, unsigned *dest, unsigned length)
{
	unsigned c, v, l;
	unsigned char i, d;
	v = l = 0;
	i = in;
	for(;;) {
		if(!(c = i)) {
			wupdatexy();
			c = wgetc(); }
		if((length == 4) && !l) switch(toupper(c)) {
			case 'X' : v=X;	goto savereg;
			case 'Y' : v=Y;	goto savereg;
			case 'U' : v=U;	goto savereg;
			case 'S' : v=S; goto savereg;
			case 'P' : v=P;
			savereg: *dest = v; return 255; }
		d = c;
		i = 0;
		if((c >= '0') && (c <= '9'))
			c -= '0';
		else if((c >= 'A') && (c <= 'F'))
			c -= ('A' - 10);
		else if((c >= 'a') && (c <= 'f'))
			c -= ('a' - 10);
		else switch(c) {
			case '?' :
			case _K1 : help(0); continue;
			case 0x1B:
			case _K10: return 0;
			case _KBS:
				if(l) {
					--l;
					v >>= 4;
					wputc('\b');
					continue; }
			default:	
				beep(1000, 250);
				if(in && !l)
					return 0;
				continue; }
		wputc(d);
		v = (v << 4) | c;
		if(++l >= length) {
			*dest = v;
			return 255; } }
}

/*
 * Update drive indicators
 */
void clrdrive(void) asm
{
		MOV		ES,DGRP:_W_BASE		; Get video segment
		MOV		BX,3984				; Offset to drives
		MOV		AX,1004h			; "Off" indicators
		MOV		ES:[BX],AX			; Clear Drive1
		MOV		ES:4[BX],AX			; Clear Drive2
		MOV		ES:8[BX],AX			; Clear Drive3
		MOV		ES:12[BX],AX		; Clear Drive4
}
void lightdrive(void) asm
{
		MOV		ES,DGRP:_W_BASE		; Get video segment
		XOR		AX,AX				; Drive 0
		MOV		BX,3984				; Offset to drives
ld1:	MOV		CX,1004h			; OFF indicator
		CMP		AX,DGRP:_Ddrive		; Is this it?
		JNZ		ld2					; No, its OK
		MOV		CH,1Ch				; Reset to ON
ld2:	MOV		ES:[BX],CX			; Write it
		ADD		BX,4				; Next drive
		INC		AX					; Do them all
		CMP		AX,4				; Finished?
		JC		ld1					; Do them all
		MOV		AX,20				; Timeout
		MOV		DGRP:_Dactive,AX	; Set timer
}

/*
 * Compute FDC cylinder/head/sector
 */
HANDLE FDCchs()
{
	unsigned l[2], ts, h;
	static unsigned L512[] = { 512, 0 };
	if(h = Dhandle[Ddrive]) {
		ts = Dcylinder * Dnhead * Dnsector;		// Cylinder offset
		ts += Dhead * Dnsector;					// Head offset
		longset(l, ts + Dsector);				// Sector offset
		longmul(l, L512);
//wprintf("\n H:%u S:%u %u/%u/%u %04x:%04x", Dnhead, Dnsector, Dcylinder, Dhead, Dsector, l[1], l[0]);
		lseek(h, l[1], l[0], 0);
		return h; }
	Dop = 0xF3;
	return h;
}

/*
 * Fake FDC delay
 */
void FDCdelay(void)
{
	unsigned i;
	if(Ddelay) {
		do {
			i = TICK; }
		while(i == Dtick);
		Dtick = i; }
}

/*
 * Perform an FDC function request
 */
void FDCfunc(unsigned f)
{
	unsigned dc, dh, ds, h;

//wprintf("\nF:%02x", f);
//wprintf(" c%u h%u s%u H%u S%u", Dcylinder, Dhead, Dsector, Dnhead, Dnsector);
//wprintf(" D%u[%04x]", Ddrive, Dhandle[Ddrive]);
	lightdrive();
	FDCdelay();
	if(!(h = Dhandle[Ddrive])) {
		Dop = 0xF3;
		return; }
	switch(f) {
	case 0x01 :	// Read
		if(h = FDCchs()) {
			read(Dbuffer, 512, h);
			Dop = 1;
			Dptr = 0; }
		return;
	case 0x02 :	// Write
		Dptr = 0;
		Dop = Dwrite[Ddrive] ? 2 : 0xF4;
		return;
	case 0x03 :	// Format
		if(!Dwrite[Ddrive]) {
			Dop = 0xF4;
			return; }
		memset(Dbuffer, 0, sizeof(Dbuffer));
		lrewind(h);
		for(dc = 0; dc < Dcylinder; ++dc) {
			clrdrive();
			for(dh = 0; dh < Dhead; ++dh) {
				for(ds = 0; ds < Dsector; ++ds) {
					if(ds & 1) { FDCdelay(); lightdrive(); }
					write(Dbuffer, 512, h); } }
			clrdrive(); }
		Dop = 0;
		return;
	case 0x04 :	// Reset drive
		lrewind(h);
		Dop = 0;
		return;
	case 0x05 :	// Initialize
		clrdrive();
		return; }
	Dop = 0xF0;
}

/*
 * Refresh console from screen buffer
 */
void Crefresh() asm
{
		MOV		SI,OFFSET DGRP:_Cdata	; Point to buffer
		XOR		DI,DI					; Beginning of video buffer
		MOV		ES,DGRP:_W_BASE			; Get video base segment
		MOV		AL,DGRP:_Cmode			; Get console mode
		AND		AL,AL					; Special mode
		JNZ		Cref1					; Yes, its ok
		ADD		SI,(DEBUG_LINES*80)		; Adjust SI
		MOV		DI,(DEBUG_LINES*160)	; Adjust DI
Cref1:	MOV		AH,07					; Assume normal video
		MOV		AL,[SI]					; Get byte from video
		AND		AL,AL					; Test character
		JNS		Cref2					; Its OK
		AND		AL,7Fh					; Back to ASCII
		MOV		AH,70h					; Inverse video		
Cref2:	MOV		ES:[DI],AX				; Write pair
		INC		SI						; Next source
		ADD		DI,2					; Next dest
		CMP		DI,(TTY_LINES*160)		; All done
		JC		Cref1					; No, keep going
;
		MOV		DX,WORD PTR DGRP:_Cx	; Get X&Y
		MOV		BH,DGRP:_W_PAGE			; Get display page
		MOV		AH,02h					; Set cursor function
		INT		10h						; Ask BIOS
}

/*
 * Scroll console down
 */
void Cscroll()
{
	unsigned i;
	if(Cdelay)
		delay(100);
	memcpy(Cdata, Cdata+80, sizeof(Cdata)-80);
	memset(&Cdata[TTY_LINES-1][0], ' ', 80);
	if(Cmode)
		copy_seg(W_BASE, 0, W_BASE, 160, (TTY_LINES-1)*160);
	else
		copy_seg(W_BASE, 160*DEBUG_LINES, W_BASE, 160*(DEBUG_LINES+1),
		((TTY_LINES-1)-DEBUG_LINES)*160);

	for(i=((TTY_LINES-1)*160); i < (TTY_LINES*160); i += 2)
		poke(W_BASE, i, ' ');
}

/*
 * Scroll console up
 */
void Cupscroll()
{
	unsigned i;
	i = (TTY_LINES-1);
	do {
		memcpy(Cdata[i], Cdata[i-1], 80); }
	while(--i);
	memset(Cdata, ' ', 80);
	Crefresh();
}

/*
 * Write a character to the console
 */
void Cout(unsigned c)
{
	if(Cstate) {
		if(Cstate == 255) {
			if((c >= ' ') && (c < (' '+TTY_LINES))) {
				Cstate = c;
				return; } }
		else {
			if((c >= ' ') && (c < (' '+80))) {
				Cy = Cstate - ' ';
				Cx = c - ' ';
				Cstate = 0;
				goto upxy; } }
		Cstate = 0; }
	switch(c) {
	case '\r' :			// Carriage return
		Cx = 0;
		goto upxy;
	case '\n' :			// Line feed
		if(Cy < (TTY_LINES-1)) {
			++Cy;
			goto upxy; }
	case 'D'-0x40:		// Scroll
		Cscroll();
		goto upxy;
	case 'P'-0x40:		// Cursor position
		Cstate = 255;
		return;
	case 'I'-0x40 :		// Tab
		do {
			if(Cx >= 79)
				return;
			Cout(' '); }
		while(Cx & 7);
		return;
	case 'E'-0x40 :		// Upward scroll
		Cupscroll();
		return;
	case 'H'-0x40 :		// Backspace
		if(Cx)
			--Cx;
		goto upxy;
	case 'U'-0x40 :		// Forward
		goto advance;
	case 'Z'-0x40 :		// Up
		if(Cy)
			--Cy;
		goto upxy;
	case 'N'-0x40 :		// Special effect
		Ceffect = 0x80;
		return;
	case 'O'-0x40 :		// Normal effect
		Ceffect = 0x00;
		return;
	case 'W'-0x40 :		// Clear EOS
		c = (Cy*80) + Cx;
		memset(Cdata+c, ' ', sizeof(Cdata)-c);
		Crefresh();
		return;
	case 'V'-0x40:		// Clear EOL
		c = (Cy*80) + Cx;
		memset(Cdata+c, ' ', 80-Cx);
		Crefresh();
		return;
	case 'X'-0x40 :		// Clear screen
		memset(Cdata, ' ', sizeof(Cdata));
		Crefresh();
	case 'Y'-0x40:		// Home cursor
		Cx = Cy = 0;
		goto upxy;
	case 'G'-0x40:		// Ring bell
		beep(1000, 250);
		return; }

	if(c < ' ')
		return;
	if(Ceffect)
		c |= 0x80;
	Cdata[Cy][Cx] = c;
	if(Cmode || (Cy >= (DEBUG_LINES))) {
		if(c & 0x80)
			c = (c & 0x7F) | 0x7000;
		else
			c = (c & 0x7F) | 0x0700;
		pokew(W_BASE, (Cy*160)+(Cx*2), c); }
advance:
	if(++Cx > 79) {
		Cx = 0;
		if(Cy >= (TTY_LINES-1))
			Cscroll();
		else
			++Cy; }
upxy:
	Cupdatexy();
}

/*
 * Read 6809 I/O memory location
 */
int RD6809(unsigned a)
{
	unsigned t;
	// wprintf("\nR:%04x", a);
	if(a >= 0x800)
		return peek(SEG, a);
	switch(a) {
	case 0 :		// 6551-A data
	case 4 :		// 6551-B data
		if(Swapio) goto kdata;
		Upending = 0;
		return Udata;
	case 1 :		// 6551-A status
	case 5 :		// 6551-B status
		if(Swapio)
			return Kbc ? 0x18 : 0x10;
		if(Upending)
			return 0x18;
		if(Uport) {
			if((t = Ctestc()) != -1) {
				Udata = t;
				Upending = 255;
				return 0x18; } }
		return 0x10;
	case 2 :		// 6551-A Command
	case 3 :		// 6551-A Control
	case 6 :		// 6551-B Command
	case 7 :		// 6551-B Control
		return peek(SEG, a);
	case 0xFF :	// Keyboard
		if(INfp) {
			if((t = getc(INfp)) != -1) {
				if(t == '\n') t = '\r';
				return t; }
			closefp(&INfp); }
		if(Swapio) {
			if((t = Ctestc()) != -1)
				return t;
			return 0; }
kdata:	t = Kbc;
		Kbc = 0;
		return t;
	case 0xF0 :	return Dop;			// FDC status
	case 0xF1 :
		if(Dop != 1)
			return Dop = 0xFD;
		t = Dbuffer[Dptr++];
		if(Dptr >= 512)
			Dop = 0;
		return t;
	}
	return 0;
}

/*
 * Write 6809 I/O memory location
 */
void WR6809(unsigned a, unsigned char v)
{
	unsigned i;
	static unsigned btable[] = {
	1,	2304,	1536,	_110,	856,	768,	_300,	_600,
	_1200,	64,	_2400,	32,		_4800,	16,		_9600,	_19200 };
	// wprintf("\nW:%04x\n=%02x", a, v);
	switch(a) {
	case 0 :	// 6551-A Data
	case 4 :	// 6551-B Data 
		if(Swapio) goto cout;
uout:	if(Uport)
			Cputc(v);
		return;
	case 2 :	// 6551-B command
	case 5 :	// 6551-B command
		Uctrl1 &= ~PAR_SPACE;			// Remove parity
		Uctrl2 &= ~(SET_RTS|SET_DTR);	// Remove signals
		Uctrl1 |= (v >> 2) & 0x38;		// Insert parity
		switch(v & 0x0C) {
		case 0x0C : Uctrl1 |= SEND_BREAK;
		default: Uctrl2 |= SET_RTS;
		case 0x00: }
		if(v & 0x01)
			Uctrl2 |= SET_DTR;
		if(Uport)
			Copen(Uport, btable[Ubaud], Uctrl1, Uctrl2|OUTPUT_2);
		return;
	case 3 :	// 6551-A control
	case 7 :	// 6551-B control
		Uctrl1 &= ~(STOP_2|DATA_8);
		if(v & 0x80)
			Uctrl1 |= STOP_2;
		Uctrl1 |= (~v >> 5) & 0x03;
		Ubaud = v & 15;
		if(Uport)
			Copen(Uport, btable[Ubaud], Uctrl1, Uctrl2|OUTPUT_2);
		return;		
	case 0xFF :	// Console
		if(Swapio) goto uout;
cout:	Cout(v);
		if(OUTfp)
			putc(v, OUTfp);
		return;
	case 0xF0 :	FDCfunc(v);		return;		// FDC function
	case 0xF1 :								// FDC data
		if(Dop != 2) {
			Dop = 0xF0;
			return; }
		if(!Dwrite[Ddrive]) {
			Dop = 0xF4;
			return; }
		Dbuffer[Dptr++] = v;
		if(Dptr >= 512) {
//for(Dop = 0; Dop < 16; ++Dop)
//	wprintf(" %02x", Dbuffer[Dop]);
			Dop = 0;
			if(i = FDCchs())
				write(Dbuffer, 512, i); }
		return;
	case 0xF2 :	Ddrive = v & 3;	return;		// Drive
	case 0xF3 :	Dcylinder = v;	return;		// Cylinder
	case 0xF4 :	Dhead = v;		return;		// Head
	case 0xF5 :	Dsector = v;	return;		// Sector
	case 0xF6 :	Dnhead = v;		return;		// #heads
	case 0xF7 :	Dnsector = v;	return;		// #sectors
	}
}

/*
 * Interactive memory editor
 */
void edit_memory(void)
{
	unsigned xc, xh, y, c, s, v;
	static int offset;
	static char mode;

	status("Memory Editor: F1=help  F2=Mode  F3=Address  Home=Align256  End=Align64 F10=Exit");
redraw:
	upmem();
	W_OPEN = mwin;
	wcursor_line();
	for(;;) {
		if(offset < 0) {
			do {
				offset += s;
				Memaddr -= s; }
			while(offset < 0);
			goto redraw; }
		if(offset > ((DEBUG_LINES*8)-1)) {
			do {
				offset -= s;
				Memaddr += s; }
			while(offset > ((DEBUG_LINES*8)-1));
			goto redraw; }
		y = offset >> 3;
		xc = offset & 7;
		xh = (xc * 3) + 5;
		xc += 29;
		wgotoxy(mode ? xc : xh, y);
		s = 1;
		switch(c = wgetc()) {
		case _KRA: ++offset;	s=1;	continue;
		case _KLA: --offset;	s=1;	continue;
		case _KUA: offset -= (s=8); 	continue;
		case _KDA: offset += (s=8);		continue;
		case _KPU: Memaddr -= 80;		goto redraw;
		case _KPD: Memaddr += 80;		goto redraw;
		case '?' : if(mode) break;
		case _K1 : help(HEDITM);	continue;
		case _K2 : mode = mode ? 0 : 255; continue;
		case _K3 :
			wgotoxy(0, 0);
			if(gethex(0, &Memaddr, 4))
				offset = 0;
			goto redraw;
		case _KHO: Memaddr &= 0xFF00;	goto redraw;
		case _KEN: Memaddr &= 0xFFC0;	goto redraw;
		case 0x1B: if(mode) break;
		case _K10:	W_OPEN = xwin;	return; }
		if(c & 0xFF00) continue;
		if(!mode) {
			if(!gethex(c, &v, 2))
				goto redraw;
			c = v; }
		poke(SEG, offset++ + Memaddr, c);
		wgotoxy(xh, y);
		wprintf("%02x", c);
		wgotoxy(xc, y);
		wputc( ((c < ' ') || (c > 0x7E)) ? '.' : c); }
}

void edit_register(void)
{
	unsigned x, y, v;
	unsigned char *bp;
	unsigned *wp;
redraw:
	upreg();
	W_OPEN = rwin;
	for(;;) {
		status("Select register: A B X Y U Sp Pc Cc Dp W ESC=exit? ");
		wcursor_line();
		switch(toupper(w_getc(swin))) {
		case 'A' : y=0; x=2; bp=&A; goto in1;
		case 'B' : y=0; x=4; bp=&B; goto in1;
		case 'W' : y=0; x=2; wp=&B; goto in2;
		case 'X' : y=1; x=2; wp=&X; goto in2;
		case 'Y' : y=2; x=2; wp=&Y; goto in2;
		case 'U' : y=3; x=2; wp=&U; goto in2;
		case 'S' : y=4; x=2; wp=&S; goto in2;
		case 'P' : y=5; x=2; wp=&P;
		in2: status("Enter value: ESC=cancel");
			wgotoxy(x, y);
			if(gethex(0, &v, 4))
				*wp = v;
			goto redraw;
		case 'D' : y=6; x=3; bp=&DP; goto in1;
		case 'C' : y=7; x=3; bp=&CC;
		in1: status("Enter value: ESC=cancel");
			wgotoxy(x, y);
			if(gethex(0, &v, 2))
				*bp = v;
			goto redraw;
		case '?' :
		case _K1 : help(HEDITR);	continue;
		case 0x1B :
		case _K10 : W_OPEN = xwin;	return; }
		beep(1000, 250); }
}

/*
 * Copy filename to Dbuffer & append extension
 */
void filename(unsigned char *file, unsigned char *extension, unsigned char *n)
{
	unsigned char *d, *p, *e;
	d = Dbuffer;
	e = 0;
	p = file;
	while(*d = toupper(*file++)) {
		switch(*d++) {
		case '.' : e = file-1; continue;
		case '\\':
		case ':' : p = file; e = 0; } }
	if(e)
		*e = 0;
	else
		strcpy(d, extension);
	if(n)
		strcpy(n, p);
}

/*
 * Load a byte from the hex file
 */
unsigned load_byte(void)
{
	unsigned char c, v, ct;
	ct = 2;
	do {
		c = *ptr++;
		if((c >= '0') && (c <= '9'))
			c -= '0';
		else if((c >= 'A') && (c <= 'F'))
			c -= ('A'-10);
		else if((c >= 'a') && (c <= 'f'))
			c -= ('a'-10);
		else {
			printf("%u: Bad hex digit '%c'[%02x]\n", P, c, c);
			exit(-1); }
		v = (v << 4) | c; }
	while(--ct);
	return v;
}

/*
 * Read file into memory image
 */
void load_file(void)
{
	unsigned a, b, chksum, addr, length, count;
	unsigned char buffer[80];
	FILE *fp;

	filename(U, ".HEX", Y=0);
	fp = fopen(Dbuffer, "rvq");
	X = -1;

	for(P = 0; fgets(ptr = buffer, 80, fp);) {
		++P;
		again: switch(*ptr++) {
			case 'S' :	/* Motorola HEX format */
				if(*ptr == '9') goto quitload;
				if(*ptr++ != '1') continue;
				length = count = (chksum = load_byte()) - 3;
				chksum += a = load_byte();
				chksum += b = load_byte();
				addr = (a << 8) + b;
				if(addr < X)
					X = addr;
				while(count--) {
					chksum += a = load_byte();
					poke(SEG, addr++, a); }
				if((addr-1) > Y)
					Y = (addr-1);
				if((255 & ~chksum) != load_byte())
					goto quitbad;
				break;
			case ':' :		/* Intel HEX format */
				if(!(length = count = load_byte())) goto quitload;
				chksum = (a = load_byte()) + length;
				chksum += b = load_byte();
				addr = (a << 8) + b;
				chksum += load_byte();
				if(addr < X)
					X = addr;
				while(count--) {
					chksum += a = load_byte();
					poke(SEG, addr++, a); }
				if((addr-1) > Y)
					Y = (addr-1);
				if((255 & -chksum) != load_byte())
					goto quitbad;
				break;
			case ' ' :		/* Space */
			case '\t' :		/* Tab */
				goto again;
			case 0 :		/* Null line */
				continue;
			default:
				printf("%u: Invalid record format\n", P);
				goto quit; }
		}

quitbad:
	printf("%u: Bad record checksum\n", P);
quit:
	fclose(fp);
	exit(-1);
quitload:
	printf("%u bytes loaded at %04x\n", (Y-X)+1, X);
}

/*
 * Parse command line parameter
 */
void cmdparm(unsigned char *p)
{
	unsigned char w;
	static char Dindex;

	while(isspace(*p))
		++p;
	w = 0;
	switch((toupper(*p++) << 8) | toupper(*p++)) {
	case '?' << 8:
	case '-?' :
	case '/?' : help(HCMD); exit(0);
	case '-D' :										// No drive delay
	case '/D' : Ddelay = 0;				return;
	case '-S' :
	case '/S' : STEP = 55;				return;		// Start stopped
	case '-U' :
	case '/U' : Swapio = 255;			return;		// Swap Uart1/console
	case 'B=' : BREAK = atox(p);		return;		// Breakpoint
	case 'C=' : Uport = atoi(p);					// COM port
		if((Uport > 4) || !Uport)
			abort("C= must be 1-3");
		return;
	case 'L=' : U = p;					return;		// Load file
	case 'S=' : Romfile = p;			return;		// ROM file
	case 'M=' : Memaddr = atox(p);		return;		// Memory address
	case 'F=' : DP = atox(p);			return;		// Fill value
	case 'W=' : w = 255;							// Writable disk
	case 'R=' :										// Read-only disk
		if(Dindex > 4) {
			printf("Too many R=/W= drives (Max 4)");
			break; }
		filename(p, ".DDI", Dname[Dindex]);
		if(Dhandle[Dindex] = open(Dbuffer, w ? (F_READ|F_WRITE) : F_READ)) {
			Dwrite[Dindex++] = w;
			return; }
		printf("Unable to access: %s", Dbuffer);
		break;
	default:
		printf("Unknown option: '%s'", p-2); }
	putc('\n', stdout);
	exit(-1);
}

/*
 * Reset virtual CPU
 */
void reset(void)
{
	CC = DP = A = B = X = Y = U = 0;
	S = 0xE000;
	P = peekw(SEG, 0xFFFE);
	P = (P >> 8) | (P << 8);
}

/*
 * Display reason for CPU abort
 */
void showcpu(unsigned c)
{
	switch(c) {
		case 1 :	error("Invalid opcode");			return;
		case 2 :	error("Bad addressing mode");		return;
		case 3 :	error("Bad TFR/EXG register");		return;
		case 254:	error("Stack Break");				return;
		case 255:	error("Breakpoint");				return;
		default:	error("Stop: %u", c);
		case 0 :	return; }
}

unsigned klook[] = {			// Special key lookup table
	_UA, _LA, _RA, _DA, _PU, _PD,
	0x7800, 0x7900, 0x7A00, 0x7B00, 0x7C00,		// Alt 1-0
	0x7D00, 0x7E00, 0x7F00, 0x8000, 0x8100,
	0x6800, 0x6900, 0x6A00, 0x6B00, 0x6C00,		// Alt F1-F10
	0x6D00, 0x6E00, 0x6F00, 0x7000, 0x7100,
	_HO, _CLA, _CRA, _EN,	0x8400, 0x7600,
	0xF000, 0x8200, 0x8300, _INS,	_DEL,
	0x5E00, 0x5F00, 0x6000, 0x6100, 0x6200,
	0x6300, 0x6400, 0x6500, 0x6600, 0x6700,
	0 };
unsigned char ktran[] = {		// Special key translate table
	'Z'-0x40, 'H'-0x40, 'U'-0x40, 'J'-0x40, 'E'-0x40, 'D'-0x40,
	'1'|0x80, '2'|0x80, '3'|0x80, '4'|0x80, '5'|0x80,
	'6'|0x80, '7'|0x80, '8'|0x80, '9'|0x80, '0'|0x80,
	'!'|0x80, '@'|0x80, '#'|0x80, '$'|0x80, '%'|0x80,
	'^'|0x80, '&'|0x80, '*'|0x80, '('|0x80, ')'|0x80,
	'Z'+0x40, 'H'+0x40, 'U'+0x40, 'J'+0x40, 'E'+0x40, 'D'+0x40,
	'I'+0x40, '-'|0x80, '='|0x80, '_'|0x80, '+'|0x80,
	','|0x80, '.'|0x80, '/'|0x80, ';'|0x80, '\''|0x80,
	'<'|0x80, '>'|0x80, '?'|0x80, ':'|0x80, '"'|0x80
	};

/*
 * Update running status information
 */
void upstat()
{
	*swin = 0x17;
	status(" F1=Help F2=%s F3=Stop F4=Reset F5=Clear F6=%s F7=Import F8=Export",
		Cmode ? "Debug" : "Tty  ", Cdelay ? "Fast" : "Slow");
	clrdrive();
	Eflag = 0;
}

/*
 * Select .TXT file and open
 */
void openfile(FILE *fp, char *attr)
{
	if(closefp(fp)) {
		if(selectfile(".TXT")) {
			filename(Directory, ".TXT", 0);
			if(!(*fp = fopen(Dbuffer, attr)))
				error("Cannot access: %s", Dbuffer); } }
}

/*
 * Execute program at full speed
 */
void run(void)
{
	unsigned i, c;
restat:
	if(!Eflag)
		upstat();
reupdate:
	Cupdatexy();
	for(;;) {
		STEP = 1024;
		if(c = go()) {
			showcpu(c);
			return; }
		if(!Cmode) {
			showdis(P);
			updis(P);
			upreg();
			upmem();
			Cupdatexy(); }
		if((c = TICK) != Rtick) {
			Rtick = c;
			if(Dactive) {
				if(!--Dactive)
					clrdrive(); } }
		if(c = kbtst()) {
			if(Eflag)
				upstat();
			switch(c) {
			case _F1 : help(HRUN);	goto reupdate;
			case _F2 :
				Cmode = Cmode ? 0 : 255;
				Crefresh();
				goto restat;
			case _F3 :	return;
			case _F4 :	reset(); 		continue;
			case _F5 :  Cout('X'-0x40);	continue;
			case _F6 :	Cdelay = Cdelay ? 0 : 255;	goto restat;
			case _F7 :	openfile(&INfp, "r");		goto restat;
			case _F8 :	openfile(&OUTfp,"w");		goto restat;
			case '\b': c = 0x7F;		goto dochar;
//			case '\n': c = 0x8D;		goto dochar;
			case 0x7F: c = 0xFF;		goto dochar;
//			case '{' : c ='['|0x80;		goto dochar;
//			case '}' : c = ']'|0x80;	goto dochar;
			case 0x1C: c = '+'|0x80;	goto dochar;
			case 0x1D: c = 0x9B;		goto dochar;
			case 0x1F: c = '_'|0x80;	goto dochar;
			default:
				if(c & 0xFF00) {		// Special key
					for(i=0; klook[i]; ++i) {
						if(klook[i] == c) {
							c = ktran[i];
							goto dochar; } }
					if((c >= 0x6800) && (c <= 0x7000))
						c = (c >> 8) + 0x49;
					else {
// wprintf("\n[%04x]", c);
						goto reupdate; } }
		dochar:	Kbc = c;
				goto reupdate; } } }
}

/*
 * View traceback log of CPU execution
 */
void trace(void)
{
	unsigned t, i, j, s, st, da;
	*swin = 0x47;
home:
	st = j = TRCPTR / 2;
reshow:
	j &= 4095;
	if(i = (st-j) & 4095)
		da = s = TRCBUF[t = j];
	else {
		t = st;
		da = s = P; }
	status("TRACE %-4u: F1=Help  F2=SearchF  F3=SearchB  F4=NextF  F5=NextB  F10=Exit", i);
	showdis(da);
	updis(da);
	for(;;) switch(wgetc()) {
	case '?' :
	case _K1 : help(HTRACE);			continue;
	case _K2 :
		status("Search Forward?"); i = 1;
		goto dosearch;
	case _K3 :
		status("Search Backward?"); i = -1;
	dosearch:
		W_OPEN = swin;
		wcursor_line();
		i = gethex(0, &s, 4);
		wcursor_off();
		W_OPEN = xwin;
		if(i) {
dosr:			j = t;
			do {
				j = (j + i) & 4095;
				if(TRCBUF[j] == s)
					goto reshow; }
			while(j != st);
			beep(1000, 100); }
		j = t;
		goto reshow;
	case _K4 : i = 1; goto dosr;
	case _K5 : i = -1; goto dosr;
	case _KHO : j = st+1;				goto reshow;
	case _KEN : goto home;
	case _KUA : j = t - 1;				goto reshow;
	case _KDA : j = t + 1;				goto reshow;
	case _KPU : j = t - 100;			goto reshow;
	case _KPD :	j = t + 100;			goto reshow;
	case _K10 :
	case 0x1B : return;
	/* default: if(eflag) { eflag = 0; goto reshow; } */ }
}

/*
 * Select a file via window panel
 */
int selectfile(unsigned char *match)
{
	unsigned a, i, Nt, Pt;
	unsigned char name[13], *n[SFILES], pool[SFILES*9], *p;
	static unsigned char *Dip;

	// Find end of directory index
	if(Dip)
		*Dip = 0;
newdir:
	status("Select file: F1=Help  F2=ManualEntry");
	Dip = Directory; while(*Dip) ++Dip;
	if(*(Dip-1) == '\\')
		--Dip;
	*Dip = '\\';
	// Scan for all files & build name list
	strcpy(Dip+1, "*.*");
	Nt = Pt = 0;
	if(!find_first(Directory, DIRECTORY, p = name, &i, &i, &a, &i, &i)) do {
		if(Nt >= SFILES)
			break;
		if(a & DIRECTORY) {
			if(!strcmp(p, "."))
				continue;
			n[Nt++] = &pool[Pt];
			pool[Pt++] = 1;	// Insure sorts first!
			while(pool[Pt++] = *p++);
			continue; }
		while(*p) {
			if(!strcmp(p, match)) {
				*p = 0;	p = name;
				n[Nt++] = &pool[Pt];
				while(pool[Pt++] = *p++);
				break; }
			++p; } }
	while !find_next(p=name, &i, &i, &a, &i, &i);

	*Dip = 0;
	if(!Nt) goto manent;

	// Sort the name list
	for(i=0; i < Nt; ++i) {
		for(a=i+1; a < Nt; ++a) {
			if(strcmp(n[a], n[i]) < 0) {
				p = n[i];
				n[i] = n[a];
				n[a] = p; } } }
	// Convert directory pref
	for(i=0; i < Nt; ++i) {
		if(*(p = n[i]) != 1)
			break;
		while(p[1]) {
			*p = p[1];
			++p; }
		*p = '\\'; }

	i = ((Nt + 4) / 5) + 2;
	wopen(4, 12-(i/2), 71, i, WSAVE|WBOX1|WCOPEN|0x70);
	wintitle(Directory);
	wcursor_off();
	a = 0;
draw:
	for(i=0; i < Nt; ++i) {
		wgotoxy((i % 5) * 14, i / 5);
		*W_OPEN = (i == a) ? 0x07 : 0x70;
		wputs(n[i]); }
	*W_OPEN = 0x70;
	for(;;) switch(wgetc()) {
		case _KLA : if(a) { --a; goto draw; }
		case _KEN : a = Nt-1; goto draw;
		case _KRA : if(++a < Nt) goto draw;
		case _KHO : a = 0;	goto draw;
		case _KUA : if(a > 4) a -= 5; goto draw;
		case _KDA : if((a+5) < Nt) a += 5; goto draw;
		case '?' :
		case _K1  : help(HFILE); continue;
		case _K2  :
			wclose();
		manent:
			strcpy(pool, Directory); Pt=Dip;
			W_OPEN = swin; wclwin(); wputs("File?");
			*Dip = '\\'; *(Dip+1) = 0;
		manent1:
			switch(wgets(0x106, 0, p = Dip = Directory, sizeof(Directory)-1)) {
			case _K1: help(HENTER);
			default : goto manent1;
			case '\n' :
				while(i = *p++) switch(i) {
					case ':' :
					case '\\': Dip = p-1; }
				W_OPEN = xwin;
				if(*(p-2) == '\\') goto newdir;
				return 255;
			case 0x1B :
				if(Nt) {
					strcpy(Directory, pool);
					Dip = Pt;
					goto newdir; }
				return 0; }
		case '\n' :
			*Dip = '\\';
			p = n[a]; while(p[1]) ++p;
			if(*p == '\\') {
				*p = 0;
				if(strcmp(p = n[a], "..")) {	// Not UP
					strcpy(Dip+1, p);
					wclose();
					goto newdir; }
				// Go up one directory
				while(Dip > Directory) {
					if(*--Dip == '\\') {
						*Dip = 0;
						break; } }
				wclose();
				goto newdir; }
			strcpy(Dip+1, n[a]);
			wclose();
			return 255;
		case 0x1B :
		case _K10 :
			wclose();
			return 0; }
}

/*
 * Execute a sub-shell
 */
doshell()
{
	unsigned char comspec[65], tail[80];

	if(!getenv("COMSPEC", comspec)) {
		error("Cannot locate COMSPEC");
		return; }
	*tail = 0;
	wopen(0, 0, 80, 25, WSAVE|WCOPEN|NORMAL);
	wcursor_line();
	wupdatexy();
	exec(comspec, tail);
	wclose();
}

/*
 * Display main status line
 */
void showstat(void)
{
	*swin = 0x70;
	status("F1=Help F2=%s F3=EdMem F4=EdReg F5=Bkpt F6=Dis F7=Mount F8=Trc F9=DOS F10=Off", Cmode ? "Debug" : "Tty  ");
	Eflag = 0;
}

unsigned char Hello[] = { "\n\x18\
6809 System Simulator - Copyright 2004-2006 Dave Dunfield.  (F1=Help)\n\r" };

static unsigned char Cops[] = { 0x17, 0x8D, 0x9D, 0xAD, 0xBD, 0x3F };

unsigned WgetC(void)
{
	unsigned c;
	W_OPEN = swin;
	c = wgetc();
	W_OPEN = xwin;
	return toupper(c);
}

/*
 * Main program
 */
main(int argc, char *argv[])
{
	unsigned i, j, k, c;
	W_PAGE;

	MemTst(7);
	*Directory = get_drive() + 'A';
	Directory[1] = ':';
	Directory[2] = '\\';
	getdir(Directory+3);
	for(i=1; i < argc; ++i)
		cmdparm(argv[i]);

	SEG = alloc_seg(4096);

	i = 0;
	c = (DP << 8) | DP;
	do {
		pokew(SEG, i, c); }
	while(i -= 2);

	filename(Romfile, ".ROM", 0);
	fp = fopen(Dbuffer, "rvqb");
	i = 0xE000;
	while((c = getc(fp)) != -1)
		poke(SEG, i++, c);
	fclose(fp);

	if(U)
		load_file();

	if(Uport) {
		if(Copen(Uport, _9600, Uctrl1 = 0, Uctrl2 = OUTPUT_2))
			abort("Unable to open COM port.");
		Cflags |= TRANSPARENT; }

	dwin = wopen(0, 0, 37, DEBUG_LINES, WSAVE|WCOPEN|0x17);
	rwin = wopen(37, 0, 6, DEBUG_LINES, WSAVE|WCOPEN|0x57);
	mwin = wopen(43, 0,37, DEBUG_LINES, WSAVE|WCOPEN|0x67);
	swin = wopen(0, 24,80, 1, WSAVE|WCOPEN|0x17);
	xwin = wopen(i=0, DEBUG_LINES, 80, TTY_LINES-DEBUG_LINES, WSAVE|WCOPEN|WSCROLL|0x07);
	if(U) {
		status("%s: loaded from %x-%x, (%u blocks)", Dbuffer, X, Y, ((Y-X)+512)/512);
		Eflag  = 255; }
	do {
		Cout(Hello[i]); }
	while(Hello[++i]);
	delay(1000);

	reset();
	showdis(P);
	if(STEP == 55) goto restat;
dorun:
	run();
	SPSTOP = 0;
	showdis(P);
restat:
	if(!Eflag)
		showstat();
	for(;;) {
		if(!Cmode) {
			updis(P);
			upreg();
			upmem(); }
		Cupdatexy();
		while(!(c = wtstc()));
		if(Eflag)
			showstat();
		switch(toupper(c)) {
		case _CPU : Memaddr -= (DEBUG_LINES*8);	continue;
		case _CPD : Memaddr += (DEBUG_LINES*8);	continue;
		case _KPU : backdis(DEBUG_LINES-2);		continue;
		case _KPD : Disaddr = Dislist[DEBUG_LINES-1];	continue;
		case _KUA : backdis(0);					continue;
		case _KDA : Disaddr = Dislist[1];		continue;
		case 'R' :	reset();					continue;
		case 'S' :
			c = peek(SEG, P);
			for(i=0; i < sizeof(Cops); ++i) {
				if(Cops[i] == c) {
		dostep:		SPSTOP = S;
					goto dorun; } }
			switch(c) {
			case 0x10 :
			case 0x11 :
				if(peek(SEG, P+1) == 0x3F)
					goto dostep; }
		case 'T' : showcpu(step());	showdis(P); goto restat;
		case 'G' :
			goto dorun;
		case 'I':
			status("Firq,Irq,Nmi,Reset,Swi,swi2,swi3?");
a1:			switch(WgetC()) {
			default	:	goto a1;
			case 0x1B:
			case _K10:	goto restat;
			case 'I':	i = 0xFFF8;	ptr = "IRQ";	break;
			case 'F':
				i = 0xFFF6;
				ptr = "FIRQ";
				CC &= 0x7F;
				goto a2;
			case 'N':	i = 0xFFFC;	ptr = "NMI";	break;
			case 'R':
				reset();
				status("RESET");
				continue;
			case 'S':	i = 0xFFFA;	ptr = "SWI";	break;
			case '2':	i = 0xFFF4;	ptr = "SWI2";	break;
			case '3':	i = 0xFFF2;	ptr = "SWI3";	}
			CC |= 0x80;
a2:			doIRQ(i);
			status(ptr);
			Crefresh();
			continue;
//		case 'R' : reset();	status("RESET"); continue;
		case 'C' :
			Ssr = Ssr ? 0 : 255;
			goto restat;
		case '?' :
		case _K1 : help(HSTOP);	continue;
		case _K2 :
			Cmode = Cmode ? 0 : 255;
			Crefresh();
			goto restat;
		case _K3 : edit_memory(); Crefresh(); goto restat;
		case _K4 : edit_register(); Crefresh(); goto restat;
		case _K5:
			status("Break=%04x ?", BREAK);
			W_OPEN = swin;
			gethex(0, &BREAK, 4);
			W_OPEN = xwin;
			goto restat;
		case _K6 :
			updis();
			W_OPEN = dwin;
			wgotoxy(0, 0);
			gethex(0, &Disaddr, 4);
			W_OPEN = xwin;
			Crefresh();
		case ' ' :
			goto restat;
		case _K7 :
			w_clwin(swin);
			for(j=i=0; j < 4; ++j)
				w_printf(swin, "%c%c:%-12s", Dwrite[j] ? 'w' : 'r', j+'A', Dname[j]);
			if(!wmenu(34, 7, WSAVE|WCOPEN|WBOX1|0x70, Dmenu, &i)) {
				k = (i >> 1);
				if(Dhandle[k]) {
					close(Dhandle[k]);
					*Dname[k] = Dhandle[k] = Dwrite[k] = 0; }
				if(selectfile(".DDI")) {
					filename(Directory, ".DDI", Dname[k]);
					if(!(Dhandle[k] = open(Dbuffer, (i&1) ? (F_READ|F_WRITE) : F_READ))) {
						*Dname[k] = 0;
						error("Cannot access: %s", Dbuffer);
						continue; }
					if(i & 1) Dwrite[k] = 255; } }
			goto restat;
		case _K8 : trace();	Crefresh(); goto restat;
		case _K9 : doshell(); goto restat;
		case _K10:
			status("Exit D6809(Y/N)?");
			for(;;) {
				switch(WgetC()) {
				case 'N' :
				case 0x1B:
					goto restat;
				case 'Y' :
				case '\n' :
					closefp(&OUTfp);
					closefp(&INfp);
					wclose();
					wclose();
					wclose();
					wclose();
					wclose();
					Cclose();
					for(i=0; i < 4; ++i) {
						if(Dhandle[i])
							close(Dhandle[i]); }
					MemTst(0);
					return; } } } }
}
