;
; Virtual H8 CPU simulation
;
; This code implements the Intel 8080 CPU processor instruction set.
;
; Software interface is via global locations:
; _A                       8080 Accumulator
; _PSW                     8080 Processor Status Word (Flags)
; _C, _B, _D, _E, _H, _L   8080 General registers
; _SP                      8080 Stack Pointer
; _PC                      8080 Program Counter
; _SEG                     Virtual Memory segment
; _TRCPTR                  Instruction trace pointer
; _TRCBUF                  Instruction trace buffer
; _STEP                    Instruction step count
; _BREAK                   Breakpoint address
; _EI                      Interrupts enabled flag
; _PIRQ                    Pending interrupt
;
;
; Modified from my ALTAIR CPU implementation, instructions to update
; front panel indicators not present on the H8 have been commented out.
;
; ?COPY.TXT 2004-2008 Dave Dunfield
;
DGRP	GROUP	DSEG,BSEG
CSEG	SEGMENT	BYTE PUBLIC 'CODE'
	ASSUME	CS:CSEG, DS:DGRP, SS:DGRP
	PUBLIC	_multi_step, _far_step, _Mread, _setslow, _setfast, _FCALL
	EXTRN	_IN8080:near, _OUT8080:near, _UP8080:near
	EXTRN	_OF0:byte
;
; Instruction decode table
; This 256 entry jump table vectors to an appropriate handler for
; Each of the 256 possible opcodes which may be fetched
; 
ITABLE	LABEL	WORD
	DW	?NOP		; 00
	DW	?LXIB		; 01
	DW	?STAXB		; 02
	DW	?INXB		; 03
	DW	?INRB		; 04
	DW	?DCRB		; 05
	DW	?MVIB		; 06
	DW	?RLC		; 07
	DW	?ILNOP		; 08
	DW	?DADB		; 09
	DW	?LDAXB		; 0A
	DW	?DCXB		; 0B
	DW	?INRC		; 0C
	DW	?DCRC		; 0D
	DW	?MVIC		; 0E
	DW	?RRC		; 0F
	DW	?ILNOP		; 10
	DW	?LXID		; 11
	DW	?STAXD		; 12
	DW	?INXD		; 13
	DW	?INRD		; 14
	DW	?DCRD		; 15
	DW	?MVID		; 16
	DW	?RAL		; 17
	DW	?ILNOP		; 18
	DW	?DADD		; 19
	DW	?LDAXD		; 1A
	DW	?DCXD		; 1B
	DW	?INRE		; 1C
	DW	?DCRE		; 1D
	DW	?MVIE		; 1E
	DW	?RAR		; 1F
	DW	?ILNOP		; 20 (8085 RIM)
	DW	?LXIH		; 21
	DW	?SHLD		; 22
	DW	?INXH		; 23
	DW	?INRH		; 24
	DW	?DCRH		; 25
	DW	?MVIH		; 26
	DW	?DAA		; 27
	DW	?ILNOP		; 28
	DW	?DADH		; 29
	DW	?LHLD		; 2A
	DW	?DCXH		; 2B
	DW	?INRL		; 2C
	DW	?DCRL		; 2D
	DW	?MVIL		; 2E
	DW	?CMA		; 2F
	DW	?ILNOP		; 30 (8085 SIM)
	DW	?LXISP		; 31
	DW	?STA		; 32
	DW	?INXSP		; 33
	DW	?INRM		; 34
	DW	?DCRM		; 35
	DW	?MVIM		; 36
	DW	?STC		; 37
	DW	?ILNOP		; 38
	DW	?DADSP		; 39
	DW	?LDA		; 3A
	DW	?DCXSP		; 3B
	DW	?INRA		; 3C
	DW	?DCRA		; 3D
	DW	?MVIA		; 3E
	DW	?CMC		; 3F
	DW	?MOVBB		; 40
	DW	?MOVBC		; 41
	DW	?MOVBD		; 42
	DW	?MOVBE		; 43
	DW	?MOVBH		; 44
	DW	?MOVBL		; 45
	DW	?MOVBM		; 46
	DW	?MOVBA		; 47
	DW	?MOVCB		; 48
	DW	?MOVCC		; 49
	DW	?MOVCD		; 4A
	DW	?MOVCE		; 4B
	DW	?MOVCH		; 4C
	DW	?MOVCL		; 4D
	DW	?MOVCM		; 4E
	DW	?MOVCA		; 4F
	DW	?MOVDB		; 50
	DW	?MOVDC		; 51
	DW	?MOVDD		; 52
	DW	?MOVDE		; 53
	DW	?MOVDH		; 54
	DW	?MOVDL		; 55
	DW	?MOVDM		; 56
	DW	?MOVDA		; 57
	DW	?MOVEB		; 58
	DW	?MOVEC		; 59
	DW	?MOVED		; 5A
	DW	?MOVEE		; 5B
	DW	?MOVEH		; 5C
	DW	?MOVEL		; 5D
	DW	?MOVEM		; 5E
	DW	?MOVEA		; 5F
	DW	?MOVHB		; 60
	DW	?MOVHC		; 61
	DW	?MOVHD		; 62
	DW	?MOVHE		; 63
	DW	?MOVHH		; 64
	DW	?MOVHL		; 65
	DW	?MOVHM		; 66
	DW	?MOVHA		; 67
	DW	?MOVLB		; 68
	DW	?MOVLC		; 69
	DW	?MOVLD		; 6A
	DW	?MOVLE		; 6B
	DW	?MOVLH		; 6C
	DW	?MOVLL		; 6D
	DW	?MOVLM		; 6E
	DW	?MOVLA		; 6F
	DW	?MOVMB		; 70
	DW	?MOVMC		; 71
	DW	?MOVMD		; 72
	DW	?MOVME		; 73
	DW	?MOVMH		; 74
	DW	?MOVML		; 75
	DW	?HLT		; 76
	DW	?MOVMA		; 77
	DW	?MOVAB		; 78
	DW	?MOVAC		; 79
	DW	?MOVAD		; 7A
	DW	?MOVAE		; 7B
	DW	?MOVAH		; 7C
	DW	?MOVAL		; 7D
	DW	?MOVAM		; 7E
	DW	?MOVAA		; 7F
	DW	?ADDB		; 80
	DW	?ADDC		; 81
	DW	?ADDD		; 82
	DW	?ADDE		; 83
	DW	?ADDH		; 84
	DW	?ADDL		; 85
	DW	?ADDM		; 86
	DW	?ADDA		; 87
	DW	?ADCB		; 88
	DW	?ADCC		; 89
	DW	?ADCD		; 8A
	DW	?ADCE		; 8B
	DW	?ADCH		; 8C
	DW	?ADCL		; 8D
	DW	?ADCM		; 8E
	DW	?ADCA		; 8F
	DW	?SUBB		; 90
	DW	?SUBC		; 91
	DW	?SUBD		; 92
	DW	?SUBE		; 93
	DW	?SUBH		; 94
	DW	?SUBL		; 95
	DW	?SUBM		; 96
	DW	?SUBA		; 97
	DW	?SBBB		; 98
	DW	?SBBC		; 99
	DW	?SBBD		; 9A
	DW	?SBBE		; 9B
	DW	?SBBH		; 9C
	DW	?SBBL		; 9D
	DW	?SBBM		; 9E
	DW	?SBBA		; 9F
	DW	?ANAB		; A0
	DW	?ANAC		; A1
	DW	?ANAD		; A2
	DW	?ANAE		; A3
	DW	?ANAH		; A4
	DW	?ANAL		; A5
	DW	?ANAM		; A6
	DW	?ANAA		; A7
	DW	?XRAB		; A8
	DW	?XRAC		; A9
	DW	?XRAD		; AA
	DW	?XRAE		; AB
	DW	?XRAH		; AC
	DW	?XRAL		; AD
	DW	?XRAM		; AE
	DW	?XRAA		; AF
	DW	?ORAB		; B0
	DW	?ORAC		; B1
	DW	?ORAD		; B2
	DW	?ORAE		; B3
	DW	?ORAH		; B4
	DW	?ORAL		; B5
	DW	?ORAM		; B6
	DW	?ORAA		; B7
	DW	?CMPB		; B8
	DW	?CMPC		; B9
	DW	?CMPD		; BA
	DW	?CMPE		; BB
	DW	?CMPH		; BC
	DW	?CMPL		; BD
	DW	?CMPM		; BE
	DW	?CMPA		; BF
	DW	?RNZ		; C0
	DW	?POPB		; C1
	DW	?JNZ		; C2
	DW	?JMP		; C3
	DW	?CNZ		; C4
	DW	?PUSHB		; C5
	DW	?ADI		; C6
	DW	?RST0		; C7
	DW	?RZ		; C8
	DW	?RET		; C9
	DW	?JZ		; CA
	DW	?ILJMP		; CB
	DW	?CZ		; CC
	DW	?CALL		; CD
	DW	?ACI		; CE
	DW	?RST1		; CF
	DW	?RNC		; D0
	DW	?POPD		; D1
	DW	?JNC		; D2
	DW	?OUT		; D3
	DW	?CNC		; D4
	DW	?PUSHD		; D5
	DW	?SUI		; D6
	DW	?RST2		; D7
	DW	?RC		; D8
	DW	?ILRET		; D9
	DW	?JC		; DA
	DW	?IN		; DB
	DW	?CC		; DC
	DW	?ILCAL		; DD
	DW	?SBI		; DE
	DW	?RST3		; DF
	DW	?RPO		; E0
	DW	?POPH		; E1
	DW	?JPO		; E2
	DW	?XTHL		; E3
	DW	?CPO		; E4
	DW	?PUSHH		; E5
	DW	?ANI		; E6
	DW	?RST4		; E7
	DW	?RPE		; E8
	DW	?PCHL		; E9
	DW	?JPE		; EA
	DW	?XCHG		; EB
	DW	?CPE		; EC
	DW	?ILCAL		; ED
	DW	?XRI		; EE
	DW	?RST5		; EF
	DW	?RP		; F0
	DW	?POPP		; F1
	DW	?JP		; F2
	DW	?DI		; F3
	DW	?CP		; F4
	DW	?PUSHP		; F5
	DW	?ORI		; F6
	DW	?RST6		; F7
	DW	?RM		; F8
	DW	?SPHL		; F9
	DW	?JM		; FA
	DW	?EI		; FB
	DW	?CM		; FC
	DW	?ILCAL		; FD
	DW	?CPI		; FE
	DW	?RST7		; FF
;
; If the "setslow" function is called to initiate CPU speed emulation,
; The main multiple-step loop is patched to call this entry point instead
; of I8080 it simply calls the slowdown delay and "falls" into the normal
; I8080 function. This way we have no overhead unless we want to slow down.
;
X8080:	CALL	GOSLOW			; Call slowdown function
;
; Execute a single 8080 instruction
;
; For efficiency, 8080 registers are cached in 8086 registers as follows:
;  AL = Accumulator
;  CX = DE register pair
;  DX = HL register pair
;  SI = Stack pointer
;  DI = Program counter
;  ES = Virtual Memory Segment
;
I8080:	MOV	BL,DGRP:_EI		; Interrupt enable
	AND	BL,BL			; Enabled?
	JZ	I80801			; Yes
	JS	I80802			; No
	DEC	BYTE PTR DGRP:_EI	; Pending, Reduce count
	JMP SHORT I80802		; And continue processing
; Check for pending interrupt
I80801:	MOV	BL,DGRP:_PIRQ		; Pending IRQ
	AND	BL,7Fh			; Pending?
	JZ	I80802			; No, continue
	XOR	BH,BH			; Zero high
	MOV	BL,DGRP:IVTAB[BX]	; Get vector
	DEC	SI			; Backup stack
	DEC	SI			; Backup stack
	MOV	ES:[SI],DI		; Save PC
	MOV	DI,BX			; Set new PC
	MOV	BYTE PTR DGRP:_EI,255	; Disable interrupts
; Fetch & execute instruction
I80802:
;	CALL	S8080
;	CALL	_XX8080
;	CALL	L8080
	MOV	DGRP:_TRCBUF[BP],DI	; Save PC
;	INC	BYTE PTR DGRP:_M1
	INC	BP			; Advance
	INC	BP			; Advance
	AND	BP,1FFFh		; Mask for 8K table
	MOV	BL,ES:[DI]		; Get opcode
	INC	DI			; Skip to next
	XOR	BH,BH			; Zero high
	SHL	BX,1			; x2 for words
	JMP	ITABLE[BX]		; Execute handler
;
; Load 8080 registers into corresponding 8086 registers for fast access.
;
L8080:	MOV	ES,DGRP:_SEG		; Read segment
	MOV	AL,DGRP:_A		; Get A
	MOV	CX,WORD PTR DGRP:_E	; Get BC
	MOV	DX,WORD PTR DGRP:_L	; Get DE
	MOV	SI,DGRP:_SP		; Get SP
	MOV	DI,DGRP:_PC		; Get PC
	MOV	BP,DGRP:_TRCPTR		; Get trace pointer
	RET
;
; Save the cached registers back into memory.
;
S8080:	MOV	DGRP:_A,AL		; Save A
	MOV 	WORD PTR DGRP:_E,CX	; Save BC
	MOV	WORD PTR DGRP:_L,DX	; Save DE
	MOV	DGRP:_SP,SI		; Save SI
	MOV	DGRP:_PC,DI		; Save PC
	MOV	DGRP:_TRCPTR,BP		; Save BP
	RET
; Far call stub
_far_step:
	MOV	DI,12			; Offset 12
	CALL	_FCALL			; Call it
;
; Step multiple 8080 instructions
; On entry, global word 'STEP' contains number of instructions
; to step.
;
_multi_step:
;	XOR	AX,AX			; Get zero
;	MOV	WORD PTR DGRP:_MEMR,AX	; Clear MEMR & IN flags
;	MOV	WORD PTR DGRP:_OUT,AX	; Clear OUT & Stack flags
;	MOV	BYTE PTR DGRP:_M1,AL	; Clear M1
	PUSH	BP			; Save callers BP
	CALL	L8080			; Load registers
	MOV	AH,DGRP:_BRKSK		; Skip breakpoint?
	AND	AH,AH			; Test
	JZ	mstep1			; No, do breakpoint
	XOR	AH,AH			; Get zero
	MOV	DGRP:_BRKSK,AH		; Clear flag
	JMP	SHORT mstep2		; Skip first breakpoint check
; Check for PC at the breakpoint location, and if so stop
mstep1:	CMP	DI,DGRP:_BREAK		; At breakpoint
	JZ	xbrk			; Yes, handle it
; Execute 8080 instructructions in a loop until STEP expires
mstep2:	CALL	I8080			; Execute one instruction
mstep3:	DEC	WORD PTR DGRP:_STEP	; Reduce STEP count
	JNZ	mstep1			; More to do.
; Normal termination - save registers & return 0 to caller
	CALL	S8080			; Save registers
	POP	BP			; Restore BP
	INC	WORD PTR DGRP:_INSTCT	; Advance block count
	XOR	AX,AX			; Indicate success
	RET
; Breakpoint encountered - save registers & indicate to caller
xbrk:	CALL	S8080			; Save registers
	POP	BP			; Restore BP
	MOV	AX,2			; Indicate breakpoint
	RET
;
; Illegal instruction was encountered
;
?ILLEGAL: MOV	AH,DGRP:_IBADOP	; Ignore bad opcodes?
	AND	AH,AH		; Flag set?
	JNZ	?NOP		; Yes - continue
	POP	BX		; Skip 1st return address
	POP	BX		; Skip 2nd return address
	DEC	DI		; Back to offending instruction
	CALL	S8080		; Save registers
	POP	BP		; Restore BP
	MOV	AX,-1		; Indicate failure
	RET

;
; Individual 8080 instruction handlers
; - Perform required operation with corresponding 8086 registers.
;
?ILNOP:	CALL	?ILLEGAL		; Test for illegal
?NOP:	RET
; LXI B,xxxx
?LXIB:	MOV	BL,ES:[DI]		; Read low
	INC	DI			; Next
	MOV	BH,ES:[DI]		; Read high
	INC	DI			; Next
	MOV	WORD PTR DGRP:_C,BX	; Save BC
	RET
;
?STAXB:	MOV	BX,WORD PTR DGRP:_C	; Get BC
	MOV	AH,AL			; Get A
	JMP	WRBX
;
?INXB:	INC	WORD PTR DGRP:_C	; Increment register
	RET
;
?INRB:	INC	BYTE PTR DGRP:_B	; Increment HI
	JMP	SFZSPA			; Save flags
;
?DCRB:	DEC	BYTE PTR DGRP:_B	; Decrement HI
	JMP	SFZSPA			; Save flags
;
?MVIB:	MOV	AH,ES:[DI]		; Get low
	INC	DI			; Next
	MOV	DGRP:_B,AH		; Save B
	RET
;
?RLC:	ROL	AL,1			; Do the shift
	JMP	SFC			; Save carry
;
?DADB:	ADD	DX,WORD PTR DGRP:_C	; Add BC
	JMP	SFC			; Save carry
;
?LDAXB:	MOV	BX,WORD PTR DGRP:_C	; Get BC
	CALL	RDBX			; Get value
	MOV	AL,AH			; Set A
	RET
;
?DCXB:	DEC	WORD PTR DGRP:_C	; Decrement register
	RET
;
?INRC:	INC	BYTE PTR DGRP:_C	; Increment LO
	JMP	SFZSPA			; Save flags
;
?DCRC:	DEC	BYTE PTR DGRP:_C	; Decrement LO
	JMP	SFZSPA			; Save flags
;
?MVIC:	MOV	AH,ES:[DI]		; Get data
	MOV	DGRP:_C,AH		; Write C
	INC	DI			; Next
	RET
;
?RRC:	ROR	AL,1			; Rotate
	JMP	SFC			; Save carry
;
?LXID:	MOV	CL,ES:[DI]		; Read value
	INC	DI			; Next
	MOV	CH,ES:[DI]		; Read value
	INC	DI			; Next
	RET
;
?STAXD:	MOV	BX,CX			; Get address
	MOV	AH,AL			; Get A
	JMP	WRBX			; Write it
;
?INXD:	INC	CX			; Increment
	RET
;
?INRD:	INC	CH			; Increment HI
	JMP	SFZSPA			; Save flags
;
?DCRD:	DEC	CH			; Decrement HI
	JMP	SFZSPA			; Save flags
;
?MVID:	MOV	CH,ES:[DI]		; Get value
	INC	DI			; Advance
	RET
;
?RAL:	MOV	AH,DGRP:_PSW		; Get PSW
	SAHF				; Set flags
	RCL	AL,1			; Do the rotate
	JMP	SFC			; Save carry
;
?DADD:	ADD	DX,CX			; Add DE
	JMP	SFC			; Save carry
;
?LDAXD:	MOV	BX,CX			; Get address
	CALL	RDBX			; Read memory
	MOV	AL,AH			; Set A
	RET
;
?DCXD:	DEC	CX			; Decrement
	RET
;
?INRE:	INC	CL			; Increment LO
	JMP	SFZSPA			; Save flags
;
?DCRE:	DEC	CL			; Decrement DL
	JMP	SFZSPA			; Save flags
;
?MVIE:	MOV	CL,ES:[DI]		; Read value
	INC	DI
	RET
;
?RAR:	MOV	AH,DGRP:_PSW		; Get PSW
	SAHF				; Set flags
	RCR	AL,1			; Do the rotate
	JMP	SFC			; Save carry

;
?LXIH:	MOV	DL,ES:[DI]		; Read low
	INC	DI			; Next
	MOV	DH,ES:[DI]		; Read high
	INC	DI			; Next
	RET
;
?SHLD:	MOV	BX,ES:[DI]		; Get address
	INC	DI			; Next
	INC	DI			; Next
	MOV	AH,DL			; Get L
	CALL	WRBX			; Write it
	MOV	AH,DH			; Get H
	JMP	WRBXI			; Write it
;
?INXH:	INC	DX
	RET
;
?INRH:	INC	DH			; Increment HI
	JMP	SFZSPA			; Save flags
;
?DCRH:	DEC	DH			; Decrement HI
	JMP	SFZSPA			; Save flags
;
?MVIH:	MOV	DH,ES:[DI]		; Get data
	INC	DI			; Next
	RET

;
?DADH:	ADD	DX,DX			; Add HL
	JMP	SFC			; Save carry
;
?LHLD:	MOV	BX,ES:[DI]		; Get address
	INC	DI			; Next
	INC	DI			; Next
	CALL	RDBX			; Get low
	MOV	DL,AH			; Set L
	CALL	RDBXI			; Get high
	MOV	DH,AH			; Set H
	RET
;
?DCXH:	DEC	DX			; Decrement
	RET
;
?INRL:	INC	DL			; Increment HI
	JMP	SFZSPA			; Save flags
;
?DCRL:	DEC	DL			; Decrement HI
	JMP	SFZSPA			; Save flags
;
?MVIL:	MOV	DL,ES:[DI]		; Get data
	INC	DI			; Next
	RET
;
?CMA:	NOT	AL			; Complement
	RET
;
?LXISP:	MOV	BL,ES:[DI]		; Get low
	INC	DI			; Next
	MOV	BH,ES:[DI]		; Get high
	INC	DI			; Next
	MOV	SI,BX			; Set SP
	RET
;
?STA:	MOV	BX,ES:[DI]		; Get address
	INC	DI			; Next
	INC	DI			; Next
	MOV	AH,AL			; Get A
	JMP	WRBX			; Write it
	RET
;
?INXSP:	INC	SI			; Increment
	RET
;
?INRM:	MOV	BX,DX			; Get HL
	CALL	RDBX			; Read value
	INC	AH			; Increment
	PUSHF				; Save flags
	CALL	WRBX			; Write it
	POPF				; Restore flags
	JMP	SFZSPA			; Save flags
;
?DCRM:	MOV	BX,DX			; Get HL
	CALL	RDBX			; Read value
	DEC	AH			; Decrement
	PUSHF				; Save flags
	CALL	WRBX			; Resave
	POPF				; Restore flags
	JMP	SFZSPA			; Save flags
;
?MVIM:	MOV	BX,DX			; Get HL
	MOV	AH,ES:[DI]		; Get immed
	INC	DI			; Next
	JMP	WRBX			; Write it
;
?STC:	MOV	AH,DGRP:_PSW		; Get PSW
	OR	AH,01h			; Set CARRY
	MOV	DGRP:_PSW,AH		; Resave
	RET
;
?DADSP:	ADD	DX,SI			; Add SP
	JMP	SFC			; Save carry
;
?LDA:	MOV	BX,ES:[DI]		; Get address
	INC	DI			; Next
	INC	DI			; Next
	CALL	RDBX			; Read it
	MOV	AL,AH			; Set A
	RET
;
?DCXSP:	DEC	SI			; Decrement
	RET
;
?INRA:	INC	AL
	JMP	SFZSPA
;
?DCRA:	DEC	AL
	JMP	SFZSPA
;
?MVIA:	MOV	AL,ES:[DI]		; Read data
	INC	DI			; Next
	RET
;
?CMC:	MOV	AH,DGRP:_PSW		; Get PSW
	XOR	AH,01h			; Toggle carry
	MOV	DGRP:_PSW,AH		; Resave
	RET
;
?MOVBC:	MOV	AH,DGRP:_C		; Get C
	MOV	DGRP:_B,AH		; Stash in B
?MOVBB:	RET
?MOVBD:	MOV	DGRP:_B,CH
	RET
?MOVBE:	MOV	DGRP:_B,CL
	RET
?MOVBH:	MOV	DGRP:_B,DH
	RET
?MOVBL:	MOV	DGRP:_B,DL
	RET
?MOVBM:	MOV	BX,DX
	CALL	RDBX			; Read value
	MOV	DGRP:_B,AH
	RET
?MOVBA:	MOV	DGRP:_B,AL
	RET
;
?MOVCB:	MOV	AH,DGRP:_B
	MOV	DGRP:_C,AH
?MOVCC:	RET
?MOVCD:	MOV	DGRP:_C,CH
	RET
?MOVCE:	MOV	DGRP:_C,CL
	RET
?MOVCH:	MOV	DGRP:_C,DH
	RET
?MOVCL:	MOV	DGRP:_C,DL
	RET
?MOVCM:	MOV	BX,DX
	CALL	RDBX
	MOV	DGRP:_C,AH
	RET
?MOVCA:	MOV	DGRP:_C,AL
	RET
;
?MOVDB:	MOV	CH,DGRP:_B
	RET
?MOVDC:	MOV	CH,DGRP:_C
?MOVDD:	RET
?MOVDE:	MOV	CH,CL
	RET
?MOVDH:	MOV	CH,DH
	RET
?MOVDL:	MOV	CH,DL
	RET
?MOVDM:	MOV	BX,DX
	CALL	RDBX
	MOV	CH,AH
	RET
?MOVDA:	MOV	CH,AL
	RET
;
?MOVEB:	MOV	CL,DGRP:_B
	RET
?MOVEC:	MOV	CL,DGRP:_C
	RET
?MOVED:	MOV	CL,CH
?MOVEE:	RET
?MOVEH:	MOV	CL,DH
	RET
?MOVEL:	MOV	CL,DL
	RET
?MOVEM:	MOV	BX,DX
	CALL	RDBX
	MOV	CL,AH
	RET
?MOVEA:	MOV	CL,AL
	RET
?MOVHB:	MOV	DH,DGRP:_B
	RET
?MOVHC:	MOV	DH,DGRP:_C
	RET
?MOVHD:	MOV	DH,CH
	RET
?MOVHE:	MOV	DH,CL
?MOVHH:	RET
?MOVHL:	MOV	DH,DL
	RET
?MOVHM:	MOV	BX,DX
	CALL	RDBX
	MOV	DH,AH
	RET
?MOVHA:	MOV	DH,AL
	RET
;
?MOVLB:	MOV	DL,DGRP:_B
	RET
?MOVLC:	MOV	DL,DGRP:_C
	RET
?MOVLD:	MOV	DL,CH
	RET
?MOVLE:	MOV	DL,CL
	RET
?MOVLH:	MOV	DL,DH
?MOVLL:	RET
?MOVLM:	MOV	BX,DX
	CALL	RDBX
	MOV	DL,AH
	RET
?MOVLA:	MOV	DL,AL
	RET
;
?MOVMB:	MOV	BX,DX
	MOV	AH,DGRP:_B
	JMP	WRBX
?MOVMC:	MOV	BX,DX
	MOV	AH,DGRP:_C
	JMP	WRBX
?MOVMD:	MOV	BX,DX
	MOV	AH,CH
	JMP	WRBX
?MOVME:	MOV	BX,DX
	MOV	AH,CL
	JMP	WRBX
?MOVMH:	MOV	BX,DX
	MOV	AH,DH
	JMP	WRBX
?MOVML:	MOV	BX,DX
	MOV	AH,DL
	JMP	WRBX
?MOVMA:	MOV	BX,DX
	MOV	AH,AL
	JMP	WRBX
;
?MOVAB:	MOV	AL,DGRP:_B
	RET
?MOVAC:	MOV	AL,DGRP:_C
	RET
?MOVAD:	MOV	AL,CH
	RET
?MOVAE:	MOV	AL,CL
	RET
?MOVAH:	MOV	AL,DH
	RET
?MOVAL:	MOV	AL,DL
	RET
?MOVAM:	MOV	BX,DX
	CALL	RDBX
	MOV	AL,AH
?MOVAA:	RET
;
?ADDB:	ADD	AL,DGRP:_B
	JMP	SFZSPCA
?ADDC:	ADD	AL,DGRP:_C
	JMP	SFZSPCA
?ADDD:	ADD	AL,CH
	JMP	SFZSPCA
?ADDE:	ADD	AL,CL
	JMP	SFZSPCA
?ADDH:	ADD	AL,DH
	JMP	SFZSPCA
?ADDL:	ADD	AL,DL
	JMP	SFZSPCA
?ADDM:	MOV	BX,DX
	CALL	RDBX
	ADD	AL,AH
	JMP	SFZSPCA
?ADDA:	ADD	AL,AL
	JMP	SFZSPCA
?ADCB:	MOV	AH,DGRP:_PSW		; Get PSW
	SAHF				; Set FLAGS
	ADC	AL,DGRP:_B
	JMP	SFZSPCA
?ADCC:	MOV	AH,DGRP:_PSW		; Get PSW
	SAHF				; Set FLAGS
	ADC	AL,DGRP:_C
	JMP	SFZSPCA
?ADCD:	MOV	AH,DGRP:_PSW		; Get PSW
	SAHF				; Set FLAGS
	ADC	AL,CH
	JMP	SFZSPCA
?ADCE:	MOV	AH,DGRP:_PSW		; Get PSW
	SAHF				; Set FLAGS
	ADC	AL,CL
	JMP	SFZSPCA
?ADCH:	MOV	AH,DGRP:_PSW		; Get PSW
	SAHF				; Set FLAGS
	ADC	AL,DH
	JMP	SFZSPCA
?ADCL:	MOV	AH,DGRP:_PSW		; Get PSW
	SAHF				; Set FLAGS
	ADC	AL,DL
	JMP	SFZSPCA
?ADCM:	MOV	BX,DX
	MOV	AH,DGRP:_PSW		; Get PSW
	SAHF				; Set FLAGS
	PUSHF
	CALL	RDBX
	POPF
	ADC	AL,AH
	JMP	SFZSPCA
?ADCA:	MOV	AH,DGRP:_PSW		; Get PSW
	SAHF				; Set FLAGS
	ADC	AL,AL
	JMP	SFZSPCA
;
?SUBB:	SUB	AL,DGRP:_B
	JMP	SFZSPCA
?SUBC:	SUB	AL,DGRP:_C
	JMP	SFZSPCA
?SUBD:	SUB	AL,CH
	JMP	SFZSPCA
?SUBE:	SUB	AL,CL
	JMP	SFZSPCA
?SUBH:	SUB	AL,DH
	JMP	SFZSPCA
?SUBL:	SUB	AL,DL
	JMP	SFZSPCA
?SUBM:	MOV	BX,DX
	CALL	RDBX
	SUB	AL,AH
	JMP	SFZSPCA
?SUBA:	SUB	AL,AL
	JMP	SFZSPCA
;
?SBBB:	MOV	AH,DGRP:_PSW
	SAHF
	SBB	AL,DGRP:_B
	JMP	SFZSPCA
?SBBC:	MOV	AH,DGRP:_PSW
	SAHF
	SBB	AL,DGRP:_C
	JMP	SFZSPCA
?SBBD:	MOV	AH,DGRP:_PSW
	SAHF
	SBB	AL,CH
	JMP	SFZSPCA
?SBBE:	MOV	AH,DGRP:_PSW
	SAHF
	SBB	AL,CL
	JMP	SFZSPCA
?SBBH:	MOV	AH,DGRP:_PSW
	SAHF
	SBB	AL,DH
	JMP	SFZSPCA
?SBBL:	MOV	AH,DGRP:_PSW
	SAHF
	SBB	AL,DL
	JMP	SFZSPCA
?SBBM:	MOV	BX,DX
	MOV	AH,DGRP:_PSW
	SAHF
	PUSHF
	CALL	RDBX
	POPF
	SBB	AL,AH
	JMP	SFZSPCA
?SBBA:	MOV	AH,DGRP:_PSW
	SAHF
	SBB	AL,AL
	JMP	SFZSPCA
;
?ANAB:	AND	AL,DGRP:_B
	JMP	SFZSPCA
?ANAC:	AND	AL,DGRP:_C
	JMP	SFZSPCA
?ANAD:	AND	AL,CH
	JMP	SFZSPCA
?ANAE:	AND	AL,CL
	JMP	SFZSPCA
?ANAH:	AND	AL,DH
	JMP	SFZSPCA
?ANAL:	AND	AL,DL
	JMP	SFZSPCA
?ANAM:	MOV	BX,DX
	CALL	RDBX
	AND	AL,AH
	JMP	SFZSPCA
?ANAA:	AND	AL,AL
	JMP	SFZSPCA
;
?XRAB:	XOR	AL,DGRP:_B
	JMP	SFZSPCA
?XRAC:	XOR	AL,DGRP:_C
	JMP	SFZSPCA
?XRAD:	XOR	AL,CH
	JMP	SFZSPCA
?XRAE:	XOR	AL,CL
	JMP	SFZSPCA
?XRAH:	XOR	AL,DH
	JMP	SFZSPCA
?XRAL:	XOR	AL,DL
	JMP	SFZSPCA
?XRAM:	MOV	BX,DX
	CALL	RDBX
	XOR	AL,AH
	JMP	SFZSPCA
?XRAA:	XOR	AL,AL
	JMP	SFZSPCA
;
?ORAB:	OR	AL,DGRP:_B
	JMP	SFZSPCA
?ORAC:	OR	AL,DGRP:_C
	JMP	SFZSPCA
?ORAD:	OR	AL,CH
	JMP	SFZSPCA
?ORAE:	OR	AL,CL
	JMP	SFZSPCA
?ORAH:	OR	AL,DH
	JMP	SFZSPCA
?ORAL:	OR	AL,DL
	JMP	SFZSPCA
?ORAM:	MOV	BX,DX
	CALL	RDBX
	OR	AL,AH
	JMP	SFZSPCA
?ORAA:	OR	AL,AL
	JMP	SFZSPCA
;
?CMPB:	CMP	AL,DGRP:_B
	JMP	SFZSPCA
?CMPC:	CMP	AL,DGRP:_C
	JMP	SFZSPCA
?CMPD:	CMP	AL,CH
	JMP	SFZSPCA
?CMPE:	CMP	AL,CL
	JMP	SFZSPCA
?CMPH:	CMP	AL,DH
	JMP	SFZSPCA
?CMPL:	CMP	AL,DL
	JMP	SFZSPCA
?CMPM:	MOV	BX,DX
	CALL	RDBX
	CMP	AL,AH
	JMP	SFZSPCA
?CMPA:	CMP	AL,AL
	JMP	SFZSPCA
;
?RZ:	MOV	AH,DGRP:_PSW
	SAHF
	JZ	?RET
	RET
?RNZ:	MOV	AH,DGRP:_PSW
	SAHF
	JNZ	?RET
	RET
?RC:	MOV	AH,DGRP:_PSW
	SAHF
	JC	?RET
	RET
?RNC:	MOV	AH,DGRP:_PSW
	SAHF
	JNC	?RET
	RET
?RP:	MOV	AH,DGRP:_PSW
	SAHF
	JNS	?RET
	RET
?RM:	MOV	AH,DGRP:_PSW
	SAHF
	JS	?RET
	RET
?RPO:	MOV	AH,DGRP:_PSW
	SAHF
	JPO	?RET
	RET
?RPE:	MOV	AH,DGRP:_PSW
	SAHF
	JPE	?RET
	RET
?ILRET:	CALL	?ILLEGAL	; Test for illegal
?RET:	MOV	DI,ES:[SI]	; Get new PC from stack
	INC	SI		; Pop
	INC	SI		; Pop
;	INC	BYTE PTR DGRP:_STACK
	RET
;
?PUSHB:	MOV	BX,WORD PTR DGRP:_C
	DEC	SI
	DEC	SI
	MOV	ES:[SI],BX
;	INC	BYTE PTR DGRP:_STACK
	RET
?POPB:	MOV	BX,ES:[SI]
	INC	SI
	INC	SI
	MOV	WORD PTR DGRP:_C,BX
;	INC	BYTE PTR DGRP:_STACK
	RET
?PUSHD:	DEC	SI
	DEC	SI
	MOV	ES:[SI],CX
;	INC	BYTE PTR DGRP:_STACK
	RET
?POPD:	MOV	CX,ES:[SI]
	INC	SI
	INC	SI
;	INC	BYTE PTR DGRP:_STACK
	RET
?PUSHH:	DEC	SI
	DEC	SI
	MOV	ES:[SI],DX
;	INC	BYTE PTR DGRP:_STACK
	RET
?POPH:	MOV	DX,ES:[SI]
	INC	SI
	INC	SI
;	INC	BYTE PTR DGRP:_STACK
	RET
?PUSHP:	MOV	AH,DGRP:_PSW
	DEC	SI
	DEC	SI
	MOV	ES:[SI],AH
	MOV	ES:1[SI],AL
;	MOV	ES:[SI],AX
;	INC	BYTE PTR DGRP:_STACK
	RET
?POPP:	MOV	AH,ES:[SI]
	MOV	AL,ES:1[SI]
;	MOV	ES:[SI],AX
	INC	SI
	INC	SI
	MOV	DGRP:_PSW,AH
;	INC	BYTE PTR DGRP:_STACK
	RET
;
?JZ:	MOV	AH,DGRP:_PSW
	SAHF
	JZ	?JMP
	INC	DI
	INC	DI
	RET
?JNZ:	MOV	AH,DGRP:_PSW
	SAHF
	JNZ	?JMP
	INC	DI
	INC	DI
	RET
?JC:	MOV	AH,DGRP:_PSW
	SAHF
	JC	?JMP
	INC	DI
	INC	DI
	RET
?JNC:	MOV	AH,DGRP:_PSW
	SAHF
	JNC	?JMP
	INC	DI
	INC	DI
	RET
?JP:	MOV	AH,DGRP:_PSW
	SAHF
	JNS	?JMP
	INC	DI
	INC	DI
	RET
?JM:	MOV	AH,DGRP:_PSW
	SAHF
	JS	?JMP
	INC	DI
	INC	DI
	RET
?JPE:	MOV	AH,DGRP:_PSW
	SAHF
	JPE	?JMP
	INC	DI
	INC	DI
	RET
?JPO:	MOV	AH,DGRP:_PSW
	SAHF
	JPO	?JMP
	INC	DI
	INC	DI
	RET
?ILJMP:	CALL	?ILLEGAL
?JMP:	MOV	DI,ES:[DI]
	RET
;
?CZ:	MOV	AH,DGRP:_PSW
	SAHF
	JZ	?CALL
	INC	DI
	INC	DI
	RET
?CNZ:	MOV	AH,DGRP:_PSW
	SAHF
	JNZ	?CALL
	INC	DI
	INC	DI
	RET
?CC:	MOV	AH,DGRP:_PSW
	SAHF
	JC	?CALL
	INC	DI
	INC	DI
	RET
?CNC:	MOV	AH,DGRP:_PSW
	SAHF
	JNC	?CALL
	INC	DI
	INC	DI
	RET
?CP:	MOV	AH,DGRP:_PSW
	SAHF
	JNS	?CALL
	INC	DI
	INC	DI
	RET
?CM:	MOV	AH,DGRP:_PSW
	SAHF
	JS	?CALL
	INC	DI
	INC	DI
	RET
?CPE:	MOV	AH,DGRP:_PSW
	SAHF
	JPE	?CALL
	INC	DI
	INC	DI
	RET
?CPO:	MOV	AH,DGRP:_PSW
	SAHF
	JPO	?CALL
	INC	DI
	INC	DI
	RET
?ILCAL:	CALL	?ILLEGAL		; Test for illegal
?CALL:	DEC	SI
	DEC	SI
	MOV	BX,ES:[DI]
	INC	DI
	INC	DI
	MOV	ES:[SI],DI
	MOV	DI,BX
;	INC	BYTE PTR DGRP:_STACK
	RET
;
?ADI:	INC	DI
	ADD	AL,ES:-1[DI]
	JMP	SFZSPCA
?ACI:	INC	DI
	MOV	AH,DGRP:_PSW
	SAHF
	ADC	AL,ES:-1[DI]
	JMP	SFZSPCA
?SUI:	INC	DI
	SUB	AL,ES:-1[DI]
	JMP	SFZSPCA
?SBI:	INC	DI
	MOV	AH,DGRP:_PSW
	SAHF
	SBB	AL,ES:-1[DI]
	JMP	SFZSPCA
?ANI:	INC	DI
	AND	AL,ES:-1[DI]
	JMP	SFZSPCA
?ORI:	INC	DI
	OR	AL,ES:-1[DI]
	JMP	SFZSPCA
?XRI:	INC	DI
	XOR	AL,ES:-1[DI]
	JMP	SFZSPCA
?CPI:	INC	DI
	CMP	AL,ES:-1[DI]
	JMP	SFZSPCA
;
?XCHG:	XCHG	CX,DX
	RET
?XTHL:	MOV	BX,ES:[SI]
	MOV	ES:[SI],DX
	MOV	DX,BX
	RET
?PCHL:	MOV	DI,DX
	RET
?SPHL:	MOV	SI,DX
	RET
;
; HALT instruction encountered
; Instead of returning to multi-step loop, clear it's return address from
; stack, save the cached regsters & return directly to original caller.
?HLT:	POP	BX		; Skip return address
	DEC	DI		; Back to offending instruction
	CALL	S8080		; Save registers
	POP	BP		; Restore BP
	MOV	AX,1		; Indicate failure
	RET
; Restart instructions
?RST0:
?RST1:
?RST2:
?RST3:
?RST4:
?RST5:
?RST6:
?RST7:	DEC	SI
	DEC	SI
	MOV	ES:[SI],DI
	MOV	BL,ES:-1[DI]
	AND	BL,38h
	XOR	BH,BH
	MOV	DI,BX
	RET
; 8080 OUT instruction
;
; Save registers & call main program handler to perform the OUT function
;
?OUT:	MOV	BL,ES:[DI]
	INC	DI
	XOR	BH,BH
	CALL	S8080
	PUSH	BX
	CALL	_OUT8080
	POP	BX
;	INC	BYTE PTR DGRP:_OUT
	AND	AX,AX			; Success?
	JZ	inret
; Handler reports an invalid I/O port accessed, terminate stepping and
; return to original called with indication.
ofail:	POP	BX			; Clear return address
	DEC	WORD PTR DGRP:_PC	; Adjust PC
	DEC	WORD PTR DGRP:_PC	; Adjust PC
	POP	BP			; Restore caller BP
	MOV	AX,-2			; Indicate bad I/O
	RET
; 8080 IN instruction
;
; Save registers & call main program handler to perform the IN function.
?IN:	MOV	BL,ES:[DI]
	INC	DI
	XOR	BH,BH
	CALL	S8080
	PUSH	BX
	CALL	_IN8080
	POP	BX
;	INC	BYTE PTR DGRP:_IN
	AND	AX,AX			; Success?
	JNZ	ofail			; Report error
inret:	JMP	L8080			; Reload registers & return
; Decimal adjust for addition
?DAA:	MOV	AH,DGRP:_PSW		; Get PSW
	SAHF				; Set FLAGS
	DAA				; Decimal adjust
	JMP	SFZSPCA
; Disable interrupts
?DI:	MOV	BYTE PTR DGRP:_EI,255	; Fully disable
	JMP	SHORT upnl		; Update panel
; Enable interrupts
?EI:	MOV	BL,1			; Assume 1
	MOV	AH,DGRP:_OF0		; Get step flag
	AND	AH,10h			; Step enabled?
	JNZ	?ei1			; No - normal
	INC	BL			; extra cycle for step	
?ei1:	MOV	DGRP:_EI,BL		; Pending enable
upnl:	CALL	S8080
	CALL	_UP8080
	JMP	L8080
;
;
; Flag update functions:
;
; Save C (Carry) flag only
SFC:	MOV	AH,DGRP:_PSW	; Get existing PSW
	JC	SFC1		; Set carry
	AND	AH,0FEh		; Clear carry bit
	MOV	DGRP:_PSW,AH	; Resave
	RET
SFC1:	OR	AH,01h		; Set carry bit
	MOV	DGRP:_PSW,AH
	RET
; save Z (Zero), S (Sign), P (Parity), and AC (Auxiliary Carry) flags
SFZSPA:	LAHF			; Get flags
	MOV	BL,DGRP:_PSW	; Get existing PSW
	AND	AH,0D4h		; Save all but carry
	AND	BL,2Bh		; Save original
	OR	AH,BL		; Set new flags
	MOV	DGRP:_PSW,AH	; Save them
	RET
; Save ALL flags: Z (Zero), S (Sign), P (Parity), C (Carry) , AC (Aux. Carry)
SFZSPCA: LAHF		; Get flags
	MOV	DGRP:_PSW,AH
	RET
;
; 8080 Memory access functions:
; we access main storage through these functions so that we can check
; for memory mapped hardware addresses.
;
; Increment BX & write value in AH to ES:[BX]
WRBXI:	INC	BX		; Increment
; Write value to ES:[BX]
WRBX:	CMP	BH,20h		; User RAM
	JAE	wrbx0		; Allow
	PUSH	BX		; Save BX
	MOV	BL,BH		; Set low
	XOR	BH,BH		; Zero high
	OR	BH,_LMWP[BX]	; Get indicator
	POP	BX		; Restore
	JNZ	wrbx1		; Do not allow
wrbx0:	MOV	ES:[BX],AH	; Write value
wrbx1:	RET	
;
; Increment BX & read value into AH from ES:[BX]
RDBXI:	INC	BX		; Increment
; Read value from ES:[BX] into AH
;RDBX:	INC	BYTE PTR DGRP:_MEMR
; Standard RAM access to ES:[BX]
RDBX:	MOV	AH,ES:[BX]	; Read value
	RET
;
; Read target memory for outside caller
;
_Mread:	MOV	BX,SP		; Get SP
	MOV	BX,2[BX]	; Get address
	MOV	ES,DGRP:_SEG	; Get segment
	MOV	AL,ES:[BX]	; Read data
	XOR	AH,AH		; Zero high
	RET
;
; Slow down execution by performing a delay loop before each
; instruction.
;
GOSLOW:	MOV	BX,DGRP:_SLOWCT	; Get slowdown count
goslo1:	DEC	BX		; Reduce count
	JNZ	goslo1		; Do them all
	RET
;
; Set system slowdown mode
;
_setslow: MOV	BX,SP		; Address parameters
	MOV	AX,2[BX]	; Get parameter
	MOV	DGRP:_SLOWCT,AX	; Set slowdown count
	MOV	AX,(OFFSET X8080)-(OFFSET mstep3) ; Get slow offset
ss1:	MOV	BX,OFFSET mstep2; Point to CALL instruction
	MOV	1[BX],AX	; Patch to call slowed version
	RET
_setfast: MOV	AX,(OFFSET I8080)-(OFFSET mstep3) ; Get FAST offset
	JMP	SHORT ss1;	; Set & return
;
; Far call to IOSEG:DI
;
;	DB	'D','A','V','E'
_FCALL:	MOV	AX,DGRP:_IOSEG	; Get I/O segment
	AND	AX,AX		; Enabled
	JZ	fcret		; No - skip it
	MOV	SI,SP		; Save SP
	CLI			; No interrupts
	MOV	SS,AX		; Get segment for stack
	MOV	SP,0		; Set new stack pointer
	STI			; Re-allow interrupts
	PUSH	SI		; Save our SP
	PUSH	DS		; Save our DS
	PUSH	DS		; Again for tx to ES
	POP	ES		; ES = out segment
	MOV	CX,OFFSET DGRP:xret	; return address
	MOV	BX,OFFSET DGRP:_SEG	; Point to data area
	MOV	DS,AX		; Get segment for data
	PUSH	CS		; Set return segment
	PUSH	CX		; Set return address
	PUSH	DS		; Set Call segment
	PUSH	DI		; Set Call offset
	DB	0CBh		; Perform long call
; Return from far caller here
xret:	POP	BX		; Get segment
	POP	SI		; Get SP
	MOV	DS,BX		; Restore DS
	CLI			; No interrupts
	MOV	SS,BX		; Set Stack seg
	MOV	SP,SI		; Set stack pointer
	STI			; Re-allow interrupts
fcret:	RET
CSEG	ENDS
;
; Initialized data
;
DSEG	SEGMENT	BYTE PUBLIC 'IDATA'
	PUBLIC	_A, _B, _C, _D, _E, _H, _L, _SP, _PC, _PSW
	PUBLIC _SEG, _IOSEG, _TRCPTR, _STEP, _BREAK, _BRKSK, _PIRQ, _LMWP
	PUBLIC	_EI, _IBADOP
; Interrupt vector table
IVTAB	DB	  0,  8, 16,  8, 24,  8, 16,  8, 32,  8, 16,  8, 24,  8, 16,  8
	DB	 40,  8, 16,  8, 24,  8, 16,  8, 32,  8, 16,  8, 24,  8, 16,  8
	DB	 48,  8, 16,  8, 24,  8, 16,  8, 32,  8, 16,  8, 24,  8, 16,  8
	DB	 40,  8, 16,  8, 24,  8, 16,  8, 32,  8, 16,  8, 24,  8, 16,  8
	DB	 56,  8, 16,  8, 24,  8, 16,  8, 32,  8, 16,  8, 24,  8, 16,  8
	DB	 40,  8, 16,  8, 24,  8, 16,  8, 32,  8, 16,  8, 24,  8, 16,  8
	DB	 48,  8, 16,  8, 24,  8, 16,  8, 32,  8, 16,  8, 24,  8, 16,  8
	DB	 40,  8, 16,  8, 24,  8, 16,  8, 32,  8, 16,  8, 24,  8, 16,  8
_IBADOP	DB	0	; Ignore bad opcode flag
_SEG	DW	0	; Virtual memory segment
_EI	DB	0	; Interrupts enabled flag
_PIRQ	DB	0	; Pending interrupt
_A	DB	0	; 8080 A register
_PSW	DB	0	; 8080 PSW register
_C	DB	0	; 8080 C register
_B	DB	0	; 8080 B register
_E	DB	0	; 8080 E register
_D	DB	0	; 8080 D register
_L	DB	0	; 8080 L register
_H	DB	0	; 8080 H register
_SP	DW	0	; 8080 SP register
_PC	DW	0	; 8080 PC register
; Low Memory Write Protect table
_LMWP	DB	255,255,255,255,000,000,000,000,000,000,000,000,000,000,000,000
	DB	000,000,000,000,000,000,000,000,000,000,000,000,000,000,000,000
_IOSEG	DW	0	; I/O handler segment (0 if none)
_TRCPTR	DW	0	; Instruction trace pointer
_STEP	DW	0	; Instruction step count
_BREAK	DW	02000h	; Hardware breakpoint address
_BRKSK	DB	0	; Breakpoint skip flag
DSEG	ENDS
;
; Uninitialized data
;
BSEG	SEGMENT	BYTE PUBLIC 'UDATA'
	PUBLIC	_TRCBUF, _SLOWCT, _INSTCT
_SLOWCT	DW	1 DUP(?)	; Slowdown delay loop value
_INSTCT	DW	1 dup(?)	; Instruction counter
_TRCBUF	DW	4096 DUP(?)	; Traceback buffer
BSEG	ENDS
	END
