/*
 * Virtual H8 - Resident Debugger
 *
 * ?COPY.TXT 2004-2008 Dave Dunfield
 */
struct WINDOW
	*awin,			// Assembly window
	*rwin,			// Register window
	*mwin;			// Memory window

unsigned
	Maddress = 0x2040, // Memory display address
	Daddress,		// Disassembler display address
	daddress[65];	// Disassembly window line address table

extern unsigned Mread(unsigned address);

/*
 * Table of instruction opcodes: MASK, COMPARE, TYPE/LENGTH, TEXT
 */
unsigned char itable[] = {
	0xFF, 0xFE, 0x02, 'C', 'P', 'I', ' ', 0,
	0xFF, 0x3A, 0x03, 'L', 'D', 'A', ' ', 0,
	0xFF, 0x32, 0x03, 'S', 'T', 'A', ' ', 0,
	0xFF, 0x2A, 0x03, 'L', 'H', 'L', 'D', ' ', 0,
	0xFF, 0x22, 0x03, 'S', 'H', 'L', 'D', ' ', 0,
	0xFF, 0xF5, 0x01, 'P', 'U', 'S', 'H', ' ', 'P', 'S', 'W', 0,
	0xFF, 0xF1, 0x01, 'P', 'O', 'P', ' ', 'P', 'S', 'W', 0,
	0xFF, 0x27, 0x01, 'D', 'A', 'A', 0,
	0xFF, 0x76, 0x01, 'H', 'L', 'T', 0,
	0xFF, 0xFB, 0x01, 'E', 'I', 0,
	0xFF, 0xF3, 0x01, 'D', 'I', 0,
	0xFF, 0x37, 0x01, 'S', 'T', 'C', 0,
	0xFF, 0x3F, 0x01, 'C', 'M', 'C', 0,
	0xFF, 0x2F, 0x01, 'C', 'M', 'A', 0,
	0xFF, 0xEB, 0x01, 'X', 'C', 'H', 'G', 0,
	0xFF, 0xE3, 0x01, 'X', 'T', 'H', 'L', 0,
	0xFF, 0xF9, 0x01, 'S', 'P', 'H', 'L', 0,
	0xFF, 0xE9, 0x01, 'P', 'C', 'H', 'L', 0,
	0xFF, 0xDB, 0x02, 'I', 'N', ' ', 0,
	0xFF, 0xD3, 0x02, 'O', 'U', 'T', ' ', 0,
	0xFF, 0x07, 0x01, 'R', 'L', 'C', 0,
	0xFF, 0x0F, 0x01, 'R', 'R', 'C', 0,
	0xFF, 0x17, 0x01, 'R', 'A', 'L', 0,
	0xFF, 0x1F, 0x01, 'R', 'A', 'R', 0,
	0xFF, 0xC6, 0x02, 'A', 'D', 'I', ' ', 0,
	0xFF, 0xCE, 0x02, 'A', 'C', 'I', ' ', 0,
	0xFF, 0xD6, 0x02, 'S', 'U', 'I', ' ', 0,
	0xFF, 0xDE, 0x02, 'S', 'B', 'I', ' ', 0,
	0xFF, 0xE6, 0x02, 'A', 'N', 'I', ' ', 0,
	0xFF, 0xF6, 0x02, 'O', 'R', 'I', ' ', 0,
	0xFF, 0xEE, 0x02, 'X', 'R', 'I', ' ', 0,
	0xFF, 0x00, 0x01, 'N', 'O', 'P', 0,
	/*  Jumps, Calls & Returns */
	0xFF, 0xC3, 0x0B, 'J', 'M', 'P', ' ', 0,
	0xFF, 0xCA, 0x43, 'J', 'Z', ' ', 0,
	0xFF, 0xC2, 0x4B, 'J', 'N', 'Z', ' ', 0,
	0xFF, 0xDA, 0x13, 'J', 'C', ' ', 0,
	0xFF, 0xD2, 0x1B, 'J', 'N', 'C', ' ', 0,
	0xFF, 0xEA, 0x23, 'J', 'P', 'E', ' ', 0,
	0xFF, 0xE2, 0x2B, 'J', 'P', 'O', ' ', 0,
	0xFF, 0xFA, 0x83, 'J', 'M', ' ', 0,
	0xFF, 0xF2, 0x8B, 'J', 'P', ' ', 0,
	0xFF, 0xCD, 0x0B, 'C', 'A', 'L', 'L', ' ', 0,
	0xFF, 0xCC, 0x43, 'C', 'Z', ' ', 0,
	0xFF, 0xC4, 0x4B, 'C', 'N', 'Z', ' ', 0,
	0xFF, 0xDC, 0x13, 'C', 'C', ' ', 0,
	0xFF, 0xD4, 0x1B, 'C', 'N', 'C', ' ', 0,
	0xFF, 0xEC, 0x23, 'C', 'P', 'E', ' ', 0,
	0xFF, 0xE4, 0x2B, 'C', 'P', 'O', ' ', 0,
	0xFF, 0xFC, 0x83, 'C', 'M', ' ', 0,
	0xFF, 0xF4, 0x8B, 'C', 'P', ' ', 0,
	0xFF, 0xC9, 0x05, 'R', 'E', 'T', 0,
	0xFF, 0xC8, 0x45, 'R', 'Z', 0,
	0xFF, 0xC0, 0x4D, 'R', 'N', 'Z', 0,
	0xFF, 0xD8, 0x15, 'R', 'C', 0,
	0xFF, 0xD0, 0x1D, 'R', 'N', 'C', 0,
	0xFF, 0xE8, 0x25, 'R', 'P', 'E', 0,
	0xFF, 0xE0, 0x2D, 'R', 'P', 'O', 0,
	0xFF, 0xF8, 0x85, 'R', 'M', 0,
	0xFF, 0xF0, 0x8D, 'R', 'P', 0,
	/*  Register based instructions */
	0xC0, 0x40, 0x01, 'M', 'O', 'V', ' ', 'd', ',', 's', 0,
	0xC7, 0x06, 0x02, 'M', 'V', 'I', ' ', 'd', ',', 0,
	0xF8, 0x90, 0x01, 'S', 'U', 'B', ' ', 's', 0,
	0xF8, 0x98, 0x01, 'S', 'B', 'B', ' ', 's', 0,
	0xF8, 0x80, 0x01, 'A', 'D', 'D', ' ', 's', 0,
	0xF8, 0x88, 0x01, 'A', 'D', 'C', ' ', 's', 0,
	0xF8, 0xA0, 0x01, 'A', 'N', 'A', ' ', 's', 0,
	0xF8, 0xB0, 0x01, 'O', 'R', 'A', ' ', 's', 0,
	0xF8, 0xA8, 0x01, 'X', 'R', 'A', ' ', 's', 0,
	0xF8, 0xB8, 0x01, 'C', 'M', 'P', ' ', 's', 0,
	0xC7, 0x04, 0x01, 'I', 'N', 'R', ' ', 'd', 0,
	0xC7, 0x05, 0x01, 'D', 'C', 'R', ' ', 'd', 0,
	/*  Register pair instructions */
	0xCF, 0x01, 0x03, 'L', 'X', 'I', ' ', 'p', ',', 0,
	0xEF, 0x0A, 0x01, 'L', 'D', 'A', 'X', ' ', 'p', 0,
	0xEF, 0x02, 0x01, 'S', 'T', 'A', 'X', ' ', 'p', 0,
	0xCF, 0x03, 0x01, 'I', 'N', 'X', ' ', 'p', 0,
	0xCF, 0x0B, 0x01, 'D', 'C', 'X', ' ', 'p', 0,
	0xCF, 0x09, 0x01, 'D', 'A', 'D', ' ', 'p', 0,
	0xCF, 0xC5, 0x01, 'P', 'U', 'S', 'H', ' ', 'p', 0,
	0xCF, 0xC1, 0x01, 'P', 'O', 'P', ' ', 'p', 0,
	/*  Restart instruction */
	0xC7, 0xC7, 0x01, 'R', 'S', 'T', ' ', 'v', 0,
	/*  This entry always matches invalid opcodes */
	0x00, 0x00, 0x01, 'D', 'B', ' ', 'o', 0 };

/* Tables to convert register index into actual names */
unsigned char regtab[]	= { 'B','C','D','E','H','L','M','A' };
unsigned char rptab[]	= { 'B','D','H','S' };

/*
 * Disassemble an 8080 instruction
 */
void disassemble(unsigned length, unsigned char display)
{
	unsigned a, i, l, ll, t;
	unsigned char c, o, *p;

	for(i=0; i < length; ++i) {
		if(daddress[i] == Daddress) {
			a = daddress[0];
			goto dodis; } }
	a = Daddress;
dodis:
	for(ll=0; ll < length; ++ll) {
		*awin = (a == Daddress) ? 0x36 : 0x67;
		o = Mread(daddress[ll] = a);
		p = itable;
again:	if((o & *p++) != *p++) {
			while(*p++);
				goto again; }
		l = *p++ & 3;
		if(display /*&& !Cmode*/) {
			w_gotoxy(0, ll, awin);
			w_printf(awin, "%04x", a);
			for(i=0; i < 3; ++i) {
				if(i < l)
					w_printf(awin, " %02x", Mread(a+i));
				else
					w_puts("   ", awin); }
			i = 0;
			w_puts("  ", awin);
			while(c = *p++) switch(c) {
			case 'v' :		/* Interrupt vector */
				w_putc(((o >> 3) & 7) + '0', awin);
				goto iplus;
			case 'p' :		/* Register PAIR */
				w_putc(c = rptab[(o >> 4) & 3], awin);
				if(c == 'S') {
					w_putc('P', awin);
					++i; }
				goto iplus;
			case 'd' :		/* Destination register */
				w_putc(regtab[(o >> 3) & 7], awin);
				goto iplus;
			case 's' :		/* Source register */
				w_putc(regtab[o & 7], awin);
				goto iplus;
			case 'o' :		/* Output opcode byte */
				i += w_printf(awin,"$%02x", o);
				break;
			case ' ' :		/* Separator */
				do
					w_putc(' ', awin);
				while(++i < 8);
				break;
			default:
				w_putc(c, awin);
			iplus:
				++i; }
			w_cleol(awin);
			switch(l) {
			case 2 :		/* Single byte operand */
				w_printf(awin, "%02x", Mread(a+1));
				break;
			case 3 :		/* Double byte operand */
				t = Mread(a+1);
				w_printf(awin, "%04x", (Mread(a+2) << 8)|t); } }
		a += l; }
}

/*
 * Dump a block of memory
 */
void memory(void)
{
	unsigned i, j, a;
	unsigned char c;
	a = Maddress;
	for(j=0; j < 16; ++j) {
		w_gotoxy(0, j, mwin);
		w_printf(mwin, "%04x ", a);
		for(i=0; i < 8; ++i)
			w_printf(mwin, " %02x", Mread(a+i));
		w_puts("  ", mwin);
		for(i=0; i < 8; ++i) {
			c = Mread(a++);
			w_putc(((c >= ' ') && (c < 0x7F)) ? c : '.', mwin); } }
}


/*
 * Update 8080 register display
 */
void dumpreg(void)
{
	unsigned i;
	unsigned char c;
	static char Fs[] = { 'S', 'Z', '?', 'A', '?', 'P', '?', 'C' };
	static char Fc[] = { 's', 'z', '-', 'a', '-', 'p', '-', 'c' };
	w_gotoxy(0, 0, rwin); w_cleol(rwin);
	w_printf(rwin, "\n A :%02x", A);			w_cleol(rwin);
	w_printf(rwin, "\n BC:%02x %02x", B, C);	w_cleol(rwin);
	w_printf(rwin, "\n DE:%02x %02x", D, E);	w_cleol(rwin);
	w_printf(rwin, "\n HL:%02x %02x", H, L);	w_cleol(rwin);
	w_printf(rwin, "\n SP:%04x", SP);			w_cleol(rwin);
	w_printf(rwin, "\n PC:%04x", PC);			w_cleol(rwin);
	w_printf(rwin, "\nPSW:%02x", c = PSW);		w_cleol(rwin);
	w_puts("\n ", rwin);
	for(i=0; i < 8; ++i) {
		w_putc( ((c & 0x80) ? Fs : Fc)[i], rwin);
		c <<= 1; }
	w_cleow(rwin);
}

/*
 * Hexidecimal input function
 */
int gethex(unsigned char in, unsigned *dest, unsigned length)
{
	unsigned c, v, l, a;
	unsigned char i, d;
	v = l = 0;
	i = in;
	for(;;) {
		if(!(c = i)) {
			wupdatexy();
			c = wgetc(); }
		d = c;
		i = 0;
		if((c >= '0') && (c <= '9'))
			c -= '0';
		else if((c >= 'A') && (c <= 'F'))
			c -= ('A' - 10);
		else if((c >= 'a') && (c <= 'f'))
			c -= ('a' - 10);
		else switch(c) {
			case '?' :
			case _K1 : help(0); continue;
			case 0x1B:
			case _K10: return 0;
			case 'A'-0x40: a = &A; goto x1;
			case 'B'-0x40: a = &B; goto x2;
			case 'C'-0x40: a = &C; goto x1;
			case 'D'-0x40: a = &D; goto x2;
			case 'E'-0x40: a = &E; goto x1;
			case 'H'-0x40: a = &H;
			x2:	if(length == 2) goto x1a;
			--a;
			x2a:*dest = *(unsigned*)a; return 255;
			case 'L'-0x40: a = &L;
			x1: if(length != 2) { xbeep(); continue; }
			x1a:*dest = *(unsigned char*)a; return 255;
			case 'F'-0x40: a = &PSW; goto x1;
			case 'S'-0x40: a = &SP; goto x3;
			case 'P'-0x40: a = &PC;
			x3:	if(length == 4) goto x2a;
				xbeep(); continue;			
			case _KBS:
				if(l) {
					--l;
					v >>= 4;
					wputc('\b');
					continue; }
			default:
				xbeep();
				if(in && !l)
					return 0;
				continue; }
		wputc(d);
		v = (v << 4) | c;
		if(++l >= length) {
			*dest = v;
			return 255; } }
}

/*
 * Interactive memory editor
 */
void edit_memory(void)
{
	unsigned xc, xh, y, c, s, v;
	static int offset;
	static char mode;

	status("Memory Editor: F1=help  F2=Address  F3=Mode  F10=Exit");
	W_OPEN = mwin;
	wcursor_line();
redraw:
	memory();
	for(;;) {
		if(offset < 0) {
			do {
				offset += s;
				Maddress -= s; }
			while(offset < 0);
			goto redraw; }
		if(offset > 127) {
			do {
				offset -= s;
				Maddress += s; }
			while(offset > 127);
			goto redraw; }
		y = offset >> 3;
		xc = offset & 7;
		xh = (xc * 3) + 6;
		xc += 31;
		wgotoxy(mode ? xc : xh, y);
		s = 1;
		switch(c = wgetc()) {
		case _KRA: ++offset;	s=1;	continue;
		case _KLA: --offset;	s=1;	continue;
		case _KUA: offset -= (s=8); 	continue;
		case _KDA: offset += (s=8);		continue;
		case _KPU: Maddress -= 128;	goto redraw;
		case _KPD: Maddress += 128;	goto redraw;
		case '?' : if(mode) break;
		case _K1 : help(HEDITM);	continue;
		case _K2:
			wgotoxy(0, 0);
			if(gethex(0, &Maddress, 4))
				offset = 0;
			goto redraw;
		case _K3:	mode = mode ? 0 : 255; continue;
		case 0x1B : if(mode) break;
		case _K10:	wcursor_off();	return; }
		if(c & 0xFF00) continue;
		if(!mode) {
			if(!gethex(c, &v, 2))
				goto redraw;
			c = v; }
		poke(SEG, offset++ + Maddress, c);
		wgotoxy(xh, y);
		wprintf("%02x", c);
		wgotoxy(xc, y);
		wputc( ((c < ' ') || (c > 0x7E)) ? '.' : c); }
}

/*
 * Interactive register editor
 */
void edit_register(void)
{
	unsigned x, y, v;
	unsigned char *bp;
	unsigned *wp;

redraw:
	W_OPEN = swin;
	dumpreg();
	for(;;) {
		status("Select Register: A B C D E H L Sp Pc psW  ESC=exit? ");
		wcursor_line(); wupdatexy();
		switch(toupper(wgetc())) {
		case 'A' : y = 1; x = 4; bp = &A; goto in1;
		case 'B' : y = 2; x = 4; bp = &B; goto in1;
		case 'C' : y = 2; x = 7; bp = &C; goto in1;
		case 'D' : y = 3; x = 4; bp = &D; goto in1;
		case 'E' : y = 3; x = 7; bp = &E; goto in1;
		case 'H' : y = 4; x = 4; bp = &H; goto in1;
		case 'L' : y = 4; x = 7; bp = &L; goto in1;
		case 'W' : y = 7; x = 4; bp = &PSW;
		in1: status("Enter value: ESC=cancel");
			W_OPEN = rwin;
			wgotoxy(x, y);
			if(gethex(0, &v, 2))
				*bp = v;
			goto redraw;
		case 'S' : y = 5; x = 4; wp = &SP; goto in2;
		case 'P' : y = 6; x = 4; wp = &PC;
		in2: status("Enter value: ESC=cancel");
			W_OPEN = rwin;
			wgotoxy(x, y);
			if(gethex(0, &v, 4))
				*wp = v;
			goto redraw;
		case '?' :
		case _K1 : help(HEDITR); continue;
		case 0x1B:
		case _K10: return; }
		xbeep(); }
}

/*
 * Edit pending IRQ flags
 */
void edit_irq()
{
	unsigned i;
	unsigned char PI;
	static unsigned char *names[] = {
	"Clock", "Monitor", "Console", "", "", "", "" };
	status("Pending IRQs: 1-7=Toggle F1=Help F2=Clear-all F3=Global ENTER=Save ESC=Cancel");
	wopen(27, 9, 20, 9, WSAVE|WCOPEN|WBOX1|0x17);
	PI = PIRQ;
	for(;;) {
		for(i=0; i < 7; ++i) {
			wgotoxy(1, i);
			wprintf("%u %c %s", i+1, ((1<<i)&PI) ? 'X' : ' ', names[i]); }
		switch(i=wgetc()) {
		case _K1: help(HPIRQ); continue;
		case _K2: PI = 0;	continue;
		case _K3: EI = EI ? 0 : 255; UP8080(); continue;
		case '\n': PIRQ = PI;
		case 0x1B: wclose(); return; }
		if((i >= '1') && (i <= '7'))
			PI ^= 1 << (i-'1'); }
}

/*
 * Perform an instruction traceback
 */
void trace(void)
{
	unsigned t, i, j, s, st;
	W_OPEN = swin;
home:
	st = j = TRCPTR / 2;
reshow:
	wcursor_off();
	j &= 4095;
	if(i = (st-j) & 4095)
		Daddress = s = TRCBUF[t = j];
	else {
		t = st;
		Daddress = s = PC; }
	status("TRACE %-4u: F1=Help  F2=SearchF  F3=SearchB  F4=NextF  F5=NextB  F10=Exit", i);
	disassemble(16, 255);
	for(;;) switch(wgetc()) {
	case '?' :
	case _K1 : help(HTRACE);			continue;
	case _K2 :
		status("Search Forward?"); i = 1;
		goto dosearch;
	case _K3 :
		status("Search Backward?"); i = -1;
	dosearch:
		wcursor_line();
		if(gethex(0, &s, 4)) {
dosr:			j = t;
			do {
				j = (j + i) & 4095;
				if(TRCBUF[j] == s)
					goto reshow; }
			while(j != st);
			beep(1000, 100); }
		j = t;
		goto reshow;
	case _K4 : i = 1; goto dosr;
	case _K5 : i = -1; goto dosr;
	case _KHO : j = st+1;				goto reshow;
	case _KEN : goto home;
	case _KUA : j = t - 1;				goto reshow;
	case _KDA : j = t + 1;				goto reshow;
	case _KPU : j = t - 100;			goto reshow;
	case _KPD :	j = t + 100;			goto reshow;
	case _K10 :
	case 0x1B : return;
	default: xbeep(); goto reshow; }
}

/*
 * Interactive debugger
 */
void debug(unsigned char *msg)
{
	unsigned i, j, k;
	unsigned char PANsave;
	struct WINDOW *Wsave;

	Wsave = W_OPEN;
	swin->WINorgy = 0;
	PANsave = Panel;
	Panel = 0xDB;
	draw_panel();
	awin = wopen(0, 9, 31, 16, WCOPEN|0x67);
	rwin = wopen(31,9, 10, 16, WCOPEN|0x57);
	mwin = wopen(41,9, 39, 16, WCOPEN|0x67);
	PIRQ &= ~1;
stop:
	Daddress = PC;
	sound_off();
reshow:
	disassemble(16, 255);
	dumpreg();
	memory();
	for(;;) {
		W_OPEN = mwin;
		wcursor_off();
		status("%-5s 1:Hlp 2:Adr 3:Trc 4:Mem 5:Reg 6:Brk 7:Rst 8:Irq 9:Tty 10:Exit SP:Stp EN:Go", msg);
		switch(i=wgetc()) {
		case _KDA: j = daddress[1];	goto dodis;	// Disassemble - down line
		case _KPD: j = daddress[15];			// Disassemble - down page
	dodis:	Daddress = daddress[0] = j;
	dodis1:	disassemble(16, 255);
			continue;
		case _KUA: i = 1; goto dotb;			// Disassemble - up line
		case _KPU: i = 15;						// Disassemble - up page
	dotb:	k = Daddress;
			Daddress = daddress[0] = (Daddress - 58);
			disassemble(64, 0);
			for(j = 15; j < 64; ++j) {
				if(daddress[j] >= k)
					break; }
			j = daddress[j-i];
			goto dodis;
		case _K1 : help(HDEBUG); 	continue;	// Help
		case _K2 :								// Disasm address
			W_OPEN = awin;
			wgotoxy(0, 0);
			wcursor_line();
			if(gethex(0, &Daddress, 4))
				daddress[0] = Daddress;
			wcursor_off();
			goto dodis1;
		case _K3 : trace();			continue;	// Traceback
		case _K4 : edit_memory();	continue;	// Edit memory
		case _K5 : edit_register();	continue;	// Edit registers
		case _K6 :								// Breakpoint
			status("Break=%04x ?", BREAK);
			W_OPEN = swin;
			wcursor_line();
			if(gethex(0, &j, 4))
				BREAK = j;
			continue;
		case _K7 :
			PC = 0; EI = 255;
			draw_panel();
			goto reshow;
		case _K8 : edit_irq(); continue;
		case _K9 :
		case _CHO:
			Panel = 0;
			Tcopy();
			if(i == _CHO)
				systat();
			swin->WINorgy = 24;
			status("TTY view: press any key to proceed");
			wgetc();
			draw_panel();
			swin->WINorgy = 0;
			goto reshow;
		case ' ' :								// Step
			STEP = 1;
			BRKSK = 255;
			switch(multi_step()) {
			case 0 : msg = "STEP";			goto stop;
			case 1 : msg = "HALT";			goto stop;
			case 2 : msg = "BREAK";			goto stop;
			case -1: msg = "?INST";			goto stop;
			case -2: msg = "?IO";			goto stop;
			default: msg = "???";			goto stop; }
		case '\n' :								// Execute
			wclose();
			wclose();
			wclose();
			swin->WINorgy = 24;
			W_OPEN = Wsave;
			if(Panel = PANsave)
				draw_panel();
			Tcopy();
			upstat();
			if(Cmode) wcursor_line(); else wcursor_off();
			BRKSK = 255;	// Skip breakpoint
			return;
		case _K10 :
			if(confirm("Exit H8 simulator")) {
				wclose();
				wclose();
				wclose();
				swin->WINorgy = 24;
				W_OPEN = Wsave;
				closeall();
				exit(0); }
			continue;
		default: xbeep(); } }
}
