/*
 * Virtual H8 - main module
 *
 * ?COPY.TXT 2004-2008 Dave Dunfield
 */
#include <stdio.h>
#include <window.h>
#include <file.h>
#define	TICK	peekw(0x40,0x6C)	// Access BIOS 55ms clock tick
//#define	STKCHK					// Enable stack monitoring
//#define	IODEBUG					// Debug disk system

// Front-Panel Digit segments
#define	SEG_M	0x01		// Middle line
#define	SEG_T	0x02		// Top line
#define	SEG_TR	0x04		// Top right bar
#define	SEG_BR	0x08		// Bottom right bar
#define	SEG_B	0x10		// Bottom line
#define	SEG_BL	0x20		// Bottom left bar
#define	SEG_TL	0x40		// Top left bar
#define	SEG_DOT	0x80		// Decimal point

// Constants for H17 emulation
#define	DRIVES	3			// Number of drives
#define	TRACKS	40			// Number of tracks/drive
#define	SECTORS	10			// Number of sectors/track
#define	SSIZE	256			// Size of each sector
#define	TTEMP	15			// Default TrackTemp counter

// Constants for select_file
#define	SFILES	110			// Number of files to allow
#define	SATTR	0x17		// File normal display attribute
#define	SHIGH	0x71		// File hilite display attribute

// Constants for HELP
#define	HTEXT	0x70		// Help text
#define HLINK	0x71		// Help link
#define	HLINKH	0x17		// Hilited link
#define	LINK_S	('N'-0x40)	// Start of a link
#define	LINK_E	('O'-0x40)	// End of a link
#define	TAB_SIZE 4			// Spacing of TABs
#define	LINKS	25			// Maximum # links per screen

// HELP file indexs
#define	HCOPT	1			// Command line option
#define	HPANEL	2			// Panel mode
#define	HTTY	3			// TTY mode
#define	HFILE	4			// File Selection
#define	HENTER	5			// Data entry
#define	HIOFILE	6			// I/O (mount) select
#define	HDEBUG	7			// Debug
#define	HEDITM	8			// Memory editor
#define	HEDITR	9			// Register editor
#define	HTRACE	10			// Traceback viewer
#define	HPIRQ	11			// IRQ editor

unsigned char
	OF0 = 0xDF,				// Front panel command output register
	Key = 0xFF,				// Front panel input
	Keyd,					// Key display
	Panel = 255,			// Panel displayed flag
	BadIO,					// Bad I/O detect flag
	Cmode,					// Console mode
	TrapHLT,				// Trap HALT
	Conctrl,				// Console control word
	Condata,				// Console input
	Conirq = 1<<2,			// Console IRQ
	Concurs = 255,			// Console cursor update flag
	Condel = 50,			// Console scroll delay
	Dnldctrl,				// Download control
	Dnldstat,				// Download status
	Owrite,					// Append to output files
	Rewind,					// Auto-rewind at EOF
	Display[9],				// Display value
	Directory[70],			// Opening directory
	Ifile[13],				// Input filename
	Ofile[13],				// Output filename
	Text[] = { ".H8T" },	// File extension
	Dext[] = { ".H8D" },	// Disk extension
	Temp[81],				// Temporary buffer
	Status,					// Status display flag
	X,						// Screen X position
	Y,						// Screen Y position
	*ptr;					// General pointer
unsigned
	Dnlddi,					// Download data in
	Keycount,				// Key duration counter
	Ltick,					// Last tick setting (18hz)
	Ftick,					// Fast tick (500hz)
	Icount,					// Count of interrupts
	Lic,					// Last interrupt count
	Speed = -1,				// instructions/sec x 1000
	Istep,					// instructions/interrupt
	screen[24][80];			// Copy of screen

struct WINDOW
	*pwin,					// Panel window
	*swin;					// Status window

FILE
	*Dnldfpi,				// Load file
	*Dnldfpo;				// Dump file

// Disk emulation variables
unsigned
	Drive,					// Currently selected drive
	Dsector,				// Current drive sector
	Dtick,					// Disk drive tick
	Tdrive,					// Drive track is from
	Ttrack,					// Track currently buffered
	Tdata,					// Track data pointer
	Tmodify,				// Track modify list
	Dselect,				// Selection list for mount
	Dsize[DRIVES+1],		// Size of drive
	Dtrack[DRIVES+1];		// Current drive track
unsigned char
	H17,					// H17 installed flag
	Dcmd,					// Drive last cmd
	Dcrc,					// Data CRC
	Dsync,					// Drive SYNC detected
	Dwp[DRIVES+1],			// Drive write protect status
	Dvol[DRIVES+1],			// Drive volume id
	Dfile[DRIVES][9],		// Drive filenames
	Tstate,					// Track access state
	Ttemp,					// Track access temp
	Tbuffer[SECTORS*SSIZE];	// Drive track buffer
HANDLE
	*Dfh[DRIVES+1];			// File pointers for drives

// Variables in 8080 processor emulation module
extern unsigned char
	A,						// Accumulator
	B,						// B register
	C,						// C register
	D,						// D register
	E,						// E register
	H,						// H register
	L,						// K register
	PSW,					// Processor Status Word
	EI,						// Interrupts enabled flags
	BRKSK,					// Break skip flag
	IBADOP,					// Ignore illegal operands
	PIRQ,					// IRQ pending flag
	LMWP[];					// Memory write protect table
extern unsigned
	SP,						// Stack pointer
	PC,						// Program counter
	SEG,					// Virtual 8080 memory segment
	IOSEG,					// I/O port handler segment
	STEP,					// Instruction step count
	BREAK,					// Breakpoint address
	SLOWCT,					// Instruction slowdown counter
	INSTCT,					// Instructions executed counter
	TRCPTR,					// Instruction trace pointer
	TRCBUF[];				// Trace buffer

// Functions in 8080 processor emulation module
int multi_step(void);			// Multi-instruction step
int FCALL(void);				// Far call to I/O handler
void setslow(unsigned count);	// Set slowdown value

void UP8080(void);				// IO: Update 8080 panel status
void Dflush(void);				// IO: Flush pending sector writes

// Required prototypes
void loadparm(char f);			// Load file of options
extern unsigned char *ARGV[];	// External access to program name

// Menu of drive select options
char *Dmenu[] = {
	"Drive 1 - RO",
	"Drive 1 - R/W",
	"Drive 2 - RO",
	"Drive 2 - R/W",
	"Drive 3 - RO",
	"Drive 3 - R/W",
	0 };

// PAM8 ROM image
unsigned char PAM8[] =
#include "pam8.h"

// H17 ROM image
unsigned char H17ROM[] =
#include "h17.h"

/*
 * Direct panel/screen & TTY update functions (high speed)
 */
unsigned
	Poffset;	// Current offset into screen
unsigned char
	Pattr;		// Current display attribute

asm {	// Required external definitinos
	EXTRN	_wtstc:NEAR
	EXTRN	_W_PAGE:byte
}

// Position panel/screen output pointer
void Pgotoxy(unsigned x, unsigned y)
{
	Poffset = ((y*80)+x)*2;
}

// Write character to panel/screen
void Pputc(c) asm
{
		MOV		ES,DGRP:_W_BASE
		MOV		BX,DGRP:_Poffset;
		MOV		AH,DGRP:_Pattr;
		MOV		AL,4[BP]
		MOV		ES:[BX],AX
		INC		BX
		INC		BX
		MOV		DGRP:_Poffset,BX
}

// Write string to panel/screen
void Pputs(s) asm
{
		MOV		ES,DGRP:_W_BASE
		MOV		BX,DGRP:_Poffset
		MOV		AH,DGRP:_Pattr
		MOV		SI,4[BP]
pputs1:	MOV		AL,[SI]
		AND		AL,AL
		JZ		pputs2
		MOV		ES:[BX],AX
		INC		SI
		INC		BX
		INC		BX
		JMP	SHORT pputs1
pputs2:	MOV		DGRP:_Poffset,BX
}

/*
 * Timed beep
 */
void xbeep(void)
{
	beep(1000, 100);
}

/*
 * Copy internal screen buffer to displayed portion of terminal window
 */
void Tcopy(void) asm
{
		MOV		ES,DGRP:_W_BASE		; Get video segment
		MOV		AL,DGRP:_Panel		; Get panel flag
		CMP		AL,0DBh				; Debug mode?
		JZ		Tcopy3				; No output
		AND		AL,AL				; Panel active?
		JZ		Tcopy1				; No, do it all
		MOV		SI,OFFSET DGRP:_screen+1600 ; Offset to screen
		MOV		DI,1600				; Starting location
		MOV		CX,1120				; Size to move
		JMP	SHORT Tcopy2			; And proceed
Tcopy1:	MOV		SI,OFFSET DGRP:_screen ;Source location
		XOR		DI,DI				; Video location
		MOV		CX,1920				; Size to move
Tcopy2:	REP		MOVSW				; Do them all
		MOV		BYTE PTR DGRP:_Concurs,255
Tcopy3:
}

/*
 * Scroll terminal screen & copy to window
 */
void Tscroll() asm
{
		PUSH	DS
		POP		ES
		MOV		SI,(OFFSET DGRP:_screen)+160
		MOV		DI,OFFSET DGRP:_screen
		MOV		CX,1840
		REP		MOVSW
		MOV		AX,0720h
		MOV		CX,80
		REP		STOSW
		CALL	_Tcopy
}

/*
 * Clear terminal screen & copy to window
 */
void Tclear() asm
{
		PUSH	DS
		POP		ES
		MOV		DI,OFFSET DGRP:_screen
		MOV		AX,0720h
		MOV		CX,1920
		REP		STOSW
		XOR		AL,AL
		MOV		DGRP:_X,AL
		MOV		AH,DGRP:_Panel
		AND		AH,AH
		JZ		Tclr1
		MOV		AL,10
Tclr1:	MOV		DGRP:_Y,AL
		CALL	_Tcopy
}

/*
 * Update terminal cursor X/Y position
 */
void Tupdatexy() asm
{
		MOV		DL,DGRP:_X
		MOV		DH,DGRP:_Y
		MOV		BH,DGRP:_W_PAGE
		MOV		AH,02h
		INT		10h
}

/*
 * Write a character to the terminal window
 */
void Tputc(unsigned c)
{
	switch(c) {
	case 8 :
		if(X) --X;
		goto xret;
	case '\r': X = 0; goto xret;
	case '\n':
		if(++Y >= 24) {
			if(Condel)
				delay(Condel);
			Tscroll();
			Y = 23; }
		goto xret;
	case 7 : xbeep(); return; }

	if((c < ' ') || (c > '~'))
		return;

	if(X >= 79) {	// Line wrap
		X=0;
		if(++Y >= 24) {	// Scroll
			Tscroll();
			Y=23; } }
	screen[Y][X] = (c |= 0x0700);
	if(Panel != 0xDB) {
		if((Y >= 10) || !Panel)
			pokew(W_BASE, (((unsigned)Y*80)+X)*2, c); }
	++X;
xret:
	Concurs = 255;
}

/*
 * Write string to terminal window
 */
void Tputs(unsigned char *s)
{
	while(*s) {
		if(*s == '\n')
			Tputc('\r');
		Tputc(*s++); }
}

/*
 * Formatted output to terminal window
 */
register Tprintf(unsigned args)
{
	unsigned char buffer[100];
	_format_(nargs()*2 + &args, buffer);
	Tputs(buffer);
}

/*
--------------------------------------------------------------------------------
         --   --   --     --   --   --     --   --   --     7     8     9    +
 O Ion  |  | |  | |  |   |  | |  | |  |   |  | |  | |  |      s  Load  Dump
 O Mon   --   --   --     --   --   --     --   --   --     4     5     6    -
 O Run  |  | |  | |  |   |  | |  | |  |   |  | |  | |  |      Go   In   Out
 O Pwr   -- . -- . -- .   -- . -- . -- .   -- . -- . -- .   1     2    3    *
        ----------- ADDRESS -----------   -- DATA/REG --      Sp   Af   BC Cncl
                                                            0     .    #    /
                                                                  Reg  Mem Altr
--------------------------------------------------------------------------------
*/

/*
 * Draw front panel digit
 */
void draw_digit(unsigned x, unsigned char d)
{
	unsigned char c, c1, c2;
	static unsigned char digitx[] = { 9, 14, 19, 26, 31, 36, 43, 48, 53 };
	if(Panel) {
		x = digitx[x];
		d = ~d;

		Pattr = 0x0C;
		c = c1 = c2 = (d & SEG_T) ? 0xDB : 0x20;
		Pgotoxy(x, 2);		Pputc((d & SEG_TL) ? c1 = 0xDB : 0x20);
		Pgotoxy(x+3, 2);	Pputc((d & SEG_TR) ? c2 = 0xDB : 0x20);
		Pgotoxy(x, 1); 		Pputc(c1); Pputc(c); Pputc(c); Pputc(c2);

		c = c1 = c2 = (d & SEG_B) ? 0xDB : 0x20;
		Pgotoxy(x, 4);		Pputc((d & SEG_BL) ? c1 = 0xDB : 0x20);
		Pgotoxy(x+3, 4);	Pputc((d & SEG_BR) ? c2 = 0xDB : 0x20);
		Pgotoxy(x, 5); 		Pputc(c1); Pputc(c); Pputc(c); Pputc(c2);

		c = c1 = c2 = (d & SEG_M) ? 0xDB : 0x20;
		if(d & (SEG_TL|SEG_BL)) c1 = 0xDB;
		if(d & (SEG_TR|SEG_BR)) c2 = 0xDB;
		Pgotoxy(x, 3); 		Pputc(c1); Pputc(c); Pputc(c); Pputc(c2);
		Pgotoxy(x+4, 5);	Pputc((d & SEG_DOT) ? 0xFE : 0x20); }
}

/*
 * Draw front panel key
 */
void draw_key(unsigned k, unsigned f)
{
	unsigned x, y;
	static unsigned char kx[] =
		{	60, 65, 70, 75,
			60, 65, 70, 75,
			60, 65, 70, 75,
			60, 65, 70, 75 };
	static unsigned char ky[] =
		{	1,  1,  1,  1,
			3,  3,  3,  3,
			5,  5,  5,  5,
			7,  7,  7,  7 };
	static unsigned char *kt1[] = {
		" 7 ", " 8 ", " 9 ", " + ",
		"4de", "5hl", "6pc", " - ",
		"1sp", "2af", "3bc", " * ",
		" 0 ", " . ", " # ", " / " };
	static unsigned char *kt2[] = {
		"   s", "Load", "Dump", "    ",
		"  Go", "  In", " Out", "    ",
		"    ", "    ", "    ", "Cncl",
		"    ", " Reg", " Mem", "Altr" };

	if(Panel) {
		Pgotoxy(x = kx[k], y=ky[k]); Pattr = f ? 0x1E : 0x07;
		Pputs(kt1[k]);
		Pgotoxy(x, y+1); Pattr = REVERSE; Pputs(kt2[k]); }
}

/*
 * Draw front panel light
 */
void draw_light(unsigned x, unsigned v)
{
	if(Panel) {
		Pgotoxy(2, x+2);
		Pattr = 0x0C;
		Pputc(v ? 0x04 : ' '); }
}

/*
 * Draw front panel
 */
void draw_panel(void)
{
	unsigned i;
	struct WINDOW *swin;
	Panel = 255;
	swin = W_OPEN;
	W_OPEN = pwin;
	*W_OPEN = REVERSE;
	for(i=0; i < 10; ++i) {
		wgotoxy(0,i);
		wcleol(); }
	wgotoxy(6, 8);
	*W_OPEN = 0x07; wputs(" Heathkit ");
	*W_OPEN = 0x47; wputs(" H8 Computer ");
	wopen(2, 1, 56, 6, WCOPEN|NORMAL);
	wgotoxy(2, 1); wputs("Ion");
	wgotoxy(2, 2); wputs("Mon");
	wgotoxy(2, 3); wputs("Run");
	wgotoxy(2, 4); wputs("Pwr");
	wgotoxy(7, 5); wputs("----------- ADDRESS -----------   -- DATA/REG --");
	wclose();
	*W_OPEN = 0x0C;
	for(i=0; i < 9; ++i)
		draw_digit(i, Display[i]);
	for(i=0; i < 16; ++i)
		draw_key(i, 0);
	draw_light(0, EI < 3);
	draw_light(1, OF0 & 0x20);
	draw_light(2, 255);
	draw_light(3, 255);
	W_OPEN = swin;
	Concurs = 255;
}

/*
 * Write formatted text into the status line
 */
register status(unsigned args)
{
	unsigned char buffer[81];
	_format_(nargs()*2 + &args, buffer);
	w_gotoxy(0, 0, swin);
	w_cleol(swin);
	w_puts(buffer, swin);
	Status = Concurs = 255;
}

/*
 * Update status line with default display information
 */
void upstat()
{
	w_gotoxy(Status = 0, 0, swin);
	w_printf(swin," F1:Help F2:Mode[%-5s] F3:Panel[%-3s] F4:Clear F5:Mount F9:Debug  F10:Exit",
		Cmode ? "TTY" : "PANEL",
		Panel ? "ON" : "OFF");
	w_cleol(swin);
	asm {
		MOV	ES,DGRP:_W_BASE
		MOV	AX,1004h
		MOV	BX,0F96h
		MOV	ES:[BX],AX
		MOV	ES:4[BX],AX
		MOV ES:8[BX],AX
	}
	Concurs = 255;
}

/*
 * Issue Y/N confirmation prompt on status line
 */
int confirm(unsigned char *msg)
{
	status("%s (Y/N)?", msg);
	for(;;) switch(wgetc()) {
		case 'y' :
		case 'Y' :
		case '\n': return 255;
		case 'n' :
		case 'N' :
		case 0x1B: return 0;
		default: xbeep(); }
}

/*
 * Close all open resources (prepare for exit)
 */
void closeall()
{
	unsigned i;
	sound_off();
	if(swin) wclose();
	if(pwin) wclose();
	if(Dnldfpo) fclose(Dnldfpo);
	if(Dnldfpi) fclose(Dnldfpi);
	for(i=1; i < 4; ++i) {
		if(Dfh[i])
			close(Dfh[i]); }
	asm " MOV DI,3";	// Simulator shutdown
	FCALL();			// Call I/O handler (if installed)
}

/*
 * Copy filename to Temp & append extension
 */
void filename(unsigned char *file, unsigned char *extension, unsigned char *n)
{
	unsigned char *d, *p, *e;
	d = Temp;
	e = 0;
	p = file;
	while(*d = *file++) {
		switch(*d++) {
		case '.' : e = file-1; continue;
		case '\\':
		case ':' : p = file; e = 0; } }
	if(e)
		*e = 0;
	else
		strcpy(d, extension);
	if(n)
		strcpy(n, p);
}

/*
 * Display a title on a window
 */
void wintitle(char *ptr)
{
	char buffer[6];
	memcpy(buffer, W_OPEN, 6);
	W_OPEN[1] &= (~WBOX3 >> 8);
	--W_OPEN[3];
	wgotoxy(0,0);
	wprintf(" %s ", ptr);
	memcpy(W_OPEN, buffer, 6);
}

/*
 * Build filename from program name + extension
 */
void mainname(unsigned char *ext)
{
	unsigned i, j;
	unsigned char *ptr;
	ptr = ARGV[i=j=0];
	while(*ptr) {
		if((Temp[i++] = *ptr++) == '.')
			j = i; }
	strcpy(Temp+j, ext);
}

/*
 * Display an error message
 * If simulator has not yet initialized, close resources & exit
 */
register error(unsigned args)
{
	unsigned char buffer[101];
	_format_(nargs()*2 + &args, buffer);
	if(!swin) {
		closeall();
		fputs(buffer, stdout);
		putc('\n', stdout);
		exit(-1); }
	*swin = 0x4E;
	w_gotoxy(0, 0, swin);
	w_cleol(swin);
	w_puts(buffer, swin);
	beep(500, 500);
	*swin = 0x17;
	Status = Concurs = 255;
}

/*
 * Help commnd - Implements "hypertext" screens
 */
void help(unsigned section)
{
	int c, lx, ly, ox, oy;
	unsigned i, j, size, ncount;
	unsigned char *ptr, *xs, *ys;
	FILE *fp;
	Concurs = 255;

	mainname("HLP");
	if(!(fp = fopen(Temp, "rb"))) {
		error("Can't access: %s", Temp);
		return; }

	Dflush(); Tdrive = 0;
	ys = (xs = Temp + LINKS) + LINKS;
	/* Locate the section in question */
lookup:
	size = getc(fp);
	size |= getc(fp) << 8;
	if(section--) {
		fseek(fp, 0, size, 1);
		goto lookup; }

	/* Draw the screen */
	wopen(5, 4, 70, 17, WSAVE|WCOPEN|WBOX1|HTEXT);
	wcursor_off();
	i = ncount = ptr = 0;
	while(Temp[i++] = getc(fp));
	wintitle(Temp);
	wgotoxy(0, 0);
	while(i++ < size) switch(c = getc(fp)) {
		case LINK_S :	/* Start link */
			ptr = (ncount*25)+Tbuffer; //names[ncount];
			xs[ncount] = W_OPEN->WINcurx;
			ys[ncount] = W_OPEN->WINcury;
			*W_OPEN = HLINK;
			break;
		case LINK_E :	/* End link */
			Temp[ncount++] = getc(fp);
			*ptr = ptr = 0;
			*W_OPEN = HTEXT;
			++i;
			break;
		case '\t' :		/* Horizontal TAB */
			do
				wputc(' ');
			while(W_OPEN->WINcurx % TAB_SIZE);
			break;
		case 0 :
			c = '\n';
		default:
			wputc(c);
			if(ptr)
				*ptr++ = c; }

	/* Allow user to select field & link to new screen */
	i = section = 0;
	for(;;) {
		wgotoxy(xs[section], ys[section]);
		*W_OPEN = HLINK;
		wputs((section*25)+Tbuffer); //names[section]);
		wgotoxy(xs[i], ys[i]);
		*W_OPEN = HLINKH;
		wputs(((section=i)*25)+Tbuffer); //names[section = i]);
		*W_OPEN = HTEXT;
		switch(c = wgetc()) {		/* User keystroke */
			case _KLA :				/* Left arrow - previous field */
				i = (i ? i : ncount) - 1;
				break;
			case _KRA :				/* Right arrow - next field */
				i = (i + 1) % ncount;
				break;
			case _KUA : ox = oy = -1000; goto dofind;
			case _KDA : ox = oy = 1000;
			dofind:
				size = i;
				for(j = 0; j < ncount; ++j) {
					lx = (int)xs[j] - (int)xs[i];
					ly = (int)ys[j] - (int)ys[i];
					if(c == _KUA) {
						if((ly >= 0) || (ly < oy)) continue; }
					else {
						if((ly <= 0) || (ly > oy))	continue; }
					if(abs(lx) > abs(ox)) continue;
					size = j;
					ox = lx;
					oy = ly; }
				i = size;
				break;
			case '\n' :				/* Enter - chain to menu */
				rewind(fp);
				section = Temp[i];
				wclose();
				goto lookup;
			case 0x1B:				/* Escape exit */
				wclose();
				fclose(fp);
				return;
			default: xbeep(); } }
}

/*
 * Skip ahead to non-blank in input line
 */
int skip()
{
	while(isspace(*ptr))
		++ptr;
	return *ptr;
}

/*
 * Extract a value from the input line
 */
unsigned get_value(unsigned base)
{
	unsigned c, v;
	unsigned char *p;
	p = ptr;
	v = 0;
	switch(skip()) {
	case '%' : base = 2;	goto xskip;
	case '@' : base = 8;	goto xskip;
	case '$' : base = 16;	goto xskip;
	case '.' : base = 10;
	xskip: ++ptr; }
	c = skip();
	do {
		if(isdigit(c))
			c -= '0';
		else if((c >= 'a') && (c <= 'f'))
			c -= ('a'-10);
		else if((c >= 'A') && (c <= 'F'))
			c -= ('A'-10);
		else
			c = 255;
		if(c >= base)
			error("Bad value: %s", p);
		v = (v * base) + c;
		++ptr; }
	while(c = skip());
	return v;
}

/*
 * Load a byte from the hex file
 */
unsigned load_byte(void)
{
	unsigned char c, v, ct;
	ct = 2;
	do {
		c = *ptr++;
		if((c >= '0') && (c <= '9'))
			c -= '0';
		else if((c >= 'A') && (c <= 'F'))
			c -= ('A'-10);
		else if((c >= 'a') && (c <= 'f'))
			c -= ('a'-10);
		else {
			printf("Line %u - Bad hex digit '%c'[%02x]\n", PC, c, c);
			exit(-1); }
		v = (v << 4) | c; }
	while(--ct);
	return v;
}

/*
 * Load I/O handler
 */
void load_io()
{
	unsigned i, c;
	FILE *fp;
	if(IOSEG)
		error("IO= already loaded");
	filename(ptr, ".H8X", 0);
	fp = fopen(Temp, "rvqb");
	IOSEG = alloc_seg(4096);
	i = 0; do {
		pokew(IOSEG, i, 0); }
	while(i += 2);
	while((c = getc(fp)) != EOF)
		poke(IOSEG, i++, c);
	fclose(fp);
}

/*
 * Read HEX file into memory image
 */
void load_file(char rom)
{
	unsigned a, b, chksum, addr, length, count, size, tf, LPC;
	unsigned char buffer[80];
	FILE *fp;

	filename(ptr, ".HEX", 0);
	fp = fopen(Temp, "rvq");

	printf("Loading: %s - ", Temp);
	for(size = PC = tf = 0; fgets(ptr = buffer, 80, fp);) {
		++PC;
		again: switch(*ptr++) {
			case 'S' :	/* Motorola HEX format */
				if(*ptr == '9') goto quitload;
				if(*ptr++ != '1') continue;
				length = count = (chksum = load_byte()) - 3;
				chksum += a = load_byte();
				chksum += b = load_byte();
				addr = (a << 8) + b;
				if(!size)
					LPC = addr;
				while(count--) {
					chksum += b = load_byte();
					poke(SEG, addr++, b); }
				if((255 & ~chksum) != load_byte()) {
					ptr = "Bad checksum"; goto quit; }
			tstrom:
				if((a < 0x20) && rom)
					LMWP[a] = 255;
				break;
			case ':' :		/* Intel HEX format */
				if(!(length = count = load_byte())) goto quitload;
				chksum = (a = load_byte()) + length;
				chksum += b = load_byte();
				addr = (a << 8) + b;
				chksum += load_byte();
				if(!size)
					LPC = addr;
				while(count--) {
					chksum += b = load_byte();
					poke(SEG, addr++, b); }
				if((255 & -chksum) != load_byte()) {
					ptr = "Bad checksum"; goto quit; }
				goto tstrom;
			case ' ' :		/* Space */
			case '\t' :		/* Tab */
				goto again;
			case '!' :		// Load direction
				while(*ptr) {
					if(tf < (sizeof(Tbuffer)-2))
						Tbuffer[tf++] = *ptr++; }
				if(tf < (sizeof(Tbuffer)-2))
					Tbuffer[tf++]= '\n';
			case ';' :		// Comment
			case 0 :		/* Null line */
				continue;
			default: ptr = "Bad format"; goto quit; }
		size += length;
		}
	ptr = "no EOF record";
quit:
	printf("Line %u - %s\n", PC, ptr);
	fclose(fp);
	exit(-1);
quitload:
	fclose(fp);
	printf("%u bytes at %04x\n", size, LPC);
}

/*
 * Get an IRQ value & validate
 */
unsigned getirq()
{
	unsigned v;
	v = get_value(10);
	if((v < 1) || (v > 7))
		error("IRQ must be 1-7");
	return 1 << (v-1);
}

/*
 * Handle command line parameters
 */
void cmdparm()
{
	unsigned i;
	unsigned char *p;
	switch(i = (toupper(*ptr++) << 8) | toupper(*ptr++)) {
	case '?' << 8:
	case '-?':
	case '/?': help(HCOPT);	exit(0);
	case '-O' :
	case '/O' : Owrite = 255;						return;
	case '-R' :
	case '/R' : Rewind = 255;						return;
	case '-T' :
	case '/T' : BadIO = 255;						return;
	case '-I' :
	case '/I' : IBADOP = 255;						return;
	case '-H' :
	case '/H' : TrapHLT = 255;						return;
	case '-P' :
	case '/P' : Panel = 0;							return;
	case 'I=' :
		filename(ptr, Text, Ifile);
		Dnldfpi = fopen(Temp, "rvqb");			return;
	case 'O=' :
		filename(ptr, Text, Ofile);
		Dnldfpo = fopen(Temp, Owrite ? "wvqb" : "wavqb");	return;
	case 'B=' : BREAK = get_value(16);			return;
	case 'L=' : load_file(0);					return; }

	if(*ptr++ == '=') switch(i) {
		case 'LR' : load_file(255);				return;
		case 'IO' :	load_io();					return;
		case 'WP' : i = get_value(16);
			if(i >= 0x20) error("WP=1st 8k only");
			LMWP[i] = 255;
			return;
		case 'TI' : Conirq = getirq();			return;
		case 'TE' : p = Text;	goto doext;
		case 'DE' : p = Dext;
	doext:	*++p = toupper(*ptr++);
			*++p = toupper(*ptr++);
			*++p = toupper(*ptr++);
			return;
		case 'W1' : i = 1; goto mountw;
		case 'W2' : i = 2; goto mountw;
		case 'W3' : i = 3;
	mountw:	p=0; goto mountx;
		case 'R1' : i = 1; goto mountr;
		case 'R2' : i = 2; goto mountr;
		case 'R3' : i = 3;
	mountr:	p=255;
	mountx:	filename(ptr, ".H8D", Dfile[i-1]);
			mount_drive(Temp, i, p);
			return;
		case 'XT' : Condel = get_value(10);			return;
		case 'XS' : Speed = get_value(10);			return;
		case 'XI' : Istep = get_value(10);			return;

	} filename(ptr-3, ".H8", 0);
	loadparm(255);
	return;
}

/*
 * Load parameter file
 */
void loadparm(char f)
{
	unsigned char buffer[100];
	FILE *fp;
	if(fp = fopen(Temp, f ? "rvq" : "r")) {
		printf("INIFILE: %s\n", Temp);
		while(fgets(ptr = buffer, sizeof(buffer)-1, fp)) {
			switch(skip()) {
			case ';' :
			case 0 : continue; }
			cmdparm(); }
		fclose(fp); }
}

#include "debug.c"

/*
 * Main H8 emulator program
 */
main(int argc, char *argv[])
{
	unsigned i;

#ifdef STKCHK
	memset(0xC000, 0x55, 0xFC00-0xC000);
#endif

	Directory[0] = get_drive()+'A';
	Directory[1] = ':';
	Directory[2] = '\\';
	getdir(Directory+3);

	// Initialize memory segment & install PAM8
	SEG = alloc_seg(4096);
	i = 0; do {	// Copy in PAM-8 monitor
		poke(SEG, i, PAM8[i]); }
	while(++i < sizeof(PAM8));
	do {		// Zero rest of segment
		pokew(SEG, i, 0x76); }
	while(i += 2);

	// Load H8.INI file if present
	mainname("INI");
	loadparm(0);

	// Process command line arguments
	for(i=1; i < argc; ++i) {
		ptr = argv[i];
		cmdparm(); }

	// Initialize I/O handler (if installed)
	asm " XOR DI,DI";
	if(i = FCALL()) {
		if(i != 0xFF00)
			error("IO init error %04x", i);
		asm {	// Patch main loop to include I/O handler in tick call
			MOV		AX,WORD PTR CS:[rnfp+1]
			SUB		AX,6
			MOV		WORD PTR CS:[rnfp+1],AX
		} }

	// Open main windows, draw panel & init termainal & status lines
	pwin = wopen(0, 0, 80, 24, WSAVE|WCOPEN|NORMAL);
	swin = wopen(0, 24, 80, 1, WSAVE|WCOPEN|0x17);
	if(Panel)
		draw_panel();
	Tclear();
	Tputs(Tbuffer);
	status(" H8 Emulator - Dave Dunfield - "#__DATE__" - F1=help");
	Status = 255;

	// Calibrate speed if required
	if(Speed) {
		if(Speed >= 10000) Speed = 500;
		if(!Istep) Istep = Speed*2;
		calibrate(); }
	else
		if(!Istep) Istep = 1000;

reset:	// Reset virtual CPU
	PC = 0;
	EI = 255;

retty:	// Reset TTY cursor (in case we reset from sub-function)
	if(Cmode) wcursor_line(); else wcursor_off();
	Tupdatexy();

	for(;;) { asm {	// Main emulation loop
rtop:	XOR		AX,AX				// Get zero
		MOV		ES,AX				// Address low memory
		MOV		AX,ES:[046Ch]		// Get Tick count
		CMP		AX,DGRP:_Ltick		// Has it changed?
		JZ		rntick				// No
		MOV		DGRP:_Ltick,AX		// Save new tick marker
		MOV		AX,DGRP:_Ftick		// Get first tick
		MOV		BX,AX				// Save for later
		SUB		AX,DGRP:_Icount		// Calc # interrupts
		MOV		DGRP:_Lic,AX		// Save interrupt count
		CMP		AX,27				// Are we right?
		JZ		rsok				// Speed it OK
		MOV		AX,DGRP:_SLOWCT		// Get slow count
		JB		rslow				// Speed is slow
		INC		AX					// Advance
		JMP		short rsslo			// And proceed
rslow:	CMP		AX,2				// At bottom
		JB		rsok				// Do not reset
		DEC		AX					// Reduce count
rsslo:	MOV		DGRP:_SLOWCT,AX		// Resave
rsok:	MOV		DGRP:_Icount,BX		// Save new inst count
rntick:	INC	WORD PTR DGRP:_Ftick	// Advance high tick count
		MOV		AX,DGRP:_Istep		// Get step count
		MOV		DGRP:_STEP,AX		// Set step count
rnfp:	CALL	_multi_step			// Perform code
		AND		AX,AX				// Success?
		JNZ		haltcpu				// Cpu has halted!
		MOV		AL,DGRP:_OF0		// Get control
		AND		AL,40h				// IRQ enabled?
		JZ		rnoi				// No
		MOV		AL,DGRP:_PIRQ		// Get pending
		OR		AL,1				// Set clock interrupt
		MOV		DGRP:_PIRQ,AL		// Resave
rnoi:	CALL	_wtstc				// Check for key
		AND		AX,AX				// Key available?
		JNZ		haltkey				// Key pressed
		MOV		AL,DGRP:_Cmode		// Console mode?
		AND		AL,DGRP:_Concurs	// Cursor update requested
		JZ		rtop				// No - keep going
		CALL	_Tupdatexy			// Update TTY cursor
		MOV	BYTE PTR DGRP:_Concurs,0 // Reset flag
		JMP	SHORT rtop				// And proceed
haltcpu:MOV		DGRP:_Temp+2,255	// Indicate CPU reset
		JMP		short haltgo		// And proceed
haltkey:MOV		DGRP:_Temp+2,0		// Indicate Key
haltgo: MOV	WORD PTR DGRP:_Temp,AX	// Save code
}
		if(Temp[2]) {
			switch(*(unsigned*)Temp) {
			case 1 : //debug("HLT");				continue;
				if((EI&0xFC)|TrapHLT) { debug("HLT"); continue; }
				++PC; PIRQ |= 1; EI=0;
				continue;
			case 2 : debug("BREAK");				continue;
			case -1: debug("?INST");				continue;
			case -2: debug("?IO");	}
			continue; }

		if(Status)
			upstat();

		switch(i = *(unsigned*)Temp) {	// Panel & TTY
			case _K3 :
				if(Panel) {
					Panel = 0;
					Tcopy(); }
				else
					draw_panel();
	restat:		upstat();
			case 0 :									continue;
			case _K4 : Tclear();						continue;
			case _K5 :
				wcursor_off();
				status(" D)rives%12s:  I)nput: %-17sO)utput: %-17s", Text+1, Ifile, Ofile);
				if(Dnldfpo) w_puts("C)lose", swin);
			K5loop: switch(toupper(wgetc())) {
				case '?' :
				case _K1 : help(HIOFILE);	goto K5loop;
				case 'D' :
					status("%s:", Dext+1);
					for(i=1; i < 4; ++i) {
						if(!Dfh[i]) { w_printf(swin, "%-25s", "  NoDrive"); continue; }
						w_printf(swin, "%7u%c%03u:%-12s",
							Dsize[i], Dwp[i] ? 'R' : 'W', Dvol[i], Dfile[i-1]); }
					if(!wmenu(30, 8, WSAVE|WBOX1|WCOPEN|0x17, Dmenu, &Dselect)) {
						if(selectfile(Dext)) {
							filename(Directory, Dext, Dfile[i = Dselect >> 1]);
							if(mount_drive(Temp, i+1, (Dselect&1) ? 0 : 255)) {
								/* status("Unable to open: %s\007", Temp); */
								goto retty; } } }
					goto ttystat;
				case 'I' :
					if(selectfile(Text)) {
						filename(Directory, Text, Ifile);
						if(Dnldfpi) fclose(Dnldfpi);
						if(!(Dnldfpi = fopen(Temp, "rb"))) {
							status("Unable to open: %s\007", Temp);
							goto retty; } }
					goto ttystat;
				case 'O' :
					if(selectfile(Text)) {
						filename(Directory, Text, Ofile);
						if(Dnldfpo) fclose(Dnldfpo);
						if(!(Dnldfpo = fopen(Temp, Owrite ? "wb" : "wab"))) {
							status("Unable to open: %s\007", Temp);
							goto retty; } }
				case 0x1B :
		ttystat:	upstat();
					goto retty;
				case 'C' :
					if(Dnldfpo) {
						fclose(Dnldfpo);
						*Ofile = Dnldfpo = 0;
						goto ttystat; }
				default: xbeep(); goto K5loop; }
			case _K9 :	debug("DEBUG");	continue;
			case _K10: if(confirm("Exit H8 simulator")) { closeall(); return; } goto restat;
#ifdef IODEBUG
			case _CEN: IOsetup();	break;
#endif
			case _CHO: systat(); }

		if(Cmode) {		// TTY only
			if(Concurs) {
				Tupdatexy();
				Concurs = 0; };
			switch(i) {
			case _K1 : help(HTTY);						continue;
			case _K2: wcursor_off(); Cmode = 0;			goto restat;
			case '\n' : i = '\r'; goto inkey;
			case _KBS : i = '\b'; goto inkey;
			case _KDL : i = 0x7F;
			default:
			inkey: if(!(i & 0xFF80)) {
				Condata = i;
				if(Conctrl & 0x02)
					PIRQ |= Conirq; }
				Concurs = 255; }
			continue; }

		switch(i) {		// Panel only
		case '0' : Key = 0xFE; i = 13;	goto ifkey;
		case '1' : Key = 0xFC; i = 9;	goto ifkey;
		case '2' : Key = 0xFA; i = 10;	goto ifkey;
		case '3' : Key = 0xF8; i = 11;	goto ifkey;
		case '4' : Key = 0xF6; i = 5;	goto ifkey;
		case '5' : Key = 0xF4; i = 6;	goto ifkey;
		case '6' : Key = 0xF2; i = 7;	goto ifkey;
		case '7' : Key = 0xF0; i = 1;	goto ifkey;
		case '8' : Key = 0xEF; i = 2;	goto ifkey;
		case '9' : Key = 0xCF; i = 3;	goto ifkey;
		case '=' :
		case _KKP:
		case '+' : Key = 0xAF; i = 4;	goto ifkey;
		case _KKM:
		case '-' : Key = 0x8F; i = 8;	goto ifkey;
		case '*' : Key = 0x6F; i = 12;	goto ifkey;
		case '/' : Key = 0x4F; i = 16;	goto ifkey;
		case '#' : Key = 0x2F; i = 15;	goto ifkey;
		case '.' : Key = 0x0F; i = 14;
		ifkey:
			if(Keyd)
				draw_key(Keyd-1, 0);
			draw_key((Keyd=i)-1, 255);
			Keycount = 5; continue;
		case 'r' :			// Reset CPU
		case 'R' :  goto reset;
		case 'm' :
		case 'M' :	Key = 0x2E; Keycount = 1; continue;
		case '?' :
		case _K1 : help(HPANEL); continue;
		case _K2 : Cmode = 255; wcursor_line(); goto restat; } }
}

/*
 * Select a file via window panel
 */
int selectfile(unsigned char *match)
{
	unsigned a, i, Nt, Pt;
	unsigned char name[13], *n[SFILES], pool[SFILES*9], *p;
	static unsigned char *Dip;

	// Find end of directory index
	if(Dip)
		*Dip = 0;
newdir:
	status("Select file: F1=Help  F2=ManualEntry");
	Dip = Directory; while(*Dip) ++Dip;
	if(*(Dip-1) == '\\')
		--Dip;
	*Dip = '\\';
	// Scan for all files & build name list
	strcpy(Dip+1, "*.*");
	Nt = Pt = 0;
	if(!find_first(Directory, DIRECTORY, p = name, &i, &i, &a, &i, &i)) do {
		if(Nt >= SFILES)
			break;
		if(a & DIRECTORY) {
			if(!strcmp(p, "."))
				continue;
			n[Nt++] = &pool[Pt];
			pool[Pt++] = 1;	// Insure sorts first!
			while(pool[Pt++] = *p++);
			continue; }
		while(*p) {
			if(!strcmp(p, match)) {
				*p = 0;	p = name;
				n[Nt++] = &pool[Pt];
				while(pool[Pt++] = *p++);
				break; }
			++p; } }
	while !find_next(p=name, &i, &i, &a, &i, &i);

	*Dip = 0;
	if(!Nt) goto manent;

	// Sort the name list
	for(i=0; i < Nt; ++i) {
		for(a=i+1; a < Nt; ++a) {
			if(strcmp(n[a], n[i]) < 0) {
				p = n[i];
				n[i] = n[a];
				n[a] = p; } } }
	// Convert directory pref
	for(i=0; i < Nt; ++i) {
		if(*(p = n[i]) != 1)
			break;
		while(p[1]) {
			*p = p[1];
			++p; }
		*p = '\\'; }

	i = ((Nt + 4) / 5) + 2;
	wopen(4, 12-(i/2), 71, i, WSAVE|WBOX1|WCOPEN|0x70);
	wintitle(Directory);
	wcursor_off();
	a = 0;
draw:
	for(i=0; i < Nt; ++i) {
		wgotoxy((i % 5) * 14, i / 5);
		*W_OPEN = (i == a) ? 0x07 : 0x70;
		wputs(n[i]); }
	*W_OPEN = 0x70;
	for(;;) switch(wgetc()) {
		case _KLA : if(a) { --a; goto draw; }
		case _KEN : a = Nt-1; goto draw;
		case _KRA : if(++a < Nt) goto draw;
		case _KHO : a = 0;	goto draw;
		case _KUA : if(a > 4) a -= 5; goto draw;
		case _KDA : if((a+5) < Nt) a += 5; goto draw;
		case '?' :
		case _K1  : help(HFILE); continue;
		case _K2  :
			wclose();
		manent:
			strcpy(pool, Directory); Pt=Dip;
			wclwin(); wputs("File?");
			*Dip = '\\'; *(Dip+1) = 0;
		manent1:
			switch(wgets(0x106, 0, p = Dip = Directory, sizeof(Directory)-1)) {
			case _K1: help(HENTER);
			default : goto manent1;
			case '\n' :
				while(i = *p++) switch(i) {
					case ':' :
					case '\\': Dip = p-1; }
				if(*(p-2) == '\\') goto newdir;
				return 255;
			case 0x1B :
				if(Nt) {
					strcpy(Directory, pool);
					Dip = Pt;
					goto newdir; }
				return 0; }
		case '\n' :
			*Dip = '\\';
			p = n[a]; while(p[1]) ++p;
			if(*p == '\\') {
				*p = 0;
				if(strcmp(p = n[a], "..")) {	// Not UP
					strcpy(Dip+1, p);
					wclose();
					goto newdir; }
				// Go up one directory
				while(Dip > Directory) {
					if(*--Dip == '\\') {
						*Dip = 0;
						break; } }
				wclose();
				goto newdir; }
			strcpy(Dip+1, n[a]);
			wclose();
			return 255;
		case 0x1B :
		case _K10 :
			wclose();
			return 0; }
}

#include "io.c"
/*
 * Calibrate CPU performance
 */
calibrate(void)
{
	unsigned t, t1, i, aj, s, sb;
	static char cal[] = { 0xEB, 0x01, 0x00, 0x00, 0x78, 0xC3, 0x00, 0x00 };
	unsigned char save[sizeof(cal)];

	if(!(s = (Speed + 9) / 18))
		return;

	for(i=PC=0; i < sizeof(cal); ++i) {
		save[i] = peek(SEG, i);
		poke(SEG, i, cal[i]); }

	sb = BREAK;
	BREAK = aj = 1000;

	wopen(15, 12, 50, 7, WSAVE|WBOX1|WCOPEN|REVERSE|WSCROLL);
	wgotoxy(9, 0); wputs(" >>> Calibrating CPU speed <<<");
	wgotoxy(10, 4); wputs(" --- Press ESC to bypass ---");

	setslow(1);
	t1 = TICK;
	while((t = TICK) == t1);	// Wait for clock to tick
	while(kbtst() != 0x1B) {
		t1 = t;
		INSTCT = 0;
		do {
			STEP = 1000;
			multi_step(); }
		while((t = TICK) == t1);
		wgotoxy(15, 2);
		wprintf("%-5u  %-5u  %-5u", INSTCT, SLOWCT, aj);
		if(INSTCT > s) {
			SLOWCT += aj;
			continue; }
		i = aj;
		if(aj /= 10) {
			SLOWCT = (SLOWCT > i) ? (SLOWCT - i) : 1;
			continue; }
		break; }
	BREAK = sb;
	wclose();
	for(i=PC=0; i < sizeof(cal); ++i)
		poke(SEG, i, save[i]);
	if(SLOWCT < 5) {
		Tprintf("This machine cannot maintain requested speed - throttle unlocked\n");
		setfast(); }
}

/*
 * Display system status information
 */
systat()
{
#ifdef STKCHK
	int i, l;
	ptr = i = l = 0; do {
		if(*ptr == 0x55)
			++i;
		else {
			if(i > l)
				l = i;
			i = 0; } }
	while(++ptr);
	Tprintf("\nS: %u %04x", l, &SEG);
#else
	Tprintf("\nS: %04x", &SEG);
#endif
	Tprintf("\nC: S=%u I=%u T=%u\nI: L=%u H=%u F=%u\n",
		Speed, Istep, SLOWCT, Ltick, Ftick, (Lic * 18) + 14);
	Tprintf("D: D=%u S=%u C=%02x Sy=%u ",
		Drive, Dsector, Dcmd, Dsync);
	Tprintf("T=%u %u %u V=%02x %02x %02x\n",
		Dtrack[1], Dtrack[2], Dtrack[3],
		Dvol[1], Dvol[2], Dvol[3]);
	Tprintf("T: D=%u T=%u Dp=%u M=%04x St=%u Tv=%u\n",
		Tdrive, Ttrack, Tdata, Tmodify, Tstate, Ttemp);
}
