/*
 * Virtual H8 - I/O device handlers
 *
 * ?COPY.TXT 2004-2008 Dave Dunfield
 */
#ifdef IODEBUG
	unsigned char IOflag;
	#define	IOTRK	0x01	// Display Disk track/sector operations
	#define	IOOUT	0x02	// Display port OUTPUT operations
	#define	IOIN	0x04	// Display port INPUT operations
	#define	IOSPEC	0x08	// Display special output
	IOsetup()
	{
		unsigned i, c;
		static char *names[] = {
			"Track",
			"Out",
			"In",
			"Special",
			0 };
		wopen(15, 10, 50, 10, WSAVE|WCOPEN|WBOX1|REVERSE);
		for(;;) {
			for(i=0; names[i]; ++i) {
				wgotoxy(1, i);
				wprintf("%u %c %s", i, ((1 << i) & IOflag) ? 'Y' : 'N', names[i]); }
			if((c = wgetc()) == 0x1B) {
				wclose();
				return; }
			c -= '0';
			if(c <  i)
				IOflag ^= 1 << c; }
	}
	register _IO_(unsigned args)
	{
		unsigned *ap;
		unsigned char buffer[100], *p, c;
		_format_(ap = (nargs() - 1) * 2 + &args, p=buffer);
		if(*ap & IOflag) {
			Tprintf("\n%04x:", PC);
			while(c = *p++) switch(c) {
			case 0x80: Tprintf(" L:%02x D:%u T:%u S:%u Ts:%u Tt:%u Td:%u\n",
				Dcmd, Drive, Dtrack[Drive], Dsector, Tstate, Ttemp, Tdata);
				continue;
			default: Tputc(c); } }
	}
	#define	IO(x)	_IO_ x;
#else
	#define	IO(x)
#endif

/* void xdelay(unsigned ms)
{
	if(IOflag) {
		delay(ms);
		switch(kbtst()) {
		case 0x1B :
			closeall();
			exit(0);
		case ' ' :
			while(kbtst() != '\r'); } }
} */

/*
 * Flush any pending sector writes
 */
void Dflush(void)
{
	unsigned o, l1[2], l2[2];
	HANDLE h;
	if(!Tdrive) return;
	if(!(h = Dfh[Tdrive])) return;
	o = 0;
	while(Tmodify & 0x3FF) {	// Modified sectors
		if(Tmodify & 1) {		// This one modified
			longset(l1, Ttrack);		// Current track
			longset(l2, SECTORS*SSIZE);	// Sectors/track
			longmul(l1, l2);			// Offset to track
			longset(l2, o);				// Offset into track
			longadd(l1, l2);			// Offset to sector
			IO((IOTRK, "Wsec: %u\x80", o))
			lseek(h, l1[1], l1[0], 0);	// Seek to track
			write(Tbuffer+o, SSIZE, h);}// Write the data
		o += SSIZE;
		Tmodify >>= 1; }
}

/*
 * Read new track into track buffer
 */
void Dload(void)
{
	unsigned l1[2], l2[2], t;
	HANDLE h;
	if(!Drive) return;				// No drive selected
	if(!(h = Dfh[Drive])) return;	// No disk mounted
	t = Dtrack[Drive];
	if((Drive == Tdrive) && (t == Ttrack)) return;
	IO((IOTRK, "Tload\x80"))
	Tdrive = Drive;
	longset(l1, Ttrack = t);
	longset(l2, SECTORS*SSIZE);
	longmul(l1, l2);
	lseek(h, l1[1], l1[0], Tmodify = 0);
	read(Tbuffer, SECTORS*SSIZE, h);
}

/*
 * Mount a disk image file on a logical drive
 */
int mount_drive(unsigned char *file, unsigned drive, unsigned char wp)
{
	unsigned h, t, s;
	unsigned char *p, sector[256];

	// Install H17 ROM & enable controller if not already so
	if(!H17) {
		h = 0; do {	// Copy in H17 ROM
			poke(SEG, h+0x1800, H17ROM[h]); }
		while(++h < 2048);
		memset(LMWP+0x18, H17 = 0xFF, 8); }

	Dflush();				// Insure updated
	if(h=Dfh[drive])		// Previous drive mounted
		close(h);
	Dwp[drive] = wp;
	Dtrack[drive] = Dsize[drive] = Dvol[drive] = t = Tdrive = 0;
	p = "Invalid image: %s";
	if(Dfh[drive] = h = open(file, wp ? F_READ : (F_READ|F_WRITE))) {
		while(s = read(sector, sizeof(sector), h)) {
			if(s != sizeof(sector)) goto fail;
			if(++t == 10)
				Dvol[drive] = *sector; }
		if((t % SECTORS) || !t) goto fail;
		Dsize[drive] = t / SECTORS;
		return 0; }
	p = "Can't open: %s";
fail:
	if(h) close(h);
	*Dfile[drive] = Dvol[drive] = Dfh[drive] = 0;
	error(p, file);
	return 255;
}

/*
 * Update the sector count and reset the state machine
 */
void bump_sector(void)
{
	if(++Dsector >= SECTORS)
		Dsector = 0;
	Ttemp = 5;
	Tstate = 0;
}

/*
 * Update track access state
 */
void update_tstate(void)
{
	if(!--Ttemp) {
		Ttemp = TTEMP;
		switch(++Tstate) {
		case 1:		Dtick = Ftick; return;		// Header: lead-in
		case 2:		Dsync = 255; return;		// Header: sync
//		case 3:		return;						// Header: volid
//		case 4:		return;						// Header: Trackno
//		case 5:		return;						// Header: Sector
//		case 6:		return;						// Header: Crc
//		case 7:		return;						// Data: Lead-in
		case 8:		Dsync = 255; return;		// Data: sync
		case 9:									// Data: data-bytes
			Tdata = Dsector * 256;
			Dcrc = 0;
			return;
//		case 10:	return;						// Data: crc
		case 11:
			if((Ftick - Dtick) < 10) {	// Was 5
				Tstate = 10;
				Ttemp = TTEMP;
				return; }
			bump_sector(); } }
}

/*
 * Read drive control register
 */
unsigned read_control(void)
{
	unsigned x;
	if(!Drive) return 0;
	x = Dtrack[Drive] ? 0 : 0x02;	// Track 0
	if(Dwp[Drive]) x |= 0x04;		// Write protect
	if(Dsync) x |= 0x08;			// Sync detected
	if(!Tstate)
		x |= 0x01;			// Hole detected
	else if((Dsector == 9) && (Tstate == 8) /* && (Ttemp == (TTEMP/2)) */)
		x |= 0x01;
	update_tstate();
	return x;
}

/*
 * Read drive status register
 */
unsigned read_status(void)
{
	switch(Tstate) {
	case 3 :	// Header: Volid
	case 4 :	// Header: Track
	case 5 :	// Header: Sector
	case 6 :	// Header: Crc
	case 9 :	// Header: Data
	case 10:	// Header: Crc
		update_tstate();
		return 0x81; }	// TX ready & Data avail
	update_tstate();
	return 0x80;		// TX ready
}

/*
 * Compute next check character for disk data
 */
unsigned xcrc(unsigned char x)
{
	Dcrc ^= x;
	if(Dcrc & 0x80)
		Dcrc = (Dcrc << 1) | 1;
	else
		Dcrc <<= 1;
	return x;
}

/*
 * Read data from drive
 */
unsigned read_data(void)
{
	unsigned char x;
	switch(Tstate) {
	case 3 : ++Tstate; Ttemp=TTEMP;	Dcrc = 0;					// Volid
		if(Dtrack[Drive]) return xcrc(Dvol[Drive]);
		return 0;
	case 4 : ++Tstate; Ttemp=TTEMP;	return xcrc(Dtrack[Drive]);	// Track
	case 5 : ++Tstate; Ttemp=TTEMP;	return xcrc(Dsector);		// Sector
	case 6 : ++Tstate; Ttemp=TTEMP;	Dload(); return xcrc(Dcrc);	// Crc
	case 9 :													// Data
		Ttemp = TTEMP;
		x = Tbuffer[Tdata];
		if(!(++Tdata & 255))
			++Tstate;
		return xcrc(x);
	case 10:
		bump_sector();
		return Dcrc; }		// End
}

/*
 * Write data to the drive
 */
void write_data(unsigned c)
{
	if(!(Dcmd & 0x01))
		debug("NWG");
	switch(Tstate) {
	case 1 :		// Header: lead-in
		if(c == 0xFD) { Tstate = 3; Ttemp = TTEMP; }
		return;
	case 2 :		// Header: SYNC
		if(c == 0xFD) { Tstate = 3; Ttemp = TTEMP; }
		return;
	case 3 :										// Volid
		if(Ttrack) Dvol[Tdrive] = c;
	case 4 : //++Tstate; Ttemp = TTEMP;	return;		// Header: Track
	case 5 : //++Tstate; Ttemp = TTEMP;	return;		// Header: Sector
	case 6 : ++Tstate; Ttemp = TTEMP;	return;		// Crc
	case 7 :		// Data: lead-in
		if(c == 0xFD) goto entdata;
		return;
	case 8 :		// Data: Sync
		if(c != 0xFD) return;
	entdata: Tdata = Dsector * 256;
		IO((IOSPEC, "Sector=%u Tdata=%u", Dsector, Tdata))
		Tstate = 9; Ttemp = TTEMP;
		Dload();
		return;
	case 9 :		// Data: Data
		Ttemp = TTEMP;
		Tbuffer[Tdata] = c;
		if(!(++Tdata & 255)) {
			Tmodify |= (1 << Dsector);
			IO((IOSPEC, "EOS %u %04x", Dsector, Tmodify))
			++Tstate; }
		return;
	case 10:		// Data: Crc
		bump_sector(); }
}

/*
 * Write a command to the drive
 */
void write_control(unsigned char cmd)
{
	unsigned char x;

	(cmd & 0x80) ? 0 : 255;		// Preset AL with Disk RAM WP state
	asm {
		MOV		BYTE PTR _LMWP+14h,AL
		MOV		BYTE PTR _LMWP+15h,AL
		MOV		BYTE PTR _LMWP+16h,AL
		MOV		BYTE PTR _LMWP+17h,AL
		MOV		ES,DGRP:_W_BASE
		MOV		DI,1004h;
		MOV		SI,1C04h;
	}
	switch(cmd & 0x0E) {
	default: asm {
		MOV ES:[0F96h],DI
		MOV	ES:[0F9Ah],DI
		MOV	ES:[0F9Eh],DI
		} Dflush();
		Drive = Dcmd = 0;
		return;
	case 0x02 :	asm {
		MOV ES:[0F96h],SI
		MOV ES:[0F9Ah],DI
		MOV ES:[0F9Eh],DI
		} x = 1; break;
	case 0x04 : asm {
		MOV ES:[0F96h],DI
		MOV ES:[0F9Ah],SI
		MOV ES:[0F9Eh],DI
		} x = 2;	break;
	case 0x08 : asm {
		MOV ES:[0F96h],DI
		MOV ES:[0F9Ah],DI
		MOV ES:[0F9Eh],SI
		} x = 3;  }

	if(x != Drive) { Dflush(); Drive = x; }

	x = Dcmd ^ cmd; Dcmd = cmd;
	if(x & 0x10) { Tstate = Tdata = 0; Ttemp = TTEMP; }	// Motor ON/OFF
	if((cmd & x) & 0x40) {	// Step
		Dflush();
		if(cmd & 0x20) {		// Step IN
			if(Dtrack[Drive] < (Dsize[Drive]-1))
				++Dtrack[Drive]; }
		else {					// Step OUT
			if(Dtrack[Drive])
				--Dtrack[Drive]; } }
}

/*
 * Perform a virtual 8080 OUT operation
 */
int OUT8080(unsigned p)
{
	unsigned char x;

	switch(p) {
	case 0xF0 :	// Front panel control
		PIRQ &= ~0x01;	// Clear interrupt
		x = (A ^ OF0);	// Compute changed
		if(x & 0x80) {		// Buzzer
			if(A & 0x80)
				sound_off();
			else
				sound(1000); }
		if(x & 0x10) {		// Single-step interrupt
			if(A & 0x10)
				PIRQ &= ~0x02;
			else
				PIRQ |= 0x02; }
		if(x & 0x20)
			draw_light(1, A & 0x20);
		OF0 = A;
		return 0;
	case 0xF1 :		// Front panel segment
		x = OF0 & 0x0F;
		if((x >= 1) && (x <= 9)) {
			if(A != Display[--x])
				draw_digit(x, Display[x] = A); }
		return 0;
	case 0xFA :		// Console data
		Tputc(A);
		Concurs = 255;
outx0:	IO((IOOUT, "OUT %02x=%02x", p, A))
		return 0;
	case 0xFB :		// Console control
		Conctrl = A;
		goto outx0;
	case 0xF8 :		// Download data
		if(Dnldfpo)
			putc(A, Dnldfpo);
		goto outx0;
	case 0xF9 :		// Download control
		Dnldctrl = A;
		goto outx0; }

	if(H17) switch(p) {
		case 0x7C : write_data(A);
		case 0x7D :
		case 0x7E : goto outx0;
		case 0x7F : write_control(A); goto outx0; }

	asm {
		MOV	DX,4[BP]		; Get port
		MOV	DI,6			; Offset for OUT
	} if(FCALL()) goto outx0;
	IO((IOOUT, "OUT %02x=%02x", p, A))
	return BadIO;
}

/*
 * Perform a virtual 8080 IN operation
 */
IN8080(unsigned p)
{
	switch(p) {
	case 0xF0 :		// Front panel
		if(Keycount) {
			A = Key;
			if(!--Keycount) {
				if(Keyd)
					draw_key(Keyd-1, Keyd = 0); } }
		else
			A = 0xFF;
		return 0;
	case 0xFA :		// Console data
		A = Condata;
		Condata = 0;
		PIRQ &= ~Conirq;
inx0:	IO((IOIN, "IN %02x=%02x", p, A))
		return 0;
	case 0xFB :		// Console status
		A = Condata ? 0x87 : 0x85;
		goto inx0;
	case 0xF8 :		// Download data
		A = Dnlddi;
		Dnldstat = 0;
		goto inx0;
	case 0xF9 :		// Download status
		A = (Dnldfpi || Dnldfpo) ? 0x05 : 0x85;
		if(Dnldfpi) {
			if(!Dnldstat) {
				if((Dnlddi = getc(Dnldfpi)) == -1) {
					if(Rewind) {
						rewind(Dnldfpi);
						if((Dnlddi = getc(Dnldfpi)) != -1)
							goto diok; }
					fclose(Dnldfpi);
					*Ifile = Dnldfpi = 0;
					goto inx0; }
	diok:		Dnldstat = 255; }
			Dnldstat = (A |= 0x02); }
		goto inx0;
	}

	if(H17) switch(p) {
		case 0x7F : A = read_control();	goto inx0;
		case 0x7E : A = 0xFD; Dsync = 0; goto inx0;
		case 0x7D : A = read_status(); goto inx0;
		case 0x7C : A = read_data(); goto inx0; }

	asm {
		MOV	DX,4[BP]		; Get port
		MOV	DI,9			; Offset for IN
	} if(FCALL()) goto inx0;

	A = 0xFF;
	IO((IOIN, "IN %02x=%02x", p, A))
	return BadIO;
}

/*
 * Update front panel
 */
void UP8080()
{
	draw_light(0, EI < 3);
}
