/*
 * Virtual Horizon - main module
 *
 * ?COPY.TXT 2003-2008 Dave Dunfield
 */
#include <stdio.h>
#include <window.h>
#include <keys.h>
#include <file.h>
#include <comm.h>
//#define	MEMCHK				// Check on memory use

#define	TICK	peekw(0x40,0x6C)	// Access BIOS tick

// Window attributes
#define	DWIN	0x30		// Debug window
#define	RUN		0x17		// While running
#define	DLOFF	1004h		// Disk LED - OFF
#define	DLON	0x1C04		// Disk LED - ON
#define	HTEXT	0x70		// Help text
#define HLINK	0x71		// Help link
#define	HLINKH	0x17		// Hilited link

// Help file indexes
#define	HCMD	1			// Command line help
#define	HSTOP	2			// Stop mode commands
#define	HRUN	3			// Run mode commands
#define	HEDITM	4			// Memory editor
#define	HEDITR	5			// Register editor
#define	HTRACE	6			// Trace mode
#define	HFILE	7			// File entry
#define	HENTER	8			// Manual name entry

// Disk system parameters
#define	DTICK	50			// Disk TICK service interval
#define	DMOTOR	255			// Disk mtor on timeout
#define	DLEDS	3984		// Offset to disk LEDs
#define	SFILES	110			// Number of files to allow select

asm " EXTRN _W_PAGE:byte";
asm " EXTRN ?heap:near";

unsigned char
	*Dip,					// Active directory pointer
	Directory[70];			// Directory name buffer

struct WINDOW
	*swin,					// Save window
	*awin,					// Assembly window
	*rwin,					// Register window
	*mwin,					// Memory window
	*dwin;					// Debug window
unsigned char
	eflag = 255,			// Error message flag
	Bport = 255,			// Bad I/O port detected
	Finput,					// Input file flag
	Foutput,				// Output file flag
	Fconsole,				// Console file flag
	Iname[9],				// Input filename
	Oname[9],				// Output filename
	Cname[9],				// Console filename
	Tflag,					// TTY delay
	Mhx,					// Write format
	Cmode=1,				// Console mode flag
	Cdata,					// Console data
	Cdata1,					// Console RX data flag
	Fdata,					// Input file data
	Fdata1,					// Input file RX data flag
	*CPUname = "HORIZON",	// Name of CPU
	*ptr,					// General pointer
	cx,						// Console X position
	cy,						// Console Y position
	cdata[24][80];			// Console data buffer
unsigned
	Step = 1024,			// Steps/update
	Speed = 0,				// Simulation rate
	LPC = 0xFFFF,			// Load PC
	CFilter,				// Console input filter
	IFilter,				// Input file filter
	Daddress,				// Last disassembly address
	Maddress,				// Last memory address
	daddress[64];			// Disassembly address

// CPU module shared globals
extern unsigned char
	A, B, C, D, E, H, L, XA, XF, XB, XC, XD, XE, XH, XL,
	PSW, XF, EI, IM, I, R, DH,
	BRKSK;
extern unsigned
	IX,
	IY,
	SP,
	PC,
	SEG,
	STEP,
	BREAK,
	TRCPTR,
	SLOWCT,
	INSTCT,
	TRCBUF[];

FILE
	*fp,			// General file pointer
	*fpi,			// IO input file pointer
	*fpo,			// IO output file pointer
	*fpc;			// Console file pointer

static unsigned char SDROM[] = {		// Single-Density BOOT ROM
0x31,0x14,0x21,0x06,0x0A,0xC5,0x3E,0x59,0x32,0x00,0x20,0x32,0x03,0x20,0x01,
0x01,0x00,0x79,0x16,0x04,0x59,0x21,0x00,0x20,0xCD,0x1E,0xE9,0xC3,0x98,0xE9,
0xF5,0xE5,0xD5,0xC5,0x06,0xEB,0xCD,0xE0,0xE9,0x21,0xFF,0x34,0x09,0x7E,0xEE,
0x59,0xE5,0xCC,0x64,0xE9,0xE1,0xF1,0xCD,0x64,0xE9,0xC1,0xCD,0xCE,0xE9,0x3A,
0x30,0xEB,0xE6,0x0F,0xB8,0xC2,0x38,0xE9,0xE1,0x0D,0xFA,0x0A,0x20,0xC2,0x07,
0x20,0x06,0x8C,0x11,0x50,0xEB,0x0E,0x00,0x3A,0x10,0xEB,0xE6,0x04,0xC2,0xAE,
0xE9,0x05,0xC2,0x53,0xE9,0x3E,0x01,0xC3,0xAB,0xE9,0x57,0x96,0x72,0xC8,0x21,
0x1D,0xEB,0x4F,0xF2,0x7B,0xE9,0x2F,0x3C,0x4F,0x3A,0x10,0xEB,0xE6,0x01,0xC0,
0x21,0x1C,0xEB,0x7E,0x3A,0x09,0xEB,0xE3,0xE3,0x3A,0x08,0xEB,0x16,0x02,0xCD,
0xD0,0xE9,0x3A,0x10,0xEB,0xE6,0x01,0xCA,0x93,0xE9,0x0E,0x01,0x0D,0xC2,0x7C,
0xE9,0xC9,0xC1,0xCA,0x04,0x20,0x05,0xC2,0x05,0xE9,0xC3,0xA0,0xE9,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0xC1,0xB7,0xC9,0x41,0x1A,0x77,0xA8,0x07,0x47,
0x23,0x0D,0xC2,0xAF,0xE9,0x1A,0xA8,0xCA,0xC4,0xE9,0x78,0x3E,0x02,0xC3,0xAB,
0xE9,0xF1,0x3D,0xC8,0xF5,0xCD,0xCE,0xE9,0xC3,0x4C,0xE9,0x16,0x01,0x3A,0x14,
0xEB,0x3A,0x90,0xEB,0xE6,0x80,0xCA,0xD3,0xE9,0x15,0xC8,0xC3,0xD0,0xE9,0x3A,
0x90,0xEB,0xE6,0x10,0xC2,0xF0,0xE9,0x16,0x32,0xCD,0xD0,0xE9,0xC3,0xF5,0xE9,
0x3A,0x03,0x20,0xB9,0xC8,0x0A,0x79,0x32,0x03,0x20,0x16,0x0D,0xC3,0xD0,0xE9,
0x00 };

static unsigned char DDROM[] = {		// Double-Density BOOT ROM
0x0E,0x0A,0x3A,0x15,0xEB,0x16,0x30,0x21,0x0D,0xE8,0xC3,0xD3,0xE8,0x3A,0x01,
0xEA,0x21,0x16,0xE8,0xC3,0x4D,0xE8,0x06,0x0C,0x21,0x1E,0xE8,0xC3,0xD1,0xE8,
0x3A,0x10,0xEB,0xE6,0x40,0xC2,0x2D,0xE8,0x05,0xC2,0x18,0xE8,0xC3,0x2A,0xE8,
0x3A,0x21,0xEA,0x3A,0x31,0xEA,0x3A,0x21,0xEA,0xC3,0x4A,0xE8,0x3A,0x20,0xEB,
0xE6,0x01,0xC2,0x52,0xE8,0x3A,0x01,0xEA,0x3A,0x11,0xEA,0x3A,0x01,0xEA,0x21,
0x39,0xE8,0x16,0x02,0xC3,0xD3,0xE8,0x21,0x58,0xE8,0xC3,0xD1,0xE8,0x3A,0x35,
0xEB,0xE6,0x0F,0xFE,0x04,0xC2,0x52,0xE8,0x3A,0x10,0xEB,0xE6,0x04,0xCA,0x62,
0xE8,0x3E,0x09,0x3D,0xC2,0x6C,0xE8,0x3A,0x10,0xEB,0xE6,0x20,0xC2,0x97,0xE8,
0x3A,0x21,0xEA,0x3A,0x31,0xEA,0x3A,0x21,0xEA,0x21,0x87,0xE8,0xC3,0x4D,0xE8,
0x21,0x8D,0xE8,0xC3,0xD1,0xE8,0x3A,0x35,0xEB,0xE6,0x0F,0xFE,0x08,0xC2,0x87,
0xE8,0x06,0x8C,0x11,0x40,0xEB,0x3A,0x10,0xEB,0x0F,0xDA,0xAE,0xE8,0x05,0xC2,
0x9C,0xE8,0x0D,0xC2,0x2D,0xE8,0xC3,0xAB,0xE8,0x1A,0x67,0x2E,0x01,0x77,0x07,
0x47,0x1A,0x77,0xA8,0x07,0x47,0x2C,0xC2,0xB5,0xE8,0x24,0x1A,0x77,0xA8,0x07,
0x47,0x2C,0xC2,0xBF,0xE8,0x1A,0xA8,0xC2,0xA7,0xE8,0x25,0x2E,0x0A,0xE9,0x16,
0x01,0x3A,0x11,0xEB,0x3A,0x10,0xEB,0xB7,0xF2,0xD6,0xE8,0x15,0x3A,0x11,0xEB,
0xC2,0xD3,0xE8,0xE9,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xE1,
0xE9 };

static unsigned L10[2] = { 10, 0 };			// Fixed long 10
static unsigned L256[2] = { 256, 0 };		// Fixed long 256
static unsigned L115200[2] = { 0xC200, 1 };	// Fixed long 115200

unsigned char *Dmenu[] = {				// Disk mount menu
	"Drive 1 - RO",
	"Drive 1 - R/W",
	"Drive 2 - RO",
	"Drive 2 - R/W",
	"Drive 3 - RO",
	"Drive 3 - R/W",
	"Drive 4 - RO",
	"Drive 4 - R/W",
	0,			// Output file options
	0,			// Input file options
	0,			// Console option-1
	0,			// Console option-2
	0 };
unsigned char *IFmenu[] = {				// Input file menu
	"No filtering",
	"Remove LF",
	"Remove CR",
	"Remove LF, CR=CRLF",
	"Remove CR, LF=CRLF",
	0 };

unsigned char *Tspeed[] = { "Fast  ", "Medium", "Slow  ", "V-slow" };
unsigned Tdelay[] = { 0, 111, 333, 1000 };

// Disk system globals
unsigned char
	Dstate,			// Disk system state
	Dcheck,			// Disk check character
	Dtrack[5],		// Current track
	Dsector,		// Current sector
	Dmotor,			// Motors on
	Dselect,		// Disk select
	Dsflag,			// Disk sector flag
	Dstep,			// Step direction
	Dfreeze,		// Freeze during read/write
	Dwindow,		// Window indicator
	Dbuffer[512],	// Sector buffer
	Dname[4][9],	// Disk names
	Ddd[5],			// Double density
	Dwrite[5];		// Write enable flags
unsigned
	Dorder,			// Side select
	Dssize,			// Disk sector size
	Dsync,			// Disk sync
	Dp;				// Disk buffer pointer
HANDLE
	Dhandle[5];	// Virtual Disk handles

//FILE *dfp;		// Log file for disk debugging

unsigned XT1, XT2, XT3, XT4;		// CPU/Disk timing counters
unsigned char
	XT5,
	DBtick;							// Debug display update flag

extern unsigned char *ARGV[];		// Global argument access
extern unsigned Longreg[];			// For long-math library

// Functions in CPU emulation module
int multi_step(void);
void setslow(unsigned count);
unsigned Mread(unsigned address);

// Functions in I/O module
void defineport(void);
void initio(void);

// Prototypes required for this module
void help(unsigned section);

/*
 * Output disk debug information whenever it changes
 */
/*register debug(unsigned args)
{
	unsigned buf1[30], buf2[60];
	static char xbuf1[30], xbuf2[60];
	_format_(nargs() * 2 + &args, buf2);
	sprintf(buf1, "%u.%u.%u.%u ", Dselect, Dtrack[Dselect], Dsector, Dp);
	if(strcmp(buf1, xbuf1) || strcmp(buf2, xbuf2)) {
		strcpy(xbuf1, buf1);
		strcpy(xbuf2, buf2);
		fputs(buf1, dfp);
		fputs(buf2, dfp);
		putc('\n', dfp); }
}*/

/*
 * Clear the disk activity LEDs
 */
void clrdiskled(void) asm
{
		MOV		ES,DGRP:_W_BASE
		MOV		BX,DLEDS
		MOV		AX,DLOFF
		CMP		BYTE PTR DGRP:_DH,0EAh
		JZ		cld1
		MOV		ES:[BX],AX
cld1:	MOV		ES:4[BX],AX
		MOV		ES:8[BX],AX
		MOV		ES:12[BX],AX
}

/*
 * Disk timer tick - service background functions
 */
void disktick(void)
{
	if(Dfreeze) {
		if(--Dfreeze)
			return; }
	Dsflag = 0x80;
	Dwindow = 3;
	if(!Dstate)
		Dstate = 1;
	if(++Dsector > 9)
		Dsector = 0;
	Dcheck = Dp = 0;
	if(Dmotor) {
		if(!--Dmotor) {
			clrdiskled();
			Dorder = Dselect = Dstate = 0; } }
}

/*
 * Read a sector from the disk controller hardware
 */
unsigned diskreadSD(unsigned a)
{
	unsigned char c, r;
	unsigned s[2], t[2];
	static char rs;		// Last status sector

	r = 0;				// Default return of zero
	if((a & 0xFF00) == 0xEA00) {
		if(Dstate < 100)
			return 0;
		if(Dstate < 116) {
			++Dstate;
			Dp = Dcheck = 0;
			return 0; }
		if(Dstate == 116)
			Dfreeze = 10;
		Dbuffer[Dp++] = a;
		if(Dp == 256) {
			if(Dwrite[Dselect]) {		// Not write protected
				longset(s, Dtrack[Dselect]);
				longmul(s, L10);
				longset(t, rs /* Dsector */);
				longadd(s, t);
				longmul(s, L256);
				if(*t = Dhandle[Dselect]) {
					lseek(*t, s[1], s[0], 0);
					write(Dbuffer, 256, *t); } }
			Dstate = Dp = 0; }
		return; }

	if(a & 0x40) {	// Read data
		if(Dstate == 1) {		// Reading
			if(Dp < 256) {
				if(!Dp) {	// Read sector from file
					Dfreeze = 10;
					longset(s, Dtrack[Dselect]);
					longmul(s, L10);
					longset(t, rs /* Dsector */);
					longadd(s, t);
					longmul(s, L256);
					if(*t = Dhandle[Dselect]) {
						lseek(*t, s[1], s[0], 0);
						read(Dbuffer, 256, *t); }
					Dcheck = 0; }
				r = c = Dbuffer[Dp++];
				c ^= Dcheck;
				Dcheck = (c << 1) | (c >> 7); }
			else if(Dp == 256) {
				STEP = 1;
				r = Dcheck;
				Dp = 500; } } // }
		if(a & 0x80)
			Dmotor = DMOTOR;
		return r; }

	rs = Dsector;
	Dfreeze = 0;

	if(a & 0x20) {	// Read B status
		r = Dsflag | Dsector;
		if(Dmotor)
			r |= 0x10; }
	else {						// Read A status
		r = Dsflag;
		if(Dmotor)				// Motors ON
			r |= 0x10;
		if(Dstate >= 100)		// Write mode
			r |= 0x08;
		else if(Dstate) {		// Read Body
			if(Dhandle[Dselect])
				r |= 0x04; }
		if(!Dwrite[Dselect])	// Write protect
			r |= 0x02;
		if(!Dtrack[Dselect])	// Track 0
			r |= 0x01; }
	switch(a & 0x1C) {		// Command
	case 0x00 :				// 000 - Select
		clrdiskled();
		if(Dselect = a & 3)
			pokew(W_BASE, ((unsigned)Dselect*4)+DLEDS, DLON);
		break;
	case 0x04 :				// 001 - Write
		Dstate = 100;
		break;
	case 0x08 :				// 010 - Step
		if(a & 1) {
			if(Dstep) {
				if(Dtrack[Dselect] < 34)
					++Dtrack[Dselect]; }
			else {
				if(Dtrack[Dselect])
					--Dtrack[Dselect]; } }
		break;
	case 0x0C :				// 011 - Interrupt armed
	case 0x10 :				// 100 - No operation
		break;
	case 0x14 :				// 101 - Reset sector flag
		Dsflag = 0;
		break;
	case 0x18:				// 110 - Reset controller
		Dselect = Dstate = Dsflag = 0;
		Dmotor = 1;
		break;
	case 0x1C:				// Load step direction
		Dstep = a & 1; }

	if(a & 0x80)
		Dmotor = DMOTOR;
	return r;
}

/*
 * Read a sector from the disk controller hardware
 */
unsigned diskreadDD(unsigned a)
{
	unsigned char c, r;
	unsigned s[2], t[2], i;
	static char rs;		// Last status sector

//debug("DR:%04x", a);
	r = 0;				// Default return of zero
	switch(a & 0xFF00) {
	case 0xE900 :					// Write data
		if(Dstate < 100)
			return 0;
		if(Dstate < Dsync) {
			++Dstate;
			Dp = Dcheck = 0;
			return 0; }
		if(Dstate == Dsync)
			Dfreeze = 10;
		Dbuffer[Dp++] = a;
		if(Dp == Dssize) {
			if(Dwrite[Dselect]) {		// Not write protected
				i = Dtrack[Dselect];
				if(Dorder & 0x40)
					i = (34-i)+35;
				longset(s, i);
				longmul(s, L10);
				longset(t, rs /* Dsector */);
				longadd(s, t);
				longset(t, Dssize);
				longmul(s, t);
				if(*t = Dhandle[Dselect]) {
//fprintf(dfp, "W: %u %u %04x:%04x\n", Dtrack[Dselect], rs, s[1], s[0]);
					lseek(*t, s[1], s[0], 0);
					write(Dbuffer, Dssize, *t); } }
			Dstate = Dp = 0; }
		return 0;

	case 0xEA00 :						// controller orders
//debug("order: %04x", a);
		i = a ^ Dorder;
		Dorder = a;
		if(i & 0x10) {					// Step
			if(a & 0x10) {				// Positive edge
				if(a & 0x20) {
					if(Dtrack[Dselect] < 34)
						++Dtrack[Dselect]; }
				else {
					if(Dtrack[Dselect])
						--Dtrack[Dselect]; } } }
		if(i & 0x0F) {					// Drive select
			clrdiskled();
			switch(a & 0x0F) {
			case 1 : Dselect = 1;	break;
			case 2 : Dselect = 2;	break;
			case 4 : Dselect = 3;	break;
			case 8 : Dselect = 4;	break;
			default: Dselect = 0; }
			if(Dselect) {
				Dssize = 256; Dsync = 116;
				if(Ddd[Dselect]) {
					Dssize = 512;
					Dsync = 133; }
				pokew(W_BASE, ((unsigned)Dselect*4)+(DLEDS-4), DLON); } }
		return 0; }

	// If it gets here, it's a standard controller access at EB00

	switch(a & 0x0F) {		// Command code
	case 1 : Dsflag = 0;		break;		// Reset sector flag
	case 5 : Dmotor = DMOTOR;	break;		// Motors on
	case 6 : Dstate = 100;		break;		// Begin write
	case 7 : Dselect = Dstate = Dsflag = 0;	// Reset controller
			 Dmotor = 1; }

	if((a & 0xF0) == 0x40) {		// Read data
		if(Dstate == 1) {		// Reading
			if(Dp < Dssize) {
				if(!Dp) {	// Read sector from file
					Dfreeze = 10;
					i = Dtrack[Dselect];
					if(Dorder & 0x40)
						i = (34-i)+35;
					longset(s, i);
					longmul(s, L10);
					longset(t, rs /* Dsector */);
					longadd(s, t);
					longset(t, Dssize);
					longmul(s, t);
//debug("seek: rs=%u %04x.%04x", rs, s[1], s[0]);
					if(*t = Dhandle[Dselect]) {
//fprintf(dfp,"R: %u %u %04x:%04x\n", Dtrack[Dselect], rs, s[1], s[0]);
						lseek(*t, s[1], s[0], 0);
						read(Dbuffer, Dssize, *t); }
					Dcheck = 0; }
				r = c = Dbuffer[Dp++];
				c ^= Dcheck;
				Dcheck = (c << 1) | (c >> 7); }
			else if(Dp == Dssize) {
				STEP = 1;
				r = Dcheck;
				Dp = 532; } } // }
//debug("CMD:%04x RD:%04x", a, r);
		return r; }

	rs = Dsector;
	Dfreeze = 0;

	r = Dsflag;						// Sector detected
//	if(Dsector == 1) r |= 0x40;		// Index detect
	if(Dsector == 0) r |= 0x40;		// Index detect
	if(Ddd[Dselect]) r |= 0x20;		// Double density
	if(Dmotor) r |= 0x10;			// Motors are ON

	if(Dwindow)
		--Dwindow;
	switch(a & 0xF0) {
	case 0x10 :			// Read A status
		if(Dwindow) r |= 0x80;
		if(Dstate < 100) r |= 0x04;					// Read enabled
		if(Dstate) if(Dhandle[Dselect]) r |= 0x01;	// Body
//debug("CMD:%04x SA:%04x", a, r);
		return r;
	case 0x20 :			// Read B status
		if(Dstate >= 100) r |= 0x08;				// Write
		if(!Dwrite[Dselect]) r |= 0x02;				// Write Protect
		if(!Dtrack[Dselect]) r |= 0x01;				// Track 0
//debug("CMD:%04x SB:%04x", a, r);
		return r;
	case 0x30 :
//debug("CMD:%04x SC:%04x", a, r|Dsector);
		return r | Dsector; }
//debug("CMD:%04x", a, r|Dsector);
	return 0;
}

/*
 * Reset virtual CPU
 */
void reset(void)
{
	memset(&A, EI = IM = Dorder = Dselect = Dstate = 0, 20);
	if(DH == 0xEA) {
		copy_seg(SEG, PC = 0xE900, get_ds(), SDROM, 256);
		XT3 = 0xE9A0; }
	else {
		copy_seg(SEG, PC = 0xE800, get_ds(), DDROM, 256);
		XT3 = 0xE8AB; }
	if(LPC == 0xFFFF)
		LPC = PC;
	PC = LPC;
}

#include "dz80.c"

/*
 * Dump a block of memory
 */
void memory(void)
{
	unsigned i, j, a;
	unsigned char c;
	if(!Cmode) {
		a = Maddress;
		for(j=0; j < 16; ++j) {
			w_gotoxy(0, j, mwin);
			w_printf(mwin, "%04x ", a);
			for(i=0; i < 8; ++i)
				w_printf(mwin, " %02x", Mread(a+i));
			w_puts("  ", mwin);
			for(i=0; i < 8; ++i) {
				c = Mread(a++);
				w_putc(((c>=' ') && (c < 0x7F)) ? c : '.', mwin); } } }
}

/*
 * Update 8080 register display
 */
void dumpreg(void)
{
	unsigned i;
	unsigned char c;
	static char Fs[] = { 'S', 'Z', '?', 'H', '?', 'P', 'N', 'C' };
	static char Fc[] = { 's', 'z', '-', 'h', '-', 'p', 'n', 'c' };
	if(!Cmode) {
		W_OPEN = rwin;
		wgotoxy(0, 0);
		wprintf("  A:%02x", A);
		wprintf("\n BC:%02x %02x", B, C);
		wprintf("\n DE:%02x %02x", D, E);
		wprintf("\n HL:%02x %02x", H, L);
		wprintf("\n IX:%04x", IX);
		wprintf("\n IY:%04x", IY);
		wprintf("\n SP:%04x", SP);
		wprintf("\n PC:%04x", PC);
		wprintf("\n  F:%02x\n ", c = PSW);
		for(i=0; i < 8; ++i) {
			wputc( ((c & 0x80) ? Fs : Fc)[i]);
			c <<= 1; }
		wprintf("\n'AF:%02x %02x", XA, XF);
		wprintf("\n'BC:%02x %02x", XB, XC);
		wprintf("\n'DE:%02x %02x", XD, XE);
		wprintf("\n'HL:%02x %02x", XH, XL);
		wprintf("\n IR:%02x %02x", I, R);
		wprintf("\nint:%c  %u", EI ? 'E' : 'D', IM);
		W_OPEN = dwin; }
}

/*
 * Update console X/Y position
 */
void cupdatexy() asm
{
		MOV		DX,WORD PTR DGRP:_cx ; Get X/Y
		MOV		BH,DGRP:_W_PAGE	; Get display page
		MOV		AH,02h			; Set cursor function
		INT		10h				; Call DOS
}

/*
 * Open the console
 */
void open_console()
{
	unsigned x, y, i;
	y = i = 0;
	if(!Cmode) {
		disassemble(16, 255);
		memory();
		w_clwin(rwin); dumpreg();
		y = 16;
		i = 2560; }
	do {
		for(x=0; x < 80; ++x) {
			pokew(W_BASE, i, (unsigned)cdata[y][x] | 0x0700);
			i += 2; } }
	while(++y < 24);
	wcursor_line();
	cupdatexy();
}

/*
 * Close the console
 */
void close_console(void)
{
	if(Cmode) {
		Cmode = 0;
		open_console(); }
}

/*
 * Perform newline delay if enabled
 */
void newline(void)
{
//	if(Tflag)
		delay(Tdelay[Tflag]);
}

/*
 * Scroll console
 */
void cscroll(void)
{
	unsigned i;

	memcpy(cdata, cdata+80, sizeof(cdata)-80);
	memset(&cdata[22][80], ' ', 80);

	if(Cmode) 
		copy_seg(W_BASE, 0, W_BASE, 160, 23*160);
	else
		copy_seg(W_BASE, 2560, W_BASE, 2720, 7*160);
	for(i=3680; i < 3840; i+=2)
		poke(W_BASE, i, ' ');
}

/*
 * Write a character to the console
 */
void cputc(unsigned char c)
{
	unsigned i, x, y;
	unsigned char *p, *pl;
	static unsigned char cstate = 255;

	// Handle leading cursor position
	switch(cstate) {
	default: cstate = 0;	// Unknown
	case 0 : break;			// Nothing active
	case 1 :				// Escape received
		cstate = 0;
		p = (pl = cdata[cy]) + cx;
		switch(c) {
		default: goto norchar;
		case '=' :	cstate = 2; return;
//case 'Z' : p = cdata; for(i=0; i < 1920; ++i) *p++ = (i & 0x3F) + ' '; break;
		case 'T' :	// Erase to end of line
			memset(p, ' ', 80-cx);
			break;
		case 'Y' :	// Erase to end of page
			i = ((unsigned)cy * 80) + cx;
			memset(p, ' ', 1920 - i);
			break;
		case 'W' :	// Delete character
			if(cx < 79) memcpy(p, p+1, 79-cx);
			cdata[cy][79] = ' ';
			break;
		case 'Q' :	// Insert charcter
			if(cx < 79) memmove(p+1, p, 79-cx);
			*p = ' ';
			break;
		case 'R' :	// Delete a line
			if(cy < 23) memcpy(pl, pl + 80, (23-cy) * 80);
			memset(cdata[23], ' ', 80);
			break;
		case 'E' :	// Insert a line
			if(cy < 23) memmove(pl+80, pl, (23-cy)*80);
			memset(cdata[cy], ' ', 80); }
		// Copy updated screen buffer to video memory
		y = i = 0;
		if(!Cmode) { y = 16; i = 2560; }
		do {
			for(x=0; x < 80; ++x) {
				pokew(W_BASE, i, (unsigned)cdata[y][x] | 0x0700);
				i += 2; } }
		while(++y < 24);
		return;
	case 2:		// 1st lead-in
		if((c < ' ') || (c > (' '+23))) {
			cstate = 0;
			break; }
		cstate = c - (' ' - 3);
		return;
	default:	// 2nd lead-in
		if((c < ' ') || (c > (' '+79))) {
			cstate = 0;
			break; }
		cy = cstate - 3;
		cx = c - ' ';
		cstate = 0;
		goto upxy;
	case 255 :
		if(!Cmode) {
			cy = 16;
			switch(c) {
			case 'K'-0x40 :
			case 'Z'-0x40 :
			case '^' - 0x40:
			case 8 :
				return; } }
		cstate = 0; }

norchar:
	switch(c) {
	case '\r' :				// Carriage return
		cx = 0;
		goto upxy;
	case '\n' :				// Line-feed
		newline();
		if(cy < 23) {
			++cy;
			goto upxy; }
		cscroll();
		goto upxy;
	case 'K'-0x40:			// Up line
		if(cy)
			--cy;
		goto upxy;
	case 'G'-0x40:			// Beep
		beep(1000, 100);
		return;
	case 'L'-0x40:			// Forward space
		goto advance;
	case 'Z'-0x40:			// Clear screen
		memset(cdata, ' ', sizeof(cdata));
		i = Cmode ? 0 : 2560;
		do {
			pokew(W_BASE, i, 0x0720); }
		while((i += 2) < 3840);
	case '^'-0x40:			// Home Cursor
		cx = cy = 0;
		if(!Cmode) cy = 16;
		goto upxy;
	case 0x1B :				// Cursor position
		cstate = 1;
		return;
	case 8 :				// Backspace
		if(cx)
			--cx;
		goto upxy; }

	if((c < ' ') || (c >= 0x7F))
		return;

	cdata[cy][cx] = c;
	if(Cmode || (cy > 15))
		poke(W_BASE, (cy*160)+(cx*2), c);

advance:
	if(++cx > 79) {
		cx = 0;
		newline();
		if(cy >= 23)
			cscroll();
		else
			++cy; }

upxy:	cupdatexy();
}

/*
 * Display error message
 */
register error(unsigned args)
{
	unsigned char emsg[81];
	_format_(nargs() * 2 + &args, emsg);
	if(!dwin) {
		fputs(emsg, stderr);
		putc('\n', stderr);
		exit(-1); }
	*dwin = DWIN; wclwin(); wputs(emsg);
	eflag = 255;
}

/*
 * Display status line
 */
register status(unsigned args)
{
	char buffer[81];
	_format_(nargs() * 2 + &args, buffer);
	if(!dwin) {
		fputs(buffer, stdout);
		putc('\n', stdout);
		return; }
	wclwin();
	wputs(buffer);
}

/*
 * "wgetc" function with background clock update
 */
#define	wgetct	wgetc

/*
 * Hexidecimal input function
 */
int gethex(unsigned char in, unsigned *dest, unsigned length)
{
	unsigned c, v, l;
	unsigned char i, d;
	v = l = 0;
	i = in;
	for(;;) {
		if(!(c = i)) {
			wupdatexy();
			c = wgetct(); }
		d = c;
		i = 0;
		if((c >= '0') && (c <= '9'))
			c -= '0';
		else if((c >= 'A') && (c <= 'F'))
			c -= ('A' - 10);
		else if((c >= 'a') && (c <= 'f'))
			c -= ('a' - 10);
		else switch(c) {
			case '?' :
			case _K1 : help(0); continue;
			case 0x1B:
			case _K10: return 0;
			case _KBS:
				if(l) {
					--l;
					v >>= 4;
					wputc('\b');
					continue; }
			default:	
				beep(1000, 250);
				if(in && !l)
					return 0;
				continue; }
		wputc(d);
		v = (v << 4) | c;
		if(++l >= length) {
			*dest = v;
			return 255; } }
}

/*
 * Interactive memory editor
 */
void edit_memory(void)
{
	unsigned xc, xh, y, c, s, v;
	static int offset;
	static char mode;

	close_console();
	status("Memory Editor: F1=help  F2=Mode  F3=Address  F10=Exit");
	W_OPEN = mwin;
	wcursor_line();
redraw:
	memory();
	for(;;) {
		if(offset < 0) {
			do {
				offset += s;
				Maddress -= s; }
			while(offset < 0);
			goto redraw; }
		if(offset > 127) {
			do {
				offset -= s;
				Maddress += s; }
			while(offset > 127);
			goto redraw; }
		y = offset >> 3;
		xc = offset & 7;
		xh = (xc * 3) + 6;
		xc += 31;
		wgotoxy(mode ? xc : xh, y);
		s = 1;
		switch(c = wgetct()) {
		case _KRA: ++offset;	s=1;	continue;
		case _KLA: --offset;	s=1;	continue;
		case _KUA: offset -= (s=8); 	continue;
		case _KDA: offset += (s=8);		continue;
		case _KPU: Maddress -= 128;	goto redraw;
		case _KPD: Maddress += 128;	goto redraw;
		case '?' : if(mode) break;
		case _K1 : help(HEDITM);	continue;
		case _K2:	mode = mode ? 0 : 255; continue;
		case _K3:
			wgotoxy(0, 0);
			if(gethex(0, &Maddress, 4))
				offset = 0;
			goto redraw;
		case 0x1B : if(mode) break;
		case _K10:	W_OPEN = dwin; wcursor_off();	return; }
		if(c & 0xFF00) continue;
		if(!mode) {
			if(!gethex(c, &v, 2))
				goto redraw;
			c = v; }
		poke(SEG, offset++ + Maddress, c);
		wgotoxy(xh, y);
		wprintf("%02x", c);
		wgotoxy(xc, y);
		wputc( ((c < ' ') || (c > 0x7E)) ? '.' : c); }
}

/*
 * Interactive register editor
 */
void edit_register(void)
{
	unsigned c, x, y, v;
	unsigned char *bp;
	unsigned *wp;
	struct REG {
		unsigned char	name;
		unsigned char	y;
		unsigned char	x;
		unsigned		addr; } *r;
	static struct REG regs[23] = {
	char 'A', 0, 4, int &A,
	char 'B', 1, 4, int &B,
	char 'C', 1, 7, int &C,
	char 'D', 2, 4, int &D,
	char 'E', 2, 7, int &E,
	char 'H', 3, 4, int &H,
	char 'L', 3, 7, int &L,
	char 'X', 4, 0, int &IX,
	char 'Y', 5, 0, int &IY,
	char 'S', 6, 0, int &SP,
	char 'P', 7, 0, int &PC,
	char 'F', 8, 4, int &PSW,
	char 'A'|0x80, 10, 4, int &XA,
	char 'F'|0x80, 10, 7, int &XF,
	char 'B'|0x80, 11, 4, int &XB,
	char 'C'|0x80, 11, 7, int &XC,
	char 'D'|0x80, 12, 4, int &XD,
	char 'E'|0x80, 12, 7, int &XE,
	char 'H'|0x80, 13, 4, int &XH,
	char 'L'|0x80, 13, 7, int &XL,
	char 'I',14, 4, int &I,
	char 'R',14, 7, int &R }

	close_console();
redraw:
	W_OPEN = dwin;
	dumpreg();
	for(;;) {
		status("Register: A B C D E H L X Y Sp Pc F ' I R ESC=exit? ");
		wcursor_line(); wupdatexy();
		switch(c = toupper(wgetct())) {
		case '?' :
		case _K1 : help(HEDITR); continue;
		case 0x1B:
		case _K10: return;
		case '\'':
			status("Alternate register: A F B C D E H L ?");
			wupdatexy();
			c = toupper(wgetct()) | 0x80; }
		for(v=0; v < (sizeof(regs)/sizeof(struct REG)); ++v) {
			if((r = regs[v])->name == c)
				goto found; }
		beep(1000, 250);
		continue;
found:	bp = wp = r->addr;
		y = r->y;
		status("Enter value: ESC=cancel");
		W_OPEN = rwin;
		if(x = r->x) {		// Byte input
			wgotoxy(x, y);
			if(gethex(0, &v, 2))
				*bp = v;
			goto redraw; }
		wgotoxy(4, y);
		if(gethex(0, &v, 4))
			*wp = v;
		goto redraw; }
}

/*
 * Step a block of 8080 instructions
 */
int dostep(unsigned s)
{
	unsigned i, t;

	if(!(t = s)) {
		*dwin = RUN;
		//      0....*....1....*....2....*....3....*....4....*....5....*....6....*....
		status("Run: F1=Help  F2=%s  F3=Stop  F4=Reset  F5=Clear  F6=%s",
			Cmode ? "Debug" : "Tty  ", Tspeed[Tflag]);
		clrdiskled();
		if(Dmotor && Dselect) {		// Drive active
			i = (DH == 0xEA) ? DLEDS : DLEDS-4;
			pokew(W_BASE, ((unsigned)Dselect*4)+i, DLON); }
		cupdatexy(); }
	INSTCT = XT3 = XT4 = XT5 = 0;
	BRKSK = 255;	// Skip first breakpoint
top:
	if(i = kbtst()) switch(i) {
		case _F1 : help(HRUN); break;
		case _F2 :					// Toggle console mode
			wgotoxy(17, 0);
			Cmode = Cmode ? 0 : 1;
			wputs(Cmode ? "Debug" : "Tty  ");
			open_console();
			break;
		case _F3 :	error("Stop");				return 1;
		case _F4 :	reset(); error("Reset");	return 1;
		case _F5 : cputc(0x1A);					break;
		case _F6 :					// Toggle TTY speed
			wgotoxy(56, 0);
			wputs(Tspeed[Tflag = (Tflag + 1) & 3]);
			cupdatexy();
			break;
		case _DEL : i = 0x7F;
		default:
			if(i < 256) {
				Cdata = i;
				Cdata1 = 255; } }

	if(s) {
		STEP = (t < Step) ? t : Step;
		t -= STEP; }
	else {
		t = 255;
		STEP = Step; }

	if(!Cmode) {
		asm " MOV ES,DGRP:_W_BASE";
		if(DBtick) {
			DBtick = 0;
			Daddress = PC;
			disassemble(16, 255);
			dumpreg();
			memory();
			cupdatexy(); } }

	asm {
; Wait for clock tick for housekeeping functions
		XOR		AX,AX			; Seg 0
		MOV		ES,AX			; Set ES
		MOV		AX,WORD PTR ES:[046Ch] ; Read tick
		CMP		AX,DGRP:_XT1	; Changed?
		JZ		diskt1			; No
		MOV		DGRP:_XT1,AX	; New value
; Tick occured - first update clock
		AND		AL,0FEh			; Mask 1s
		MOV		DGRP:_DBtick,AL	; Set tick
; Update instruction rate
		MOV		AL,DGRP:_XT5	; Get update flag
		AND		AL,AL			; Test
		JZ		newsp			; First tick
		MOV		AX,DGRP:_INSTCT	; Get current speed
		XOR		BX,BX			; Get zero
		MOV		DGRP:_INSTCT,BX	; Reset counter
		MOV		BX,DGRP:_SLOWCT	; Get slow count
		CMP		AX,DGRP:_Speed	; Compare to desired
		JC		spdup			; Must go faster
		INC		BX				; Slow down a little
		JMP		SHORT savspd	; And continue
newsp:	MOV		AX,DGRP:_Speed	; Get speed setting
		OR		AL,AH			; Test for zero
		MOV		DGRP:_XT5,AL	; Save
		JMP		SHORT nospd		; And proceed
spdup:	CMP		BX,2			; Can we go faster?
		JC		nospd			; No - stay here
		DEC		BX				; Speed up a little
savspd:	MOV		DGRP:_SLOWCT,BX	; Resave it
; Update disk controller value
nospd:	MOV		AX,DGRP:_XT2	; Get count
		XOR		DX,DX			; Get zero
		MOV		BX,3			; / 3
		IDIV	BX				; AX = AX / 3
		MOV		DGRP:_XT3,AX	; Save 1st third
		ADD		AX,AX			; Calculate 2nd third
		MOV		DGRP:_XT4,AX	; Save step
		CALL	_disktick		; Perform disk tick
		XOR		AX,AX			; Get zero
		MOV		DGRP:_XT2,AX	; Reset count
		JMP	SHORT diskt3		; And proceed
diskt1:	INC		WORD PTR DGRP:_XT2 ; Advance count
		MOV		AX,DGRP:_XT2	; Get step
		CMP		AX,DGRP:_XT3	; Time to step?
		JZ		diskt2			; Yes
		CMP		AX,DGRP:_XT4	; Time to step?
		JNZ		diskt3			; No
diskt2:	CALL	_disktick		; Perform tick
diskt3:
	}

	switch(i = multi_step()) {
	case 0 : if(t) goto top; *dwin = DWIN;					return 0;
	case 1 : error("HLT instruction");						break;
	case 2 : error("Breakpoint!");							break;
	case -1 : error("Illegal instruction encountered.");	break;
	case -2 : error("Unknown I/O port");					break;
	default: error("STOP reason: %d", i); }
	return -1;
}

/*
 * Load a byte from the hex file
 */
unsigned load_byte(void)
{
	unsigned char c, v, ct;
	ct = 2;
	do {
		c = *ptr++;
		if((c >= '0') && (c <= '9'))
			c -= '0';
		else if((c >= 'A') && (c <= 'F'))
			c -= ('A'-10);
		else if((c >= 'a') && (c <= 'f'))
			c -= ('a'-10);
		else {
			error("%u: Bad hex digit '%c'[%02x]\n", PC, c, c);
			exit(-1); }
		v = (v << 4) | c; }
	while(--ct);
	return v;
}

/*
 * Copy filename to Dbuffer & append extension
 */
void filename(unsigned char *file, unsigned char *extension, unsigned char *n)
{
	unsigned char *d, *p, *e;
	d = Dbuffer;
	e = 0;
	p = file;
	while(*d = *file++) {
		switch(*d++) {
		case '.' : e = file-1; continue;
		case '\\':
		case ':' : p = file; e = 0; } }
	if(e)
		*e = 0;
	else
		strcpy(d, extension);
	if(n)
		strcpy(n, p);
}

/*
 * Read file into memory image
 */
void load_file(unsigned char *File)
{
	unsigned a, b, chksum, addr, length, count, size, line, XPC;
	unsigned char buffer[80];

	filename(File, ".HEX", 0);
	if(!(fp = fopen(Dbuffer, "r"))) {
		error("Unable to access: %s", Dbuffer);
		return; }

	for(size = line = XPC = 0; fgets(ptr = buffer, 80, fp);) {
		++line;
		again: switch(*ptr++) {
			case 'S' :	/* Motorola HEX format */
				if(*ptr == '9') goto quitload;
				if(*ptr++ != '1') continue;
				length = count = (chksum = load_byte()) - 3;
				chksum += a = load_byte();
				chksum += b = load_byte();
				addr = (a << 8) + b;
				if(!size)
					XPC = addr;
				while(count--) {
					chksum += a = load_byte();
					poke(SEG, addr++, a); }
				if((255 & ~chksum) != load_byte())
					goto quitbad;
				break;
			case ':' :		/* Intel HEX format */
				if(!(length = count = load_byte())) goto quitload;
				chksum = (a = load_byte()) + length;
				chksum += b = load_byte();
				addr = (a << 8) + b;
				chksum += load_byte();
				if(!size)
					XPC = addr;
				while(count--) {
					chksum += a = load_byte();
					poke(SEG, addr++, a); }
				if((255 & -chksum) != load_byte())
					goto quitbad;
				break;
			case ' ' :		/* Space */
			case '\t' :		/* Tab */
				goto again;
			case 0 :		/* Null line */
				continue;
			default:
				error("%u: Invalid record format", line);
				goto quit; }
		size += length;
		}

quitbad:
	error("%u: Bad record checksum", PC);
quit:
	fclose(fp);
	return;
quitload:
	fclose(fp);
	status("%u bytes loaded at %04x", size, XPC);
	if(LPC == 0xFFFF)
		LPC = XPC;
}

void write_record(unsigned a, unsigned char l)
{
	unsigned char c, d;
	if(Mhx)
		fprintf(fp, "S1%02x%04x", c = l+3, a);
	else
		fprintf(fp, ":%02x%04x00", c = l, a);
	c += (a >> 8);
	c += (a & 255);
	do {
		fprintf(fp, "%02x", d = peek(SEG, a++));
		c += d; }
	while(--l);
	fprintf(fp,"%02x\n", (Mhx ? ~c : -c) & 255);
}

/*
 * Open a disk drive
 */
OPEN(unsigned index, unsigned char mode)
{
	unsigned h, s, t;
	unsigned char xb[512];
	if(h = open(Dbuffer, mode ? (F_READ|F_WRITE) : F_READ)) {
		s = 0;
		while(t = read(xb, sizeof(xb), h))
			++s;
		if(!t) switch(s) {
			case 175 : ptr = "SD"; t = 0;	goto ok;
			case 350 : ptr = "DD"; t = 255; goto okdd;
			case 700 : ptr = "DQ"; t = 255;
	okdd:	if(DH == 0xEA) {
				close(h);
				error("DD/DQ image cannot be read on SD controller");
				return 0; }
	ok:		Dwrite[index] = mode ? 255 : 0;
			Dhandle[index] = h;
			Ddd[index] = t;
			if(dwin) error("Drive %u = %s (%uk)", index, ptr, s/2);
			return 255; }
		close(h);
		error("Bad format: %s", Dbuffer);
		return 0; }
	error("Unable to access: %s", Dbuffer);
	return 0;
}

/*
 * Skip ahead to non-blank in input
 */
int skip()
{
	while(isspace(*ptr))
		++ptr;
	if(*ptr == ';')
		return 0;
	return *ptr;
}

/*
 * Get value from input line
 */
unsigned getval(unsigned h)
{
	unsigned c, v;
	skip();
	v = 0;
	for(;;) {
		c = *ptr++;
		if((c >= '0') && (c <= '9'))
			c -= '0';
		else if((c >= 'A') && (c <= 'F'))
			c -= ('A'-10);
		else if((c >= 'a') && (c <= 'f'))
			c -= ('a'-10);
		else switch(c) {
			case ';':
			case 0 : --ptr;
			case ',':
			case '\t':
			case ' ':
				if(v > h) {
					error("Value too high");
					return 0; }
				return v;
			default:
			xfail: error("Bad digit: '%c'", c);
				return 0; }
		if(c >= PC) goto xfail;
		v = (v * PC) + c; }
}

/*
 * Parse a command line parameter
 */
void cmdparm()
{
	unsigned char w;
	static char Dindex;
	FILE *fp;

	w = 0;
	PC = 16;
	switch((toupper(*ptr++) << 8) | toupper(*ptr++)) {
	case 'L=' : strcpy(Dbuffer+384, ptr);	return;	// Load file
	case 'M=' : Maddress = getval();		return;	// Initial Memory view
	case 'F=' :										// Fill memory value
		IX = getval(255);
		IX |= IX << 8;
		return;
	case 'W=' : w = 255;							// Writeable disk
	case 'R=' :										// Read only disk
		if(++Dindex > 3) {
			printf("Too many R=/W= drives (Max 3)");
			break; }
		filename(ptr, ".NSI", Dname[Dindex-1]);
		OPEN(Dindex, w);
		return;
	case 'O=' :
		filename(ptr, ".TXT", Oname);
		fpo = fopen(Dbuffer, "wvqb");
		return;
	case 'I=' :
		filename(ptr, ".TXT", Iname);
		fpi = fopen(Dbuffer, "rvqb");
		return;
	case 'P=' : defineport();				return;	// Define I/O port
	case '/G' : XT5 = 'G';					return;	// GO
	case '/I' : Bport = 0;					return;	// Ignore bad I/O
	case '/M' : Mhx = 255;					return;	// Write MHX
	case '/V' : CPUname = "VECTOR";			return;	// Alternate ID
	case '?'<<8 :
	case '-?' :
	case '/?' : help(HCMD); exit(0);
	case '/S' : DH = 0xEA;			return;
	case '/P' : PC = 10;
		switch(toupper(*ptr++)) {
		case 'T' :	SP = 0x1234;			return;
		case ':' :	Speed = getval();		return;
		case 0	:	Speed = 500;			return;
		case 'D' :	w = 255;
		case 'I' :	Speed = 0;
			if(*ptr++ == ':') {
				if(w) setslow(getval()); else Step = getval();
				return; } }
		--ptr; }
	filename(ptr -= 2, ".CMD", 0);
	if(fp = fopen(Dbuffer, "r")) {
		while(fgets(ptr = Dbuffer+128, 255, fp)) {
			while(*ptr) {
				if(*ptr == ';')
					break;
				++ptr; }
			while((ptr > Dbuffer) && isspace(*(ptr-1)))
				--ptr;
			*ptr = 0; ptr = Dbuffer+128;
			if(skip())
				cmdparm(); }
		fclose(fp);
		return; }
	error("Unknown option: '%s'", ptr);
}

/*
 * Execute a sub-shell
 */
void doshell(void)
{
	unsigned char comspec[65], tail[80];

	if(!getenv("COMSPEC", comspec)) {
		error("Cannot locate COMSPEC");
		return; }
	*tail = 0;
	wopen(0, 0, 80, 25, WSAVE|WCOPEN|NORMAL);
	wcursor_line();
	wupdatexy();
	exec(comspec, tail);
	wclose();
}

/*
 * Calibrate CPU performance
 */
void calibrate(void)
{
	unsigned t, t1, i, aj, SPC;

	if(!(Speed = (Speed + 9) / 18))
		return;

	SPC = PC;
	aj = 1000;

	wopen(15, 12, 50, 7, WSAVE|WBOX1|WCOPEN|REVERSE|WSCROLL);
	wgotoxy(9, 0); wputs(" >>> Calibrating CPU speed <<<");
	wgotoxy(10, 4); wputs(" --- Press ESC to bypass ---");

	PC = XT3;	// Jump to self
	setslow(1);
	t1 = TICK;
	while((t = TICK) == t1);	// Wait for clock to tick
	while(kbtst() != 0x1B) {
		t1 = t;
		INSTCT = 0;
		do {
			STEP = Step;
			multi_step(); }
		while((t = TICK) == t1);
		wgotoxy(15, 2);
		wprintf("%-5u  %-5u  %-5u", INSTCT, SLOWCT, aj);
		if(INSTCT > Speed) {
			SLOWCT += aj;
			continue; }
		i = aj;
		if(aj /= 10) {
			SLOWCT = (SLOWCT > i) ? (SLOWCT - i) : 1;
			continue; }
		break; }
	wclose();
	PC = SPC;
}

/*
 * Debug trace back up to 4095 instructions
 */
void trace(void)
{
	unsigned t, i, j, s, st;
	close_console();
home:
	st = j = TRCPTR / 2;
reshow:
	wcursor_off();
	j &= 4095;
	if(i = (st-j) & 4095)
		Daddress = s = TRCBUF[t = j];
	else {
		t = st;
		Daddress = s = PC; }
	status("TRACE %-4u: F1=Help  F2=SearchF  F3=SearchB  F4=NextF  F5=NextB  F10=Exit", i);
	disassemble(16, 255);
	for(;;) switch(wgetct()) {
	case '?' :
	case _K1 : help(HTRACE);			continue;
	case _K2 :
		status("Search Forward?"); i = 1;
		goto dosearch;
	case _K3 :
		status("Search Backward?"); i = -1;
	dosearch:
		wcursor_line();
		if(gethex(0, &s, 4)) {
dosr:			j = t;
			do {
				j = (j + i) & 4095;
				if(TRCBUF[j] == s)
					goto reshow; }
			while(j != st);
			beep(1000, 100); }
		j = t;
		goto reshow;
	case _K4 : i = 1; goto dosr;
	case _K5 : i = -1; goto dosr;
	case _KHO : j = st+1;				goto reshow;
	case _KEN : goto home;
	case _KUA : j = t - 1;				goto reshow;
	case _KDA : j = t + 1;				goto reshow;
	case _KPU : j = t - 100;			goto reshow;
	case _KPD :	j = t + 100;			goto reshow;
	case _K10 :
	case 0x1B : return;
	default: if(eflag) { eflag = 0; goto reshow; } }
}

/*
 * Process a file open request
 */
FILE *dofile(FILE *fp, unsigned char *name, unsigned char *mode)
{
	if(fp) {
		fclose(fp);
		return *name = 0; }
	if(selectfile(".TXT")) {
		filename(Directory, ".TXT", name);
		if(!(fp = fopen(Dbuffer, mode)))
			error("Failed to open: %s", Dbuffer); }
	return fp;
}

/*
 * Main Horizon simulator
 */
main(int argc, char *argv[])
{
	unsigned i, j, k, t;
	static char pflag;

#ifdef MEMCHK
	asm {
		PUSH	DS
		POP		ES
		MOV		DI,(offset DGRP:?heap)+1
		MOV		CX,0FF00h
		SUB		CX,DI
		MOV		AL,55h
	REP	STOSB
	}
#endif

	Directory[0] = get_drive() + 'A';
	Directory[1] = ':';
	Directory[2] = '\\';
	getdir(Directory+3);
	for(i=1; i < argc; ++i) {
		ptr = argv[i];
		cmdparm(); }

	SEG = alloc_seg(4096);
	i = 0;
	do {
		pokew(SEG, i, IX); }
	while(i += 2);

	if(SP == 0x1234) {		// Performance analysis
		printf("Blocks of %u insts/sec., press ESC to exit\n", Step);
		reset();
		PC = XT3;
		while(kbtst() != 0x1B) {
			i = TICK;
			while((t = TICK) == i);
			i = 0;
			ptr = "%u\n";
			do {
				STEP = Step; 
				multi_step();
				if(!++i)
					ptr = ">65535\n"; }
			while((TICK - t) < 18);
			printf(ptr, i); }
		return; }

	if(Dbuffer[384])
		load_file(Dbuffer+384);

//dfp = fopen("D:LOG", "wvq");
	swin = wopen(0, 0, 80, 25, WSAVE|WCOPEN|0x07);
	awin = wopen(0, 0, 31, 16, 0x67);
	rwin = wopen(31,0, 10, 16, 0x57);
	mwin = wopen(41, 0, 39, 16, 0x67);
	dwin = wopen(0, 24, 80, 1, WCOPEN|DWIN);
	open_console();
	W_OPEN = dwin;
	wcursor_off();
	status(" Virtual %s - Dave Dunfield - "#__DATE__"", CPUname);
	initio();
	memset(cdata, ' ', sizeof(cdata));
	reset();
	calibrate();
	wgotoxy(71, 0); wputs("F1=Help");
	if(k = XT5) goto docmd;
stop:
	t = TRCPTR/2;
//	j = Mread(Address = Daddress = PC);
	Daddress = PC;
	disassemble(16, 255);
	dumpreg();
	memory();
cmsg:
	if(!eflag)
		status("F1=Help F2=%s F3=EdMem F4=EdReg F5=BkPt F6=Dis F7=Mount F8=Trc F9=Dos F10=Off",
			Cmode ? "Debug" : "Tty  ");
	eflag = 0;
	wcursor_off();
cmd:
	k = toupper(wgetct());
docmd:
	if(!pflag) { cputc('Z'-0x40); pflag = 255; }
	switch(k) {
	case _K10 :
		status("Exit %s simulator (Y/N)?", CPUname);
		for(;;) switch(wgetct()) {
		case 'n' :
		case 'N' :
		case 0x1B:
			goto cmsg;
		case 'y' :
		case 'Y' :
		case '\n' :
//fclose(dfp);
			wclose();
			wclose();
			wclose();
			wclose();
			wclose();
			for(i=0; i < 4; ++i) {
				if(Dhandle[i])
					close(Dhandle[i]); }
			if(fpi) fclose(fpi);
			if(fpo) fclose(fpo);
			if(fpc) fclose(fpc);
#ifdef MEMCHK
			i = t = 0;
			ptr = 0xB000;
			do {
				if(*ptr++ == 0x55)
					++i;
				else {
					if(i > t)
						t = i;
					i = 0; } }
			while(ptr < 0xFFF0);
			printf("%u\n", t);
#endif
			return; }
	case 'G' :
//		clrdiskled();
		wcursor_line();
		dostep(0);
		goto stop;
	case 'S' :
		dostep(1);
		goto stop;
	case 'R' :
		reset();
		error("Reset");
		goto stop;
	case 'L' :
		if(selectfile(".HEX")) {
			load_file(Directory);
			eflag = 255; }
		goto stop;
	case 'W' :
		wopen(27, 9, 20, 3, WSAVE|WBOX1|WCOPEN|REVERSE);
		//	    ....*....1....*
		wputs("From ____ to ____");
		wcursor_line();
		j = 0x2000; k = 0xE7FF;
		wgotoxy(5, 0);
		if(gethex(0, &j, 4)) {
			wgotoxy(13, 0);
			if(gethex(0, &k, 4)) {
				wclose();
				if(selectfile(".HEX")) {
					filename(Directory, ".HEX", 0);
					if(!(fp = fopen(Dbuffer, "w"))) {
						error("Unable to access: %s", Dbuffer);
						goto stop; }
					do {
						if((i = k - j) > 31) i = 31;
						write_record(j, i+1);
						j += i; }
					while(j++ < k);
					fputs(Mhx ? "S9030000FC\n" : ":00000001FF\n", fp);
					fclose(fp);
					goto stop; }
				goto stop; } }
		wclose();
		goto stop;
	case _K2 :	// TTY mode
		Cmode = Cmode ? 0 : 1;
		open_console();
		wcursor_off();
		goto cmsg;
	case _KDA:
		j = daddress[1];
		goto dodis;
	case _KPD:
		j = daddress[15];
	dodis:
		Daddress = daddress[0] = j;
	dodis1:
		close_console();
		disassemble(16, 255);
		goto cmsg;
	case _KUA: i = 1; goto dotb;
	case _KPU:
		i = 15;
	dotb:
		close_console();
		k = Daddress;
		Daddress = daddress[0] = (Daddress - 58);
		disassemble(64, 0);
		for(j=15; j < 64; ++j) {
			if(daddress[j] >= k)
				break; }
		j = daddress[j-i];
		goto dodis;
	case _K3:	edit_memory();		goto cmsg;
	case _K4:	edit_register();	goto cmsg;
	case _K5:
		status("Break=%04x ?", BREAK);
		wcursor_line();
		if(gethex(0, &j, 4))
			BREAK = j;
		goto cmsg;
	case _K6:
		close_console();
		W_OPEN = awin;
		wgotoxy(0, 0);
		wcursor_line();
		if(gethex(0, &Daddress, 4))
			daddress[0] = Daddress;
		W_OPEN = dwin;
		wcursor_off();
		goto dodis1;
	case _K7 :
		wclwin();
		if(DH == 0xEA)
			{ j = 4; k = 6; }
		else
			{ j = 5; k = 8; }
		for(i=1; i < j; ++i)
			wprintf("%c%u:%-9s", Dwrite[i] ? 'W' : 'R', i, Dname[i-1]);
		if(Finput) {
			wprintf("I:%-9s", Iname);
			Dmenu[Finput = k++] = fpi ? "Close input" : "Input  file"; }
		if(Foutput) {
			wprintf("O:%-9s", Oname);
			Dmenu[Foutput = k++] = fpo ? "Close output" : "Output file"; }
		wprintf("C:%-8s", Cname);
		if(fpc)
			Dmenu[t = k++] = "Close console";
		else {
			Dmenu[t = k++] = "Console input";
			Dmenu[k++] = "Console output"; }
		/* *Temp = */ Dmenu[k] = j = 0;
		if(!wmenu(30, 5, WSAVE|WCOPEN|WBOX1|0x70, Dmenu, &j)) {
			if(j) {
				if(j == Foutput) {
					fpo = dofile(fpo, Oname, "wb");
					goto cmsg; }
				if(j == Finput) {
					Fdata1 = 0;
					if(fpi = dofile(fpi, Iname, "rb"))
						wmenu(28, 7, WSAVE|WCOPEN|WBOX1|0x70, IFmenu, &IFilter);
					goto cmsg; }
				if(j == t) {
					Fconsole = 0x55;
					if(fpc = dofile(fpc, Cname, "rb"))
						wmenu(28, 7, WSAVE|WCOPEN|WBOX1|0x70, IFmenu, &CFilter);
					goto cmsg; }
				if(j == (k-1)) {
					Fconsole = 0xAA;
					fpc = dofile(fpc, Cname, "wb");
					goto cmsg; } }
			k = (j >> 1) + 1;	// Drive number
			if(Dhandle[k]) {
				close(Dhandle[k]);
				*Dname[k-1] = Dhandle[k] = Dwrite[k] = 0; }
			if(selectfile(".NSI")) {
				filename(Directory, ".NSI", Dname[k-1]);
				if(!OPEN(k, j&1)) {
					*Dname[k-1] = 0;
					goto cmd; }
				if(j & 1) Dwrite[k] = 255; } }
#if 0
			wclwin(); wputs("File?");
			if(wgets(6, 0, Temp, sizeof(Temp)-1) == '\n') {
				wcursor_off();
				if(!*Temp) goto cmsg;
				filename(Temp, ".NSI", Dname[k-1]);
				if(!(OPEN(k, j&1)) {
					*Dname[k-1] = 0;
					goto cmd; }
				if(j & 1) Dwrite[k] = 255; } }
#endif
		wcursor_off();
		goto cmsg;
	case _K8 : trace();	goto cmsg;
	case _K9 : doshell(); goto cmsg;
	case _K1 :
	case '?' : help(HSTOP); goto cmd;
	default: goto cmsg; }
}

/* General parameters */
#define	LINK_S		('N'-0x40)		/* Start of a link */
#define	LINK_E		('O'-0x40)		/* End of a link */
//#define	USER_CODE	('Q'-0x40)		/* User code */

#define	TAB_SIZE	4				/* Spacing of TABs */
#define	LINKS		25				/* Maximum # links per screen */

/*
 * Display a title on a window
 */
void wintitle(char *ptr)
{
	char buffer[6];
	memcpy(buffer, W_OPEN, 6);
	W_OPEN[1] &= (~WBOX3 >> 8);
	--W_OPEN[3];
	wgotoxy(0,0);
	wprintf(" %s ", ptr);
	memcpy(W_OPEN, buffer, 6);
}

/*
 * Help commnd - Implements "hypertext" screens
 */
void help(unsigned section)
{
	int c, lx, ly, ox, oy;
	unsigned i, j, size, ncount;
	unsigned char buffer[65], *ptr;
	unsigned char xs[LINKS], ys[LINKS], names[LINKS][25], link[LINKS];

	ptr = ARGV[i=j=0];
	while(*ptr) {
		if((buffer[i++] = *ptr++) == '.')
			j = i; }
	strcpy(buffer+j, "HLP");
	if(!(fp = fopen(buffer, "rb"))) {
		beep(1000, 250);
		return; }

	/* Locate the section in question */
lookup:
	size = getc(fp);
	size |= getc(fp) << 8;
	if(section--) {
		fseek(fp, 0, size, 1);
		goto lookup; }

	/* Draw the screen */
	wopen(5, 4, 70, 17, WSAVE|WCOPEN|WBOX1|HTEXT);
	wcursor_off();
	i = ncount = ptr = 0;
	while(buffer[i++] = getc(fp));
	wintitle(buffer);
	wgotoxy(0, 0);
	while(i++ < size) switch(c = getc(fp)) {
		case LINK_S :	/* Start link */
			ptr = names[ncount];
			xs[ncount] = W_OPEN->WINcurx;
			ys[ncount] = W_OPEN->WINcury;
			*W_OPEN = HLINK;
			break;
		case LINK_E :	/* End link */
			link[ncount++] = getc(fp);
			*ptr = ptr = 0;
			*W_OPEN = HTEXT;
#ifndef USER_CODE
			++i;
			break;
#else
		extra:
			++i;
			break;
		case USER_CODE:	/* User control code */
			user_help(getc(fp));
			goto extra;
#endif
		case '\t' :		/* Horizontal TAB */
			do
				wputc(' ');
			while(W_OPEN->WINcurx % TAB_SIZE);
			break;
		case 0 :
			c = '\n';
		default:
			wputc(c);
			if(ptr)
				*ptr++ = c; }

	/* Allow user to select field & link to new screen */
	i = section = 0;
	for(;;) {
		wgotoxy(xs[section], ys[section]);
		*W_OPEN = HLINK;
		wputs(names[section]);
		wgotoxy(xs[i], ys[i]);
		*W_OPEN = HLINKH;
		wputs(names[section = i]);
		*W_OPEN = HTEXT;
		switch(c = wgetct()) {		/* User keystroke */
			case _KLA :				/* Left arrow - previous field */
				i = (i ? i : ncount) - 1;
				break;
			case _KRA :				/* Right arrow - next field */
				i = (i + 1) % ncount;
				break;
			case _KUA : ox = oy = -1000; goto dofind;
			case _KDA : ox = oy = 1000;
			dofind:
				size = i;
				for(j = 0; j < ncount; ++j) {
					lx = (int)xs[j] - (int)xs[i];
					ly = (int)ys[j] - (int)ys[i];
					if(c == _KUA) {
						if((ly >= 0) || (ly < oy)) continue; }
					else {
						if((ly <= 0) || (ly > oy))	continue; }
					if(abs(lx) > abs(ox)) continue;
					size = j;
					ox = lx;
					oy = ly; }
				i = size;
				break;
			case '\n' :				/* Enter - chain to menu */
				rewind(fp);
				section = link[i];
				wclose();
				goto lookup;
			case 0x1B:				/* Escape exit */
				wclose();
				fclose(fp);
				return; } }
}

/*
 * Select a file via window panel
 */
int selectfile(unsigned char *match)
{
	unsigned a, i, Nt, Pt;
	unsigned char name[13], *n[SFILES], pool[SFILES*9], *p;
	static unsigned char *Dip;

	// Find end of directory index
	if(Dip)
		*Dip = 0;
newdir:
	status("Select file: F1=Help  F2=ManualEntry");
	Dip = Directory; while(*Dip) ++Dip;
	if(*(Dip-1) == '\\')
		--Dip;
	*Dip = '\\';
	// Scan for all files & build name list
	strcpy(Dip+1, "*.*");
	Nt = Pt = 0;
	if(!find_first(Directory, DIRECTORY, p = name, &i, &i, &a, &i, &i)) do {
		if(Nt >= SFILES)
			break;
		if(a & DIRECTORY) {
			if(!strcmp(p, "."))
				continue;
			n[Nt++] = &pool[Pt];
			pool[Pt++] = 1;	// Insure sorts first!
			while(pool[Pt++] = *p++);
			continue; }
		while(*p) {
			if(!strcmp(p, match)) {
				*p = 0;	p = name;
				n[Nt++] = &pool[Pt];
				while(pool[Pt++] = *p++);
				break; }
			++p; } }
	while !find_next(p=name, &i, &i, &a, &i, &i);

	*Dip = 0;
	if(!Nt) goto manent;

	// Sort the name list
	for(i=0; i < Nt; ++i) {
		for(a=i+1; a < Nt; ++a) {
			if(strcmp(n[a], n[i]) < 0) {
				p = n[i];
				n[i] = n[a];
				n[a] = p; } } }
	// Convert directory pref
	for(i=0; i < Nt; ++i) {
		if(*(p = n[i]) != 1)
			break;
		while(p[1]) {
			*p = p[1];
			++p; }
		*p = '\\'; }

	i = ((Nt + 4) / 5) + 2;
	wopen(4, 12-(i/2), 71, i, WSAVE|WBOX1|WCOPEN|0x70);
	wintitle(Directory);
	wcursor_off();
	a = 0;
draw:
	for(i=0; i < Nt; ++i) {
		wgotoxy((i % 5) * 14, i / 5);
		*W_OPEN = (i == a) ? 0x07 : 0x70;
		wputs(n[i]); }
	*W_OPEN = 0x70;
	for(;;) switch(wgetc()) {
		case _KLA : if(a) { --a; goto draw; }
		case _KEN : a = Nt-1; goto draw;
		case _KRA : if(++a < Nt) goto draw;
		case _KHO : a = 0;	goto draw;
		case _KUA : if(a > 4) a -= 5; goto draw;
		case _KDA : if((a+5) < Nt) a += 5; goto draw;
		case '?' :
		case _K1  : help(HFILE); continue;
		case _K2  :
			wclose();
		manent:
			strcpy(pool, Directory); Pt=Dip;
			wclwin(); wputs("File?");
			*Dip = '\\'; *(Dip+1) = 0;
		manent1:
			switch(wgets(0x106, 0, p = Dip = Directory, sizeof(Directory)-1)) {
			case _K1: help(HENTER);
			default : goto manent1;
			case '\n' :
				while(i = *p++) switch(i) {
					case ':' :
					case '\\': Dip = p-1; }
				if(*(p-2) == '\\') goto newdir;
				return 255;
			case 0x1B :
				if(Nt) {
					strcpy(Directory, pool);
					Dip = Pt;
					goto newdir; }
				return 0; }
		case '\n' :
			*Dip = '\\';
			p = n[a]; while(p[1]) ++p;
			if(*p == '\\') {
				*p = 0;
				if(strcmp(p = n[a], "..")) {	// Not UP
					strcpy(Dip+1, p);
					wclose();
					goto newdir; }
				// Go up one directory
				while(Dip > Directory) {
					if(*--Dip == '\\') {
						*Dip = 0;
						break; } }
				wclose();
				goto newdir; }
			strcpy(Dip+1, n[a]);
			wclose();
			return 255;
		case 0x1B :
		case _K10 :
			wclose();
			return 0; }
}

#include "io.c"
