;
; Virtual Horizon CPU simulation
;
; This code implements the Zilog Z80 CPU processor instruction set.
;
; Software interface is via global locations:
; _A                       Z80 Accumulator
; _PSW                     Z80 Processor Status Word (Flags)
; _C, _B, _D, _E, _H, _L   Z80 General registers
; _IX, _IY		   Z80 Index registers
; _XA, _XF, _XC, _XB       Z80 alternate registers
; _XE, _XD, _XL, _XH       "" ""
; _SP                      8080 Stack Pointer
; _PC                      8080 Program Counter
; _SEG                     Virtual Memory segment
; _TRCPTR                  Instruction trace pointer
; _TRCBUF                  Instruction trace buffer
; _STEP                    Instruction set count
; _BREAK                   Breakpoint address
; _EI                      Interrupts enabled flag
; _IM                      Interrupt mode
; _I                       Interrupt vector
; _R                       Refresh register
; _DH                      Disk controller hardware address
;
; Modified from my ALTAIR 8080 CPU simulation (some comments may not
; reflect Z80).
;
; ?COPY.TXT 2003-2008 Dave Dunfield
;
DGRP	GROUP	DSEG,BSEG
CSEG	SEGMENT	BYTE PUBLIC 'CODE'
	ASSUME	CS:CSEG, DS:DGRP, SS:DGRP
	PUBLIC	_multi_step, _Mread, _setslow
	EXTRN	_IN8080:near, _OUT8080:near
	EXTRN	_diskreadSD:near, _diskreadDD:near
;
; Instruction decode table
; This 256 entry jump table vectors to an appropriate handler for
; Each of the 256 possible opcodes which may be fetched
; 
ITABLE	LABEL	WORD
	DW	?NOP		;00
	DW	?LXIB		;01
	DW	?STAXB		;02
	DW	?INXB		;03
	DW	?INRB		;04
	DW	?DCRB		;05
	DW	?MVIB		;06
	DW	?RLC		;07
	DW	?EXAF		;08
	DW	?DADB		;09
	DW	?LDAXB		;0A
	DW	?DCXB		;0B
	DW	?INRC		;0C
	DW	?DCRC		;0D
	DW	?MVIC		;0E
	DW	?RRC		;0F
	DW	?DJNZ		;10
	DW	?LXID		;11
	DW	?STAXD		;12
	DW	?INXD		;13
	DW	?INRD		;14
	DW	?DCRD		;15
	DW	?MVID		;16
	DW	?RAL		;17
	DW	?JR		;18
	DW	?DADD		;19
	DW	?LDAXD		;1A
	DW	?DCXD		;1B
	DW	?INRE		;1C
	DW	?DCRE		;1D
	DW	?MVIE		;1E
	DW	?RAR		;1F
	DW	?JRNZ		;20
	DW	?LXIH		;21
	DW	?SHLD		;22
	DW	?INXH		;23
	DW	?INRH		;24
	DW	?DCRH		;25
	DW	?MVIH		;26
	DW	?DAA		;27
	DW	?JRZ		;28
	DW	?DADH		;29
	DW	?LHLD		;2A
	DW	?DCXH		;2B
	DW	?INRL		;2C
	DW	?DCRL		;2D
	DW	?MVIL		;2E
	DW	?CMA		;2F
	DW	?JRNC		;30
	DW	?LXISP		;31
	DW	?STA		;32
	DW	?INXSP		;33
	DW	?INRM		;34
	DW	?DCRM		;35
	DW	?MVIM		;36
	DW	?STC		;37
	DW	?JRC		;38
	DW	?DADSP		;39
	DW	?LDA		;3A
	DW	?DCXSP		;3B
	DW	?INRA		;3C
	DW	?DCRA		;3D
	DW	?MVIA		;3E
	DW	?CMC		;3F
	DW	?MOVBB		;40
	DW	?MOVBC		;41
	DW	?MOVBD		;42
	DW	?MOVBE		;43
	DW	?MOVBH		;44
	DW	?MOVBL		;45
	DW	?MOVBM		;46
	DW	?MOVBA		;47
	DW	?MOVCB		;48
	DW	?MOVCC		;49
	DW	?MOVCD		;4A
	DW	?MOVCE		;4B
	DW	?MOVCH		;4C
	DW	?MOVCL		;4D
	DW	?MOVCM		;4E
	DW	?MOVCA		;4F
	DW	?MOVDB		;50
	DW	?MOVDC		;51
	DW	?MOVDD		;52
	DW	?MOVDE		;53
	DW	?MOVDH		;54
	DW	?MOVDL		;55
	DW	?MOVDM		;56
	DW	?MOVDA		;57
	DW	?MOVEB		;58
	DW	?MOVEC		;59
	DW	?MOVED		;5A
	DW	?MOVEE		;5B
	DW	?MOVEH		;5C
	DW	?MOVEL		;5D
	DW	?MOVEM		;5E
	DW	?MOVEA		;5F
	DW	?MOVHB		;60
	DW	?MOVHC		;61
	DW	?MOVHD		;62
	DW	?MOVHE		;63
	DW	?MOVHH		;64
	DW	?MOVHL		;65
	DW	?MOVHM		;66
	DW	?MOVHA		;67
	DW	?MOVLB		;68
	DW	?MOVLC		;69
	DW	?MOVLD		;6A
	DW	?MOVLE		;6B
	DW	?MOVLH		;6C
	DW	?MOVLL		;6D
	DW	?MOVLM		;6E
	DW	?MOVLA		;6F
	DW	?MOVMB		;70
	DW	?MOVMC		;71
	DW	?MOVMD		;72
	DW	?MOVME		;73
	DW	?MOVMH		;74
	DW	?MOVML		;75
	DW	?HLT		;76
	DW	?MOVMA		;77
	DW	?MOVAB		;78
	DW	?MOVAC		;79
	DW	?MOVAD		;7A
	DW	?MOVAE		;7B
	DW	?MOVAH		;7C
	DW	?MOVAL		;7D
	DW	?MOVAM		;7E
	DW	?MOVAA		;7F
	DW	?ADDB		;80
	DW	?ADDC		;81
	DW	?ADDD		;82
	DW	?ADDE		;83
	DW	?ADDH		;84
	DW	?ADDL		;85
	DW	?ADDM		;86
	DW	?ADDA		;87
	DW	?ADCB		;88
	DW	?ADCC		;89
	DW	?ADCD		;8A
	DW	?ADCE		;8B
	DW	?ADCH		;8C
	DW	?ADCL		;8D
	DW	?ADCM		;8E
	DW	?ADCA		;8F
	DW	?SUBB		;90
	DW	?SUBC		;91
	DW	?SUBD		;92
	DW	?SUBE		;93
	DW	?SUBH		;94
	DW	?SUBL		;95
	DW	?SUBM		;96
	DW	?SUBA		;97
	DW	?SBBB		;98
	DW	?SBBC		;99
	DW	?SBBD		;9A
	DW	?SBBE		;9B
	DW	?SBBH		;9C
	DW	?SBBL		;9D
	DW	?SBBM		;9E
	DW	?SBBA		;9F
	DW	?ANAB		;A0
	DW	?ANAC		;A1
	DW	?ANAD		;A2
	DW	?ANAE		;A3
	DW	?ANAH		;A4
	DW	?ANAL		;A5
	DW	?ANAM		;A6
	DW	?ANAA		;A7
	DW	?XRAB		;A8
	DW	?XRAC		;A9
	DW	?XRAD		;AA
	DW	?XRAE		;AB
	DW	?XRAH		;AC
	DW	?XRAL		;AD
	DW	?XRAM		;AE
	DW	?XRAA		;AF
	DW	?ORAB		;B0
	DW	?ORAC		;B1
	DW	?ORAD		;B2
	DW	?ORAE		;B3
	DW	?ORAH		;B4
	DW	?ORAL		;B5
	DW	?ORAM		;B6
	DW	?ORAA		;B7
	DW	?CMPB		;B8
	DW	?CMPC		;B9
	DW	?CMPD		;BA
	DW	?CMPE		;BB
	DW	?CMPH		;BC
	DW	?CMPL		;BD
	DW	?CMPM		;BE
	DW	?CMPA		;BF
	DW	?RNZ		;C0
	DW	?POPB		;C1
	DW	?JNZ		;C2
	DW	?JMP		;C3
	DW	?CNZ		;C4
	DW	?PUSHB		;C5
	DW	?ADI		;C6
	DW	?RST0		;C7
	DW	?RZ		;C8
	DW	?RET		;C9
	DW	?JZ		;CA
	DW	PREFCB		;CB
	DW	?CZ		;CC
	DW	?CALL		;CD
	DW	?ACI		;CE
	DW	?RST1		;CF
	DW	?RNC		;D0
	DW	?POPD		;D1
	DW	?JNC		;D2
	DW	?OUT		;D3
	DW	?CNC		;D4
	DW	?PUSHD		;D5
	DW	?SUI		;D6
	DW	?RST2		;D7
	DW	?RC		;D8
	DW	?EXX		;D9
	DW	?JC		;DA
	DW	?IN		;DB
	DW	?CC		;DC
	DW	PREFIX		;DD - IX override
	DW	?SBI		;DE
	DW	?RST3		;DF
	DW	?RPO		;E0
	DW	?POPH		;E1
	DW	?JPO		;E2
	DW	?XTHL		;E3
	DW	?CPO		;E4
	DW	?PUSHH		;E5
	DW	?ANI		;E6
	DW	?RST4		;E7
	DW	?RPE		;E8
	DW	?PCHL		;E9
	DW	?JPE		;EA
	DW	?XCHG		;EB
	DW	?CPE		;EC
	DW	PREFED		;ED - Extended Opcodes
	DW	?XRI		;EE
	DW	?RST5		;EF
	DW	?RP		;F0
	DW	?POPP		;F1
	DW	?JP		;F2
	DW	?DI		;F3
	DW	?CP		;F4
	DW	?PUSHP		;F5
	DW	?ORI		;F6
	DW	?RST6		;F7
	DW	?RM		;F8
	DW	?SPHL		;F9
	DW	?JM		;FA
	DW	?EI		;FB
	DW	?CM		;FC
	DW	PREFIY		;FD - IY override
	DW	?CPI		;FE
	DW	?RST7		;FF
; Instruction table for ED operand
ITABED	DW	ILLEGAL		;00
	DW	ILLEGAL		;01
	DW	ILLEGAL		;02
	DW	ILLEGAL		;03
	DW	ILLEGAL		;04
	DW	ILLEGAL		;05
	DW	ILLEGAL		;06
	DW	ILLEGAL		;07
	DW	ILLEGAL		;08
	DW	ILLEGAL		;09
	DW	ILLEGAL		;0A
	DW	ILLEGAL		;0B
	DW	ILLEGAL		;0C
	DW	ILLEGAL		;0D
	DW	ILLEGAL		;0E
	DW	ILLEGAL		;0F
	DW	ILLEGAL		;10
	DW	ILLEGAL		;11
	DW	ILLEGAL		;12
	DW	ILLEGAL		;13
	DW	ILLEGAL		;14
	DW	ILLEGAL		;15
	DW	ILLEGAL		;16
	DW	ILLEGAL		;17
	DW	ILLEGAL		;18
	DW	ILLEGAL		;19
	DW	ILLEGAL		;1A
	DW	ILLEGAL		;1B
	DW	ILLEGAL		;1C
	DW	ILLEGAL		;1D
	DW	ILLEGAL		;1E
	DW	ILLEGAL		;1F
	DW	ILLEGAL		;20
	DW	ILLEGAL		;21
	DW	ILLEGAL		;22
	DW	ILLEGAL		;23
	DW	ILLEGAL		;24
	DW	ILLEGAL		;25
	DW	ILLEGAL		;26
	DW	ILLEGAL		;27
	DW	ILLEGAL		;28
	DW	ILLEGAL		;29
	DW	ILLEGAL		;2A
	DW	ILLEGAL		;2B
	DW	ILLEGAL		;2C
	DW	ILLEGAL		;2D
	DW	ILLEGAL		;2E
	DW	ILLEGAL		;2F
	DW	ILLEGAL		;30
	DW	ILLEGAL		;31
	DW	ILLEGAL		;32
	DW	ILLEGAL		;33
	DW	ILLEGAL		;34
	DW	ILLEGAL		;35
	DW	ILLEGAL		;36
	DW	ILLEGAL		;37
	DW	ILLEGAL		;38
	DW	ILLEGAL		;39
	DW	ILLEGAL		;3A
	DW	ILLEGAL		;3B
	DW	ILLEGAL		;3C
	DW	ILLEGAL		;3D
	DW	ILLEGAL		;3E
	DW	ILLEGAL		;3F
	DW	?INBC		;40
	DW	?OUTCB		;41
	DW	?DSBCB		;42
	DW	?SBCD		;43
	DW	?NEG		;44
	DW	?RETN		;45
	DW	?IM0		;46
	DW	?LDIA		;47
	DW	?INCC		;48
	DW	?OUTCC		;49
	DW	?DADCB		;4A
	DW	?LBCD		;4B
	DW	ILLEGAL		;4C
	DW	?RETI		;4D
	DW	ILLEGAL		;4E
	DW	?LDRA		;4F
	DW	?INDC		;50
	DW	?OUTCD		;51
	DW	?DSBCD		;52
	DW	?SDED		;53
	DW	ILLEGAL		;54
	DW	ILLEGAL		;55
	DW	?IM1		;56
	DW	?LDAI		;57
	DW	?INEC		;58
	DW	?OUTCE		;59
	DW	?DADCD		;5A
	DW	?LDED		;5B
	DW	ILLEGAL		;5C
	DW	ILLEGAL		;5D
	DW	?IM2		;5E
	DW	?LDAR		;5F
	DW	?INHC		;60
	DW	?OUTCH		;61
	DW	?DSBCH		;62
	DW	ILLEGAL		;63
	DW	ILLEGAL		;64
	DW	ILLEGAL		;65
	DW	ILLEGAL		;66
	DW	?RRD		;67
	DW	?INLC		;68
	DW	?OUTCL		;69
	DW	?DADCH		;6A
	DW	ILLEGAL		;6B
	DW	ILLEGAL		;6C
	DW	ILLEGAL		;6D
	DW	ILLEGAL		;6E
	DW	?RLD		;6F
	DW	ILLEGAL		;70
	DW	ILLEGAL		;71
	DW	?DSBCS		;72
	DW	?SSPD		;73
	DW	ILLEGAL		;74
	DW	ILLEGAL		;75
	DW	ILLEGAL		;76
	DW	ILLEGAL		;77
	DW	?INAC		;78
	DW	?OUTCA		;79
	DW	?DADCS		;7A
	DW	?LSPD		;7B
	DW	ILLEGAL		;7C
	DW	ILLEGAL		;7D
	DW	ILLEGAL		;7E
	DW	ILLEGAL		;7F
	DW	ILLEGAL		;80
	DW	ILLEGAL		;81
	DW	ILLEGAL		;82
	DW	ILLEGAL		;83
	DW	ILLEGAL		;84
	DW	ILLEGAL		;85
	DW	ILLEGAL		;86
	DW	ILLEGAL		;87
	DW	ILLEGAL		;88
	DW	ILLEGAL		;89
	DW	ILLEGAL		;8A
	DW	ILLEGAL		;8B
	DW	ILLEGAL		;8C
	DW	ILLEGAL		;8D
	DW	ILLEGAL		;8E
	DW	ILLEGAL		;8F
	DW	ILLEGAL		;90
	DW	ILLEGAL		;91
	DW	ILLEGAL		;92
	DW	ILLEGAL		;93
	DW	ILLEGAL		;94
	DW	ILLEGAL		;95
	DW	ILLEGAL		;96
	DW	ILLEGAL		;97
	DW	ILLEGAL		;98
	DW	ILLEGAL		;99
	DW	ILLEGAL		;9A
	DW	ILLEGAL		;9B
	DW	ILLEGAL		;9C
	DW	ILLEGAL		;9D
	DW	ILLEGAL		;9E
	DW	ILLEGAL		;9F
	DW	?ZLDI		;A0
	DW	?ZCPI		;A1
	DW	?ZINI		;A2
	DW	?ZOUTI		;A3
	DW	ILLEGAL		;A4
	DW	ILLEGAL		;A5
	DW	ILLEGAL		;A6
	DW	ILLEGAL		;A7
	DW	?ZLDD		;A8
	DW	?ZCPD		;A9
	DW	?ZIND		;AA
	DW	?ZOUTD		;AB
	DW	ILLEGAL		;AC
	DW	ILLEGAL		;AD
	DW	ILLEGAL		;AE
	DW	ILLEGAL		;AF
	DW	?ZLDIR		;B0
	DW	?ZCPIR		;B1
	DW	?ZINIR		;B2
	DW	?ZOTIR		;B3
	DW	ILLEGAL		;B4
	DW	ILLEGAL		;B5
	DW	ILLEGAL		;B6
	DW	ILLEGAL		;B7
	DW	?ZLDDR		;B8
	DW	?ZCPDR		;B9
	DW	?ZINDR		;BA
	DW	?ZOTDR		;BB
	DW	ILLEGAL		;BC
	DW	ILLEGAL		;BD
	DW	ILLEGAL		;BE
	DW	ILLEGAL		;BF
	DW	ILLEGAL		;C0
	DW	ILLEGAL		;C1
	DW	ILLEGAL		;C2
	DW	ILLEGAL		;C3
	DW	ILLEGAL		;C4
	DW	ILLEGAL		;C5
	DW	ILLEGAL		;C6
	DW	ILLEGAL		;C7
	DW	ILLEGAL		;C8
	DW	ILLEGAL		;C9
	DW	ILLEGAL		;CA
	DW	ILLEGAL		;CB
	DW	ILLEGAL		;CC
	DW	ILLEGAL		;CD
	DW	ILLEGAL		;CE
	DW	ILLEGAL		;CF
	DW	ILLEGAL		;D0
	DW	ILLEGAL		;D1
	DW	ILLEGAL		;D2
	DW	ILLEGAL		;D3
	DW	ILLEGAL		;D4
	DW	ILLEGAL		;D5
	DW	ILLEGAL		;D6
	DW	ILLEGAL		;D7
	DW	ILLEGAL		;D8
	DW	ILLEGAL		;D9
	DW	ILLEGAL		;DA
	DW	ILLEGAL		;DB
	DW	ILLEGAL		;DC
	DW	ILLEGAL		;DD
	DW	ILLEGAL		;DE
	DW	ILLEGAL		;DF
	DW	ILLEGAL		;E0
	DW	ILLEGAL		;E1
	DW	ILLEGAL		;E2
	DW	ILLEGAL		;E3
	DW	ILLEGAL		;E4
	DW	ILLEGAL		;E5
	DW	ILLEGAL		;E6
	DW	ILLEGAL		;E7
	DW	ILLEGAL		;E8
	DW	ILLEGAL		;E9
	DW	ILLEGAL		;EA
	DW	ILLEGAL		;EB
	DW	ILLEGAL		;EC
	DW	ILLEGAL		;ED
	DW	ILLEGAL		;EE
	DW	ILLEGAL		;EF
	DW	ILLEGAL		;F0
	DW	ILLEGAL		;F1
	DW	ILLEGAL		;F2
	DW	ILLEGAL		;F3
	DW	ILLEGAL		;F4
	DW	ILLEGAL		;F5
	DW	ILLEGAL		;F6
	DW	ILLEGAL		;F7
	DW	ILLEGAL		;F8
	DW	ILLEGAL		;F9
	DW	ILLEGAL		;FA
	DW	ILLEGAL		;FB
	DW	ILLEGAL		;FC
	DW	ILLEGAL		;FD
	DW	ILLEGAL		;FE
	DW	ILLEGAL		;FF
;
; If the "setslow" function is called to initiate CPU speed emulation,
; The main multiple-step loop is patched to call this entry point instead
; of I8080 it simply calls the slowdown delay and "falls" into the normal
; I8080 function. This way we have no overhead unless we want to slow down.
;
X8080:	CALL	GOSLOW			; Call slowdown function
;
; Execute a single Z80 instruction
;
; 8086 register usage:
;  AL = Z80 Accumulator cached
;  SI = Z80 Stack pointer cached
;  DI = Z80 Program counter cached
;  BX = Index register address (_IX, _IY or _L) - also transiant mem
;  CX = Cached DE register
;  DX = General
;  BP = Trace buffer adress
;  ES = Virtual Memory Segment
;
I8080:	MOV	DGRP:_TRCBUF[BP],DI	; Save PC
	INC	BP			; Advance
	INC	BP			; Advance
	AND	BP,1FFFh		; Mask for 8K table
	MOV	BL,ES:[DI]		; Get opcode
	INC	DI			; Skip to next
	XOR	BH,BH			; Zero high
	SHL	BX,1			; x2 for words
	MOV	DX,ITABLE[BX]		; Get handler
	MOV	BX,OFFSET DGRP:_L	; Point at HL (default)
	JMP	DX			; Execute handler
; IX override
PREFIX:	MOV	BL,ES:[DI]		; Get following opcode
	INC	DI			; Advance
	XOR	BH,BH			; Zero high
	SHL	BX,1			; x2 for words
	MOV	DX,ITABLE[BX]		; Get handler
	MOV	BX,OFFSET DGRP:_IX	; Point at IX
	JMP	DX			; Execute handler
; IY override
PREFIY:	MOV	BL,ES:[DI]		; Get following opcode
	INC	DI			; Advance
	XOR	BH,BH			; Zero high
	SHL	BX,1			; x2 for words
	MOV	DX,ITABLE[BX]		; Get handler
	MOV	BX,OFFSET DGRP:_IY	; Point at IY
	JMP	DX			; Execute handler
; Extended opcodes
PREFED:	XCHG	BX,DX			; Save index
	MOV	BL,ES:[DI]		; Get following opcode
	INC	DI			; Advance
	XOR	BH,BH			; Zero high
	SHL	BX,1			; x2 for words
	MOV	BX,ITABED[BX]		; Get handler
	XCHG	BX,DX			; Restore index
	JMP	DX			; Execute handler
; CB prefix BIT/SET/REG/ROT
PREFCB:	CALL	GETM			; BX = M
	MOV	DGRP:_A,AL		; Save A
	MOV	WORD PTR DGRP:_E,CX	; Save DE
	MOV	AL,ES:[DI]		; Get opcode
	INC	DI			; Skip
	MOV	CL,AL			; Copy for later
	MOV	DL,AL			; Copy for later
	SHR	CL,1			; / 2
	SHR	CL,1			; / 4
	SHR	CL,1			; / 8
	SHL	DL,1			; *2
	SHL	DL,1			; *4
	AND	DL,1Ch			; Save only mask
	XOR	DH,DH			; Zero high
	ADD	DX,OFFSET GRfunc	; DX = function offset
	AND	AL,0C0h			; Determine inst type
	JZ	pcbrot			; Do the rotates
; Bit manipulation SET/RES/BIT
	AND	CL,07h			; CL = bit number
	MOV	CH,1			; Start with one
	SHL	CH,CL			; Determine bit
	CMP	AL,40h			; Is it BIT?
	JNZ	pcbres			; No, try next
	CALL	DX			; Get value
	AND	AL,CH			; Test bit
	CALL	SFZSPA			; Set flags
	JMP	SHORT pcbr2		; And exit
pcbres:	CMP	AL,80h			; Is it RES?
	JNZ	pcbset			; No, try next
	CALL	DX			; Get value
	NOT	CH			; Invert mask
	AND	AL,CH			; Clear bits
pcbr1:	ADD	DX,32			; Offset to write
	CALL	DX			; Write value
pcbr2:	MOV	CX,WORD PTR DGRP:_E	; Restore DE
	MOV	AL,DGRP:_A		; Restore A
	RET
pcbset:	CALL	DX			; Get value
	OR	AL,CH			; Set bits
	JMP	SHORT pcbr1
; Rotate functions
pcbrot:	CALL	DX			; Read value
	AND	CX,0007h		; Clear CH, tests bit
	MOV	AH,DGRP:_PSW		; Get flags
	JNZ	pcbrrc			; Not RLC
pcbrlc:	ROL	AL,1
pcbret:	JC	PCBrt1
	AND	AL,AL
	JMP	SHORT pcbrt2
PCBrt1:	AND	AL,AL
	STC
pcbrt2:	CALL	SFZSPCA			; Save carry flag
	ADD	DX,32			; Offset to write
	CALL	DX			; Write value
	MOV	CX,WORD PTR DGRP:_E	; Restore DE
	MOV	AL,DGRP:_A		; Restore A
	RET
pcbrrc:	LOOP	pcbrl			; 1
	ROR	AL,1
	JMP	SHORT pcbret
pcbrl:	LOOP	pcbrr			; 2
	SAHF
	RCL	AL,1
	JMP	SHORT pcbret
pcbrr:	LOOP	pcbsla			; 3
	SAHF
	RCR	AL,1
	JMP	SHORT pcbret
pcbsla:	LOOP	pcbsra			; 4
	SAHF
	SAL	AL,1
	JMP	SHORT pcbret
pcbsra:	LOOP	pcbsrl			; 5
	SAHF
	SAR	AL,1
	JMP	SHORT pcbret
pcbsrl:	SAHF				; 6
	SHR	AL,1
	JMP	SHORT pcbret
; Register read/write function table
GRfunc:	MOV	AL,DGRP:_B		; 3 - B
	RET				; 4
	MOV	AL,DGRP:_C		; 3 - C
	RET				; 4
	MOV	AL,DGRP:_D		; 2 - D
	RET				; 3
	MOV	AL,DGRP:_E		; 2 - C
	RET				; 3
	MOV	AL,DGRP:_H		; 3 - H
	RET				; 4
	MOV	AL,DGRP:_L		; 3 - L
	RET				; 4
	MOV	AL,ES:[BX]		; 3 - M
	RET				; 4
	MOV	AL,DGRP:_A		; 3 - A
	RET
GWfunc:	MOV	DGRP:_B,AL		; 3 - B
	RET				; 4
	MOV	DGRP:_C,AL		; 3 - C
	RET				; 4
	MOV	DGRP:_D,AL		; 2 - D
	RET				; 3
	MOV	DGRP:_E,AL		; 2 - C
	RET				; 3
	MOV	DGRP:_H,AL		; 3 - H
	RET				; 4
	MOV	DGRP:_L,AL		; 3 - L
	RET				; 4
	MOV	ES:[BX],AL		; 3 - M
	RET				; 4
	MOV	DGRP:_A,AL		; 3 - A
	RET
;
; Load 8080 registers into corresponding 8086 registers for fast access.
;
L8080:	MOV	ES,DGRP:_SEG		; Read segment
	MOV	AL,DGRP:_A		; Get A
	MOV	CX,WORD PTR DGRP:_E	; Get DE
	MOV	SI,DGRP:_SP		; Get SP
	MOV	DI,DGRP:_PC		; Get PC
	MOV	BP,DGRP:_TRCPTR		; Get trace pointer
	RET
;
; Save the cached registers back into memory.
;
S8080:	MOV	DGRP:_A,AL		; Save A
	MOV 	WORD PTR DGRP:_E,CX	; Save DE
	MOV	DGRP:_SP,SI		; Save SI
	MOV	DGRP:_PC,DI		; Save PC
	MOV	DGRP:_TRCPTR,BP		; Save BP
	RET
;
; Step multiple 8080 instructions
; On entry, global word 'STEP' contains number of instructions
; to step.
;
_multi_step:
	PUSH	BP			; Save callers BP
	MOV	DGRP:_SAVSP,SP		; Save stack
	CALL	L8080			; Load registers
	MOV	AH,DGRP:_BRKSK		; Skip breakpoint?
	AND	AH,AH			; Test
	JZ	mstep1			; No, do breakpoint
	XOR	AH,AH			; Get zero
	MOV	DGRP:_BRKSK,AH		; Clear flag
	JMP	SHORT mstep2		; Skip first breakpoint check
; Check for PC at the breakpoint location, and if so stop
mstep1:	CMP	DI,DGRP:_BREAK		; At breakpoint
	JZ	xbrk
; Execute 8080 instructructions in a loop until STEP expires
mstep2:	CALL	I8080			; Execute one instruction
mstep3:	DEC	WORD PTR DGRP:_STEP	; Reduce STEP count
	JNZ	mstep1			; More to do.
; Normal termination - save registers & return 0 to caller
	CALL	S8080			; Save registers
	POP	BP			; Restore BP
	INC	WORD PTR DGRP:_INSTCT	; Advance block count
	XOR	AX,AX			; Indicate success
	RET
; Breakpoint encountered - save registers & indicate to caller
xbrk:	CALL	S8080			; Save registers
	POP	BP			; Restore BP
	MOV	AX,2			; Indicate breakpoint
	RET
;
; Individual 8080 instruction handlers
; - Perform required operation with corresponding 8086 registers.
;
; LXI B,xxxx
?LXIB:	MOV	BL,ES:[DI]		; Read low
	INC	DI			; Next
	MOV	BH,ES:[DI]		; Read high
	INC	DI			; Next
	MOV	WORD PTR DGRP:_C,BX	; Save BC
?NOP:	RET
;
?STAXB:	MOV	BX,WORD PTR DGRP:_C	; Get BC
	MOV	AH,AL			; Get A
	JMP	WRBX
;
?INXB:	INC	WORD PTR DGRP:_C	; Increment register
	RET
;
?INRB:	INC	BYTE PTR DGRP:_B	; Increment HI
	JMP	SFZSPA			; Save flags
;
?DCRB:	DEC	BYTE PTR DGRP:_B	; Decrement HI
	JMP	SFZSPA			; Save flags
;
?MVIB:	MOV	AH,ES:[DI]		; Get low
	INC	DI			; Next
	MOV	DGRP:_B,AH		; Save B
	RET
;
?RLC:	ROL	AL,1			; Do the shift
	JMP	SFC			; Save carry
;
?DADB:	MOV	DX,[BX]			; Get value
	ADD	DX,WORD PTR DGRP:_C	; Add BC
	MOV	[BX],DX			; Resave
	JMP	SFC			; Save carry
?DADCB:	MOV	DX,WORD PTR DGRP:_C	;
dadcrp:	MOV	AH,DGRP:_PSW		; Get flags
	SAHF				; Set 'C'
	ADC	DX,[BX]			; Add to HL
	MOV	[BX],DX			; Resave
	JMP	SFZSPCA			; And save flags
?DSBCB:	MOV	DX,WORD PTR DGRP:_C
dsbcrp:	PUSH	AX			; Save AX
	MOV	AH,DGRP:_PSW		; Get flags
	SAHF				; Set 'C'
	MOV	AX,[BX]			; Get HL
	SBB	AX,DX			; Subtract (with borrow)
	MOV	[BX],AX			; Resave HL
	POP	AX			; Restore ACC
	JMP	SNZSPCA			; Save flags
;
?LDAXB:	MOV	BX,WORD PTR DGRP:_C	; Get BC
	CALL	RDBX			; Get value
	MOV	AL,AH			; Set A
	RET
;
?DCXB:	DEC	WORD PTR DGRP:_C	; Decrement register
	RET
;
?INRC:	INC	BYTE PTR DGRP:_C	; Increment LO
	JMP	SFZSPA			; Save flags
;
?DCRC:	DEC	BYTE PTR DGRP:_C	; Decrement LO
	JMP	SFZSPA			; Save flags
;
?MVIC:	MOV	AH,ES:[DI]		; Get data
	MOV	DGRP:_C,AH		; Write C
	INC	DI			; Next
	RET
;
?RRC:	ROR	AL,1			; Rotate
	JMP	SFC			; Save carry
;
?LXID:	MOV	CL,ES:[DI]		; Read value
	INC	DI			; Next
	MOV	CH,ES:[DI]		; Read value
	INC	DI			; Next
	RET
;
?STAXD:	MOV	BX,CX			; Get address
	MOV	AH,AL			; Get A
	JMP	WRBX			; Write it
;
?INXD:	INC	CX			; Increment
	RET
;
?INRD:	INC	CH			; Increment HI
	JMP	SFZSPA			; Save flags
;
?DCRD:	DEC	CH			; Decrement HI
	JMP	SFZSPA			; Save flags
;
?MVID:	MOV	CH,ES:[DI]		; Get value
	INC	DI			; Advance
	RET
;
?RAL:	MOV	AH,DGRP:_PSW		; Get PSW
	SAHF				; Set flags
	RCL	AL,1			; Do the rotate
	JMP	SFC			; Save carry
;
?DADD:	MOV	DX,[BX]			; Get value
	ADD	DX,CX			; Add DE
	MOV	[BX],DX			; Resave
	JMP	SFC			; Save carry
?DADCD:	MOV	DX,CX
	JMP	dadcrp
?DSBCD:	MOV	DX,CX
	JMP	dsbcrp
;
?LDAXD:	MOV	BX,CX			; Get address
	CALL	RDBX			; Read memory
	MOV	AL,AH			; Set A
	RET
;
?DCXD:	DEC	CX			; Decrement
	RET
;
?INRE:	INC	CL			; Increment LO
	JMP	SFZSPA			; Save flags
;
?DCRE:	DEC	CL			; Decrement DL
	JMP	SFZSPA			; Save flags
;
?MVIE:	MOV	CL,ES:[DI]		; Read value
	INC	DI
	RET
;
?RAR:	MOV	AH,DGRP:_PSW		; Get PSW
	SAHF				; Set flags
	RCR	AL,1			; Do the rotate
	JMP	SFC			; Save carry
;
?LXIH:	MOV	DL,ES:[DI]		; Read low
	INC	DI			; Next
	MOV	DH,ES:[DI]		; Read high
	INC	DI			; Next
	MOV	[BX],DX			; Save
	RET
;
; Store register pairs direct
?SBCD:	MOV	DX,WORD PTR DGRP:_C
	JMP	SHORT srpd
?SDED:	MOV	DX,CX
	JMP	SHORT srpd
?SSPD:	MOV	DX,SI
	JMP	SHORT srpd
?SHLD:	MOV	DX,[BX]
srpd:	MOV	BX,ES:[DI]		; Get address
	INC	DI			; Next
	INC	DI			; Next
	MOV	AH,DL			; Get L
	CALL	WRBX			; Write it
	MOV	AH,DH			; Get H
	JMP	WRBXI			; Write it
;
?INXH:	INC	WORD PTR [BX]
	RET
;
?INRH:	INC	BYTE PTR DGRP:_H
	JMP	SFZSPA			; Save flags
;
?DCRH:	DEC	BYTE PTR DGRP:_H
	JMP	SFZSPA			; Save flags
;
?MVIH:	MOV	AH,ES:[DI]		; Get data
	INC	DI			; Next
	MOV	DGRP:_H,AH
	RET
;
?DADH:	MOV	DX,[BX]
	ADD	DX,[BX]			; Add HL
	MOV	[BX],DX
	JMP	SFC			; Save carry
?DADCH:	MOV	DX,[BX]
	JMP	dadcrp
?DSBCH:	MOV	DX,[BX]
	JMP	dsbcrp
;
?LBCD:	MOV	BX,OFFSET DGRP:_C
	JMP	SHORT ?LHLD
?LDED:	CALL	ltmpd
	MOV	CX,[BX]
	RET
?LSPD:	CALL	ltmpd
	MOV	SI,[BX]
	RET
ltmpd:	MOV	BX,OFFSET DGRP:_TEMP
?LHLD:	PUSH	BX
	MOV	BX,ES:[DI]		; Get address
	INC	DI			; Next
	INC	DI			; Next
	CALL	RDBX			; Get low
	MOV	DL,AH			; Set L
	CALL	RDBXI			; Get high
	MOV	DH,AH			; Set H
	POP	BX
	MOV	[BX],DX
	RET
;
?DCXH:	DEC	WORD PTR [BX]		; Decrement
	RET
;
?INRL:	INC	BYTE PTR DGRP:_L	; Increment L
	JMP	SFZSPA			; Save flags
;
?DCRL:	DEC	BYTE PTR DGRP:_L	; Decrement L
	JMP	SFZSPA			; Save flags
;
?MVIL:	MOV	AH,ES:[DI]		; Get data
	INC	DI			; Next
	MOV	DGRP:_L,AH
	RET
;
?CMA:	NOT	AL			; Complement
	RET
;
?NEG:	XOR	AH,AH
	SUB	AH,AL
	MOV	AL,AH
	JMP	SNZSPCA
;
?LXISP:	MOV	BL,ES:[DI]		; Get low
	INC	DI			; Next
	MOV	BH,ES:[DI]		; Get high
	INC	DI			; Next
	MOV	SI,BX			; Set SP
	RET
;
?STA:	MOV	BX,ES:[DI]		; Get address
	INC	DI			; Next
	INC	DI			; Next
	MOV	AH,AL			; Get A
	JMP	WRBX			; Write it
	RET
;
?INXSP:	INC	SI			; Increment
	RET
;
?INRM:	CALL	GETM
	CALL	RDBX			; Read value
	INC	AH			; Increment
	PUSHF				; Save flags
	CALL	WRBX			; Write it
	POPF				; Restore flags
	JMP	SFZSPA			; Save flags
;
?DCRM:	CALL	GETM			; Get HL
	CALL	RDBX			; Read value
	DEC	AH			; Decrement
	PUSHF				; Save flags
	CALL	WRBX			; Resave
	POPF				; Restore flags
	JMP	SFZSPA			; Save flags
;
?MVIM:	CALL	GETM			; Get HL
	MOV	AH,ES:[DI]		; Get immed
	INC	DI			; Next
	JMP	WRBX			; Write it
;
?STC:	MOV	AH,DGRP:_PSW		; Get PSW
	OR	AH,01h			; Set CARRY
	MOV	DGRP:_PSW,AH		; Resave
	RET
;
?DADSP:	MOV	DX,[BX]
	ADD	DX,SI			; Add SP
	MOV	[BX],DX
	JMP	SFC			; Save carry
?DADCS:	MOV	DX,SI
	JMP	dadcrp
?DSBCS:	MOV	DX,SI
	JMP	dsbcrp
;
?LDA:	MOV	BX,ES:[DI]		; Get address
	INC	DI			; Next
	INC	DI			; Next
	CALL	RDBX			; Read it
	MOV	AL,AH			; Set A
	RET
;
?DCXSP:	DEC	SI			; Decrement
	RET
;
?INRA:	INC	AL
	JMP	SFZSPA
;
?DCRA:	DEC	AL
	JMP	SFZSPA
;
?MVIA:	MOV	AL,ES:[DI]		; Read data
	INC	DI			; Next
	RET
;
?CMC:	MOV	AH,DGRP:_PSW		; Get PSW
	XOR	AH,01h			; Toggle carry
	MOV	DGRP:_PSW,AH		; Resave
	RET
;
?MOVBC:	MOV	AH,DGRP:_C		; Get C
	MOV	DGRP:_B,AH		; Stash in B
?MOVBB:	RET
?MOVBD:	MOV	DGRP:_B,CH
	RET
?MOVBE:	MOV	DGRP:_B,CL
	RET
?MOVBH:	MOV	AH,DGRP:_H
	MOV	DGRP:_B,AH
	RET
?MOVBL:	MOV	AH,DGRP:_L
	MOV	DGRP:_B,AH
	RET
?MOVBM:	CALL	GETM
	CALL	RDBX			; Read value
	MOV	DGRP:_B,AH
	RET
?MOVBA:	MOV	DGRP:_B,AL
	RET
;
?MOVCB:	MOV	AH,DGRP:_B
	MOV	DGRP:_C,AH
?MOVCC:	RET
?MOVCD:	MOV	DGRP:_C,CH
	RET
?MOVCE:	MOV	DGRP:_C,CL
	RET
?MOVCH:	MOV	AH,DGRP:_H
	MOV	DGRP:_C,AH
	RET
?MOVCL:	MOV	AH,DGRP:_L
	MOV	DGRP:_C,AH
	RET
?MOVCM:	CALL	GETM
	CALL	RDBX
	MOV	DGRP:_C,AH
	RET
?MOVCA:	MOV	DGRP:_C,AL
	RET
;
?MOVDB:	MOV	CH,DGRP:_B
	RET
?MOVDC:	MOV	CH,DGRP:_C
?MOVDD:	RET
?MOVDE:	MOV	CH,CL
	RET
?MOVDH:	MOV	CH,DGRP:_H
	RET
?MOVDL:	MOV	CH,DGRP:_L
	RET
?MOVDM:	CALL	GETM
	CALL	RDBX
	MOV	CH,AH
	RET
?MOVDA:	MOV	CH,AL
	RET
;
?MOVEB:	MOV	CL,DGRP:_B
	RET
?MOVEC:	MOV	CL,DGRP:_C
	RET
?MOVED:	MOV	CL,CH
?MOVEE:	RET
?MOVEH:	MOV	CL,DGRP:_H
	RET
?MOVEL:	MOV	CL,DGRP:_L
	RET
?MOVEM:	CALL	GETM
	CALL	RDBX
	MOV	CL,AH
	RET
?MOVEA:	MOV	CL,AL
	RET
;
?MOVHB:	MOV	AH,DGRP:_B
	MOV	DGRP:_H,AH
	RET
?MOVHC:	MOV	AH,DGRP:_C
	MOV	DGRP:_H,AH
	RET
?MOVHD:	MOV	DGRP:_H,CH
	RET
?MOVHE:	MOV	DGRP:_H,CL
?MOVHH:	RET
?MOVHL:	MOV	AH,DGRP:_L
	MOV	DGRP:_H,AH
	RET
?MOVHM:	CALL	GETM
	CALL	RDBX
	MOV	DGRP:_H,AH
	RET
?MOVHA:	MOV	DGRP:_H,AL
	RET
;
?MOVLB:	MOV	AH,DGRP:_B
	MOV	DGRP:_L,AH
	RET
?MOVLC:	MOV	AH,DGRP:_C
	MOV	DGRP:_L,AH
	RET
?MOVLD:	MOV	DGRP:_L,CH
	RET
?MOVLE:	MOV	DGRP:_L,CL
	RET
?MOVLH:	MOV	AH,DGRP:_H
	MOV	DGRP:_L,AH
?MOVLL:	RET
?MOVLM:	CALL	GETM
	CALL	RDBX
	MOV	DGRP:_L,AH
	RET
?MOVLA:	MOV	DGRP:_L,AL
	RET
;
?MOVMB:	CALL	GETM
	MOV	AH,DGRP:_B
	JMP	WRBX
?MOVMC:	CALL	GETM
	MOV	AH,DGRP:_C
	JMP	WRBX
?MOVMD:	CALL	GETM
	MOV	AH,CH
	JMP	WRBX
?MOVME:	CALL	GETM
	MOV	AH,CL
	JMP	WRBX
?MOVMH:	CALL	GETM
	MOV	AH,DGRP:_H
	JMP	WRBX
?MOVML:	CALL	GETM
	MOV	AH,DGRP:_L
	JMP	WRBX
?MOVMA:	CALL	GETM
	MOV	AH,AL
	JMP	WRBX
;
?MOVAB:	MOV	AL,DGRP:_B
	RET
?MOVAC:	MOV	AL,DGRP:_C
	RET
?MOVAD:	MOV	AL,CH
	RET
?MOVAE:	MOV	AL,CL
	RET
?MOVAH:	MOV	AL,DGRP:_H
	RET
?MOVAL:	MOV	AL,DGRP:_L
	RET
?MOVAM:	CALL	GETM
	CALL	RDBX
	MOV	AL,AH
?MOVAA:	RET
;
?ADDB:	ADD	AL,DGRP:_B
	CALL	FLAGOV
	JMP	XFZSPCA
?ADDC:	ADD	AL,DGRP:_C
	CALL	FLAGOV
	JMP	XFZSPCA
?ADDD:	ADD	AL,CH
	CALL	FLAGOV
	JMP	XFZSPCA
?ADDE:	ADD	AL,CL
	CALL	FLAGOV
	JMP	XFZSPCA
?ADDH:	ADD	AL,DGRP:_H
	CALL	FLAGOV
	JMP	XFZSPCA
?ADDL:	ADD	AL,DGRP:_L
	CALL	FLAGOV
	JMP	XFZSPCA
?ADDM:	CALL	GETM
	CALL	RDBX
	ADD	AL,AH
	CALL	FLAGOV
	JMP	XFZSPCA
?ADDA:	ADD	AL,AL
	CALL	FLAGOV
	JMP	XFZSPCA
;
?ADCB:	MOV	AH,DGRP:_PSW		; Get PSW
	SAHF				; Set FLAGS
	ADC	AL,DGRP:_B
	CALL	FLAGOV
	JMP	XFZSPCA
?ADCC:	MOV	AH,DGRP:_PSW		; Get PSW
	SAHF				; Set FLAGS
	ADC	AL,DGRP:_C
	CALL	FLAGOV
	JMP	XFZSPCA
?ADCD:	MOV	AH,DGRP:_PSW		; Get PSW
	SAHF				; Set FLAGS
	ADC	AL,CH
	CALL	FLAGOV
	JMP	XFZSPCA
?ADCE:	MOV	AH,DGRP:_PSW		; Get PSW
	SAHF				; Set FLAGS
	ADC	AL,CL
	CALL	FLAGOV
	JMP	XFZSPCA
?ADCH:	MOV	AH,DGRP:_PSW		; Get PSW
	SAHF				; Set FLAGS
	ADC	AL,DGRP:_H
	CALL	FLAGOV
	JMP	XFZSPCA
?ADCL:	MOV	AH,DGRP:_PSW		; Get PSW
	SAHF				; Set FLAGS
	ADC	AL,DGRP:_L
	CALL	FLAGOV
	JMP	XFZSPCA
?ADCM:	CALL	GETM
	MOV	AH,DGRP:_PSW		; Get PSW
	SAHF				; Set FLAGS
	PUSHF
	CALL	RDBX
	POPF
	ADC	AL,AH
	CALL	FLAGOV
	JMP	XFZSPCA
?ADCA:	MOV	AH,DGRP:_PSW		; Get PSW
	SAHF				; Set FLAGS
	ADC	AL,AL
	CALL	FLAGOV
	JMP	XFZSPCA
;
?SUBB:	SUB	AL,DGRP:_B
	CALL	FLAGOV
	JMP	XNZSPCA
?SUBC:	SUB	AL,DGRP:_C
	CALL	FLAGOV
	JMP	XNZSPCA
?SUBD:	SUB	AL,CH
	CALL	FLAGOV
	JMP	XNZSPCA
?SUBE:	SUB	AL,CL
	CALL	FLAGOV
	JMP	XNZSPCA
?SUBH:	SUB	AL,DGRP:_H
	CALL	FLAGOV
	JMP	XNZSPCA
?SUBL:	SUB	AL,DGRP:_L
	CALL	FLAGOV
	JMP	XNZSPCA
?SUBM:	CALL	GETM
	CALL	RDBX
	SUB	AL,AH
	CALL	FLAGOV
	JMP	XNZSPCA
?SUBA:	SUB	AL,AL
	CALL	FLAGOV
	JMP	XNZSPCA
;
?SBBB:	MOV	AH,DGRP:_PSW
	SAHF
	SBB	AL,DGRP:_B
	CALL	FLAGOV
	JMP	XNZSPCA
?SBBC:	MOV	AH,DGRP:_PSW
	SAHF
	SBB	AL,DGRP:_C
	CALL	FLAGOV
	JMP	XNZSPCA
?SBBD:	MOV	AH,DGRP:_PSW
	SAHF
	SBB	AL,CH
	CALL	FLAGOV
	JMP	XNZSPCA
?SBBE:	MOV	AH,DGRP:_PSW
	SAHF
	SBB	AL,CL
	CALL	FLAGOV
	JMP	XNZSPCA
?SBBH:	MOV	AH,DGRP:_PSW
	SAHF
	SBB	AL,DGRP:_H
	CALL	FLAGOV
	JMP	XNZSPCA
?SBBL:	MOV	AH,DGRP:_PSW
	SAHF
	SBB	AL,DGRP:_L
	CALL	FLAGOV
	JMP	XNZSPCA
?SBBM:	CALL	GETM
	MOV	AH,DGRP:_PSW
	SAHF
	PUSHF
	CALL	RDBX
	POPF
	SBB	AL,AH
	CALL	FLAGOV
	JMP	XNZSPCA
?SBBA:	MOV	AH,DGRP:_PSW
	SAHF
	SBB	AL,AL
	CALL	FLAGOV
	JMP	XNZSPCA
;
?ANAB:	AND	AL,DGRP:_B
	JMP	SFZSPCA
?ANAC:	AND	AL,DGRP:_C
	JMP	SFZSPCA
?ANAD:	AND	AL,CH
	JMP	SFZSPCA
?ANAE:	AND	AL,CL
	JMP	SFZSPCA
?ANAH:	AND	AL,DGRP:_H
	JMP	SFZSPCA
?ANAL:	AND	AL,DGRP:_L
	JMP	SFZSPCA
?ANAM:	CALL	GETM
	CALL	RDBX
	AND	AL,AH
	JMP	SFZSPCA
?ANAA:	AND	AL,AL
	JMP	SFZSPCA
;
?XRAB:	XOR	AL,DGRP:_B
	JMP	SFZSPCA
?XRAC:	XOR	AL,DGRP:_C
	JMP	SFZSPCA
?XRAD:	XOR	AL,CH
	JMP	SFZSPCA
?XRAE:	XOR	AL,CL
	JMP	SFZSPCA
?XRAH:	XOR	AL,DGRP:_H
	JMP	SFZSPCA
?XRAL:	XOR	AL,DGRP:_L
	JMP	SFZSPCA
?XRAM:	CALL	GETM
	CALL	RDBX
	XOR	AL,AH
	JMP	SFZSPCA
?XRAA:	XOR	AL,AL
	JMP	SFZSPCA
;
?ORAB:	OR	AL,DGRP:_B
	JMP	SFZSPCA
?ORAC:	OR	AL,DGRP:_C
	JMP	SFZSPCA
?ORAD:	OR	AL,CH
	JMP	SFZSPCA
?ORAE:	OR	AL,CL
	JMP	SFZSPCA
?ORAH:	OR	AL,DGRP:_H
	JMP	SFZSPCA
?ORAL:	OR	AL,DGRP:_L
	JMP	SFZSPCA
?ORAM:	CALL	GETM
	CALL	RDBX
	OR	AL,AH
	JMP	SFZSPCA
?ORAA:	OR	AL,AL
	JMP	SFZSPCA
;
?CMPB:	CMP	AL,DGRP:_B
	JMP	SNZSPCA
?CMPC:	CMP	AL,DGRP:_C
	JMP	SNZSPCA
?CMPD:	CMP	AL,CH
	JMP	SNZSPCA
?CMPE:	CMP	AL,CL
	JMP	SNZSPCA
?CMPH:	CMP	AL,DGRP:_H
	JMP	SNZSPCA
?CMPL:	CMP	AL,DGRP:_L
	JMP	SNZSPCA
?CMPM:	CALL	GETM
	CALL	RDBX
	CMP	AL,AH
	JMP	SNZSPCA
?CMPA:	CMP	AL,AL
	JMP	SNZSPCA
;
?RZ:	MOV	AH,DGRP:_PSW
	SAHF
	JZ	?RET
	RET
?RNZ:	MOV	AH,DGRP:_PSW
	SAHF
	JNZ	?RET
	RET
?RC:	MOV	AH,DGRP:_PSW
	SAHF
	JC	?RET
	RET
?RNC:	MOV	AH,DGRP:_PSW
	SAHF
	JNC	?RET
	RET
?RP:	MOV	AH,DGRP:_PSW
	SAHF
	JNS	?RET
	RET
?RM:	MOV	AH,DGRP:_PSW
	SAHF
	JS	?RET
	RET
?RPO:	MOV	AH,DGRP:_PSW
	SAHF
	JPO	?RET
	RET
?RPE:	MOV	AH,DGRP:_PSW
	SAHF
	JPE	?RET
	RET
?RETI:
?RETN:
?RET:	MOV	DI,ES:[SI]	; Get new PC from stack
	INC	SI		; Pop
	INC	SI		; Pop
	RET
;
?PUSHB:	MOV	BX,WORD PTR DGRP:_C
	DEC	SI
	DEC	SI
	MOV	ES:[SI],BX
	RET
?POPB:	MOV	BX,ES:[SI]
	INC	SI
	INC	SI
	MOV	WORD PTR DGRP:_C,BX
	RET
?PUSHD:	DEC	SI
	DEC	SI
	MOV	ES:[SI],CX
	RET
?POPD:	MOV	CX,ES:[SI]
	INC	SI
	INC	SI
	RET
?PUSHH:	MOV	DX,[BX]
	DEC	SI
	DEC	SI
	MOV	ES:[SI],DX
	RET
?POPH:	MOV	DX,ES:[SI]
	INC	SI
	INC	SI
	MOV	[BX],DX
	RET
?PUSHP:	MOV	AH,DGRP:_PSW
	DEC	SI
	DEC	SI
	MOV	ES:[SI],AH
	MOV	ES:1[SI],AL
	RET
?POPP:	MOV	AH,ES:[SI]
	MOV	AL,ES:1[SI]
	INC	SI
	INC	SI
	MOV	DGRP:_PSW,AH
	RET
;
?JZ:	MOV	AH,DGRP:_PSW
	SAHF
	JZ	?JMP
	INC	DI
	INC	DI
	RET
?JNZ:	MOV	AH,DGRP:_PSW
	SAHF
	JNZ	?JMP
	INC	DI
	INC	DI
	RET
?JC:	MOV	AH,DGRP:_PSW
	SAHF
	JC	?JMP
	INC	DI
	INC	DI
	RET
?JNC:	MOV	AH,DGRP:_PSW
	SAHF
	JNC	?JMP
	INC	DI
	INC	DI
	RET
?JP:	MOV	AH,DGRP:_PSW
	SAHF
	JNS	?JMP
	INC	DI
	INC	DI
	RET
?JM:	MOV	AH,DGRP:_PSW
	SAHF
	JS	?JMP
	INC	DI
	INC	DI
	RET
?JPE:	MOV	AH,DGRP:_PSW
	SAHF
	JPE	?JMP
	INC	DI
	INC	DI
	RET
?JPO:	MOV	AH,DGRP:_PSW
	SAHF
	JPO	?JMP
	INC	DI
	INC	DI
	RET
?JMP:	MOV	DI,ES:[DI]
	RET
;
?JRZ:	MOV	AH,DGRP:_PSW
	SAHF
	JZ	?JR
	INC	DI
	RET
?JRNZ:	MOV	AH,DGRP:_PSW
	SAHF
	JNZ	?JR
	INC	DI
	RET
?JRC:	MOV	AH,DGRP:_PSW
	SAHF
	JC	?JR
	INC	DI
	RET
?JRNC:	MOV	AH,DGRP:_PSW
	SAHF
	JNC	?JR
	INC	DI
	RET
?JR:	PUSH	AX
	MOV	AL,ES:[DI]
	INC	DI
	CBW
	ADD	DI,AX
	POP	AX
	RET
?DJNZ:	DEC	BYTE PTR DGRP:_B
	JNZ	?JR
	INC	DI
	RET
;
?CZ:	MOV	AH,DGRP:_PSW
	SAHF
	JZ	?CALL
	INC	DI
	INC	DI
	RET
?CNZ:	MOV	AH,DGRP:_PSW
	SAHF
	JNZ	?CALL
	INC	DI
	INC	DI
	RET
?CC:	MOV	AH,DGRP:_PSW
	SAHF
	JC	?CALL
	INC	DI
	INC	DI
	RET
?CNC:	MOV	AH,DGRP:_PSW
	SAHF
	JNC	?CALL
	INC	DI
	INC	DI
	RET
?CP:	MOV	AH,DGRP:_PSW
	SAHF
	JNS	?CALL
	INC	DI
	INC	DI
	RET
?CM:	MOV	AH,DGRP:_PSW
	SAHF
	JS	?CALL
	INC	DI
	INC	DI
	RET
?CPE:	MOV	AH,DGRP:_PSW
	SAHF
	JPE	?CALL
	INC	DI
	INC	DI
	RET
?CPO:	MOV	AH,DGRP:_PSW
	SAHF
	JPO	?CALL
	INC	DI
	INC	DI
	RET
?CALL:	DEC	SI
	DEC	SI
	MOV	BX,ES:[DI]
	INC	DI
	INC	DI
	MOV	ES:[SI],DI
	MOV	DI,BX
	RET
;
?ADI:	INC	DI
	ADD	AL,ES:-1[DI]
	JMP	SFZSPCA
?ACI:	INC	DI
	MOV	AH,DGRP:_PSW
	SAHF
	ADC	AL,ES:-1[DI]
	JMP	SFZSPCA
?SUI:	INC	DI
	SUB	AL,ES:-1[DI]
	JMP	SNZSPCA
?SBI:	INC	DI
	MOV	AH,DGRP:_PSW
	SAHF
	SBB	AL,ES:-1[DI]
	JMP	SNZSPCA
?ANI:	INC	DI
	AND	AL,ES:-1[DI]
	JMP	SFZSPCA
?ORI:	INC	DI
	OR	AL,ES:-1[DI]
	JMP	SFZSPCA
?XRI:	INC	DI
	XOR	AL,ES:-1[DI]
	JMP	SFZSPCA
?CPI:	INC	DI
	CMP	AL,ES:-1[DI]
	JMP	SFZSPCA
;
?XCHG:	XCHG	CX,[BX]
	RET
?XTHL:	MOV	DX,[BX]		; Get HL
	PUSH	DX		; Save
	MOV	DX,ES:[SI]	; Get T
	MOV	[BX],DX		; Save T
	POP	DX		; Get HL
	MOV	ES:[SI],DX	; Save HL
	RET
?PCHL:	MOV	DI,[BX]
	RET
?SPHL:	MOV	SI,[BX]
	RET
;
; HALT instruction encountered
; Instead of returning to multi-step loop, clear it's return address from
; stack, save the cached regsters & return directly to original caller.
?HLT:	POP	BX		; Skip return address
	DEC	DI		; Back to offending instruction
	CALL	S8080		; Save registers
	POP	BP		; Restore BP
	MOV	AX,1		; Indicate failure
	RET
; Restart instructions
?RST0:
?RST1:
?RST2:
?RST3:
?RST4:
?RST5:
?RST6:
?RST7:	DEC	SI
	DEC	SI
	MOV	ES:[SI],DI
	MOV	BL,ES:-1[DI]
	AND	BL,38h
	XOR	BH,BH
	MOV	DI,BX
	RET
; 8080 OUT instruction
?OUT:	MOV	DL,ES:[DI]		; Get port
	MOV	AH,AL			; Get value
	CALL	XOUT			; Do the out
	INC	DI			; Update PC
	RET
?INAC:	CALL	CIN
	MOV	AL,AH
XINF:	TEST	AH,AH
	JMP	SFZSPA		; Save Z,S,P,A
?INBC:	CALL	CIN
	MOV	DGRP:_B,AH
	JMP	SHORT XINF
?INCC:	CALL	CIN
	MOV	DGRP:_C,AH
	JMP	SHORT XINF
?INDC:	CALL	CIN
	MOV	CH,AH
	JMP	SHORT XINF
?INEC:	CALL	CIN
	MOV	CL,AH
	JMP	SHORT XINF
?INHC:	CALL	CIN
	MOV	DGRP:_H,AH
	JMP	SHORT XINF
?INLC:	CALL	CIN
	MOV	DGRP:_L,AH
	JMP	SHORT XINF
; 8080 IN instruction
?IN:	MOV	DL,ES:[DI]		; Get port
	CALL	XIN			; Read port
	INC	DI			; Next
	MOV	AL,AH			; Set value
	RET
; Read I/O port[DL] and return in AH
CIN:	MOV	DL,DGRP:_C		; Get port
XIN:	XOR	DH,DH			; Zero high
	CALL	S8080			; Save register
	PUSH	DX			; Save pointer
	CALL	_IN8080			; Perform the in
	POP	BX			; Clean stack
	AND	AH,AH			; Error?
	JNZ	iofail			; Operation failed
	MOV	AH,AL			; AH = value
	JMP	L8080			; Restore registers
?OUTCB:	MOV	AH,DGRP:_B
	JMP	SHORT COUT
?OUTCC:	MOV	AH,DGRP:_C
	JMP	SHORT COUT
?OUTCD:	MOV	AH,CH
	JMP	SHORT COUT
?OUTCE:	MOV	AH,DL
	JMP	SHORT COUT
?OUTCH:	MOV	AH,DGRP:_H
	JMP	SHORT COUT
?OUTCL:	MOV	AH,DGRP:_L
	JMP	SHORT COUT
?OUTCA:	MOV	AH,AL			; Get A
; Write value in AH to I/O port[DL]
COUT:	MOV	DL,DGRP:_C		; Get port
XOUT:	XOR	DH,DH			; Zero high
	CALL	S8080			; Save registers
	PUSH	DX			; Pass address
	MOV	AL,AH			; Copy
	PUSH	AX			; Pass value
	CALL	_OUT8080		; Perform the out
	POP	BX			; Clean stack
	POP	BX			; Clean stack
	AND	AX,AX			; Error?
	JNZ	iofail			; Operation failed
	JMP	L8080			; Load registers
iofail:	MOV	SP,DGRP:_SAVSP		; Restore stack
	DEC	WORD PTR DGRP:_PC	; Adjust PC
	POP	BP			; Restore caller BP
	MOV	AX,-2			; Indicate bad I/O
	RET
; Decimal adjust for addition/subtraction
?DAA:	MOV	AH,DGRP:_PSW		; Get PSW
	TEST	AH,02h			; Was it subtract?
	JNZ	?DAS			; Yes, do DAS
	SAHF				; Set FLAGS
	DAA				; Decimal adjust
	JMP	SFZSPCA
?DAS:	SAHF
	DAS
	JMP	SNZSPCA
; Disable interrupts
?DI:	XOR	AH,AH
	MOV	DGRP:_EI,AH
	RET
; Enable interrupts
?EI:	MOV	AH,255
	MOV	DGRP:_EI,AH
	RET
; Swap all register
?EXX:	MOV	DX,WORD PTR DGRP:_C
	XCHG	DX,WORD PTR DGRP:_XC
	MOV	WORD PTR DGRP:_C,DX
	XCHG	CX,WORD PTR DGRP:_XE
	MOV	DX,WORD PTR DGRP:_L
	XCHG	DX,WORD PTR DGRP:_XL
	MOV	WORD PTR DGRP:_L,DX
	RET
; Swap AF and AF'
?EXAF:	MOV	AH,DGRP:_PSW
	XCHG	AX,WORD PTR DGRP:_XA
	MOV	DGRP:_PSW,AH
	RET
; nibble rotates 
?RLD:	CALL	GETM
	MOV	DL,[BX]		; Get memory
	XOR	DH,DH		; Zero high
	MOV	AH,AL		; Copy ACC
	PUSH	CX		; Save CX
	MOV	CL,4		; Shift by 4
	SHL	AH,CL		; put in top
	XOR	CH,CH		; Zero high
rld1:	SHL	AH,1		; Do one bit
	RCL	DX,1		; Into DX
	LOOP	rld1		; Do 4 bits
	POP	CX
	MOV	[BX],DL		; New memory location
	AND	AL,0F0h		; Remove old
	OR	AL,DH		; Add high
	JMP	SFZSPA		; Save flags
?RRD:	CALL	GETM
	MOV	DH,[BX]		; Get memory
	MOV	AH,AL		; Copy ACC
	SHR	AH,1
	RCR	DX,1		; 1-bit
	SHR	AH,1
	RCR	DX,1		; 2-bits
	SHR	AH,1
	RCR	DX,1		; 3-bits
	SHR	AH,1
	RCR	DX,1		; 4-bits
	SHR	DX,1
	SHR	DX,1
	SHR	DX,1
	SHR	DX,1
	MOV	[BX],DL		; Save M
	AND	AL,0F0h		; Clear low
	OR	AL,DH		; Include new low
	JMP	SFZSPA		; Save flags
; Interrupt modes
?IM0:	XOR	AH,AH
setim:	MOV	DGRP:_IM,AH
?IM1:	MOV	AH,1
	JMP	SHORT setim
?IM2:	MOV	AH,2
	JMP	SHORT setim
?LDAI:	MOV	AL,DGRP:_I
	RET
?LDIA:	MOV	DGRP:_I,AL
	RET
?LDAR:	MOV	AL,DGRP:_R
	RET
?LDRA:	MOV	DGRP:_R,AL
	RET
;
ZCP:	CALL	RDBX
	CMP	AL,AH
	JMP	SFZSPCA
ZIN:	MOV	WORD PTR DGRP:_L,BX	; Save BX
	CALL	CIN			; Read the port
	MOV	BX,WORD PTR DGRP:_L	; Restore BX
	MOV	[BX],AH			; Save value
	RET
ZLD:	CALL	RDBX			; Read [HL]
	XCHG	BX,CX			; BX = DE
	CALL	WRBX			; Write [DE]
	XCHG	BX,CX			; Swap back
	RET
ZOUT:	MOV	WORD PTR DGRP:_L,BX	; Save BX
	CALL	RDBX			; Read value
	CALL	COUT			; Write it
	MOV	BX,WORD PTR DGRP:_L	; Restore BX
	RET
;
?ZCPI:	MOV	AH,01h			; incHL
	JMP	SHORT ZCPGO
?ZCPIR:	MOV	AH,61h			; Repeat, Zexit, incHL
	JMP	SHORT ZCPGO
?ZCPD:	MOV	AH,02h			; decHL
	JMP	SHORT ZCPGO
?ZCPDR:	MOV	AH,62h			; Repeat, Zexit, decHL
ZCPGO:	MOV	DX,OFFSET ZCP
	JMP	SHORT RINS
?ZLDI:	MOV	AH,05h			; incDE, incHL
	JMP	SHORT	ZLDGO
?ZLDIR:	MOV	AH,45h			; Repeat, incDE, incHL
	JMP	SHORT	ZLDGO
?ZLDD:	MOV	AH,0Ah			; dec DE, decHL
	JMP	SHORT	ZLDGO
?ZLDDR:	MOV	AH,4Ah			; Repeat, decDE, decHL
ZLDGO:	MOV	DX,OFFSET ZLD
	JMP	SHORT RINS
?ZINI:	MOV	AH,11h			; byteCT, incHL
	JMP	SHORT	ZINGO
?ZINIR:	MOV	AH,51h			; Repeat, byteCT, incHL
	JMP	SHORT	ZINGO
?ZIND:	MOV	AH,12h			; byteCT, decHL
	JMP	SHORT	ZINGO
?ZINDR:	MOV	AH,52h			; Repeat, byteCT, decHL
ZINGO:	MOV	DX,OFFSET ZIN
	JMP	SHORT RINS
?ZOUTI:	MOV	AH,11h			; byteCT, incHL
	JMP	SHORT	ZOTGO
?ZOTIR:	MOV	AH,51h			; Repeat, byteCT, incHL
	JMP	SHORT	ZOTGO
?ZOUTD:	MOV	AH,12h			; byteCT, decHL
	JMP	SHORT	ZOTGO
?ZOTDR:	MOV	AH,52h			; Repeat, byteCT, decHL
ZOTGO:	MOV	DX,OFFSET ZOUT
	JMP	SHORT RINS
;
; Execute repeated instruction: xx RP ZE BT  DC IC DH IH
;
RINS:	MOV	DGRP:_TEMP,DX	; Save address
	CALL	GETM		; BX = M
rins0:	PUSH	AX		; Save flags
	MOV	DX,DGRP:_TEMP	; Get address
	CALL	DX		; Execute function
	POP	AX		; Restore
	PUSH	AX		; & Resave
	SHR	AH,1		; B0=incHL
	JNC	rins1
	INC	BX
rins1:	SHR	AH,1		; B1=decHL
	JNC	rins2
	DEC	BX
rins2:	SHR	AH,1		; B2=incCX
	JNC	rins3
	INC	CX
rins3:	SHR	AH,1		; b3=decCX
	JNC	rins4
	DEC	CX
rins4:	SHR	AH,1		; b4=ByteCT
	JNC	rins5
	DEC	BYTE PTR DGRP:_B
	JZ	rexit
	JMP	short rins6
rins5:	DEC	WORD PTR DGRP:_C
	JZ	rexit
rins6:	SHR	AH,1		; b5=Zexit
	JNC	rins7
	MOV	DL,DGRP:_PSW	; Get flags
	AND	DL,40h		; Z set?
	JNZ	rexit		; Stop
rins7:	SHR	AH,1		; b6=Repeat
	POP	AX		; Restore flags
	JC	rins0
	MOV	WORD PTR DGRP:_L,BX	; Resave HL
	RET
rexit:	POP	AX		; Clean stack
rexit1:	MOV	WORD PTR DGRP:_L,BX	; Resave HL
	RET
; Illegal instruction was encountered
ILLEGAL: POP	BX		; Skip return address
	DEC	DI		; Back to offending instruction
	DEC	DI		; Back to offending instruction
	CALL	S8080		; Save registers
	POP	BP		; Restore BP
	MOV	AX,-1		; Indicate failure
	RET
;
; Get memory address
;
GETM:	CMP	BX,OFFSET DGRP:_L	; Is it HL?
	JNZ	getix
	MOV	BX,[BX]		; BX = HL
	RET
; IX or IY ... add following signed offset
getix:	MOV	BX,[BX]		; Read value
	XOR	DX,DX		; Begin with zero
	OR	DL,ES:[DI]	; Get lower
	JNS	getix1		; Positive - OK
	NOT	DH		; DH=0xFF
getix1:	INC	DI		; Advance
	ADD	BX,DX		; offset
	RET
;
; Flag update functions:
;
; Save C (Carry) flag only
SFC:	MOV	AH,DGRP:_PSW	; Get existing PSW
	JC	SFC1		; Set carry
	AND	AH,0FCh		; Clear C + N
	MOV	DGRP:_PSW,AH	; Resave
	RET
SFC1:	AND	AH,0FDh		; Clear N
	OR	AH,01h		; Set carry bit
	MOV	DGRP:_PSW,AH
	RET
; save Z (Zero), S (Sign), P (Parity), and AC (Auxiliary Carry) flags
SFZSPA:	LAHF			; Get flags
XFZSPA:	MOV	BL,DGRP:_PSW	; Get existing PSW
	AND	AH,0D4h		; Save all but carry (clear N)
	AND	BL,29h		; Save original (clear N)
	OR	AH,BL		; Set new flags
	MOV	DGRP:_PSW,AH	; Save them
	RET
; Save ALL flags: Z (Zero), S (Sign), P (Parity), C (Carry) , AC (Aux. Carry)
SFZSPCA: LAHF			; Get flags
XFZSPCA: AND	AH,0FDh		; Clear N
	MOV	DGRP:_PSW,AH
	RET
; As above, except that 'N' is set
; Save C (Carry) flag only
SNC:	MOV	AH,DGRP:_PSW	; Get existing PSW
	JC	SNC1		; Set carry
	AND	AH,0FEh		; Clear C
	OR	AH,02h		; Set 'N'
	MOV	DGRP:_PSW,AH	; Resave
	RET
SNC1:	AND	AH,0FDh		; Clear N
	OR	AH,01h		; Set carry bit
	MOV	DGRP:_PSW,AH
	RET
; save Z (Zero), S (Sign), P (Parity), and AC (Auxiliary Carry) flags
SNZSPA:	LAHF			; Get flags
XBZSPA:	MOV	BL,DGRP:_PSW	; Get existing PSW
	AND	AH,0D4h		; Save all but carry (clear N)
	AND	BL,2Bh		; Save original (clear N)
	OR	AH,BL		; Set new flags
	OR	AH,02h		; Set N
	MOV	DGRP:_PSW,AH	; Save them
	RET
; Save ALL flags: Z (Zero), S (Sign), P (Parity), C (Carry) , AC (Aux. Carry)
SNZSPCA: LAHF			; Get flags
XNZSPCA: OR	AH,02h		; Set N
	MOV	DGRP:_PSW,AH
	RET
; Copy 8086 Overflow flag to Z80 Parity flag
FLAGOV:	JO	flago1
	LAHF
	AND	AH,0FBh		; Clear P
	RET
flago1:	LAHF
	OR	AH,04h		; Set P
	RET
;
; 8080 Memory access functions:
; we access main storage through these functions so that we can check
; for memory mapped hardware addresses.
;
; Increment BX & write value in AH to ES:[BX]
WRBXI:	INC	BX		; Increment
; Write value to ES:[BX]
WRBX:	MOV	ES:[BX],AH	; Write value
	RET
;
; Increment BX & read value into AH from ES:[BX]
RDBXI:	INC	BX		; Increment
; Read value from ES:[BX] into AH
RDBX:	CMP	BH,DGRP:_DH	; 0E9h; In Disk Controller range?
	JB	RDBXn		; No, normal RAM access
	CMP	BH,0ECh		; In Disk Controller range?
	JB	RDBXs		; Yes, perform "special" read.
; Standard RAM access to ES:[BX]
RDBXn:	MOV	AH,ES:[BX]	; Read value
	RET
; Access to memory-mapped Disk Controller - call mainline
; program to simulate the MDS hardware registers.
RDBXs:	PUSH	BX		; Pass address
	CALL	S8080		; Save registers
	CMP	BYTE PTR DGRP:_DH,0EAh ; Single-Density
	JNZ	RDBXdd		; Read double
	CALL	_diskreadSD	; Read single
	JMP	SHORT RDBXe	;
RDBXdd:	CALL	_diskreadDD	; Read double
RDBXe:	MOV	AH,AL		; Save value
	POP	BX		; Restore BX
	JMP	L8080		; Load registers
;
; Read target memory for outside caller
;
_Mread:	MOV	BX,SP		; Get SP
	MOV	BX,2[BX]	; Get address
	MOV	ES,DGRP:_SEG	; Get segment
	MOV	AL,ES:[BX]	; Read data
	XOR	AH,AH		; Zero high
	RET
;
; Slow down execution by performing a delay loop before each
; instruction.
;
GOSLOW:	MOV	BX,DGRP:_SLOWCT	; Get slowdown count
goslo1:	DEC	BX		; Reduce count
	JNZ	goslo1		; Do them all
	RET
;
; Set system slowdown mode
;
_setslow: MOV	BX,SP		; Address parameters
	MOV	AX,2[BX]	; Get parameter
	MOV	DGRP:_SLOWCT,AX	; Set slowdown count
	MOV	BX,OFFSET mstep2; Point to CALL instruction
	MOV	AX,(OFFSET X8080)-(OFFSET mstep3) ; Get slow offset
	MOV	1[BX],AX	; Patch to call slowed version
	RET
CSEG	ENDS
;
; Initialized data
;
DSEG	SEGMENT	BYTE PUBLIC 'IDATA'
	PUBLIC	_A, _B, _C, _D, _E, _H, _L, _IX, _IY
	PUBLIC	_XA, _XF, _XB, _XC, _XD, _XE, _XH, _XL
	PUBLIC _SP, _PC, _PSW, _SEG, _TRCPTR, _STEP, _BREAK, _BRKSK
_BRKSK	DB	0
_A	DB	0	;1
_PSW	DB	0	;2
_C	DB	0	;3
_B	DB	0	;4
_E	DB	0	;5
_D	DB	0	;6
_L	DB	0	;7
_H	DB	0	;8
_XA	DB	0	;9
_XF	DB	0	;10
_XC	DB	0	;11
_XB	DB	0	;12
_XE	DB	0	;13
_XD	DB	0	;14
_XL	DB	0	;15
_XH	DB	0	;16
_IX	DW	0	;18
_IY	DW	0	;19
_SP	DW	0	;20
_PC	DW	0
_SEG	DW	0
_TRCPTR	DW	0
_STEP	DW	0
_BREAK	DW	0EA00h		; Controller hardware
_TEMP	DW	0
_SAVSP	DW	0		; Saved SP for premature exit
; Status bytes
	PUBLIC	_EI, _IM, _I, _R, _DH
_DH	DB	0E9h		; Disk hardware address
_EI	DB	0		; Interrupts enabled
_IM	DB	0		; Interrupt mode
_I	DB	0		; Interrupt vector
_R	DB	0		; Refresh register
DSEG	ENDS
;
; Uninitialized data
;
BSEG	SEGMENT	BYTE PUBLIC 'UDATA'
	PUBLIC	_TRCBUF, _SLOWCT, _INSTCT
_SLOWCT	DW	1 DUP(?)
_INSTCT	DW	1 dup(?)
_TRCBUF	DW	4096 DUP(?)
BSEG	ENDS
	END
