#include <stdio.h>
#include <window.h>
#include <file.h>

#define	MAXMEM	0x4000
#define	SFILES	110

// Attributes
#define	HTEXT	0x70		// Help text
#define HLINK	0x71		// Help link
#define	HLINKH	0x17		// Hilited link

// Help indexes
#define	HCMD	1			// Command line
#define	HMAIN	2			// Main menu
#define	HRUN	3			// Running
#define	HEDITM	4			// Memory editor
#define	HEDITR	5			// Register editor
#define	HEDITF	6			// Flags editor
#define	HEDITP	7			// EPROM editor
#define	HFILE	8			// File selection
#define	HENTER	9			// Manual entry
#define	HGETV	10			// Get value
#define	HSIO	11			// Set I/O

struct WINDOW
	*dwin,					// Disassembly window
	*swin,					// Status window
	*rwin,					// Register window
	*mwin,					// Memory window
	*twin;					// TTY window

unsigned
	Break = -1,				// Breakpoint
	Mdisplay,				// Memory display address
	Ddisplay,				// Disassembler display address
	Daddr,					// Disassembler current address
	Daddrs[9];				// Disassembler on-screen addresses

unsigned char
	I0, I1, I2,				// Input ports
	O0, O1, O2,				// Output ports
	Ptype,					// PROM type
	Eflag = 0x55,			// Error output
	Octal = 255,			// Octal display
	Octals = 255,			// Split octal
	Kbuf,					// Input key buffer
	*ptr,					// General pointer
	*Dip,					// Active directory pointer
	Directory[70],			// Directory name buffer
	temp[80],				// Temp storage
	ttybuf[24][81],			// TTY buffer
	ttyX,					// TTY x position
	ttyY,					// TTY y position
	*Dout;					// Disassembler output pointer

FILE
	*fp;					// General file pointer

extern unsigned
	PC, FLAGS, SEG, STACK[8], CYCLE;

extern unsigned char
	HLT, A, B, C, D, E, H, L, SP;

extern unsigned char
	*ARGV[];

/* Table of instruction opcodes: MASK, COMPARE, TYPE/LENGTH, TEXT */
unsigned char itable[] = {
	0xFE, 0x00, 0x01, 'H', 'L', 'T', 0,
	0xFF, 0xFF, 0x01, 'H', 'L', 'T', 0,

	0xC0, 0xC0, 0x01, 'L', 'd', 's', 0,
	0xC7, 0x06, 0x02, 'L', 'd', 'I', ' ', 0,
	0xC7, 0x00, 0x01, 'I', 'N', 'd', 0,
	0xC7, 0x01, 0x01, 'D', 'C', 'd', 0,
	0xF8, 0x80, 0x01, 'A', 'D', 's', 0,
	0xF8, 0x88, 0x01, 'A', 'C', 's', 0,
	0xF8, 0x90, 0x01, 'S', 'U', 's', 0,
	0xF8, 0x98, 0x01, 'S', 'B', 's', 0,
	0xF8, 0xA0, 0x01, 'N', 'D', 's', 0,
	0xF8, 0xA8, 0x01, 'X', 'R', 's', 0,
	0xF8, 0xB0, 0x01, 'O', 'R', 's', 0,
	0xF8, 0xB8, 0x01, 'C', 'P', 's', 0,

	0xFF, 0x04, 0x02, 'A', 'D', 'I', ' ', 0,
	0xFF, 0x0C, 0x02, 'A', 'C', 'I', ' ', 0,
	0xFF, 0x14, 0x02, 'S', 'U', 'I', ' ', 0,
	0xFF, 0x1C, 0x02, 'S', 'B', 'I', ' ', 0,
	0xFF, 0x24, 0x02, 'N', 'D', 'I', ' ', 0,
	0xFF, 0x2C, 0x02, 'X', 'R', 'I', ' ', 0,
	0xFF, 0x34, 0x02, 'O', 'R', 'I', ' ', 0,
	0xFF, 0x3C, 0x02, 'C', 'P', 'I', ' ', 0,

	0xFF, 0x02, 0x01, 'R', 'L', 'C', 0,
	0xFF, 0x0A, 0x01, 'R', 'R', 'C', 0,
	0xFF, 0x12, 0x01, 'R', 'A', 'L', 0,
	0xFF, 0x1A, 0x01, 'R', 'A', 'R', 0,

	0xC7, 0x44, 0x03, 'J', 'M', 'P', ' ', 0,
	0xC7, 0x40, 0x03, 'J', 'c', ' ', 0,
	0xC7, 0x46, 0x03, 'C', 'A', 'L', ' ', 0,
	0xC7, 0x42, 0x03, 'C', 'c', ' ', 0,
	0xC7, 0x07, 0x01, 'R', 'E', 'T', 0,
	0xC7, 0x03, 0x01, 'R', 'c', 0,

	0xC7, 0x05, 0x01, 'R', 'S', 'T', ' ', 'r', 0,
	0xF1, 0x41, 0x01, 'I', 'N', 'P', ' ', 'i', 0,
	0xC1, 0x41, 0x01, 'O', 'U', 'T', ' ', 'o', 0,

	/*  This entry always matches invalid opcodes */
	0x00, 0x00, 0x01, '?', 0 };

/* Tables to convert register index into actual names */
unsigned char regtab[]	= { 'A', 'B', 'C', 'D', 'E', 'H', 'L', 'M' };
unsigned char *cctab[] = { "FC", "FZ", "FS", "FP", "TC", "TZ", "TS", "TP" };

unsigned char mgo[] = { "GO:1=Hlp 2=Clr 3=Stp 4=" };

unsigned char MON8[] =
#include "R:MONITOR8.H"

/*
 * Shared format routine used by the various versions of "printf".
 * Format spec. and operands are passed as a pointer to the calling
 * functions argument list.
 */
_format_(unsigned *inptr, char *outptr)
{
	char outstk[17], *ptr, *optr, *format, justify, zero, minus, chr;
	unsigned width, value, i, pad;

	optr = outptr;
	format = *--inptr;

	while(chr = *format++) {
		if(chr == '%') {					/* format code */
			chr = *format++;
			*(ptr = &outstk[16]) = justify = minus = width = value = i = pad = 0;
			zero = ' ';
			if(chr == '-') {				/* left justify */
				justify = -1;
				chr = *format++; }
			if(chr == '0')					/* leading zeros */
				zero = '0';
			while(isdigit(chr)) {			/* field width specifier */
				width = (width * 10) + (chr - '0');
				chr = *format++; }

			value = *--inptr;				/* get parameter value */

			switch(chr) {
				case 'd' :					/* decimal number */
					if(value & 0x8000) {
						value = -value;
						minus = -1; }
				case 'u' :					/* unsigned number */
					i = 10;
					break;
				case 'o' :					/* octal number */
					i = 8;
					break;
				case 'b' :					/* binary number */
					i = 2;
					break;
				case 'B' :
					zero = '0';
					if(Octal) {
						i = 8;
						width = 3;
						break; }
					width = 2;
					pad = 1;
				case 'x' :					/* hexidecimal number */
					i = 16;
					break;
					break;
				case 'W' :
					zero = '0';
					if(Octal) {
						i = 8;
						if(Octals) {
							*optr++ = (value >> 14) + '0';
							*optr++ = ((value >> 11) & 7) + '0';
							*optr++ = ((value >> 8) & 7) + '0';
							value &= 255;
							width = 3;
							break; }
						width = 6;
						break; }
					width = 4;
					pad = 2;
					i = 16;
					break;
				default:					/* Unknown */
					++inptr;
					value = chr;
				case 'c' :					/* character data */
					*--ptr = value;
					break;
				case 's' :					/* string */
					ptr = value; }

			if(i)		/* for all numbers, generate the ASCII string */
				do {
					if((chr = (value % i) + '0') > '9')
						chr += 7;
					*--ptr = chr; }
				while(value /= i);

/* output sign if any */
			if(minus) {
				*optr++ = '-';
				if(width)
					--width; }

/* pad with 'zero' value if right justify enabled  */
			if(width && !justify) {
				for(i = strlen(ptr); i < width; ++i)
					*optr++ = zero; }

/* move in data */
			i = 0;
			value = width - 1;
			while((*ptr) && (i <= value)) {
				*optr++ = *ptr++;
				++i; }
			while(pad) {
				*optr++ = ' ';
				--pad; }

/* pad with 'zero' value if left justify enabled */
			if(width && justify) {
				while(i < width) {
					*optr++ = zero;
					++i; } } }
		else
/* not a format code, simply display the character */
			*optr++ = chr; }

	*optr = 0;
	return optr - outptr;
}

/*
 * Display a title on a window
 */
void wintitle(char *ptr)
{
	char buffer[6];
	memcpy(buffer, W_OPEN, 6);
	W_OPEN[1] &= (~WBOX3 >> 8);
	--W_OPEN[3];
	wgotoxy(0,0);
	wprintf(" %s ", ptr);
	memcpy(W_OPEN, buffer, 6);
}

/*
 * Help commnd - Implements "hypertext" screens
 */
/* General parameters */
#define	LINK_S		('N'-0x40)		/* Start of a link */
#define	LINK_E		('O'-0x40)		/* End of a link */
//#define	USER_CODE	('Q'-0x40)		/* User code */
#define	TAB_SIZE	4				/* Spacing of TABs */
#define	LINKS		25				/* Maximum # links per screen */
help(unsigned section)
{
	int c, lx, ly, ox, oy;
	unsigned i, j, size, ncount;
	unsigned char buffer[65], *ptr;
	unsigned char xs[LINKS], ys[LINKS], names[LINKS][25], link[LINKS];

	ptr = ARGV[i=j=0];
	while(*ptr) {
		if((buffer[i++] = *ptr++) == '.')
			j = i; }
	strcpy(buffer+j, "HLP");
	if(!(fp = fopen(buffer, "rb"))) {
		beep(1000, 250);
		return; }

	/* Locate the section in question */
lookup:
	size = getc(fp);
	size |= getc(fp) << 8;
	if(section--) {
		fseek(fp, 0, size, 1);
		goto lookup; }

	/* Draw the screen */
	wopen(5, 4, 70, 17, WSAVE|WCOPEN|WBOX1|HTEXT);
	wcursor_off();
	i = ncount = ptr = 0;
	while(buffer[i++] = getc(fp));
	wintitle(buffer);
	wgotoxy(0, 0);
	while(i++ < size) switch(c = getc(fp)) {
		case LINK_S :	/* Start link */
			ptr = names[ncount];
			xs[ncount] = W_OPEN->WINcurx;
			ys[ncount] = W_OPEN->WINcury;
			*W_OPEN = HLINK;
			break;
		case LINK_E :	/* End link */
			link[ncount++] = getc(fp);
			*ptr = ptr = 0;
			*W_OPEN = HTEXT;
#ifndef USER_CODE
			++i;
			break;
#else
		extra:
			++i;
			break;
		case USER_CODE:	/* User control code */
			user_help(getc(fp));
			goto extra;
#endif
		case '\t' :		/* Horizontal TAB */
			do
				wputc(' ');
			while(W_OPEN->WINcurx % TAB_SIZE);
			break;
		case 0 :
			c = '\n';
		default:
			wputc(c);
			if(ptr)
				*ptr++ = c; }

	/* Allow user to select field & link to new screen */
	i = section = 0;
	for(;;) {
		wgotoxy(xs[section], ys[section]);
		*W_OPEN = HLINK;
		wputs(names[section]);
		wgotoxy(xs[i], ys[i]);
		*W_OPEN = HLINKH;
		wputs(names[section = i]);
		*W_OPEN = HTEXT;
		switch(c = wgetc()) {		/* User keystroke */
			case _KLA :				/* Left arrow - previous field */
				i = (i ? i : ncount) - 1;
				break;
			case _KRA :				/* Right arrow - next field */
				i = (i + 1) % ncount;
				break;
			case _KUA : ox = oy = -1000; goto dofind;
			case _KDA : ox = oy = 1000;
			dofind:
				size = i;
				for(j = 0; j < ncount; ++j) {
					lx = (int)xs[j] - (int)xs[i];
					ly = (int)ys[j] - (int)ys[i];
					if(c == _KUA) {
						if((ly >= 0) || (ly < oy)) continue; }
					else {
						if((ly <= 0) || (ly > oy))	continue; }
					if(abs(lx) > abs(ox)) continue;
					size = j;
					ox = lx;
					oy = ly; }
				i = size;
				break;
			case '\n' :				/* Enter - chain to menu */
				rewind(fp);
				section = link[i];
				wclose();
				goto lookup;
			case 0x1B:				/* Escape exit */
				wclose();
				fclose(fp);
				return; } }
}

/*
 * Update status line
 */
register status(unsigned args)
{
	unsigned char buffer[81];
	_format_(nargs()*2 + &args, buffer);
	*swin = 0x70;
	w_clwin(swin);
	w_puts(buffer, swin);
}

/*
 * Display error message
 */
register error(unsigned args)
{
	unsigned char buffer[81];
	_format_(nargs()*2 + &args, buffer);
	*swin = 0x47;
	w_clwin(swin);
	w_puts(buffer, swin);
	Eflag = 255;
}

/*
 * Display a character in the TTY window
 */
void Tputc(int c)
{
	switch(c) {
	case 0x0A :
		if(++ttyY >= 24)
			ttyY = 0;
		memset(ttybuf[ttyY], ' ', 80);
		w_putc(c, twin);
		goto dopos;
	case 0x0D :
		ttyX = 0;
	dopos:
		w_gotoxy(ttyX, twin->WINheight-1, twin);
	case 0x7F :
		return;
	case 0x08 :
		if(!ttyX)
			return;
		--ttyX;
		goto dopos; }

	if(c < ' ')
		return;
	if(ttyX < 80) {
		ttybuf[(unsigned)ttyY][(unsigned)ttyX++] = c;
		w_putc(c, twin);
		goto dopos; }
}

/* register Tprintf(unsigned args)
{
	unsigned char buffer[81];
	_format_(nargs()*2 + &args, buffer);
	w_puts(buffer, twin);
} */

/*
 * Output character to disassembly output buffer
 */
void Dputc(unsigned char c)
{
	if(Dout)
		*Dout++ = c;
}

/*
 * Output formatted string to disassembly output buffer
 */
register Dprintf(unsigned args)
{
	unsigned char buffer[50], *p;
	_format_(nargs() * 2 + &args, p = buffer);
	while(*p)
		Dputc(*p++);
}

/*
 * Disassemble one instruction
 */
unsigned *disasm(void)
{
	unsigned l, v;
	unsigned char c, o, *ip;

	o = peek(SEG, v = Daddr++);
	Dprintf("%W %B ", v, o);

	ip = itable;
	while((*ip++ & o) != *ip++) {
		++ip;
		while(*ip++); }
	l = *ip++;
	if(l > 1) {
		v = peek(SEG, Daddr++);
		Dprintf("%B ", v); }
	else
		Dprintf("    ");
	if(l > 2) {
		v |= (peek(SEG, Daddr++) << 8);
		Dprintf("%B ", v >> 8); }
	else
		Dprintf("    ");

	while(*ip) switch(c = *ip++) {
		case 'r' :		/* Interrupt vector */
			Dputc(((o >> 3) & 7) + '0');
			continue;
		case 'd' :		/* Destination register */
			Dputc(regtab[(o >> 3) & 7]);
			continue;
		case 's' :		/* Source register */
			Dputc(regtab[o & 7]);
			continue;
		case 'c' :		/* Condition */
			Dprintf("%s", cctab[(o>>3) & 7]);
			continue;
		case 'i' :		/* Input port */
			Dprintf("%o", (o >> 1) & 0x07);
			continue;
		case 'o' :		/* Output port */
			Dprintf("%o", (o >> 1) & 0x1F);
			continue;
		default:
			Dputc(c); }

	switch(l) {
		case 2 :		/* Single byte operand */
			Dprintf("%B", v);
			break;
		case 3 :		/* Double byte operand */
			Dprintf("%W", v); }

	if(Dout)
		*Dout = 0;
	return l;
}

/*
 * Terminate program
 */
void terminate()
{
	w_close(twin);
	if(mwin) {
		w_close(mwin);
		w_close(rwin);
		w_close(dwin); }
	w_close(swin);
	exit(0);
}

/*
 * Refresh register display window
 */
void refresh_registers(void)
{
	unsigned i, j;

	j = SP / 2;
	W_OPEN = rwin;
	wgotoxy(0, 0);
	i = *(unsigned*)&L;
	wprintf("A:%B\nB:%B\nC:%B\nD:%B\nE:%B\nH:%B\nL:%B\nM:%B\n",
		A, B, C, D, E, H, L, peek(SEG, i & (MAXMEM-1)));
	wputc((FLAGS & 0x01) ? 'C' : 'c');
	wputc((FLAGS & 0x04) ? 'P' : 'p');
	wputc((FLAGS & 0x40) ? 'Z' : 'z');
	wputc((FLAGS & 0x80) ? 'S' : 's');
	wputc(HLT ? 'H' : 'h');
	wprintf(" S:%u", j);
	for(i=0; i < 8; ++i) {
		wgotoxy(5, i);
		if(i == j) {
			wprintf(">%W", PC);
			continue; }
		wprintf(" %W", STACK[i]); }
}

/*
 * Refresh memory window
 */
void refresh_memory(void)
{
	unsigned x, y, a, c, L, M, W, S;
	unsigned char *F;
	if(Octal) {
		W = 6;
		L = 4;
		S = 10;
		M = MAXMEM - (9*8);
		F = "%03o"; }
	else {
		W = 3;
		L = 8;
		S = 7;
		M = MAXMEM - (9*4);
		F = "%02x"; }

	W_OPEN = mwin;
	Mdisplay &= (MAXMEM-1);
	if(Mdisplay > M)
		Mdisplay = M;
	for(y=0; y < 9; ++y) {
		wgotoxy(0, y);
		wprintf("%W", a = (y*L)+Mdisplay);
		for(x=0; x < L; ++x) {
			wgotoxy((x*W)+S, y);
			wprintf(F, c = peek(SEG, a+x));
			if((c < ' ') || (c > 0x7E))
				c = '.';
			wgotoxy((39-L)+x, y);
			wputc(c); } }
}

/*
 * Refresh disassembler window
 */
void refresh_disassembler(void)
{
	unsigned i;

	W_OPEN = dwin;
	Daddr = Ddisplay;
	for(i=0; i < 9; ++i) {
		Dout = temp;
		wgotoxy(0, i);
		if(Daddr == PC)
			*W_OPEN = (Daddr == Break) ? 0x31 : 0x71;
		else
			*W_OPEN = (Daddr == Break) ? 0x13 : 0x17;
		Daddrs[i] = Daddr;
		disasm();
		wcleol();
		wputs(temp); }
}

void update_octal()
{
	struct WINDOW *w;
	w = W_OPEN;
	Octal = Octal ? 0 : 255;
	refresh_disassembler();
	refresh_registers();
	w_clwin(mwin);
	refresh_memory();
	W_OPEN = w;
}

/*
 * Move to position if not already on screen
 */
int showdis(unsigned addr)
{
	unsigned i;
	for(i=0; i < 9; ++i) {
		if(Daddrs[i] == addr)
			return 0; }
	Ddisplay = addr;
	return 255;
}

/*
 * Backup disassembler
 */
void backdis(unsigned n)
{
	unsigned i, a[33];
	Daddr = (Ddisplay < 32) ? 0 : Ddisplay - 32;
	Dout = 0;
	for(i=0; i < (sizeof(a)/2); ++i) {
		a[i] = Daddr;
		if(Daddr >= Ddisplay) {
			Ddisplay = (i < n) ? a[0] : a[i-n];
			return; }
		disasm(); }
}

/*
 * Convert ASCII<>HEX
 */
unsigned cvdig(unsigned *a, unsigned base)
{
	unsigned c;
	if(base & 0x80) {
		base &= 0x7F;
		*a = wgetc(); }
	c = *a;
	if((c >= '0') && (c <= '9'))
		c -= '0';
	else if((c >= 'A') && (c <= 'F'))
		c -= ('A'-10);
	else if((c >= 'a') && (c <= 'f'))
		c -= ('a'-10);
	else
		return 255;
	if(c >= base)
		return 255;
	wprintf("%x", *a = c);
	return 0;
}

/*
 * Get value from prompted input
 */
int getval(unsigned char *prompt, unsigned *r, unsigned w)
{
	unsigned i, s, c, d[6];
	unsigned char *bp;
	static unsigned char hb[] = { 16, 16, 16, 16 };
	static unsigned char os[] = { 4, 8, 8, 4, 8, 8 };
	static unsigned char ol[] = { 2, 8, 8, 8, 8, 8 };

	if(Octal) {
		bp = os;
		s = 3;
		if(w) {
			s = 6;
			if(!Octals)
				bp = ol; } }
	else {
		bp = hb;
		s = w ? 4 : 2; }
	status("Enter %s (%u %s digits)?", prompt, s, Octal ? "Octal" : "Hex");
	wcursor_line();
	i = 0;
	for(;;) switch(c = wgetc()) {
		case _K1 : help(HGETV);		continue;
		case _KBS:
			if(i) {
				--i;
				wputc('\b'); }
			continue;
		case 0x1B: return 255;
		default:
			if(!cvdig(&c, bp[i])) {
				d[i] = c;
				if(++i >= s) {
					if(Octal) {
						if(w && !Octals) {
							*r = (d[0]<<15)|(d[1]<<12)|(d[2]<<9)|(d[3]<<6)|(d[4]<<3)|d[5];
							return 0; }
						c = (d[0]<<6) | (d[1]<<3) | d[2];
						*r = w ? (c<<8) | (d[3]<<6) | (d[4]<<3) | d[5] : c; }
					else {
						c = (d[0] << 4) | d[1];
						*r = w ? (c<<8) | (d[2]<<4) | d[3] : c; }
					return 0; }
				continue; }
			beep(1000, 100); }
}

/*
 * Memory editor
 */
void edit_memory()
{
	unsigned c, a, b, x, y, L, P, M, X, B, W, S;
	unsigned char f;
	static int o;
	static unsigned char m;

reoct:
	if(Octal) {
		B = 2;
		W = 6;
		S = 10; }
	else {
		B = 3;
		W = 3;
		S = 7; }
	a = L = 1 << B;
	X = L - 1;
	P = L * 9;
	M = MAXMEM - P;

restat:
	status("Memory: F1=Help F2=%s F3=Adr F4=%s F10=Quit", Octal ? "Hex" : "Oct", m ? "Num" : "Chr");
redraw:
	refresh_memory();
	wcursor_line();
	for(;;) {
		f = 0;
		while(o < 0) {
			if(Mdisplay < a) {
				Mdisplay = o = 0;
				break; }
			Mdisplay -= a;
			o += a;
			f = 255; }
		while(o >= P) {
			Mdisplay += a;
			o -= a;
			f = 255; }
		if(Mdisplay > M) {
			Mdisplay = M;
			goto redraw; }
		if(f) goto redraw;
		x = o & X;
		y = o >> B;
		wgotoxy(m ? (39-L)+x : (x*W)+S, y);
		switch(c = wgetc()) {
		case _KLA : o -= (a = 1);							continue;
		case _KRA : o += (a = 1);							continue;
		case _KHO : if(o & X) { o &= ~X;					continue; }
		case _KUA : o -= (a=L);								continue;
		case _KEN : if((o&X)!=X) { o |= X;					continue; }
		case _KDA : o += (a=L);								continue;
		case _KPU : o -= (a=P);								continue;
		case _KPD : o += (a=P);								continue;
		case _CHO : Mdisplay &= ~X; o &= ~X;				goto redraw;
		case _CEN : Mdisplay &= ~X; o |= X;					goto redraw;
		case _CPU : Mdisplay = o = 0;						goto redraw;
		case _CPD : Mdisplay = M; o = P-1;					goto redraw;
		case _CRA : if(Mdisplay < M) ++Mdisplay;			goto redraw;
		case _CLA : if(Mdisplay) --Mdisplay;				goto redraw;
		case _K1  : help(HEDITM);							continue;
		case _K2  : update_octal();							goto reoct;
		case _K3  :
			wgotoxy(0, 0);
			if(!getval("Address", &Mdisplay, 255))
				o = 0;
			goto restat;
		case _K4  : m = m ? 0 : 255;						goto restat;
		case 0x1B : if(m) break;
		case _K10:	wcursor_off(); return; }

		if(c & 0xFF00) continue;
		if(m) {
			wputc(c);
			wgotoxy((x*W)+S, y);
			wprintf("%B", c); }
		else {
			if(Octal) {
				if(cvdig(&c, 4)) goto redraw;
				if(cvdig(&a, 0x88)) goto redraw;
				if(cvdig(&b, 0x88)) goto redraw;
				c = (c << 6) | (a << 3) | b; }
			else {
				if(cvdig(&c, 16)) goto redraw;
				if(cvdig(&a, 0x90)) goto redraw;
				c = (c << 4) | a; }
			wgotoxy((39-L)+x, y);
			wputc( ((c < ' ') || (c > 0x7E)) ? '.' : c); }
		poke(SEG, Mdisplay+o, c);
		o += (a = 1); }
}

/*
 * Register editor
 */
void edit_registers()
{
	unsigned c, v;
	W_OPEN = rwin;

redraw:
	status("Register: A, B, C, D, E, H, L, F, S, P, 0-7 F1=Help F2=%s F10=Quit?", Octal ? "Hex" : "Oct");
	refresh_registers();
	wcursor_off();
	for(;;) switch(c=toupper(wgetc())) {
	case 'A' : wgotoxy(2, 0); if(!getval("A", &v, 0)) A = v;	goto redraw;
	case 'B' : wgotoxy(2, 1); if(!getval("B", &v, 0)) B = v;	goto redraw;
	case 'C' : wgotoxy(2, 2); if(!getval("C", &v, 0)) C = v;	goto redraw;
	case 'D' : wgotoxy(2, 3); if(!getval("D", &v, 0)) D = v;	goto redraw;
	case 'E' : wgotoxy(2, 4); if(!getval("E", &v, 0)) E = v;	goto redraw;
	case 'H' : wgotoxy(2, 5); if(!getval("H", &v, 0)) H = v;	goto redraw;
	case 'L' : wgotoxy(2, 6); if(!getval("L", &v, 0)) L = v;	goto redraw;
	case 'M' : wgotoxy(2, 7);
		if(!getval("M", &v, 0)) {
			poke(SEG, (*(unsigned*)&L) & (MAXMEM-1), v);
			refresh_disassembler();
			refresh_memory(); }
		goto redraw;
	case 'S' : wgotoxy(8, 8); wcursor_line(); v = wgetc() - '0';
		if(v < 8)
			SP = v*2;
		goto redraw;
	case 'F' : status("Flag: C, P, Z, S, H ?");
	ref: switch(toupper(wgetc())) {
		case _K1 : help(HEDITF);	goto ref;
		case 'C' : FLAGS ^= 0x01;	goto redraw;
		case 'P' : FLAGS ^= 0x04;	goto redraw;
		case 'Z' : FLAGS ^= 0x40;	goto redraw;
		case 'S' : FLAGS ^= 0x80;	goto redraw;
		case 'H' : HLT = HLT ? 0 : 255; }
		goto redraw;
	case _K1 :	help(HEDITR);		continue;
	case _K2 :	update_octal();		continue;
	case 0x1B:
	case _K10 :	return;
	case 'P' : c = (SP / 2) + '0';
	default:
		if((c -= '0') < 8) {
			wgotoxy(6, c);
			if(!getval("PC", &v, 255)) {
				STACK[c] = (v &= (MAXMEM-1));
				if((c*2) == SP) {
					PC = v;
					refresh_disassembler(); } }
			goto redraw; } }
}

/*
 * Select a file via window panel
 */
int selectfile(unsigned char *match)
{
	unsigned a, i, Nt, Pt;
	unsigned char name[13], *n[SFILES], pool[SFILES*9], *p;
	static unsigned char *Dip;

	W_OPEN = swin;
	// Find end of directory index
	if(Dip)
		*Dip = 0;
newdir:
	status("Select file: F1=Help  F2=ManualEntry");
	Dip = Directory; while(*Dip) ++Dip;
	if(*(Dip-1) == '\\')
		--Dip;
	*Dip = '\\';
	// Scan for all files & build name list
	strcpy(Dip+1, "*.*");
	Nt = Pt = 0;
	if(!find_first(Directory, DIRECTORY, p = name, &i, &i, &a, &i, &i)) do {
		if(Nt >= SFILES)
			break;
		if(a & DIRECTORY) {
			if(!strcmp(p, "."))
				continue;
			n[Nt++] = &pool[Pt];
			pool[Pt++] = 1;	// Insure sorts first!
			while(pool[Pt++] = *p++);
			continue; }
		while(*p) {
			if(!strcmp(p, match)) {
				*p = 0;	p = name;
				n[Nt++] = &pool[Pt];
				while(pool[Pt++] = *p++);
				break; }
			++p; } }
	while !find_next(p=name, &i, &i, &a, &i, &i);

	*Dip = 0;
	if(!Nt) goto manent;

	// Sort the name list
	for(i=0; i < Nt; ++i) {
		for(a=i+1; a < Nt; ++a) {
			if(strcmp(n[a], n[i]) < 0) {
				p = n[i];
				n[i] = n[a];
				n[a] = p; } } }
	// Convert directory pref
	for(i=0; i < Nt; ++i) {
		if(*(p = n[i]) != 1)
			break;
		while(p[1]) {
			*p = p[1];
			++p; }
		*p = '\\'; }

	i = ((Nt + 4) / 5) + 2;
	wopen(4, 12-(i/2), 71, i, WSAVE|WBOX1|WCOPEN|0x70);
	wintitle(Directory);
	wcursor_off();
	a = 0;
draw:
	for(i=0; i < Nt; ++i) {
		wgotoxy((i % 5) * 14, i / 5);
		*W_OPEN = (i == a) ? 0x07 : 0x70;
		wputs(n[i]); }
	*W_OPEN = 0x70;
	for(;;) switch(wgetc()) {
		case _KLA : if(a) { --a; goto draw; }
		case _KEN : a = Nt-1; goto draw;
		case _KRA : if(++a < Nt) goto draw;
		case _KHO : a = 0;	goto draw;
		case _KUA : if(a > 4) a -= 5; goto draw;
		case _KDA : if((a+5) < Nt) a += 5; goto draw;
		case '?' :
		case _K1  : help(HFILE); continue;
		case _K2  :
			wclose();
		manent:
			strcpy(pool, Directory); Pt=Dip;
			wclwin(); wputs("File?");
			*Dip = '\\'; *(Dip+1) = 0;
		manent1:
			switch(wgets(0x106, 0, p = Dip = Directory, sizeof(Directory)-1)) {
			case _K1: help(HENTER);
			default : goto manent1;
			case '\n' :
				while(i = *p++) switch(i) {
					case ':' :
					case '\\': Dip = p-1; }
				if(*(p-2) == '\\') goto newdir;
				return 255;
			case 0x1B :
				if(Nt) {
					strcpy(Directory, pool);
					Dip = Pt;
					goto newdir; }
				return 0; }
		case '\n' :
			*Dip = '\\';
			p = n[a]; while(p[1]) ++p;
			if(*p == '\\') {
				*p = 0;
				if(strcmp(p = n[a], "..")) {	// Not UP
					strcpy(Dip+1, p);
					wclose();
					goto newdir; }
				// Go up one directory
				while(Dip > Directory) {
					if(*--Dip == '\\') {
						*Dip = 0;
						break; } }
				wclose();
				goto newdir; }
			strcpy(Dip+1, n[a]);
			wclose();
			return 255;
		case 0x1B :
		case _K10 :
			wclose();
			return 0; }
}

/*
 * Load a byte from the hex file
 */
unsigned load_byte(void)
{
	unsigned char c, v, ct;
	ct = 2;
	do {
		c = *ptr++;
		if((c >= '0') && (c <= '9'))
			c -= '0';
		else if((c >= 'A') && (c <= 'F'))
			c -= ('A'-10);
		else if((c >= 'a') && (c <= 'f'))
			c -= ('a'-10);
		else {
			error("%u: Bad hex digit '%c'[%02x]\n", PC, c, c);
			exit(-1); }
		v = (v << 4) | c; }
	while(--ct);
	return v;
}

/*
 * Copy filename to temp & append extension
 */
void filename(unsigned char *file, unsigned char *extension, unsigned char *n)
{
	unsigned char *d, *p, *e;
	d = temp;
	e = 0;
	p = file;
	while(*d = *file++) {
		switch(*d++) {
		case '.' : e = file-1; continue;
		case '\\':
		case ':' : p = file; e = 0; } }
	if(e)
		*e = 0;
	else
		strcpy(d, extension);
	if(n)
		strcpy(n, p);
}

/*
 * Read file into memory image
 */
void load_file(unsigned char *File)
{
	unsigned a, b, chksum, addr, length, count, size, line, XPC;
	unsigned char buffer[80];

	filename(File, ".HEX", 0);
	if(!(fp = fopen(temp, "r"))) {
		error("Unable to access: %s", temp);
		return; }

	XPC;
	for(size = line = 0; fgets(ptr = buffer, 80, fp);) {
		++line;
		again: switch(*ptr++) {
			case 'S' :	/* Motorola HEX format */
				if(*ptr == '9') goto quitload;
				if(*ptr++ != '1') continue;
				length = count = (chksum = load_byte()) - 3;
				chksum += a = load_byte();
				chksum += b = load_byte();
				addr = (a << 8) + b;
				if(!size)
					XPC = addr;
				while(count--) {
					chksum += a = load_byte();
					poke(SEG, addr++, a); }
				if((255 & ~chksum) != load_byte())
					goto quitbad;
				break;
			case ':' :		/* Intel HEX format */
				if(!(length = count = load_byte())) goto quitload;
				chksum = (a = load_byte()) + length;
				chksum += b = load_byte();
				addr = (a << 8) + b;
				chksum += load_byte();
				if(!size)
					XPC = addr;
				while(count--) {
					chksum += a = load_byte();
					poke(SEG, addr++, a); }
				if((255 & -chksum) != load_byte())
					goto quitbad;
				break;
			case ' ' :		/* Space */
			case '\t' :		/* Tab */
				goto again;
			case 0 :		/* Null line */
				continue;
			default:
				error("%u: Invalid record format", line);
				goto quit; }
		size += length;
		}

quitbad:
	error("%u: Bad record checksum", PC);
quit:
	fclose(fp);
	return;
quitload:
	fclose(fp);
	status("%u bytes loaded at %W", size, XPC);
//	if(LPC == 0xFFFF)
//		LPC = XPC;
}

void writebin(unsigned char b, struct WINDOW *w)
{
	unsigned i;
	for(i=0; i < 8; ++i) {
		w_putc((b & 0x80) ? '1' : '0', w);
		b <<= 1; }
}

void updateio()
{
	w_gotoxy(23, 0, swin);
	w_puts("I:", swin);
	writebin(I0, swin);
	w_putc(' ', swin);
	writebin(I1, swin);
	w_putc(' ', swin);
	writebin(I2, swin);
	w_puts(" O:", swin);
	writebin(O0, swin);
	w_putc(' ', swin);
	writebin(O1, swin);
	w_putc(' ', swin);
	writebin(O2, swin);
}

/*
 * Update I/O port via terminal command
 */
void setio(unsigned char n, unsigned char *p)
{
	unsigned b, c, d, s;
	s = W_OPEN;
	W_OPEN = swin;
	wgotoxy(0, 0);
	wprintf("%23s\n%cport(0-2)?", "", n);
	wcursor_line();
	while((b = wgetc()) == _K1)
		help(HSIO);
	b -= '0';
	if(b < 3) {
		wprintf("\n%23s\nbit(0-7)/Z/O/V?", "");
		p += b;
a1:		switch(b = toupper(wgetc())) {
		case _K1 : help(HSIO);	goto a1;
		case 'Z' : *p = 0;		break;
		case 'O' : *p = 255;	break;
		case 'V' :
			wprintf("\n%23s\nValue(%s)?", "", Octal ? "3-oct" : "2-hex");
			if(Octal) {
				if(!(cvdig(&b, 0x84) || cvdig(&c, 0x88) || cvdig(&d, 0x88)))
					*p = (b << 6) | (c << 3) | d;
				break; }
			if(!(cvdig(&b, 0x90) || cvdig(&c, 0x90)))
				*p = (b << 4) | c;
			break;
		default:
			b -= '0';
			if(b < 8)
				*p ^= (1 << b); } }
	W_OPEN = s;
}

/*
 * Test for input key - stack non-control key for 8008 input
 */
int tstkey()
{
	unsigned c;
	if(c = wtstc()) {
		switch(c) {
		case '\n' : Kbuf = '\r';	return 0;
		case _K1:	help(HRUN);		return 0;
		case _K2:
			w_clwin(twin);
			w_gotoxy(0, twin->WINheight-1, twin);
			memset(ttybuf, 0, sizeof(ttybuf));
			return ttyX = ttyY = 0;
		case _K4 : setio('I', &I0);
			status(mgo);
			updateio();
			wupdatexy();
			return 0;
		case _KBS : Kbuf = 0x7F;	return 0; }
		if(c & 0xFF00)
			return c;
		Kbuf = c; }
	return 0;
}

/*
 * Set teletype screen size
 */
void settty(unsigned size)
{
	unsigned i, s;
	int p;

	w_close(twin);
	twin = wopen(0, size ? 0 : 9, 80, s = size ? 24 : 15, WSAVE|WCOPEN|WSCROLL|0x07);
	p = ttyY - (s-1);
	if(p < 0)
		p += 24;
	for(i=0; i < s; ++i) {
		w_gotoxy(0, i, twin);
		ttybuf[p][80] = 0;
		w_puts(ttybuf[p], twin);
		if(++p >= 24)
			p = 0; }
	w_gotoxy(ttyX, twin->WINheight-1, twin);
}
main(int argc, char *argv[])
{
	unsigned i;
	static unsigned char mprompt[] = {
	"1=Help 2=%s 3=adr 4=mem 5=reg 6=brk 7=epm 8=i/o Go St Nx Ex Rs Ld 10=Quit" };

	for(i=1; i < argc; ++i) {
		ptr = argv[i];
		switch((toupper(*ptr++) << 8) | toupper(*ptr++)) {
		case '-A' :
		case '/A' : Ptype = 255;		continue;
		case '-H' :
		case '/H' : Octal = 0;			continue;
		case '-G' :
		case '/G' : Kbuf = 255;			continue;
		case '-W' :
		case '/W' : Octals = 0;			continue;
		case 'L=' : Daddr = ptr;		continue;
		case '-?' :
		case '/?' :
		case '?'<<8: help(HCMD);		return;
		} printf("Bad option: %s\n", ptr-2);
		exit(-1); }

	SEG = alloc_seg(4096);
	i = 0; do {
		pokew(SEG, i, 0); }
	while(i += 2);
	do {
		poke(SEG, i, MON8[i]); }
	while(++i < sizeof(MON8));
	erase_prom();

	Directory[0] = get_drive() + 'A';
	Directory[1] = ':';
	Directory[2] = '\\';
	getdir(Directory+3);

	swin = wopen(0,24, 80, 1, WSAVE|WCOPEN|0x70);
	twin = wopen(0, 9,  80,15, WSAVE|WCOPEN|WWRAP|WSCROLL|0x07);
	wgotoxy(0, 14);
	status("MOD8 Simulator - Dave Dunfield - "#__DATE__"");
	if(Daddr) {
		load_file(Daddr);
		Eflag = 0x55; }
	if(Kbuf) {
		Kbuf = 0;
		goto dogo; }
reopen:
	dwin = wopen( 0, 0, 29, 9, WSAVE|WCOPEN|0x17);
	rwin = wopen(29, 0, 12, 9, WSAVE|WCOPEN|0x47);
	mwin = wopen(41, 0, 39, 9, WSAVE|WCOPEN|0x17);

restat:
	if(Eflag != 0x55) {
		status(mprompt, Octal ? "Hex" : "Oct");
		Eflag = 0; }

redraw:
	wcursor_off();
	refresh_disassembler();
	refresh_registers();
	refresh_memory();
	Kbuf = 0;
	for(;;) {
		i = toupper(wgetc());
		if(Eflag) {
			status(mprompt, Octal ? "Hex" : "Oct");
			Eflag = 0; }
		switch(i) {
		case _K10: status("Confirm quit (Y/N)?");
			for(;;) switch(wgetc()) {
				case '\n':
				case 'y' :
				case 'Y' :
					terminate();
				case 'n' :
				case 'N' :
				case 0x1B: goto restat; }
		case _KDA: Ddisplay = Daddrs[1];	goto redis;
		case _KPD: Ddisplay = Daddrs[8];	goto redis;
		case _KUA: backdis(1);				goto redis;
		case _KPU: backdis(8);
		redis:		refresh_disassembler();	continue;
		case _CPD: Mdisplay += 64;			goto remem;
		case _CPU: if(Mdisplay > 64) { Mdisplay -= 64; goto remem; }
		case _CHO: Mdisplay = 0;			goto remem;
		case _CEN: Mdisplay = 0x3FFF;
		remem:		refresh_memory();		continue;
		case '?':
		case _K1: help(HMAIN);				continue;
		case _K2: update_octal();			goto restat;
		case _K3:
			W_OPEN = dwin;
			wgotoxy(0, 0);
			getval("Address", &Ddisplay, 255);
			wcursor_off();
			goto restat;
		case _K4 : edit_memory();			goto restat;
		case _K5 : edit_registers();		goto restat;
		case _K6 :
			W_OPEN = swin;
			sprintf(temp, "Breakpoint [%W]", Break);
			if(!getval(temp, &i, 255))
				Break = i;
			goto restat;
		case _K7 : edit_prom();				goto restat;
		case 'I' : updateio();	setio('I', &I0);	goto uio;
		case 'O' : updateio();	setio('O', &O0);
		case _K8:
uio:		status("I/O ports");
			updateio();
			Eflag = 255;
			wcursor_off();
			continue;
		case 'S': S8008(); showdis(PC);		goto restat;
		case 'R' :
			A = B = C = D = E = H = L = SP = PC = HLT = Ddisplay = 0;
			goto redraw;
		case 'G' :
			w_close(mwin);
			w_close(rwin);
			w_close(dwin);
dogo:		status(mgo); updateio();
			settty(255);
			do {
				S8008(); }
			while((tstkey() != _K3) && (PC != Break));
			settty(0);
			showdis(PC);
			goto reopen;
		case 'E' :
			i = SP;
			status(mgo); updateio();
			do {
				S8008();
				if(PC == Break)
					break;
				refresh_registers(); }
			while((tstkey() != _K3) && (SP >= i));
			showdis(PC);
			goto restat;
		case 'N' :
			i = PC;
			status(mgo); updateio();
			do {
				S8008();
				if(PC == Break)
					break;
				refresh_registers(); }
			while((tstkey() != _K3) && (i != PC));
			showdis(PC);
			goto restat;
		case 'L' :
			if(selectfile(".HEX")) {
				load_file(Directory);
				Eflag = 255;
				goto redraw; }
			goto restat;
	} }
}

unsigned
	SRcycle,			// Serial RX cycle count
	STcycle;			// Serial TX cycle count
unsigned char
	PROM[256],			// Eprom
	PROMprg[256],		// Prom programmed
	PROMadr,			// Eprom address
	PROMdata,			// Eprom data
	SRstate,			// Serial RX state
	SRchar,				// Serial RX char
	STstate,			// Serial TX state
	STchar;				// Serial TX character

/*
 * Erase content of EPROM
 */
erase_prom()
{
	memset(PROM, Ptype ? 0 : 255, sizeof(PROM));
	memset(PROMprg, 5, sizeof(PROMprg));
}

/*
 * Edit content of EPROM
 */
edit_prom()
{
	unsigned a, b, c, y;
	unsigned char xf;
	static unsigned char p, m;
	wopen(0, 0, 80, 16, WSAVE|WCOPEN|0x67);
	wcursor_line();
redraw:
	for(y=xf=0; y < 16; ++y) {
upline:
		wgotoxy(0, y);
		a = y << 4;
		for(c=0; c < 16; ++c)
			wprintf("%B ", PROM[a+c]);
		do {
			c = PROM[a++];
			if((c < ' ') || (c > 0x7E))
				c = '.';
			wputc(c); }
		while(a & 0x0F);
		if(xf)
			break; }
	for(;;) {
		y = p >> 4;
		a = p & 15;
		status("EPROM %W  F1=Help F2=%s F3=Erase F4=%s F10=Quit", p,
			Octal ? "Hex" : "Oct", m ? "Num" : "Chr");
		wgotoxy(m ? a+64 : a*4, y);
		switch(c=wgetc()) {
		case _KLA: --p;								continue;
		case _KRA: ++p;								continue;
		case _KHO: if(p & 15) { p &= 0xF0;			continue; }
		case _KUA: p -= 16;							continue;
		case _KEN: if((p & 15) != 15) { p |= 0x0F;	continue; }
		case _KDA: p += 16;							continue;
		case _KPU: p = 0;							continue;
		case _KPD: p = 255;							continue;
		case _K1 : help(HEDITP);					continue;
		case _K2 : Octal = Octal ? 0 : 255;			goto redraw;
		case _K3 : erase_prom();					goto redraw;
		case _K4 : m = m ? 0 : 255;					goto redraw;
		case 0x1B: if(m) break;
		case _K10:	wclose();	return; }

		if(c & 0xFF00) continue;
		if(m)
			wputc(c);
		else {
			if(Octal) {
				if(cvdig(&c, 4)) goto upline;
				if(cvdig(&a, 0x88)) goto upline;
				if(cvdig(&b, 0x88)) goto upline;
				c = (c << 6) | (a << 3) | b; }
			else {
				if(cvdig(&c, 16)) goto upline;
				if(cvdig(&a, 0x90)) goto upline;
			c = (c << 4) | a; } }
		PROM[p++] = c;
		goto upline; }
}

void hltirq(void)
{
	if(Kbuf) {
		SRchar = toupper(Kbuf);
		if(SRchar == '\n')
			SRchar = '\r';
		Kbuf = HLT = 0; }
}

/*
 * 8008 input character function
 */
inport(unsigned port)
{
	unsigned i;
	switch(port) {
	case 000 :			// Serial data
		i = SRchar;
		SRchar >>= 1;
		return ~(i & 1);
	case 001 :			// Eprom data
		return PROM[PROMadr];
//case 004 : i = Kbuf; Kbuf = 0; return i;
	case 005 : return I0;
	case 006 : return I1;
	case 007 : return I2;
	} return  0;
}

/*
 * 8008 output port handler
 */
void outport(unsigned port, unsigned value)
{
	unsigned i;
	switch(port) {
	case 010 :		// Eprom address
		PROMadr = value;
		return;
	case 011 :		// Eprom data
		PROMdata = ~value;
		return;
	case 012 :		// Serial output
		if(!STstate) {						// Not active
dos0:		if(!(value &  0x01)) {			// Start bit
				STstate = 1;
				STcycle = CYCLE;
				STchar = 0; }
			return; }
		i = CYCLE - STcycle;
		STcycle = CYCLE;
		if(i > 2500) {						// Too long
			STstate = 0;
			goto dos0; }
		if(++STstate < 9) {					// Not RXed all bits
			if(value & 1)
				STchar |= 1 << (STstate-2);
			return; }
		// Stop bit - output data
		STstate = 0;
		Tputc(STchar & 0x7F);
		return;
	case 013 :
		if(value & 4) {
			if(PROMprg[PROMadr])
				if(!--PROMprg[PROMadr])
					if(Ptype)
						PROM[PROMadr] |= PROMdata;
					else
						PROM[PROMadr] &= PROMdata; }
		return;
//case 014 : Tputc(value & 0x7F);	return;
	case 022 : O0 = value; updateio();	return;
	case 023 : O1 = value; updateio();	return;
	case 024 : O2 = value; updateio();
	}
}
