#ifndef INLINE
#include <stdio.h>

#define	Debug(x)		//printf x;

unsigned
	Line,				// Current line number
	Csum[2],			// Checksum Accumulator
	Crc32table[256][2],	// CRC table
	M_addr,				// Memory access address
	R_eg[3][2],			// Register values
	Seg,				// External segment
	Stop,				// Top of external segment
	Spos,				// Position in segment
	Position,			// Current buffer position
	Ibase = 10,			// Inline number base
	Abase = 10,			// Address number base
	Xoutp;				// Current output position

unsigned char
	*ptr,				// General pointer
	Uline,				// Unget line flag
	M_size,				// Size of memory access
	M_size1,			// Size of first memory access
	M_oper,				// Memory operation
	M_oper1,			// First memory operation
	Dsize,				// Data block size
	Xaddr,				// Display addresses in output
	Dbuffer[128],		// Data block buffer
	Pbuffer[4096];		// Packet input buffer

FILE
	*fp;				// General file pointer

#endif

unsigned				// Long constants
	L1[] = { 1, 0 },
	L128[] = { 128, 0 },
	L256[] = { 256, 0 },
	L32896[] = { 32896, 0 },
	L65792[] = { 0x0100, 1 };

#define	F_SETX		0xE0
#define	F_SETY		0xE1
#define	F_SETZ		0xE2
#define	F_CRCX		0xE3
#define	F_CRCY		0xE4
#define	F_CRCZ		0xE5
#define	F_POSA		0xE6
#define	F_POSP		0xE7
#define	F_POSN		0xE8
#define	F_DATA		0xE9
#define	F_DUMP		0xEA
#define	F_COND		0xEB
#define	F_JUMP		0xEC
#define	F_SWITCH	0xED
#define	F_CASE		0xEE
#define	F_END		0xEF

#define	S_IF		1
#define	S_ELSE		2
#define	S_SWITCH	3

extern unsigned Longreg[];

#ifndef INLINE
#define	Lputc(c)	putc(c,stdout)

unsigned
	Ftype,				// Type filter
	Ptop,				// Size of current packet
	Pcurrent,			// Number of packet in buffer
	Pstart,				// Starting packet
	Pend = 65535,		// Ending packet
	Lines = 65535;		// Lines

unsigned char
	Etype,				// Error type
	PKTseg,				// 0== Get data from packet segment
	Pfile[65],			// Packet filename
	Tfile[65],			// Template filename
	Bfile[65];			// Binary template

unsigned char Help[] = { "\n\
Use:	SNIFFD	capture_file [options]\n\n\
opts:	/A		- include Address when decoding templates\n\
	/F		- Full dump (with header) when no template\n\
	A=2|8|10|16	- numeric-base for decoded Address values	[10]\n\
	B=file[.PTB]	- load template from Binary file		[None]\n\
	F=hextype	- Filter: only show packets of this type	[All]\n\
	L=[n]		- limit Lines in dump displays			[All]\n\
	I=2|8|10|16	- numeric-base for decoded Inline values	[10]\n\
	P=n[,n]		- select Packet[s] to dump			[All]\n\
	T=file[.PTT]	- load template from Text file			[None]\n\
\nIf both T= & B= are supplied, the first is converted into the second.\n\
\n2010-2012 Dave Dunfield ** See COPY.TXT **.\n" };

/*
 * Skip to non-blank
 */
int skip()
{
	while(isspace(*ptr))
		++ptr;
	if(*ptr == ';')
		return 0;
	return *ptr;
}

/*
 * Report and error and terminate
 */
register error(unsigned args)
{
	unsigned char buf[128], *p;
	_format_(nargs() * 2 + &args, buf);
	switch(Etype) {
	case 1 :	// Definition parser
		if(Line)
			printf("%u: ", Line);
		fputs(buf, stdout);
		putc('\n', stdout);
		if(Line) {
			fputs(p=Dbuffer, stdout);
			putc('\n', stdout);
			while(p < ptr) {
				++p;
				putc(' ', stdout); }
			fputs("^\n", stdout); }
		break;
	case 2 :	// Definition decoder
		printf("%04x: ", Spos);
	case 0 :
		fputs(buf, stdout);
		putc('\n', stdout); }
	exit(-1);
}

/*
 * Read data from packet detecting out of bounds
 */
unsigned char PKTdata(unsigned o)
{
	if(!PKTseg)
		return peek(Seg, o);

	if((o += Position) >= Ptop)
		error("Read outside packet %04x\n", o);

	return Pbuffer[o];
}

/*
 * Display a memory dump
 */
void dump(unsigned d, unsigned l, unsigned a)
{
	unsigned i, j, k;
	i = 0;
	while(i < l) {
		printf("%04x", a+i);
		for(j=0; j < 16; ++j) {
			if(!(j&3))
				Lputc(' ');
			if((k = i+j) < l)
				printf(" %02x", PKTdata(d+k));
			else
				printf("   "); }
		printf("  ");
		for(j=0; j < 16; ++j) {
			k = PKTdata(d+i);
			if((k < ' ') || (k > 0x7E))
				k = '.';
			Lputc(k);
			if(++i >= l)
				break; }
		Lputc('\n'); }
}

// Dump within line limit
void xdump(unsigned addr, unsigned nbytes, unsigned lines)
{
	if(Lines)
		lines = Lines;
	if(nbytes && lines) {
		if(((nbytes+15) / 16) > lines)
			nbytes = lines * 16;
		dump(addr, nbytes, 0); }
}

// Read next packet from stream
int read_packet()
{
	if(!fget(&Ptop, 2, fp))
		return 0;
	if(!fget(Pbuffer, Ptop, fp))
		return 0;
	++Pcurrent;
	return 255;
}

void decode_section(void);

/*
 * Adjust filename to include extension if not supplied
 */
void file(unsigned char *dest, unsigned char *source, unsigned char *ext)
{
	unsigned char f;
	f = 0;
	while(*dest = *source) {
		++dest;
		switch(*source++) {
		case ':' :
		case '\\': f = 0;	continue;
		case '.' : f = 255; } }
	if(!f)
		strcpy(dest, ext);
}

unsigned getbase()
{
	unsigned b;
	switch(b = atoi(ptr)) {
	default: error("Bad number-base, must be: 2, 8, 10 or 16");
	case 2 :
	case 8 :
	case 10:
	case 16: }
	return b;
}

main(int argc, char *argv[])
{
	unsigned i, e;
	unsigned char f;
	static unsigned char dumps, mode;

	e = 0;
	for(i=1; i < argc; ++i) {
		ptr = argv[i];
		switch((toupper(*ptr++)<<8) | toupper(*ptr++)) {
		case '-A' :
		case '/A' : Xaddr = 255;		continue;	// Include address
		case '-F' :
		case '/F' : mode = 255;			continue;	// Dump all
		case '-S' :
		case '/S' : dumps = 255;		continue;	// Incl source
		case '?'<<8:
		case '-?' :
		case '/?' : goto help;
		case 'I=' : Ibase = getbase();					continue;
		case 'A=' : Abase = getbase();					continue;
		case 'T=' : file(Tfile, ptr, ".PTT"); e += e|3;	continue;	// Text tpl
		case 'B=' : file(Bfile, ptr, ".PTB"); e += e|4;	continue;	// Bin tpl
		case 'L=' : Lines = atoi(ptr);					continue;	// Limit lines
		case 'F=' : Ftype = atox(ptr);					continue;	// Filter
		case 'P=' :
			Pstart = Pend = atoi(ptr);
			while(*ptr) {
				if(*ptr++ == ',') {
					Pend = atoi(ptr);
					break; } }
			continue;
		}
		if(!*Pfile) {
			file(Pfile, ptr-2, "");
			continue; }
		help: abort(Help); }

	if(!(*Pfile | dumps)) switch(e) {
	default: goto help;
	case 10 :
	case 11 : ; }

	if(e) {
		Seg = alloc_seg(4096);
		i = 0; do {
			pokew(Seg, i, 0); }
		while(i -= 2);
		switch(e) {
		default: error("Invalid T=/B= arguments");
		case 10:		// T= B=
		case 3 :		// T=
			fp = fopen(Tfile, "rvq");
			Etype = 1;
			while(define_section());
			fclose(fp);
			break;
		case 11:		// B= T=
		case 4 :		// B=
			fp = fopen(Bfile, "rvqb");
			while((i = getc(fp)) != EOF)
				poke(Seg, Stop++, i);
			fclose(fp); }
		printf("Template size=%u\n", Stop);
		switch(e) {
		case 10 : // Write B
			fp = fopen(Bfile, "wvqb");
			for(i=0; i < Stop; ++i)
				putc(peek(Seg, i), fp);
			fclose(fp);
			break;
		case 11 : // Write T
			fp = fopen(Tfile, "wvq");
			Etype = 2;
			while(Spos < Stop)
				decode_section();
			fclose(fp);
			Spos = 0; }
		if(dumps)		// Display template
			xdump(0, Stop, 65535); }

	Etype = 0;		// Normal error output
	PKTseg = 255;	// Get data from packet

	if(*Pfile) {
		fp = fopen(Pfile, "rvqb");
		while(read_packet()) {
			if((Pcurrent >= Pstart) && (Pcurrent <= Pend)) {
				i = (Pbuffer[12]<<8)|Pbuffer[13];
				if(Ftype && (Ftype != i))
					continue;
				printf("%u To:%02x%02x%02x%02x%02x%02x", Pcurrent,
					Pbuffer[0], Pbuffer[1], Pbuffer[2], Pbuffer[3], Pbuffer[4], Pbuffer[5]);
				printf(" Fr:%02x%02x%02x%02x%02x%02x",
					Pbuffer[6], Pbuffer[7], Pbuffer[8], Pbuffer[9], Pbuffer[10], Pbuffer[11]);
				printf(" Ty:%04x Size: %u/%x\n", i, Ptop, Ptop);
				Spos = f = Position = 0;
				if(Seg) {
					while(Spos < Stop) {
						longset(R_eg[2], Ptop);
						if(test_section(f))
							f = 255; } }
				if(!f) {
					if(mode)
						xdump(0, Ptop, 65535);
					else
						xdump(14, Ptop-14, 65535); }
				putc('\n', stdout); } }
		fclose(fp); }

//	for(i=0; i < 3; ++i)
//		printf("R[%u] = %04x%04x\n", i, R_eg[i][1], *R_eg[i]);
}
#endif

/*
0000 8u			- 8-Bit unsigned		0000 = Nothing
0001 8s			- 8-bit signed			0001 = +
0010 16lu		- 16-bit LE unsigned	0010 = -
0011 16ls		- 15-bit LE signed		0011 = *
0100 16bu		- 16-bit BE unsigned	0100 = /
0101 16bs		- 16-bit BE signed		0101 = %
0110 32l		- 32-bit LE				0110 = &
0111 32b		- 32-bit BE				0111 = |
1000 Imm8+		- Imm 8-bit unsigned	1000 = ^
1001 imm8-		- Imm 8-bit signed		1001 = <
1010 imm16+		- Imm 16-bit unsigned	1010 = >
1011 imm16-		- Imm 16-bit signed		1011 = <<	[
1100 imm32		- Imm 32-bit			1100 = >>	]
1101 reg0		- Content of R0			1101 = ==	=
1110 reg1		- Content of R1			1110 = !=	!
1111 reg3		- Content of R2
*/

/*
 * Get data value from memory
 */
void get_M_data(unsigned char d[])
{
	unsigned i, j, k, a, l[2];
	unsigned char c, f;
	static unsigned char mb[] =
		{ 1, 1, 2, 2, 2, 2, 4, 4, 1, 1, 2, 2, 4, 4, 4, 4 };	// # bytes

	M_size = mb[i = M_oper >> 4];
	if(!(i & 8)) {		// Memory access
		a = M_addr;
		j = 1;
		if((0x0D<<i) & 0x80) {		// Big endian? (00001101)
			a += (M_size-1);
			j = -1; }
		for(k=f=0; k < M_size; ++k) {	// Get data value
			d[k] = c = PKTdata(a);
			a += j; }
		if(i & 1)					// Signed
			f = (c & 0x80) ? 0xFF : 0;
		for(k=M_size; k < 4; ++k)		// Fill top
			d[k] = f;
		Debug((" M%u %02x%02x%02x%02x\n", M_size, d[3], d[2], d[1], d[0]))
		return; }

	if(i < 0x0D) {			// Inline
		for(k=f=0; k < M_size; ++k)
			d[k] = c = peek(Seg, Spos++);
		while(k < 4)
			d[k++] = 0;
		if(M_size == 2)		// 16-bit
			longadd(d, L256);
		if(i & 1) {			// Negative number
			longcpy(l, d);
			memset(d, 0xFF, 4);		// Base is -1
			longsub(d, l); }
		Debug((" I%u %02x%02x%02x%02x\n", M_size, d[3], d[2], d[1], d[0]))
		return; }

	switch(i) {
	case 0x0D :	longcpy(d, R_eg[0]);	break;		// Reg-X
	case 0x0E :	longcpy(d, R_eg[1]);	break;		// Reg-Y
	case 0x0F :	longcpy(d, R_eg[2]);		}		// Reg-Z
	Debug((" R%u %02x%02x%02x%02x\n", M_size, d[3], d[2], d[1], d[0]))
}

/*
 * Decode and get data value from memory
 */
void get_M1(unsigned l[2], unsigned n)
{
	unsigned r[2];
	unsigned char x, f;
	M_oper = x = f = 0;
	M_size1 = 1;
again:
	M_oper = peek(Seg, Spos++);
	if((M_oper & 15) == 15) {
		Debug(("Open (\n"))
		get_M1(r, (M_oper >> 4) + 1);
		Debug(("Close ) - %02x %04x%04x\n", M_oper, l[1], *l))
		goto doop; }

	if(!(M_oper & 0x80)) {			// Address is specified
		M_addr = peek(Seg, Spos++);
		if(M_addr & 0x80) {			// Extended
			M_addr &= 0x7F;
			M_addr = ((M_addr << 8) | peek(Seg, Spos++)) + 128; } }
	else
		M_addr = Spos;
	Debug(("MEM: %02x %04x", M_oper, M_addr))
	get_M_data(r);					// Get value
	if(!f) {
		if(!(M_oper & 0x80)) {
			M_oper1 = M_oper;
			M_size1 = M_size;
			f = 255; } }
doop:
	switch(x) {
	case 0 : longcpy(l, r);									break;	// None
	case 1 : longadd(l, r);									break;	// +
	case 2 : longsub(l, r);									break;	// -
	case 3 : longmul(l, r);									break;	// *
	case 4 : longdiv(l, r);									break;	// /
	case 5 : longdiv(l, r);	longcmp(l, Longreg);			break;	// %
	case 6 : *l &= *r; l[1] &= r[1];						break;	// &
	case 7 : *l |= *r; l[1] |= r[1];						break;	// |
	case 8 : *l ^= *r; l[1] ^= r[1];						break;	// ^
	case 9 : if(longcmp(l, r) < 0) goto R1;	goto R0;				// <
	case 10: if(longcmp(l, r) > 0) goto R1;	goto R0;				// >
	case 11: x = *r; while(x--) longshl(l);					break;	// <<
	case 12: x = *r; while(x--) longshr(l);					break;	// >>
	case 13: if(longcmp(l, r)) goto R0; R1:longset(l,1);	break;	// ==
	case 14: if(longcmp(l, r)) goto R1; R0:longset(l,0);	}		// !=

	if(!--n)
		return;
	if(x = M_oper & 15) {		// Arithmetic operation
		Debug(("Operator: %u\n", x))
		longcpy(r, l);
		goto again; }
}

get_M(unsigned l[])
{
	get_M1(l, 0);
}

/*
 * Process and display an encoded value
 */
void output_value(unsigned char t)
{
	unsigned b, i, l[2], l1[2];
	unsigned char *p, f, r, s, w, c, m, st[33];

	f=r=s=0;
	if(t & 0x20)
		f = peek(Seg, Spos++);
	if(t & 0x10) {
		r = peek(Seg, Spos++);
		if(r & 0x80)
			s = peek(Seg, Spos++);
		r &= 0x7F; }
	get_M(l);
	Debug(("{V:%02x %02x %02x}", t, f, r))
repeat:
	*(p=&st[32])=m=0;
	switch(t & 15) {
	case 0 :	b = 16;	don:			// Hex.
		longset(l1, b);
		do {
			longdiv(l, l1);
			c = *Longreg + '0';
			if(c > '9')
				c += 7;
			*--p = c; }
		while(l[0]|l[1]);
		break;
	case 1 :	b = 2;	goto don;		// Bin.
	case 2 :	b = 8;	goto don;		// Oct.
	case 4 :							// S-dec.
		if(l[1] & 0x8000) {
			longcpy(l1, l);
			longset(l, 0);
			longsub(l, l1);
			m = '-'; }
	case 3 :	b = 10;	goto don;		// U-dec.
	case 5 :	*--p = *l;				// Character
	case 6 :	p = *l;					// String
	}
	if(m)								// Negative
		Lputc(m);
	if(t & 0x20) {						// Formatted
		i = strlen(p);
		w = (f & 0x3F)+1;
		if(m)
			--w;
		if(!(f & 0x80)) {				// Right justify
			c = (f & 0x40) ? '0' : ' ';
			while(i < w) {
				Lputc(c);
				++i; } } }
	while(c = *p++)
		Lputc(c);
	if(f & 0x80) {
		while(i < w) {
			Lputc(' ');
			++i; } }
	if(r) {
		--r;
		if(s)
			Lputc(s);
		M_addr += M_size;
		get_M_data(l);
		goto repeat; }
}

/*
 * Build the pre-computed 32 bit CRC table
 */
void build_crc32(unsigned p[2])
{
	int i, j;
	unsigned crc[2];

	for(i=0; i < 256; ++i) {
		longset(crc, i);
		for(j=0; j < 8; ++j) {
			if(longshr(crc)) {
				*crc ^= *p;
				crc[1] ^= p[1]; } }
		longcpy(Crc32table[i], crc); }
}

/*
 * Calculate checksum
 *	1 = 8-bit checksum
 *	2 = 8-bit XOR
 *	3 = 16-bit 1's compliment
 *	4 = 16-bit 2's compliment
 *	5 = CRC-32
 */
void calchk(unsigned type, unsigned char *block, unsigned length)
{
	unsigned char crc[4], temp[4];

	switch(type) {
	case 1 :					// 8-bit sum
		while(length--) {
			longset(temp, *block++);
			longadd(Csum, temp); }
		return;
	case 2 :					// 8-bit XOR
		while(length--)
			*Csum ^= *block++;
		return;
	case 3 :					// 16-bit BE 1's compliment (internet)
	case 4 : asm {				// 16-bit BE 2's compliment
		MOV		BX,6[BP]		; Get pointer
		MOV		CX,4[BP]		; Get length
ca1:	AND		CX,CX			; Test length
		JZ		ca3				; Nothing to do
		XOR		AL,AL			; Assume no low
		MOV		AH,[BX]			; Get data
		INC		BX				; Next
		DEC		CX				; Reduce count
		JZ		ca2				; We have data
		MOV		AL,[BX]			; Get data
		INC		BX				; Next
		DEC		CX				; And remove from length
ca2:	ADD		DGRP:_Csum,AX	; Add to total
		ADC		WORD PTR DGRP:_Csum+2,0	; Propogate carry
		JMP SHORT ca1			; And keep going
ca3:
		} return;
	case 5 :					// 16-bit LE 1's compliment
	case 6 : asm {				// 16-bit BE 2's compliment
		MOV		BX,6[BP]		; Get pointer
		MOV		CX,4[BP]		; Get length
cb1:	AND		CX,CX			; Test length
		JZ		cb3				; Nothing to do
		XOR		AH,AH			; Assume no HIGH
		MOV		AL,[BX]			; Get data
		INC		BX				; Next
		DEC		CX				; Reduce count
		JZ		cb2				; We have data
		MOV		AH,[BX]			; Get data
		INC		BX				; Next
		DEC		CX				; And remove from length
cb2:	ADD		DGRP:_Csum,AX	; Add to total
		ADC		WORD PTR DGRP:_Csum+2,0	; Propogate carry
		JMP SHORT cb1			; And keep going
cb3:
		} return;
	case 7 :					// CRC-32
		longcpy(crc, Csum);
		while(length--) {
			longcpy(temp, crc);
			longcpy(crc, Crc32table[*crc ^ *block++]);
			crc[0] ^= temp[1];
			crc[1] ^= temp[2];
			crc[2] ^= temp[3]; }
		longcpy(Csum, crc);
		return;
	}
}

/*
 * End checksum calculation - assign to dest in network format,
 * return in native format
 */
unsigned calend(unsigned type, unsigned dest[])
{
	switch(type) {
	case 3 :					// BE Internet 1's
	case 5 : asm {				// LE Internet 1's
ce1:	MOV		AX,DGRP:_Csum+2	; Get low
		AND		AX,AX			; Any data?
		JZ		ce2				; No
		MOV		WORD PTR DGRP:_Csum+2,0	; Reset
		ADD		DGRP:_Csum,AX	; Add
		ADC		WORD PTR DGRP:_Csum+2,0
		JMP SHORT ce1
ce2:	MOV		AX,DGRP:_Csum	; Get low
		NOT		AX				; 1s comp
		AND		AX,AX
		JNZ		ce3				; Not 0000
		NOT		AX				; 0000 = FFFF
ce3:	MOV		BX,4[BP]		; Get dests
		MOV		[BX],AX			; Save dest
		MOV	WORD PTR 2[BX],0	; Zero high
ce4:
	} return; }

	// For anything else, copy checksum to destination
	longcpy(dest, Csum);
}

/*
 * Perform a checksum test
 */
void test_checksum(unsigned l[])
{
	unsigned ca[2], op[2];
	unsigned char *p, t, n;

	t = peek(Seg, Spos++);		// Type
	get_M(Csum);				// Initial value
	p = Pbuffer + Position;

	switch(t) {					// Initialization
	default: error("Unknown CHECK type %u @%04x", t, Spos-2);
	case 7 :
		get_M(op);				// CRC-32
		build_crc32(op);
	case 1 :
	case 2 :
	case 3 :
	case 4 :
	case 5 :
	case 6 : }

	n = peek(Seg, Spos++);		// # pairs
	Debug(("\nT:%u I:%04x%04x N:%u\n", t, Csum[1], *Csum, n))
	if(Dsize) {					// Pseudo-header supplied
		calchk(t, Dbuffer, Dsize);
		Dsize = 0; }
	while(n) {					// Process blocks
		--n;
		get_M(ca);				// Segment address
		get_M(l);				// Segment length
		PKTdata(*ca);
		Debug(("A:%04x L:%u\n", *ca, *l))
		calchk(t, p + *ca, *l); }
	calend(t, l);
	Debug(("L=%04x%04x\n", l[1], *l))
}

/*
 * Test for matching section
 */
int test_section(unsigned char skiplast)
{
	unsigned i, j, ss, sb, se, l[2];
	unsigned char *sp, c, np, lf;
	static unsigned Sp, Sv[32];

	// Get section header information
	if(peek(Seg, sb = Spos) != 0xFF)
		error("Not a section @%04x", Spos);
	np = (c = peek(Seg, Spos+1)) & 0x7F;	// Number of match parameters
	ss = peekw(Seg, Spos+2);				// Section size
	se = sb + ss;
	Debug(("Sect b:%04x p:%u s:%04x e:%04x\n", sb, np, ss, se))
	sp = Position;
	Position = Dsize = Sp = 0;
	Spos += 4;

	// See it match parameters check out
	for(i=lf=0; i < np; ++i) {				// Scan parameters
		get_M(l);
		if(l[1]|*l) {						// True
			lf = 255;
			continue; }
		if(!(c & 0x80))						// False - exit if not |
			goto skip; }
	if(!lf) {								// No match was found
		if(np | skiplast) {					// Not default
skip:		Debug(("No-Match\n"))
			Spos = se;
			Position = sp;
			return 0; } }
	// This section matches - process it
	lf = 0;
	while(Spos < se) {
#ifdef INLINE
		*W_OPEN = WA_MAIN;
#endif
		c = peek(Seg, Spos++);
		if(!(c & 0x80)) {			// Normal character
#ifdef INLINE
			if(c == '\t') {
				do {
					Lputc(' '); }
				while(W_OPEN->WINcurx & 7);
				continue; }
#endif
			Lputc(c);
			continue; }
		if((c & 0xC0) == 0x80) {	// Value output
			output_value(c);
			continue; }
//		if((c & 0xE0) == 0xC0) {	// Dump
//			i = ((c & 0x1F) << 8) | peek(Seg, Spos++);
//			get_M(l);
//			dump(*l, i & 255, 0);
//			continue; }
		switch(c) {
		default: error("Unknown action %02x @%04x", c, Spos-1);
		case F_SETX :	get_M(R_eg[0]);			continue;		// set X
		case F_SETY :	get_M(R_eg[1]);			continue;		// set Y
		case F_SETZ :	get_M(R_eg[2]);			continue;		// set Z
		case F_CRCX : test_checksum(R_eg[0]);	continue;		// check X
		case F_CRCY : test_checksum(R_eg[1]);	continue;		// check Y
		case F_CRCZ : test_checksum(R_eg[2]);	continue;		// check Z
		case F_POSA : get_M(l); Position = *l;	continue;		// Offset
		case F_POSP : get_M(l); Position += *l;	continue;		// Offset +
		case F_POSN : get_M(l); Position -= *l;	continue;		// Offset -
		case F_DATA :					// Data block definition
			Dsize = 0;
			i = peek(Seg, Spos++);
			while(i--) {
				get_M(l);
				if(M_oper1 & 0x80) {	// Inline
					M_size1 = 1;
					if(*l & 0xFF00)		M_size1 = 2;
					if(l[1] & 0x00FF)	M_size1 = 3;
					if(l[1] & 0xFF00)	M_size1 = 4; }
				switch(M_size1) {
				case 4 : Dbuffer[Dsize++] = l[1] >> 8;
				case 3 : Dbuffer[Dsize++] = l[1];
				case 2 : Dbuffer[Dsize++] = *l >> 8;
				case 1 : Dbuffer[Dsize++] = *l; } }
//			dump(Dbuffer, Dsize, 0);
			continue;
		case F_DUMP :				// Dump
			get_M(l); i = *l;		// address
			get_M(l); j = *l;		// length
			get_M(l);				// Max # lines
			xdump(i, j, *l);
			continue;
		case F_COND :				// If
			get_M(l);
			i = peek(Seg, Spos++) + 1;
			Debug(("COND: (%u) %u += %u\n", l[1]|*l, Spos, i))
			if(!(l[1]|*l))
				Spos += i;
			continue;
		case F_JUMP :				// JMP
			i = peek(Seg, Spos++) + 1;
			Debug(("JMP: %u += %u\n", Spos, i))
			Spos += i;
			continue;
		case F_SWITCH :				// SWITCH
			get_M(l);
			Sv[Sp++] = Spos;						// Final address
			Spos += 2;
			Sv[Sp++] = *l;							// Key value low
			Sv[Sp++] = l[1];						// Key value high
			Sv[Sp++] = 0;							// Flag
			i = peek(Seg, Spos++) + 1;				// Offset
			Spos += i;
			continue;
		case F_CASE :				// CASE
			if(Sv[Sp-1])
				goto endswit;
			get_M(l);
			i = peek(Seg, Spos++) + 1;
			if((Sv[Sp-2] != l[1]) || (Sv[Sp-3] != *l))
				Spos += i;
			else
				Sv[Sp-1] = 1;
			continue;
		case F_END :				// End of switch
			if(Sv[Sp-1]) {
endswit:		Spos = peekw(Seg, Sv[Sp-4]);
				Sp -= 4;
				continue; }
			Sv[Sp-1] = 1;
			Spos = Sv[Sp-4] + 3;
			continue;
		case 0xFF :					// Nested section
			--Spos;
			if(test_section(lf)) {
				lf = 255; }
			continue; }
	}
	Position = sp;
	return 255;
}

/*
 * Get arithmetic operator from the input line
 */
unsigned getop(void)
{
	unsigned c;
	c = skip();
	switch((c << 8) | ptr[1]) {
		default: switch(c) {
			default: return 0;
			case '+' :
			case '-' :
			case '*' :
			case '/' :
			case '%' :
			case '&' :
			case '|' :
			case '^' :
			case '<' :
			case '>' :
			case '=' :
				++ptr;
				return c; }
		case '<<' : c = '[';	break;
		case '>>' : c = ']';	break;
		case '!=' : c = '!';	}
	ptr += 2;
	return c;
}

/*
 * Get a number element from the input line
 */
void getexpr(unsigned l[2], unsigned char t);
void getnum(unsigned l[2])
{
	unsigned c, b, l1[2];
	unsigned char f;

	skip();

	longset(l, f=0);
	b = 10;

	switch(c = *ptr++) {
	case '(' :
		getexpr(l, ')');
		return;
	case '-' :
		getnum(l1);
		longsub(l, l1);
		return;
	case '~' :
		getnum(l1);
		l[0] = ~l1[0];
		l[1] = ~l1[1];
		return;
	case '!' :
		getnum(l1);
		if(!longtst(l1))
			l[0] = 1;
		return;
	case '$' : b = 16;	break;
	case '@' : b = 8;	break;
	case '%' : b = 2;	break;
	default: --ptr; }

	for(;;) {
		c = *ptr;
		if((c >= '0') && (c <= '9'))
			c -= '0';
		else if((c >= 'a') && (c <= 'f'))
			c -= ('a'-10);
		else if((c >= 'A') && (c <= 'F'))
			c -= ('A'-10);
		else
			c = 255;
		if(c >= b)
			break;
		longset(l1, b);
		longmul(l, l1);
		longset(l1, c);
		longadd(l, l1);
		f = 255;
		++ptr; }

	if(!f)
		error("Number expected");
}

/*
 * Get an expression value from the input line
 */
void getexpr(unsigned l[2], unsigned char t)
{
	unsigned l1[2];
	unsigned char c;

	getnum(l);

	for(;;) {
		c = skip();
		if(c == t) {
			++ptr;
			return; }

		if(!(c = getop()))
			error("Unknown operator");
		getnum(l1);

		switch(c) {
		case '+' : longadd(l, l1);						continue;
		case '-' : longsub(l, l1);						continue;
		case '*' : longmul(l, l1);						continue;
		case '/' : longdiv(l, l1);						continue;
		case '%' : longdiv(l, l1); longcpy(l, Longreg);	continue;
		case '&' : *l &= *l1; l[1] &= l1[1];			continue;
		case '|' : *l |= *l1; l[1] |= l1[1];			continue;
		case '^' : *l ^= *l1; l[1] ^= l1[1];			continue;
		case '=' : if(longcmp(l, l1))	goto ret0;					// ==
		ret1: longset(l, 1);							continue;
		case '!' : if(longcmp(l, l1)) 	goto ret1;					// !=
		ret0: longset(l, 0);							continue;
		case '[' : if(longcmp(l, l1) < 0) goto ret1;	goto ret0;	// <
		case ']' : if(longcmp(l, l1) > 0) goto ret1;	goto ret0;	// >
		case '<' :
			c = *l1;
			while(c) {
				longshl(l);
				--c; }
			continue;
		case '>' :
			c = *l1;
			while(c) {
				longshr(l);
				--c; }
			continue; }
	}
}

/*
0000 8u			- 8-Bit unsigned		0000 = Nothing
0001 8s			- 8-bit signed			0001 = +
0010 16lu		- 16-bit LE unsigned	0010 = -
0011 16ls		- 15-bit LE signed		0011 = *
0100 16bu		- 16-bit BE unsigned	0100 = /
0101 16bs		- 16-bit BE signed		0101 = %
0110 32l		- 32-bit LE				0110 = &
0111 32b		- 32-bit BE				0111 = |
1000 Imm8u		- Imm 8-bit unsigned	1000 = ^
1001 imm8s		- Imm 8-bit signed		1001 = <
1010 imm16u		- Imm 16-bit unsigned	1010 = <
1011 imm16s		- Imm 16-bit signed		1011 = <<	[
1100 imm32		- Imm 32-bit			1100 = >>	]
1101 reg0		- Content of R0			1101 = ==	=
1110 reg1		- Content of R1			1110 = !=	!
1111 reg3		- Content of R2			1111 = $
*/
unsigned getmem(void)
{
	unsigned i, l[2], l1[2];
	unsigned char v, n, ob[4], o;
	static unsigned Mcount;
	static unsigned char ops[] = { "+-*/%&|^<>[]=!" };
again:
	n = o = 0;
	skip();
	switch(*ptr++) {
		default: err: error("?Unknown MEM");
	case '(' :
		l[1] = Stop++;
		*l1 = Mcount;
		*l = getmem();
		i = (Mcount - *l1) - 1;
		Mcount = *l1 + 1;
		if(*ptr++ != ')')
			error("Expected ')'");
		poke(Seg, l[1], (i << 4) | 15);
		if(n = getop()) {
			for(i=0; ops[i]; ++i) {
				if(ops[i] == n)
					break; }
			poke(Seg, *l, peek(Seg, *l) | (i+1));
			goto again; }
		return *l;
	case '1' : v = 0x00;	goto doend;		// 8-bit
	case '2' : v = 0x20;	goto doend;		// 16-bit
	case '4' : v = 0x60;	doend:			// 32-bit
		switch(*ptr++) {
		default: goto err;
		case '>' :		// Big endian
			switch(v) {
			case 0x20 : v += 0x20;	break;	// 16-bit
			case 0x60 : v += 0x10; }		// 32-bit
		case '<' :		// Little endian
			getnum(l);
			if(l[1] & 0x8000) {
				longcpy(l1, l);
				longset(l, 0);
				longsub(l, l1);
				if(v < 0x60)
					v |= 0x10; }
			if(longcmp(l, L128) < 0) {
				ob[o++] = *l;
				break; }
			if(longcmp(l, L32896) < 0) {
				longsub(l, L128);
				ob[o++] = (*l >> 8) | 0x80;
				ob[o++] = *l & 255;
				break; }
			error("Offset too large"); }
		break;
	case '#' :
		getnum(l);
		if(l[1] & 0x8000) {
			longcpy(l1, l);
			memset(l, 0xFF, 4);
			longsub(l, l1);
			n = 0x10; }
		if(longcmp(l, L256) < 0) {		// Fits in 8-bits
			v = 0x80 + n;
			ob[o++] = *l;
			break; }
		if(longcmp(l, L65792) < 0) {	// Fits in 16-bits
			v = 0xA0 + n;
			longsub(l, L256);
			ob[o++] = *l;
			ob[o++] = *l >> 8;
			break; }
		if(n)							// Reget original (not -)
			longcpy(l, l1);				// Since 32-bit is signed
		v = 0xC0;
		ob[o++] = *l;
		ob[o++] = *l >> 8;
		ob[o++] = l[1];
		ob[o++] = l[1] >> 8;
		break;
	case 'x' :
	case 'X' : v = 0xD0;	break;
	case 'y' :
	case 'Y' : v = 0xE0;	break;
	case 'z' :
	case 'Z' : v = 0xF0; }

	++Mcount;

	if(n = getop()) {
		for(i=0; ops[i]; ++i) {
			if(ops[i] == n)
				break; }
		v |= (i+1); }

	poke(Seg, i = Stop++, v);
	for(n=0; n < o; ++n)
		poke(Seg, Stop++, ob[n]);

	if(v & 0x0F)			// Arithmetic operation
		goto again;

	return i;
}

/*
 * Get line from input, ignoring blank lines
 */
int getline(char eof)
{
	if(Uline) {
		Uline = 0;
		ptr = Dbuffer;
		skip();
		return 255; }
again:
	if(fgets(ptr = Dbuffer, sizeof(Dbuffer)-1, fp)) {
		++Line;
		if(Xaddr) {
			printf("%-6u%04x: ", Line, Stop);
			fputs(Dbuffer, stdout);
			putc('\n', stdout); }
		switch(skip()) {
		case ';' :
		case 0 :	goto again; }
		Debug(("[%s]\n", Dbuffer))
		return 255; }
	*(ptr = Dbuffer) = 0;
	if(eof)
		error("Unexpected End-Of-File");
	return 0;
}

// Get a simple decimal number from the input line
unsigned getdecimal()
{
	unsigned v;
	v = 0;
	while(isdigit(*ptr))
		v = (v * 10) + (*ptr++ - '0');
	return v;
}

// Patch jump length at beginning of block
void patchlen(unsigned i)
{
	unsigned l;
	l = Stop - i;
	Debug(("#endblk %u %u\n", Stop, l))
	if(l < 2) error("Block empty");
	if(l > 257) error("Block too large");
	poke(Seg, i, l-2);
}

/*
 * Define a parsing section
 */
int define_section(void)
{
	int c;
	unsigned i, l, sp, Sp, Sa[8], Sv[8];
	unsigned char t, f, r, lf, ae, St[8];

	// Read and parse section header
	if(!getline(0))
		return 0;
	if(!strbeg(ptr, "#section"))
		error("#section header expected");
	ptr += 8;

	// Section header:
	// 0xFF
	// # matchparms
	// Total section size
	sp = Stop;
	poke(Seg, sp, 0xFF);		// Block start flag
	Stop += 4;					// Reserve header

	i = lf = Sp = ae = 0;

	switch(skip()) {			// Handle &/| prefix
	case '|' : i = 0x80;
	case '&' : ++ptr; }

	while(skip()) {				// Process match terms
		getmem();
		++i; }

	poke(Seg, sp+1, i);			// Save # match terms
	Debug(("{%u\n", i))
	while(getline(255)) {
		if(lf)
			poke(Seg, Stop++, '\n');
		i = lf = 0;
		if(ae)
			goto autoend;
pline:	switch(*ptr++) {
		case '#' :						// Special command
			if(strbeg(ptr, "end")) {
				if(Sp) {
autoend:			switch(St[--Sp]) {
					default: error("stack error");
					case S_IF:
					case S_ELSE:
						patchlen(Sa[Sp]);
						break;
					case S_SWITCH:
						patchlen(Sa[Sp]);
						poke(Seg, Stop++, F_END);
						pokew(Seg, Sv[Sp], Stop); }
					if(ae) {
						ae = 0;
						goto pline; }
					continue; }
				pokew(Seg, sp+2, Stop-sp);	// Total length of section
				return 255; }
			if(strbeg(ptr, "section")) {
				Uline = 255;
				define_section();
				continue; }
			if(strbeg(ptr, "set")) {
				ptr += 3;
				switch(toupper(skip())) {
				default: error("Unknown register");
				case 'X' : t = F_SETX;	break;
				case 'Y' : t = F_SETY;	break;
				case 'Z' : t = F_SETZ;	}
				++ptr;
				poke(Seg, Stop++, t);
				getmem();
				continue; }
			if(strbeg(ptr, "check")) {
				ptr += 5;
				switch(toupper(skip())) {
				default: error("Unknown register");
				case 'X' : t = F_CRCX;	break;
				case 'Y' : t = F_CRCY;	break;
				case 'Z' : t = F_CRCZ;	}
				++ptr; skip();
				poke(Seg, Stop++, t);
				poke(Seg, Stop++, getdecimal());	// Type
				getmem();							// Initial value
				switch(t) {
				case 7 :	getmem();				// CRC-32 polynomial
				} l = Stop++;						// # pairs
				t = 0;
				while(skip()) {
					++t;
					getmem();
					getmem(); }
				poke(Seg, l, t);
				continue; }
			if(strbeg(ptr, "offset")) {
				ptr += 5;
				t = F_POSA;
				switch(skip()) {
				case '-' : ++t;
				case '+' : ++t; ++ptr; }
				poke(Seg, Stop++, t);
				getmem();
				continue; }
			if(strbeg(ptr, "data")) {
				ptr += 4;
				poke(Seg, Stop++, F_DATA);
				l = Stop++;
				t = 0;
				do {
					++t;
					getmem(); }
				while(skip());
				poke(Seg, l, t);
				continue; }
			if(strbeg(ptr, "dump")) {
				ptr += 4;
				poke(Seg, Stop++, F_DUMP);
				getmem();		// Address
				getmem();		// Length
				getmem();		// # lines
				continue; }
			if(strbeg(ptr, "if")) {
				ptr += 2;
				poke(Seg, Stop++, F_COND);
				getmem();		// Conditional term
				St[Sp] = S_IF;		// If
				Debug(("#if %u\n", Stop))
				Sa[Sp++] = Stop++;
doae:			if(skip()) {
					ae = 255;
					goto xline; }
				continue; }
			if(strbeg(ptr, "else")) {
				ptr += 4;
				if((!Sp) || (St[Sp-1] != S_IF))
					error("#if mismatch");
				poke(Seg, Stop++, F_JUMP);
				poke(Seg, Stop++, 0);
				patchlen(Sa[Sp-1]);
				St[Sp-1] = S_ELSE;
				Sa[Sp-1] = Stop-1;
				goto doae; }
			if(strbeg(ptr, "switch")) {
				ptr += 6;
				poke(Seg, Stop++, F_SWITCH);
				getmem();			// Match case
				Sv[Sp] = Stop;		// Final address
				Stop += 2;
				Sa[Sp] = Stop++;	// Fixup length
				St[Sp++] = S_SWITCH;
				goto tstline; }
			if(strbeg(ptr, "case")) {
				ptr += 4;
				if((!Sp) && (St[Sp] != S_SWITCH))
					error("#case mismatch");
				patchlen(Sa[Sp-1]);
				poke(Seg, Stop++, F_CASE);
				getmem();			// Match case
				Sa[Sp-1] = Stop++;
tstline:		if(skip()) {
					goto xline; }
				continue; }
		}
		// Data to display
// { 2>12==#$800
// IP V/I=~02x1>0 TOS=~02x1>1 LEN=~u2>2 ID=~04x2>4 FRAG=~04x2>6 TTL=~02x1>9
// CHK=~04x2>10 Src-IP=~u*4.1>12 DstIP=~u*4.1>10
// }
		ptr = Dbuffer;
xline:	lf = 255;
		while(c = *ptr++) switch(c) {
			default:
				poke(Seg, Stop++, c);
				continue;
			case ';' : ptr = "";	continue;
			case '~' :	// Formatted output
// 0ccccccc				= Normal character
// 10frtttt [fmt] [rpt]	= Display value
//  fmt: lznnnnnn
//	rpt: srrrrrrr [ss]
// 110aaaaa				= Reserved
// 111xxxxx				= Function
				f = r = 0;
				if(*ptr == '-') {	// Left justify
					f |= 0x80;
					++ptr; }
				if(*ptr == '0')		// Zero fill
					f |= 0x40;
				l = getdecimal();
				switch(toupper(*ptr++)) {
				default: error("Unknown format");
				case 0 :				// End of line
					lf = 0;
					--ptr;
					continue;
				case '.' :				// Force line-feed
					poke(Seg, Stop++, 0x0A);
					continue;
				case ';' :				// Inject comment-char
					poke(Seg, Stop++, ';');
				case '=' :				// Nothing (stop scanner)
					continue;
				case 'X' :	t = 0;	break;		// Hexidecimal
				case 'B' :	t = 1;	break;		// Binary
				case 'O' :	t = 2;	break;		// Octal
				case 'U' :	t = 3;	break;		// Unsigned
				case 'D' :	t = 4;	break;		// Decimal
				case 'C' :	t = 5;	break;		// Character
				case 'S' :	t = 6;	}			// String
				if(l) {							// Format width
					if(l > 64)
						error("FormatWidth >64");
					t |= 0x20;
					f |= l-1; }
				if(*ptr == '*') {
					++ptr;
					r = getdecimal();
					if((c = *ptr++) != '~')
						r |= 0x80;
					t |= 0x10; }
				poke(Seg, Stop++, t | 0x80);
				if(t & 0x20)
					poke(Seg, Stop++, f);
				if(t & 0x10) {
					poke(Seg, Stop++, r);
					if(r & 0x80)
						poke(Seg, Stop++, c); }
				getmem();
				while((ptr > Dbuffer) && isspace(*(ptr-1)))
					--ptr;
			}
	}
}

/*
 * --- Decoder ---
 */
void Xputc(unsigned char c)
{
	if(c == '\n') {
		Xoutp = 0;
		if(Xaddr) {
			fprintf(fp, "\n%-6u%04x: ", ++Line, Spos);
			return; } }
	putc(c, fp);
	++Xoutp;
}
void Xputs(unsigned char *s)
{
	while(*s)
		Xputc(*s++);
}

// Display numeric base indicator
void Xbase(unsigned base)
{
	switch(base) {
	case 2 : Xputc('%');	return;
	case 8 : Xputc('@');	return;
	case 16: Xputc('$');	}
}

// Display 16-bit value
void Xvalue(unsigned v, unsigned base)
{
	unsigned l[2];
	unsigned char buf[12];
	longset(l, v);
	Xbase(base);
	ltoa(l, buf, base);
	Xputs(buf);
}

// Decode memory access operand
void decode_M(void)
{
	unsigned i, ms, lw[2], Bsp;
	unsigned char t, lb[4], buf[12], Bs[16];
	static unsigned char *mpt[] = {
		"1<",	"1<-", "2<",	"2<-",	"2>",	"2>-", "4<",	"4>",
		"#", "#", "#", "#", "#", "X", "Y", "Z" };
	static unsigned char mst[] = {
		1, 1, 2, 2, 2, 2, 4, 4, 1, 1, 2, 2, 4, 4, 4, 4 };
	static unsigned char *opt[] = {
		"+","-","*","/","%","&","|","^","<",">","<<",">>","==","!=", "?" };

	Bsp = 0;
again:
	t = peek(Seg, Spos++);
	if((t & 15) == 15) {			// Bracket
		Xputc('(');
		Bs[Bsp++] = (t >> 4) + 1;
		goto again; }
	ms = mst[i = t >> 4];
	Xputs(mpt[i]);

	if(t < 0x80) {					// Memory address
		i = peek(Seg, Spos++);		// Get base address
		if(i & 0x80) {
			i &= 0x7F;
			i = ((i << 8) | peek(Seg, Spos++)) + 128; }
		Xvalue(i, Abase); }
	else if(t < 0xD0) {				// Inline value
		longset(lb, 0);
		for(i=0; i < ms; ++i)
			lb[i] = peek(Seg, Spos++);
		if(ms == 2)
			longadd(lb, L256);
		if(t & 0x10) {		// Force negative
			longcpy(lw, lb);
			longset(lb, 0);
			longsub(lb, lw); }
		if(lb[3] & 0x80) {
			Xputc('-');
			longcpy(lw, lb);
			longset(lb, 0);
			longsub(lb, lw); }
		Xbase(Ibase);
		ltoa(lb, buf, Ibase);
		Xputs(buf); }

xs:	if(Bsp) {
		if(!--Bs[Bsp-1]) {
			Xputc(')');
			--Bsp;
			goto xs; } }

	if(t &= 15) {
		Xputs(opt[t-1]);
		goto again; }
}

// Decode memory operand with leading space
void decode_Msp(void)
{
	Xputc(' ');
	decode_M();
}

// Decode section block
void decode_section(void)
{
	unsigned i, ss, se, Sp, Sa[16];
	unsigned char t, c, f, r, s, ls, St[16];
	static unsigned char nft[] = { 'x', 'b', 'o', 'u', 'd', 'c', 's' };

	if(peek(Seg, Spos) != 0xFF)
		error("Not a section");
	Xputs("\n#section");
	if((i = peek(Seg, Spos+1)) & 0x80) {
		i &= 0x7F;
		Xputs(" &"); }
	se = Spos + (ss = peekw(Seg, Spos+2));
	Spos += 4;
	while(i) {
		--i;
		decode_Msp(); }
	Xputc('\n');
	Sp = 0;
	ls = 1;
	while(Spos < se) {
		for(i=t=0; i < Sp; ++i) {
			if(Sa[i] == Spos) {
				if(!ls) {
					Xputs("~\n");
					ls = 1; }
				Xputs("#end\n");
				++t; } }
		Sp -= t;
		c = peek(Seg, Spos++);
		switch(c & 0xE0) {
		case 0x00 :
		case 0x20 :
		case 0x40 :
		case 0x60 :
		if(!(c & 0x80)) {
			switch(c) {
			case '\n':
				if(ls) {
					Xputs("~.");
					ls = 0;
					continue; }
				ls = 1;
				break;
			case ' ' :
				if(Xoutp > 60)
					Xputs("~\n");
				break;
			case '~' :
			case ';' : Xputc('~');
			default: ls=0; }
			Xputc(c);
			continue; }
		case 0x80:				// Print output
		case 0xA0:				// ""
			f = s = r = ls = 0;
			Xputc('~');
			if(c & 0x20) {
				f = peek(Seg, Spos++);
				if(f & 0x80)
					Xputc('-');
				if(f & 0x40)
					Xputc('0');
				Xvalue((f & 0x3F)+1, 10); }
			if((t = c & 15) > 6)
				error("Unimplemented number format: %02x", c);
			Xputc(nft[t]);
			if(c & 0x10) {
				Xputc('*');
				r = peek(Seg, Spos++);
				Xvalue(r & 0x7F, 10);
				s = '~';
				if(r & 0x80)
					s = peek(Seg, Spos++);
				Xputc(s); }
			decode_M();
			continue; }
		if(!ls) {
			Xputs("~\n");
			ls = 1; }
		switch(c) {
		case 0xE0 :	c = 'X';	goto doset;			// #set X
		case 0xE1 : c = 'Y';	goto doset;			// #set Y
		case 0xE2 : c = 'Z';	doset:				// #set Z
			Xputs("#set ");
			Xputc(c);
xnlDM:		decode_Msp();
xnl:		Xputc('\n');
			continue;
		case 0xE3 :	c = 'X';	goto docheck;		// #check X
		case 0xE4 : c = 'Y';	goto docheck;		// #check Y
		case 0xE5 : c = 'Z';	docheck:			// #check Z
			t = peek(Seg, Spos++);
			if((t < 1) || (t > 7))
				error("Unknown #check type: %u", t);
			Xputs("#check ");
			Xputc(c);
			Xputc(' ');
			Xvalue(t, 10);
			decode_Msp();			// Initial
			if(t == 7) {
				decode_Msp(); }	// CRC polynomial
			i = peek(Seg, Spos++);
			while(i) {
				--i;
				decode_Msp();
				decode_Msp(); }
			goto xnl;
		case 0xE6 : c = 0; dooffset:				// #offset
			Xputs("#offset ");
			if(c)
				Xputc(c);
			goto xnlDM;
		case 0xE7 : c = '+'; goto dooffset;			// #offset +
		case 0xE8 : c = '-'; goto dooffset;			// #offset -
		case 0xE9 :									// #data
			Xputs("#data");
			i = peek(Seg, Spos++);
			while(i) {
				--i;
				decode_Msp(); }
			goto xnl;
		case 0xEA :									// #dump
			Xputs("#dump");
			decode_Msp();
			decode_Msp();
			goto xnlDM;
		case 0xEB :									// #if
			Xputs("#if");
			decode_Msp();
			i = peek(Seg, Spos++) + 1;
			Sa[Sp] = Spos + i;
			St[Sp++] = 1;
			goto xnl;
		case 0xEC :									// #else
			Xputs("#else");
			i = peek(Seg, Spos+1);
			Sa[Sp-1] = Spos+i;
			St[Sp-1] = 2;
			goto xnl;
		case 0xED :									// #switch
			Xputs("#switch");
			decode_Msp();
			Spos += 3;
			goto xnl;
		case 0xEE :									// #case
			Xputs("#case");
			decode_Msp();
			++Spos;
			goto xnl;
		case 0xEF :									// #end
			Xputs("#end");
			goto xnl;
		case 0xFF :									// sub-section
			--Spos;
			decode_section();
			continue;
	} error("Unimplemented control code: %02x", c); }
	Xputs("#end\n");
}
