#include <stdio.h>
#include <window.h>
#include <file.h>

/*
 * -------------------- PACKET DRIVER INTERFACE --------------------
 */

/* PKT_INT must be in DECIMAL, as it is used by C and inline assembly */
#define	PKT_INT		96				/* Packet driver interrupt (0x60) */
#define	PKT_DATA	1000			/* Size of data area in packet */
#define	PKT_SIZE	PKT_DATA+14		/* Size of packet with overhead */
#define	PKT_EASIZE	6				/* Size of ethernet address */
/* PKT_WINDOW must be a power of 2, so we can mask with PKT_WINDOW-1 */
#define	PKT_WINDOW	32				/* Size of packet receive queue */

/*
 * Structure of information returned by pkt_driver_info()
 */
struct PD_driver_info {
	unsigned version;				/* Driver version */
	unsigned char class;			/* Interface class */
	unsigned type;					/* Interface type */
	unsigned char number;			/* Interface number */
	unsigned char functionality;	/* Driver functionality */
	unsigned char name[9]; };		/* Driver name (?) */

/*
 * Structure of packet as sent/received by driver
 */
struct PD_packet {
	unsigned char to_addr[PKT_EASIZE];
	unsigned char from_addr[PKT_EASIZE];
	unsigned ptype;
	unsigned char data[PKT_DATA]; };

/*
 * Structure of a packet receive queue entry
 */
struct PD_queue {
	unsigned handle;
	unsigned length;
	struct PD_packet packet; };

/*
 * Packet driver interface global variables
 */
struct PD_queue
	pkt_queue[PKT_WINDOW];			/* Queue for received packets */

unsigned
	pkt_head = 0,					/* Head of RX packet queue */
	pkt_tail = 0;					/* Tail of RX packet queue */

unsigned reverse(value) asm
{
	MOV		AH,4[BP]
	MOV		AL,5[BP]
}

/*
 * Obtain packet driver information
 *
 * This function *MUST* be called before pkt_access_type() is called, in
 * order to insure that the SAVEDS location is initialized with the Micro-C
 * data segment. Another reason is that you need to obtain the interface
 * class/type/number so that you can pass them to pkt_access_type();
 *
 * Arguments:
 *	info_struct	= "PD_driver_info" structure to fill in.
 *
 * Returns:
 *	0			= Success
 *	!0			= Packet driver error code
 */
int pkt_driver_info(info_struct) asm
{
		MOV		CS:SAVEDS,DS		; Save out data seg
		MOV		AX,01ffh			; Request info
pi1:	INT		PKT_INT				; Ask packet driver
		JC		di2					; Error occured
		PUSH	DS					; Save driver DS
		MOV		DS,CS:SAVEDS		; Restore DATA segment
		POP		ES					; Point to driver data
		MOV		DI,4[BP]			; Get struct
		MOV		0[DI],BX			; Save version
		MOV		2[DI],CH			; Save Class
		MOV		3[DI],DX			; Save type
		MOV		5[DI],CL			; Save number
		MOV		6[DI],AL			; Save functionality
		LEA		BX,7[DI]			; Point to name
di1:	MOV		AL,ES:[SI]			; Get data
		MOV		[BX],AL				; Copy it
		INC		SI					; Advance source
		INC		DI					; Advance dest
		AND		AL,AL				; End of string?
		JNZ		di1					; Copy it all
		XOR		AX,AX				; Return 0
		JMP		di3
; Report driver error
di2:	MOV		AL,DH				; Get error code
		MOV		AH,0FFh				; Insure non-zero
di3:
}

/*
 * Obtain access to a packet type
 *
 * Arguments:
 *	if_class	= Interface class  \
 *	if_type		= Interface type    > - From pkt_driver_info()
 *	if_number	= Interface number /
 *	type		= Pointer to packet type to access
 *	typelen		= Length of packet type (0=Access ALL packets)
 *	handle		= Pointer to 'int' variable to receive handle
 *
 * Returns:
 *	0			= Success
 *	!0			= Packet driver error code
 */
int pkt_access_type(if_class, if_type, if_number, type, typelen, handle) asm
{
		MOV		AL,14[BP]			; Get if_class
		MOV		BX,12[BP]			; Get if_type
		MOV		DL,10[BP]			; Get if_number
		MOV		SI,8[BP]			; Get type pointer
		MOV		CX,6[BP]			; Get typelen
		MOV		DI,OFFSET pktrdy	; Point to handler
		PUSH	CS					; Save code segment
		POP		ES					; ES = handler segment
		MOV		AH,02h				; Indicate "access type" request
pi2:	INT		PKT_INT				; Call packet driver
		JC		di2					; Report error
		MOV		BX,4[BP]			; Get handle
		MOV		[BX],AX				; Save handle
		XOR		AX,AX				; Indicate success
}

/*
 * Release access to a packet type
 *
 * Arguments:
 *	handle		= Handle returned by pkt_access_type()
 *
 * Returns:
 *	0			= Success
 *	!0			= Packet driver error code
 */
int pkt_release_type(handle) asm
{
		MOV		BX,4[BP]			; Get handle
		MOV		AH,03h				; Indicate "release type" request
pi3:	INT		PKT_INT				; Ask for it
		JC		di2					; Indicate failed
		XOR		AX,AX				; Indicate success
}

/*
 * Send a packet
 *
 * Arguments:
 *	buffer		= Pointer to buffer to send
 *	length		= Length of buffer to send
 *
 * Returns:
 *	0			= Success
 *	!0			= Packet driver error code
 */
int pkt_send(buffer, length) asm
{
		MOV		SI,6[BP]			; Get buffer address
		MOV		CX,4[BP]			; Get length
		MOV		AH,04h				; Indicate "send packet" request
pi4:	INT		PKT_INT				; Ask for it
		JC		di2					; Report error
		XOR		AX,AX				; Indicate success
}

/*
 * Get our ethernet address
 *
 * Arguments:
 *	handle		= Handle returned by pkt_access_type()
 *	buffer		= Pointer to buffer to receive address
 *	length		= Size of buffer (maximum)
 *	alen		= Pointer to 'int' variable to receive size of address
 *
 * Returns:
 *	0			= Success
 *	!0			= Packet driver error code
 */
int pkt_get_address(handle, buffer, length, alen) asm
{
		MOV		BX,10[BP]			; Get handle
		MOV		DI,8[BP]			; Get buffer
		MOV		CX,6[BP]			; Get length
		PUSH	DS					; Save out segment
		POP		ES					; Set ES
		MOV		AH,06h				; Indicate "get address" request
pi5:	INT		PKT_INT				; Ask driver
		JC		di2					; Report error
		MOV		BX,4[BP]			; Get received length
		MOV		[BX],CX				; Save length
		XOR		AX,AX				; Indicate success
}

/*
 * Set the receive mode for a packet
 *
 * Arguments:
 *	handle		= Handle returned by pkt_access_type()
 *	mode		=	1: Turn off receiver
 *					2: Receive only packets sent to our address
 *					3: mode 2 + broadcast packets
 *					4: mode 3 + limited multicast packets
 *					5: mode 3 + all multicast packets
 *					6: all packets
 *
 * Returns:
 *	0			= Success
 *	!0			= Packet driver error code
 */
int pkt_set_receive_mode(handle, mode) asm
{
		MOV		BX,6[BP]			; Get handle
		MOV		CX,4[BP]			; Get mode
		MOV		AH,20				; Set function
pi6:	INT		PKT_INT				; Ask driver
		JC		di2					; Report error
		XOR		AX,AX				; Report success
}

/*
 * Receive packet handler. Called by packet driver when a packet
 * has been received for us.
 *
 * Check to see if we have enough space, and buffer the packet in our
 * receive packet queue, notify the mainline when received.
 */
asm {
; Entry:
;  AX = flag: 0=Allocate buffer, 1=Transfer complete
;  BX = handle
;  CX = Length
; Exit:
;  ES:DI = Buffer to receive packet (0:0=Not available)
pktrdy:	PUSH	DS					; Save our data segment
		MOV		DS,CS:SAVEDS		; Restore Micro-C DS
		AND		AX,AX				; Is this allocate?
		JNZ		pktr2				; No, try next
; Allocate space for a packet
		CMP		CX,PKT_SIZE			; Is packet too large?
		JA		pktr1				; Yes, Fail reception
		MOV		AX,DGRP:_pkt_head	; Get head of queue
		INC		AX					; Advance to next
		AND		AX,PKT_WINDOW-1		; Mask for wrap
		CMP		AX,DGRP:_pkt_tail	; Is queue full?
		JZ		pktr1				; Yes, Fail reception
		MOV		AX,DGRP:_pkt_head	; Get head back
		MOV		DI,PKT_SIZE+4		; Get size
		MUL		DI					; Compute offset
		ADD		AX,OFFSET DGRP:_pkt_queue ; Offset to location
		MOV		DI,AX				; Point to location
		MOV		[DI],BX				; Save handle
		MOV		2[DI],CX			; Save length
		ADD		DI,4				; Skip to data
		PUSH	DS					; Get DS
		POP		ES					; Copy to ES
		JMP SHORT pktrx				; And exit
; No space for buffer, or packet too large - reject packet
pktr1:	MOV		DI,0				; Indicate no buffer
		MOV		ES,DI				; ""
		JMP SHORT pktrx				; And exit
; Packet has been copied into our buffer, notify application
pktr2:	MOV		AX,DGRP:_pkt_head ; Get packet queue head
		INC		AX					; Skip to next
		AND		AX,PKT_WINDOW-1		; Wrap circular buffer
		MOV		DGRP:_pkt_head,AX ; Save new queue head
pktrx:	POP		DS					; Restore data seg
		DB		0CBh				; Far return
SAVEDS	DW		?
}

/*
 * Set the packet interrupt by patching the code
 */
set_interrupt(inum) asm
{
		MOV		AL,4[BP]			; Get the interrupt number
		MOV		byte ptr pi1+1,AL	; Patch
		MOV		byte ptr pi2+1,AL	; Patch
		MOV		byte ptr pi3+1,AL	; Patch
		MOV		byte ptr pi4+1,AL	; Patch
		MOV		byte ptr pi5+1,AL	; Patch
		MOV		byte ptr pi6+1,AL	; Patch
}

/*
 * -------------------- SNIFFER APPLICATION --------------------
 */

#define	ADDRS		10				/* Number of addresses in our list */
#define	TRIGSIZE	50				/* Number of trigger bytes */
#define	TRIGSCREEN	23				/* Width of trigger screen entry */

#define	MAIN		0				/* Index to main window */
#define	TRIG		1				/* Index to trigger window */

unsigned c_attrs[] = {
	WSAVE|WCOPEN|WSCROLL|0x17,		/* Main window attributes */
	WSAVE|WCOPEN|0x67 };			/* Trigger window attributes */
unsigned m_attrs[] = {
	WSAVE|WCOPEN|WSCROLL|NORMAL,	/* Main window attributes */
	WSAVE|WCOPEN|NORMAL };			/* Trigger window attributes */
unsigned *attrs = &c_attrs;

unsigned char
	Pkt_int = PKT_INT,				/* Packet interrupt number */
	trig_mask[TRIGSIZE],			/* Trigger mask bytes */
	trig_data[TRIGSIZE],			/* Trigger data */
	addr_list[ADDRS][PKT_EASIZE],	/* List of received addresses */
	send_buffer[2000];				/* Packet to send */

unsigned
	snif_pass = -1,					/* Packet passes validity check */
	snif_mode = 6,					/* Access for sniffing */
	snif_type,						/* Type of packets to snif */
	snif_stype = 0x0010,			/* Type of packet to send */
	snif_slen = 100,				/* Length of packet to send */
	dump_mode = 1,					/* #lines to dump */
	trig_top,						/* Maximum trigger value */
	addr_top = 0;					/* Top of address list */

struct WINDOW
	*twin,							/* Title window */
	*mwin;							/* Main window */

unsigned char bcast[] = { 0xFF,0xFF,0xFF,0xFF,0xFF,0xFF };

FILE *wfp;

static char hello[] = {
#ifndef DEMO
"Network Monitor 1.1 - 2001-2007 Dave Dunfield ** See COPY.TXT **." };
#else
"Network Monitor 1.1 -DEMO- ?COPY.TXT 2005 Dave Dunfield - **See COPY.TXT**." };
#endif

static char help[] = { "\n\
Use:	SNIFF <options>\n\n\
opts:	B=mac	= set Broadcast mac address		[FFFFFFFFFFFF]\n\
	F=file	= write monitored packets to File	[none]\n\
	I=xx	= specify packet driver Interrupt	[60]\n\
	L=n	= set Length of send packets		[100]\n\
	M=n	= set default Monitor mode (1-6)	[6]\n\
	S=xxxx	= set packet type to Send		[1000]\n\
	T=xxxx	= monitor packets of this Type only	[all]\n\
	/M	= force Monochrome screen\n" };

/*
 * Display an ethernet address address
 */
void show_address(unsigned char *address, unsigned alen)
{
	unsigned i;

	for(i=0; i < alen; ++i) {
		if(i) wputc(':');
		wprintf("%02x", *address++); }
}

/*
 * Test for a new address, and if so, collect into address list
 */
int collect_address(unsigned char *address, unsigned alen)
{
	unsigned i;

	for(i=0; i < addr_top; ++i)
		if(!memcmp(addr_list[i], address, alen))
			return 0;
	wputs("\nNew address: ");
	show_address(address, alen);
	if(i >= ADDRS) {
		wputs(" - List full!");
		return 0; }
	memcpy(addr_list[addr_top], address, alen);
	wprintf(" - Added at position: %u", ++addr_top);
	return -1;
}

/*
 * Dump a buffer of memory with the specified address & size
 */
void dump(unsigned char *buffer, unsigned size, unsigned addr)
{
	unsigned i, j, c, l;

	for(i=l=0; i < size; i += 16) {
		if(++l > dump_mode)
			return;
		wprintf("\n%04x ", addr+i);
		for(j=0; j < 16; ++j) {						/* Display HEX */
			if(!(j & 0x03))
				wputc(' ');
			if((i+j) < size) {
				wprintf("%02x ", buffer[j]); }
			else
				wputs("   "); }
		wputc(' ');
		for(j=0; (j < 16) && ((i+j) < size); ++j) {	/* Display ASCII */
			c = *buffer++;
			wputc(((c >= ' ') && (c < 0x7f)) ? c : '.'); } }
}

/*
 * Set trigger values
 */
set_trigger()
{
	unsigned i, j, bp, bf;
	unsigned char m, b, a, d, *ptr;
	static unsigned p;
	static char *msg[] = {
		"Edit packet trigger:",
		"",
		"To select byte, use:",
		" Up/Down/Left/Right/Home/End",
		"0-9/A-F  = enter HEX data",
		"Space/BS = edit bits",
		"F1       = set DON'T CARE",
		"F2       = Toggle sense",
		"F10/ESC/ENTER = exit",
		0 };

	wopen(0, 0, 80, 25, attrs[TRIG]);
	for(i=0; ptr = msg[i]; ++i) {
		wgotoxy(TRIGSCREEN*2, i+5);
		wputs(ptr); }

	a = *W_OPEN;
	bf = -1;

	ptr = "SI\xDA\xB3TO\xB3\xC0\xDAFROM\xC0TY";
	i = 0;
	while(*ptr) {
		wgotoxy(0, i++);
		wputc(*ptr++); }

top:
	for(i=0; i < TRIGSIZE; ++i) {
		wgotoxy(((i/25)*TRIGSCREEN)+1, i % 25);
		if(i == p)
			*W_OPEN = (a >> 4) | (a << 4);
		wprintf("%3u: ", i+1);
		m = trig_mask[i];
		d = trig_data[i];
		if(i == bf) goto dobit;
		switch(m) {
		case 0x00 :
			wputs("Don't care");
			break;
		case 0xFF :
			wprintf("%08b (%02x) [%c]", d, d, ((d >= ' ') && (d < 0x7F))
				? d : 0xA8);
			break;
		default:
		dobit:
			for(j=0; j < 8; ++j) {
				b = 0x80 >> j;
				if(m & b)
					wputc((d & b) ? '1' : '0');
				else
					wputc('x'); } }
		while(W_OPEN->WINcurx % TRIGSCREEN)
			wputc(' ');		
		*W_OPEN = a; }
	wgotoxy((TRIGSCREEN*2)+3, 22);
	wprintf("Trigger sense: %s", snif_pass ? "TRUE " : "FALSE");
	if(p == bf) {
		wgotoxy(((p/25)*TRIGSCREEN)+bp+6, p%25);
		wcursor_block(); }
	else
		wcursor_off();
	switch(i = toupper(wgetc())) {
	case _KUA :
		p = p ? p-1 : (TRIGSIZE-1);
		break;
	case _KDA :
		if(++p > (TRIGSIZE-1)) p = 0;
		break;
	case _KRA :
	case _KLA :
		p = (p >= 25) ? p - 25 : p + 25;
		break;
	case _KHO :
		p = 0;
		break;
	case _KEN :
		p = TRIGSIZE-1;
		break;
	case 'X' :
		if(p != bf)
			break;
	case '0' :
	case '1' :
		if(p != bf)
			goto do_num;
		b = 0x80 >> bp;
		trig_data[p] &= ~b;
		if(i == 'X')
			trig_mask[p] &= ~b;
		else {
			trig_mask[p] |= b;
			if(i-'0')
				trig_data[p] |= b; }
	case ' ' :
		if(p != bf) {
			bf = p;
			bp = 0; }
		else {
			if(bp < 7)
				++bp; }
		goto top;
	case _KBS :
		if(p != bf) {
			bf = p;
			bp = 8; }
		if(bp)
			--bp;
		goto top;
	case 'A' :
	case 'B' :
	case 'C' :
	case 'D' :
	case 'E' :
	case 'F' :
		i -= 7;
	case '2' :
	case '3' :
	case '4' :
	case '5' :
	case '6' :
	case '7' :
	case '8' :
	case '9' :
	do_num:
		if(p == bf) {
			bf = -1;
			goto top; }
		if(trig_mask[p] != 0xFF) {
			trig_mask[p] = 0xFF;
			trig_data[p] = 0; }
		trig_data[p] = (trig_data[p] << 4) + (i-'0');
		bf = -1;
		break;
	case _K1 :
		trig_mask[p] = trig_data[p] = 0;
		bf = -1;
		break;
	case _K2 :
		snif_pass = snif_pass ? 0 : -1;
		break;
	case _K10 :
	case 0x1B :
	case '\n' :
		trig_top = TRIGSIZE;
		while(trig_top) {
			if(trig_mask[trig_top-1])
				break;
			trig_data[--trig_top] = 0; }
		wclose();
		return; }
	bf = -1;
	bp = 0;
	goto top;
}

/*
 * Validate the packet with trigger
 */
validate_packet(packet) asm
{
		MOV		CX,DGRP:_trig_top	; Get top marker
		AND		CX,CX				; Free running?
		JZ		vp2					; Indicate it is OK
		MOV		BX,4[BP]			; Get packet pointer
		MOV		SI,OFFSET DGRP:_trig_mask; Point to mask
		MOV		DI,OFFSET DGRP:_trig_data; Point to data
vp1:	MOV		AL,[BX]				; Get byte from packet
		AND		AL,[SI]				; And with mask
		CMP		AL,[DI]				; Does it match?
		JNZ		vp3					; No, fail it
		INC		BX					; Next in packet
		INC		SI					; Next in mask
		INC		DI					; Next in data
		LOOP	vp1					; Do all bytes
vp2:	MOV		AX,DGRP:_snif_pass	; Get pass flag
		POP		BP					; Restore BP
		RET
vp3:	MOV		AX,DGRP:_snif_pass	; Get fail flag
		NOT		AX					; Indicate failure
}


/*
 * Main demonstration program
 */
main(int argc, char *argv[])
{
	unsigned ps, po;							/* Driver segment,offset */
	unsigned h1;								/* Driver handle */
	unsigned char my_address[PKT_EASIZE];		/* Our ethernet address */
	unsigned alen;								/* Length of address */
	struct PD_driver_info info;					/* Info from driver */
	static char pkt_driver_id[] = {"PKT DRVR"};	/* Driver ID string */
	static unsigned packet_rcount = 0;			/* Count of packets received */
	static unsigned packet_vcount = 0;			/* Count of packets validated */
	static unsigned packet_scount = 0;			/* Count of packets sent */
	struct PD_packet *p;						/* Pointer to current packet */
	unsigned l;									/* Length of current packet */
	unsigned i, t;								/* Misc variables */
static unsigned lp;

	printf("%s\n", hello);

	IOB_size = 4096;
	for(i=1; i < argc; ++i) {
		p = argv[i];
		switch((toupper(*p++) << 8) | toupper(*p++)) {
		case '-M' :
		case '/M' : attrs = m_attrs;					continue;
		case 'I=' : set_interrupt(Pkt_int = atox(p));	continue;
#ifndef DEMO
		case 'F=' : wfp = fopen(p, "wvqb");				continue;
		case 'M=' : snif_mode = atoi(p);				continue;
		case 'S=' : snif_stype = reverse(atox(p));		continue;
		case 'T=' : snif_type = reverse(atox(p));		continue;
		case 'L=' : snif_slen = atoi(p);				continue;
		case 'B=' :
			for(h1=0; h1 < 12; ++h1) {
				if(isdigit(t = *p++))
					t -= '0';
				else if((t >= 'A') && (t <= 'F'))
					t -= ('A'-10);
				else if((t >= 'a') && (t <= 'f'))
					t -= ('a'-10);
				else
					abort("Bad HEX");
				if(h1 & 1)
					bcast[h1/2] = (l << 4) | t;
				l = t; }
			continue;
#else
		case 'F=' :
		case 'M=' :
		case 'S=' :
		case 'T=' :
			abort("F= M= S= T= options not available in demo.");
#endif
		case '?'<<8:
		case '-?' :
		case '/?' : abort(help); }
		printf("Bad option: '%s' - use ? for help\n", argv[i]);
		exit(-1); }

	/* Check that a packet driver is present */
	po = peekw(0, Pkt_int*4) + 3;	/* Offset to packet ID text */
	ps = peekw(0, (Pkt_int*4)+2);	/* Segment of packet driver */
	t = l = 0;
	do {
		if(pkt_driver_id[t] != peek(ps, po+t)) {
			printf("No packet driver on INT %02x [%04x:%04x]\n", Pkt_int, ps, po-3);
			exit(-1); } }
	while(pkt_driver_id[++t]);

	/* Obtain information from driver and display */
	if(t = pkt_driver_info(info)) {
		printf("Unable to obtain packet driver information, RC=%04x\n", t);
		exit(-1); }
	printf("Pkt driver: version=%u ", info.version);
	printf("Class/Type/Num=%02x/%04x/%02x ", info.class, info.type, info.number);
	printf("Func=%02x ", info.functionality);
	printf("Name='%s'\n", info.name);

	/* Obtain access to our packet type */
	t = snif_type ? sizeof(snif_type) : 0;
	if(t = pkt_access_type(info.class, info.type, 0, &snif_type, t, &h1)) {
		printf("Unable to obtain packet access, RC=%04x\n", t);
		exit(-1); }

	/* Get our ethernet address */
	if(t=pkt_get_address(h1, my_address, sizeof(my_address), &alen)) {
		printf("Unable to read ethernet address!, RC=%04x\n", t);
		goto do_exit; }

	if(t=pkt_set_receive_mode(h1, snif_mode)) {
		printf("Unable to set access mode!, RC=%04x\n", t);
		goto do_exit; }

	twin = wopen(0, 24, 80, 1, WSAVE|WCOPEN|REVERSE);
	if(W_BASE == 0xB000) attrs = m_attrs;

	mwin = wopen(0, 0, 80, 24, attrs[MAIN]);
	wcursor_off();
	wputs(hello);

	/*
	 * Enter into a loop, monitoring the receive packet queue.
	 * If a packet is received, display it and remove from the queue.
	 * Also monitor keyboard, and exit if the ESC key is pressed.
	 * '0' causes transmission of a multicast packet.
	 * '1-9' transmits to collected addresses.
	 */
	memset(send_buffer, 0, sizeof(send_buffer));

remode:
	wputs("\nMonitor: ");
	switch(snif_mode) {
	case 6 : wputs("All packets"); goto dotype;
	case 5 : wputs("All"); goto domulti;
	case 4 : wputs("Limited");
	domulti: wputs(" multicast+");
	case 3 : wputs("Broadcast+");
	case 2 : show_address(my_address, alen);
	dotype:
		wprintf(snif_type ? " of type:%04x" : " of all types", reverse(snif_type));
		break;
	case 1 : wputs("No packets!"); }

retitle:
	w_printf(twin, "\r\
%-5u %-5u Addr:%-3u +/-Dump:%-5u <>mode:%u  0-9send:%-5u  C/E/M/R/T/ESC",
		packet_rcount, packet_vcount, addr_top, dump_mode, snif_mode, packet_scount);
	w_cleow(twin);
	for(;;) {
		/* Check for received packets, and display */
		if(pkt_head != pkt_tail) {		/* Packet has been received */
			w_printf(twin, "\r%-5u", ++packet_rcount);
			if(validate_packet(pkt_queue[pkt_tail]+2)) {
				p = pkt_queue[pkt_tail].packet;
l = pkt_queue[pkt_tail].length;
++packet_vcount;
/*				w_printf(twin, " %-5u", ++packet_vcount);
				wprintf("\n[%2u] %u ", pkt_tail, l = pkt_queue[pkt_tail].length);
				wputs("To:");
				show_address(p->to_addr, alen);
				wputs(" Fr:");
				show_address(p->from_addr, alen);
				wprintf(" Type:%04x", reverse(p->ptype));
				dump(p->data, l-(PKT_SIZE-PKT_DATA), 0); */
				t = collect_address(pkt_queue[pkt_tail].packet.from_addr, alen);
#ifndef DEMO
				if(wfp)
					fput(p-2, l+2, wfp);
#endif
				}
			pkt_tail = (pkt_tail+1) & (PKT_WINDOW-1);
			if(t)
				goto retitle; }

	if(packet_vcount != lp)
		w_printf(twin, " %-5u", lp = packet_vcount);

		/* Check for transmit packets */
		if(t = wtstc()) {
			if(isdigit(t)) {
				if((t-='0') >= addr_top) {		/* From address list */
					wputs("\nAddress not defined!");
					continue; }
				memcpy(send_buffer.to_addr, addr_list[t], alen);
				wputs("\nSending");
			send_packet:
				wprintf(" %u bytes to ", snif_slen);
				show_address(send_buffer.to_addr, alen);
				memcpy(send_buffer.from_addr, my_address, alen);
				send_buffer.ptype = snif_stype;
				for(i=14; i < snif_slen; ++i)
					send_buffer[i] = snif_slen;
				send_buffer[i] = 0x00;
				sprintf(send_buffer.data, "This is packet number %u ...", ++packet_scount);
				pkt_send(send_buffer, snif_slen);
				wprintf(" as packet #%u", packet_scount);
				goto retitle; }
			switch(toupper(t)) {
			case 'B' :	// Broadcast
			case 'M' :
//				memset(send_buffer.to_addr, 0xFF, alen);
				memcpy(send_buffer.to_addr, bcast, alen);
				wputs("\nBroadcast");
				goto send_packet;
			case 0x1B :
			case _K10 :
				wclose();
				wclose();
			do_exit:
#ifndef DEMO
				if(wfp)
					fclose(wfp);
#endif
				if(t = pkt_release_type(h1))
					printf("Failed to release packet driver, RC=%04x\n", t);
				return;
			case '+' : ++dump_mode; goto retitle;
			case '-' : --dump_mode; goto retitle;
			case '>' : ++snif_mode; goto domode;
			case '<' : --snif_mode;
			domode:
				if(snif_mode < 1) snif_mode = 6;
				if(snif_mode > 6) snif_mode = 1;
					if(!(t = pkt_set_receive_mode(h1, snif_mode)))
					goto remode;
				wprintf("\nUnable to set receive mode!, packet driver RC=%04x", t);
				continue;
			case 'C' :
				wclwin();
				continue;
			case 'E' :
				addr_top = 0;
				goto retitle;
			case 'R' :
				packet_rcount = packet_vcount = packet_scount = 0;
				goto retitle;
			case 'D' :
				if(l) {
					wputc('\n');
					t = dump_mode;
					dump_mode = -1;
					dump(p-2, l+2, 0);
					dump_mode = t; }
				continue;
			case 'A' :
				wputs("\nAddress list:");
				for(i=0; i < addr_top; ++i) {
					wprintf("\n%3u: ", i);
					show_address(addr_list[i], alen); }
					continue;
			case _KPU :
				if(snif_slen < sizeof(send_buffer))
					++snif_slen;
				goto showlen;
			case _KPD :
				if(snif_slen > 14)
					--snif_slen;
	showlen:	wprintf("\nTX length: %u", snif_slen);
				continue;
			case 'T' :
				set_trigger(); } } }
}
