#include <stdio.h>
#include <file.h>

// Allowable copy operations
#define	DEL_SRC		0x0001			// Delete from source
#define	DEL_DST		0x0002			// Delete from dest
#define	CPY_S2D		0x0008			// Copy source to dest
#define	CPY_D2S		0x0004			// Copy dest to source
#define	AUTOCOPY	0x0100			// Autocopy allowed
#define	PROMPT		0x0200			// Prompt to be sure
#define	ALERT		0x0400			// Sound alert

// DOS return codes
#define	RC_NOFILE	2				// File not found
#define	RC_NODIR	3				// Directory not found
#define	RC_NOMORE	18				// No more files

unsigned
	Aseg,				// Attribute segment
	Atop,				// Attribute segment top
	Nseg,				// Name segment
	Ntop,				// Name segment top
	Dseg,				// Directory segment
	Dtop,				// Directory segment top
	Etop,				// Top of exclude list
	Ptop,				// Top of string pool
	FFtmp,				// Find First/Next temp
	AutoDirLevel,		// Auto-directory level
	SkipTop;			// Skip at top level

unsigned char
	*Path,				// Display path
	*Send,				// Source end
	*Dend,				// Destination end
	*Elist[100],		// Exclude file list
	Autocopy,			// Auto-copy
	Autoskip,			// Auto-skip
	Autodir,			// Auto-create
	Attrset,			// Set attributes
	Chktime = 255,		// Check timestamps
	Safe = 255,			// Prompt for unsafe
	Fast = 255,			// Skip content check
	Hour,				// 1-hour fix
	Xdir = 255,			// Error on extra directories
	Hidden = 255,		// Copy Hidden/System files
	ReadOnly = 255,		// Auto-override ReadOnly
	Verbose = 255,		// Verbose output option
	Halt = 255,			// Halt on errors
	ScanSFN,			// Scan for possible SFNs in source
	Debug,				// Debug output
	Bypass,				// Bypass directories
	Source[1024],		// Source path
	Dest[1024],			// Dest path
	TextPool[4096],		// Text pool
	Temp[16384];		// Temp storage

char Help[] = { "\n\
Use: SYNC source_path dest_path [options]\n\n\
opts:	/1[!]	- ignore/update! timestamp if exactly 1 hour difference\n\
	/A	- copy file Attributes\n\
	/C	- auto-Copy new files\n\
	/D	- auto-create new Directories\n\
	/E	- continue on Error\n\
	/H	- ignore Hidden&system files/dirs\n\
	/I	- Ignore time/date, compare size/content only\n\
	/L	- disable Long-filename support\n\
	/Q	- Quiet: reduce informational output\n\
	/R	- auto-override Read-only attribute\n\
	/S	- auto-Skip safe copies\n\
	/T	- Test content of files\n\
	/V	- Verbose: display files processed\n\
	/X	- ignore eXtra dirs in dest.\n\
	/Y	- auto-Yes to unsafe operations\n\
	/~	- scan for SFNs in source\n\
	D=n	- do not create new Dirs in top n levels of tree\n\
	E=file	- specify file of files to Exclude\n\
\nOptions can be preset in SYNCOPT environment variable.\n\n\
?COPY.TXT 1998-2012 Dave Dunfield -  -- see COPY.TXT --." };

/*
 * Longfilename lookup structure
 */
struct LFN {
	unsigned char	Attrib;			// File attributes
	unsigned char	Reserved_1[19];	// Attrib[3]+Ctime[8]+Atime[8]
	unsigned		Time;			// Modification Time
	unsigned		Date;			// Modification Date
	unsigned char	Reserved_2[4];	// Mtime[4]
	unsigned		SizeH[2];
	unsigned		SizeL[2];
	unsigned char	Reserved_3[8];
	unsigned char	Lname[260];
	unsigned char	Sname[14];
	} Slfn, Dlfn;

#if 0
/*
 * Rename LFN file
 */
int lfn_rename(old, new) asm
{
		MOV		AX,7156h			// LFN rename
		MOV		DX,6[BP]			// Old name
		MOV		DI,4[BP]			// New name
		PUSH	DS					// Copy
		POP		ES					// ES = DS
		STC							// Assume failure
		INT		21h					// Ask DOS
		JNC		rz					// Success
}

/*
 * Change LFN directory
 *
int lfn_cd(directory) asm
{
		MOV		AX,713Bh			// LFN chdir
		MOV		DX,4[BP]			// Directory
		STC							// Assume fail
		INT		21h					// Ask DOS
		JNC		rz					// Success
}
#endif

/*
 * Create directory
 */
int lfn_mkdir(name) asm
{
		MOV		AX,7139h			// Make directory
		MOV		DX,4[BP]			// Get directory
		STC							// Assume fail
		INT		21h					// Ask DOS
		JNC		rz					// Success
}

/*
 * Remove directory
 */
int lfn_rmdir(name) asm
{
		MOV		AX,713Ah			// Remove directory
		MOV		DX,4[BP]			// Get directory
		STC							// Assume fail
		INT		21h					// Ask DOS
		JNC		rz					// Success
}

/*
 * Get LFN current directory
 */
int lfn_getdir(buffer) asm
{
		MOV		AX,7147h			// LFN getdir
		XOR		DL,DL				// Current
		MOV		SI,4[BP]			// Buffer
		STC							// Assume fail
		INT		21h					// Ask DOS
		JNC		rz					// Success
}

/*
 * Delete LFN file
 */
int lfn_delete(file) asm
{
		MOV		AX,7141h			// LFN delete
		XOR		SI,SI				// No search attributes
		XOR		CX,CX				// No search attrbutes
delf:	MOV		DX,4[BP]			// Filename
		STC							// Assume fail
		INT		21h					// Ask DOS
		JNC		rz					// Success
}
int sfn_delete(name) asm
{
		MOV		AH,41h				// SFN delete
		JMP short delf				// Do it
}

/*
 * Lookup first LFN name
 */
unsigned lfn_find_first(name, attributes, lfn) asm
{
		MOV		AX,714Eh			// LFN find-first
		MOV		DI,4[BP]			// LFN structure
		MOV		CX,6[BP]			// Required/Allowed attributes
		MOV		DX,8[BP]			// Filename
		MOV		SI,1				// DOS format date/time
		PUSH	DS					// Copy
		POP		ES					// ES = DS
		STC							// Assume fail
		INT		21h					// Ask DOS
		JC		ok1					// Failed
		MOV		DGRP:_FFtmp,AX		// Save handle
rz:		XOR		AX,AX				// Indicate Success
ok1:
}

/*
 * Lookup next LFN name
 */
int lfn_find_next(lfn) asm
{
		MOV		AX,714Fh			// LFN find-next
		MOV		DI,4[BP]			// Structure
		MOV		BX,DGRP:_FFtmp		// Handle
		MOV		SI,1				// DOS format date/time
		PUSH	DS					// Copy
		POP		ES					// ES = DS
		INT		21h					// Ask DOS
		JNC		rz					// Success
}

/*
 * Close LFN search
 */
void lfn_find_close() asm
{
		MOV		AX,71A1h			// LFN find close
		MOV		BX,DGRP:_FFtmp		// Handle
		INT		21h					// Ask DOS
}

/*
 * Set LFN file attributes
 */
int lfn_set_attr(file, attr) asm
{
		MOV		AX,7143h			// LFN set attributes
		MOV		BL,01h				// Set attributes
seta:	MOV		DX,6[BP]			// Get filename
		MOV		CX,4[BP]			// Get attributes
		STC							// Assume fail
		INT		21h					// Ask DOS
		JNC		rz					// Success
}
int sfn_set_attr(file, attr) asm
{
		MOV		AX,4301h			// SFN set attributes
		JMP	short seta				// Do it
}
		
/*
 * Set timestamp for file
 */
int touch(handle, time, date) asm
{
		MOV		DX,4[BP]		// Get date
		MOV		CX,6[BP]		// Get time
		MOV		BX,8[BP]		// Get handle
		MOV		AX,5701h		// Set date & time function
		INT		21h				// Ask DOS
		JNC		rz				// Success
		MOV		AX,1			// Report fail
}

/*
 * Open LFN file
 */
unsigned lfn_open(name, mode) asm
{
		MOV		AX,4[BP]			// Access mode
		MOV		DX,1				// Read file
		MOV		BX,0				// Read-only
		CMP		AX,2				// Write?
		JNZ		lo1					// No - go with it
		MOV		DX,012h				// Open/Truncate
		MOV		BX,1				// Write-only
lo1:	MOV		AX,716Ch			// LFN open
		XOR		CX,CX				// Attributes = None
		MOV		SI,6[BP]			// Name
		STC							// Assume fail
		INT		21h					// Ask DOS
		JNC		ok7					// Success!
		XOR		AX,AX				// Indicate fail
ok7:
}

/*
 * Lookup first LFN name
 */
unsigned sfn_find_first(unsigned char *name, unsigned attributes, struct LFN *lfn)
{
	unsigned a, r;
	if(r = find_first(name, attributes, lfn->Lname, &lfn->SizeL[1], &lfn->SizeL[0],
		&a, &lfn->Time, &lfn->Date))
		return r;
	FFtmp = attributes >> 8;
	do {
		if((a & FFtmp) || !FFtmp) {
			lfn->Attrib = a;
			return 0; } }
	while(!(r = find_next(lfn->Lname, &lfn->SizeL[1], &lfn->SizeL[0], &a,
		&lfn->Time, &lfn->Date)));
	return r;
}

/*
 * Lookup next LFN name
 */
int sfn_find_next(handle, struct LFN *lfn)
{
	unsigned a, r;
	while(!(r=find_next(lfn->Lname, &lfn->SizeL[1], &lfn->SizeL[0], &a,
		&lfn->Time, &lfn->Date))) {
		if((a & FFtmp) || !FFtmp) {
			lfn->Attrib = a;
			return 0; } }
	return r;
}

void null() { }
extern open(), close(), mkdir(), rmdir();
unsigned patch_table[] = {
	&lfn_find_first,	&sfn_find_first,
	&lfn_find_next,		&sfn_find_next,
	&lfn_find_close,	&null,
	&lfn_mkdir,			&mkdir,
	&lfn_rmdir,			&rmdir,
	&lfn_delete,		&sfn_delete,
	&lfn_set_attr,		&sfn_set_attr,
	&lfn_open,			&open,
	0 };

/*
 * Patch file functions for non-LFN
 */
void patch_lfn() asm
{
		MOV		BX,OFFSET DGRP:_patch_table
p1:		MOV		DI,[BX]			// Get patch target
		AND		DI,DI			// End of table?
		JZ		p2				// Exit
		LEA		SI,3[DI]		// Point to end
		MOV		AX,2[BX]		// Get destination
		SUB		AX,SI			// Calculate offset for JMP
		MOV		BYTE PTR [DI],0E9h	// JMP instruction
		MOV		1[DI],AX		// Set target address
		ADD		BX,4			// Skip to next
		JMP short p1			// And proceed
p2:
}

/*
 * High speed memory compare
 */
int mcompare(block1, block2, size) asm
{
		PUSH	DS				// Save DS
		POP		ES				// ES=DS
		MOV		SI,8[BP]		// Get first block
		MOV		DI,6[BP]		// Get second block
		MOV		CX,4[BP]		// Get size
		XOR		AX,AX			// Assume success
	REPE CMPSB					// Do the compare
		JZ		cok				// Compare OK
		DEC		AX				// Set -1 (fail)
cok:
}

/*
 * Scan memory for a character
 *
int scan(string, c) asm
{
		MOV		BX,6[BP]		// Get memory address 
		MOV		AH,4[BP]		// Character looking for
sc1:	MOV		AL,[BX]			// Get char
		INC		BX				// Next
		CMP		AL,AH			// Match?
		JZ		sc2				// We found
		AND		AL,AL			// End of string?
		JNZ		sc1				// Keep looking
		XOR		AX,AX			// Not found
sc2:
} */

/*
 * Test to see if a name is a possible "short" filename.
 */
int issfn(name) asm
{
		MOV		SI,4[BP]		// Point to name
		XOR		BX,BX			// Index 0
issfn1:	MOV		AL,[SI+BX]		// Get data
		INC		BX				// Advance
		AND		AL,AL			// End of string?
		JZ		notsf			// Not a SFN
		CMP		AL,' '			// Embedded space?
		JZ		notsf			// Not a SFN
		CMP		BX,7			// At '~' position?
		JB		issfn1			// Too early
		JA		notsf			// Too late - not SFN
		CMP		AL,'~'			// Possible short?
		JNZ		issfn1			// No, keep looking
; '~' in sixth place - possible SFN
		MOV		AL,[SI+BX]		// Get next char
		INC		BX				// Advance
		CMP		AL,'0'			// < '0'
		JB		notsf			// Not a SFN
		CMP		AL,'9'			// < '9'
		JA		notsf			// Not a SFN
; '0-9' in seventh place - still possible
		MOV		AL,[SI+BX]		// Get next
		INC		BX				// Advance
		AND		AL,AL			// End of name?
		JZ		issf			// Is an SFN!
		CMP		AL,'.'			// Dot allowed
		JNZ		notsf			// Not an SFN
; '.' in eighth place - still possible
issfn2:	MOV		AL,[SI+BX]		// Get next
		INC		BX				// Advance
		AND		AL,AL			// END of string?
		JZ		issf			// Is an LFN!
		CMP		AL,' '			// Embedded space?
		JZ		notsf			// Not sf
		CMP		BX,13			// Too far?
		JB		issfn2			// Keep looking
notsf:	XOR		AX,AX			// Not an SFN!
		POP		BP				// Clean stack
		RET
issf:	MOV		AX,255			// Indicate YES
}

/*
 * Report a segment overflow condition
 */
void segover(void)
{
	abort("Segment overflow");
}

/*
 * Match a filename against a pattern using unix rules
 * - '?' matches any single character
 * - '*' matches any substring
 * - '.' is treated like any other character
 */
int fmatch(unsigned char *name, unsigned char *pattern)
{
	unsigned char c;

	for(;;) switch(c = *pattern++) {
	case 0 : return *name == 0;
	case '?' :
		if(!*name++)
			return 0;
		break;
	case '*' :
		if(!*pattern)
			return 1;
		while(*name) {
			if(fmatch(name, pattern))
				return 1;
			++name; }
		return 0;
	default:
		if(toupper(*name++) != c)
			return 0; }
}

/*
 * Add string to segment
 */
void Astring(seg, string) asm
{
		MOV		BX,6[BP]		// Segment pointer
		MOV		ES,[BX]			// Get segment
		MOV		DI,2[BX]		// Get dest
		MOV		SI,4[BP]		// Get source
as1:	MOV		AL,[SI]			// Get from source
		INC		SI				// Next
		MOV		ES:[DI],AL		// Write to dest
		INC		DI				// Next
		AND		AL,AL			// End of string?
		JNZ		as1				// Keep going
		CMP		DI,2[BX]		// Overflow?
		JA		as2				// It's OK
		CALL	_segover		// Report overflow
as2:	MOV		2[BX],DI		// Resave
}

/*
 * Get string from segment
 */
unsigned Gstring(seg, source, string) asm
{
		MOV		ES,8[BP]		// Get segment
		MOV		SI,6[BP]		// Get source
		MOV		DI,4[BP]		// Get dest
gs1:	MOV		AL,ES:[SI]		// Get from source
		INC		SI				// Next
		MOV		[DI],AL			// Write to dest
		INC		DI				// Next
		AND		AL,AL			// End?
		JNZ		gs1				// Keep going
		MOV		AX,SI			// Return source
}

/*
 * Compare two strings (case insensitive)
 */
int Xcmp(s1, s2) asm
{
		MOV		SI,6[BP]		// Get source-1
		MOV		DI,4[BP]		// Get source-2
xc1:	MOV		AH,[SI]			// Get from source-1
		INC		SI				// Next
		CMP		AH,61h			// < 'a'
		JB		xc2				// OK
		CMP		AH,7Ah			// > 'z'
		JA		xc2				// OK
		AND		AH,5Fh			// Convert to upper
xc2:	MOV		AL,[DI]			// Get from source-2
		INC		DI				// Next
		CMP		AL,61h			// < 'a'
		JB		xc3				// OK
		CMP		AL,7Ah			// >  'z'
		JA		xc3				// OK
		AND		AL,5Fh			// Convert to upper
xc3:	CMP		AL,AH			// Same?
		JNZ		xc4				// Fail
		AND		AL,AL			// End of string?
		JNZ		xc1				// Keep going
		POP		BP				// Clean stack
		RET
xc4:	MOV		AX,0FFFFh		// Indicate fail
}

/*
 * Add file information to segment
 */
void add_file(struct LFN *lfn)
{
	unsigned i;

	poke(Aseg, i=Atop, lfn->Attrib & 0x7F);
	pokew(Aseg, Atop+1, lfn->Time);
	pokew(Aseg, Atop+3, lfn->Date);
	pokew(Aseg, Atop+5, lfn->SizeL[0]);
	pokew(Aseg, Atop+7, lfn->SizeL[1]);
	pokew(Aseg, Atop+9, lfn->SizeH[0]);
	pokew(Aseg, Atop+11, lfn->SizeH[1]);
	pokew(Aseg, Atop+13, Ntop);
	Atop += 15;
	if(Atop < i)
		segover();
	Astring(&Nseg, lfn->Lname);
}

/*
 * Get file information from segment
 */
void get_file(unsigned i, struct LFN *lfn)
{
	lfn->Attrib = peek(Aseg, i);
	lfn->Time = peekw(Aseg, i+1);
	lfn->Date = peekw(Aseg, i+3);
	lfn->SizeL[0] = peekw(Aseg, i+5);
	lfn->SizeL[1] = peekw(Aseg, i+7);
	lfn->SizeH[0] = peekw(Aseg, i+9);
	lfn->SizeH[1] = peekw(Aseg, i+11);
	Gstring(Nseg, peekw(Aseg, i+13), lfn->Lname);
}

/*
 * Display file information from LFN block
 */
void show_file(struct LFN *lfn)
{
	unsigned h, mi, s, d, mo, y;
	unsigned char st[12];

	// Decode attributes & timestamp
	h	= lfn->Time >> 11;
	mi	= (lfn->Time >> 5) & 63;
	s	= lfn->Time & 0x1F;
	d	= lfn->Date & 0x1F;
	mo	= (lfn->Date >> 5) & 15;
	y	= lfn->Date >> 9;
	if(longtst(lfn->SizeH))
		stpcpy(st, "(over 4G)");
	else
		ltoa(lfn->SizeL, st, 10);
	printf("%2u/%02u/%u%3u:%02u:%02u ", mo, d, y+1980, h, mi, s*2, st);
	for(h=0; h < 6; ++h)
		putc( (((1<<h)&lfn->Attrib) ? "RHSVDA"[h] : '-'), stdout);
	printf("%11s ", st);
	fputs(lfn->Lname, stdout);
	putc('\n', stdout);
}

/*
 * Scan for string in external segment
 */
int scanfor(unsigned seg[2], unsigned B, unsigned char *string)
{
	unsigned S, T, i;

	S = seg[i=0];
	T = seg[1];
	while(B < T) {
		++i;
		B = Gstring(S, B, Temp);
		if(!Xcmp(string, Temp)) {
			return i; } }
	return 0;
}

/*
 * Sound warning alert
 */
void alert(void)
{
	beep(1500, 250);
}

/*
 * Handle a severe error condition
 */
void severe_error(void)
{
	alert();
loop:
	printf("Abort Continue Dos-shell ?");
	for(;;) switch(toupper(kbget())) {
		case 'A' :
			printf("Abort\n");
			exit(-1);
		case 'C' :
			printf("Continue\n");
			return;
		case 'D' :
			if(!getenv("COMSPEC", Temp)) {
				printf("Cannot locate COMSPEC\n");
				goto loop; }
			exec(Temp, "");
			goto loop; }
}
void serious_error()
{
	if(Halt)
		severe_error();
	else
		alert();
}

/*
 * Prompt for Yes/No response
 */
int yesno(char *prompt)
{
loop:
	if(*prompt == '~')
		printf("%s (Y/N/C)?", prompt+1);
	else
		printf("%s (Y/N)?", prompt);
	for(;;) switch(kbget()) {
		case 'y' :
		case 'Y' :
			printf("Yes\n");
			return 255;
		case 'n' :
		case 'N' :
			printf("No\n");
			return 0;
		case 0x1B:
			putc(' ', stdout);
			severe_error();
			goto loop;
		case 'c' :
		case 'C' :
			if(*prompt == '~') {
				printf("Copy\n");
				return 0x55; } }
}

/*
 * Test return code
 */
int test_rc(int rc)
{
	if(rc)
		printf("Operation failed, return code=%d\n", rc);
	return rc;
}

/*
 * Copy a file
 */
void copy_file(unsigned char *src, unsigned char *dst, unsigned time, unsigned date)
{
	unsigned i, hr, hw;

	if(!(hr = lfn_open(src, F_READ))) {
		printf("Cannot READ: ");
		fputs(src, stdout);
		putc('\n', stdout);
		serious_error();
		return; }

	if(!(hw = lfn_open(dst, F_WRITE))) {
		close(hr);
		printf("Cannot WRITE: ");
		fputs(dst, stdout);
		putc('\n', stdout);
		serious_error();
		return; }

	do {
		if(i = read(Temp, sizeof(Temp), hr)) {
			if(test_rc(write(Temp, i, hw)))
				goto quit; } }
	while(i == sizeof(Temp));

	touch(hw, time, date);
quit:
	close(hw);
	close(hr);
}

/*
 * Compare two files
 */
int compare_file()
{
	int r;
	unsigned n, s, hs, hd;
	unsigned char *d;

	stpcpy(Send, Slfn.Lname);
	stpcpy(Dend, Dlfn.Lname);

	if(!(hs = lfn_open(Source, F_READ))) {
		printf("Cannot READ: ");
		fputs(Source, stdout);
		putc('\n', stdout);
		serious_error();
		return; }

	if(!(hd = lfn_open(Dest, F_READ))) {
		close(hs);
		printf("Cannot READ: ");
		fputs(Dest, stdout);
		putc('\n', stdout);
		serious_error();
		return; }

	s = sizeof(Temp)/2;
	d = Temp + s;

	r = 0;
	do {
		n = read(Temp, s, hs);
		if(read(d, s, hd) != n) {
			r = 255;
			break; }
		if(n && mcompare(Temp, d, n)) {
			r = 255;
			break; } }
	while(n == s);

	close(hd);
	close(hs);
	return r;
}

/*
 * Check for read-only file
 */
int check_ro(unsigned char *file, struct LFN *lfn)
{
	if(lfn->Attrib & READONLY) {
		if(ReadOnly && !yesno("Override READ-ONLY attribute?"))
			return 0;
		if(test_rc(lfn_set_attr(file, lfn->Attrib & ~READONLY)))
			return 0; }
	return 255;
}

/*
 * Restore read-only attribute if removed
 */
void fix_attr(unsigned char *file, unsigned da, unsigned sa)
{
	unsigned char f;
	f = 0;
	if(da & READONLY)				// We changed to set RO
		f = 255;
	if(Attrset && (da != sa)) {		// Copy attributes
		da = sa;
		f = 255; }
	if(f)
		test_rc(lfn_set_attr(file, da & 0x7F));
}

/*
 * Take action on file
 */
void file_action(unsigned char *reason, unsigned p)
{
	stpcpy(Send, Slfn.Lname);
	stpcpy(Dend, Dlfn.Lname);

	fputs(reason, stdout);
	printf(":\n");

	if(p & (CPY_S2D|DEL_SRC)) {
		if(p & (CPY_D2S|DEL_DST))
			printf("S ");
		show_file(Slfn); }

	if(p & (CPY_D2S|DEL_DST)) {
		if(p & (CPY_S2D|DEL_SRC))
			printf("D ");
		show_file(Dlfn); }

	if(p & ALERT) alert();

	if(Autoskip && (p & AUTOCOPY)) goto doskip;
	if(Autocopy && (p & AUTOCOPY)) goto docopy;

loop1:
	if(p & CPY_S2D) printf("All Copy(s->d) ");
	if(p & CPY_D2S)	printf("Kopy(d->s) ");
	if(p & DEL_DST) printf("Delete(d) ");
	if(p & DEL_SRC) printf("Erase(s) ");
	printf("Skip Bypass Leave?");
loop2: switch(toupper(kbget())) {
	default: goto loop2;
	case 0x1B:
		putc('\n', stdout);
		severe_error();
		goto loop1;
	case 'A': if(!(p & CPY_S2D)) goto loop2;
		Autocopy = 0x55;
		printf("All");
	case 'C' : if(!(p & CPY_S2D)) goto loop2;
	docopy:
		printf("Copy\n");
		if(p & PROMPT) {
			if(Safe && !yesno("Are you sure"))
				goto loop1; }
		if(check_ro(Dest, Dlfn)) {
			copy_file(Source, Dest, Slfn.Time, Slfn.Date);
			fix_attr(Dest, Dlfn.Attrib, Slfn.Attrib); }
		break;
	case 'K' : if(!(p & CPY_D2S)) goto loop2;
		printf("Kopy\n");
		if(Safe && !yesno("Are you sure"))
			goto loop1;
		if(check_ro(Source, Slfn)) {
			copy_file(Dest, Source, Dlfn.Time, Dlfn.Date);
			fix_attr(Source, Slfn.Attrib, Dlfn.Attrib); }
		break;
	case 'D' : if(!(p & DEL_DST)) goto loop2;
		printf("Delete\n");
		if(Safe && !yesno("Are you sure"))
			goto loop1;
		if(check_ro(Dest, Dlfn))
			test_rc(lfn_delete(Dest));
		break;
	case 'E' : if(!(p & DEL_SRC)) goto loop2;
		printf("Erase\n");
		if(Safe && !yesno("Are you sure"))
			goto loop1;
		if(check_ro(Source, Slfn))
			test_rc(lfn_delete(Source));
		break;
	case 'B' : Bypass = 1; printf("Bypass\n");	break;
	case 'L' : Bypass = 2; printf("Leave\n");	break;
	case 'S' : doskip:
		printf("Skip\n"); }
}

/*
 * Correct 1-hour difference
 */
int chk1hour()
{
	unsigned t, h, mi, s, H, MI, S;

	if(!Hour)
		return 0;

	// If size different not 1-hour fix
	if(	(Slfn.SizeL[0] != Dlfn.SizeL[0]) || (Slfn.SizeL[1] != Dlfn.SizeL[1])
	||	(Slfn.SizeH[0] != Dlfn.SizeH[0]) || (Slfn.SizeH[1] != Dlfn.SizeH[1]) ) {
		return 0; }

	// Read timestamps
	t	= Slfn.Time;
	h	= t>>11;
	mi	= (t>>5)&0x3F;
	s	= t & 0x1F;
	t	= Dlfn.Time;
	H	= t>>11;
	MI	= (t>>5)&0x3F;
	S	= t & 0x1F;

	if((mi != MI) || (s != S))			// Minites/seconds mismatch
		return 0;

	if(Slfn.Date != Dlfn.Date) {		// dates differ
		if( ((h-H) == 23) || ((H-h) == 23) )
			goto dofix;
		return 0; }

	if( ((h-H) != 1) && ((H-h) != 1) )	// Exactly 1 hour
		return 0;

dofix:
	if(Hour != 255) {
		printf("Update TS: ");
		stpcpy(Dend, Dlfn.Lname);
		fputs(Dest, stdout);
		if(t = s = lfn_open(Dest, F_WRITE|F_APPEND)) {
			if(touch(t, Slfn.Time, Slfn.Date))
				s = 0;
			close(t); }
		if(!s) {
			printf(" Failed!");
			serious_error(); }
		putc('\n', stdout); }
	return 255;
}

/*
 * Test for "system" directory ("." or "..")
 */
int issys(unsigned char *n)
{
	if(*n == '.') switch(n[1]) {
		case '.' :
			if(n[2])
				break;
		case 0 : return 255; }
	return 0;
}

/*
 * Delete a directory tree
 */
deltree()
{
	unsigned i, j, dir;
	unsigned char *sbase, *dbase;
	static int rc;

	Atop = Ntop = 0;
	sbase = Send;
	dbase = Dend;
	dir = Dtop;

	if(Verbose) {
		fputs("RmPath: ", stdout);
		fputs(Dest, stdout);
		putc('\n', stdout); }
	stpcpy(Dend, "*.*");

	if(rc = lfn_find_first(Dest, 0xFF, Dlfn)) {
		if((rc != RC_NOFILE) && (rc != RC_NOMORE)) {
			printf("Unable to access dest path, code=%u\n", rc);
			severe_error();
			return;  } }
	else {
		do {
			if(Dlfn.Attrib & DIRECTORY) {
				if(issys(Dlfn.Lname))
					continue;
				Astring(&Dseg, Dlfn.Lname);
				continue; }
			add_file(Dlfn); }
		while !lfn_find_next(Dlfn);
		lfn_find_close(); }

	// Delete all files in directory
	for(i=0; i < Atop; i += 15) {
		get_file(i, Dlfn);
		stpcpy(Dend, Dlfn.Lname);
		if(Debug) {
			printf("Rmfile: ");
			fputs(Dest, stdout);
			putc('\n', stdout); }
		if(check_ro(Dest, Dlfn)) {
			if(test_rc(lfn_delete(Dest))) {
				printf("Failed to DELETE:");
				fputs(Dest, stdout);
				putc('\n', stdout);
				severe_error(); } } }

	// Recurse into sub-directories
	i = Dtop;
	j = dir;
	while(j < Dtop) {
		j = Gstring(Dseg, j, Temp);
		Dend = stpcpy(stpcpy(Dend, Temp), "\\");
		deltree();
		Dend = dbase; }

	// Remove the parent directory
	*(Dend-1) = 0;
	if(Debug) {
		printf("RmDir: ");
		fputs(Dest, stdout);
		putc('\n', stdout); }
	if(test_rc(lfn_rmdir(Dest)))
		severe_error();

	Dtop = dir;
}

/*
 * Synchronize a directory
 */
void sync_dir(void)
{
	unsigned i, j, dir;
	unsigned char *sbase, *dbase;
	static int rc;
	static unsigned ddir, sync_level;

	Atop = Ntop = Bypass = 0;
	sbase = Send;		// Save source position
	dbase = Dend;		// Save dest position
	dir = Dtop;			// Save directory position
	if(Autocopy == 0x55)
		Autocopy = 0;
	++sync_level;		// Keep track of how deep

	if(Verbose) {
		fputs(Path, stdout);
		putc('\n', stdout); }
	stpcpy(Send, "*.*");
	stpcpy(Dend, "*.*");

	// Scan source for files & directories
	if(rc = lfn_find_first(Source, 0xFF, Slfn)) {
		if((rc != RC_NOFILE) && (rc != RC_NOMORE)) {
			printf("Unable to access source path, code=%u\n", rc);
			severe_error();
			goto doexit; } }
	else {
		do {
			if(Slfn.Attrib & VOLUME)
				continue;
			if(Slfn.Attrib & (HIDDEN|SYSTEM)) {
				if(!Hidden)
					continue; }
			stpcpy(Send, Slfn.Lname);
			for(i=0; i < Etop; ++i) {
				if(fmatch(Path, Elist[i]))
					goto skipsource; }
			if(ScanSFN) {
				if(issfn(Slfn.Lname)) {
					printf("Possible SFN in source:\n");
					fputs(Slfn.Lname, stdout);
					putc('\n', stdout);
					severe_error(); } }
			if(Slfn.Attrib & DIRECTORY) {
				if(issys(Slfn.Lname))
					continue;
				if(Debug) {
					printf("Recording source dir: ");
					fputs(Slfn.Lname, stdout);
					putc('\n', stdout); }
				Astring(&Dseg, Slfn.Lname);
				continue; }
			if(Debug) {
				printf("Recording source file: ");
				fputs(Slfn.Lname, stdout);
				putc('\n', stdout); }
			add_file(Slfn);
skipsource: }
		while(!lfn_find_next(Slfn));
		lfn_find_close(); }

	// Scan dest for files & directories, reporting any that do not
	// exist in the source
	ddir = Dtop;
retry1:
	if(rc = lfn_find_first(Dest, 0xFF, Dlfn)) {
		if(rc == RC_NODIR) {
			i = *--Dend;
			*Dend = 0;
			printf("DEST path does not exist:\n");
			fputs(Dest, stdout);
			putc('\n', stdout);
			if(sync_level <= SkipTop)
				goto doexit;
			j = 0;
			if(Autodir || (j = yesno("~Create"))) {
				rc = lfn_mkdir(Dest);
				*Dend++ = i;
				if(!test_rc(rc)) {
					lfn_find_close();
					if(j == 0x55) {
						AutoDirLevel = sync_level;
						if(!Autocopy) Autocopy = 0xAA;
						Autodir = 255; }
					goto retry1; }
				severe_error(); }
			goto doexit; }
		if((rc != RC_NOFILE) && (rc != RC_NOMORE)) {
			printf("Unable to access dest path, code=%u\n", rc);
			severe_error();
			goto doexit; }
		goto checknew; }

	do {
		if(Dlfn.Attrib & VOLUME)
			continue;
		if(Dlfn.Attrib & (HIDDEN|SYSTEM)) {
			if(!Hidden)
				continue; }
		*Slfn.Lname = 0;
		stpcpy(Send, Dlfn.Lname);
		for(i=0; i < Etop; ++i) {
			if(fmatch(Path, Elist[i]))
				goto skipdest; }
		if(Dlfn.Attrib & DIRECTORY) {
			if(issys(Dlfn.Lname))
				continue;
			if(!scanfor(&Dseg, dir, Dlfn.Lname)) {
				if(Xdir)
					Astring(&Dseg, Dlfn.Lname); }
			continue; }
		if(Bypass == 1) continue;
		if(Bypass == 2) goto doexit;
		if(Debug) {
			printf("Testing dest file: ");
			fputs(Dlfn.Lname, stdout);
			putc('\n', stdout); }
		if(!(i=scanfor(&Nseg, 0, Dlfn.Lname))) {
			stpcpy(Slfn.Lname, Dlfn.Lname);
			file_action("File exists in DEST but not in SOURCE",
				CPY_D2S|DEL_DST);
			continue; }
		get_file(i=(i-1)*15, Slfn);
		poke(Aseg, i, Slfn.Attrib | 0x80);
		if(Chktime) {
			if(Dlfn.Date < Slfn.Date) goto older;
			if(Dlfn.Date > Slfn.Date) goto younger;
			if(Dlfn.Time < Slfn.Time) goto older;
			if(Dlfn.Time > Slfn.Time) goto younger; }
		// Files have same time/date
		if(	(Slfn.SizeL[0] != Dlfn.SizeL[0]) || (Slfn.SizeL[1] != Dlfn.SizeL[1])
		||	(Slfn.SizeH[0] != Dlfn.SizeH[0]) || (Slfn.SizeH[1] != Dlfn.SizeH[1]) ) {
			file_action("Files TIME/DATE match, but SIZE is different",
				CPY_D2S|CPY_S2D|DEL_DST|DEL_SRC|PROMPT|ALERT);
			continue; }
		// TIME/DATE & SIZE all match
		if(!Fast) {
			if(compare_file()) {
				file_action("Files TIME/DATE and SIZE match, but content differs",
					CPY_S2D|CPY_D2S|DEL_DST|DEL_SRC|PROMPT|ALERT); } }
		continue;
younger:
	if(!chk1hour()) {
		file_action("Destination file is NEWER than source",
			CPY_D2S|CPY_S2D|DEL_DST|DEL_SRC|PROMPT|ALERT); }
	continue;
older:
	if(!chk1hour()) {
		file_action("Destination file is OLDER than source",
			CPY_D2S|CPY_S2D|DEL_DST|DEL_SRC|AUTOCOPY); }
skipdest:
	} while(!lfn_find_next(Dlfn));
	lfn_find_close();

	// Scan for files in source not found in dest
checknew:
	*Dlfn.Lname = 0;
	for(i=0; i < Atop; i += 15) {
		if(peek(Aseg, i) & 0x80)
			continue;
		if(Bypass == 1) break;
		if(Bypass == 2) goto doexit;
		get_file(i, Slfn);
		stpcpy(Dlfn.Lname, Slfn.Lname);
		file_action("File exists in SOURCE but not in DEST",
			CPY_S2D|DEL_SRC|AUTOCOPY); }

	// Scan for directories in dest but not in source
	i = ddir;
	while(i < Dtop) {
		i = Gstring(Dseg, i, Temp);
		printf("Path exists in DEST but not in SOURCE:\n");
		stpcpy(Dend, Temp);
		fputs(Dest, stdout);
		putc('\n', stdout);
		if(Safe) {
			if(!yesno("Remove"))
				continue;
			if(Safe && !yesno("Are you sure"))
				continue; }
		Dend = stpcpy(stpcpy(Dend, Temp), "\\");
		deltree(Temp);
		Dend = dbase; }

		
	// Recurse into subdirectories
	i = Dtop = ddir;
	j = dir;
	while(j < i) {
		j = Gstring(Dseg, j, Temp);
		Send = stpcpy(stpcpy(Send, Temp), "\\");
		Dend = stpcpy(stpcpy(Dend, Temp), "\\");
		sync_dir();
		Send = sbase;
		Dend = dbase; }

doexit:
	if(--sync_level < AutoDirLevel) {
		if(Autocopy == 0xAA)
			Autocopy = 0;
		AutoDirLevel = Autodir = 0; }
	Dtop = dir;
}

/*
 * Add a string to the text pool
 */
unsigned char *add_pool(unsigned char *s)
{
	unsigned t;
	while(isspace(*s))
		++s;
	t = Ptop;
	while(*s)
		TextPool[Ptop++] = toupper(*s++);
	while((Ptop > t) && isspace(TextPool[Ptop-1]))
		--Ptop;
	if(Ptop >= sizeof(TextPool))
		abort("String pool exhausted");
	TextPool[Ptop++] = 0;
	return TextPool + t;
}

/*
 * Add a file to the exclude list
 */
void exclude(unsigned char *file)
{
	unsigned h;
	unsigned char *ptr;

	h = 0;
	ptr = Temp;
	for(;;) switch(*ptr++ = *file++) {
		case 0 :
			if(!h)
				stpcpy(ptr-1, ".SNC");
			if(!(h = lfn_open(Temp, F_READ))) {
				printf("Cannot READ: ");
				fputs(Temp, stdout);
				abort("\n"); }
			while(lgets(ptr = Temp, sizeof(Temp)-1, h)) {
				while(isspace(*ptr))
					++ptr;
				switch(*ptr) {
				case ';' :
				case 0 : continue; }
				Elist[Etop++] = add_pool(ptr); }
			close(h);
			return;
		case ':' :
		case '\\': h = 0; continue;
		case '.' : h = 255; }
}

/*
 * Process a command line ARG
 */
void cmdarg(unsigned char *ptr)
{
	int r;
	unsigned i;
	unsigned char *p;

	r = 0;
	for(;;) {
		while(isspace(*ptr)) {
			++ptr;
			r = 0; }
		if(!*ptr)
			return;
		switch(i = (toupper(*ptr++)<<8) | toupper(*ptr++)) {
		case '-1' :		// One-hour fix
		case '/1' : p=&Hour;		//goto set1;
			if(toupper(*ptr) != '!')
				goto set1;
			++ptr;
			*p = 0x55;
			goto setn;
		case '-A' :		// copy Attributes
		case '/A' : p=&Attrset;		goto set1;
		case '-C' :		// Auto-copy
		case '/C' : p=&Autocopy;	goto set1;
		case '-D' :		// Auto-create directories
		case '/D' : p=&Autodir;		goto set1;
		case '-E' :
		case '/E' : p=&Halt;		goto set0;
		case '-H' :		// Ignore Hidden/System
		case '/H' : p=&Hidden;		goto set0;
		case '-I' :		// Ingore timestamp
		case '/I' : p=&Chktime;		goto set0;
		case '-L' :		// Disable LFN
		case '/L' : p=&Aseg;		goto set1;
		case '-Q' :		// Quiet mode
		case '/Q' : p=&Verbose;		goto set0;
		case '-R' :		// Override ReadOnly
		case '/R' : p=&ReadOnly;	goto set0;
		case '-S' :		// Auto-skip
		case '/S' : p=&Autoskip;	goto set1;
		case '-T' :		// Test file content
		case '/T' : p=&Fast;		goto set0;
		case '-V' :		// Debug messages
		case '/V' : p=&Debug;		goto set1;
		case '-X' :		// ignore eXtra directories
		case '/X' : p=&Xdir;		goto set0;
		case '-Y' :		// Suppress "are you sure?"
		case '/Y' : p=&Safe;
		set0: *p = 0; setn:
			if(*ptr == '-') {
				++ptr;
				*p = ~*p; }
			r = 255;
			continue;
		case '-~' :
		case '/~' : p=&ScanSFN;
		set1: *p = 255; goto setn; }

		if(r)
			goto fail;

		switch(i) {
		case 'D=' :
			if(!(SkipTop = atoi(ptr)))
				goto fail;
			return;
		case 'E=' :
			stpcpy(TextPool, ptr);
			return; }

		if(!Send) {
			Send = stpcpy(Source, ptr-2);
			return; }
		if(!Dend) {
			Dend = stpcpy(Dest, ptr-2);
			return; }

fail:	printf("Unknown option: %s\n", ptr-2);
		exit(-1); }
}

/*
 * Re-parse the command line arguments to obtain values in quotations
 *
 * New arguments are parsed into the same holding buffer as the original
 * ones, which means that the value of ARGV will not change. The global
 * ARGC is updated by this function, and the new argc value is returned
 * so that local copies may be easily updated as well.
 */
int parse_args(address) asm
{
		EXTRN	_ARGC, _ARGV
		MOV		SI,4[BP]				// Input address
		MOV		AH,[SI]					// Get length
		INC		SI						// Advance
		MOV		BX,offset DGRP:_ARGV+2	// BX = ARGV output
		LEA		DI,38[BX]				// Argument storage
		XOR		CH,CH					// flag = 0;
; Process each char from arguments
qa1:	AND		AH,AH					// End of arguments
		JZ		qa7						// Yes, exit
		DEC		AH						// Reduce count
		MOV		AL,[SI]					// Get from source
		AND		AL,AL					// End of string?
		JZ		qa7						// Exit
		CMP		AL,0Dh					// End if line?
		JZ		qa7						// Exit
		INC		SI						// Advance
		AND		CH,CH					// Already processing string?
		JNZ		qa3						// Yes
		CMP		AL,' '					// Space?
		JZ		qa1						// Keep going
		CMP		AL,9					// Tab
		JZ		qa1						// Keep going
		MOV		[BX],DI					// Save pointer
		ADD		BX,2					// Advance
		CMP		AL,'"'					// Start of quote?
		JNZ		qa2						// No, continune
		MOV		CH,2					// Inside quite
		JMP short qa1					// And continue
; Processing a string
qa2:	MOV		CH,1					// Not-quoted
qa3:	CMP		AL,' '					// Space
		JZ		qa5						// Process
		CMP		AL,09h					// Tab?
		JZ		qa5						// Process
		CMP		AL,'"'					// Quote?
		JNZ		qa6						// No processing
; Encountered '"' within a string
		CMP		CH,2					// Processing within quotes?
		JNZ		qa6						// No, process
		CMP		AL,[SI]					// Double quotes?
		JNZ		qa4						// No, back to non-quote
		INC		SI						// Advance
		DEC		AH						// Reduce length
		JMP short qa6					// and process
; Encountered CR
qa4:	MOV		CH,1					// Reset flag
qa5:	CMP		CH,1					// Quoted?
		JNZ		qa6						// No
		XOR		AL,AL					// Zero terminator
		MOV		CH,AL					// Flag = 0
qa6:	MOV		[DI],AL					// Write to arg
		INC		DI						// Advance
		JMP short qa1					// and proceed
qa7:	XOR		AL,AL					// Get zero
		MOV		[DI],AL					// Zero terminate
		MOV		AX,BX					// Get address
		SUB		AX,offset DGRP:_ARGV	// Calculate offset
		SHR		AX,1					// /2
		MOV		DGRP:_ARGC,AX			// Set new value
}

main(int argc, char *argv[])
{
	unsigned i;

//	printf("%04x\n", malloc(1));

	Aseg = lfn_getdir(Temp);		// Determine if LFN present

	if(getenv("SYNCOPT", Temp+1)) {	// Process SYNCOPT arguments
		printf("SYNCOPT=%s\n", Temp+1);
		*Temp = 255;
		argc = parse_args(Temp);
		for(i=1; i < argc; ++i)
			cmdarg(argv[i]); }

	argc = parse_args(0x80);		// Process command arguments
	for(i=1; i < argc; ++i)
		cmdarg(argv[i]);

	if(!Dend)						// Not enough args
		abort(Help);

	switch(*(Send-1)) {				// Set up source prefix
		default: *Send++ = '\\';
		case ':' :
		case '\\' : }
	switch(*(Dend-1)) {				// Set up dest prefix
		default: *Dend++ = '\\';
		case ':' :
		case '\\' : }
	Path = Send;

	if(Aseg)						// No LFN - switch to SFN
		patch_lfn();

	if(*TextPool)
		exclude(TextPool);
		
	if(!(Aseg = alloc_seg(4096*3)))
		abort("No memory");
	Dseg = (Nseg = Aseg + 4096) + 4096;

	if(Verbose) {
		printf("Source: ");
		fputs(Source, stdout);
		printf("\nDest  : ");
		fputs(Dest, stdout);
		putc('\n', stdout); }

	sync_dir();
}	
