/*
 * 6502 Cross Assembler
 *
 * ?COPY.TXT 1983-2005 Dave Dunfield
 * **See COPY.TXT**.
 */
#include <stdio.h>
#include <ctype.h>

#include "xasm.h"

/* 6502 opcode table */
static char opcodes[] = {
	'B','R','K',		0x81, 0x00,
	'C','L','C',		0x81, 0x18,
	'C','L','D',		0x81, 0xd8,
	'C','L','I',		0x81, 0x58,
	'C','L','V',		0x81, 0xb8,
	'D','E','X',		0x81, 0xca,
	'D','E','Y',		0x81, 0x88,
	'I','N','X',		0x81, 0xe8,
	'I','N','Y',		0x81, 0xc8,
	'N','O','P',		0x81, 0xea,
	'P','H','A',		0x81, 0x48,
	'P','H','P',		0x81, 0x08,
	'P','L','A',		0x81, 0x68,
	'P','L','P',		0x81, 0x28,
	'R','T','I',		0x81, 0x40,
	'R','T','S',		0x81, 0x60,
	'S','E','C',		0x81, 0x38,
	'S','E','D',		0x81, 0xf8,
	'S','E','I',		0x81, 0x78,
	'T','A','X',		0x81, 0xaa,
	'T','A','Y',		0x81, 0xa8,
	'T','S','X',		0x81, 0xba,
	'T','X','A',		0x81, 0x8a,
	'T','X','S',		0x81, 0x9a,
	'T','Y','A',		0x81, 0x98,
	'A','D','C',		0x82, 0x60,
	'A','N','D',		0x82, 0x20,
	'C','M','P',		0x82, 0xc0,
	'E','O','R',		0x82, 0x40,
	'L','D','A',		0x82, 0xa0,
	'O','R','A',		0x82, 0x00,
	'S','B','C',		0x82, 0xe0,
	'S','T','A',		0x83, 0x80,
	'A','S','L',		0x84, 0x00,
	'L','S','L',		0x84, 0x00,
	'L','S','R',		0x84, 0x40,
	'R','O','L',		0x84, 0x20,
	'R','O','R',		0x84, 0x60,
	'B','C','C',		0x85, 0x90,
	'B','C','S',		0x85, 0xb0,
	'B','E','Q',		0x85, 0xf0,
	'B','M','I',		0x85, 0x30,
	'B','N','E',		0x85, 0xd0,
	'B','P','L',		0x85, 0x10,
	'B','V','C',		0x85, 0x50,
	'B','V','S',		0x85, 0x70,
	'B','I','T',		0x86, 0x20,
	'C','P','X',		0x87, 0xd0,
	'C','P','Y',		0x87, 0xb0,
	'D','E','C',		0x88, 0xc0,
	'I','N','C',		0x88, 0xe0,
	'L','D','Y',		0x89, 0x90,
	'L','D','X',		0x8a, 0xa0,
	'S','T','X',		0x8b, 0x80,
	'S','T','Y',		0x8c, 0x80,
	'J','S','R',		0x8d, 0x00,
	'J','M','P',		0x8e, 0x00,
/* directives */
	'E','Q','U',		228,	0,
	'O','R','G',		229,	0,
	'F','C','B',		230,	0,
	'F','D','B',		231,	0,
	'R','D','B',		231,	-1,
	'F','C','C',		232,	0,
	'F','C','C','H',	232,	1,
	'F','C','C','Z',	232,	2,
	'R','M','B',		233,	0,
	'D','B',			230,	0,
	'D','W',			231,	0,
	'D','R','W',		231,	-1,
	'S','T','R',		232,	0,
	'S','T','R','H',	232,	1,
	'S','T','R','Z',	232,	2,
	'D','S',			233,	0,
	'E','N','D',		234,	0,
	'P','A','G','E',	248,	0,
	'T','I','T','L','E',249,0,
	'S','P','A','C','E',250,0,
	'L','I','S','T',	251,	0,
	'N','O','L','I','S','T',252,0,
	'L','I','N','E',	253,	0,
	0 };	/* end of table */

/* addressing mode tables, gives value added to opcode for all   */
/* applicable addressing modes, zero indicates that mode invalid */
/* -------------- Imm.  Abs.  Zero  [n,X] [n],Y <n,X  n,x   n,y   <n,y  Ind.*/
char optab1[] = { 0x09, 0x0d, 0x05, 0x01, 0x11, 0x15, 0x1d, 0x19, 0x00, 0x00 };
char optab2[] = { 0x00, 0x0d, 0x05, 0x01, 0x11, 0x15, 0x1d, 0x19, 0x00, 0x00 };
char optab3[] = { 0x00, 0x0c, 0x04, 0,00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 };
char optab4[] = { 0x10, 0x1c, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 };
char optab5[] = { 0x00, 0x0e, 0x06, 0x00, 0x00, 0x16, 0x1e, 0x00, 0x00, 0x00 };
char optab6[] = { 0x10, 0x1c, 0x14, 0x00, 0x00, 0x24, 0x2c, 0x00, 0x00, 0x00 };
char optab7[] = { 0x02, 0x0e, 0x06, 0x00, 0x00, 0x00, 0x00, 0x1e, 0x16, 0x00 };
char optab8[] = { 0x00, 0x0e, 0x06, 0x00, 0x00, 0x00, 0x00, 0x00, 0x16, 0x00 };
char optab9[] = { 0x00, 0x0c, 0x04, 0x00, 0x00, 0x14, 0x00, 0x00, 0x00, 0x00 };
char optaba[] = { 0x00, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 };
char optabb[] = { 0x00, 0x4c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x6c };

/* error messages */
static char *etext[] = { "?",
	"Unknown instruction",
	"Out of range",
	"Invalid addressing mode",
	"Invalid register specification",
	"Undefined symbol",
	"Invalid expression syntax",
	"Invalid argument format",
	"Improperly delimited string" } ;

/* Symbol table & free pointer */
char symtab[SYMB_POOL], *symtop;

/* misc. global variables */
char buffer[LINE_SIZE+1], label[SYMB_SIZE+1], title[50], operand[80], optr,
	itype, opcode;
unsigned char instruction[80], outrec[35];

char optf=3, symf=0, intel=0, fulf=0, quietf=0, nlist=0, casf=0;

unsigned addr, line=0, pcount=0, lcount=0, value, length, emsg, ecnt,
	ocnt=0, pagel=59, pagew=80;

FILE *asm_fp, *hex_fp, *lst_fp;

/*
 * Define a symbol in the symbol table
 */
char *create_symbol(symbol, value)
	char *symbol;
	int value;
{
	register char *ptr, *ptr1;

	ptr = ptr1 = symtop;
	while(*symbol)
		*++ptr = *symbol++;
	*ptr1 = ptr - ptr1;
	*++ptr = value >> 8;
	*++ptr = value;
	if(ptr < (symtab+sizeof(symtab))) {
		symtop = ptr + 1;
		return ptr1; }
	return 0;
}

/*
 * Lookup a symbol in the symbol table
 */
char *lookup_symbol(symbol)
	char *symbol;
{
	register int l;
	register unsigned char *ptr, *ptr1;
	char *ptr2;

	ptr = symtab;
again:
	if((ptr2 = ptr) >= symtop) {
		value = 0x8888;
		return 0; }
	ptr1 = symbol;
	l = *ptr++ & SYMMASK;
	while(l--) {
		if(*ptr1++ != *ptr++) {
			ptr += l + 2;
			goto again; } }
	if(*ptr1) {
		ptr += 2;
		goto again; }
	value = *ptr++ << 8;
	value |= *ptr;
	return ptr2;
}

/*
 * Set the value for a symbol in the symbol table
 */
set_symbol(ptr, value)
	char *ptr;
	unsigned value;
{
	ptr += (*ptr & SYMMASK);
	*++ptr = value >> 8;
	*++ptr = value;
}

/*
 * Display the symbol table (sorted)
 */
dump_symbols()
{
	unsigned char *ptr, *hptr;
	unsigned int l, h, w;

	fprintf(lst_fp, "SYMBOL TABLE:\n\n");
	w = 0;
	for(;;) {
		for(hptr = symtab; hptr < symtop; hptr += (*hptr & SYMMASK) + 3)
			if(!(*hptr & SYMSORT))
				goto found;
		putc('\n', lst_fp);
		return;
found:	for(ptr = (*hptr & SYMMASK) + hptr + 3; ptr < symtop; ptr += (*ptr & SYMMASK) + 3) {
			if(*ptr & SYMSORT)
				continue;
			if(compsym(ptr, hptr) < 0)
				hptr = ptr; }
		*(ptr = hptr) |= SYMSORT;
		h = l = *ptr++ & SYMMASK;
		if((w + l + 7) >= pagew) {			/* Overrun page length */
			putc('\n', lst_fp);
			w = 0; }
		if(w) {								/* Not a start of line - separate */
			fprintf(lst_fp, "   ");
			w += 3; }
		while(l--)
			putc(*ptr++, lst_fp);
		w += (l = (h > 8) ? 24 : 8) + 5;	/* Calculate extended length */
		while(h++ < l)
			putc(' ', lst_fp);
		l = *ptr++ << 8;
		l |= *ptr++;
		fprintf(lst_fp, "-%04x", l); }
}

/*
 * Compare two symbol table entries
 */
compsym(sym1, sym2)
	char *sym1;
	char *sym2;
{
	int l, l1, l2;
	l1 = *sym1++ & SYMMASK;
	l2 = *sym2++ & SYMMASK;
	l = (l1 < l2) ? l1 : l2;
	do {
		if(*sym1 > *sym2)
			return 1;
		if(*sym1++ < *sym2++)
			return -1; }
	while(--l);
	if(l1 > l2)
		return 1;
	if(l1 < l2)
		return -1;
	return 0;
}

/************************************/
/* get a character from the operand */
/************************************/
char getchr()
{
	register char chr;

	if(chr=operand[optr]) ++optr;
	return(chupper(chr));
}

/*
 * Convert character to upper case if NOT case sensitive
 */
chupper(c)
	char c;
{
	return casf ? c : ((c >= 'a') && (c <= 'z')) ? c - ('a'-'A') : c;
}

/*
 * Open a filename with the appriopriate extension &
 * report an error if not possible
 */
FILE *open_file(filename, extension, options)
	char *filename, *extension, *options;
{
	char buffer[100], *ptr, *dot;
	FILE *fp;

	dot = 0;

	for(ptr = buffer; *ptr = *filename; ++ptr) {	/* Copy filename */
		if(*filename == '.')
			dot = filename;
		else if(*filename == '\\')
			dot = 0;
		++filename; }

	if(!dot) {									/* No extension supplied */
		*ptr++ = '.';
		do
			*ptr++ = *extension;
		while(*extension++); }
	else
		*dot = 0;

	if(!(fp = fopen(buffer, options))) {
		fprintf(stderr,"Unable to access: '%s'\n", buffer);
		exit(-1); }

	return fp;
}

/*
 * Main program
 */
main(argc, argv)
	unsigned argc;
	char *argv[];
{	unsigned i, temp, daddr, vbyt;
	int stemp;
	char chr, cflg, pflg, *ptr, *lfile, *cfile;

	if(argc < 2)
		abort("\nUse: asm02 <filename> [-cfiqst c=file l=file o=n p=length w=width]\n\n?COPY.TXT 1983-2005 Dave Dunfield\n**See COPY.TXT**.\n");

	pflg = 0;

/* parse for command line options. */
	lfile = cfile = argv[1];
	for(i=2; i < argc; ++i) {
		if(*(ptr = argv[i]) == '-') {		/* Enable switch */
			while(*++ptr) switch(toupper(*ptr)) {
				case 'C' : casf = -1;		break;
				case 'F' : fulf = -1;		break;
				case 'I' : intel = -1;		break;
				case 'Q' : quietf = -1;		break;
				case 'S' : symf = -1;		break;
				case 'T' : lfile = 0;		break;
				default: goto badopt; }
			continue; }
		if(*++ptr == '=') switch(toupper(*(ptr++ - 1))) {
			case 'L' : lfile = ptr;			continue;
			case 'C' : cfile = ptr;			continue;
			case 'O' : optf=atoi(ptr);		continue;
			case 'P' : pagel=atoi(ptr)-1;	continue;
			case 'W' : pagew=atoi(ptr);		continue; }
	badopt:
		fprintf(stderr,"Invalid option: %s\n", argv[i]);
		exit(-1); }

	asm_fp = open_file(argv[1], "ASM", "r");
	hex_fp = open_file(cfile, "HEX", "w");
	lst_fp = (lfile) ? open_file(lfile, "LST", "w") : stdout;
	strncpy(title,argv[1],sizeof(title)-1);

/* first pass - build symbol table */
	symtop = symtab;
	do {
		addr = cflg = 0; line = 1;
		if(!quietf) if(pflg) fprintf(stderr,"Opt... ");
			else	fprintf(stderr,"First pass... ");
		while(readline()) {
			if(!optr) {
				locins(instruction);
				if(label[0]) {				/* label exists */
					if(pflg) {				/* optomization pass */
						ptr = lookup_symbol(label);
						if(itype==100) {	/* EQU statement */
							optr = 0; vbyt = value;
							if((temp=eval()) != vbyt) {
								if(pflg >= optf) optfail();
								set_symbol(ptr, temp);
								cflg=1; } }
						else if(value != addr) {
							if(pflg >= optf) optfail();
							set_symbol(ptr, addr);
							cflg=1; } }
					else {							/* original pass-1 */
						if(lookup_symbol(label)) {	/* duplicate label */
							fprintf(lst_fp,"** Line %u - Duplicate symbol: '%s'\n",line,label);
							++ecnt; }
						else {
							cflg=1;			/* indicate symbol table change */
							if(!(ptr = create_symbol(label, addr))) {
								xabort("symbol");
								fprintf(lst_fp,"** Line %u - Symbol table overflow\n", line);
								++ecnt;
								break; } } } }
				emsg = length = optr = 0;
				switch(itype) {
					case 1 :		/* inherent addressing */
						++length;
						break;
					case 2 :
						memop(optab1);
						break;
					case 3 :
						memop(optab2);
						break;
					case 4 :		/* accumulator */
						if((chupper(operand[0])=='A') && !issymbol(operand[1]))
							++length;
						else
							memop(optab5);
						break;
					case 5 :		/* branches */
						length = 2;
						break;
					case 6 :
						memop(optab3);
						break;
					case 7 :
						memop(optab4);
						break;
					case 8 :
						memop(optab5);
						break;
					case 9 :
						memop(optab6);
						break;
					case 10 :
						memop(optab7);
						break;
					case 11 :
						memop(optab8);
						break;
					case 12 :
						memop(optab9);
						break;
					case 13 :
						memop(optaba);
						break;
					case 14 :
						memop(optabb);
						break;
					case 100 :			/* EQU directive */
						if(!pflg) {
							set_symbol(ptr, eval());
						err1st:
							if(emsg && !optf) {
								fprintf(lst_fp,"** Line %u - %s\n",line,etext[emsg]);
								++ecnt; } }
						break;
					case 101 :			/* ORG directive */
						addr=eval();
						goto err1st;
					case 102 :			/* FCB statement */
						length=1;
						while(nxtelem()) ++length;
						break;
					case 103 :			/* FDB statement */
						length=2;
						while(nxtelem()) length += 2;
						break;
					case 104 :			/* FCC statements */
						chr=operand[0];
						for(i=1;(operand[i])&&(chr!=operand[i]); ++i) ++length;
						if(opcode == 2) ++length;
						break;
					case 105 :			/* RMB statement */
						addr = addr + eval();
						goto err1st;
					case 106 :			/* END statement */
						goto end1;
					case 125 :			/* LINE directive */
						line = eval(); } }
			else
				length = 0;
			addr += length;
			++line; /* += (itype < 120); */ }
end1:	++pflg;
		rewind(asm_fp); }
	while((optf)&&(cflg)&&(!ecnt));

/* second pass - generate object code */

	if(!quietf)
		fprintf(stderr,"Second pass... ");
	addr = emsg = 0; line = pcount = 1; lcount = 9999;
	while(readline()) {
		daddr = addr;
		if(!optr) {
			if(!locins(instruction))
				reperr(1);			/* unknown opcode */
			length = temp = optr = 0;
			switch(itype) {
				case 1 :		/* inherent addressing */
					instruction[length++] = opcode;
					break;
				case 2 :		/* full memory reference */
					memop(optab1);
					goto wrinst;
				case 3 :		/* as above, no immed */
					memop(optab2);
					goto wrinst;
				case 4 :		/* accumulator instructions */
					if((chupper(operand[0])=='A') && !issymbol(operand[1])) {
						itype = 0x0a;
						length = 1; }
					else
						memop(optab5);
					goto wrinst;
				case 5 :		/* branches */
					instruction[length++] = opcode;
					stemp = (eval() - addr) - 2;
					if((stemp > 127) || (stemp < -128))
						reperr(2);				/* out or range */
					instruction[length++] = stemp;
					break;
				case 6 :		/* abs. or zero only */
					memop(optab3);
					goto wrinst;
				case 7 :		/* immed, abs. or zero only */
					memop(optab4);
					goto wrinst;
				case 8 :		/* abs, zero, zpagex or abs,x only */
					memop(optab5);
					goto wrinst;
				case 9 :
					memop(optab6);
					goto wrinst;
				case 10 :
					memop(optab7);
					goto wrinst;
				case 11 :
					memop(optab8);
					goto wrinst;
				case 12 :
					memop(optab9);
					goto wrinst;
				case 13 :
					memop(optaba);
					goto wrinst;
				case 14 :
					memop(optabb);
wrinst:				instruction[0] = opcode + itype;
					instruction[1] = value & 255;
					if(length > 2)
						instruction[2] = value >> 8;
					break;
				case 100 :			/* equate statement */
					daddr=eval();	/* generate errors if any */
					break;
				case 101 :			/* ORG statement */
					if(ocnt) wrrec();
					daddr=addr=eval();
					break;
				case 102 :			/* FCB statement */
					do {
						instruction[length++] = eval(); }
					while(operand[optr-1] == ',');
					break;
				case 103 :			/* FDB statement */
					do {
						temp = eval();
						if(opcode) {
							instruction[length++] = temp >> 8;
							instruction[length++] = temp; }
						else {
							instruction[length++] = temp;
							instruction[length++] = temp >> 8; } }
					while(operand[optr-1] == ',');
					break;
				case 104 :			/* FCC statements */
					chr=operand[0];
					for(i=1;((operand[i])&&(chr!=operand[i])); ++i)
						instruction[length++] = operand[i];
					if(!operand[i])	reperr(8);
					if(opcode == 1)
						instruction[length-1] = instruction[length-1] | 0x80;
					if(opcode == 2)
						instruction[length++]=0;
					break;
				case 105 :			/* RMB statement */
					if(ocnt) wrrec();
					addr += eval();
					break;
				case 106 :			/* END statement */
					goto end2;
				case 120 :			/* PAGE statement */
					lcount=9999;
					break;
				case 121 :			/* TITLE directive */
					strncpy(title, operand, sizeof(title)-1);
					break;
				case 122 :			/* SPACE directive */
					fprintf(lst_fp,"\n");
					++lcount;
					break;
				case 123 :			/* LIST directive */
					if(nlist) --nlist;
					break;
				case 124 :			/* NOLIST directive */
					++nlist;
					break;
				case 125 :			/* LINE directive */
					line = eval(); } }
		else
			length=itype=0;

/* generate listing */
	if(((itype<120) && fulf && !nlist) || emsg) {
		if(++lcount >= pagel)
			write_title();
		fprintf(lst_fp,"%04x ",daddr);
			for(i=0; i < 6; ++i) {
				if(i < length)
					fprintf(lst_fp," %02x", instruction[i]);
				else
					fprintf(lst_fp,"   "); }
			fprintf(lst_fp," %c%5u  %s\n",(length <= 6) ? ' ' : '+',line,buffer); }
		if(emsg) {
			fprintf(lst_fp,"  ** ERROR ** - %u - %s\n", emsg, etext[emsg]);
			++ecnt; ++lcount;
			emsg=0; }
		++line; /* += (itype<120); */
/* write code to output record */
	for(i=0; i<length; ++i) {
		if(!ocnt) {			/* first byte of record */
			outrec[ocnt++]=addr>>8;
			outrec[ocnt++]=addr; }
		++addr;
		outrec[ocnt++]=instruction[i];
		if(ocnt>33) wrrec(); } }	/* record is full, write it */

end2:
	if(ocnt) wrrec();
	if(intel) fprintf(hex_fp,":00000001FF\n");
	else fprintf(hex_fp,"S9030000FC\n");

	if(ecnt) fprintf(lst_fp,"\n %u error(s) occurred in this assembly.\n",ecnt);
	if(!quietf) fprintf(stderr,"%u error(s).\n",ecnt);

/* display the symbol table */
	if(symf) {
		write_title();
		dump_symbols(); }

	fclose(asm_fp);
	fclose(hex_fp);
	fclose(lst_fp);

	return ecnt ? -2 : 0;
}

/***********************************************/
/* lookup instruction in the instruction table */
/***********************************************/
int locins(inst)
	char inst[];
{
	register char *ptr, *ptr1;

	ptr = opcodes;
	do {
		ptr1 = inst;
		while(*ptr == *ptr1) {
			++ptr;
			++ptr1; }
		if((*ptr < 0) && !*ptr1) {
			itype = *ptr++ & 127;
			opcode = *ptr++;
			return 1; }
		while(*ptr > 0)
			++ptr; }
	while(*(ptr += 2));

	itype = 119;
	return(inst[0] == 0);
}

/*************************************************/
/* read a line from input file, and break it up. */
/*************************************************/
int readline()
{
	register int i;
	register char *ptr;

	if(!fgets(ptr = buffer, LINE_SIZE, asm_fp))
		return 0;

	i = 0;
	while(issymbol(*ptr)) {
		label[i] = chupper(*ptr++);
		if(i < SYMB_SIZE)
			++i; }
	label[i]=0;
	if(*ptr == ':')
		++ptr;
	while(isspace(*ptr))
		++ptr;
	if(((*ptr != '*') && (*ptr != ';')) || i) {
		i = 0;
		while(*ptr && !isspace(*ptr))
			instruction[i++] = toupper(*ptr++);
		instruction[i]=0;
		while(isspace(*ptr))
			++ptr;
		optr = i = 0;
		while(*ptr && (i < 79))
			operand[i++] = *ptr++;
		operand[i] = 0;
		return 1; }

	return optr = 1;
}

/***************************************/
/* test for valid terminator character */
/***************************************/
int isterm(chr)
	register char chr;
{
	switch(chr) {
		case 0 :
		case ' ' :
		case '\t':
		case ',' :
		case ']' :
		case ')' :
			return 1; }
	return 0;
}

/*************************************/
/* Test for a valid symbol character */
/*************************************/
int issymbol(c)
	char c;
{
	switch(c) {			/* Special allowed characters */
		case '_' :
		case '?' :
		case '!' :
		case '.' :
			return 1; }

	return isalnum(c);
}

/* report an error */
reperr(errn)
	register int errn;
{	if(!emsg) emsg=errn;
}

/*******************************/
/* write record to output file */
/*******************************/
wrrec()
{	register unsigned i, chk, chr;

	xhex(ocnt-2);
	if(intel) {					/* intel hex format */
		chk = outrec[0] + outrec[1] + ocnt - 2;
		fprintf(hex_fp,":%02x%02x%02x00", ocnt-2, outrec[0], outrec[1]);
		for(i=2; i<ocnt; ++i) {
			fprintf(hex_fp,"%02x", chr = outrec[i]);
			chk += chr; }
		fprintf(hex_fp,"%02x\n", 255 & (0-chk)); }
	else {						/* motorola hex format */
		chk = ocnt + 1;
		fprintf(hex_fp,"S1%02x", ocnt + 1);
		for(i=0; i<ocnt; ++i) {
			fprintf(hex_fp,"%02x", chr = outrec[i]);
			chk += chr; }
		fprintf(hex_fp,"%02x\n", 255 & ~chk); }
	ocnt = 0;
}

/*
 * Evaluate an expression.
 */
int eval()
{	register char c;
	unsigned result;

	result=getval();
	while(!isterm(c = getchr())) switch(c) {
		case '+' : result += getval();	break;
		case '-' : result -= getval();	break;
		case '*' : result *= getval();	break;
		case '/' : result /= getval();	break;
		case '\\': result %= getval();	break;
		case '&' : result &= getval();	break;
		case '|' : result |= getval();	break;
		case '^' : result ^= getval();	break;
		case '<' : result <<= getval();	break;
		case '>' : result >>= getval();	break;
		default: reperr(6); }

	return result;
}

/*
 * Get a single value (number or symbol)
 */
int getval()
{
	unsigned i, b, val;
	char chr, array[25], *ptr;

	switch(chr = operand[optr++]) {
		case '-' :				/* Negated value */
			return 0 - getval();
		case '~' :				/* Complemented value */
			return ~getval();
		case '=' :				/* Swap high and low bytes */
			i=getval();
			return (i<<8)+(i>>8);
		case '(' :				/* Nested expression */
			val = eval();
			if(operand[optr-1] != ')')
				reperr(6);
			return val; }
	--optr;

	val=b=i=0;
	switch(chr) {
		case '$' :	b = 16;	goto getnum;
		case '@' :	b = 8;	goto getnum;
		case '%' :	b = 2;	goto getnum; }
	if(isdigit(chr)) {
		array[i++] = chr;
	getnum:
		while(isalnum(chr = toupper(operand[++optr])))
			array[i++] = (chr < 'A') ? chr : chr - 7;
		if((b == 16) && !i)
			val = addr;
		else {
			if(!b) {
				b = 10;
				switch(array[i-1]) {
					case 'H'-7 :	b = 16;	goto deci;
					case 'T'-7 :
					case 'D'-7 :	b = 10;	goto deci;
					case 'Q'-7 :
					case 'O'-7 :	b = 8;	goto deci;
					case 'B'-7 :	b = 2;
					deci:	--i;	} }
			if(!i)
				reperr(6);
			for(ptr = array; i; --i) {
				if((chr = *ptr++ - '0') >= b)
					reperr(6);
				val = (val * b) + chr; } } }
	else if(chr=='\'') {				/* Single quote. */
		++optr;
		while(((chr=operand[optr++]) != '\'')&& chr)
			val=(val << 8)+chr;
		if(!chr) reperr(8); }
	else if(chr=='*') {					/* Program counter */
		++optr;
		val=addr; }
	else {								/* Must be a label */
		i = 0;
		while(issymbol(chr = chupper(operand[optr]))) {
			++optr;
			label[i]=chr;
			if(i < SYMB_SIZE)
				++i; }
		label[i]=0;

		if(!lookup_symbol(label)) reperr(5);
		val=value; }

	return val;
}

/* evaluate memory operand, and determine type of */
/* addressing, and size of instruction.           */
int memop(optab)
	char optab[];
{
	register char chr;
	char immed, extend, zero, indir;

	immed = extend = zero = indir = optr = length = 0; itype = -1;
	if((chr = chupper(operand[optr++])) == '#' ) {	/* immediate addressing */
		itype = optab[0];
		length = 2;
		++immed; }
	else if(chr == '>')			/* extended reference */
		++extend;
	else if(chr == '<')			/* zero page addressing */
		++zero;
	else if(chr == '[')			/* indirect addressing */
		++indir;
	else
		--optr;					/* no special modes */
/* we have options */

	value = eval();				/* get value for offset */

	if(((chr = operand[optr-1]) == ']') && (operand[optr] == ',')) {	/* [n],Y */
		if(!indir)
			reperr(7);		/* bad format */
		if(chupper(operand[optr+1]) != 'Y')
			reperr(3);		/* bad addressing mode */
		itype = optab[4];
		length = 2; }
	else if(chr == ',') {		/* indirect */
		if(immed)
			reperr(3);
		if((chr = chupper(operand[optr++])) == 'X') {	/* n,X */
			if(zero) {						/* <n,x */
				itype = optab[5];
				length = 2; }
			else if(extend) {				/* >n,x */
				itype = optab[6];
				length = 3; }
			else if(indir) {				/* [n,X] */
				if(operand[optr] != ']')
					reperr(7);	/* bad format */
				itype = optab[3];
				length = 2; }
			else {							/* default, use zero or abs */
				if((optf) && (value < 256) && (chr=optab[5])) {
					itype = chr;
					length = 2; }
				else {
					itype = optab[6];
					length = 3; } } }
		else if(chr == 'Y') {				/* n,y */
			if(zero) {						/* <n,y */
				itype = optab[8];
				length = 2; }
			else if(extend) {				/* >n,y */
				itype = optab[7];
				length = 3; }
			else if(indir)					/* [n,y] not allowed */
				reperr(3);
			else {							/* default, use zero or abs */
				if((optf) && (value < 256) && (chr=optab[8])) {
					itype = chr;
					length = 2; }
				else {
					itype = optab[7];
					length = 3; } } }
		else
			reperr(4); }
	if(itype == -1) {			/* extended or dp */
		if(indir) {
			if(operand[optr-1] != ']')
				reperr(7);		/* bad format */
			itype = optab[9];
			length = 3; }
		else if(zero) {
			itype = optab[2];
			length = 2; }
		else if(extend) {
			itype = optab[1];
			length = 3; }
		else {
			if((optf) && (value < 256) && (chr = optab[2])) {
				itype = chr;
				length = 2; }
			else {
				itype = optab[1];
				length = 3; } } }
	if(!itype)
		reperr(3);			/* does not apply */
}

/*
 * Write out title
 */
write_title()
{
	int w;
	char *ptr;

	if(pcount > 1)
		putc('\f',lst_fp);
	lcount = 1;
	fprintf(lst_fp,"DUNFIELD 6502 ASSEMBLER: ");
	ptr = title;
	for(w = 35; w < pagew; ++w)
		putc(*ptr ? *ptr++ : ' ', lst_fp);
	fprintf(lst_fp,"PAGE: %u\n\n", pcount);
	++pcount;
}

/**************************************/
/* find next element in argument list */
/**************************************/
int	nxtelem() {
	register char chr;
	while((chr=operand[optr])&&(chr != ' ')&&(chr != 9)) {
		++optr;
		if(chr==39) {
			while((chr=operand[optr])&&(chr!=39)) ++optr;
			++optr; }
		if(chr==',') return(1); }
	return(0);
}

/*
 * Too many optimization passes - report failure
 */
optfail()
{
	fprintf(lst_fp,"** Line %u - Unable to resolve: %s\n", line, label);
	++ecnt;
}
