/*
 * 6808 Cross Assembler
 *
 * ?COPY.TXT 1983-2005 Dave Dunfield
 * **See COPY.TXT**.
 */
#include <stdio.h>
#include <ctype.h>

#include "xasm.h"

/* 6808 opcode table  <text>, <type>, <opcode> */
static char opcodes[] = {
/* Type 1 - Imherent addressing */
	'A','S','L','A',	0x81, 0x48,
	'A','S','L','X',	0x81, 0x58,
	'A','S','R','A',	0x81, 0x47,
	'A','S','R','X',	0x81, 0x57,
	'C','L','R','A',	0x81, 0x4F,
	'C','L','R','X',	0x81, 0x5F,
	'C','L','R','H',	0x81, 0x8C,
	'C','O','M','A',	0x81, 0x43,
	'C','O','M','X',	0x81, 0x53,
	'D','E','C','A',	0x81, 0x4A,
	'D','E','C','X',	0x81, 0x5A,
	'I','N','C','A',	0x81, 0x4C,
	'I','N','C','X',	0x81, 0x5C,
	'L','S','L','A',	0x81, 0x48,
	'L','S','L','X',	0x81, 0x58,
	'L','S','R','A',	0x81, 0x44,
	'L','S','R','X',	0x81, 0x54,
	'N','E','G','A',	0x81, 0x40,
	'N','E','G','X',	0x81, 0x50,
	'R','O','L','A',	0x81, 0x49,
	'R','O','L','X',	0x81, 0x59,
	'R','O','R','A',	0x81, 0x46,
	'R','O','R','X',	0x81, 0x56,
	'T','A','X',		0x81, 0x97,
	'T','X','A',		0x81, 0x9F,
	'T','A','P',		0x81, 0x84,
	'T','P','A',		0x81, 0x85,
	'T','S','X',		0x81, 0x95,
	'T','X','S',		0x81, 0x94,
	'T','S','T','A',	0x81, 0x4D,
	'T','S','T','X',	0x81, 0x5D,
	'R','T','I',		0x81, 0x80,
	'R','T','S',		0x81, 0x81,
	'P','S','H','A',	0x81, 0x87,
	'P','S','H','X',	0x81, 0x89,
	'P','S','H','H',	0x81, 0x8B,
	'P','U','L','A',	0x81, 0x86,
	'P','U','L','X',	0x81, 0x88,
	'P','U','L','H',	0x81, 0x8A,
	'R','S','P',		0x81, 0x9C,
	'M','U','L',		0x81, 0x42,
	'D','I','V',		0x81, 0x52,
	'C','L','C',		0x81, 0x98,
	'C','L','I',		0x81, 0x9A,
	'S','E','C',		0x81, 0x99,
	'S','E','I',		0x81, 0x9B,
	'S','W','I',		0x81, 0x83,
	'N','O','P',		0x81, 0x9D,
	'N','S','A',		0x81, 0x62,
	'D','A','A',		0x81, 0x72,
	'S','T','O','P',	0x81, 0x8E,
	'W','A','I','T',	0x81, 0x8F,
/* Type 2 - Immed, Dir, Ext, Ix2, Ix1, Ix, Sp2, Sp1 */
	'A','D','C',		0x82, 0xA9,
	'A','D','D',		0x82, 0xAB,
	'A','N','D',		0x82, 0xA4,
	'B','I','T',		0x82, 0xA5,
	'C','M','P',		0x82, 0xA1,
	'C','P','X',		0x82, 0xA3,
	'E','O','R',		0x82, 0xA8,
	'L','D','A',		0x82, 0xA6,
	'L','D','X',		0x82, 0xAE,
	'O','R','A',		0x82, 0xAA,
	'S','B','C',		0x82, 0xA2,
	'S','U','B',		0x82, 0xA0,
/* Type 3 -  Dir, Ext, Ix2, Ix1, Ix, Sp2, Sp1 */
	'J','M','P',		0x83, 0xBC,
	'J','S','R',		0x83, 0xBD,
	'S','T','A',		0x83, 0xB7,
	'S','T','X',		0x83, 0xBF,
/* Type 4 - Direct, Ix1, Ix, Sp1 */
	'A','S','L',		0x84, 0x38,
	'A','S','R',		0x84, 0x37,
	'C','L','R',		0x84, 0x3F,
	'C','O','M',		0x84, 0x33,
	'D','E','C',		0x84, 0x3A,
	'I','N','C',		0x84, 0x3C,
	'L','S','L',		0x84, 0x38,
	'L','S','R',		0x84, 0x34,
	'N','E','G',		0x84, 0x30,
	'R','O','L',		0x84, 0x39,
	'R','O','R',		0x84, 0x36,
	'T','S','T',		0x84, 0x3D,
/* Type 5 - Imm */
	'A','I','S',		0x85, 0xA7,
	'A','I','X',		0x85, 0xAF,
/* Type 6 - Imm16, Dir */
	'L','D','H','X',	0x86, 0x44,
	'C','P','H','X',	0x86, 0x64,
/* Type 7 - Dir only */
	'S','T','H','X',	0x87, 0x35,
/* Type 8 - Relative */
	'B','C','C',		0x88, 0x24,
	'B','C','S',		0x88, 0x25,
	'B','E','Q',		0x88, 0x27,
	'B','H','C','C',	0x88, 0x28,
	'B','H','C','S',	0x88, 0x29,
	'B','H','I',		0x88, 0x22,
	'B','H','S',		0x88, 0x24,
	'B','I','H',		0x88, 0x2F,
	'B','I','L',		0x88, 0x2E,
	'B','L','O',		0x88, 0x25,
	'B','L','S',		0x88, 0x23,
	'B','M','C',		0x88, 0x2C,
	'B','M','I',		0x88, 0x2B,
	'B','M','S',		0x88, 0x2D,
	'B','N','E',		0x88, 0x26,
	'B','P','L',		0x88, 0x2A,
	'B','R','A',		0x88, 0x20,
	'B','R','N',		0x88, 0x21,
	'B','S','R',		0x88, 0xAD,
	'B','G','E',		0x88, 0x90,
	'B','L','T',		0x88, 0x91,
	'B','G','T',		0x88, 0x92,
	'B','L','E',		0x88, 0x93,
	'D','B','N','Z','A',0x88, 0x4B,
	'D','B','N','Z','X',0x88, 0x5B,
/* Type 9 - Bit & direct */
	'B','S','E','T',	0x89, 0x10,
	'B','C','L','R',	0x89, 0x11,
/* Type 10 - Bit , Direct & Relative */
	'B','R','S','E','T',0x8A, 0x00,
	'B','R','C','L','R',0x8A, 0x01,
/* Type 11 - MOV */
	'M','O','V',		0x8B, 0x00,
/* Type 12 - DBNZ */
	'D','B','N','Z',	0x8C, 0x00,
/* Type 13 - CBEQ */
	'C','B','E','Q',	0x8D, 0x00,
/* Type 14 - CBEQA and CBEQX */
	'C','B','E','Q','A',0x8E, 0x41,
	'C','B','E','Q','X',0x8E, 0x51,
/* Type 100+ - directives */
	'E','Q','U',		228,	0,
	'O','R','G',		229,	0,
	'F','C','B',		230,	0,
	'F','D','B',		231,	-1,
	'R','D','B',		231,	0,
	'F','C','C',		232,	0,
	'F','C','C','H',	232,	1,
	'F','C','C','Z',	232,	2,
	'R','M','B',		233,	0,
	'D','B',			230,	0,
	'D','W',			231,	-1,
	'D','R','W',		231,	0,
	'S','T','R',		232,	0,
	'S','T','R','H',	232,	1,
	'S','T','R','Z',	232,	2,
	'D','S',			233,	0,
	'E','N','D',		234,	0,
	'P','A','G','E',	248,	0,
	'T','I','T','L','E',249,	0,
	'S','P','A','C','E',250,	0,
	'L','I','S','T',	251,	0,
	'N','O','L','I','S','T',252,0,
	'L','I','N','E',	253,	0,
	0 };	/* end of table */

/* addressing mode offset tables */
/* -------------- Imm.  Dir.  Ext.  IX2,  IX1,  IX,   Sp2,  Sp1 */
char optab1[] = { 0x00, 0x10, 0x20, 0x30, 0x40, 0x50, 0x30, 0x40 };
char optab2[] = { 0xFF, 0x00, 0x10, 0x20, 0x30, 0x40, 0x20, 0x30 };
char optab3[] = { 0xFF, 0x00, 0xFF, 0xFF, 0x30, 0x40, 0xFF, 0x30 };
char optab4[] = { 0x00, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF };
char optab5[] = { 0x01, 0x11, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF };
char optab6[] = { 0xFF, 0x00, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF };

/* error messages */
static char *etext[] = { "?",
	"Unknown instruction",
	"Out of range",
	"Invalid addressing mode",
	"Invalid register specification",
	"Undefined symbol",
	"Invalid expression syntax",
	"Invalid argument format",
	"Improperly delimited string" } ;

/* Symbol table & free pointer */
char symtab[SYMB_POOL], *symtop;

/* misc. global variables */
char buffer[LINE_SIZE+1], label[SYMB_SIZE+1], title[50], operand[80], optr,
	itype, opcode;
unsigned char instruction[80], outrec[35];

char optf=3, symf=0, intel=0, fulf=0, quietf=0, nlist=0, casf=0, prefix;

unsigned addr, line=0, pcount=0, lcount=0, value, length, emsg, ecnt,
	ocnt=0, pagel=59, pagew=80;

FILE *asm_fp, *hex_fp, *lst_fp;

/*
 * Define a symbol in the symbol table
 */
char *create_symbol(symbol, value)
	char *symbol;
	int value;
{
	register char *ptr, *ptr1;

	ptr = ptr1 = symtop;
	while(*symbol)
		*++ptr = *symbol++;
	*ptr1 = ptr - ptr1;
	*++ptr = value >> 8;
	*++ptr = value;
	if(ptr < (symtab+sizeof(symtab))) {
		symtop = ptr + 1;
		return ptr1; }
	return 0;
}

/*
 * Lookup a symbol in the symbol table
 */
char *lookup_symbol(symbol)
	char *symbol;
{
	register int l;
	register unsigned char *ptr, *ptr1;
	char *ptr2;

	ptr = symtab;
again:
	if((ptr2 = ptr) >= symtop) {
		value = 0x8888;
		return 0; }
	ptr1 = symbol;
	l = *ptr++ & SYMMASK;
	while(l--) {
		if(*ptr1++ != *ptr++) {
			ptr += l + 2;
			goto again; } }
	if(*ptr1) {
		ptr += 2;
		goto again; }
	value = *ptr++ << 8;
	value |= *ptr;
	return ptr2;
}

/*
 * Set the value for a symbol in the symbol table
 */
set_symbol(ptr, value)
	char *ptr;
	unsigned value;
{
	ptr += (*ptr & SYMMASK);
	*++ptr = value >> 8;
	*++ptr = value;
}

/*
 * Display the symbol table (sorted)
 */
dump_symbols()
{
	unsigned char *ptr, *hptr;
	unsigned int l, h, w;

	fprintf(lst_fp, "SYMBOL TABLE:\n\n");
	w = 0;
	for(;;) {
		for(hptr = symtab; hptr < symtop; hptr += (*hptr & SYMMASK) + 3)
			if(!(*hptr & SYMSORT))
				goto found;
		putc('\n', lst_fp);
		return;
found:	for(ptr = (*hptr & SYMMASK) + hptr + 3; ptr < symtop; ptr += (*ptr & SYMMASK) + 3) {
			if(*ptr & SYMSORT)
				continue;
			if(compsym(ptr, hptr) < 0)
				hptr = ptr; }
		*(ptr = hptr) |= SYMSORT;
		h = l = *ptr++ & SYMMASK;
		if((w + l + 7) >= pagew) {			/* Overrun page length */
			putc('\n', lst_fp);
			w = 0; }
		if(w) {								/* Not a start of line - separate */
			fprintf(lst_fp, "   ");
			w += 3; }
		while(l--)
			putc(*ptr++, lst_fp);
		w += (l = (h > 8) ? 24 : 8) + 5;	/* Calculate extended length */
		while(h++ < l)
			putc(' ', lst_fp);
		l = *ptr++ << 8;
		l |= *ptr++;
		fprintf(lst_fp, "-%04x", l); }
}

/*
 * Compare two symbol table entries
 */
compsym(sym1, sym2)
	char *sym1;
	char *sym2;
{
	int l, l1, l2;
	l1 = *sym1++ & SYMMASK;
	l2 = *sym2++ & SYMMASK;
	l = (l1 < l2) ? l1 : l2;
	do {
		if(*sym1 > *sym2)
			return 1;
		if(*sym1++ < *sym2++)
			return -1; }
	while(--l);
	if(l1 > l2)
		return 1;
	if(l1 < l2)
		return -1;
	return 0;
}

/************************************/
/* get a character from the operand */
/************************************/
char getchr()
{
	register char chr;

	if(chr=operand[optr]) ++optr;
	return(chupper(chr));
}

/*
 * Convert character to upper case if NOT case sensitive
 */
chupper(c)
	char c;
{
	return casf ? c : ((c >= 'a') && (c <= 'z')) ? c - ('a'-'A') : c;
}

/*
 * Open a filename with the appriopriate extension &
 * report an error if not possible
 */
FILE *open_file(filename, extension, options)
	char *filename, *extension, *options;
{
	char buffer[100], *ptr, *dot;
	FILE *fp;

	dot = 0;

	for(ptr = buffer; *ptr = *filename; ++ptr) {	/* Copy filename */
		if(*filename == '.')
			dot = filename;
		else if(*filename == '\\')
			dot = 0;
		++filename; }

	if(!dot) {									/* No extension supplied */
		*ptr++ = '.';
		do
			*ptr++ = *extension;
		while(*extension++); }
	else
		*dot = 0;

	if(!(fp = fopen(buffer, options))) {
		fprintf(stderr,"Unable to access: '%s'\n", buffer);
		exit(-1); }

	return fp;
}

/*
 * Main program
 */
main(argc, argv)
	unsigned argc;
	char *argv[];
{
	unsigned i, temp, daddr, vbyt;
	int stemp;
	char chr, cflg, pflg, *ptr, *lfile, *cfile;

	if(argc < 2)
		abort("\nUse: asm08 <filename> [-cfiqst c=file l=file o=n p=length w=width]\n\n?COPY.TXT 1983-2005 Dave Dunfield\n**See COPY.TXT**.\n");

	pflg = 0;

/* parse for command line options. */
	lfile = cfile = argv[1];
	for(i=2; i < argc; ++i) {
		if(*(ptr = argv[i]) == '-') {		/* Enable switch */
			while(*++ptr) switch(toupper(*ptr)) {
				case 'C' : casf = -1;		break;
				case 'F' : fulf = -1;		break;
				case 'I' : intel = -1;		break;
				case 'Q' : quietf = -1;		break;
				case 'S' : symf = -1;		break;
				case 'T' : lfile = 0;		break;
				default: goto badopt; }
			continue; }
		if(*++ptr == '=') switch(toupper(*(ptr++ - 1))) {
			case 'L' : lfile = ptr;			continue;
			case 'C' : cfile = ptr;			continue;
			case 'O' : optf=atoi(ptr);		continue;
			case 'P' : pagel=atoi(ptr)-1;	continue;
			case 'W' : pagew=atoi(ptr);		continue; }
	badopt:
		fprintf(stderr,"Invalid option: %s\n", argv[i]);
		exit(-1); }

	asm_fp = open_file(argv[1], "ASM", "r");
	hex_fp = open_file(cfile, "HEX", "w");
	lst_fp = (lfile) ? open_file(lfile, "LST", "w") : stdout;
	strncpy(title,argv[1],sizeof(title)-1);

/* first pass - build symbol table */
	symtop = symtab;
	do {
		addr = cflg = 0; line = 1;
		if(!quietf) if(pflg) fprintf(stderr,"Opt... ");
			else	fprintf(stderr,"First pass... ");
		while(readline()) {
			if(!optr) {
				locins(instruction);
				if(label[0]) {				/* label exists */
					if(pflg) {				/* optomization pass */
						ptr = lookup_symbol(label);
						if(itype==100) {	/* EQU statement */
							optr = 0; vbyt = value;
							if((temp=eval()) != vbyt) {
								if(pflg >= optf) optfail();
								set_symbol(ptr, temp);
								cflg=1; } }
						else if(value != addr) {
							if(pflg >= optf) optfail();
							set_symbol(ptr, addr);
							cflg=1; } }
					else {							/* original pass-1 */
						if(lookup_symbol(label)) {	/* duplicate label */
							fprintf(lst_fp,"** Line %u - Duplicate symbol: '%s'\n",line,label);
							++ecnt; }
						else {
							cflg=1;			/* indicate symbol table change */
							if(!(ptr = create_symbol(label, addr))) {
								xabort("symbol");
								fprintf(lst_fp,"** Line %u - Symbol table overflow\n", line);
								++ecnt;
								break; } } } }
				emsg = length = optr = 0;
				switch(itype) {
					case 1 :		/* inherent addressing */
						length = 1;
						break;
					case 2 :		/* Imm, Dir, Ext, Ix2, Ix1, Ix */
						memop(optab1);
						break;
					case 3 :		/* Dir, Ext, Ix2, Ix1, Ix */
						memop(optab2);
						break;
					case 4 :		/* Dir, Ix1, Ix */
						memop(optab3);
						break;
					case 5 :		/* Imm only */
						memop(optab4);
						break;
					case 6 :		/* Imm16 + Dir */
						memop(optab5);
						break;
					case 7 :		/* Dir only */
						memop(optab6);
						break;
					case 8 :		/* Relative */
					case 9 :		/* Bit & Direct */
						length = 2;
						break;
					case 10 :		/* Bit & Direct & Relative */
					case 14 :		/* CBEQA & CBEQX */
						length = 3;
						break;
					case 11 :		/* MOV */
						do_mov();
						break;
					case 12 :		/* DBNZ */
						do_dbnz();
						break;
					case 13 :		/* CBEQ */
						do_cbeq();
						break;
					case 100 :			/* EQU directive */
						if(!pflg) {
							set_symbol(ptr, eval());
						err1st:
							if(emsg && !optf) {
								fprintf(lst_fp,"** Line %u - %s\n",line,etext[emsg]);
								++ecnt; } }
						break;
					case 101 :			/* ORG directive */
						addr=eval();
						goto err1st;
					case 102 :			/* FCB statement */
						length=1;
						while(nxtelem())
							++length;
						break;
					case 103 :			/* FDB statement */
						length=2;
						while(nxtelem()) length += 2;
						break;
					case 104 :			/* FCC statements */
						chr=operand[0];
						for(i=1;(operand[i])&&(chr!=operand[i]); ++i)
							++length;
						if(opcode == 2) ++length;
						break;
					case 105 :			/* RMB statement */
						addr = addr + eval();
						goto err1st;
					case 106 :			/* END statement */
						goto end1;
					case 125 :			/* LINE directive */
						line = eval(); } }
			else
				length = 0;
			addr += length;
			++line; /* += (itype < 120); */ }
end1:	++pflg;
		rewind(asm_fp); }
	while((optf)&&(cflg)&&(!ecnt));

/* second pass - generate object code */

	if(!quietf)
		fprintf(stderr,"Second pass... ");
	addr = emsg = 0; line = pcount = 1; lcount = 9999;
	while(readline()) {
		daddr = addr;
		if(!optr) {
			if(!locins(instruction))
				reperr(1);			/* unknown opcode */
			length = temp = optr = prefix = 0;
			switch(itype) {
				case 1 :		/* inherent addressing */
					instruction[length++] = opcode;
					break;
				case 2 :		/* Imm, Dir, Ext, Ix2, Ix1, Ix */
					memop(optab1);
					goto wrinst;
				case 3 :		/* Dir, Ext, Ix2, Ix1, Ix */
					memop(optab2);
					goto wrinst;
				case 4 :		/* Dir, Ix1, Ix */
					memop(optab3);
					goto wrinst;
				case 5 :		/* Imm only */
					memop(optab4);
					goto wrinst;
				case 6 :		/* Imm16 + dir */
					memop(optab5);
					goto wrinst;
				case 7 :		/* Dir only */
					memop(optab6);
wrinst:				if(prefix) {
						instruction[0] = prefix;
						instruction[1] = opcode + itype;
						if(length > 3)
							instruction[2] = value >> 8;
						if(length > 2)
							instruction[length-1] = value & 255;
						break; }
					instruction[0] = opcode + itype;
					if(length > 2)
						instruction[1] = value >> 8;
					if(length > 1)
						instruction[length-1] = value & 255;
					break;
				case 8 :			/* branches */
					instruction[length++] = opcode;
gorel:				stemp = (eval() - addr) - (length + 1);
					if((stemp > 127) || (stemp < -128))
						reperr(2);				/* out or range */
					instruction[length++] = stemp;
					break;
				case 9 :			/* Bit + Direct */
				case 10 :			/* Bit + Direct + Releative */
					temp = eval();
					if(temp > 7)
						reperr(2);				/* Out of range */
					instruction[length++] = opcode + (temp * 2);
					if(operand[optr-1] != ',')
						reperr(6);				/* Invalid arg format */
					if(operand[optr] == '<')
						++optr;
					temp = eval();
					if(temp > 255)
						reperr(3);				/* Illegal addressing mode */
					instruction[length++] = temp;
					if(itype == 10) {
						if(operand[optr-1] != ',')
							reperr(6);				/* Invalid arg format */
						goto gorel; }
					break;
				case 11 :			/* MOV */
					do_mov();
					break;
				case 12 :			/* DBNZ */
					do_dbnz();
					break;
				case 13 :			/* CBEQ */
					do_cbeq();
					break;
				case 14 :			/* CBEQA and CBEQX */
					do_cbeq1();
					break;
				case 100 :			/* equate statement */
					daddr=eval();	/* generate errors if any */
					break;
				case 101 :			/* ORG statement */
					if(ocnt) wrrec();
					daddr=addr=eval();
					break;
				case 102 :			/* FCB statement */
					do {
						instruction[length++] = eval(); }
					while(operand[optr-1] == ',');
					break;
				case 103 :			/* FDB statement */
					do {
						temp = eval();
						if(opcode) {
							instruction[length++] = temp >> 8;
							instruction[length++] = temp; }
						else {
							instruction[length++] = temp;
							instruction[length++] = temp >> 8; } }
					while(operand[optr-1] == ',');
					break;
				case 104 :			/* FCC statements */
					chr=operand[0];
					for(i=1;((operand[i])&&(chr!=operand[i])); ++i)
						instruction[length++] = operand[i];
					if(!operand[i])	reperr(8);
					if(opcode == 1)
						instruction[length-1] = instruction[length-1] | 0x80;
					if(opcode == 2)
						instruction[length++]=0;
					break;
				case 105 :			/* RMB statement */
					if(ocnt) wrrec();
					addr += eval();
					break;
				case 106 :			/* END statement */
					goto end2;
				case 120 :			/* PAGE statement */
					lcount=9999;
					break;
				case 121 :			/* TITLE directive */
					strncpy(title, operand, sizeof(title)-1);
					break;
				case 122 :			/* SPACE directive */
					fprintf(lst_fp,"\n");
					++lcount;
					break;
				case 123 :			/* LIST directive */
					if(nlist) --nlist;
					break;
				case 124 :			/* NOLIST directive */
					++nlist;
					break;
				case 125 :			/* LINE directive */
					line = eval(); } }
		else
			length=itype=0;

/* generate listing */
	if(((itype<120) && fulf && !nlist) || emsg) {
		if(++lcount >= pagel)
			write_title();
		fprintf(lst_fp,"%04x ",daddr);
			for(i=0; i < 6; ++i) {
				if(i<length)
					fprintf(lst_fp," %02x",instruction[i]);
				else
					fprintf(lst_fp,"   "); }
			fprintf(lst_fp," %c%5u  %s\n",(length <= 6) ? ' ' : '+', line,buffer); }
		if(emsg) {
			fprintf(lst_fp,"  ** ERROR ** - %u - %s\n", emsg, etext[emsg]);
			++ecnt; ++lcount;
			emsg=0; }
		++line; /* += (itype<120); */
/* write code to output record */
	for(i=0; i<length; ++i) {
		if(!ocnt) {			/* first byte of record */
			outrec[ocnt++]=addr>>8;
			outrec[ocnt++]=addr; }
		++addr;
		outrec[ocnt++]=instruction[i];
		if(ocnt>33) wrrec(); } }	/* record is full, write it */

end2:
	if(ocnt) wrrec();
	if(intel) fprintf(hex_fp,":00000001FF\n");
	else fprintf(hex_fp,"S9030000FC\n");

	if(ecnt) fprintf(lst_fp,"\n %u error(s) occurred in this assembly.\n",ecnt);
	if(!quietf) fprintf(stderr,"%u error(s).\n",ecnt);

/* display the symbol table */
	if(symf) {
		write_title();
		dump_symbols(); }

	fclose(asm_fp);
	fclose(hex_fp);
	fclose(lst_fp);

	return ecnt ? -2 : 0;
}

/***********************************************/
/* lookup instruction in the instruction table */
/***********************************************/
int locins(inst)
	char inst[];
{
	register char *ptr, *ptr1;

	ptr = opcodes;
	do {
		ptr1 = inst;
		while(*ptr == *ptr1) {
			++ptr;
			++ptr1; }
		if((*ptr < 0) && !*ptr1) {
			itype = *ptr++ & 127;
			opcode = *ptr++;
			return 1; }
		while(*ptr > 0)
			++ptr; }
	while(*(ptr += 2));

	itype = 119;
	return(inst[0] == 0);
}

/*************************************************/
/* read a line from input file, and break it up. */
/*************************************************/
int readline()
{
	register int i;
	register char *ptr;

	if(!fgets(ptr = buffer, LINE_SIZE, asm_fp))
		return 0;

	i = 0;
	while(issymbol(*ptr)) {
		label[i] = chupper(*ptr++);
		if(i < SYMB_SIZE)
			++i; }
	label[i]=0;
	if(*ptr == ':')
		++ptr;
	while(isspace(*ptr))
		++ptr;
	if(((*ptr != '*') && (*ptr != ';')) || i) {
		i = 0;
		while(*ptr && !isspace(*ptr))
			instruction[i++] = toupper(*ptr++);
		instruction[i]=0;
		while(isspace(*ptr))
			++ptr;
		optr = i = 0;
		while(*ptr && (i < 79))
			operand[i++] = *ptr++;
		operand[i] = 0;
		return 1; }

	return optr = 1;
}

/***************************************/
/* test for valid terminator character */
/***************************************/
int isterm(chr)
	register char chr;
{
	switch(chr) {
		case 0 :
		case ' ' :
		case '\t':
		case ',' :
		case ')' :
			return 1; }
	return 0;
}

/*************************************/
/* Test for a valid symbol character */
/*************************************/
int issymbol(c)
	char c;
{
	switch(c) {			/* Special allowed characters */
		case '_' :
		case '?' :
		case '!' :
		case '.' :
			return 1; }

	return isalnum(c);
}

/* report an error */
reperr(errn)
	register int errn;
{
	if(!emsg) emsg=errn;
}

/*******************************/
/* write record to output file */
/*******************************/
wrrec()
{
	register unsigned i, chk, chr;

	xhex(ocnt-2);
	if(intel) {					/* intel hex format */
		chk = outrec[0] + outrec[1] + ocnt - 2;
		fprintf(hex_fp,":%02x%02x%02x00", ocnt-2, outrec[0], outrec[1]);
		for(i=2; i<ocnt; ++i) {
			fprintf(hex_fp,"%02x", chr = outrec[i]);
			chk += chr; }
		fprintf(hex_fp,"%02x\n", 255 & (0-chk)); }
	else {						/* motorola hex format */
		chk = ocnt + 1;
		fprintf(hex_fp,"S1%02x", ocnt + 1);
		for(i=0; i<ocnt; ++i) {
			fprintf(hex_fp,"%02x", chr = outrec[i]);
			chk += chr; }
		fprintf(hex_fp,"%02x\n", 255 & ~chk); }
	ocnt = 0;
}

/*
 * Evaluate an expression.
 */
int eval()
{	register char c;
	unsigned result;

	result=getval();
	while(!isterm(c = getchr())) switch(c) {
		case '+' : result += getval();	break;
		case '-' : result -= getval();	break;
		case '*' : result *= getval();	break;
		case '/' : result /= getval();	break;
		case '\\': result %= getval();	break;
		case '&' : result &= getval();	break;
		case '|' : result |= getval();	break;
		case '^' : result ^= getval();	break;
		case '<' : result <<= getval();	break;
		case '>' : result >>= getval();	break;
		default: reperr(6); }

	return result;
}

/*
 * Get a single value (number or symbol)
 */
int getval()
{
	unsigned i, b, val;
	char chr, array[25], *ptr;

	switch(chr = operand[optr++]) {
		case '-' :				/* Negated value */
			return 0 - getval();
		case '~' :				/* Complemented value */
			return ~getval();
		case '=' :				/* Swap high and low bytes */
			i=getval();
			return (i<<8)+(i>>8);
		case '(' :				/* Nested expression */
			val = eval();
			if(operand[optr-1] != ')')
				reperr(6);
			return val; }
	--optr;

	val=b=i=0;
	switch(chr) {
		case '$' :	b = 16;	goto getnum;
		case '@' :	b = 8;	goto getnum;
		case '%' :	b = 2;	goto getnum; }
	if(isdigit(chr)) {
		array[i++] = chr;
	getnum:
		while(isalnum(chr = toupper(operand[++optr])))
			array[i++] = (chr < 'A') ? chr : chr - 7;
		if((b == 16) && !i)
			val = addr;
		else {
			if(!b) {
				b = 10;
				switch(array[i-1]) {
					case 'H'-7 :	b = 16;	goto deci;
					case 'T'-7 :
					case 'D'-7 :	b = 10;	goto deci;
					case 'Q'-7 :
					case 'O'-7 :	b = 8;	goto deci;
					case 'B'-7 :	b = 2;
					deci:	--i;	} }
			if(!i)
				reperr(6);
			for(ptr = array; i; --i) {
				if((chr = *ptr++ - '0') >= b)
					reperr(6);
				val = (val * b) + chr; } } }
	else if(chr=='\'') {				/* Single quote. */
		++optr;
		while(((chr=operand[optr++]) != '\'')&& chr)
			val=(val << 8)+chr;
		if(!chr) reperr(8); }
	else if(chr=='*') {					/* Program counter */
		++optr;
		val=addr; }
	else {								/* Must be a label */
		i = 0;
		while(issymbol(chr = chupper(operand[optr]))) {
			++optr;
			label[i]=chr;
			if(i < SYMB_SIZE)
				++i; }
		label[i]=0;

		if(!lookup_symbol(label)) reperr(5);
		val=value; }

	return val;
}

/* evaluate memory operand, and determine type of */
/* addressing, and size of instruction.           */
int memop(optab)
	char optab[];
{
	register char chr;

	if((chr = chupper(operand[optr++])) == '#' ) {	/* immediate addressing */
		value = eval();
		length = (itype = optab[0]) ? 3 : 2; }
	else if(chr == '>') {			/* extended reference */
		value = eval();
		itype = optab[2];
		length = 3; }
	else if(chr == '<') {			/* Direct addressing */
		value = eval();
		itype = optab[1];
		length = 2; }
	else if(chr == ',') {			/* Indexed (no offset) 'IX' */
		if(chupper(operand[optr]) != 'X')
			reperr(3);
		value = 0;
		itype = optab[5];
		length = 1; }
	else {							/* Not simple case */
		--optr;
		value = eval();					/* get value for offset */
		if(operand[optr-1] == ',') {	/* indexed */
			if(chupper(operand[optr]) == 'X') {
				if((optf) && (value < 256)) {		/* IX1 */
					itype = optab[4];
					length = 2; }
				else {								/* IX2 */
					itype = optab[3];
					length = 3; } }
			else {
				if((chupper(operand[optr]) != 'S') || (chupper(operand[optr+1]) != 'P'))
					reperr(3);
				prefix = 0x9E;
				if((optf) && (value < 256)) {		/* SP1 */
					itype = optab[7];
					length = 3; }
				else {
					itype = optab[6];
					length = 4; } } }
		else {		/* extended or direct addressing */
			if((optf) && (value < 256)) {		/* Direct */
				itype = optab[1];
				length = 2; }
			else {								/* extended */
				itype = optab[2];
				length = 3; } } }
	if(itype == -1)
		reperr(3);
}

/*
 * Write out title
 */
write_title()
{
	int w;
	char *ptr;

	if(pcount > 1)
		putc('\f',lst_fp);
	lcount = 1;
	fprintf(lst_fp,"DUNFIELD 6808 ASSEMBLER: ");
	ptr = title;
	for(w = 35; w < pagew; ++w)
		putc(*ptr ? *ptr++ : ' ', lst_fp);
	fprintf(lst_fp,"PAGE: %u\n\n", pcount);
	++pcount;
}

/**************************************/
/* find next element in argument list */
/**************************************/
int	nxtelem()
{
	register char chr;
	while((chr=operand[optr])&&(chr != ' ')&&(chr != 9)) {
		++optr;
		if(chr==39) {
			while((chr=operand[optr])&&(chr!=39)) ++optr;
			++optr; }
		if(chr==',') return(1); }
	return(0);
}

/*
 * Too many optimization passes - report failure
 */
optfail()
{
	fprintf(lst_fp,"** Line %u - Unable to resolve: %s\n", line, label);
	++ecnt;
}

/*
 * Process a MOV instruction
 */
do_mov()
{
	if(operand[0] == '#') {							/* #,D */
		instruction[length++] = 0x6E;
		optr = 1;
		instruction[length++] = eval();
		if(operand[optr-1] != ',')
			reperr(6);
		instruction[length++] = eval();
		return; }

	if((chupper(operand[0]) == 'X') && (operand[1] == '+')) {	/* X+.D */
		instruction[length++] = 0x7E;
		if(operand[2] != ',')
			reperr(6);
		optr = 3;
		instruction[length++] = eval();
		return; }

	value = eval();		/* Get first value */
	if(operand[optr-1] != ',')
		reperr(6);

	if((chupper(operand[optr]) == 'X') && (operand[optr+1] == '+')) {	/* D,X+ */
		instruction[length++] = 0x5E;
		instruction[length++] = value;
		return; }

	instruction[length++] = 0x4E;		/* D,D */
	instruction[length++] = value;
	instruction[length++] = eval();
}

/*
 * Process a DBNZ instruction
 */
do_dbnz()
{
	int stemp;
	char a, b;

	if((chupper(operand[0]) == 'X') && (operand[1] == ',')) {	/* X,dir */
		optr = 2;
		instruction[length++] = 0x7B; }
	else {
		value = eval();
		if(operand[optr-1] != ',')
			reperr(6);
		a = chupper(operand[optr]);
		b = chupper(operand[optr+1]);
		if((a == 'X') && (b == ',')) {		/*  nn,X */
			optr += 2;
			instruction[length++] = 0x6B; }
		else if((a == 'S') && (b == 'P') && (operand[optr+2] == ',')) {
			optr += 3;
			instruction[length++] = 0x9E;
			instruction[length++] = 0x6B; }
		else
			instruction[length++] = 0x3B;
		instruction[length++] = value; }

	if(operand[optr-1] != ',')
		reperr(6);

	stemp = (eval() - addr) - (length + 1);
	if((stemp > 127) || (stemp < -128))
		reperr(2);				/* out or range */
	instruction[length++] = stemp;
}

/*
 * Process a CBEQ instruction
 */
do_cbeq()
{
	int stemp;
	char a, b;

	if((chupper(operand[0]) == 'X') && (operand[1] == '+')) {	/* X+,rel */
		optr = 3;
		instruction[length++] = 0x71; }
	else {
		value = eval();
		if(operand[optr-1] != ',')
			reperr(6);
		a = chupper(operand[optr]);
		b = chupper(operand[optr+1]);
		if((a == 'X') && (b == '+')) {				/* n,X,rel */
			optr += 3;
			instruction[length++] = 0x61; }
		else if((a == 'S') && (b == 'P') && (operand[optr+2] == ',')) {
			optr += 3;
			instruction[length++] = 0x9E;
			instruction[length++] = 0x61; }
		else
			instruction[length++] = 0x31;
		instruction[length++] = value; }

	if(operand[optr-1] != ',')
		reperr(6);

	stemp = (eval() - addr) - (length + 1);
	if((stemp > 127) || (stemp < -128))
		reperr(2);				/* out or range */
	instruction[length++] = stemp;
}

/*
 * Process CBEQA and CBEQX instructions
 */
do_cbeq1()
{
	int stemp;

	instruction[length++] = opcode;
	if(operand[optr++] != '#')
		reperr(3);
	instruction[length++] = eval();

	if(operand[optr-1] != ',')
		reperr(6);
	
	stemp = (eval() - addr) - (length + 1);
	if((stemp > 127) || (stemp < -128))
		reperr(2);				/* out or range */
	instruction[length++] = stemp;
}
