/*
 * 68HC12 Cross Assembler
 *
 * ?COPY.TXT 1983-2005 Dave Dunfield
 * **See COPY.TXT**.
 */
#include <stdio.h>
#include <ctype.h>

#include "xasm.h"

/* 68HC12 opcode table
 *
 * ITYPE High nibble:
 *		8 = Standard opcode
 *		C = 18 prebyte
 * ITYPE Low nibble:
 *		1 = Inherent addressing (no operands)
 *		2 = Full addressing (IMM, DIR, IDX, EXT) (8  bit data)
 *		3 = Full addressing (IMM, DIR, IDX, EXT) (16 bit data)
 *		4 = MOV operand
 *		5 = 8 bit Relative branch
 *		6 = 16 bit relative branch
 *		7 = Bit addressing (BCLR)
 *		8 = Bit test & branch (BRSET)
 *		9 = REG, branch (DBEQ)
 *		A = EXG registers
 *		B = CALL
 *		C = TRAP
 */
static char opcodes[] = {
	'A','B','A',		0xC1,0x06,0x00,
	'A','B','X',		0x81,0x1A,0xE5,
	'A','B','Y',		0x81,0x19,0xED,
	'A','D','C','A',	0x82,0x89,0x00,
	'A','D','C','B',	0x82,0xC9,0x00,
	'A','D','D','A',	0x82,0x8B,0x00,
	'A','D','D','B',	0x82,0xCB,0x00,
	'A','D','D','D',	0x83,0xC3,0x00,
	'A','N','D','A',	0x82,0x84,0x00,
	'A','N','D','B',	0x82,0xC4,0x00,
	'A','N','D','C','C',0x82,0x10,0x01,
	'A','S','L',		0x82,0x68,0x04,
	'A','S','L','A',	0x81,0x48,0x00,
	'A','S','L','B',	0x81,0x58,0x00,
	'A','S','L','D',	0x81,0x59,0x00,
	'A','S','R',		0x82,0x67,0x04,
	'A','S','R','A',	0x81,0x47,0x00,
	'A','S','R','B',	0x81,0x57,0x00,
	'B','C','C',		0x85,0x24,0x00,
	'B','C','L','R',	0x87,0x0D,0x0A,
	'B','C','S',		0x85,0x25,0x00,
	'B','E','Q',		0x85,0x27,0x00,
	'B','G','E',		0x85,0x2C,0x00,
	'B','G','N','D',	0x81,0x00,0x00,
	'B','G','T',		0x85,0x2E,0x00,
	'B','H','I',		0x85,0x22,0x00,
	'B','H','S',		0x85,0x24,0x00,
	'B','I','T','A',	0x82,0x85,0x00,
	'B','I','T','B',	0x82,0xC5,0x00,
	'B','L','E',		0x85,0x2F,0x00,
	'B','L','O',		0x85,0x25,0x00,
	'B','L','S',		0x85,0x23,0x00,
	'B','L','T',		0x85,0x2D,0x00,
	'B','M','I',		0x85,0x2B,0x00,
	'B','N','E',		0x85,0x26,0x00,
	'B','P','L',		0x85,0x2A,0x00,
	'B','R','A',		0x85,0x20,0x00,
	'B','R','C','L','R',0x88,0x0F,0x0A,
	'B','R','N',		0x85,0x21,0x00,
	'B','R','S','E','T',0x88,0x0E,0x0A,
	'B','S','E','T',	0x87,0x0C,0x0A,
	'B','S','R',		0x85,0x07,0x00,
	'B','V','C',		0x85,0x28,0x00,
	'B','V','S',		0x85,0x29,0x00,
	'C','A','L','L',	0x8B,0x4A,0x05,
	'C','B','A',		0xC1,0x17,0x00,
	'C','L','C',		0x81,0x10,0xFE,
	'C','L','I',		0x81,0x10,0xEF,
	'C','L','R',		0x82,0x69,0x04,
	'C','L','R','A',	0x81,0x87,0x00,
	'C','L','R','B',	0x81,0xC7,0x00,
	'C','L','V',		0x81,0x10,0xFD,
	'C','M','P','A',	0x82,0x81,0x00,
	'C','M','P','B',	0x82,0xC1,0x00,
	'C','O','M',		0x82,0x61,0x04,
	'C','O','M','A',	0x81,0x41,0x00,
	'C','O','M','B',	0x81,0x51,0x00,
	'C','P','D',		0x83,0x8C,0x00,
	'C','P','S',		0x83,0x8F,0x00,
	'C','P','X',		0x83,0x8E,0x00,
	'C','P','Y',		0x83,0x8D,0x00,
	'D','A','A',		0xC1,0x07,0x00,
	'D','B','E','Q',	0x89,0x04,0x00,
	'D','B','N','E',	0x89,0x04,0x20,
	'D','E','C',		0x82,0x63,0x04,
	'D','E','C','A',	0x81,0x43,0x00,
	'D','E','C','B',	0x81,0x53,0x00,
	'D','E','S',		0x81,0x1B,0x9F,
	'D','E','X',		0x81,0x09,0x00,
	'D','E','Y',		0x81,0x03,0x00,
	'E','D','I','V',	0x81,0x11,0x00,
	'E','D','I','V','S',0xC1,0x14,0x00,
	'E','M','A','C','S',0xC2,0x12,0x03,
	'E','M','A','X','D',0xC2,0x1A,0x02,
	'E','M','A','X','M',0xC2,0x1E,0x02,
	'E','M','I','N','D',0xC2,0x1B,0x02,
	'E','M','I','N','M',0xC2,0x1F,0x02,
	'E','M','U','L',	0x81,0x13,0x00,
	'E','M','U','L','S',0xC1,0x13,0x00,
	'E','O','R','A',	0x82,0x88,0x00,
	'E','O','R','B',	0x82,0xC8,0x00,
	'E','T','B','L',	0xC2,0x3F,0x02,
	'E','X','G',		0x8A,0xB7,0x80,
	'F','D','I','V',	0xC1,0x11,0x00,
	'I','B','E','Q',	0x89,0x04,0x80,
	'I','B','N','E',	0x89,0x04,0xA0,
	'I','D','I','V',	0xC1,0x10,0x00,
	'I','D','I','V','S',0xC1,0x15,0x00,
	'I','N','C',		0x82,0x62,0x04,
	'I','N','C','A',	0x81,0x42,0x00,
	'I','N','C','B',	0x81,0x52,0x00,
	'I','N','S',		0x81,0x1B,0x81,
	'I','N','X',		0x81,0x08,0x00,
	'I','N','Y',		0x81,0x02,0x00,
	'J','M','P',		0x82,0x05,0x06,
	'J','S','R',		0x82,0x15,0x07,
	'L','B','C','C',	0xC6,0x24,0x00,
	'L','B','C','S',	0xC6,0x25,0x00,
	'L','B','E','Q',	0xC6,0x27,0x00,
	'L','B','G','E',	0xC6,0x2C,0x00,
	'L','B','G','T',	0xC6,0x2E,0x00,
	'L','B','H','I',	0xC6,0x22,0x00,
	'L','B','H','S',	0xC6,0x24,0x00,
	'L','B','L','E',	0xC6,0x2F,0x00,
	'L','B','L','O',	0xC6,0x25,0x00,
	'L','B','L','S',	0xC6,0x23,0x00,
	'L','B','L','T',	0xC6,0x2D,0x00,
	'L','B','M','I',	0xC6,0x2B,0x00,
	'L','B','N','E',	0xC6,0x26,0x00,
	'L','B','P','L',	0xC6,0x2A,0x00,
	'L','B','R','A',	0xC6,0x20,0x00,
	'L','B','R','N',	0xC6,0x21,0x00,
	'L','B','V','C',	0xC6,0x28,0x00,
	'L','B','V','S',	0xC6,0x29,0x00,
	'L','D','A','A',	0x82,0x86,0x00,
	'L','D','A','B',	0x82,0xC6,0x00,
	'L','D','D',		0x83,0xCC,0x00,
	'L','D','S',		0x83,0xCF,0x00,
	'L','D','X',		0x83,0xCE,0x00,
	'L','D','Y',		0x83,0xCD,0x00,
	'L','E','A','S',	0x82,0x1B,0x02,
	'L','E','A','X',	0x82,0x1A,0x02,
	'L','E','A','Y',	0x82,0x19,0x02,
	'L','S','L',		0x82,0x68,0x04,
	'L','S','L','A',	0x81,0x48,0x00,
	'L','S','L','B',	0x81,0x58,0x00,
	'L','S','L','D',	0x81,0x59,0x00,
	'L','S','R',		0x82,0x64,0x04,
	'L','S','R','A',	0x81,0x44,0x00,
	'L','S','R','B',	0x81,0x54,0x00,
	'L','S','R','D',	0x81,0x49,0x00,
	'M','A','X','A',	0xC2,0x18,0x02,
	'M','A','X','M',	0xC2,0x1C,0x02,
	'M','E','M',		0x81,0x01,0x00,
	'M','I','N','A',	0xC2,0x19,0x02,
	'M','I','N','M',	0xC2,0x1D,0x02,
	'M','O','V','B',	0xC4,0x08,0x00,
	'M','O','V','W',	0xC4,0x00,0x01,
	'M','U','L',		0x81,0x12,0x00,
	'N','E','G',		0x82,0x60,0x04,
	'N','E','G','A',	0x81,0x40,0x00,
	'N','E','G','B',	0x81,0x50,0x00,
	'N','O','P',		0x81,0xA7,0x00,
	'O','R','A','A',	0x82,0x8A,0x00,
	'O','R','A','B',	0x82,0xCA,0x00,
	'O','R','C','C',	0x82,0x14,0x01,
	'P','S','H','A',	0x81,0x36,0x00,
	'P','S','H','B',	0x81,0x37,0x00,
	'P','S','H','C',	0x81,0x39,0x00,
	'P','S','H','D',	0x81,0x3B,0x00,
	'P','S','H','X',	0x81,0x34,0x00,
	'P','S','H','Y',	0x81,0x35,0x00,
	'P','U','L','A',	0x81,0x32,0x00,
	'P','U','L','B',	0x81,0x33,0x00,
	'P','U','L','C',	0x81,0x38,0x00,
	'P','U','L','D',	0x81,0x3A,0x00,
	'P','U','L','X',	0x81,0x30,0x00,
	'P','U','L','Y',	0x81,0x31,0x00,
	'R','E','V',		0xC1,0x3A,0x00,
	'R','E','V','W',	0xC1,0x3B,0x00,
	'R','O','L',		0x82,0x65,0x04,
	'R','O','L','A',	0x81,0x45,0x00,
	'R','O','L','B',	0x81,0x55,0x00,
	'R','O','R',		0x82,0x66,0x04,
	'R','O','R','A',	0x81,0x46,0x00,
	'R','O','R','B',	0x81,0x56,0x00,
	'R','T','C',		0x81,0x0A,0x00,
	'R','T','I',		0x81,0x0B,0x00,
	'R','T','S',		0x81,0x3D,0x00,
	'S','B','A',		0xC1,0x16,0x00,
	'S','B','C','A',	0x82,0x82,0x00,
	'S','B','C','B',	0x82,0xC2,0x00,
	'S','E','C',		0x81,0x14,0x01,
	'S','E','I',		0x81,0x14,0x10,
	'S','E','V',		0x81,0x14,0x02,
	'S','E','X',		0x8A,0xB7,0x00,
	'S','T','A','A',	0x82,0x5A,0x08,
	'S','T','A','B',	0x82,0x5B,0x08,
	'S','T','D',		0x82,0x5C,0x08,
	'S','T','O','P',	0xC1,0x3E,0x00,
	'S','T','S',		0x82,0x5F,0x08,
	'S','T','X',		0x82,0x5E,0x08,
	'S','T','Y',		0x82,0x5D,0x08,
	'S','U','B','A',	0x82,0x80,0x00,
	'S','U','B','B',	0x82,0xC0,0x00,
	'S','U','B','D',	0x83,0x83,0x00,
	'S','W','I',		0x81,0x3F,0x00,
	'T','A','B',		0xC1,0x0E,0x00,
	'T','A','P',		0x81,0xB7,0x02,
	'T','B','A',		0xC1,0x0F,0x00,
	'T','B','E','Q',	0x89,0x04,0x40,
	'T','B','L',		0xC2,0x3D,0x02,
	'T','B','N','E',	0x89,0x04,0x60,
	'T','F','R',		0x8A,0xB7,0x00,
	'T','P','A',		0x81,0xB7,0x20,
	'T','R','A','P',	0x8C,0x18,0x00,
	'T','S','T',		0x82,0xE7,0x04,
	'T','S','T','A',	0x81,0x97,0x00,
	'T','S','T','B',	0x81,0xD7,0x00,
	'T','S','X',		0x81,0xB7,0x75,
	'T','S','Y',		0x81,0xB7,0x76,
	'T','X','S',		0x81,0xB7,0x57,
	'T','Y','S',		0x81,0xB7,0x67,
	'W','A','I',		0x81,0x3E,0x00,
	'W','A','V',		0xC1,0x3C,0x00,
	'X','G','D','X',	0x81,0xB7,0xC5,
	'X','G','D','Y',	0x81,0xB7,0xC6,
/* directives */
	'E','Q','U',		228,	0,	0,
	'O','R','G',		229,	0,	0,
	'F','C','B',		230,	0,	0,
	'F','D','B',		231,   -1,	0,
	'R','D','B',		231,	0,	0,
	'F','C','C',		232,	0,	0,
	'F','C','C','H',	232,	1,	0,
	'F','C','C','Z',	232,	2,	0,
	'R','M','B',		233,	0,	0,
	'D','B',			230,	0,	0,
	'D','W',			231,	-1,	0,
	'D','R','W',		231,	0,	0,
	'S','T','R',		232,	0,	0,
	'S','T','R','H',	232,	1,	0,
	'S','T','R','Z',	232,	2,	0,
	'D','S',			233,	0,	0,
	'E','N','D',		234,	0,	0,
	'P','A','G','E',	248,	0,	0,
	'T','I','T','L','E',249,	0,	0,
	'S','P','A','C','E',250,	0,	0,
	'L','I','S','T',	251,	0,	0,
	'N','O','L','I','S','T',252,0,	0,
	'L','I','N','E',	253,	0,	0,
	0 };	/* end of table */

/*
 * Addressing mode adjustment tables (0xFF = Not allowed)
 */
unsigned char modeadj[11][4] = {
/*	IMM,	DIR,	IDX,	EXT */
/*------------------------------*/
	0x00,	0x10,	0x20,	0x30,	/* 0: Standard - all modes */
	0x00,	0xFF,	0xFF,	0xFF,	/* 1: Immediate only */
	0xFF,	0xFF,	0x00,	0xFF,	/* 2: Indexed only */
	0xFF, 	0xFF,	0xFF,	0x00,	/* 3: Extended only */
	0xFF,	0xFF,	0x00,	0x10,	/* 4: IDX and EXT only */
	0xFF,	0xFF,	0x01,	0x00,	/* 5: IDX & EXT (CALL) */
	0xFF,	0xFF,	0x00,	0x01,	/* 6: IDX and EXT (JMP) */
	0xFF,	0x02,	0x00,	0x01,	/* 7: DIR, IDX and EXT (JSR) */
	0xFF,	0x00,	0x10,	0x20,	/* 8: DIR, IDX, EXT (STA) */
	0x00,	0xFF,	0x00,	0x00,	/* 9: IMM, IDX and EXT */
	0xFF,	0x40,	0x00,	0x10 };	/*10: DIR, IDX, EXT (BSET) */

/* error messages */
static char *etext[] = { "?",
	"Unknown instruction",
	"Out of range",
	"Invalid addressing mode",
	"Invalid register specification",
	"Undefined symbol",
	"Invalid expression syntax",
	"Invalid argument format",
	"Improperly delimited string" } ;

/* Symbol table & free pointer */
char symtab[SYMB_POOL], *symtop;

/* misc. global variables */
char buffer[LINE_SIZE+1], label[SYMB_SIZE+1], title[50], operand[80], optr,
	itype, opcode, opcode1, prebyte, postbyte;
unsigned char instruction[80], outrec[35];

char optf=3, symf=0, intel=0, fulf=0, quietf=0, nlist=0, casf=0;

unsigned addr, line=0, pcount=0, lcount=0, value, length, ilength, emsg,
	ecnt, ocnt=0, pagel=59, pagew=80;

FILE *asm_fp, *hex_fp, *lst_fp;

/*
 * Define a symbol in the symbol table
 */
char *create_symbol(symbol, value)
	char *symbol;
	int value;
{
	register char *ptr, *ptr1;

	ptr = ptr1 = symtop;
	while(*symbol)
		*++ptr = *symbol++;
	*ptr1 = ptr - ptr1;
	*++ptr = value >> 8;
	*++ptr = value;
	if(ptr < (symtab+sizeof(symtab))) {
		symtop = ptr + 1;
		return ptr1; }
	return 0;
}

/*
 * Lookup a symbol in the symbol table
 */
char *lookup_symbol(symbol)
	char *symbol;
{
	register int l;
	register unsigned char *ptr, *ptr1;
	char *ptr2;

	ptr = symtab;
again:
	if((ptr2 = ptr) >= symtop) {
		value = 0x8888;
		return 0; }
	ptr1 = symbol;
	l = *ptr++ & SYMMASK;
	while(l--) {
		if(*ptr1++ != *ptr++) {
			ptr += l + 2;
			goto again; } }
	if(*ptr1) {
		ptr += 2;
		goto again; }
	value = *ptr++ << 8;
	value |= *ptr;
	return ptr2;
}

/*
 * Set the value for a symbol in the symbol table
 */
set_symbol(ptr, value)
	char *ptr;
	unsigned value;
{
	ptr += (*ptr & SYMMASK);
	*++ptr = value >> 8;
	*++ptr = value;
}

/*
 * Display the symbol table (sorted)
 */
dump_symbols()
{
	unsigned char *ptr, *hptr;
	unsigned int l, h, w;

	fprintf(lst_fp, "SYMBOL TABLE:\n\n");
	w = 0;
	for(;;) {
		for(hptr = symtab; hptr < symtop; hptr += (*hptr & SYMMASK) + 3)
			if(!(*hptr & SYMSORT))
				goto found;
		putc('\n', lst_fp);
		return;
found:	for(ptr = (*hptr & SYMMASK) + hptr + 3; ptr < symtop; ptr += (*ptr & SYMMASK) + 3) {
			if(*ptr & SYMSORT)
				continue;
			if(compsym(ptr, hptr) < 0)
				hptr = ptr; }
		*(ptr = hptr) |= SYMSORT;
		h = l = *ptr++ & SYMMASK;
		if((w + l + 7) >= pagew) {			/* Overrun page length */
			putc('\n', lst_fp);
			w = 0; }
		if(w) {								/* Not a start of line - separate */
			fprintf(lst_fp, "   ");
			w += 3; }
		while(l--)
			putc(*ptr++, lst_fp);
		w += (l = (h > 8) ? 24 : 8) + 5;	/* Calculate extended length */
		while(h++ < l)
			putc(' ', lst_fp);
		l = *ptr++ << 8;
		l |= *ptr++;
		fprintf(lst_fp, "-%04x", l); }
}

/*
 * Compare two symbol table entries
 */
compsym(sym1, sym2)
	char *sym1;
	char *sym2;
{
	int l, l1, l2;
	l1 = *sym1++ & SYMMASK;
	l2 = *sym2++ & SYMMASK;
	l = (l1 < l2) ? l1 : l2;
	do {
		if(*sym1 > *sym2)
			return 1;
		if(*sym1++ < *sym2++)
			return -1; }
	while(--l);
	if(l1 > l2)
		return 1;
	if(l1 < l2)
		return -1;
	return 0;
}

/************************************/
/* get a character from the operand */
/************************************/
char getchr()
{
	register char c;

	if(c=operand[optr])
		++optr;
	return(chupper(c));
}

/*
 * Convert character to upper case if NOT case sensitive
 */
chupper(c)
	char c;
{
	return casf ? c : ((c >= 'a') && (c <= 'z')) ? c - ('a'-'A') : c;
}

/*
 * Open a filename with the appriopriate extension &
 * report an error if not possible
 */
FILE *open_file(filename, extension, options)
	char *filename, *extension, *options;
{
	char buffer[100], *ptr, *dot;
	FILE *fp;

	dot = 0;

	for(ptr = buffer; *ptr = *filename; ++ptr) {	/* Copy filename */
		if(*filename == '.')
			dot = filename;
		else if(*filename == '\\')
			dot = 0;
		++filename; }

	if(!dot) {									/* No extension supplied */
		*ptr++ = '.';
		do
			*ptr++ = *extension;
		while(*extension++); }
	else
		*dot = 0;

	if(!(fp = fopen(buffer, options))) {
		fprintf(stderr,"Unable to access: '%s'\n", buffer);
		exit(-1); }

	return fp;
}

/*
 * Main program
 */
main(argc, argv)
	unsigned argc;
	char *argv[];
{
	unsigned i, j, temp, daddr, vbyt;
	int stemp;
	char chr, cflg, pflg, *ptr, *lfile, *cfile;

	if(argc < 2)
		abort("\nUse: asm12 <filename> [-cfiqst c=file l=file o=n p=length w=width]\n\n?COPY.TXT 1983-2005 Dave Dunfield\n**See COPY.TXT**.\n");

	pflg = 0;

/* parse for command line options. */
	lfile = cfile = argv[1];
	for(i=2; i < argc; ++i) {
		if(*(ptr = argv[i]) == '-') {		/* Enable switch */
			while(*++ptr) switch(toupper(*ptr)) {
				case 'C' : casf = -1;		break;
				case 'F' : fulf = -1;		break;
				case 'I' : intel = -1;		break;
				case 'Q' : quietf = -1;		break;
				case 'S' : symf = -1;		break;
				case 'T' : lfile = 0;		break;
				default: goto badopt; }
			continue; }
		if(*++ptr == '=') switch(toupper(*(ptr++ - 1))) {
			case 'L' : lfile = ptr;			continue;
			case 'C' : cfile = ptr;			continue;
			case 'O' : optf=atoi(ptr);		continue;
			case 'P' : pagel=atoi(ptr)-1;	continue;
			case 'W' : pagew=atoi(ptr);		continue; }
	badopt:
		fprintf(stderr,"Invalid option: %s\n", argv[i]);
		exit(-1); }

	asm_fp = open_file(argv[1], "ASM", "r");
	hex_fp = open_file(cfile, "HEX", "w");
	lst_fp = (lfile) ? open_file(lfile, "LST", "w") : stdout;
	strncpy(title,argv[1],sizeof(title)-1);

/* first pass - build symbol table */
	symtop = symtab;
	do {
		addr = cflg = 0; line = 1;
		if(!quietf)
			fprintf(stderr, pflg ? "Opt... " : "First pass... ");
		while(readline()) {
			if(!optr) {
				locins(instruction);
				if(label[0]) {				/* label exists */
					if(pflg) {				/* optomization pass */
						ptr = lookup_symbol(label);
						if(itype==100) {	/* EQU statement */
							optr = 0; vbyt = value;
							if((temp=eval()) != vbyt) {
								if(pflg >= optf) optfail();
								set_symbol(ptr, temp);
								cflg=1; } }
						else if(value != addr) {
							if(pflg >= optf) optfail();
							set_symbol(ptr, addr);
							cflg=1; } }
					else {							/* original pass-1 */
						if(lookup_symbol(label)) {	/* duplicate label */
							fprintf(lst_fp,"** Line %u - Duplicate symbol: '%s'\n",line,label);
							++ecnt; }
						else {
							cflg=1;			/* indicate symbol table change */
							if(!(ptr = create_symbol(label, addr))) {
								xabort("symbol");
								fprintf(lst_fp,"** Line %u - Symbol table overflow\n", line);
								++ecnt;
								break; } } } }
				emsg = length = optr = prebyte = 0;
				if((itype & 0x70) == 0x40) {	/* Prebyte */
					prebyte = 0x18;
					itype &= 0x3F;
					++length; }
				switch(itype) {
					case 1 :			/* Inherent */
						length += (opcode1) ? 2 : 1;
						break;
					case 8 :			/* Bit test and branch */
						++length;
					case 7 :			/* BIT addressing */
					case 11 :			/* CALL */
						++length;
					case 2 :			/* Full addressing - 8 */
					case 3 :			/* Full memory address - 16 */
						i = memop();
						if((itype == 3) && !i)
							++length;
						length += ilength + 1;
						break;
					case 4 :			/* Move operands */
						j = opcode1;
						opcode1 = 9;	/* Disallow direct */
						i = memop();
						temp = ilength;
						vbyt = value;
						switch((i << 4) +  memop()) {
							case 0x03 :			/* IMM -> EXT */
								length += (4 + j);
								break;
							case 0x02 :			/* IMM -> IND */
								length += (3 + j);
								break;
							case 0x33 :			/* EXT -> EXT */
								length += 5;
								break;
							case 0x32 :			/* EXT->IDX */
							case 0x23 :			/* IDX->EXT */
								length += 4;
								break;
							case 0x22 :			/* IDX->IDX */
								length += 3; }
						break;
					case 5 :			/* 8 bit relative */
					case 10:			/* EXT */
					case 12:			/* TRAP */
						length += 2;
						break;
					case 6 :			/* 16 bit relative */
					case 9 :			/* DBEQ etc. */
						length += 3;
						break;
					case 100 :			/* EQU directive */
						if(!pflg) {
							set_symbol(ptr, eval());
						err1st:
							if(emsg && !optf) {
								fprintf(lst_fp,"** Line %u - %s\n",line,etext[emsg]);
								++ecnt; } }
						break;
					case 101 :			/* ORG directive */
						addr=eval();
						goto err1st;
					case 102 :			/* FCB statement */
						length=1;
						while(nxtelem()) ++length;
						break;
					case 103 :			/* FDB statement */
						length=2;
						while(nxtelem()) length += 2;
						break;
					case 104 :			/* FCC statements */
						chr=operand[0];
						for(i=1;(operand[i])&&(chr!=operand[i]); ++i) ++length;
						if(opcode == 2) ++length;
						break;
					case 105 :			/* RMB statement */
						addr = addr + eval();
						goto err1st;
					case 106 :			/* END statement */
						goto end1;
					case 125:			/* LINE directive */
						line = eval(); } }
			else
				length = 0;
			addr += length;
			++line; /* += (itype < 120); */ }
end1:	++pflg;
		rewind(asm_fp); }
	while((optf)&&(cflg)&&(!ecnt));

/* second pass - generate object code */

	if(!quietf)
		fprintf(stderr,"Second pass... ");
	addr = emsg = 0; line = pcount = 1; lcount = 9999;
	while(readline()) {
		daddr = addr;
		if(!optr) {
			if(!locins(instruction))
				reperr(1);			/* unknown opcode */
			length = temp = optr = 0;
			if((itype & 0x70) == 0x40) {	/* Prebyte */
				instruction[length++] = prebyte = 0x18;
				itype &= 0x3F; }
			switch(itype) {
				case 1 :			/* Inherent */
					instruction[length++] = opcode;
					if(opcode1)
						instruction[length++] = opcode1;
					break;
				case 2 :			/* Full memory address - 8 */
				case 3 :			/* Full memory address - 16 */
					i = memop();
					instruction[length++] = opcode + modeadj[opcode1][i];
					switch(i) {
						case 2 :		/* Indexed */
							instruction[length++] = postbyte;
							if(ilength == 3)
								goto wr16;
							if(ilength == 2)
								goto wr8;
							break;
						case 0 :		/* Immediate */
							if(itype != 3)
								goto wr8;
						case 3 :		/* Extended */
							goto wr16;
						case 1 :		/* Direct */
							goto wr8; }
					break;
					case 4 :			/* Move operands */
						j = opcode1;
						opcode1 = 9;	/* Disallow DIR */
						i = memop();
						temp = ilength;
						vbyt = value;
						stemp = postbyte;
						if(operand[optr-1] != ';')
							reperr(6);
						switch((i << 4) +  memop()) {
							default:
								reperr(3);
								break;
							case 0x03 :			/* IMM -> EXT */
								instruction[length++] = opcode + 3;
								if(j)
									goto wr16a;
								goto wr8a;
							case 0x02 :			/* IMM -> IDX */
								if(ilength > 1)
									reperr(3);
								instruction[length++] = opcode;
								instruction[length++] = postbyte;
								value = vbyt;
								if(j)
									goto wr16;
								goto wr8;
							case 0x33 :			/* EXT -> EXT */
								instruction[length++] = opcode + 4;
							wr16a:
									instruction[length++] = vbyt >> 8;
							wr8a:
								instruction[length++] = vbyt;
								goto wr16;
							case 0x32 :			/* EXT->IDX */
								if(ilength > 1)
									reperr(3);
								instruction[length++] = opcode + 1;
								instruction[length++] = postbyte;
								value = vbyt;
								goto wr16;
							case 0x23 :			/* IDX->EXT */
								if(temp > 1)
									reperr(3);
								instruction[length++] = opcode + 5;
								instruction[length++] = stemp;
								goto wr16;
							case 0x22 :			/* IDX->IDX */
								if((temp > 1) || (ilength > 1))
									reperr(3);
								instruction[length++] = opcode + 2;
								instruction[length++] = stemp;
								instruction[length++] = postbyte; }
						break;
				case 5 :			/* 8 bit relative */
				case 6 :			/* 16 bit relative */
					instruction[length++] = opcode;
				dobra:
					value = stemp = eval() - addr - (length+1);
					if(itype != 6) {
						if((stemp > 127) || (stemp < -128))
						reperr(2);		/* Out of range */
						goto wr8; }
					--value;
					wr16:
						instruction[length++] = value >> 8;
					wr8:
						instruction[length++] = value;
					break;
				case 7 :			/* BIT instructions */
				case 8 :			/* Bit test and branch */
					i = memop();
					instruction[length++] = opcode + modeadj[opcode1][i];
					if(operand[optr-1] != ';')
						reperr(6);
					switch(i) {
						case 2 :		/* Indexed */
							instruction[length++] = postbyte;
							if(ilength == 3)
								goto wr16b;
							if(ilength == 2)
								goto wr8b;
							break;
						case 0 :		/* Immediate */
							if(itype != 3)
								goto wr8b;
						case 3 :		/* Extended */
						wr16b:
							instruction[length++] = value >> 8;
						case 1 :		/* Direct */
						wr8b:
							instruction[length++] = value; }
					instruction[length++] = eval();
					if(itype != 8)
						break;
					if(((chr = operand[optr-1]) != ';') && (chr != ','))
						reperr(6);
					goto dobra;
				case 9 :			/* DBEQ etc. */
					instruction[length++] = opcode;
					switch(getchr()) {
						case 'A' :	i = 0;	break;
						case 'B' :	i = 1;	break;
						case 'D' :	i = 4;	break;
						case 'X' :	i = 5;	break;
						case 'Y' :	i = 6;	break;
						case 'S' :
							if(getchr() == 'P') {
								i = 7; break; }
							--optr;
						default:
							reperr(4); }
					if(((chr = getchr()) != ';') && (chr != ','))
						reperr(6);
					value = stemp = eval() - addr - (length+2);
					if((stemp > 255) || (stemp < -256))
						reperr(2);		/* Out of range */
					if(stemp < 0)
						i += 0x10;
					instruction[length++] = opcode1 + i;
					instruction[length++] = value;
					break;
				case 10 :			/* TFR and EXG */
					instruction[length++] = opcode;
					stemp = -1;
				dotfr:
					switch(getchr()) {
						case 'A' :	i = 0;	break;
						case 'B' :	i = 1;	break;
						case 'C' :	i = 2;
							if(getchr() == 'C')
								if(getchr() == 'R')
									break;
							goto badreg;
						case 'D' :	i = 4;	break;
						case 'X' :	i = 5;	break;
						case 'Y' :	i = 6;	break;
						case 'S' :	i = 7;
							if(getchr() == 'P')
								break;
						default:
						badreg:
							--optr;
							reperr(4); }
					if(!isterm(getchr()))
						reperr(4);
					if(stemp == -1) {
						stemp = i;
						if(operand[optr-1] != ',')
							reperr(6);
						goto dotfr; }
					instruction[length++] = (stemp<<4) + i + opcode1;
					break;
				case 11 :			/* CALL */
					i = memop();
					instruction[length++] = opcode + modeadj[opcode1][i];
					switch(i) {
						case 2 :		/* Indexed */
							instruction[length++] = postbyte;
							if(ilength == 3)
								goto wr16c;
							if(ilength == 2)
								goto wr8c;
							break;
						case 3 :		/* Extended */
						wr16c:
							instruction[length++] = value >> 8;
						wr8c:
							instruction[length++] = value; }
					if(((chr = operand[optr-1]) != ';') && (chr != ','))
						reperr(6);
					value = eval();
					goto wr8;
				case 12 :			/* TRAP */
					instruction[length++] = opcode;
					instruction[length++] = i = eval();
					if((i < 0x30) || (i > 0xFF))
						reperr(2);
					if((i > 0x39) && (i < 0x40))
						reperr(2);
					break;
				case 100 :			/* equate statement */
					daddr=eval();	/* generate errors if any */
					break;
				case 101 :			/* ORG statement */
					if(ocnt) wrrec();
					daddr=addr=eval();
					break;
				case 102 :			/* FCB statement */
					do {
						instruction[length++] = eval(); }
					while(operand[optr-1] == ',');
					break;
				case 103 :			/* FDB statement */
					do {
						temp = eval();
						if(opcode) {
							instruction[length++] = temp >> 8;
							instruction[length++] = temp; }
						else {
							instruction[length++] = temp;
							instruction[length++] = temp >> 8; } }
					while(operand[optr-1] == ',');
					break;
				case 104 :			/* FCC statements */
					chr=operand[0];
					for(i=1;((operand[i])&&(chr!=operand[i])); ++i)
						instruction[length++] = operand[i];
					if(!operand[i])	reperr(8);
					if(opcode == 1)
						instruction[length-1] = instruction[length-1] | 0x80;
					if(opcode == 2)
						instruction[length++]=0;
					break;
				case 105 :			/* RMB statement */
					if(ocnt) wrrec();
					addr += eval();
					break;
				case 106 :			/* END statement */
					goto end2;
				case 120 :			/* PAGE statement */
					lcount=9999;
					break;
				case 121 :			/* TITLE directive */
					strncpy(title, operand, sizeof(title)-1);
					break;
				case 122 :			/* SPACE directive */
					fprintf(lst_fp,"\n");
					++lcount;
					break;
				case 123 :			/* LIST directive */
					if(nlist) --nlist;
					break;
				case 124 :			/* NOLIST directive */
					++nlist;
					break;
				case 125 :			/* LINE directive */
					line = eval(); } }
		else
			length=itype=0;

/* generate listing */
	if(((itype<120) && fulf && !nlist) || emsg) {
		if(++lcount >= pagel)
			write_title();
		fprintf(lst_fp,"%04x ",daddr);
			for(i=0; i < 6; ++i) {
				if(i < length)
					fprintf(lst_fp," %02x",instruction[i]);
				else
					fprintf(lst_fp,"   "); }
			fprintf(lst_fp," %c%5u  %s\n",(length <= 6) ? ' ' : '+',line,buffer); }
		if(emsg) {
			fprintf(lst_fp,"  ** ERROR ** - %u - %s\n", emsg, etext[emsg]);
			++ecnt; ++lcount;
			emsg=0; }
		++line; /* += (itype<120); */
/* write code to output record */
	for(i=0; i<length; ++i) {
		if(!ocnt) {			/* first byte of record */
			outrec[ocnt++]=addr>>8;
			outrec[ocnt++]=addr; }
		++addr;
		outrec[ocnt++]=instruction[i];
		if(ocnt>33) wrrec(); } }	/* record is full, write it */

end2:
	if(ocnt) wrrec();
	if(intel) fprintf(hex_fp,":00000001FF\n");
	else fprintf(hex_fp,"S9030000FC\n");

	if(ecnt) fprintf(lst_fp,"\n %u error(s) occurred in this assembly.\n",ecnt);
	if(!quietf) fprintf(stderr,"%u error(s).\n",ecnt);

/* display the symbol table */
	if(symf) {
		write_title();
		dump_symbols(); }

	fclose(asm_fp);
	fclose(hex_fp);
	fclose(lst_fp);

	return ecnt ? -2 : 0;
}

/***********************************************/
/* lookup instruction in the instruction table */
/***********************************************/
int locins(inst)
	char inst[];
{
	register char *ptr, *ptr1;

	ptr = opcodes;
	do {
		ptr1 = inst;
		while(*ptr == *ptr1) {
			++ptr;
			++ptr1; }
		if((*ptr < 0) && !*ptr1) {
			itype = *ptr++ & 127;
			opcode = *ptr++;
			opcode1 = *ptr++;
			return 1; }
		while(*ptr > 0)
			++ptr; }
	while(*(ptr += 3));

	itype = 119;
	return(inst[opcode1 = 0] == 0);
}

/*************************************************/
/* read a line from input file, and break it up. */
/*************************************************/
int readline()
{
	register int i;
	register char *ptr;

	if(!fgets(ptr = buffer, LINE_SIZE, asm_fp))
		return 0;

	i = 0;
	while(issymbol(*ptr)) {
		label[i] = chupper(*ptr++);
		if(i < SYMB_SIZE)
			++i; }
	label[i]=0;
	if(*ptr == ':')
		++ptr;
	while(isspace(*ptr))
		++ptr;
	if(((*ptr != '*') && (*ptr != ';')) || i) {
		i = 0;
		while(*ptr && !isspace(*ptr))
			instruction[i++] = toupper(*ptr++);
		instruction[i]=0;
		while(isspace(*ptr))
			++ptr;
		optr = i = 0;
		while(*ptr && (i < 79))
			operand[i++] = *ptr++;
		operand[i] = 0;
		return 1; }

	return optr = 1;
}

/***************************************/
/* test for valid terminator character */
/***************************************/
int isterm(chr)
	register char chr;
{
	switch(chr) {
		case 0 :
		case ' ' :
		case '\t':
		case ',' :
		case ';' :
		case ')' :
			return 1; }
	return 0;
}

/*************************************/
/* Test for a valid symbol character */
/*************************************/
int issymbol(c)
	char c;
{
	switch(c) {			/* Special allowed characters */
		case '_' :
		case '?' :
		case '!' :
		case '.' :
			return 1; }

	return isalnum(c);
}

/* report an error */
reperr(errn)
	register int errn;
{	if(!emsg) emsg=errn;
}

/*******************************/
/* write record to output file */
/*******************************/
wrrec()
{	register unsigned i, chk, chr;

	xhex(ocnt-2);
	if(intel) {					/* intel hex format */
		chk = outrec[0] + outrec[1] + ocnt - 2;
		fprintf(hex_fp,":%02x%02x%02x00", ocnt-2, outrec[0], outrec[1]);
		for(i=2; i<ocnt; ++i) {
			fprintf(hex_fp,"%02x", chr = outrec[i]);
			chk += chr; }
		fprintf(hex_fp,"%02x\n", 255 & (0-chk)); }
	else {						/* motorola hex format */
		chk = ocnt + 1;
		fprintf(hex_fp,"S1%02x", ocnt + 1);
		for(i=0; i<ocnt; ++i) {
			fprintf(hex_fp,"%02x", chr = outrec[i]);
			chk += chr; }
		fprintf(hex_fp,"%02x\n", 255 & ~chk); }
	ocnt = 0;
}

/*
 * Evaluate an expression.
 */
int eval()
{	register char c;
	unsigned result;

	result=getval();
	while(!isterm(c = getchr())) switch(c) {
		case '+' : result += getval();	break;
		case '-' : result -= getval();	break;
		case '*' : result *= getval();	break;
		case '/' : result /= getval();	break;
		case '\\': result %= getval();	break;
		case '&' : result &= getval();	break;
		case '|' : result |= getval();	break;
		case '^' : result ^= getval();	break;
		case '<' : result <<= getval();	break;
		case '>' : result >>= getval();	break;
		default: reperr(6); }

	return result;
}

/*
 * Get a single value (number or symbol)
 */
int getval()
{
	unsigned i, b, val;
	char chr, array[25], *ptr;

	switch(chr = operand[optr++]) {
		case '-' :				/* Negated value */
			return 0 - getval();
		case '~' :				/* Complemented value */
			return ~getval();
		case '=' :				/* Swap high and low bytes */
			i=getval();
			return (i<<8)+(i>>8);
		case '(' :				/* Nested expression */
			val = eval();
			if(operand[optr-1] != ')')
				reperr(6);
			return val; }
	--optr;

	val=b=i=0;
	switch(chr) {
		case '$' :	b = 16;	goto getnum;
		case '@' :	b = 8;	goto getnum;
		case '%' :	b = 2;	goto getnum; }
	if(isdigit(chr)) {
		array[i++] = chr;
	getnum:
		while(isalnum(chr = toupper(operand[++optr])))
			array[i++] = (chr < 'A') ? chr : chr - 7;
		if((b == 16) && !i)
			val = addr;
		else {
			if(!b) {
				b = 10;
				switch(array[i-1]) {
					case 'H'-7 :	b = 16;	goto deci;
					case 'T'-7 :
					case 'D'-7 :	b = 10;	goto deci;
					case 'Q'-7 :
					case 'O'-7 :	b = 8;	goto deci;
					case 'B'-7 :	b = 2;
					deci:	--i;	} }
			if(!i)
				reperr(6);
			for(ptr = array; i; --i) {
				if((chr = *ptr++ - '0') >= b)
					reperr(6);
				val = (val * b) + chr; } } }
	else if(chr=='\'') {				/* Single quote. */
		++optr;
		while(((chr=operand[optr++]) != '\'')&& chr)
			val=(val << 8)+chr;
		if(!chr) reperr(8); }
	else if(chr=='*') {					/* Program counter */
		++optr;
		val=addr; }
	else {								/* Must be a label */
		i = 0;
		while(issymbol(chr = chupper(operand[optr]))) {
			++optr;
			label[i]=chr;
			if(i < SYMB_SIZE)
				++i; }
		label[i]=0;

		if(!lookup_symbol(label)) reperr(5);
		val=value; }

	return val;
}

/*
 * Write out title
 */
write_title()
{
	int w;
	char *ptr;

	if(pcount > 1)
		putc('\f',lst_fp);
	lcount = 1;
	fprintf(lst_fp,"DUNFIELD 68HC12 ASSEMBLER: ");
	ptr = title;
	for(w = 37; w < pagew; ++w)
		putc(*ptr ? *ptr++ : ' ', lst_fp);
	fprintf(lst_fp,"PAGE: %u\n\n", pcount);
	++pcount;
}

/**************************************/
/* find next element in argument list */
/**************************************/
int	nxtelem() {
	register char chr;
	while((chr=operand[optr])&&(chr != ' ')&&(chr != 9)) {
		++optr;
		if(chr==39) {
			while((chr=operand[optr])&&(chr!=39)) ++optr;
			++optr; }
		if(chr==',') return(1); }
	return(0);
}

/*
 * Too many optimization passes - report failure
 */
optfail()
{
	fprintf(lst_fp,"** Line %u - Unable to resolve: %s\n", line, label);
	++ecnt;
}

/*
 * Evaluate a memory reference and determine it's type, calculate
 * the postbyte and operand length.
 */
memop()
{
	int v;
	register unsigned char ind, acc, reg, incdec;

	ilength = 1;
	value = incdec = ind = acc = v = 0;
	switch(operand[optr++]) {
		case '#' :								/* Immediate */
			value = eval();
			if(modeadj[opcode1][0] == 255)
				reperr(3);
			return 0;
		case '<' :								/* Direct */
			value = eval();
			if(modeadj[opcode1][1] == 255)
				reperr(3);
			return 1;
		case '>' :								/* Extended */
			ilength = 2;
			value = eval();
			if(modeadj[opcode1][3] == 255)
				reperr(3);
			return 3;
		case '[' :								/* Indirect */
			ind = -1;
			break;
		default:
			--optr; }

	if(operand[optr] == ',') {					/* Comma .. no offset */
		++optr;
		goto newreg; }

	if(operand[optr+1] == ',') switch(chupper(operand[optr])) {
		case 'A' :	acc=0x80;	goto gotacc;	/* A Accumulator offset */
		case 'B' :	acc=0x81;	goto gotacc;	/* B accumulator offset */
		case 'D' :	acc=0x82;					/* D accumulator offset */
		gotacc: optr += 2; goto newreg; }

	value = v = eval();

	if(operand[optr-1] != ',') {				/* Extended or direct */
		if(acc)
			reperr(3);
		if((value > 255) || (modeadj[opcode1][1] == 255) ) {
			ilength = 2;
			return 3; }
		if(modeadj[opcode1][3] == 255)
			reperr(3);
		return 1; }

newreg:
	switch(chupper(operand[optr++])) {
		case '-' : if(incdec) reperr(3); incdec = 1; goto newreg;
		case '+' : if(incdec) reperr(3); incdec = 2; goto newreg;
		case 'X' :	reg = 0x00;	break;
		case 'Y' :	reg = 0x01;	break;
		case 'S' :
			if(chupper(operand[optr++]) != 'P')
				goto badreg;
			reg = 0x02;
			break;
		case 'P' :
			if(chupper(operand[optr++]) == 'C') {
				reg=0x03;
				break; }
		default:
		badreg:
			--optr;
			reperr(4); }

	switch(operand[optr]) {
		case '-' : if(incdec) reperr(3); incdec = 3; goto adjoptr;
		case '+' : if(incdec) reperr(3); incdec = 4;
		adjoptr: ++optr; }

	if(ind) {								/* Handle indirect modes */
		if((operand[optr] == ']') && !incdec)
			++optr;
		else
			reperr(3);
		switch(acc) {
			case 0x00 :		/* Offset value */
				postbyte = (reg << 3) | 0xE3;
				ilength = 3;
				break;
			case 0x82 :		/* D offset indexed-indirect */
				postbyte = (reg << 3) | 0xE7;
				break;
			default:
				reperr(3); } }

	if(!isterm(getchr()))
		reperr(3);

	if(modeadj[opcode1][2] == 255)
		reperr(3);

	if(ind)
		return 2;

	if(acc) {								/* Handle ACC offsets */
		if(incdec)
			reperr(3);
		postbyte = (reg << 3) | (acc & 0x0F) | 0xE4;
		return 2; }

	if(incdec) {							/* Handle Auto inc/dec modes */
		if(reg == 0x03)
			reperr(4);
		if(!value)
			reperr(3);
		if(value > 8)
			reperr(2);
		switch(incdec) {
			case 1 : postbyte = -value & 15;	 		break;	/* pre-dec */
			case 2 : postbyte = value-1;				break;	/* pre-inc */
			case 3 : postbyte = 0x10 | (-value & 15);	break;	/* Post-dec */
			case 4 : postbyte = 0x10 | (value-1); 	}			/* Post inc */
		postbyte |= ((reg << 6) | 0x20);
		return 2; }

	if((v >= -16) && (v < 16)) {			/* 5 bit offset */
		postbyte = (reg << 6) | (value & 0x1F);
		return 2; }

	if((v >= -256) && (v < 256)) {			/* 9 bit offset */
		postbyte = (reg << 3) | 0xE0 | (v < 0);
		ilength = 2;
		return 2; }

	postbyte = (reg << 3) | 0xE2;			/* 16 bit offset */
	ilength = 3;
	return 2;
}
