/*
 * MICRO 80186 ASSEMBLER:
 *
 * This 80186 assembler, supports a "motorola" style syntax.
 *
 * ?COPY.TXT 1983-2005 Dave Dunfield
 * **See COPY.TXT**.
 */
#include <stdio.h>
#include <ctype.h>

#include "xasm.h"

/* Fixed assembler parameters */
#define	NUMREG		20		/* Number of registers defined */
#define	NUMSEG		4		/* Maximum number of segments */

/* Operand types */
#define	REGIS		1		/* Simple register */
#define	VALUE		2		/* Direct memory reference */
#define	INDEX		3		/* Indexed memory reference */
#define	IMMED		4		/* Immediate value */

/* Symbol flags */
#define	SDUP		0x80	/* Duplicate symbol */
#define	SPUB		0x40	/* Public symbol */
#define	SEXT		0x20	/* External reference */

/* Instruction opcode table */
#define	IDATSIZE	5		/* Number of bytes of instruction data */
unsigned char inst_table[] = {
	/* Type 1 - all bytes up to 0x00 are written */
	'A','A','A'+128,		1, 0x37, 0x00, 0x00, 0x00,
	'A','A','D'+128,		1, 0xD5, 0x0A, 0x00, 0x00,
	'A','A','M'+128,		1, 0xD4, 0x0A, 0x00, 0x00,
	'A','A','S'+128,		1, 0x3F, 0x00, 0x00, 0x00,
	'C','B','W'+128,		1, 0x98, 0x00, 0x00, 0x00,
	'C','L','C'+128,		1, 0xF8, 0x00, 0x00, 0x00,
	'C','L','D'+128,		1, 0xFC, 0x00, 0x00, 0x00,
	'C','L','I'+128,		1, 0xFA, 0x00, 0x00, 0x00,
	'C','M','C'+128,		1, 0xF5, 0x00, 0x00, 0x00,
	'C','M','P','S','B'+128,1, 0xA6, 0x00, 0x00, 0x00,
	'C','M','P','S','W'+128,1, 0xA7, 0x00, 0x00, 0x00,
	'C','W','D'+128,		1, 0x99, 0x00, 0x00, 0x00,
	'D','A','A'+128,		1, 0x27, 0x00, 0x00, 0x00,
	'D','A','S'+128,		1, 0x2F, 0x00, 0x00, 0x00,
	'H','L','T'+128,		1, 0xF4, 0x00, 0x00, 0x00,
	'I','N','T','O'+128,	1, 0xCE, 0x00, 0x00, 0x00,
	'I','R','E','T'+128,	1, 0xCF, 0x00, 0x00, 0x00,
	'L','A','H','F'+128,	1, 0x9F, 0x00, 0x00, 0x00,
	'L','O','D','S','B'+128,1, 0xAC, 0x00, 0x00, 0x00,
	'L','O','D','S','W'+128,1, 0xAD, 0x00, 0x00, 0x00,
	'M','O','V','S','B'+128,1, 0xA4, 0x00, 0x00, 0x00,
	'M','O','V','S','W'+128,1, 0xA5, 0x00, 0x00, 0x00,
	'N','O','P'+128,		1, 0x90, 0x00, 0x00, 0x00,
	'P','O','P','F'+128,	1, 0x9D, 0x00, 0x00, 0x00,
	'P','U','S','H','F'+128,1, 0x9C, 0x00, 0x00, 0x00,
	'S','A','H','F'+128,	1, 0x9E, 0x00, 0x00, 0x00,
	'S','C','A','S','B'+128,1, 0xAE, 0x00, 0x00, 0x00,
	'S','C','A','S','W'+128,1, 0xAF, 0x00, 0x00, 0x00,
	'S','T','C'+128,		1, 0xF9, 0x00, 0x00, 0x00,
	'S','T','D'+128,		1, 0xFD, 0x00, 0x00, 0x00,
	'S','T','I'+128,		1, 0xFB, 0x00, 0x00, 0x00,
	'S','T','O','S','B'+128,1, 0xAA, 0x00, 0x00, 0x00,
	'S','T','O','S','W'+128,1, 0xAB, 0x00, 0x00, 0x00,
	'W','A','I','T'+128,	1, 0x9B, 0x00, 0x00, 0x00,
	'X','L','A','T'+128,	1, 0xD7, 0x00, 0x00, 0x00,
	'R','E','P'+128,		1, 0xF3, 0x00, 0x00, 0x00,
	'R','E','P','Z'+128,	1, 0xF3, 0x00, 0x00, 0x00,
	'R','E','P','E'+128,	1, 0xF3, 0x00, 0x00, 0x00,
	'R','E','P','N','Z'+128,1, 0xF2, 0x00, 0x00, 0x00,
	'R','E','P','N','E'+128,1, 0xF2, 0x00, 0x00, 0x00,
	'L','O','C','K'+128,	1, 0xF0, 0x00, 0x00, 0x00,
/* 80186 extensions */
	'I','N','S','B'+128,	1, 0x6C, 0x00, 0x00, 0x00,
	'I','N','S','W'+128,	1, 0x6D, 0x00, 0x00, 0x00,
	'O','U','T','S','B'+128,1, 0x6E, 0x00, 0x00, 0x00,
	'O','U','T','S','W'+128,1, 0x6F, 0x00, 0x00, 0x00,
	'L','E','A','V','E'+128,1, 0xC9, 0x00, 0x00, 0x00,
	'P','U','S','H','A'+128,1, 0x60, 0x00, 0x00, 0x00,
	'P','O','P','A'+128,	1, 0x61, 0x00, 0x00, 0x00,
	/* Type 2 - MEM, REGIS, 16BIT */
	'D','E','C'+128,		2, 0xFE, 0x01, 0x48, 0x00,
	'D','I','V'+128,		2, 0xF6, 0x06, 0x00, 0x00,
	'I','D','I','V'+128,	2, 0xF6, 0x07, 0x00, 0x00,
/*	'I','M','U','L'+128,	2, 0xF6, 0x05, 0x00, 0x00, */
	'I','N','C'+128,		2, 0xFE, 0x00, 0x40, 0x00,
	'M','U','L'+128,		2, 0xF6, 0x04, 0x00, 0x00,
	'N','E','G'+128,		2, 0xF6, 0x03, 0x00, 0x00,
	'N','O','T'+128,		2, 0xF6, 0x02, 0x00, 0x00,
	/* Type 3 - opcode */
	'L','D','S'+128,		3, 0xC5, 0x00, 0x00, 0x00,
	'L','E','A'+128,		3, 0x8D, 0x00, 0x00, 0x00,
	'L','E','S'+128,		3, 0xC4, 0x00, 0x00, 0x00,
	'B','O','U','N','D'+128,3, 0x62, 0x00, 0x00, 0x00,
	/* Type 4 - ACC/IMMED, REGIS,IMMED, INDEX,REGIS, XXX (Fn ='S') */
	'A','D','C'+128,		4, 0x14, 0x80, 0x10, 0xF2,
	'A','D','D'+128,		4, 0x04, 0x80, 0x00, 0xF0,
	'S','U','B'+128,		4, 0x2C, 0x80, 0x28, 0xF5,
	'S','B','B'+128,		4, 0x1C, 0x80, 0x18, 0xF3,
	'C','M','P'+128,		4, 0x3C, 0x80, 0x38, 0xF7,
	'A','N','D'+128,		4, 0x24, 0x80, 0x20, 0x04,
	'O','R'+128,			4, 0x0C, 0x80, 0x08, 0x01,
	'X','O','R'+128,		4, 0x34, 0x80, 0x30, 0x06,
	'T','E','S','T'+128,	4, 0xA8, 0xF6, 0x84, 0x00,
	/* Type 5 - MOV */
	'M','O','V'+128,		5, 0x00, 0x00, 0x00, 0x00,
	/* Type 6 - single byte relative jumps */
	'J','A'+128,			6, 0x77, 0x00, 0x00, 0x00,
	'J','N','B','E'+128,	6, 0x77, 0x00, 0x00, 0x00,
	'J','A','E'+128,		6, 0x73, 0x00, 0x00, 0x00,
	'J','N','B'+128,		6, 0x73, 0x00, 0x00, 0x00,
	'J','C'+128,			6, 0x72, 0x00, 0x00, 0x00,
	'J','N','C'+128,		6, 0x73, 0x00, 0x00, 0x00,
	'J','B'+128,			6, 0x72, 0x00, 0x00, 0x00,
	'J','N','A','E'+128,	6, 0x72, 0x00, 0x00, 0x00,
	'J','B','E'+128,		6, 0x76, 0x00, 0x00, 0x00,
	'J','N','A'+128,		6, 0x76, 0x00, 0x00, 0x00,
	'J','C','X','Z'+128,	6, 0xE3, 0x00, 0x00, 0x00,
	'J','E'+128,			6, 0x74, 0x00, 0x00, 0x00,
	'J','Z'+128,			6, 0x74, 0x00, 0x00, 0x00,
	'J','G'+128,			6, 0x7F, 0x00, 0x00, 0x00,
	'J','N','L','E'+128,	6, 0x7F, 0x00, 0x00, 0x00,
	'J','G','E'+128,		6, 0x7D, 0x00, 0x00, 0x00,
	'J','N','L'+128,		6, 0x7D, 0x00, 0x00, 0x00,
	'J','L'+128,			6, 0x7C, 0x00, 0x00, 0x00,
	'J','N','G','E'+128,	6, 0x7C, 0x00, 0x00, 0x00,
	'J','L','E'+128,		6, 0x7E, 0x00, 0x00, 0x00,
	'J','N','G'+128,		6, 0x7e, 0x00, 0x00, 0x00,
	'J','N','E'+128,		6, 0x75, 0x00, 0x00, 0x00,
	'J','N','Z'+128,		6, 0x75, 0x00, 0x00, 0x00,
	'J','O'+128,			6, 0x70, 0x00, 0x00, 0x00,
	'J','N','O'+128,		6, 0x71, 0x00, 0x00, 0x00,
	'J','S'+128,			6, 0x78, 0x00, 0x00, 0x00,
	'J','N','S'+128,		6, 0x79, 0x00, 0x00, 0x00,
	'J','N','P'+128,		6, 0x7B, 0x00, 0x00, 0x00,
	'J','P','O'+128,		6, 0x7B, 0x00, 0x00, 0x00,
	'J','P'+128,			6, 0x7A, 0x00, 0x00, 0x00,
	'J','P','E'+128,		6, 0x7A, 0x00, 0x00, 0x00,
	'L','O','O','P'+128,	6, 0xE2, 0x00, 0x00, 0x00,
	'L','O','O','P','E'+128,6, 0xE1, 0x00, 0x00, 0x00,
	'L','O','O','P','Z'+128,6, 0xE1, 0x00, 0x00, 0x00,
	'L','O','O','P','N','E'+128,6,0xE0,0x00,0x00,0x00,
	'L','O','O','P','N','Z'+128,6,0xE0,0x00,0x00,0x00,
	/* Type 7 - JMP/CALL */
	'J','M','P'+128,		7, 0xE9, 0xEA, 0x04, 0x05,
	'C','A','L','L'+128,	7, 0xE8, 0x9A, 0x02, 0x03,
	/* Type 8 - RET/RETF */
	'R','E','T'+128,		8, 0xC3, 0xC2, 0x00, 0x00,
	'R','E','T','F'+128,	8, 0xCB, 0xCA, 0x00, 0x00,
	/* Type 9 - SEGMENT, REGIS, MEMOP */
	'P','O','P'+128,		9, 0x07, 0x58, 0x8F, 0x00,
	'P','U','S','H'+128,	9, 0x06, 0x50, 0xFF, 0x06,
	/* Type 10 - STRING instructions */
	'C','M','P','S'+128,	10,0xA6, 0x00, 0x00, 0x00,
	'L','O','D','S'+128,	10,0xAC, 0x00, 0x00, 0x00,
	'M','O','V','S'+128,	10,0xA4, 0x00, 0x00, 0x00,
	'S','C','A','S'+128,	10,0xAE, 0x00, 0x00, 0x00,
	'S','T','O','S'+128,	10,0xAA, 0x00, 0x00, 0x00,
	/* Type 11 - Shifts & Rotates */
	'R','C','L'+128,		11,0xD0, 0x02, 0x00, 0x00,
	'R','C','R'+128,		11,0xD0, 0x03, 0x00, 0x00,
	'R','O','L'+128,		11,0xD0, 0x00, 0x00, 0x00,
	'R','O','R'+128,		11,0xD0, 0x01, 0x00, 0x00,
	'S','A','L'+128,		11,0xD0, 0x04, 0x00, 0x00,
	'S','H','L'+128,		11,0xD0, 0x04, 0x00, 0x00,
	'S','A','R'+128,		11,0xD0, 0x07, 0x00, 0x00,
	'S','H','R'+128,		11,0xD0, 0x05, 0x00, 0x00,
	/* Type 12 - INT */
	'I','N','T'+128,		12, 0x00, 0x00, 0x00, 0x00,
	/* Type 13 - ESC */
	'E','S','C'+128,		13,0x00, 0x00, 0x00, 0x00,
	/* Type 14 - IN/OUT */
	'I','N'+128,			14,0xE4, 0xEC, 0x00, 0x00,
	'O','U','T'+128,		14,0xE6, 0xEE, 0xFF, 0x00,
	/* Type 15 - XCHG */
	'X','C','H','G'+128,	15,0x86, 0x90, 0x00, 0x00,
	/* Type 16 - 80186: ENTER */
	'E','N','T','E','R'+128,16,0xC8, 0x00, 0x00, 0x00,
	/* Type 17 - 80186: IMUL */
	'I','M','U','L'+128,	17,0xF6, 0x05, 0x00, 0x00,
	/* Directives */
	'E','Q','U'+128,		100, 0, 0, 0, 0,
	'O','R','G'+128,		101, 0, 0, 0, 0,
	'S','E','G'+128,		102, 0, 0, 0, 0,
	'D','B'+128,			103, 0, 0, 0, 0,
	'D','W'+128,			104,-1, 0, 0, 0,
	'D','R','W'+128,		104, 0, 0, 0, 0,
	'D','S'+128,			105, 0, 0, 0, 0,
	'S','T','R'+128,		106, 0, 0, 0, 0,
	'S','T','R','Z'+128,	106, 1, 0, 0, 0,
	'S','T','R','H'+128,	106, 2, 0, 0, 0,
	'F','C','B'+128,		103, 0, 0, 0, 0,
	'F','D','B'+128,		104,-1, 0, 0, 0,
	'R','D','B'+128,		104, 0, 0, 0, 0,
	'R','M','B'+128,		105, 0, 0, 0, 0,
	'F','C','C'+128,		106, 0, 0, 0, 0,
	'F','C','C','Z'+128,	106, 1, 0, 0, 0,
	'F','C','C','H'+128,	106, 2, 0, 0, 0,
	'P','U','B','L','I','C'+128,107,0,0,0,0,
	'E','X','T','E','R','N'+128,108,0,0,0,0,
	'E','N','D'+128,		109, 0, 0, 0, 0,
	'P','A','G','E'+128,	120, 0, 0, 0, 0,
	'T','I','T','L','E'+128,121, 0, 0, 0, 0,
	'S','P','A','C','E'+128,122, 0, 0, 0, 0,
	'L','I','S','T'+128,	123, 0, 0, 0, 0,
	'N','O','L','I','S','T'+128,124,0,0,0,0,
	'L','I','N','E'+128,	125, 0, 0, 0, 0,
	'E','S',':'+128,		1, 0x26, 0, 0, 0,
	'C','S',':'+128,		1, 0x2E, 0, 0, 0,
	'S','S',':'+128,		1, 0x36, 0, 0, 0,
	'D','S',':'+128,		1, 0x3E, 0, 0, 0,
	0 }, *inst_ptr;

/* 8086 register table */
char *registers[] = {
	"AL", "CL", "DL", "BL", "AH", "CH", "DH", "BH",
	"AX", "CX", "DX", "BX", "SP", "BP", "SI", "DI",
	"ES", "CS", "SS", "DS" };

/* Error messages */
char *error_text[] = {
	"Phase error",						/* 0 */
	"Syntax error",						/* 1 */
	"Invalid instruction",				/* 2 */
	"Additional operands reqired",		/* 3 */
	"Constant value required",			/* 4 */
	"16 bit operand required",			/* 5 */
	"Illegal addressing mode",			/* 6 */
	"Illegal segment override",			/* 7 */
	"Illegal use of segment register",	/* 8 */
	"Invalid register operation",		/* 9 */
	"Invalid immediate expression",		/* 10 */
	"Invalid index register",			/* 11 */
	"Cannot preset register size",		/* 12 */
	"Unterminated character constant",	/* 13 */
	"Incompatible operand sizes",		/* 14 */
	"Size not known",					/* 15 */
	"Out of range",						/* 16 */
	"Undefined symbol",					/* 17 */
	"Duplicate sumbol",					/* 18 */
	"Symbol table overflow",			/* 19 */
	"Shift count must be CL or 0-255",	/* 20 */
	"Port must be DX or value",			/* 21 */
	"Accumulator required" };			/* 22 */

/* Symbol table & associated variables */
char symtab[SYMB_POOL], *symtop, sflags;

/* Input line buffer & holding areas */
char inline[LINE_SIZE+1], *input_ptr;
unsigned char label[SYMB_SIZE+1], instruction[80], operand[80];

/* Assembler Segment holding areas */
unsigned active_seg, seg_pc[NUMSEG], seg_size[NUMSEG] = { 0 };

/* Global locations for operand results */
int type, length, value, xvalue,
	type1, value1, xvalue1;

/* Command line options flags */
char casf = 0, fulf = 0, symf = 0, quif = 0, intel = 0,
	*lfile, *cfile;

/* Global counters and misc variables */
char dir, title[50], list = 0, outrec[35];
unsigned pc, line, ocount, daddr, pass = 0, error_flag = 0, error_count = 0,
	lcount = 9999, pcount = 1, ocnt = 0, pagel = 59, pagew = 80;

FILE *asm_fp, *hex_fp, *lst_fp;

/*
 * Lookup a symbol in the symbol table
 */
char *lookup_symbol(symbol, value)
	char *symbol;
	unsigned *value;
{
	register unsigned l;
	register unsigned char *ptr, *ptr1;
	char *ptr2;

	ptr = symtab;
again:
	if((ptr2 = ptr) >= symtop) {
		*value = 0x8888;
		return 0; }
	ptr1 = symbol;
	l = *ptr++ & SYMMASK;
	while(l--) {
		if(*ptr1++ != *ptr++) {
			ptr += l + 3;
			goto again; } }
	if(*ptr1) {
		ptr += 3;
		goto again; }
	l = *ptr++ << 8;
	*value = l | *ptr++;
	sflags = *ptr;
	return ptr2;
}

/*
 * Define a symbol
 */
char *create_symbol(symbol, value, flags)
	char *symbol;
	int value;
	char flags;
{
	register char *ptr, *ptr1;
	int j;

	if(ptr = lookup_symbol(symbol, &j)) {
		set_flags(ptr, SDUP);
		return symtop; }

	ptr = ptr1 = symtop;
	while(*symbol)
		*++ptr = *symbol++;
	*ptr1 = ptr - ptr1;
	*++ptr = value >> 8;
	*++ptr = value;
	*++ptr = flags;
	if(ptr < (symtab+sizeof(symtab))) {
		symtop = ptr + 1;
		return ptr1; }
	xabort("symbol");
	show_error(19);
	exit(-1);
}

/*
 * Set the value for a symbol
 */
set_symbol(ptr, value)
	char *ptr;
	unsigned value;
{
	ptr += (*ptr & SYMMASK);
	*++ptr = value >> 8;
	*++ptr = value;
}

/*
 * Set the flags for a symbol
 */
set_flags(ptr, value)
	char *ptr;
	unsigned value;
{
	ptr += (*ptr & SYMMASK) + 3;
	*ptr = value;
}

/*
 * Display the sorted symbol table
 */
dump_symbols()
{
	unsigned char *ptr, *hptr;
	unsigned int l, h, w;

	fprintf(lst_fp, "SYMBOL TABLE:\n\n");
	w = 0;
	for(;;) {
		for(hptr = symtab; hptr < symtop; hptr += (*hptr & SYMMASK) + 4)
			if(!(*hptr & SYMSORT))
				goto found;
		putc('\n', lst_fp);
		return;
found:	for(ptr = (*hptr & SYMMASK) + hptr + 4; ptr < symtop; ptr += (*ptr & SYMMASK) + 4) {
			if(*ptr & SYMSORT)
				continue;
			if(compsym(ptr, hptr) < 0)
				hptr = ptr; }
		*(ptr = hptr) |= SYMSORT;
		h = l = *ptr++ & SYMMASK;
		if((w + l + 7) >= pagew) {			/* Overrun page length */
			putc('\n', lst_fp);
			w = 0; }
		if(w) {								/* Not a start of line - separate */
			fprintf(lst_fp, "   ");
			w += 3; }
		while(l--)
			putc(*ptr++, lst_fp);
		w += (l = (h > 8) ? 24 : 8) + 5;	/* Calculate extended length */
		while(h++ < l)
			putc(' ', lst_fp);
		l = *ptr++ << 8;
		l |= *ptr++;
		fprintf(lst_fp, "-%04x", l); }
}

/*
 * Compare two symbol table entries
 */
compsym(sym1, sym2)
	char *sym1;
	char *sym2;
{
	int l, l1, l2;
	l1 = *sym1++ & SYMMASK;
	l2 = *sym2++ & SYMMASK;
	l = (l1 < l2) ? l1 : l2;
	do {
		if(*sym1 > *sym2)
			return 1;
		if(*sym1++ < *sym2++)
			return -1; }
	while(--l);
	if(l1 > l2)
		return 1;
	if(l1 < l2)
		return -1;
	return 0;
}

/*
 * Open a filename with the appriopriate extension &
 * report an error if not possible
 */
FILE *open_file(filename, extension, options)
	char *filename, *extension, *options;
{
	char buffer[100], *ptr, *dot;
	FILE *fp;

	dot = 0;

	for(ptr = buffer; *ptr = *filename; ++ptr) {	/* Copy filename */
		if(*filename == '.')
			dot = filename;
		else if(*filename == '\\')
			dot = 0;
		++filename; }

	if(!dot) {									/* No extension supplied */
		*ptr++ = '.';
		do
			*ptr++ = *extension;
		while(*extension++); }
	else
		*dot = 0;

	if(!(fp = fopen(buffer, options))) {
		fprintf(stderr,"Unable to access: '%s'\n", buffer);
		exit(-1); }

	return fp;
}

/*
 * Main program
 */
main(argc, argv)
	int argc;
	char *argv[];
{
	int i;
	char *ptr;

	if(argc < 2)
		abort("\nUse: asm186 <filename> [-cfiqst c=file l=file p=length w=width]\n\n?COPY.TXT 1983-2005 Dave Dunfield\n**See COPY.TXT**.\n");

	/* Parse for command line options */
	lfile = cfile = argv[1];
	for(i=2; i < argc; ++i) {
		if(*(ptr = argv[i]) == '-') {		/* Enable switch */
			while(*++ptr) switch(toupper(*ptr)) {
				case 'C' : casf = -1;		break;
				case 'F' : fulf = -1;		break;
				case 'I' : intel = -1;		break;
				case 'Q' : quif = -1;		break;
				case 'S' : symf = -1;		break;
				case 'T' : lfile = 0;		break;
				default: goto badopt; }
			continue; }
		if(*++ptr == '=') switch(toupper(*(ptr++ - 1))) {
			case 'L' : lfile = ptr;			continue;
			case 'C' : cfile = ptr;			continue;
			case 'P' : pagel=atoi(ptr)-1;	continue;
			case 'W' : pagew=atoi(ptr);		continue; }
	badopt:
		fprintf(stderr,"Invalid option: %s\n", argv[i]);
		exit(-1); }

	/* Open input & output files */
	asm_fp = open_file(argv[1], "ASM", "r");
	hex_fp = open_file(cfile, "HEX", "w");
	lst_fp = (lfile) ? open_file(lfile, "LST", "w") : stdout;
	strncpy(title, argv[1], sizeof(title)-1);	/* Initial title is filename */

	symtop = symtab;
	do {
		if(!quif)
			fprintf(stderr,"Pass %u... ", pass+1);
		for(i = pc = line = active_seg = 0; i < NUMSEG; ++i)
			seg_pc[i] = 0;
		while(fgets(ptr = inline, LINE_SIZE, asm_fp)) {
			error_flag = ocount = 0;
			daddr = pc;
			++line;
			i = 0;						/* Parse label */
			while(issymbol(*ptr)) {
				label[i] = chupper(*ptr++);
				if(i < SYMB_SIZE)
					++i; }
			label[i]=0;
			if(*ptr == ':')
				++ptr;
			while(isspace(*ptr))
				++ptr;
			if(((*ptr == '*') || (*ptr == ';')) && !i)
				inst_ptr = "";
			else {
				i = 0;						/* Parse instruction */
				while((*ptr != ' ') && (*ptr != '\t') && *ptr)
					instruction[i++] = toupper(*ptr++);
				instruction[i]=0;
				while(isspace(*ptr))
					++ptr;
				i = 0;						/* Parse operands */
				while(*ptr && (i < 79))
					operand[i++] = *ptr++;
				operand[i] = 0;
				if(*label && !pass)			/* Label this line */
					ptr = create_symbol(label, pc, active_seg);
				if(!lookup_inst())
					asmerr(2);
				input_ptr = operand;
				switch(*inst_ptr) {
case 1 :	/* No operand */
	i = 1;
	do
		instruction[ocount++] = inst_ptr[i];
	while(inst_ptr[++i]);
	break;
case 2 :	/* DEC ... */
	oper3();
doimul:
	switch(type) {
		case REGIS :
			test_register(value, 0x02);
			if((value > 7) && inst_ptr[3]) {
				instruction[ocount++] = inst_ptr[3] | (value & 7);
				break; }
		case INDEX :
		case VALUE :
			instruction[ocount++] = inst_ptr[1] | (length > 1);
			write_ea(inst_ptr[2]);
			break;
		default:
			asmerr(6); }
	break;
case 3 :	/* LDS ... */
	switch(oper4()) {
		case (REGIS<<8)+INDEX :
		case (REGIS<<8)+VALUE :
			test_register(value, 0x03);
			direction();
			instruction[ocount++] = inst_ptr[1];
			write_ea(value1);
			break;
		default:
			asmerr(6); }
	break;
case 4 :	/* ADC ... */
	dir = 0;
	switch(oper4()) {
		case (REGIS<<8)+IMMED :		/* AX,#nn */
			test_register(value, 0x02);
			if(!(value & 0x07)) {
				instruction[ocount++] = inst_ptr[1] | (length > 1);
				write_value1();
				break; }
		case (INDEX<<8)+IMMED :		/* [BX],#nn */
		case (VALUE<<8)+IMMED :		/* SYM,#nn */
			i = inst_ptr[2];
			if(length > 1) {
				i |= 0x01;
				if((inst_ptr[4] & 0xF0) && (value1 <= 127) && (value1 >= -128)) {
					i |= 0x02;
					--length; } }
			instruction[ocount++] = i;
			write_ea(inst_ptr[4]);
			write_value1();
			break;
		case (REGIS<<8)+REGIS :		/* BX,AX */
			test_register(value, 0x02);
			goto adc1;
		case (REGIS<<8)+INDEX :		/* AX,[BX] */
		case (REGIS<<8)+VALUE :		/* AX,nn */
			direction();
		case (INDEX<<8)+REGIS :		/* [BX],AX */
		case (VALUE<<8)+REGIS :		/* 123,AX */
			if(!inst_ptr[4])		/* Special case for "test" */
				dir = 0;
		adc1:
			test_register(value1, 0x02);
			instruction[ocount++] = (inst_ptr[3] | (length > 1)) ^ dir;
			write_ea(value1);
			break;
		default:
			asmerr(6); }
	break;
case 5 :	/* MOV .. */
	dir = 0;
	switch(oper4()) {
		case (REGIS<<8)+IMMED :
			test_register(value, 0x02);
			instruction[ocount++] = 0xB0 + (value & 0x07) + ((length > 1) * 0x08);
			write_value1();
			break;
		case (VALUE<<8)+IMMED :
		case (INDEX<<8)+IMMED :
			instruction[ocount++] = 0xC6 + (length > 1);
			write_ea(0);
			write_value1();
			break;
		case (REGIS<<8)+INDEX :		/* AX,[BX] */
		case (REGIS<<8)+VALUE :		/* AX,nn */
			direction();
		case (INDEX<<8)+REGIS :		/* [BX],AX */
		case (VALUE<<8)+REGIS :		/* 123,AX */
			if(value1 > 15) {		/* Segment register */
				instruction[ocount++] = 0x8C | dir;
				write_ea(value1); }
			else if((type == VALUE) && !(value1 & 0x07)) {
				instruction[ocount++] = (0xA0 | length > 1) ^ dir ^ 0x02;
				instruction[ocount++] = value;
				instruction[ocount++] = value >> 8; }
			else {
				instruction[ocount++] = (0x88 | (length > 1)) ^ dir;
				write_ea(value1); }
			break;
		case (REGIS<<8)+REGIS :		/* BX,AX */
			if(value > 15) {
				test_register(value1, 0x03);
				direction();
				i = 0x8E; }
			else if(value1 > 15) {
				test_register(value, 0x03);
				i = 0x8C; }
			else
				i = 0x88 | (length > 1);
			instruction[ocount++] = i;
			write_ea(value1);
			break;
		default:
			asmerr(6); }
	break;
case 6 :	/* short jumps */
	if(oper2() == VALUE) {
		value -= pc + 2;
		if((value > 127) || (value < -128))
			asmerr(16);
		instruction[ocount++] = inst_ptr[1];
		instruction[ocount++] = value; }
	else
		asmerr(6);
	break;
case 7 :	/* JMP and CALL */
	i = 0;
	type = oper2();
	if(test_next(':')) {
		if(type != VALUE)
			asmerr(6);
		i = -1;
		value1 = value;
		type = oper2(); }
	switch(type) {
		case VALUE :
			if(length != 2) {
				if(i) {
					instruction[ocount++] = inst_ptr[2];
					instruction[ocount++] = value;
					instruction[ocount++] = value >> 8;
					length = 2;
					write_value1();
					break; }
				if(length == 1) {
					if(inst_ptr[1] != 0xE9)
						asmerr(6);
					value -= pc + 2;
					if((value > 127) || (value < -128))
						asmerr(16);
					instruction[ocount++] = 0xEB;
					instruction[ocount++] = value;
					break; }
				value -= pc + 3;
				instruction[ocount++] = inst_ptr[1];
				instruction[ocount++] = value;
				instruction[ocount++] = value >> 8;
				break; }
			goto doindex;
		case REGIS :
			test_register(value, 0x03);
		case INDEX :
		doindex:
			instruction[ocount++] = 0xFF;
			write_ea(inst_ptr[i ? 4 : 3]);
			break;
		default:
			asmerr(6); }
	break;
case 8 :	/* RET and RETF */
	if((*input_ptr ==';') || !*input_ptr)
		instruction[ocount++] = inst_ptr[1];
	else {
		get_constant(-1);
		instruction[ocount++] = inst_ptr[2];
		instruction[ocount++] = value;
		instruction[ocount++] = value >> 8; }
	break;
case 9 :	/* PUSH and POP */
	switch(oper2()) {
		case REGIS :
			chkterm();
			chklz();
			if(length == 1)
				asmerr(5);
			if(value > 15)
				instruction[ocount++] = inst_ptr[1] | ((value & 7) << 3);
			else
				instruction[ocount++] = inst_ptr[2] | (value & 7);
			break;
		case VALUE :
		case INDEX :
			chkterm();
			chklz();
			if(length == 1)
				asmerr(5);
			instruction[ocount++] = inst_ptr[3];
			write_ea(inst_ptr[4]);
			break;
		case IMMED :
			if(inst_ptr[1] == 0x06) {	/* 80186: PUSH Immediate */
				if((value & 0xFF00) && !length)
					length = 2;
				instruction[ocount++] = (length != 2) ? 0x6A : 0x68;
				instruction[ocount++] = value;
				if(length > 1)
					instruction[ocount++] = value >> 8;
				break; }
		default:
			asmerr(6); }
	break;
case 10 :	/* String instructions */
	if(oper3() == VALUE)
		instruction[ocount++] = inst_ptr[1] | (length > 1);
	else
		asmerr(6);
	break;
case 11 :	/* Shifts & Rotates */
	if((type = oper2()) == IMMED)
		asmerr(6);
	if(type == REGIS)
		test_register(value, 0x02);
	chklz();
	i = length > 1;
	comma();
	direction();
	type = oper2();
	direction();
	if(type1 == REGIS) {
		if(value1 != 1)
			asmerr(20);
		instruction[ocount++] = inst_ptr[1] | 2 | i;
		write_ea(inst_ptr[2]); }
	else if(type1 == VALUE) {
		if(value1 > 255)
			asmerr(20);
		instruction[ocount++] = inst_ptr[1] | i;
		write_ea(inst_ptr[2]);
		if(value1 != 1) {
			instruction[ocount++] = value1;
			instruction[0] &= ~0x10; } }
	else
		asmerr(20);
	break;
case 12 :	/* INT Statement */
	get_constant(255);
	if((value == 3) && (length < 2))
		instruction[ocount++] = 0xCC;
	else {
		instruction[ocount++] = 0xCD;
		instruction[ocount++] = value; }
	break;
case 13 :	/* ESC opcode */
	switch(oper4()) {
		case (VALUE<<8)+REGIS :
			test_register(value1, 0x02);
		case (VALUE<<8)+VALUE :
		case (VALUE<<8)+INDEX :
			direction();
			instruction[ocount++] = 0xD8 | ((value1 >> 3) & 7);
			write_ea(value1);
			break;
		default:
			asmerr(6); }
		break;
case 14 :	/* IN and OUT */
	type = oper2();
	i = length;
	comma();
	direction();
	type = oper2();
	if(!inst_ptr[3]) {
		direction();
		length = i; }
	if((type != REGIS) || (value & 0x17))
		asmerr(22);
	switch(type1) {
		case VALUE :
			if((value1 < 0 ) || (value1 > 255))
				asmerr(16);
			instruction[ocount++] = inst_ptr[1] | (length > 1);
			instruction[ocount++] = value1;
			break;
		case REGIS :
			if(value1 != 10)
				asmerr(21);
			instruction[ocount++] = inst_ptr[2] | (length > 1);
			break;
		default:
			asmerr(21); }
	break;
case 15 :	/* XCHG */
	switch(oper4()) {
		case (REGIS<<8)+REGIS :
			test_register(value, 0x02);
			test_register(value1, 0x02);
			if((value > 7) && !(value & 7))
				instruction[ocount++] = 0x90 | (value1 & 7);
			else if((value1 > 7) && !(value1 & 7))
				instruction[ocount++] = 0x90 | (value & 7);
			else
				goto doxchg;
			break;
		case (REGIS<<8)+INDEX :
		case (REGIS<<8)+VALUE :
			direction();
		case (INDEX<<8)+REGIS :
		case (VALUE<<8)+REGIS :
			test_register(value1, 0x02);
		doxchg:
			instruction[ocount++] = inst_ptr[1] | (length > 1);
			write_ea(value1);
			break;
		default:
			asmerr(6); }
	break;
case 16 :	/* 80186: ENTER */
	if(oper2() == VALUE) {
		comma();
		value1 = value;
		if(oper2() == VALUE) {
			instruction[0] = 0xC8;
			instruction[1] = value1;
			instruction[2] = value1 >> 8;
			instruction[3] = value;
			ocount = 4;
			break; } }
	asmerr(6);
	break;
case 17 :	/* 80186: IMUL */
	type = oper2();		/* Get register */
	if(*input_ptr != ',') {
		chklz();
		goto doimul; }
	if(type != REGIS)
		asmerr(1);
	test_register(i = value, 0x03);
	comma();
	type = oper2();
	if(length != 2)
		asmerr(14);
	++ocount;
	write_ea(i);
	comma();
	if(oper2() != IMMED)
		asmerr(6);
	if((value & 0xFF00) && !length)
		length = 2;
	i = 0x6B;
	instruction[ocount++] = value;
	if(length > 1) {
		instruction[ocount++] = value >> 8;
		i = 0x69; }
	instruction[0] = i;
	break;
case 100 :	/* EQU statement */
	if(pass)
		daddr = get_constant(-1);
	else {
		set_symbol(ptr, get_constant(-1));
		goto err1st; }
	break;
case 101 :	/* ORG statement */
	wrrec();
	daddr = pc = get_constant(-1);
err1st:
	if(error_flag && !pass)
		show_error(error_flag);
	break;
case 102 :	/* SEG directive */
	seg_pc[active_seg] = pc;
	if(seg_size[active_seg] < pc)
		seg_size[active_seg] = pc;
	daddr = pc = seg_pc[active_seg = get_constant(NUMSEG-1) & (NUMSEG-1)];
	break;
case 103 :	/* DB statement */
	do
		instruction[ocount++] = get_constant(-1);
	while(test_next(','));
	break;
case 104 :	/* DW statement */
	do {
		get_constant(-1);
		if(inst_ptr[1]) {
			instruction[ocount++] = value;
			instruction[ocount++] = value >> 8; }
		else {
			instruction[ocount++] = value >> 8;
			instruction[ocount++] = value; } }
	while(test_next(','));
	break;
case 105 :	/* DS statement */
	wrrec();
	pc += get_constant(-1);
	goto err1st;
case 106 :	/* STR statement */
	i = *input_ptr++;
	while(*input_ptr != i) {
		if(!*input_ptr) {
			asmerr(13);
			break; }
		instruction[ocount++] = *input_ptr++; }
	switch(inst_ptr[1]) {
		case 1 :
			instruction[ocount++] = 0;
			break;
		case 2 :
			instruction[ocount-1] |= 0x80; }
	break;
case 107 :	/* PUBLIC statement */
	if(pass) do {
		ptr = input_ptr;
		while(!isterm(i = *input_ptr))
			*input_ptr++ = chupper(i);
		*input_ptr++ = 0;
		if(ptr = lookup_symbol(ptr, &value))
			set_flags(ptr, sflags | SPUB);
		else
			asmerr(17); }
	while(i == ',');
	break;
case 108 :	/* EXTERN statement */
	if(!pass) do {
		ptr = input_ptr;
		while(!isterm(i = *input_ptr))
			*input_ptr++ = chupper(i);
		*input_ptr++ = 0;
		create_symbol(ptr, 0, SEXT | active_seg); }
	while(i == ',');
	break;
case 109 :	/* END directive */
	goto end;
case 120 :	/* PAGE directive */
	lcount = 9999;
	break;
case 121 :	/* TITLE directive */
	strncpy(title, operand, sizeof(title)-1);
	break;
case 122 :	/* SPACE directive */
	if(pass && fulf && !list) {
		putc('\n', lst_fp);
		++lcount; }
	break;
case 123 :	/* LIST directive */
	if(list)
		--list;
	break;
case 124 :	/* NOLIST directive */
	++list;
	break;
case 125 :	/* LINE directive */
	line = get_constant(-1);
}
			/* Check for out of PHASE errors & duplicated symbols */
			if(*label && pass) {
				lookup_symbol(label, &i);
				if(sflags & SDUP)
					error_flag = 18;
				else if(i != daddr) {
					show_error(0);
					break; } } }

		if(pass) {
	/* Generate formatted output listing */
			if((fulf && (*inst_ptr < 120) && !list) || error_flag) {
				if(++lcount >= pagel)
					write_title();
				fprintf(lst_fp, "%04x ", daddr);
				for(i=0; i < 6; ++i)
					if(i < ocount)
						fprintf(lst_fp," %02x", instruction[i]);
					else
						fputs("   ", lst_fp);
				fprintf(lst_fp, " %c%5u  %s\n", (ocount <= 6) ? ' ' : '+', line, inline);
				if(error_flag)
					show_error(error_flag); }
	/* Write the code file (just output binary for now) */
			for(i=0; i<ocount; ++i) {
				if(!ocnt) {			/* first byte of record */
					daddr = pc + i;
					outrec[ocnt++]=daddr>>8;
					outrec[ocnt++]=daddr; }
				outrec[ocnt++]=instruction[i];
				if(ocnt>33) wrrec(); } }	/* record is full, write it */
			pc += ocount; }

end:
		if(seg_size[active_seg] < pc)
			seg_size[active_seg] = pc;
		rewind(asm_fp); }
	while(++pass < 2);

	wrrec();
	if(intel) fprintf(hex_fp,":00000001FF\n");
	else fprintf(hex_fp,"S9030000FC\n");

	if(error_count)
		fprintf(lst_fp,"\n %u error(s) occurred in this assembly.\n",error_count);

	if(!quif)
		fprintf(stderr,"%u error(s).\n",error_count);

/* display the symbol table */
	if(symf) {
		write_title();
		dump_symbols(); }

	fclose(asm_fp);
	fclose(hex_fp);
	fclose(lst_fp);

	return error_count ? -2 : 0;
}
comma()
{
	if(!test_next(','))
		asmerr(3);
}

/*
 * Write out title
 */
write_title()
{
	int w;
	char *ptr;

	if(pcount > 1)
		putc('\f',lst_fp);
	lcount = 1;
	fprintf(lst_fp,"DUNFIELD 80186 ASSEMBLER: ");
	ptr = title;
	for(w = 36; w < pagew; ++w)
		putc(*ptr ? *ptr++ : ' ', lst_fp);
	fprintf(lst_fp,"PAGE: %u\n\n", pcount);
	++pcount;
}

/*
 * Reverse operands, setting diretion flag
 */
direction()
{
	int i;

	i = type;	type = type1;	type1 = i;
	i = value;	value = value1;	value1 = i;
	i = xvalue; xvalue=xvalue1; xvalue1 = i;
	dir = 0x02;
}

/*
 * Test a register for valid ranges
 */
test_register(reg, flags)
	int reg;
	char flags;
{
	if(pass) {
		if((flags & 0x01) && (reg < 8))		/* No 8 bit registers */
			asmerr(5);
		if((flags & 0x02) && (reg > 15))	/* No segment registers */
			asmerr(8); }
}

/*
 * Writes the value of value1
 */
write_value1()
{
	instruction[ocount++] = value1;
	if(length > 1)
		instruction[ocount++] = value1 >> 8;
}

/*
 * Build the EA value from parsed data
 */
write_ea(xxx)
	int xxx;
{
	int dflag;

	dflag = 0;
	xxx = (xxx & 7) << 3;
	switch(type) {
		case REGIS :	/* Register reference */
			xxx |= 0xC0 + (value & 7);
			break;
		case VALUE :	/* Direct memory reference */
			xxx |= 6;
			dflag = 2;
			break;
		case INDEX :	/* Indexed addressing mode */
			if((value > 127) || (value < -128)) {
				xxx |= 0x80;	/* 16 bit displacement */
				dflag = 2; }
			else if(value || (xvalue == 6)) {
				xxx |= 0x40;	/* 8 bit displacement */
				dflag = 1; }
			xxx |= xvalue;
			break;
		default:
			asmerr(6); }
		instruction[ocount++] = xxx;
		if(dflag)
			instruction[ocount++] = value;
		if(dflag > 1)
			instruction[ocount++] = value >> 8;
	return dflag+1;
}

/*
 * Get a constant value
 */
get_constant(limit)
	unsigned limit;
{
	if(oper2() != VALUE)
		asmerr(4);
	if(value > limit) {
		value = 0;
		asmerr(16); }
	return value;
}

/*
 * Get two complete operands with error checking
 */
oper4()
{
	int l, v, x;

	type = oper2();
	l = length;
	v = value;
	x = xvalue;
	comma();
	type1 = oper2();
	chkterm();

	/* Make sure that the lengths are compatible */
	if(!(length || l))
		asmerr(15);
	else if(length && l && (length != l))
		asmerr(14);
	else if(l)
		length = l;

	value1 = value; value = v;
	xvalue1 = xvalue; xvalue = x;

	return (type << 8) + type1;
}

/*
 * Check for zero length & generate error
 */
chklz()
{
	if(!length)
		asmerr(15);
}

/*
 * Check for non-terminator & generate error
 */
chkterm()
{
	if(!isterm(*input_ptr))
		asmerr(1);
}

/*
 * Get a one complete operand with error checking
 */
oper3()
{
	type = oper2();
	chkterm();
	chklz();
	return type;
}

/*
 * Get and evaluate a single operand expression
 */
oper2()
{
	int i, j;
	char o, immed;

	immed = test_next('#');

	if(isterm(*input_ptr))
		asmerr(3);

	if(oper1(&value, &length, 0) == REGIS) {	/* Simple register */
		if((value > 15) && (*input_ptr == ':')) {
			++input_ptr;
			instruction[ocount++] = 0x26 | ((value & 3) << 3);
			i = oper1(&value, &length, 0);
			if((i != VALUE) && (i != INDEX))
				asmerr(7); }
		else {
			if(!isterm(*input_ptr))
				asmerr(9);
			return REGIS; } }

	/* A value of a memory reference */
	while(*input_ptr && !isterm(*input_ptr)) {
		if(isoper(o = *input_ptr)) {
			++input_ptr;
			if(oper1(&i, &j, 0) == REGIS)
				asmerr(9);
			if(length && j && (length != j))
				asmerr(14);
			if(j)
				length = j;
			switch(o) {
				case '+' : value += i;	break;
				case '-' : value -= i;	break;
				case '*' : value *= i;	break;
				case '/' : value /= i;	break;
				case '%' : value %= i;	break;
				case '&' : value &= i;	break;
				case '|' : value |= i;	break;
				case '^' : value ^= i;	break;
				case '<' : value <<=i;	break;
				case '>' : value >>=i; } }
		else if(o == '[') {				/* Indexing operation */
			++input_ptr;
			xvalue = test_index(255);
			if(test_next('+'))			/* Two registers */
				xvalue = test_index(xvalue);
			if(!test_next(']'))
				asmerr(1);
			if(immed)
				asmerr(10);
			return INDEX; }
		else {
			asmerr(1);
			break; } }

	return immed ? IMMED : VALUE;
}

/*
 * Scan for a single operand
 */
oper1(val, len, l)
	int *val, *len;
	unsigned l;
{
	unsigned base, v, v1, l1;
	char symbol[SYMB_SIZE+1], c, *ptr;

	v = 0;
	for(;;) switch(*input_ptr++) {
		case '(' :			/* Nested expression */
			v1 = value;
			l1 = length;
			if(oper2() != VALUE)
				asmerr(4);
			v = value;
			l = length;
			value = v1;
			length = l1;
			if(!test_next(')'))
				asmerr(1);
			goto retval;
		case '<' :			/* Force 8 bit value */
			l = 1;
			break;
		case '>' :			/* Force 16 bit value */
			l = 2;
			break;
		case '-' :			/* Unary minus */
			base = oper1(&v, &l, l);
			*val = -v;
			*len = l;
			return base;
		case '~' :			/* Unary NOT */
			base = oper1(&v, &l, l);
			*val = ~v;
			*len = l;
			return base;
		case '=' :			/* Swap high and low */
			base = oper1(&v, &l, l);
			*val = (v<<8)+(v>>8);
			*len = l;
			return base;
		case '$' :	base = 16;	goto getn;	/* Hexidecimal number */
		case '%' :	base = 2;	goto getn;	/* Binary number */
		case '@' :	base = 8;	goto getn;	/* Octal number */
		case '0' :	case '1' :	case '2' :	/* Decimal number */
		case '3' :	case '4' :	case '5' :
		case '6' :	case '7' :	case '8' :	case '9' :
			base = 0;
			--input_ptr;
		getn:
			v1 = 0;
			while(isalnum(c = toupper(*input_ptr))) {	/* Accept till end */
				++input_ptr;
				symbol[v1++] = (c < 'A') ? c : c - 7; }
			if((base == 16) && !v1)
				v = pc;
			else {
				if(!base) {
					base = 10;
					switch(symbol[v1-1]) {
						case 'H'-7 :	base = 16;	goto deci;
						case 'T'-7 :
						case 'D'-7 :	base = 10;	goto deci;
						case 'Q'-7 :
						case 'O'-7 :	base = 8;	goto deci;
						case 'B'-7 :	base = 2;
						deci: --v1;	break; } }
				if(!v1)
					asmerr(1);
				for(ptr = symbol; v1; --v1) {
					if((c = *ptr++ - '0') >= base)
						asmerr(1);
					v = (v * base) + c; } }
			goto retval;
		case '\'' :			/* Quoted value */
			while((c = *input_ptr) && (c != '\'')) {
				++input_ptr;
				v = (v << 8) + c; }
			if(*input_ptr)
				++input_ptr;
			else
				asmerr(13);
			goto retval;
		case '*' :			/* Program counter */
			v = pc;
			goto retval;
		default :			/* Symbol value */
			--input_ptr;
			while(issymbol(c = *input_ptr)) {
				++input_ptr;
				symbol[v++] = chupper(c); }
			symbol[v] = 0;
			if(v) {
				for(v = 0; v < NUMREG; ++v)
					if(!strcmp(symbol, registers[v])) {
						if(l)
							asmerr(12);
						*val = v;
						*len = (v > 7) + 1;
						return REGIS; }
				if(!lookup_symbol(symbol, &v))
					asmerr(17); }
		retval:
			*val = v;
			*len = l;
			return VALUE; }
}

/*
 * Test the next character in the stream
 */
test_next(c)
	char c;
{
	if(*input_ptr == c) {
		++input_ptr;
		return -1; }
	return 0;
}

/*
 * Convert character to upper case if NOT case sensitive
 */
chupper(c)
	char c;
{
	return casf ? c : ((c >= 'a') && (c <= 'z')) ? c - ('a'-'A') : c;
}

/*
 * Test for a valid INDEX register combination
 */
test_index(reg)
	int reg;
{
	int v, l;

	if(oper1(&v, &l, 0) == VALUE)	/* Invalid, force an error */
		reg = 0;
	switch((reg << 8) + v) {
		case 0xFF0B :	/* [BX] */
			return 7;
		case 0xFF0D :	/* [BP] */
			return 6;
		case 0xFF0E :	/* [SI] */
			return 4;
		case 0xFF0F :	/* [DI] */
			return 5;
		case 0x070E :	/* [BX+SI] */
		case 0x040B :	/* [SI+BX] */
			return 0;
		case 0x070F :	/* [BX+DI] */
		case 0x050B :	/* [DI+BX] */
			return 1;
		case 0x060E :	/* [BP+SI] */
		case 0x040D :	/* [SI+BP] */
			return 2;
		case 0x060F :	/* [BP+DI] */
		case 0x050D :	/* [DI+BP] */
			return 3;
		default:
			asmerr(11); }
}

/*
 * Report an assembly error
 */
asmerr(errno)
	int errno;
{
	if(!error_flag)
		error_flag = errno;
}

/*
 * Display an error
 */
show_error(errno)
	int errno;
{
	fprintf(lst_fp,"** Error %u in line %u : %s\n", errno, line, error_text[errno]);
	++error_count;
}

/*
 * Test for a valid symbol name character
 */
issymbol(c)
	char c;
{
	switch(c) {			/* Special allowed characters */
		case '_' :
		case '?' :
		case '!' :
		case '.' :
			return 1; }

	return isalnum(c);
}

/*
 * Test for a valid operation character
 */
isoper(c)
	char c;
{
	char d, *ptr;

	ptr = "+-*/%&|^<>";		/* Table of valid operators */
	while(d = *ptr++)
		if(c == d)
			return -1;

	return 0;
}

/*
 * Test for terminater character
 */
isterm(c)
	char c;
{
	switch(c) {
		case 0 :
		case ' ' :
		case '\t':
		case ',' :
		case ':' :
		case ')' :
			return 1; }
	return 0;
}

/*
 * Lookup instruction in table
 */
lookup_inst()
{
	char *ptr1, *ptr2;

	ptr1 = inst_table;
	do {
		ptr2 = instruction;
		while(*ptr1 == *ptr2) {
			++ptr1;
			++ptr2; }
		if(((*ptr1 & 0x7f) == *ptr2) && !*(ptr2+1)) {
			inst_ptr = ptr1 + 1;
			return 1; }
		while(!(*ptr1++ & 0x80)); }
	while(*(ptr1 += IDATSIZE));

	inst_ptr = ptr1;
	return !*instruction;
}

/*******************************/
/* write record to output file */
/*******************************/
wrrec()
{
	register unsigned i, chk, chr;

	if(pass && ocnt) {
		xhex(ocnt-2);
		if(intel) {					/* intel hex format */
			chk = outrec[0] + outrec[1] + ocnt - 2;
			fprintf(hex_fp,":%02x%02x%02x00", ocnt-2, outrec[0] & 255, outrec[1] & 255);
			for(i=2; i<ocnt; ++i) {
				fprintf(hex_fp,"%02x", chr = outrec[i] & 255);
				chk += chr; }
			fprintf(hex_fp,"%02x\n", 255 & (0-chk)); }
		else {						/* motorola hex format */
			chk = ocnt + 1;
			fprintf(hex_fp,"S1%02x", ocnt + 1);
			for(i=0; i<ocnt; ++i) {
				fprintf(hex_fp,"%02x", chr = outrec[i] & 255);
				chk += chr; }
			fprintf(hex_fp,"%02x\n", 255 & ~chk); }
		ocnt = 0; }
}
