/*
 * 80C320 Cross Assembler
 *
 * ?COPY.TXT 1983-2005 Dave Dunfield
 * **See COPY.TXT**.
 */
#include <stdio.h>
#include <ctype.h>

#include "xasm.h"

#define	BIT_TYPE	0x40		/* Indicate BIT type equate */

/* 80C320 opcode table */
static char opcodes[] = {
	'N','O','P',		0x81,	0x00,
	'R','E','T',		0x81,	0x22,
	'R','E','T','I',	0x81,	0x32,
	'M','U','L',		0x81,	0xa4,
	'D','I','V',		0x81,	0x84,
	'A','D','D',		0x82,	0x24,
	'A','D','D','C',	0x82,	0x34,
	'S','U','B','B',	0x82,	0x94,
	'M','O','V',		0x83,	0x74,
	'A','N','L',		0x84,	0x52,
	'O','R','L',		0x84,	0x42,
	'X','C','H',		0x85,	0xc5,
	'X','R','L',		0x86,	0x62,
	'X','C','H','D',	0x87,	0xd6,
	'A','C','A','L','L',0x88,0x11,
	'A','J','M','P',	0x88,	0x01,
	'C','J','N','E',	0x89,	0xb4,
	'D','A',			0x8a,	0xd4,
	'R','L',			0x8a,	0x23,
	'R','L','C',		0x8a,	0x33,
	'R','R',			0x8a,	0x03,
	'R','R','C',		0x8a,	0x13,
	'S','W','A','P',	0x8a,	0xc4,
	'S','J','M','P',	0x8b,	0x80,
	'J','C',			0x8b,	0x40,
	'J','N','C',		0x8b,	0x50,
	'J','N','Z',		0x8b,	0x70,
	'J','Z',			0x8b,	0x60,
	'P','O','P',		0x8c,	0xd0,
	'P','U','S','H',	0x8c,	0xc0,
	'C','L','R',		0x8d,	0x00,
	'C','P','L',		0x8e,	0x00,
	'S','E','T','B',	0x8f,	0xd0,
	'J','B',			0x90,	0x20,
	'J','B','C',		0x90,	0x10,
	'J','N','B',		0x90,	0x30,
	'D','J','N','Z',	0x91,	0xd5,
	'J','M','P',		0x92,	0x73,
	'L','C','A','L','L',0x93,0x12,
	'L','J','M','P',	0x93,	0x02,
	'M','O','V','X',	0x94,	0xe0,
	'M','O','V','C',	0x95,	0x83,
	'I','N','C',		0x96,	0x04,
	'D','E','C',		0x97,	0x14,
/* directives */
	'E','Q','U',		228,	0,
	'B','I','T',		228,	1,
	'D','A','T','A',	228,	2,
	'O','R','G',		229,	0,
	'D','B',			230,	0,
	'D','W',			231,	-1,
	'D','R','W',		231,	0,
	'S','T','R',		232,	0,
	'S','T','R','H',	232,	1,
	'S','T','R','Z',	232,	2,
	'D','S',			233,	0,
	'F','C','B',		230,	0,
	'F','D','B',		231,	-1,
	'R','D','B',		231,	0,
	'F','C','C',		232,	0,
	'F','C','C','H',	232,	1,
	'F','C','C','Z',	232,	2,
	'R','M','B',		233,	0,
	'E','N','D',		234,	0,
	'P','A','G','E',	248,	0,
	'T','I','T','L','E',249,0,
	'S','P','A','C','E',250,0,
	'L','I','S','T',	251,	0,
	'N','O','L','I','S','T',252,0,
	'L','I','N','E',	253,	0,
	0 };	/* end of table */

			/* A,A A,# A,D A,@ A,R		#,A #,# #,D #,@ #,R		D,A D,# D,D D,@ D,R		@,A @,# @,D @,@ @,R		R,A R,# R,D R,@ R,R */
char it1[] = {	-1,	0,	1,	2,	4,		-1,	-1,	-1,	-1,	-1,		-1,	-1,	-1,	-1,	-1,		-1,	-1,	-1,	-1,	-1,		-1,	-1,	-1,	-1,	-1 };
char it2[] = {	-1,	0, 113,114,	116,	-1,	-1,	-1,	-1,	-1,		129,1,	17,	18,	20,		130,2,	50,	-1,	-1,		132,4,	52,	-1,	-1 };
char it3[] = {	-1,	2,	3,	4,	6,		-1,	-1,	-1,	-1,	-1,		0,	1,	-1,	-1,	-1,		-1,	-1,	-1,	-1,	-1,		-1,	-1,	-1,	-1,	-1 };
char it4[] = {	-1,	-1,	0,	1,	3,		-1,	-1,	-1,	-1,	-1,		-1,	-1,	-1,	-1,	-1,		-1,	-1,	-1,	-1,	-1,		-1,	-1,	-1,	-1,	-1 };
char it5[] = {	-1,	-1,	-1,	0,	-1,		-1,	-1,	-1,	-1,	-1,		-1,	-1,	-1,	-1,	-1,		-1,	-1,	-1,	-1,	-1,		-1,	-1,	-1,	-1,	-1 };
char it6[] = {	-1,	0,	1,	-1,	-1,		-1,	-1,	-1,	-1,	-1,		-1,	-1,	-1,	-1,	-1,		-1,	2,	-1,	-1,	-1,		-1,	4,	-1,	-1,	-1 };
char it7[] = {	-1,	-1,	-1,	2,	-1,		-1,	-1,	-1,	-1,	-1,		-1,	-1,	-1,	-1,	-1,		18,	-1,	-1,	-1,	-1,		-1,	-1,	-1,	-1,	-1 };

/* internal register name and address tables */
char *iregnam[] = {
	"A", "B", "PSW", "SP", "DPL", "DPH", "DPL1", "DPH1", "DPS",
	"P0", "P1", "P2", "P3", "IP", "IE", "EIP", "EIE", "EXIF",
	"TMOD", "TCON", "TH0", "TL0", "TH1", "TL1", "TH2", "TL2", "T2MOD", "T2CON",
	"SCON", "SBUF", "PCON", "RCAP2H", "RCAP2L", "ACC",
	"SCON0", "SBUF0", "SADDR0", "SADEN0", "SCON1", "SBUF1", "SADDR1", "SADEN1",
	"CKCON", "TA", "WDCON",
	0 };
int iregadr[] = {
	0xE0, 0xF0, 0xD0, 0x81, 0x82, 0x83, 0x84, 0x85, 0x86,
	0x80, 0x90, 0xA0, 0xB0, 0xB8, 0xA8, 0xF8, 0xE8, 0x91,
	0x89, 0x88, 0x8C, 0x8A, 0x8D, 0x8B, 0xCD, 0xCC, 0xC9, 0xC8,
	0x98, 0x99, 0x87, 0xCB, 0xCA, 0xE0,
	0x98, 0x99, 0xA9, 0xB9, 0xC0, 0xC1, 0xAA, 0xBA,
	0x8E, 0xC7, 0xD8
	};

/* error messages */
static char *etext[] = { "?",
	"Unknown instruction",
	"Out of range",
	"Invalid addressing mode",
	"Invalid register specification",
	"Undefined symbol",
	"Invalid expression syntax",
	"Invalid argument format",
	"Improperly delimited string" } ;

/* Symbol table & free pointer */
char symtab[SYMB_POOL], *symtop;

/* misc. global variables */
char buffer[LINE_SIZE+1], label[SYMB_SIZE+1], title[50], operand[80], optr,
	itype, opcode, bit, bit_comp;
unsigned char instruction[80], outrec[35];

char symf=0, intel=0, fulf=0, quietf=0, nlist=0, casf=0, ljmp = -1;

unsigned addr, line=0, pcount=0, lcount=0, value, length, emsg, ecnt,
	ocnt=0, pagel=59, pagew=80;

FILE *asm_fp, *hex_fp, *lst_fp;

/*
 * Define a symbol in the symbol table
 */
char *create_symbol(symbol, value)
	char *symbol;
	int value;
{
	register char *ptr, *ptr1;

	ptr = ptr1 = symtop;
	while(*symbol)
		*++ptr = *symbol++;
	*ptr1 = ptr - ptr1;
	*++ptr = value >> 8;
	*++ptr = value;
	if(ptr < (symtab+sizeof(symtab))) {
		symtop = ptr + 1;
		return ptr1; }
	return 0;
}

/*
 * Lookup a symbol in the symbol table
 */
char *lookup_symbol(symbol)
	char *symbol;
{
	register int l;
	register unsigned char *ptr, *ptr1;
	char *ptr2;
	static char ireg_type = 0;

	l = 0;
	while(ptr2 = iregnam[l]) {
		ptr1 = symbol;
		while(*ptr1)
			if(*ptr1++ != *ptr2++)
				goto next;
		if(!*ptr2) {
			value = iregadr[l];
			return &ireg_type; }
	next: ++l; }
	
	ptr = symtab;
again:
	if((ptr2 = ptr) >= symtop) {
		value = 0x8888;
		return 0; }
	ptr1 = symbol;
	l = *ptr++ & SYMMASK;
	while(l--) {
		if(*ptr1++ != *ptr++) {
			ptr += l + 2;
			goto again; } }
	if(*ptr1) {
		ptr += 2;
		goto again; }
	value = *ptr++ << 8;
	value |= *ptr;
	return ptr2;
}

/*
 * Set the value for a symbol in the symbol table
 */
set_symbol(ptr, value)
	char *ptr;
	unsigned value;
{
	if(ptr) {
		ptr += (*ptr & SYMMASK);
		*++ptr = value >> 8;
		*++ptr = value; }
}

/*
 * Display the symbol table (sorted)
 */
dump_symbols()
{
	unsigned char *ptr, *hptr;
	unsigned int l, h, w;

	fprintf(lst_fp, "SYMBOL TABLE:\n\n");
	w = 0;
	for(;;) {
		for(hptr = symtab; hptr < symtop; hptr += (*hptr & SYMMASK) + 3)
			if(!(*hptr & SYMSORT))
				goto found;
		putc('\n', lst_fp);
		return;
found:	for(ptr = (*hptr & SYMMASK) + hptr + 3; ptr < symtop; ptr += (*ptr & SYMMASK) + 3) {
			if(*ptr & SYMSORT)
				continue;
			if(compsym(ptr, hptr) < 0)
				hptr = ptr; }
		*(ptr = hptr) |= SYMSORT;
		h = l = *ptr++ & SYMMASK;
		if((w + l + 7) >= pagew) {			/* Overrun page length */
			putc('\n', lst_fp);
			w = 0; }
		if(w) {								/* Not a start of line - separate */
			fprintf(lst_fp, "   ");
			w += 3; }
		while(l--)
			putc(*ptr++, lst_fp);
		w += (l = (h > 8) ? 24 : 8) + 5;	/* Calculate extended length */
		while(h++ < l)
			putc(' ', lst_fp);
		l = *ptr++ << 8;
		l |= *ptr++;
		fprintf(lst_fp, "-%04x", l); }
}

/*
 * Compare two symbol table entries
 */
compsym(sym1, sym2)
	char *sym1;
	char *sym2;
{
	int l, l1, l2;
	l1 = *sym1++ & SYMMASK;
	l2 = *sym2++ & SYMMASK;
	l = (l1 < l2) ? l1 : l2;
	do {
		if(*sym1 > *sym2)
			return 1;
		if(*sym1++ < *sym2++)
			return -1; }
	while(--l);
	if(l1 > l2)
		return 1;
	if(l1 < l2)
		return -1;
	return 0;
}

/************************************/
/* get a character from the operand */
/************************************/
char getchr()
{
	register char chr;

	if(chr=operand[optr]) ++optr;
	return(chupper(chr));
}

/*
 * Convert character to upper case if NOT case sensitive
 */
chupper(c)
	char c;
{
	return casf ? c : ((c >= 'a') && (c <= 'z')) ? c - ('a'-'A') : c;
}

/*
 * Open a filename with the appriopriate extension &
 * report an error if not possible
 */
FILE *open_file(filename, extension, options)
	char *filename, *extension, *options;
{
	char buffer[100], *ptr, *dot;
	FILE *fp;

	dot = 0;

	for(ptr = buffer; *ptr = *filename; ++ptr) {	/* Copy filename */
		if(*filename == '.')
			dot = filename;
		else if(*filename == '\\')
			dot = 0;
		++filename; }

	if(!dot) {									/* No extension supplied */
		*ptr++ = '.';
		do
			*ptr++ = *extension;
		while(*extension++); }
	else
		*dot = 0;

	if(!(fp = fopen(buffer, options))) {
		fprintf(stderr,"Unable to access: '%s'\n", buffer);
		exit(-1); }

	return fp;
}

/*
 * Main program
 */
main(argc, argv)
	unsigned argc;
	char *argv[];
{
	unsigned i, temp, daddr;
	int stemp;
	char chr, *ptr, *lfile, *cfile;

	if(argc < 2)
		abort("\nUse: asm320 <filename> [-acfiqst c=file l=file p=length w=width]\n\n?COPY.TXT 1983-2005 Dave Dunfield\n**See COPY.TXT**.\n");

/* parse for command line options. */
	lfile = cfile = argv[1];
	for(i=2; i < argc; ++i) {
		if(*(ptr = argv[i]) == '-') {		/* Enable switch */
			while(*++ptr) switch(toupper(*ptr)) {
				case 'A' : ljmp = 0;		break;
				case 'C' : casf = -1;		break;
				case 'F' : fulf = -1;		break;
				case 'I' : intel = -1;		break;
				case 'Q' : quietf = -1;		break;
				case 'S' : symf = -1;		break;
				case 'T' : lfile = 0;		break;
				default: goto badopt; }
			continue; }
		if(*++ptr == '=') switch(toupper(*(ptr++ - 1))) {
			case 'L' : lfile = ptr;			continue;
			case 'C' : cfile = ptr;			continue;
			case 'P' : pagel=atoi(ptr)-1;	continue;
			case 'W' : pagew=atoi(ptr);		continue; }
	badopt:
		fprintf(stderr,"Invalid option: %s\n", argv[i]);
		exit(-1); }

	asm_fp = open_file(argv[1], "ASM", "r");
	hex_fp = open_file(cfile, "HEX", "w");
	lst_fp = (lfile) ? open_file(lfile, "LST", "w") : stdout;
	strncpy(title,argv[1],sizeof(title)-1);

/* first pass - build symbol table */
	symtop = symtab;
	addr = 0;
	line = 1;
	if(!quietf)	fprintf(stderr,"First pass... ");
	while(readline()) {
		if(!optr) {
			emsg = length = optr = ptr = 0;
			if(label[0]) {
				if(lookup_symbol(label)) {
					fprintf(lst_fp,"** Line %u - Duplicate symbol: '%s'\n",line,label);
					++ecnt; }
				else {
					if(!(ptr = create_symbol(label, addr))) {
						xabort("symbol");
						fprintf(lst_fp,"** Line %u - Symbol table overflow\n", line);
						++ecnt;
						break; } } }
			locins(instruction);
			switch(itype) {
				case 1 :			/* no operand */
				case 7 :			/* XCHD */
				case 10 :			/* accumulator only */
				case 18 :			/* JMP */
				case 20 :			/* MOVX */
				case 21 :			/* MOVC */
					++length;
					break;
				case 2 :			/* complex, table1 */
					eval_cplx(-1, 0, it1, 0, 0);
					break;
				case 3 :			/* MOV */
					if(partial_match(operand,"DPTR,"))
						length = 3;
					else
						eval_cplx(-1, -1, it2, 0x2e, 0x1e);
					break;
				case 4 :			/* anl etc */
					eval_cplx(-1, -1, it3, 0x30, 0);
					break;
				case 5 :			/* xch etc */
					eval_cplx(-1, 0, it4, 0, 0);
					break;
				case 6 :			/* XRL etc. */
					eval_cplx(-1, -1, it3, 0, 0);
					break;
				case 19 :			/* LCALL & LJMP */
					if(ljmp) {
						length = 3;
						break; }
				case 8 :			/* ACALL & AJMP */
				case 11 :			/* SJMP */
				case 12 :			/* PUSH, POP etc */
					length = 2;
					break;
				case 9 :			/* CJNE */
				case 16 :			/* JB, JNB, JBC */
					length = 3;
					break;
				case 13 :			/* CLR */
					eval_bit(0xe4, 0xc3, 0xc2);
					break;
				case 14 :			/* CPL */
					eval_bit(0xf4, 0xb3, 0xb2);
					break;
				case 15 :			/* SETB */
					eval_bit(0, 3, 2);
					break;
				case 17 :			/* DJNZ */
					length = 2;
					if(eval_type(0) == 2)
						++length;
					break;
				case 22 :			/* INC */
				case 23 :			/* DEC */
					++length;
					if(!partial_match(operand,"DPTR"))
						if(eval_type(-1) == 2)
							++length;
					break;
				case 100 :			/* EQU directive */
					bit = 0;
					set_symbol(ptr, eval());
					if((opcode == 1) || (bit && !opcode))
						*ptr |= BIT_TYPE;
				err1st:
					if(emsg) {
						fprintf(lst_fp,"** Line %u - %s\n",line,etext[emsg]);
						++ecnt; }
					break;
				case 101 :			/* ORG directive */
					addr=eval();
					goto err1st;
				case 102 :			/* DB statement */
					length=1;
					while(nxtelem()) ++length;
					break;
				case 103 :			/* DW statements */
					length=2;
					while(nxtelem()) length += 2;
					break;
				case 104 :			/* STR statements */
					chr=operand[0];
					for(i=1;(operand[i])&&(chr!=operand[i]); ++i)
						++length;
					if(opcode == 2) ++length;
					break;
				case 105 :			/* DS statement */
					addr = addr + eval();
					goto err1st;
				case 106 :			/* END directive */
					goto end1;
				case 125 :			/* LINE directive */
					line = eval(); } }
		else
			length = 0;
		addr += length;
		++line; /* += (itype < 120); */ }
end1:
	rewind(asm_fp);

/* second pass - generate object code */

	if(!quietf)
		fprintf(stderr,"Second pass... ");
	addr = emsg = 0; line = pcount = 1; lcount = 9999;
	while(readline()) {
		daddr = addr;
		if(!optr) {
			if(!locins(instruction))
				reperr(1);			/* unknown opcode */
			length = temp = optr = 0;
			switch(itype) {
				case 10 :		/* 'A' operand only */
					if(eval_type(-1)) reperr(3);
				case 1 :			/* no operand */
					instruction[length++] = opcode;
					break;
				case 2 :			/* complex, table 1 */
					eval_cplx(-1, 0, it1, 0, 0);
					break;
				case 3 :			/* MOV */
					if(partial_match(operand,"DPTR,")) {
						optr += 5;
						if(eval_type(-1) != 1)
							reperr(3);
						instruction[length++] = 0x90;
						instruction[length++] = value >> 8;
						instruction[length++] = value; }
					else
						eval_cplx(-1, -1, it2, 0x2e, 0x1e);
					break;
				case 4 :			/* ANL etc */
					eval_cplx(-1, -1, it3, 0x30, 0);
					instruction[0] += bit_comp;
					break;
				case 5 :			/* XCH etc */
					eval_cplx(-1, 0, it4, 0, 0);
					break;
				case 6 :			/* XRL etc. */
					eval_cplx(-1, -1, it3, 0, 0);
					break;
				case 7 :			/* XCHD */
					eval_cplx(-1, -1, it5, 0, 0);
					break;
				case 19 :			/* LCALL & LJMP */
					if(ljmp) {
						instruction[length++] = opcode;
						value = eval();
						instruction[length++] = value >> 8;
						instruction[length++] = value;
						break; }
					--opcode;		/* Convert */
				case 8 :			/* ACALL & AJMP */
					value = eval();
					if((value & 0xF800) != ((addr+2) & 0xF800))
						reperr(2);
					instruction[length++] = ((value >> 3) & 0xe0) | opcode;
					instruction[length++] = value;
					break;
				case 9 :			/* CJNE */
					eval_cplx(-1, 0, it6, 0, 0);
gojmp1:				if(operand[optr++] != ',')
						reperr(7);
					goto gojmp;
				case 11 :			/* SJMP */
					instruction[length++] = opcode;
gojmp:				stemp = (eval() - addr) - (length + 1);
					if((stemp > 127) || (stemp < -128))
						reperr(2);				/* out or range */
					instruction[length++] = stemp;
					break;
				case 12 :			/* push, pop etc */
					if(eval_type(0) != 2)
						reperr(3);
					instruction[length++] = opcode;
					instruction[length++] = value;
					break;
				case 13 :			/* CLR */
					eval_bit(0xe4, 0xc3, 0xc2);
					break;
				case 14 :			/* CPL */
					eval_bit(0xf4, 0xb3, 0xb2);
					break;
				case 15 :			/* SETB */
					eval_bit(0, 3, 2);
					break;
				case 16 :			/* JB, JNB, JBC */
					instruction[length++] = opcode;
					instruction[length++] = bit_address();
					goto gojmp1;
				case 17 :			/* DJNZ */
					++length;
					if((temp=eval_type(0)) == 4) {	/* Gen'l Reg. */
						instruction[0] = opcode + 3 + value;
						goto gojmp1; }
					else if(temp == 2) {			/* direct */
						instruction[0] = opcode;
						instruction[length++] = value;
						goto gojmp1; }
					else
						reperr(3);
					break;
				case 18 :			/* JMP */
					if(!(partial_match(operand,"[A+DPTR]")
					||	partial_match(operand,"@A+DPTR")))
						reperr(7);
					instruction[length++] = opcode;
					break;
				case 20 :			/* MOVX */
					if(	partial_match(operand,"A,[DPTR]")
					||	partial_match(operand,"A,@DPTR"))
						instruction[length++] = opcode;
					else if(partial_match(operand,"[DPTR],A")
					||	partial_match(operand,"@DPTR,A"))
						instruction[length++] = opcode + 0x10;
					else
						eval_cplx(-1, -1, it7, 0, 0);
					break;
				case 21 :			/* MOVC */
					++length;
					if(	partial_match(operand,"A,[A+PC]")
					||	partial_match(operand,"A,@A+PC"))
						instruction[0] = opcode;
					else if(partial_match(operand,"A,[A+DPTR]")
					||	partial_match(operand,"A,@A+DPTR"))
						instruction[0] = opcode + 0x10;
					else
						reperr(7);
					break;
				case 22 :			/* INC */
					if(partial_match(operand,"DPTR")) {
						instruction[length++] = 0xa3;
						break; }
				case 23 :			/* DEC */
					++length;
					if((temp = eval_type(-1)) == 1)
						reperr(3);
					else if(temp == 2) {
						++opcode;
						instruction[length++] = value; }
					else if(temp == 3)
						opcode += 2 + value;
					else if(temp == 4)
						opcode += 4 + value;
					instruction[0] = opcode;
					break;
				case 100 :			/* equate statement */
					daddr=eval();	/* generate errors if any */
					break;
				case 101 :			/* ORG statement */
					if(ocnt) wrrec();
					daddr=addr=eval();
					break;
				case 102 :			/* DB statement */
					do {
						instruction[length++] = eval(); }
					while(operand[optr-1] == ',');
					break;
				case 103 :			/* DW statements */
					do {
						temp = eval();
						if(opcode) {		/* Normal order */
							instruction[length++] = temp >> 8;
							instruction[length++] = temp; }
						else {				/* Reverse order */
							instruction[length++] = temp;
							instruction[length++] = temp >> 8; } }
					while(operand[optr-1] == ',');
					break;
				case 104 :			/* STR statements */
					chr=operand[0];
					for(i=1;((operand[i])&&(chr!=operand[i])); ++i)
						instruction[length++] = operand[i];
					if(!operand[i])	reperr(8);
					if(opcode == 1)
						instruction[length-1] = instruction[length-1] | 0x80;
					if(opcode == 2)
						instruction[length++]=0;
					break;
				case 105 :			/* DS statement */
					if(ocnt) wrrec();
					addr += eval();
					break;
				case 106 :			/* END statement */
					goto end2;
				case 120 :			/* PAGE statement */
					lcount=9999;
					break;
				case 121 :			/* TITLE directive */
					strncpy(title, operand, sizeof(title)-1);
					break;
				case 122 :			/* SPACE directive */
					fprintf(lst_fp,"\n");
					++lcount;
					break;
				case 123 :			/* LIST directive */
					if(nlist) --nlist;
					break;
				case 124 :			/* NOLIST directive */
					++nlist;
					break;
				case 125 :			/* LINE directive */
					line = eval(); } }
		else
			length=itype=0;

/* generate listing */
		if(((itype<120) && fulf && !nlist) || emsg) {
			if(++lcount >= pagel)
				write_title();
			fprintf(lst_fp,"%04x ",daddr);
				for(i=0; i < 6; ++i) {
					if(i < length)
						fprintf(lst_fp," %02x", instruction[i]);
					else
						fprintf(lst_fp,"   "); }
				fprintf(lst_fp," %c%5u  %s\n",(length <=6) ? ' ' : '+',line,buffer); }
			if(emsg) {
				fprintf(lst_fp,"  ** ERROR ** - %u - %s\n", emsg, etext[emsg]);
				++ecnt; ++lcount;
				emsg=0; }
			++line; /* += (itype<120); */
/* write code to output record */
		for(i=0; i<length; ++i) {
			if(!ocnt) {			/* first byte of record */
				outrec[ocnt++]=addr>>8;
				outrec[ocnt++]=addr; }
			++addr;
			outrec[ocnt++]=instruction[i];
			if(ocnt>33) wrrec(); } }	/* record is full, write it */

end2:
	if(ocnt) wrrec();
	if(intel) fprintf(hex_fp,":00000001FF\n");
	else fprintf(hex_fp,"S9030000FC\n");

	if(ecnt) fprintf(lst_fp,"\n %u error(s) occurred in this assembly.\n",ecnt);
	if(!quietf) fprintf(stderr,"%u error(s).\n",ecnt);

/* display the symbol table */
	if(symf) {
		write_title();
		dump_symbols(); }

	fclose(asm_fp);
	fclose(hex_fp);
	fclose(lst_fp);

	return ecnt ? -2 : 0;
}

/***********************************************/
/* lookup instruction in the instruction table */
/***********************************************/
int locins(inst)
	char inst[];
{
	register char *ptr, *ptr1;

	ptr = opcodes;
	do {
		ptr1 = inst;
		while(*ptr == *ptr1) {
			++ptr;
			++ptr1; }
		if((*ptr < 0) && !*ptr1) {
			itype = *ptr++ & 127;
			opcode = *ptr++;
			return 1; }
		while(*ptr > 0)
			++ptr; }
	while(*(ptr += 2));

	itype = 119;
	return(inst[0] == 0);
}

/*************************************************/
/* read a line from input file, and break it up. */
/*************************************************/
int readline()
{
	register int i;
	register char *ptr;

	if(!fgets(ptr = buffer, LINE_SIZE, asm_fp))
		return 0;

	i = 0;
	while(issymbol(*ptr)) {
		label[i] = chupper(*ptr++);
		if(i < SYMB_SIZE)
			++i; }
	label[i]=0;
	if(*ptr == ':')
		++ptr;
	while(isspace(*ptr))
		++ptr;
	if(((*ptr != '*') && (*ptr != ';')) || i) {	/* non comment line */
		i = 0;
		while(*ptr && !isspace(*ptr))
			instruction[i++] = toupper(*ptr++);
		instruction[i]=0;
		while(isspace(*ptr))
			++ptr;
		optr = i = 0;
		while(*ptr && (i < 79))
			operand[i++] = *ptr++;
		operand[i] = 0;
		return 1; }

	return optr = 1;
}

/***************************************/
/* test for valid terminator character */
/***************************************/
int isterm(chr)
	register char chr;
{
	switch(chr) {
		case 0 :
		case ' ' :
		case '\t':
		case ',' :
		case '.' :
		case ')' :
			return 1; }
	return 0;
}

/*************************************/
/* Test for a valid symbol character */
/*************************************/
int issymbol(c)
	char c;
{
	switch(c) {			/* Special allowed characters */
		case '_' :
		case '?' :
		case '!' :
			return 1; }

	return isalnum(c);
}

/* report an error */
reperr(errn)
	register int errn;
{
	if(!emsg) emsg=errn;
}

/*******************************/
/* write record to output file */
/*******************************/
wrrec()
{
	register unsigned i, chk, chr;

	xhex(ocnt-2);
	if(intel) {					/* intel hex format */
		chk = outrec[0] + outrec[1] + ocnt - 2;
		fprintf(hex_fp,":%02x%02x%02x00", ocnt-2, outrec[0], outrec[1]);
		for(i=2; i<ocnt; ++i) {
			fprintf(hex_fp,"%02x", chr = outrec[i]);
			chk += chr; }
		fprintf(hex_fp,"%02x\n", (0-chk) & 255); }
	else {						/* motorola hex format */
		chk = ocnt + 1;
		fprintf(hex_fp,"S1%02x", ocnt + 1);
		for(i=0; i<ocnt; ++i) {
			fprintf(hex_fp,"%02x", chr = outrec[i]);
			chk += chr; }
		fprintf(hex_fp,"%02x\n", 255 & ~chk); }
	ocnt = 0;
}

/*
 * Evaluate an expression.
 */
int eval()
{	register char c;
	unsigned result;

	result=getval();
	while(!isterm(c = getchr())) switch(c) {
		case '+' : result += getval();	break;
		case '-' : result -= getval();	break;
		case '*' : result *= getval();	break;
		case '/' : result /= getval();	break;
		case '\\': result %= getval();	break;
		case '&' : result &= getval();	break;
		case '|' : result |= getval();	break;
		case '^' : result ^= getval();	break;
		case '<' : result <<= getval();	break;
		case '>' : result >>= getval();	break;
		default: reperr(6); }

	return result;
}

/*
 * Get a single value (number or symbol)
 */
int getval()
{
	unsigned i, b, val;
	char chr, *ptr, array[25], bflag;

	bflag = 0;
	switch(chr = operand[optr++]) {
		case '-' :				/* Negated value */
			return 0 - getval();
		case '~' :				/* Complemented value */
			return ~getval();
		case '=' :				/* Swap high and low bytes */
			i=getval();
			return (i<<8)+(i>>8);
		case '(' :				/* Nested expression */
			val = eval();
			if(operand[optr-1] != ')')
				reperr(6);
			return val; }
	--optr;

	val=b=i=0;
	switch(chr) {
		case '$' :	b = 16;	goto getnum;
		case '@' :	b = 8;	goto getnum;
		case '%' :	b = 2;	goto getnum; }
	if(isdigit(chr)) {
		array[i++] = chr;
	getnum:
		while(isalnum(chr = toupper(operand[++optr])))
			array[i++] = (chr < 'A') ? chr : chr - 7;
		if((b == 16) && !i)
			val = addr;
		else {
			if(!b) {
				b = 10;
				switch(array[i-1]) {
					case 'H'-7 :	b = 16;	goto deci;
					case 'T'-7 :
					case 'D'-7 :	b = 10;	goto deci;
					case 'Q'-7 :
					case 'O'-7 :	b = 8;	goto deci;
					case 'B'-7 :	b = 2;
					deci:	--i;	} }
			if(!i)
				reperr(6);
			for(ptr = array; i; --i) {
				if((chr = *ptr++ - '0') >= b)
					reperr(6);
				val = (val * b) + chr; } }
		bflag = -1; }
	else if(chr=='\'') {				/* Single quote. */
		++optr;
		while(((chr=operand[optr++]) != '\'') && chr)
			val=(val << 8) + chr;
		if(!chr) reperr(8); }
	else if(chr=='*') {					/* Program counter */
		++optr;
		val=addr; }
	else {								/* Must be a label */
		i = 0;
		while(issymbol(chr = chupper(operand[optr]))) {
			++optr;
			label[i]=chr;
			if(i < SYMB_SIZE)
				++i; }
		label[i]=0;

		if(!(ptr = lookup_symbol(label)))
			reperr(5);
		val = value;
		if(*ptr & BIT_TYPE)
			bit = -1;
		else
			bflag = -1; }

	if(bflag && (operand[optr] == '.')) {
		bit = -1;
		++optr;
		if((i = getval()) > 7)
			reperr(2);
		if((val >= 0x20) && (val <= 0x2F))
			val = (val - 0x20) * 8;
		else if((val < 0x80) || (val > 0xFF) || (val & 0x07))
			reperr(2);
		val += i; }

	return val;
}

/***********************************************/
/* evaluate an operand, and determine its type */
/* 0 = accumulator           1 = immediate     */
/* 2 = direct                3 = indirect      */
/* 4 = gen'l register                          */
/* operand value is set in global var. 'value' */
/* non-zero passed parm indicates that 'A'     */
/* should be resolved to an accumulator ref.   */
/***********************************************/
eval_type(acc)
	int acc;
{
	register char chr;

	bit = 0;
	if((chr = chupper(operand[optr++])) == '#') {	/* immediate type */
		value = eval();
		bit = 0;
		--optr;
		return(1); }
	else if((chr == 'A') && acc && (operand[optr] != '.') && !issymbol(operand[optr]))	/* accumulator */
		return(0);
	else if((chr == '@') && (chupper(operand[optr]) == 'R')) {
		if((value = operand[++optr] - '0') > 1)
			reperr(4);
		++optr;
		return 3; }
	else if(chr == '[') {				/* indirect */
		if(chupper(operand[optr++]) != 'R')
			reperr(4);
		else {
			if((value = operand[optr++] - '0') > 1)
				reperr(4);
			if(operand[optr++] != ']')
				reperr(7);
			return(3); } }
	else if((chr == 'R') && isdigit(operand[optr]) && !issymbol(operand[optr+1])) {
		if((value = operand[optr++] - '0') > 7)
				reperr(4);
		return(4); }
	else {					/* direct reference */
		--optr;
		value = eval();
		--optr;
		return(2); }
}

/* evaluates a bit address */
bit_address()
{
	unsigned val;
	bit = 0;
	if(operand[optr] == '/') {
		++optr;
		bit_comp = 0x2e; }
	val = eval();
	--optr;
	if(!bit)
		reperr(3);
	return val;
}

/* evaluate bit oriented or accumlator operands */
eval_bit(acc, cy, badr)
	char acc, cy, badr;
{
	register char chr1, chr2;
	chr1 =chupper(operand[optr]);
	chr2 = operand[optr+1];

	if((chr1 == 'A') && (chr2 != '.') && !issymbol(chr2)) {	/* A */
		if(!acc) reperr(3);
		instruction[length++] = opcode + acc; }
	else if((chr1 == 'C') && !issymbol(chr2)) {				/* C */
		if(!cy) reperr(3);
		instruction[length++] = opcode + cy; }
	else {													/* bit addr */
		if(!badr) reperr(3);
		instruction[length++] = opcode + badr;
		instruction[length++] = bit_address(); }
}

/*
 * Write out title
 */
write_title()
{
	int w;
	char *ptr;

	if(pcount > 1)
		putc('\f',lst_fp);
	lcount = 1;
	fprintf(lst_fp,"DUNFIELD 80C320 ASSEMBLER: ");
	ptr = title;
	for(w = 37; w < pagew; ++w)
		putc(*ptr ? *ptr++ : ' ', lst_fp);
	fprintf(lst_fp,"PAGE: %u\n\n", pcount);
	++pcount;
}

/**************************************/
/* find next element in argument list */
/**************************************/
int	nxtelem() {
	register char chr;
	while((chr=operand[optr])&&(chr != ' ')&&(chr != 9)) {
		++optr;
		if(chr==39) {
			while((chr=operand[optr])&&(chr!=39)) ++optr;
			++optr; }
		if(chr==',') return(1); }
	return(0);
}

/*
 * Evaluate complex arguments
 */
eval_cplx(d1, d2, table, c_bit, bit_c)
	char d1, d2, table[], c_bit, bit_c;
{
	char t1, t2, opc, bit1;
	unsigned v1, v2;

	bit_comp = bit = 0;
	t2 = opc = 0;

	if((chupper(operand[optr]) == 'C') && (operand[optr+1] == ',')) {
		if(!c_bit) reperr(3);
		opcode += c_bit;
		optr += 2;
		t1 = 2;
		bit1 = -1;
		v1 = bit_address(); }
	else {
		if(!(t1 = eval_type(d1)))		/* only one acc. ref allowed */
			d2 = 0;
		v1 = value;
		bit1 = bit;
		bit = 0;
		if(operand[optr++] != ',')
			reperr(7);

		if((chupper(operand[optr]) == 'C') && (operand[optr+1] <= ' ')) {
			if(!bit_c) reperr(3);
			opcode += bit_c;
			bit = -1; }
		else {
			t2 = eval_type(d2);
			v2 = value;

			if(((opc = table[(t1*5)+t2]) == -1) || bit)
				reperr(3);

			if(t1 > 2)
				opc += v1;
			if(t2 > 2)
				opc += v2; } }

	if(bit != bit1)
		reperr(3);

	length = 0;
	instruction[length++] = opcode + opc;
	if((t1 == 2) && (t2 == 2)) {		/* direct to direct move */
		instruction[length++] = v2;
		instruction[length++] = v1; }
	else {
		if((t1 == 1) || (t1 == 2))
			instruction[length++] = v1;
		if((t2 == 1) || (t2 == 2))
			instruction[length++] = v2; }
}

partial_match(s1, s2)
	char *s1, *s2;
{
	while(*s2)
		if(chupper(*s1++) != *s2++)
			return(0);
	return(1);
}
